/* Copyright (c) 2021-2022 Intel Corporation

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
==============================================================================*/

///////////////////////////////////////////////////////////////////////////
// Code to be compatible with the pb generated by internal Quantization Tools,
// such as QuantizedFusedMatMul, QuantizedFusedMatMulAndRequantize, etc.
///////////////////////////////////////////////////////////////////////////

#include "itex/core/kernels/common/no_ops.h"
#include "itex/core/kernels/legacy/matmul_common.h"
#include "itex/core/kernels/onednn/block/quantized_ops.h"
#include "itex/core/utils/errors.h"
#include "itex/core/utils/onednn/onednn_layout_util.h"
#include "itex/core/utils/onednn/onednn_post_op_util.h"
#include "itex/core/utils/onednn/onednn_util.h"
#include "itex/core/utils/op_kernel.h"
#include "itex/core/utils/op_requires.h"
#include "itex/core/utils/plugin_tensor.h"
#include "itex/core/utils/quantization_util.h"
#include "itex/core/utils/register_types.h"
#include "itex/core/utils/types.h"
#include "third_party/eigen3/unsupported/Eigen/CXX11/Tensor"

namespace itex {

using memory = dnnl::memory;

// Currently, Targs = Tbias. We may improve such design
template <typename Device, typename Tinput, typename Tweight, typename Tbias,
          typename Toutput>
class OneDnnQuantizedFusedMatMulOp
    : public LegacyOneDnnQuantizedMatMulOpBase<Device, Tinput, Tweight, Tbias,
                                               Toutput> {
 public:
  explicit OneDnnQuantizedFusedMatMulOp(OpKernelConstruction* context)
      : LegacyOneDnnQuantizedMatMulOpBase<Device, Tinput, Tweight, Tbias,
                                          Toutput>(context) {
    // Quantize mode assignment
    string mode_string;
    OP_REQUIRES_OK(context, context->GetAttr("input_quant_mode", &mode_string));
    if (mode_string == "MIN_FIRST") {
      this->mode_ = QuantizeMode::MIN_FIRST;
    } else if (mode_string == "SCALED") {
      this->mode_ = QuantizeMode::SCALED;
    } else {
      context->CtxFailure(errors::InvalidArgument(
          "Quantization mode must be either MIN_FIRST or SCALED, but received ",
          mode_string));
    }

    OP_REQUIRES_OK(context,
                   context->GetAttr("transpose_a", &this->transpose_a_));
    OP_REQUIRES_OK(context,
                   context->GetAttr("transpose_b", &this->transpose_b_));

    // weight/bias const flag set
    OP_REQUIRES_OK(context, context->GetAttr("is_filter_const",
                                             &(this->is_weight_const_)));
    OP_REQUIRES_OK(context,
                   context->GetAttr("is_bias_const", &(this->is_bias_const_)));
    // Set alpha if get `LeakyRelu` after adding ops.

    // PostOpUtil set
    OP_REQUIRES_OK(context, context->GetAttr("fused_ops", &fused_ops_));
    OP_REQUIRES(context, fused_ops_.size() <= 2,
                errors::InvalidArgument(
                    "_QuantizedFusedMatMul supports maximum 2 post ops"));
    OP_REQUIRES(context, fused_ops_.size() == 0 || fused_ops_[0] == "BiasAdd",
                errors::InvalidArgument(
                    "_QuantizedFusedMatMul first post op must be BiasAdd"));
    OP_REQUIRES(context, this->post_op_util_.AddOps(fused_ops_),
                errors::InvalidArgument(
                    "Found unsupported fusion in _QuantizedFusedMatMul."));

    // Set alpha if get `LeakyRelu` after adding ops.
    if (this->post_op_util_.HasLeakyRelu()) {
      float alpha;
      OP_REQUIRES_OK(context, context->GetAttr("leakyrelu_alpha", &alpha));
      this->post_op_util_.SetLeakyReluAlpha(alpha);
    }

    // Set input/output tensor index
    bool has_postop_add = std::find(fused_ops_.begin(), fused_ops_.end(),
                                    "Add") != fused_ops_.end();
    this->kSrcMinRangeIndex = has_postop_add + 3;
    this->kSrcMaxRangeIndex = has_postop_add + 4;
    this->kFilterMinRangeIndex = has_postop_add + 5;
    this->kFilterMaxRangeIndex = has_postop_add + 6;
    this->kMinFreezedIndex = has_postop_add + 7;
    this->kMaxFreezedIndex = has_postop_add + 8;
    this->kDstMinRangeIndex = 1;
    this->kDstMaxRangeIndex = 2;
  }

  void Compute(OpKernelContext* context) override {
    LegacyOneDnnQuantizedMatMulOpBase<Device, Tinput, Tweight, Tbias,
                                      Toutput>::Compute(context);
  }

  void ExtendInt8PostOps(OpKernelContext* context) override {
    if (std::is_same<Toutput, qint32>::value) {
      // If output is qint32, the scale for MatMul is 1.0f and the scales for
      // activations is 1.0f (default).
      this->post_op_util_.SetOutputScale({1.0f});
    } else if (std::is_same<Toutput, qint8>::value ||
               std::is_same<Toutput, quint8>::value ||
               std::is_same<Toutput, float>::value ||
               std::is_same<Toutput, Eigen::bfloat16>::value ||
               std::is_same<Toutput, Eigen::half>::value) {
      // When Toutput is float, the fusion semantic has its output dequantized,
      // and when Toutput is q{u}int8 the fusion semantic has its output
      // requantized.
      const float min_input =
          context->input(this->kSrcMinRangeIndex).template flat<float>()(0);
      const float max_input =
          context->input(this->kSrcMaxRangeIndex).template flat<float>()(0);
      const Tensor& min_weight_tensor =
          context->input(this->kFilterMinRangeIndex);
      const Tensor& max_weight_tensor =
          context->input(this->kFilterMaxRangeIndex);
      const float* min_weight = min_weight_tensor.flat<float>().data();
      const float* max_weight = max_weight_tensor.flat<float>().data();
      const size_t num_output_channels = min_weight_tensor.NumElements();

      const float max_int8_input =
          (std::is_same<Tinput, quint8>::value) ? 255.0f : 127.0f;
      const float max_int8_weight =
          (std::is_same<Tweight, quint8>::value) ? 255.0f : 127.0f;
      const float range_input =
          (this->mode_ == QuantizeMode::MIN_FIRST)
              ? max_input - min_input
              : std::max(std::abs(min_input), std::abs(max_input));

      std::vector<float> scale_output(num_output_channels);
      for (size_t i = 0; i < num_output_channels; ++i) {
        float range_weight =
            std::max(std::abs(min_weight[i]), std::abs(max_weight[i]));
        scale_output[i] =
            (range_input * range_weight) / (max_int8_input * max_int8_weight);
      }

      float scale_post_op = 1.0;

      // Note: When Toutput is u8/s8, in other words Requantize mode, the scale
      // calculation needs to take min/max_freezed_output into account.
      if (std::is_same<Toutput, qint8>::value ||
          std::is_same<Toutput, quint8>::value) {
        // Requantize condition
        const float min_output =
            context->input(this->kMinFreezedIndex).template flat<float>()(0);
        const float max_output =
            context->input(this->kMaxFreezedIndex).template flat<float>()(0);
        const float range_output =
            std::max(std::abs(min_output), std::abs(max_output));

        // Note: INT8 primitive needs different scale factor calculation with or
        // without post op activations, when output is qint8 or quint8.
        if (fused_ops_.size() == 1) {
          // No post op activations
          if (std::is_same<Toutput, qint8>::value) {
            for (size_t i = 0; i < scale_output.size(); ++i) {
              scale_output[i] = scale_output[i] * (127.0f / range_output);
            }
          } else if (std::is_same<Toutput, quint8>::value) {
            for (size_t i = 0; i < scale_output.size(); ++i) {
              scale_output[i] = scale_output[i] * (255.0f / range_output);
            }
          }
        } else {
          // Has post op activations
          if (std::is_same<Toutput, qint8>::value) {
            scale_post_op = 127.0f / range_output;
          } else if (std::is_same<Toutput, quint8>::value) {
            scale_post_op = 255.0f / range_output;
          }
        }
      }

      // Set output scale
      this->post_op_util_.SetOutputScale(scale_output);

      // Set postop scale if needed
      if (fused_ops_.size() == 2) {
        string postop = fused_ops_[1];
        if (postop != "Add") {
          this->post_op_util_.SetPostOpScale(postop, scale_post_op);
        } else {
          this->post_op_util_.SetPostOpScale("Add", 1.0f);
        }
      }
    }
  }

 protected:
  // QuantizedFusedMatMul needs the fused ops information during runtime. This
  // information is necessary in calculating scales.
  std::vector<string> fused_ops_;
};

#ifdef INTEL_CPU_ONLY
#define REGISTER_ONEDNN_KERNEL(op, kernel, input_type, args_type, output_type) \
  REGISTER_KERNEL_BUILDER(                                                     \
      Name(op)                                                                 \
          .Device(DEVICE_CPU)                                                  \
          .TypeConstraint<input_type>("T1")                                    \
          .TypeConstraint<qint8>("T2")                                         \
          .TypeConstraint<args_type>("Targs")                                  \
          .TypeConstraint<output_type>("Toutput"),                             \
      kernel TEMPLATE_ARGS(CPUDevice, input_type, args_type, output_type));
#else
#define REGISTER_ONEDNN_KERNEL(op, kernel, input_type, args_type, output_type) \
  REGISTER_KERNEL_BUILDER(                                                     \
      Name(op)                                                                 \
          .Device(DEVICE_GPU)                                                  \
          .TypeConstraint<input_type>("T1")                                    \
          .TypeConstraint<qint8>("T2")                                         \
          .TypeConstraint<args_type>("Targs")                                  \
          .TypeConstraint<output_type>("Toutput") HOSTMEMORYLIST,              \
      kernel TEMPLATE_ARGS(GPUDevice, input_type, args_type, output_type));
#endif  // INTEL_CPU_ONLY

#define REGISTER_ONEDNN_KERNEL_ALL_INPUT_TYPES(op, kernel, args_type, \
                                               output_type)           \
  REGISTER_ONEDNN_KERNEL(op, kernel, qint8, args_type, output_type);  \
  REGISTER_ONEDNN_KERNEL(op, kernel, quint8, args_type, output_type);

#ifdef INTEL_CPU_ONLY
#define REGISTER_ONEDNN_KERNEL_ALL_INPUT_AND_ARGS_TYPES(op, kernel,        \
                                                        output_type)       \
  REGISTER_ONEDNN_KERNEL_ALL_INPUT_TYPES(op, kernel, float, output_type);  \
  REGISTER_ONEDNN_KERNEL_ALL_INPUT_TYPES(op, kernel, Eigen::bfloat16,      \
                                         output_type);                     \
  REGISTER_ONEDNN_KERNEL_ALL_INPUT_TYPES(op, kernel, qint32, output_type); \
  REGISTER_ONEDNN_KERNEL_ALL_INPUT_TYPES(op, kernel, quint8, output_type); \
  REGISTER_ONEDNN_KERNEL_ALL_INPUT_TYPES(op, kernel, qint8, output_type);
#else
#define REGISTER_ONEDNN_KERNEL_ALL_INPUT_AND_ARGS_TYPES(op, kernel,        \
                                                        output_type)       \
  REGISTER_ONEDNN_KERNEL_ALL_INPUT_TYPES(op, kernel, float, output_type);  \
  REGISTER_ONEDNN_KERNEL_ALL_INPUT_TYPES(op, kernel, Eigen::bfloat16,      \
                                         output_type);                     \
  REGISTER_ONEDNN_KERNEL_ALL_INPUT_TYPES(op, kernel, Eigen::half,          \
                                         output_type);                     \
  REGISTER_ONEDNN_KERNEL_ALL_INPUT_TYPES(op, kernel, qint32, output_type); \
  REGISTER_ONEDNN_KERNEL_ALL_INPUT_TYPES(op, kernel, quint8, output_type); \
  REGISTER_ONEDNN_KERNEL_ALL_INPUT_TYPES(op, kernel, qint8, output_type);
#endif  // INTEL_CPU_ONLY

// Concrete OneDnn MatMul INT8 kernel implementation
#define TEMPLATE_ARGS(Device, input_type, args_type, output_type) \
<Device, input_type, qint8, args_type, output_type>

#define HOSTMEMORYLIST                                                \
  .HostMemoryList4("min_a", "max_a", "min_b", "max_b")                \
      .HostMemoryList7("a_meta", "b_meta", "args_meta", "min_a_meta", \
                       "max_a_meta", "min_b_meta", "max_b_meta")      \
      .HostMemoryList2("min_product", "max_product")                  \
      .HostMemoryList3("product_meta", "min_product_meta", "max_product_meta")
REGISTER_ONEDNN_KERNEL_ALL_INPUT_AND_ARGS_TYPES("_OneDnnQuantizedFusedMatMul",
                                                OneDnnQuantizedFusedMatMulOp,
                                                qint32);
#undef HOSTMEMORYLIST

#define HOSTMEMORYLIST                                                \
  .HostMemoryList4("min_a", "max_a", "min_b", "max_b")                \
      .HostMemoryList7("a_meta", "b_meta", "args_meta", "min_a_meta", \
                       "max_a_meta", "min_b_meta", "max_b_meta")      \
      .HostMemoryList1("product_meta")
#ifdef INTEL_CPU_ONLY
REGISTER_ONEDNN_KERNEL_ALL_INPUT_AND_ARGS_TYPES(
    "_OneDnnQuantizedFusedMatMulAndDequantize", OneDnnQuantizedFusedMatMulOp,
    float);
REGISTER_ONEDNN_KERNEL_ALL_INPUT_AND_ARGS_TYPES(
    "_OneDnnQuantizedFusedMatMulAndDequantize", OneDnnQuantizedFusedMatMulOp,
    Eigen::bfloat16);
#else
REGISTER_ONEDNN_KERNEL_ALL_INPUT_AND_ARGS_TYPES(
    "_OneDnnQuantizedFusedMatMulAndDequantize", OneDnnQuantizedFusedMatMulOp,
    float);
REGISTER_ONEDNN_KERNEL_ALL_INPUT_AND_ARGS_TYPES(
    "_OneDnnQuantizedFusedMatMulAndDequantize", OneDnnQuantizedFusedMatMulOp,
    Eigen::bfloat16);
REGISTER_ONEDNN_KERNEL_ALL_INPUT_AND_ARGS_TYPES(
    "_OneDnnQuantizedFusedMatMulAndDequantize", OneDnnQuantizedFusedMatMulOp,
    Eigen::half);
#endif  // INTEL_CPU_ONLY
#undef HOSTMEMORYLIST

#define HOSTMEMORYLIST                                                       \
  .HostMemoryList6("min_a", "max_a", "min_b", "max_b", "min_freezed_output", \
                   "max_freezed_output")                                     \
      .HostMemoryList9("a_meta", "b_meta", "args_meta", "min_a_meta",        \
                       "max_a_meta", "min_b_meta", "max_b_meta",             \
                       "min_freezed_output_meta", "max_freezed_output_meta") \
      .HostMemoryList2("min_product", "max_product")                         \
      .HostMemoryList3("product_meta", "min_product_meta", "max_product_meta")
REGISTER_ONEDNN_KERNEL_ALL_INPUT_AND_ARGS_TYPES(
    "_OneDnnQuantizedFusedMatMulAndRequantize", OneDnnQuantizedFusedMatMulOp,
    quint8);
REGISTER_ONEDNN_KERNEL_ALL_INPUT_AND_ARGS_TYPES(
    "_OneDnnQuantizedFusedMatMulAndRequantize", OneDnnQuantizedFusedMatMulOp,
    qint8);
#undef HOSTMEMORYLIST
#undef TEMPLATE_ARGS

}  // namespace itex
