use crate::{config::Endpoint, tunnel::utils::iptables::IptablesRuleGenerator};

use anyhow::{Context, Result};
use async_trait::async_trait;

use super::NetfilterIngress;

const IPTABLES_FW_MARK_DEFAULT: u32 = 566;
const IP_ROUTE_TABLE_NUM_DEFAULT: u32 = 239;

#[async_trait]
impl IptablesRuleGenerator for NetfilterIngress {
    async fn gen_script(&self) -> Result<(String, String)> {
        // Detect required external tools
        which::which("iptables")
            .context("The external tool \"iptables\" is not found, please install it")?;
        which::which("ip").context("The external tool \"ip\" is not found, please install it")?;

        let id = self.id;
        let fw_mark = IPTABLES_FW_MARK_DEFAULT;
        let route_table = IP_ROUTE_TABLE_NUM_DEFAULT;
        let listen_port = self.listen_port;

        if self.capture_cgroup.is_empty() {
            tracing::info!(
                "capture_cgroup is empty, will capture all traffic from all cgroups (\"/\")"
            );
        }

        if self.capture_dst.is_empty() {
            tracing::info!("capture_dst is empty, will capture all TCP traffic")
        }

        let mut invoke_script = "".to_owned();
        let mut revoke_script = "".to_owned();

        let mut tproxy_invoke_script = "".to_owned();

        // TODO: add comment to all iptables rules, e.g. "-m comment --comment 'generated by tng'"
        let clean_up_iptables_script = format!(
            "\
            iptables -t mangle -D PREROUTING -p tcp -j TNG_INGRESS_{id}_PREROUTING 2>/dev/null || true ; \
            iptables -t mangle -D OUTPUT -p tcp -j TNG_INGRESS_{id}_OUTPUT_STAGE_1 2>/dev/null || true ; \
            iptables -t mangle -D OUTPUT -p tcp -j TNG_INGRESS_{id}_OUTPUT_STAGE_2 2>/dev/null || true ; \
            iptables -t mangle -F TNG_INGRESS_{id}_PREROUTING 2>/dev/null || true ; \
            iptables -t mangle -X TNG_INGRESS_{id}_PREROUTING 2>/dev/null || true ; \
            iptables -t mangle -F TNG_INGRESS_{id}_OUTPUT_STAGE_1 2>/dev/null || true ; \
            iptables -t mangle -X TNG_INGRESS_{id}_OUTPUT_STAGE_1 2>/dev/null || true ; \
            iptables -t mangle -F TNG_INGRESS_{id}_OUTPUT_STAGE_2 2>/dev/null || true ; \
            iptables -t mangle -X TNG_INGRESS_{id}_OUTPUT_STAGE_2 2>/dev/null || true ; \
            ip route flush table {route_table} 2>/dev/null || true ; \
            ip rule del table {route_table} 2>/dev/null || true ; \
            ",
        );

        tproxy_invoke_script += &clean_up_iptables_script;

        // PREROUTING rules
        tproxy_invoke_script += &format!(
            "\
            iptables -t mangle -N TNG_INGRESS_{id}_PREROUTING ; \
            "
        );

        // In the first stage, we ignore the packets that:
        // - The destination is a local adress on the host
        // - The destination is not a unicast address
        // - The source is not a local adress on the host (to prevent the tng from forwarding packets from other hosts)
        tproxy_invoke_script += &format!(
            "\
            iptables -t mangle -A TNG_INGRESS_{id}_PREROUTING -m addrtype --dst-type LOCAL -j RETURN ; \
            iptables -t mangle -A TNG_INGRESS_{id}_PREROUTING -m addrtype ! --dst-type UNICAST -j RETURN ; \
            iptables -t mangle -A TNG_INGRESS_{id}_PREROUTING -m addrtype ! --src-type LOCAL -j RETURN ; \
            ",
        );

        tproxy_invoke_script += &format!(
            "iptables -t mangle -A TNG_INGRESS_{id}_PREROUTING -p tcp -j TPROXY --on-ip 127.0.0.1 --on-port {listen_port} --tproxy-mark {fw_mark}/0xffffff ;"
        );

        tproxy_invoke_script +=
            &format!("iptables -t mangle -A PREROUTING -p tcp -j TNG_INGRESS_{id}_PREROUTING ;");

        // OUTPUT rules
        tproxy_invoke_script += &format!(
            "\
            ip route add local default dev lo table {route_table} ; \
            ip rule add fwmark {fw_mark}/0xffffff table {route_table} ; \
            iptables -t mangle -N TNG_INGRESS_{id}_OUTPUT_STAGE_1 ; \
            iptables -t mangle -N TNG_INGRESS_{id}_OUTPUT_STAGE_2 ; \
            "
        );

        // In the first stage, we ignore the packets that:
        // - The destination is a local adress on the host
        // - The destination is not a unicast address
        // - The source is not a local adress on the host
        // - The packet is marked with SO_MARK set to {self.so_mark}
        tproxy_invoke_script += &format!(
            "\
            iptables -t mangle -A TNG_INGRESS_{id}_OUTPUT_STAGE_1 -m addrtype --dst-type LOCAL -j RETURN ; \
            iptables -t mangle -A TNG_INGRESS_{id}_OUTPUT_STAGE_1 -m addrtype ! --dst-type UNICAST -j RETURN ; \
            iptables -t mangle -A TNG_INGRESS_{id}_OUTPUT_STAGE_1 -m addrtype ! --src-type LOCAL -j RETURN ; \
            iptables -t mangle -A TNG_INGRESS_{id}_OUTPUT_STAGE_1 -p tcp -m mark --mark {} -j RETURN ; \
            ",
            self.so_mark
        );

        // In the first stage, if the packet match any capture_cgroup, then jump to the next stage.
        if self.capture_cgroup.is_empty() {
            tproxy_invoke_script += &format!(
                "iptables -t mangle -A TNG_INGRESS_{id}_OUTPUT_STAGE_1 -m cgroup --path / -j TNG_INGRESS_{id}_OUTPUT_STAGE_2 ;"
            );
        } else {
            for cgroup in &self.capture_cgroup {
                tproxy_invoke_script += &format!(
                    "iptables -t mangle -A TNG_INGRESS_{id}_OUTPUT_STAGE_1 -m cgroup --path {cgroup} -j TNG_INGRESS_{id}_OUTPUT_STAGE_2 ;"
                );
            }
        }

        // In the second stage, if the packet match any nocapture_cgroup, then we skip it.
        for cgroup in &self.nocapture_cgroup {
            tproxy_invoke_script += &format!(
                "iptables -t mangle -A TNG_INGRESS_{id}_OUTPUT_STAGE_2 -m cgroup --path {cgroup} -j RETURN ;"
            );
        }

        // In the second stage, if the packet match any capture_dst, then we set the netfilter mark on the packet it so that it can be re-routed to local with rule based route rule, and we can handle them in the prerouting chain
        if self.capture_dst.is_empty() {
            tproxy_invoke_script +=
                &format!("iptables -t mangle -A TNG_INGRESS_{id}_OUTPUT_STAGE_2 -p tcp -j MARK --set-mark {fw_mark}/0xffffff ;");
        } else {
            for Endpoint { host, port } in &self.capture_dst {
                if let Some(addr) = &host {
                    tproxy_invoke_script += &format!(
                        "iptables -t mangle -A TNG_INGRESS_{id}_OUTPUT_STAGE_2 -p tcp --dst {addr}/32 --dport {port} -j MARK --set-mark {fw_mark}/0xffffff ;"
                    );
                } else {
                    tproxy_invoke_script += &format!(
                        "iptables -t mangle -A TNG_INGRESS_{id}_OUTPUT_STAGE_2 -p tcp --dport {port} -j MARK --set-mark {fw_mark}/0xffffff ;"
                    );
                }
            }
        }

        // Insert the first stage rule into the OUTPUT chain
        tproxy_invoke_script +=
            &format!("iptables -t mangle -A OUTPUT -p tcp -j TNG_INGRESS_{id}_OUTPUT_STAGE_1 ;");

        invoke_script += &tproxy_invoke_script;
        revoke_script += &clean_up_iptables_script;

        Ok((invoke_script, revoke_script))
    }
}
