#!/usr/bin/python
# @lint-avoid-python-3-compatibility-imports
#
# alisyscount   Summarize syscall counts and latencies.
#               Enhanced version of BCC syscount.
#               For Linux, uses BCC, eBPF.
#
# USAGE: alisyscount [-h] [-T] [-N] [-D] [-S] [-P] [-s syscall] [-p pid] [interval] [outputs]
#
# Copyright (c) 2015 Brendan Gregg.
# Copyright (c) 2019 Jeffle Xu, Alibaba, Inc.
# Licensed under the Apache License, Version 2.0 (the "License")

from __future__ import print_function
from bcc import BPF
from time import sleep, strftime
import argparse
import ctypes
from bcc.syscall import syscall_name, syscalls

# arguments
examples = """examples:
    ./alisyscount            # sum syscall counts and latencies.
    ./alisyscount -S         # show per-syscall statistics
    ./alisyscount -P         # show per-thread syscall statistics
    ./alisyscount -D         # show syscall statistics as histograms
    ./alisyscount -s 0       # show syscall statistics of syscall 0 only
    ./alisyscount -p 25      # show syscall statistics of PID 25 only
    ./alisyscount 1 10       # print 1 second summaries, 10 times
    ./alisyscount -NT 1      # 1s summaries, nanoseconds, and timestamps
"""
parser = argparse.ArgumentParser(
    description="Summarize syscall counts and latencies.",
    formatter_class=argparse.RawDescriptionHelpFormatter,
    epilog=examples)
parser.add_argument("-T", "--timestamp", action="store_true",
    help="include timestamp on output")
parser.add_argument("-N", "--nanoseconds", action="store_true",
    help="output in nanoseconds")
parser.add_argument("-S", "--syscalls", action="store_true",
    help="show per-syscall statistics")
parser.add_argument("-P", "--pids", action="store_true",
    help="show per-thread syscall statistics")
parser.add_argument("-D", "--dist", action="store_true",
    help="show distributions as histograms")
parser.add_argument("-s", "--syscall",
    help="show statistics of specific syscall only")
parser.add_argument("-p", "--pid",
    help="show syscall statistics of specific PID only")
parser.add_argument("interval", nargs="?", default=99999999,
    help="output interval, in seconds")
parser.add_argument("outputs", nargs="?", default=99999999,
    help="number of outputs")
parser.add_argument("--ebpf", action="store_true",
    help=argparse.SUPPRESS)

args = parser.parse_args()
countdown = int(args.outputs)
if args.pid:
    args.pids = True
if args.syscall:
    args.syscalls = True
if (not args.syscalls) and (not args.pids):
    args.syscalls = True

if args.nanoseconds:
    factor = 1
    label = "nsecs"
else:
    factor = 1000
    label = "usecs"
debug = 0

# define BPF program
bpf_text = """
#include <uapi/linux/ptrace.h>

BPF_PERCPU_ARRAY(start, u64, 1);

STORAGE

TRACEPOINT_PROBE(raw_syscalls, sys_enter)
{
    COND_FILTER

    u32 idx = 0;
    u64 ts = bpf_ktime_get_ns();
    start.update(&idx, &ts);

    return 0;
}

TRACEPOINT_PROBE(raw_syscalls, sys_exit)
{
    COND_FILTER

    u32 idx = 0;
    u64 *tsp, delta;
    tsp = start.lookup(&idx);
    if (!tsp || *tsp == 0) {
        return 0;   // missed start
    }

    delta = bpf_ktime_get_ns() - *tsp;
    u64 zero = 0;
    start.update(&idx, &zero);

    u32 sysnr = args->id;
    u32 pid = bpf_get_current_pid_tgid();

    UPDATE

    return 0;
}
"""


# replace STORAGE
map_key = """
typedef struct map_key {
    u32 sysnr;
    u32 pid;
} map_key_t;
"""

map_val = """
typedef struct map_val {
    u64 time;
    u64 count;
} map_val_t;
"""

if args.dist:
    if args.syscalls and args.pids:
        storage = map_key + """
typedef struct map_key_slot {
    map_key_t key;
    u64 slot;
} map_key_slot_t;
"""
    else:
        storage = """
typedef struct map_key_slot {
    u64 key;
    u64 slot;
} map_key_slot_t;
"""
    storage += "BPF_HISTOGRAM(dist, map_key_slot_t);"

else:
    storage = map_val
    if args.syscalls and args.pids:
        storage += map_key + "BPF_HASH(res, map_key_t, map_val_t);"
    else:
        storage += "BPF_HASH(res, u32, map_val_t);"
bpf_text = bpf_text.replace('STORAGE', storage)


# replace UPDATE
if args.dist:
    if args.syscalls and args.pids:
        key_exp = "{.sysnr = sysnr, .pid = pid,}"
    elif args.syscalls:
        key_exp = "sysnr"
    elif args.pids:
        key_exp = "pid"

    update = """
    map_key_slot_t key_slot = {
        .key = %s,
        .slot = bpf_log2l(delta / FACTOR),
    };
    dist.increment(key_slot);
""" % key_exp

else:
    if args.syscalls and args.pids:
        update = """
    map_key_t key = {
        .sysnr = sysnr,
        .pid = pid,
    };
"""
    elif args.syscalls:
        update = "u32 key = sysnr;"
    elif args.pids:
        update = "u32 key = pid;"

    update += """
    map_val_t val = {};
    map_val_t *valp = res.lookup_or_init(&key, &val);
    if (valp) {
        valp->count += 1;
        valp->time += delta;
    }
"""
bpf_text = bpf_text.replace('UPDATE', update)
bpf_text = bpf_text.replace('FACTOR', str(factor))


# replace FILTER
sysnr_match = "args->id == %s" % args.syscall
pid_match = "((u32)bpf_get_current_pid_tgid()) == %s" % args.pid

if args.syscall and args.pid:
    cond_match = "(%s) && (%s)" % (sysnr_match, pid_match)
elif args.syscall:
    cond_match = sysnr_match
elif args.pid:
    cond_match = pid_match
else:
    cond_match = ""

cond_filter = "if (!(%s)) {return 0;}" % cond_match if cond_match else ""
bpf_text = bpf_text.replace('COND_FILTER', cond_filter)


# output eBPF program C code after it is replaced, used by debugging
if debug or args.ebpf:
    print(bpf_text)
    if args.ebpf:
        exit()


# load BPF program
b = BPF(text=bpf_text)
print("Tracing syscall statistics... Hit Ctrl-C to end.")

if args.dist:
    tab = b.get_table("dist")
else:
    tab = b.get_table("res")


def pid_to_comm(pid):
    try:
        return open("/proc/%d/comm" % pid, "r").read().rstrip("\n")
    except IOError:
        return ""


def get_pid_info(pid):
    return "%10d %30s" % (pid, pid_to_comm(pid))

def get_syscall_info(sysnr):
    return "%10d %30s" % (sysnr, syscall_name(sysnr))

def skip_errata(k):
    if args.syscalls and args.pids:
        sysnr = k.sysnr
    else:  # args.syscalls or args.pids
        sysnr = k.value

    # happens occasionally, we don't need it
    if sysnr == 0xFFFFFFFF:
        return True
    else:
        return False


def print_stat(tab):
    if args.syscalls and args.pids:
        section_hdr = "%10s %30s %10s %30s" % ("SysNR", "Syscall", "PID", "Thread")
    elif args.pids:
        section_hdr = "%10s %30s" % ("PID", "Thread")
    else:  # args.syscalls
        section_hdr = "%10s %30s" % ("SysNR", "Syscall")

    section_hdr += " %10s %10s" % ("Count", "TOTAL_" + label)
    print(section_hdr)

    # k is struct map_key; v is struct map_val
    for k, v in sorted(tab.items(), key=lambda res: (-res[1].count, -res[1].time)):
        if skip_errata(k):
            continue

        if args.syscalls and args.pids:
            record = "%40s %40s" % (get_syscall_info(k.sysnr), get_pid_info(k.pid))
        elif args.pids:
            record = "%40s" % (get_pid_info(k.value))
        else:  # args.syscalls
            record = "%40s" % (get_syscall_info(k.value))

        record += "%10d %10d" % (v.count, v.time / factor)
        print(record)


def get_dist_pid_info(pid):
    return "PID %d (%s)" % (pid, pid_to_comm(pid))

def get_dist_syscall_info(sysnr):
    return "Syscall %d (%s)" % (sysnr, syscall_name(sysnr))

def print_dist_section(k):
    if args.syscalls and args.pids:
        return "%s %s" % (get_dist_syscall_info(k[0]), get_dist_pid_info(k[1]))
    elif args.pids:
        return "%40s" % (get_dist_pid_info(k))
    else:
        return "%40s" % (get_dist_syscall_info(k))

def print_dist_stat(tab):
    section = "section" if args.syscalls and args.pids else "Thread" if args.pids else "Syscall"
    tab.print_log2_hist("time_" + label, section,
        section_print_fn=print_dist_section,
        bucket_fn=(lambda k: (k.sysnr, k.pid)) if args.syscalls and args.pids else None,
        bucket_sort_fn=lambda k: sorted(k))


# output
while (1):
    try:
        sleep(int(args.interval))
    except KeyboardInterrupt:
        countdown = 1
    print()
    if args.timestamp:
        print("%-8s\n" % strftime("%H:%M:%S"), end="")

    if args.dist:
        print_dist_stat(tab)
    else:
        print_stat(tab)

    tab.clear()

    countdown -= 1
    if countdown == 0:
        exit()
