#!/usr/bin/python
# @lint-avoid-python-3-compatibility-imports
#
# alirunqinfo  Summarize task on-CPU time before being descheduled.
#              For Linux, uses BCC, eBPF.
#
# USAGE: alirunqinfo [-h] [-T] [-N] [-D] [-C] [-P] [-c CPU] [-p PID]
#                    [interval] [outputs]
#
# Copyright (c) 2019 Jeffle Xu, Alibaba, Inc.
# Licensed under the Apache License, Version 2.0 (the "License")

from __future__ import print_function
from bcc import BPF
from time import sleep, strftime
import argparse
import ctypes
import signal

# arguments
examples = """examples:
    ./alirunqinfo            # sum runqueue statistics
    ./alirunqinfo -C         # show per-CPU runqueue statistics
    ./alirunqinfo -P         # show per-thread runqueue statistics
    ./alirunqinfo -D         # show runqueue statistics as histograms
    ./alirunqinfo -c 0       # show runqueue statistics of CPU 0 only
    ./alirunqinfo -p 25      # show runqueue statistics of PID 25 only
    ./alirunqinfo 1 10       # print 1 second summaries, 10 times
    ./alirunqinfo -NT 1      # 1s summaries, nanoseconds, and timestamps
"""
parser = argparse.ArgumentParser(
    description="Summarize runqueue statistics.",
    formatter_class=argparse.RawDescriptionHelpFormatter,
    epilog=examples)
parser.add_argument("-T", "--timestamp", action="store_true",
    help="include timestamp on output")
parser.add_argument("-N", "--nanoseconds", action="store_true",
    help="output in nanoseconds")
parser.add_argument("-C", "--cpus", action="store_true",
    help="show per-CPU runqueue statistics")
parser.add_argument("-P", "--pids", action="store_true",
    help="show per-thread runqueue statistics")
parser.add_argument("-D", "--dist", action="store_true",
    help="show distributions as histograms")
parser.add_argument("-c", "--cpu",
    help="show runqueue statistics on specific CPU only")
parser.add_argument("-p", "--pid",
    help="show runqueue statistics of specific PID only")
parser.add_argument("interval", nargs="?", default=99999999,
    help="output interval, in seconds")
parser.add_argument("outputs", nargs="?", default=99999999,
    help="number of outputs")
parser.add_argument("--ebpf", action="store_true",
    help=argparse.SUPPRESS)

args = parser.parse_args()
countdown = int(args.outputs)
if args.pid:
    args.pids = True
if args.cpu:
    args.cpus = True
if (not args.cpus) and (not args.pids):
    args.cpus = True

if args.nanoseconds:
    factor = 1
    label = "nsecs"
else:
    factor = 1000
    label = "usecs"
debug = 0

# define BPF program
bpf_text = """
#include <uapi/linux/ptrace.h>

typedef struct time_stamp {
    u64 stamp;
    u32 pid;
} time_stamp_t;

BPF_PERCPU_ARRAY(start, time_stamp_t, 1);

STORAGE


static inline int sched_switch_exit(u32 cpu, u32 pid, u64 now, time_stamp_t *new) {
    PID_FILTER

    u32 idx = 0;
    // fetch timestamp and calculate delta
    time_stamp_t *stampp = start.lookup(&idx);
    if (!stampp || stampp->pid != pid) {
        return 0;   // missed start
    }
    u64 delta = now - stampp->stamp;
    start.update(&idx, new);

    UPDATE

    return 0;
}

static inline int sched_switch_enter(u32 pid, u64 now, time_stamp_t *new) {
    PID_FILTER

    u32 idx = 0;
    new->stamp = now;
    new->pid = pid;

    start.update(&idx, new);
    return 0;
}

TRACEPOINT_PROBE(sched, sched_switch)
{
    u32 cpu = bpf_get_smp_processor_id();
    CPU_FILTER

    u64 now = bpf_ktime_get_ns();
    u32 pid, idx = 0;
    time_stamp_t new = {};

    pid = args->prev_pid;
    if (pid != 0) {
        sched_switch_exit(cpu, pid, now, &new);
    }

    pid = args->next_pid;
    if (pid != 0) {
        sched_switch_enter(pid, now, &new);
    }

    return 0;
}
"""


# replace STORAGE
map_key = """
typedef struct map_key {
    u32 cpu;
    u32 pid;
} map_key_t;
"""

map_val = """
typedef struct map_val {
    u64 time;
    u64 count;
} map_val_t;
"""

if args.dist:
    # diagram mode, we need BPF_HISTOGRAM
    if args.cpus and args.pids:
        storage = map_key + """
typedef struct map_key_slot {
    map_key_t key;
    u64 slot;
} map_key_slot_t;
"""
    else:
        storage = """
typedef struct map_key_slot {
    u64 key;
    u64 slot;
} map_key_slot_t;
"""
    storage += """BPF_HISTOGRAM(dist, map_key_slot_t);"""

else:
    # normal mode, we need BPF_HASH
    storage = map_val
    if args.cpus and args.pids:
        storage += map_key + """BPF_HASH(res, map_key_t, map_val_t);"""
    else:
        storage += """BPF_HASH(res, u32, map_val_t);"""

bpf_text = bpf_text.replace('STORAGE', storage)


# replace UPDATE
if args.dist:
    if args.cpus and args.pids:
        key_exp = "{.cpu = cpu, .pid = pid,}"
    elif args.cpus:
        key_exp = "cpu"
    elif args.pids:
        key_exp = "pid"

    update = """
    map_key_slot_t key_slot = {
        .key = %s,
        .slot = bpf_log2l(delta / FACTOR),
    };
    dist.increment(key_slot);
""" % key_exp

else:
    if args.cpus and args.pids:
        update = """
    map_key_t key = {
        .cpu = cpu,
        .pid = pid,
    };
"""
    elif args.cpus:
        update = "u32 key = cpu;"
    elif args.pids:
        update = "u32 key = pid;"

    update += """
    map_val_t *valp = res.lookup(&key);
    if (valp) {
        valp->count += 1;
        valp->time += delta;
    }
    else {
        map_val_t val = {
            .count = 1,
            .time  = delta,
        };
        res.update(&key, &val);
    }
"""
bpf_text = bpf_text.replace('UPDATE', update)
bpf_text = bpf_text.replace('FACTOR', str(factor))


# replace FILTER
cpu_filter = "if (cpu != %s) {return 0;}" % args.cpu if args.cpu else ""
bpf_text = bpf_text.replace('CPU_FILTER', cpu_filter)

pid_filter = "if (pid != %s) {return 0;}" % args.pid if args.pid else ""
bpf_text = bpf_text.replace('PID_FILTER', pid_filter)


# output eBPF program C code after it is replaced, used by debugging
if debug or args.ebpf:
    print(bpf_text)
    if args.ebpf:
        exit()


# load BPF program
b = BPF(text=bpf_text)

# print header
print("Tracing runqueue statistics", end="")
if args.pid:
    print(" of PID %s" % args.pid, end="")
if args.cpu:
    print(" on CPU %s" % args.cpu, end="")
if args.dist:
    print(", in diagram mode", end="")
print("...Hit Ctrl-C to end.")

if args.dist:
    tab = b.get_table("dist")
else:
    tab = b.get_table("res")


def pid_to_comm(pid):
    try:
        return open("/proc/%d/comm" % pid, "r").read().rstrip("\n")
    except IOError:
        return ""


def print_per_cpu_thread(tab):
    cur_cpu = -1
    # k is struct map_key; v is struct map_val
    for k, v in sorted(tab.items(), key=lambda res: (res[0].cpu, res[0].pid)):
        if cur_cpu != k.cpu:
            print("\nCPU%d:" % k.cpu)
            print("%10s %40s %10s %10s" % ("PID", "Thread", "Count", "TOTAL_" + label))
            cur_cpu = k.cpu
        print("%10d %40s %10d %10d" % (k.pid, pid_to_comm(k.pid), v.count, v.time / factor))


def print_per_cpu(tab):
    print("%10s %10s %10s" % ("CPU", "Count", "TOTAL_" + label))
    # k is c_uint (representing CPU), v is struct map_val
    for k, v in sorted(tab.items(), key=lambda res: res[0].value):
        print("%10d %10d %10d" % (k.value, v.count, v.time / factor))


def print_per_thread(tab):
    print("%10s %40s %10s %10s" % ("PID", "Thread", "Count", "TOTAL_" + label))
    # k is c_uint (representing PID), v is struct map_val
    for k, v in sorted(tab.items(), key=lambda res: res[0].value):
        print("%10d %40s %10d %10d" % (k.value, pid_to_comm(k.value), v.count, v.time / factor))


def print_dist_per_cpu_thread(tab):
    tab.print_log2_hist("time_" + label, "section",
            # k is (cpuN, pidN) formatted by bucket_fn
            section_print_fn=lambda k: ("CPU %d, PID %d (%s)" % (k[0], k[1], pid_to_comm(k[1]))),
            bucket_fn=lambda k: (k.cpu, k.pid),  # k is struct map_key
            bucket_sort_fn=lambda k: sorted(k))  # k is struct map_key


def print_dist_per_cpu(tab):
    tab.print_log2_hist("time_" + label, "CPU",
            section_print_fn=lambda k: str(k),   # k is u64 representing CPU
            bucket_sort_fn=lambda k: sorted(k))  # k is u64 representing CPU


def print_dist_per_thread(tab):
    tab.print_log2_hist("time_" + label, "Thread",
            section_print_fn=lambda k: ("%d (%s)" % (k, pid_to_comm(k))),  # k is u64 representing PID
            bucket_sort_fn=lambda k: sorted(k))  # k is u64 representing PID


# signal handler
def signal_ignore(signal, frame):
    print()


# output
while (1):
    try:
        sleep(int(args.interval))
    except KeyboardInterrupt:
        # as cleanup can take many seconds, trap Ctrl-C:
        signal.signal(signal.SIGINT, signal_ignore)
        countdown = 1

    print()
    if args.timestamp:
        print("%-8s\n" % strftime("%H:%M:%S"), end="")

    if args.dist:
        if args.cpus and args.pids:
            print_dist_per_cpu_thread(tab)
        elif args.cpus:
            print_dist_per_cpu(tab)
        elif args.pids:
            print_dist_per_thread(tab)
        else:
            print("Invalid parameter")
            countdown = 1
    else:
        if args.cpus and args.pids:
            print_per_cpu_thread(tab)
        elif args.cpus:
            print_per_cpu(tab)
        elif args.pids:
            print_per_thread(tab)
        else:
            print("Invalid parameter")
            countdown = 1

    tab.clear()

    countdown -= 1
    if countdown == 0:
        exit()
