#!/usr/bin/python
# @lint-avoid-python-3-compatibility-imports
#
# alihardirqs   Summarize hard IRQ (interrupt) event time.
#               Enhanced version of BCC hardirqs.
#               For Linux, uses BCC, eBPF.
#
# USAGE: alihardirqs [-h] [-T] [-N] [-D] [-c CPU] [-i IRQ] [interval] [outputs]
#
# Copyright (c) 2015 Brendan Gregg.
# Copyright (c) 2019 Jeffle Xu, Alibaba, Inc.
# Licensed under the Apache License, Version 2.0 (the "License")

from __future__ import print_function
from bcc import BPF
from time import sleep, strftime
import argparse

# arguments
examples = """examples:
    ./alihardirqs            # sum hard irq event time
    ./alihardirqs -D         # show hard irq event time as histograms
    ./alihardirqs -c 0       # show hard irq event time of CPU 0 only
    ./alihardirqs -i 25      # show event time of IRQ 25 only
    ./alihardirqs -c 0 -i 25 # show event time of IRQ 25 on CPU 0 only
    ./alihardirqs 1 10       # print 1 second summaries, 10 times
    ./alihardirqs -NT 1      # 1s summaries, nanoseconds, and timestamps
"""
parser = argparse.ArgumentParser(
    description="Summarize hard irq statistics, including total count, time, etc.",
    formatter_class=argparse.RawDescriptionHelpFormatter,
    epilog=examples)
parser.add_argument("-T", "--timestamp", action="store_true",
    help="include timestamp on output")
parser.add_argument("-N", "--nanoseconds", action="store_true",
    help="output in nanoseconds")
parser.add_argument("-D", "--dist", action="store_true",
    help="show distributions as histograms")
parser.add_argument("-c", "--cpu",
    help="output hardIRQ statistics on specific CPU only")
parser.add_argument("-i", "--irq",
    help="output statistics of specific hardIRQ")
parser.add_argument("interval", nargs="?", default=99999999,
    help="output interval, in seconds")
parser.add_argument("outputs", nargs="?", default=99999999,
    help="number of outputs")
parser.add_argument("--ebpf", action="store_true",
    help=argparse.SUPPRESS)

args = parser.parse_args()
countdown = int(args.outputs)
distmode = "1" if args.dist else "0"
if args.nanoseconds:
    factor = 1
    label = "nsecs"
else:
    factor = 1000
    label = "usecs"
debug = 0

# define BPF program
bpf_text = """
#include <uapi/linux/ptrace.h>

typedef struct irq_key {
    u32 cpu;
    u32 irq;
    char name[32];
} irq_key_t;

typedef struct irq_key_slot {
    irq_key_t key;
    u64 slot;
} irq_key_slot_t;

typedef struct irq_val {
    u64 count;
    u64 time;
    u64 time_max;
} irq_val_t;

typedef struct stamp {
    u64 ts;
    char name[32];
} stamp_t;

BPF_PERCPU_ARRAY(start, stamp_t, 1);
BPF_HASH(res, irq_key_t, irq_val_t);
BPF_HISTOGRAM(dist, irq_key_slot_t);


TRACEPOINT_PROBE(irq, irq_handler_entry)
{
    COND_FILTER

    u32 idx = 0;
    stamp_t stamp = {
        .ts = bpf_ktime_get_ns(),
    };
    TP_DATA_LOC_READ_CONST(stamp.name, name, sizeof(stamp.name));
    start.update(&idx, &stamp);

    return 0;
}

TRACEPOINT_PROBE(irq, irq_handler_exit)
{
    u64 delta;
    u32 cpu = bpf_get_smp_processor_id();

    u32 idx = 0;
    // fetch timestamp and calculate delta
    stamp_t *stampp = start.lookup(&idx);
    if (!stampp || stampp->ts == 0) {
        return 0;   // missed start
    }

    delta = bpf_ktime_get_ns() - stampp->ts;
    irq_key_t key = {.cpu = cpu, .irq = args->irq};
    char *name = (char *)stampp->name;
    bpf_probe_read(key.name, sizeof(key.name), name);

#if DIST_MODE
    irq_key_slot_t key_slot = {
        .key = key,
        .slot = bpf_log2l(delta / FACTOR),
    };
    dist.increment(key_slot);
#else
    struct irq_val *valp, val;
    valp = res.lookup(&key);

    if (valp) {
        valp->count += 1;
        valp->time += delta;
        if (valp->time_max < delta) {valp->time_max = delta;}
    }
    else {
        val.count = 1;
        val.time = val.time_max = delta;
        res.update(&key, &val);
    }
#endif
    stamp_t zero = {};
    start.update(&idx, &zero);

    return 0;
}
"""

cond_filter = ''
cpu_filter = "1"
if args.cpu:
    cpu_filter = "bpf_get_smp_processor_id() == %s" % args.cpu

irq_filter = "1"
if args.irq:
    irq_filter = "args->irq == %s" % args.irq

cond_filter = "if (!(%s && %s)) {return 0;}" % (cpu_filter, irq_filter)
bpf_text = bpf_text.replace('COND_FILTER', cond_filter)

bpf_text = bpf_text.replace('FACTOR', "%d" % factor)
bpf_text = bpf_text.replace('DIST_MODE', distmode)


# output eBPF program C code after it is replaced, used by debugging
if debug or args.ebpf:
    print(bpf_text)
    if args.ebpf:
        exit()


def print_section(key):
    return "CPU %d, hardIRQ %d (%s)" % (key[0], key[1], key[2].decode('utf-8', 'replace'))


# load BPF program
b = BPF(text=bpf_text)
print("Tracing hard irq event time... Hit Ctrl-C to end.")

if args.dist:
    tab = b.get_table("dist")
else:
    tab = b.get_table("res")


# output
while (1):
    try:
        sleep(int(args.interval))
    except KeyboardInterrupt:
        countdown = 1
    print()
    if args.timestamp:
        print("%-8s\n" % strftime("%H:%M:%S"), end="")

    if args.dist:
        tab.print_log2_hist("time_" + label, "section",
            section_print_fn=print_section,
            bucket_fn=lambda k: (k.cpu, k.irq, k.name),
            bucket_sort_fn=lambda keys: sorted(keys))
    else:
        cur_cpu = -1
        for k, v in sorted(tab.items(), key=lambda res: (res[0].cpu, res[0].irq)):
            if cur_cpu != k.cpu:
                print("\nCPU%d:" % k.cpu)
                print("%10s %20s %10s %10s %10s" %
                    ("HARDIRQ", "IRQHandler", "Count", "TOTAL_" + label, "MAX_" + label))
                cur_cpu = k.cpu

            print("%10d %20s %10d %10d %10d" %
                (k.irq, k.name.decode('utf-8', 'replace'), v.count, v.time / factor, v.time_max / factor))

        tab.clear()

    countdown -= 1
    if countdown == 0:
        exit()
