"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildPackageTask = void 0;
const tslib_1 = require("tslib");
const chalk_1 = tslib_1.__importDefault(require("chalk"));
const execa = require("execa");
const fs_1 = require("fs");
const globby_1 = tslib_1.__importDefault(require("globby"));
const lodash_1 = require("lodash");
const path = tslib_1.__importStar(require("path"));
const useSpinner_1 = require("../utils/useSpinner");
const task_1 = require("./task");
const clean = (cwd) => (0, useSpinner_1.useSpinner)('Cleaning', () => execa('npm', ['run', 'clean'], { cwd }));
const compile = (cwd) => (0, useSpinner_1.useSpinner)('Compiling sources', () => execa('tsc', ['-p', './tsconfig.build.json'], { cwd }));
const bundle = (cwd) => (0, useSpinner_1.useSpinner)('Bundling', () => execa('npm', ['run', 'bundle'], { cwd }));
const preparePackage = (packageDist, pkg) => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
    pkg = (0, lodash_1.cloneDeep)(pkg); // avoid mutations
    pkg.main = 'index.js';
    pkg.types = 'index.d.ts';
    const version = pkg.version;
    const name = pkg.name;
    const deps = pkg.dependencies;
    // Below we are adding cross-dependencies to Grafana's packages
    // with the version being published
    if (name.endsWith('/ui')) {
        deps['@grafana/data'] = version;
    }
    else if (name.endsWith('/runtime')) {
        deps['@grafana/data'] = version;
        deps['@grafana/ui'] = version;
    }
    else if (name.endsWith('/toolkit')) {
        deps['@grafana/data'] = version;
        deps['@grafana/ui'] = version;
    }
    yield (0, useSpinner_1.useSpinner)('Updating package.json', () => fs_1.promises.writeFile(`${packageDist}/package.json`, JSON.stringify(pkg, null, 2)));
});
const moveFiles = (fromPath, toPath) => {
    const files = ['README.md', 'CHANGELOG.md', 'index.js'];
    return (0, useSpinner_1.useSpinner)(`Moving ${files.join(', ')} files`, () => {
        const promises = files.map((file) => fs_1.promises.copyFile(`${fromPath}/${file}`, `${toPath}/${file}`));
        return Promise.all(promises);
    });
};
const moveStaticFiles = (packageRoot, pkg) => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
    if (pkg.name.endsWith('/ui')) {
        return (0, useSpinner_1.useSpinner)('Moving static files', () => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
            const staticFiles = yield (0, globby_1.default)(`${packageRoot}/src/**/*.{png,svg,gif,jpg}`);
            const pathSearch = new RegExp(`^${packageRoot}/src`);
            const pathReplace = `${packageRoot}/compiled`;
            const promises = staticFiles.map((file) => fs_1.promises.copyFile(file, file.replace(pathSearch, pathReplace)));
            yield Promise.all(promises);
        }));
    }
});
const buildTaskRunner = ({ scope }) => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
    if (!scope) {
        throw new Error('Provide packages with -s, --scope <packages>');
    }
    const scopes = scope.split(',').map((s) => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
        const packageRoot = path.resolve(__dirname, `../../../../grafana-${s}`);
        const packageDist = `${packageRoot}/dist`;
        const pkg = require(`${packageRoot}/package.json`);
        console.log(chalk_1.default.yellow(`Building ${pkg.name} (package.json version: ${pkg.version})`));
        yield clean(packageRoot);
        yield compile(packageRoot);
        yield moveStaticFiles(packageRoot, pkg);
        yield bundle(packageRoot);
        yield preparePackage(packageDist, pkg);
        yield moveFiles(packageRoot, packageDist);
    }));
    yield Promise.all(scopes);
});
exports.buildPackageTask = new task_1.Task('Package build', buildTaskRunner);
//# sourceMappingURL=package.build.js.map