"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.readGitLog = exports.findImagesInFolder = exports.getPackageDetails = exports.getFileSizeReportInFolder = exports.getGrafanaVersions = void 0;
const tslib_1 = require("tslib");
const execa_1 = tslib_1.__importDefault(require("execa"));
const fs_1 = tslib_1.__importDefault(require("fs"));
const path_1 = tslib_1.__importDefault(require("path"));
const md5File = require('md5-file');
function getGrafanaVersions() {
    const dir = path_1.default.resolve(process.cwd(), 'node_modules', '@grafana');
    const versions = {};
    try {
        fs_1.default.readdirSync(dir).forEach((file) => {
            const json = require(path_1.default.resolve(dir, file, 'package.json'));
            versions[file] = json.version;
        });
    }
    catch (err) {
        console.warn('Error reading toolkit versions', err);
    }
    return versions;
}
exports.getGrafanaVersions = getGrafanaVersions;
function getFileSizeReportInFolder(dir, info) {
    const acc = info ? info : {};
    const files = fs_1.default.readdirSync(dir);
    if (files) {
        files.forEach((file) => {
            const newbase = path_1.default.join(dir, file);
            const stat = fs_1.default.statSync(newbase);
            if (stat.isDirectory()) {
                getFileSizeReportInFolder(newbase, info);
            }
            else {
                let ext = '_none_';
                const idx = file.lastIndexOf('.');
                if (idx > 0) {
                    ext = file.substring(idx + 1).toLowerCase();
                }
                const current = acc[ext];
                if (current) {
                    current.count += 1;
                    current.bytes += stat.size;
                }
                else {
                    acc[ext] = { bytes: stat.size, count: 1 };
                }
            }
        });
    }
    return acc;
}
exports.getFileSizeReportInFolder = getFileSizeReportInFolder;
function getPackageDetails(zipFile, zipSrc, writeChecksum = true) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const zipStats = fs_1.default.statSync(zipFile);
        if (zipStats.size < 100) {
            throw new Error('Invalid zip file: ' + zipFile);
        }
        const info = {
            name: path_1.default.basename(zipFile),
            size: zipStats.size,
            contents: getFileSizeReportInFolder(zipSrc),
        };
        try {
            const exe = yield (0, execa_1.default)('shasum', [zipFile]);
            const idx = exe.stdout.indexOf(' ');
            const sha1 = exe.stdout.substring(0, idx);
            if (writeChecksum) {
                fs_1.default.writeFile(zipFile + '.sha1', sha1, (err) => { });
            }
            info.sha1 = sha1;
        }
        catch (_a) {
            console.warn('Unable to read SHA1 Checksum');
        }
        try {
            info.md5 = md5File.sync(zipFile);
        }
        catch (_b) {
            console.warn('Unable to read MD5 Checksum');
        }
        return info;
    });
}
exports.getPackageDetails = getPackageDetails;
function findImagesInFolder(dir, prefix = '', append) {
    const imgs = append || [];
    const files = fs_1.default.readdirSync(dir);
    if (files) {
        files.forEach((file) => {
            if (file.endsWith('.png')) {
                imgs.push(file);
            }
        });
    }
    return imgs;
}
exports.findImagesInFolder = findImagesInFolder;
function readGitLog() {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        try {
            let exe = yield (0, execa_1.default)('git', [
                'log',
                '-1',
                '--pretty=format:{%n  "commit": "%H",%n  "tree": "%T",%n  "subject": "%s",%n  "author": {%n    "name": "%aN",%n    "email": "%aE",%n    "time":"%at"  },%n  "commiter": {%n    "name": "%cN",%n    "email": "%cE",%n    "time":"%ct"  }%n}',
            ]);
            const info = JSON.parse(exe.stdout);
            // Read the body
            exe = yield (0, execa_1.default)('git', [
                'log',
                '-1',
                '--pretty=format:%b', // Just the body (with newlines!)
            ]);
            if (exe.stdout && exe.stdout.length) {
                info.body = exe.stdout.trim();
            }
            // Read any commit notes
            exe = yield (0, execa_1.default)('git', [
                'log',
                '-1',
                '--pretty=format:%N', // commit notes (with newlines!)
            ]);
            if (exe.stdout && exe.stdout.length) {
                info.notes = exe.stdout.trim();
            }
            return info;
        }
        catch (err) {
            console.warn('Error REading Git log info', err);
        }
        return undefined;
    });
}
exports.readGitLog = readGitLog;
//# sourceMappingURL=utils.js.map