/*
 * Copyright 2020 The TensorFlow Runtime Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

// This file defines utilities related to setting up unit tests.
#ifndef TFRT_CPP_TESTS_TEST_UTIL_H_
#define TFRT_CPP_TESTS_TEST_UTIL_H_

#include "tfrt/dtype/dtype.h"
#include "tfrt/host_context/concurrent_work_queue.h"
#include "tfrt/host_context/host_allocator.h"
#include "tfrt/host_context/host_buffer.h"
#include "tfrt/host_context/host_context.h"
#include "tfrt/support/forward_decls.h"
#include "tfrt/tensor/dense_host_tensor.h"
#include "tfrt/tensor/dense_host_tensor_view.h"
#include "tfrt/tensor/tensor_metadata.h"

namespace tfrt {

// TODO(jingdong): Move the test functions to the ::tfrt::test namespace to
// avoid name collision.

inline std::unique_ptr<HostContext> CreateHostContext() {
  auto decoded_diagnostic_handler = [&](const DecodedDiagnostic& diag) {
    abort();
  };
  std::unique_ptr<ConcurrentWorkQueue> work_queue =
      CreateSingleThreadedWorkQueue();
  std::unique_ptr<HostAllocator> host_allocator = CreateMallocAllocator();
  return std::make_unique<HostContext>(decoded_diagnostic_handler,
                                       std::move(host_allocator),
                                       std::move(work_queue));
}

template <typename T>
DenseHostTensor CreateDummyTensor(ArrayRef<Index> dims, HostContext* host_ctx) {
  const TensorMetadata metadata(GetDType<T>(), dims);
  auto dht = *DenseHostTensor::CreateUninitialized(metadata, host_ctx);
  MutableDHTArrayView<T> view(&dht);
  for (int i = 0, s = dht.NumElements(); i < s; i++) {
    view[i] = i;
  }
  return dht;
}

inline RCReference<HostBuffer> CreateHostBufferOnHeap(size_t byte_size) {
  std::unique_ptr<char[]> buf{new char[byte_size]};
  auto ptr = buf.get();
  return HostBuffer::CreateFromExternal(
      ptr, byte_size, [buf = std::move(buf)](void*, size_t) {});
}

template <typename T>
DenseHostTensor CreateTensorFromValues(ArrayRef<Index> dims,
                                       llvm::ArrayRef<T> values,
                                       HostContext* host_ctx = nullptr) {
  const TensorMetadata metadata(GetDType<T>(), dims);
  DenseHostTensor dht;
  if (host_ctx) {
    dht = *DenseHostTensor::CreateUninitialized(metadata, host_ctx);
  } else {
    auto host_buf = CreateHostBufferOnHeap(metadata.GetHostSizeInBytes());
    dht = DenseHostTensor(metadata, std::move(host_buf));
  }

  MutableDHTArrayView<T> view(&dht);
  CHECK(view.NumElements() == values.size());

  std::copy(values.begin(), values.end(), view.begin());
  return dht;
}

}  // namespace tfrt

#endif  // TFRT_CPP_TESTS_TEST_UTIL_H_
