/*
 * Unix SMB/CIFS implementation.
 *
 * Copyright (C) 2019 Guenther Deschner <gd@samba.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdarg.h>
#include <stddef.h>
#include <stdint.h>
#include <setjmp.h>
#include <cmocka.h>

#include "includes.h"
#include "auth/gensec/schannel.c"

static void torture_schannel_seal_flags(void **state, uint32_t flags,
					const DATA_BLOB session_key,
					const DATA_BLOB seq_num_initial,
					const DATA_BLOB confounder_initial,
					const DATA_BLOB confounder_expected,
					const DATA_BLOB clear_initial,
					const DATA_BLOB crypt_expected)
{
	NTSTATUS status;
	struct schannel_state *schannel_state;
	struct netlogon_creds_CredentialState *creds;
	uint8_t confounder[8];
	DATA_BLOB io;

	assert_int_equal(session_key.length, 16);
	assert_int_equal(seq_num_initial.length, 8);
	assert_int_equal(confounder_initial.length, 8);
	assert_int_equal(confounder_expected.length, 8);
	assert_int_equal(clear_initial.length, crypt_expected.length);

	DEBUG(0,("checking buffer size: %d\n", (int)clear_initial.length));

	schannel_state = talloc_zero(NULL, struct schannel_state);
	assert_non_null(schannel_state);
	creds = talloc_zero(schannel_state,
			    struct netlogon_creds_CredentialState);
	assert_non_null(creds);
	schannel_state->creds = creds;

	io = data_blob_dup_talloc(schannel_state, clear_initial);
	assert_non_null(io.data);
	assert_int_equal(io.length, clear_initial.length);

	schannel_state->creds->negotiate_flags = flags;
	memcpy(schannel_state->creds->session_key, session_key.data, 16);

	memcpy(confounder, confounder_initial.data, 8);

	DEBUG(0,("confounder before crypt:\n"));
	dump_data(0, confounder, 8);
	dump_data(0, seq_num_initial.data, 8);
	dump_data(0, io.data, io.length);

	status = netsec_do_seal(schannel_state,
				seq_num_initial.data,
			        confounder,
				io.data,
				io.length,
				true);

	assert_true(NT_STATUS_IS_OK(status));
	dump_data(0, io.data, io.length);
	DEBUG(0,("confounder after crypt:\n"));
	dump_data(0, confounder, 8);
	dump_data(0, seq_num_initial.data, 8);
	assert_memory_equal(io.data, crypt_expected.data, crypt_expected.length);
	assert_memory_equal(confounder, confounder_expected.data, confounder_expected.length);

	status = netsec_do_seal(schannel_state,
				seq_num_initial.data,
			        confounder,
				io.data,
				io.length,
				false);

	assert_true(NT_STATUS_IS_OK(status));
	dump_data(0, io.data, io.length);
	DEBUG(0,("confounder after decrypt:\n"));
	dump_data(0, confounder, 8);
	dump_data(0, seq_num_initial.data, 8);
	assert_memory_equal(io.data, clear_initial.data, clear_initial.length);
	assert_memory_equal(confounder, confounder_initial.data, confounder_initial.length);

	talloc_free(schannel_state);
}

static void torture_schannel_seal_rc4(void **state)
{
	const uint8_t _session_key[16] = {
		0x14, 0xD5, 0x7F, 0x8D, 0x8E, 0xCF, 0xFB, 0x56,
		0x71, 0x29, 0x9D, 0x9C, 0x2A, 0x75, 0x00, 0xA1
	};
	const DATA_BLOB session_key = data_blob_const(_session_key, 16);
	const uint8_t _seq_num_initial[8] = {
		0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00
	};
	const DATA_BLOB seq_num_initial =
		data_blob_const(_seq_num_initial, 8);
	const uint8_t _confounder_initial[8] = {
		0x1A, 0x5A, 0xE8, 0xC7, 0xBE, 0x4F, 0x1F, 0x07
	};
	const DATA_BLOB confounder_initial =
		data_blob_const(_confounder_initial, 8);
	const uint8_t _confounder_expected[8] = {
		0x25, 0x4A, 0x9C, 0x15, 0x82, 0x3E, 0x4A, 0x42
	};
	const DATA_BLOB confounder_expected =
		data_blob_const(_confounder_expected, 8);
	const uint8_t _clear_initial[] = {
		0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00,
		0x01, 0x00, 0x00, 0x00, 0x04, 0x00, 0x02, 0x00,
		0x01, 0x00, 0x00, 0x00, 0x01, 0x01, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x8A, 0xE3, 0x13, 0x71, 0x02, 0xF4, 0x36, 0x71,
		0x01, 0x00, 0x04, 0x00, 0x01, 0x00, 0x00, 0x00,
		0x02, 0x40, 0x28, 0x00, 0x78, 0x57, 0x34, 0x12,
		0x34, 0x12, 0xCD, 0xAB, 0xEF, 0x00, 0x01, 0x23,
		0x45, 0x67, 0x89, 0xAB, 0x00, 0x00, 0x00, 0x00,
		0x04, 0x5D, 0x88, 0x8A, 0xEB, 0x1C, 0xC9, 0x11,
		0x9F, 0xE8, 0x08, 0x00, 0x2B, 0x10, 0x48, 0x60,
		0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
	};
	const DATA_BLOB clear_initial = data_blob_const(_clear_initial,
			sizeof(_clear_initial));
	const uint8_t crypt_buffer[] = {
		0x3E, 0x10, 0x74, 0xD2, 0x3C, 0x71, 0x57, 0x45,
		0xB8, 0xAA, 0xCF, 0xE3, 0x84, 0xBE, 0xC4, 0x00,
		0xF4, 0x4D, 0x88, 0x0A, 0x9B, 0xCC, 0x53, 0xFC,
		0x32, 0xAA, 0x8E, 0x4B, 0x0E, 0xDE, 0x5F, 0x7D,
		0x6D, 0x31, 0x4E, 0xAB, 0xE0, 0x7D, 0x37, 0x9D,
		0x3D, 0x16, 0xD8, 0xBA, 0x6A, 0xB0, 0xD0, 0x99,
		0x14, 0x05, 0x37, 0xCF, 0x63, 0xD3, 0xD7, 0x60,
		0x63, 0x3C, 0x03, 0x0A, 0x30, 0xA0, 0x3E, 0xC7,
		0xDA, 0x94, 0x3B, 0x40, 0x63, 0x74, 0xEF, 0xCF,
		0xE5, 0x48, 0x87, 0xE9, 0x6A, 0x5A, 0xC7, 0x61,
		0xF7, 0x09, 0xB7, 0x7C, 0xDE, 0xDB, 0xB0, 0x94,
		0x9B, 0x99, 0xC0, 0xA7, 0x7E, 0x78, 0x09, 0x35,
		0xB4, 0xF4, 0x11, 0xC3, 0xB3, 0x77, 0xB5, 0x77,
		0x25, 0xEE, 0xFD, 0x2F, 0x9A, 0x15, 0x95, 0x27,
		0x08, 0xDA, 0xD0, 0x28, 0xD6, 0x31, 0xB4, 0xB7,
		0x7A, 0x19, 0xBB, 0xF3, 0x78, 0xF8, 0xC2, 0x5B
	};
	const DATA_BLOB crypt_expected = data_blob_const(crypt_buffer,
							 sizeof(crypt_buffer));
	int buffer_sizes[] = {
		0, 1, 3, 7, 8, 9, 15, 16, 17
	};
	int i;

	torture_schannel_seal_flags(state, 0,
				    session_key,
				    seq_num_initial,
				    confounder_initial,
				    confounder_expected,
				    clear_initial,
				    crypt_expected);

	/* repeat the test for varying buffer sizes */

	for (i = 0; i < ARRAY_SIZE(buffer_sizes); i++) {
		DATA_BLOB clear_initial_trunc =
			data_blob_const(clear_initial.data, buffer_sizes[i]);
		DATA_BLOB crypt_expected_trunc =
			data_blob_const(crypt_expected.data, buffer_sizes[i]);
		torture_schannel_seal_flags(state, 0,
					    session_key,
					    seq_num_initial,
					    confounder_initial,
					    confounder_expected,
					    clear_initial_trunc,
					    crypt_expected_trunc);
	}
}

static void torture_schannel_seal_aes(void **state)
{
	const uint8_t _session_key[16] = {
		0x8E, 0xE8, 0x27, 0x85, 0x83, 0x41, 0x3C, 0x8D,
		0xC9, 0x54, 0x70, 0x75, 0x8E, 0xC9, 0x69, 0x91
	};
	const DATA_BLOB session_key = data_blob_const(_session_key, 16);
	const uint8_t _seq_num_initial[8] = {
		0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00
	};
	const DATA_BLOB seq_num_initial =
		data_blob_const(_seq_num_initial, 8);
	const uint8_t _confounder_initial[8] = {
		0x6E, 0x09, 0x25, 0x94, 0x01, 0xA0, 0x09, 0x31
	};
	const DATA_BLOB confounder_initial =
		data_blob_const(_confounder_initial, 8);
	const uint8_t _confounder_expected[8] = {
		0xCA, 0xFB, 0xAC, 0xFB, 0xA8, 0x26, 0x75, 0x2A
	};
	const DATA_BLOB confounder_expected =
		data_blob_const(_confounder_expected, 8);
	const uint8_t _clear_initial[] = {
		0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00,
		0x01, 0x00, 0x00, 0x00, 0x04, 0x00, 0x02, 0x00,
		0x01, 0x00, 0x00, 0x00, 0x01, 0x01, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x8A, 0xE3, 0x13, 0x71, 0x02, 0xF4, 0x36, 0x71,
		0x01, 0x00, 0x04, 0x00, 0x01, 0x00, 0x00, 0x00,
		0x02, 0x40, 0x28, 0x00, 0x78, 0x57, 0x34, 0x12,
		0x34, 0x12, 0xCD, 0xAB, 0xEF, 0x00, 0x01, 0x23,
		0x45, 0x67, 0x89, 0xAB, 0x00, 0x00, 0x00, 0x00,
		0x04, 0x5D, 0x88, 0x8A, 0xEB, 0x1C, 0xC9, 0x11,
		0x9F, 0xE8, 0x08, 0x00, 0x2B, 0x10, 0x48, 0x60,
		0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
	};
	const DATA_BLOB clear_initial = data_blob_const(_clear_initial,
			sizeof(_clear_initial));
	const uint8_t crypt_buffer[] = {
		0xE2, 0xE5, 0xE3, 0x26, 0x45, 0xFB, 0xFC, 0xF3,
		0x9C, 0x14, 0xDD, 0xE1, 0x39, 0x23, 0xE0, 0x55,
		0xED, 0x8F, 0xF4, 0x92, 0xA1, 0xBD, 0xDC, 0x40,
		0x58, 0x6F, 0xD2, 0x5B, 0xF9, 0xC9, 0xA3, 0x87,
		0x46, 0x4B, 0x7F, 0xB2, 0x03, 0xD2, 0x35, 0x22,
		0x3E, 0x70, 0x9F, 0x1E, 0x3F, 0x1F, 0xDB, 0x7D,
		0x79, 0x88, 0x5A, 0x3D, 0xD3, 0x40, 0x1E, 0x69,
		0xD7, 0xE2, 0x1D, 0x5A, 0xE9, 0x3B, 0xE1, 0xE2,
		0x98, 0xFD, 0xCB, 0x3A, 0xF7, 0xB5, 0x1C, 0xF8,
		0xCA, 0x02, 0x00, 0x99, 0x9F, 0x0C, 0x01, 0xE6,
		0xD2, 0x00, 0xAF, 0xE0, 0x51, 0x88, 0x62, 0x50,
		0xB7, 0xE8, 0x6D, 0x63, 0x4B, 0x97, 0x05, 0xC1,
		0xD4, 0x83, 0x96, 0x29, 0x80, 0xAE, 0xD8, 0xA2,
		0xED, 0xC9, 0x5D, 0x0D, 0x29, 0xFF, 0x2C, 0x23,
		0x02, 0xFA, 0x3B, 0xEE, 0xE8, 0xBA, 0x06, 0x01,
		0x95, 0xDF, 0x80, 0x76, 0x0B, 0x17, 0x0E, 0xD8
	};
	const DATA_BLOB crypt_expected = data_blob_const(crypt_buffer,
							 sizeof(crypt_buffer));
	int buffer_sizes[] = {
		0, 1, 3, 7, 8, 9, 15, 16, 17
	};
	int i;

	torture_schannel_seal_flags(state, NETLOGON_NEG_SUPPORTS_AES,
				    session_key,
				    seq_num_initial,
				    confounder_initial,
				    confounder_expected,
				    clear_initial,
				    crypt_expected);

	/* repeat the test for varying buffer sizes */

	for (i = 0; i < ARRAY_SIZE(buffer_sizes); i++) {
		DATA_BLOB clear_initial_trunc =
			data_blob_const(clear_initial.data, buffer_sizes[i]);
		DATA_BLOB crypt_expected_trunc =
			data_blob_const(crypt_expected.data, buffer_sizes[i]);
		torture_schannel_seal_flags(state, NETLOGON_NEG_SUPPORTS_AES,
					    session_key,
					    seq_num_initial,
					    confounder_initial,
					    confounder_expected,
					    clear_initial_trunc,
					    crypt_expected_trunc);
	}
}

int main(int argc, char *argv[])
{
	int rc;
	const struct CMUnitTest tests[] = {
		cmocka_unit_test(torture_schannel_seal_rc4),
		cmocka_unit_test(torture_schannel_seal_aes),
	};

	if (argc == 2) {
		cmocka_set_test_filter(argv[1]);
	}
	cmocka_set_message_output(CM_OUTPUT_SUBUNIT);

	rc = cmocka_run_group_tests(tests, NULL, NULL);

	return rc;
}
