// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at https://mozilla.org/MPL/2.0/.

use crate::system;

/// Metrics included in every ping as `client_info`.
#[derive(Debug)]
pub struct ClientInfoMetrics {
    /// The build identifier generated by the CI system (e.g. "1234/A").
    pub app_build: String,
    /// The user visible version string (e.g. "1.0.3").
    pub app_display_version: String,
    /// The product-provided release channel (e.g. "beta").
    pub channel: Option<String>,
}

impl ClientInfoMetrics {
    /// Creates the client info with dummy values for all.
    pub fn unknown() -> Self {
        ClientInfoMetrics {
            app_build: "Unknown".to_string(),
            app_display_version: "Unknown".to_string(),
            channel: None,
        }
    }
}

impl From<ClientInfoMetrics> for glean_core::ClientInfoMetrics {
    fn from(metrics: ClientInfoMetrics) -> Self {
        glean_core::ClientInfoMetrics {
            app_build: metrics.app_build,
            app_display_version: metrics.app_display_version,
            channel: metrics.channel,
            os_version: system::get_os_version(),
            architecture: system::ARCH.to_string(),
            ..Default::default()
        }
    }
}
