"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getCircleDownloadBaseURL = exports.writeJobStats = exports.getCiFolder = exports.getJobFolder = exports.getPullRequestNumber = exports.getBuildNumber = exports.getPluginBuildInfo = exports.job = void 0;
const tslib_1 = require("tslib");
const execa_1 = tslib_1.__importDefault(require("execa"));
const fs_1 = tslib_1.__importDefault(require("fs"));
const path_1 = tslib_1.__importDefault(require("path"));
const getJobFromProcessArgv = () => {
    const arg = process.argv[2];
    if (arg && arg.startsWith('plugin:ci-')) {
        const task = arg.substring('plugin:ci-'.length);
        if ('build' === task) {
            if ('--backend' === process.argv[3] && process.argv[4]) {
                return task + '_' + process.argv[4];
            }
            return 'build_plugin';
        }
        return task;
    }
    return 'unknown_job';
};
exports.job = (process.env.DRONE_STEP_NAME ? process.env.DRONE_STEP_NAME : process.env.CIRCLE_JOB) || getJobFromProcessArgv();
const getPluginBuildInfo = () => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
    if (process.env.CI === 'true') {
        let repo;
        let branch;
        let hash;
        let build;
        let pr;
        if (process.env.DRONE === 'true') {
            repo = process.env.DRONE_REPO_LINK;
            branch = process.env.DRONE_BRANCH;
            hash = process.env.DRONE_COMMIT_SHA;
            build = parseInt(process.env.DRONE_BUILD_NUMBER || '', 10);
            pr = parseInt(process.env.DRONE_PULL_REQUEST || '', 10);
        }
        else if (process.env.CIRCLECI === 'true') {
            repo = process.env.CIRCLE_REPOSITORY_URL;
            branch = process.env.CIRCLE_BRANCH;
            hash = process.env.CIRCLE_SHA1;
            build = parseInt(process.env.CIRCLE_BUILD_NUM || '', 10);
            const url = process.env.CIRCLE_PULL_REQUEST || '';
            const idx = url.lastIndexOf('/') + 1;
            pr = parseInt(url.substring(idx), 10);
        }
        const info = {
            time: Date.now(),
            repo,
            branch,
            hash,
        };
        if (pr) {
            info.pr = pr;
        }
        if (build) {
            info.number = build;
        }
        return info;
    }
    const branch = yield (0, execa_1.default)('git', ['rev-parse', '--abbrev-ref', 'HEAD']);
    const hash = yield (0, execa_1.default)('git', ['rev-parse', 'HEAD']);
    return {
        time: Date.now(),
        branch: branch.stdout,
        hash: hash.stdout,
    };
});
exports.getPluginBuildInfo = getPluginBuildInfo;
const getBuildNumber = () => {
    if (process.env.DRONE === 'true') {
        return parseInt(process.env.DRONE_BUILD_NUMBER || '', 10);
    }
    else if (process.env.CIRCLECI === 'true') {
        return parseInt(process.env.CIRCLE_BUILD_NUM || '', 10);
    }
    return undefined;
};
exports.getBuildNumber = getBuildNumber;
const getPullRequestNumber = () => {
    if (process.env.DRONE === 'true') {
        return parseInt(process.env.DRONE_PULL_REQUEST || '', 10);
    }
    else if (process.env.CIRCLECI === 'true') {
        const url = process.env.CIRCLE_PULL_REQUEST || '';
        const idx = url.lastIndexOf('/') + 1;
        return parseInt(url.substring(idx), 10);
    }
    return undefined;
};
exports.getPullRequestNumber = getPullRequestNumber;
const getJobFolder = () => {
    const dir = path_1.default.resolve(process.cwd(), 'ci', 'jobs', exports.job);
    if (!fs_1.default.existsSync(dir)) {
        fs_1.default.mkdirSync(dir, { recursive: true });
    }
    return dir;
};
exports.getJobFolder = getJobFolder;
const getCiFolder = () => {
    const dir = path_1.default.resolve(process.cwd(), 'ci');
    if (!fs_1.default.existsSync(dir)) {
        fs_1.default.mkdirSync(dir, { recursive: true });
    }
    return dir;
};
exports.getCiFolder = getCiFolder;
const writeJobStats = (startTime, workDir) => {
    const endTime = Date.now();
    const stats = {
        job: exports.job,
        startTime,
        endTime,
        elapsed: endTime - startTime,
        buildNumber: (0, exports.getBuildNumber)(),
    };
    const f = path_1.default.resolve(workDir, 'job.json');
    fs_1.default.writeFile(f, JSON.stringify(stats, null, 2), (err) => {
        if (err) {
            throw new Error('Unable to stats: ' + f);
        }
    });
};
exports.writeJobStats = writeJobStats;
// https://circleci.com/api/v1.1/project/github/NatelEnergy/grafana-discrete-panel/latest/artifacts
function getCircleDownloadBaseURL() {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        try {
            const axios = require('axios');
            const repo = process.env.CIRCLE_PROJECT_REPONAME;
            const user = process.env.CIRCLE_PROJECT_USERNAME;
            let url = `https://circleci.com/api/v1.1/project/github/${user}/${repo}/latest/artifacts`;
            const rsp = yield axios.get(url);
            for (const s of rsp.data) {
                const { path, url } = s;
                if (url && path && path.endsWith('report.json')) {
                    return url.substring(url.length - 'report.json'.length);
                }
            }
        }
        catch (e) {
            console.log('Error reading CircleCI artifact URL', e);
        }
        return undefined;
    });
}
exports.getCircleDownloadBaseURL = getCircleDownloadBaseURL;
//# sourceMappingURL=env.js.map