"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.run = void 0;
const tslib_1 = require("tslib");
const chalk_1 = tslib_1.__importDefault(require("chalk"));
const commander_1 = require("commander");
const changelog_1 = require("./tasks/changelog");
const cherrypick_1 = require("./tasks/cherrypick");
const closeMilestone_1 = require("./tasks/closeMilestone");
const component_create_1 = require("./tasks/component.create");
const nodeVersionChecker_1 = require("./tasks/nodeVersionChecker");
const package_build_1 = require("./tasks/package.build");
const plugin_build_1 = require("./tasks/plugin.build");
const plugin_ci_1 = require("./tasks/plugin.ci");
const plugin_create_1 = require("./tasks/plugin.create");
const plugin_dev_1 = require("./tasks/plugin.dev");
const plugin_sign_1 = require("./tasks/plugin.sign");
const plugin_tests_1 = require("./tasks/plugin.tests");
const plugin_update_1 = require("./tasks/plugin.update");
const plugin_utils_1 = require("./tasks/plugin.utils");
const bundle_managed_1 = require("./tasks/plugin/bundle.managed");
const searchTestDataSetup_1 = require("./tasks/searchTestDataSetup");
const template_1 = require("./tasks/template");
const toolkit_build_1 = require("./tasks/toolkit.build");
const execTask_1 = require("./utils/execTask");
const run = (includeInternalScripts = false) => {
    if (includeInternalScripts) {
        commander_1.program.option('-d, --depreciate <scripts>', 'Inform about npm script deprecation', (v) => v.split(','));
        commander_1.program
            .command('package:build')
            .option('-s, --scope <packages>', 'packages=[data|runtime|ui|toolkit|e2e|e2e-selectors]')
            .description('Builds @grafana/* package to packages/grafana-*/dist')
            .action((cmd) => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
            yield (0, execTask_1.execTask)(package_build_1.buildPackageTask)({
                scope: cmd.scope,
            });
        }));
        commander_1.program
            .command('changelog')
            .option('-m, --milestone <milestone>', 'Specify milestone')
            .description('Builds changelog markdown')
            .action((cmd) => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
            if (!cmd.milestone) {
                console.log('Please specify milestone, example: -m <milestone id from github milestone URL>');
                return;
            }
            yield (0, execTask_1.execTask)(changelog_1.changelogTask)({
                milestone: cmd.milestone,
                silent: true,
            });
        }));
        commander_1.program
            .command('cherrypick')
            .option('-e, --enterprise', 'Run task for grafana-enterprise')
            .description('Helps find commits to cherry pick')
            .action((cmd) => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
            yield (0, execTask_1.execTask)(cherrypick_1.cherryPickTask)({ enterprise: !!cmd.enterprise });
        }));
        commander_1.program
            .command('node-version-check')
            .description('Verify node version')
            .action((cmd) => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
            yield (0, execTask_1.execTask)(nodeVersionChecker_1.nodeVersionCheckerTask)({});
        }));
        commander_1.program
            .command('debug:template')
            .description('Just testing')
            .action((cmd) => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
            yield (0, execTask_1.execTask)(template_1.templateTask)({});
        }));
        commander_1.program
            .command('toolkit:build')
            .description('Prepares grafana/toolkit dist package')
            .action((cmd) => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
            yield (0, execTask_1.execTask)(toolkit_build_1.toolkitBuildTask)({});
        }));
        commander_1.program
            .command('searchTestData')
            .option('-c, --count <number_of_dashboards>', 'Specify number of dashboards')
            .description('Setup test data for search')
            .action((cmd) => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
            yield (0, execTask_1.execTask)(searchTestDataSetup_1.searchTestDataSetupTask)({ count: cmd.count });
        }));
        commander_1.program
            .command('close-milestone')
            .option('-m, --milestone <milestone>', 'Specify milestone')
            .option('--dryRun', 'Only simulate actions')
            .description('Helps ends a milestone by removing the cherry-pick label and closing it')
            .action((cmd) => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
            if (!cmd.milestone) {
                console.log('Please specify milestone, example: -m <milestone id from github milestone URL>');
                return;
            }
            yield (0, execTask_1.execTask)(closeMilestone_1.closeMilestoneTask)({
                milestone: cmd.milestone,
                dryRun: !!cmd.dryRun,
            });
        }));
        // React generator
        commander_1.program
            .command('component:create')
            .description('Scaffold React components. Optionally add test, story and .mdx files. The components are created in the same dir the script is run from.')
            .action(() => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
            yield (0, execTask_1.execTask)(component_create_1.componentCreateTask)({});
        }));
    }
    commander_1.program.option('-v, --version', 'Toolkit version').action(() => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
        const version = (0, plugin_utils_1.getToolkitVersion)();
        console.log(`v${version}`);
    }));
    commander_1.program
        .command('plugin:create [name]')
        .description('Creates plugin from template')
        .action((cmd) => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
        yield (0, execTask_1.execTask)(plugin_create_1.pluginCreateTask)({ name: cmd, silent: true });
    }));
    commander_1.program
        .command('plugin:build')
        .option('--maxJestWorkers <num>|<string>', 'Limit number of Jest workers spawned')
        .option('--coverage', 'Run code coverage', false)
        .option('--skipTest', 'Skip running tests (for pipelines that run it separate)', false)
        .option('--skipLint', 'Skip running lint (for pipelines that run it separate)', false)
        .option('--preserveConsole', 'Preserves console calls', false)
        .description('Prepares plugin dist package')
        .action((cmd) => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
        yield (0, execTask_1.execTask)(plugin_build_1.pluginBuildTask)({
            coverage: cmd.coverage,
            silent: true,
            maxJestWorkers: cmd.maxJestWorkers,
            preserveConsole: cmd.preserveConsole,
            skipLint: cmd.skipLint,
            skipTest: cmd.skipTest,
        });
    }));
    commander_1.program
        .command('plugin:dev')
        .option('-w, --watch', 'Run plugin development mode with watch enabled')
        .description('Starts plugin dev mode')
        .action((cmd) => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
        yield (0, execTask_1.execTask)(plugin_dev_1.pluginDevTask)({
            watch: !!cmd.watch,
            silent: true,
        });
    }));
    commander_1.program
        .command('plugin:test')
        .option('-u, --updateSnapshot', 'Run snapshots update')
        .option('--coverage', 'Run code coverage')
        .option('--watch', 'Run tests in interactive watch mode')
        .option('--testPathPattern <regex>', 'Run only tests with a path that matches the regex')
        .option('--testNamePattern <regex>', 'Run only tests with a name that matches the regex')
        .option('--maxWorkers <num>|<string>', 'Limit number of workers spawned')
        .description('Executes plugin tests')
        .action((cmd) => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
        yield (0, execTask_1.execTask)(plugin_tests_1.pluginTestTask)({
            updateSnapshot: !!cmd.updateSnapshot,
            coverage: !!cmd.coverage,
            watch: !!cmd.watch,
            testPathPattern: cmd.testPathPattern,
            testNamePattern: cmd.testNamePattern,
            maxWorkers: cmd.maxWorkers,
            silent: true,
        });
    }));
    commander_1.program
        .command('plugin:sign')
        .option('--signatureType <type>', 'Signature Type')
        .option('--rootUrls <urls...>', 'Root URLs', function (url, urls) {
        if (typeof url !== 'string') {
            return urls;
        }
        const parts = url.split(',');
        urls.push(...parts);
        return urls;
    }, [])
        .description('Create a plugin signature')
        .action((cmd) => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
        yield (0, execTask_1.execTask)(plugin_sign_1.pluginSignTask)({
            signatureType: cmd.signatureType,
            rootUrls: cmd.rootUrls,
            silent: true,
        });
    }));
    commander_1.program
        .command('plugin:ci-build')
        .option('--finish', 'move all results to the jobs folder', false)
        .option('--maxJestWorkers <num>|<string>', 'Limit number of Jest workers spawned')
        .description('[deprecated] Build the plugin, leaving results in /dist and /coverage')
        .action((cmd) => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
        yield (0, execTask_1.execTask)(plugin_ci_1.ciBuildPluginTask)({
            finish: cmd.finish,
            maxJestWorkers: cmd.maxJestWorkers,
        });
    }));
    commander_1.program
        .command('plugin:ci-package')
        .option('--signatureType <type>', 'Signature Type')
        .option('--rootUrls <urls...>', 'Root URLs')
        .option('--signing-admin', 'Use the admin API endpoint for signing the manifest. (deprecated)', false)
        .description('[deprecated] Create a zip packages for the plugin')
        .action((cmd) => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
        yield (0, execTask_1.execTask)(plugin_ci_1.ciPackagePluginTask)({
            signatureType: cmd.signatureType,
            rootUrls: cmd.rootUrls,
        });
    }));
    commander_1.program
        .command('plugin:ci-report')
        .description('[deprecated] Build a report for this whole process')
        .option('--upload', 'upload packages also')
        .action((cmd) => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
        yield (0, execTask_1.execTask)(plugin_ci_1.ciPluginReportTask)({
            upload: cmd.upload,
        });
    }));
    commander_1.program
        .command('plugin:bundle-managed')
        .description('Builds managed plugins')
        .action((cmd) => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
        yield (0, execTask_1.execTask)(bundle_managed_1.bundleManagedTask)({});
    }));
    commander_1.program
        .command('plugin:github-publish')
        .option('--dryrun', 'Do a dry run only', false)
        .option('--verbose', 'Print verbose', false)
        .option('--commitHash <hashKey>', 'Specify the commit hash')
        .description('Publish to github')
        .action((cmd) => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
        yield (0, execTask_1.execTask)(plugin_utils_1.githubPublishTask)({
            dryrun: cmd.dryrun,
            verbose: cmd.verbose,
            commitHash: cmd.commitHash,
        });
    }));
    commander_1.program
        .command('plugin:update-circleci')
        .description('Update plugin')
        .action((cmd) => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
        yield (0, execTask_1.execTask)(plugin_update_1.pluginUpdateTask)({});
    }));
    commander_1.program.on('command:*', () => {
        console.error('Invalid command: %s\nSee --help for a list of available commands.', commander_1.program.args.join(' '));
        process.exit(1);
    });
    commander_1.program.parse(process.argv);
    const options = commander_1.program.opts();
    if (options.depreciate && options.depreciate.length === 2) {
        console.log(chalk_1.default.yellow.bold(`[NPM script depreciation] ${options.depreciate[0]} is deprecated! Use ${options.depreciate[1]} instead!`));
    }
};
exports.run = run;
//# sourceMappingURL=index.js.map