/*
 * COPYRIGHT (c) International Business Machines Corp. 2011-2017
 *
 * This program is provided under the terms of the Common Public License,
 * version 1.0 (CPL-1.0). Any use, reproduction or distribution for this
 * software constitutes recipient's acceptance of CPL-1.0 terms which can be
 * found in the file LICENSE file or at
 * https://opensource.org/licenses/cpl1.0.php
 */

enum curve_type {
    CURVE_BRAINPOOL = 1,
    CURVE_PRIME = 2,
    CURVE_EDWARDS = 3,
    CURVE_MONTGOMERY = 4,
};

struct EC_TEST_VECTOR {
    CK_BYTE name[32];
    enum curve_type curve_type;
    CK_BYTE params[64];
    CK_ULONG params_len;
    CK_BYTE privkey[196];
    CK_ULONG privkey_len;
    CK_BYTE pubkey[196];
    CK_ULONG pubkey_len;
};

struct EC_TEST_VECTOR ec_tv[] = {
    {                           // #0 [PRIME192v1]
        .name = "PRIME192v1",
        .curve_type = CURVE_PRIME,
        .params = {0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x03, 0x01, 0x01},
        .params_len = 10,
        .privkey = {0x0a, 0x38, 0x68, 0x89, 0xdc, 0xe6, 0xe1, 0x93,
                    0xba, 0xc4, 0xcf, 0xd1, 0xd2, 0x0d, 0xd3, 0x2c,
                    0xc9, 0xcf, 0x08, 0x14, 0x51, 0x23, 0xd4, 0x7a},
        .privkey_len = 24,
        .pubkey = {0x04, 0x31, 0x04, 0xd9, 0xf6, 0x79, 0xa6, 0xc2,
                   0xf9, 0x16, 0x11, 0x6e, 0x8a, 0xbe, 0xe6, 0xa2,
                   0x2c, 0x64, 0xb7, 0x88, 0xaa, 0xa5, 0x7e, 0x22,
                   0x97, 0xc0, 0x0e, 0xf5, 0x6f, 0xe1, 0xd4, 0x4a,
                   0x47, 0xb4, 0x76, 0x3e, 0xd6, 0xa3, 0x90, 0x4c,
                   0x7d, 0xb4, 0xee, 0x61, 0x9b, 0x01, 0x37, 0x71,
                   0xfe, 0x65, 0x37},
        .pubkey_len = 51,
    },
    {                           // #1 [PRIME256v1]
        .name = "PRIME256v1",
        .curve_type = CURVE_PRIME,
        .params = {0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x03, 0x01, 0x07},
        .params_len = 10,
        .privkey = {0x00, 0xd4, 0x37, 0x30, 0xbb, 0x01, 0x9d, 0x1b,
                    0xaa, 0x72, 0x9c, 0x31, 0x5f, 0x5d, 0x25, 0xab,
                    0x9f, 0x4d, 0xeb, 0x09, 0xc5, 0x1f, 0xaf, 0xa5,
                    0xea, 0x8b, 0x37, 0x30, 0xc3, 0x93, 0x2e, 0x08,
                    0xf4},
        .privkey_len = 33,
        .pubkey = {0x04, 0x41, 0x04, 0x4f, 0x55, 0xee, 0x90, 0xba,
                   0x52, 0x17, 0x41, 0xeb, 0x58, 0xf1, 0xf5, 0x75,
                   0xc7, 0x90, 0x6b, 0x8f, 0x0e, 0xda, 0x12, 0x11,
                   0xa0, 0xed, 0xa9, 0xd0, 0xc3, 0x36, 0xc3, 0xa3,
                   0x0c, 0x64, 0x59, 0xa8, 0x8f, 0x18, 0x6f, 0x65,
                   0x1c, 0x72, 0x2a, 0xa8, 0x59, 0xeb, 0x6e, 0xe3,
                   0x0e, 0xa3, 0x79, 0xf2, 0x04, 0xd0, 0x0f, 0x67,
                   0x88, 0xf2, 0x1d, 0x43, 0x24, 0x47, 0x98, 0xf0,
                   0xd7, 0xb5, 0x60},
        .pubkey_len = 67,
    },
    {                           // #2 [SECP224r1]
        .name = "SECP224r1",
        .curve_type = CURVE_PRIME,
        .params = {0x06, 0x05, 0x2B, 0x81, 0x04, 0x00, 0x21},
        .params_len = 7,
        .privkey = {0x51, 0xbf, 0xef, 0xae, 0xb9, 0x3b, 0xa7, 0xd7,
                    0xa0, 0x6d, 0xbf, 0x10, 0xc2, 0x19, 0xa3, 0x83,
                    0x79, 0xf8, 0x4d, 0xd7, 0x98, 0x91, 0x7a, 0xcb,
                    0x88, 0xed, 0x9d, 0x0d},
        .privkey_len = 28,
        .pubkey = {0x04, 0x39, 0x04, 0xa7, 0x21, 0x25, 0xe5, 0xfd,
                   0xe4, 0xb9, 0x0d, 0x86, 0xa6, 0x32, 0x43, 0x1d,
                   0x75, 0x92, 0x38, 0x8d, 0xd1, 0x17, 0xee, 0xf3,
                   0x12, 0x66, 0xde, 0x47, 0x13, 0xde, 0xdb, 0xfc,
                   0x26, 0xb2, 0x46, 0xb4, 0xd4, 0x47, 0x61, 0x03,
                   0x71, 0x2b, 0xc5, 0xae, 0xd7, 0x63, 0x42, 0xbe,
                   0xad, 0x7b, 0x81, 0x9c, 0x70, 0xe5, 0x3c, 0x67,
                   0x65, 0x13, 0x0e},
        .pubkey_len = 59,
    },
    {                           // #3 [SECP384r1]
        .name = "SECP384r1",
        .curve_type = CURVE_PRIME,
        .params = {0x06, 0x05, 0x2B, 0x81, 0x04, 0x00, 0x22},
        .params_len = 7,
        .privkey = {0x00, 0xd4, 0xb1, 0xb8, 0x20, 0xc6, 0x54, 0x2e,
                    0x79, 0xe5, 0x6e, 0xae, 0xa5, 0xc5, 0xa3, 0x4e,
                    0x67, 0x1e, 0xe2, 0x72, 0xcc, 0x75, 0xf4, 0x7f,
                    0xbc, 0x76, 0x3c, 0xbc, 0xe1, 0x20, 0x21, 0x44,
                    0xa5, 0x7d, 0xcd, 0xac, 0x5b, 0x10, 0xda, 0xd5,
                    0xf8, 0xff, 0x97, 0x08, 0x55, 0x7c, 0x53, 0xb2,
                    0x06},
        .privkey_len = 49,
        .pubkey = {0x04, 0x61, 0x04, 0x03, 0xc6, 0x87, 0xa9, 0xb2,
                   0xff, 0x76, 0x4f, 0xb3, 0x14, 0x8f, 0x5e, 0x5f,
                   0x36, 0x38, 0x90, 0xc1, 0x15, 0xc7, 0x61, 0x38,
                   0x09, 0xec, 0x16, 0xfd, 0xdd, 0xbf, 0xdf, 0x6b,
                   0x10, 0xc6, 0xbb, 0x39, 0x1a, 0x73, 0xd8, 0x7a,
                   0x6e, 0xc6, 0xcc, 0x3b, 0x3e, 0x0f, 0x9a, 0x4d,
                   0x21, 0xa5, 0x13, 0xfe, 0x6c, 0x65, 0x98, 0x74,
                   0xa0, 0x66, 0x7d, 0x9d, 0x6b, 0x35, 0x6a, 0xed,
                   0xd3, 0xdf, 0x9c, 0x5e, 0x55, 0xc1, 0xf0, 0x7a,
                   0xbf, 0x0e, 0x66, 0xef, 0xd3, 0xb3, 0xc1, 0xf2,
                   0x7c, 0xa6, 0x8b, 0x7f, 0x41, 0x43, 0xe8, 0xc0,
                   0xf0, 0x82, 0x41, 0x44, 0xe3, 0xe1, 0x94, 0x4c,
                   0x75, 0x64, 0x63},
        .pubkey_len = 99,
    },
    {                           // #4 [SECP521r1]
        .name = "SECP521r1",
        .curve_type = CURVE_PRIME,
        .params = {0x06, 0x05, 0x2B, 0x81, 0x04, 0x00, 0x23},
        .params_len = 7,
        .privkey = {0x01, 0x56, 0xec, 0x45, 0xe0, 0x40, 0x11, 0x58,
                    0xbd, 0x4f, 0xf3, 0xba, 0xfa, 0x89, 0x7a, 0xcf,
                    0x66, 0x21, 0xe6, 0x3e, 0xc6, 0x30, 0x54, 0x50,
                    0xef, 0x0e, 0x4d, 0x7a, 0xb7, 0xa9, 0x71, 0x77,
                    0xa8, 0x83, 0xd4, 0x08, 0x37, 0x65, 0xc6, 0xe1,
                    0x7e, 0x72, 0x0d, 0x0a, 0xdf, 0xb7, 0x37, 0xc5,
                    0x3a, 0xc9, 0x96, 0x2e, 0x34, 0xe5, 0x55, 0x38,
                    0x79, 0x18, 0x93, 0x78, 0x48, 0x3f, 0x46, 0xd5,
                    0x5a, 0xe8},
        .privkey_len = 66,
        .pubkey = {0x04, 0x81, 0x85, 0x04, 0x01, 0x60, 0xe5, 0x4c,
                   0xea, 0xbf, 0x54, 0x7e, 0x45, 0xbd, 0x6b, 0xb5,
                   0x8d, 0x55, 0xb8, 0x56, 0xd2, 0x0d, 0x88, 0x49,
                   0x78, 0x72, 0x46, 0x5c, 0x1e, 0xe6, 0x09, 0x8f,
                   0xc5, 0x96, 0x12, 0x16, 0x1c, 0xf4, 0x25, 0x28,
                   0x24, 0x18, 0x7c, 0xdd, 0x23, 0xa7, 0x3b, 0x99,
                   0x2c, 0xff, 0xa2, 0xd5, 0x25, 0x57, 0xa1, 0x65,
                   0x42, 0x47, 0x2c, 0x84, 0x59, 0x8b, 0x4e, 0x14,
                   0xea, 0x20, 0x7e, 0xd0, 0x69, 0x6d, 0x00, 0x06,
                   0x4a, 0x33, 0x75, 0x2a, 0x72, 0x81, 0x5d, 0xe0,
                   0x33, 0xce, 0x7c, 0xa0, 0xa4, 0x83, 0x84, 0xf2,
                   0x2f, 0x6f, 0x2b, 0x73, 0x4c, 0x7f, 0xb5, 0x8d,
                   0xae, 0xf7, 0xce, 0xdd, 0x35, 0x1d, 0x20, 0x8d,
                   0xd6, 0x8c, 0x3f, 0x44, 0x49, 0x1b, 0xd5, 0xc0,
                   0x4a, 0x4e, 0x30, 0x6f, 0x54, 0x84, 0x10, 0xb6,
                   0x78, 0x8a, 0xfe, 0x1a, 0xdc, 0xd4, 0x49, 0x59,
                   0xb0, 0x86, 0xdc, 0x7c, 0x13, 0xd1, 0x6b, 0x2f},
        .pubkey_len = 136,
    },
    {                           // #5 [BrainpoolP160r1]
        .name = "BrainpoolP160r1",
        .curve_type = CURVE_BRAINPOOL,
        .params = {0x06, 0x09, 0x2B, 0x24, 0x03, 0x03, 0x02, 0x08,
                   0x01, 0x01, 0x01},
        .params_len = 11,
        .privkey = {0x00, 0xd7, 0x9e, 0x70, 0x72, 0x32, 0xef, 0x19,
                    0x8e, 0xa7, 0x91, 0x3a, 0x52, 0xdd, 0x69, 0x38,
                    0x4e, 0xc8, 0x9f, 0x68, 0x49},
        .privkey_len = 21,
        .pubkey = {0x04, 0x29, 0x04, 0xa6, 0x1e, 0x8d, 0x7e, 0xfd,
                   0xf6, 0x88, 0xb0, 0x88, 0xda, 0x6e, 0x83, 0x35,
                   0x82, 0x0d, 0xb7, 0xee, 0xd6, 0x35, 0x7b, 0x78,
                   0x3c, 0x8a, 0xce, 0xf9, 0xf7, 0xa3, 0x4a, 0x12,
                   0x9d, 0x3f, 0x78, 0x06, 0x75, 0x87, 0x86, 0x17,
                   0x70, 0x5c, 0x5e},
        .pubkey_len = 43,
    },
    {                           // #6 [BrainpoolP160t1]
        .name = "BrainpoolP160t1",
        .curve_type = CURVE_BRAINPOOL,
        .params = {0x06, 0x09, 0x2B, 0x24, 0x03, 0x03, 0x02, 0x08,
                   0x01, 0x01, 0x02},
        .params_len = 11,
        .privkey = {0x00, 0xda, 0xc3, 0x24, 0x84, 0x71, 0xd9, 0x46,
                    0x80, 0x5c, 0x81, 0xaa, 0x80, 0x22, 0x3e, 0xd2,
                    0x36, 0xf0, 0x77, 0x8f, 0xbc},
        .privkey_len = 21,
        .pubkey = {0x04, 0x29, 0x04, 0x34, 0xfd, 0xe0, 0x1f, 0x2a,
                   0xf0, 0x0c, 0x19, 0x02, 0x96, 0x5e, 0x11, 0xa3,
                   0x94, 0xe3, 0xe7, 0xa4, 0x14, 0xa0, 0x27, 0x6b,
                   0xc1, 0xdf, 0x9a, 0xe7, 0x8c, 0x79, 0xdf, 0xe4,
                   0xcb, 0x9b, 0xa2, 0xd4, 0x4e, 0xa9, 0x21, 0x7b,
                   0x87, 0x6b, 0x61},
        .pubkey_len = 43,
    },
    {                           // #7 [BrainpoolP192r1]
        .name = "BrainpoolP192r1",
        .curve_type = CURVE_BRAINPOOL,
        .params = {0x06, 0x09, 0x2B, 0x24, 0x03, 0x03, 0x02, 0x08,
                   0x01, 0x01, 0x03},
        .params_len = 11,
        .privkey = {0x00, 0x96, 0xe2, 0xbc, 0x9b, 0x5d, 0x81, 0x64,
                    0xe4, 0x50, 0x4a, 0x9e, 0xa0, 0xe5, 0x70, 0x59,
                    0x60, 0x39, 0x71, 0x80, 0x34, 0x2b, 0xce, 0x97,
                    0x84},
        .privkey_len = 25,
        .pubkey = {0x04, 0x31, 0x04, 0xb4, 0x17, 0xb3, 0x3b, 0x89,
                   0xe4, 0x1e, 0x05, 0x55, 0x22, 0x4e, 0x50, 0x91,
                   0xa3, 0x10, 0xaa, 0xb3, 0xa9, 0x23, 0xd6, 0x92,
                   0xf8, 0xbb, 0xeb, 0xb1, 0xf3, 0x3a, 0xc8, 0xc5,
                   0x55, 0x8d, 0x8e, 0x9d, 0x2e, 0xbf, 0x0c, 0x63,
                   0xd4, 0x51, 0xe9, 0xd4, 0x9d, 0x5e, 0x9f, 0x37,
                   0xf4, 0x03, 0xb0},
        .pubkey_len = 51,
    },
    {                           // #8 [BrainpoolP192t1]
        .name = "BrainpoolP192t1",
        .curve_type = CURVE_BRAINPOOL,
        .params = {0x06, 0x09, 0x2B, 0x24, 0x03, 0x03, 0x02, 0x08,
                   0x01, 0x01, 0x04},
        .params_len = 11,
        .privkey = {0x4c, 0xa7, 0x5b, 0x36, 0x82, 0x71, 0x45, 0x50,
                    0x09, 0x9e, 0xc3, 0xe2, 0x7c, 0x38, 0xb3, 0x0d,
                    0x94, 0x0d, 0x0e, 0x7b, 0x69, 0x65, 0xf8, 0xab},
        .privkey_len = 24,
        .pubkey = {0x04, 0x31, 0x04, 0x5b, 0x04, 0x54, 0xf7, 0xfc,
                   0xfe, 0xe8, 0x07, 0x63, 0x97, 0x1f, 0x03, 0x2d,
                   0x7e, 0x46, 0xcf, 0xf3, 0xe9, 0x66, 0x5c, 0x4e,
                   0xe4, 0x97, 0xcf, 0xc1, 0xac, 0xaa, 0x00, 0xdb,
                   0x48, 0x60, 0xa7, 0x18, 0xea, 0x6a, 0x76, 0x18,
                   0xd1, 0x06, 0x17, 0xf1, 0x39, 0x58, 0x8b, 0x63,
                   0x59, 0x26, 0xd7},
        .pubkey_len = 51,
    },
    {                           // #9 [BrainpoolP224r1]
        .name = "BrainpoolP224r1",
        .curve_type = CURVE_BRAINPOOL,
        .params = {0x06, 0x09, 0x2B, 0x24, 0x03, 0x03, 0x02, 0x08,
                   0x01, 0x01, 0x05},
        .params_len = 11,
        .privkey = {0x51, 0x09, 0x54, 0x86, 0x77, 0x1f, 0x7a, 0x31,
                    0x5a, 0xd8, 0x85, 0xa4, 0x00, 0xe1, 0xac, 0xad,
                    0xa1, 0xa6, 0x2b, 0x73, 0x8d, 0x32, 0x21, 0xf1,
                    0x65, 0xe1, 0x9f, 0x92},
        .privkey_len = 28,
        .pubkey = {0x04, 0x39, 0x04, 0x8d, 0x9b, 0xb5, 0xb1, 0x8c,
                   0xc6, 0x45, 0xb3, 0x1b, 0xdc, 0x96, 0xfe, 0x2e,
                   0xc0, 0xce, 0xe5, 0x02, 0x4c, 0xfe, 0xa1, 0xe3,
                   0x52, 0x32, 0xe3, 0xb3, 0xa3, 0xc8, 0x6c, 0x19,
                   0x79, 0xc6, 0xa0, 0xdc, 0x2a, 0x5a, 0x04, 0xcf,
                   0x1b, 0x39, 0x31, 0x1f, 0x58, 0x96, 0x63, 0x96,
                   0x6a, 0x6f, 0x45, 0x96, 0xd7, 0x7c, 0xbe, 0xbf,
                   0x83, 0xd3, 0xa2},
        .pubkey_len = 59,
    },
    {                           // #10 [BrainpoolP224t1]
        .name = "BrainpoolP224t1",
        .curve_type = CURVE_BRAINPOOL,
        .params = {0x06, 0x09, 0x2B, 0x24, 0x03, 0x03, 0x02, 0x08,
                   0x01, 0x01, 0x06},
        .params_len = 11,
        .privkey = {0x00, 0xb4, 0xa2, 0x01, 0x0a, 0xfc, 0x53, 0xe8,
                    0xe4, 0x3e, 0xf0, 0xfb, 0xc2, 0x56, 0xe8, 0xa8,
                    0x43, 0xce, 0x22, 0x65, 0x7a, 0xa7, 0x03, 0xf7,
                    0x0f, 0x6e, 0xa2, 0x16, 0x42},
        .privkey_len = 29,
        .pubkey = {0x04, 0x39, 0x04, 0xa6, 0x0a, 0x6f, 0x38, 0xec,
                   0x7c, 0xa4, 0x62, 0x4d, 0x9b, 0x8e, 0xeb, 0xd8,
                   0xf0, 0x80, 0x2e, 0x2d, 0xcf, 0x34, 0x90, 0xac,
                   0xb8, 0xf9, 0x0c, 0x84, 0x99, 0x60, 0xb0, 0x69,
                   0x1d, 0xf1, 0xa7, 0x82, 0xb5, 0x75, 0x3a, 0x26,
                   0x08, 0x89, 0x1f, 0x0c, 0xc2, 0xce, 0x68, 0x23,
                   0xad, 0x68, 0xe0, 0x96, 0xe3, 0xc7, 0x23, 0x44,
                   0xfe, 0xca, 0xc2},
        .pubkey_len = 59,
    },
    {                           // #11 [BrainpoolP256r1]
        .name = "BrainpoolP256r1",
        .curve_type = CURVE_BRAINPOOL,
        .params = {0x06, 0x09, 0x2B, 0x24, 0x03, 0x03, 0x02, 0x08,
                   0x01, 0x01, 0x07},
        .params_len = 11,
        .privkey = {0x66, 0xbb, 0xbb, 0x8d, 0x08, 0xbd, 0xd7, 0x02,
                    0x4a, 0xa0, 0x72, 0x20, 0x1e, 0x05, 0x60, 0x9c,
                    0xe4, 0x7a, 0x32, 0xab, 0x66, 0x4a, 0x7c, 0x00,
                    0x30, 0x14, 0x92, 0x3a, 0xb1, 0x8e, 0xe1, 0x90},
        .privkey_len = 32,
        .pubkey = {0x04, 0x41, 0x04, 0x6e, 0x9e, 0x8b, 0x83, 0xef,
                   0xda, 0x7e, 0x7d, 0x4b, 0x0d, 0x91, 0xee, 0x7e,
                   0x05, 0xe5, 0x09, 0x03, 0xe4, 0xbd, 0x36, 0x75,
                   0x44, 0x20, 0xd8, 0xf8, 0xf8, 0x2f, 0x92, 0x45,
                   0x97, 0xf4, 0x59, 0x8e, 0x4c, 0xd3, 0x41, 0x08,
                   0x2c, 0x2b, 0x3d, 0x7f, 0xfc, 0x8a, 0x5b, 0x4f,
                   0xaa, 0x29, 0xba, 0x4a, 0x0a, 0x99, 0x14, 0x0a,
                   0xa6, 0xe7, 0xb4, 0x6c, 0x9d, 0x55, 0xbf, 0x14,
                   0xa2, 0x7e, 0xdc},
        .pubkey_len = 67,
    },
    {                           // #12 [BrainpoolP256t1]
        .name = "BrainpoolP256t1",
        .curve_type = CURVE_BRAINPOOL,
        .params = {0x06, 0x09, 0x2B, 0x24, 0x03, 0x03, 0x02, 0x08,
                   0x01, 0x01, 0x08},
        .params_len = 11,
        .privkey = {0x32, 0x73, 0x2a, 0xf1, 0x10, 0xb5, 0x07, 0xf1,
                    0x05, 0xc0, 0x3c, 0x16, 0x1c, 0xcc, 0xf6, 0xa9,
                    0x0f, 0xdb, 0x3f, 0xeb, 0xc3, 0xd1, 0x69, 0x19,
                    0xac, 0x9b, 0x7c, 0x9d, 0xbb, 0x75, 0xfb, 0x40},
        .privkey_len = 32,
        .pubkey = {0x04, 0x41, 0x04, 0x91, 0x3b, 0xd0, 0x87, 0xc3,
                   0x92, 0x59, 0x00, 0x30, 0x01, 0x7e, 0x93, 0x1f,
                   0x79, 0x10, 0xd0, 0x7a, 0xe6, 0xe8, 0x49, 0x78,
                   0x62, 0x69, 0x8a, 0x92, 0x12, 0x9a, 0x05, 0x6e,
                   0xd3, 0x76, 0x6e, 0x5b, 0xab, 0xf6, 0xe8, 0x23,
                   0x34, 0xb0, 0xcb, 0xc6, 0x1c, 0xb3, 0x18, 0x8b,
                   0x51, 0x11, 0x2f, 0x6b, 0xca, 0x0a, 0xf7, 0xf4,
                   0x27, 0x6d, 0x16, 0x09, 0x7c, 0x27, 0x03, 0xd2,
                   0x5e, 0x8e, 0x14},
        .pubkey_len = 67,
    },
    {                           // #13 [BrainpoolP320r1]
        .name = "BrainpoolP320r1",
        .curve_type = CURVE_BRAINPOOL,
        .params = {0x06, 0x09, 0x2B, 0x24, 0x03, 0x03, 0x02, 0x08,
                   0x01, 0x01, 0x09},
        .params_len = 11,
        .privkey = {0x00, 0xa5, 0xba, 0x71, 0x28, 0xa0, 0x90, 0xcd,
                    0x5a, 0x8f, 0xda, 0x07, 0xfa, 0xac, 0x9b, 0x00,
                    0x81, 0x36, 0x70, 0x6b, 0x0b, 0x32, 0xc9, 0x6b,
                    0x4b, 0xc2, 0x58, 0x35, 0xd6, 0x0b, 0x57, 0x12,
                    0x43, 0x65, 0x4b, 0xc6, 0xa3, 0x41, 0x7d, 0x7c,
                    0xda},
        .privkey_len = 41,
        .pubkey = {0x04, 0x51, 0x04, 0xaa, 0x22, 0x58, 0x21, 0xfe,
                   0xe3, 0xfc, 0x71, 0x82, 0x2a, 0xb0, 0x52, 0xe5,
                   0xe5, 0x6d, 0x7c, 0x01, 0x05, 0x7b, 0x97, 0x7d,
                   0xd7, 0xf3, 0xf8, 0x0e, 0xce, 0xe7, 0xd1, 0xfa,
                   0xbd, 0x0c, 0x2c, 0xb8, 0x7b, 0x2b, 0xbf, 0xfe,
                   0x9e, 0x0b, 0xf1, 0x9e, 0x6a, 0x1a, 0x76, 0xe1,
                   0xf3, 0xc6, 0x77, 0x38, 0x58, 0x24, 0xc2, 0x9b,
                   0xa0, 0x71, 0xe7, 0x04, 0x8b, 0xe3, 0x96, 0x49,
                   0xdd, 0x9d, 0x43, 0xc2, 0x91, 0x20, 0x59, 0xa9,
                   0x2b, 0xd2, 0x77, 0x10, 0x16, 0xc3, 0xa6, 0xe3,
                   0x89, 0x57, 0xcc},
        .pubkey_len = 83,
    },
    {                           // #14 [BrainpoolP320t1]
        .name = "BrainpoolP320t1",
        .curve_type = CURVE_BRAINPOOL,
        .params = {0x06, 0x09, 0x2B, 0x24, 0x03, 0x03, 0x02, 0x08,
                   0x01, 0x01, 0x0A},
        .params_len = 11,
        .privkey = {0x1d, 0x4c, 0xdc, 0x1e, 0x66, 0x15, 0x35, 0x80,
                    0xf8, 0xf2, 0x2e, 0x4f, 0x2c, 0x1a, 0x88, 0xfb,
                    0xef, 0xa5, 0xce, 0x29, 0xf2, 0xf9, 0x9b, 0x1f,
                    0xc5, 0x9f, 0xf7, 0x18, 0x73, 0x13, 0x84, 0x44,
                    0xe5, 0x34, 0x83, 0x4b, 0x73, 0x98, 0x44, 0x3b},
        .privkey_len = 40,
        .pubkey = {0x04, 0x51, 0x04, 0x8a, 0x02, 0x82, 0x6d, 0xe4,
                   0x33, 0x28, 0xd3, 0x79, 0x8a, 0x39, 0x1f, 0x30,
                   0x5b, 0xe7, 0xae, 0xaa, 0x3b, 0xca, 0x2f, 0x64,
                   0xc0, 0x1c, 0x8c, 0x83, 0x96, 0x82, 0xfe, 0x1c,
                   0x1f, 0x9e, 0x9a, 0x29, 0xc8, 0xb7, 0x8b, 0x36,
                   0x0d, 0x20, 0xff, 0x71, 0x22, 0x8b, 0xe7, 0xa6,
                   0x28, 0xbf, 0x1f, 0x45, 0x9d, 0xca, 0x67, 0xef,
                   0x9b, 0xba, 0xbc, 0x6c, 0x47, 0xa9, 0xf1, 0x9f,
                   0x22, 0x57, 0x54, 0x32, 0xd2, 0x3d, 0x43, 0x17,
                   0x79, 0x8e, 0x74, 0xb0, 0xd8, 0x67, 0xf7, 0xa1,
                   0x4d, 0x29, 0x13},
        .pubkey_len = 83,
    },
    {                           // #15 [BrainpoolP384r1]
        .name = "BrainpoolP384r1",
        .curve_type = CURVE_BRAINPOOL,
        .params = {0x06, 0x09, 0x2B, 0x24, 0x03, 0x03, 0x02, 0x08,
                   0x01, 0x01, 0x0B},
        .params_len = 11,
        .privkey = {0x49, 0x56, 0xb3, 0xba, 0xe7, 0xbb, 0x27, 0xcb,
                    0x7f, 0x10, 0x84, 0xc7, 0xb4, 0x14, 0xfe, 0x2c,
                    0x8b, 0xc4, 0xbe, 0x94, 0x0d, 0x2e, 0x20, 0xc9,
                    0x54, 0x73, 0x17, 0x89, 0x8e, 0x4d, 0xa2, 0x6d,
                    0xa8, 0x6d, 0xa0, 0xfb, 0x89, 0x1f, 0xf6, 0xce,
                    0x4a, 0x45, 0xe7, 0xa9, 0x02, 0x0c, 0xa2, 0x7b},
        .privkey_len = 48,
        .pubkey = {0x04, 0x61, 0x04, 0x31, 0xb9, 0xa2, 0x0a, 0x47,
                   0x60, 0xc3, 0xa0, 0x27, 0x87, 0x04, 0x44, 0xdc,
                   0x6c, 0xd0, 0xa3, 0x83, 0x09, 0xdb, 0x6d, 0x9c,
                   0x48, 0xdf, 0x4d, 0xe5, 0x72, 0x0c, 0x71, 0x69,
                   0x28, 0xbb, 0x1f, 0x33, 0x63, 0xdb, 0x9f, 0x94,
                   0xb0, 0x28, 0x0e, 0xed, 0x39, 0x41, 0x56, 0x56,
                   0x92, 0xba, 0xed, 0x06, 0x3b, 0xb0, 0x1d, 0x68,
                   0x20, 0x5d, 0xcc, 0xf4, 0x4a, 0xcd, 0x3f, 0x09,
                   0x73, 0x51, 0x12, 0xbb, 0x94, 0x7d, 0xfd, 0x0a,
                   0x89, 0xd1, 0xfa, 0xa9, 0xd4, 0xe5, 0x39, 0xcd,
                   0x13, 0x8b, 0x23, 0x19, 0xbe, 0x12, 0xae, 0xc7,
                   0x41, 0x18, 0x55, 0x04, 0x30, 0xee, 0xb5, 0x4c,
                   0x2c, 0x33, 0xe2},
        .pubkey_len = 99,
    },
    {                           // #16 [BrainpoolP384t1]
        .name = "BrainpoolP384t1",
        .curve_type = CURVE_BRAINPOOL,
        .params = {0x06, 0x09, 0x2B, 0x24, 0x03, 0x03, 0x02, 0x08,
                   0x01, 0x01, 0x0C},
        .params_len = 11,
        .privkey = {0x76, 0xe2, 0x62, 0x73, 0x3b, 0xb5, 0x42, 0x2a,
                    0x9c, 0x29, 0x3b, 0x8c, 0x07, 0x23, 0xcc, 0x5a,
                    0x6b, 0x4f, 0x30, 0x50, 0x1e, 0x00, 0xbf, 0xd6,
                    0xab, 0x52, 0xda, 0x1c, 0x76, 0xf0, 0x6e, 0x56,
                    0xbd, 0x52, 0x83, 0x05, 0x4b, 0xb0, 0x8c, 0xa8,
                    0x69, 0xab, 0x5f, 0x6b, 0x8c, 0x3f, 0xda, 0x09},
        .privkey_len = 48,
        .pubkey = {0x04, 0x61, 0x04, 0x49, 0xc3, 0xd5, 0x8d, 0x45,
                   0xf1, 0x97, 0x6a, 0xd5, 0xb3, 0x06, 0x93, 0x14,
                   0xe2, 0x94, 0x32, 0xb3, 0xe1, 0xdf, 0x7a, 0x35,
                   0x98, 0x22, 0x11, 0x90, 0x55, 0x13, 0x77, 0x31,
                   0x88, 0xf5, 0xe0, 0x55, 0xdf, 0x49, 0x0e, 0x6b,
                   0x84, 0x18, 0xea, 0xe9, 0x49, 0x73, 0x46, 0x86,
                   0x9e, 0x6a, 0x3b, 0x3a, 0x80, 0x6e, 0x1c, 0x4c,
                   0x3b, 0xd9, 0x78, 0x16, 0x3c, 0x02, 0x11, 0x6e,
                   0x1f, 0x59, 0x21, 0xbb, 0x20, 0x89, 0xa2, 0x30,
                   0x06, 0x6b, 0x3e, 0x5a, 0x07, 0xf3, 0x80, 0xd3,
                   0xc0, 0xb7, 0x5a, 0x84, 0x4b, 0x62, 0x5b, 0x17,
                   0xb2, 0xc6, 0xcf, 0x1a, 0x00, 0xdc, 0x87, 0xc2,
                   0xcd, 0x59, 0x17},
        .pubkey_len = 99,
    },
    {                           // #17 [BrainpoolP512r1]
        .name = "BrainpoolP512r1",
        .curve_type = CURVE_BRAINPOOL,
        .params = {0x06, 0x09, 0x2B, 0x24, 0x03, 0x03, 0x02, 0x08,
                   0x01, 0x01, 0x0D},
        .params_len = 11,
        .privkey = {0x72, 0xfc, 0xe0, 0x63, 0xea, 0xbc, 0x9a, 0x03,
                    0xb5, 0x42, 0x82, 0x00, 0x75, 0xa4, 0x72, 0xcf,
                    0x3c, 0x12, 0xe3, 0xf5, 0xf4, 0xf5, 0xc4, 0x9a,
                    0xf8, 0xe0, 0xa9, 0x40, 0xfa, 0xe9, 0xe0, 0xf6,
                    0x10, 0x99, 0x18, 0x60, 0xce, 0xce, 0x07, 0x08,
                    0x0e, 0x02, 0xb8, 0x7c, 0xa6, 0xa2, 0x78, 0x40,
                    0x83, 0xf1, 0xa2, 0x53, 0x7b, 0xf3, 0x53, 0x0f,
                    0xca, 0xfa, 0x30, 0xe2, 0x9a, 0xae, 0xba, 0x6e},
        .privkey_len = 64,
        .pubkey = {0x04, 0x81, 0x81, 0x04, 0x78, 0x38, 0x8a, 0xdf,
                   0xbf, 0xe7, 0xd9, 0xf5, 0x23, 0xaf, 0x22, 0x52,
                   0xe5, 0xe7, 0x0f, 0x7f, 0x88, 0x50, 0x8b, 0x31,
                   0xd3, 0x31, 0x05, 0x4a, 0xc9, 0xb2, 0xed, 0x5d,
                   0x93, 0x58, 0x3a, 0x1f, 0x26, 0x77, 0x50, 0x24,
                   0x27, 0xc4, 0x89, 0xfb, 0xdb, 0x1a, 0x2d, 0x87,
                   0x30, 0x85, 0x4e, 0x68, 0x5b, 0x1b, 0xa3, 0x45,
                   0x92, 0x8f, 0x4c, 0x8d, 0x79, 0xd6, 0xd6, 0x12,
                   0x35, 0x80, 0xbe, 0x34, 0x1c, 0xa1, 0xec, 0x10,
                   0x59, 0xb1, 0x4e, 0x8b, 0xe2, 0xb7, 0x7d, 0x64,
                   0x58, 0x00, 0xd1, 0x73, 0xb3, 0x5b, 0x16, 0x97,
                   0x42, 0x0b, 0x2b, 0x84, 0xd8, 0x46, 0xd7, 0xa1,
                   0x8e, 0x98, 0xd7, 0x59, 0x96, 0x7d, 0x88, 0x9d,
                   0x0b, 0xc0, 0x32, 0xaa, 0x76, 0x42, 0x90, 0xd2,
                   0x61, 0xe1, 0xe0, 0x8f, 0xa0, 0x57, 0x53, 0xfd,
                   0xa4, 0x07, 0xaf, 0x21, 0x8b, 0x77, 0x2e, 0x31,
                   0xa4, 0x4b, 0x84, 0xa1},
        .pubkey_len = 132,
    },
    {                           // #18 [BrainpoolP512t1]
        .name = "BrainpoolP512t1",
        .curve_type = CURVE_BRAINPOOL,
        .params = {0x06, 0x09, 0x2B, 0x24, 0x03, 0x03, 0x02, 0x08,
                   0x01, 0x01, 0x0E},
        .params_len = 11,
        .privkey = {0x00, 0xa3, 0xdb, 0xf8, 0x98, 0x71, 0x23, 0x93,
                    0x92, 0xf7, 0x62, 0xe5, 0xc7, 0x10, 0xb0, 0xc8,
                    0xb0, 0xef, 0xae, 0x4b, 0x0e, 0x99, 0x0d, 0x89,
                    0x56, 0x4b, 0xa7, 0x9c, 0xc3, 0x18, 0x0b, 0x13,
                    0x00, 0x9f, 0x95, 0x7e, 0x2f, 0x52, 0x20, 0xcc,
                    0xcc, 0x55, 0x99, 0x45, 0x9a, 0x8e, 0x0c, 0x5f,
                    0xf4, 0xe4, 0x5c, 0x2d, 0xd5, 0xcb, 0x9f, 0xa5,
                    0x9f, 0x3f, 0xa3, 0xc2, 0x00, 0xd1, 0xfe, 0xeb,
                    0x6c},
        .privkey_len = 65,
        .pubkey = {0x04, 0x81, 0x81, 0x04, 0x93, 0x0a, 0x7d, 0xe1,
                   0x95, 0x97, 0x84, 0x1d, 0xf3, 0xa9, 0x93, 0xed,
                   0x23, 0x86, 0x89, 0x44, 0xb5, 0x7e, 0xec, 0xc9,
                   0xd8, 0x92, 0x0d, 0xca, 0xf6, 0x46, 0x5b, 0xd3,
                   0x16, 0xc6, 0xc4, 0x05, 0x50, 0x98, 0x99, 0xeb,
                   0xab, 0x5e, 0x2a, 0xea, 0x4e, 0x25, 0x60, 0x23,
                   0x2c, 0x8a, 0x8a, 0x6f, 0xd3, 0xc4, 0xab, 0xff,
                   0x93, 0x34, 0xb2, 0x9c, 0x76, 0x2a, 0x55, 0x00,
                   0xb9, 0x16, 0xea, 0xfd, 0x39, 0x17, 0x2e, 0xae,
                   0x62, 0x33, 0x5f, 0x62, 0x91, 0xcc, 0x06, 0xcc,
                   0xc9, 0x3f, 0x86, 0xbb, 0x11, 0x52, 0x81, 0xfb,
                   0xc6, 0x10, 0xef, 0xc5, 0x19, 0x2f, 0xb6, 0x61,
                   0x7e, 0xde, 0xad, 0xbe, 0x43, 0x05, 0x0c, 0xfd,
                   0xe1, 0x8a, 0x99, 0x26, 0x28, 0x38, 0xaa, 0x3b,
                   0x0f, 0x8c, 0x68, 0x7e, 0x4f, 0xfc, 0x47, 0x83,
                   0xd7, 0x16, 0x2a, 0x3c, 0x9f, 0x60, 0xf1, 0x59,
                   0x82, 0x91, 0x29, 0x0c},
        .pubkey_len = 132,
    },
    {                           // #19 [curve25519]
        .name = "curve25519",
        .curve_type = CURVE_MONTGOMERY,
        .params = {0x06, 0x03, 0x2B, 0x65, 0x6E},
        .params_len = 5,
        .privkey = {0x77, 0x07, 0x6D, 0x0A, 0x73, 0x18, 0xA5, 0x7D,
                    0x3C, 0x16, 0xC1, 0x72, 0x51, 0xB2, 0x66, 0x45,
                    0xDF, 0x4C, 0x2F, 0x87, 0xEB, 0xC0, 0x99, 0x2A,
                    0xB1, 0x77, 0xFB, 0xA5, 0x1D, 0xB9, 0x2C, 0x2A},
        .privkey_len = 32,
        .pubkey = {0x04, 0x20, 0x85, 0x20, 0xF0, 0x09, 0x89, 0x30,
                   0xA7, 0x54, 0x74, 0x8B, 0x7D, 0xDC, 0xB4, 0x3E,
                   0xF7, 0x5A, 0x0D, 0xBF, 0x3A, 0x0D, 0x26, 0x38,
                   0x1A, 0xF4, 0xEB, 0xA4, 0xA9, 0x8E, 0xAA, 0x9B,
                   0x4E, 0x6A},
        .pubkey_len = 34,
    },
    {                           // #20 [curve448]
        .name = "curve448",
        .curve_type = CURVE_MONTGOMERY,
        .params = {0x06, 0x03, 0x2B, 0x65, 0x6F},
        .params_len = 5,
        .privkey = {0x9A, 0x8F, 0x49, 0x25, 0xD1, 0x51, 0x9F, 0x57,
                    0x75, 0xCF, 0x46, 0xB0, 0x4B, 0x58, 0x00, 0xD4,
                    0xEE, 0x9E, 0xE8, 0xBA, 0xE8, 0xBC, 0x55, 0x65,
                    0xD4, 0x98, 0xC2, 0x8D, 0xD9, 0xC9, 0xBA, 0xF5,
                    0x74, 0xA9, 0x41, 0x97, 0x44, 0x89, 0x73, 0x91,
                    0x00, 0x63, 0x82, 0xA6, 0xF1, 0x27, 0xAB, 0x1D,
                    0x9A, 0xC2, 0xD8, 0xC0, 0xA5, 0x98, 0x72, 0xEB},
        .privkey_len = 56,
        .pubkey = {0x04, 0x38, 0x9B, 0x08, 0xF7, 0xCC, 0x31, 0xB7,
                   0xE3, 0xE6, 0x7D, 0x22, 0xD5, 0xAE, 0xA1, 0x21,
                   0x07, 0x4A, 0x27, 0x3B, 0xD2, 0xB8, 0x3D, 0xE0,
                   0x9C, 0x63, 0xFA, 0xA7, 0x3D, 0x2C, 0x22, 0xC5,
                   0xD9, 0xBB, 0xC8, 0x36, 0x64, 0x72, 0x41, 0xD9,
                   0x53, 0xD4, 0x0C, 0x5B, 0x12, 0xDA, 0x88, 0x12,
                   0x0D, 0x53, 0x17, 0x7F, 0x80, 0xE5, 0x32, 0xC4,
                   0x1F, 0xA0 },
        .pubkey_len = 58,
    },
    {                           // #21 [ed25519]
        .name = "ed25519",
        .curve_type = CURVE_EDWARDS,
        .params = {0x06, 0x03, 0x2B, 0x65, 0x70},
        .params_len = 5,
        .privkey = {0x9D, 0x61, 0xB1, 0x9D, 0xEF, 0xFD, 0x5A, 0x60,
                    0xBA, 0x84, 0x4A, 0xF4, 0x92, 0xEC, 0x2C, 0xC4,
                    0x44, 0x49, 0xC5, 0x69, 0x7B, 0x32, 0x69, 0x19,
                    0x70, 0x3B, 0xAC, 0x03, 0x1C, 0xAE, 0x7F, 0x60},
        .privkey_len = 32,
        .pubkey = {0x04, 0x20, 0xD7, 0x5A, 0x98, 0x01, 0x82, 0xB1,
                   0x0A, 0xB7, 0xD5, 0x4B, 0xFE, 0xD3, 0xC9, 0x64,
                   0x07, 0x3A, 0x0E, 0xE1, 0x72, 0xF3, 0xDA, 0xA6,
                   0x23, 0x25, 0xAF, 0x02, 0x1A, 0x68, 0xF7, 0x07,
                   0x51, 0x1A},
        .pubkey_len = 34,
    },
    {                           // #22 [ed448]
        .name = "ed448",
        .curve_type = CURVE_EDWARDS,
        .params = {0x06, 0x03, 0x2B, 0x65, 0x71},
        .params_len = 5,
        .privkey = {0x6C, 0x82, 0xA5, 0x62, 0xCB, 0x80, 0x8D, 0x10,
                    0xD6, 0x32, 0xBE, 0x89, 0xC8, 0x51, 0x3E, 0xBF,
                    0x6C, 0x92, 0x9F, 0x34, 0xDD, 0xFA, 0x8C, 0x9F,
                    0x63, 0xC9, 0x96, 0x0E, 0xF6, 0xE3, 0x48, 0xA3,
                    0x52, 0x8C, 0x8A, 0x3F, 0xCC, 0x2F, 0x04, 0x4E,
                    0x39, 0xA3, 0xFC, 0x5B, 0x94, 0x49, 0x2F, 0x8F,
                    0x03, 0x2E, 0x75, 0x49, 0xA2, 0x00, 0x98, 0xF9,
                    0x5B},
        .privkey_len = 57,
        .pubkey = {0x04, 0x39, 0x5F, 0xD7, 0x44, 0x9B, 0x59, 0xB4,
                   0x61, 0xFD, 0x2C, 0xE7, 0x87, 0xEC, 0x61, 0x6A,
                   0xD4, 0x6A, 0x1D, 0xA1, 0x34, 0x24, 0x85, 0xA7,
                   0x0E, 0x1F, 0x8A, 0x0E, 0xA7, 0x5D, 0x80, 0xE9,
                   0x67, 0x78, 0xED, 0xF1, 0x24, 0x76, 0x9B, 0x46,
                   0xC7, 0x06, 0x1B, 0xD6, 0x78, 0x3D, 0xF1, 0xE5,
                   0x0F, 0x6C, 0xD1, 0xFA, 0x1A, 0xBE, 0xAF, 0xE8,
                   0x25, 0x61, 0x80 },
        .pubkey_len = 59,
    },
};

#define EC_TV_NUM sizeof(ec_tv)/sizeof(struct EC_TEST_VECTOR)

// Elliptic Curve lengths in bits
#define CURVE160_LENGTH        0x00A0
#define CURVE192_LENGTH        0x00C0
#define CURVE224_LENGTH        0x00E0
#define CURVE256_LENGTH        0x0100
#define CURVE320_LENGTH        0x0140
#define CURVE384_LENGTH        0x0180
#define CURVE456_LENGTH        0x01C8
#define CURVE512_LENGTH        0x0200
#define CURVE521_LENGTH        0x0209

// EC test vectors for ECDH
#define NUM_SECRET_KEY_LENGTHS sizeof(secret_key_len)/sizeof(CK_ULONG)
CK_ULONG secret_key_len[] = { 0, 8, 16, 24, 32, 48, 64, 66, 80, 123, 3456, 56789 };

#define NUM_SHARED_DATA (sizeof(shared_data)/sizeof(shared_data_t))

typedef struct {
    unsigned int length;
    unsigned char data[128];
} shared_data_t;

static shared_data_t shared_data[] = {
      {0, {0}},
      {1, {0x00}},
      {2, {0x01,0x02}},
     {16, {0x32,0x3F,0xA3,0x16,0x9D,0x8E,0x9C,0x65,0x93,0xF5,0x94,0x76,0xBC,0x14,0x20,0x00}},
     {64, {0x16,0x30,0x2F,0xF0,0xDB,0xBB,0x5A,0x8D,0x73,0x3D,0xAB,0x71,0x41,0xC1,0xB4,0x5A,
           0xCB,0xC8,0x71,0x59,0x39,0x67,0x7F,0x6A,0x56,0x85,0x0A,0x38,0xBD,0x87,0xBD,0x59,
           0xB0,0x9E,0x80,0x27,0x96,0x09,0xFF,0x33,0x3E,0xB9,0xD4,0xC0,0x61,0x23,0x1F,0xB2,
           0x6F,0x92,0xEE,0xB0,0x49,0x82,0xA5,0xF1,0xD1,0x76,0x4C,0xAD,0x57,0x66,0x54,0x22,}},
    {123, {0xCB,0xC8,0x71,0x59,0x39,0x67,0x7F,0x6A,0x56,0x85,0x0A,0x38,0xBD,0x87,0xBD,0x59,
           0xB0,0x9E,0x80,0x27,0x96,0x09,0xFF,0x33,0x3E,0xB9,0xD4,0xC0,0x61,0x23,0x1F,0xB2,
           0xCB,0xC8,0x71,0x59,0x39,0x67,0x7F,0x6A,0x56,0x85,0x0A,0x38,0xBD,0x87,0xBD,0x59,
           0xB0,0x9E,0x80,0x27,0x96,0x09,0xFF,0x33,0x3E,0xB9,0xD4,0xC0,0x61,0x23,0x1F,0xB2,
           0xCB,0xC8,0x71,0x59,0x39,0x67,0x7F,0x6A,0x56,0x85,0x0A,0x38,0xBD,0x87,0xBD,0x59,
           0xB0,0x9E,0x80,0x27,0x96,0x09,0xFF,0x33,0x3E,0xB9,0xD4,0xC0,0x61,0x23,0x1F,0xB2,
           0xCB,0xC8,0x71,0x59,0x39,0x67,0x7F,0x6A,0x56,0x85,0x0A,0x38,0xBD,0x87,0xBD,0x59,
           0xB0,0x9E,0x80,0x27,0x96,0x09,0xFF,0x33,0x3E,0xB9,0xD4,}},
};

// ECDH KATs
struct ECDH_TEST_VECTOR {
    CK_BYTE  name[32];
    CK_BYTE  params[64];
    CK_ULONG params_len;
    CK_BYTE  privkeyA[196];
    CK_BYTE  pubkeyA[196];
    CK_ULONG privkey_len;
    CK_BYTE  privkeyB[196];
    CK_BYTE  pubkeyB[196];
    CK_ULONG pubkey_len;
    CK_ULONG kdf;
    CK_BYTE  shared_data[196];
    CK_ULONG shared_data_len;
    CK_BYTE  derived_key[512];
    CK_ULONG derived_key_len;
};

/**
 * Test vectors for ECDH.
 *
 *    Note: these test vectors contain compressed public keys that get uncompressed
 *          using openssl. If openssl is built in fips-mode, only few curves are
 *          supported. Therefore, compressed keys are used only for curves that
 *          are supported by openssl-fips.
 */
struct ECDH_TEST_VECTOR ecdh_tv[] = {

    {   // #0 [PRIME192v1]
        .name   = "PRIME192v1",
        .params = {0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x03, 0x01, 0x01},
        .params_len = 10,
        .privkeyA = {0x0a,0x38,0x68,0x89,0xdc,0xe6,0xe1,0x93,0xba,0xc4,0xcf,0xd1,0xd2,0x0d,0xd3,0x2c,
                     0xc9,0xcf,0x08,0x14,0x51,0x23,0xd4,0x7a},
        .pubkeyA =  {0x04,0x31,0x04,0xd9,0xf6,0x79,0xa6,0xc2,0xf9,0x16,0x11,0x6e,0x8a,0xbe,0xe6,0xa2,
                     0x2c,0x64,0xb7,0x88,0xaa,0xa5,0x7e,0x22,0x97,0xc0,0x0e,0xf5,0x6f,0xe1,0xd4,0x4a,
                     0x47,0xb4,0x76,0x3e,0xd6,0xa3,0x90,0x4c,0x7d,0xb4,0xee,0x61,0x9b,0x01,0x37,0x71,
                     0xfe,0x65,0x37},
        .privkey_len = 24,
        .privkeyB = {0x63,0x1F,0x95,0xBB,0x4A,0x67,0x63,0x2C,0x9C,0x47,0x6E,0xEE,0x9A,0xB6,0x95,0xAB,
                     0x24,0x0A,0x04,0x99,0x30,0x7F,0xCF,0x62,},
        .pubkeyB =  {0x04,0x31,0x04,0x51,0x9A,0x12,0x16,0x80,0xE0,0x04,0x54,0x66,0xBA,0x21,0xDF,0x2E,
                     0xEE,0x47,0xF5,0x97,0x3B,0x50,0x05,0x77,0xEF,0x13,0xD5,0xFF,0x61,0x3A,0xB4,0xD6,
                     0x4C,0xEE,0x3A,0x20,0x87,0x5B,0xDB,0x10,0xF9,0x53,0xF6,0xB3,0x0C,0xA0,0x72,0xC6,
                     0x0A,0xA5,0x7F,},
        .pubkey_len = 51,
        .kdf = CKD_SHA1_KDF,
        .shared_data = {0},
        .shared_data_len = 0,
        .derived_key = {0x70,0x39,0xA4,0x26,0x35,0xF2,0xDB,0xE5,0xE6,0x9E,0xB0,0xF1,0xC7,0xD0,0x56,0xD6,
                        0x86,0xC0},
        .derived_key_len = 18,
    },

    {   // #1 [PRIME256v1]
        .name   =  "PRIME256v1",
        .params =  {0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x03, 0x01, 0x07},
        .params_len = 10,
        .privkeyA =  {0x81,0x42,0x64,0x14,0x5F,0x2F,0x56,0xF2,0xE9,0x6A,0x8E,0x33,0x7A,0x12,0x84,0x99,
                      0x3F,0xAF,0x43,0x2A,0x5A,0xBC,0xE5,0x9E,0x86,0x7B,0x72,0x91,0xD5,0x07,0xA3,0xAF,},
        .pubkeyA =   {0x04,0x41,0x04,0x2A,0xF5,0x02,0xF3,0xBE,0x89,0x52,0xF2,0xC9,0xB5,0xA8,0xD4,0x16,
                      0x0D,0x09,0xE9,0x71,0x65,0xBE,0x50,0xBC,0x42,0xAE,0x4A,0x5E,0x8D,0x3B,0x4B,0xA8,
                      0x3A,0xEB,0x15,0xEB,0x0F,0xAF,0x4C,0xA9,0x86,0xC4,0xD3,0x86,0x81,0xA0,0xF9,0x87,
                      0x2D,0x79,0xD5,0x67,0x95,0xBD,0x4B,0xFF,0x6E,0x6D,0xE3,0xC0,0xF5,0x01,0x5E,0xCE,
                      0x5E,0xFD,0x85,},
        .privkey_len = 32,
        .privkeyB =  {0x2C,0xE1,0x78,0x8E,0xC1,0x97,0xE0,0x96,0xDB,0x95,0xA2,0x00,0xCC,0x0A,0xB2,0x6A,
                      0x19,0xCE,0x6B,0xCC,0xAD,0x56,0x2B,0x8E,0xEE,0x1B,0x59,0x37,0x61,0xCF,0x7F,0x41,},
        .pubkeyB =   {0x04,0x41,0x04,0xB1,0x20,0xDE,0x4A,0xA3,0x64,0x92,0x79,0x53,0x46,0xE8,0xDE,0x6C,
                      0x2C,0x86,0x46,0xAE,0x06,0xAA,0xEA,0x27,0x9F,0xA7,0x75,0xB3,0xAB,0x07,0x15,0xF6,
                      0xCE,0x51,0xB0,0x9F,0x1B,0x7E,0xEC,0xE2,0x0D,0x7B,0x5E,0xD8,0xEC,0x68,0x5F,0xA3,
                      0xF0,0x71,0xD8,0x37,0x27,0x02,0x70,0x92,0xA8,0x41,0x13,0x85,0xC3,0x4D,0xDE,0x57,
                      0x08,0xB2,0xB6,},
        .pubkey_len = 67,
        .kdf = CKD_SHA224_KDF,
        .shared_data = {0},
        .shared_data_len = 0,
        .derived_key = {0xCA,0x11,0x2E,0x5E,0xDC,0xBE,0x71,0x78,0xB3,0xB7,0xBE,0x67,0xAF,0x54,0xE2,0x58,
                        0xB7,0x29,0x2A,0xA1,0x8A,0x85,0xBA,0x4C,0xA1,0xC6,0x58,0x44,0x94,},
        .derived_key_len = 29,
    },

    {   // #1a [PRIME256v1 with compressed odd pubkeyA and even pubkeyB, pubkey_len = 33]
        .name   =  "PRIME256v1",
        .params =  {0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x03, 0x01, 0x07},
        .params_len = 10,
        .privkeyA =  {0x81,0x42,0x64,0x14,0x5F,0x2F,0x56,0xF2,0xE9,0x6A,0x8E,0x33,0x7A,0x12,0x84,0x99,
                      0x3F,0xAF,0x43,0x2A,0x5A,0xBC,0xE5,0x9E,0x86,0x7B,0x72,0x91,0xD5,0x07,0xA3,0xAF,},
        .pubkeyA =   {0x04,0x21,0x03,0x2A,0xF5,0x02,0xF3,0xBE,0x89,0x52,0xF2,0xC9,0xB5,0xA8,0xD4,0x16,
                      0x0D,0x09,0xE9,0x71,0x65,0xBE,0x50,0xBC,0x42,0xAE,0x4A,0x5E,0x8D,0x3B,0x4B,0xA8,
                      0x3A,0xEB,0x15,},
        .privkey_len = 32,
        .privkeyB =  {0x2C,0xE1,0x78,0x8E,0xC1,0x97,0xE0,0x96,0xDB,0x95,0xA2,0x00,0xCC,0x0A,0xB2,0x6A,
                      0x19,0xCE,0x6B,0xCC,0xAD,0x56,0x2B,0x8E,0xEE,0x1B,0x59,0x37,0x61,0xCF,0x7F,0x41,},
        .pubkeyB =   {0x04,0x21,0x02,0xB1,0x20,0xDE,0x4A,0xA3,0x64,0x92,0x79,0x53,0x46,0xE8,0xDE,0x6C,
                      0x2C,0x86,0x46,0xAE,0x06,0xAA,0xEA,0x27,0x9F,0xA7,0x75,0xB3,0xAB,0x07,0x15,0xF6,
                      0xCE,0x51,0xB0,},
        .pubkey_len = 35,
        .kdf = CKD_SHA224_KDF,
        .shared_data = {0},
        .shared_data_len = 0,
        .derived_key = {0xCA,0x11,0x2E,0x5E,0xDC,0xBE,0x71,0x78,0xB3,0xB7,0xBE,0x67,0xAF,0x54,0xE2,0x58,
                        0xB7,0x29,0x2A,0xA1,0x8A,0x85,0xBA,0x4C,0xA1,0xC6,0x58,0x44,0x94,},
        .derived_key_len = 29,
    },

    {   // #1b [PRIME256v1 with hybrid odd pubkeyA and even pubkeyB, pubkey_len = 65]
        .name   =  "PRIME256v1",
        .params =  {0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x03, 0x01, 0x07},
        .params_len = 10,
        .privkeyA =  {0x81,0x42,0x64,0x14,0x5F,0x2F,0x56,0xF2,0xE9,0x6A,0x8E,0x33,0x7A,0x12,0x84,0x99,
                      0x3F,0xAF,0x43,0x2A,0x5A,0xBC,0xE5,0x9E,0x86,0x7B,0x72,0x91,0xD5,0x07,0xA3,0xAF,},
        .pubkeyA =   {0x04,0x41,0x07,0x2A,0xF5,0x02,0xF3,0xBE,0x89,0x52,0xF2,0xC9,0xB5,0xA8,0xD4,0x16,
                      0x0D,0x09,0xE9,0x71,0x65,0xBE,0x50,0xBC,0x42,0xAE,0x4A,0x5E,0x8D,0x3B,0x4B,0xA8,
                      0x3A,0xEB,0x15,0xEB,0x0F,0xAF,0x4C,0xA9,0x86,0xC4,0xD3,0x86,0x81,0xA0,0xF9,0x87,
                      0x2D,0x79,0xD5,0x67,0x95,0xBD,0x4B,0xFF,0x6E,0x6D,0xE3,0xC0,0xF5,0x01,0x5E,0xCE,
                      0x5E,0xFD,0x85,},
        .privkey_len = 32,
        .privkeyB =  {0x2C,0xE1,0x78,0x8E,0xC1,0x97,0xE0,0x96,0xDB,0x95,0xA2,0x00,0xCC,0x0A,0xB2,0x6A,
                      0x19,0xCE,0x6B,0xCC,0xAD,0x56,0x2B,0x8E,0xEE,0x1B,0x59,0x37,0x61,0xCF,0x7F,0x41,},
        .pubkeyB =   {0x04,0x41,0x06,0xB1,0x20,0xDE,0x4A,0xA3,0x64,0x92,0x79,0x53,0x46,0xE8,0xDE,0x6C,
                      0x2C,0x86,0x46,0xAE,0x06,0xAA,0xEA,0x27,0x9F,0xA7,0x75,0xB3,0xAB,0x07,0x15,0xF6,
                      0xCE,0x51,0xB0,0x9F,0x1B,0x7E,0xEC,0xE2,0x0D,0x7B,0x5E,0xD8,0xEC,0x68,0x5F,0xA3,
                      0xF0,0x71,0xD8,0x37,0x27,0x02,0x70,0x92,0xA8,0x41,0x13,0x85,0xC3,0x4D,0xDE,0x57,
                      0x08,0xB2,0xB6,},
        .pubkey_len = 67,
        .kdf = CKD_SHA224_KDF,
        .shared_data = {0},
        .shared_data_len = 0,
        .derived_key = {0xCA,0x11,0x2E,0x5E,0xDC,0xBE,0x71,0x78,0xB3,0xB7,0xBE,0x67,0xAF,0x54,0xE2,0x58,
                        0xB7,0x29,0x2A,0xA1,0x8A,0x85,0xBA,0x4C,0xA1,0xC6,0x58,0x44,0x94,},
        .derived_key_len = 29,
    },

    {   // #2 [SECP384r1]
        .name   =  "SECP384r1",
        .params =  {0x06, 0x05, 0x2B, 0x81, 0x04, 0x00, 0x22},
        .params_len =   7,
        .privkeyA = {0xD2,0x73,0x35,0xEA,0x71,0x66,0x4A,0xF2,0x44,0xDD,0x14,0xE9,0xFD,0x12,0x60,0x71,
                     0x5D,0xFD,0x8A,0x79,0x65,0x57,0x1C,0x48,0xD7,0x09,0xEE,0x7A,0x79,0x62,0xA1,0x56,
                     0xD7,0x06,0xA9,0x0C,0xBC,0xB5,0xDF,0x29,0x86,0xF0,0x5F,0xEA,0xDB,0x93,0x76,0xF1,},
        .pubkeyA =  {0x04,0x61,0x04,0x79,0x31,0x48,0xF1,0x78,0x76,0x34,0xD5,0xDA,0x4C,0x6D,0x90,0x74,
                     0x41,0x7D,0x05,0xE0,0x57,0xAB,0x62,0xF8,0x20,0x54,0xD1,0x0E,0xE6,0xB0,0x40,0x3D,
                     0x62,0x79,0x54,0x7E,0x6A,0x8E,0xA9,0xD1,0xFD,0x77,0x42,0x7D,0x01,0x6F,0xE2,0x7A,
                     0x8B,0x8C,0x66,0xC6,0xC4,0x12,0x94,0x33,0x1D,0x23,0xE6,0xF4,0x80,0xF4,0xFB,0x4C,
                     0xD4,0x05,0x04,0xC9,0x47,0x39,0x2E,0x94,0xF4,0xC3,0xF0,0x6B,0x8F,0x39,0x8B,0xB2,
                     0x9E,0x42,0x36,0x8F,0x7A,0x68,0x59,0x23,0xDE,0x3B,0x67,0xBA,0xCE,0xD2,0x14,0xA1,
                     0xA1,0xD1,0x28,},
        .privkey_len =  48,
        .privkeyB = {0x52,0xD1,0x79,0x1F,0xDB,0x4B,0x70,0xF8,0x9C,0x0F,0x00,0xD4,0x56,0xC2,0xF7,0x02,
                     0x3B,0x61,0x25,0x26,0x2C,0x36,0xA7,0xDF,0x1F,0x80,0x23,0x11,0x21,0xCC,0xE3,0xD3,
                     0x9B,0xE5,0x2E,0x00,0xC1,0x94,0xA4,0x13,0x2C,0x4A,0x6C,0x76,0x8B,0xCD,0x94,0xD2,},
        .pubkeyB =  {0x04,0x61,0x04,0x5C,0xD4,0x2A,0xB9,0xC4,0x1B,0x53,0x47,0xF7,0x4B,0x8D,0x4E,0xFB,
                     0x70,0x8B,0x3D,0x5B,0x36,0xDB,0x65,0x91,0x53,0x59,0xB4,0x4A,0xBC,0x17,0x64,0x7B,
                     0x6B,0x99,0x99,0x78,0x9D,0x72,0xA8,0x48,0x65,0xAE,0x2F,0x22,0x3F,0x12,0xB5,0xA1,
                     0xAB,0xC1,0x20,0xE1,0x71,0x45,0x8F,0xEA,0xA9,0x39,0xAA,0xA3,0xA8,0xBF,0xAC,0x46,
                     0xB4,0x04,0xBD,0x8F,0x6D,0x5B,0x34,0x8C,0x0F,0xA4,0xD8,0x0C,0xEC,0xA1,0x63,0x56,
                     0xCA,0x93,0x32,0x40,0xBD,0xE8,0x72,0x34,0x15,0xA8,0xEC,0xE0,0x35,0xB0,0xED,0xF3,
                     0x67,0x55,0xDE,},
        .pubkey_len = 99,
        .kdf = CKD_SHA256_KDF,
        .shared_data = {0},
        .shared_data_len = 0,
        .derived_key = {0xE5,0xCA,0xFD,0x5E,0x16,0x08,0x67,0xCF,0x60,0x3F,0x46,0x0F,0xFC,0x53,0x36,0x08,
                        0x29,0xF8,0xDD,0x89,0x04,0x55,0x4A,0xAB,0x4F,0xD4,0x32,0x1A,0x44,0x60,0x5C,0x68,
                        0x1B,0xCE,0xAA,0x8C,0x2C,0x97,0x36,0xE4,0x07,0x24,0x2F,0xB0,0x3C,0xA9,0x68,0x2F,
                        0xCC,0x31,0xE3,0x4B,0xBC,0xA4,0xF6,0xC0,0x5C,0x1F,0x06,0xF4,0x69,0xE8,0x7B,0xFB,
                        0x1B,0xBC,0xB6,0x92,0xB6,0xC7,0x3A,0x87,0xBA,0x51,0x72,0xE2,0xE3,0x83,0xD1,0x93,
                        0x8C,0x30,0x8F,0xB9,0x3B,0x8A,0x3A,0x48,0x2E,0xED,0x5F,0x9C,0x08,0x39,0x46,0x52,
                        0x00,0xF9,0xA7,0x19,},
        .derived_key_len = 100,
    },

    {   // #3 [SECP521r1]
        .name   = "SECP521r1",
        .params = {0x06, 0x05, 0x2B, 0x81, 0x04, 0x00, 0x23},
        .params_len = 7,
        .privkeyA = {0x00,0x37,0xAD,0xE9,0x31,0x9A,0x89,0xF4,0xDA,0xBD,0xB3,0xEF,0x41,0x1A,0xAC,0xCC,
                     0xA5,0x12,0x3C,0x61,0xAC,0xAB,0x57,0xB5,0x39,0x3D,0xCE,0x47,0x60,0x81,0x72,0xA0,
                     0x95,0xAA,0x85,0xA3,0x0F,0xE1,0xC2,0x95,0x2C,0x67,0x71,0xD9,0x37,0xBA,0x97,0x77,
                     0xF5,0x95,0x7B,0x26,0x39,0xBA,0xB0,0x72,0x46,0x2F,0x68,0xC2,0x7A,0x57,0x38,0x2D,
                     0x4A,0x52,},
        .pubkeyA =  {0x04,0x81,0x85,0x04,0x00,0x15,0x41,0x7E,0x84,0xDB,0xF2,0x8C,0x0A,0xD3,0xC2,0x78,
                     0x71,0x33,0x49,0xDC,0x7D,0xF1,0x53,0xC8,0x97,0xA1,0x89,0x1B,0xD9,0x8B,0xAB,0x43,
                     0x57,0xC9,0xEC,0xBE,0xE1,0xE3,0xBF,0x42,0xE0,0x0B,0x8E,0x38,0x0A,0xEA,0xE5,0x7C,
                     0x2D,0x10,0x75,0x64,0x94,0x18,0x85,0x94,0x2A,0xF5,0xA7,0xF4,0x60,0x17,0x23,0xC4,
                     0x19,0x5D,0x17,0x6C,0xED,0x3E,0x01,0x7C,0xAE,0x20,0xB6,0x64,0x1D,0x2E,0xEB,0x69,
                     0x57,0x86,0xD8,0xC9,0x46,0x14,0x62,0x39,0xD0,0x99,0xE1,0x8E,0x1D,0x5A,0x51,0x4C,
                     0x73,0x9D,0x7C,0xB4,0xA1,0x0A,0xD8,0xA7,0x88,0x01,0x5A,0xC4,0x05,0xD7,0x79,0x9D,
                     0xC7,0x5E,0x7B,0x7D,0x5B,0x6C,0xF2,0x26,0x1A,0x6A,0x7F,0x15,0x07,0x43,0x8B,0xF0,
                     0x1B,0xEB,0x6C,0xA3,0x92,0x6F,0x95,0x82,},
        .privkey_len = 66,
        .privkeyB = {0x01,0x45,0xBA,0x99,0xA8,0x47,0xAF,0x43,0x79,0x3F,0xDD,0x0E,0x87,0x2E,0x7C,0xDF,
                     0xA1,0x6B,0xE3,0x0F,0xDC,0x78,0x0F,0x97,0xBC,0xCC,0x3F,0x07,0x83,0x80,0x20,0x1E,
                     0x9C,0x67,0x7D,0x60,0x0B,0x34,0x37,0x57,0xA3,0xBD,0xBF,0x2A,0x31,0x63,0xE4,0xC2,
                     0xF8,0x69,0xCC,0xA7,0x45,0x8A,0xA4,0xA4,0xEF,0xFC,0x31,0x1F,0x5C,0xB1,0x51,0x68,
                     0x5E,0xB9,},
        .pubkeyB =  {0x04,0x81,0x85,0x04,0x00,0xD0,0xB3,0x97,0x5A,0xC4,0xB7,0x99,0xF5,0xBE,0xA1,0x6D,
                     0x5E,0x13,0xE9,0xAF,0x97,0x1D,0x5E,0x9B,0x98,0x4C,0x9F,0x39,0x72,0x8B,0x5E,0x57,
                     0x39,0x73,0x5A,0x21,0x9B,0x97,0xC3,0x56,0x43,0x6A,0xDC,0x6E,0x95,0xBB,0x03,0x52,
                     0xF6,0xBE,0x64,0xA6,0xC2,0x91,0x2D,0x4E,0xF2,0xD0,0x43,0x3C,0xED,0x2B,0x61,0x71,
                     0x64,0x00,0x12,0xD9,0x46,0x0F,0x01,0x5C,0x68,0x22,0x63,0x83,0x95,0x6E,0x3B,0xD0,
                     0x66,0xE7,0x97,0xB6,0x23,0xC2,0x7C,0xE0,0xEA,0xC2,0xF5,0x51,0xA1,0x0C,0x2C,0x72,
                     0x4D,0x98,0x52,0x07,0x7B,0x87,0x22,0x0B,0x65,0x36,0xC5,0xC4,0x08,0xA1,0xD2,0xAE,
                     0xBB,0x8E,0x86,0xD6,0x78,0xAE,0x49,0xCB,0x57,0x09,0x1F,0x47,0x32,0x29,0x65,0x79,
                     0xAB,0x44,0xFC,0xD1,0x7F,0x0F,0xC5,0x6A,},
        .pubkey_len = 136,
        .kdf = CKD_SHA384_KDF,
        .shared_data = {0x01,0x02,0x03},
        .shared_data_len = 3,
        .derived_key = {0xA6,0x39,0x99,0x84,0xAD,0xC3,0xAA,0x63,0x6C,0xE1,0x43,0x78,0x32,0x9C,0xAF,0x8D,
                        0x1A,0x74,0xE8,0x3E,0xCE,0xB8,0x58,0xCC,0xF7,0x5B,0x02,0x22,0x70,0x58,0x09,0xF5,
                        0x04,0xFC,0xBE,0x47,0xE7,0xB0,0x45,0xCB,0x69,0x3D,0x3F,0x84,0xDE,0x04,0xCA,0x19,
                        0xB6,0x23,0xAA,0x55,0xEA,0xE4,0x0F,0x1A,0x9C,0xD8,0xCF,0x63,0x07,0x08,0xB6,0x34,
                        0x6F,0x5E,0x38,0x7A,0xE9,0xF7,0xFF,0x44,0x3F,0xF3,0x15,0xFD,0x1E,0xE9,0xA0,0xFA,
                        0x95,0xB7,0xD7,0xC1,0xC1,0x14,0xC0,0xC9,0x22,0x01,0xF7,0xF6,0xDC,0xB9,0x0D,0xD6,
                        0x8D,0xEA,0x41,0x2A,0xAC,0x19,0xB6,0xF3,0x09,0x85,0x61,0x61,0x4E,0xF1,0x5D,0x23,
                        0x9F,0xD8,0x4E,0xC8,0xED,0x53,0x2F,0xCA,0x65,0x36,0x3E,0x1F,0xE6,0xB8,0x54,0x42,
                        0x3E,0xA5,0x38,0xB4,},
        .derived_key_len = 132,
    },

    {   // #3a [SECP521r1 with compressed even pubkeyA and pubkeyB: pubkey_len = 67]
        .name   = "SECP521r1",
        .params = {0x06, 0x05, 0x2B, 0x81, 0x04, 0x00, 0x23},
        .params_len = 7,
        .privkeyA = {0x00,0x37,0xAD,0xE9,0x31,0x9A,0x89,0xF4,0xDA,0xBD,0xB3,0xEF,0x41,0x1A,0xAC,0xCC,
                     0xA5,0x12,0x3C,0x61,0xAC,0xAB,0x57,0xB5,0x39,0x3D,0xCE,0x47,0x60,0x81,0x72,0xA0,
                     0x95,0xAA,0x85,0xA3,0x0F,0xE1,0xC2,0x95,0x2C,0x67,0x71,0xD9,0x37,0xBA,0x97,0x77,
                     0xF5,0x95,0x7B,0x26,0x39,0xBA,0xB0,0x72,0x46,0x2F,0x68,0xC2,0x7A,0x57,0x38,0x2D,
                     0x4A,0x52,},
        .pubkeyA =  {0x04,0x43,0x02,0x00,0x15,0x41,0x7E,0x84,0xDB,0xF2,0x8C,0x0A,0xD3,0xC2,0x78,0x71,
                     0x33,0x49,0xDC,0x7D,0xF1,0x53,0xC8,0x97,0xA1,0x89,0x1B,0xD9,0x8B,0xAB,0x43,0x57,
                     0xC9,0xEC,0xBE,0xE1,0xE3,0xBF,0x42,0xE0,0x0B,0x8E,0x38,0x0A,0xEA,0xE5,0x7C,0x2D,
                     0x10,0x75,0x64,0x94,0x18,0x85,0x94,0x2A,0xF5,0xA7,0xF4,0x60,0x17,0x23,0xC4,0x19,
                     0x5D,0x17,0x6C,0xED,0x3E,},
        .privkey_len = 66,
        .privkeyB = {0x01,0x45,0xBA,0x99,0xA8,0x47,0xAF,0x43,0x79,0x3F,0xDD,0x0E,0x87,0x2E,0x7C,0xDF,
                     0xA1,0x6B,0xE3,0x0F,0xDC,0x78,0x0F,0x97,0xBC,0xCC,0x3F,0x07,0x83,0x80,0x20,0x1E,
                     0x9C,0x67,0x7D,0x60,0x0B,0x34,0x37,0x57,0xA3,0xBD,0xBF,0x2A,0x31,0x63,0xE4,0xC2,
                     0xF8,0x69,0xCC,0xA7,0x45,0x8A,0xA4,0xA4,0xEF,0xFC,0x31,0x1F,0x5C,0xB1,0x51,0x68,
                     0x5E,0xB9,},
        .pubkeyB =  {0x04,0x43,0x02,0x00,0xD0,0xB3,0x97,0x5A,0xC4,0xB7,0x99,0xF5,0xBE,0xA1,0x6D,0x5E,
                     0x13,0xE9,0xAF,0x97,0x1D,0x5E,0x9B,0x98,0x4C,0x9F,0x39,0x72,0x8B,0x5E,0x57,0x39,
                     0x73,0x5A,0x21,0x9B,0x97,0xC3,0x56,0x43,0x6A,0xDC,0x6E,0x95,0xBB,0x03,0x52,0xF6,
                     0xBE,0x64,0xA6,0xC2,0x91,0x2D,0x4E,0xF2,0xD0,0x43,0x3C,0xED,0x2B,0x61,0x71,0x64,
                     0x00,0x12,0xD9,0x46,0x0F,},
        .pubkey_len = 69,
        .kdf = CKD_SHA384_KDF,
        .shared_data = {0x01,0x02,0x03},
        .shared_data_len = 3,
        .derived_key = {0xA6,0x39,0x99,0x84,0xAD,0xC3,0xAA,0x63,0x6C,0xE1,0x43,0x78,0x32,0x9C,0xAF,0x8D,
                        0x1A,0x74,0xE8,0x3E,0xCE,0xB8,0x58,0xCC,0xF7,0x5B,0x02,0x22,0x70,0x58,0x09,0xF5,
                        0x04,0xFC,0xBE,0x47,0xE7,0xB0,0x45,0xCB,0x69,0x3D,0x3F,0x84,0xDE,0x04,0xCA,0x19,
                        0xB6,0x23,0xAA,0x55,0xEA,0xE4,0x0F,0x1A,0x9C,0xD8,0xCF,0x63,0x07,0x08,0xB6,0x34,
                        0x6F,0x5E,0x38,0x7A,0xE9,0xF7,0xFF,0x44,0x3F,0xF3,0x15,0xFD,0x1E,0xE9,0xA0,0xFA,
                        0x95,0xB7,0xD7,0xC1,0xC1,0x14,0xC0,0xC9,0x22,0x01,0xF7,0xF6,0xDC,0xB9,0x0D,0xD6,
                        0x8D,0xEA,0x41,0x2A,0xAC,0x19,0xB6,0xF3,0x09,0x85,0x61,0x61,0x4E,0xF1,0x5D,0x23,
                        0x9F,0xD8,0x4E,0xC8,0xED,0x53,0x2F,0xCA,0x65,0x36,0x3E,0x1F,0xE6,0xB8,0x54,0x42,
                        0x3E,0xA5,0x38,0xB4,},
        .derived_key_len = 132,
    },

    {   // #3b [SECP521r1 without format byte in pubkeyA and pubkeyB, pubkey_len = 132]
            .name   = "SECP521r1",
            .params = {0x06, 0x05, 0x2B, 0x81, 0x04, 0x00, 0x23},
            .params_len = 7,
            .privkeyA = {0x00,0x37,0xAD,0xE9,0x31,0x9A,0x89,0xF4,0xDA,0xBD,0xB3,0xEF,0x41,0x1A,0xAC,0xCC,
                         0xA5,0x12,0x3C,0x61,0xAC,0xAB,0x57,0xB5,0x39,0x3D,0xCE,0x47,0x60,0x81,0x72,0xA0,
                         0x95,0xAA,0x85,0xA3,0x0F,0xE1,0xC2,0x95,0x2C,0x67,0x71,0xD9,0x37,0xBA,0x97,0x77,
                         0xF5,0x95,0x7B,0x26,0x39,0xBA,0xB0,0x72,0x46,0x2F,0x68,0xC2,0x7A,0x57,0x38,0x2D,
                         0x4A,0x52,},
            .pubkeyA =  {0x04,0x81,0x84,0x00,0x15,0x41,0x7E,0x84,0xDB,0xF2,0x8C,0x0A,0xD3,0xC2,0x78,0x71,
                         0x33,0x49,0xDC,0x7D,0xF1,0x53,0xC8,0x97,0xA1,0x89,0x1B,0xD9,0x8B,0xAB,0x43,0x57,
                         0xC9,0xEC,0xBE,0xE1,0xE3,0xBF,0x42,0xE0,0x0B,0x8E,0x38,0x0A,0xEA,0xE5,0x7C,0x2D,
                         0x10,0x75,0x64,0x94,0x18,0x85,0x94,0x2A,0xF5,0xA7,0xF4,0x60,0x17,0x23,0xC4,0x19,
                         0x5D,0x17,0x6C,0xED,0x3E,0x01,0x7C,0xAE,0x20,0xB6,0x64,0x1D,0x2E,0xEB,0x69,0x57,
                         0x86,0xD8,0xC9,0x46,0x14,0x62,0x39,0xD0,0x99,0xE1,0x8E,0x1D,0x5A,0x51,0x4C,0x73,
                         0x9D,0x7C,0xB4,0xA1,0x0A,0xD8,0xA7,0x88,0x01,0x5A,0xC4,0x05,0xD7,0x79,0x9D,0xC7,
                         0x5E,0x7B,0x7D,0x5B,0x6C,0xF2,0x26,0x1A,0x6A,0x7F,0x15,0x07,0x43,0x8B,0xF0,0x1B,
                         0xEB,0x6C,0xA3,0x92,0x6F,0x95,0x82,},
            .privkey_len = 66,
            .privkeyB = {0x01,0x45,0xBA,0x99,0xA8,0x47,0xAF,0x43,0x79,0x3F,0xDD,0x0E,0x87,0x2E,0x7C,0xDF,
                         0xA1,0x6B,0xE3,0x0F,0xDC,0x78,0x0F,0x97,0xBC,0xCC,0x3F,0x07,0x83,0x80,0x20,0x1E,
                         0x9C,0x67,0x7D,0x60,0x0B,0x34,0x37,0x57,0xA3,0xBD,0xBF,0x2A,0x31,0x63,0xE4,0xC2,
                         0xF8,0x69,0xCC,0xA7,0x45,0x8A,0xA4,0xA4,0xEF,0xFC,0x31,0x1F,0x5C,0xB1,0x51,0x68,
                         0x5E,0xB9,},
            .pubkeyB =  {0x04,0x81,0x84,0x00,0xD0,0xB3,0x97,0x5A,0xC4,0xB7,0x99,0xF5,0xBE,0xA1,0x6D,0x5E,
                         0x13,0xE9,0xAF,0x97,0x1D,0x5E,0x9B,0x98,0x4C,0x9F,0x39,0x72,0x8B,0x5E,0x57,0x39,
                         0x73,0x5A,0x21,0x9B,0x97,0xC3,0x56,0x43,0x6A,0xDC,0x6E,0x95,0xBB,0x03,0x52,0xF6,
                         0xBE,0x64,0xA6,0xC2,0x91,0x2D,0x4E,0xF2,0xD0,0x43,0x3C,0xED,0x2B,0x61,0x71,0x64,
                         0x00,0x12,0xD9,0x46,0x0F,0x01,0x5C,0x68,0x22,0x63,0x83,0x95,0x6E,0x3B,0xD0,0x66,
                         0xE7,0x97,0xB6,0x23,0xC2,0x7C,0xE0,0xEA,0xC2,0xF5,0x51,0xA1,0x0C,0x2C,0x72,0x4D,
                         0x98,0x52,0x07,0x7B,0x87,0x22,0x0B,0x65,0x36,0xC5,0xC4,0x08,0xA1,0xD2,0xAE,0xBB,
                         0x8E,0x86,0xD6,0x78,0xAE,0x49,0xCB,0x57,0x09,0x1F,0x47,0x32,0x29,0x65,0x79,0xAB,
                         0x44,0xFC,0xD1,0x7F,0x0F,0xC5,0x6A,},
            .pubkey_len = 135,
            .kdf = CKD_SHA384_KDF,
            .shared_data = {0x01,0x02,0x03},
            .shared_data_len = 3,
            .derived_key = {0xA6,0x39,0x99,0x84,0xAD,0xC3,0xAA,0x63,0x6C,0xE1,0x43,0x78,0x32,0x9C,0xAF,0x8D,
                            0x1A,0x74,0xE8,0x3E,0xCE,0xB8,0x58,0xCC,0xF7,0x5B,0x02,0x22,0x70,0x58,0x09,0xF5,
                            0x04,0xFC,0xBE,0x47,0xE7,0xB0,0x45,0xCB,0x69,0x3D,0x3F,0x84,0xDE,0x04,0xCA,0x19,
                            0xB6,0x23,0xAA,0x55,0xEA,0xE4,0x0F,0x1A,0x9C,0xD8,0xCF,0x63,0x07,0x08,0xB6,0x34,
                            0x6F,0x5E,0x38,0x7A,0xE9,0xF7,0xFF,0x44,0x3F,0xF3,0x15,0xFD,0x1E,0xE9,0xA0,0xFA,
                            0x95,0xB7,0xD7,0xC1,0xC1,0x14,0xC0,0xC9,0x22,0x01,0xF7,0xF6,0xDC,0xB9,0x0D,0xD6,
                            0x8D,0xEA,0x41,0x2A,0xAC,0x19,0xB6,0xF3,0x09,0x85,0x61,0x61,0x4E,0xF1,0x5D,0x23,
                            0x9F,0xD8,0x4E,0xC8,0xED,0x53,0x2F,0xCA,0x65,0x36,0x3E,0x1F,0xE6,0xB8,0x54,0x42,
                            0x3E,0xA5,0x38,0xB4,},
            .derived_key_len = 132,
        },

        {   // #3c [SECP521r1 without format byte and leading nulls in pubkeyA and pubkeyB, pubkey_len = 131]
                .name   = "SECP521r1",
                .params = {0x06, 0x05, 0x2B, 0x81, 0x04, 0x00, 0x23},
                .params_len = 7,
                .privkeyA = {0x00,0x37,0xAD,0xE9,0x31,0x9A,0x89,0xF4,0xDA,0xBD,0xB3,0xEF,0x41,0x1A,0xAC,0xCC,
                             0xA5,0x12,0x3C,0x61,0xAC,0xAB,0x57,0xB5,0x39,0x3D,0xCE,0x47,0x60,0x81,0x72,0xA0,
                             0x95,0xAA,0x85,0xA3,0x0F,0xE1,0xC2,0x95,0x2C,0x67,0x71,0xD9,0x37,0xBA,0x97,0x77,
                             0xF5,0x95,0x7B,0x26,0x39,0xBA,0xB0,0x72,0x46,0x2F,0x68,0xC2,0x7A,0x57,0x38,0x2D,
                             0x4A,0x52,},
                .pubkeyA =  {0x04,0x81,0x83,0x15,0x41,0x7E,0x84,0xDB,0xF2,0x8C,0x0A,0xD3,0xC2,0x78,0x71,0x33,
                             0x49,0xDC,0x7D,0xF1,0x53,0xC8,0x97,0xA1,0x89,0x1B,0xD9,0x8B,0xAB,0x43,0x57,0xC9,
                             0xEC,0xBE,0xE1,0xE3,0xBF,0x42,0xE0,0x0B,0x8E,0x38,0x0A,0xEA,0xE5,0x7C,0x2D,0x10,
                             0x75,0x64,0x94,0x18,0x85,0x94,0x2A,0xF5,0xA7,0xF4,0x60,0x17,0x23,0xC4,0x19,0x5D,
                             0x17,0x6C,0xED,0x3E,0x01,0x7C,0xAE,0x20,0xB6,0x64,0x1D,0x2E,0xEB,0x69,0x57,0x86,
                             0xD8,0xC9,0x46,0x14,0x62,0x39,0xD0,0x99,0xE1,0x8E,0x1D,0x5A,0x51,0x4C,0x73,0x9D,
                             0x7C,0xB4,0xA1,0x0A,0xD8,0xA7,0x88,0x01,0x5A,0xC4,0x05,0xD7,0x79,0x9D,0xC7,0x5E,
                             0x7B,0x7D,0x5B,0x6C,0xF2,0x26,0x1A,0x6A,0x7F,0x15,0x07,0x43,0x8B,0xF0,0x1B,0xEB,
                             0x6C,0xA3,0x92,0x6F,0x95,0x82,},
                .privkey_len = 66,
                .privkeyB = {0x01,0x45,0xBA,0x99,0xA8,0x47,0xAF,0x43,0x79,0x3F,0xDD,0x0E,0x87,0x2E,0x7C,0xDF,
                             0xA1,0x6B,0xE3,0x0F,0xDC,0x78,0x0F,0x97,0xBC,0xCC,0x3F,0x07,0x83,0x80,0x20,0x1E,
                             0x9C,0x67,0x7D,0x60,0x0B,0x34,0x37,0x57,0xA3,0xBD,0xBF,0x2A,0x31,0x63,0xE4,0xC2,
                             0xF8,0x69,0xCC,0xA7,0x45,0x8A,0xA4,0xA4,0xEF,0xFC,0x31,0x1F,0x5C,0xB1,0x51,0x68,
                             0x5E,0xB9,},
                .pubkeyB =  {0x04,0x81,0x83,0xD0,0xB3,0x97,0x5A,0xC4,0xB7,0x99,0xF5,0xBE,0xA1,0x6D,0x5E,0x13,
                             0xE9,0xAF,0x97,0x1D,0x5E,0x9B,0x98,0x4C,0x9F,0x39,0x72,0x8B,0x5E,0x57,0x39,0x73,
                             0x5A,0x21,0x9B,0x97,0xC3,0x56,0x43,0x6A,0xDC,0x6E,0x95,0xBB,0x03,0x52,0xF6,0xBE,
                             0x64,0xA6,0xC2,0x91,0x2D,0x4E,0xF2,0xD0,0x43,0x3C,0xED,0x2B,0x61,0x71,0x64,0x00,
                             0x12,0xD9,0x46,0x0F,0x01,0x5C,0x68,0x22,0x63,0x83,0x95,0x6E,0x3B,0xD0,0x66,0xE7,
                             0x97,0xB6,0x23,0xC2,0x7C,0xE0,0xEA,0xC2,0xF5,0x51,0xA1,0x0C,0x2C,0x72,0x4D,0x98,
                             0x52,0x07,0x7B,0x87,0x22,0x0B,0x65,0x36,0xC5,0xC4,0x08,0xA1,0xD2,0xAE,0xBB,0x8E,
                             0x86,0xD6,0x78,0xAE,0x49,0xCB,0x57,0x09,0x1F,0x47,0x32,0x29,0x65,0x79,0xAB,0x44,
                             0xFC,0xD1,0x7F,0x0F,0xC5,0x6A,},
                .pubkey_len = 134,
                .kdf = CKD_SHA384_KDF,
                .shared_data = {0x01,0x02,0x03},
                .shared_data_len = 3,
                .derived_key = {0xA6,0x39,0x99,0x84,0xAD,0xC3,0xAA,0x63,0x6C,0xE1,0x43,0x78,0x32,0x9C,0xAF,0x8D,
                                0x1A,0x74,0xE8,0x3E,0xCE,0xB8,0x58,0xCC,0xF7,0x5B,0x02,0x22,0x70,0x58,0x09,0xF5,
                                0x04,0xFC,0xBE,0x47,0xE7,0xB0,0x45,0xCB,0x69,0x3D,0x3F,0x84,0xDE,0x04,0xCA,0x19,
                                0xB6,0x23,0xAA,0x55,0xEA,0xE4,0x0F,0x1A,0x9C,0xD8,0xCF,0x63,0x07,0x08,0xB6,0x34,
                                0x6F,0x5E,0x38,0x7A,0xE9,0xF7,0xFF,0x44,0x3F,0xF3,0x15,0xFD,0x1E,0xE9,0xA0,0xFA,
                                0x95,0xB7,0xD7,0xC1,0xC1,0x14,0xC0,0xC9,0x22,0x01,0xF7,0xF6,0xDC,0xB9,0x0D,0xD6,
                                0x8D,0xEA,0x41,0x2A,0xAC,0x19,0xB6,0xF3,0x09,0x85,0x61,0x61,0x4E,0xF1,0x5D,0x23,
                                0x9F,0xD8,0x4E,0xC8,0xED,0x53,0x2F,0xCA,0x65,0x36,0x3E,0x1F,0xE6,0xB8,0x54,0x42,
                                0x3E,0xA5,0x38,0xB4,},
                .derived_key_len = 132,
            },

    {   // #4 [BrainpoolP256r1]
        .name   = "BrainpoolP256r1",
        .params = {0x06, 0x09, 0x2B, 0x24, 0x03, 0x03, 0x02, 0x08, 0x01, 0x01, 0x07},
        .params_len = 11,
        .privkeyA = {0x81,0xDB,0x1E,0xE1,0x00,0x15,0x0F,0xF2,0xEA,0x33,0x8D,0x70,0x82,0x71,0xBE,0x38,
                     0x30,0x0C,0xB5,0x42,0x41,0xD7,0x99,0x50,0xF7,0x7B,0x06,0x30,0x39,0x80,0x4F,0x1D,},
        .pubkeyA =  {0x04,0x41,0x04,0x44,0x10,0x6E,0x91,0x3F,0x92,0xBC,0x02,0xA1,0x70,0x5D,0x99,0x53,
                     0xA8,0x41,0x4D,0xB9,0x5E,0x1A,0xAA,0x49,0xE8,0x1D,0x9E,0x85,0xF9,0x29,0xA8,0xE3,
                     0x10,0x0B,0xE5,0x8A,0xB4,0x84,0x6F,0x11,0xCA,0xCC,0xB7,0x3C,0xE4,0x9C,0xBD,0xD1,
                     0x20,0xF5,0xA9,0x00,0xA6,0x9F,0xD3,0x2C,0x27,0x22,0x23,0xF7,0x89,0xEF,0x10,0xEB,
                     0x08,0x9B,0xDC,},
        .privkey_len = 32,
        .privkeyB = {0x55,0xE4,0x0B,0xC4,0x1E,0x37,0xE3,0xE2,0xAD,0x25,0xC3,0xC6,0x65,0x45,0x11,0xFF,
                     0xA8,0x47,0x4A,0x91,0xA0,0x03,0x20,0x87,0x59,0x38,0x52,0xD3,0xE7,0xD7,0x6B,0xD3,},
        .pubkeyB =  {0x04,0x41,0x04,0x8D,0x2D,0x68,0x8C,0x6C,0xF9,0x3E,0x11,0x60,0xAD,0x04,0xCC,0x44,
                     0x29,0x11,0x7D,0xC2,0xC4,0x18,0x25,0xE1,0xE9,0xFC,0xA0,0xAD,0xDD,0x34,0xE6,0xF1,
                     0xB3,0x9F,0x7B,0x99,0x0C,0x57,0x52,0x08,0x12,0xBE,0x51,0x26,0x41,0xE4,0x70,0x34,
                     0x83,0x21,0x06,0xBC,0x7D,0x3E,0x8D,0xD0,0xE4,0xC7,0xF1,0x13,0x6D,0x70,0x06,0x54,
                     0x7C,0xEC,0x6A,},
        .pubkey_len = 67,
        .kdf = CKD_SHA512_KDF,
        .shared_data = {0x16,0x30,0x2F,0xF0,0xDB,0xBB,0x5A,0x8D,0x73,0x3D,0xAB,0x71,0x41,0xC1,0xB4,0x5A,
                        0xCB,0xC8,0x71,0x59,0x39,0x67,0x7F,0x6A,0x56,0x85,0x0A,0x38,0xBD,0x87,0xBD,0x59,
                        0xB0,0x9E,0x80,0x27,0x96,0x09,0xFF,0x33,0x3E,0xB9,0xD4,0xC0,0x61,0x23,0x1F,0xB2,
                        0x6F,0x92,0xEE,0xB0,0x49,0x82,0xA5,0xF1,0xD1,0x76,0x4C,0xAD,0x57,0x66,0x54,0x22,
                        0xBB,0xEF,0xCA},
        .shared_data_len = 67,
        .derived_key = {0x01,0x11,0xC7,0xD3,0xAC,0x19,0x45,0x20,0x8C,0x55,0xCB,0xCB,0x95,0x2E,0xFF,0x02,
                        0x10,0x08,0xF1,0x1D,0xFE,0xF3,0xB1,0x09,0x5D,0x1D,0xE5,0x96,0xEB,0xE7,0x9D,0x1E,
                        0xC9,0x35,0xD4,0x2D,0x41,0x12,0xC3,0xEA,0xE6,0x8F,0xDA,0xB6,0xAE,0xB5,0xB7,0x44,
                        0x98,0x14,0xDE,0x5B,0x8B,0xDB,0x3E,0x10,0x28,0x69,0x9F,0xEC,0x28,0x13,0x0F,0x27,
                        0x31,0xCE,0x09,0x3C,0x74,0x7A,0x08,0xAB,0x93,0x80,0xE9,0x0A,0x0C,0xA0,0xFA,0xD5,
                        0xB2,0x28,0x59,0x50,0x4F,0x6E,0x22,0xD0,0x25,0x8F,0xEF,0x1F,0x9F,0xC0,0x9B,0x06,
                        0x15,0x72,0x82,0x3F,0x5E,0xD1,0xB5,0xA9,0x5D,0x0E,0x5B,0x42,0xFC,0x05,0x8D,0x94,
                        0xD0,0x0B,0x60,0xAF,0xAD,0x34,0xCF,0xFF,0xFF,0x4D,0x9B,0x44,0x79,0xF6,0x54,0x4A,
                        0x49,0x55,0x3E,0xD4,0x6A,0x67,0xB3,0x58,0xD0,0x63,0x4D,0xB1,0xE3,0x75,0xF2,0xEE,
                        0xC7,0x4A,0xE0,0x57,0x35,0x74,0xF8,0x24,0x1F,0xD9,0x71,0x9A,0x26,0x73,0xEE,0x33,
                        0x3A,0xAD,0x56,0x63,0x5E,0x6C,0xDE,0x38,0x99,0x18,0x13,0x94,0x75,0x97,0x4A,0x7E,
                        0xC2,0x63,0x05,0x9D,0x4B,0xAA,0xFC,0x2C,0xF5,0xD2,0xCB,0x9E,0x91,0x26,0x29,0x85,
                        0x1E,0x7D,0x38,0xDD,0x26,0xAA,0x94,0x10,0x35,0x2A,0xCB,0x98,0x13,0xE9,0x62,0xAB,
                        0xB5,0xE4,0xF6,0xFB,0x37,0xA1,0xCF,0xC2,0x16,0xB2,0x60,0xA7,0xC7,0xB7,0xBE,0x01,
                        0xA8,0x97,0x46,0xD1,0x59,0xB8,0xD1,0xC8,0x1B,0x4C,0x1C,0x46,0xE5,0x2C,0xAF,0xF3,
                        0xB8,0x25,0xA9,0x43,0xB4,0x3C,0x31,0x2C,0xF8,0xAC,0x7F,0x96,0x40,0x23,0x36,0x66,
                        0xDB,0x15,0x48,0x53,0xCF,0xE3,0x15,0xDA,0x66,0x66,0x03,0x5C,0x01,0x8B,0x41,0xCB,
                        0x6B,0xF7,0xD5,0x4B,0x51,0x15,0xED,0x37,0x30,0xAC,0xE1,0xF3,0x14,0x5F,0x8B,0xA9,
                        0x63,0x4E,0x49,0xC0,0x49,0x1F,0x42,0x7E,0x6C,0xB1,0x1E,0x55,0xC3,0x6F,0xE2,0xBC,
                        0x99,0x3F,0xA2,0xA6,0xD3,0x0A,0x27,0x0A,0x2B,0x73,0xED,0x3F,0xF7,0x5A,0xD8,0xF2,
                        0xE2,0x75,0xE5,0x01,0x9F,0x26,0x78,0x90,0x9A,0xB9,0x18,0xF3,0x6F,0x11,0x9E,0xE4,
                        0x86,0x5D,0xBE,0xA0,0xC9,0x06,0x90,0xBA,0x28,},
        .derived_key_len = 345,
    },

    {   // #5 [BrainpoolP384r1]
        .name   = "BrainpoolP384r1",
        .params = {0x06, 0x09, 0x2B, 0x24, 0x03, 0x03, 0x02, 0x08, 0x01, 0x01, 0x0B},
        .params_len = 11,
        .privkeyA = {0x1E,0x20,0xF5,0xE0,0x48,0xA5,0x88,0x6F,0x1F,0x15,0x7C,0x74,0xE9,0x1B,0xDE,0x2B,
                     0x98,0xC8,0xB5,0x2D,0x58,0xE5,0x00,0x3D,0x57,0x05,0x3F,0xC4,0xB0,0xBD,0x65,0xD6,
                     0xF1,0x5E,0xB5,0xD1,0xEE,0x16,0x10,0xDF,0x87,0x07,0x95,0x14,0x36,0x27,0xD0,0x42,},
        .pubkeyA =  {0x04,0x61,0x04,0x68,0xB6,0x65,0xDD,0x91,0xC1,0x95,0x80,0x06,0x50,0xCD,0xD3,0x63,
                     0xC6,0x25,0xF4,0xE7,0x42,0xE8,0x13,0x46,0x67,0xB7,0x67,0xB1,0xB4,0x76,0x79,0x35,
                     0x88,0xF8,0x85,0xAB,0x69,0x8C,0x85,0x2D,0x4A,0x6E,0x77,0xA2,0x52,0xD6,0x38,0x0F,
                     0xCA,0xF0,0x68,0x55,0xBC,0x91,0xA3,0x9C,0x9E,0xC0,0x1D,0xEE,0x36,0x01,0x7B,0x7D,
                     0x67,0x3A,0x93,0x12,0x36,0xD2,0xF1,0xF5,0xC8,0x39,0x42,0xD0,0x49,0xE3,0xFA,0x20,
                     0x60,0x74,0x93,0xE0,0xD0,0x38,0xFF,0x2F,0xD3,0x0C,0x2A,0xB6,0x7D,0x15,0xC8,0x5F,
                     0x7F,0xAA,0x59,},
        .privkey_len = 48,
        .privkeyB = {0x03,0x26,0x40,0xBC,0x60,0x03,0xC5,0x92,0x60,0xF7,0x25,0x0C,0x3D,0xB5,0x8C,0xE6,
                     0x47,0xF9,0x8E,0x12,0x60,0xAC,0xCE,0x4A,0xCD,0xA3,0xDD,0x86,0x9F,0x74,0xE0,0x1F,
                     0x8B,0xA5,0xE0,0x32,0x43,0x09,0xDB,0x6A,0x98,0x31,0x49,0x7A,0xBA,0xC9,0x66,0x70,},
        .pubkeyB =  {0x04,0x61,0x04,0x4D,0x44,0x32,0x6F,0x26,0x9A,0x59,0x7A,0x5B,0x58,0xBB,0xA5,0x65,
                     0xDA,0x55,0x56,0xED,0x7F,0xD9,0xA8,0xA9,0xEB,0x76,0xC2,0x5F,0x46,0xDB,0x69,0xD1,
                     0x9D,0xC8,0xCE,0x6A,0xD1,0x8E,0x40,0x4B,0x15,0x73,0x8B,0x20,0x86,0xDF,0x37,0xE7,
                     0x1D,0x1E,0xB4,0x62,0xD6,0x92,0x13,0x6D,0xE5,0x6C,0xBE,0x93,0xBF,0x5F,0xA3,0x18,
                     0x8E,0xF5,0x8B,0xC8,0xA3,0xA0,0xEC,0x6C,0x1E,0x15,0x1A,0x21,0x03,0x8A,0x42,0xE9,
                     0x18,0x53,0x29,0xB5,0xB2,0x75,0x90,0x3D,0x19,0x2F,0x8D,0x4E,0x1F,0x32,0xFE,0x9C,
                     0xC7,0x8C,0x48,},
        .pubkey_len = 99,
        .kdf = CKD_SHA384_KDF,
        .shared_data = {0x70,0xE6,0xD8,0x9A,0x6D,0x75,0xD2,0x91,0xB6,0x99,0xAB},
        .shared_data_len = 11,
        .derived_key = {0xAC,0xEA,0x1C,0xE5,0xF2,0x3C,0x62,0xC9,0x68,0xF5,0x77,0x8B,0x94,0x67,0x2C,0xD8,
                        0x3D,0xE1,0x65,0xBC,0x9C,0x4D,0xB9,0xA4,0x3B,0x92,0x18,0x1A,0x0B,0x0E,0x07,0x55,
                        0x8A,0xB4,0x8C,0xF1,0xF8,0x65,0x7D,0xAF,0xF3,0x3F,0xAE,0x60,0x8C,0x4A,0xA9,0x12,
                        0xEE,0x33,0xDF,0xE2,0x15,0x20,0xDF,0x27,0xE3,0x47,0xA6,0x04,0xC3,0x06,0xCA,0x41,
                        0x6A,0x4B,0x52,0x50,0x32,0x72,0xAD,0x77,0x22,0xA9,0xB5,0xA8,0x6C,0x1F,0xA8,0x1E,
                        0xB9,0xC9,0x83,0xC4,0x77,0x7F,0x89,0xF6,0x77,},
        .derived_key_len = 89,
    },

    {   // #6 [BrainpoolP512r1]
        .name   = "BrainpoolP512r1",
        .params = {0x06, 0x09, 0x2B, 0x24, 0x03, 0x03, 0x02, 0x08, 0x01, 0x01, 0x0D},
        .params_len = 11,
        .privkeyA = {0x16,0x30,0x2F,0xF0,0xDB,0xBB,0x5A,0x8D,0x73,0x3D,0xAB,0x71,0x41,0xC1,0xB4,0x5A,
                     0xCB,0xC8,0x71,0x59,0x39,0x67,0x7F,0x6A,0x56,0x85,0x0A,0x38,0xBD,0x87,0xBD,0x59,
                     0xB0,0x9E,0x80,0x27,0x96,0x09,0xFF,0x33,0x3E,0xB9,0xD4,0xC0,0x61,0x23,0x1F,0xB2,
                     0x6F,0x92,0xEE,0xB0,0x49,0x82,0xA5,0xF1,0xD1,0x76,0x4C,0xAD,0x57,0x66,0x54,0x22,},
        .pubkeyA =  {0x04,0x81,0x81,0x04,0x0A,0x42,0x05,0x17,0xE4,0x06,0xAA,0xC0,0xAC,0xDC,0xE9,0x0F,
                     0xCD,0x71,0x48,0x77,0x18,0xD3,0xB9,0x53,0xEF,0xD7,0xFB,0xEC,0x5F,0x7F,0x27,0xE2,
                     0x8C,0x61,0x49,0x99,0x93,0x97,0xE9,0x1E,0x02,0x9E,0x06,0x45,0x7D,0xB2,0xD3,0xE6,
                     0x40,0x66,0x8B,0x39,0x2C,0x2A,0x7E,0x73,0x7A,0x7F,0x0B,0xF0,0x44,0x36,0xD1,0x16,
                     0x40,0xFD,0x09,0xFD,0x72,0xE6,0x88,0x2E,0x8D,0xB2,0x8A,0xAD,0x36,0x23,0x7C,0xD2,
                     0x5D,0x58,0x0D,0xB2,0x37,0x83,0x96,0x1C,0x8D,0xC5,0x2D,0xFA,0x2E,0xC1,0x38,0xAD,
                     0x47,0x2A,0x0F,0xCE,0xF3,0x88,0x7C,0xF6,0x2B,0x62,0x3B,0x2A,0x87,0xDE,0x5C,0x58,
                     0x83,0x01,0xEA,0x3E,0x5F,0xC2,0x69,0xB3,0x73,0xB6,0x07,0x24,0xF5,0xE8,0x2A,0x6A,
                     0xD1,0x47,0xFD,0xE7,},
        .privkey_len = 64,
        .privkeyB = {0x23,0x0E,0x18,0xE1,0xBC,0xC8,0x8A,0x36,0x2F,0xA5,0x4E,0x4E,0xA3,0x90,0x20,0x09,
                     0x29,0x2F,0x7F,0x80,0x33,0x62,0x4F,0xD4,0x71,0xB5,0xD8,0xAC,0xE4,0x9D,0x12,0xCF,
                     0xAB,0xBC,0x19,0x96,0x3D,0xAB,0x8E,0x2F,0x1E,0xBA,0x00,0xBF,0xFB,0x29,0xE4,0xD7,
                     0x2D,0x13,0xF2,0x22,0x45,0x62,0xF4,0x05,0xCB,0x80,0x50,0x36,0x66,0xB2,0x54,0x29,},
        .pubkeyB =  {0x04,0x81,0x81,0x04,0x9D,0x45,0xF6,0x6D,0xE5,0xD6,0x7E,0x2E,0x6D,0xB6,0xE9,0x3A,
                     0x59,0xCE,0x0B,0xB4,0x81,0x06,0x09,0x7F,0xF7,0x8A,0x08,0x1D,0xE7,0x81,0xCD,0xB3,
                     0x1F,0xCE,0x8C,0xCB,0xAA,0xEA,0x8D,0xD4,0x32,0x0C,0x41,0x19,0xF1,0xE9,0xCD,0x43,
                     0x7A,0x2E,0xAB,0x37,0x31,0xFA,0x96,0x68,0xAB,0x26,0x8D,0x87,0x1D,0xED,0xA5,0x5A,
                     0x54,0x73,0x19,0x9F,0x2F,0xDC,0x31,0x30,0x95,0xBC,0xDD,0x5F,0xB3,0xA9,0x16,0x36,
                     0xF0,0x7A,0x95,0x9C,0x8E,0x86,0xB5,0x63,0x6A,0x1E,0x93,0x0E,0x83,0x96,0x04,0x9C,
                     0xB4,0x81,0x96,0x1D,0x36,0x5C,0xC1,0x14,0x53,0xA0,0x6C,0x71,0x98,0x35,0x47,0x5B,
                     0x12,0xCB,0x52,0xFC,0x3C,0x38,0x3B,0xCE,0x35,0xE2,0x7E,0xF1,0x94,0x51,0x2B,0x71,
                     0x87,0x62,0x85,0xFA,},
        .pubkey_len = 132,
        .kdf = CKD_SHA1_KDF,
        .shared_data = {0x70,0xE6,0xD8,0x9A,0x6D,0x75,0xD2,0x91,0xB6,0x99,0xAB,0xAC,0xEA,0x1C,0xE5,0xF2,
                        0x3C,0x62,0xDD},
        .shared_data_len = 19,
        .derived_key = {0xE4,0x3B,0xDD,0xDC,0x03,0x13,0xBA,0xB8,0x8C,0x25,0x22,0x73,0xA5,0x14,0x29,0xC9,
                        0xB2,0x9F,0x41,0x05,0x78,0x1B,0x74,0x45,0xC8,0xE6,0x1F,0xC7,0x09,0x0A,0xAA,0x44,
                        0xEA,0x99,0x1D,0x32,0xC2,0x7A,0x70,0x75,0xEB,0x3D,0xFC,0xF0,0xF4,0x03,0xA8,0x2C,
                        0x93,0x4B,0x0B,0x73,0x60,0x51,0x96,0xA2,0xA2,0x14,0x71,0x74,0x9F,0xA2,0xAD,0x86,
                        0x49,0x85,0xB0,0x05,0xBA,0x23,0x8F,0x59,0x07,0xB1,0xA5,0x46,0xBF,0x59,0xB6,0xE1,
                        0x1E,},
        .derived_key_len = 81,
    },
};

#define ECDH_TV_NUM sizeof(ecdh_tv)/sizeof(struct ECDH_TEST_VECTOR)
