# API style guidelines

Generally follow guidance at https://cloud.google.com/apis/design/, in
particular for proto3 as described at:

* https://cloud.google.com/apis/design/proto3
* https://cloud.google.com/apis/design/naming_convention
* https://developers.google.com/protocol-buffers/docs/style

A key aspect of our API style is maintaining stability by following the [API versioning
guidelines](API_VERSIONING.md). All developers must familiarize themselves with these guidelines,
any PR which makes breaking changes to the API will not be merged.

In addition, the following conventions should be followed:

* Every proto directory should have a `README.md` describing its content. See
  for example [envoy.service](envoy/service/README.md).

* The data plane APIs are primarily intended for machine generation and consumption.
  It is expected that the management server is responsible for mapping higher
  level configuration concepts to concrete API concepts. Similarly, static configuration
  fragments may be generated by tools and UIs, etc. The APIs and tools used
  to generate xDS configuration are beyond the scope of the definitions in this
  repository.

* Use [wrapped scalar
  types](https://github.com/google/protobuf/blob/master/src/google/protobuf/wrappers.proto)
  where there is a real need for the field to have a default value that does not
  match the proto3 defaults (0/false/""). This should not be done for fields
  where the proto3 defaults make sense. All things being equal, pick appropriate
  logic, e.g. enable vs. disable for a `bool` field, such that the proto3
  defaults work, but only where this doesn't result in API gymnastics.

* Use a `[#not-implemented-hide:]` `protodoc` annotation in comments for fields that lack Envoy
  implementation. These indicate that the entity is not implemented in Envoy and the entity
  should be hidden from the Envoy documentation.

* Always use plural field names for `repeated` fields, such as `filters`.

* Due to the fact that we consider JSON/YAML to be first class inputs, we cannot easily change a
  a singular field to a repeated field (both due to JSON/YAML array structural differences as well
  as singular vs. plural field naming). If there is a reasonable expectation that a field may need
  to be repeated in the future, but we don't need it to be repeated right away, consider making it
  repeated now but using constraints to enforce a maximum repeated size of 1. E.g.:

  ```proto
  repeated OutputSink sinks = 1 [(validate.rules).repeated = {min_items: 1, max_items: 1}];
  ```

* Always use upper camel case names for message types and enum types without embedded
  acronyms, such as `HttpRequest`.

* Prefer `oneof` selections to boolean overloads of fields, for example, prefer:

  ```proto
  oneof path_specifier {
    string simple_path = 1;
    string regex_path = 2;
  }
  ```

  to

  ```proto
  string path = 1;
  bool path_is_regex = 2;
  ```

  This is more efficient, extendable and self-describing.

* The API includes two types for representing [percents](envoy/type/percent.proto). `Percent` is
  effectively a double value in the range 0.0-100.0. `FractionalPercent` is an integral fraction
  that can be used to create a truncated percentage also in the range 0.0-100.0. In high performance
  paths, `FractionalPercent` is preferred as randomness calculations can be performed using integral
  modulo and comparison operations only without any floating point conversions. Typically, most
  users do not need infinite precision in these paths.

* For enum types, if one of the enum values is used for most cases, make it the
  first enum value with `0` numeric value. Otherwise, define the first enum
  value like `TYPE_NAME_UNSPECIFIED = 0`, and treat it as an error. This design
  pattern forces developers to explicitly choose the correct enum value for
  their use case, and avoid misunderstanding of the default behavior.

* Proto fields should be sorted logically, not by field number. For large protos, place a comment
  at the top that specifies the next free field number. E.g.,

  ```
  // [#comment:next free field: 28]
  ```

## Package organization

API definitions are layered hierarchically in packages from top-to-bottom:

- `envoy.service` contains gRPC definitions of supporting services;
- `envoy.config` contains definitions for service configuration, filter
configuration, and bootstrap;
- `envoy.api.v2` contains definitions for EDS, CDS, RDS, LDS, and top-level
resources such as `Cluster`;
- `envoy.api.v2.endpoint`, `envoy.api.v2.cluster`, `envoy.api.v2.route`,
`envoy.api.v2.listener`, `envoy.api.v2.ratelimit` define sub-messages of the top-level resources;
- `envoy.api.v2.core` and `envoy.api.v2.auth` hold core definitions consumed
throughout the API.

Dependencies are enforced from top-to-bottom using visibility constraints in
the build system to prevent circular dependency formation. Package group
`//envoy/api/v2:friends` selects consumers of the core API package (services and configs)
and is the default visibility for the core API packages. The default visibility
for services and configs should be `//docs` (proto documentation tool).

Extensions should use the regular hierarchy. For example, configuration for network filters belongs
in a package under `envoy.config.filter.network`.
