/** BEGIN COPYRIGHT BLOCK
 * This Program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; version 2 of the License.
 *
 * This Program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this Program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Copyright (C) 2005 Red Hat, Inc.
 * All rights reserved.
 * END COPYRIGHT BLOCK **/

#define FORCE_PR_LOG 1
#define nsnull NULL
#define NS_OK 0
#define NS_ERROR_FAILURE -1
#include <stdio.h>
#include "rhCoolKey.h"
#include "CoolKey.h"
#include "prlink.h"
#include "prthread.h"
#include <plstr.h>
#include <vector>
#include <string>
#include <time.h>

#define STRINGIFY(x) #x
#define GETSTRING(x) STRINGIFY(x)


#include <prlog.h>
#define COOL_MAX_PATH 1024
#define MAX_STR_LEN COOL_MAX_PATH

std::list<CoolKeyNode*>rhCoolKey::gASCAvailableKeys;

//std::list< nsCOMPtr <rhIKeyNotify>  > rhCoolKey::gNotifyListeners;

PRLock* rhCoolKey::certCBLock=NULL;
PRLock* rhCoolKey::eventLock= NULL;

PRBool rhCoolKey::gAutoEnrollBlankTokens = PR_FALSE; 

static PRLogModuleInfo *coolKeyLog = PR_NewLogModule("coolKey");

static rhCoolKey *single = NULL;


class CoolKeyShutdownObserver
 {
 public:

  void Observe(const char *aSubject,
                                 const char *aTopic,
                                 const char *someData);
  private:
   ~CoolKeyShutdownObserver(); 
 };

 CoolKeyShutdownObserver::~CoolKeyShutdownObserver()
 {
    char tBuff[56];
    PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s CoolKeyShutdownObserver::~CoolKeyShutdownObserver \n",GetTStamp(tBuff,56)));
 }


void CoolKeyShutdownObserver::Observe(const char *aSubject,
                                 const char *aTopic,
                                 const char *someData)
 {
     char tBuff[56];
     PR_LOG(coolKeyLog, PR_LOG_DEBUG, ("%s CoolKeyShutdownObserver::Observe shutting down",GetTStamp(tBuff,56)));
     if(single)
     {
          single->ShutDownInstance();
     }

 }

unsigned int
ASCCalcBase64DecodedLength(const char *aBase64Str)
{
  // The Base64 string might contain whitespace
  // formatting, so we need to scan the string and
  // count the non whitespace characters.

  unsigned int numValidChars = 0;
  unsigned int numEqualSigns = 0;

  const char *c = aBase64Str;

  while (c && *c) {
    if (!isspace(*c)) {
      if (*c == '=')
        numEqualSigns++;
      numValidChars++;
    }
    c++;
  }

  // The number of Base64 characters we have should
  // always be a divisible by 4, but we also need to
  // subtract off the '=' padding characters from our
  // final calculation.

  return ((numValidChars / 4) * 3) - numEqualSigns;
}
class CoolKeyResultTask 
{
public:
  ~CoolKeyResultTask() {

      char tBuff[56];
      PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s ~CoolKeyResultTask thread: %p \n",GetTStamp(tBuff,56),PR_GetCurrentThread()));

      if (mKeyID) {
          free(mKeyID);
      }

      if (mStrData) {
          free(mStrData);
      }

  }

  CoolKeyResultTask(unsigned long keyType, const char *keyID, unsigned long keyState,
                    unsigned long data, const char *strData, CoolKeyListener *listener)
    : mKeyType(keyType), mKeyState(keyState), 
      mData(data), mListener(listener)
  {

     mStrData = NULL;
     mKeyID   = NULL;

     
     if (keyID) {
         mKeyID = strdup(keyID);
     }

     if (strData) {
         mStrData = strdup(strData);
     }

     char tBuff[56];

     PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s CoolKeyResultTask::CoolKeyResultTask thread:   %p keyID %s \n",GetTStamp(tBuff,56),PR_GetCurrentThread(),mKeyID));

  }


 void  Run(void) {

     char tBuff[56];
     PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s CoolKeyResultTask::Run thread: %p \n",GetTStamp(tBuff,56),PR_GetCurrentThread()));
   
   }

private: 
   unsigned long mKeyType;
   char *mKeyID;
   unsigned long mKeyState;
   unsigned long mData;
   char *mStrData; 
   CoolKeyListener *mListener;
};

unsigned int
ASCCalcBase64EncodedLength(unsigned int aDataLength)
{
  // The Base64 data we generate will always be a
  // multiple of 4 in length since it includes no
  // whitespace!

  return ((aDataLength + 2) / 3) * 4;
}

rhCoolKey::rhCoolKey(const char* dbDir)
{
    char tBuff[56];
    PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s rhCoolKey::rhCoolKey: %p \n",GetTStamp(tBuff,56),this));

    if(!single)
    {
        single = this;

    }
    else
    {
        return;
    }

    certCBLock = PR_NewLock();

    if(!certCBLock) {
       PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s Failed to create lock exiting! \n",GetTStamp(tBuff,56)));
        exit(1);
    }

    eventLock = PR_NewLock();

    if(!eventLock) {
       PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s Failed to create event lock exiting! \n", GetTStamp(tBuff,56)));
       exit(1);
    }


    PRBool res = InitInstance(dbDir);

    if(res == PR_FALSE)
    {
        PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s ESC InitInstance failed,exiting. CoolKey instance %p\n",GetTStamp(tBuff,56),single));
        exit(1);
     }

  /* member initializers and constructor code */ 
}

rhCoolKey::~rhCoolKey()
{
   /* destructor code */

    char tBuff[56];
    PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s rhCoolKey::~rhCoolKey: %p \n",GetTStamp(tBuff,56),this));

    if(certCBLock) {
        PR_DestroyLock(certCBLock);
    }

    if(eventLock) {
        PR_DestroyLock(eventLock);
    }
}

void rhCoolKey::ShutDownInstance()
{
    char tBuff[56];
    PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s rhCoolKey::ShutDownInstance. %p \n",GetTStamp(tBuff,56),this));    

    //ClearNotifyKeyList();

    CoolKeyShutdown();
}

HRESULT rhCoolKey::Dispatch( CoolKeyListener *listener,
    unsigned long keyType, const char *keyID, unsigned long keyState,
    unsigned long data, const char *strData)
{
    char tBuff[56];

    PR_Lock(eventLock);
    PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s rhCoolKey::Dispatch: thead:  %p \n",GetTStamp(tBuff,56),PR_GetCurrentThread()));

    if(listener) {
        listener->notify(listener,keyType,keyID, keyState, data, strData);
    }

    PR_Unlock(eventLock);
    return  1;  
}

HRESULT rhCoolKey::Reference( CoolKeyListener *listener )
{
    return S_OK;
}

HRESULT rhCoolKey::Release( CoolKeyListener *listener )
{
    return S_OK;
}

struct BadCertData {
     PRErrorCode error; 
     PRInt32 port;
};  

typedef struct BadCertData BadCertData;

SECStatus rhCoolKey::badCertHandler(void *arg, PRFileDesc *fd)
{
    SECStatus    secStatus = SECFailure;
    PRErrorCode    err;
    char *host = NULL;
    PRInt32 port = 0;
    CERTCertificate *serverCert = NULL;
    char tBuff[56];
    
    PR_Lock(certCBLock);

    if (!arg || !fd) {
        PR_Unlock(certCBLock);
        return secStatus;
    }

    // Retrieve callback data from NssHttpClient
    // Caller cleans up this data
    BadCertData *data = (BadCertData *) arg;
    data->error = err = PORT_GetError();


    /* If any of the cases in the switch are met, then we will proceed   */

    switch (err) {
    case SEC_ERROR_INVALID_AVA:
    case SEC_ERROR_INVALID_TIME:
    case SEC_ERROR_BAD_SIGNATURE:
    case SEC_ERROR_EXPIRED_CERTIFICATE:
    case SEC_ERROR_UNKNOWN_ISSUER:
    case SEC_ERROR_UNTRUSTED_CERT:
    case SEC_ERROR_CERT_VALID:
    case SEC_ERROR_EXPIRED_ISSUER_CERTIFICATE:
    case SEC_ERROR_CRL_EXPIRED:
    case SEC_ERROR_CRL_BAD_SIGNATURE:
    case SEC_ERROR_EXTENSION_VALUE_INVALID:
    case SEC_ERROR_CA_CERT_INVALID:
    case SEC_ERROR_CERT_USAGES_INVALID:
    case SEC_ERROR_UNKNOWN_CRITICAL_EXTENSION:
    case SEC_ERROR_EXTENSION_NOT_FOUND: // Added by Rob 5/21/2002
        secStatus = SECSuccess;
    break;
    default:
        secStatus = SECFailure;
    break;
    }

    if(secStatus == SECSuccess)  {
        PR_Unlock(certCBLock);
        return secStatus;
    }

    // Now proceed to see if we have an exception.
    // Get the server certificate that was rejected.
    serverCert = SSL_PeerCertificate(fd);

    if(!serverCert) {
        PR_Unlock(certCBLock);
        return secStatus;
    }

    port = data->port;
    host = SSL_RevealURL(fd);

    if(!host || port <= 0) {
        PR_Unlock(certCBLock);
        return secStatus;
    }

    PR_LOG(coolKeyLog, PR_LOG_DEBUG,
                          ("%s rhCoolKey::badCertHandler enter: error: %d  url: %s port: %d \n",
                          GetTStamp(tBuff,56),err,host,port)
    );

    PR_Free(host);
    host = NULL;

    PR_Unlock(certCBLock);

    return secStatus;
}


HRESULT rhCoolKey::doSetCoolKeyConfigValue(const char *aName, const char *aValue) 
{

    if(!aName || !aValue)
    {
        return E_FAIL;
    }


    return S_OK;

}          

const char *rhCoolKey::doGetCoolKeyConfigValue(const char *aName )
{
    const char* prefValue = NULL;
    if(!aName)
    {
        return NULL;
    }

    return (const char *) prefValue;
}

void rhCoolKey::RegisterCoolKeyListener(CoolKeyListener *listener) {
    if(listener) {
        CoolKeyRegisterListener(listener);
    }
}

PRBool rhCoolKey::InitInstance(const char* dbDir)
{
    char tBuff[56];
    PRBool ret = PR_TRUE;

    PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s rhCoolKey::InitInstance %p \n",GetTStamp(tBuff,56),this));

    CoolKeySetCallbacks(Dispatch,Reference, Release,doGetCoolKeyConfigValue ,doSetCoolKeyConfigValue,badCertHandler);

    //CoolKeyRegisterListener();

    // Now setup CoolKey.

    CoolKeyInit(dbDir);

    return ret;
}

CoolKeyNode* rhCoolKey::GetCoolKeyInfo(unsigned long aKeyType, const char * aKeyID)
{
    char tBuff[56];
    PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s rhCoolKey::GetCoolKeyInfo: gASCAvailableKeys %p looking for key %s type %d \n",GetTStamp(tBuff,56),&gASCAvailableKeys,aKeyID,aKeyType));

    std::list<CoolKeyNode*>::const_iterator it;
    for(it=gASCAvailableKeys.begin(); it!=gASCAvailableKeys.end(); ++it) {

        CoolKeyNode *node = (CoolKeyNode *) (*it);

        PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s rhCoolKey::GetCoolKeyInfo: current key %s type %d, looking for key %s type %d \n",GetTStamp(tBuff,56),node->mKeyID,node->mKeyType,aKeyID,aKeyType));

        if (node->mKeyType == aKeyType && !strcmp(node->mKeyID, aKeyID))
            return *it;
    }

    return 0;
}

// Internal methods

PRBool rhCoolKey::ASCCoolKeyIsAvailable(unsigned long aKeyType, char * aKeyID)
{
    char tBuff[56];
    PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s rhCoolKey::ASCCoolKeyIsAvailable type %d id %s \n",GetTStamp(tBuff,56),aKeyType,aKeyID));
    return GetCoolKeyInfo(aKeyType, aKeyID) ? PR_TRUE : PR_FALSE;
}

HRESULT  rhCoolKey::ASCGetAvailableCoolKeyAt(unsigned long aIndex,
                                           unsigned long *aKeyType,
                                           char **aKeyID)
{
    char tBuff[56]; 
    PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s rhCoolKey::ASCGetAvailableCoolKeyAt: index %d type %d id %s \n",GetTStamp(tBuff,56),aIndex,aKeyType,aKeyID)); 
    if (!aKeyType || !aKeyID)
        return E_FAIL;

    *aKeyType = 0;
    *aKeyID = NULL;

    if (!gASCAvailableKeys.empty() && 
		aIndex < (unsigned long) ASCGetNumAvailableCoolKeys()) {
        std::list<CoolKeyNode*>::const_iterator it;
        for(it=gASCAvailableKeys.begin(); it!=gASCAvailableKeys.end(); ++it) {
          if (aIndex-- == 0) {
              *aKeyType = (*it)->mKeyType;
              *aKeyID = (char *) (*it)->mKeyID;
               return S_OK;
          }
        }
    }

  return E_FAIL;

}

int   rhCoolKey::ASCGetNumAvailableCoolKeys(void)
{
    char tBuff[56];
    int size = (int) gASCAvailableKeys.size();
    PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s rhCoolKey::ASCGetNumAvailableCoolKeys %d \n",GetTStamp(tBuff,56),size));
    return size;

}

void rhCoolKey::InsertKeyIntoAvailableList(unsigned long aKeyType, const char * aKeyID,CoolKeyStatus aStatus)
{
    char tBuff[56];
    PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s rhCoolKey::InsertKeyIntoAvailableList: \n",GetTStamp(tBuff,56)));
    if (ASCCoolKeyIsAvailable(aKeyType, (char *)aKeyID))
    {
         PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s rhCoolKey::InsertKeyIntoAvailableList: Key Not Available \n",GetTStamp(tBuff,56)));

        return;
    }

    CoolKeyNode *node = new CoolKeyNode(aKeyType, aKeyID, aStatus);

    if (!node)
    {
        PR_LOG( coolKeyLog, PR_LOG_ERROR, ("%s Can't create new  CoolKey Data Structure. \n",GetTStamp(tBuff,56)));
        return;
    }

    gASCAvailableKeys.push_back(node);

}

void rhCoolKey::RemoveKeyFromAvailableList(unsigned long aKeyType, const char * aKeyID)
{
    char tBuff[56];
    PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s rhCoolKey::RemoveKeyFromAvailableList type %d id %s \n",GetTStamp(tBuff,56),aKeyType,aKeyID));
    CoolKeyNode *node = GetCoolKeyInfo(aKeyType, aKeyID);

    if (!node)
        return;

    gASCAvailableKeys.remove(node);
    delete node;
}

void rhCoolKey::ClearAvailableList()
{
    char tBuff[56];
    PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s rhCoolKey::ClearAvailableList \n",GetTStamp(tBuff,56)));
    while (gASCAvailableKeys.size() > 0) {
        CoolKeyNode *node = gASCAvailableKeys.front();
        delete node;
        gASCAvailableKeys.pop_front();
    }
}

HRESULT rhCoolKey::ASCSetCoolKeyPin(unsigned long aKeyType, const char * aKeyID, const char * aPin)
{
    char tBuff[56];
    PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s rhCoolKey::ASCSetCoolKeyPin type %d id %s pin %s \n",GetTStamp(tBuff,56),aKeyType,aKeyID,aPin));
    CoolKeyNode *node = GetCoolKeyInfo(aKeyType, aKeyID);
    if (!node)
        return E_FAIL;

    node->mPin = aPin;
    return S_OK;
}

// Interface method implementations

HRESULT rhCoolKey::RhNotifyKeyStateChange(PRUint32 aKeyType,const char *aKeyID, PRUint32 aKeyState, PRUint32 aData,const char* strData)
{
    char tBuff[56];
    PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s rhCoolKey::RhNotifyKeyStateChange: id: %s type: %d state %d data: %d \n",GetTStamp(tBuff,56),aKeyID,aKeyType, aKeyState,aData));

    CoolKeyNode tempKey(aKeyType, aKeyID,(CoolKeyStatus) aKeyState);
    CoolKeyNode *node = NULL;

    AutoCoolKey key(aKeyType, aKeyID);

    switch (aKeyState)
    {
        case eCKState_KeyInserted:
        {
         // The assumption we're making here is that we
         // will never get notified for a non CoolKey.

             CoolKeyStatus keyStatus = eAKS_AppletNotFound;
            if (CoolKeyIsEnrolled(&key))
                keyStatus = eAKS_Available;       else if (CoolKeyHasApplet(&key))                keyStatus = eAKS_Uninitialized;

             PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s Key Inserted. ID %s \n",GetTStamp(tBuff,56),aKeyID));
             InsertKeyIntoAvailableList(tempKey.mKeyType,aKeyID ,(CoolKeyStatus) keyStatus);
          break;
        }
        case eCKState_KeyRemoved:
            PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s Key Removed. ID %s \n",GetTStamp(tBuff,56),aKeyID));
            RemoveKeyFromAvailableList(tempKey.mKeyType, aKeyID);
        break;
        case eCKState_EnrollmentComplete:
        case eCKState_EnrollmentError:
        case eCKState_PINResetComplete:
        case eCKState_PINResetError:
        case eCKState_FormatComplete:
        case eCKState_FormatError:
        case eCKState_BlinkComplete:
        case eCKState_BlinkError:
        case eCKState_OperationCancelled:
        {
          node = GetCoolKeyInfo(tempKey.mKeyType, aKeyID);

          if (node) {
            node->mStatus = eAKS_AppletNotFound;
            if (CoolKeyIsEnrolled(&key))
              node->mStatus = eAKS_Available;
            else if (CoolKeyHasApplet(&key))
              node->mStatus = eAKS_Uninitialized;
           }
          break;
         }
        case eCKState_EnrollmentStart:
        case eCKState_PINResetStart:
        case eCKState_FormatStart:
        case eCKState_BlinkStart:
        default:
          break;
      };

      //Now notify all the listeners of the event


     return NS_OK;
}

/* HRESULT rhCoolKey::RhCoolKeyUnSetNotifyCallback(rhIKeyNotify *jsNotify)
{
    char tBuff[56];
    PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s rhCoolKey::RhCoolKeyUnSetNotifyCallback Object: input %p  this %p \n",GetTStamp(tBuff,56),jsNotify,this));

    RemoveNotifyKeyListener(jsNotify);

    PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s rhCoolKey::RhCoolKeyUnSetNotifyCallback Object: removed listener, size now %d \n",GetTStamp(tBuff,56),GetNotifyKeyListenerListSize()));

   if(GetNotifyKeyListenerListSize() == 0)
   {
       PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s rhCoolKey::RhCoolKeyUnSetNotifyCallback Object: input %p  this %p Listener size 0. \n",GetTStamp(tBuff,56),jsNotify,this));
   }

   return NS_OK;
 }

HRESULT rhCoolKey::RhCoolKeySetNotifyCallback(rhIKeyNotify *jsNotify)
{
    char tBuff[56];
    PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s rhCoolKey::RhCoolKeySetNotifyCallback Object: %p this %p\n",GetTStamp(tBuff,56),jsNotify,this));

    AddNotifyKeyListener(jsNotify);    

    return NS_OK;

}
 */
/* void CoolKeyInitializeLog (in string aPathName, in unsigned long aMaxLines); */
HRESULT rhCoolKey::CoolKeyInitializeLog(const char *aPathName, PRUint32 aMaxLines)
{
    ::CoolKeyInitializeLog((char *)aPathName, aMaxLines);  

    return NS_OK;
}

/* void CoolKeyLogMsg (in unsigned long aLogLevel, in string aMessage); */
HRESULT rhCoolKey::CoolKeyLogMsg(unsigned int aLogLevel, const char *aMessage)
{
    char tBuff[56];

    if(aMessage )
    {
        ///::CoolKeyLogMsg(Level, "%s %s \n",GetTStamp(tBuff,56),aMessage);
        PR_LOG( coolKeyLog, PR_LOG_ERROR, ("%s %s",GetTStamp(tBuff,56),aMessage));
    }

    return NS_OK;
}

HRESULT rhCoolKey::BlinkCoolKey(PRUint32 aKeyType, const char *aKeyID, PRUint32 aRate, PRUint32 aDuration)
{
    char tBuff[56];
    PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s rhCoolKey::RhBlinkCoolKey thread: %p \n",GetTStamp(tBuff,56),PR_GetCurrentThread()));

   CoolKeyNode *node = GetCoolKeyInfo(aKeyType, aKeyID);

  if (!node)
  {
      PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s rhCoolKey::RhBlinkCoolKey: GetCoolKeyInfo failed. \n",GetTStamp(tBuff,56)));
      return NS_ERROR_FAILURE;
  }

  if (!aKeyID)
  {
    return NS_ERROR_FAILURE ;
  }

  AutoCoolKey key(aKeyType, aKeyID);
  HRESULT hres = CoolKeyBlinkToken(&key, aRate, aDuration);

  if (hres == S_OK)
  {
    node->mStatus = eAKS_BlinkInProgress;
    return NS_OK;
  }

  return NS_ERROR_FAILURE;

}

/* void rhEnrollCoolKey (in unsigned long aKeyType, in string aKeyID, in string aEnrollmentType, in string aScreenName, in string aPin, in string aScreenNamePWord, in string aTokenCode); */

HRESULT rhCoolKey::EnrollCoolKey(PRUint32 aKeyType, const char *aKeyID, const char *aEnrollmentType, const char *aScreenName, const char *aPin, const char *aScreenNamePWord, const char *aTokenCode)
{

    char tBuff[56];
    ::CoolKeyLogMsg(1, "%s Attempting to Enroll Key ,ID: %s \n",GetTStamp(tBuff,56),aKeyID);
    
    CoolKeyNode *node = GetCoolKeyInfo(aKeyType, aKeyID);

    if (!node)
        return NS_ERROR_FAILURE;

  // Prevent multiple windows from trying to enroll the same
  // key at the same time. If the key is already in the process
  // of being enrolled, just return S_OK.

    if (node->mStatus == eAKS_EnrollmentInProgress)
        return NS_OK;

    AutoCoolKey key(aKeyType, aKeyID);
    HRESULT hres = CoolKeyEnrollToken(&key, aEnrollmentType, aScreenName, aPin,aScreenNamePWord,aTokenCode);

    if (hres == S_OK)
    {
        node->mStatus = eAKS_EnrollmentInProgress;
        return NS_OK;
    }

    return NS_OK;

}

/* void rhResetCoolKeyPIN (in unsigned long aKeyType, in string aKeyID, in string aScreenName, in string aPIN, in string aScreenNamePwd); */

HRESULT rhCoolKey::ResetCoolKeyPIN(PRUint32 aKeyType, const char *aKeyID, const char *aScreenName, const char *aPIN, const char *aScreenNamePwd)
{
    char tBuff[56];
    ::CoolKeyLogMsg( 1, "%s Attempting to Reset Key Password, ID: %s \n",GetTStamp(tBuff,56),aKeyID);
    CoolKeyNode *node = GetCoolKeyInfo(aKeyType, aKeyID);

    if (!node)
    {
        PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s rhCoolKey::RhResetCoolKeyPIN no node: thread: %p \n",GetTStamp(tBuff,56),PR_GetCurrentThread()));
        return NS_ERROR_FAILURE;
    }

  // Prevent multiple windows from trying to reset the pin for the same
  // key at the same time. If the key is already in the process
  // of being reset, just return S_OK.

    if (node->mStatus == eAKS_PINResetInProgress)
        return NS_OK;

  // Key must be available, or we throw an error!

    if (node->mStatus != eAKS_Available)
    {
        PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s rhCoolKey::RhResetCoolKeyPIN thread: token unavailable %p \n",GetTStamp(tBuff,56),PR_GetCurrentThread()));
        return NS_ERROR_FAILURE;
    }

    AutoCoolKey key(aKeyType, aKeyID);

    HRESULT hres = CoolKeyResetTokenPIN(&key, aScreenName, aPIN,aScreenNamePwd);

    if (hres == S_OK) 
    {
        node->mStatus = eAKS_PINResetInProgress;
        return NS_OK;
    }

    return NS_OK;
}

/* void rhFormatCoolKey (in unsigned long aKeyType, in string aKeyID, in string aEnrollmentType, in string aScreenName, in string aPIN, in string aScreenNamePWord, in string aTokenCode); */

HRESULT rhCoolKey::FormatCoolKey(PRUint32 aKeyType, const char *aKeyID, const char *aEnrollmentType, const char *aScreenName, const char *aPIN, const char *aScreenNamePWord, const char *aTokenCode)
{
    char tBuff[56];
    ::CoolKeyLogMsg( 1, "%s Attempting to Format Key, ID: %s. ",GetTStamp(tBuff,56),aKeyID);
    CoolKeyNode *node = GetCoolKeyInfo(aKeyType, aKeyID);

    if (!node)
        return NS_ERROR_FAILURE;

  // Prevent multiple windows from trying to format the same
  // key at the same time. If the key is already in the process
  // of being formatted, just return S_OK.

    if (node->mStatus == eAKS_FormatInProgress)
        return NS_OK;

  // Throw an error if the key is already busy!
    if (node->mStatus != eAKS_AppletNotFound &&
        node->mStatus != eAKS_Uninitialized &&
        node->mStatus != eAKS_Available)
    {
        return NS_ERROR_FAILURE;
    }

  // Prevent multiple windows from trying to enroll the same
  // key at the same time. If the key is already in the process
  // of being enrolled, just return S_OK.

    AutoCoolKey key(aKeyType, aKeyID);

    HRESULT hres = CoolKeyFormatToken(&key, aEnrollmentType, aScreenName,aPIN,aScreenNamePWord,aTokenCode);

    if (hres == S_OK) 
    {
        node->mStatus = eAKS_FormatInProgress;
        return NS_OK;
    }

    return NS_OK;
}

/* void rhCancelCoolKeyOperation (in unsigned long aKeyType, in string aKeyID); */

HRESULT rhCoolKey::CancelCoolKeyOperation(PRUint32 aKeyType, const char *aKeyID)
{
    char tBuff[56];
    CoolKeyNode *node = GetCoolKeyInfo(aKeyType, aKeyID);

    if (!node)
        return NS_ERROR_FAILURE;

    PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s rhCoolKey::RhCancelCoolKeyOperation type %d id %s status %d: \n",GetTStamp(tBuff,56),aKeyType,aKeyID,node->mStatus));

  // If the key isn't busy, then there's nothing to do.

    if (node->mStatus != eAKS_EnrollmentInProgress &&
      node->mStatus != eAKS_UnblockInProgress &&
      node->mStatus != eAKS_PINResetInProgress &&
      node->mStatus != eAKS_RenewInProgress &&
      node->mStatus != eAKS_FormatInProgress) 
          return NS_OK;

    AutoCoolKey key(aKeyType, aKeyID);
    HRESULT hres = CoolKeyCancelTokenOperation(&key);

    if(hres == S_OK)
    {
        return NS_OK;
    }

    return NS_ERROR_FAILURE;
}

/* void GetCoolKeyCertNicknames (in unsigned long aKeyType, in string aKeyID, out PRUint32 count, [array, size_is (count), retval] out string str); */
HRESULT rhCoolKey::GetCoolKeyCertNicknames(PRUint32 aKeyType, const char *aKeyID, PRUint32 *count, char ***str)
{
    char tBuff[56];
    if(!aKeyID || !count)
    {
        return NS_ERROR_FAILURE;
    }

    vector<string>  nicknames;
    AutoCoolKey key(aKeyType, aKeyID);

    HRESULT res =  CoolKeyGetCertNicknames( &key,nicknames);
  
    if(res != S_OK)
    {
        return NS_OK;    
    }

    char **array = NULL;
    int num = nicknames.size();

    array = (char **) PR_Malloc((sizeof(char *) *num));

    if(!array)
    {
        return NS_ERROR_FAILURE;
    }
 
    vector<string>::iterator i; 
    int j = 0;
    for(i = nicknames.begin(); i!= nicknames.end(); i++)
    {
        char *tName = (char *) (*i).c_str();

        PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s rhCoolKey::GetCoolKeyCertNicknames  name %s  \n",GetTStamp(tBuff,56),tName));

        array[j] = NULL;
        if(tName)
        {
            array[j] =(char *) PL_strdup(tName);
        }

        j++;
    }

    *count = num;
    *str = array;
    
    return NS_OK;
}


/* void rhGetAvailableCoolKeys (out PRUint32 count, [array, size_is (count), retval] out string str); */

HRESULT rhCoolKey::GetAvailableCoolKeys(PRUint32 *count, char ***str)
{
    char tBuff[56];
    PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s rhCoolKey::RhGetAvailableCoolKeys thread: %p \n",GetTStamp(tBuff,56),PR_GetCurrentThread()));

    if(!count || !str)
    {
        return NS_ERROR_FAILURE;
    }
 
    char **array = NULL;

    long numKeys = ASCGetNumAvailableCoolKeys();

    PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s Attempting to get number of keys. Value:  %d \n",GetTStamp(tBuff,56),numKeys));

    if(numKeys == 0)
    {
        return NS_OK;
    }

    array = (char **) PR_Malloc((sizeof(char *) * numKeys));

    if(!array)
    {
        return NS_ERROR_FAILURE;
    }

    int i;

    for (i = 0; i < numKeys; i++) {

        unsigned long keyType;
        char* keyID;

        ASCGetAvailableCoolKeyAt(i, &keyType, &keyID);

        const char *id = keyID;

        array[i] = NULL;

        if(id)
        {
            array[i] =(char *) PL_strdup(id);

            if(!array[i])
            {
                return NS_ERROR_FAILURE;
            }

        }
    }

    *count = numKeys;
    *str = array;
    return NS_OK;
}

/* unsigned long rhGetCoolKeyStatus (in unsigned long aKeyType, in string aKeyID); */

HRESULT rhCoolKey::GetCoolKeyStatus(PRUint32 aKeyType, const char *aKeyID, PRUint32 *_retval)
{
    char tBuff[56]; 
    PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s rhCoolKey::RhGetCoolKeyStatus thread: %p \n",GetTStamp(tBuff,56),PR_GetCurrentThread()));
    CoolKeyNode *node = GetCoolKeyInfo(aKeyType, aKeyID);

    if(node)
    {
        *_retval = node->mStatus;
    }
    else
    {
        *_retval = eAKS_Unavailable;
    }
    
    PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s rhCoolKey::RhGetCoolKeyStatus retval: %d \n",GetTStamp(tBuff,56),*_retval));

    return NS_OK;
}


/* boolean GetCoolKeyIsReallyCoolKey (in unsigned long aKeyType, in string aKeyID); */


HRESULT rhCoolKey::GetCoolKeyIsReallyCoolKey(PRUint32 aKeyType, const char *aKeyID, bool *_retval)
{
    char tBuff[56];
    PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s rhCoolKey::GetCoolKeyIsReallyCoolKey thread: %p \n",GetTStamp(tBuff,56),PR_GetCurrentThread()));

    if (aKeyType && aKeyID && ASCCoolKeyIsAvailable(aKeyType, (char *) aKeyID)) {
        if (aKeyID) {
            AutoCoolKey key(aKeyType, aKeyID);
            PRBool isCool = CoolKeyIsReallyCoolKey(&key);
            PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s rhCoolKey::GetCoolKeyIsReallyCoolKey isCool: %d \n",GetTStamp(tBuff,56),(int) isCool));
            *_retval= isCool;
            return NS_OK;
        }
    }
    *_retval = PR_FALSE;
    return NS_OK;
}

/* long GetCoolKeyGetAppletVer (in unsigned long aKeyType, in string aKeyID, in boolean aIsMajor); */
HRESULT rhCoolKey::GetCoolKeyGetAppletVer(PRUint32 aKeyType, const char *aKeyID, bool aIsMajor, PRInt32 *_retval)
{
    char tBuff[56];
    PR_LOG(coolKeyLog, PR_LOG_DEBUG, ("%s rhCoolKey::GetCoolKeyAppletVer thread: %p \n",GetTStamp(tBuff,56),PR_GetCurrentThread()));

    AutoCoolKey key(aKeyType, aKeyID);

    int ver = CoolKeyGetAppletVer(&key, aIsMajor);

    *_retval = ver;

    return NS_OK;
}

/* boolean rhCoolKeyIsEnrolled (in unsigned long aKeyType, in string aKeyID); */

HRESULT rhCoolKey::GetCoolKeyIsEnrolled(PRUint32 aKeyType, const char *aKeyID, bool *_retval)
{
    char tBuff[56];
    PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s rhCoolKey::RhCoolKeyIsEnrolled thread: %p \n",GetTStamp(tBuff,56),PR_GetCurrentThread()));
    if (ASCCoolKeyIsAvailable(aKeyType, (char *) aKeyID)) {

    if (aKeyID) {
      AutoCoolKey key(aKeyType, aKeyID);
      PRBool isEnrolled = CoolKeyIsEnrolled(&key);
      *_retval= isEnrolled;
      return NS_OK;
    }
  }

  *_retval = PR_FALSE;

  return NS_OK;
}

/* string GetCoolKeyCertInfo (in unsigned long aKeyType, in string aKeyID, in string aCertNickname); */   
HRESULT rhCoolKey::GetCoolKeyCertInfo(PRUint32 aKeyType, const char *aKeyID, const char *aCertNickname, char **aCertInfo)
{
    char tBuff[56];
    string certInfo = "";
    *aCertInfo = NULL;

    PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s rhCoolKey::GetCoolKeyCertInfo thread: %p \n",GetTStamp(tBuff,56),PR_GetCurrentThread()));

    AutoCoolKey key(aKeyType, aKeyID);

    HRESULT res =  CoolKeyGetCertInfo(&key,(char *) aCertNickname, certInfo);

    if(res == S_OK)
    {
        char *info = (char *) certInfo.c_str();

        char *temp =  (char *) PL_strdup(info);
        *aCertInfo = temp;
    }

    return NS_OK;
}

/* string GetCoolKeyATR (in unsigned long aKeyType, in string aKeyID); */
HRESULT rhCoolKey::GetCoolKeyATR(PRUint32 aKeyType, const char *aKeyID, char **_retval)
{
    char tBuff[56];
    *_retval  = NULL;
    AutoCoolKey key(aKeyType, aKeyID);
    char atr[128];
    HRESULT res =   CoolKeyGetATR(&key, (char *)&atr,sizeof(atr));
     PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s Attempting to get the key's ATR: Key: %s, ATR  %s. \n",GetTStamp(tBuff,56),aKeyID, (char *) atr));
    if(res == S_OK)
    {
        char *temp =  (char *) PL_strdup(atr);
        *_retval  = temp;
    }
      return NS_OK;
  }

/* string GetCoolKeyTokenName (in unsigned long aKeyType, in string aKeyID); */
HRESULT rhCoolKey::GetCoolKeyTokenName(PRUint32 aKeyType, const char *aKeyID, char **_retval)
{
  char tBuff[56];

  *_retval = NULL;

  if(!aKeyType && !aKeyID)
      return NS_OK;

  AutoCoolKey key(aKeyType,aKeyID);
  
  char *tokenName = NULL;

  tokenName = (char *) CoolKeyGetTokenName(&key);

  PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s rhCoolKey::GetCoolKeyTokenName %s \n",GetTStamp(tBuff,56),tokenName));
  if(tokenName)
  {
      char *temp =  (char *) PL_strdup(tokenName);
      *_retval  = temp;

  }

  return NS_OK;

}

HRESULT rhCoolKey::GetCoolKeyIssuerInfo(PRUint32 aKeyType, const char *aKeyID, char **_retval)
{
    char tBuff[56];
    *_retval  = NULL;

    AutoCoolKey key(aKeyType, aKeyID);

    char issuerInfo[256];

    HRESULT res =  CoolKeyGetIssuerInfo(&key, (char *)&issuerInfo,256);

    ::CoolKeyLogMsg( 1, "%s Attempting to get the key's Issuer: Key: %s, Issuer  %s. \n",GetTStamp(tBuff,56),aKeyID, (char *) issuerInfo);

    if(res == S_OK)
    {
        char *temp =  (char *) PL_strdup(issuerInfo);
        *_retval  = temp;

    }
    return NS_OK;

}

/* void rhGetCoolKeyPolicy (in unsigned long aKeyType, in string aKeyID, out string policy); */
HRESULT rhCoolKey::GetCoolKeyPolicy(PRUint32 aKeyType, const char *aKeyID, char **policy)
{
    char tBuff[56];
    PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s rhCoolKey::RhGetCoolKeyPolicy thread: %p \n",GetTStamp(tBuff,56),PR_GetCurrentThread()));

    if (!aKeyID) {
        return NS_ERROR_FAILURE;
    }

    char policyChar[MAX_STR_LEN] ;
    policyChar[0] = 0;

    AutoCoolKey key(aKeyType, aKeyID);
    HRESULT hres =  CoolKeyGetPolicy(&key, policyChar, MAX_STR_LEN);

    PR_LOG(coolKeyLog,PR_LOG_DEBUG,("%s rhCoolKey::RhGetCoolKeyPolicy hres: %d \n",GetTStamp(tBuff,56),hres));
    if (hres == E_FAIL)
    {
        return NS_ERROR_FAILURE;

    }

    PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s rhCoolKey::RhGetCoolKeyPolicy policy: %s \n",GetTStamp(tBuff,56),policyChar));

     char *temp =  (char *) PL_strdup(policyChar);

    *policy = temp;

    return NS_OK;
}

/* string GetCoolKeyIssuedTo (in unsigned long aKeyType, in string aKeyID); */
HRESULT rhCoolKey::GetCoolKeyUID(PRUint32 aKeyType, const char *aKeyID, char **uid)
{
    char tBuff[56];
    if (!aKeyID) {
        return NS_ERROR_FAILURE;
    }

    AutoCoolKey key(aKeyType, ( char *)aKeyID);

    char buff[512];
    int bufLength = 512;
    buff[0] = 0;
   
    CoolKeyGetUID(&key, (char *) buff, bufLength);

    if(!buff[0])
    {
        return NS_OK;
    }

    PR_LOG(coolKeyLog,PR_LOG_DEBUG,("%s rhCoolKey::RhGetCoolKeyGetUID  %s \n",GetTStamp(tBuff,56),(char *) buff));

    char *temp =  (char *) PL_strdup(buff);

    *uid = temp;

    return NS_OK;

}


/* string GetCoolKeyIssuedTo (in unsigned long aKeyType, in string aKeyID); */
HRESULT rhCoolKey::GetCoolKeyIssuedTo(PRUint32 aKeyType, const char *aKeyID, char **issuedTo)
{
    char tBuff[56];
    if (!aKeyID) {
        return NS_ERROR_FAILURE;
    }

    AutoCoolKey key(aKeyType, ( char *)aKeyID);

  //  const char *keyName = CoolKeyGetTokenName(&key);

    char buff[512];
    int bufLength = 512;
    buff[0] = 0;
    
    CoolKeyGetIssuedTo(&key, (char *) buff, bufLength);

    if(!buff[0])
    {
        return NS_OK;
    }

    PR_LOG(coolKeyLog,PR_LOG_DEBUG,("%s rhCoolKey::RhGetCoolKeyGetIssuedTo  %s \n",GetTStamp(tBuff,56),(char *) buff));

    char *temp =  (char *) PL_strdup(buff);

    *issuedTo = temp;

    return NS_OK;

}

/* string GetCoolKeyIssuer (in unsigned long aKeyType, in string aKeyID); */
HRESULT rhCoolKey::GetCoolKeyIssuer(PRUint32 aKeyType, const char *aKeyID, char **issuer)
{
    char tBuff[56];
    if (!aKeyID) {
        return NS_ERROR_FAILURE;
    }

    AutoCoolKey key(aKeyType, ( char *)aKeyID);

  //  const char *keyName = CoolKeyGetTokenName(&key);

    char buff[512];
    int bufLength = 512;
    buff[0] = 0;
   
    CoolKeyGetIssuer(&key, (char *) buff, bufLength);

    if(!buff[0])
    {
        return NS_OK;
    }

    PR_LOG(coolKeyLog,PR_LOG_DEBUG,("%s rhCoolKey::RhGetCoolKeyGetIssuer  %s \n",GetTStamp(tBuff,56),(char *) buff));

    char *temp =  (char *) PL_strdup(buff);

    *issuer = temp;

    return NS_OK;

}

/* boolean SetCoolKeyConfigValue (in string aName, in string aValue); */
HRESULT rhCoolKey::SetCoolKeyConfigValue(const char *aName, const char *aValue, bool *_retval)
{
    char tBuff[56];
    PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("rhCoolKey::SetCoolKeyConfigValue thread: %p \n",GetTStamp(tBuff,56),PR_GetCurrentThread()));
    if(!aName || !aValue)
    {
        *_retval = 0;
        return NS_ERROR_FAILURE;
    }

    *_retval = (PRBool)  doSetCoolKeyConfigValue(aName,aValue);
   
    return NS_OK;
}

/* string GetCoolKeyConfigValue (in string aName); */
HRESULT rhCoolKey::GetCoolKeyConfigValue(const char *aName, char **_retval)
{
    char tBuff[56];
    PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s rhCoolKey::GetCoolKeyConfigValue thread: %p \n",GetTStamp(tBuff,56),PR_GetCurrentThread()));

    if(!aName)
    {
        return NS_ERROR_FAILURE;
    }

   *_retval = (char *) doGetCoolKeyConfigValue(aName);

   return NS_OK;   

}

/* boolean rhCoolKeyRequiresAuthentication (in unsigned long aKeyType, in string aKeyID); */

HRESULT rhCoolKey::GetCoolKeyRequiresAuthentication(PRUint32 aKeyType, const char *aKeyID, bool *_retval)
{
    char tBuff[56];
    PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s rhCoolKey::RhCoolKeyRequiresAuthentication thread: %p \n",GetTStamp(tBuff,56),PR_GetCurrentThread()));
    PRBool requiresAuth = PR_FALSE;

    *_retval = PR_TRUE;

    if (aKeyID) {
        AutoCoolKey key(aKeyType, aKeyID);
        requiresAuth = CoolKeyRequiresAuthentication(&key);

        *_retval = requiresAuth;
    }

    return NS_OK;
}

/* boolean rhGetCoolKeyIsAuthenticated (in unsigned long aKeyType, in string aKeyID); */

HRESULT rhCoolKey::GetCoolKeyIsAuthenticated(PRUint32 aKeyType, const char *aKeyID, bool *_retval)
{
    char tBuff[56];
    PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s rhCoolKey::RhGetCoolKeyIsAuthenticated thread: %p \n",GetTStamp(tBuff,56),PR_GetCurrentThread()));
    PRBool isAuthed = PR_FALSE;

    *_retval = PR_TRUE;

    if (aKeyID) {
        AutoCoolKey key(aKeyType, aKeyID);
        isAuthed = CoolKeyIsAuthenticated(&key);

        *_retval = isAuthed;
    }

    return NS_OK;
}

/* boolean rhAuthenticateCoolKey (in unsigned long aKeyType, in string aKeyID, in string aPIN); */

HRESULT rhCoolKey::AuthenticateCoolKey(PRUint32 aKeyType, const char *aKeyID, const char *aPIN, bool *_retval) 
{
    char tBuff[56];
    PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s rhCoolKey::RhAuthenticateCoolKey thread: %p \n",GetTStamp(tBuff,56),PR_GetCurrentThread()));
    *_retval = PR_FALSE;

    if(!aKeyID || !aPIN)
    {
        return NS_ERROR_FAILURE;
    }

    AutoCoolKey key(aKeyType, aKeyID);

    PRBool didAuth = CoolKeyAuthenticate(&key, aPIN);

    if (didAuth)
        ASCSetCoolKeyPin(aKeyType, aKeyID, aPIN);

    *_retval = PR_TRUE;

    return NS_OK;
}

/* void SetCoolKeyDataValue (in unsigned long aKeyType, in string aKeyID, in string name, in string value); */

HRESULT rhCoolKey::SetCoolKeyDataValue(PRUint32 aKeyType, const char *aKeyID, const char *name, const char *value)
{
    char tBuff[56];
    PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s rhCoolKey::SetCoolKeyDataValue \n",GetTStamp(tBuff,56)));
    CoolKeyNode *node = GetCoolKeyInfo(aKeyType, aKeyID);

    if (!node)
         return NS_ERROR_FAILURE;

    AutoCoolKey key(aKeyType, aKeyID);

    CoolKeySetDataValue(&key,name, value);

    return NS_OK;
}

/* string GetCoolKeyVersion (); */
HRESULT rhCoolKey::GetCoolKeyVersion(char **_retval)
{
    char tBuff[56];
    PR_LOG( coolKeyLog, PR_LOG_DEBUG, ("%s rhCoolKey::GetCoolKeyVersion \n",GetTStamp(tBuff,56)));

    char *version = (char *) GETSTRING(ESC_VERSION);
    
    char *versionVal =  (char *) PL_strdup(version);
    
    *_retval = versionVal;   

    return NS_OK;
}
