/**
 * Copyright (c) 2021 OceanBase
 * OceanBase CE is licensed under Mulan PubL v2.
 * You can use this software according to the terms and conditions of the Mulan PubL v2.
 * You may obtain a copy of Mulan PubL v2 at:
 *          http://license.coscl.org.cn/MulanPubL-2.0
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PubL v2 for more details.
 * This file contains implementation for eval_priv_st_touches.
 */

#define USING_LOG_PREFIX SQL_ENG

#include "lib/geo/ob_geo_func_register.h"
#include "ob_expr_priv_st_touches.h"
#include "sql/engine/expr/ob_geo_expr_utils.h"

using namespace oceanbase::common;
using namespace oceanbase::sql;

namespace oceanbase
{
namespace sql
{

ObExprPrivSTTouches::ObExprPrivSTTouches(ObIAllocator &alloc)
    : ObFuncExprOperator(alloc, T_FUN_SYS_PRIV_ST_TOUCHES, N_PRIV_ST_TOUCHES, 2,
          VALID_FOR_GENERATED_COL, NOT_ROW_DIMENSION)
{}

ObExprPrivSTTouches::~ObExprPrivSTTouches()
{}

int ObExprPrivSTTouches::calc_result_type2(ObExprResType &type, ObExprResType &type1,
    ObExprResType &type2, common::ObExprTypeCtx &type_ctx) const
{
  UNUSED(type_ctx);
  INIT_SUCC(ret);
  int unexpected_types = 0;
  int null_types = 0;

  if (type1.get_type() == ObNullType) {
    null_types++;
  } else if (!ob_is_geometry(type1.get_type()) && !ob_is_string_type(type1.get_type())) {
    unexpected_types++;
    LOG_WARN("invalid type", K(type1.get_type()));
  }
  if (type2.get_type() == ObNullType) {
    null_types++;
  } else if (!ob_is_geometry(type2.get_type()) && !ob_is_string_type(type2.get_type())) {
    unexpected_types++;
    LOG_WARN("invalid type", K(type2.get_type()));
  }
  // an invalid type and a nullptr type will return nullptr
  // an invalid type and a valid type return error
  if (unexpected_types > 0) {
    ret = OB_ERR_GIS_INVALID_DATA;
    LOG_USER_ERROR(OB_ERR_GIS_INVALID_DATA, N_PRIV_ST_TOUCHES);
    LOG_WARN("invalid type", K(ret));
  }
  if (OB_SUCC(ret)) {
    type.set_int();
    type.set_scale(common::ObAccuracy::DDL_DEFAULT_ACCURACY[common::ObIntType].scale_);
    type.set_precision(common::ObAccuracy::DDL_DEFAULT_ACCURACY[common::ObIntType].precision_);
  }
  return ret;
}

int ObExprPrivSTTouches::get_input_geometry(omt::ObSrsCacheGuard &srs_guard, MultimodeAlloctor &temp_allocator, ObEvalCtx &ctx, ObExpr *gis_arg,
    ObDatum *gis_datum, const ObSrsItem *&srs, ObGeometry *&geo, bool &is_geo_empty)
{
  int ret = OB_SUCCESS;
  ObString wkb = gis_datum->get_string();
  ObGeoType type = ObGeoType::GEOTYPEMAX;
  uint32_t srid = -1;
  if (OB_FAIL(ObTextStringHelper::read_real_string_data_with_copy(temp_allocator,
          *gis_datum,
          gis_arg->datum_meta_,
          gis_arg->obj_meta_.has_lob_header(),
          wkb))) {
    LOG_WARN("fail to get real string data", K(ret), K(wkb));
  } else if (FALSE_IT(temp_allocator.add_baseline_size(wkb.length()))) {
  } else if (OB_FAIL(ObGeoTypeUtil::get_type_srid_from_wkb(wkb, type, srid))) {
    if (ret == OB_ERR_GIS_INVALID_DATA) {
      LOG_USER_ERROR(OB_ERR_GIS_INVALID_DATA, N_PRIV_ST_TOUCHES);
    }
    LOG_WARN("get type and srid from wkb failed", K(wkb), K(ret));
  } else if (OB_FAIL(ObGeoExprUtils::get_srs_item(
                  ctx, srs_guard, wkb, srs, true, N_PRIV_ST_TOUCHES))) {
    LOG_WARN("fail to get srs item", K(ret), K(wkb));
  } else if (OB_FAIL(ObGeoExprUtils::build_geometry(temp_allocator,
                  wkb,
                  geo,
                  srs,
                  N_PRIV_ST_TOUCHES,
                  GEO_ALLOW_3D_DEFAULT | GEO_NOT_COPY_WKB))) {
    LOG_WARN("get first geo by wkb failed", K(ret));
  } else if (OB_FAIL(ObGeoExprUtils::check_empty(geo, is_geo_empty))) {
    LOG_WARN("check geo empty failed", K(ret));
  }
  return ret;
}

int ObExprPrivSTTouches::eval_priv_st_touches(const ObExpr &expr, ObEvalCtx &ctx, ObDatum &res)
{
  int ret = OB_SUCCESS;
  ObExpr *gis_arg1 = expr.args_[0];
  ObExpr *gis_arg2 = expr.args_[1];
  bool is_geo1_empty = false;
  bool is_geo2_empty = false;
  bool is_geo1_null = false;
  bool is_geo2_null = false;
  ObGeometry *geo1 = nullptr;
  ObGeometry *geo2 = nullptr;
  omt::ObSrsCacheGuard srs_guard;
  const ObSrsItem *srs1 = nullptr;
  const ObSrsItem *srs2 = nullptr;
  ObEvalCtx::TempAllocGuard tmp_alloc_g(ctx);
  uint64_t tenant_id = ObMultiModeExprHelper::get_tenant_id(ctx.exec_ctx_.get_my_session());
  MultimodeAlloctor temp_allocator(tmp_alloc_g.get_allocator(), expr.type_, tenant_id, ret, N_PRIV_ST_TOUCHES);
  ObDatum *gis_datum1 = nullptr;
  ObDatum *gis_datum2 = nullptr;
  if (OB_FAIL(gis_arg1->eval(ctx, gis_datum1)) || OB_FAIL(gis_arg2->eval(ctx, gis_datum2))) {
    LOG_WARN("eval geo args failed", K(ret));
  } else if (gis_datum1->is_null() || gis_datum2->is_null()) {
    res.set_null();
  } else if (OB_FAIL(get_input_geometry(
                 srs_guard, temp_allocator, ctx, gis_arg1, gis_datum1, srs1, geo1, is_geo1_empty))) {
    LOG_WARN("fail to get input geometry", K(ret));
  } else if (!is_geo1_null && OB_FAIL(get_input_geometry(
                 srs_guard, temp_allocator, ctx, gis_arg2, gis_datum2, srs2, geo2, is_geo2_empty))) {
    LOG_WARN("fail to get input geometry", K(ret));
  } else if (is_geo1_null || is_geo2_null) {
    res.set_null();
  } else {
    uint32_t srid1 = srs1 == nullptr ? 0 : srs1->get_srid();
    uint32_t srid2 = srs2 == nullptr ? 0 : srs2->get_srid();
    ObGeoBoostAllocGuard guard(tenant_id);
    lib::MemoryContext *mem_ctx = nullptr;
    if (srid1 != srid2) {
      ret = OB_ERR_GIS_DIFFERENT_SRIDS;
      LOG_WARN("srid not the same", K(ret), K(srid1), K(srid2));
      LOG_USER_ERROR(OB_ERR_GIS_DIFFERENT_SRIDS, N_PRIV_ST_TOUCHES, srid1, srid2);
    } else if (is_geo1_empty || is_geo2_empty) {
      res.set_bool(false);
    } else if (OB_FAIL(ObGeoExprUtils::zoom_in_geos_for_relation(srs1, *geo1, *geo2))) {
      LOG_WARN("zoom in geos failed", K(ret));
    } else if (OB_FAIL(guard.init())) {
      LOG_WARN("fail to init geo allocator guard", K(ret));
    } else if (OB_ISNULL(mem_ctx = guard.get_memory_ctx())) {
      ret = OB_ERR_NULL_VALUE;
      LOG_WARN("fail to get mem ctx", K(ret));
    } else {
      bool result = false;
      ObGeoEvalCtx gis_context(*mem_ctx, srs1);
      if (OB_FAIL(gis_context.append_geo_arg(geo1)) || OB_FAIL(gis_context.append_geo_arg(geo2))) {
        LOG_WARN("build gis context failed", K(ret), K(gis_context.get_geo_count()));
      } else if (OB_FAIL(ObGeoFunc<ObGeoFuncType::Touches>::geo_func::eval(gis_context, result))) {
        LOG_WARN("eval st intersection failed", K(ret));
        ObGeoExprUtils::geo_func_error_handle(ret, N_PRIV_ST_TOUCHES);
      } else {
        res.set_bool(result);
      }
    }
    if (mem_ctx != nullptr) {
      temp_allocator.add_ext_used((*mem_ctx)->arena_used());
    }
  }
  return ret;
}

int ObExprPrivSTTouches::cg_expr(
    ObExprCGCtx &expr_cg_ctx, const ObRawExpr &raw_expr, ObExpr &rt_expr) const
{
  UNUSED(expr_cg_ctx);
  UNUSED(raw_expr);
  rt_expr.eval_func_ = eval_priv_st_touches;
  return OB_SUCCESS;
}

}  // namespace sql
}  // namespace oceanbase
