/*
 * Copyright (c) 2024 OceanBase.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package api

import (
	"github.com/gin-gonic/gin"

	"github.com/oceanbase/obshell/agent/api/common"
	"github.com/oceanbase/obshell/agent/constant"
	"github.com/oceanbase/obshell/agent/errors"
	"github.com/oceanbase/obshell/agent/executor/pool"
	"github.com/oceanbase/obshell/agent/meta"
)

//	@ID				poolDrop
//	@Summary		drop resource pool
//	@Description	drop resource pool which is not used by any tenant
//	@Tags			resource-pool
//	@Accept			application/json
//	@Produce		application/json
//	@Param			X-OCS-Header	header	string	true	"Authorization"
//	@Param			name			path	string	true	"resource pool name"
//	@Success		200				object	http.OcsAgentResponse
//	@Failure		400				object	http.OcsAgentResponse
//	@Failure		401				object	http.OcsAgentResponse
//	@Failure		500				object	http.OcsAgentResponse
//	@Router			/api/v1/resource-pool/{name} [delete]
func poolDropHandler(c *gin.Context) {
	name := c.Param(constant.URI_PARAM_NAME)
	if name == "" {
		common.SendResponse(c, nil, errors.Occur(errors.ErrIllegalArgument, "Resource pool name is empty."))
		return
	}
	if !meta.OCS_AGENT.IsClusterAgent() {
		common.SendResponse(c, nil, errors.Occurf(errors.ErrKnown, "%s is not cluster agent.", meta.OCS_AGENT.String()))
		return
	}
	err := pool.DropResourcePool(name)
	common.SendResponse(c, nil, err)
}

//	@ID				poolList
//	@Summary		list all resource pools
//	@Description	list all resource pools in the cluster
//	@Tags			resource-pool
//	@Accept			application/json
//	@Produce		application/json
//	@Param			X-OCS-Header	header	string	true	"Authorization"
//	@Success		200				object	http.OcsAgentResponse{data=[]oceanbase.DbaObResourcePool}
//	@Failure		400				object	http.OcsAgentResponse
//	@Failure		401				object	http.OcsAgentResponse
//	@Failure		500				object	http.OcsAgentResponse
//	@Router			/api/v1/resources-pool [get]
func poolListHandler(c *gin.Context) {
	if !meta.OCS_AGENT.IsClusterAgent() {
		common.SendResponse(c, nil, errors.Occurf(errors.ErrKnown, "%s is not cluster agent.", meta.OCS_AGENT.String()))
		return
	}
	pools, err := pool.GetAllResourcePools()
	common.SendResponse(c, pools, err)
}
