/*
 * Copyright 2020 The TensorFlow Runtime Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

// Tensorflow Tile operation.

#include "tile_op.h"

#include <sys/types.h>

#include <cstdint>

#include "../../kernels/tile_kernel.h"
#include "tfrt/common/compat/eigen/eigen_dtype.h"
#include "tfrt/core_runtime/op_utils.h"
#include "tfrt/cpu/core_runtime/cpu_op_registry.h"
#include "tfrt/host_context/async_value_ref.h"
#include "tfrt/host_context/diagnostic.h"
#include "tfrt/host_context/kernel_utils.h"
#include "tfrt/support/forward_decls.h"
#include "tfrt/tensor/string_host_tensor.h"

namespace tfrt {
namespace {

static AsyncValueRef<HostTensor> TfTileOp(const HostTensor& input_arg,
                                          const DenseHostTensor& multiples_arg,
                                          const ExecutionContext& exec_ctx) {
  HostContext* host = exec_ctx.host();

  // Parse multiples tensor.
  Expected<llvm::SmallVector<Index, 5>> expected_multiples =
      cpu::TileMultiples(multiples_arg);
  if (auto err = expected_multiples.takeError())
    return EmitErrorAsync(exec_ctx,
                          absl::InternalError(toString(std::move(err))));

  // Check that multiples are valid.
  const TensorShape& input_shape = input_arg.shape();
  if (expected_multiples->size() != input_shape.GetRank()) {
    return EmitErrorAsync(
        exec_ctx, "Tile multiples must have the same size as input rank");
  }

  // Compute the output shape from the input shape and multiples.
  llvm::SmallVector<Index, 5> output_dims;
  for (int d = 0; d < expected_multiples->size(); ++d) {
    output_dims.push_back(input_shape.GetDimensionSize(d) *
                          (*expected_multiples)[d]);
  }

  TensorShape output_shape(output_dims);
  TensorMetadata output_md(input_arg.dtype(), output_shape);

  if (isa<DenseHostTensor>(input_arg)) {
    const DenseHostTensor& input = cast<DenseHostTensor>(input_arg);

    // Allocate output tensor.
    auto dest = DenseHostTensor::CreateUninitialized(output_md, host);
    if (!dest) {
      return EmitErrorAsync(exec_ctx, "out of memory allocating result");
    }

    // Call tile kernel.
    AsyncValueRef<Chain> chain;

    switch (input.dtype()) {
      default:
        chain = EmitErrorAsync(exec_ctx,
                               StrCat("Unsupported dtype: ", input.dtype()));
        break;
      case DType::I1: {
        using T = EigenTypeForDTypeKind<DType::I1>;
        chain = ::tfrt::cpu::Tile<T, compat::AsyncEigenEvaluator>(
            input, *expected_multiples, &*dest, exec_ctx);
      } break;
#define DTYPE_NUMERIC(ENUM)                                    \
  case DType::ENUM: {                                          \
    using T = EigenTypeForDTypeKind<DType::ENUM>;              \
    chain = ::tfrt::cpu::Tile<T, compat::AsyncEigenEvaluator>( \
        input, *expected_multiples, &*dest, exec_ctx);         \
  } break;
#include "tfrt/dtype/dtype.def"  // NOLINT
    }

    return ForwardValue(dest.value(), std::move(chain));

  } else if (isa<StringHostTensor>(input_arg)) {
    const StringHostTensor& input = cast<StringHostTensor>(input_arg);

    auto dest = StringHostTensor::CreateUninitialized(output_md, host);
    if (!dest) {
      return EmitErrorAsync(exec_ctx, "out of memory allocating result");
    }

    cpu::TileStringTensor(input, &*dest);

    return MakeAvailableAsyncValueRef<StringHostTensor>(std::move(*dest));

  } else {
    return EmitErrorAsync(exec_ctx, "Unsupported tensor type");
  }
}

}  // namespace

void RegisterTfTileCpuOp(CpuOpRegistry* op_registry) {
  op_registry->AddOp("tf.Tile", TFRT_CPU_OP(TfTileOp),
                     CpuOpFlags::NoSideEffects | CpuOpFlags::AllowsString);
}

}  // namespace tfrt
