"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.changelogTask = void 0;
const tslib_1 = require("tslib");
const chalk_1 = tslib_1.__importDefault(require("chalk"));
const lodash_1 = require("lodash");
const githubClient_1 = tslib_1.__importDefault(require("../utils/githubClient"));
const useSpinner_1 = require("../utils/useSpinner");
const task_1 = require("./task");
const filterBugs = (item) => {
    if (item.title.match(/fix|fixes/i)) {
        return true;
    }
    if (item.labels.find((label) => label.name === 'type/bug')) {
        return true;
    }
    return false;
};
const getPackageChangelog = (packageName, issues) => {
    if (issues.length === 0) {
        return '';
    }
    let markdown = chalk_1.default.bold.yellow(`\n\n/*** ${packageName} changelog  ***/\n\n`);
    const bugs = (0, lodash_1.sortBy)(issues.filter(filterBugs), 'title');
    const notBugs = (0, lodash_1.sortBy)((0, lodash_1.difference)(issues, bugs), 'title');
    if (notBugs.length > 0) {
        markdown += '### Features / Enhancements\n';
        for (const item of notBugs) {
            markdown += getMarkdownLineForIssue(item);
        }
    }
    if (bugs.length > 0) {
        markdown += '\n### Bug Fixes\n';
        for (const item of bugs) {
            markdown += getMarkdownLineForIssue(item);
        }
    }
    return markdown;
};
const changelogTaskRunner = ({ milestone }) => (0, useSpinner_1.useSpinner)('Generating changelog', () => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
    const githubClient = new githubClient_1.default();
    const client = githubClient.client;
    if (!/^\d+$/.test(milestone)) {
        console.log('Use milestone number not title, find number in milestone url');
        return;
    }
    let res = yield client.get('/issues', {
        params: {
            state: 'closed',
            per_page: 100,
            labels: 'add to changelog',
            milestone: milestone,
        },
    });
    const data = res.data;
    while (res.headers.link) {
        const links = parseLink(res.headers.link);
        if (links.next) {
            res = yield client.get(links.next);
            data.push(...res.data);
        }
        else {
            break;
        }
    }
    const mergedIssues = [];
    for (const item of data) {
        if (!item.pull_request) {
            // it's an issue, not pull request
            mergedIssues.push(item);
            continue;
        }
        const isMerged = yield client.get(item.pull_request.url + '/merge');
        if (isMerged.status === 204) {
            mergedIssues.push(item);
        }
    }
    const issues = (0, lodash_1.sortBy)(mergedIssues, 'title');
    const toolkitIssues = issues.filter((item) => item.labels.find((label) => label.name === 'area/grafana/toolkit'));
    const grafanaUiIssues = issues.filter((item) => item.labels.find((label) => label.name === 'area/grafana/ui'));
    let markdown = '';
    markdown += getPackageChangelog('Grafana', issues);
    markdown += getPackageChangelog('grafana-toolkit', toolkitIssues);
    markdown += getPackageChangelog('grafana-ui', grafanaUiIssues);
    console.log(markdown);
}));
function getMarkdownLineForIssue(item) {
    const githubGrafanaUrl = 'https://github.com/grafana/grafana';
    let markdown = '';
    let title = item.title.replace(/^([^:]*)/, (_match, g1) => {
        return `**${g1}**`;
    });
    title = title.trim();
    if (title[title.length - 1] === '.') {
        title = title.slice(0, -1);
    }
    if (!item.pull_request) {
        markdown += '* ' + title + '.';
        markdown += ` [#${item.number}](${githubGrafanaUrl}/issues/${item.number})`;
    }
    else {
        markdown += '* ' + title + '.';
        markdown += ` [#${item.number}](${githubGrafanaUrl}/pull/${item.number})`;
        markdown += `, [@${item.user.login}](${item.user.html_url})`;
    }
    markdown += '\n';
    return markdown;
}
function parseLink(s) {
    const output = {};
    const regex = /<([^>]+)>; rel="([^"]+)"/g;
    let m;
    while ((m = regex.exec(s))) {
        const [, v, k] = m;
        output[k] = v;
    }
    return output;
}
exports.changelogTask = new task_1.Task('Changelog generator task', changelogTaskRunner);
//# sourceMappingURL=changelog.js.map