/*
 *  Licensed to the Apache Software Foundation (ASF) under one or more
 *  contributor license agreements.  See the NOTICE file distributed with
 *  this work for additional information regarding copyright ownership.
 *  The ASF licenses this file to You under the Apache License, Version 2.0
 *  (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package security.tomcat;

import java.io.File;
import java.net.InetSocketAddress;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSocket;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;

import org.junit.Assert;
import org.junit.Assume;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import org.junit.runners.Parameterized.Parameter;

import org.apache.catalina.Context;
import org.apache.catalina.startup.Tomcat;
import org.apache.catalina.startup.TomcatBaseTest;
import org.apache.tomcat.util.buf.ByteChunk;
import org.apache.tomcat.util.net.Constants;
import org.apache.tomcat.util.net.TesterSupport;
import org.apache.tomcat.websocket.server.WsContextListener;

@RunWith(Parameterized.class)
public class TestRekeyingSupport extends TomcatBaseTest {

    @Parameterized.Parameters(name = "{0}")
    public static Collection<Object[]> parameters() {
        List<Object[]> parameterSets = new ArrayList<>();
        parameterSets.add(new Object[] {
                "JSSE", Boolean.TRUE, "org.apache.tomcat.util.net.jsse.JSSEImplementation"});
        // We can skip testing openssl impl since we wont have tomcat-native
        // parameterSets.add(new Object[] {
        //         "OpenSSL", Boolean.TRUE, "org.apache.tomcat.util.net.openssl.OpenSSLImplementation"});
        // parameterSets.add(new Object[] {
        //         "OpenSSL-FFM", Boolean.TRUE, "org.apache.tomcat.util.net.openssl.panama.OpenSSLImplementation"});

        return parameterSets;
    }

    @Parameter(0)
    public String connectorName;

    @Parameter(1)
    public boolean useOpenSSL;

    @Parameter(2)
    public String sslImplementationName;

    @Test
    public void testRekeyFlood() throws Exception {
        Assume.assumeTrue(TesterSupport.isTlsv13Available());

        TesterSupport.configureClientSsl();

        Tomcat tomcat = getTomcatInstance();
        TesterSupport.initSsl(tomcat);
        // useOpenSSL wasn't added until 10.1.24
        // TesterSupport.configureSSLImplementation(tomcat, sslImplementationName, useOpenSSL);
        TesterSupport.configureSSLImplementation(tomcat, sslImplementationName);

        File appDir = new File(getBuildDirectory(), "webapps/examples");
        Context ctxt  = tomcat.addWebapp(null, "/examples", appDir.getAbsolutePath());
        ctxt.addApplicationListener(WsContextListener.class.getName());

        tomcat.start();

        // Do a raw socket connection with a custom context
        SSLContext context = SSLContext.getInstance(Constants.SSL_PROTO_TLSv1_3);
        context.init(null, new TrustManager[] { new X509TrustManager() {
            public void checkClientTrusted(X509Certificate[] arg0, String arg1) throws CertificateException {}
            public void checkServerTrusted(X509Certificate[] arg0, String arg1) throws CertificateException {}
            public X509Certificate[] getAcceptedIssuers() {return null;}
        } }, null);

        SSLSocket sock = (SSLSocket) context.getSocketFactory().createSocket();
        sock.setEnabledProtocols(new String[] { Constants.SSL_PROTO_TLSv1_3 });
        sock.connect(new InetSocketAddress("localhost", getPort()));
        for (int i = 0; i < 100000000; ++i) {
            try {
                sock.startHandshake();
                if (i % 10000 == 0) {
                    System.out.println(i);
                }
            } catch (Exception e) {
                log.info("Stopped at " + i, e);
                break;
            }
        }

        // Check that the server is alive
        ByteChunk res = getUrl("https://localhost:" + getPort() +
            "/examples/servlets/servlet/HelloWorldExample");
        Assert.assertTrue(res.toString().indexOf("<a href=\"../helloworld.html\">") > 0);

        sock.close();
    }

}
