"use strict";

var _generateFuncSignature = require("./generateFuncSignature");

var _jsdocParser = require("../../../lib/jsdocParser");

describe('generateFuncSignature', function () {
  it('should return an empty string when there is no @params and @returns tags', function () {
    var result = (0, _generateFuncSignature.generateFuncSignature)(null, null);
    expect(result).toBe('');
  });
  it('should return a signature with a single arg when there is a @param tag with a name', function () {
    var _parseJsDoc$extracted = (0, _jsdocParser.parseJsDoc)('@param event').extractedTags,
        params = _parseJsDoc$extracted.params,
        returns = _parseJsDoc$extracted.returns;
    var result = (0, _generateFuncSignature.generateFuncSignature)(params, returns);
    expect(result).toBe('(event)');
  });
  it('should return a signature with a single arg when there is a @param tag with a name and a type', function () {
    var _parseJsDoc$extracted2 = (0, _jsdocParser.parseJsDoc)('@param {SyntheticEvent} event').extractedTags,
        params = _parseJsDoc$extracted2.params,
        returns = _parseJsDoc$extracted2.returns;
    var result = (0, _generateFuncSignature.generateFuncSignature)(params, returns);
    expect(result).toBe('(event: SyntheticEvent)');
  });
  it('should return a signature with a single arg when there is a @param tag with a name, a type and a desc', function () {
    var _parseJsDoc$extracted3 = (0, _jsdocParser.parseJsDoc)('@param {SyntheticEvent} event - React event').extractedTags,
        params = _parseJsDoc$extracted3.params,
        returns = _parseJsDoc$extracted3.returns;
    var result = (0, _generateFuncSignature.generateFuncSignature)(params, returns);
    expect(result).toBe('(event: SyntheticEvent)');
  });
  it('should support @param of record type', function () {
    var _parseJsDoc$extracted4 = (0, _jsdocParser.parseJsDoc)('@param {{a: number}} event').extractedTags,
        params = _parseJsDoc$extracted4.params,
        returns = _parseJsDoc$extracted4.returns;
    var result = (0, _generateFuncSignature.generateFuncSignature)(params, returns);
    expect(result).toBe('(event: ({a: number}))');
  });
  it('should support @param of union type', function () {
    var _parseJsDoc$extracted5 = (0, _jsdocParser.parseJsDoc)('@param {(number|boolean)} event').extractedTags,
        params = _parseJsDoc$extracted5.params,
        returns = _parseJsDoc$extracted5.returns;
    var result = (0, _generateFuncSignature.generateFuncSignature)(params, returns);
    expect(result).toBe('(event: (number|boolean))');
  });
  it('should support @param of array type', function () {
    var _parseJsDoc$extracted6 = (0, _jsdocParser.parseJsDoc)('@param {number[]} event').extractedTags,
        params = _parseJsDoc$extracted6.params,
        returns = _parseJsDoc$extracted6.returns;
    var result = (0, _generateFuncSignature.generateFuncSignature)(params, returns);
    expect(result).toBe('(event: number[])');
  });
  it('should support @param with a nullable type', function () {
    var _parseJsDoc$extracted7 = (0, _jsdocParser.parseJsDoc)('@param {?number} event').extractedTags,
        params = _parseJsDoc$extracted7.params,
        returns = _parseJsDoc$extracted7.returns;
    var result = (0, _generateFuncSignature.generateFuncSignature)(params, returns);
    expect(result).toBe('(event: number)');
  });
  it('should support @param with a non nullable type', function () {
    var _parseJsDoc$extracted8 = (0, _jsdocParser.parseJsDoc)('@param {!number} event').extractedTags,
        params = _parseJsDoc$extracted8.params,
        returns = _parseJsDoc$extracted8.returns;
    var result = (0, _generateFuncSignature.generateFuncSignature)(params, returns);
    expect(result).toBe('(event: number)');
  });
  it('should support optional @param with []', function () {
    var _parseJsDoc$extracted9 = (0, _jsdocParser.parseJsDoc)('@param {number} [event]').extractedTags,
        params = _parseJsDoc$extracted9.params,
        returns = _parseJsDoc$extracted9.returns;
    var result = (0, _generateFuncSignature.generateFuncSignature)(params, returns);
    expect(result).toBe('(event: number)');
  });
  it('should support optional @param with =', function () {
    var _parseJsDoc$extracted10 = (0, _jsdocParser.parseJsDoc)('@param {number=} event').extractedTags,
        params = _parseJsDoc$extracted10.params,
        returns = _parseJsDoc$extracted10.returns;
    var result = (0, _generateFuncSignature.generateFuncSignature)(params, returns);
    expect(result).toBe('(event: number)');
  });
  it('should support @param of type any', function () {
    var _parseJsDoc$extracted11 = (0, _jsdocParser.parseJsDoc)('@param {*} event').extractedTags,
        params = _parseJsDoc$extracted11.params,
        returns = _parseJsDoc$extracted11.returns;
    var result = (0, _generateFuncSignature.generateFuncSignature)(params, returns);
    expect(result).toBe('(event: any)');
  });
  it('should support multiple @param tags', function () {
    var _parseJsDoc$extracted12 = (0, _jsdocParser.parseJsDoc)('@param {SyntheticEvent} event\n@param {string} customData').extractedTags,
        params = _parseJsDoc$extracted12.params,
        returns = _parseJsDoc$extracted12.returns;
    var result = (0, _generateFuncSignature.generateFuncSignature)(params, returns);
    expect(result).toBe('(event: SyntheticEvent, customData: string)');
  });
  it('should return a signature with a return type when there is a @returns with a type', function () {
    var _parseJsDoc$extracted13 = (0, _jsdocParser.parseJsDoc)('@returns {string}').extractedTags,
        params = _parseJsDoc$extracted13.params,
        returns = _parseJsDoc$extracted13.returns;
    var result = (0, _generateFuncSignature.generateFuncSignature)(params, returns);
    expect(result).toBe('() => string');
  });
  it('should support @returns of record type', function () {
    var _parseJsDoc$extracted14 = (0, _jsdocParser.parseJsDoc)('@returns {{a: number, b: string}}').extractedTags,
        params = _parseJsDoc$extracted14.params,
        returns = _parseJsDoc$extracted14.returns;
    var result = (0, _generateFuncSignature.generateFuncSignature)(params, returns);
    expect(result).toBe('() => ({a: number, b: string})');
  });
  it('should support @returns of array type', function () {
    var _parseJsDoc$extracted15 = (0, _jsdocParser.parseJsDoc)('@returns {integer[]}').extractedTags,
        params = _parseJsDoc$extracted15.params,
        returns = _parseJsDoc$extracted15.returns;
    var result = (0, _generateFuncSignature.generateFuncSignature)(params, returns);
    expect(result).toBe('() => integer[]');
  });
  it('should support @returns of union type', function () {
    var _parseJsDoc$extracted16 = (0, _jsdocParser.parseJsDoc)('@returns {(number|boolean)}').extractedTags,
        params = _parseJsDoc$extracted16.params,
        returns = _parseJsDoc$extracted16.returns;
    var result = (0, _generateFuncSignature.generateFuncSignature)(params, returns);
    expect(result).toBe('() => (number|boolean)');
  });
  it('should support @returns type any', function () {
    var _parseJsDoc$extracted17 = (0, _jsdocParser.parseJsDoc)('@returns {*}').extractedTags,
        params = _parseJsDoc$extracted17.params,
        returns = _parseJsDoc$extracted17.returns;
    var result = (0, _generateFuncSignature.generateFuncSignature)(params, returns);
    expect(result).toBe('() => any');
  });
  it('should support @returns of type void', function () {
    var _parseJsDoc$extracted18 = (0, _jsdocParser.parseJsDoc)('@returns {void}').extractedTags,
        params = _parseJsDoc$extracted18.params,
        returns = _parseJsDoc$extracted18.returns;
    var result = (0, _generateFuncSignature.generateFuncSignature)(params, returns);
    expect(result).toBe('() => void');
  });
  it('should return a full signature when there is a single @param tag and a @returns', function () {
    var _parseJsDoc$extracted19 = (0, _jsdocParser.parseJsDoc)('@param {SyntheticEvent} event - React event.\n@returns {string}').extractedTags,
        params = _parseJsDoc$extracted19.params,
        returns = _parseJsDoc$extracted19.returns;
    var result = (0, _generateFuncSignature.generateFuncSignature)(params, returns);
    expect(result).toBe('(event: SyntheticEvent) => string');
  });
  it('should return a full signature when there is a multiple @param tags and a @returns', function () {
    var _parseJsDoc$extracted20 = (0, _jsdocParser.parseJsDoc)('@param {SyntheticEvent} event - React event.\n@param {string} data\n@returns {string}').extractedTags,
        params = _parseJsDoc$extracted20.params,
        returns = _parseJsDoc$extracted20.returns;
    var result = (0, _generateFuncSignature.generateFuncSignature)(params, returns);
    expect(result).toBe('(event: SyntheticEvent, data: string) => string');
  });
});
describe('generateShortFuncSignature', function () {
  it('should return an empty string when there is no @params and @returns tags', function () {
    var result = (0, _generateFuncSignature.generateShortFuncSignature)(null, null);
    expect(result).toBe('');
  });
  it('should return ( ... ) when there is @params', function () {
    var _parseJsDoc$extracted21 = (0, _jsdocParser.parseJsDoc)('@param event').extractedTags,
        params = _parseJsDoc$extracted21.params,
        returns = _parseJsDoc$extracted21.returns;
    var result = (0, _generateFuncSignature.generateShortFuncSignature)(params, returns);
    expect(result).toBe('( ... )');
  });
  it('should return ( ... ) => returnsType when there is @params and a @returns', function () {
    var _parseJsDoc$extracted22 = (0, _jsdocParser.parseJsDoc)('@param event\n@returns {string}').extractedTags,
        params = _parseJsDoc$extracted22.params,
        returns = _parseJsDoc$extracted22.returns;
    var result = (0, _generateFuncSignature.generateShortFuncSignature)(params, returns);
    expect(result).toBe('( ... ) => string');
  });
  it('should return () => returnsType when there is only a @returns', function () {
    var _parseJsDoc$extracted23 = (0, _jsdocParser.parseJsDoc)('@returns {string}').extractedTags,
        params = _parseJsDoc$extracted23.params,
        returns = _parseJsDoc$extracted23.returns;
    var result = (0, _generateFuncSignature.generateShortFuncSignature)(params, returns);
    expect(result).toBe('() => string');
  });
});