"use strict";

require("core-js/modules/es.array.for-each");

require("core-js/modules/es.function.name");

var _acornParser = require("./acornParser");

var _types = require("./types");

describe('parse', function () {
  describe('expression', function () {
    it('support HTML element', function () {
      var result = (0, _acornParser.parse)('<div>Hello!</div>');
      var inferredType = result.inferredType;
      expect(inferredType.type).toBe(_types.InspectionType.ELEMENT);
      expect(inferredType.identifier).toBe('div');
      expect(result.ast).toBeDefined();
    });
    it('support React declaration', function () {
      var result = (0, _acornParser.parse)('<FunctionalComponent />');
      var inferredType = result.inferredType;
      expect(inferredType.type).toBe(_types.InspectionType.ELEMENT);
      expect(inferredType.identifier).toBe('FunctionalComponent');
      expect(result.ast).toBeDefined();
    });
    it('support anonymous functional React component', function () {
      var result = (0, _acornParser.parse)('() => { return <div>Hey!</div>; }');
      var inferredType = result.inferredType;
      expect(inferredType.type).toBe(_types.InspectionType.ELEMENT);
      expect(inferredType.identifier).toBeUndefined();
      expect(result.ast).toBeDefined();
    });
    it('support named functional React component', function () {
      var result = (0, _acornParser.parse)('function NamedFunctionalComponent() { return <div>Hey!</div>; }');
      var inferredType = result.inferredType;
      expect(inferredType.type).toBe(_types.InspectionType.ELEMENT);
      expect(inferredType.identifier).toBe('NamedFunctionalComponent');
      expect(result.ast).toBeDefined();
    });
    it('support class React component', function () {
      var result = (0, _acornParser.parse)("\n        class ClassComponent extends React.PureComponent {\n          render() {\n            return <div>Hey!</div>;\n          }\n      }");
      var inferredType = result.inferredType;
      expect(inferredType.type).toBe(_types.InspectionType.ELEMENT);
      expect(inferredType.identifier).toBe('ClassComponent');
      expect(result.ast).toBeDefined();
    });
    it('support PropTypes.shape', function () {
      var result = (0, _acornParser.parse)('PropTypes.shape({ foo: PropTypes.string })');
      var inferredType = result.inferredType;
      expect(inferredType.type).toBe(_types.InspectionType.OBJECT);
      expect(inferredType.depth).toBe(1);
      expect(result.ast).toBeDefined();
    });
    it('support deep PropTypes.shape', function () {
      var result = (0, _acornParser.parse)('PropTypes.shape({ foo: PropTypes.shape({ bar: PropTypes.string }) })');
      var inferredType = result.inferredType;
      expect(inferredType.type).toBe(_types.InspectionType.OBJECT);
      expect(inferredType.depth).toBe(2);
      expect(result.ast).toBeDefined();
    });
    it('support shape', function () {
      var result = (0, _acornParser.parse)('shape({ foo: string })');
      var inferredType = result.inferredType;
      expect(inferredType.type).toBe(_types.InspectionType.OBJECT);
      expect(inferredType.depth).toBe(1);
      expect(result.ast).toBeDefined();
    });
    it('support deep shape', function () {
      var result = (0, _acornParser.parse)('shape({ foo: shape({ bar: string }) })');
      var inferredType = result.inferredType;
      expect(inferredType.type).toBe(_types.InspectionType.OBJECT);
      expect(inferredType.depth).toBe(2);
      expect(result.ast).toBeDefined();
    });
    it('support single prop object literal', function () {
      var result = (0, _acornParser.parse)('{ foo: PropTypes.string }');
      var inferredType = result.inferredType;
      expect(inferredType.type).toBe(_types.InspectionType.OBJECT);
      expect(inferredType.depth).toBe(1);
      expect(result.ast).toBeDefined();
    });
    it('support multi prop object literal', function () {
      var result = (0, _acornParser.parse)("\n      {\n          foo: PropTypes.string,\n          bar: PropTypes.string\n      }");
      var inferredType = result.inferredType;
      expect(inferredType.type).toBe(_types.InspectionType.OBJECT);
      expect(inferredType.depth).toBe(1);
      expect(result.ast).toBeDefined();
    });
    it('support deep object literal', function () {
      var result = (0, _acornParser.parse)("\n      {\n          foo: {\n            hey: PropTypes.string\n          },\n          bar: PropTypes.string,\n          hey: {\n            ho: PropTypes.string\n          }\n      }");
      var inferredType = result.inferredType;
      expect(inferredType.type).toBe(_types.InspectionType.OBJECT);
      expect(inferredType.depth).toBe(2);
      expect(result.ast).toBeDefined();
    });
    it('support required prop', function () {
      var result = (0, _acornParser.parse)('{ foo: PropTypes.string.isRequired }');
      var inferredType = result.inferredType;
      expect(inferredType.type).toBe(_types.InspectionType.OBJECT);
      expect(inferredType.depth).toBe(1);
      expect(result.ast).toBeDefined();
    });
    it('support array', function () {
      var result = (0, _acornParser.parse)("['bottom-left', 'botton-center', 'bottom-right']");
      var inferredType = result.inferredType;
      expect(inferredType.type).toBe(_types.InspectionType.ARRAY);
      expect(inferredType.depth).toBe(1);
      expect(result.ast).toBeDefined();
    });
    it('support deep array', function () {
      var result = (0, _acornParser.parse)("['bottom-left', { foo: string }, [['hey', 'ho']]]");
      var inferredType = result.inferredType;
      expect(inferredType.type).toBe(_types.InspectionType.ARRAY);
      expect(inferredType.depth).toBe(3);
      expect(result.ast).toBeDefined();
    });
    it('support object identifier', function () {
      var result = (0, _acornParser.parse)('NAMED_OBJECT');
      var inferredType = result.inferredType;
      expect(inferredType.type).toBe(_types.InspectionType.IDENTIFIER);
      expect(inferredType.identifier).toBe('NAMED_OBJECT');
      expect(result.ast).toBeDefined();
    });
    it('support anonymous function', function () {
      var result = (0, _acornParser.parse)('() => {}');
      var inferredType = result.inferredType;
      expect(inferredType.type).toBe(_types.InspectionType.FUNCTION);
      expect(inferredType.identifier).toBeUndefined();
      expect(inferredType.hasParams).toBeFalsy();
      expect(inferredType.params.length).toBe(0);
      expect(result.ast).toBeDefined();
    });
    it('support anonymous function with arguments', function () {
      var result = (0, _acornParser.parse)('(a, b) => {}');
      var inferredType = result.inferredType;
      expect(inferredType.type).toBe(_types.InspectionType.FUNCTION);
      expect(inferredType.identifier).toBeUndefined();
      expect(inferredType.hasParams).toBeTruthy();
      expect(inferredType.params.length).toBe(2);
      expect(result.ast).toBeDefined();
    });
    it('support named function', function () {
      var result = (0, _acornParser.parse)('function concat() {}');
      var inferredType = result.inferredType;
      expect(inferredType.type).toBe(_types.InspectionType.FUNCTION);
      expect(inferredType.identifier).toBe('concat');
      expect(inferredType.hasParams).toBeFalsy();
      expect(inferredType.params.length).toBe(0);
      expect(result.ast).toBeDefined();
    });
    it('support named function with arguments', function () {
      var result = (0, _acornParser.parse)('function concat(a, b) {}');
      var inferredType = result.inferredType;
      expect(inferredType.type).toBe(_types.InspectionType.FUNCTION);
      expect(inferredType.identifier).toBe('concat');
      expect(inferredType.hasParams).toBeTruthy();
      expect(inferredType.params.length).toBe(2);
      expect(result.ast).toBeDefined();
    });
    it('support class', function () {
      var result = (0, _acornParser.parse)('class Foo {}');
      var inferredType = result.inferredType;
      expect(inferredType.type).toBe(_types.InspectionType.CLASS);
      expect(inferredType.identifier).toBe('Foo');
      expect(result.ast).toBeDefined();
    });
    [{
      name: 'string',
      value: "'string value'"
    }, {
      name: 'numeric',
      value: '1'
    }, {
      name: 'boolean (true)',
      value: 'true'
    }, {
      name: 'boolean (false)',
      value: 'false'
    }, {
      name: 'null',
      value: 'null'
    }].forEach(function (x) {
      it("support ".concat(x.name), function () {
        var result = (0, _acornParser.parse)(x.value);
        var inferredType = result.inferredType;
        expect(inferredType.type).toBe(_types.InspectionType.LITERAL);
        expect(result.ast).toBeDefined();
      });
    });
    it("returns Unknown when it's not supported", function () {
      var result = (0, _acornParser.parse)("Symbol('foo')");
      var inferredType = result.inferredType;
      expect(inferredType.type).toBe(_types.InspectionType.UNKNOWN);
      expect(result.ast).toBeDefined();
    });
  });
});