//  BEGIN COPYRIGHT BLOCK
//  This Program is free software; you can redistribute it and/or modify it under
//  the terms of the GNU General Public License as published by the Free Software
//  Foundation; version 2 of the License.
//
//  This Program is distributed in the hope that it will be useful, but WITHOUT
//  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
//  FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License along with
//  this Program; if not, write to the Free Software Foundation, Inc., 59 Temple
//  Place, Suite 330, Boston, MA 02111-1307 USA.
//
//  Copyright (C) 2005 Red Hat, Inc.
//  All rights reserved.
//  END COPYRIGHT BLOCK


#include "coolkey-mgr.h"
#include "coolkey-api.h"
#include <stdio.h>

#include <dbus/dbus.h>
#include <stdbool.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>

enum
{
    PROP_0,
    PROP_DBUS_UNIQUE_NAME,
    PROP_CONFIG_DIR,
    N_PROPERTIES
};

static GParamSpec *obj_properties[N_PROPERTIES] = { NULL, };

#define COOLKEY_MGR_GET_PRIVATE(o)      \
    (G_TYPE_INSTANCE_GET_PRIVATE ((o), COOLKEY_TYPE_MGR, CoolkeyMgrPrivate))

struct _CoolkeyMgrPrivate {
    CoolKeyNotifyCallback cb;
    char *dbusUniqueName;
    char *configDir;
    DBusConnection* dbusConn; 
};

int notify(void *data,
   unsigned long aKeyType, const char *aKeyID, unsigned long aKeyState,
   unsigned long aData, const char *aStrData);

void dbus_notify_esc(const char* dbusUniqueName, 
     DBusConnection* dbusConn, 
     long aKeyType, const char* aKeyID,
     unsigned long aKeyState, 
     unsigned long aData, const char *sStrData);

// private members go in this structure
struct _CoolkeyMgr
{
  GObject parent_instance;
};

// implement CoolkeyMgr as a child of GObject
G_DEFINE_TYPE (CoolkeyMgr, coolkey_mgr, G_TYPE_OBJECT)

// instance cleanup
void
coolkey_mgr_cleanup(CoolkeyMgr *self)
{
    coolkey_destroy();
}
// instance initialization
static void
coolkey_mgr_init (CoolkeyMgr *self)
{
  /* initialize all public and private members to reasonable default values.
   * They are all automatically initialized to 0 to begin with. */

  CoolkeyMgrPrivate *priv = COOLKEY_MGR_GET_PRIVATE (self);
  priv->cb = notify;
}

static void
coolkey_mgr_set_property (GObject      *object,
                          guint         property_id,
                          const GValue *value,
                          GParamSpec   *pspec)
{
    CoolkeyMgrPrivate *priv = COOLKEY_MGR_GET_PRIVATE (object);

    switch (property_id) {
    case PROP_DBUS_UNIQUE_NAME:
        g_free (priv->dbusUniqueName);
        priv->dbusUniqueName = g_value_dup_string (value);
        break;

    case PROP_CONFIG_DIR:
        g_free(priv->configDir);
        priv->configDir = g_value_dup_string(value);
        break;

    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
        break;
    }
}

static void
coolkey_mgr_get_property (GObject    *object,
                          guint       property_id,
                          GValue     *value,
                          GParamSpec *pspec)
{
    CoolkeyMgrPrivate *priv = COOLKEY_MGR_GET_PRIVATE (object);

    switch (property_id) {
    case PROP_DBUS_UNIQUE_NAME:
        g_value_set_string (value, priv->dbusUniqueName);
        break;

    case PROP_CONFIG_DIR:
        g_value_set_string(value,priv->configDir);
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
        break;
    }
}

static void
coolkey_mgr_finalize (GObject *object)
{
    CoolkeyMgrPrivate *priv = COOLKEY_MGR_GET_PRIVATE (object);

    if(priv) {
        g_free (priv->dbusUniqueName);
        g_free(priv->configDir);
        if(priv->dbusConn) {
            dbus_connection_unref(priv->dbusConn);
            priv->dbusConn = NULL;
        }
    }

    G_OBJECT_CLASS (coolkey_mgr_parent_class)->finalize (object);
}

static void
coolkey_mgr_constructed (GObject *object)
{
    CoolkeyMgrPrivate *priv = COOLKEY_MGR_GET_PRIVATE (object);
    coolkey_init(priv->configDir,priv->cb,(void *) object,priv->dbusUniqueName);
}

// class initialization
static void
coolkey_mgr_class_init (CoolkeyMgrClass *klass)
{
    GObjectClass *object_class = G_OBJECT_CLASS (klass);

    object_class->set_property = coolkey_mgr_set_property;
    object_class->get_property = coolkey_mgr_get_property;

    object_class->finalize = coolkey_mgr_finalize;
    object_class->constructed = coolkey_mgr_constructed;

    obj_properties[PROP_DBUS_UNIQUE_NAME] =
        g_param_spec_string ("dbus_unique_name",
                             "Dbus_unique_name",
                             "The the name of the dbus service",
                             "World",
                             G_PARAM_READWRITE |
                             G_PARAM_CONSTRUCT);

   obj_properties[PROP_CONFIG_DIR] = 
    g_param_spec_string ("config_dir",
                             "Config_dir",
                             "User config directory path",
                             "World",
                             G_PARAM_READWRITE |
                             G_PARAM_CONSTRUCT);

    g_object_class_install_properties (object_class,
                                       N_PROPERTIES,
                                       obj_properties);

    g_type_class_add_private (object_class, sizeof (CoolkeyMgrPrivate));

}

/**
 * coolkey_mgr_new:
 *
 * Allocates a new #CoolkeyMgr
 *
 * Returns: A new #CoolkeyMgr
 *
 */
CoolkeyMgr *
coolkey_mgr_new (void)
{
  CoolkeyMgr *mgr;

  mgr = g_object_new(COOLKEY_TYPE_MGR, NULL);
  return mgr;
}

void 
coolkey_mgr_get_token_info(CoolkeyMgr* self, CoolkeyToken* token) {

    gchar *cuid = NULL;
    gchar *keyType = NULL; 
    int keyTypeInt = 0;
    tokenInfo * tInfo = NULL;

    g_object_get(token,"key_type", &keyType,NULL);

    g_object_get(token,"cuid", &cuid, NULL);

    if(keyType == NULL) {
       goto cleanup;
    }

    keyTypeInt = atoi(keyType);

    tInfo = coolkey_get_token_info(keyTypeInt, cuid);

    if(tInfo != NULL) {
        g_object_set(token,"atr", tInfo->atr, 
            "issuer_info", tInfo->issuerInfo,
            "issuer",tInfo->issuer, 
            "issued_to", tInfo->issuedTo, 
            "status", tInfo->status,NULL);
    }

cleanup:

    coolkey_free_token_info(tInfo);
    g_free (keyType);
    g_free (cuid);

    
}


/**
 * coolkey_mgr_speak:
 * @words: what you want it to say
 *
 * Returns: what the mgr actually says
 *
 */
gchar *
coolkey_mgr_speak (CoolkeyMgr* self, gchar *words)
{
  gchar *retval= 0;

  printf("Coolkey says: %s \n", words);

  return retval;
}

int notify(void *data,
   unsigned long aKeyType, const char *aKeyID, unsigned long aKeyState,
   unsigned long aData, const char *aStrData) {

    const char *noData = "no data";
    const char * finalStrData = NULL;
    const char * finalKeyID = NULL;

    /* printf("Inside coolkey_mgr.notify: aKeyType: %ld aKeyID %s aKeyState: %ld aData: %ld aStrData: %s \n",
        aKeyType, aKeyID, aKeyState, aData, aStrData);
    */

    CoolkeyMgr *mgr = (CoolkeyMgr *) data;
    CoolkeyMgrPrivate *priv = COOLKEY_MGR_GET_PRIVATE (mgr);

    if(aStrData == NULL) {
        finalStrData = noData;
    } else {
        finalStrData = aStrData;
    }
       
    if(aKeyID == NULL) {
        finalKeyID = noData;
    } else {
        finalKeyID = aKeyID;
    }
        
    dbus_notify_esc(priv->dbusUniqueName, priv->dbusConn,aKeyType, finalKeyID, aKeyState, aData, finalStrData);  

    /* use dbus to send this message to the UI layer */ 
    return 0;
}

void dbus_notify_esc(const char* dbusUniqueName, 
    DBusConnection* conn, 
    long aKeyType, 
    const char* aKeyID,
    unsigned long aKeyState, 
    unsigned long aData, 
    const char *strData)
{
   DBusMessage* msg = NULL;
   DBusMessageIter args;
   DBusError err;
   DBusPendingCall* pending = NULL;
   int ret;
   bool stat = false;
   dbus_uint32_t level;

   if(dbusUniqueName == NULL) {
       goto cleanup;
   }

   dbus_error_init(&err);

   conn = dbus_bus_get(DBUS_BUS_SESSION, &err);
   if (dbus_error_is_set(&err)) { 
      fprintf(stderr, "Connection Error (%s)\n", err.message); 
      dbus_error_free(&err);
   }
   if (NULL == conn) { 
      goto cleanup; 
   }
   msg = dbus_message_new_method_call(dbusUniqueName, // target for the method call
                                      "/com/jmagne/CoolKeyNotify", // object to call on
                                      "com.jmagne.CoolKeyNotify", // interface to call on
                                      "notifyCoolKeyEvent"); // method name
   if (NULL == msg) { 
       goto cleanup;
   }

   // append arguments
   dbus_message_iter_init_append(msg, &args);
   if (!dbus_message_iter_append_basic(&args, DBUS_TYPE_INT64 , &aKeyType)) {
       goto cleanup;
   }

   if (!dbus_message_iter_append_basic(&args, DBUS_TYPE_STRING , &aKeyID)) {
      goto cleanup; 
   }  
 
   if (!dbus_message_iter_append_basic(&args, DBUS_TYPE_UINT64 , &aKeyState)) {
      goto cleanup;
   }

   if (!dbus_message_iter_append_basic(&args, DBUS_TYPE_UINT64 , &aData)) {
      goto cleanup; 
   }

   if (!dbus_message_iter_append_basic(&args, DBUS_TYPE_STRING , &strData)) {
      goto cleanup; 
   }
 
   if (!dbus_connection_send_with_reply (conn, msg, &pending, -1)) { // -1 is default timeout
       goto cleanup;
   }
   if (NULL == pending) { 
       goto cleanup; 
   }
   dbus_connection_flush(conn);
   dbus_message_unref(msg);
   msg = NULL;
   
   // block until we recieve a reply
   dbus_pending_call_block(pending);

   // get the reply message
   msg = dbus_pending_call_steal_reply(pending);
   if (NULL == msg) {
       goto cleanup;
   }

   dbus_pending_call_unref(pending);
   pending = NULL;
   // we don't really care about what is in the the reply here 
   // This is a one way relationship

cleanup:

    if(msg != NULL) {
        dbus_message_unref(msg);
    }

    if(pending != NULL) {
       dbus_pending_call_unref(pending);
    }

   /* if(conn != NULL) {
       dbus_connection_close(conn);
   }
   */
}

