/**
 * Copyright (c) 2021 OceanBase
 * OceanBase CE is licensed under Mulan PubL v2.
 * You can use this software according to the terms and conditions of the Mulan PubL v2.
 * You may obtain a copy of Mulan PubL v2 at:
 *          http://license.coscl.org.cn/MulanPubL-2.0
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PubL v2 for more details.
 * This file contains implementation for eval_st_symdifference.
 */
#define USING_LOG_PREFIX SQL_ENG
#include "ob_expr_st_symdifference.h"
#include "sql/engine/expr/ob_geo_expr_utils.h"
#include "lib/geo/ob_geo_elevation_visitor.h"
#include "lib/geo/ob_geo_func_utils.h"

using namespace oceanbase::common;
using namespace oceanbase::sql;
namespace oceanbase
{
namespace sql
{
ObExprSTSymDifference::ObExprSTSymDifference(ObIAllocator &alloc)
    : ObFuncExprOperator(alloc, T_FUN_SYS_ST_SYMDIFFERENCE, N_ST_SYMDIFFERENCE, 2,
          VALID_FOR_GENERATED_COL, NOT_ROW_DIMENSION)
{}
ObExprSTSymDifference::~ObExprSTSymDifference()
{}
int ObExprSTSymDifference::calc_result_type2(ObExprResType &type, ObExprResType &type1,
    ObExprResType &type2, common::ObExprTypeCtx &type_ctx) const
{
  UNUSED(type_ctx);
  INIT_SUCC(ret);
  if (type1.get_type() == ObNullType) {
  } else if (!ob_is_geometry(type1.get_type()) && !ob_is_string_type(type1.get_type())) {
    type1.set_calc_type(ObVarcharType);
    type1.set_calc_collation_type(CS_TYPE_BINARY);
  }
  if (type2.get_type() == ObNullType) {
  } else if (!ob_is_geometry(type2.get_type()) && !ob_is_string_type(type2.get_type())) {
    type2.set_calc_type(ObVarcharType);
    type2.set_calc_collation_type(CS_TYPE_BINARY);
  }
    type.set_geometry();
    type.set_length((ObAccuracy::DDL_DEFAULT_ACCURACY[ObGeometryType]).get_length());
  return ret;
}
int ObExprSTSymDifference::process_input_geometry(omt::ObSrsCacheGuard &srs_guard, const ObExpr &expr, ObEvalCtx &ctx,
    MultimodeAlloctor &allocator, ObGeometry *&geo1, ObGeometry *&geo2, bool &is_null_res,
    const ObSrsItem *&srs)
{
  int ret = OB_SUCCESS;
  ObDatum *gis_datum1 = nullptr;
  ObDatum *gis_datum2 = nullptr;
  ObExpr *gis_arg1 = expr.args_[0];
  ObExpr *gis_arg2 = expr.args_[1];
  ObObjType input_type1 = gis_arg1->datum_meta_.type_;
  ObObjType input_type2 = gis_arg2->datum_meta_.type_;
  is_null_res = false;
  if (OB_FAIL(allocator.eval_arg(gis_arg1, ctx, gis_datum1)) || OB_FAIL(allocator.eval_arg(gis_arg2, ctx, gis_datum2))) {
    LOG_WARN("eval geo args failed", K(ret));
  } else if (gis_datum1->is_null() || gis_datum2->is_null()) {
    is_null_res = true;
  } else {
    ObGeoType type1;
    ObGeoType type2;
    uint32_t srid1;
    uint32_t srid2;
    ObString wkb1 = gis_datum1->get_string();
    ObString wkb2 = gis_datum2->get_string();
    bool is_geo1_valid = false;
    bool is_geo2_valid = false;
    if (OB_FAIL(ObTextStringHelper::read_real_string_data_with_copy(allocator,
            *gis_datum1,
            gis_arg1->datum_meta_,
            gis_arg1->obj_meta_.has_lob_header(),
            wkb1))) {
      LOG_WARN("fail to get real string data", K(ret), K(wkb1));
    } else if (OB_FAIL(ObTextStringHelper::read_real_string_data_with_copy(allocator,
                   *gis_datum2,
                   gis_arg2->datum_meta_,
                   gis_arg2->obj_meta_.has_lob_header(),
                   wkb2))) {
      LOG_WARN("fail to get real string data", K(ret), K(wkb2));
    } else if (FALSE_IT(allocator.set_baseline_size(wkb1.length() + wkb2.length()))) {
    } else if (OB_FAIL(ObGeoTypeUtil::get_type_srid_from_wkb(wkb1, type1, srid1))) {
      if (ret == OB_ERR_GIS_INVALID_DATA) {
        LOG_USER_ERROR(OB_ERR_GIS_INVALID_DATA, N_ST_SYMDIFFERENCE);
      }
      LOG_WARN("get type and srid from wkb failed", K(wkb1), K(ret));
    } else if (OB_FAIL(ObGeoTypeUtil::get_type_srid_from_wkb(wkb2, type2, srid2))) {
      if (ret == OB_ERR_GIS_INVALID_DATA) {
        LOG_USER_ERROR(OB_ERR_GIS_INVALID_DATA, N_ST_SYMDIFFERENCE);
      }
      LOG_WARN("get type and srid from wkb failed", K(wkb2), K(ret));
    } else if (srid1 != srid2) {
      ret = OB_ERR_GIS_DIFFERENT_SRIDS;
      LOG_WARN("srid not the same", K(ret), K(srid1), K(srid2));
      LOG_USER_ERROR(OB_ERR_GIS_DIFFERENT_SRIDS, N_ST_SYMDIFFERENCE, srid1, srid2);
    } else if (OB_FAIL(ObGeoExprUtils::get_srs_item(
                   ctx, srs_guard, wkb1, srs, true, N_ST_SYMDIFFERENCE))) {
      LOG_WARN("fail to get srs item", K(ret), K(wkb1));
    } else if (OB_FAIL(ObGeoExprUtils::build_geometry(allocator,
                   wkb1,
                   geo1,
                   srs,
                   N_ST_SYMDIFFERENCE,
                   ObGeoBuildFlag::GEO_ALLOW_3D_DEFAULT | GEO_RESERVE_3D | GEO_NOT_COPY_WKB))) {
      LOG_WARN("get first geo by wkb failed", K(ret));
    } else if (OB_FAIL(ObGeoExprUtils::build_geometry(allocator,
                   wkb2,
                   geo2,
                   srs,
                   N_ST_SYMDIFFERENCE,
                   ObGeoBuildFlag::GEO_ALLOW_3D_DEFAULT | GEO_RESERVE_3D | GEO_NOT_COPY_WKB))) {
      LOG_WARN("get second geo by wkb failed", K(ret));
    }
  }
  return ret;
}

int ObExprSTSymDifference::eval_st_symdifference(const ObExpr &expr, ObEvalCtx &ctx, ObDatum &res)
{
  int ret = OB_SUCCESS;
  bool is_geo1_empty = false;
  bool is_geo2_empty = false;
  ObGeometry *geo1_3d = nullptr;
  ObGeometry *geo2_3d = nullptr;
  bool is_null_res = false;
  omt::ObSrsCacheGuard srs_guard;
  const ObSrsItem *srs = nullptr;
  ObEvalCtx::TempAllocGuard tmp_alloc_g(ctx);
  uint64_t tenant_id = ObMultiModeExprHelper::get_tenant_id(ctx.exec_ctx_.get_my_session());
  MultimodeAlloctor temp_allocator(tmp_alloc_g.get_allocator(), expr.type_, tenant_id, ret, N_ST_SYMDIFFERENCE);
  ObGeometry *diff_res = nullptr;
  if (OB_FAIL(
          process_input_geometry(srs_guard, expr, ctx, temp_allocator, geo1_3d, geo2_3d, is_null_res, srs))) {
    LOG_WARN("fail to process input geometry", K(ret));
  }
  ObGeoBoostAllocGuard guard(tenant_id);
  lib::MemoryContext *mem_ctx = nullptr;
  if (OB_SUCC(ret) && !is_null_res) {
    ObGeometry *geo1 = nullptr;
    ObGeometry *geo2 = nullptr;
    bool is_3d_geo1 = ObGeoTypeUtil::is_3d_geo_type(geo1_3d->type());
    bool is_3d_geo2 = ObGeoTypeUtil::is_3d_geo_type(geo2_3d->type());
    if ((is_3d_geo1 && !is_3d_geo2) || (!is_3d_geo1 && is_3d_geo2)) {
      ret = OB_ERR_GIS_INVALID_DATA;
      LOG_WARN("mixed dimension geometries", K(ret), K(is_3d_geo1), K(is_3d_geo2));
    } else if (is_3d_geo1) {
      if (OB_FAIL(ObGeoTypeUtil::convert_geometry_3D_to_2D(
              srs, temp_allocator, geo1_3d, ObGeoBuildFlag::GEO_DEFAULT, geo1))) {
        LOG_WARN("fail to convert 3D geometry to 2D", K(ret));
      }
    } else {
      geo1 = geo1_3d;
    }
    if (OB_FAIL(ret)) {
    } else if (is_3d_geo2) {
      if (OB_FAIL(ObGeoTypeUtil::convert_geometry_3D_to_2D(
              srs, temp_allocator, geo2_3d, ObGeoBuildFlag::GEO_DEFAULT, geo2))) {
        LOG_WARN("fail to convert 3D geometry to 2D", K(ret));
      }
    } else {
      geo2 = geo2_3d;
    }
    if (OB_FAIL(ret)) {
    } else if (OB_FAIL(guard.init())) {
      LOG_WARN("fail to init geo allocator guard", K(ret));
    } else if (OB_ISNULL(mem_ctx = guard.get_memory_ctx())) {
      ret = OB_ERR_NULL_VALUE;
      LOG_WARN("fail to get mem ctx", K(ret));
    }
    if (OB_SUCC(ret)) {
      ObGeoEvalCtx gis_context(*mem_ctx, srs);
      bool is_empty_res = false;
      if (OB_FAIL(gis_context.append_geo_arg(geo1)) || OB_FAIL(gis_context.append_geo_arg(geo2))) {
        LOG_WARN("build gis context failed", K(ret), K(gis_context.get_geo_count()));
      } else if (OB_FAIL(ObGeoFunc<ObGeoFuncType::SymDifference>::geo_func::eval(
                     gis_context, diff_res))) {
        LOG_WARN("eval st symdifference failed", K(ret));
        ObGeoExprUtils::geo_func_error_handle(ret, N_ST_SYMDIFFERENCE);
      } else if (OB_FAIL(ObGeoExprUtils::check_empty(diff_res, is_empty_res))) {
        LOG_WARN("check geo empty failed", K(ret));
      } else if (is_empty_res) {
        // 2D return GEOMETRYCOLLECTION EMPTY, 3D return GEOMETRYCOLLECTION Z EMPTY
        if (OB_FAIL(ObGeoExprUtils::create_3D_empty_collection(temp_allocator, geo1->get_srid(), is_3d_geo1,
                      geo1->crs() == ObGeoCRS::Geographic, diff_res))) {
          LOG_WARN("fail to create 3D empty collection", K(ret));
        }
      } else if (OB_FAIL(ObGeoTypeUtil::remove_duplicate_geo(diff_res, *mem_ctx, srs))) {
        // should not do simplify in symdifference functor, it may affect
        // ObGeoFuncUtils::ob_geo_gc_union
        LOG_WARN("fail to simplify result", K(ret));
      } else if (is_3d_geo1 && is_3d_geo2) {
        // populate Z coordinates
        ObGeoElevationVisitor visitor(*mem_ctx, srs);
        ObGeometry *diff_res_bin = nullptr;
        if (OB_FAIL(visitor.init(*geo1_3d, *geo2_3d))) {
          LOG_WARN("fail to init elevation visitor", K(ret));
        } else if (OB_FAIL(
                       ObGeoTypeUtil::tree_to_bin(temp_allocator, diff_res, diff_res_bin, srs))) {
          LOG_WARN("fail to do tree to bin", K(ret));
        } else if (OB_FAIL(diff_res_bin->do_visit(visitor))) {
          LOG_WARN("fail to do elevation visitor", K(ret));
        } else if (OB_FAIL(visitor.get_geometry_3D(diff_res))) {
          LOG_WARN("failed get geometry 3D", K(ret));
        }
      }
    }
  }
  if (OB_FAIL(ret)) {
  } else if (is_null_res) {
    res.set_null();
  } else {
    ObString res_wkb;
    if (OB_FAIL(ObGeoExprUtils::geo_to_wkb(*diff_res, expr, ctx, srs, res_wkb))) {
      LOG_WARN("failed to write geometry to wkb", K(ret));
    } else {
      res.set_string(res_wkb);
    }
  }
  if (mem_ctx != nullptr) {
    temp_allocator.add_ext_used((*mem_ctx)->arena_used());
  }
  return ret;
}
int ObExprSTSymDifference::cg_expr(
    ObExprCGCtx &expr_cg_ctx, const ObRawExpr &raw_expr, ObExpr &rt_expr) const
{
  UNUSED(expr_cg_ctx);
  UNUSED(raw_expr);
  rt_expr.eval_func_ = eval_st_symdifference;
  return OB_SUCCESS;
}
}  // namespace sql
}  // namespace oceanbase
