/**
 * Copyright (c) 2021 OceanBase
 * OceanBase CE is licensed under Mulan PubL v2.
 * You can use this software according to the terms and conditions of the Mulan PubL v2.
 * You may obtain a copy of Mulan PubL v2 at:
 *          http://license.coscl.org.cn/MulanPubL-2.0
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PubL v2 for more details.
 */

#define USING_LOG_PREFIX RS

#include "ob_ddl_service.h"
#include "share/schema/ob_schema_printer.h"
#include "share/schema/ob_schema_service_sql_impl.h"
#include "share/ob_tablet_autoincrement_service.h"
#include "share/ob_primary_zone_util.h"
#include "share/ob_index_builder_util.h"
#include "share/sequence/ob_sequence_ddl_proxy.h"
#include "share/ob_global_stat_proxy.h"
#include "share/ob_service_epoch_proxy.h"
#include "rootserver/standby/ob_standby_service.h" // ObStandbyService
#include "sql/resolver/ddl/ob_ddl_resolver.h"
#include "sql/resolver/expr/ob_raw_expr_modify_column_name.h"
#include "rootserver/ob_index_builder.h"
#include "rootserver/ob_ddl_sql_generator.h"
#include "rootserver/ob_ddl_help.h"
#include "rootserver/ob_root_service.h"
#include "rootserver/ob_vertical_partition_builder.h"
#include "rootserver/freeze/ob_major_freeze_helper.h"
#include "rootserver/ob_tenant_thread_helper.h"//get_zone_priority
#include "src/sql/engine/px/ob_dfo.h"
#include "observer/omt/ob_tenant_timezone_mgr.h"
#include "rootserver/ob_table_creator.h"
#include "rootserver/ob_balance_group_ls_stat_operator.h"
#include "rootserver/ob_primary_ls_service.h" // ObDupLSCreateHelper
#include "rootserver/ob_tablet_drop.h"
#include "share/schema/ob_context_ddl_proxy.h"
#include "share/ob_global_context_operator.h"
#include "share/ls/ob_ls_creator.h"
#include "ob_lob_meta_builder.h"
#include "ob_lob_piece_builder.h"
#include "share/ls/ob_ls_life_manager.h"//ObLSLifeAgentManager
#include "storage/ddl/ob_ddl_lock.h"
#include "logservice/data_dictionary/ob_data_dict_storager.h" // ObDataDictStorage
#include "share/schema/ob_mlog_info.h"
#ifdef OB_BUILD_ARBITRATION
#include "share/arbitration_service/ob_arbitration_service_table_operator.h"
#include "share/arbitration_service/ob_arbitration_service_utils.h" // ObArbitrationServiceUtils
#endif
#ifdef OB_BUILD_ORACLE_PL
#include "pl/sys_package/ob_dbms_audit_mgmt.h" // ObDbmsAuditMgmt
#endif
#include "storage/tx_storage/ob_ls_map.h"
#include "storage/tx_storage/ob_ls_service.h"
#include "share/backup/ob_log_restore_config.h"//ObLogRestoreSourceServiceConfigParser
#include "storage/tablelock/ob_lock_inner_connection_util.h"
#include "storage/compaction/ob_compaction_schedule_util.h"
#include "share/schema/ob_mview_info.h"
#include "storage/vector_index/ob_vector_index_sched_job_utils.h"
#include "rootserver/restore/ob_tenant_clone_util.h"
#include "rootserver/ob_split_partition_helper.h"
#include "ob_disaster_recovery_task_utils.h" // DisasterRecoveryUtils
#include "rootserver/mview/ob_mview_dependency_service.h"
#include "rootserver/parallel_ddl/ob_ddl_helper.h"
#include "storage/ddl/ob_ddl_alter_auto_part_attr.h"
#include "src/share/ob_vec_index_builder_util.h"
#include "rootserver/direct_load/ob_direct_load_partition_exchange.h"
#include "share/schema/ob_mview_info.h"
#include "sql/resolver/mv/ob_mv_provider.h"
#include "rootserver/mview/ob_mview_alter_service.h"
#include "storage/tablet/ob_tablet_binding_helper.h"


namespace oceanbase
{
using namespace common;
using namespace share;
using namespace obrpc;
using namespace storage;
using namespace palf;
namespace rootserver
{

#define GRANT_SYS_ROLE_NUM 2       /* len of role array is 2 */
#define GRANT_ROLE_MIN_ROLE_NUM 3  /* min len of role array is 3 */

ObDDLService::ObDDLService()
  : inited_(false),
    stopped_(false),
    rpc_proxy_(NULL),
    common_rpc_(NULL),
    sql_proxy_(NULL),
    schema_service_(NULL),
    lst_operator_(NULL),
    snapshot_mgr_(NULL),
    ddl_lock_(),
    index_name_checker_(),
    non_partitioned_tablet_allocator_()
{
}

int ObDDLService::init(obrpc::ObSrvRpcProxy &rpc_proxy,
                       obrpc::ObCommonRpcProxy &common_rpc,
                       common::ObMySQLProxy &sql_proxy,
                       share::schema::ObMultiVersionSchemaService &schema_service,
                       share::ObLSTableOperator &lst_operator,
                       ObSnapshotInfoManager &snapshot_mgr,
                       ObTenantDDLService &tenant_ddl_service)
{
  int ret = OB_SUCCESS;
  if (inited_) {
    ret = OB_INIT_TWICE;
    LOG_WARN("init twice", KR(ret));
  } else if (OB_FAIL(index_name_checker_.init(sql_proxy))) {
    LOG_WARN("fail to init index name checker", KR(ret));
  } else if (OB_FAIL(non_partitioned_tablet_allocator_.init(sql_proxy))) {
    LOG_WARN("fail to init non partitioned tablet allocator", KR(ret));
  } else {
    rpc_proxy_ = &rpc_proxy;
    common_rpc_ = &common_rpc;
    sql_proxy_ = &sql_proxy;
    schema_service_ = &schema_service;
    lst_operator_ = &lst_operator;
    snapshot_mgr_ = &snapshot_mgr;
    tenant_ddl_service_ = &tenant_ddl_service;
    stopped_ = false;
    inited_ = true;
  }
  return ret;
}

int ObDDLService::get_tenant_schema_guard_with_version_in_inner_table(const uint64_t tenant_id, ObSchemaGetterGuard &schema_guard)
{
  int ret = OB_SUCCESS;
  bool is_restore = false;
  bool use_local = false;
  int64_t version_in_inner_table = OB_INVALID_VERSION;
  ObRefreshSchemaStatus schema_status;
  if (OB_INVALID_TENANT_ID == tenant_id) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("invalid tenant_id", K(ret), K(tenant_id));
  } else if (OB_ISNULL(GCTX.schema_service_) || OB_ISNULL(GCTX.sql_proxy_)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("pointer is null", K(ret), KP(GCTX.schema_service_), KP(GCTX.sql_proxy_));
  } else if (OB_FAIL(GCTX.schema_service_->check_tenant_is_restore(NULL, tenant_id, is_restore))) {
    LOG_WARN("fail to check tenant is restore", KR(ret), K(tenant_id));
  } else if (is_restore && OB_SYS_TENANT_ID != tenant_id) {
    ObSchemaStatusProxy *schema_status_proxy = GCTX.schema_status_proxy_;
    if (OB_ISNULL(schema_status_proxy)) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("schema_status_proxy is null", KR(ret));
    } else if (OB_FAIL(schema_status_proxy->get_refresh_schema_status(tenant_id, schema_status))) {
      LOG_WARN("failed to get tenant refresh schema status", KR(ret), K(tenant_id));
    } else if (OB_INVALID_VERSION == schema_status.readable_schema_version_) {
      // The second of physical recovery, after reset schema status, modify_schema can be modified
      use_local = false;
    } else if (is_restore) {
      ret = OB_NOT_SUPPORTED;
      LOG_WARN("tenant is still restoring, ddl not supported", KR(ret), K(tenant_id), K(schema_status));
    }
  }
  if (OB_FAIL(ret)) {
  } else if (use_local) {
    // Only for the failover/switchover stage of the standalone cluster
    if (OB_FAIL(GCTX.schema_service_->get_tenant_schema_guard(tenant_id, schema_guard))) {
      LOG_WARN("fail to get schema guard", K(ret), K(tenant_id));
    } else {
      LOG_INFO("use local tenant schema guard", K(ret), K(tenant_id));
    }
  } else {
    // 1. the normal tenants do DDL in primary cluster.
    // 2. restore tenant is in modify_schema stage in primary cluster.
    schema_status.tenant_id_ = tenant_id;
    // Ensure that the user tenant schema is updated to the latest
    if (OB_FAIL(GCTX.schema_service_->get_schema_version_in_inner_table(*GCTX.sql_proxy_,
            schema_status, version_in_inner_table))) {
      LOG_WARN("fail to get latest schema version in inner table", K(ret));
    } else if (OB_FAIL(GCTX.schema_service_->get_tenant_schema_guard(tenant_id, schema_guard, version_in_inner_table))) {
      if (OB_SCHEMA_EAGAIN == ret) {
        int t_ret = OB_SUCCESS;
        ObArray<uint64_t> tenant_ids;
        if (OB_SUCCESS != (t_ret = tenant_ids.push_back(tenant_id))) {
          LOG_WARN("fail to push back tenant_id", K(t_ret), K(tenant_id));
        } else if (OB_SUCCESS != (t_ret = GCTX.schema_service_->refresh_and_add_schema(tenant_ids))) {
          LOG_WARN("fail to refresh and add schema", K(t_ret), K(tenant_id));
        } else if (OB_FAIL(GCTX.schema_service_->get_tenant_schema_guard(tenant_id, schema_guard, version_in_inner_table))) {
          LOG_WARN("fail to retry get schema guard", K(ret), K(tenant_id), K(version_in_inner_table));
        }
      } else {
        LOG_WARN("get schema manager failed!", K(ret));
      }
    }
  }
  return ret;
}

int ObDDLService::get_tenant_schema_guard_with_version_in_inner_table(
    const uint64_t src_tenant_id,
    const uint64_t dst_tenant_id,
    share::schema::ObSchemaGetterGuard &hold_buf_src_tenant_schema_guard,
    share::schema::ObSchemaGetterGuard &hold_buf_dst_tenant_schema_guard,
    share::schema::ObSchemaGetterGuard *&src_tenant_schema_guard,
    share::schema::ObSchemaGetterGuard *&dst_tenant_schema_guard)
{
  int ret = OB_SUCCESS;
  src_tenant_schema_guard = nullptr;
  dst_tenant_schema_guard = nullptr;
  if (OB_UNLIKELY(OB_INVALID_TENANT_ID == src_tenant_id || OB_INVALID_TENANT_ID == dst_tenant_id)) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("invalid tenant_id", K(ret), K(src_tenant_id), K(dst_tenant_id));
  } else if (OB_FAIL(get_tenant_schema_guard_with_version_in_inner_table(dst_tenant_id, hold_buf_dst_tenant_schema_guard))) {
    LOG_WARN("get tenant schema guard failed", K(dst_tenant_id));
  } else if (src_tenant_id == dst_tenant_id) {
    src_tenant_schema_guard = &hold_buf_dst_tenant_schema_guard;
    dst_tenant_schema_guard = &hold_buf_dst_tenant_schema_guard;
  } else {
    if (OB_FAIL(get_tenant_schema_guard_with_version_in_inner_table(src_tenant_id, hold_buf_src_tenant_schema_guard))) {
      LOG_WARN("get tenant schema guard failed", K(src_tenant_id));
    } else {
      src_tenant_schema_guard = &hold_buf_src_tenant_schema_guard;
      dst_tenant_schema_guard = &hold_buf_dst_tenant_schema_guard;
    }
  }
  if (OB_SUCC(ret)) {
    if (OB_UNLIKELY(nullptr == src_tenant_schema_guard || nullptr == dst_tenant_schema_guard)) {
      ret = OB_TENANT_NOT_EXIST;
      LOG_WARN("tenant not exist", K(ret), K(src_tenant_id), K(dst_tenant_id), KP(src_tenant_schema_guard), KP(dst_tenant_schema_guard));
    }
  }
  return ret;
}

int ObDDLService::check_tenant_in_alter_locality(
    const uint64_t tenant_id,
    bool &in_alter_locality)
{
  int ret = OB_SUCCESS;
  share::schema::ObSchemaGetterGuard schema_guard;
  const share::schema::ObTenantSchema *tenant_schema = NULL;
  in_alter_locality = false;
  if (OB_FAIL(check_inner_stat())) {
    LOG_WARN("not init", K(ret));
  } else if (OB_UNLIKELY(OB_INVALID_ID == tenant_id)) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("invalid argument", K(ret), K(tenant_id));
  } else if (OB_FAIL(get_tenant_schema_guard_with_version_in_inner_table(OB_SYS_TENANT_ID, schema_guard))) {
    LOG_WARN("fail to get schema guard", K(ret));
  } else if (OB_FAIL(schema_guard.get_tenant_info(tenant_id, tenant_schema))) {
    LOG_WARN("tenant not exists", K(ret), K(tenant_id));
  } else if (OB_UNLIKELY(NULL == tenant_schema)) {
    ret = OB_TENANT_NOT_EXIST;
    LOG_WARN("tenant not exist", K(ret), K(tenant_id));
  } else if (!tenant_schema->get_previous_locality_str().empty()) {
    // previous locality is not null, alter locality is processing
    in_alter_locality = true;
  } else {
    // do nothing
  }
  return ret;
}

int ObDDLService::create_user_tables(
    const bool if_not_exist,
    const ObString &ddl_stmt_str,
    const ObErrorInfo &error_info,
    ObIArray<ObTableSchema> &table_schemas,
    ObSchemaGetterGuard &schema_guard,
    const obrpc::ObSequenceDDLArg &sequence_ddl_arg,
    const uint64_t last_replay_log_id,
    const ObIArray<ObDependencyInfo> *dep_infos,
    ObIArray<ObMockFKParentTableSchema> &mock_fk_parent_table_schema_array,
    int64_t &ddl_task_id)
{
  int ret = OB_SUCCESS;
  ddl_task_id = 0;
  RS_TRACE(create_user_tables_begin);
  uint64_t tenant_id = OB_INVALID_TENANT_ID;
  bool have_duplicate_table = false;
  bool is_compatible = false;
  if (OB_FAIL(check_inner_stat())) {
    LOG_WARN("not init", K(ret));
  } else if (table_schemas.count() < 1) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("table_schemas have no element", K(ret));
  } else {
    tenant_id = table_schemas.at(0).get_tenant_id();
    have_duplicate_table = table_schemas.at(0).is_duplicate_table();
    // for checking unique index name duplicate when create user table in oracle mode
    bool is_oracle_mode = false;
    if (OB_FAIL(table_schemas.at(0).check_if_oracle_compat_mode(is_oracle_mode))) {
      LOG_WARN("fail to check if tenant mode is oracle mode", K(ret));
    } else if (is_oracle_mode) { // oracle mode
      uint64_t i = 1;
      bool idx_name_is_exist = false;
      for (i = 1; OB_SUCC(ret) && !idx_name_is_exist && i < table_schemas.count(); ++i) {
        if (table_schemas.at(i).is_index_table()) {
          if (OB_FAIL(table_schemas.at(i).generate_origin_index_name())) {
            LOG_WARN("generate origin index name failed", K(ret), K(table_schemas.at(i).get_table_name_str()));
          } else if (OB_FAIL(check_index_table_exist(table_schemas.at(i).get_tenant_id(),
                                                     table_schemas.at(i).get_database_id(),
                                                     table_schemas.at(0).get_table_id(),
                                                     table_schemas.at(i).get_origin_index_name_str(),
                                                     schema_guard,
                                                     idx_name_is_exist))) {
            LOG_WARN("failed to check index table", K(ret));
          }
        }
      }
      if (OB_SUCC(ret)) {
        if(idx_name_is_exist && ((--i) < table_schemas.count())) {
          ret = OB_ERR_KEY_NAME_DUPLICATE;
          LOG_USER_ERROR(OB_ERR_KEY_NAME_DUPLICATE, table_schemas.at(i).get_origin_index_name_str().length(),
                         table_schemas.at(i).get_origin_index_name_str().ptr());
          LOG_WARN("duplicate index name", K(ret), K(table_schemas.at(i).get_table_name_str()));
        }
      }
    } else {
      // mysql mode
      // do nothing, only oracle mode need this
    }
  }

  if (OB_FAIL(ret)) {
    //do nothing
  } else if (!have_duplicate_table) {
    // do nothing
  } else if (OB_FAIL(ObShareUtil::check_compat_version_for_readonly_replica(tenant_id, is_compatible))) {
    LOG_WARN("fail to check compat version for duplicate log stream", KR(ret), K(tenant_id));
  } else if (!is_compatible) {
    ret = OB_NOT_SUPPORTED;
    LOG_WARN("duplicate table is not supported below 4.2", KR(ret), K(tenant_id));
    LOG_USER_ERROR(OB_NOT_SUPPORTED, "create duplicate table below 4.2");
  } else if (!is_user_tenant(tenant_id)) {
    ret = OB_NOT_SUPPORTED;
    LOG_WARN("not user tenant, create duplicate table not supported", KR(ret), K(tenant_id));
    LOG_USER_ERROR(OB_NOT_SUPPORTED, "not user tenant, create duplicate table");
  }

  if (OB_FAIL(ret)) {
    //do nothing
  } else if (OB_FAIL(create_tables_in_trans(if_not_exist, ddl_stmt_str, error_info, table_schemas,
                                            sequence_ddl_arg,
                                            last_replay_log_id, dep_infos, mock_fk_parent_table_schema_array,
                                            ddl_task_id))) {
    LOG_WARN("create_tables_in_trans failed", K(ret));
  }
  RS_TRACE_EXT(create_user_tables_end, OB_ID(ret), ret);
  return ret;
}

int ObDDLService::create_inner_expr_index(ObMySQLTransaction &trans,
                                          const ObTableSchema &orig_table_schema,
                                          const uint64_t tenant_data_version,
                                          ObTableSchema &new_table_schema,
                                          ObIArray<ObColumnSchemaV2*> &new_columns,
                                          ObTableSchema &index_schema)
{
  int ret = OB_SUCCESS;
  ObSchemaGetterGuard schema_guard;
  uint64_t tenant_id = new_table_schema.get_tenant_id();
  int64_t refreshed_schema_version = 0;
  if (OB_FAIL(check_inner_stat())) {
    LOG_WARN("variable is not init");
  } else if (OB_FAIL(get_tenant_schema_guard_with_version_in_inner_table(tenant_id, schema_guard))) {
    LOG_WARN("fail to get schema guard with version in inner table", K(ret), K(tenant_id));
  } else if (OB_FAIL(schema_guard.get_schema_version(tenant_id, refreshed_schema_version))) {
    LOG_WARN("failed to get tenant schema version", KR(ret), K(tenant_id));
  } else if (OB_FAIL(check_table_exist(index_schema))) {
    if (OB_ERR_TABLE_EXIST != ret) {
       LOG_WARN("check_table_exist failed", K(index_schema), K(ret));
    }
  } else {
    ObSchemaService *schema_service = schema_service_->get_schema_service();
    uint64_t new_table_id = index_schema.get_table_id();
    if (OB_UNLIKELY(NULL == schema_service)) {
      ret = OB_ERR_SYS;
      LOG_WARN("schema service must be not null", K(ret));
    } else if (OB_FAIL(schema_service->fetch_new_table_id(
            index_schema.get_tenant_id(), new_table_id))) {
      LOG_WARN("failed to fetch_new_table_id", K(ret));
    } else {
      index_schema.set_table_id(new_table_id);
    }
  }
  if (OB_SUCC(ret)) {
    ObDDLOperator ddl_operator(*schema_service_, *sql_proxy_);
    for (int64_t i = 0; OB_SUCC(ret) && i < new_columns.count(); ++i) {
      ObColumnSchemaV2 *new_column_schema = new_columns.at(i);
      if (OB_ISNULL(new_column_schema)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("new column schema is null");
      } else if (OB_FAIL(ddl_operator.insert_single_column(
              trans, new_table_schema, *new_column_schema))) {
        LOG_WARN("failed to create table schema, ", K(ret));
      }
    }
    if (OB_SUCC(ret)) {
      if (OB_FAIL(ddl_operator.alter_table_options(schema_guard,
                                                   new_table_schema,
                                                   orig_table_schema,
                                                   false,
                                                   trans))) {
        LOG_WARN("alter table options failed", K(ret), K(new_table_schema));
      } else if (OB_FAIL(ddl_operator.create_table(
              index_schema, trans, nullptr/*ddl_stmt_str*/, true, false))) {
        // record the create index operation when index enables rather than schema generates.
        LOG_WARN("failed to create index schema", K(ret));
      }
    }

		if (OB_SUCC(ret)) {
      int64_t last_schema_version = OB_INVALID_VERSION;
      if (OB_FAIL(get_last_schema_version(last_schema_version))) {
        LOG_WARN("fail to get last schema_version", KR(ret));
      } else if (OB_FAIL(ddl_operator.insert_ori_schema_version(trans,
                tenant_id, index_schema.get_table_id(), last_schema_version))) {
        LOG_WARN("failed to insert_ori_schema_version!", K(ret));
      }
      if (OB_SUCC(ret) && index_schema.has_tablet()
          && OB_FAIL(create_index_tablet(index_schema, trans, schema_guard,
                                         true/*need_check_tablet_cnt*/, tenant_data_version))) {
        LOG_WARN("fail to create_index_tablet", KR(ret), K(index_schema));
      }
    }
  }
  return ret;
}

int ObDDLService::create_global_index(
    ObMySQLTransaction &trans,
    const obrpc::ObCreateIndexArg &arg,
    const share::schema::ObTableSchema &table_schema,
    const uint64_t tenant_data_version,
    share::schema::ObTableSchema &index_schema)
{
  int ret = OB_SUCCESS;
  if (OB_FAIL(check_inner_stat())) {
    LOG_WARN("variable is not init", K(ret));
  } else if (OB_FAIL(table_schema.check_create_index_on_hidden_primary_key(index_schema))) {
    LOG_WARN("fail to check create global index on table", K(ret), K(index_schema));
  } else if (OB_FAIL(create_index_table(arg, tenant_data_version, index_schema, trans))) {
    LOG_WARN("fail to create global index", K(ret));
  }
  return ret;
}

int ObDDLService::create_global_inner_expr_index(
    ObMySQLTransaction &trans,
    const share::schema::ObTableSchema &orig_table_schema,
    const uint64_t tenant_data_version,
    share::schema::ObTableSchema &new_table_schema,
    common::ObIArray<share::schema::ObColumnSchemaV2*> &new_columns,
    share::schema::ObTableSchema &index_schema)
{
  int ret = OB_SUCCESS;
  if (OB_FAIL(check_inner_stat())) {
    LOG_WARN("variable is not init", K(ret));
  } else if (OB_FAIL(create_inner_expr_index(trans, orig_table_schema,
          tenant_data_version, new_table_schema, new_columns, index_schema))) {
    LOG_WARN("fail to create inner expr index", K(ret));
  }
  return ret;
}

// create_index_table is used by create index
int ObDDLService::create_index_table(
  const obrpc::ObCreateIndexArg &arg,
  const uint64_t tenant_data_version,
  ObTableSchema &table_schema,
  ObMySQLTransaction &sql_trans)
{
  int ret = OB_SUCCESS;
  uint64_t new_table_id = table_schema.get_table_id(); // You can specify the data table id to build an index
  uint64_t tenant_id = table_schema.get_tenant_id();
  ObSchemaService *schema_service = NULL;
  const ObDatabaseSchema *database_schema = NULL;
  ObSchemaGetterGuard schema_guard;

  if (OB_FAIL(get_tenant_schema_guard_with_version_in_inner_table(tenant_id, schema_guard))) {
    LOG_WARN("fail to get schema guard with version in inner table", K(ret), K(tenant_id));
  } else if (OB_FAIL(check_inner_stat())) {
    LOG_WARN("variable is not init");
  } else if (OB_FAIL(check_table_exist(table_schema))) {
    if (OB_ERR_TABLE_EXIST != ret) {
       LOG_WARN("check_table_exist failed", K(table_schema), K(ret));
    }
  } else {
    schema_service = schema_service_->get_schema_service();
    if (OB_ISNULL(schema_service)) {
      ret = OB_ERR_SYS;
      LOG_ERROR("schema_service must not null");
    } else if (OB_FAIL(schema_service->fetch_new_table_id(
            table_schema.get_tenant_id(), new_table_id))) {
      LOG_WARN("fail to fetch new table id", K(ret));
    } else {
      table_schema.set_table_id(new_table_id);
      // zone_list, resource_pool_list, primary_zone not set, copy from all_database_schema
      if (OB_FAIL(schema_guard.get_database_schema(
          tenant_id, table_schema.get_database_id(), database_schema))) {
        LOG_WARN("get_database_schema failed", K(tenant_id),
            K(table_schema.get_database_id()), K(ret));
      } else if (OB_ISNULL(database_schema)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("database schema should not be null", K(ret));
      } else if (!arg.is_inner_ && database_schema->is_in_recyclebin()) {
        ret = OB_ERR_OPERATION_ON_RECYCLE_OBJECT;
        LOG_WARN("can not create index in recyclebin", K(ret), K(*database_schema));
      } else {} // no more to do
    }
    if (OB_SUCC(ret)) {
      // For create index operation, generate ddl_stmt_str when index enables, but
      // for alter table add index operation, keep generating ddl_stmt_str same as 3.x while generating index schema.
      if (OB_FAIL(create_index_or_mlog_table_in_trans(table_schema,
              nullptr/* ddl_stmt_str */, &sql_trans, schema_guard, true/*need_check_tablet_cnt*/, tenant_data_version))) {
        LOG_WARN("create_table_in_trans failed", KR(ret), K(arg), K(table_schema));
      }
    }
  }

  return ret;
}

int ObDDLService::create_mlog_table(
    ObMySQLTransaction &sql_trans,
    const obrpc::ObCreateMLogArg &arg,
    const uint64_t tenant_data_version,
    ObSchemaGetterGuard &schema_guard,
    ObTableSchema &table_schema)
{
  int ret = OB_SUCCESS;
  uint64_t new_table_id = table_schema.get_table_id();
  uint64_t tenant_id = table_schema.get_tenant_id();
  ObSchemaService *schema_service = nullptr;
  const ObDatabaseSchema *database_schema = nullptr;
  bool is_oracle_mode = false;
  ObArray<ObSchemaType> conflict_schema_types;
  if (OB_FAIL(check_inner_stat())) {
    LOG_WARN("variable is not init", KR(ret));
  } else if (OB_FAIL(check_table_exist(table_schema))) {
    if (OB_ERR_TABLE_EXIST != ret) {
      LOG_WARN("failed to check table exist", KR(ret), K(table_schema));
    } else {
      ret = OB_ERR_MLOG_EXIST;
      LOG_WARN("a materialized view log already exists on table",
          KR(ret), K(arg.table_name_));
      LOG_USER_ERROR(OB_ERR_MLOG_EXIST, to_cstring(arg.table_name_));
    }
  } else if (OB_FAIL(ObCompatModeGetter::check_is_oracle_mode_with_tenant_id(
      tenant_id, is_oracle_mode))) {
    LOG_WARN("fail to check is oracle mode with tenant id", KR(ret), K(tenant_id));
  } else if (is_oracle_mode && OB_FAIL(schema_guard.check_oracle_object_exist(
      tenant_id, table_schema.get_database_id(), table_schema.get_table_name_str(),
      TABLE_SCHEMA, INVALID_ROUTINE_TYPE, false/*if_not_exist*/, conflict_schema_types))) {
    LOG_WARN("failed to check oracle object exist", KR(ret));
  } else if (conflict_schema_types.count() > 0) {
    ret = OB_ERR_EXIST_OBJECT;
    LOG_WARN("mlog name is already used by an existing object in oralce mode",
        KR(ret), K(table_schema.get_table_name_str()), K(conflict_schema_types));
  } else if (OB_FAIL(schema_guard.get_database_schema(
      tenant_id, table_schema.get_database_id(), database_schema))) {
    LOG_WARN("failed to get database schema",
        KR(ret), K(tenant_id), K(table_schema.get_database_id()));
  } else if (OB_ISNULL(database_schema)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("database schema should not be null",
        KR(ret), K(tenant_id), K(table_schema.get_database_id()));
  } else if (database_schema->is_in_recyclebin()) {
    ret = OB_ERR_OPERATION_ON_RECYCLE_OBJECT;
    LOG_WARN("cannot create mlog in recyclebin", KR(ret));
  } else if (OB_ISNULL(schema_service = schema_service_->get_schema_service())){
    ret = OB_ERR_SYS;
    LOG_ERROR("schema sevice cannot be null", KR(ret));
  } else if (OB_FAIL(schema_service->fetch_new_table_id(
      table_schema.get_tenant_id(), new_table_id))) {
    LOG_WARN("failed to fetch new table id", KR(ret));
  } else if (OB_FALSE_IT(table_schema.set_table_id(new_table_id))) {
  } else if (is_oracle_mode) {
    // add pk constraint
    ObArenaAllocator allocator("CreateMlog");
    ObString cst_name;
    ObConstraint cst;
    if (OB_FAIL(ObTableSchema::create_cons_name_automatically(
        cst_name,
        table_schema.get_table_name_str(),
        allocator,
        CONSTRAINT_TYPE_PRIMARY_KEY,
        is_oracle_mode))) {
      LOG_WARN("failed to create cons name automatically", KR(ret));
    } else if (cst_name.length() > OB_MAX_CONSTRAINT_NAME_LENGTH_ORACLE) {
      ret = OB_ERR_TOO_LONG_IDENT;
      LOG_WARN("constraint_name length overflow", KR(ret), K(cst_name.length()));
    } else if (OB_FAIL(cst.set_constraint_name(cst_name))) {
      LOG_WARN("failed to set constraint name", KR(ret));
    } else {
      uint64_t new_cst_id = OB_INVALID_ID;
      bool cst_name_exist = false;
      cst.set_name_generated_type(GENERATED_TYPE_SYSTEM);
      cst.set_constraint_type(CONSTRAINT_TYPE_PRIMARY_KEY);
      cst.set_tenant_id(tenant_id);
      cst.set_table_id(table_schema.get_table_id());
      if (OB_FAIL(schema_service->fetch_new_constraint_id(tenant_id, new_cst_id))) {
        LOG_WARN("failed to fetch new constraint id", K(ret));
      } else if (FALSE_IT(cst.set_constraint_id(new_cst_id))) {
      } else if (cst.get_constraint_name_str().empty()) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("cst name is empty", KR(ret));
      } else if (OB_FAIL(check_constraint_name_is_exist(
          schema_guard, table_schema, cst.get_constraint_name_str(), false, cst_name_exist))) {
        LOG_WARN("failed to check constraint name is exist", KR(ret), K(cst.get_constraint_name_str()));
      } else if (cst_name_exist) {
        ret = OB_ERR_CONSTRAINT_NAME_DUPLICATE;
        if (!is_oracle_mode) {
          LOG_USER_ERROR(OB_ERR_CONSTRAINT_NAME_DUPLICATE,
              cst.get_constraint_name_str().length(), cst.get_constraint_name_str().ptr());
        }
        LOG_WARN("cst name is duplicate", KR(ret), K(cst.get_constraint_name_str()));
      } else if (OB_FAIL(table_schema.add_constraint(cst))) {
        LOG_WARN("failed to add constraint", KR(ret));
      }
    }
  }

  if (OB_SUCC(ret)) {
    if (OB_FAIL(create_index_or_mlog_table_in_trans(
        table_schema,
        &arg.ddl_stmt_str_,
        &sql_trans,
        schema_guard,
        true /*need_check_tablet_cnt*/,
        tenant_data_version))) {
      LOG_WARN("failed to create index or mlog table in trans", KR(ret), K(arg.ddl_stmt_str_), K(table_schema));
    } else if (OB_FAIL(add_mlog(sql_trans, arg, schema_guard, table_schema))) {
      LOG_WARN("failed to add mlog", KR(ret));
    }
  }
  return ret;
}

int ObDDLService::create_mlog_tablet(
    ObMySQLTransaction &trans,
    ObSchemaGetterGuard &schema_guard,
    const ObTableSchema &mlog_schema,
    const bool need_check_tablet_cnt,
    const uint64_t tenant_data_version)
{
  int ret = OB_SUCCESS;
  int64_t tenant_id = mlog_schema.get_tenant_id();
  SCN frozen_scn;
  if (OB_FAIL(check_inner_stat())) {
    LOG_WARN("check_inner_stat error", K(is_inited()), KR(ret));
  } else if (!mlog_schema.is_mlog_table() || tenant_data_version <= 0) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("arg must be materialized view log table", KR(ret), K(tenant_id), K(tenant_data_version), K(mlog_schema));
  } else if (OB_ISNULL(GCTX.root_service_)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("root service is null", KR(ret));
  } else if (OB_FAIL(ObMajorFreezeHelper::get_frozen_scn(tenant_id, frozen_scn))) {
    LOG_WARN("failed to get frozen status for create tablet", KR(ret), K(tenant_id));
  } else {
    ObTableCreator table_creator(tenant_id, frozen_scn, trans);
    ObNewTableTabletAllocator new_table_tablet_allocator(tenant_id, schema_guard, sql_proxy_);
    common::ObArray<share::ObLSID> ls_id_array;
    if (OB_FAIL(table_creator.init(need_check_tablet_cnt))) {
      LOG_WARN("fail to init table creator", KR(ret));
    } else if (OB_FAIL(new_table_tablet_allocator.init())) {
      LOG_WARN("fail to init new table tablet allocator", KR(ret));
    } else {
      const ObTableSchema *data_table_schema = NULL;
      const uint64_t data_table_id = mlog_schema.get_data_table_id();
      ObSEArray<const share::schema::ObTableSchema*, 1> schemas;
      ObSEArray<bool, 1> need_create_empty_majors;
      if (OB_FAIL(schema_guard.get_table_schema(tenant_id, data_table_id, data_table_schema))) {
        LOG_WARN("failed to get table schema", KR(ret), K(tenant_id), K(data_table_id));
      } else if (OB_ISNULL(data_table_schema)) {
        ret = OB_TABLE_NOT_EXIST;
        LOG_WARN("data table schema not exists", KR(ret), K(data_table_id));
      } else if (OB_FAIL(schemas.push_back(&mlog_schema))
        || OB_FAIL(need_create_empty_majors.push_back(true))) {
        LOG_WARN("failed to push back mlog schema", KR(ret), K(mlog_schema));
      } else if (OB_FAIL(new_table_tablet_allocator.prepare_like(*data_table_schema))) {
        LOG_WARN("failed to prepare like data table schema", KR(ret), KPC(data_table_schema));
      } else if (OB_FAIL(new_table_tablet_allocator.get_ls_id_array(ls_id_array))) {
        LOG_WARN("failed to get ls id array", KR(ret));
      } else if (OB_FAIL(table_creator.add_create_tablets_of_local_aux_tables_arg(
          schemas, data_table_schema, ls_id_array,
          tenant_data_version,
          need_create_empty_majors))) {
        LOG_WARN("failed to add create tablets of local aux tables arg", KR(ret));
      } else if (OB_FAIL(table_creator.execute())) {
        LOG_WARN("failed to execute create tablet", KR(ret));
      }
    }

    // finishing is always invoked for new table tablet allocator
    int tmp_ret = OB_SUCCESS;
    if (OB_SUCCESS != (tmp_ret = new_table_tablet_allocator.finish(OB_SUCCESS == ret))) {
      LOG_WARN("fail to finish new table tablet allocator", KR(tmp_ret));
    }
  }
  return ret;
}

int ObDDLService::add_mlog(
    ObMySQLTransaction &trans,
    const obrpc::ObCreateMLogArg &arg,
    ObSchemaGetterGuard &schema_guard,
    const ObTableSchema &mlog_schema)
{
  int ret = OB_SUCCESS;
  const uint64_t tenant_id = mlog_schema.get_tenant_id();
  const uint64_t exec_tenant_id = ObSchemaUtils::get_exec_tenant_id(tenant_id);
  const uint64_t mlog_table_id = mlog_schema.get_table_id();
  const obrpc::ObCreateMLogArg::PurgeOptions &purge_options = arg.purge_options_;
  ObMLogPurgeMode purge_mode = purge_options.purge_mode_;
  int64_t purge_start = purge_options.start_datetime_expr_.get_timestamp();
  ObString purge_next = purge_options.next_datetime_expr_;
  ObString purge_job;
  ObArenaAllocator allocator("mlog_sched");

  if (!purge_options.start_datetime_expr_.is_null()
      || !purge_options.next_datetime_expr_.empty()) {
    const ObDatabaseSchema *database = NULL;
    // job_name is generated as "job_prefix+job_id"
    if (OB_FAIL(schema_guard.get_database_schema(
        tenant_id, mlog_schema.get_database_id(), database))) {
      LOG_WARN("failed to get database_schema", KR(ret),
          K(tenant_id), "database_id", mlog_schema.get_database_id());
    } else if (OB_ISNULL(database)) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("database_schema is null", KR(ret),
          "database_id", mlog_schema.get_database_id());
    } else {
      const ObString &db_name = database->get_database_name_str();
      const ObString &table_name = arg.table_name_;
      // job_action is generated as "mlog_purge_func('db_name.table_name')"
      if (OB_FAIL(ObMViewSchedJobUtils::create_mlog_scheduler_job(
          trans,
          tenant_id,
          mlog_table_id,
          db_name,
          table_name,
          purge_options.start_datetime_expr_,
          purge_next,
          arg.purge_options_.exec_env_,
          allocator,
          purge_job))) {
        LOG_WARN("failed to add purge mlog scheduler job", KR(ret), K(mlog_table_id), K(db_name), K(table_name),
            K(purge_options.start_datetime_expr_), K(purge_next));
      } else {
        LOG_INFO("succeed to add purge mlog scheduler job", K(mlog_table_id),
            K(purge_options.start_datetime_expr_), K(purge_next), K(purge_job));
      }
    }
  }

  if (OB_SUCC(ret)) {
    ObMLogInfo mlog_info;
    mlog_info.set_tenant_id(tenant_id);
    mlog_info.set_mlog_id(mlog_table_id);
    mlog_info.set_purge_mode(purge_options.purge_mode_);
    if (!purge_options.start_datetime_expr_.is_null()) {
      mlog_info.set_purge_start(purge_options.start_datetime_expr_.get_timestamp());
    }
    mlog_info.set_schema_version(mlog_schema.get_schema_version());
    if (OB_FAIL(mlog_info.set_purge_next(purge_next))) {
      LOG_WARN("fail to set purge next", KR(ret));
    } else if (OB_FAIL(mlog_info.set_purge_job(purge_job))) {
      LOG_WARN("fail to set purge job", KR(ret));
    } else if (OB_FAIL(ObMLogInfo::insert_mlog_info(trans, mlog_info))) {
      LOG_WARN("fail to insert mlog info", KR(ret), K(mlog_info));
    }
  }
  return ret;
}

int ObDDLService::check_create_with_db_id(ObDatabaseSchema &schema)
{
  int ret = OB_SUCCESS;
  const uint64_t db_id = schema.get_database_id();
  if (OB_INVALID_ID != db_id) {
    const bool enable_sys_table_ddl = common::ObServerConfig::get_instance().enable_sys_table_ddl;
    char err_msg[number::ObNumber::MAX_PRINTABLE_SIZE];
    if (!enable_sys_table_ddl) { //Only when the configuration item switch is turned on can the internal table be created
      ret = OB_OP_NOT_ALLOW;
      (void)snprintf(err_msg, sizeof(err_msg),
               "%s", "create database with database_id");
    } else if (!is_inner_db(db_id)) {
      ret = OB_OP_NOT_ALLOW;
      (void)snprintf(err_msg, sizeof(err_msg),
               "%s %lu", "create inner db with invalid database_id: ", db_id);
    } else {
      schema.set_database_id(db_id);
    }
    if (OB_FAIL(ret)) {
      LOG_USER_ERROR(OB_OP_NOT_ALLOW, err_msg);
    }
  }
  return ret;
}

int ObDDLService::replace_table_schema_type(ObTableSchema &schema)
{
  int ret = OB_SUCCESS;
  const uint64_t table_id = schema.get_table_id();
  if (OB_INVALID_ID != table_id) {
    if (!schema.is_user_table() && !schema.is_view_table()) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("should not reach here");
    } else {
      const bool enable_sys_table_ddl = common::ObServerConfig::get_instance().enable_sys_table_ddl;
      char err_msg[number::ObNumber::MAX_PRINTABLE_SIZE];
      if (!enable_sys_table_ddl) { // Only when the configuration item switch is turned on can the internal table is created
        ret = OB_OP_NOT_ALLOW;
        (void)snprintf(err_msg, sizeof(err_msg),
                 "%s", "create table with table_id");
      } else if (!is_inner_table(table_id)) { // Cannot specify table_id to create non-internal table
        ret = OB_OP_NOT_ALLOW;
        (void)snprintf(err_msg, sizeof(err_msg),
                 "%s %lu", "create inner table with invalid table_id: ", table_id);
      } else if (schema.is_user_table() && is_sys_view(table_id)) { // table_id of table cannot fall into the system view id range
        ret = OB_OP_NOT_ALLOW;
        (void)snprintf(err_msg, sizeof(err_msg),
                 "%s", "create table with table_id which fall into sys_view interval");
      } else if (schema.is_user_view() && !is_sys_view(table_id)) { // table_id of view cannot fall into the non-system view range
        ret = OB_OP_NOT_ALLOW;
        (void)snprintf(err_msg, sizeof(err_msg),
                 "%s", "create view with table_id which fall into non-sys_view interval");
      } else if (is_reserved_table_id(table_id)) {
        ret = OB_OP_NOT_ALLOW;
        (void)snprintf(err_msg, sizeof(err_msg),
                 "%s", "table_id which fall into reserved interval");
      } else {
        ObTableType type = get_inner_table_type_by_id(table_id);
        if (MAX_TABLE_TYPE != type) {
          schema.set_table_type(type);
          schema.set_def_type(TABLE_DEF_TYPE_INTERNAL);
          schema.set_table_id(table_id);
        } else {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("type is error", K(type), K(ret));
        }
      }
      if (OB_FAIL(ret)) {
        if (OB_OP_NOT_ALLOW == ret) {
          LOG_USER_ERROR(OB_OP_NOT_ALLOW, err_msg);
        } else {
          LOG_USER_ERROR(OB_ERR_UNEXPECTED, err_msg);
        }
      }
    }
  }
  return ret;
}

int ObDDLService::generate_object_id_for_partition_schemas(
    ObIArray<ObTableSchema> &partition_schemas)
{
  int ret = OB_SUCCESS;
  const bool gen_subpart_only = false;
  int64_t total_object_cnt = 0;

  for (int64_t i = 0; OB_SUCC(ret) && i < partition_schemas.count(); i++) {
    const ObTableSchema &partition_schema = partition_schemas.at(i);
    int64_t object_cnt = 0;
    if (OB_FAIL(calc_partition_object_id_cnt_(partition_schema, gen_subpart_only, object_cnt))) {
      LOG_WARN("fail to calc partition object id cnt", KR(ret));
    } else {
      total_object_cnt += object_cnt;
    }
  } // end for

  if (OB_SUCC(ret) && total_object_cnt > 0) {
    ObIDGenerator id_generator;
    ObObjectID max_object_id = OB_INVALID_ID;
    ObObjectID min_object_id = OB_INVALID_ID;
    const uint64_t tenant_id = partition_schemas.at(0).get_tenant_id();
    if (OB_ISNULL(schema_service_) || OB_ISNULL(schema_service_->get_schema_service())) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("schema_service is empty", KR(ret), KP_(schema_service));
    } else if (OB_FAIL(schema_service_->get_schema_service()
                       ->fetch_new_partition_ids(tenant_id, total_object_cnt, max_object_id))) {
      LOG_WARN("fail to get max object id", KR(ret), K(tenant_id), K(total_object_cnt));
    } else if (OB_UNLIKELY(0 >= (min_object_id = max_object_id - total_object_cnt + 1))) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("min_object_id should be greator than 0",
               KR(ret), K(min_object_id), K(max_object_id), K(total_object_cnt));
    } else if (OB_FAIL(id_generator.init(1/*step*/, min_object_id, max_object_id))) {
      LOG_WARN("fail to init id generator", KR(ret), K(min_object_id), K(max_object_id));
    }

    for (int64_t i = 0; OB_SUCC(ret) && i < partition_schemas.count(); i++) {
      ObTableSchema &partition_schema = partition_schemas.at(i);
      if (OB_FAIL(generate_object_id_for_partition_schema(partition_schema, gen_subpart_only, &id_generator))) {
        LOG_WARN("fail to generate object_id for partition schema", KR(ret));
      }
    } // end for
  }
  return ret;
}

int ObDDLService::calc_partition_object_id_cnt_(
    const ObPartitionSchema &partition_schema,
    const bool gen_subpart_only,
    int64_t &object_cnt)
{
  int ret = OB_SUCCESS;
  const int64_t schema_id = partition_schema.get_table_id();
  const ObPartitionLevel part_level = partition_schema.get_part_level();
  const int64_t partition_num = partition_schema.get_partition_num();
  const int64_t first_part_num = partition_schema.get_first_part_num();
  object_cnt = 0;
  if (OB_UNLIKELY(part_level >= PARTITION_LEVEL_MAX)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("part level is unexpected", KR(ret), K(part_level));
  } else if (is_inner_table(schema_id)
             || is_sys_tablegroup_id(schema_id)
             || PARTITION_LEVEL_ZERO == part_level) {
    // 1. For non-partitioned schemas'(include view、virtual table):
    //    object_id is equal to its schema_id.
    // 2. For partitioned virtual table(list columns only):
    //    object_id for its partition is hard code by schema.
    // For the above reasons, we won't allocate object_id for table/tablegroup
    object_cnt = 0;
  } else if (OB_UNLIKELY(first_part_num != partition_num)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("first_part_num is not equal to partition_num", KR(ret), K(partition_schema));
  } else if (OB_UNLIKELY(0 >= (object_cnt = partition_schema.get_all_part_num()))) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("invalid partition num", KR(ret), K(object_cnt), K(partition_schema));
  } else if (PARTITION_LEVEL_TWO == part_level && !gen_subpart_only) {
    object_cnt += first_part_num;
  }
  return ret;
}

// Actually, object_id can be allocated within the same ddl trans.
// To avoid refactor more codes, generate_object_id_for_partition_schema() should be called with generate_tablet_id().
//
// [@input]gen_subpart_only:
// - True  : for add/drop subpartition situations, part_id is valid and should not be generated again.
// - False : other situations.
int ObDDLService::generate_object_id_for_partition_schema(
    ObPartitionSchema &partition_schema,
    const bool gen_subpart_only /* = false */,
    ObIDGenerator *batch_id_generator /* = NULL */)
{
  int ret = OB_SUCCESS;
  const uint64_t tenant_id = partition_schema.get_tenant_id();
  // table_id/tablegroup_id
  const int64_t schema_id = partition_schema.get_table_id();
  ObPartitionLevel part_level = partition_schema.get_part_level();
  ObPartition** part_array = partition_schema.get_part_array();
  int64_t partition_num = partition_schema.get_partition_num();
  int64_t object_cnt = 0;
  if (OB_FAIL(calc_partition_object_id_cnt_(partition_schema, gen_subpart_only, object_cnt))) {
    LOG_WARN("fail to calc partition object id cnt", KR(ret), K(partition_schema));
  } else if (0 == object_cnt) {
    // skip
  } else if (OB_ISNULL(part_array)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("part_array is empty", KR(ret), KP(part_array));
  } else {
    ObIDGenerator tmp_id_generator;
    if (OB_ISNULL(batch_id_generator)) {
      ObObjectID max_object_id = OB_INVALID_ID;
      ObObjectID min_object_id = OB_INVALID_ID;
      if (OB_ISNULL(schema_service_) || OB_ISNULL(schema_service_->get_schema_service())) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("schema_service is empty", KR(ret), KP_(schema_service));
      } else if (OB_FAIL(schema_service_->get_schema_service()
                         ->fetch_new_partition_ids(tenant_id, object_cnt, max_object_id))) {
        LOG_WARN("fail to get max object id", KR(ret), K(tenant_id), K(object_cnt));
      } else if (OB_UNLIKELY(0 >= (min_object_id = max_object_id - object_cnt + 1))) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("min_object_id should be greator than 0",
                 KR(ret), K(min_object_id), K(max_object_id), K(object_cnt));
      } else if (OB_FAIL(tmp_id_generator.init(1/*step*/, min_object_id, max_object_id))) {
        LOG_WARN("fail to init id generator", KR(ret), K(min_object_id), K(max_object_id));
      }
    }

    ObObjectID object_id = OB_INVALID_ID;
    ObIDGenerator &id_generator = OB_ISNULL(batch_id_generator) ?
                                  tmp_id_generator :
                                  *batch_id_generator;
    // 1. generate object_id for partitions
    if (OB_SUCC(ret) && !gen_subpart_only) {
      for (int64_t i = 0; OB_SUCC(ret) && i < partition_num; i++) {
        ObPartition* part = part_array[i];
        if (OB_ISNULL(part)) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("part is null", KR(ret), K(i));
        } else if (OB_FAIL(id_generator.next(object_id))) {
          LOG_WARN("fail to get next object id", KR(ret));
        } else {
          part->set_part_id(object_id);
        }
      } // end for
    }
    // 2. generate object_id for subpartitions
    if (OB_SUCC(ret) && PARTITION_LEVEL_TWO == part_level) {
      for (int64_t i = 0; OB_SUCC(ret) && i < partition_num; i++) {
        ObPartition* part = part_array[i];
        if (OB_ISNULL(part)) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("part is null", KR(ret), K(i));
        } else if (OB_UNLIKELY(part->get_part_id() <= 0)) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("part_id is invalid", KR(ret), KPC(part));
        } else if (OB_ISNULL(part->get_subpart_array())
                   || part->get_subpartition_num() <= 0) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("sub_part_array is null or invalid subpartition num", KR(ret), KPC(part));
        } else {
          for (int64_t j = 0; OB_SUCC(ret) && j < part->get_subpartition_num(); j++) {
            ObSubPartition *subpart = part->get_subpart_array()[j];
            if (OB_ISNULL(subpart)) {
              ret = OB_ERR_UNEXPECTED;
              LOG_WARN("subpart is null", KR(ret), KPC(part), K(j));
            } else if (OB_FAIL(id_generator.next(object_id))) {
              LOG_WARN("fail to get next object id", KR(ret));
            } else {
              subpart->set_part_id(part->get_part_id());
              subpart->set_sub_part_id(object_id);
            }
          } // end for subpart
        }
      } // end for part
    }
  }
  return ret;
}

int ObDDLService::generate_tables_tablet_id(ObIArray<ObTableSchema> &table_schemas)
{
  int ret = OB_SUCCESS;
  int64_t total_normal_tablet_cnt = 0;
  int64_t total_extended_tablet_cnt = 0;
  ObIDGenerator normal_tablet_id_generator;
  ObIDGenerator extended_tablet_id_generator;
  uint64_t tenant_id = OB_INVALID_TENANT_ID;

  if (OB_ISNULL(schema_service_) || OB_ISNULL(schema_service_->get_schema_service())) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("schema service is null", KR(ret), KP_(schema_service));
  } else if (table_schemas.count() > 0) {
    tenant_id = table_schemas.at(0).get_tenant_id();
  }

  for (int64_t i = 0; OB_SUCC(ret) && i < table_schemas.count(); i++) {
    const ObTableSchema &table_schema = table_schemas.at(i);
    uint64_t tablet_cnt = 0;
    if (OB_FAIL(calc_table_tablet_id_cnt_(table_schema, tablet_cnt))) {
      LOG_WARN("fail to calc tablet_id cnt", KR(ret));
    } else if (table_schema.gen_normal_tablet()) {
      total_normal_tablet_cnt += tablet_cnt;
    } else {
      total_extended_tablet_cnt += tablet_cnt;
    }
  }

  if (OB_SUCC(ret) && total_normal_tablet_cnt > 0) {
    uint64_t min_tablet_id = OB_INVALID_ID;
    uint64_t max_tablet_id = OB_INVALID_ID;
    if (OB_FAIL(schema_service_->get_schema_service()->fetch_new_tablet_ids(
                tenant_id, true /*gen_normal_tablet*/, total_normal_tablet_cnt, min_tablet_id))) {
      LOG_WARN("fail to fetch new tablet id", KR(ret), K(total_normal_tablet_cnt));
    } else if (FALSE_IT(max_tablet_id = min_tablet_id + total_normal_tablet_cnt - 1)) {
    } else if (OB_FAIL(normal_tablet_id_generator.init(1/*step*/, min_tablet_id, max_tablet_id))) {
      LOG_WARN("fail to init id generator", KR(ret), K(min_tablet_id), K(max_tablet_id));
    }
  }

  if (OB_SUCC(ret) && total_extended_tablet_cnt> 0) {
    uint64_t min_tablet_id = OB_INVALID_ID;
    uint64_t max_tablet_id = OB_INVALID_ID;
    if (OB_FAIL(schema_service_->get_schema_service()->fetch_new_tablet_ids(
                tenant_id, false/*gen_normal_tablet*/, total_extended_tablet_cnt, min_tablet_id))) {
      LOG_WARN("fail to fetch new tablet id", KR(ret), K(total_extended_tablet_cnt));
    } else if (FALSE_IT(max_tablet_id = min_tablet_id + total_extended_tablet_cnt - 1)) {
    } else if (OB_FAIL(extended_tablet_id_generator.init(1/*step*/, min_tablet_id, max_tablet_id))) {
      LOG_WARN("fail to init id generator", KR(ret), K(min_tablet_id), K(max_tablet_id));
    }
  }

  for (int64_t i = 0; OB_SUCC(ret) && i < table_schemas.count(); i++) {
    ObTableSchema &table_schema = table_schemas.at(i);
    ObIDGenerator &id_generator = table_schema.gen_normal_tablet() ?
                                  normal_tablet_id_generator :
                                  extended_tablet_id_generator;
    if (OB_FAIL(generate_tablet_id(table_schema, &id_generator))) {
      LOG_WARN("fail to generator tablet_id", KR(ret));
    }
  }
  return ret;
}

int ObDDLService::calc_table_tablet_id_cnt_(
    const ObTableSchema &table_schema,
    uint64_t &tablet_cnt)
{
  int ret = OB_SUCCESS;
  tablet_cnt = 0;
  if (is_sys_table(table_schema.get_table_id())
      || table_schema.is_vir_table()
      || table_schema.is_view_table()
      || table_schema.is_external_table()) {
    // 1. sys table use table_id as tablet_id
    // 2. virtual table/view/external table don't have tablet_id
  } else if (OB_UNLIKELY((tablet_cnt = table_schema.get_all_part_num()) <= 0)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("fail to get tablet cnt", KR(ret), K(tablet_cnt), K(table_schema));
  }
  return ret;
}

int ObDDLService::generate_tablet_id(
    ObTableSchema &table_schema,
    ObIDGenerator *batch_id_generator /* = NULL*/)
{
  int ret = OB_SUCCESS;
  uint64_t tablet_num = OB_INVALID_ID;
  if (is_sys_table(table_schema.get_table_id())) {
    table_schema.set_tablet_id(table_schema.get_table_id());
  } else if (table_schema.is_external_table()) {
    //skip
  } else if (OB_FAIL(calc_table_tablet_id_cnt_(table_schema, tablet_num))) {
    LOG_WARN("fail to calc tablet num", KR(ret), K(table_schema));
  } else if (0 == tablet_num) {
    // skip
  } else {
    ObIDGenerator tmp_id_generator;
    if (OB_ISNULL(batch_id_generator)) {
      const uint64_t tenant_id = table_schema.get_tenant_id();
      uint64_t min_tablet_id = OB_INVALID_ID;
      uint64_t max_tablet_id = OB_INVALID_ID;
      if (OB_ISNULL(schema_service_) || OB_ISNULL(schema_service_->get_schema_service())) {
        ret = OB_ERR_UNEXPECTED;
        LOG_ERROR("schema service is null", KR(ret), KP_(schema_service));
      } else if (OB_FAIL(schema_service_->get_schema_service()->fetch_new_tablet_ids(
                         tenant_id, table_schema.gen_normal_tablet(), tablet_num, min_tablet_id))) {
        LOG_WARN("fail to fetch new tablet id", KR(ret), K(tablet_num));
      } else if (FALSE_IT(max_tablet_id = min_tablet_id + tablet_num - 1)) {
      } else if (OB_FAIL(tmp_id_generator.init(1/*step*/, min_tablet_id, max_tablet_id))) {
        LOG_WARN("fail to init id generator", KR(ret), K(min_tablet_id), K(max_tablet_id));
      }
    }

    ObIDGenerator &id_generator = OB_ISNULL(batch_id_generator) ?
                                  tmp_id_generator :
                                  *batch_id_generator;
    uint64_t new_tablet_id = OB_INVALID_ID;
    ObPartitionLevel part_level = table_schema.get_part_level();
    if (OB_FAIL(ret)) {
    } else if (OB_UNLIKELY(part_level >= PARTITION_LEVEL_MAX)) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("part level is unexpected", KR(ret), K(part_level));
    } else if (PARTITION_LEVEL_ZERO == part_level) {
      if (OB_FAIL(id_generator.next(new_tablet_id))) {
        LOG_WARN("fail to get next tablet_id", KR(ret));
      } else {
        (void) table_schema.set_tablet_id(new_tablet_id);
      }
    } else {
      ObPartition **part_array = table_schema.get_part_array();
      int64_t part_num = table_schema.get_partition_num();
      if (OB_ISNULL(part_array)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("part array is null", KR(ret), KP(part_array));
      } else {
        for (int64_t i = 0; i < part_num && OB_SUCC(ret); i++) {
          if (OB_ISNULL(part_array[i])) {
            ret = OB_ERR_UNEXPECTED;
            LOG_WARN("NULL ptr", KR(ret), K(i));
          } else if (PARTITION_LEVEL_ONE == part_level) {
            if (OB_FAIL(id_generator.next(new_tablet_id))) {
              LOG_WARN("fail to get next tablet_id", KR(ret));
            } else {
              part_array[i]->set_tablet_id(new_tablet_id);
            }
          } else if (PARTITION_LEVEL_TWO == part_level) {
            ObSubPartition **sub_part_array = part_array[i]->get_subpart_array();
            int64_t sub_part_num = part_array[i]->get_subpartition_num();
            if (OB_ISNULL(sub_part_array)) {
              ret = OB_ERR_UNEXPECTED;
              LOG_WARN("sub part array is null", KR(ret), K(i));
            } else {
              for (int64_t j = 0; j < sub_part_num && OB_SUCC(ret); j++) {
                if (OB_ISNULL(sub_part_array[j])) {
                  ret = OB_ERR_UNEXPECTED;
                  LOG_WARN("NULL ptr", KR(ret), K(i), K(j));
                } else if (OB_FAIL(id_generator.next(new_tablet_id))) {
                  LOG_WARN("fail to get next tablet_id", KR(ret));
                } else {
                  sub_part_array[j]->set_tablet_id(new_tablet_id);
                }
              } // end for
            }
          } else {
            ret = OB_NOT_SUPPORTED;
            LOG_WARN("4.0 not support part type", KR(ret), K(part_level));
          }
        } // end for
      }
    }
  }
  return ret;
}

// For create table/tablegroup
// 1. Add missing partition/subpartition schema.
// 2. Reorganize part_idx/subpart_idx.
int ObDDLService::try_format_partition_schema(ObPartitionSchema &partition_schema)
{
  int ret = OB_SUCCESS;
  // 1. generate missing partition/subpartition.
  bool generated = false;
  if (partition_schema.is_external_table()) {
    //do nothing
  } else if (OB_FAIL(partition_schema.try_generate_hash_part())) {
    LOG_WARN("fail to generate hash part", KR(ret), K(partition_schema));
  } else if (OB_FAIL(partition_schema.try_generate_hash_subpart(generated))) {
    LOG_WARN("fail to generate hash part", KR(ret), K(partition_schema));
  } else if (generated) {
    // skip
  } else if (OB_FAIL(partition_schema.try_generate_subpart_by_template(generated))) {
    LOG_WARN("fail to generate_subpart_by_template", KR(ret), K(partition_schema));
  }
  if (OB_SUCC(ret) && generated) {
    partition_schema.set_sub_part_template_def_valid();
    LOG_INFO("convert schema to nontemplate", K(partition_schema));
  }
  // 2. generate part_idx/subpart_idx.
  if (FAILEDx(partition_schema.try_init_partition_idx())) {
    LOG_WARN("fail to init partition idx", KR(ret));
  }
  return ret;
}

/* generate_schema is called when creating a data table,
 * IndexBuilder::generate_schema is called when create a index table.
 */
int ObDDLService::generate_schema(
    const ObCreateTableArg &arg,
    ObTableSchema &schema)
{
  int ret = OB_SUCCESS;
  const ObIArray<ObConstraint> &constraints = arg.constraint_list_;
  const uint64_t tenant_id = schema.get_tenant_id();
  uint64_t new_table_id = schema.get_table_id();
  uint64_t compat_version = 0;
  ObSchemaService *schema_service = NULL;
  const ObDatabaseSchema *database_schema = NULL;
  const ObTenantSchema *tenant_schema = NULL;
  const ObTablespaceSchema *tablespace_schema = NULL;
  bool is_oracle_mode = false;
  ObSchemaGetterGuard guard;
  if (OB_FAIL(check_inner_stat())) {
    LOG_WARN("variable is not init");
  } else if (OB_FAIL(GET_MIN_DATA_VERSION(tenant_id, compat_version))) {
    LOG_WARN("fail to get data version", K(ret), K(tenant_id));
  } else if (not_compat_for_queuing_mode(compat_version) && arg.schema_.is_new_queuing_table_mode()) {
    ret = OB_NOT_SUPPORTED;
    LOG_WARN(QUEUING_MODE_NOT_COMPAT_WARN_STR, K(ret), K(tenant_id), K(compat_version), K(arg));
    LOG_USER_ERROR(OB_NOT_SUPPORTED, QUEUING_MODE_NOT_COMPAT_USER_ERROR_STR);
  } else if (compat_version < DATA_VERSION_4_3_5_1 && arg.schema_.get_enable_macro_block_bloom_filter()) {
    ret = OB_NOT_SUPPORTED;
    LOG_WARN("fail to generate schema, not support enable_macro_block_bloom_filter for this version",
             KR(ret), K(tenant_id), K(compat_version), K(arg));
    LOG_USER_ERROR(OB_NOT_SUPPORTED, "this version not support enable_macro_block_bloom_filter");
  } else if (OB_FAIL(schema_service_->get_tenant_schema_guard(tenant_id, guard))) {
    LOG_WARN("get schema guard failed", K(ret));
  } else {
    schema_service = schema_service_->get_schema_service();
    if (OB_ISNULL(schema_service)) {
      ret = OB_ERR_SYS;
      LOG_ERROR("schema_service must not null", K(ret));
    }
  }

  // support to create inner table if enable_sys_table_ddl is opened
  // -- system view                            ----> table_type will be TABLE_TYPE_VIEW
  // -- other(virtual table or core table)     ----> table_type will be TABLE_TYPE_SYS
  if (OB_SUCC(ret) && OB_FAIL(replace_table_schema_type(schema))) {
    LOG_WARN("not supported operator", K(ret));
  }

  // set basic schema info, will be checked by check_table_exist
  if (OB_SUCC(ret)) {
    LOG_DEBUG("hualong schema is ", K(schema));
    if (OB_FAIL(set_tablegroup_id(schema))) {
      LOG_WARN("set_tablegroup_id failed", "tablegroup name",
               schema.get_tablegroup_name(), K(ret));
    } else if (OB_FAIL(guard.get_database_schema(
                tenant_id, schema.get_database_id(), database_schema))) {
      LOG_WARN("get_database_schema failed", K(tenant_id), "database_id",
               schema.get_database_id(), K(ret));
    } else if (NULL == database_schema) {
      ret = OB_ERR_NO_DB_SELECTED;
      LOG_WARN("not find this database schema", K(schema.get_database_id()), K(ret));
    } else if (!arg.is_inner_ && database_schema->is_in_recyclebin()) {
      ret = OB_ERR_OPERATION_ON_RECYCLE_OBJECT;
      LOG_WARN("can not create table in recyclebin", K(ret), K(schema));
    } else if (OB_FAIL(guard.get_tenant_info(tenant_id, tenant_schema))) {
      LOG_WARN("fail to get tenant schema", K(ret), K(tenant_id));
    } else if (OB_UNLIKELY(NULL == tenant_schema)) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("tenant schema is null", K(ret), KP(tenant_schema));
    } else if (OB_INVALID_ID != new_table_id && is_inner_table(new_table_id)) {
      // Specify table_id to create a system table, it may create a tenant-level system table, skip
    } else if (OB_FAIL(schema_service->fetch_new_table_id(
            schema.get_tenant_id(), new_table_id))) {
      LOG_WARN("fail to fetch new table id", K(ret));
    } else if (FALSE_IT(schema.set_table_id(new_table_id))) {
    } else if (OB_FAIL(try_format_partition_schema(schema))) {
      LOG_WARN("fail to try_format_partition_schema", K(schema), KR(ret));
    } else if (OB_FAIL(generate_object_id_for_partition_schema(schema))) {
      LOG_WARN("fail to generate object_id for partition schema", KR(ret), K(schema));
    } else if (OB_FAIL(generate_tablet_id(schema))) {
      LOG_WARN("fail to fetch new table id", K(schema), K(ret));
    }
    if (OB_SUCC(ret) && OB_INVALID_ID != schema.get_tablespace_id()) {
      if (OB_FAIL(guard.get_tablespace_schema(tenant_id, schema.get_tablespace_id(), tablespace_schema))) {
        LOG_WARN("fail to get tablespace schema", K(schema), K(ret));
      } else if (OB_UNLIKELY(NULL == tablespace_schema)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("tablespace schema is null", K(ret), K(schema));
      } else if (OB_FAIL(schema.set_encrypt_key(tablespace_schema->get_encrypt_key()))) {
        LOG_WARN("fail to set encrypt key", K(ret), K(schema));
      } else {
        schema.set_master_key_id(tablespace_schema->get_master_key_id());
      }
    }
  }

  if (OB_SUCC(ret)) {
    if (schema.has_partition()) { // include table of standalone and binding
      common::ObArray<share::ObResourcePoolName> pool_names;
      if (OB_FAIL(try_check_and_set_table_schema_in_tablegroup(guard, schema))) {
        LOG_WARN("check table schema in tablegroup failed", K(ret));
      }
    } else {} // has no partition
  }

  // constraints
  if (OB_FAIL(ret)) {
  } else if (OB_FAIL(schema.check_if_oracle_compat_mode(is_oracle_mode))) {
    LOG_WARN("fail to check is oracle mode", K(ret));
  }
  for (int64_t i = 0; OB_SUCC(ret) && i < constraints.count(); ++i) {
    uint64_t new_cst_id = OB_INVALID_ID;
    ObConstraint &cst = const_cast<ObConstraint &>(constraints.at(i));
    cst.set_tenant_id(tenant_id);
    cst.set_table_id(schema.get_table_id());
    if (OB_FAIL(schema_service->fetch_new_constraint_id(tenant_id, new_cst_id))) {
      LOG_WARN("failed to fetch new constraint id", K(ret));
    } else if (FALSE_IT(cst.set_constraint_id(new_cst_id))) {
    } else if (cst.get_constraint_name_str().empty()) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("cst name is empty", K(ret));
    } else {
       // Check whether the name of the constraint is repeated
      bool is_constraint_name_exist = false;
      if (OB_FAIL(check_constraint_name_is_exist(guard, schema, cst.get_constraint_name_str(), false, is_constraint_name_exist))) {
        LOG_WARN("fail to check constraint name is exist or not", K(ret), K(cst.get_constraint_name_str()));
      } else if (is_constraint_name_exist) {
        ret = OB_ERR_CONSTRAINT_NAME_DUPLICATE;
        if (!is_oracle_mode) {
          LOG_USER_ERROR(OB_ERR_CONSTRAINT_NAME_DUPLICATE, cst.get_constraint_name_str().length(), cst.get_constraint_name_str().ptr());
        }
        LOG_WARN("cst name is duplicate", K(ret), K(cst.get_constraint_name_str()));
      }
    }
    // The process that check whether the constraint name is repeated is end
    if (OB_SUCC(ret)) {
      if (OB_FAIL(schema.add_constraint(cst))) {
        LOG_WARN("add constraint failed", K(ret), K(i));
      }
    }
  }

  // check udt id invalid
  if (OB_SUCC(ret) && is_oracle_mode) {
    if (OB_FAIL(check_table_udt_id_is_exist(guard, schema, tenant_id))) {
      LOG_WARN("check udt id failed", K(ret));
    }
  }

  // fill table schema for interval part
  if (OB_SUCC(ret) && schema.has_partition()
      && schema.is_interval_part()) {
    int64_t part_num = schema.get_part_option().get_part_num();
    ObPartition **part_array = schema.get_part_array();
    const ObRowkey *transition_point = NULL;
    if (PARTITION_LEVEL_TWO == schema.get_part_level()
        && !schema.has_sub_part_template_def()) {
      ret = OB_NOT_SUPPORTED;
      LOG_WARN("interval part of composited-partitioned table not support", K(ret), K(schema));
      LOG_USER_ERROR(OB_NOT_SUPPORTED, "interval part of composited-partitioned table without template");
    } else if (1 != schema.get_partition_key_column_num()) {
      ret = OB_OP_NOT_ALLOW;
      LOG_WARN("more than one partition key not support", K(ret), K(schema));
      LOG_USER_ERROR(OB_OP_NOT_ALLOW, "more than one partition key");
    } else if (OB_ISNULL(part_array)) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("range part array is null", K(ret), K(schema));
    } else if (0 == part_num) {
      ret = OB_ERR_UNEXPECTED;
      LOG_INFO("range part num is 0", K(ret), K(schema));
    } else if (FALSE_IT(transition_point = &part_array[part_num - 1]->get_high_bound_val())) {
    } else if (OB_FAIL(ObPartitionUtils::check_interval_partition_table(*transition_point, schema.get_interval_range()))) {
      LOG_WARN("fail to check_interval_partition_table", K(ret), K(schema));
    } else if (OB_FAIL(schema.set_transition_point(*transition_point))) {
      LOG_WARN("fail to set transition point", K(ret), K(schema));
    }
  }

  if (OB_SUCC(ret)) {
    if (schema.is_materialized_view()) {
      if (OB_FAIL(ObResolverUtils::check_schema_valid_for_mview(schema))) {
        LOG_WARN("failed to check schema valid for mview", KR(ret), K(schema));
      } else if (arg.mv_ainfo_.count() <= 0) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("there should be mv ainfo", KR(ret), K(arg.mv_ainfo_.count()));
      } else {
        schema.get_view_schema().set_mv_refresh_info(&(arg.mv_ainfo_.at(0).mv_refresh_info_));
      }
    }
  }

  // check auto_partition validity
  if (OB_FAIL(ret)) {
  } else if (OB_FAIL(schema.check_validity_for_auto_partition())) {
    LOG_WARN("fail to check auto partition setting", KR(ret), K(schema), K(arg));
  }

  if (OB_SUCC(ret)) {
    schema.set_micro_index_clustered(arg.schema_.get_micro_index_clustered());
  }
  return ret;
}

int ObDDLService::get_uk_cst_id_for_self_ref(const ObIArray<ObTableSchema> &table_schemas,
                                             const ObCreateForeignKeyArg &foreign_key_arg,
                                             ObForeignKeyInfo &foreign_key_info)
{
  int ret = OB_SUCCESS;

  bool is_match = false;
  for (int64_t i = 1; OB_SUCC(ret) && !is_match && i < table_schemas.count(); ++i) {
    const ObTableSchema &index_table_schema = table_schemas.at(i);
    if (index_table_schema.get_data_table_id() != table_schemas.at(0).get_table_id()) {
      ret = OB_INVALID_ARGUMENT;
      LOG_WARN("the data table of the index table is wrong.", K(ret), K(index_table_schema.get_data_table_id()), K(table_schemas.at(0).get_table_id()));
    } else if (index_table_schema.is_unique_index()) {
      const ObColumnSchemaV2 *index_col = NULL;
      const ObIndexInfo &index_info = index_table_schema.get_index_info();
      ObSEArray<ObString, 8> uk_columns;
      for (int64_t j = 0; OB_SUCC(ret) && j < index_info.get_size(); ++j) {
        if (OB_ISNULL(index_col = index_table_schema.get_column_schema(index_info.get_column(j)->column_id_))) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("get index column schema failed", K(ret));
        } else if (index_col->is_hidden() || index_col->is_shadow_column()) { // do nothing
        } else if (OB_FAIL(uk_columns.push_back(index_col->get_column_name()))) {
          LOG_WARN("push back index column failed", K(ret));
        } else {} // do nothing
      }
      if (OB_SUCC(ret)) {
        const ObIArray<ObString> &parent_columns = foreign_key_arg.parent_columns_;
        if (OB_FAIL(sql::ObResolverUtils::check_match_columns(parent_columns, uk_columns, is_match))) {
          LOG_WARN("Failed to check_match_columns", K(ret));
        } else if (is_match) {
          foreign_key_info.fk_ref_type_ = foreign_key_arg.fk_ref_type_;
          foreign_key_info.ref_cst_id_ = index_table_schema.get_table_id();
        }
      }
    }
  }

  return ret;
}


int ObDDLService::get_index_cst_id_for_self_ref(const ObIArray<ObTableSchema> &table_schemas,
                                             const ObCreateForeignKeyArg &foreign_key_arg,
                                             ObForeignKeyInfo &foreign_key_info)
{
  int ret = OB_SUCCESS;

  bool is_match = false;

  /* 1.检查自引用primary key的前缀 */
  if (table_schemas.count() < 1) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("the parameters passed to get_index_cst_id_for_self_ref is wrong.", K(ret), K(table_schemas));
  } else if (!table_schemas.at(0).is_user_table()) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("the first table in table_schemas is wrong.", K(ret), K(table_schemas.at(0)));
  } else {
    const ObTableSchema& parent_table_schema = table_schemas.at(0);
    const ObRowkeyInfo &rowkey_info = parent_table_schema.get_rowkey_info();
    common::ObSEArray<ObString, 8> pk_columns;
    for (int64_t i = 0; OB_SUCC(ret) && i < rowkey_info.get_size(); ++i) {
      uint64_t column_id = 0;
      const ObColumnSchemaV2 *col_schema = NULL;
      if (OB_FAIL(rowkey_info.get_column_id(i, column_id))) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("fail to get rowkey info", K(ret), K(i), K(rowkey_info));
      } else if (NULL == (col_schema = parent_table_schema.get_column_schema(column_id))) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("get pk column schema failed", K(ret));
      } else if (col_schema->is_hidden() || col_schema->is_shadow_column()) {
        // do nothing
      } else if(OB_FAIL(pk_columns.push_back(col_schema->get_column_name()))) {
        LOG_WARN("push back pk column failed", K(ret));
      } else { } // do nothing
    }
    if (OB_SUCC(ret)) {
      const ObIArray<ObString> &parent_columns = foreign_key_arg.parent_columns_;
      if (OB_FAIL(sql::ObResolverUtils::check_partial_match_columns(parent_columns, pk_columns, is_match))) {
        LOG_WARN("Failed to check_match_columns", K(ret));
      } else if (is_match) {
        foreign_key_info.fk_ref_type_ = foreign_key_arg.fk_ref_type_;
        foreign_key_info.ref_cst_id_ = parent_table_schema.get_table_id();
      }
    }
  }

  /* 2.检查自引用unique key的前缀 */
  for (int64_t i = 1; OB_SUCC(ret) && !is_match && i < table_schemas.count(); ++i) {
    const ObTableSchema &index_table_schema = table_schemas.at(i);
    const ObColumnSchemaV2 *index_col = NULL;
    const ObIndexInfo &index_info = index_table_schema.get_index_info();
    ObSEArray<ObString, 8> index_columns;
    if (index_table_schema.get_data_table_id() != table_schemas.at(0).get_table_id()) {
      ret = OB_INVALID_ARGUMENT;
      LOG_WARN("the data table of the index table is wrong.", K(ret), K(index_table_schema.get_data_table_id()), K(table_schemas.at(0).get_table_id()));
    }
    for (int64_t j = 0; OB_SUCC(ret) && j < index_info.get_size(); ++j) {
      if (OB_ISNULL(index_info.get_column(j))) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("index_info get_column returns nullptr.", K(ret), K(j));
      } else if (OB_ISNULL(index_col = index_table_schema.get_column_schema(index_info.get_column(j)->column_id_))) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("get index column schema failed", K(ret));
      } else if (index_col->is_hidden() || index_col->is_shadow_column()) { // do nothing
      } else if (OB_FAIL(index_columns.push_back(index_col->get_column_name()))) {
        LOG_WARN("push back index column failed", K(ret));
      } else {} // do nothing
    }
    if (OB_SUCC(ret)) {
      const ObIArray<ObString> &parent_columns = foreign_key_arg.parent_columns_;
      if (OB_FAIL(sql::ObResolverUtils::check_partial_match_columns(parent_columns, index_columns, is_match))) {
        LOG_WARN("Failed to check_match_columns", K(ret));
      } else if (is_match) {
        foreign_key_info.fk_ref_type_ = foreign_key_arg.fk_ref_type_;
        foreign_key_info.ref_cst_id_ = index_table_schema.get_table_id();
      }
    }
  }

  return ret;
}

int ObDDLService::check_table_udt_id_is_exist(share::schema::ObSchemaGetterGuard &schema_guard,
                                              const share::schema::ObTableSchema &table_schema,
                                              const uint64_t tenant_id)
{
  int ret = OB_SUCCESS;
  ObTableSchema::const_column_iterator tmp_begin = table_schema.column_begin();
  ObTableSchema::const_column_iterator tmp_end = table_schema.column_end();
  for (; OB_SUCC(ret) && tmp_begin != tmp_end; tmp_begin++) {
    ObColumnSchemaV2 *col = (*tmp_begin);
    if (OB_ISNULL(col)) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("get column schema failed", K(ret));
    } else if (col->get_meta_type().is_user_defined_sql_type()) {
      // delete hidden primary key
      if (OB_FAIL(check_udt_id_is_exist(schema_guard, *col, tenant_id))) {
        LOG_WARN("fail to check column udt id", K(ret));
      }
    }
  }
  return ret;
}

int ObDDLService::check_udt_id_is_exist(share::schema::ObSchemaGetterGuard &schema_guard,
                                        const share::schema::ObColumnSchemaV2 &col_schema,
                                        const uint64_t tenant_id)
{
  int ret = OB_SUCCESS;
  uint64_t udt_id = col_schema.get_sub_data_type();
  const ObUDTTypeInfo *udt_info = NULL;

  if (OB_FAIL(schema_guard.get_udt_info(tenant_id, udt_id, udt_info))) {
    LOG_WARN("get udt info failed", K(ret), K(tenant_id), K(udt_id));
  } else if (OB_NOT_NULL(udt_info)) {
    // do nothing
  } else if (OB_FAIL(schema_guard.get_udt_info(OB_SYS_TENANT_ID, udt_id, udt_info))) {
    LOG_WARN("get udt info failed", K(ret), K(OB_SYS_TENANT_ID), K(udt_id));
  } else if (OB_NOT_NULL(udt_info)) {
    // do nothing
  } else {
    ret = OB_ERR_INVALID_DATATYPE;
    LOG_WARN("invalid column udt id", K(ret), K(udt_id));
  }

  return ret;
}

// In oracle mode, check constraints and foreign key constraints are in the same namespace.
// So we need to check if the new constraint name dup with all kinds of constraints in oracle mode.
// In mysql mode, check constraints and foreign key constraints are in the separate namespaces.
// So we need to check if the new check constraint name dup with check constraints and fk constraint name dup with fk constraints in mysql mode.
int ObDDLService::check_constraint_name_is_exist(share::schema::ObSchemaGetterGuard &schema_guard,
                                                 const share::schema::ObTableSchema &table_schema,
                                                 const common::ObString &constraint_name,
                                                 const bool is_foreign_key, // this param is only effective in mysql mode
                                                 bool &is_constraint_name_exist)
{
  int ret = OB_SUCCESS;
  uint64_t constraint_id = OB_INVALID_ID;
  bool is_oracle_mode = false;
  is_constraint_name_exist = false;

  if (OB_FAIL(table_schema.check_if_oracle_compat_mode(is_oracle_mode))) {
    LOG_WARN("check if oracle compat mode failed", K(ret));
  } else {
    // check if fk name is exist
    if (is_oracle_mode || is_foreign_key) {
      if (OB_FAIL(schema_guard.get_foreign_key_id(table_schema.get_tenant_id(),
                                                  table_schema.get_database_id(),
                                                  constraint_name,
                                                  constraint_id))) {
        LOG_WARN("get foreign key id failed", K(ret),
                                              K(table_schema.get_tenant_id()),
                                              K(table_schema.get_database_id()),
                                              K(constraint_name));
      } else if (OB_INVALID_ID != constraint_id) {
        is_constraint_name_exist = true;
      }
    }
    // check if cst name is exist
    if (OB_SUCC(ret) && !is_constraint_name_exist && (is_oracle_mode || !is_foreign_key)) {
      if (table_schema.is_mysql_tmp_table()) {
        // tmp table in mysql mode, do nothing
      } else if (OB_FAIL(schema_guard.get_constraint_id(table_schema.get_tenant_id(),
                                                        table_schema.get_database_id(),
                                                        constraint_name,
                                                        constraint_id))) {
        LOG_WARN("get constraint id failed", K(ret),
                                             K(table_schema.get_tenant_id()),
                                             K(table_schema.get_database_id()),
                                             K(constraint_name));
      } else if (OB_INVALID_ID != constraint_id) {
        is_constraint_name_exist = true;
      }
    }
  }

  return ret;
}

int ObDDLService::deal_with_cst_for_alter_table(
    share::schema::ObSchemaGetterGuard &schema_guard,
    const uint64_t tenant_id,
    const ObTableSchema *orig_table_schema,
    obrpc::ObAlterTableArg &alter_table_arg,
    ObMockFKParentTableSchema &mock_fk_parent_table_schema)
{
  int ret = OB_SUCCESS;
  AlterTableSchema &alter_table_schema = alter_table_arg.alter_table_schema_;
  const ObTableSchema *table_schema = NULL;
  bool is_oracle_mode = false;
  if (OB_ISNULL(orig_table_schema)) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("table schema is null", KR(ret), K(alter_table_arg));
  } else if (OB_FAIL(schema_guard.get_table_schema(tenant_id,
             alter_table_schema.get_origin_database_name(),
             alter_table_schema.get_origin_table_name(),
             false,
             table_schema))) {
    LOG_WARN("get table schema failed", K(ret), K(alter_table_schema.get_origin_database_name()), K(alter_table_schema.get_origin_table_name()));
  } else if (OB_FAIL(table_schema->check_if_oracle_compat_mode(is_oracle_mode))) {
    LOG_WARN("fail to check is oracle mode", K(ret));
  }
  // check check constraint info
  if (OB_FAIL(ret)) {
  } else if (alter_table_arg.alter_constraint_type_ == obrpc::ObAlterTableArg::ADD_CONSTRAINT) {
    ObTableSchema::const_constraint_iterator iter = alter_table_schema.constraint_begin();
    for (; OB_SUCC(ret) && iter != alter_table_schema.constraint_end(); iter++) {
      if (OB_ISNULL(iter) || OB_ISNULL(*iter)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("constraint is null", K(ret));
      } else if ((*iter)->get_constraint_name_str().empty()) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("check constraint name is null", K(ret));
      } else {
        bool is_check_constraint_name_exist = true;
        if (OB_FAIL(check_constraint_name_is_exist(
                    schema_guard,
                    *table_schema,
                    (*iter)->get_constraint_name_str(),
                    false,
                    is_check_constraint_name_exist))) {
          LOG_WARN("fail to check check constraint name is exist or not",
                  K(ret), K((*iter)->get_constraint_name_str()));
        } else if (is_check_constraint_name_exist) {
          ret = OB_ERR_CONSTRAINT_NAME_DUPLICATE;
          LOG_WARN("check constraint name is duplicate",
                  K(ret), K((*iter)->get_constraint_name_str()));
          if (!is_oracle_mode) {
            LOG_USER_ERROR(OB_ERR_CONSTRAINT_NAME_DUPLICATE, (*iter)->get_constraint_name_str().length(), (*iter)->get_constraint_name_str().ptr());
          }
        }
      }
    }
  }

  // check foreign key info
  for (int64_t i = 0; OB_SUCC(ret) && i < alter_table_arg.foreign_key_arg_list_.count(); i++) {
    const ObCreateForeignKeyArg &foreign_key_arg = alter_table_arg.foreign_key_arg_list_.at(i);
    ObForeignKeyInfo foreign_key_info;
    // Check for duplicate foreign key constraint names
    if (foreign_key_arg.foreign_key_name_.empty()) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("fk name is empty", K(ret));
    } else {
      bool is_foreign_key_name_exist = true;
      if (OB_FAIL(check_constraint_name_is_exist(schema_guard, *table_schema, foreign_key_arg.foreign_key_name_, true, is_foreign_key_name_exist))) {
        LOG_WARN("fail to check foreign key name is exist or not", K(ret), K(foreign_key_arg.foreign_key_name_));
      } else if (is_foreign_key_name_exist) {
        if (foreign_key_arg.is_modify_fk_state_) {
          // Check whether the constraint is a foreign key constraint
          uint64_t fk_constraint_id = OB_INVALID_ID;
          if (OB_FAIL(schema_guard.get_foreign_key_id(table_schema->get_tenant_id(),
                                                      table_schema->get_database_id(),
                                                      foreign_key_arg.foreign_key_name_,
                                                      fk_constraint_id))) {
            LOG_WARN("get foreign key id failed", K(ret),
                                                  K(table_schema->get_tenant_id()),
                                                  K(table_schema->get_database_id()),
                                                  K(foreign_key_arg.foreign_key_name_));
          } else if (OB_INVALID_ID != fk_constraint_id) {
            // There is a corresponding foreign key, do nothing
          } else {
            ret = OB_ERR_UNEXPECTED;
            LOG_WARN("no such fk constraint", K(ret), K(foreign_key_arg.foreign_key_name_));
          }
        } else if (is_oracle_mode) {
          ret = OB_ERR_CONSTRAINT_NAME_DUPLICATE;
          LOG_WARN("fk name is duplicate", K(ret), K(foreign_key_arg.foreign_key_name_));
        } else { // mysql mode
          ret = OB_ERR_DUP_KEY;
          LOG_USER_ERROR(OB_ERR_DUP_KEY, table_schema->get_table_name_str().length(),
                         table_schema->get_table_name_str().ptr());
        }
      }
    }
    // Check whether the foreign key constraint name is repeated end
    if (OB_SUCC(ret) && foreign_key_arg.is_modify_fk_state_) {
      bool is_found = false;
      const ObIArray<ObForeignKeyInfo> &foreign_key_infos = orig_table_schema->get_foreign_key_infos();
      for (int64_t i = 0; !is_found && i < foreign_key_infos.count(); ++i) {
        if (0 == foreign_key_arg.foreign_key_name_.case_compare(foreign_key_infos.at(i).foreign_key_name_)) {
          is_found = true;
          foreign_key_info = foreign_key_infos.at(i);
          foreign_key_info.is_modify_fk_state_ = true;
          if (foreign_key_arg.is_modify_enable_flag_) {
            foreign_key_info.is_modify_enable_flag_ = true;
            foreign_key_info.enable_flag_ = foreign_key_arg.enable_flag_;
          }
          if (foreign_key_arg.is_modify_validate_flag_) {
            foreign_key_info.is_modify_validate_flag_ = true;
            foreign_key_info.validate_flag_ = foreign_key_arg.validate_flag_;
          }
          if (foreign_key_arg.is_modify_rely_flag_) {
            foreign_key_info.is_modify_rely_flag_ = true;
            foreign_key_info.rely_flag_ = foreign_key_arg.rely_flag_;
          }
        }
      }
      if (!is_found) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("no such fk constraint", K(ret), K(foreign_key_arg.foreign_key_name_));
      }
      if (OB_SUCC(ret)) {
        if (foreign_key_info.child_table_id_ != foreign_key_info.parent_table_id_) {
          // If the reference table is itself, there is no need to update sync_versin_for_cascade_table
          if (OB_FAIL(alter_table_schema.add_depend_table_id(foreign_key_info.parent_table_id_))) {
            LOG_WARN("failed to add depend table id", K(ret), K(foreign_key_info));
          }
        }
      }
      if (OB_SUCC(ret)) {
        if (OB_FAIL(alter_table_schema.add_foreign_key_info(foreign_key_info))) {
          LOG_WARN("failed to push foreign key info", K(ret), K(foreign_key_info));
        }
      }
    } else {
      const ObTableSchema *parent_schema = NULL;
      if (OB_SUCC(ret)) {
        // get parent table schema
        // Determine whether it is self-referential if ture, partition table schema is child table schema
        if (0 == foreign_key_arg.parent_table_.case_compare(alter_table_schema.get_origin_table_name())
              && 0 == foreign_key_arg.parent_database_.case_compare(alter_table_schema.get_origin_database_name())) {
          parent_schema = table_schema;
        } else if (OB_FAIL(schema_guard.get_table_schema(table_schema->get_tenant_id(),
                                                         foreign_key_arg.parent_database_,
                                                         foreign_key_arg.parent_table_,
                                                         false, parent_schema))) {
          LOG_WARN("failed to get parent table schema", K(ret), K(foreign_key_arg));
        }
      }
      if (OB_SUCC(ret)) {
        if (foreign_key_arg.is_parent_table_mock_) {
          if (OB_NOT_NULL(parent_schema)) {
            ret = OB_ERR_UNEXPECTED;
            LOG_WARN("parent_schema is exist", K(ret), KPC(parent_schema));
          } else if (OB_FAIL(gen_mock_fk_parent_table_for_create_fk(schema_guard,
                             table_schema->get_tenant_id(), foreign_key_arg, NULL, foreign_key_info,
                             mock_fk_parent_table_schema))) {
            LOG_WARN("failed to generate_mock_fk_parent_table_schema", K(ret), K(table_schema->get_tenant_id()), K(foreign_key_arg));
          }
        } else if (OB_ISNULL(parent_schema)) {
          ret = OB_TABLE_NOT_EXIST;
          LOG_WARN("parent table is not exist", K(ret), K(foreign_key_arg));
        } else if (parent_schema->is_in_recyclebin()) {
          ret = OB_ERR_OPERATION_ON_RECYCLE_OBJECT;
          LOG_WARN("parent table is in recyclebin", K(ret), K(foreign_key_arg));
        } else if (parent_schema->get_table_id() != table_schema->get_table_id()) {
          // If the reference table is itself, there is no need to update sync_versin_for_cascade_table
          if (OB_FAIL(alter_table_schema.add_depend_table_id(parent_schema->get_table_id()))) {
            LOG_WARN("failed to add depend table id", K(ret), K(foreign_key_arg));
          }
        }
      }
      // get child column schema.
      if (OB_SUCC(ret)) {
        foreign_key_info.child_table_id_ = table_schema->get_table_id();
        foreign_key_info.parent_table_id_ = foreign_key_arg.is_parent_table_mock_ ? mock_fk_parent_table_schema.get_mock_fk_parent_table_id() : parent_schema->get_table_id();
        for (int64_t j = 0; OB_SUCC(ret) && j < foreign_key_arg.child_columns_.count(); j++) {
          const ObString &column_name = foreign_key_arg.child_columns_.at(j);
          const ObColumnSchemaV2 *column_schema = table_schema->get_column_schema(column_name);
          if (OB_ISNULL(column_schema)) {
            ret = OB_ERR_COLUMN_NOT_FOUND;
            LOG_WARN("child column is not exist", K(ret), K(column_name));
          } else if (OB_FAIL(foreign_key_info.child_column_ids_.push_back(column_schema->get_column_id()))) {
            LOG_WARN("failed to push child column id", K(ret), K(column_name));
          }
        }
      }
      // get parent column schema.
      if (OB_SUCC(ret) && !foreign_key_arg.is_parent_table_mock_) {
        for (int64_t j = 0; OB_SUCC(ret) && j < foreign_key_arg.parent_columns_.count(); j++) {
          const ObString &column_name = foreign_key_arg.parent_columns_.at(j);
          const ObColumnSchemaV2 *column_schema = parent_schema->get_column_schema(column_name);
          if (OB_ISNULL(column_schema)) {
            ret = OB_ERR_COLUMN_NOT_FOUND;
            LOG_WARN("parent column is not exist", K(ret), K(column_name));
          } else if (OB_FAIL(foreign_key_info.parent_column_ids_.push_back(column_schema->get_column_id()))) {
            LOG_WARN("failed to push parent column id", K(ret), K(column_name));
          }
        }
      }
      // get reference option and foreign key name.
      if (OB_SUCC(ret)) {
        foreign_key_info.update_action_ = foreign_key_arg.update_action_;
        foreign_key_info.delete_action_ = foreign_key_arg.delete_action_;
        foreign_key_info.foreign_key_name_ = foreign_key_arg.foreign_key_name_;
        foreign_key_info.enable_flag_ = foreign_key_arg.enable_flag_;
        foreign_key_info.validate_flag_ = foreign_key_arg.validate_flag_;
        foreign_key_info.rely_flag_ = foreign_key_arg.rely_flag_;
        foreign_key_info.fk_ref_type_ = foreign_key_arg.fk_ref_type_;
        foreign_key_info.ref_cst_id_ = foreign_key_arg.ref_cst_id_;
        foreign_key_info.is_parent_table_mock_ = foreign_key_arg.is_parent_table_mock_;
        foreign_key_info.name_generated_type_ = foreign_key_arg.name_generated_type_;
      }
      // add foreign key info.
      if (OB_SUCC(ret)) {
        ObSchemaService *schema_service = schema_service_->get_schema_service();
        if (OB_FAIL(schema_service->fetch_new_constraint_id(table_schema->get_tenant_id(),
                                                            foreign_key_info.foreign_key_id_))) {
          LOG_WARN("failed to fetch new foreign key id", K(ret), K(foreign_key_arg));
        } else if (OB_FAIL(alter_table_schema.add_foreign_key_info(foreign_key_info))) {
          LOG_WARN("failed to push foreign key info", K(ret), K(foreign_key_info));
        }
      }
    }
  } // for


  return ret;
}

int ObDDLService::check_cst_name_dup_for_rename_table_mysql(
    share::schema::ObSchemaGetterGuard &schema_guard,
    const share::schema::ObTableSchema *from_table_schema,
    const uint64_t to_database_id)
{
  int ret = OB_SUCCESS;
  bool is_oracle_mode = false;
  if (OB_FAIL(from_table_schema->check_if_oracle_compat_mode(is_oracle_mode))) {
    LOG_WARN("failed to check if oracle compat mode", K(ret));
  } else if (is_oracle_mode) {
    // do nothing, not support rename table to another db in oracle mode
  } else if (from_table_schema->get_database_id() != to_database_id) {
    // check if constraint/foreign key name is exist when rename table to another database in mysql mode
    ObTableSchema tmp_schema;
    bool is_constraint_name_exist = true;
    tmp_schema.set_table_id(from_table_schema->get_table_id());
    tmp_schema.set_tenant_id(from_table_schema->get_tenant_id());
    tmp_schema.set_table_type(from_table_schema->get_table_type());
    tmp_schema.set_database_id(to_database_id);
    // for check constraint
    ObTableSchema::const_constraint_iterator iter = from_table_schema->constraint_begin();
    for (;OB_SUCC(ret) && iter != from_table_schema->constraint_end(); ++iter) {
      if (OB_FAIL(check_constraint_name_is_exist(schema_guard, tmp_schema, (*iter)->get_constraint_name_str(), false, is_constraint_name_exist))) {
        LOG_WARN("fail to check check constraint name is exist or not", K(ret), K((*iter)->get_constraint_name_str()));
      } else if (is_constraint_name_exist) {
        ret = OB_ERR_CONSTRAINT_NAME_DUPLICATE;
        LOG_USER_ERROR(OB_ERR_CONSTRAINT_NAME_DUPLICATE, (*iter)->get_constraint_name_str().length(), (*iter)->get_constraint_name_str().ptr());
        LOG_WARN("check constraint name is duplicate", K(ret), K((*iter)->get_constraint_name_str()));
      }
    }
    // for foreign key constraint
    if (OB_SUCC(ret)) {
      const common::ObIArray<ObForeignKeyInfo>& foreign_keys = from_table_schema->get_foreign_key_infos();
      for (int i = 0; OB_SUCC(ret) && i < foreign_keys.count(); ++i) {
        if (OB_FAIL(check_constraint_name_is_exist(
                    schema_guard, tmp_schema, foreign_keys.at(i).foreign_key_name_, true, is_constraint_name_exist))) {
          LOG_WARN("fail to check foreign key name is exist or not", K(ret), K(foreign_keys.at(i).foreign_key_name_));
        } else if (is_constraint_name_exist) {
          ret = OB_ERR_DUP_KEY;
          LOG_USER_ERROR(OB_ERR_DUP_KEY, foreign_keys.at(i).foreign_key_name_.length(), foreign_keys.at(i).foreign_key_name_.ptr());
          LOG_WARN("foreign key name is duplicate", K(ret), K(foreign_keys.at(i).foreign_key_name_));
        }
      }
    }
  }

  return ret;
}

int ObDDLService::check_database_exist(
  const uint64_t tenant_id,
  const common::ObString &database_name,
  uint64_t &database_id)
{
  int ret = OB_SUCCESS;
  bool exist = false;
  if (OB_FAIL(check_inner_stat())) {
    LOG_WARN("variable is not init");
  } else if (OB_FAIL(schema_service_->check_database_exist(
        tenant_id, database_name, database_id, exist))) {
    LOG_WARN("check database exist failed", "database", database_name, K(ret));
  } else {
    if (!exist) {
      ret = OB_ERR_BAD_DATABASE;
      LOG_USER_ERROR(OB_ERR_BAD_DATABASE, database_name.length(), database_name.ptr());
    }
  }
  return ret;
}

int ObDDLService::check_table_exist(ObTableSchema &table_schema)
{
  int ret = OB_SUCCESS;
  if (OB_FAIL(check_inner_stat())) {
    LOG_WARN("variable is not init");
  } else {
    bool is_table_name_exist = false;
    bool is_table_id_exist = false;
    const int64_t table_schema_version = OB_INVALID_VERSION; // get the latest local schema_guard
    if (OB_FAIL(schema_service_->check_table_exist(
                table_schema.get_tenant_id(),
                table_schema.get_database_id(),
                table_schema.get_table_name_str(),
                table_schema.is_index_table(),
                table_schema_version,
                is_table_name_exist))) {
      LOG_WARN("failed to check is table exist",
               K(table_schema.get_tenant_id()),
               K(table_schema.get_database_id()),
               K(table_schema.is_index_table()),
               K(table_schema.get_table_name()), K(ret));
    } else if (OB_INVALID_ID != table_schema.get_table_id()
               && OB_FAIL(schema_service_->check_table_exist(
                          table_schema.get_tenant_id(),
                          table_schema.get_table_id(),
                          table_schema_version,
                          is_table_id_exist))) {
      LOG_WARN("failed to check is table exist",
               K(table_schema.get_tenant_id()),
               K(table_schema.get_table_id()), K(ret));
    } else if (is_table_name_exist || is_table_id_exist) {
      ret = OB_ERR_TABLE_EXIST;
      LOG_WARN("table is exist, cannot create it twice,",
               K(table_schema.get_tenant_id()),
               K(table_schema.get_database_id()),
               K(table_schema.get_table_id()),
               K(table_schema.get_table_name()), K(ret));
    }
  }
  return ret;
}

int ObDDLService::check_inner_stat() const
{
  int ret = OB_SUCCESS;
  if (!inited_) {
    ret = OB_NOT_INIT;
    LOG_WARN("not init", K(ret));
  } else if (OB_ISNULL(schema_service_) || OB_ISNULL(sql_proxy_)
      || OB_ISNULL(rpc_proxy_)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("schema_service_,sql_proxy_  or rpc_proxy_ is null", K(ret));
  } else if (OB_ISNULL(lst_operator_)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("lst_operator_ is null", KR(ret));
  } else if (OB_ISNULL(tenant_ddl_service_)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("tenant_ddl_service_ is null", K(ret), KP(tenant_ddl_service_));
  }
  return ret;
}

int ObDDLService::set_tablegroup_id(ObTableSchema &table_schema)
{
  int ret = OB_SUCCESS;

  uint64_t tablegroup_id = OB_INVALID_ID;
  uint64_t tenant_id = table_schema.get_tenant_id();
  ObSchemaGetterGuard schema_guard;
  if (OB_FAIL(check_inner_stat())) {
    LOG_WARN("variable is not init");
  } else if (OB_FAIL(get_tenant_schema_guard_with_version_in_inner_table(tenant_id, schema_guard))) {
    LOG_WARN("fail to get schema guard with version in inner table", K(ret), K(tenant_id));
  } else if (table_schema.get_tablegroup_name().empty()) {
    table_schema.set_tablegroup_id(OB_INVALID_ID);
  } else if (OB_FAIL(schema_guard.get_tablegroup_id(table_schema.get_tenant_id(),
      table_schema.get_tablegroup_name(), tablegroup_id))) {
    LOG_WARN("get_tablegroup_id failed", "tenant_id", table_schema.get_tenant_id(),
        "tablegroup_name", table_schema.get_tablegroup_name(), K(ret));
  } else if (OB_INVALID_ID == tablegroup_id) {
    ret = OB_TABLEGROUP_NOT_EXIST;
    LOG_WARN("group name not exist ", K(ret));
  } else {
    table_schema.set_tablegroup_id(tablegroup_id);
  }
  // if table's tablegroup does not exist, use database's default tablegroup
  if (OB_SUCC(ret) && OB_INVALID_ID == table_schema.get_tablegroup_id()) {
    const ObDatabaseSchema *db_schema = NULL;
    if (OB_FAIL(schema_guard.get_database_schema(tenant_id, table_schema.get_database_id(), db_schema))) {
      LOG_WARN("fail to get database schema", K(ret), K(tenant_id), "db_id", table_schema.get_database_id());
    } else if (OB_UNLIKELY(NULL == db_schema)) {
      ret = OB_ERR_BAD_DATABASE;
      LOG_WARN("fail to get database schema", K(ret), K(tenant_id), "db_id", table_schema.get_database_id());
    } else {
      table_schema.set_tablegroup_id(db_schema->get_default_tablegroup_id());
    }
  }
  // if database's default_tablegroup_id does not exist, use tenant's default tablegroup
  if (OB_SUCC(ret) && OB_INVALID_ID == table_schema.get_tablegroup_id()) {
    const ObTenantSchema *tenant_schema = NULL;
    if (OB_FAIL(schema_guard.get_tenant_info(table_schema.get_tenant_id(), tenant_schema))) {
      LOG_WARN("fail to get tenant schema", K(ret), "tenant_id", table_schema.get_tenant_id());
    } else if (OB_UNLIKELY(NULL == tenant_schema)) {
      ret = OB_TENANT_NOT_EXIST;
      LOG_WARN("fail to get tenant schema", K(ret), "tenant_id", table_schema.get_tenant_id());
    } else {
      table_schema.set_tablegroup_id(tenant_schema->get_default_tablegroup_id());
    }
  }

  // TODO: (2019.6.24 wendu) Cannot add replicated table to tablegroup
  if (OB_SUCC(ret)) {
    if (ObDuplicateScope::DUPLICATE_SCOPE_NONE != table_schema.get_duplicate_scope()
        && OB_INVALID_ID != table_schema.get_tablegroup_id()) {
      ret = OB_NOT_SUPPORTED;
      LOG_WARN("changing tablegroup of duplicate table is not supported", K(ret),
               "table_id", table_schema.get_table_id(),
               "tablegroup_id", table_schema.get_tablegroup_id());
      LOG_USER_ERROR(OB_NOT_SUPPORTED, "changing tablegroup of duplicate table is");
    }
  }

  if (OB_SUCC(ret)) {
    uint64_t table_id = table_schema.get_table_id();
    if (!(is_inner_table(table_id)
          || table_schema.is_user_table()
          || table_schema.is_mysql_tmp_table())) {
      table_schema.set_tablegroup_id(OB_INVALID_ID);
    }
  }

  return ret;
}

int ObDDLService::print_view_expanded_definition(
    const ObTableSchema &table_schema,
    ObString &ddl_stmt_str,
    common::ObIAllocator &allocator,
    ObSchemaGetterGuard &schema_guard,
    bool if_not_exist)
{
  int ret = OB_SUCCESS;
  char *buf = NULL;
  int64_t buf_len = OB_MAX_VARCHAR_LENGTH;
  int64_t pos = 0;
  bool is_oracle_mode;
  const ObDatabaseSchema *database_schema = NULL;
  const uint64_t tenant_id = table_schema.get_tenant_id();
  const int64_t database_id = table_schema.get_database_id();

  if (OB_ISNULL(buf = static_cast<char *>(allocator.alloc(buf_len)))) {
    ret = OB_ALLOCATE_MEMORY_FAILED;
    LOG_WARN("fail to allocate memory", K(ret), K(OB_MAX_VARCHAR_LENGTH));
  } else if (OB_FAIL(schema_guard.get_database_schema(tenant_id, database_id, database_schema))) {
    LOG_WARN("failed to get database schema", K(ret), K(tenant_id), K(database_id));
  } else if (OB_ISNULL(database_schema)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("database not exist", K(ret), K(database_id));
  } else if (OB_FAIL(table_schema.check_if_oracle_compat_mode(is_oracle_mode))) {
    LOG_WARN("fail to check is oracle mode", K(ret));
  } else if (OB_FAIL(databuff_printf(buf, buf_len, pos,
        is_oracle_mode ? "CREATE%s %sVIEW \"%s\".\"%s\" AS %.*s" : "CREATE%s %sVIEW `%s`.`%s` AS %.*s",
        if_not_exist ? " OR REPLACE" : "",
        table_schema.is_materialized_view() ? "MATERIALIZED " : "",
        database_schema->get_database_name(),
        table_schema.get_table_name(),
        table_schema.get_view_schema().get_view_definition_str().length(),
        table_schema.get_view_schema().get_view_definition_str().ptr()))) {
   LOG_WARN("fail to print view definition", K(ret));
  } else if (OB_FAIL(databuff_printf(buf, buf_len, pos, ";"))) {
   LOG_WARN("fail to print end", K(ret));
  } else {
   ddl_stmt_str.assign_ptr(buf, static_cast<int32_t>(pos));
  }
  return ret;
}

int ObDDLService::get_obj_privs_ora(const uint64_t tenant_id,
                                    const uint64_t obj_id,
                                    const uint64_t obj_type,
                                    ObSchemaGetterGuard &schema_guard,
                                    ObIArray<ObObjPriv> &obj_privs) {
  int ret = OB_SUCCESS;
  ObArray<const ObObjPriv*> obj_privs_pointer;
  if (OB_FAIL(schema_guard.get_obj_priv_with_obj_id(tenant_id,
                                                    obj_id,
                                                    obj_type,
                                                    obj_privs_pointer,
                                                    true))) {
    LOG_WARN("fail to get_obj_priv_with_obj_id", KR(ret), K(tenant_id), K(obj_id), K(obj_type));
  } else {
    for (int64_t i = 0; OB_SUCC(ret) && i < obj_privs_pointer.count(); ++i) {
      if (OB_ISNULL(obj_privs_pointer.at(i))) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("obj_privs_pointer contains NULL", KR(ret), K(i));
      } else if(OB_FAIL(obj_privs.push_back(*(obj_privs_pointer.at(i))))) {
        LOG_WARN("obj_privs fail to push back", KR(ret), K(i));
      }
    }
  }

  return ret;
}

int ObDDLService::create_tablets_in_trans_for_mv_(ObIArray<ObTableSchema> &table_schemas,
                                                  ObDDLOperator &ddl_operator,
                                                  ObMySQLTransaction &trans,
                                                  ObSchemaGetterGuard &schema_guard,
                                                  const uint64_t tenant_data_version)
{
  int ret = OB_SUCCESS;
  SCN frozen_scn;
  uint64_t tenant_id = OB_INVALID_ID;
  if (table_schemas.count() < 2) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("table_schemas count should not be smaller than 2", KR(ret), K(table_schemas.count()));
  } else {
    tenant_id = table_schemas.at(0).get_tenant_id();
  }

  if (OB_FAIL(ret)) {
  } else if (OB_ISNULL(GCTX.root_service_)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("root service is null", KR(ret));
  } else if (OB_FAIL(frozen_scn.convert_for_tx(
                 compaction::ObBasicMergeScheduler::INIT_COMPACTION_SCN))) {
    LOG_WARN("failed to get frozen status for create tablet", KR(ret), K(tenant_id));
  } else {
    ObTableCreator table_creator(
                   tenant_id,
                   frozen_scn,
                   trans);
    ObNewTableTabletAllocator new_table_tablet_allocator(
                              tenant_id,
                              schema_guard,
                              sql_proxy_);
    common::ObArray<share::ObLSID> ls_id_array;
    if (OB_FAIL(table_creator.init(true/*need_tablet_cnt_check*/))) {
      LOG_WARN("fail to init table creator", KR(ret));
    } else if (OB_FAIL(new_table_tablet_allocator.init())) {
      LOG_WARN("fail to init new table tablet allocator", KR(ret));
    }

    ObArray<const ObTableSchema*> tablet_schemas;
    ObArray<bool> need_create_empty_majors;
    int64_t last_schema_version = OB_INVALID_VERSION;
    // create tablets for container table and lob tables
    for (int64_t i = 0; OB_SUCC(ret) && (i < table_schemas.count()); ++i) {
      const ObTableSchema &this_table = table_schemas.at(i);
      const int64_t table_id = this_table.get_table_id();
      if (OB_INVALID_VERSION == last_schema_version
          && OB_FAIL(get_last_schema_version(last_schema_version))) {
        LOG_WARN("get last schema version failed", KR(ret));
      } else if (OB_FAIL(ddl_operator.insert_ori_schema_version(
          trans, tenant_id, table_id, last_schema_version))) {
        LOG_WARN("failed to insert_ori_schema_version!",
                KR(ret), K(tenant_id), K(table_id), K(last_schema_version));
      } else if (!this_table.is_materialized_view()) {
        // no need to create tablets for mv because its data is stored in container table
        if (OB_FAIL(tablet_schemas.push_back(&this_table))) {
          LOG_WARN("failed to push back", KR(ret), K(this_table));
        } else if (OB_FAIL(need_create_empty_majors.push_back(true))) {
          LOG_WARN("failed to push_back", KR(ret), K(this_table));
        }
      }
    }
    if (OB_SUCC(ret)) {
      const ObTableSchema &container_table = table_schemas.at(1);
      const ObTablegroupSchema *container_tablegroup_schema = NULL; // keep NULL if no tablegroup
      if (OB_INVALID_ID != container_table.get_tablegroup_id()) {
        if (OB_FAIL(schema_guard.get_tablegroup_schema(
            container_table.get_tenant_id(),
            container_table.get_tablegroup_id(),
            container_tablegroup_schema))) {
          LOG_WARN("get tablegroup schema failed", KR(ret), K(container_table));
        } else if (OB_ISNULL(container_tablegroup_schema)) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("container_tablegroup_schema is null", KR(ret), K(container_table));
        }
      }
      if (FAILEDx(new_table_tablet_allocator.prepare(trans, container_table, container_tablegroup_schema))) {
        LOG_WARN("fail to prepare ls for container table schema tablets", KR(ret));
      } else if (OB_FAIL(new_table_tablet_allocator.get_ls_id_array(ls_id_array))) {
        LOG_WARN("fail to get ls id array", KR(ret));
      } else if (OB_FAIL(table_creator.add_create_tablets_of_tables_arg(
          tablet_schemas,
          ls_id_array,
          tenant_data_version,
          need_create_empty_majors/*need_create_empty_major_sstable*/))) {
        LOG_WARN("create table partitions failed", KR(ret), K(tablet_schemas));
      } else if (OB_FAIL(table_creator.execute())) {
        LOG_WARN("execute create partition failed", KR(ret));
      }
    }

    // finishing is always invoked for new table tablet allocator
    int tmp_ret = OB_SUCCESS;
    if (OB_SUCCESS != (tmp_ret = new_table_tablet_allocator.finish(OB_SUCCESS == ret))) {
      LOG_WARN("fail to finish new table tablet allocator", KR(tmp_ret));
    }
  }
  return ret;
}


int ObDDLService::create_tablets_in_trans_(ObIArray<ObTableSchema> &table_schemas,
                                           ObDDLOperator &ddl_operator,
                                           ObMySQLTransaction &trans,
                                           ObSchemaGetterGuard &schema_guard,
                                           const uint64_t tenant_data_version)
{
  int ret = OB_SUCCESS;
  SCN frozen_scn;
  share::schema::ObTableSchema *first_table = nullptr;
  uint64_t tenant_id = OB_INVALID_ID;
  if (table_schemas.count() <= 0) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("table_schemas should not be emtpy", KR(ret), K(table_schemas.count()));
  } else {
    first_table = &table_schemas.at(0);
    tenant_id = first_table->get_tenant_id();
  }

  if (OB_FAIL(ret)) {
  } else if (OB_ISNULL(GCTX.root_service_)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("root service is null", KR(ret));
  } else if (OB_ISNULL(first_table)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("first_table is null", KR(ret));
  } else if (OB_FAIL(ObMajorFreezeHelper::get_frozen_scn(tenant_id, frozen_scn))) {
    LOG_WARN("failed to get frozen status for create tablet", KR(ret), K(tenant_id));
  } else {
    ObTableCreator table_creator(
                   tenant_id,
                   frozen_scn,
                   trans);
    ObNewTableTabletAllocator new_table_tablet_allocator(
                              tenant_id,
                              schema_guard,
                              sql_proxy_);
    common::ObArray<share::ObLSID> ls_id_array;
    const ObTablegroupSchema *data_tablegroup_schema = NULL; // keep NULL if no tablegroup
    if (OB_FAIL(table_creator.init(true/*need_tablet_cnt_check*/))) {
      LOG_WARN("fail to init table creator", KR(ret));
    } else if (OB_FAIL(new_table_tablet_allocator.init())) {
      LOG_WARN("fail to init new table tablet allocator", KR(ret));
    } else if (OB_INVALID_ID != first_table->get_tablegroup_id()) {
      if (OB_FAIL(schema_guard.get_tablegroup_schema(
          first_table->get_tenant_id(),
          first_table->get_tablegroup_id(),
          data_tablegroup_schema))) {
        LOG_WARN("get tablegroup_schema failed", KR(ret), KPC(first_table));
      } else if (OB_ISNULL(data_tablegroup_schema)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("data_tablegroup_schema is null", KR(ret), KPC(first_table));
      }
    }

    ObArray<const ObTableSchema*> schemas;
    ObArray<bool> need_create_empty_majors;
    int64_t last_schema_version = OB_INVALID_VERSION;
    for (int64_t i = 0; OB_SUCC(ret) && i < table_schemas.count(); i++) {
      const share::schema::ObTableSchema &this_table = table_schemas.at(i);
      const int64_t table_id = this_table.get_table_id();
      if (!this_table.has_tablet()) {
      } else if (!this_table.is_global_index_table()) {
        if (OB_FAIL(schemas.push_back(&this_table)) || OB_FAIL(need_create_empty_majors.push_back(true))) {
          LOG_WARN("failed to push_back", KR(ret), K(this_table));
        }
      } else {
        if (OB_FAIL(new_table_tablet_allocator.prepare(trans, this_table, data_tablegroup_schema))) {
          LOG_WARN("fail to prepare ls for index schema tablets", KR(ret));
        } else if (OB_FAIL(new_table_tablet_allocator.get_ls_id_array(
            ls_id_array))) {
          LOG_WARN("fail to get ls id array", KR(ret));
        } else if (OB_FAIL(table_creator.add_create_tablets_of_table_arg(
            this_table,
            ls_id_array,
            tenant_data_version,
            true /*need_create_empty_major_sstable*/))) {
          LOG_WARN("create table partitions failed", KR(ret), K(this_table));
        }
      }
      if (OB_SUCC(ret)) {
        if (OB_INVALID_VERSION == last_schema_version
          && OB_FAIL(get_last_schema_version(last_schema_version))) {
          LOG_WARN("get last schema version failed", KR(ret));
        } else if (OB_FAIL(ddl_operator.insert_ori_schema_version(
            trans, tenant_id, table_id, last_schema_version))) {
          LOG_WARN("failed to insert_ori_schema_version!",
                   KR(ret), K(tenant_id), K(table_id), K(last_schema_version));
        }
      }
    }

    if (OB_FAIL(ret)) {
    } else if (schemas.count() <= 0) {
      // virtual tablet and view skip
    } else if (OB_FAIL(new_table_tablet_allocator.prepare(trans, *first_table, data_tablegroup_schema))) {
      LOG_WARN("fail to prepare ls for first table", KR(ret), KPC(first_table));
    } else if (OB_FAIL(new_table_tablet_allocator.get_ls_id_array(
            ls_id_array))) {
      LOG_WARN("fail to get ls id array", KR(ret));
    } else if (OB_FAIL(table_creator.add_create_tablets_of_tables_arg(
            schemas,
            ls_id_array,
            tenant_data_version,
            need_create_empty_majors/*need_create_empty_major_sstable*/))) {
      LOG_WARN("create table partitions failed", KR(ret), KPC(first_table),
           K(last_schema_version));
    } else if (OB_FAIL(table_creator.execute())) {
      LOG_WARN("execute create partition failed", KR(ret));
    }

    // finishing is always invoked for new table tablet allocator
    int tmp_ret = OB_SUCCESS;
    if (OB_SUCCESS != (tmp_ret = new_table_tablet_allocator.finish(OB_SUCCESS == ret))) {
      LOG_WARN("fail to finish new table tablet allocator", KR(tmp_ret));
    }
  }
  return ret;
}

int ObDDLService::create_tables_in_trans(const bool if_not_exist,
                                         const ObString &ddl_stmt_str,
                                         const ObErrorInfo &error_info,
                                         ObIArray<ObTableSchema> &table_schemas,
                                         const obrpc::ObSequenceDDLArg &sequence_ddl_arg,
                                         const uint64_t last_replay_log_id,
                                         const ObIArray<ObDependencyInfo> *dep_infos,
                                         ObIArray<ObMockFKParentTableSchema> &mock_fk_parent_table_schema_array,
                                         int64_t &ddl_task_id)
{
  int ret = OB_SUCCESS;
  uint64_t tenant_data_version = 0;
  ObArenaAllocator allocator(ObModIds::OB_RS_PARTITION_TABLE_TEMP);
  RS_TRACE(create_tables_in_trans_begin);

  DEBUG_SYNC(BEFOR_EXECUTE_CREATE_TABLE_WITH_FTS_INDEX);

  if (OB_FAIL(check_inner_stat())) {
    LOG_WARN("variable is not init");
  } else if (table_schemas.count() < 1) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("table_schemas have no element", K(ret));
  } else if (OB_INVALID_ID == last_replay_log_id) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("invalid arg", K(ret), K(last_replay_log_id));
  } else {
    ObDDLSQLTransaction trans(schema_service_);
    ObDDLOperator ddl_operator(*schema_service_, *sql_proxy_);
    ObSchemaGetterGuard schema_guard;
    uint64_t tenant_id = table_schemas.at(0).get_tenant_id();
    share::schema::ObTableSchema *first_table = &table_schemas.at(0);
    uint64_t data_table_id = first_table->get_table_id();
    ObArray<ObObjPriv> orig_obj_privs_ora;
    const ObTableSchema *old_view_schema = NULL;
    bool is_oracle_mode = false;
    int64_t refreshed_schema_version = 0;
    ObDDLTaskRecord task_record;
    ddl_task_id = 0;
    ObMViewInfo mview_info;
    if (OB_FAIL(get_tenant_schema_guard_with_version_in_inner_table(tenant_id, schema_guard))) {
      LOG_WARN("fail to get schema guard with version in inner table", K(ret), K(tenant_id));
    } else if (OB_FAIL(schema_guard.get_schema_version(tenant_id, refreshed_schema_version))) {
      LOG_WARN("failed to get tenant schema version", KR(ret), K(tenant_id));
    } else if (OB_FAIL(trans.start(sql_proxy_, tenant_id, refreshed_schema_version))) {
      LOG_WARN("failed to start trans", KR(ret), K(tenant_id),
               K(tenant_id), K(refreshed_schema_version));
    } else if (OB_FAIL(first_table->check_if_oracle_compat_mode(is_oracle_mode))) {
      LOG_WARN("fail to check is oracle mode", KR(ret), KPC(first_table));
    } else if (OB_FAIL(GET_MIN_DATA_VERSION(tenant_id, tenant_data_version))) {
      LOG_WARN("get min data version failed", K(ret), K(tenant_id));
    }
    if (OB_SUCC(ret)) {
      ObString tmp_ddl_stmt_str = ddl_stmt_str;
      if (table_schemas.count() > 0) {
        const ObTableSchema &table_schema = table_schemas.at(0);
        if (table_schema.is_view_table()) {
          if (OB_FAIL(print_view_expanded_definition(
              table_schema,
              tmp_ddl_stmt_str,
              allocator,
              schema_guard,
              if_not_exist))) {
            LOG_WARN("fail to print_view_expanded_definition", K(ret), K(table_schema.get_table_id()));
          }
        }
        //create or replace view xxx
        if (OB_FAIL(ret)) {
        } else if (table_schema.is_view_table() && if_not_exist) {
          const ObString &view_name = table_schema.get_table_name();
          if (OB_FAIL(schema_guard.get_table_schema(table_schema.get_tenant_id(),
                                                    table_schema.get_database_id(),
                                                    view_name,
                                                    false, //is index
                                                    old_view_schema))) {
            LOG_WARN("failed to get table schema", K(view_name), K(ret));
          } else if (OB_ISNULL(old_view_schema)) {
            ret = OB_SUCCESS;
          } else {
            if (!is_oracle_mode) {
              // no need to store obj privs
            } else if (OB_FAIL(get_obj_privs_ora(table_schema.get_tenant_id(),
                                                 old_view_schema->get_table_id(),
                                                 static_cast<uint64_t>(ObObjectType::TABLE),
                                                 schema_guard,
                                                 orig_obj_privs_ora))) {
              LOG_WARN("fial to get obj privs ora", KR(ret), K(table_schema.get_tenant_id()),
                       K(old_view_schema->get_table_id()));
            }

            if (OB_FAIL(ret)) {
            } else if (OB_FAIL(ObPLDDLService::drop_trigger_in_drop_table(trans,
                                                                          ddl_operator,
                                                                          schema_guard,
                                                                          *old_view_schema,
                                                                          false))) {
              // 兼容oracle,create or replace view时drop trigger, 且不进回收站
              LOG_WARN("failed to drop trigger", KR(ret), K(old_view_schema->get_table_id()));
            } else if (OB_FAIL(ddl_operator.drop_table(*old_view_schema, trans))) {
              LOG_WARN("failed to drop old view schema", KR(ret));
            }
          }
        }
      }
      RS_TRACE(operator_create_table_begin);

      // Fill index/lob table for system table
      if (OB_SUCC(ret)
          && ObSysTableChecker::is_sys_table_has_index(data_table_id)) {
        if (OB_FAIL(ObSysTableChecker::append_sys_table_index_schemas(
                    tenant_id, data_table_id, table_schemas))) {
          LOG_WARN("fail to add sys table index", KR(ret), K(tenant_id), "table_id", data_table_id);
        } else {
          first_table = &table_schemas.at(0);
        }
      }

      if (OB_SUCC(ret) && is_system_table(data_table_id)) {
        HEAP_VARS_2((ObTableSchema, lob_meta_schema), (ObTableSchema, lob_piece_schema)) {
          if (OB_FAIL(add_sys_table_lob_aux(tenant_id, data_table_id, lob_meta_schema, lob_piece_schema))) {
            LOG_WARN("fail to get sys table lob aux schema", KR(ret), K(data_table_id));
          } else if (OB_FAIL(table_schemas.push_back(lob_meta_schema))) {
            LOG_WARN("fail to push back lob meta table", KR(ret), K(lob_meta_schema));
          } else if (OB_FAIL(table_schemas.push_back(lob_piece_schema))) {
            LOG_WARN("fail to push back lob piece table", KR(ret), K(lob_piece_schema));
          } else {
            first_table = &table_schemas.at(0);
          }
        }
      }

      for (int64_t i = 0; OB_SUCC(ret) && i < table_schemas.count(); i++) {
        ObTableSchema &table_schema = table_schemas.at(i);
        if (OB_FAIL(ObFtsIndexBuilderUtil::try_load_and_lock_dictionary_tables(table_schema, trans))) {
          LOG_WARN("fail to try load and lock dictionary tables", K(ret));
        } else if (OB_FAIL(ddl_operator.create_sequence_in_create_table(table_schema,
                                                                 trans,
                                                                 schema_guard,
                                                                 &sequence_ddl_arg))) {
          LOG_WARN("create sequence in create table fail", K(ret));
        } else if (OB_FAIL(ddl_operator.create_table(table_schema,
                                                     trans,
                                                     0 == i ? &tmp_ddl_stmt_str : NULL,
                                                     i == table_schemas.count() - 1))) {
          LOG_WARN("failed to create table schema, ", K(ret));
        } else if (OB_FAIL(ddl_operator.insert_temp_table_info(trans, table_schema))) {
          LOG_WARN("failed to insert_temp_table_info!", K(ret));
        } else if (table_schema.is_view_table() && dep_infos != nullptr && 0 == i) {
          for (int64_t i = 0 ; OB_SUCC(ret) && i < dep_infos->count(); ++i) {
            ObDependencyInfo dep;
            if (OB_FAIL(dep.assign(dep_infos->at(i)))) {
              LOG_WARN("failed to assign dependency info", K(ret));
            } else {
              dep.set_tenant_id(tenant_id);
              dep.set_dep_obj_id(table_schema.get_table_id());
              dep.set_dep_obj_owner_id(table_schema.get_table_id());
              dep.set_schema_version(table_schema.get_schema_version());
              OZ (dep.insert_schema_object_dependency(trans));
            }
          }
        }

        if (OB_SUCC(ret) && (0 == i) && table_schema.is_view_table() &&
            OB_NOT_NULL(old_view_schema) && is_oracle_mode) {
          const uint64_t db_id = table_schema.get_database_id();
          const ObSimpleDatabaseSchema *db_schema = NULL;
          if (OB_FAIL(schema_guard.get_database_schema(tenant_id, db_id, db_schema))) {
            LOG_WARN("failed to get database schema", KR(ret), K(db_id));
          } else if (OB_ISNULL(db_schema)) {
            ret = OB_ERR_BAD_DATABASE;
            LOG_WARN("db schema is NULL", KR(ret), K(tenant_id), K(db_id));
          } else if (OB_FAIL(restore_obj_privs_for_table(table_schema.get_table_id(),
                                                         db_schema->get_database_name_str(),
                                                         table_schema.get_table_name_str(),
                                                         ddl_operator,
                                                         trans,
                                                         orig_obj_privs_ora))) {
            LOG_WARN("restore_obj_privs_for_table failed", KR(ret), K(table_schema.get_table_id()),
                     K(db_schema->get_database_name_str()), K(table_schema.get_table_name_str()));
          }
        }

        if (OB_SUCC(ret) && (0 == i) && table_schema.is_materialized_view()) {
          const uint64_t db_id = table_schema.get_database_id();
          const ObSimpleDatabaseSchema *db_schema = NULL;
          if (OB_FAIL(schema_guard.get_database_schema(tenant_id, db_id, db_schema))) {
            LOG_WARN("failed to get database schema", KR(ret), K(db_id));
          } else if (OB_ISNULL(db_schema)) {
            ret = OB_ERR_BAD_DATABASE;
            LOG_WARN("db schema is NULL", KR(ret), K(tenant_id), K(db_id));
          } else if (OB_FAIL(ObMViewSchedJobUtils::add_mview_info_and_refresh_job(trans,
                                  tenant_id,
                                  table_schema.get_table_id(),
                                  db_schema->get_database_name_str(),
                                  table_schema.get_table_name_str(),
                                  table_schema.get_view_schema().get_mv_refresh_info(),
                                  table_schema.get_schema_version(),
                                  mview_info))) {
            LOG_WARN("fail to start mview refresh job", KR(ret));
          }
        }
      }

      // add error info for create force view
      if (OB_SUCC(ret) && 1 == table_schemas.count() && first_table->is_user_view()) {
        if (OB_LIKELY(ERROR_STATUS_HAS_ERROR != error_info.get_error_status())) {
          /* do nothing */
        } else if (OB_UNLIKELY(!is_oracle_mode)) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("unexpected compat mode add create view error info", K(ret), K(is_oracle_mode));
        } else {
          ObErrorInfo tmp_error_info;
          if (OB_FAIL(tmp_error_info.assign(error_info))) {
            LOG_WARN("failed to assign error info", K(ret));
          } else {
            tmp_error_info.set_obj_id(first_table->get_table_id());
            tmp_error_info.set_obj_type(static_cast<uint64_t>(ObObjectType::VIEW));
            tmp_error_info.set_database_id(first_table->get_database_id());
            tmp_error_info.set_tenant_id(first_table->get_tenant_id());
            tmp_error_info.set_schema_version(first_table->get_schema_version());
            if (OB_FAIL(tmp_error_info.handle_error_info(trans, NULL))) {
              LOG_WARN("insert create error info failed.", K(ret));
            }
          }
        }
      }
      if (FAILEDx(ddl_operator.deal_with_mock_fk_parent_tables(trans, schema_guard, mock_fk_parent_table_schema_array))) {
        LOG_WARN("fail to deal_with_mock_fk_parent_tables", K(ret), K(tenant_id));
      }

      if (OB_SUCC(ret)) {
        if (first_table->is_materialized_view()) {
          if (OB_FAIL(create_tablets_in_trans_for_mv_(table_schemas, ddl_operator, trans, schema_guard, tenant_data_version))) {
            LOG_WARN("fail to create tablets in trans for mv", KR(ret));
          }
        } else {
          if (OB_FAIL(create_tablets_in_trans_(table_schemas, ddl_operator, trans, schema_guard, tenant_data_version))) {
            LOG_WARN("fail to create tablets in trans", KR(ret));
          }
        }
        if (OB_SUCC(ret)
            && first_table->is_materialized_view()
            && table_schemas.count() >= 2) {
          ObTableSchema &mview_schema = table_schemas.at(0);
          ObTableSchema &container_table_schema = table_schemas.at(1);
          if (OB_FAIL(start_mview_complete_refresh_task(trans,
                                                        schema_guard,
                                                        mview_schema,
                                                        container_table_schema,
                                                        dep_infos,
                                                        allocator,
                                                        tenant_data_version,
                                                        mview_info,
                                                        task_record))) {
            LOG_WARN("failed to start mview complete refresh task", KR(ret));
          } else {
            ddl_task_id = task_record.task_id_;
          }
        }
      }
      RS_TRACE(operator_create_table_end);
    }
    DEBUG_SYNC(BEFORE_CREATE_TABLE_TRANS_COMMIT);
    if (OB_SUCC(ret)
        && THIS_WORKER.is_timeout_ts_valid()
        && THIS_WORKER.is_timeout()) {
      ret = OB_TIMEOUT;
      LOG_WARN("already timeout", KR(ret));
    }
    if (OB_SUCC(ret)) {
      ret = OB_E(EventTable::EN_CREATE_TABLE_TRANS_END_FAIL) OB_SUCCESS;
    }
    if (trans.is_started()) {
      int temp_ret = OB_SUCCESS;
      if (OB_SUCCESS != (temp_ret = trans.end(OB_SUCC(ret)))) {
        LOG_WARN("trans end failed", "is_commit", OB_SUCCESS == ret, K(temp_ret));
        ret = (OB_SUCC(ret)) ? temp_ret : ret;
      }
    }
    RS_TRACE(public_schema_begin);
    if (OB_SUCC(ret)) {
      DEBUG_SYNC(CREATE_TABLE_BEFORE_PUBLISH_SCHEMA);
      if (OB_FAIL(publish_schema(tenant_id))) {
        LOG_WARN("publish_schema failed", KR(ret), K(tenant_id));
      } else {
        RS_TRACE(public_schema_end);
      }
    }
    if (OB_SUCC(ret)
        && first_table->is_materialized_view()) {
      int tmp_ret = OB_SUCCESS;
      if (OB_TMP_FAIL(GCTX.root_service_->get_ddl_scheduler().schedule_ddl_task(task_record))) {
        LOG_WARN("fail to schedule ddl task", KR(tmp_ret), K(task_record));
      }
    }
  }
  RS_TRACE(create_tables_in_trans_end);
  return ret;
}

int ObDDLService::start_mview_complete_refresh_task(
    ObMySQLTransaction &trans,
    ObSchemaGetterGuard &schema_guard,
    const ObTableSchema &mview_schema,
    const ObTableSchema &container_table_schema,
    const ObIArray<ObDependencyInfo> *dep_infos,
    common::ObIAllocator &allocator,
    const uint64_t tenant_data_version,
    const ObMViewInfo &mview_info,
    ObDDLTaskRecord &task_record)
{
  int ret = OB_SUCCESS;
  int64_t max_dependency_version = 0;
  uint64_t tenant_id = mview_schema.get_tenant_id();
  uint64_t compat_version = 0;
  const ObMVRefreshInfo *mv_refresh_info = mview_schema.get_view_schema().get_mv_refresh_info();
  ObFixedLengthString<common::OB_MAX_TIMESTAMP_TZ_LENGTH> time_zone;
  const ObSysVarSchema *data_format_schema = nullptr;
  const ObSysVarSchema *nls_timestamp_format = nullptr;
  const ObSysVarSchema *nls_timestamp_tz_format = nullptr;
  obrpc::ObMViewCompleteRefreshArg arg;
  arg.tenant_id_ = tenant_id;
  arg.table_id_ = mview_schema.get_table_id();
  arg.consumer_group_id_ = THIS_WORKER.get_group_id();
  arg.session_id_ = 100;// FIXME
  arg.exec_tenant_id_ = tenant_id;
  if (OB_UNLIKELY(nullptr == dep_infos || nullptr == mv_refresh_info)) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("dep_infos is nullptr", KR(ret) , KP(dep_infos), KP(mv_refresh_info));
  } else if (OB_FAIL(GET_MIN_DATA_VERSION(tenant_id, compat_version))) {
    LOG_WARN("fail to get data version", KR(ret), K(tenant_id));
  } else if (OB_FAIL(arg.last_refresh_scn_.convert_for_inner_table_field(mview_info.get_last_refresh_scn()))) {
    LOG_WARN("fail to covert for inner table field", KR(ret), K(mview_info));
  } else if (OB_FAIL(share::ObBackupUtils::get_tenant_sys_time_zone_wrap(tenant_id,
                                                                  time_zone,
                                                                  arg.tz_info_wrap_))) {
    LOG_WARN("failed to get tenant sys timezoen wrap", K(ret));
  } else if (OB_FAIL(schema_guard.get_tenant_system_variable(tenant_id,
                                                             share::SYS_VAR_NLS_DATE_FORMAT,
                                                             data_format_schema))) {
    LOG_WARN("fail to get tenant system variable", K(ret));
  } else if (OB_FAIL(schema_guard.get_tenant_system_variable(tenant_id,
                                                             share::SYS_VAR_NLS_TIMESTAMP_FORMAT,
                                                             nls_timestamp_format))) {
    LOG_WARN("fail to get tenant system variable", K(ret));
  } else if (OB_FAIL(schema_guard.get_tenant_system_variable(tenant_id,
                                                             share::SYS_VAR_NLS_TIMESTAMP_TZ_FORMAT,
                                                             nls_timestamp_tz_format))) {
    LOG_WARN("fail to get tenant system variable", K(ret));
  } else if (OB_ISNULL(data_format_schema) || OB_ISNULL(nls_timestamp_format) || OB_ISNULL(nls_timestamp_tz_format)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("var schema must not be null", K(ret));
  } else {
    arg.parallelism_ = mv_refresh_info->parallel_;
    arg.tz_info_ =  arg.tz_info_wrap_.get_tz_info_offset();
    arg.nls_formats_[ObNLSFormatEnum::NLS_DATE] = data_format_schema->get_value();
    arg.nls_formats_[ObNLSFormatEnum::NLS_TIMESTAMP] = nls_timestamp_format->get_value();
    arg.nls_formats_[ObNLSFormatEnum::NLS_TIMESTAMP_TZ] = nls_timestamp_tz_format->get_value();
  }
  for (int64_t i = 0; OB_SUCC(ret) && i < dep_infos->count(); ++i) {
    const ObDependencyInfo &dep = dep_infos->at(i);
    const ObSimpleTableSchemaV2 *base_table_schema = nullptr;
    if ((compat_version < DATA_VERSION_4_3_5_0) && OB_UNLIKELY(ObObjectType::TABLE != dep.get_ref_obj_type())) {
      ret = OB_NOT_SUPPORTED;
      LOG_WARN("ref obj type is not table, not supported", KR(ret), K(dep));
      LOG_USER_ERROR(OB_NOT_SUPPORTED, "the ref obj type of materialized view not user table is");
    } else if ((compat_version >= DATA_VERSION_4_3_5_0)
        && OB_UNLIKELY((ObObjectType::TABLE != dep.get_ref_obj_type())
                        && ObObjectType::VIEW != dep.get_ref_obj_type())) {
      ret = OB_NOT_SUPPORTED;
      LOG_WARN("ref obj type is not table or mview, not supported", KR(ret), K(dep));
      LOG_USER_ERROR(OB_NOT_SUPPORTED, "the ref obj type of materialized view not user table or mview is");
    } else if (OB_FAIL(
          schema_guard.get_simple_table_schema(tenant_id, dep.get_ref_obj_id(), base_table_schema))) {
      LOG_WARN("fail to get table schema", K(ret), K(dep));
    } else if (OB_ISNULL(base_table_schema)) {
      ret = OB_TABLE_NOT_EXIST;
      LOG_WARN("base table schema is nullptr", K(ret));
    } else {
      ObBasedSchemaObjectInfo base_info;
      base_info.schema_id_ = dep.get_ref_obj_id();
      base_info.schema_type_ = ObSchemaType::TABLE_SCHEMA;
      base_info.schema_version_ = base_table_schema->get_schema_version();
      base_info.schema_tenant_id_ = tenant_id;
      max_dependency_version = MAX(max_dependency_version, base_table_schema->get_schema_version());
      if (OB_FAIL(arg.based_schema_object_infos_.push_back(base_info))) {
        LOG_WARN("fail to push back base info", KR(ret));
      }
    }
  }
  if (OB_SUCC(ret)) {
    if (OB_UNLIKELY(max_dependency_version > mview_schema.get_max_dependency_version())) {
      ret = OB_OLD_SCHEMA_VERSION;
      LOG_WARN("base table schema version is old",
          KR(ret), K(max_dependency_version), KPC(dep_infos),
          "old_max_dependency_version", mview_schema.get_max_dependency_version());
    } else {
      ObCreateDDLTaskParam param(tenant_id,
                        ObDDLType::DDL_CREATE_MVIEW,
                        &mview_schema,
                        &container_table_schema,
                        0/*object_id*/,
                        container_table_schema.get_schema_version(),
                        arg.parallelism_,
                        arg.consumer_group_id_,
                        &allocator,
                        &arg);
      param.tenant_data_version_ = tenant_data_version;
      if (OB_FAIL(GCTX.root_service_->get_ddl_task_scheduler().create_ddl_task(param, trans, task_record))) {
        LOG_WARN("submit create index ddl task failed", K(ret));
      }
    }
  }

  return ret;
}

// Create table information is written to the internal table within a transaction.
// If sql_trans is NULL, it need to create a transaction inside the function.
int ObDDLService::create_index_or_mlog_table_in_trans(
    ObTableSchema &table_schema,
    const ObString *ddl_stmt_str,
    ObMySQLTransaction *sql_trans,
    share::schema::ObSchemaGetterGuard &schema_guard,
    const bool need_check_tablet_cnt,
    const uint64_t tenant_data_version)
{
  int ret = OB_SUCCESS;
  if (OB_FAIL(check_inner_stat())) {
    LOG_WARN("variable is not init");
  } else {
    const uint64_t tenant_id = table_schema.get_tenant_id();
    int64_t refreshed_schema_version = 0;
    ObDDLSQLTransaction tmp_trans(schema_service_);
    ObMySQLTransaction &trans = OB_ISNULL(sql_trans) ? tmp_trans : *sql_trans;
    ObDDLOperator ddl_operator(*schema_service_, *sql_proxy_);
    if (OB_FAIL(schema_guard.get_schema_version(tenant_id, refreshed_schema_version))) {
      LOG_WARN("failed to get tenant schema version", KR(ret), K(tenant_id));
    } else if (OB_ISNULL(sql_trans)
        && OB_FAIL(trans.start(sql_proxy_, tenant_id, refreshed_schema_version))) {
      LOG_WARN("failed to start trans", KR(ret), K(tenant_id), K(refreshed_schema_version));
    } else if (OB_FAIL(ddl_operator.create_table(table_schema,
                                                 trans,
                                                 ddl_stmt_str))) {
      LOG_WARN("failed to create table schema, ", KR(ret));
		} else if (OB_FAIL(ddl_operator.insert_temp_table_info(trans, table_schema))) {
			LOG_WARN("failed to insert temp table info!", KR(ret));
    } else {
      LOG_INFO("succeed to insert table schema in schema tables",
          K(table_schema.get_tenant_id()),
          K(table_schema.get_database_id()),
          K(table_schema.get_table_id()),
          K(table_schema.get_table_name()));
    }
    if (OB_SUCC(ret)) {
      int64_t last_schema_version = OB_INVALID_VERSION;
      if (OB_FAIL(get_last_schema_version(last_schema_version))) {
        LOG_WARN("fail to get last schema version", KR(ret));
      } else if (OB_FAIL(ddl_operator.insert_ori_schema_version(trans, tenant_id, table_schema.get_table_id(), last_schema_version))) {
          LOG_WARN("failed to insert_ori_schema_version!", KR(ret), K(tenant_id), K(last_schema_version));
      }
    }
    if (OB_SUCC(ret) && table_schema.has_tablet()) {
      if (table_schema.is_mlog_table()) {
        if (OB_FAIL(create_mlog_tablet(trans, schema_guard, table_schema, need_check_tablet_cnt, tenant_data_version))) {
          LOG_WARN("failed to create_mlog_tablet", KR(ret), K(table_schema));
        }
      } else {
        if (OB_FAIL(create_index_tablet(table_schema, trans, schema_guard, need_check_tablet_cnt, tenant_data_version))) {
          LOG_WARN("fail to create_index_tablet", KR(ret), K(table_schema));
        }
      }
    }
    if (OB_ISNULL(sql_trans) && trans.is_started()) {
      int temp_ret = OB_SUCCESS;
      if (OB_SUCCESS != (temp_ret = trans.end(OB_SUCC(ret)))) {
        LOG_WARN("trans end failed", "is_commit", OB_SUCCESS == ret, K(temp_ret));
        ret = (OB_SUCC(ret)) ? temp_ret : ret;
      }
    }
  }
  return ret;
}

int ObDDLService::set_new_table_options(
    const obrpc::ObAlterTableArg &alter_table_arg,
    const share::schema::AlterTableSchema &alter_table_schema,
    const share::schema::ObTenantSchema &tenant_schema,
    share::schema::ObTableSchema &new_table_schema,
    const share::schema::ObTableSchema &orig_table_schema,
    share::schema::ObSchemaGetterGuard &schema_guard,
    bool &need_update_index_table)
{
  int ret = OB_SUCCESS;
  if ((alter_table_schema.alter_option_bitset_.has_member(obrpc::ObAlterTableArg::LOCALITY)
        || alter_table_schema.alter_option_bitset_.has_member(obrpc::ObAlterTableArg::PRIMARY_ZONE))
       && OB_INVALID_ID != new_table_schema.get_tablegroup_id()) {
    // 2.0 introduces constraints, it is not allowed to directly modify the primary_zone/locality
    // of the table in the tablegroup
    ret = OB_OP_NOT_ALLOW;
    LOG_WARN("modify primary_zone/locality of table in a tablegroup is not allowed", K(ret));
    LOG_USER_ERROR(OB_OP_NOT_ALLOW, "modify primary_zone/locality of table in a tablegroup");
  } else if (alter_table_schema.alter_option_bitset_.has_member(obrpc::ObAlterTableArg::LOCALITY)
             && alter_table_schema.alter_option_bitset_.has_member(obrpc::ObAlterTableArg::TABLEGROUP_NAME)) {
    // 2.0 introduces constraints, It is not allowed to change tablegroup and locality at the same time
    ret = OB_OP_NOT_ALLOW;
    LOG_WARN("modify locality and tablegroup at the same time is not allowed", K(ret));
    LOG_USER_ERROR(OB_OP_NOT_ALLOW, "modify locality and tablegroup at the same time");
  } else if (alter_table_schema.alter_option_bitset_.has_member(obrpc::ObAlterTableArg::STORAGE_FORMAT_VERSION)
      && alter_table_schema.get_storage_format_version() < orig_table_schema.get_storage_format_version()) {
    ret = OB_OP_NOT_ALLOW;
    LOG_WARN("cannot modify storage format version to lower version", K(ret), K(alter_table_schema.get_storage_format_version()),
        K(orig_table_schema.get_storage_format_version()));
    LOG_USER_ERROR(OB_OP_NOT_ALLOW, "target storage format version cannot be smaller than current version");
  } else if (alter_table_schema.alter_option_bitset_.has_member(obrpc::ObAlterTableArg::PROGRESSIVE_MERGE_ROUND)
      && alter_table_schema.get_progressive_merge_round() <= orig_table_schema.get_progressive_merge_round()) {
    ret = OB_OP_NOT_ALLOW;
    LOG_WARN("cannot modify progressive merge round to lower version", K(ret), K(alter_table_schema.get_progressive_merge_round()),
        K(orig_table_schema.get_progressive_merge_round()));
    LOG_USER_ERROR(OB_OP_NOT_ALLOW, "target progressive merge round cannot be smaller or equal to current version");
  } else if (OB_FAIL(set_raw_table_options(
          alter_table_schema, new_table_schema, schema_guard, need_update_index_table))) {
    LOG_WARN("fail to set raw table options", K(ret), K(new_table_schema), K(orig_table_schema));
  } else if (ObDuplicateScope::DUPLICATE_SCOPE_NONE != new_table_schema.get_duplicate_scope()
             && OB_INVALID_ID != new_table_schema.get_tablegroup_id()) {
    ret = OB_NOT_SUPPORTED;
    LOG_WARN("changing duplicate_scope of table in tablegroup is not supported", K(ret),
             "table_id", new_table_schema.get_table_id(),
             "tablegroup_id", new_table_schema.get_tablegroup_id());
    LOG_USER_ERROR(OB_NOT_SUPPORTED, "changing duplicate_scope of table in tablegroup is");
  } else {
    if (OB_SUCC(ret)
        && alter_table_schema.alter_option_bitset_.has_member(obrpc::ObAlterTableArg::TABLEGROUP_NAME)) {
      ObTableGroupHelp helper(*this, *schema_service_, *sql_proxy_);
      if (OB_FAIL(helper.check_table_alter_tablegroup(
                         schema_guard, NULL, orig_table_schema, new_table_schema))) {
        LOG_WARN("fail to check table schema in tablegroup", K(ret));
      } else {} // good
    }
    bool alter_collation = alter_table_schema.alter_option_bitset_.has_member(
                           obrpc::ObAlterTableArg::COLLATION_TYPE);
    bool alter_charset = alter_table_schema.alter_option_bitset_.has_member(
                         obrpc::ObAlterTableArg::CHARSET_TYPE);
    if (OB_SUCC(ret) && (alter_collation || alter_charset)) {
      ObCharsetType charset_type = alter_table_schema.get_charset_type();
      ObCollationType collation_type = alter_table_schema.get_collation_type();
      if (alter_collation && alter_charset) {
        if (!ObCharset::is_valid_collation(charset_type, collation_type)) {
          ret = OB_ERR_COLLATION_MISMATCH;
          const char *cs_name = ObCharset::charset_name(charset_type);
          const char *coll_name = ObCharset::collation_name(collation_type);
          ObString charset = ObString::make_string(cs_name);
          ObString collation = ObString::make_string(coll_name);
          LOG_USER_ERROR(OB_ERR_COLLATION_MISMATCH, collation.length(), collation.ptr(),
                           charset.length(), charset.ptr());
        } else {
          new_table_schema.set_collation_type(collation_type);
          new_table_schema.set_charset_type(charset_type);
        }
      } else if (alter_collation) {
        new_table_schema.set_collation_type(collation_type);
        new_table_schema.set_charset_type(ObCharset::charset_type_by_coll(collation_type));
      } else if (alter_charset) {
        new_table_schema.set_collation_type(ObCharset::get_default_collation(charset_type));
        new_table_schema.set_charset_type(charset_type);
      }
    }
  }
  LOG_DEBUG("set new table options", K(alter_table_arg), K(alter_table_schema),
            K(tenant_schema), K(new_table_schema), K(orig_table_schema));
  return ret;
}

//set new table options to new table schema
int ObDDLService::set_raw_table_options(
    const AlterTableSchema &alter_table_schema,
    ObTableSchema &new_table_schema,
    ObSchemaGetterGuard &schema_guard,
    bool &need_update_index_table)
{
  int ret = OB_SUCCESS;
  //replace alter options
  need_update_index_table = false;
  uint64_t tenant_id = new_table_schema.get_tenant_id();
  for (int32_t i = ObAlterTableArg::AUTO_INCREMENT;
       OB_SUCC(ret) && i < ObAlterTableArg::MAX_OPTION; ++i) {
    if (alter_table_schema.alter_option_bitset_.has_member(i)) {
      switch (i) {
        case ObAlterTableArg::TABLE_DOP: {
          new_table_schema.set_dop(alter_table_schema.get_dop());
          break;
        }
        case ObAlterTableArg::AUTO_INCREMENT: {
          new_table_schema.set_auto_increment(alter_table_schema.get_auto_increment());
          break;
        }
        case ObAlterTableArg::BLOCK_SIZE: {
          need_update_index_table = true;
          new_table_schema.set_block_size(alter_table_schema.get_block_size());
          break;
        }
        case ObAlterTableArg::CHARSET_TYPE: {
          new_table_schema.set_charset_type(alter_table_schema.get_charset_type());
          break;
        }
        case ObAlterTableArg::COLLATION_TYPE: {
          new_table_schema.set_collation_type(alter_table_schema.get_collation_type());
          break;
        }
        case ObAlterTableArg::STORE_FORMAT: {
          new_table_schema.set_row_store_type(alter_table_schema.get_row_store_type());
          new_table_schema.set_store_format(alter_table_schema.get_store_format());
          need_update_index_table = true;
          break;
        }
        case ObAlterTableArg::ENABLE_MACRO_BLOCK_BLOOM_FILTER: {
          new_table_schema.set_enable_macro_block_bloom_filter(alter_table_schema.get_enable_macro_block_bloom_filter());
          need_update_index_table = true;
          break;
        }
        case ObAlterTableArg::STORAGE_FORMAT_VERSION: {
          new_table_schema.set_storage_format_version(alter_table_schema.get_storage_format_version());
          need_update_index_table = true;
          break;
        }
        case ObAlterTableArg::COMPRESS_METHOD: {
          ret = new_table_schema.set_compress_func_name(alter_table_schema.get_compress_func_name());
          need_update_index_table = true;
          break;
        }
        case ObAlterTableArg::COMMENT: {
          ret = new_table_schema.set_comment(alter_table_schema.get_comment_str());
          break;
        }
        case ObAlterTableArg::EXPIRE_INFO: {
          ret = new_table_schema.set_expire_info(alter_table_schema.get_expire_info());
          break;
        }
        case ObAlterTableArg::PRIMARY_ZONE: {
          LOG_INFO("changing table's primay_zone does not take effect");
          ret = OB_SUCCESS; // do nothing
          break;
        }
        case ObAlterTableArg::REPLICA_NUM: {
          // ignore alter replica num
          break;
        }
        case ObAlterTableArg::TABLET_SIZE: {
          new_table_schema.set_tablet_size(alter_table_schema.get_tablet_size());
          need_update_index_table = true;
          break;
        }
        case ObAlterTableArg::PCTFREE: {
          new_table_schema.set_pctfree(alter_table_schema.get_pctfree());
          need_update_index_table = true;
          break;
        }
        case ObAlterTableArg::PROGRESSIVE_MERGE_NUM: {
          need_update_index_table = true;
          new_table_schema.set_progressive_merge_num(alter_table_schema.get_progressive_merge_num());
          break;
        }
        case ObAlterTableArg::PROGRESSIVE_MERGE_ROUND: {
          need_update_index_table = true;
          new_table_schema.set_progressive_merge_round(alter_table_schema.get_progressive_merge_round());
          break;
        }
        case ObAlterTableArg::TABLE_NAME: {
          //check table rename
          const ObString &new_table_name = alter_table_schema.get_table_name_str();
          const ObString &origin_table_name = alter_table_schema.get_origin_table_name();
          const ObString &new_database_name = alter_table_schema.get_database_name();
          const ObString &origin_database_name = alter_table_schema.get_origin_database_name();
          ObNameCaseMode mode = OB_NAME_CASE_INVALID;
          bool is_oracle_mode = false;
          const ObTableSchema *orig_table_schema = NULL;
          if (OB_FAIL(schema_guard.get_table_schema(tenant_id,
                                                    origin_database_name,
                                                    origin_table_name,
                                                    false,
                                                    orig_table_schema))) {
            LOG_WARN("fail to get table schema", K(ret), K(tenant_id),
                     K(origin_table_name), K(origin_database_name));
          } else if (NULL == orig_table_schema) {
            ret = OB_ERR_TABLE_EXIST;
            LOG_WARN("table not exist", K(ret));
          } else if (OB_FAIL(schema_guard.get_tenant_name_case_mode(tenant_id, mode))) {
            LOG_WARN("fail to get tenant name case mode", K(tenant_id), K(ret));
          } else if (OB_FAIL(orig_table_schema->check_if_oracle_compat_mode(is_oracle_mode))) {
            LOG_WARN("failed to get compat mode", K(ret), K(tenant_id));
          } else {
            //TODO rename database_name need to update all index table @hualong
            ObString database_name;
            if (!is_oracle_mode) {
              // mysql mode
              if (!new_database_name.empty() && !ObCharset::case_mode_equal(mode, new_database_name, origin_database_name)) {
                database_name = new_database_name;
              } else {
                database_name = origin_database_name;
              }
            } else {
              // oracle mode
              if (!new_database_name.empty() && !ObCharset::case_sensitive_equal(new_database_name, origin_database_name)) {
                database_name = new_database_name;
              } else {
                database_name = origin_database_name;
              }
            }
            ObString table_name;
            uint64_t database_id = OB_INVALID_ID;
            if (!is_oracle_mode) {
              // mysql mode
              if (!new_table_name.empty() && !ObCharset::case_mode_equal(mode, new_table_name, origin_table_name)) {
                table_name = new_table_name;
              } else {
                table_name = origin_table_name;
              }
            } else {
              // oracle mode
              if (!new_table_name.empty() && !ObCharset::case_sensitive_equal(new_table_name, origin_table_name)) {
                table_name = new_table_name;
              } else {
                table_name = origin_table_name;
              }
            }
            if (OB_FAIL(schema_guard.get_database_id(tenant_id, database_name, database_id))) {
              LOG_WARN("fail to get database id", K(tenant_id), K(database_name), K(ret));
            } else if (database_name != origin_database_name || table_name != origin_table_name) {
              if (OB_UNLIKELY(OB_RECYCLEBIN_SCHEMA_ID == database_id
                  || OB_PUBLIC_SCHEMA_ID == database_id)) {
                ret = OB_OP_NOT_ALLOW;
                LOG_WARN("rename table to hidden database is not allowd",
                         KR(ret), K(database_id), K(database_name), K(table_name));
              } else if (!is_oracle_mode) {
                const ObTableSchema *tmp_schema = NULL;
                const ObSynonymInfo *synonym_info = NULL;
                bool is_index = false;
                if (OB_FAIL(schema_guard.get_synonym_info(tenant_id, database_id, table_name, synonym_info))) {
                  LOG_WARN("fail to check synonym exist", KR(ret), K(tenant_id), K(database_name), K(table_name));
                } else if (OB_NOT_NULL(synonym_info)) {
                  ret = OB_ERR_EXIST_OBJECT;
                  LOG_WARN("Name is already used by an existing object", KR(ret), K(database_name), K(table_name));
                } else if (OB_FAIL(schema_guard.get_table_schema(tenant_id, database_name, table_name, is_index, tmp_schema))) {
                  LOG_WARN("fail to get table schema", KR(ret), K(tenant_id), K(database_name), K(table_name));
                } else if (OB_ISNULL(tmp_schema)) {
                  LOG_INFO("table not exist, can rename to new table name", K(new_table_name));
                } else {
                  ret = OB_ERR_TABLE_EXIST;
                  LOG_USER_ERROR(OB_ERR_TABLE_EXIST, table_name.length(), table_name.ptr());
                }
              } else {
                ObArray<ObSchemaType> conflict_schema_types;
                if (OB_FAIL(schema_guard.check_oracle_object_exist(
                    tenant_id, database_id, table_name, TABLE_SCHEMA,
                    INVALID_ROUTINE_TYPE, false /*if_not_eixst*/, conflict_schema_types))) {
                  LOG_WARN("fail to check oracle object exist", KR(ret), K(tenant_id),
                           K(database_id), K(database_name), K(table_name));
                 } else if (conflict_schema_types.count() > 0) {
                   ret = OB_ERR_EXIST_OBJECT;
                   LOG_WARN("Name is already used by an existing object in oralce mode",
                            KR(ret), K(tenant_id), K(database_id), K(database_name),
                            K(table_name), K(conflict_schema_types));
                }
              }
            }
            if (OB_SUCC(ret)) {
              if (OB_FAIL(new_table_schema.set_table_name(table_name))) {
                LOG_WARN("fail to set table name", K(table_name), K(ret));
              } else {
                need_update_index_table = true;
                new_table_schema.set_database_id(database_id);
              }
            }
            if (OB_SUCC(ret)
                && OB_FAIL(check_cst_name_dup_for_rename_table_mysql(
                           schema_guard, orig_table_schema, new_table_schema.get_database_id()))) {
              LOG_WARN("fail to check cst name dup for rename table mysql", K(ret));
            }
          }
          break;
        }
        case ObAlterTableArg::TABLEGROUP_NAME: {
          const ObString &tablegroup_name = alter_table_schema.get_tablegroup_name();
          if (!tablegroup_name.empty()) {
            uint64_t tablegroup_id = OB_INVALID_ID;
            uint64_t tenant_id = alter_table_schema.get_tenant_id();
            //tablegroup_id not set in resolver, only record tablegroup name
            if (OB_FAIL(schema_guard.get_tablegroup_id(tenant_id,
                                                       tablegroup_name,
                                                       tablegroup_id))) {
              LOG_WARN("failed to get tablegroup id", K(ret), K(tablegroup_name));
            } else if (OB_INVALID_ID == tablegroup_id) {
              ret = OB_TABLEGROUP_NOT_EXIST;
              LOG_WARN("invalid tablegroup name", K(ret), K(tablegroup_name));
            } else {
              new_table_schema.set_tablegroup_id(tablegroup_id);
              need_update_index_table = true;
            }
          } else {
            new_table_schema.set_tablegroup_id(OB_INVALID_ID);
            need_update_index_table = true;
          }
          break;
        }
        case ObAlterTableArg::SEQUENCE_COLUMN_ID: {
          //TODO
          break;
        }
        case ObAlterTableArg::USE_BLOOM_FILTER: {
          new_table_schema.set_is_use_bloomfilter(alter_table_schema.is_use_bloomfilter());
          break;
        }
        case ObAlterTableArg::READ_ONLY: {
          new_table_schema.set_read_only(alter_table_schema.is_read_only());
          need_update_index_table = true;
          break;
        }
        case ObAlterTableArg::LOCALITY: {
          if (is_sys_table(new_table_schema.get_table_id())) {
            ret = OB_NOT_SUPPORTED;
            LOG_WARN("change system table's locality is not allowed", K(ret));
            LOG_USER_ERROR(OB_NOT_SUPPORTED, "change system table's locality");
          } else {
            LOG_INFO("changing table's locality does not take effect");
          }
          break;
        }
        case ObAlterTableArg::SESSION_ID: {
          bool is_index = false;
          const ObString &table_name = new_table_schema.get_table_name_str();
          const uint64_t database_id = new_table_schema.get_database_id();
          uint64_t tenant_id = new_table_schema.get_tenant_id();
          const ObTableSchema *found_table_schema = NULL;
          if (0 == new_table_schema.get_session_id()
              || alter_table_schema.get_session_id() == new_table_schema.get_session_id()) {
            ret = OB_ERR_UNEXPECTED;
            LOG_WARN("table session id is reset already", K(ret), K(new_table_schema), K(alter_table_schema));
          } else {
            // bug18197606, In the process of querying or building a table,
            // another session creates a table with the same name,
            // Modify the first two can lead to the same name table and session_id = 0;
            // It is necessary to check whether there is a normal table with session_id = 0
            // in the last step of querying the table creation. If it is error CTAS will be rolled back.
            uint64_t  org_session_id = schema_guard.get_session_id();
            schema_guard.set_session_id(alter_table_schema.get_session_id());
            if (OB_FAIL(schema_guard.get_table_schema(tenant_id,
                                                      database_id,
                                                      table_name,
                                                      is_index,
                                                      found_table_schema))) {
              LOG_WARN("get_table_schema failed", K(tenant_id),
                      K(database_id), K(table_name), K(is_index), K(ret));
            } else if (OB_NOT_NULL(found_table_schema)) {
              ret = OB_ERR_TABLE_EXIST;
              LOG_USER_ERROR(OB_ERR_TABLE_EXIST, table_name.length(), table_name.ptr());
            } else {
              new_table_schema.set_session_id(alter_table_schema.get_session_id());
            }
            schema_guard.set_session_id(org_session_id);
          }
          break;
        }
        case ObAlterTableArg::SESSION_ACTIVE_TIME: {
          new_table_schema.set_sess_active_time(alter_table_schema.get_sess_active_time());
          break;
        }
        case ObAlterTableArg::DUPLICATE_SCOPE: {
          // alter table duplicate scope not allowed in master now
          new_table_schema.set_duplicate_attribute(alter_table_schema.get_duplicate_scope(),
                                                   alter_table_schema.get_duplicate_read_consistency());
          need_update_index_table = true;
          break;
        }
        case ObAlterTableArg::ENABLE_ROW_MOVEMENT: {
          new_table_schema.set_enable_row_movement(alter_table_schema.is_enable_row_movement());
          break;
        }
        case ObAlterTableArg::FORCE_LOCALITY: {
          // do nothing
          break;
        }
        case ObAlterTableArg::TABLE_MODE: {
          uint64_t compat_version = 0;
          if (OB_FAIL(GET_MIN_DATA_VERSION(tenant_id, compat_version))) {
            LOG_WARN("get min data_version failed", K(ret), K(tenant_id));
          } else if (not_compat_for_queuing_mode(compat_version) && alter_table_schema.is_new_queuing_table_mode()) {
            ret = OB_NOT_SUPPORTED;
            LOG_WARN(QUEUING_MODE_NOT_COMPAT_WARN_STR, K(ret), K(compat_version), K(alter_table_schema.get_table_mode()));
            LOG_USER_ERROR(OB_NOT_SUPPORTED, QUEUING_MODE_NOT_COMPAT_USER_ERROR_STR);
          } else {
            new_table_schema.set_table_mode(alter_table_schema.get_table_mode());
            need_update_index_table = true;
          }
          break;
        }
        case ObAlterTableArg::INCREMENT_MODE : {
          new_table_schema.set_table_auto_increment_mode(
            alter_table_schema.get_table_auto_increment_mode());
          break;
        }
        case ObAlterTableArg::INCREMENT_CACHE_SIZE : {
          uint64_t data_version = OB_INVALID_VERSION;
          if (OB_FAIL(GET_MIN_DATA_VERSION(tenant_id, data_version))) {
            LOG_WARN("get min data_version failed", K(ret), K(tenant_id));
          } else if (data_version < MOCK_DATA_VERSION_4_2_3_0 ||
              (data_version >= DATA_VERSION_4_3_0_0 && data_version < DATA_VERSION_4_3_2_0)) {
            ret = OB_NOT_SUPPORTED;
            LOG_WARN("table auto_increment_cache_size less than 4.2.3 not support", K(ret),
                     K(data_version));
            LOG_USER_ERROR(OB_NOT_SUPPORTED, "table auto_increment_cache_size less than 4.2.3");
          } else {
            new_table_schema.set_auto_increment_cache_size(
              alter_table_schema.get_auto_increment_cache_size());
          }
          break;
        }
        case ObAlterTableArg::ENABLE_EXTENDED_ROWID: {
          new_table_schema.set_table_rowid_mode(alter_table_schema.get_table_rowid_mode());
          break;
        }
        case ObAlterTableArg::ENCRYPTION: {
          if (OB_FAIL(new_table_schema.set_encryption_str(alter_table_schema.get_encryption_str()))) {
            LOG_WARN("fail to set encryption_str", K(ret), K(alter_table_schema.get_encryption_str()));
          }
          break;
        }
        case ObAlterTableArg::TABLESPACE_ID: {
          new_table_schema.set_tablespace_id(alter_table_schema.get_tablespace_id());
          if (OB_FAIL(new_table_schema.set_encryption_str(alter_table_schema.get_encryption_str()))) {
            LOG_WARN("fail to set encryption_str", K(ret), K(alter_table_schema.get_encryption_str()));
          }
          break;
        }
        case ObAlterTableArg::TTL_DEFINITION: {
          uint64_t compat_version = OB_INVALID_VERSION;
          if (OB_FAIL(GET_MIN_DATA_VERSION(tenant_id, compat_version))) {
            LOG_WARN("get min data_version failed", K(ret), K(tenant_id));
          } else if (compat_version < DATA_VERSION_4_2_1_0) {
            ret = OB_NOT_SUPPORTED;
            LOG_WARN("ttl definition less than 4.2.1 not support", K(ret), K(compat_version));
            LOG_USER_ERROR(OB_NOT_SUPPORTED, "ttl definition less than 4.2.1");
          } else if (OB_FAIL(new_table_schema.set_ttl_definition(alter_table_schema.get_ttl_definition()))) {
            LOG_WARN("fail to set ttl definition", K(ret));
          }
          break;
        }
        case ObAlterTableArg::KV_ATTRIBUTES: {
          uint64_t compat_version = OB_INVALID_VERSION;
          if (OB_FAIL(GET_MIN_DATA_VERSION(tenant_id, compat_version))) {
            LOG_WARN("get min data_version failed", K(ret), K(tenant_id));
          } else if (compat_version < DATA_VERSION_4_2_1_0) {
            ret = OB_NOT_SUPPORTED;
            LOG_WARN("kv attributes less than 4.2.1 not support", K(ret), K(compat_version));
            LOG_USER_ERROR(OB_NOT_SUPPORTED, "kv attributes less than 4.2");
          } else if (OB_FAIL(new_table_schema.set_kv_attributes(alter_table_schema.get_kv_attributes()))) {
            LOG_WARN("fail to set kv attributes", K(ret));
          }
          break;
        }
        case ObAlterTableArg::LOB_INROW_THRESHOLD: {
          uint64_t compat_version = OB_INVALID_VERSION;
          if (OB_FAIL(GET_MIN_DATA_VERSION(tenant_id, compat_version))) {
            LOG_WARN("get min data_version failed", K(ret), K(tenant_id));
          } else if (compat_version < DATA_VERSION_4_2_1_2) {
            ret = OB_NOT_SUPPORTED;
            LOG_WARN("lob inrow threshold less than 4.2.1.2 not support", K(ret), K(compat_version));
            LOG_USER_ERROR(OB_NOT_SUPPORTED, "lob inrow threshold less than 4.2.1.2");
          } else if (OB_FAIL(ObVectorIndexUtil::check_ivf_lob_inrow_threshold(
              tenant_id,
              alter_table_schema.get_origin_database_name(),
              alter_table_schema.get_origin_table_name(),
              schema_guard,
              alter_table_schema.get_lob_inrow_threshold()))) {
            LOG_WARN("fail to check table has vector ivf index", K(ret), K(tenant_id));
          } else {
            new_table_schema.set_lob_inrow_threshold(alter_table_schema.get_lob_inrow_threshold());
            need_update_index_table = true;
          }
          break;
        }
        default: {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("Unknown option!", K(i));
        }
      }
    }
  }
  return ret;
}

int ObDDLService::generate_index_name(ObCreateIndexArg &create_index_arg,
                                      const ObTableSchema &origin_table_schema,
                                      const AddIndexNameHashSet &add_index_name_set,
                                      const DropIndexNameHashSet &drop_index_name_set,
                                      ObSchemaGetterGuard &schema_guard,
                                      ObArenaAllocator &allocator)
{
  int ret = OB_SUCCESS;
  //inspect whether first_column_name is exist
  if (create_index_arg.index_columns_.count() < 1) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("create_index_arg.index_columns_ have no element", K(ret));
  } else {
    const ObColumnSortItem &sort_column = create_index_arg.index_columns_.at(0);
    const ObString &first_col_name = sort_column.column_name_;
    //check in old database
    const uint64_t tenant_id = origin_table_schema.get_tenant_id();
    const uint64_t database_id = origin_table_schema.get_database_id();

    //short index name
    ObString index_name = first_col_name;
    bool exist = true;
    //use first column name as index name
    if (OB_FAIL(check_index_table_exist(tenant_id,
                                        database_id,
                                        origin_table_schema.get_table_id(),
                                        index_name,
                                        schema_guard,
                                        exist))) {
      LOG_WARN("failed to check index table exist!", K(index_name),
               K(origin_table_schema), K(ret));
    } else {
      ObIndexNameHashWrapper index_key(index_name);
      if (!exist) {
        if (OB_HASH_EXIST == add_index_name_set.exist_refactored(index_key)) {
          exist = true;
        }
      } else {
        if (OB_HASH_EXIST == drop_index_name_set.exist_refactored(index_key)) {
          exist = false;
        }
      }
    }
    //genereate new index name use the first column c1_2, c1_3..
    if (OB_SUCC(ret) && exist) {
      index_name.reset();
      char buffer[number::ObNumber::MAX_PRINTABLE_SIZE];
      for (int32_t i = 2; OB_SUCC(ret) && exist; ++i) {
        (void)snprintf(buffer, sizeof(buffer), "%.*s_%d", first_col_name.length(),
                 first_col_name.ptr(), i);
        if (OB_FAIL(ob_write_string(allocator,
                                    ObString::make_string(buffer),
                                    index_name))) {
          LOG_WARN("Can not malloc space for index name", K(ret));
        } else if (OB_FAIL(check_index_table_exist(tenant_id,
                                                   database_id,
                                                   origin_table_schema.get_table_id(),
                                                   index_name,
                                                   schema_guard,
                                                   exist))) {
          LOG_WARN("failed to check table schema",
                   K(database_id), K(tenant_id), K(index_name),
                   K(origin_table_schema), K(ret));
        } else if (exist) {
          continue;
        } else {
          ObIndexNameHashWrapper index_key(index_name);
          if (!exist) {
            if (OB_HASH_EXIST == add_index_name_set.exist_refactored(index_key)) {
              exist = true;
            }
          } else {
            if (OB_HASH_EXIST == drop_index_name_set.exist_refactored(index_key)) {
              exist = false;
            }
          }
        }
      }
    }
    if (OB_SUCC(ret)) {
      create_index_arg.index_name_ = index_name;
    }
  }
  return ret;
}

int ObDDLService::check_index_table_exist(const uint64_t tenant_id,
                                          const uint64_t database_id,
                                          const uint64_t table_id,
                                          const ObString &index_name,
                                          ObSchemaGetterGuard &schema_guard,
                                          bool &is_exist)
{
  int ret = OB_SUCCESS;
  is_exist = false;
  ObString index_table_name;
  const ObTableSchema *index_schema = NULL;
  bool is_index = true;
  ObArenaAllocator allocator(ObModIds::OB_RS_PARTITION_TABLE_TEMP);
  if (OB_FAIL(ObTableSchema::build_index_table_name(allocator,
                                                    table_id,
                                                    index_name,
                                                    index_table_name))) {
    LOG_WARN("failed to build index table name", K(index_name), K(index_table_name), K(ret));
  } else if (OB_FAIL(schema_guard.get_table_schema(tenant_id,
                                                  database_id,
                                                  index_table_name,
                                                  is_index,
                                                  index_schema))) {
    LOG_WARN("failed to check table schema",
         K(database_id), K(tenant_id), K(index_name),
         K(index_table_name), K(ret));
  } else if (NULL != index_schema) {
    is_exist = true;
  }
  allocator.clear();
  return ret;
}

int ObDDLService::check_hidden_index_exist(
    const uint64 tenant_id,
    const uint64_t database_id,
    const ObTableSchema &hidden_table_schema,
    const ObString &index_name,
    ObSchemaGetterGuard &schema_guard,
    bool &is_exist)
{
  int ret = OB_SUCCESS;
  bool is_oracle_mode = false;
  is_exist = false;
  ObArray<const ObSimpleTableSchemaV2 *> table_schemas;
  if (OB_FAIL(hidden_table_schema.check_if_oracle_compat_mode(is_oracle_mode))) {
    LOG_WARN("check if oracle mode failed", K(ret), K(hidden_table_schema));
  } else if (is_oracle_mode) {
    // in oracle mode, we need iterate hidden table map within database
    if (OB_FAIL(schema_guard.get_table_schemas_in_database(tenant_id, database_id, table_schemas))) {
      LOG_WARN("get table schemas in database", K(ret));
    }
  } else {
    // is mysql mode, just check index_name exists in hidden_table_schema
    if (OB_FAIL(schema_guard.get_index_schemas_with_data_table_id(tenant_id, hidden_table_schema.get_table_id(), table_schemas))) {
      LOG_WARN("get index schemas failed", K(ret));
    }
  }

  // check if the index name exist in index_schemas array
  if (OB_SUCC(ret)) {
    ObString tmp_index_name;
    ObCompareNameWithTenantID column_name_cmp(hidden_table_schema.get_tenant_id());
    for (int64_t i = 0; OB_SUCC(ret) && i < table_schemas.count() && !is_exist; ++i) {
      const ObSimpleTableSchemaV2 *table_schema = table_schemas.at(i);
      if (OB_ISNULL(table_schema)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("error unexpected, index schema must not be nullptr", K(ret));
      } else if (table_schema->is_index_table() && OB_FAIL(table_schema->get_index_name(tmp_index_name))) {
        LOG_WARN("get index name failed", K(ret));
      } else {
        is_exist = 0 == column_name_cmp.compare(tmp_index_name, index_name);
      }
    }
  }
  return ret;
}

// Used for alter table xxx drop primary key.
// reset origin rowkey info and add heap table hidden pk column.
int ObDDLService::drop_primary_key(
    ObTableSchema &new_table_schema)
{
  int ret = OB_SUCCESS;
  // step1: clear origin primary key.
  const ObRowkeyInfo &rowkey_info = new_table_schema.get_rowkey_info();
  for (int64_t i = 0; OB_SUCC(ret) && i < rowkey_info.get_size(); i++) {
    const ObRowkeyColumn *rowkey_column = rowkey_info.get_column(i);
    ObColumnSchemaV2 *col = nullptr;
    if (OB_ISNULL(col = new_table_schema.get_column_schema(rowkey_column->column_id_))) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("col is nullptr", K(ret), K(rowkey_column->column_id_), K(new_table_schema));
    } else if (OB_HIDDEN_SESSION_ID_COLUMN_ID == col->get_column_id()) {
      ret = OB_NOT_SUPPORTED;
      LOG_USER_ERROR(OB_NOT_SUPPORTED, "Dropping primary key of global temporary table");
    } else {
      col->set_rowkey_position(0);
    }
  }
  if (OB_SUCC(ret)) {
    // step2: set table pk mode and origanization mode.
    new_table_schema.set_table_pk_mode(ObTablePKMode::TPKM_TABLET_SEQ_PK);
    new_table_schema.set_table_pk_exists_mode(ObTablePrimaryKeyExistsMode::TOM_TABLE_WITHOUT_PK);
    // step3: add hidden pk column.
    new_table_schema.reset_rowkey_info();
    int32_t rowkey_position = 1;
    ObColumnSchemaV2 hidden_pk;
    hidden_pk.reset();
    hidden_pk.set_column_id(OB_HIDDEN_PK_INCREMENT_COLUMN_ID);
    hidden_pk.set_data_type(ObUInt64Type);
    hidden_pk.set_nullable(false);
    hidden_pk.set_is_hidden(true);
    hidden_pk.set_charset_type(CHARSET_BINARY);
    hidden_pk.set_collation_type(CS_TYPE_BINARY);
    if (OB_FAIL(hidden_pk.set_column_name(OB_HIDDEN_PK_INCREMENT_COLUMN_NAME))) {
      LOG_WARN("failed to set column name", K(ret));
    } else {
      hidden_pk.set_rowkey_position(rowkey_position);
      if (OB_FAIL(new_table_schema.add_column(hidden_pk))) {
        LOG_WARN("add column to table_schema failed", K(ret), K(hidden_pk));
      }
    }
  }
  return ret;
}

int ObDDLService::add_primary_key(const ObIArray<ObString> &pk_column_names, ObTableSchema &new_table_schema)
{
  int ret = OB_SUCCESS;
  // step1: clear origin primary key
  ObTableSchema::const_column_iterator tmp_begin = new_table_schema.column_begin();
  ObTableSchema::const_column_iterator tmp_end = new_table_schema.column_end();
  ObColumnSchemaV2 *del_hidden_pk_column = nullptr;
  if (pk_column_names.count() > common::OB_USER_MAX_ROWKEY_COLUMN_NUMBER) {
    ret = OB_ERR_TOO_MANY_ROWKEY_COLUMNS;
    LOG_USER_ERROR(OB_ERR_TOO_MANY_ROWKEY_COLUMNS, OB_USER_MAX_ROWKEY_COLUMN_NUMBER);
    LOG_WARN("too many rowkey columns", K(ret), K(pk_column_names.count()));
  }
  for (; OB_SUCC(ret) && tmp_begin != tmp_end; tmp_begin++) {
    ObColumnSchemaV2 *col = (*tmp_begin);
    if (OB_ISNULL(col)) {
      ret = OB_ERR_UNEXPECTED;
    } else if (OB_HIDDEN_PK_INCREMENT_COLUMN_ID == col->get_column_id()) {
      del_hidden_pk_column = col;
    } else if (OB_HIDDEN_SESSION_ID_COLUMN_ID == col->get_column_id()) {
      ret = OB_NOT_SUPPORTED;
      LOG_USER_ERROR(OB_NOT_SUPPORTED, "Adding primary key to global temporary table");
    } else {
      col->set_rowkey_position(0);
    }
    new_table_schema.set_table_pk_mode(ObTablePKMode::TPKM_OLD_NO_PK);
    new_table_schema.set_table_pk_exists_mode(ObTablePrimaryKeyExistsMode::TOM_TABLE_WITH_PK);
  }

  if (OB_SUCC(ret) && nullptr != del_hidden_pk_column) {
    // delete hidden primary key
    if (OB_FAIL(new_table_schema.delete_column(del_hidden_pk_column->get_column_name_str()))) {
      LOG_WARN("fail to delete hidden primary key", K(ret));
    }
  }

  if (OB_SUCC(ret)) {
    // step2: set new primary key rowkey_position
    int64_t rowkey_position = 1;
    new_table_schema.reset_rowkey_info();
    for (int64_t i = 0; OB_SUCC(ret) && i < pk_column_names.count(); i++) {
      ObColumnSchemaV2 *col_schema = nullptr;
      const ObString &col_name = pk_column_names.at(i);
      if (OB_ISNULL(col_schema = new_table_schema.get_column_schema(col_name))) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("unexpected err, can not find column", K(ret), K(col_name), K(new_table_schema));
      } else {
        col_schema->set_rowkey_position(rowkey_position++);
        col_schema->set_nullable(false);
        if (OB_FAIL(new_table_schema.set_rowkey_info(*col_schema))) {
          LOG_WARN("set rowkey info failed", K(ret));
        }
      }
    }
    if (OB_SUCC(ret) && OB_FAIL(new_table_schema.check_primary_key_cover_partition_column())) {
      LOG_WARN("fail to check primary key cover partition column", K(ret));
    }
  }
  return ret;
}

int ObDDLService::create_hidden_table_with_pk_changed(
                  const obrpc::ObAlterTableArg &alter_table_arg,
                  const ObSArray<ObString> &index_columns,
                  const ObTableSchema &origin_table_schema,
                  ObTableSchema &new_table_schema,
                  ObSchemaGetterGuard &schema_guard,
                  ObDDLOperator &ddl_operator,
                  ObMySQLTransaction &trans,
                  ObArenaAllocator &allocator,
                  const ObIndexArg::IndexActionType &index_action_type,
                  const uint64_t tenant_data_version)
{
  int ret = OB_SUCCESS;
  const bool bind_tablets = false;
  ObString index_name("");
  const bool is_drop_pk = ObIndexArg::DROP_PRIMARY_KEY == index_action_type;
  const bool is_add_or_alter_pk = (ObIndexArg::ADD_PRIMARY_KEY == index_action_type) || (ObIndexArg::ALTER_PRIMARY_KEY == index_action_type);
  // For add primary key and modify column in one sql, create user hidden table when modifing column.
  const bool create_user_hidden_table_now = !(ObIndexArg::ADD_PRIMARY_KEY == index_action_type && alter_table_arg.is_alter_columns_);
  if ((!is_drop_pk && !is_add_or_alter_pk)
    || (is_drop_pk && 0 != index_columns.count())
    || (is_add_or_alter_pk && 0 == index_columns.count())) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("invalid argument", K(ret), K(index_action_type), K(index_columns));
  } else if (is_add_or_alter_pk && OB_FAIL(add_primary_key(index_columns, new_table_schema))) {
    LOG_WARN("failed to add pk", K(ret), K(index_columns), K(new_table_schema));
  }

  if (OB_FAIL(ret)) {
  } else if (new_table_schema.is_auto_partitioned_table()) {
    bool is_match_rowkey_prefix = true;
    if (is_drop_pk) {
      ret = OB_NOT_SUPPORTED;
      LOG_WARN("not support to drop primary keys of auto-partitioned table", KR(ret), K(index_action_type),
                                                                             K(new_table_schema));
      LOG_USER_ERROR(OB_NOT_SUPPORTED, "drop primary keys of auto-partitioning table is");
    } else if (new_table_schema.get_part_level() == PARTITION_LEVEL_ZERO) {
      if (OB_FAIL(new_table_schema.is_presetting_partition_key_match_rowkey_prefix(is_match_rowkey_prefix))){
        LOG_WARN("fail to check whether presetting partition key matches rowkey prefix", KR(ret), K(new_table_schema));
      }
    } else if (OB_FAIL(new_table_schema.is_partition_key_match_rowkey_prefix(is_match_rowkey_prefix))) {
      LOG_WARN("fail to check whether partition key matches rowkey prefix", KR(ret), K(new_table_schema));
    }

    if (OB_FAIL(ret)) {
    } else if (!is_match_rowkey_prefix) {
      // auto-partitioning require that partition key must be equal to primary key prefix.
      ret = OB_NOT_SUPPORTED;
      LOG_WARN("mismatching between primary key prefix and partition key", KR(ret), K(index_action_type),
                                                                           K(new_table_schema));
      LOG_USER_ERROR(OB_NOT_SUPPORTED, "mismatching between primary key prefix and partition key is");
    } else if (!new_table_schema.is_partitioned_table() &&
               origin_table_schema.get_part_option().get_part_func_expr_str().empty()) {
      // for auto-partitioned non-partitioned table, empty part_func_expr means the presetting partition key
      // of the data_table will always be "current" primary key.
      // thus, when modify primary key, we need to check whether the part_func_type is need to be modified
      const bool is_single_pk_column_new = new_table_schema.get_rowkey_column_num() == 1;
      ObPartitionFuncType part_func_type = PARTITION_FUNC_TYPE_RANGE;
      if (is_single_pk_column_new) {
        ObObjMeta type;
        ObRowkeyColumn row_key_col;
        const common::ObRowkeyInfo &row_key_info = new_table_schema.get_rowkey_info();
        if (OB_FAIL(row_key_info.get_column(0/*since there is only one row key, we only need to check the first one*/, row_key_col))) {
          LOG_WARN("get row key column failed", K(ret), K(row_key_info));
        } else {
          type = row_key_col.get_meta_type();
        }
        if (ObResolverUtils::is_partition_range_column_type(type.get_type())) {
          part_func_type = PARTITION_FUNC_TYPE_RANGE_COLUMNS;
        }
      } else {
        part_func_type = PARTITION_FUNC_TYPE_RANGE_COLUMNS;
      }
      if (OB_SUCC(ret)) {
        new_table_schema.get_part_option().set_part_func_type(part_func_type);
      }
    }
  }

  if (OB_FAIL(ret)) {
  } else if (is_drop_pk && OB_FAIL(drop_primary_key(new_table_schema))) {
    LOG_WARN("failed to add hidden pk column for heap table", K(ret));
  } else if (!create_user_hidden_table_now) {
  } else if (OB_FAIL(adjust_cg_for_offline(new_table_schema))) {
    LOG_WARN("failed to adjust for create hiddent table with pk changed", K(ret));
  } else if (OB_FAIL(get_add_pk_index_name(origin_table_schema,
                                           new_table_schema,
                                           index_action_type,
                                           alter_table_arg.index_arg_list_,
                                           schema_guard,
                                           index_name))) {
    LOG_WARN("fail to rename hidden table's pk constraint", K(ret));
  } else if (OB_FAIL(create_user_hidden_table(origin_table_schema,
                                              new_table_schema,
                                              &alter_table_arg.sequence_ddl_arg_,
                                              bind_tablets,
                                              schema_guard,
                                              schema_guard,
                                              ddl_operator,
                                              trans,
                                              allocator,
                                              tenant_data_version,
                                              index_name))) {
    LOG_WARN("failed to alter table offline", K(ret));
  }
  return ret;
}

int ObDDLService::get_add_pk_index_name(const ObTableSchema &origin_table_schema,
                                        ObTableSchema &new_table_schema,
                                        const ObIndexArg::IndexActionType &index_action_type,
                                        const ObIArray<ObIndexArg *> &index_arg_list,
                                        ObSchemaGetterGuard &schema_guard,
                                        ObString &index_name)
{
  int ret = OB_SUCCESS;
  bool is_oracle_mode = false;
  bool is_exist = false;

  if (OB_FAIL(origin_table_schema.check_if_oracle_compat_mode(is_oracle_mode))) {
    LOG_WARN("fail to check oracle mode", K(ret));
  } else if (is_oracle_mode && index_action_type == ObIndexArg::ADD_PRIMARY_KEY) {
    // find pk name;
    ObIndexArg *tmp_index_arg = nullptr;
    bool found = false;
    for (int64_t i = 0; OB_SUCC(ret) && !found && i < index_arg_list.count(); i++) {
      if (OB_ISNULL(tmp_index_arg = index_arg_list.at(i))) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("fail to get index arg", K(ret));
      } else if (tmp_index_arg->index_action_type_ == ObIndexArg::ADD_PRIMARY_KEY) {
        found = true;
      }
    }
    if (OB_SUCC(ret) && found && tmp_index_arg->index_name_.length() != 0) {
      index_name.assign_ptr(tmp_index_arg->index_name_.ptr(),
                            tmp_index_arg->index_name_.length());
      // check constraint name is not duplicated.
      if (OB_FAIL(check_constraint_name_is_exist(schema_guard,
                                                 origin_table_schema,
                                                 index_name,
                                                 false,
                                                 is_exist))) {
        LOG_WARN("fail to check constraint exist", K(ret));
      } else if (is_exist) {
        ret = OB_ERR_CONSTRAINT_NAME_DUPLICATE;
        LOG_WARN("check constraint name is duplicate", K(ret), K(index_name));
      }
    }
  }
  return ret;
}

int ObDDLService::check_is_change_column_order(
    const ObTableSchema &table_schema,
    const AlterColumnSchema &alter_column_schema,
    bool &is_change_column_order) const
{
  int ret = OB_SUCCESS;
  const ObString &this_name = alter_column_schema.get_origin_column_name();
  const ObString &next_name = alter_column_schema.get_next_column_name();
  const ObString &prev_name = alter_column_schema.get_prev_column_name();
  const bool is_first = alter_column_schema.is_first_;
  const bool is_before = !next_name.empty();
  const bool is_after = !prev_name.empty();
  const int flag_cnt = static_cast<int>(is_first) + static_cast<int>(is_before) + static_cast<int>(is_after);
  const ObSchemaOperationType op_type = alter_column_schema.alter_type_;
  if (OB_UNLIKELY(1 < flag_cnt)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("change column order flag conflict", K(ret), K(is_first), K(is_before), K(is_after));
  } else {
    is_change_column_order = 0 != flag_cnt;
  }
  if (OB_SUCC(ret) && is_change_column_order) {
    bool is_oracle_mode = false;
    if (OB_FAIL(table_schema.check_if_oracle_compat_mode(is_oracle_mode))) {
      LOG_WARN("failed to get oracle mode", K(ret));
    } else if (OB_DDL_CHANGE_COLUMN == op_type || OB_DDL_MODIFY_COLUMN == op_type) {
      // same order iff no non-hidden column exists in (left, right)
      const ObString &left_name = is_before ? this_name : prev_name;
      const ObString &right_name = is_before ? next_name : this_name;
      bool same_order = false;
      ObColumnIterByPrevNextID iter(table_schema);
      const ObColumnSchemaV2 *column = nullptr;
      // iter to left column, for is_first the left column is just iter start
      while (OB_SUCC(ret) && !same_order && !is_first) {
        if (OB_FAIL(iter.next(column))) {
        } else if (OB_ISNULL(column)) {
          ret = OB_ERR_UNEXPECTED;
        } else if ((is_oracle_mode && left_name == column->get_column_name_str())
            || (!is_oracle_mode && 0 == left_name.case_compare(column->get_column_name_str()))) {
          if (left_name == right_name) {
            same_order = true;
          }
          break;
        }
      }
      // iter from left to first non-shadow non-hidden column
      while (OB_SUCC(ret) && !same_order) {
        if (OB_FAIL(iter.next(column))) {
        } else if (OB_ISNULL(column)) {
          ret = OB_ERR_UNEXPECTED;
        } else if (column->is_shadow_column() || column->is_hidden()) {
          // skip
        } else {
          if ((is_oracle_mode && right_name == column->get_column_name_str())
              || (!is_oracle_mode && 0 == right_name.case_compare(column->get_column_name_str()))) {
            same_order = true;
          }
          break;
        }
      }
      if (same_order) {
        is_change_column_order = false;
      }
      if (OB_FAIL(ret)) {
        if (ret == OB_ITER_END) {
          ret = OB_SUCCESS;
        } else {
          LOG_WARN("failed to check is change column order", K(ret));
        }
      }
    } else if (OB_DDL_ADD_COLUMN == op_type) {
      const ObColumnSchemaV2 *column = nullptr;
      if (is_after && nullptr != (column = table_schema.get_column_schema(prev_name))
          && column->get_next_column_id() == BORDER_COLUMN_ID) {
        // add trailing column won't affect the order of old columns
        is_change_column_order = false;
      }
    }
  }
  return ret;
}

int ObDDLService::check_alter_column_is_offline(
    const ObTableSchema &orig_table_schema,
    ObSchemaGetterGuard &schema_guard,
    const ObColumnSchemaV2 &orig_column_schema,
    AlterColumnSchema &alter_column_schema,
    bool &is_offline) const
{
  int ret = OB_SUCCESS;
  bool is_change_column_order = false;
  bool need_rewrite_data = false;
  bool add_pk = false;
  if (OB_FAIL(check_is_change_column_order(orig_table_schema, alter_column_schema, is_change_column_order))) {
    LOG_WARN("failed to check is change column order", K(ret));
  } else if (OB_FAIL(orig_table_schema.check_alter_column_is_offline(
      &orig_column_schema, &alter_column_schema, schema_guard, need_rewrite_data))) {
    LOG_WARN("fail to check column can be altered", K(ret));
  } else {
    //check_alter_column_is_offline will check whether the column type, precision, type is changed,
    if ((orig_table_schema.is_heap_organized_table() && need_rewrite_data && orig_column_schema.is_heap_table_primary_key_column())
     || (orig_table_schema.is_heap_organized_table() && alter_column_schema.is_primary_key_)
     || (orig_table_schema.is_heap_organized_table() && orig_column_schema.is_heap_table_primary_key_column() && alter_column_schema.is_nullable())) {
      ret = OB_NOT_SUPPORTED;
      LOG_WARN("Heap table primary key column not support change", K(ret), K(orig_table_schema), K(need_rewrite_data), K(alter_column_schema.is_nullable()), K(alter_column_schema.is_primary_key_));
      LOG_USER_ERROR(OB_NOT_SUPPORTED, "performing offline DDL operations on primary key columns or changing primary key columns to be NOT NULL in a heap table is");
    }
    add_pk = orig_table_schema.is_table_without_pk() && alter_column_schema.is_primary_key_;
    is_offline = is_change_column_order || need_rewrite_data || add_pk;
  }
  return ret;
}

// check wheter the hidden table of offline ddl contains stored generated column.
int ObDDLService::check_exist_stored_gen_col(
    const ObTableSchema &orig_table_schema,
    const AlterTableSchema &alter_table_schema,
    bool &is_exist)
{
  int ret = OB_SUCCESS;
  is_exist = false;
  int64_t column_cnt = 0;
  AlterColumnSchema *alter_column_schema = nullptr;
  ObTableSchema::const_column_iterator it_begin = orig_table_schema.column_begin();
  ObTableSchema::const_column_iterator it_end = orig_table_schema.column_end();
  for (; OB_SUCC(ret) && it_begin != it_end; it_begin++) {
    ObColumnSchemaV2 *column_schema = (*it_begin);
    if (OB_ISNULL(column_schema)) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("unexpected error", K(ret));
    } else if (column_schema->is_stored_generated_column()) {
      column_cnt++;
    } else {/* do nothing. */}
  }
  it_begin = alter_table_schema.column_begin();
  it_end = alter_table_schema.column_end();
  for (; OB_SUCC(ret) && !is_exist && it_begin != it_end; it_begin++) {
    if (OB_ISNULL(alter_column_schema = static_cast<AlterColumnSchema *>(*it_begin))) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("*it_begin is NULL", K(ret));
    } else if (alter_column_schema->is_stored_generated_column()) {
      if (OB_DDL_DROP_COLUMN == alter_column_schema->alter_type_) {
        column_cnt--;
      } else if (OB_DDL_ADD_COLUMN == alter_column_schema->alter_type_) {
        is_exist = true;
      } else {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("unexpected ddl type found here", K(ret), K(*alter_column_schema));
      }
    }
  }
  if (OB_FAIL(ret)) {
  } else if (!is_exist && column_cnt > 0) {
    is_exist = true;
  } else {/* do nothing. */}
  return ret;
}

int ObDDLService::check_can_add_column_use_instant_(const bool is_oracle_mode,
                                                    const uint64_t tenant_data_version,
                                                    bool &can_add_column_instant)
{
  int ret = OB_SUCCESS;
  can_add_column_instant = false;
  if (is_oracle_mode) {
    // oracle mode can not do add column instant
    ret = OB_NOT_SUPPORTED;
    LOG_WARN("in oracle mode not supported to add column instant", KR(ret), K(tenant_data_version));
  } else {
    if (OB_UNLIKELY(tenant_data_version < MOCK_DATA_VERSION_4_2_5_0)) {
      // mysql mode less than 4.2.5 not support add column instant
      ret = OB_NOT_SUPPORTED;
      LOG_WARN("in mysql mode not supported instant algorithm under this version", KR(ret), K(tenant_data_version));
    } else if (OB_UNLIKELY(tenant_data_version >= DATA_VERSION_4_3_0_0
                            && tenant_data_version < DATA_VERSION_4_3_5_0)) {
      // [4.3.0, 4.3.5) mysql mode not support instant algorithm
      ret = OB_NOT_SUPPORTED;
      LOG_WARN("in mysql mode not supported instant algorithm under this version", KR(ret), K(tenant_data_version));
    } else {
      can_add_column_instant = true;
    }
  }

  return ret;
}

int ObDDLService::check_can_drop_column_instant_(const bool is_oracle_mode,
                                                 const uint64_t tenant_data_version)
{
  int ret = OB_SUCCESS;
  if (!is_oracle_mode) {
    ret = OB_NOT_SUPPORTED;
    LOG_WARN("in mysql mode not supported to drop column instant", KR(ret), K(tenant_data_version));
  } else if ((tenant_data_version >= MOCK_DATA_VERSION_4_2_5_0 && tenant_data_version < DATA_VERSION_4_3_0_0)
             || tenant_data_version >= DATA_VERSION_4_3_5_1) {
    // can drop column instant
  } else {
    ret = OB_NOT_SUPPORTED;
    LOG_WARN("in oracle mode not supported instant algorithm under this version", KR(ret), K(tenant_data_version));
  }
  return ret;
}

int ObDDLService::check_is_add_column_online_(const AlterTableSchema &alter_table_schema,
                                              const ObTableSchema &table_schema,
                                              const AlterColumnSchema &alter_column_schema,
                                              const obrpc::ObAlterTableArg::AlterAlgorithm &algorithm,
                                              const bool is_oracle_mode,
                                              const uint64_t tenant_data_version,
                                              ObDDLType &tmp_ddl_type)
{
  int ret = OB_SUCCESS;
  tmp_ddl_type = ObDDLType::DDL_INVALID;
  bool can_add_column_instant = false;
  bool is_change_column_order = false;
  if (OB_DDL_ADD_COLUMN != alter_column_schema.alter_type_) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("alter_type is not add column", KR(ret), K(alter_column_schema));
  } else if (algorithm == obrpc::ObAlterTableArg::AlterAlgorithm::INSTANT) {
    if (OB_FAIL(check_can_add_column_use_instant_(is_oracle_mode,
                                                  tenant_data_version,
                                                  can_add_column_instant))) {
      LOG_WARN("fail to check can add column use instant algorithm", KR(ret), K(is_oracle_mode), K(table_schema));
    }
  }

  if (OB_SUCC(ret)) {
    if (alter_column_schema.is_autoincrement_ || alter_column_schema.is_primary_key_ || alter_column_schema.has_not_null_constraint()) {
      tmp_ddl_type = ObDDLType::DDL_TABLE_REDEFINITION;
    } else if (nullptr != table_schema.get_column_schema(alter_column_schema.get_column_name())) {
      ObTableSchema::const_column_iterator it_begin = alter_table_schema.column_begin();
      ObTableSchema::const_column_iterator it_end = alter_table_schema.column_end();
      for (; OB_SUCC(ret) && it_begin != it_end; it_begin++) {
        const AlterColumnSchema *column_schema = nullptr;
        if (OB_ISNULL(column_schema = static_cast<AlterColumnSchema *>(*it_begin))) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("*it_begin is NULL", K(ret));
        } else if (ObSchemaOperationType::OB_DDL_DROP_COLUMN == column_schema->alter_type_) {
          lib::Worker::CompatMode compat_mode = (is_oracle_mode ?
          lib::Worker::CompatMode::ORACLE : lib::Worker::CompatMode::MYSQL);
          lib::CompatModeGuard guard(compat_mode);
          const ObString &drop_column_name = column_schema->get_origin_column_name();
          const ObString &add_column_name = alter_column_schema.get_column_name();
          if (ObColumnNameHashWrapper(drop_column_name) == ObColumnNameHashWrapper(add_column_name)) {
            tmp_ddl_type = ObDDLType::DDL_TABLE_REDEFINITION;
          }
        }
      }
    }
  }

  if (OB_SUCC(ret)) {
    if (ObDDLType::DDL_INVALID != tmp_ddl_type) {
    } else if (alter_column_schema.is_stored_generated_column()) {
      tmp_ddl_type = ObDDLType::DDL_ADD_COLUMN_OFFLINE;
    } else if (OB_FAIL(check_is_change_column_order(table_schema, alter_column_schema, is_change_column_order))) {
      LOG_WARN("fail to check is change column order", KR(ret));
    } else if (!can_add_column_instant) {
      // before update to 435, we use DDL_ADD_COLUMN_OFFLINE when change column order
      tmp_ddl_type = is_change_column_order ? ObDDLType::DDL_ADD_COLUMN_OFFLINE : ObDDLType::DDL_ADD_COLUMN_ONLINE;
    } else {
      // After update to 435
      //We still use DDL_ADD_COLUMN_ONLINE to represent add column online when not change column order
      //To distinguish, we use DDL_ADD_COLUMN_INSTANT to represent add column online when change column order
      tmp_ddl_type = is_change_column_order ? ObDDLType::DDL_ADD_COLUMN_INSTANT : ObDDLType::DDL_ADD_COLUMN_ONLINE;
    }
  }
  return ret;
}

int ObDDLService::check_is_modify_partition_key(const ObTableSchema &orig_table_schema,
                                                const AlterTableSchema &alter_table_schema,
                                                bool &is_modify_partition_key)
{
  int ret = OB_SUCCESS;
  is_modify_partition_key = false;
  const common::ObPartitionKeyInfo &partition_keys = orig_table_schema.get_partition_key_info();
  const common::ObPartitionKeyInfo &subpartition_keys = orig_table_schema.get_subpartition_key_info();
  ObTableSchema::const_column_iterator iter = alter_table_schema.column_begin();
  ObTableSchema::const_column_iterator iter_end = alter_table_schema.column_end();
  AlterColumnSchema *alter_column_schema = nullptr;
  for(; OB_SUCC(ret) && !is_modify_partition_key && iter != iter_end; iter++) {
    if (OB_ISNULL(alter_column_schema = static_cast<AlterColumnSchema *>(*iter))) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("iter is NULL", K(ret));
    } else if (OB_FAIL(orig_table_schema.is_column_in_partition_key(alter_column_schema->get_column_id(),
                                                                    is_modify_partition_key))) {
      LOG_WARN("fail to check if column in partition key", K(ret), "column_id", alter_column_schema->get_column_id());
    }
  }
  return ret;
}

int ObDDLService::check_is_change_cst_column_name(const ObTableSchema &table_schema,
                                                  const AlterTableSchema &alter_table_schema,
                                                  bool &change_cst_column_name)
{
  int ret = OB_SUCCESS;
  change_cst_column_name = false;
  ObTableSchema::const_column_iterator iter = alter_table_schema.column_begin();
  ObTableSchema::const_column_iterator iter_end = alter_table_schema.column_end();
  AlterColumnSchema *alter_column_schema = nullptr;
  for(; OB_SUCC(ret) && !change_cst_column_name && iter != iter_end; iter++) {
    const ObColumnSchemaV2 *col_schema = nullptr;
    if (OB_ISNULL(alter_column_schema = static_cast<AlterColumnSchema *>(*iter))) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("iter is NULL", K(ret));
    } else if (OB_ISNULL(col_schema = table_schema.get_column_schema(alter_column_schema->get_column_id()))) {
    } else if (col_schema->get_column_name_str() != alter_column_schema->get_column_name_str()) {
      change_cst_column_name = true;
    }
  }
  return ret;
}

int ObDDLService::check_is_alter_decimal_int_offline(const share::ObDDLType &ddl_type,
                                                     const ObTableSchema &table_schema,
                                                     const AlterTableSchema &alter_table_schema,
                                                     bool &is_alter_decimal_int_offline)
{
  int ret = OB_SUCCESS;
  is_alter_decimal_int_offline = false;
  if (DDL_MODIFY_COLUMN == ddl_type) {
    ObTableSchema::const_column_iterator iter = alter_table_schema.column_begin();
    ObTableSchema::const_column_iterator iter_end = alter_table_schema.column_end();
    AlterColumnSchema *alter_column_schema = nullptr;
    for(; OB_SUCC(ret) && iter != iter_end; iter++) {
      const ObColumnSchemaV2 *col_schema = nullptr;
      if (OB_ISNULL(alter_column_schema = static_cast<AlterColumnSchema *>(*iter))) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("iter is NULL", K(ret));
      } else if (OB_DDL_MODIFY_COLUMN != alter_column_schema->alter_type_ ||
          OB_ISNULL(col_schema = table_schema.get_column_schema(alter_column_schema->get_column_id()))) {
        is_alter_decimal_int_offline = false;
        break;
      } else if (ob_is_decimal_int_tc(col_schema->get_data_type()) &&
          ob_is_number_or_decimal_int_tc(alter_column_schema->get_data_type())) {
        is_alter_decimal_int_offline = true;
      } else {
        is_alter_decimal_int_offline = false;
        break;
      }
    }
  }
  return ret;
}

int ObDDLService::check_can_add_column_instant_(const ObTableSchema &orig_table_schema,
                                                const AlterTableSchema &alter_table_schema,
                                                const obrpc::ObAlterTableArg::AlterAlgorithm &algorithm,
                                                const uint64_t tenant_data_version,
                                                const bool is_oracle_mode,
                                                ObSchemaGetterGuard &schema_guard,
                                                ObDDLType &ddl_type)
{
  int ret = OB_SUCCESS;
  bool add_column_instant = false;
  ddl_type = ObDDLType::DDL_INVALID;
  bool has_other_ddl_type = false;
  AlterColumnSchema *alter_column_schema = NULL;
  ObTableSchema::const_column_iterator it_begin = alter_table_schema.column_begin();
  ObTableSchema::const_column_iterator it_end = alter_table_schema.column_end();
  // 1.add column instant only under mysql
  // 2.add column instant except modify column, with other ddl should be offline
  for (; OB_SUCC(ret) && !is_oracle_mode && !(add_column_instant && has_other_ddl_type) && it_begin != it_end; it_begin++) {
    if (OB_ISNULL(alter_column_schema = static_cast<AlterColumnSchema *>(*it_begin))) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("alter_column_schema is NULL", KR(ret), K(alter_table_schema));
    } else {
      const ObSchemaOperationType op_type = alter_column_schema->alter_type_;
      switch (op_type) {
        case OB_DDL_ADD_COLUMN: {
          ObDDLType tmp_ddl_type = ObDDLType::DDL_INVALID;
          if (!add_column_instant && OB_FAIL(check_is_add_column_online_(alter_table_schema, orig_table_schema, *alter_column_schema, algorithm,
                                                                         is_oracle_mode, tenant_data_version, tmp_ddl_type))) {
            LOG_WARN("fail to check is add column online", KR(ret));
          } else if (ObDDLType::DDL_ADD_COLUMN_INSTANT == tmp_ddl_type) {
            add_column_instant = true;
          }
          break;
        }
        case OB_DDL_MODIFY_COLUMN: {
          // do nothing
          break;
        }
        case OB_DDL_DROP_COLUMN:
        case OB_DDL_CHANGE_COLUMN:
        case OB_DDL_ALTER_COLUMN: {
          has_other_ddl_type = true;
          break;
        }
        default: {
          ret = OB_INVALID_ARGUMENT;
          LOG_WARN("unhandled operator type!", KR(ret), K(op_type));
          break;
        }
      }
    }
  }
  if (OB_SUCC(ret) && add_column_instant && has_other_ddl_type) {
    ddl_type = ObDDLType::DDL_TABLE_REDEFINITION;
    LOG_INFO("add column instant with other ddl except modify column, set ddl type to DDL_TABLE_REDEFINITION", K(ddl_type));
  }
  return ret;
}

int ObDDLService::check_alter_unused_column(const ObSchemaOperationType &operation_type,
                                            const ObColumnSchemaV2 *orig_column_schema)
{
  int ret = OB_SUCCESS;
  if (nullptr == orig_column_schema) {
    // not unused column, do nothing.
  } else if (OB_UNLIKELY(orig_column_schema->is_unused())) {
    const ObString &column_name = orig_column_schema->get_column_name_str();
    switch (operation_type) {
      case OB_DDL_ADD_COLUMN: {
        ret = OB_ERR_COLUMN_DUPLICATE;
        LOG_USER_ERROR(OB_ERR_COLUMN_DUPLICATE, column_name.length(), column_name.ptr());
        LOG_WARN("duplicate column name", KR(ret), K(column_name));
        break;
      }
      default: {
        ret = OB_WRONG_COLUMN_NAME;
        LOG_USER_ERROR(OB_WRONG_COLUMN_NAME, column_name.length(), column_name.ptr());
        LOG_WARN("incorrect column name", KR(ret), K(column_name));
        break;
      }
    }
  }
  return ret;
}

int is_list_part_val_changed(const ObBasePartition &orig_part, const ObBasePartition &new_part)
{
  int ret = OB_SUCCESS;
  if (!orig_part.is_valid() || !new_part.is_valid()) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("invalid argument", K(ret), K(orig_part), K(new_part));
  } else {
    const ObIArray<common::ObNewRow>* orig_list_values = &(orig_part.get_list_row_values());
    const ObIArray<common::ObNewRow>* new_list_values   = &(new_part.get_list_row_values());
    if (OB_ISNULL(orig_list_values) && OB_ISNULL(new_list_values)) {
      /* both empty, skip */
    } else if ((OB_ISNULL(orig_list_values) ^ OB_ISNULL(new_list_values))
             ||(orig_list_values->count() != new_list_values->count())) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("error unexpected, val list changed", K(ret), KPC(orig_list_values), KPC(new_list_values));
    } else {
      for (int64_t i = 0; i < orig_list_values->count() && OB_SUCC(ret); i++) {
        if (orig_list_values->at(i) == new_list_values->at(i)) {
        } else {
          ret = OB_ERR_PARTITION_CONST_DOMAIN_ERROR;
          LOG_WARN("partition list val have been changed", K(ret), K(orig_list_values->at(i)), K(new_list_values->at(i)));
        }
      }
    }
  }
  return ret;
}

int ObDDLService::check_alter_table_column(obrpc::ObAlterTableArg &alter_table_arg,
                                           const ObTableSchema &orig_table_schema,
                                           ObSchemaGetterGuard &schema_guard,
                                           const bool is_oracle_mode,
                                           const uint64_t tenant_data_version,
                                           ObDDLType &ddl_type,
                                           bool &ddl_need_retry_at_executor)
{
  int ret = OB_SUCCESS;
  bool is_modify_partition_key = false;
  common::ObIAllocator &allocator = alter_table_arg.allocator_;
  const AlterTableSchema &alter_table_schema = alter_table_arg.alter_table_schema_;
  char err_msg[number::ObNumber::MAX_PRINTABLE_SIZE] = {0};
  AlterColumnSchema *alter_column_schema = NULL;
  int64_t drop_lob_cols_cnt = 0;
  ObTableSchema::const_column_iterator it_begin = alter_table_schema.column_begin();
  ObTableSchema::const_column_iterator it_end = alter_table_schema.column_end();
  ddl_need_retry_at_executor = false;
  const obrpc::ObAlterTableArg::AlterAlgorithm &algorithm = alter_table_arg.alter_algorithm_;

  if (OB_FAIL(check_can_add_column_instant_(orig_table_schema, alter_table_schema, algorithm,
                                            is_oracle_mode, tenant_data_version, schema_guard, ddl_type))) {
    LOG_WARN("fail to check can add column instant", KR(ret), K(orig_table_schema), K(alter_table_schema));
  }
  for (; OB_SUCC(ret) && it_begin != it_end; it_begin++) {
    if (OB_ISNULL(alter_column_schema = static_cast<AlterColumnSchema *>(*it_begin))) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("*it_begin is NULL", K(ret));
    } else if (OB_FAIL(check_alter_unused_column(alter_column_schema->alter_type_,
        orig_table_schema.get_column_schema(alter_column_schema->get_column_name())))) {
      LOG_WARN("check alter unused colunm failed", KR(ret));
    } else {
      const ObString &orig_column_name = alter_column_schema->get_origin_column_name();
      const ObColumnSchemaV2 *orig_column_schema = orig_table_schema.get_column_schema(orig_column_name);
      const ObSchemaOperationType op_type = alter_column_schema->alter_type_;
      switch (op_type) {
        case OB_DDL_ADD_COLUMN: {
          ObDDLType tmp_ddl_type = ObDDLType::DDL_INVALID;
          if (OB_FAIL(check_is_add_column_online_(alter_table_schema, orig_table_schema, *alter_column_schema, algorithm,
                                                  is_oracle_mode, tenant_data_version, tmp_ddl_type))) {
            LOG_WARN("fail to check is add column online", K(ret));
          } else if (tmp_ddl_type == ObDDLType::DDL_ADD_COLUMN_ONLINE) {
            if (ObDDLType::DDL_INVALID == ddl_type
                || ObDDLType::DDL_ADD_COLUMN_ONLINE == ddl_type) {
              ddl_type = ObDDLType::DDL_ADD_COLUMN_ONLINE;
            } else if (ObDDLType::DDL_ADD_COLUMN_INSTANT == ddl_type) {
              // ddl_type = ObDDLType::DDL_ADD_COLUMN_INSTANT;
            } else if (ObDDLType::DDL_COMPOUND_INSTANT == ddl_type) {
              // ddl_type = ObDDLType::DDL_COMPOUND_INSTANT;
            } else if (ObDDLType::DDL_NORMAL_TYPE == ddl_type) {
              // ddl_type = ObDDLType::DDL_NORMAL_TYPE;
            } else if (ObDDLType::DDL_ADD_COLUMN_OFFLINE == ddl_type) {
              // ddl_type = ObDDLType::DDL_ADD_COLUMN_OFFLINE;
            } else if (ObDDLType::DDL_DROP_COLUMN == ddl_type
                    || ObDDLType::DDL_COLUMN_REDEFINITION == ddl_type) {
              ddl_type = ObDDLType::DDL_COLUMN_REDEFINITION;
            } else {
              ddl_type = ObDDLType::DDL_TABLE_REDEFINITION;
            }
          } else if (tmp_ddl_type == ObDDLType::DDL_ADD_COLUMN_INSTANT) {
            if (ObDDLType::DDL_INVALID == ddl_type
                || ObDDLType::DDL_ADD_COLUMN_ONLINE == ddl_type
                || ObDDLType::DDL_ADD_COLUMN_INSTANT == ddl_type) {
              ddl_type = ObDDLType::DDL_ADD_COLUMN_INSTANT;
            // 1.there is no new type to represent modify column online, we use DDL_NORMAL_TYPE to represent it
            // 2. after check_can_add_column_instant_, add column instant except modify column compund with other ddl will be DDL_TABLE_REDEFINITION,
            // so only modify column online can be DDL_NORMAL_TYPE here
            } else if (ObDDLType::DDL_NORMAL_TYPE == ddl_type
                    || ObDDLType::DDL_COMPOUND_INSTANT == ddl_type) {
              ddl_type = ObDDLType::DDL_COMPOUND_INSTANT;
            } else if (ObDDLType::DDL_ADD_COLUMN_OFFLINE == ddl_type) {
              // ddl_type = ObDDLType::DDL_ADD_COLUMN_OFFLINE;
            } else if (ObDDLType::DDL_DROP_COLUMN == ddl_type
                    || ObDDLType::DDL_COLUMN_REDEFINITION == ddl_type) {
              ddl_type = ObDDLType::DDL_COLUMN_REDEFINITION;
            } else {
              ddl_type = ObDDLType::DDL_TABLE_REDEFINITION;
            }
          } else if (tmp_ddl_type == ObDDLType::DDL_TABLE_REDEFINITION) {
            ddl_type = ObDDLType::DDL_TABLE_REDEFINITION;
          } else {
            if (ObDDLType::DDL_INVALID == ddl_type
                || ObDDLType::DDL_ADD_COLUMN_ONLINE == ddl_type
                || ObDDLType::DDL_ADD_COLUMN_OFFLINE == ddl_type) {
              ddl_type = ObDDLType::DDL_ADD_COLUMN_OFFLINE;
            } else if (ObDDLType::DDL_DROP_COLUMN == ddl_type
                    || ObDDLType::DDL_COLUMN_REDEFINITION == ddl_type) {
              ddl_type = ObDDLType::DDL_COLUMN_REDEFINITION;
            } else {
              ddl_type = ObDDLType::DDL_TABLE_REDEFINITION;
            }
          }
          break;
        }
        case OB_DDL_DROP_COLUMN: {
          if (OB_ISNULL(orig_column_schema)) {
            ret = OB_ERR_CANT_DROP_FIELD_OR_KEY;
            LOG_USER_ERROR(OB_ERR_CANT_DROP_FIELD_OR_KEY, orig_column_name.length(), orig_column_name.ptr());
            LOG_WARN("fail to find old column schema", KR(ret), K(orig_column_name), KPC(orig_column_schema));
          } else if (orig_column_schema->is_heap_table_primary_key_column() && orig_table_schema.is_heap_organized_table()) {
            ret = OB_NOT_SUPPORTED;
            LOG_WARN("Heap table primary key column not support drop", K(ret));
            LOG_USER_ERROR(OB_NOT_SUPPORTED, "dropping primary key columns in a heap table is");
          } else if (is_lob_storage(orig_column_schema->get_data_type())
                     && !orig_column_schema->is_udt_hidden_column()) {
            drop_lob_cols_cnt++;
          } else if (orig_column_schema->is_xmltype()) {
            // The implicitly added lob column will be dropped when drop the xml column.
            drop_lob_cols_cnt++;
          }
          if (OB_SUCC(ret)) {
            if (ObDDLType::DDL_INVALID == ddl_type
                || ObDDLType::DDL_DROP_COLUMN == ddl_type) {
              ddl_type = ObDDLType::DDL_DROP_COLUMN;
            } else if (ObDDLType::DDL_ADD_COLUMN_ONLINE == ddl_type
                    || ObDDLType::DDL_ADD_COLUMN_OFFLINE == ddl_type
                    || ObDDLType::DDL_COLUMN_REDEFINITION == ddl_type) {
              ddl_type = ObDDLType::DDL_COLUMN_REDEFINITION;
            } else {
              ddl_type = ObDDLType::DDL_TABLE_REDEFINITION;
            }
          }
          break;
        }
        case OB_DDL_CHANGE_COLUMN:
        case OB_DDL_MODIFY_COLUMN: {
          bool is_offline = false;
          bool add_pk = false;

          if (OB_ISNULL(orig_column_schema)) {
            ret = OB_ERR_BAD_FIELD_ERROR;
            LOG_USER_ERROR(OB_ERR_BAD_FIELD_ERROR, orig_column_name.length(), orig_column_name.ptr(),
                orig_table_schema.get_table_name_str().length(), orig_table_schema.get_table_name_str().ptr());
            LOG_WARN("unknown column", KR(ret), K(orig_column_name), K(orig_table_schema));
          }

          // check is partition list val changed
          if (OB_SUCC(ret)) {
            bool is_partition_key = orig_column_schema->is_part_key_column();
            bool is_sub_partition_key = orig_column_schema->is_subpart_key_column();
            if (!is_partition_key && !is_sub_partition_key) {
            } else if (!ob_is_integer_type(orig_column_schema->get_data_type())
                      || !ob_is_integer_type(alter_column_schema->get_data_type())) {
            } else if (orig_column_schema->get_data_type() != alter_column_schema->get_data_type()) {
              if (orig_table_schema.get_partition_num() != alter_table_schema.get_partition_num()) {
                ret = OB_ERR_UNEXPECTED;
                LOG_WARN("partition num have been changed when change column type", K(ret), K(orig_table_schema), K(alter_table_arg));
              }
              for (int64_t i = 0; i < orig_table_schema.get_partition_num() && OB_SUCC(ret); i++) {
                const ObPartition *orig_part = orig_table_schema.get_part_array()[i];
                const ObPartition *new_part  = alter_table_schema.get_part_array()[i];
                if (OB_ISNULL(orig_part) || OB_ISNULL(new_part)) {
                  ret = OB_ERR_UNEXPECTED;
                  LOG_WARN("invalid part", K(ret), KPC(orig_part), K(new_part));
                } else if (is_partition_key && OB_FAIL(is_list_part_val_changed(*orig_part, *new_part))) {
                  LOG_WARN("failed to check new part valid", K(ret), K(orig_table_schema), K(alter_table_arg));
                } else if (is_sub_partition_key) {
                  if (orig_part->get_sub_part_num() != new_part->get_sub_part_num()) {
                    ret = OB_ERR_UNEXPECTED;
                    LOG_WARN("invalid part num", K(ret), KPC(orig_part), KPC(new_part), K(orig_table_schema), K(alter_table_arg));
                  }
                  for (int64_t j = 0; OB_SUCC(ret) && j < orig_part->get_sub_part_num(); j++) {
                    ObSubPartition *orig_sub_part = orig_part->get_subpart_array()[j];
                    ObSubPartition *new_sub_part  = new_part->get_subpart_array()[j];
                    if (OB_ISNULL(orig_sub_part) || OB_ISNULL(new_sub_part)) {
                      ret = OB_ERR_UNEXPECTED;
                      LOG_WARN("invalid part", K(ret), KPC(orig_part), KPC(new_part));
                    } else if (OB_FAIL(is_list_part_val_changed(*orig_sub_part, *new_sub_part))) {
                      LOG_WARN("failed to check sub part changed", K(ret), KPC(orig_sub_part), K(new_sub_part), K(orig_table_schema), K(alter_table_arg));
                    }
                  }
                }
              }
            }
          }

          if (OB_FAIL(ret)) {
          } else if (OB_FAIL(fill_column_collation(alter_table_schema.get_sql_mode(),
                                            is_oracle_mode,
                                            orig_table_schema,
                                            allocator,
                                            *alter_column_schema))) {
            LOG_WARN("failed to fill column collation", K(ret));
         } else if (OB_FAIL(check_alter_column_is_offline(
              orig_table_schema, schema_guard, *orig_column_schema, *alter_column_schema, is_offline))) {
            LOG_WARN("failed to check is offline", K(ret));
          } else if (is_offline) {
            if (alter_column_schema->is_primary_key_) {
              if (orig_table_schema.get_rowkey_column_num() > 0) {
                if (orig_table_schema.is_table_with_pk()) {
                  ret = OB_ERR_MULTIPLE_PRI_KEY;
                  RS_LOG(WARN, "multiple primary key defined", K(ret));
                } else {
                  add_pk = true;
                }
              } else {
                ret = OB_ERR_UNEXPECTED;
                LOG_WARN("rowkey_column_num must be greater than 0", K(ret),
                K(orig_table_schema.get_rowkey_column_num()));
              }
            }
          }
          if (OB_FAIL(ret)) {
          } else if (add_pk) {
            ddl_type = ObDDLType::DDL_TABLE_REDEFINITION;
          } else if (is_offline) {
            if (ObDDLType::DDL_INVALID == ddl_type
                || ObDDLType::DDL_MODIFY_COLUMN == ddl_type) {
              ddl_type = ObDDLType::DDL_MODIFY_COLUMN;
            } else {
              ddl_type = ObDDLType::DDL_TABLE_REDEFINITION;
            }
          } else {
            if (alter_column_schema->is_autoincrement() && !orig_column_schema->is_autoincrement()) {
              if (orig_column_schema->is_generated_column()) {
                ret = OB_NOT_SUPPORTED;
                LOG_USER_ERROR(OB_NOT_SUPPORTED, "Changing the STORED status for generated columns");
              } else if (orig_table_schema.get_autoinc_column_id() == 0) {
                if (orig_table_schema.is_heap_organized_table() && orig_column_schema->is_heap_table_primary_key_column()) {
                  ret = OB_NOT_SUPPORTED;
                  LOG_WARN("Heap table primary key column not support adding an autoincrement attribute", K(ret));
                  LOG_USER_ERROR(OB_NOT_SUPPORTED, "adding an autoincrement attribute to primary key column in a heap table is");
                } else if (orig_column_schema->is_nullable()) {
                  // if the original table has null, we need to do double write to fill the nulls
                  if (ObDDLType::DDL_INVALID == ddl_type || ObDDLType::DDL_MODIFY_COLUMN == ddl_type) {
                    ddl_type = ObDDLType::DDL_MODIFY_COLUMN;
                  } else {
                    ddl_type = ObDDLType::DDL_TABLE_REDEFINITION;
                  }
                  ddl_need_retry_at_executor = true;
                } else {
                  if (ObDDLType::DDL_INVALID == ddl_type) {
                    ddl_type = ObDDLType::DDL_MODIFY_AUTO_INCREMENT;
                  } else if (ObDDLType::DDL_MODIFY_COLUMN == ddl_type) {
                    // ddl_type = ObDDLType::DDL_MODIFY_COLUMN;
                  } else {
                    ddl_type = ObDDLType::DDL_TABLE_REDEFINITION;
                  }
                }

              } else {
                ret = OB_ERR_WRONG_AUTO_KEY;
                LOG_USER_ERROR(OB_ERR_WRONG_AUTO_KEY);
              }
            } else if (OB_DDL_CHANGE_COLUMN == op_type) {
              if (ObDDLType::DDL_INVALID == ddl_type
                || ObDDLType::DDL_ADD_COLUMN_ONLINE == ddl_type
                || ObDDLType::DDL_NORMAL_TYPE == ddl_type) {
                ddl_type = ObDDLType::DDL_NORMAL_TYPE;
              } else {
                ddl_type = ObDDLType::DDL_TABLE_REDEFINITION;
              }
            } else if (OB_DDL_MODIFY_COLUMN == op_type) {
              if (ObDDLType::DDL_INVALID == ddl_type
                || ObDDLType::DDL_ADD_COLUMN_ONLINE == ddl_type
                || ObDDLType::DDL_NORMAL_TYPE == ddl_type) {
                ddl_type = ObDDLType::DDL_NORMAL_TYPE;
              } else if (ObDDLType::DDL_ADD_COLUMN_INSTANT == ddl_type
                      || ObDDLType::DDL_COMPOUND_INSTANT == ddl_type) {
                ddl_type = ObDDLType::DDL_COMPOUND_INSTANT;
              } else {
                ddl_type = ObDDLType::DDL_TABLE_REDEFINITION;
              }
            }
          }
          break;
        }
        case OB_DDL_ALTER_COLUMN: {
          if (ObDDLType::DDL_INVALID == ddl_type
              || ObDDLType::DDL_ADD_COLUMN_ONLINE == ddl_type
              || ObDDLType::DDL_NORMAL_TYPE == ddl_type) {
            ddl_type = ObDDLType::DDL_NORMAL_TYPE;
          } else {
            ddl_type = ObDDLType::DDL_TABLE_REDEFINITION;
          }
          break;
        }
        default: {
          ret = OB_INVALID_ARGUMENT;
          LOG_WARN("unhandled operator type!", K_(alter_column_schema->alter_type));
          break;
        }
      }
      if (OB_SUCC(ret)
          && ObDDLType::DDL_TABLE_REDEFINITION == ddl_type
          && alter_column_schema->is_identity_column()
          && OB_DDL_DROP_COLUMN != op_type) {
        if (nullptr == orig_column_schema) {
          ddl_need_retry_at_executor = true;
        } else if (orig_column_schema->is_nullable()) {
          // only nullable column need generate value from sequence
          ddl_need_retry_at_executor = true;
        }
      }
    }
  }
  if (OB_FAIL(ret)) {
  } else if (ObDDLType::DDL_DROP_COLUMN == ddl_type
          || ObDDLType::DDL_ADD_COLUMN_OFFLINE == ddl_type
          || ObDDLType::DDL_COLUMN_REDEFINITION == ddl_type) {
    bool is_exist_stored_gen_col = false; // whether the target table contain stored generated column.
    bool is_column_group_store = false;
    bool need_process_cs_replica = false;
    // oracle drop column instant will be drop column instant
    if (is_oracle_mode && ObDDLType::DDL_DROP_COLUMN == ddl_type) {
      // ignore
    } else if (OB_FAIL(ObCODDLUtil::need_column_group_store(orig_table_schema, is_column_group_store))) {
      LOG_WARN("fail to check schema is column group store", K(ret));
    } else if (is_column_group_store) {
      ddl_type = ObDDLType::DDL_TABLE_REDEFINITION;
    } else if (OB_FAIL(check_exist_stored_gen_col(orig_table_schema,
                                           alter_table_schema,
                                           is_exist_stored_gen_col))) {
      LOG_WARN("fail to check exist stored generated column", K(ret));
    } else if (is_exist_stored_gen_col) {
      // column redefinition cannot handle stored gen column, use table redefinition instead
      ddl_type = ObDDLType::DDL_TABLE_REDEFINITION;
    } else if (OB_FAIL(ObCSReplicaUtil::check_need_process_cs_replica_for_offline_ddl(orig_table_schema, need_process_cs_replica))) {
      LOG_WARN("fail to check need process cs replica", K(ret));
    } else if (need_process_cs_replica) {
      ddl_type = ObDDLType::DDL_TABLE_REDEFINITION;
    } else {/* do nothing. */}
  }
  if (OB_FAIL(ret)) {
  } else if (is_double_table_long_running_ddl(ddl_type)
             && OB_FAIL(check_is_modify_partition_key(orig_table_schema,
                                                      alter_table_schema,
                                                      is_modify_partition_key))) {
    LOG_WARN("fail to check is modify partition key", K(ret));
  } else if (is_modify_partition_key) {
    ddl_type = ObDDLType::DDL_ALTER_PARTITION_BY;
  }
  if (OB_SUCC(ret) && is_oracle_mode && ObDDLType::DDL_DROP_COLUMN == ddl_type && drop_lob_cols_cnt > 0) {
    // DDL_DROP_COLUMN will be replaced with DDL_DROP_COLUMN_INSTANT by the caller under oracle mode.
    // And to drop all lob columns needs to change ddl_type to DDL_COLUMN_REDEFINITION,
    // so as to execute offline drop column.
    int64_t lob_cols_cnt_in_table = 0;
    const ObColumnSchemaV2 *col = nullptr;
    ObColumnIterByPrevNextID iter(orig_table_schema);
    while (OB_SUCC(ret) && OB_SUCC(iter.next(col))) {
      if (OB_ISNULL(col)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("unexpected null column", K(ret), K(lob_cols_cnt_in_table), K(orig_table_schema));
      } else if (is_lob_storage(col->get_data_type())) {
        lob_cols_cnt_in_table++;
      }
    }
    if (OB_ITER_END != ret) {
      LOG_WARN("iter table column failed", KR(ret), K(orig_table_schema));
    } else {
      ret = OB_SUCCESS;
      alter_table_arg.alter_algorithm_ = drop_lob_cols_cnt == lob_cols_cnt_in_table ?
        obrpc::ObAlterTableArg::AlterAlgorithm::INPLACE : alter_table_arg.alter_algorithm_; // to execute offline.
    }
  }
  return ret;
}

// Check whether alter primary key and alter column in one sql is allowed.
// Currently, support MODIFY column and ADD PRIMARY KEY in one sql.
int ObDDLService::check_support_alter_pk_and_columns(
      const obrpc::ObAlterTableArg &alter_table_arg,
      const obrpc::ObIndexArg::IndexActionType &index_action_type,
      bool &is_support)
{
  int ret = OB_SUCCESS;
  is_support = true;
  if (ObIndexArg::ADD_PRIMARY_KEY != index_action_type) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("invalid argument", K(ret), K(index_action_type), K(alter_table_arg));
  } else if (!alter_table_arg.is_alter_columns_) {
    // without column operation in the sql.
  } else {
    const AlterTableSchema &alter_table_schema = alter_table_arg.alter_table_schema_;
    for (ObTableSchema::const_column_iterator it_begin = alter_table_schema.column_begin();
      OB_SUCC(ret) && is_support && it_begin != alter_table_schema.column_end(); it_begin++) {
      AlterColumnSchema *alter_column_schema = nullptr;
      if (OB_ISNULL(alter_column_schema = static_cast<AlterColumnSchema *>(*it_begin))) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("*it_begin is NULL", K(ret));
      } else if (OB_DDL_MODIFY_COLUMN != alter_column_schema->alter_type_) {
        is_support = false;
      } else if (alter_column_schema->is_primary_key_) {
        ret = OB_ERR_MULTIPLE_PRI_KEY;
        LOG_WARN("multiple primary key defined", K(ret), KPC(alter_column_schema));
      } else {
        // do nothing.
      }
    }
  }
  return ret;
}

int ObDDLService::check_alter_heap_table_index(const obrpc::ObIndexArg::IndexActionType type,
                                               const ObTableSchema &orig_table_schema,
                                               obrpc::ObIndexArg *index_arg)
{
  int ret = OB_SUCCESS;
  ObString index_table_name;
  char err_msg[number::ObNumber::MAX_PRINTABLE_SIZE] = {0};
  uint64_t tenant_id = orig_table_schema.get_tenant_id();
  ObArenaAllocator allocator("HeapTblCheck", OB_MALLOC_NORMAL_BLOCK_SIZE);
  bool is_oracle_mode = false;
  if (OB_FAIL(ObCompatModeGetter::check_is_oracle_mode_with_tenant_id(tenant_id, is_oracle_mode))) {
    LOG_WARN("fail to check is oracle mode", K(ret));
  } else if (OB_ISNULL(index_arg)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("index_arg is NULL", K(ret));
  } else if (OB_FAIL(ObTableSchema::build_index_table_name(allocator,
                                                      orig_table_schema.get_table_id(),
                                                      index_arg->index_name_,
                                                      index_table_name))) {
    LOG_WARN("failed to build index table name", K(index_arg->index_name_), K(index_table_name), K(ret));
  }
  const ObIArray<ObAuxTableMetaInfo> &simple_index_infos = orig_table_schema.get_simple_index_infos();
  ObSchemaGetterGuard schema_guard;
  const ObTableSchema *index_schema = NULL;
  if (OB_FAIL(ret)) {
  } else if (OB_FAIL(schema_service_->get_tenant_schema_guard(tenant_id, schema_guard))) {
    LOG_WARN("fail to get schema guard", K(ret));
  }
  ObIndexSchemaHashWrapper dest_index_schema_name_wrapper(tenant_id,
                                                         orig_table_schema.get_database_id(),
                                                         is_oracle_mode ? common::OB_INVALID_ID : orig_table_schema.get_table_id(),
                                                         index_table_name);
  for (int64_t i = 0; OB_SUCC(ret) && i < simple_index_infos.count(); i++) {
    if (OB_FAIL(schema_guard.get_table_schema(tenant_id, simple_index_infos.at(i).table_id_, index_schema))) {
      LOG_WARN("fail to get index schema", K(ret));
    } else if (OB_ISNULL(index_schema)) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("the index to be altered is null", K(ret));
    } else if (ObIndexType::INDEX_TYPE_HEAP_ORGANIZED_TABLE_PRIMARY != index_schema->get_index_type()) {
      continue;
    }
    ObIndexSchemaHashWrapper orgin_index_schema_name_wrapper(index_schema->get_tenant_id(),
                                                             orig_table_schema.get_database_id(),
                                                             is_oracle_mode ? common::OB_INVALID_ID : orig_table_schema.get_table_id(),
                                                             index_schema->get_table_name_str());
    //RENAME_INDEX ddl needs to check whether it reuses the primary key name in the heap table
    if (OB_FAIL(ret)) {
    } else if (ObIndexArg::RENAME_INDEX == type) {
      ObRenameIndexArg *rename_index_arg = static_cast<ObRenameIndexArg *>(index_arg);
      const ObString &ori_index_name = rename_index_arg->origin_index_name_;
      const ObString &new_index_name = rename_index_arg->new_index_name_;
      ObString ori_index_table_name;
      ObString new_index_table_name;
      if (OB_FAIL(ObTableSchema::build_index_table_name(allocator,
                                                        orig_table_schema.get_table_id(),
                                                        ori_index_name,
                                                        ori_index_table_name))) {
        LOG_WARN("failed to build index table name", K(index_arg->index_name_), K(ori_index_table_name), K(ret));
      } else if (OB_FAIL(ObTableSchema::build_index_table_name(allocator,
                                                        orig_table_schema.get_table_id(),
                                                        new_index_name,
                                                        new_index_table_name))) {
        LOG_WARN("failed to build index table name", K(index_arg->index_name_), K(ori_index_table_name), K(ret));
      }
      ObIndexSchemaHashWrapper rename_from_name_wrapper(tenant_id,
                                                        orig_table_schema.get_database_id(),
                                                        is_oracle_mode ? common::OB_INVALID_ID : index_schema->get_data_table_id(),
                                                        ori_index_table_name);
      ObIndexSchemaHashWrapper rename_to_name_wrapper(tenant_id,
                                                      orig_table_schema.get_database_id(),
                                                      is_oracle_mode ? common::OB_INVALID_ID : index_schema->get_data_table_id(),
                                                      new_index_table_name);
      if (OB_FAIL(ret)) {
      } else if (rename_from_name_wrapper == orgin_index_schema_name_wrapper) {
        ret = OB_NOT_SUPPORTED;
        (void)snprintf(err_msg, sizeof(err_msg), "can't RENAME '%s'; RENAME from a column/key that is reserved by heap table is is",
                      ori_index_name.ptr());
        LOG_USER_ERROR(OB_NOT_SUPPORTED, err_msg);
      } else if (rename_to_name_wrapper == orgin_index_schema_name_wrapper) {
        ret = OB_NOT_SUPPORTED;
        (void)snprintf(err_msg, sizeof(err_msg), "switch to another name; RENAME a column/key to a name that is reserved by heap table is");
        LOG_USER_ERROR(OB_NOT_SUPPORTED, err_msg);
      }
    } else if (dest_index_schema_name_wrapper == orgin_index_schema_name_wrapper) {
      ret = OB_NOT_SUPPORTED;
      (void)snprintf(err_msg, sizeof(err_msg), "can't %s '%s'; %s with a name that is reserved by heap table is",
                      ObIndexArg::to_type_str(type),
                      index_arg->index_name_.ptr(),
                      ObIndexArg::to_type_str(type));
      LOG_USER_ERROR(OB_NOT_SUPPORTED, err_msg);
    }
  }
  return ret;
}
int ObDDLService::check_alter_table_index(const obrpc::ObAlterTableArg &alter_table_arg,
                                          const ObTableSchema &orig_table_schema,
                                          ObDDLType &ddl_type,
                                          share::schema::ObSchemaGetterGuard &schema_guard,
                                          bool &has_drop_and_add_index)
{
  int ret = OB_SUCCESS;
  char err_msg[number::ObNumber::MAX_PRINTABLE_SIZE] = {0};
  ObIndexArg::IndexActionType last_type = ObIndexArg::INVALID_ACTION;
  const ObSArray<ObIndexArg *> &index_arg_list = alter_table_arg.index_arg_list_;
  bool has_drop_index = false;
  bool has_create_index = false;
  bool is_add_many_fts_indexes = false;
  bool is_drop_fts_or_multivalue_or_vector_index = false;
  common::ObArray<ObString> dropping_indexes;
  const share::schema::ObTableSchema *table_schema = nullptr;
  const int64_t data_table_id = alter_table_arg.alter_table_schema_.get_table_id();
  const uint64_t tenant_id = alter_table_arg.alter_table_schema_.get_tenant_id();
  has_drop_and_add_index = false;
  if (OB_FAIL(schema_guard.get_table_schema(tenant_id, data_table_id, table_schema))) {
    LOG_WARN("fail to get table schema", K(ret), K(tenant_id), K(data_table_id));
  } else if (OB_ISNULL(table_schema)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("the table schema is null", K(ret));
  }
  for (int64_t i = 0; OB_SUCC(ret) && i < index_arg_list.size(); ++i) {
    ObIndexArg *index_arg = const_cast<ObIndexArg *>(index_arg_list.at(i));
    if (OB_ISNULL(index_arg)) {
      ret = OB_INVALID_ARGUMENT;
      LOG_WARN("index arg should not be null", K(ret));
    } else {
      const ObIndexArg::IndexActionType type = index_arg->index_action_type_;
      ObCreateIndexArg *create_index_arg = nullptr;
      bool is_add_fts_or_multivalue_index = false;
      bool is_drop_fts_or_multivalue_index = false;
      if (ObIndexArg::ADD_INDEX == type) {
        ObCreateIndexArg *create_index_arg = static_cast<ObCreateIndexArg *>(index_arg);
        if (OB_ISNULL(create_index_arg)) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("the create index arg is null", K(ret));
        } else {
          is_add_fts_or_multivalue_index = share::schema::is_fts_or_multivalue_index(create_index_arg->index_type_);
        }
      } else if (ObIndexArg::DROP_INDEX == type) {
        ObDropIndexArg *drop_index_arg = static_cast<ObDropIndexArg *>(index_arg);
        ObTableSchema *index_schema = nullptr;
        if (OB_ISNULL(drop_index_arg)) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("the drop index arg is null", K(ret));
        } else {
          const share::schema::ObTableSchema *index_schema = nullptr;
          const int64_t index_table_id = drop_index_arg->index_table_id_;
          if (OB_FAIL(get_index_schema_by_name(data_table_id,
                                               table_schema->get_database_id(),
                                               *drop_index_arg,
                                               schema_guard,
                                               index_schema))) {
            LOG_WARN("fail to get index schema by name",
                K(ret), K(data_table_id), K(table_schema->get_database_id()), KPC(drop_index_arg));
          } else if (OB_ISNULL(index_schema)) {
            ret = OB_TABLE_NOT_EXIST;
            LOG_WARN("get table schema failed", K(ret), K(tenant_id), K(index_table_id));
          } else {
            is_drop_fts_or_multivalue_or_vector_index = index_schema->is_fts_or_multivalue_index()
                                                        || index_schema->is_vec_index();
            if (is_drop_fts_or_multivalue_or_vector_index &&
                OB_FAIL(dropping_indexes.push_back(drop_index_arg->index_name_))) {
              LOG_WARN("fail to push back", K(ret), K(drop_index_arg->index_name_));
            }
          }
        }
      } else if (ObIndexArg::RENAME_INDEX == type) {
        ObRenameIndexArg *rename_index_arg = static_cast<ObRenameIndexArg *>(index_arg);
        if (OB_ISNULL(rename_index_arg)) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("the rename index arg is null", K(ret));
        } else {
          ObString &new_index_name = rename_index_arg->new_index_name_;
          for (int64_t i = 0; OB_SUCC(ret) && i < dropping_indexes.count(); ++i) {
            ObString &dropping_index_name = dropping_indexes.at(i);
            if (ObIndexNameHashWrapper(dropping_index_name) == ObIndexNameHashWrapper(new_index_name)) {
              ret = OB_NOT_SUPPORTED;
              LOG_WARN("dropping an fts or multivalue index and renaming another index " \
                  "to the droped index name in a single ALTER TABLE statement is not surppted", K(ret));
              LOG_USER_ERROR(OB_NOT_SUPPORTED, "dropping an fts or multivalue index and renaming another index " \
                  "to the droped index name in a single ALTER TABLE statement");
            }
          }
        }
      }
      if (OB_SUCC(ret)) {
      switch(type) {
        case ObIndexArg::DROP_PRIMARY_KEY: {
          if (!is_invalid_ddl_type(ddl_type)) {
            ret = OB_NOT_SUPPORTED;
            (void)snprintf(err_msg, sizeof(err_msg), "%s and %s in single statment",
                          ObIndexArg::to_type_str(last_type), ObIndexArg::to_type_str(type));
            LOG_USER_ERROR(OB_NOT_SUPPORTED, err_msg);
          } else {
            ddl_type = DDL_DROP_PRIMARY_KEY;
            last_type = type;
          }
          break;
        }
        case ObIndexArg::ADD_PRIMARY_KEY: {
          bool is_support = true;
          if (orig_table_schema.is_heap_organized_table()) {
            ret = OB_NOT_SUPPORTED;
            (void)snprintf(err_msg, sizeof(err_msg), "%s in heap organized table",
                          ObIndexArg::to_type_str(type));
            LOG_USER_ERROR(OB_NOT_SUPPORTED, err_msg);
          } else if (ObDDLType::DDL_INVALID == ddl_type) {
            ddl_type = DDL_ADD_PRIMARY_KEY;
            last_type = type;
          } else if (OB_FAIL(check_support_alter_pk_and_columns(alter_table_arg, type, is_support))) {
            LOG_WARN("check support column operation and add primary key in one sql failed", K(ret), K(alter_table_arg));
          } else if (is_support) {
            ddl_type = DDL_TABLE_REDEFINITION;
            last_type = type;
          } else {
            ret = OB_NOT_SUPPORTED;
            (void)snprintf(err_msg, sizeof(err_msg), "%s and %s in single statment",
                            ObIndexArg::to_type_str(last_type), ObIndexArg::to_type_str(type));
            LOG_USER_ERROR(OB_NOT_SUPPORTED, err_msg);
            LOG_WARN("Not Supported DDL", K(ret), K(ddl_type), K(last_type), K(type));
          }
          break;
        }
        case ObIndexArg::ALTER_PRIMARY_KEY: {
          if (!is_invalid_ddl_type(ddl_type)) {
            ret = OB_NOT_SUPPORTED;
            (void)snprintf(err_msg, sizeof(err_msg), "%s and %s in single statment",
                          ObIndexArg::to_type_str(last_type), ObIndexArg::to_type_str(type));
            LOG_USER_ERROR(OB_NOT_SUPPORTED, err_msg);
          } else if (orig_table_schema.is_heap_organized_table()) {
            ret = OB_NOT_SUPPORTED;
            (void)snprintf(err_msg, sizeof(err_msg), "%s in heap organized table",
                          ObIndexArg::to_type_str(type));
            LOG_USER_ERROR(OB_NOT_SUPPORTED, err_msg);
          } else {
            ddl_type = DDL_ALTER_PRIMARY_KEY;
            last_type = type;
          }
          break;
        }
        case ObIndexArg::ADD_INDEX:
        case ObIndexArg::REBUILD_INDEX:
        case ObIndexArg::DROP_INDEX:
        case ObIndexArg::ALTER_INDEX:
        case ObIndexArg::ALTER_INDEX_PARALLEL:
        case ObIndexArg::RENAME_INDEX:
        case ObIndexArg::ALTER_INDEX_TABLESPACE: {
          // offline ddl cannot appear at the same time with other ddl
          if (orig_table_schema.is_heap_organized_table() && OB_FAIL(check_alter_heap_table_index(type, orig_table_schema, index_arg))) {
            ret = OB_NOT_SUPPORTED;
            LOG_WARN("Alter heap table index failed", K(ret));
          }
          if (OB_FAIL(ret)) {
          } else if ((DDL_MODIFY_COLUMN == ddl_type || DDL_ADD_COLUMN_OFFLINE == ddl_type
              || DDL_ADD_COLUMN_ONLINE == ddl_type || DDL_TABLE_REDEFINITION == ddl_type
              || DDL_ADD_COLUMN_INSTANT == ddl_type)
              && ObIndexArg::ADD_INDEX == type) {
            // TODO(shuangcan): distinguish simple table and double table ddl
            ddl_type = ObDDLType::DDL_TABLE_REDEFINITION;
            last_type = type;
          } else if (is_long_running_ddl(ddl_type)) {
            ret = OB_NOT_SUPPORTED;
            (void)snprintf(err_msg, sizeof(err_msg), "%s and %s in single statment",
                          ddl_type_str(ddl_type), ObIndexArg::to_type_str(type));
            LOG_USER_ERROR(OB_NOT_SUPPORTED, err_msg);
          } else {
            ddl_type = ObDDLType::DDL_NORMAL_TYPE;
            last_type = type;
          }
          break;
        }
        case ObIndexArg::DROP_FOREIGN_KEY: {
          if (DDL_DROP_COLUMN == ddl_type) {
             // In oracle mode, we support to drop foreign key implicitly caused by drop column.
          } else if (is_long_running_ddl(ddl_type)) {
            ret = OB_NOT_SUPPORTED;
            (void)snprintf(err_msg, sizeof(err_msg), "%s and %s in single statment",
                          ddl_type_str(ddl_type), ObIndexArg::to_type_str(type));
            LOG_USER_ERROR(OB_NOT_SUPPORTED, err_msg);
          } else {
            ddl_type = ObDDLType::DDL_NORMAL_TYPE;
            last_type = type;
          }
          break;
        }
        default: {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("Unknown index action type!", K(type), K(ret));
        }
      }
      }
      if (OB_SUCC(ret)) {
        if (!has_drop_index) {
          has_drop_index = ObIndexArg::DROP_INDEX == type;
        }
        if (!has_create_index) {
          has_create_index = ObIndexArg::ADD_INDEX == type;
        }
        if (!has_drop_and_add_index) {
          has_drop_and_add_index = has_drop_index && has_create_index;
        }
        if (!is_add_many_fts_indexes) {
          is_add_many_fts_indexes =  is_add_fts_or_multivalue_index;
        } else if (is_add_fts_or_multivalue_index) {
          ret = OB_OP_NOT_ALLOW;
          LOG_WARN("adding many fulltext or multivalue indexes at the same time is a high-risk operation, which is not support", K(ret));
          LOG_USER_ERROR(OB_NOT_SUPPORTED,
              "adding many fulltext or multivalue indexes at the same time is a high-risk operation, which is");
        }

      }
    }
  }
  return ret;
}

int ObDDLService::check_convert_to_character(obrpc::ObAlterTableArg &alter_table_arg,
                                             const ObTableSchema &orig_table_schema,
                                             ObDDLType &ddl_type)
{
  int ret = OB_SUCCESS;
  // alter table table_name CONVERT TO CHARACTER SET charset_name [COLLATE collation_name]
  AlterTableSchema &alter_table_schema = alter_table_arg.alter_table_schema_;
  ObCharsetType charset_type = alter_table_schema.get_charset_type();
  ObCollationType collation_type = alter_table_schema.get_collation_type();
  if (CS_TYPE_INVALID == collation_type) {
    // If collation_type is not given, the default collation_type of charset_type is used
    collation_type = ObCharset::get_default_collation(charset_type);
    alter_table_schema.set_collation_type(collation_type);
    alter_table_schema.set_charset_type(charset_type);
  } else if (!ObCharset::is_valid_collation(charset_type, collation_type)) {
    ret = OB_ERR_COLLATION_MISMATCH;
    const char *cs_name = ObCharset::charset_name(charset_type);
    const char *coll_name = ObCharset::collation_name(collation_type);
    ObString charset = ObString::make_string(cs_name);
    ObString collation = ObString::make_string(coll_name);
    LOG_USER_ERROR(OB_ERR_COLLATION_MISMATCH, collation.length(), collation.ptr(),
                      charset.length(), charset.ptr());
  }
  // This is to do a performance optimization. If the collation_type of the original table is
  // equivalent to the new collation_type, do nothing
  if (OB_SUCC(ret) && orig_table_schema.get_collation_type() != collation_type) {
    if (is_long_running_ddl(ddl_type)) {
      ret = OB_NOT_SUPPORTED;
      LOG_USER_ERROR(OB_NOT_SUPPORTED, "There are several mutually exclusive DDL in single statement");
    } else {
      bool can_convert = false;
      bool convert_partition_key = false;
      ObTableSchema::const_column_iterator tmp_begin = orig_table_schema.column_begin();
      ObTableSchema::const_column_iterator tmp_end = orig_table_schema.column_end();
      for (; OB_SUCC(ret) && tmp_begin != tmp_end; tmp_begin++) {
        ObColumnSchemaV2 *col = (*tmp_begin);
        if (OB_ISNULL(col)) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("col is NULL", K(ret));
        } else if (ObDDLUtil::check_can_convert_character(col->get_meta_type())) {
          if (orig_table_schema.is_column_in_foreign_key(col->get_column_id())) {
            ret = OB_NOT_SUPPORTED;
            LOG_USER_ERROR(OB_NOT_SUPPORTED, "Alter column charset or collation with foreign key");
          } else if (OB_FAIL(orig_table_schema.is_column_in_partition_key(col->get_column_id(),
                                                                          convert_partition_key))) {
            LOG_WARN("fail to check if column in partition key", K(ret), "column id", col->get_column_id());
          }
        }
      }
      // change charset of a partitioned table can lead to repartition, we should handle it seperately.
      if (convert_partition_key) {
        if (is_long_running_ddl(ddl_type)) {
          // override the ret code here is by design.
          ret = OB_NOT_SUPPORTED;
          LOG_USER_ERROR(OB_NOT_SUPPORTED, "There are several mutually exclusive DDL in single statement");
        } else {
          ddl_type = ObDDLType::DDL_ALTER_PARTITION_BY;
        }
      } else {
        ddl_type = ObDDLType::DDL_CONVERT_TO_CHARACTER;
      }
    }
  }
  return ret;
}


int ObDDLService::check_is_add_identity_column(const share::schema::ObTableSchema &orig_table_schema,
                                               const share::schema::ObTableSchema &hidden_table_schema,
                                               bool &is_add_identity_column)
{
  int ret = OB_SUCCESS;
  ObTableSchema::const_column_iterator iter = orig_table_schema.column_begin();
  ObTableSchema::const_column_iterator end = orig_table_schema.column_end();
  ObTableSchema::const_column_iterator hidden_iter = hidden_table_schema.column_begin();
  ObTableSchema::const_column_iterator hidden_end = hidden_table_schema.column_end();
  is_add_identity_column = false;
  for (; OB_SUCC(ret) && hidden_iter != hidden_end; ++hidden_iter) {
    const ObColumnSchemaV2 *column = *hidden_iter;
    if (OB_ISNULL(column)) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("invalid column schema", K(column));
    } else if (column->is_identity_column()) {
      if (is_add_identity_column) {
        ret = OB_ERR_IDENTITY_COLUMN_COUNT_EXCE_LIMIT;
        LOG_USER_ERROR(OB_ERR_IDENTITY_COLUMN_COUNT_EXCE_LIMIT);
        LOG_WARN("add more than one identity column is not allowed");
      } else {
        is_add_identity_column = true;
      }
    }
  }
  for (; OB_SUCC(ret) && iter != end; ++iter) {
    const ObColumnSchemaV2 *column = *iter;
    if (OB_ISNULL(column)) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("invalid column schema", K(column));
    } else if (column->is_identity_column()) {
      is_add_identity_column = false;
    }
  }
  return ret;
}

int ObDDLService::check_alter_table_partition(const obrpc::ObAlterTableArg &alter_table_arg,
                                              const ObTableSchema &orig_table_schema,
                                              const bool is_oracle_mode,
                                              ObDDLType &ddl_type)
{
  int ret = OB_SUCCESS;
  uint64_t compat_version = OB_INVALID_VERSION;
  const uint64_t tenant_id = orig_table_schema.get_tenant_id();
  const uint64_t tablegroup_id = orig_table_schema.get_tablegroup_id();
  const ObPartitionLevel part_level = orig_table_schema.get_part_level();
  if (OB_FAIL(GET_MIN_DATA_VERSION(tenant_id, compat_version))) {
    LOG_WARN("get min data_version failed", K(ret), K(tenant_id));
  } else if (obrpc::ObAlterTableArg::REPARTITION_TABLE == alter_table_arg.alter_part_type_) {
    if (is_oracle_mode && PARTITION_LEVEL_ZERO != part_level) {
      ret = OB_NOT_SUPPORTED;
      LOG_USER_ERROR(OB_NOT_SUPPORTED, "re-partition a patitioned table");
    } else if (OB_INVALID_ID != tablegroup_id && compat_version < DATA_VERSION_4_2_0_0) {
      ret = OB_NOT_SUPPORTED;
      LOG_USER_ERROR(OB_NOT_SUPPORTED, "can't modify table partition in tablegroup");
    } else {
      if (is_long_running_ddl(ddl_type)) {
        ret = OB_NOT_SUPPORTED;
        LOG_USER_ERROR(OB_NOT_SUPPORTED, "There are several mutually exclusive DDL in single statement");
      } else {
        ddl_type = ObDDLType::DDL_ALTER_PARTITION_BY;
      }
    }
  } else {
    if (is_long_running_ddl(ddl_type)) {
      ret = OB_NOT_SUPPORTED;
      LOG_USER_ERROR(OB_NOT_SUPPORTED, "There are several mutually exclusive DDL in single statement");
    } else {
      ddl_type = ObDDLType::DDL_NORMAL_TYPE;
    }
  }
  return ret;
}

int ObDDLService::add_column_group(const obrpc::ObAlterTableArg &alter_table_arg,
                                   const share::schema::ObTableSchema &ori_table_schema,
                                   share::schema::ObTableSchema &new_table_schema)
{
  int ret = OB_SUCCESS;
  if (alter_table_arg.alter_table_schema_.get_column_group_count() <= 0) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("alter table arg has no column groups", K(ret), K(alter_table_arg));
  } else if (alter_table_arg.based_schema_object_infos_.count() <= 0) {
    /* based schema object infos is checked in the alter column group, here: only check count*/
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("based info object count <=0 cannot promise consist", K(ret));
  } else {
    ObTableSchema::const_column_group_iterator iter_begin =
        alter_table_arg.alter_table_schema_.column_group_begin();
    ObTableSchema::const_column_group_iterator iter_end =
        alter_table_arg.alter_table_schema_.column_group_end();

    for (; OB_SUCC(ret) && iter_begin != iter_end; iter_begin++) {
      bool cg_exist = false;
      ObColumnGroupSchema *column_group = *iter_begin;

      if (OB_ISNULL(column_group)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("column group should not be null", K(ret), K(alter_table_arg));
      } else if (column_group->get_column_group_id() <= new_table_schema.get_max_used_column_group_id()) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("added column group should have greater id than used column id",
                 K(ret), K(new_table_schema.get_max_used_column_group_id()),
                 K(column_group->get_column_group_id()));
      } else if (OB_FAIL(new_table_schema.add_column_group(*column_group))) {
        if (OB_HASH_EXIST == ret) {
          ret = OB_ERR_COLUMN_GROUP_DUPLICATE;
          LOG_WARN("fail to add column group, column group duplicate", K(ret), K(new_table_schema));
          char err_msg[OB_MAX_COLUMN_GROUP_NAME_LENGTH] = {'\0'};
          ObString err_msg_str(OB_MAX_COLUMN_GROUP_NAME_LENGTH, 0 /*length*/, err_msg);
          int tmp_ret = column_group->get_column_group_type_name(err_msg_str);
          if (tmp_ret != OB_SUCCESS) {
            LOG_WARN("fail to get readable column group name", K(tmp_ret), KPC(column_group));
          } else {
            LOG_USER_ERROR(OB_ERR_COLUMN_GROUP_DUPLICATE, err_msg_str.length(), err_msg_str.ptr());
          }
        } else {
          LOG_WARN("fail to add column group to table schema", K(ret), K(new_table_schema), KPC(column_group));
        }
      }
    }
    int tmp_ret = OB_SUCCESS;
#ifdef ERRSIM
    tmp_ret = OB_E(EventTable::EN_DDL_CREATE_OLD_VERSION_COLUMN_GROUP) OB_SUCCESS;
    if (OB_TMP_FAIL(tmp_ret)) {
      // EN_DDL_CREATE_OLD_VERSION_COLUMN_GROUP, code before v4.3.5.0
      /* note must alter rowkey cg first, else will affect default cg*/
      if (OB_FAIL(ObSchemaUtils::alter_rowkey_column_group(new_table_schema))) {
        LOG_WARN("fail to adjust rowkey column group when add column group", K(ret));
      } else if (OB_FAIL(ObSchemaUtils::alter_default_column_group(new_table_schema))) {
        LOG_WARN("fail to alter default column group", K(ret));
      }
    }
#endif
    if (OB_SUCC(ret) && OB_SUCCESS == tmp_ret) {
      if (OB_FAIL(adjust_cg_for_offline(new_table_schema))) {
        LOG_WARN("fail to adjust cg for offline", KR(ret));
      }
    }
  }
  return ret;
}

int ObDDLService::drop_column_group(const obrpc::ObAlterTableArg &alter_table_arg,
                                    const share::schema::ObTableSchema &ori_table_schema,
                                    share::schema::ObTableSchema &new_table_schema)
{
  int ret = OB_SUCCESS;
  bool has_unused_column = false;
  if (alter_table_arg.alter_table_schema_.get_column_group_count() <= 0) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("alter table arg has no column group", K(ret), K(alter_table_arg.alter_table_schema_));
  } else if (alter_table_arg.based_schema_object_infos_.count() <= 0) {
    /* based schema object infos is checked in the alter column group, here only check count*/
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("based schema object info count <= 0, cannot promise column consist", K(ret));
  } else {
    ObTableSchema::const_column_group_iterator iter_begin =
        alter_table_arg.alter_table_schema_.column_group_begin();
    ObTableSchema::const_column_group_iterator iter_end =
        alter_table_arg.alter_table_schema_.column_group_end();

    for (; OB_SUCC(ret) && iter_begin != iter_end; iter_begin++) {
      const ObColumnGroupSchema *column_group = *iter_begin;
      ObColumnGroupSchema *ori_column_group = nullptr;
      /* drop column group use column group name to get real column*/
      if (OB_ISNULL(column_group)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("column group in origin table should not be null", K(ret));
      } else if (OB_FAIL(ori_table_schema.get_column_group_by_name(column_group->get_column_group_name(),
                                                                   ori_column_group))) {
        /* if not exist in origin*/
        if (OB_HASH_NOT_EXIST == ret) {
          ret = OB_COLUMN_GROUP_NOT_FOUND;
          LOG_WARN("cannot found column group", KPC(column_group));
          char err_msg[OB_MAX_COLUMN_GROUP_NAME_LENGTH] = {'\0'};
          ObString err_msg_str(OB_MAX_COLUMN_GROUP_NAME_LENGTH, 0, err_msg);
          int tmp_ret = column_group->get_column_group_type_name(err_msg_str);
          if (tmp_ret != OB_SUCCESS){
            LOG_WARN("fail to get readable column group name");
          } else {
            LOG_USER_ERROR(OB_COLUMN_GROUP_NOT_FOUND, err_msg_str.length(), err_msg_str.ptr());
          }
        } else {
          LOG_WARN("fail to get column group by name", K(ret), K(ori_table_schema), KPC(column_group));
        }
      } else if (OB_ISNULL(ori_column_group)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("column group should not be null", K(ret), KPC(column_group));
      } else if (OB_FAIL(ori_table_schema.has_unused_column(has_unused_column))) {
        LOG_WARN("fail to check orig table schema has unused column", KR(ret), K(ori_table_schema));
      } else if (OB_FAIL(new_table_schema.remove_column_group(ori_column_group->get_column_group_id()))) {
        if (OB_HASH_NOT_EXIST == ret && has_unused_column) {
          // unused column and its' column group has deleted in delete_unused_columns_and_redistribute_schema
          ret = OB_SUCCESS;
        } else {
          LOG_WARN("fail to remove column group from new table schema", KR(ret));
        }
      }
    }
    int tmp_ret = OB_SUCCESS;
#ifdef ERRSIM
    tmp_ret = OB_E(EventTable::EN_DDL_CREATE_OLD_VERSION_COLUMN_GROUP) OB_SUCCESS;
    if (OB_TMP_FAIL(tmp_ret)) {
      // EN_DDL_CREATE_OLD_VERSION_COLUMN_GROUP, code before v4.3.5.0
      /* note must alter rowkey cg first, else will affect default cg*/
      if (OB_FAIL(ObSchemaUtils::alter_rowkey_column_group(new_table_schema))) {
        LOG_WARN("fail to adjust rowkey column group when add column group", K(ret));
      } else if (OB_FAIL(ObSchemaUtils::alter_default_column_group(new_table_schema))) {
        LOG_WARN("fail to alter default column group", K(ret));
      }
    }
#endif
    if (OB_SUCC(ret) && OB_SUCCESS == tmp_ret) {
      if (OB_FAIL(adjust_cg_for_offline(new_table_schema))) {
        LOG_WARN("fail to adjust cg for offline", KR(ret));
      }
    }
  }
  return ret;
}


int ObDDLService::alter_column_group(obrpc::ObAlterTableArg &alter_table_arg,
                                     const share::schema::ObTableSchema &orig_table_schema,
                                     share::schema::ObTableSchema &new_table_schema,
                                     share::schema::ObSchemaGetterGuard &schema_guard,
                                     ObDDLOperator &ddl_operator,
                                     common::ObMySQLTransaction &trans)
{
  int ret = OB_SUCCESS;
  bool bind_tablets = false;
  uint64_t compat_version = 0;
  if (alter_table_arg.alter_table_schema_.get_column_group_count() == 0) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("alter table arg has no column group", K(ret), K(alter_table_arg.alter_table_schema_));
  } else if (OB_FAIL(GET_MIN_DATA_VERSION(orig_table_schema.get_tenant_id(), compat_version))) {
    LOG_WARN("fail to get compat version", K(ret), K(orig_table_schema), K(compat_version));
  } else if (compat_version < DATA_VERSION_4_3_0_0) {
    ret = OB_NOT_SUPPORTED;
    LOG_WARN("compat version not support", K(ret), K(compat_version));
    LOG_USER_ERROR(OB_NOT_SUPPORTED, "tenant data version is less than 4.3, alter column group");
  } else if (alter_table_arg.based_schema_object_infos_.count() <= 0) {
    /* alter_table() has use check_parallel_ddl_conflict() before
       so here only need to check count
    */
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("there is no schema object infos to promise consit", K(ret));
  } else {
    new_table_schema.set_column_store(true);
    switch (alter_table_arg.alter_table_schema_.alter_type_) {
      case share::schema::OB_DDL_ADD_COLUMN_GROUP: {
        if (OB_FAIL(add_column_group(alter_table_arg, orig_table_schema, new_table_schema))) {
          LOG_WARN("fail to add column group to new table schema", K(ret));
        }
        break;
      }
      case share::schema::OB_DDL_DROP_COLUMN_GROUP: {
        if (OB_FAIL(drop_column_group(alter_table_arg, orig_table_schema, new_table_schema))) {
          LOG_WARN("fail to dorp column in new table schema", K(ret));
        }
        break;
      }
      default: {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("recevive unexpected alter table actions", K(ret),
                 K(alter_table_arg.alter_table_schema_.alter_type_));
      }
    }
    LOG_DEBUG("ddl service alter column group finish", K(ret), K(orig_table_schema), K(new_table_schema));
  }
  return ret;
}

int ObDDLService::update_column_group_table_inplace(const share::schema::ObTableSchema &origin_table_schema,
                                                    const share::schema::ObTableSchema &new_table_schema,
                                                    ObDDLOperator &ddl_operator,
                                                    common::ObMySQLTransaction &trans)
{
  int ret = OB_SUCCESS;
  if (OB_UNLIKELY(!origin_table_schema.is_valid()
                  || !new_table_schema.is_valid()
                  || !new_table_schema.is_column_store_supported())) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("invalid argument", K(ret), K(new_table_schema));
  } else if (OB_FAIL(ddl_operator.update_origin_column_group_with_new_schema(trans,
                                                                             origin_table_schema,
                                                                             new_table_schema))) {
    LOG_WARN("fail to clear origin table schema and insert new schema", K(ret),
                                                                        K(origin_table_schema),
                                                                        K(new_table_schema));
  }
  return ret;
}

int ObDDLService::adjust_cg_for_offline(ObTableSchema &new_table_schema)
{
  /* do adjustment on column group when add or drop column/primary key*/
  int ret = OB_SUCCESS;
  int tmp_ret = OB_SUCCESS;
#ifdef ERRSIM
  tmp_ret = OB_E(EventTable::EN_DDL_CREATE_OLD_VERSION_COLUMN_GROUP) OB_SUCCESS;
#endif
  bool is_each_cg_exist = false;
  bool is_all_cg_exist = false;
  if (!new_table_schema.is_column_store_supported()) {
    /*skip*/
  } else if (OB_FAIL(new_table_schema.is_column_group_exist(OB_ALL_COLUMN_GROUP_NAME, is_all_cg_exist))) {
    LOG_WARN("fail to check is all column group exist", K(ret));
  } else if (OB_FAIL(new_table_schema.is_column_group_exist(OB_EACH_COLUMN_GROUP_NAME, is_each_cg_exist))) {
    LOG_WARN("fail to check is each column group exist", K(ret));
  } else {
    /* for double_table_ddl reset all column groups*/
    new_table_schema.reset_column_group_info();
    /* add each column group*/
    ObTableSchema::const_column_iterator col_iter = new_table_schema.column_begin();
    for (; OB_SUCC(ret) && is_each_cg_exist && col_iter != new_table_schema.column_end(); col_iter++) {
      ObColumnSchemaV2 *col = *col_iter;
      ObColumnGroupSchema new_single_cg;
      new_single_cg.reset();
      if (OB_ISNULL(col)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("column group pointer should not be null", K(ret));
      } else if (col->is_virtual_generated_column()) {
        /* skip virtual column group*/
      } else if (OB_FAIL(ObSchemaUtils::build_single_column_group(new_table_schema, col, new_table_schema.get_tenant_id(),
                                                                  OB_SUCCESS != tmp_ret ? new_table_schema.get_max_used_column_group_id() + 1 : new_table_schema.get_next_single_column_group_id(),
                                                                  new_single_cg))) {
        LOG_WARN("fail to build single column group", K(ret));
      } else if (OB_FAIL(new_table_schema.add_column_group(new_single_cg))) {
        LOG_WARN("fail to add new column group to table schema", K(ret));
      }
    }
    /* add all column group*/
    if (OB_SUCC(ret) && is_all_cg_exist) {
      ObColumnGroupSchema new_cg;
      new_cg.reset();
      if (OB_FAIL(ObSchemaUtils::build_all_column_group(
                                   new_table_schema, new_table_schema.get_tenant_id(),
                                   OB_SUCCESS != tmp_ret ? new_table_schema.get_max_used_column_group_id() + 1 : ALL_COLUMN_GROUP_ID, new_cg))) {
        LOG_WARN("fail to build new all column group schema", K(ret));
      } else if (OB_FAIL(new_table_schema.add_column_group(new_cg))) {
        LOG_WARN("fail to add new column group to table schema", K(ret));
      }
    }
    /* adjust rowkey & default column group*/
    if (OB_SUCC(ret)) {
      ObArray<uint64_t> column_ids;
      ObColumnGroupSchema default_cg;
      default_cg.reset();
      if (OB_FAIL(ObSchemaUtils::build_column_group(new_table_schema, new_table_schema.get_tenant_id(),
                                        ObColumnGroupType::DEFAULT_COLUMN_GROUP,
                                        OB_DEFAULT_COLUMN_GROUP_NAME, column_ids,
                                        DEFAULT_TYPE_COLUMN_GROUP_ID, default_cg))) {
        LOG_WARN("fail to build column group", K(ret));
      } else if (OB_FAIL(new_table_schema.add_column_group(default_cg))) {
        LOG_WARN("failt to add default column group", K(ret));
      } else if (OB_FAIL(ObSchemaUtils::alter_rowkey_column_group(new_table_schema))) {
        LOG_WARN("fail to alter rowkey column group", K(ret));
      } else if (OB_FAIL(ObSchemaUtils::alter_default_column_group(new_table_schema))) {
        LOG_WARN("fail to alter default column grouop schema", K(ret));
      }
    }
    // adjust column group array order
    if (OB_SUCC(ret) && OB_SUCCESS == tmp_ret) {
      if (OB_FAIL(new_table_schema.adjust_column_group_array())) {
        LOG_WARN("fail to adjust column group array", K(ret), K(new_table_schema));
      }
    }
  }
  return ret;
}

int ObDDLService::gen_alter_partition_new_table_schema_offline(
    obrpc::ObAlterTableArg &alter_table_arg,
    const AlterTableSchema & alter_table_schema,
    const ObTableSchema &orig_table_schema,
    ObTableSchema &new_table_schema)
{
  int ret = OB_SUCCESS;
  bool has_drop_column_instant = false;
  if (OB_FAIL(new_table_schema.assign(orig_table_schema))) {
    LOG_WARN("fail to assign table schema", K(ret));
  } else if (OB_FAIL(new_table_schema.assign_partition_schema(alter_table_schema))) {
    LOG_WARN("fail to assign table partition schema", K(ret));
  } else {
    new_table_schema.reset_column_info();
    new_table_schema.reset_column_part_key_info();

    ObTableSchema::const_column_iterator tmp_begin = alter_table_schema.column_begin();
    ObTableSchema::const_column_iterator tmp_end = alter_table_schema.column_end();
    for (; OB_SUCC(ret) && tmp_begin != tmp_end; tmp_begin++) {
      ObColumnSchemaV2 *col = (*tmp_begin);
      if (OB_ISNULL(col)) {
        ret = OB_ERR_UNEXPECTED;
      } else if (col->is_unused()) {
        // by pass unused column.
        has_drop_column_instant = true;
      } else if (OB_FAIL(new_table_schema.add_column(*col))) {
        LOG_WARN("fail to add column", K(ret));
      }
    }
  }
  OZ (fill_interval_info_for_offline(alter_table_schema, new_table_schema), new_table_schema);
  if (OB_SUCC(ret) && alter_table_arg.is_alter_partitions_) {
    // In the alter table partition by range(xxx) (partitions...) case,
    // it is necessary to reset attributes of old table related to automatic partitioning here
    new_table_schema.get_part_option().assign_auto_partition_attr(orig_table_schema.get_part_option());
  }
  if (OB_SUCC(ret) && has_drop_column_instant) {
    if (OB_FAIL(redistribute_column_ids(new_table_schema))) {
      LOG_WARN("redistribute column id failed", KR(ret));
    }
  }
  return ret;
}

int ObDDLService::check_alter_partition_with_tablegroup(const ObTableSchema *orig_table_schema,
                                                        ObTableSchema &new_table_schema,
                                                        ObSchemaGetterGuard &schema_guard)
{
  int ret = OB_SUCCESS;
  const uint64_t tablegroup_id = new_table_schema.get_tablegroup_id();
  if (OB_FAIL(check_inner_stat())) {
    LOG_WARN("variable is not init", KR(ret));
  } else if (OB_INVALID_ID != tablegroup_id) {
    ObTableGroupHelp helper(*this, *schema_service_, *sql_proxy_);
    if (OB_FAIL(helper.check_table_partition_in_tablegroup(orig_table_schema, new_table_schema, schema_guard))) {
      LOG_WARN("check partition option failed", KR(ret));
    }
  }
  return ret;
}

int ObDDLService::alter_table_partition_by(
    obrpc::ObAlterTableArg &alter_table_arg,
    const ObTableSchema &orig_table_schema,
    ObTableSchema &new_table_schema,
    ObSchemaGetterGuard &schema_guard,
    ObDDLOperator &ddl_operator,
    ObMySQLTransaction &trans,
    const uint64_t tenant_data_version)
{
  int ret = OB_SUCCESS;
  const bool bind_tablets = false;
  AlterTableSchema &alter_table_schema = alter_table_arg.alter_table_schema_;
  OZ (gen_alter_partition_new_table_schema_offline(
      alter_table_arg, alter_table_schema, orig_table_schema, new_table_schema));
  if (FAILEDx(new_table_schema.check_validity_for_auto_partition())) {
    LOG_WARN("fail to check enable auto partitioning", KR(ret), K(new_table_schema));
  }
  OZ (create_user_hidden_table(orig_table_schema,
                              new_table_schema,
                              &alter_table_arg.sequence_ddl_arg_,
                              bind_tablets,
                              schema_guard,
                              schema_guard,
                              ddl_operator,
                              trans,
                              alter_table_arg.allocator_,
                              tenant_data_version));
  return ret;
}

// convert character set for ObBasePartition, inluding high_bound_val and list_row_values.
int ObDDLService::convert_to_character_for_partition(
    const ObCollationType &to_collation,
    ObTableSchema &new_table_schema)
{
  int ret = OB_SUCCESS;
  const ObPartitionLevel part_level = new_table_schema.get_part_level();
  const ObPartitionFuncType part_func_type = new_table_schema.get_part_option().get_part_func_type();
  const ObPartitionFuncType subpart_func_type = new_table_schema.get_sub_part_option().get_sub_part_func_type();
  if (PARTITION_LEVEL_MAX <= part_level) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("invalid arg", K(ret), K(part_level), K(new_table_schema));
  } else if (PARTITION_LEVEL_ZERO == part_level) {
    // non-partitioned, do nothing.
  } else {
    const int64_t part_num = new_table_schema.get_partition_num();
    ObPartition **part_array = new_table_schema.get_part_array();
    if (OB_ISNULL(part_array)) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("unexpected null first part array", K(ret), K(part_level), K(part_num), K(part_func_type));
    }
    // for the first-level part.
    for (int64_t i = 0; OB_SUCC(ret) && i < part_num; i++) {
      ObPartition *partition = part_array[i];
      if (OB_ISNULL(partition)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("unexpected err", K(ret));
      } else if (PARTITION_FUNC_TYPE_RANGE_COLUMNS == part_func_type
        && OB_FAIL(partition->convert_character_for_range_columns_part(to_collation))) {
          LOG_WARN("convert charset failed", K(ret), K(to_collation));
      } else if (PARTITION_FUNC_TYPE_LIST_COLUMNS == part_func_type
        && OB_FAIL(partition->convert_character_for_list_columns_part(to_collation))) {
        LOG_WARN("convert charset failed", K(ret), K(to_collation));
      } else if (PARTITION_LEVEL_TWO == part_level) {
        // for the second-level part.
        const int64_t subpart_num = partition->get_subpartition_num();
        ObSubPartition **subpart_array = partition->get_subpart_array();
        if (OB_ISNULL(subpart_array)) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("part array is null", K(ret));
        } else if (subpart_num < 1) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("sub_part_num less than 1", K(ret));
        } else {
          for (int64_t j = 0; OB_SUCC(ret) && j < subpart_num; j++) {
            ObSubPartition *subpart = subpart_array[j];
            if (OB_ISNULL(subpart)) {
              ret = OB_ERR_UNEXPECTED;
              LOG_WARN("unexpected err", K(ret));
            } else if (PARTITION_FUNC_TYPE_RANGE_COLUMNS == subpart_func_type
              && OB_FAIL(subpart->convert_character_for_range_columns_part(to_collation))) {
                LOG_WARN("convert charset failed", K(ret), K(to_collation));
            } else if (PARTITION_FUNC_TYPE_LIST_COLUMNS == subpart_func_type
              && OB_FAIL(subpart->convert_character_for_list_columns_part(to_collation))) {
              LOG_WARN("convert charset failed", K(ret), K(to_collation));
            }
          }
        }
      }
    }
  }
  // for def subpartition array.
  if (OB_SUCC(ret) && new_table_schema.has_sub_part_template_def()) {
    ObSubPartition **def_subpart_array = new_table_schema.get_def_subpart_array();
    const int64_t def_subpart_num = new_table_schema.get_def_sub_part_num();
    if (OB_ISNULL(def_subpart_array)) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("unexpected err, def subpart arr is null", K(ret), K(new_table_schema));
    } else {
      ObSubPartition *subpart_info = nullptr;
      for (int64_t i = 0; OB_SUCC(ret) && i < def_subpart_num; i++) {
        if (OB_ISNULL(subpart_info = def_subpart_array[i])) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("sub part is nullptr", K(ret));
        } else if (PARTITION_FUNC_TYPE_RANGE_COLUMNS == subpart_func_type
          && OB_FAIL(subpart_info->convert_character_for_range_columns_part(to_collation))) {
            LOG_WARN("convert charset failed", K(ret), K(to_collation));
        } else if (PARTITION_FUNC_TYPE_LIST_COLUMNS == subpart_func_type
          && OB_FAIL(subpart_info->convert_character_for_list_columns_part(to_collation))) {
          LOG_WARN("convert charset failed", K(ret), K(to_collation));
        }
      }
    }
  }
  return ret;
}

int ObDDLService::convert_to_character(
    obrpc::ObAlterTableArg &alter_table_arg,
    const ObTableSchema &orig_table_schema,
    ObTableSchema &new_table_schema,
    ObSchemaGetterGuard &schema_guard,
    ObDDLOperator &ddl_operator,
    ObMySQLTransaction &trans,
    const uint64_t tenant_data_version)
{
  int ret = OB_SUCCESS;
  const bool bind_tablets = false;
  bool can_convert = false;
  const ObSQLMode sql_mode = alter_table_arg.alter_table_schema_.get_sql_mode();
  bool is_oracle_mode = false;
  AlterTableSchema &alter_table_schema = alter_table_arg.alter_table_schema_;
  ObCollationType collation_type = alter_table_schema.get_collation_type();
  new_table_schema.set_collation_type(collation_type);
  new_table_schema.set_charset_type(ObCharset::charset_type_by_coll(collation_type));
  ObColumnIterByPrevNextID iter(orig_table_schema);
  if (OB_FAIL(orig_table_schema.check_if_oracle_compat_mode(is_oracle_mode))) {
    LOG_WARN("failed to get oracle mode", K(ret));
  } else if (is_oracle_mode) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("unexpected compat mode", K(ret), K(orig_table_schema));
  } else {
    while (OB_SUCC(ret)) {
      const ObColumnSchemaV2 *orig_col = nullptr;
      if (OB_FAIL(iter.next(orig_col))) {
        if (OB_ITER_END == ret) {
          ret = OB_SUCCESS;
          break;
        } else {
          LOG_WARN("iter failed", KR(ret), K(orig_table_schema));
        }
      } else if (OB_ISNULL(orig_col)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("col is NULL", K(ret));
      } else if (ObDDLUtil::check_can_convert_character(orig_col->get_meta_type())) {
        ObColumnSchemaV2 *col = new_table_schema.get_column_schema(orig_col->get_column_name());
        if (OB_ISNULL(col)) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("col is NULL", K(ret));
        } else {
          col->set_collation_type(collation_type);
          col->set_charset_type(ObCharset::charset_type_by_coll(collation_type));
          if (OB_FAIL(fill_column_collation(sql_mode,
                                            is_oracle_mode,
                                            new_table_schema,
                                            alter_table_arg.allocator_,
                                            *col))) {
            LOG_WARN("failed to fill column collation", K(ret));
          }
        }
      }
    }
    // convert character set for partition.
    if (OB_SUCC(ret)) {
      if (OB_FAIL(convert_to_character_for_partition(collation_type, new_table_schema))) {
        LOG_WARN("convert collation type for partition failed", K(ret));
      }
    }
    OZ (create_user_hidden_table(orig_table_schema,
                                new_table_schema,
                                &alter_table_arg.sequence_ddl_arg_,
                                bind_tablets,
                                schema_guard,
                                schema_guard,
                                ddl_operator,
                                trans,
                                alter_table_arg.allocator_,
                                tenant_data_version));
  }
  return ret;
}

// check whether drop primary key is allowed.
int ObDDLService::check_can_drop_primary_key(const ObTableSchema &origin_table_schema)
{
  int ret = OB_SUCCESS;
  bool is_oracle_mode = false;
  const ObIArray<ObForeignKeyInfo> &fk_infos = origin_table_schema.get_foreign_key_infos();
  ObPartitionFuncType part_func_type = origin_table_schema.get_part_option().get_part_func_type();
  // disallow to drop pk if the table is implicit key partition table.
  if (origin_table_schema.is_table_without_pk()) {
    const ObString pk_name = "PRIMAY";
    ret = OB_ERR_CANT_DROP_FIELD_OR_KEY;
    LOG_WARN("can't DROP 'PRIMARY', check primary key exists", K(ret), K(origin_table_schema));
    LOG_USER_ERROR(OB_ERR_CANT_DROP_FIELD_OR_KEY, pk_name.length(), pk_name.ptr());
  } else if (share::schema::PARTITION_FUNC_TYPE_KEY_IMPLICIT == part_func_type) {
    ret = OB_ERR_FIELD_NOT_FOUND_PART;
    LOG_WARN("can't drop primary key if table is implicit key partition table to be compatible with mysql mode", K(ret));
  } else if (fk_infos.empty()) {
    // allowed to drop primary key.
  } else if (OB_FAIL(origin_table_schema.check_if_oracle_compat_mode(is_oracle_mode))) {
    LOG_WARN("fail to check if oracle compat mode", K(ret));
  } else if (is_oracle_mode) {
    // disallowed to drop pk if the table is parent table.
    for (int64_t i = 0; OB_SUCC(ret) && i < fk_infos.count(); i++) {
      if (fk_infos.at(i).parent_table_id_ == origin_table_schema.get_table_id()) {
        ret = OB_ERR_ATLER_TABLE_ILLEGAL_FK;
        LOG_USER_ERROR(OB_ERR_ATLER_TABLE_ILLEGAL_FK);
        LOG_WARN("drop primary key is disallowed, due to referenced foreign key", K(ret));
      }
    }
  } else {
    // disallow to drop pk if,
    // 1. the table is parent table,
    // 2. the table is child table and the primary key columns contain foreign key refernce columns.
    for (int64_t i = 0; OB_SUCC(ret) && i < fk_infos.count(); i++) {
      if (fk_infos.at(i).parent_table_id_ == origin_table_schema.get_table_id()) {
        ret = OB_ERR_ATLER_TABLE_ILLEGAL_FK;
        LOG_USER_ERROR(OB_ERR_ATLER_TABLE_ILLEGAL_FK);
        LOG_WARN("drop primary key is disallowed, due to referenced foreign key", K(ret));
      } else {
        bool is_all_rowkey_columns = true; // check the refernce columns is all part of rowkey columns.
        FOREACH_CNT_X(col_id, fk_infos.at(i).child_column_ids_, OB_SUCC(ret) && is_all_rowkey_columns) {
          const ObColumnSchemaV2 *col_schema = nullptr;
          if (OB_ISNULL(col_schema = origin_table_schema.get_column_schema(*col_id))) {
            ret = OB_ERR_UNEXPECTED;
            LOG_WARN("child column of foreign key is nullptr", K(ret), K(*col_id));
          } else if (!col_schema->is_rowkey_column()) {
            is_all_rowkey_columns = false;
          }
        }
        if (OB_SUCC(ret) && is_all_rowkey_columns) {
          ret = OB_ERR_ATLER_TABLE_ILLEGAL_FK;
          LOG_USER_ERROR(OB_ERR_ATLER_TABLE_ILLEGAL_FK);
          LOG_WARN("drop primary key is disallowed, due to referenced foreign key", K(ret));
        }
      }
    }
  }
  return ret;
}

int ObDDLService::alter_table_primary_key(obrpc::ObAlterTableArg &alter_table_arg,
                                          const ObTableSchema &origin_table_schema,
                                          ObTableSchema &new_table_schema,
                                          ObSchemaGetterGuard &schema_guard,
                                          ObDDLOperator &ddl_operator,
                                          ObMySQLTransaction &trans,
                                          common::ObArenaAllocator &allocator,
                                          const uint64_t tenant_data_version)
{
  int ret = OB_SUCCESS;
  int64_t index_count = new_table_schema.get_index_count();
  int64_t index_aux_count = new_table_schema.get_index_tid_count();
  const ObSArray<ObIndexArg *> &index_arg_list = alter_table_arg.index_arg_list_;
  for (int64_t i = 0; OB_SUCC(ret) && i < index_arg_list.size(); ++i) {
    ObIndexArg *index_arg = const_cast<ObIndexArg *>(index_arg_list.at(i));
    if (OB_ISNULL(index_arg)) {
      ret = OB_INVALID_ARGUMENT;
      LOG_WARN("index arg should not be null", K(ret));
    } else {
      const ObIndexArg::IndexActionType type = index_arg->index_action_type_;
      switch(type) {
        case ObIndexArg::ADD_INDEX: {
          break;
        }
        case ObIndexArg::DROP_PRIMARY_KEY: {
          ObCreateIndexArg *create_index_arg = static_cast<ObCreateIndexArg *>(index_arg);
          ObSArray<ObString> index_columns;
          index_columns.reset();
          if (OB_FAIL(check_can_drop_primary_key(origin_table_schema))) {
            LOG_WARN("fail to check whether to allow to drop primary key", K(ret));
          } else if (OB_FAIL(create_hidden_table_with_pk_changed(alter_table_arg,
                                                                index_columns,
                                                                origin_table_schema,
                                                                new_table_schema,
                                                                schema_guard,
                                                                ddl_operator,
                                                                trans,
                                                                allocator,
                                                                type,
                                                                tenant_data_version))) {
            LOG_WARN("failed to add hidden primary key for heap table", K(ret));
          }
          break;
        }
        case ObIndexArg::ADD_PRIMARY_KEY:
        case ObIndexArg::ALTER_PRIMARY_KEY: {
          if (ObIndexArg::ADD_PRIMARY_KEY == type) {
            if (OB_MAX_AUX_TABLE_PER_MAIN_TABLE <= index_aux_count || OB_MAX_INDEX_PER_TABLE <= index_count) {
              ret = OB_ERR_TOO_MANY_KEYS;
              LOG_USER_ERROR(OB_ERR_TOO_MANY_KEYS, OB_MAX_INDEX_PER_TABLE);
              LOG_WARN("too many index or index aux for table!",
                       K(index_count), K(OB_MAX_INDEX_PER_TABLE), K(index_aux_count), K(OB_MAX_AUX_TABLE_PER_MAIN_TABLE));
            } else if (new_table_schema.is_table_with_pk()) {
              ret = OB_ERR_MULTIPLE_PRI_KEY;
              LOG_WARN("multiple primary key defined", K(ret));
            }
          } else {
            if (new_table_schema.is_table_without_pk()) {
              ret = OB_ERR_UNEXPECTED;
              LOG_WARN("primary key does not exist!", K(ret));
            }
          }
          if (OB_SUCC(ret)) {
            ObCreateIndexArg *create_index_arg = static_cast<ObCreateIndexArg *>(index_arg);
            ObSArray<ObString> index_columns;
            for (int32_t i = 0; OB_SUCC(ret)
                && i < create_index_arg->index_columns_.count(); ++i) {
              const ObColumnSortItem &col_item = create_index_arg->index_columns_.at(i);
              if (OB_FAIL(index_columns.push_back(col_item.column_name_))) {
                LOG_WARN("failed to add index column name", K(ret));
              }
            }
            if (OB_SUCC(ret) &&
                OB_FAIL(create_hidden_table_with_pk_changed(alter_table_arg,
                                                            index_columns,
                                                            origin_table_schema,
                                                            new_table_schema,
                                                            schema_guard,
                                                            ddl_operator,
                                                            trans,
                                                            allocator,
                                                            type,
                                                            tenant_data_version))) {
              LOG_WARN("failed to add primary key", K(ret));
            }
          }
          break;
        }
        default: {
          ret = OB_NOT_SUPPORTED;
          LOG_WARN("offline ddl cannot appear at the same time with other ddl", K(ret));
        }
      }
    }
  }
  return ret;
}

int ObDDLService::check_alter_table_constraint(
    const obrpc::ObAlterTableArg &alter_table_arg,
    const ObTableSchema &orig_table_schema,
    share::ObDDLType &ddl_type)
{
  int ret = OB_SUCCESS;
  char err_msg[number::ObNumber::MAX_PRINTABLE_SIZE] = {0};
  const ObAlterTableArg::AlterConstraintType type = alter_table_arg.alter_constraint_type_;
  bool change_cst_column_name = false;
  bool is_alter_decimal_int_offline = false;
  bool is_column_group_store = false;
  switch(type) {
    case obrpc::ObAlterTableArg::ADD_CONSTRAINT:
    case obrpc::ObAlterTableArg::ALTER_CONSTRAINT_STATE: {
      if (OB_FAIL(check_is_change_cst_column_name(orig_table_schema,
                                                  alter_table_arg.alter_table_schema_,
                                                  change_cst_column_name))) {
        LOG_WARN("failed to check change cst column name", K(ret));
      } else if ((ObDDLType::DDL_TABLE_REDEFINITION == ddl_type || ObDDLType::DDL_MODIFY_COLUMN == ddl_type)
                  && !change_cst_column_name) {
        ddl_type = ObDDLType::DDL_TABLE_REDEFINITION;
      } else if (is_long_running_ddl(ddl_type)) {
        // if modify auto_increment and constraint together, treat it as normal modify column
        ret = OB_NOT_SUPPORTED;
      } else if (change_cst_column_name) {
        ddl_type = ObDDLType::DDL_CHANGE_COLUMN_NAME;
        ret = OB_NOT_SUPPORTED;
      } else {
        ddl_type = ObDDLType::DDL_NORMAL_TYPE;
      }
      break;
    }
    // to avoid ddl type being modified from DROP_COLUMN to NORMAL_TYPE
    case obrpc::ObAlterTableArg::DROP_CONSTRAINT: {
      bool is_drop_col_only = false;
      if (ObDDLType::DDL_DROP_COLUMN == ddl_type) {
        // In oracle mode, we support to drop constraint implicitly caused by drop column.
      } else if (OB_FAIL(ObCODDLUtil::need_column_group_store(orig_table_schema, is_column_group_store))) {
        LOG_WARN("fail to check schema is column group store", K(ret));
      } else if (OB_FAIL(ObSchemaUtils::is_drop_column_only(alter_table_arg.alter_table_schema_, is_drop_col_only))) {
        LOG_WARN("fail to check is drop column only", K(ret), K(alter_table_arg.alter_table_schema_));
      } else if (ObDDLType::DDL_TABLE_REDEFINITION == ddl_type && is_drop_col_only && is_column_group_store) {
        // for column store, drop column is table redefinition
      } else if (OB_FAIL(check_is_alter_decimal_int_offline(ddl_type,
                                                            orig_table_schema,
                                                            alter_table_arg.alter_table_schema_,
                                                            is_alter_decimal_int_offline))) {
        LOG_WARN("fail to check is alter decimal int offline ddl", K(ret));
      } else if (is_long_running_ddl(ddl_type) && !is_alter_decimal_int_offline) {
        ret = OB_NOT_SUPPORTED;
      } else if (is_alter_decimal_int_offline) {
        ddl_type = ObDDLType::DDL_MODIFY_COLUMN;
      } else {
        ddl_type = ObDDLType::DDL_NORMAL_TYPE;
      }
      break;
    }
    default: {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("Unknown alter constraint action type!", K(ret), K(type));
    }
  }
  if (OB_NOT_SUPPORTED == ret) {
    (void)snprintf(err_msg, sizeof(err_msg), "%s and alter constraint in single statement", ddl_type_str(ddl_type));
    LOG_USER_ERROR(OB_NOT_SUPPORTED, err_msg);
  }
  return ret;
}

int ObDDLService::drop_constraint_caused_by_drop_column(
    const obrpc::ObAlterTableArg &alter_table_arg,
    share::schema::ObSchemaGetterGuard &schema_guard,
    const ObTableSchema &orig_table_schema,
    ObTableSchema &new_table_schema,
    ObDDLOperator &ddl_operator,
    common::ObMySQLTransaction &trans)
{
  int ret = OB_SUCCESS;
  const AlterTableSchema &alter_table_schema = alter_table_arg.alter_table_schema_;
  const ObSArray<ObIndexArg *> &index_arg_list = alter_table_arg.index_arg_list_;
  if (OB_SUCC(ret)) {
    if (OB_FAIL(check_can_alter_table_constraints(obrpc::ObAlterTableArg::DROP_CONSTRAINT, schema_guard, orig_table_schema, alter_table_schema))) {
      LOG_WARN("fail to check can alter constraints", K(ret), K(alter_table_schema));
    } else if (OB_FAIL(ddl_operator.drop_table_constraints(orig_table_schema, alter_table_schema, new_table_schema, trans))) {
      LOG_WARN("failed to drop table constraints", K(ret), K(alter_table_schema));
    }
  }
  return ret;
}

int ObDDLService::drop_rls_policy_caused_by_drop_column_online(
    share::schema::ObSchemaGetterGuard &schema_guard,
    const share::schema::ObTableSchema &origin_table_schema,
    const common::ObIArray<uint64_t> &drop_cols_id_arr,
    share::schema::ObTableSchema &new_table_schema,
    ObDDLOperator &ddl_operator,
    common::ObMySQLTransaction &trans)
{
  int ret = OB_SUCCESS;
  ObString empty_ddl_stmt;
  bool rls_object_changed = false;
  const uint64_t tenant_id = origin_table_schema.get_tenant_id();
  if (OB_FAIL(check_inner_stat())) {
    LOG_WARN("check inner stat failed", KR(ret));
  } else {
    for (int64_t i = 0; OB_SUCC(ret) && i < origin_table_schema.get_rls_policy_ids().count(); i++) {
      ObRlsPolicySchema new_rls_policy;
      const ObRlsPolicySchema *policy_schema = nullptr;
      const uint64_t policy_id = origin_table_schema.get_rls_policy_ids().at(i);
      if (OB_FAIL(schema_guard.get_rls_policy_schema_by_id(tenant_id, policy_id, policy_schema))) {
        LOG_WARN("get rls policy schema failed", KR(ret), K(policy_id));
      } else if (OB_ISNULL(policy_schema)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("unexpected null policy", KR(ret), K(policy_id));
      } else if (OB_FAIL(new_rls_policy.rebuild_with_table_schema(*policy_schema, new_table_schema))) {
        LOG_WARN("rebuild with table schema failed", KR(ret), K(policy_id));
      } else if (new_rls_policy.get_sec_column_count() == policy_schema->get_sec_column_count()) {
        // do nothing, the policy does not releated to the dropped column.
      } else if (OB_FALSE_IT(rls_object_changed = true)) {
      } else if (OB_FAIL(ddl_operator.drop_rls_policy(*policy_schema, trans, empty_ddl_stmt,
          false/*is_update_table_schema*/, nullptr/*table_schema*/))) {
        LOG_WARN("drop origin policy failed", K(ret));
      } else if (policy_schema->is_column_level_policy() && !new_rls_policy.is_column_level_policy()) {
        // column level policy will be dropped after drop column
      } else if (OB_FAIL(ddl_operator.create_rls_policy(new_rls_policy, trans, empty_ddl_stmt,
          false/*is_update_table_schema*/, nullptr/*table_schema*/))) {
        LOG_WARN("create new policy failed", KR(ret));
      }
    }
    if (OB_SUCC(ret) && rls_object_changed) {
      if (OB_FAIL(update_new_table_rls_flag(schema_guard, drop_cols_id_arr, new_table_schema))) {
        LOG_WARN("fail to update new table flags", KR(ret));
      } else if (OB_FAIL(ddl_operator.update_table_attribute(new_table_schema, trans, OB_DDL_ALTER_TABLE))) {
        LOG_WARN("fail to update table schema", KR(ret));
      }
    }
  }
  return ret;
}

int ObDDLService::drop_index_caused_by_drop_column_online(
    share::schema::ObSchemaGetterGuard &schema_guard,
    const share::schema::ObTableSchema &origin_table_schema,
    const common::ObIArray<uint64_t> &drop_cols_id_arr,
    common::ObIAllocator &allocator,
    ObDDLOperator &ddl_operator,
    common::ObMySQLTransaction &trans,
    common::ObIArray<ObDDLTaskRecord> &ddl_task_records)
{
  int ret = OB_SUCCESS;
  bool is_oracle_mode = false;
  ObSEArray<ObAuxTableMetaInfo, 16> simple_index_infos;
  const uint64_t tenant_id = origin_table_schema.get_tenant_id();
  const ObDatabaseSchema *database_schema = nullptr;
  if (OB_FAIL(check_inner_stat())) {
    LOG_WARN("check inner stat failed", KR(ret));
  } else if (OB_UNLIKELY(drop_cols_id_arr.empty())) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("invalid arg", KR(ret), K(drop_cols_id_arr));
  } else if (OB_FAIL(origin_table_schema.check_if_oracle_compat_mode(is_oracle_mode))) {
    LOG_WARN("check compat mode failed", KR(ret));
  } else if (OB_UNLIKELY(!is_oracle_mode)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("not oracle mode", KR(ret), K(tenant_id), "table_id", origin_table_schema.get_table_id());
  } else if (origin_table_schema.is_tmp_table()) {
    ret = OB_OP_NOT_ALLOW;
    char err_msg[OB_MAX_ERROR_MSG_LEN] = {0};
    (void)snprintf(err_msg, sizeof(err_msg), "drop column on temporary table is");
    LOG_WARN("drop column on temporary table is disallowed", KR(ret));
    LOG_USER_ERROR(OB_OP_NOT_ALLOW, err_msg);
  } else if (OB_FAIL(schema_guard.get_database_schema(tenant_id, origin_table_schema.get_database_id(), database_schema))) {
    LOG_WARN("get database schema failed", KR(ret), K(tenant_id), "db_id", origin_table_schema.get_database_id());
  } else if (OB_ISNULL(database_schema)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("unexpected err", KR(ret), K(tenant_id), "db_id", origin_table_schema.get_database_id());
  } else if (OB_UNLIKELY(database_schema->is_in_recyclebin())) {
    ret = OB_ERR_OPERATION_ON_RECYCLE_OBJECT;
    LOG_WARN("can not drop table column in recyclebin", KR(ret), K(tenant_id), "db_id", origin_table_schema.get_database_id());
  } else if (OB_FAIL(origin_table_schema.get_simple_index_infos(simple_index_infos))) {
    LOG_WARN("fail to get simple index infos", KR(ret));
  } else {
    for (int64_t i = 0; OB_SUCC(ret) && i < simple_index_infos.count(); i++) {
      const ObTableSchema *index_schema = nullptr;
      if (OB_FAIL(schema_guard.get_table_schema(tenant_id, simple_index_infos.at(i).table_id_, index_schema))) {
        LOG_WARN("get index schema failed", KR(ret), K(tenant_id), "table id", simple_index_infos.at(i).table_id_);
      } else if (OB_ISNULL(index_schema)) {
        ret = OB_TABLE_NOT_EXIST;
        LOG_WARN("index not exist", K(ret), K(tenant_id), "table id", simple_index_infos.at(i).table_id_);
      } else if (OB_UNLIKELY(index_schema->is_in_recyclebin())) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("unexpected error", KR(ret), K(tenant_id), "index_tid", simple_index_infos.at(i).table_id_);
      } else {
        ObString index_name;
        bool has_this_col = false;
        FOREACH_CNT_X(each_col_id, drop_cols_id_arr, OB_SUCC(ret) && !has_this_col) {
          if (OB_FAIL(index_schema->has_column(*each_col_id, has_this_col))) {
            LOG_WARN("check has column failed", KR(ret), K(*each_col_id), KPC(index_schema));
          } else if (!has_this_col) {
          } else if (OB_FAIL(index_schema->get_index_name(index_name))) {
            LOG_WARN("get index name without prefix failed", KR(ret), KPC(index_schema));
          } else {
            ObDDLTaskRecord drop_index_task_record;
            obrpc::ObDropIndexArg drop_index_arg;
            drop_index_arg.tenant_id_         = tenant_id;
            drop_index_arg.exec_tenant_id_    = tenant_id;
            drop_index_arg.index_table_id_    = index_schema->get_table_id();
            drop_index_arg.session_id_        = schema_guard.get_session_id();
            drop_index_arg.index_name_        = index_name;
            drop_index_arg.table_name_        = origin_table_schema.get_table_name();
            drop_index_arg.database_name_     = database_schema->get_database_name_str();
            drop_index_arg.index_action_type_ = obrpc::ObIndexArg::DROP_INDEX;
            // drop_index_arg.ddl_stmt_str_      = // empty.
            drop_index_arg.is_add_to_scheduler_ = false;
            drop_index_arg.is_hidden_         = origin_table_schema.is_user_hidden_table();
            drop_index_arg.is_in_recyclebin_  = index_schema->is_in_recyclebin();
            drop_index_arg.is_inner_          = true;
            obrpc::ObAlterTableRes ddl_res;
            if (OB_FAIL(drop_index_to_scheduler_(trans, schema_guard, allocator, origin_table_schema,
                                                 nullptr/*inc_tablet_ids*/, nullptr/*del_tablet_ids*/,
                                                 &drop_index_arg, ddl_operator,
                                                 ddl_res, ddl_task_records))) {
              LOG_WARN("fail to drop index to scheduler", KR(ret), K(drop_index_arg));
            }
          }
        }
      }
    }
  }

  return ret;
}

int ObDDLService::drop_lob_caused_by_drop_column_online_if_need(
    const obrpc::ObAlterTableArg &alter_table_arg,
    const share::schema::ObTableSchema &origin_table_schema,
    const share::schema::ObTableSchema &new_table_schema,
    common::ObIAllocator &allocator,
    common::ObMySQLTransaction &trans,
    common::ObIArray<ObDDLTaskRecord> &ddl_task_records,
    obrpc::ObAlterTableRes &res)
{
  int ret = OB_SUCCESS;
  bool is_oracle_mode = false;
  // To check whether there is any lob column before and after this drop column operation,
  // If so, a drop lob tablet task will be created to remove lob meta/piece tablet.
  const bool has_lob_column_before = origin_table_schema.has_lob_column(true/*ignore_unused_column*/);
  const bool has_lob_column_after = new_table_schema.has_lob_column(true/*ignore_unused_column*/);
  if (OB_FAIL(check_inner_stat())) {
    LOG_WARN("check inner stat failed", KR(ret));
  } else if (OB_FAIL(origin_table_schema.check_if_oracle_compat_mode(is_oracle_mode))) {
    LOG_WARN("check compat mode failed", KR(ret));
  } else if (OB_UNLIKELY(!is_oracle_mode)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("not oracle mode", KR(ret), "table_id", origin_table_schema.get_table_id());
  } else if (has_lob_column_before && !has_lob_column_after) {
    ObDDLTaskRecord task_record;
    if (OB_FAIL(submit_drop_lob_task_(trans, new_table_schema, alter_table_arg, allocator, task_record))) {
      LOG_WARN("submit drop lob task failed", KR(ret));
    } else if (OB_FAIL(ddl_task_records.push_back(task_record))) {
      LOG_WARN("push back failed", KR(ret));
    } else {
      // for wait ddl finish.
      res.task_id_ = task_record.task_id_;
    }
  }
  return ret;
}

int ObDDLService::get_all_dropped_udt_hidden_column_ids(const ObTableSchema &orig_table_schema, const ObColumnSchemaV2 &orig_column_schema,
                                                        common::ObIArray<uint64_t> &drop_cols_id_arr, int64_t &columns_cnt_in_new_table)
{
  int ret = OB_SUCCESS;
  bool is_oracle_mode = false;
  if (OB_FAIL(orig_table_schema.check_if_oracle_compat_mode(is_oracle_mode))) {
    LOG_WARN("fail to check if oracle compat mode", K(ret));
  } else if (is_oracle_mode && orig_column_schema.is_xmltype()) {
    ObSEArray<ObColumnSchemaV2 *, 1> hidden_cols;
    if (OB_FAIL(orig_table_schema.get_column_schema_in_same_col_group(orig_column_schema.get_column_id(),
                                                                      orig_column_schema.get_udt_set_id(),
                                                                      hidden_cols))) {
      LOG_WARN("failed to get column schema", K(ret));
    } else {
      for (int i = 0; i < hidden_cols.count() && OB_SUCC(ret); i++) {
        if (OB_FAIL(drop_cols_id_arr.push_back(hidden_cols.at(i)->get_column_id()))) {
          LOG_WARN("fail to push back column id", K(ret), KPC(hidden_cols.at(i)));
        } else {
          columns_cnt_in_new_table--;
        }
      }
    }
  }
  return ret;
}

int ObDDLService::get_all_dropped_column_ids(
    const obrpc::ObAlterTableArg &alter_table_arg,
    const ObTableSchema &orig_table_schema,
    common::ObIArray<uint64_t> &drop_cols_id_arr,
    int64_t *new_table_cols_cnt)
{
  int ret = OB_SUCCESS;
  int64_t columns_cnt_in_new_table = orig_table_schema.get_column_count();
  ObTableSchema::const_column_iterator it = nullptr;
  AlterColumnSchema *alter_column_schema = nullptr;
  const AlterTableSchema &alter_table_schema = alter_table_arg.alter_table_schema_;
  for (it = alter_table_schema.column_begin(); OB_SUCC(ret) && it != alter_table_schema.column_end(); it++) {
    if (OB_ISNULL(alter_column_schema = static_cast<AlterColumnSchema *>(*it))) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("alter_column_schema is NULL", K(ret), K(alter_table_schema));
    } else if (OB_DDL_ADD_COLUMN == alter_column_schema->alter_type_) {
      columns_cnt_in_new_table++;
    } else if (OB_DDL_DROP_COLUMN == alter_column_schema->alter_type_) {
      const ObString &orig_column_name = alter_column_schema->get_origin_column_name();
      const ObColumnSchemaV2 *orig_column_schema = orig_table_schema.get_column_schema(orig_column_name);
      if (OB_ISNULL(orig_column_schema)) {
        ret = OB_ERR_CANT_DROP_FIELD_OR_KEY;
        LOG_USER_ERROR(OB_ERR_CANT_DROP_FIELD_OR_KEY, orig_column_name.length(), orig_column_name.ptr());
        LOG_WARN("fail to find old column schema!", K(ret), K(orig_column_name), KPC(orig_column_schema));
      } else if (OB_FAIL(drop_cols_id_arr.push_back(orig_column_schema->get_column_id()))) {
        LOG_WARN("fail to push back column id", K(ret), KPC(orig_column_schema));
      } else {
        columns_cnt_in_new_table--;
        if (OB_FAIL(get_all_dropped_udt_hidden_column_ids(orig_table_schema, *orig_column_schema,
                                                          drop_cols_id_arr, columns_cnt_in_new_table))) {
           LOG_WARN("fail to push back udt hidden column id", K(ret), KPC(orig_column_schema));
        }
      }
    } else {/* do nothing. */}
  }
  if (OB_SUCC(ret) && nullptr != new_table_cols_cnt) {
    *new_table_cols_cnt = columns_cnt_in_new_table;
  }
  return ret;
}

int ObDDLService::check_can_drop_column(
    const ObString &orig_column_name,
    const ObColumnSchemaV2 *orig_column_schema,
    const ObTableSchema &orig_table_schema,
    const ObTableSchema &new_table_schema,
    const int64_t new_table_cols_cnt,
    ObSchemaGetterGuard &schema_guard)
{
  int ret = OB_SUCCESS;
  int64_t column_count = new_table_cols_cnt;
  if (OB_ISNULL(orig_column_schema) || OB_ISNULL(new_table_schema.get_column_schema(orig_column_name))) {
    ret = OB_ERR_CANT_DROP_FIELD_OR_KEY;
    LOG_USER_ERROR(OB_ERR_CANT_DROP_FIELD_OR_KEY, orig_column_name.length(), orig_column_name.ptr());
    LOG_WARN("fail to find old column schema!", K(ret), K(orig_column_name), KP(orig_column_schema),
        K(new_table_schema));
  } else if (orig_table_schema.is_oracle_tmp_table()) {
    ret = OB_NOT_SUPPORTED;
    LOG_USER_ERROR(OB_NOT_SUPPORTED, "drop oracle temporary table column is");
    LOG_WARN("oracle temporary table column is not allowed to be dropped", K(ret));
  } else if (orig_column_schema->has_generated_column_deps()) {
    bool has_func_idx_col_deps = false;
    bool is_oracle_mode = false;
    if (OB_FAIL(orig_table_schema.check_if_oracle_compat_mode(is_oracle_mode))) {
      LOG_WARN("fail to check if oracle compat mode", K(ret));
    } else if (!is_oracle_mode && OB_FAIL(orig_table_schema.check_functional_index_columns_depend(*orig_column_schema, schema_guard, has_func_idx_col_deps))) {
      LOG_WARN("fail to check if column has functional index dependency.", K(ret));
    } else if (!has_func_idx_col_deps) {
      ret = OB_ERR_DEPENDENT_BY_GENERATED_COLUMN;
      LOG_USER_ERROR(OB_ERR_DEPENDENT_BY_GENERATED_COLUMN, orig_column_name.length(), orig_column_name.ptr());
      LOG_WARN("Dropping column has generated column deps", K(ret), K(orig_column_name));
    } else {
      ret = OB_ERR_DEPENDENT_BY_FUNCTIONAL_INDEX;
      LOG_USER_ERROR(OB_ERR_DEPENDENT_BY_FUNCTIONAL_INDEX, orig_column_name.length(), orig_column_name.ptr());
      LOG_WARN("Dropping column has functional index column deps", K(ret), K(orig_column_name));
    }
  } else if (OB_FAIL(check_is_drop_partition_key(orig_table_schema, *orig_column_schema, schema_guard))) {
    LOG_WARN("check drop partition column failed", K(ret));
  } else if ((!new_table_schema.is_table_with_hidden_pk_column() && column_count <= ObTableSchema::MIN_COLUMN_COUNT_WITH_PK_TABLE)
      || (new_table_schema.is_table_with_hidden_pk_column() && column_count <= ObTableSchema::MIN_COLUMN_COUNT_WITH_HEAP_TABLE)) {
    ret = OB_CANT_REMOVE_ALL_FIELDS;
    LOG_USER_ERROR(OB_CANT_REMOVE_ALL_FIELDS);
    LOG_WARN("Can not delete all columns in table", K(ret));
  } else if (orig_column_schema->is_rowkey_column()) {
    ret = OB_NOT_SUPPORTED;
    LOG_USER_ERROR(OB_NOT_SUPPORTED, "drop rowkey column is");
    LOG_WARN("rowkey column is not allowed to be dropped", K(ret), K(orig_column_schema->get_column_name_str()));
  }
  return ret;
}

int ObDDLService::check_is_drop_partition_key(
    const share::schema::ObTableSchema &orig_table_schema,
    const ObColumnSchemaV2 &to_drop_column,
    ObSchemaGetterGuard &schema_guard)
{
  int ret = OB_SUCCESS;
  bool is_tbl_partition_key = false;
  ObSEArray<ObAuxTableMetaInfo, 16> simple_index_infos;
  const uint64_t tenant_id = orig_table_schema.get_tenant_id();
  const uint64_t to_drop_column_id = to_drop_column.get_column_id();
  const ObPartitionOption &part_option = orig_table_schema.get_part_option();
  const ObString &part_func_str = part_option.get_part_func_expr_str();
  if (OB_FAIL(orig_table_schema.get_simple_index_infos(simple_index_infos))) {
    LOG_WARN("get index infos failed", K(ret));
  } else if (!part_func_str.empty()/*nonempty means user-defined part key*/ && OB_FAIL(orig_table_schema.is_tbl_partition_key(to_drop_column_id, is_tbl_partition_key,
                                                           false /* ignore_presetting_key */))) {
    LOG_WARN("fail to check tbl partition key", K(ret), K(to_drop_column), K(orig_table_schema));
  } else {
    // to check whether the column is the partition key of the global index.
    for (int64_t idx = 0; OB_SUCC(ret) && !is_tbl_partition_key && idx < simple_index_infos.count(); idx++) {
      const ObTableSchema *index_schema = nullptr;
      const uint64_t index_tid = simple_index_infos.at(idx).table_id_;
      if (OB_FAIL(schema_guard.get_table_schema(tenant_id, index_tid, index_schema))) {
        LOG_WARN("get index schema failed", K(ret), K(tenant_id), K(index_tid));
      } else if (OB_ISNULL(index_schema)) {
        ret = OB_TABLE_NOT_EXIST;
        LOG_WARN("index not exist", K(ret), K(tenant_id), K(index_tid));
      } else if (index_schema->is_global_index_table()) {
        const ObPartitionOption &part_option = index_schema->get_part_option();
        const ObString &part_func_str = part_option.get_part_func_expr_str();
        if (!part_func_str.empty()/*nonempty means user-defined part key*/
            && OB_FAIL(index_schema->is_tbl_partition_key(to_drop_column_id, is_tbl_partition_key, false /* ignore_presetting_key */))) {
          LOG_WARN("check column in part key failed", K(ret), K(index_tid));
        }
      }
    }
  }
  if (OB_SUCC(ret) && is_tbl_partition_key) {
    ret = OB_ERR_DEPENDENT_BY_PARTITION_FUNC;
    LOG_USER_ERROR(OB_ERR_DEPENDENT_BY_PARTITION_FUNC,
                  to_drop_column.get_column_name_str().length(),
                  to_drop_column.get_column_name_str().ptr());
    LOG_WARN("drop column has table part key deps", K(ret), "column_name", to_drop_column.get_column_name_str());
  }
  return ret;
}

// to check whether the dropped column is related to constraint, and check can drop the column.
int ObDDLService::check_drop_column_with_drop_constraint(
    const obrpc::ObAlterTableArg &alter_table_arg,
    share::schema::ObSchemaGetterGuard &schema_guard,
    const ObTableSchema &orig_table_schema,
    const common::ObIArray<uint64_t> &drop_cols_id_arr)
{
  int ret = OB_SUCCESS;
  bool is_oracle_mode = false;
  const AlterTableSchema &alter_table_schema = alter_table_arg.alter_table_schema_;
  const ObAlterTableArg::AlterConstraintType type = alter_table_arg.alter_constraint_type_;
  if (OB_FAIL(orig_table_schema.check_if_oracle_compat_mode(is_oracle_mode))) {
    LOG_WARN("fail to check if oracle compat mode", K(ret));
  } else if (OB_FAIL(check_can_alter_table_constraints(type, schema_guard, orig_table_schema, alter_table_schema))) {
    LOG_WARN("fail to check can alter constraints", K(ret), K(type), K(alter_table_schema));
  } else {
    FOREACH_CNT_X(dropped_col, drop_cols_id_arr, OB_SUCC(ret)) {
      for (ObTableSchema::const_constraint_iterator iter = orig_table_schema.constraint_begin(); OB_SUCC(ret) &&
        iter != orig_table_schema.constraint_end(); iter++) {
        if (CONSTRAINT_TYPE_CHECK != (*iter)->get_constraint_type()) {
        } else if (0 == (*iter)->get_column_cnt()) {
        } else {
          const ObString &cst_name = (*iter)->get_constraint_name_str();
          for (ObConstraint::const_cst_col_iterator cst_col_iter = (*iter)->cst_col_begin();
            OB_SUCC(ret) && (cst_col_iter != (*iter)->cst_col_end()); ++cst_col_iter) {
            if (*cst_col_iter == *dropped_col) {
              // the dropped column is related to check constraint.
              const ObString &dropped_column_name = orig_table_schema.get_column_schema(*dropped_col)->get_column_name_str();
              ObConstraint* const* res = std::find_if(alter_table_schema.constraint_begin(), alter_table_schema.constraint_end(), [&cst_name](const ObConstraint* cst)
                             { return 0 == cst_name.case_compare(cst->get_constraint_name_str()); });
              if (alter_table_schema.constraint_end() == res) {
                LOG_WARN("the column is related to check constraint, can not be dropped", K(ret), K(cst_name), K(dropped_column_name));
                if (is_oracle_mode) {
                  ret = OB_ERR_MODIFY_OR_DROP_MULTI_COLUMN_CONSTRAINT;
                  LOG_USER_ERROR(OB_ERR_DROP_COL_REFERENCED_MULTI_COLS_CONSTRAINT,
                                 dropped_column_name.length(), dropped_column_name.ptr(),
                                 cst_name.length(), cst_name.ptr());
                } else {
                  ret = OB_ERR_DROP_COL_REFERENCED_MULTI_COLS_CONSTRAINT;
                  LOG_USER_ERROR(OB_ERR_DROP_COL_REFERENCED_MULTI_COLS_CONSTRAINT,
                                 cst_name.length(), cst_name.ptr(), dropped_column_name.length(),
                                 dropped_column_name.ptr());
                }
              }
            }
          }
        }
      }
    }
  }
  return ret;
}

// To be compatible with Mysql 5.6 and 8.0, follwing behavior on child table are allowed on OB 4.0:
// 1. drop foreign key non-related columns and drop any foreign key in single stmt;
// 2. drop the foreign key and its' some/all related columns in single stmt.
// 3. drop foreign key in the same time when drop fk-related column under oracle mode, but report error under mysql mode,
// ensured by the ObAlterTableArg.
// Notice that, drop fk related column on parent table has been processed in phase ddl resolver.
// Here, only need to report OB_ERR_ALTER_COLUMN_FK if drop foreign key related columns without drop the fk.
int ObDDLService::check_drop_column_with_drop_foreign_key(
    const obrpc::ObAlterTableArg &alter_table_arg,
    const ObTableSchema &orig_table_schema,
    const common::ObIArray<uint64_t> &drop_cols_id_arr)
{
  int ret = OB_SUCCESS;
  FOREACH_CNT_X(dropped_col, drop_cols_id_arr, OB_SUCC(ret)) {
    // 1. to iter all foreign keys related to the dropped column.
    const ObIArray<ObForeignKeyInfo> &foreign_key_infos = orig_table_schema.get_foreign_key_infos();
    for (int64_t i = 0; OB_SUCC(ret) && i < foreign_key_infos.count(); i++) {
      const ObForeignKeyInfo &fk_info = foreign_key_infos.at(i);
      const ObString &curr_fk_name = fk_info.foreign_key_name_;
      if (fk_info.child_table_id_ == orig_table_schema.get_table_id()) {
        bool is_drop_child_col = false;
        FOREACH_CNT_X(col_id, fk_info.child_column_ids_, OB_SUCC(ret) && !is_drop_child_col) {
          if (*dropped_col == *col_id) {
            is_drop_child_col = true;
          }
        }
        if (is_drop_child_col) {
          // 2. to check whether to drop the related foreign key.
          bool is_drop_curr_fk = false;
          const ObSArray<ObIndexArg *> &index_arg_list = alter_table_arg.index_arg_list_;
          for (int64_t i = 0; OB_SUCC(ret) && !is_drop_curr_fk && i < index_arg_list.size(); ++i) {
            ObIndexArg *index_arg = const_cast<ObIndexArg *>(index_arg_list.at(i));
            if (OB_ISNULL(index_arg)) {
              ret = OB_INVALID_ARGUMENT;
              LOG_WARN("invalid argument, index arg should not be null", K(ret), K(index_arg_list));
            } else if (ObIndexArg::DROP_FOREIGN_KEY != index_arg->index_action_type_) {
              ret = OB_INVALID_ARGUMENT;
              LOG_WARN("invalid argument", K(ret), K(index_arg));
            } else {
              const ObDropForeignKeyArg *drop_foreign_key_arg = static_cast<ObDropForeignKeyArg *>(index_arg);
              if (0 == curr_fk_name.case_compare(drop_foreign_key_arg->foreign_key_name_)) {
                is_drop_curr_fk = true;
              }
            }
          }
          // 3. drop child column of fk, but the fk is not dropped, should report error.
          if (OB_FAIL(ret)) {
          } else if (!is_drop_curr_fk) {
            const ObString &column_name = orig_table_schema.get_column_schema(*dropped_col)->get_column_name_str();
            ret = OB_ERR_ALTER_COLUMN_FK;
            LOG_USER_ERROR(OB_ERR_ALTER_COLUMN_FK, column_name.length(), column_name.ptr());
            LOG_WARN("the column is related to foreign key, and can not be dropped", K(ret), K(column_name), K(curr_fk_name));
          }
        }
      }
    }
  }
  return ret;
}

int ObDDLService::delete_column_from_schema(obrpc::ObAlterTableArg &alter_table_arg)
{
  int ret = OB_SUCCESS;
  if (OB_FAIL(check_inner_stat())) {
    LOG_WARN("variable is not init", K(ret));
  } else {
    ObSchemaGetterGuard schema_guard;
    AlterTableSchema &alter_table_schema = alter_table_arg.alter_table_schema_;
    uint64_t tenant_id = alter_table_schema.get_tenant_id();
    const ObString &orig_database_name = alter_table_schema.get_origin_database_name();
    const ObString &orig_table_name = alter_table_schema.get_origin_table_name();
    ObTableSchema new_table_schema;
    const ObTableSchema *orig_table_schema = nullptr;
    const ObTenantSchema *tenant_schema = nullptr;
    schema_guard.set_session_id(alter_table_arg.session_id_);
    ObDDLOperator ddl_operator(*schema_service_, *sql_proxy_);

    if (OB_FAIL(get_tenant_schema_guard_with_version_in_inner_table(tenant_id, schema_guard))) {
      LOG_WARN("fail to get schema guard with version in inner table", K(ret), K(tenant_id));
    } else if (OB_FAIL(schema_guard.get_tenant_info(tenant_id, tenant_schema))) {
      LOG_WARN("fail to get tenant schema", K(ret), K(tenant_id));
    } else if (OB_ISNULL(tenant_schema)) {
      ret =  OB_ERR_UNEXPECTED;
      LOG_WARN("tenant schema is null", K(ret), K(tenant_id));
    } else if (OB_FAIL(schema_guard.get_table_schema(tenant_id,
        orig_database_name,
        orig_table_name,
        false,  // is_index
        orig_table_schema))) {
      LOG_WARN("fail to get table schema", K(ret), K(orig_database_name), K(orig_table_name), KP(orig_table_schema));
    } else if (OB_ISNULL(orig_table_schema)) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("err unexpected", K(ret), K(orig_database_name), K(orig_table_name));
    } else if (OB_FAIL(new_table_schema.assign(*orig_table_schema))) {
      LOG_WARN("fail to assign schema", K(ret));
    } else {
      ObDDLSQLTransaction trans(schema_service_);
      int64_t refreshed_schema_version = 0;
      new_table_schema.set_in_offline_ddl_white_list(true);
      if (OB_FAIL(schema_guard.get_schema_version(tenant_id, refreshed_schema_version))) {
        LOG_WARN("failed to get tenant schema version", KR(ret), K(tenant_id));
      } else if (OB_FAIL(trans.start(sql_proxy_, tenant_id, refreshed_schema_version))) {
        LOG_WARN("start transaction failed", KR(ret), K(tenant_id), K(refreshed_schema_version));
      } else if (OB_FAIL(drop_constraint_caused_by_drop_column(alter_table_arg, schema_guard, *orig_table_schema, new_table_schema, ddl_operator, trans))) {
        LOG_WARN("fail to drop constraint caused by drop column", K(ret), K(alter_table_arg));
      } else if (OB_FAIL(delete_column_from_schema_in_trans(alter_table_schema, schema_guard, *orig_table_schema, new_table_schema,
          ddl_operator, trans))) {
        LOG_WARN("fail to set column to no minor", K(ret), K(new_table_schema));
      } else {
        ObSchemaOperationType operation_type = OB_DDL_ALTER_TABLE;
        if (OB_FAIL(ddl_operator.update_table_attribute(new_table_schema,
                                                        trans,
                                                        operation_type,
                                                        &alter_table_arg.ddl_stmt_str_))) {
          LOG_WARN("failed to update data table schema attribute", K(ret));
        }
      }
      if (trans.is_started()) {
        int temp_ret = OB_SUCCESS;
        if (OB_SUCCESS != (temp_ret = trans.end(OB_SUCC(ret)))) {
          LOG_WARN("trans end failed", "is_commit", OB_SUCCESS == ret, K(temp_ret));
          ret = (OB_SUCC(ret)) ? temp_ret : ret;
        }
      }
      if (OB_SUCC(ret)) {
        if (OB_FAIL(delete_auto_increment_attribute(orig_table_schema, new_table_schema, alter_table_schema))) {
          LOG_WARN("fail to delete auto-increment attri", K(ret), K(new_table_schema), K(alter_table_schema));
        } else if (OB_FAIL(publish_schema(tenant_id))) {
          LOG_WARN("fail to publish schema", K(ret), K(tenant_id));
        }
      }
    }
  }
  return ret;
}

int ObDDLService::delete_column_from_schema_in_trans(
    const AlterTableSchema &alter_table_schema,
    ObSchemaGetterGuard &schema_guard,
    const ObTableSchema &orig_table_schema,
    ObTableSchema &new_table_schema,
    ObDDLOperator &ddl_operator,
    common::ObMySQLTransaction &trans)
{
  int ret = OB_SUCCESS;
  int64_t new_schema_version = OB_INVALID_VERSION;
  ObTableSchema::const_column_iterator it = nullptr;
  AlterColumnSchema *alter_column_schema = nullptr;
  for (it = alter_table_schema.column_begin(); OB_SUCC(ret) && it != alter_table_schema.column_end(); it++) {
    if (OB_ISNULL(alter_column_schema = static_cast<AlterColumnSchema *>(*it))) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("alter_column_schema is NULL", K(ret), K(alter_table_schema));
    } else {
      const uint64_t tenant_id = orig_table_schema.get_tenant_id();
      const ObString &orig_column_name = alter_column_schema->get_origin_column_name();
      const ObColumnSchemaV2 *orig_column_schema = orig_table_schema.get_column_schema(orig_column_name);
      if (OB_ISNULL(orig_column_schema)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("ObColumnSchemaV2 should not be null", K(ret), K(*it));
      } else if (OB_FAIL(ddl_operator.drop_sequence_in_drop_column(
          *orig_column_schema,
          trans,
          schema_guard))) {
        LOG_WARN("alter table drop identity column fail", K(ret));
      } else if (OB_FAIL(ddl_operator.update_prev_id_for_delete_column(orig_table_schema,
          new_table_schema, *orig_column_schema, trans))) {
        LOG_WARN("failed to update column previous id for delele column", K(ret));
      } else if (OB_FAIL(schema_service_->gen_new_schema_version(tenant_id, new_schema_version))) {
        LOG_WARN("fail to gen new schema version", KR(ret), K(tenant_id));
      } else if (OB_FAIL(ddl_operator.delete_single_column(trans, new_schema_version, new_table_schema, orig_column_name))) {
        LOG_WARN("fail to delete column", K(ret), K(alter_column_schema));
      } else {
        LOG_INFO("delete column from schema", K(orig_column_name));
      }
    }
  }
  return ret;
}

int ObDDLService::delete_auto_increment_attribute(
    const ObTableSchema *orig_table_schema,
    ObTableSchema &new_table_schema,
    AlterTableSchema &alter_table_schema)
{
  int ret = OB_SUCCESS;
  uint64_t tenant_id = new_table_schema.get_tenant_id();
  uint64_t table_id  = new_table_schema.get_table_id();
  uint64_t part_num  = new_table_schema.get_all_part_num();
  // drop auto-increment attr in drop column
  ObAutoincrementService &autoinc_service = ObAutoincrementService::get_instance();
  if (0 != orig_table_schema->get_autoinc_column_id()
      && 0 == new_table_schema.get_autoinc_column_id()) {
    LOG_INFO("begin to clear all auto-increment cache",
        K(tenant_id), K(table_id), K(part_num),
        K(orig_table_schema->get_autoinc_column_id()));
    if (OB_FAIL(autoinc_service.clear_autoinc_cache_all(tenant_id,
        table_id,
        orig_table_schema->get_autoinc_column_id(),
        orig_table_schema->is_order_auto_increment_mode()))) {
      LOG_WARN("failed to clear auto-increment cache", K(tenant_id), K(table_id));
    }
  }
  // sync sync_value(auto_increment)
  uint64_t auto_increment = alter_table_schema.get_auto_increment();
  if (OB_SUCC(ret)) {
    if (new_table_schema.get_autoinc_column_id() > 0 && auto_increment > 0) {
      LOG_INFO("begin to sync auto_increment",
          K(tenant_id), K(table_id), K(part_num), K(auto_increment));
      if (OB_FAIL(autoinc_service.sync_auto_increment_all(tenant_id,
          table_id,
          new_table_schema.get_autoinc_column_id(),
          0 == auto_increment ? 0 : auto_increment - 1))) {
        LOG_WARN("failed to sync auto_increment", K(tenant_id), K(table_id), K(auto_increment));
      }
    }
  }
  return ret;
}

// rename and remap all index tables.
int ObDDLService::remap_index_tablets_to_new_indexs(
    obrpc::ObAlterTableArg &alter_table_arg,
    const ObTableSchema &orig_table_schema,
    const ObTableSchema &hidden_table_schema,
    ObSchemaGetterGuard &schema_guard,
    ObSArray<ObTableSchema> &table_schemas,
    common::ObMySQLTransaction &trans)
{
  int ret = OB_SUCCESS;
  int64_t new_schema_version = OB_INVALID_VERSION;
  const uint64_t tenant_id = orig_table_schema.get_tenant_id();
  ObSchemaService *schema_service = schema_service_->get_schema_service();
  ObSEArray<ObAuxTableMetaInfo, 16>simple_index_infos;
  if (OB_FAIL(check_inner_stat())) {
    LOG_WARN("variable is not inited", K(ret));
  } else if (OB_ISNULL(schema_service)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("schema service is null", K(ret));
  } else if (OB_FAIL(orig_table_schema.get_simple_index_infos(simple_index_infos))) {
    LOG_WARN("fail to get simple index infos failed", K(ret));
  } else {
    for (int64_t i = 0; OB_SUCC(ret) && i < simple_index_infos.count(); ++i) {
      const ObTableSchema *index_table_schema = nullptr;
      if (OB_FAIL(schema_guard.get_table_schema(tenant_id, simple_index_infos.at(i).table_id_, index_table_schema))) {
        LOG_WARN("fail to get index_table_schema", K(ret), K(tenant_id), "table_id", simple_index_infos.at(i).table_id_);
      } else if (OB_ISNULL(index_table_schema)) {
        ret = OB_TABLE_NOT_EXIST;
        LOG_WARN("index table schema should not be null", K(ret), "table_id", simple_index_infos.at(i).table_id_);
      } else {
        ObString new_index_table_name;
        HEAP_VAR(ObTableSchema, new_index_table_schema) {
          if (OB_FAIL(new_index_table_schema.assign(*index_table_schema))) {
            LOG_WARN("fail to assign new index table schema", K(ret));
          } else if (OB_FAIL(gen_new_index_table_name(index_table_schema->get_table_name_str(),
                                                      orig_table_schema.get_table_id(),
                                                      hidden_table_schema.get_table_id(),
                                                      alter_table_arg.allocator_,
                                                      new_index_table_name))) {
            LOG_WARN("fail to build new index table name!", K(ret), K(hidden_table_schema.get_table_id()), K(new_index_table_name));
          } else {
            uint64_t new_index_tid = OB_INVALID_ID;
            const ObString *ddl_stmt_str = nullptr;
            const bool need_sync_schema_version = true;
            const bool is_truncate_table = false;
            if (OB_FAIL(schema_service->fetch_new_table_id(tenant_id, new_index_tid))) {
              LOG_WARN("fail to fetch new table id", K(ret), K(tenant_id));
            } else if (OB_FAIL(schema_service_->gen_new_schema_version(tenant_id, new_schema_version))) {
              LOG_WARN("fail to gen new schema version", K(ret), K(tenant_id));
            } else {
              new_index_table_schema.set_table_id(new_index_tid);
              new_index_table_schema.set_data_table_id(hidden_table_schema.get_table_id());
              new_index_table_schema.set_table_name(new_index_table_name);
              new_index_table_schema.set_schema_version(new_schema_version);
              new_index_table_schema.set_tablet_id(index_table_schema->get_tablet_id());
              new_index_table_schema.set_in_offline_ddl_white_list(true);
              new_index_table_schema.set_table_state_flag(ObTableStateFlag::TABLE_STATE_NORMAL);
              if (OB_FAIL(schema_service->get_table_sql_service().create_table(new_index_table_schema,
                                                                               trans,
                                                                               ddl_stmt_str,
                                                                               need_sync_schema_version,
                                                                               is_truncate_table))) {
                LOG_WARN("fail to create table", K(ret), K(need_sync_schema_version), K(is_truncate_table));
              } else if (OB_FAIL(table_schemas.push_back(new_index_table_schema))) {
                LOG_WARN("fail to push back table schmea", K(ret));
              }
            }
          }
        }
      }
    }
  }
  return ret;
}

// remap and rename index tables, and then take effect.
int ObDDLService::swap_orig_and_hidden_table_state(
    obrpc::ObAlterTableArg &alter_table_arg,
    const ObTableSchema &orig_table_schema,
    const ObTableSchema &hidden_table_schema,
    ObSchemaGetterGuard &schema_guard,
    ObDDLOperator &ddl_operator,
    common::ObMySQLTransaction &trans,
    ObSArray<ObTableSchema> &table_schemas)
{
  int ret = OB_SUCCESS;
  const uint64_t tenant_id = orig_table_schema.get_tenant_id();
  if (OB_FAIL(check_inner_stat())) {
    LOG_WARN("variable is not init", K(ret));
  } else {
    HEAP_VARS_2((ObTableSchema, new_orig_table_schema),
                (ObTableSchema, new_hidden_table_schema)) {
      ObSEArray<ObAuxTableMetaInfo, 16> orig_simple_index_infos;
      if (OB_FAIL(new_orig_table_schema.assign(orig_table_schema))
                || OB_FAIL(new_hidden_table_schema.assign(hidden_table_schema))) {
        LOG_WARN("fail to assign schema", K(ret));
      } else if (OB_FAIL(orig_table_schema.get_simple_index_infos(
                 orig_simple_index_infos))) {
        LOG_WARN("fail to get simple index infos", K(ret));
      } else {
        // if the original table is a parent table, after swap the status of the two tables
        // need to rename the original fk name to the newly generated hidden fk name , and then
        // rename the newly generated hidden fk name to the original fk name on the child table
        if (orig_table_schema.is_parent_table()) {
          if (OB_FAIL(swap_all_child_table_fk_name(orig_table_schema,
                                                   hidden_table_schema,
                                                   schema_guard,
                                                   trans,
                                                   alter_table_arg.allocator_))) {
            LOG_WARN("fail to modify child table fk name", K(ret));
          }
        }
        if (OB_SUCC(ret) && alter_table_arg.need_rebuild_trigger_) {
          if (OB_FAIL(ObPLDDLService::rebuild_triggers_on_hidden_table(alter_table_arg,
                                                                       orig_table_schema,
                                                                       hidden_table_schema,
                                                                       schema_guard,
                                                                       schema_guard,
                                                                       ddl_operator,
                                                                       trans))) {
            LOG_WARN("fail to create triggers on hidden table", K(ret));
          }
        }

        if (OB_SUCC(ret)) {
          ObTableSchema tmp_schema;
          for (int64_t i = 0; OB_SUCC(ret) && i < orig_simple_index_infos.count(); i++) {
            tmp_schema.reset();
            const ObTableSchema *orig_index_table_schema = nullptr;
            if (OB_FAIL(schema_guard.get_table_schema(tenant_id,
                        orig_simple_index_infos.at(i).table_id_, orig_index_table_schema))) {
              LOG_WARN("fail to get orig index table schema", K(ret),
                       K(tenant_id), "table_id", orig_simple_index_infos.at(i).table_id_);
            } else if (OB_ISNULL(orig_index_table_schema)) {
              ret = OB_TABLE_NOT_EXIST;
              LOG_WARN("orig index table schema is null", K(ret), "table_id", orig_simple_index_infos.at(i).table_id_);
            } else if (OB_FAIL(tmp_schema.assign(*orig_index_table_schema))) {
              LOG_WARN("fail to assign orig index table schema", K(ret));
            } else {
              tmp_schema.set_table_state_flag(ObTableStateFlag::TABLE_STATE_HIDDEN_OFFLINE_DDL);
              if (OB_FAIL(table_schemas.push_back(tmp_schema))) {
                LOG_WARN("fail to add table schema", K(ret));
              }
            }
          }
          if (OB_SUCC(ret)) {
            new_orig_table_schema.set_table_state_flag(ObTableStateFlag::TABLE_STATE_HIDDEN_OFFLINE_DDL);
            new_hidden_table_schema.set_table_state_flag(ObTableStateFlag::TABLE_STATE_OFFLINE_DDL);
            new_orig_table_schema.set_table_name(hidden_table_schema.get_table_name_str());
            new_hidden_table_schema.set_table_name(orig_table_schema.get_table_name_str());
            if (OB_FAIL(table_schemas.push_back(new_orig_table_schema))
                || OB_FAIL(table_schemas.push_back(new_hidden_table_schema))) {
              LOG_WARN("fail to add table schema!", K(ret));
            }
          }
        }
        for (int64_t i = 0; OB_SUCC(ret) && i < table_schemas.count(); i++) {
          const ObString *ddl_stmt_str = nullptr;
          ObTableSchema &tmp_schema = table_schemas.at(i);
          ObSchemaOperationType operation_type = OB_DDL_ALTER_TABLE;
          tmp_schema.set_in_offline_ddl_white_list(true);
          if (tmp_schema.get_table_id() == orig_table_schema.get_table_id()) {
            ddl_stmt_str = &alter_table_arg.ddl_stmt_str_;
          }
          if (OB_FAIL(ddl_operator.update_table_attribute(tmp_schema,
                                                          trans,
                                                          operation_type,
                                                          ddl_stmt_str))) {
            LOG_WARN("fail to update data table schema attribute", K(ret));
          }
        }
      }
    }
  }
  return ret;
}

int ObDDLService::remap_index_tablets_and_take_effect(
    obrpc::ObAlterTableArg &alter_table_arg)
{
  int ret = OB_SUCCESS;
  ObSArray<uint64_t> index_ids;
  AlterTableSchema &alter_table_schema = alter_table_arg.alter_table_schema_;
  const uint64_t tenant_id = alter_table_schema.get_tenant_id();
  const uint64_t dest_tenant_id = alter_table_arg.exec_tenant_id_;
  if (OB_FAIL(check_inner_stat())) {
    LOG_WARN("variable is not init", K(ret));
  } else {
    ObDDLSQLTransaction trans(schema_service_);
    ObSArray<ObTableSchema> table_schemas;
    ObSchemaGetterGuard schema_guard;
    ObSchemaGetterGuard dest_schema_guard;
    const ObTableSchema *orig_table_schema = nullptr;
    const ObTableSchema *hidden_table_schema = nullptr;
    schema_guard.set_session_id(alter_table_arg.session_id_);
    int64_t refreshed_schema_version = 0;
    ObDDLOperator ddl_operator(*schema_service_, *sql_proxy_);
    if (OB_FAIL(get_tenant_schema_guard_with_version_in_inner_table(tenant_id, schema_guard))) {
      LOG_WARN("fail to get schema guard with version in inner table", K(ret), K(tenant_id));
    } else if (OB_FAIL(get_tenant_schema_guard_with_version_in_inner_table(dest_tenant_id, dest_schema_guard))) {
      LOG_WARN("fail to get schema guard with version in inner table", K(ret), K(dest_tenant_id));
    } else if (OB_FAIL(schema_guard.get_schema_version(tenant_id, refreshed_schema_version))) {
      LOG_WARN("fail to get tenant schema version", KR(ret), K(tenant_id));
    } else if (OB_FAIL(trans.start(sql_proxy_, tenant_id, refreshed_schema_version))) {
      LOG_WARN("fail to start trans, ", KR(ret), K(tenant_id), K(refreshed_schema_version));
    } else if (OB_FAIL(get_orig_and_hidden_table_schema(alter_table_arg,
                                                        schema_guard,
                                                        dest_schema_guard,
                                                        alter_table_schema,
                                                        orig_table_schema,
                                                        hidden_table_schema))) {
      LOG_WARN("fail to get orig and hidden table schema", K(ret));
    } else if (OB_FAIL(remap_index_tablets_to_new_indexs(alter_table_arg,
                                                          *orig_table_schema,
                                                          *hidden_table_schema,
                                                          schema_guard,
                                                          table_schemas,
                                                          trans))) {
      LOG_WARN("fail to remap all index tables to the hidden table", K(ret));
    } else if (OB_FAIL(swap_orig_and_hidden_table_state(alter_table_arg,
                                                        *orig_table_schema,
                                                        *hidden_table_schema,
                                                        schema_guard,
                                                        ddl_operator,
                                                        trans,
                                                        table_schemas))) {
      LOG_WARN("fail to swap origin and hidden table state", K(ret));
    }
    if (trans.is_started()) {
      int temp_ret = OB_SUCCESS;
      if (OB_SUCCESS != (temp_ret = trans.end(OB_SUCC(ret)))) {
        LOG_WARN("trans end failed", "is_commit", OB_SUCCESS == ret, K(temp_ret));
        ret = (OB_SUCC(ret)) ? temp_ret : ret;
      }
    }
  }
  if (OB_FAIL(ret)) {
  } else if (OB_FAIL(publish_schema(tenant_id))) {
    LOG_WARN("publish_schema failed", K(ret));
  }
  return ret;
}

int ObDDLService::switch_index_name_and_status_for_vec_index_table(obrpc::ObAlterTableArg &alter_table_arg)
{
  int ret = OB_SUCCESS;

  LOG_DEBUG("switch_index_name_and_status_for_vec_index_table", K(alter_table_arg));

  const int64_t old_index_id = alter_table_arg.table_id_;
  const int64_t new_index_id = alter_table_arg.hidden_table_id_;
  const int64_t tenant_id = alter_table_arg.alter_table_schema_.get_tenant_id();
  common::ObIAllocator &allocator = alter_table_arg.allocator_;
  if (OB_FAIL(check_inner_stat())) {
    LOG_WARN("variable is not init", K(ret));
  } else if (old_index_id == OB_INVALID_ID || new_index_id == OB_INVALID_ID
          || tenant_id == OB_INVALID_TENANT_ID) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("invalid argument", K(ret), K(old_index_id), K(new_index_id), K(tenant_id));
  } else {
    ObDDLSQLTransaction trans(schema_service_);
    const ObTableSchema *old_table_schema = NULL;
    const ObTableSchema *new_table_schema = NULL;
    ObSchemaGetterGuard schema_guard;
    ObDDLOperator ddl_operator(*schema_service_, *sql_proxy_);
    schema_guard.set_session_id(alter_table_arg.session_id_);
    int64_t refreshed_schema_version = 0;
    if (OB_FAIL(get_tenant_schema_guard_with_version_in_inner_table(tenant_id, schema_guard))) {
      LOG_WARN("fail to get schema guard with version in inner table", K(ret), K(tenant_id));
    } else if (OB_FAIL(schema_guard.get_schema_version(tenant_id, refreshed_schema_version))) {
      LOG_WARN("failed to get tenant schema version", KR(ret), K(tenant_id));
    } else if (OB_FAIL(trans.start(sql_proxy_, tenant_id, refreshed_schema_version))) {
      LOG_WARN("fail to start trans, ", KR(ret), K(tenant_id), K(refreshed_schema_version));
    } else if (OB_FAIL(schema_guard.get_table_schema(tenant_id, old_index_id, old_table_schema))) {
      LOG_WARN("fail to get old index table schema", K(ret), K(old_index_id));
    } else if (OB_ISNULL(old_table_schema)) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("unexpectd null pointer", K(ret));
    } else if (!old_table_schema->is_vec_index() || !old_table_schema->is_vec_domain_index()) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("old index table must be vec domain index", K(ret), K(old_table_schema));
    } else if (OB_FAIL(schema_guard.get_table_schema(tenant_id, new_index_id, new_table_schema))) {
      LOG_WARN("fail to get new index table schema", K(ret), K(new_index_id));
    } else if (OB_ISNULL(new_table_schema)) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("unexpectd null pointer", K(ret));
    } else if (!new_table_schema->is_vec_index() || !new_table_schema->is_vec_domain_index()) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("new index table must be vec delta_buffer_table", K(ret), K(new_table_schema));
    } else if (old_table_schema->get_database_id() != new_table_schema->get_database_id()) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("old table and new table should in same database", K(ret));
    } else if (old_table_schema->get_data_table_id() != new_table_schema->get_data_table_id()) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("old table and new table should in same data table", K(ret));
    } else {
      const int64_t database_id = old_table_schema->get_database_id();
      const int64_t data_table_id = old_table_schema->get_data_table_id();
      const bool is_index = true;
      ObString old_domain_index_name = old_table_schema->get_table_name_str();
      ObString new_domain_index_name = new_table_schema->get_table_name_str();
      ObArray<ObString> new_table_names;
      ObArray<ObString> old_table_names;
      const int64_t unused_id = 1;    // not set OB_INVALID_ID for sava check
      const int64_t EXPECTED_HNSW_UPDATE_TABLE_CNT = 7;    // 3 old index table + 3 new index table + 1 data data
      const int64_t EXPECTED_IVFFLAT_UPDATE_TABLE_CNT = 7;    // 3 old index table + 3 new index table + 1 data data
      const int64_t EXPECTED_IVFSQ8_UPDATE_TABLE_CNT = 9;    // 4 old index table + 4 new index table + 1 data data
      const int64_t EXPECTED_IVFPQ_UPDATE_TABLE_CNT = 9;    // 4 old index table + 4 new index table + 1 data data
      int64_t update_table_cnt = 0;
      const ObIndexType index_type = old_table_schema->get_index_type();
      if (share::schema::is_vec_hnsw_index(index_type)) {
        update_table_cnt = EXPECTED_HNSW_UPDATE_TABLE_CNT;
      } else if (share::schema::is_vec_ivfflat_index(index_type)) {
        update_table_cnt = EXPECTED_IVFFLAT_UPDATE_TABLE_CNT;
      } else if (share::schema::is_vec_ivfsq8_index(index_type)) {
        update_table_cnt = EXPECTED_IVFSQ8_UPDATE_TABLE_CNT;
      } else if (share::schema::is_vec_ivfpq_index(index_type)) {
        update_table_cnt = EXPECTED_IVFPQ_UPDATE_TABLE_CNT;
      }
      SMART_VAR(ObSArray<ObTableSchema>, table_schemas) {
        if (OB_FAIL(ObVectorIndexUtil::generate_switch_index_names(old_domain_index_name,
                                                                   new_domain_index_name,
                                                                   index_type,
                                                                   allocator,
                                                                   old_table_names,
                                                                   new_table_names))) {
          LOG_WARN("fail to generate switch index names");
        } else if (OB_FAIL(ObVectorIndexUtil::update_index_tables_status(tenant_id,
                                                                         database_id,
                                                                         old_table_names,
                                                                         new_table_names,
                                                                         ddl_operator,
                                                                         schema_guard,
                                                                         trans,
                                                                         table_schemas))) {
          LOG_WARN("fail to update index table status", K(ret), K(tenant_id));
        } else if (OB_FAIL(ObVectorIndexUtil::update_index_tables_attributes(tenant_id,
                                                                             database_id,
                                                                             data_table_id,
                                                                             update_table_cnt,
                                                                             old_table_names,
                                                                             new_table_names,
                                                                             ddl_operator,
                                                                             schema_guard,
                                                                             trans,
                                                                             table_schemas))) {
          LOG_WARN("fail to update index table attribute", K(ret), K(tenant_id));
        }
      } // end smart_var table_schemas
    }
    if (trans.is_started()) {
      int temp_ret = OB_SUCCESS;
      if (OB_SUCCESS != (temp_ret = trans.end(OB_SUCC(ret)))) {
        LOG_WARN("trans end failed", "is_commit", OB_SUCCESS == ret, K(temp_ret));
        ret = (OB_SUCC(ret)) ? temp_ret : ret;
      }
    }
  }

  if (OB_FAIL(ret)) {
  } else if (OB_FAIL(publish_schema(tenant_id))) {
    LOG_WARN("publish_schema failed", K(ret));
  }
  return ret;
}

int ObDDLService::update_autoinc_schema(obrpc::ObAlterTableArg &alter_table_arg)
{
  int ret = OB_SUCCESS;
  if (OB_FAIL(check_inner_stat())) {
    LOG_WARN("variable is not init", K(ret));
  } else {
    ObSchemaGetterGuard schema_guard;
    AlterTableSchema &alter_table_schema = alter_table_arg.alter_table_schema_;
    uint64_t tenant_id = alter_table_schema.get_tenant_id();
    uint64_t table_id = alter_table_schema.get_table_id();
    ObDDLOperator ddl_operator(*schema_service_, *sql_proxy_);
    ObDDLSQLTransaction trans(schema_service_);
    ObSchemaOperationType operation_type = OB_DDL_ALTER_TABLE;
    const ObTableSchema *curr_table_schema;
    ObTableSchema new_table_schema;
    int64_t refreshed_schema_version = 0;
    ObAutoincrementService &auto_inc_service = ObAutoincrementService::get_instance();

    if (OB_FAIL(schema_service_->get_tenant_schema_guard(tenant_id, schema_guard))) {
      LOG_WARN("get schema guard failed", K(ret), K(tenant_id));
    } else if (OB_FAIL(schema_guard.get_table_schema(tenant_id, table_id, curr_table_schema))) {
      LOG_WARN("get data table schema failed", K(ret), K(tenant_id), K(table_id));
    } else if (OB_ISNULL(curr_table_schema)) {
      ret = OB_TABLE_NOT_EXIST;
      LOG_WARN("cannot find orig table", K(ret), K(alter_table_arg));
    } else if (OB_FAIL(schema_guard.get_schema_version(tenant_id, refreshed_schema_version))) {
      LOG_WARN("failed to get tenant schema version", KR(ret), K(tenant_id));
    } else if (OB_FAIL(trans.start(sql_proxy_, tenant_id, refreshed_schema_version))) {
      LOG_WARN("failed to start trans, ", KR(ret), K(tenant_id), K(refreshed_schema_version));
    } else if (OB_UNLIKELY(curr_table_schema->get_table_state_flag() != ObTableStateFlag::TABLE_STATE_OFFLINE_DDL)) {
      ret = OB_NO_NEED_UPDATE;
      LOG_WARN("already updated", K(ret), K(tenant_id), K(table_id), K(curr_table_schema->get_schema_version()));
    } else if (OB_FAIL(new_table_schema.assign(*curr_table_schema))) {
      LOG_WARN("fail to assign table schema", K(ret));
    } else {
      ObTableSchema::const_column_iterator iter = alter_table_schema.column_begin();
      ObTableSchema::const_column_iterator iter_end = alter_table_schema.column_end();
      AlterColumnSchema *alter_column_schema = nullptr;
      ObColumnSchemaV2 *new_column_schema = nullptr;
      int64_t alter_column_num = 0;
      new_table_schema.set_in_offline_ddl_white_list(true);
      new_table_schema.set_table_state_flag(ObTableStateFlag::TABLE_STATE_NORMAL);
      for(; OB_SUCC(ret) && iter != iter_end; iter++) {
        if (alter_column_num != 0 || OB_ISNULL(alter_column_schema = static_cast<AlterColumnSchema *>(*iter))) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("unexpected alter_column_num or iter is NULL", K(ret), K(alter_column_num));
        } else {
          const ObString &orig_column_name = alter_column_schema->get_origin_column_name();
          const ObColumnSchemaV2 *curr_column_schema = curr_table_schema->get_column_schema(orig_column_name);
          const bool need_del_stat = false;
          new_column_schema = new_table_schema.get_column_schema(orig_column_name);
          if (OB_ISNULL(new_column_schema)) {
            ret = OB_ERR_UNEXPECTED;
            LOG_WARN("alter column schema is null", KR(ret), K(new_table_schema.get_table_id()),
                                                    K(orig_column_name));
          } else if (OB_ISNULL(curr_column_schema)) {
            ret = OB_ERR_UNEXPECTED;
            LOG_WARN("cur column schema is null", KR(ret), K(curr_table_schema->get_table_id()),
                                                  K(orig_column_name));
          } else {
            new_column_schema->set_autoincrement(alter_column_schema->is_autoincrement());
            new_column_schema->set_nullable(alter_column_schema->is_nullable());
            new_table_schema.set_auto_increment(alter_table_schema.get_auto_increment());
            new_table_schema.set_autoinc_column_id(alter_column_schema->get_column_id());

            // we need clear inner autoinc when add autoinc attribute bug/53305960
            if (new_column_schema->is_autoincrement() && !curr_column_schema->is_autoincrement()) {
              if (OB_FAIL(ddl_operator.try_reinit_autoinc_row(new_table_schema, trans))) {
                LOG_WARN("fail to reinit autoinc row", KR(ret), K(new_table_schema));
              }
            }
            if (FAILEDx(ddl_operator.update_single_column(trans,
                                                          *curr_table_schema,
                                                          new_table_schema,
                                                          *new_column_schema,
                                                          need_del_stat))) {
              LOG_WARN("update single column failed", K(ret), K(*new_column_schema));
            }
          }
          alter_column_num++;
        }
      }

      if (OB_SUCC(ret)) {
        const bool is_commit = alter_column_num == 1;
        if (OB_FAIL(ddl_operator.update_table_attribute(new_table_schema,
                                                        trans,
                                                        operation_type,
                                                        is_commit ? &alter_table_arg.ddl_stmt_str_ : nullptr))) {
          LOG_WARN("update table attribute failed", K(ret), K(new_table_schema));
        }
      }

      if (trans.is_started()) {
        int temp_ret = OB_SUCCESS;
        if (OB_SUCCESS != (temp_ret = trans.end(OB_SUCC(ret)))) {
          LOG_WARN("trans end failed", "is_commit", OB_SUCCESS == ret, K(temp_ret));
          ret = (OB_SUCC(ret)) ? temp_ret : ret;
        }
      }

      if (OB_FAIL(ret)) {
      } else if (OB_FAIL(publish_schema(tenant_id))) {
        LOG_WARN("publish_schema failed", K(ret));
      }
    }
  }
  if (OB_NO_NEED_UPDATE == ret) {
    ret = OB_SUCCESS;
  }
  return ret;
}

int ObDDLService::lock_partitions(ObMySQLTransaction &trans,
                                  const ObTableSchema &table_schema)
{
  int ret = OB_SUCCESS;
  ObTabletIDArray tablet_ids;
  const int64_t tenant_id = table_schema.get_tenant_id();
  const int64_t table_id = table_schema.get_table_id();
  // skip those type table for lock table
  if (!table_schema.has_tablet()
      || table_schema.is_aux_table()
      || table_schema.is_mlog_table()
      || table_schema.is_sys_table()) {
  } else if (OB_FAIL(table_schema.get_tablet_ids(tablet_ids))) {
    LOG_WARN("failed to get tablet ids", KR(ret), K(table_schema));
  } else if (OB_FAIL(lock_tablets(trans, tenant_id, table_id, tablet_ids))) {
    LOG_WARN("failed to get tablet ids", KR(ret), K(table_schema), K(table_schema));
  }
  return ret;
}

int ObDDLService::lock_tablets(ObMySQLTransaction &trans,
                               const int64_t tenant_id,
                               const int64_t table_id,
                               const ObTabletIDArray &tablet_ids)
{
  int ret = OB_SUCCESS;
  const int64_t timeout = 0;

  observer::ObInnerSQLConnection *conn = NULL;
  if (OB_ISNULL(conn = dynamic_cast<observer::ObInnerSQLConnection *>
                       (trans.get_connection()))) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("conn_ is NULL", KR(ret));
  } else {
    LOG_INFO("lock tablet", KR(ret), K(tablet_ids), K(table_id), K(tenant_id), KPC(conn));
    if (GET_MIN_CLUSTER_VERSION() < CLUSTER_VERSION_4_2_0_0) {
      for (int i = 0; i < tablet_ids.count() && OB_SUCC(ret); i++) {
        if (OB_FAIL(ObInnerConnectionLockUtil::lock_tablet(tenant_id,
                                                           table_id,
                                                           tablet_ids.at(i),
                                                           EXCLUSIVE,
                                                           timeout,
                                                           conn))) {
          LOG_WARN("lock dest table failed", KR(ret), K(table_id), K(tenant_id));
        }
      }
    } else {
        if (OB_FAIL(ObInnerConnectionLockUtil::lock_tablet(tenant_id,
                                                           table_id,
                                                           tablet_ids,
                                                           EXCLUSIVE,
                                                           timeout,
                                                           conn))) {
        LOG_WARN("lock dest table failed", KR(ret), K(table_id), K(tenant_id));
      }
    }
  }
  return ret;
}

int ObDDLService::lock_table(ObMySQLTransaction &trans,
                             const ObSimpleTableSchemaV2 &table_schema)
{
  int ret = OB_SUCCESS;

  const uint64_t table_id = table_schema.get_table_id();
  const int64_t tenant_id = table_schema.get_tenant_id();
  const int64_t timeout = 0;

  observer::ObInnerSQLConnection *conn = NULL;
  // skip those type table for lock table
  if (!table_schema.has_tablet()
      || table_schema.is_aux_table()
      || table_schema.is_mlog_table()
      || table_schema.is_sys_table()) {
  } else if (OB_ISNULL(conn = dynamic_cast<observer::ObInnerSQLConnection *>
                       (trans.get_connection()))) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("conn_ is NULL", KR(ret));
  } else {
    LOG_INFO("lock table", KR(ret), K(table_id), K(tenant_id), KPC(conn));
    if (OB_FAIL(ObInnerConnectionLockUtil::lock_table(tenant_id,
                                                      table_id,
                                                      EXCLUSIVE,
                                                      timeout,
                                                      conn))) {
      LOG_WARN("lock dest table failed", KR(ret), K(table_schema));
    }
  }
  return ret;
}

int ObDDLService::lock_mview(ObMySQLTransaction &trans, const ObSimpleTableSchemaV2 &table_schema)
{
  int ret = OB_SUCCESS;
  const int64_t tenant_id = table_schema.get_tenant_id();
  const uint64_t mview_id = table_schema.get_table_id();
  observer::ObInnerSQLConnection *conn = nullptr;
  uint64_t data_version = 0;
  if (OB_FAIL(GET_MIN_DATA_VERSION(tenant_id, data_version))) {
    LOG_WARN("fail to get data version", KR(ret), K(tenant_id));
  } else if (OB_UNLIKELY(data_version < DATA_VERSION_4_3_0_0)) {
    ret = OB_NOT_SUPPORTED;
    LOG_WARN("version lower than 4.3 does not support this operation", KR(ret));
    LOG_USER_ERROR(OB_NOT_SUPPORTED, "tenant's data version is below 4.3.0.0, mview is ");
  } else if (OB_UNLIKELY(!table_schema.is_materialized_view())) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("invalid args", KR(ret), K(table_schema));
  } else if (OB_ISNULL(conn = dynamic_cast<observer::ObInnerSQLConnection *>
                       (trans.get_connection()))) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("conn_ is NULL", KR(ret));
  } else {
    LOG_INFO("lock mview", KR(ret), K(mview_id), K(tenant_id), KPC(conn));
    ObLockObjRequest lock_arg;
    lock_arg.obj_type_ = ObLockOBJType::OBJ_TYPE_MATERIALIZED_VIEW;
    lock_arg.obj_id_ = mview_id;
    lock_arg.owner_id_ = ObTableLockOwnerID::default_owner();
    lock_arg.lock_mode_ = EXCLUSIVE;
    lock_arg.op_type_ = ObTableLockOpType::IN_TRANS_COMMON_LOCK;
    lock_arg.timeout_us_ = 0;
    if (OB_FAIL(ObInnerConnectionLockUtil::lock_obj(tenant_id, lock_arg, conn))) {
      LOG_WARN("fail to lock mview obj", KR(ret), K(tenant_id), K(lock_arg), KPC(conn));
    }
  }
  return ret;
}

int ObDDLService::lock_tables_of_database(const ObDatabaseSchema &database_schema,
                                          ObMySQLTransaction &trans)
{
  int ret = OB_SUCCESS;
  ObSchemaGetterGuard schema_guard;
  const uint64_t tenant_id = database_schema.get_tenant_id();
  const uint64_t database_id = database_schema.get_database_id();
  ObArray<const ObSimpleTableSchemaV2*> table_schemas;
  if (OB_ISNULL(schema_service_)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("schema_service is null", KR(ret));
  } else if (OB_FAIL(schema_service_->get_tenant_schema_guard(tenant_id, schema_guard))) {
    LOG_WARN("fail to get schema guard", KR(ret), K(tenant_id));
  } else if (OB_FAIL(schema_guard.get_table_schemas_in_database(tenant_id,
                                                                database_id,
                                                                table_schemas))) {
    LOG_WARN("fail to get table ids in database", K(tenant_id), K(database_id), K(ret));
  } else {
    const ObSimpleTableSchemaV2 *table_schema = NULL;
    for (int64_t i = 0; OB_SUCC(ret) && i < table_schemas.count(); i++) {
      table_schema = table_schemas.at(i);
      if (OB_ISNULL(table_schema)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("table schema should not be null", K(ret));
      } else if (table_schema->is_materialized_view() && OB_FAIL(lock_mview(trans, *table_schema))) {
        LOG_WARN("fail to lock mview", KR(ret), KPC(table_schema));
      }
    }
    for (int64_t i = 0; OB_SUCC(ret) && i < table_schemas.count(); i++) {
      table_schema = table_schemas.at(i);
      if (OB_ISNULL(table_schema)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("table schema should not be null", K(ret));
      } else if (!table_schema->check_can_do_ddl()) {
        ret = OB_NOT_SUPPORTED;
        LOG_WARN("offline ddl is being executed, other ddl operations are not allowed",
                 K(table_schema->get_table_id()), K(ret));
      } else if (OB_FAIL(lock_table(trans, *table_schema))) {
        LOG_WARN("fail to lock_table", KR(ret), KPC(table_schema));
      }
    }
  }
  return ret;
}

int ObDDLService::check_aux_index_schema_exist(
    const uint64_t tenant_id,
    const obrpc::ObCreateIndexArg &arg,
    ObSchemaGetterGuard &schema_guard,
    const ObTableSchema *data_schema,
    bool &is_exist,
    const ObTableSchema *&index_schema)
{
  int ret = OB_SUCCESS;
  is_exist = false;
  index_schema = nullptr;
  ObArenaAllocator allocator(ObModIds::OB_SCHEMA);
  ObIndexType index_type = arg.index_type_;
  ObString index_table_name;
  if (tenant_id == OB_INVALID_ID ||
      !arg.is_valid() ||
      OB_ISNULL(data_schema)) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("invalid argument", K(ret), K(tenant_id), K(arg), KPC(data_schema));
  } else if (OB_FAIL(ObTableSchema::build_index_table_name(allocator,
                                                           data_schema->get_table_id(),
                                                           arg.index_name_,
                                                           index_table_name))) {
    LOG_WARN("failed to construct index table name", K(ret),
        K(arg.index_name_));
  } else if (share::schema::is_fts_or_multivalue_index(index_type)
             || share::schema::is_vec_index(index_type)) {
    const uint64_t database_id = data_schema->get_database_id();
    if (OB_FAIL(schema_guard.get_table_schema(tenant_id,
                                              database_id,
                                              index_table_name,
                                              true/*is_index*/,
                                              index_schema,
                                              false/*with_hidden_flag*/,
                                              true/*is_built_in_index*/))) {
      LOG_WARN("failed to get index schema",
          K(ret), K(tenant_id), K(index_table_name));
    }

    if (OB_SUCC(ret) && OB_ISNULL(index_schema)) {
      if (OB_FAIL(schema_guard.get_table_schema(tenant_id,
                                                database_id,
                                                index_table_name,
                                                true/*is_index*/,
                                                index_schema,
                                                false/*with_hidden_flag*/,
                                                false/*is_built_in_index*/))) {
        LOG_WARN("failed to get index schema",
            K(ret), K(tenant_id), K(index_table_name));
      }
    }

    if (OB_FAIL(ret)) {
    } else if (OB_NOT_NULL(index_schema)) {
      if (index_schema->get_index_type() == index_type) {
        is_exist = true;
        LOG_INFO("fts index aux table already exist, no need to generate",
            K(index_table_name), K(index_schema->get_index_type()), K(index_type));
      } else {
        ret = OB_ERR_KEY_NAME_DUPLICATE;
        LOG_WARN("the index name is too special" \
            "the index name is not supported", K(ret), K(arg.index_name_));
      }
    }
  } else {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("unexpected index type", K(ret), K(index_type));
  }
  return ret;
}

int ObDDLService::generate_aux_index_schema_(
    const uint64_t tenant_id,
    ObSchemaGetterGuard &schema_guard,
    ObCreateIndexArg &create_index_arg,
    ObTableSchema &nonconst_data_schema,
    const ObTableSchema *data_schema,
    ObIArray<ObColumnSchemaV2*> &gen_columns,
    ObDDLSQLTransaction &trans,
    const uint64_t tenant_data_version,
    ObTableSchema &index_schema)
{
  int ret = OB_SUCCESS;
  if (!create_index_arg.is_valid() || OB_INVALID_TENANT_ID == tenant_id ||
      OB_ISNULL(data_schema)) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("invalid argument", K(ret), K(create_index_arg), KPC(data_schema));
  } else {
    ObIndexBuilder index_builder(*this);
    const bool global_index_without_column_info = true;
    if (create_index_arg.is_rebuild_index_) {
      if (OB_FAIL(ObVectorIndexUtil::generate_index_schema_from_exist_table(tenant_id,
                                                                            schema_guard,
                                                                            *this,
                                                                            create_index_arg,
                                                                            *data_schema,
                                                                            index_schema))) {
        LOG_WARN("fail to generate index schema from exist table", K(ret), K(tenant_id), K(create_index_arg));
      }
    }
    if (OB_FAIL(ret)) {
    } else if (!create_index_arg.is_rebuild_index_ &&
               OB_FAIL(index_builder.generate_schema(create_index_arg,
                                              nonconst_data_schema,
                                              global_index_without_column_info,
                                              true/*generate_id*/,
                                              index_schema))) {
      LOG_WARN("fail to generate schema", K(ret), K(create_index_arg));
    } else if (OB_FAIL(nonconst_data_schema.check_create_index_on_hidden_primary_key(index_schema))) {
      LOG_WARN("failed to check create index on table", K(ret), K(index_schema));
    } else if (gen_columns.empty()) {
      if (OB_FAIL(create_index_table(create_index_arg,
                                     tenant_data_version,
                                     index_schema,
                                     trans))) {
        LOG_WARN("fail to create index", K(ret), K(index_schema));
      }
    } else {
      if (OB_FAIL(create_inner_expr_index(trans,
                                          *data_schema,
                                          tenant_data_version,
                                          nonconst_data_schema,
                                          gen_columns,
                                          index_schema))) {
        LOG_WARN("fail to create inner expr index", K(ret));
      }
    }
  }
  return ret;
}

int ObDDLService::create_aux_index_task_(
    const ObTableSchema *data_schema,
    const ObTableSchema *idx_schema,
    ObCreateIndexArg &create_index_arg,
    ObArenaAllocator &allocator,
    const int64_t parent_task_id,
    const uint64_t tenant_data_version,
    ObDDLSQLTransaction &trans,
    ObDDLTaskRecord &task_record,
    const int64_t snapshot_version)
{
  int ret = OB_SUCCESS;
  if (OB_ISNULL(data_schema) ||
      OB_ISNULL(idx_schema) ||
      OB_ISNULL(GCTX.root_service_) ||
      OB_ISNULL(GCTX.sql_proxy_) ||
      !create_index_arg.is_valid()) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("invalid argument", KPC(data_schema), KPC(idx_schema),
        KP(GCTX.root_service_), KP(GCTX.sql_proxy_), K(create_index_arg));
  } else {
    bool need_partitioned = ((DATA_VERSION_4_2_2_0 <= tenant_data_version &&
                              tenant_data_version < DATA_VERSION_4_3_0_0) ||
                             tenant_data_version >= DATA_VERSION_4_3_2_0) &&
                            idx_schema->is_storage_local_index_table() &&
                            idx_schema->is_partitioned_table();
    ObCreateDDLTaskParam param(data_schema->get_tenant_id(),
                               need_partitioned ? ObDDLType::DDL_CREATE_PARTITIONED_LOCAL_INDEX : ObDDLType::DDL_CREATE_INDEX,
                               data_schema,
                               idx_schema,
                               0/*object_id*/,
                               idx_schema->get_schema_version(),
                               create_index_arg.parallelism_,
                               create_index_arg.consumer_group_id_,
                               &allocator,
                               &create_index_arg,
                               parent_task_id);
    param.tenant_data_version_ = tenant_data_version;
    param.fts_snapshot_version_ = snapshot_version;
    if (OB_FAIL(GCTX.root_service_->get_ddl_task_scheduler().
        create_ddl_task(param, trans, task_record))) {
      if (OB_ENTRY_EXIST == ret) {
        trans.reset_last_error();
        ret = OB_SUCCESS;
      } else {
        LOG_WARN("submit create index ddl task failed", K(ret));
      }
    }
  }
  return ret;
}

int ObDDLService::create_aux_index(
    const obrpc::ObCreateAuxIndexArg &arg,
    obrpc::ObCreateAuxIndexRes &result)
{
  int ret = OB_SUCCESS;
  const uint64_t tenant_id = arg.tenant_id_;
  const uint64_t data_table_id = arg.data_table_id_;
  int64_t index_status = ObIndexStatus::INDEX_STATUS_NOT_FOUND;
  ObArenaAllocator allocator(lib::ObLabel("DdlTaskTmp"));
  SMART_VARS_2((obrpc::ObCreateIndexArg, create_index_arg),
               (ObTableSchema, nonconst_data_schema)) {
    ObDDLSQLTransaction trans(&get_schema_service());
    int64_t refreshed_schema_version = 0;
    uint64_t tenant_data_version = 0;
    bool schema_already_exist = false;
    uint64_t index_table_id = OB_INVALID_ID;
    ObSchemaGetterGuard schema_guard;
    const ObTableSchema *data_schema = nullptr;
    const ObTableSchema *idx_schema = nullptr;
    ObSEArray<ObColumnSchemaV2 *, 1> gen_columns;
    ObArenaAllocator allocator(lib::ObLabel("DdlTaskTmp"));
    ObDDLTaskRecord task_record;
    if (!arg.is_valid()) {
      ret = OB_INVALID_ARGUMENT;
      LOG_WARN("invalid argument", K(ret), K(arg));
    } else if (OB_ISNULL(GCTX.root_service_)) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("root service is nullptr", K(ret));
    } else if (OB_FAIL(get_tenant_schema_guard_with_version_in_inner_table(tenant_id,
                                                                           schema_guard))) {
      LOG_WARN("get schema guard failed", K(ret));
    } else if (OB_FAIL(GET_MIN_DATA_VERSION(tenant_id, tenant_data_version))) {
      LOG_WARN("get min data version failed", K(ret), K(tenant_id));
    } else if (OB_FAIL(schema_guard.get_schema_version(tenant_id,
                                                       refreshed_schema_version))) {
      LOG_WARN("failed to get tenant schema version", KR(ret), K(tenant_id));
    } else if (OB_FAIL(trans.start(&get_sql_proxy(),
                                   tenant_id,
                                   refreshed_schema_version))) {
      LOG_WARN("start transaction failed", KR(ret), K(tenant_id),
          K(refreshed_schema_version));
    } else if (OB_FAIL(schema_guard.get_table_schema(tenant_id,
                                                     data_table_id,
                                                     data_schema))) {
      LOG_WARN("get table schema failed", K(ret), K(tenant_id), K(data_table_id));
    } else if (OB_ISNULL(data_schema)) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("error unexpected, table schema is nullptr", K(ret), K(data_table_id));
    } else if (OB_FAIL(nonconst_data_schema.assign(*data_schema))) {
      LOG_WARN("failed to assign to nonconst data schema", K(ret));
    } else if (OB_FAIL(create_index_arg.assign(arg.create_index_arg_))) {
      LOG_WARN("fail to assign create index arg", K(ret));
    } else if (tenant_data_version < DATA_VERSION_4_3_5_1
        && (share::schema::is_fts_index_aux(create_index_arg.index_type_) || share::schema::is_fts_doc_word_aux(create_index_arg.index_type_))
        && !create_index_arg.index_option_.parser_properties_.empty()) {
      ret = OB_NOT_SUPPORTED;
      LOG_WARN("parser properties isn't supported before version 4.3.5.1", K(ret));
      LOG_USER_ERROR(OB_NOT_SUPPORTED, "parser properties before version 4.3.5.1 is");
    /* is_fts_index means: rowkey-doc, doc-rowkey, fts, word-doc multivalue-index, fts-index all run here*/
    } else if (share::schema::is_fts_index(create_index_arg.index_type_)
      && OB_FAIL(ObFtsIndexBuilderUtil::adjust_fts_args(create_index_arg,
                                                        nonconst_data_schema,
                                                        allocator,
                                                        gen_columns))) {
      LOG_WARN("fail to adjust create index args", K(ret), K(create_index_arg));
    } else if (share::schema::is_multivalue_index_aux(create_index_arg.index_type_) /* only multivalue index 3rd table run here */
      && OB_FAIL(ObMulValueIndexBuilderUtil::adjust_mulvalue_index_args(create_index_arg,
                                                                        nonconst_data_schema,
                                                                        allocator,
                                                                        gen_columns))) {
      LOG_WARN("fail to adjust create index args", K(ret), K(create_index_arg));
    } else if (!create_index_arg.is_rebuild_index_
              && share::schema::is_vec_index(create_index_arg.index_type_)
              && OB_FAIL(ObVecIndexBuilderUtil::adjust_vec_args(create_index_arg,
                                                                nonconst_data_schema,
                                                                allocator,
                                                                gen_columns))) {
      LOG_WARN("fail to adjust expr index args", K(ret));
    } else if (!(share::schema::is_fts_index(create_index_arg.index_type_) ||
                share::schema::is_multivalue_index_aux(create_index_arg.index_type_) ||
                share::schema::is_vec_index(create_index_arg.index_type_))) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("fail to create aux index, index type invalid", K(ret), K(create_index_arg));
    } else if (OB_FAIL(check_aux_index_schema_exist(tenant_id,
                                                     arg.create_index_arg_,
                                                     schema_guard,
                                                     data_schema,
                                                     schema_already_exist,
                                                     idx_schema))) {
      LOG_WARN("failed to check if schema is generated for aux index table", K(ret), K(arg));
    } else if (schema_already_exist) {
      if (OB_ISNULL(idx_schema)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("idx_schema is nullptr", K(ret));
      } else if (FALSE_IT(index_table_id = idx_schema->get_table_id())) {
      } else if (FALSE_IT(index_status = idx_schema->get_index_status())) {
      } else if (ObIndexStatus::INDEX_STATUS_AVAILABLE == index_status) {
      // 1. index schema exists && index available, fetch index table id
        result.schema_generated_ = true;
        result.aux_table_id_ = index_table_id;
        result.ddl_task_id_ = OB_INVALID_ID; // no need to wait task
      } else { // 2. index schema exists && not available, create ddl task
        result.schema_generated_ = true;
        result.aux_table_id_ = index_table_id;
        if (OB_FAIL(create_aux_index_task_(data_schema,
                                           idx_schema,
                                           create_index_arg,
                                           allocator,
                                           arg.task_id_/*parent fts*/,
                                           tenant_data_version,
                                           trans,
                                           task_record,
                                           arg.snapshot_version_))) {
          LOG_WARN("failed to create aux index ddl task", K(ret), K(create_index_arg));
        } else if (FALSE_IT(result.ddl_task_id_ = task_record.task_id_)) {
        }
      }
      if (OB_FAIL(ret)) {
      } else if (OB_FAIL(ObDDLTaskRecordOperator::update_parent_task_message(tenant_id,
          arg.task_id_, *idx_schema, result.aux_table_id_, result.ddl_task_id_, ObDDLUpdateParentTaskIDType::UPDATE_CREATE_INDEX_ID, allocator, trans))) {
        LOG_WARN("fail to update parent task message", K(ret), K(arg.task_id_), K(idx_schema));
      }
    } else { // 3. index scheme not exist, generate schema && create ddl task
      ObTableSchema index_schema;
      if (OB_FAIL(generate_aux_index_schema_(tenant_id,
                                             schema_guard,
                                             create_index_arg,
                                             nonconst_data_schema,
                                             data_schema,
                                             gen_columns,
                                             trans,
                                             tenant_data_version,
                                             index_schema))) {
        LOG_WARN("failed to generate aux index schema", K(ret), K(create_index_arg));
      } else if (FALSE_IT(result.schema_generated_ = true)) {
      } else if (FALSE_IT(result.aux_table_id_ = index_schema.get_table_id())) {
      } else if (OB_FAIL(create_aux_index_task_(data_schema,
                                                &index_schema,
                                                create_index_arg,
                                                allocator,
                                                arg.task_id_/*parent fts*/,
                                                tenant_data_version,
                                                trans,
                                                task_record))) {
        LOG_WARN("failed to create aux index ddl task", K(ret), K(create_index_arg));
      } else if (FALSE_IT(result.ddl_task_id_ = task_record.task_id_)) {
      } else if (OB_FAIL(ObDDLTaskRecordOperator::update_parent_task_message(tenant_id,
          arg.task_id_, index_schema, result.aux_table_id_, result.ddl_task_id_, ObDDLUpdateParentTaskIDType::UPDATE_CREATE_INDEX_ID, allocator, trans))) {
        LOG_WARN("fail to update parent task message", K(ret), K(arg.task_id_), K(index_schema));
      }
    }
    if (trans.is_started()) {
      int temp_ret = OB_SUCCESS;
      if (OB_SUCCESS != (temp_ret = trans.end(OB_SUCC(ret)))) {
        LOG_WARN("trans end failed", "is_commit", OB_SUCCESS == ret, K(temp_ret));
        ret = (OB_SUCC(ret)) ? temp_ret : ret;
      }
    }
    if (OB_SUCC(ret)) {
      if (OB_FAIL(publish_schema(tenant_id))) {
        LOG_WARN("fail to publish schema", K(ret), K(tenant_id));
      } else if (OB_INVALID_ID == result.ddl_task_id_) { // no need to schedule
      } else {
        DEBUG_SYNC(CREATE_AUX_INDEX_TABLE);
        if (OB_FAIL(GCTX.root_service_->get_ddl_task_scheduler().
                         schedule_ddl_task(task_record))) {
          LOG_WARN("fail to schedule ddl task", K(ret), K(task_record));
        }
      }
    }
  }
  LOG_INFO("finish create aux index", K(ret), K(arg), K(result), "ddl_event_info", ObDDLEventInfo());
  return ret;
}

int ObDDLService::lock_tables_in_recyclebin(const ObDatabaseSchema &database_schema,
                                            ObMySQLTransaction &trans)
{
  int ret = OB_SUCCESS;
  ObSchemaGetterGuard schema_guard;
  ObArray<ObRecycleObject> recycle_objs;
  ObSchemaService *schema_service = nullptr;
  const uint64_t tenant_id = database_schema.get_tenant_id();
  const uint64_t database_id = database_schema.get_database_id();
  if (OB_ISNULL(schema_service_)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("schema_service is null", K(ret));
  } else if (OB_ISNULL(schema_service = schema_service_->get_schema_service())) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("schema service is null", K(ret));
  } else if (OB_FAIL(schema_service->fetch_recycle_objects_of_db(tenant_id,
                                                                 database_id,
                                                                 trans,
                                                                 recycle_objs))) {
    LOG_WARN("fetch recycle objects of db failed", K(ret));
  } else if (OB_FAIL(schema_service_->get_tenant_schema_guard(tenant_id, schema_guard))) {
    LOG_WARN("fail to get schema guard", KR(ret), K(tenant_id));
  } else {
    for (int64_t i = 0; OB_SUCC(ret) && i < recycle_objs.count(); ++i) {
      const ObRecycleObject &recycle_obj = recycle_objs.at(i);
      const ObSimpleTableSchemaV2* table_schema = NULL;
      if (OB_FAIL(schema_guard.get_simple_table_schema(recycle_obj.get_tenant_id(),
          recycle_obj.get_table_id(), table_schema))) {
        LOG_WARN("get table schema failed", K(ret), K(recycle_obj));
      } else if (OB_ISNULL(table_schema)) {
        ret = OB_TABLE_NOT_EXIST;
        LOG_WARN("table is not exist", K(ret), K(recycle_obj));
        LOG_USER_ERROR(OB_TABLE_NOT_EXIST, to_cstring(database_schema.get_database_name_str()),
                       to_cstring(recycle_obj.get_object_name()));
      } else if (table_schema->is_materialized_view() && OB_FAIL(lock_mview(trans, *table_schema))) {
        LOG_WARN("fail to lock mview", KR(ret), KPC(table_schema));
      }
    }
    for (int64_t i = 0; OB_SUCC(ret) && i < recycle_objs.count(); ++i) {
      const ObRecycleObject &recycle_obj = recycle_objs.at(i);
      const ObSimpleTableSchemaV2* table_schema = NULL;
      if (OB_FAIL(schema_guard.get_simple_table_schema(recycle_obj.get_tenant_id(),
          recycle_obj.get_table_id(), table_schema))) {
        LOG_WARN("get table schema failed", K(ret), K(recycle_obj));
      } else if (OB_ISNULL(table_schema)) {
        ret = OB_TABLE_NOT_EXIST;
        LOG_WARN("table is not exist", K(ret), K(recycle_obj));
        LOG_USER_ERROR(OB_TABLE_NOT_EXIST, to_cstring(database_schema.get_database_name_str()),
                       to_cstring(recycle_obj.get_object_name()));
      } else if (OB_FAIL(lock_table(trans, *table_schema))) {
        LOG_WARN("fail to lock_table", KR(ret), KPC(table_schema));
      }
    }
  }
  return ret;
}

int ObDDLService::create_index_tablet(const ObTableSchema &index_schema,
                                      ObMySQLTransaction &trans,
                                      share::schema::ObSchemaGetterGuard &schema_guard,
                                      const bool need_check_tablet_cnt,
                                      const uint64_t tenant_data_version)
{
  int ret = OB_SUCCESS;
  int64_t tenant_id = index_schema.get_tenant_id();
  SCN frozen_scn;
  if (OB_FAIL(check_inner_stat())) {
    LOG_WARN("check_inner_stat error", K(is_inited()), KR(ret));
  } else if (!index_schema.is_index_table() || tenant_data_version <= 0) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("arg must be index table", KR(ret), K(tenant_id), K(tenant_data_version), K(index_schema));
  } else if (OB_ISNULL(GCTX.root_service_)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("root service is null", KR(ret));
  } else if (OB_FAIL(ObMajorFreezeHelper::get_frozen_scn(tenant_id, frozen_scn))) {
    LOG_WARN("failed to get frozen status for create tablet", KR(ret), K(tenant_id));
  } else {
    int64_t start_usec = ObTimeUtility::current_time();
    ObTableCreator table_creator(
                   tenant_id,
                   frozen_scn,
                   trans);
    ObNewTableTabletAllocator new_table_tablet_allocator(
                              tenant_id,
                              schema_guard,
                              sql_proxy_);
    common::ObArray<share::ObLSID> ls_id_array;
    const uint64_t data_table_id = index_schema.get_data_table_id();
    const ObTableSchema *data_table_schema = NULL;
    const ObTablegroupSchema *data_tablegroup_schema = NULL; // keep NULL if no tablegroup
    if (OB_FAIL(table_creator.init(need_check_tablet_cnt))) {
      LOG_WARN("fail to init table creator", KR(ret));
    } else if (OB_FAIL(new_table_tablet_allocator.init())) {
      LOG_WARN("fail to init new table tablet allocator", KR(ret));
    } else if (OB_FAIL(schema_guard.get_table_schema(tenant_id, data_table_id, data_table_schema))) {
      LOG_WARN("failed to get table schema", KR(ret), K(tenant_id), K(data_table_id));
    } else if (OB_ISNULL(data_table_schema)) {
      ret = OB_TABLE_NOT_EXIST;
      LOG_WARN("data table schema not exists", KR(ret), K(data_table_id));
    } else if (OB_INVALID_ID != data_table_schema->get_tablegroup_id()) {
      if (OB_FAIL(schema_guard.get_tablegroup_schema(
          data_table_schema->get_tenant_id(),
          data_table_schema->get_tablegroup_id(),
          data_tablegroup_schema))) {
        LOG_WARN("get tablegroup_schema failed", KR(ret), KPC(data_table_schema));
      } else if (OB_ISNULL(data_tablegroup_schema)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("data_tablegroup_schema is null", KR(ret), KPC(data_table_schema));
      }
    }
    if (OB_FAIL(ret)) {
    } else if (index_schema.is_index_local_storage()) {
      ObSEArray<const share::schema::ObTableSchema*, 1> schemas;
      ObSEArray<bool, 1> need_create_empty_majors;
      if (OB_FAIL(schema_guard.get_table_schema(tenant_id, data_table_id, data_table_schema))) {
        LOG_WARN("failed to get table schema", KR(ret), K(tenant_id), K(data_table_id));
      } else if (OB_ISNULL(data_table_schema)) {
        ret = OB_TABLE_NOT_EXIST;
        LOG_WARN("data table schema not exists", KR(ret), K(data_table_id));
      } else if (OB_FAIL(schemas.push_back(&index_schema))
        || OB_FAIL(need_create_empty_majors.push_back(false))) {
        LOG_WARN("failed to push_back", KR(ret), K(index_schema));
      } else if (OB_FAIL(new_table_tablet_allocator.prepare(trans, index_schema, data_tablegroup_schema))) {
        LOG_WARN("fail to prepare ls for index schema tablets", KR(ret));
      } else if (OB_FAIL(new_table_tablet_allocator.get_ls_id_array(
              ls_id_array))) {
        LOG_WARN("fail to get ls id array", KR(ret));
      } else if (OB_FAIL(table_creator.add_create_tablets_of_local_aux_tables_arg(
              schemas,
              data_table_schema,
              ls_id_array,
              tenant_data_version,
              need_create_empty_majors /* add_index, need_create_empty_major_sstable*/))) {
        LOG_WARN("create table tablet failed", KR(ret), K(index_schema));
      }
    } else {
      if (OB_FAIL(new_table_tablet_allocator.prepare(trans, index_schema, data_tablegroup_schema))) {
        LOG_WARN("fail to prepare ls for index schema tablets");
      } else if (OB_FAIL(new_table_tablet_allocator.get_ls_id_array(
              ls_id_array))) {
        LOG_WARN("fail to get ls id array", KR(ret));
      } else if (OB_FAIL(table_creator.add_create_tablets_of_table_arg(
            index_schema,
            ls_id_array,
            tenant_data_version,
            false /*need_create_empty_major_sstable*/))) {
        LOG_WARN("create table tablet failed", KR(ret), K(index_schema));
      }
    }

    // execute create tablet
    if (OB_FAIL(ret)) {
    } else if (OB_FAIL(table_creator.execute())) {
      LOG_WARN("execute create partition failed", KR(ret));
    }

    // finishing is always invoked for new table tablet allocator
    int tmp_ret = OB_SUCCESS;
    if (OB_SUCCESS != (tmp_ret = new_table_tablet_allocator.finish(OB_SUCCESS == ret))) {
      LOG_WARN("fail to finish new table tablet allocator", KR(tmp_ret));
    }

    int64_t cost_usec = ObTimeUtility::current_time() - start_usec;
    LOG_INFO("create table partitions cost: ", K(cost_usec));
  }
  return ret;
}

int ObDDLService::check_index_on_foreign_key(const ObTableSchema *index_table_schema,
                                             const common::ObIArray<ObForeignKeyInfo> &foreign_key_infos,
                                             bool &have_index)
{
  int ret = OB_SUCCESS;
  have_index = false;
  if (foreign_key_infos.count() <= 0) {
  } else if (OB_ISNULL(index_table_schema)) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("index table schema is nullptr", K(ret));
  } else {
    const uint64_t index_table_id = index_table_schema->get_table_id();
    for (int64_t i = 0;  OB_SUCC(ret) && i < foreign_key_infos.count(); i++) {
      if (foreign_key_infos.at(i).ref_cst_id_ == index_table_id) {
        have_index = true;
        break;
      }
    }
  }
  return ret;
}

int ObDDLService::alter_table_index(obrpc::ObAlterTableArg &alter_table_arg,
                                    const ObTableSchema &origin_table_schema,
                                    ObTableSchema &new_table_schema,
                                    ObSchemaGetterGuard &schema_guard,
                                    ObDDLOperator &ddl_operator,
                                    ObMySQLTransaction &trans,
                                    ObArenaAllocator &allocator,
                                    const uint64_t tenant_data_version,
                                    obrpc::ObAlterTableRes &res,
                                    ObIArray<ObDDLTaskRecord> &ddl_tasks,
                                    int64_t &new_fetched_snapshot)
{
  int ret = OB_SUCCESS;
  ObIndexBuilder index_builder(*this);
  ObSArray<ObIndexArg *> &index_arg_list = alter_table_arg.index_arg_list_;
  common::ObArray<const ObForeignKeyInfo *> drop_parent_table_mock_foreign_key_infos_array;
  ObIArray<obrpc::ObDDLRes> &ddl_res_array = res.ddl_res_array_;
  new_fetched_snapshot = 0;
  // To many hashset will fill up the stack, construct them on heap instead
  HEAP_VAR(AddIndexNameHashSet, add_index_name_set) {
  HEAP_VAR(DropIndexNameHashSet, drop_index_name_set) {
  HEAP_VAR(RenameIndexNameHashSet, rename_ori_index_name_set) {
  HEAP_VAR(RenameIndexNameHashSet, rename_new_index_name_set) {
  HEAP_VAR(AlterIndexNameHashSet, alter_index_name_set) {
    for (int64_t i = 0; OB_SUCC(ret) && i < index_arg_list.size(); ++i) {
      ObIndexArg *index_arg = index_arg_list.at(i);
      int64_t index_count = new_table_schema.get_index_count();
      int64_t index_aux_count = new_table_schema.get_index_tid_count();
      if (OB_ISNULL(index_arg)) {
        ret = OB_INVALID_ARGUMENT;
        LOG_WARN("index arg should not be null", K(ret));
      } else {
        if (index_arg->index_action_type_ == ObIndexArg::ADD_INDEX) {
          ObCreateIndexArg *create_index_arg = static_cast<ObCreateIndexArg *>(index_arg);
          uint64_t tenant_data_version = 0;
          const bool is_check_fts_index_conflict = !create_index_arg->is_inner_ && share::schema::is_fts_or_multivalue_index(create_index_arg->index_type_);
          if (is_check_fts_index_conflict && OB_FAIL(check_fts_index_conflict(origin_table_schema.get_tenant_id(), origin_table_schema.get_table_id()))) {
            if (OB_EAGAIN != ret) {
              LOG_WARN("failed to check fts index ", K(ret));
            }
          } else if (OB_FAIL(GET_MIN_DATA_VERSION(create_index_arg->tenant_id_, tenant_data_version))) {
            LOG_WARN("get min data version failed", K(ret), KPC(create_index_arg));
          } else if (OB_MAX_AUX_TABLE_PER_MAIN_TABLE <= index_aux_count || OB_MAX_INDEX_PER_TABLE <= index_count) {
            ret = OB_ERR_TOO_MANY_KEYS;
            LOG_USER_ERROR(OB_ERR_TOO_MANY_KEYS, OB_MAX_INDEX_PER_TABLE);
            LOG_WARN("too many index or index aux for table!",
                     K(index_count), K(OB_MAX_INDEX_PER_TABLE), K(index_aux_count), K(OB_MAX_AUX_TABLE_PER_MAIN_TABLE));
          }
          if (!new_table_schema.is_partitioned_table()
              && !new_table_schema.is_auto_partitioned_table()
              && !create_index_arg->index_schema_.is_partitioned_table()) {
            if (INDEX_TYPE_NORMAL_GLOBAL == create_index_arg->index_type_) {
              create_index_arg->index_type_ = INDEX_TYPE_NORMAL_GLOBAL_LOCAL_STORAGE;
            } else if (INDEX_TYPE_UNIQUE_GLOBAL == create_index_arg->index_type_) {
              create_index_arg->index_type_ = INDEX_TYPE_UNIQUE_GLOBAL_LOCAL_STORAGE;
            } else if (INDEX_TYPE_SPATIAL_GLOBAL == create_index_arg->index_type_) {
              create_index_arg->index_type_ = INDEX_TYPE_SPATIAL_GLOBAL_LOCAL_STORAGE;
            }
          }
          if (OB_SUCC(ret)) {
            if (create_index_arg->index_type_ == INDEX_TYPE_PRIMARY) {
              if (new_table_schema.get_rowkey_column_num() > 0) {
                if (new_table_schema.is_table_without_pk()) {
                  ret = OB_NOT_SUPPORTED;
                  LOG_WARN("not support to add primary key!", K(ret));
                } else {
                  ret = OB_ERR_MULTIPLE_PRI_KEY;
                  LOG_WARN("multiple primary key defined", K(ret));
                }
              }
              continue;
            }
          }
          if (OB_SUCC(ret) && new_table_schema.mv_major_refresh()) {
            ret = OB_NOT_SUPPORTED;
            LOG_WARN("not support to add index on mv", K(ret));
            LOG_USER_ERROR(OB_NOT_SUPPORTED, "add index on major refresh materialized view is");
          }
          if (OB_FAIL(ret)) {
          } else if (create_index_arg->index_name_.empty()) {
            if (OB_FAIL(generate_index_name(*create_index_arg,
                                            new_table_schema,
                                            add_index_name_set,
                                            drop_index_name_set,
                                            schema_guard,
                                            allocator))) {
              LOG_WARN("generate index name failed", K(ret));
            }
          } else {
            ObString index_name = create_index_arg->index_name_;
            ObString index_table_name;
            bool is_exist = false;
            if (OB_FAIL(check_index_table_exist(origin_table_schema.get_tenant_id(),
                                                origin_table_schema.get_database_id(),
                                                origin_table_schema.get_table_id(),
                                                index_name,
                                                schema_guard,
                                                is_exist))) {
              LOG_WARN("failed to check index table", K(ret));
            } else {
              ObIndexNameHashWrapper index_key(index_name);
              if (!is_exist) {
                if (OB_HASH_EXIST == add_index_name_set.exist_refactored(index_key)) {
                  //alter table t1 add index c1, add index c1
                  is_exist = true;
                } else if (OB_HASH_EXIST == rename_new_index_name_set.exist_refactored(index_key)) {
                  //alter table t1 rename index ori_idx to new_idx, add index new_idx // MySQL supports this behavior
                  is_exist = true;
                }
              } else {
                if (OB_HASH_EXIST == drop_index_name_set.exist_refactored(index_key)) {
                  //alter table t1 drop index c1, add index c1
                  is_exist = false;
                } else if (OB_HASH_EXIST == rename_ori_index_name_set.exist_refactored(index_key)) {
                  //alter table t1 rename index ori_idx to new_idx, add index ori_idx // MySQL supports this behavior
                  is_exist = false;
                }
              }
              if (is_exist) {
                ret = OB_ERR_KEY_NAME_DUPLICATE;
                LOG_USER_ERROR(OB_ERR_KEY_NAME_DUPLICATE, index_name.length(), index_name.ptr());
                LOG_WARN("duplicate index name", K(index_name), K(ret));
              }
            }
          }
          if (OB_SUCC(ret)) {
            HEAP_VARS_3((ObTableSchema, index_schema),
                        (obrpc::ObCreateIndexArg, my_arg),
                        (obrpc::ObCreateIndexArg, tmp_arg)){
              ObArray<ObColumnSchemaV2*> gen_columns;
              bool global_storage = INDEX_TYPE_NORMAL_GLOBAL == create_index_arg->index_type_
                                    || INDEX_TYPE_UNIQUE_GLOBAL == create_index_arg->index_type_
                                    || INDEX_TYPE_SPATIAL_GLOBAL == create_index_arg->index_type_;
              if (global_storage) {
                //FIXME: Cannot build index on tablegroup at present,
                //       For a global index, its locality and primary take the value of the primary table
                //       The current implementation has the following two problems:
                //       1. If the locality/primary_zone of the primary table is inherited semantics,
                //       and the primary table is in the tablegroup, then the locality/primary_zone of
                //       the inheritance relationship between the global index table and the primary table
                //       is different, and the inherited values may be different;
                //       2. In addition, for the case where the primary_zone is random,
                //       currently the leader_coordinator cannot guarantee that the results of
                //       the random breakup of the primary table and the global index table primary_zone are consistent.
                if (OB_FAIL(ret)) {
                } else if (OB_FAIL(index_schema.assign(create_index_arg->index_schema_))) {
                  LOG_WARN("fail to assign schema", K(ret));
                } else if (FALSE_IT(index_schema.set_tenant_id(origin_table_schema.get_tenant_id()))) {
                }
              }
              bool global_index_without_column_info = (create_index_arg->index_schema_.is_partitioned_table() ||
                                                       create_index_arg->index_schema_.is_auto_partitioned_table()) ?
                                                       false : true;
              bool rowkey_doc_exist = false;
              bool is_generate_rowkey_doc = false;
              if (OB_FAIL(ret)) {
              } else if (OB_FAIL(my_arg.assign(*create_index_arg))) {
                LOG_WARN("fail to assign arg", K(ret));
              } else if (share::schema::is_fts_or_multivalue_index(my_arg.index_type_)) {
                const ObTableSchema *rowkey_doc_schema = nullptr;
                if (OB_FAIL(tmp_arg.assign(my_arg))) {
                  LOG_WARN("fail to assign arg", K(ret));
                } else if (!tmp_arg.is_valid()) {
                  ret = OB_ERR_UNEXPECTED;
                  LOG_WARN("fail to copy create index arg", K(ret));
                } else if (FALSE_IT(tmp_arg.index_type_ = INDEX_TYPE_ROWKEY_DOC_ID_LOCAL)) {
                } else if (OB_FAIL(ObFtsIndexBuilderUtil::generate_fts_aux_index_name(tmp_arg, &allocator))) {
                  LOG_WARN("failed to adjust fts index name", K(ret));
                } else if (OB_FAIL(check_aux_index_schema_exist(origin_table_schema.get_tenant_id(),
                                                                tmp_arg,
                                                                schema_guard,
                                                                &new_table_schema,
                                                                rowkey_doc_exist,
                                                                rowkey_doc_schema))) {
                  LOG_WARN("fail to check rowkey doc schema existence", K(ret));
                }
              }
              if (OB_FAIL(ret)) {
              } else if (share::schema::is_fts_or_multivalue_index(my_arg.index_type_) &&
                         !rowkey_doc_exist &&
                         (FALSE_IT(my_arg.index_type_ = INDEX_TYPE_ROWKEY_DOC_ID_LOCAL) ||
                          FALSE_IT(is_generate_rowkey_doc = true))) {
                // 1. generate rowkey doc schema if not exist
                // 2. otherwise generate fts index aux schema
              } else if (share::schema::is_fts_index(my_arg.index_type_) &&
                  OB_FAIL(ObFtsIndexBuilderUtil::generate_fts_aux_index_name(my_arg, &allocator))) {
                LOG_WARN("failed to adjust fts index name", K(ret));
              } else if (OB_FAIL(ObIndexBuilderUtil::adjust_expr_index_args(
                      my_arg, new_table_schema, allocator, gen_columns))) {
                LOG_WARN("adjust fulltext args failed", K(ret));
              } else if (is_generate_rowkey_doc &&
                         OB_FAIL(ObDDLLock::lock_table_in_trans(new_table_schema, transaction::tablelock::EXCLUSIVE, trans))) {
                LOG_WARN("fail to lock for offline ddl", K(ret), K(new_table_schema));
              } else if (OB_FAIL(index_builder.generate_schema(my_arg,
                                                        new_table_schema,
                                                        global_index_without_column_info,
                                                        true, /*generate_id*/
                                                        index_schema))) {
                LOG_WARN("failed to generate index schema!", K(ret));
              } else if (OB_FAIL(ddl_operator.alter_table_create_index(new_table_schema,
                                                                      gen_columns,
                                                                      index_schema,
                                                                      trans))) {
                LOG_WARN("failed to alter table add index!", K(index_schema), K(ret));
              } else {
                // The index data is stored separately from the main table,
                // the partition needs to be built, and insert ori_schema_version in the outer insert
                if (index_schema.has_tablet()
                    && OB_FAIL(create_index_tablet(index_schema, trans, schema_guard, true/*need_check_tablet_cnt*/, tenant_data_version))) {
                  LOG_WARN("fail to create_index_tablet", KR(ret), K(index_schema));
                }
                if (OB_SUCC(ret) && is_generate_rowkey_doc) {
                  if (new_fetched_snapshot <= 0 &&
                      OB_FAIL(ObDDLUtil::obtain_snapshot(trans, new_table_schema, index_schema, new_fetched_snapshot))) {
                    LOG_WARN("fail to obtain snapshot",
                        K(ret), K(new_table_schema), K(index_schema), K(new_fetched_snapshot));
                  }
                }
                if (OB_SUCC(ret)) {
                  ObIndexNameHashWrapper index_key(create_index_arg->index_name_);
                  if (OB_FAIL(create_index_arg->index_schema_.assign(index_schema))) {
                    LOG_WARN("fail to assign schema", K(ret));
                  } else if (OB_FAIL(add_index_name_set.set_refactored(index_key))) {
                    LOG_WARN("set index name to hash set failed",
                          K(create_index_arg->index_name_), K(ret));
                  }
                }
              }
            }
          }
        } else if (ObIndexArg::DROP_INDEX == index_arg->index_action_type_) {
          ObDropIndexArg *drop_index_arg = static_cast<ObDropIndexArg *>(index_arg);
          drop_index_arg->tenant_id_ = origin_table_schema.get_tenant_id();
          const ObString &index_name = drop_index_arg->index_name_;
          ObIndexNameHashWrapper index_key(index_name);
          if (OB_HASH_EXIST == drop_index_name_set.exist_refactored(index_key)) {
            //already drop in the same alter table clause
            ret = OB_ERR_KEY_COLUMN_DOES_NOT_EXITS;
            LOG_USER_ERROR(OB_ERR_KEY_COLUMN_DOES_NOT_EXITS, index_name.length(), index_name.ptr());
            LOG_WARN("duplicate index name", K(index_name), K(ret));
          } else if (OB_FAIL(rename_ori_index_name_set.set_refactored(index_key))) {
            // alter table rename ori_idx to new_idx, drop ori_idx
            ret = OB_ERR_KEY_DOES_NOT_EXISTS;
            const ObString &data_table_name = origin_table_schema.get_table_name_str();
            LOG_USER_ERROR(OB_ERR_KEY_DOES_NOT_EXISTS, index_name.length(), index_name.ptr(), data_table_name.length(),
                          data_table_name.ptr());
            LOG_WARN("index does not exist in table", K(index_name), K(data_table_name), K(ret));
          } else if (OB_FAIL(drop_index_name_set.set_refactored(index_key))) {
            LOG_WARN("set index name to hash set failed", K(ret), K(index_name));
          }
          if (OB_SUCC(ret)) {
            const ObTableSchema *index_table_schema = nullptr;
            bool have_index = false;
            const common::ObIArray<ObForeignKeyInfo> &foreign_key_infos = origin_table_schema.get_foreign_key_infos();
            if (OB_FAIL(get_index_schema_by_name(
                origin_table_schema.get_table_id(),
                origin_table_schema.get_database_id(),
                *drop_index_arg,
                schema_guard,
                index_table_schema))) {
              LOG_WARN("get index schema by name failed", K(ret));
            } else if (OB_ISNULL(index_table_schema)) {
              ret = OB_ERR_CANT_DROP_FIELD_OR_KEY;
              LOG_WARN("index table schema should not be null", K(*drop_index_arg), K(ret));
              LOG_USER_ERROR(OB_ERR_CANT_DROP_FIELD_OR_KEY, drop_index_arg->index_name_.length(), drop_index_arg->index_name_.ptr());
            } else if (index_table_schema->is_in_recyclebin()) {
              ret = OB_ERR_OPERATION_ON_RECYCLE_OBJECT;
              LOG_WARN("index table is in recyclebin", KR(ret));
            } else if (OB_FAIL(check_index_on_foreign_key(index_table_schema,
                                                          foreign_key_infos,
                                                          have_index))) {
              LOG_WARN("fail to check index on foreign key", K(ret), K(foreign_key_infos), KPC(index_table_schema));
            } else if (have_index) {
              if (index_table_schema->is_unique_index()) {
                ret = OB_ERR_ATLER_TABLE_ILLEGAL_FK;
              } else {
                ObString index_name;
                if (OB_FAIL(ObTableSchema::get_index_name(allocator,
                  index_table_schema->get_data_table_id(),
                  index_table_schema->get_table_name_str(),
                  index_name))) {
                  LOG_WARN("failed to build index table name", K(ret));
                } else {
                  ret = OB_ERR_ATLER_TABLE_ILLEGAL_FK_DROP_INDEX;
                  LOG_USER_ERROR(OB_ERR_ATLER_TABLE_ILLEGAL_FK_DROP_INDEX, index_name.length(), index_name.ptr());
                }
              }
              LOG_WARN("cannot delete index with foreign key dependency", K(ret));
            } else if (!drop_index_arg->is_inner_ && index_table_schema->is_unavailable_index()) {
              ret = OB_NOT_SUPPORTED;
              LOG_WARN("not support to drop a building index", K(ret), K(drop_index_arg->is_inner_), KPC(index_table_schema));
              LOG_USER_ERROR(OB_NOT_SUPPORTED, "dropping a building index is");
            } else if (drop_index_arg->is_add_to_scheduler_) {
              if (OB_FAIL(drop_index_to_scheduler_(trans, schema_guard, alter_table_arg.allocator_, origin_table_schema,
                                                   nullptr /*inc_tablet_ids*/, nullptr /*del_tablet_ids*/, drop_index_arg,
                                                   ddl_operator, res, ddl_tasks))) {
                LOG_WARN("fail to drop index to scheduler", KR(ret), K(drop_index_arg));
              }
            } else {
              if (OB_FAIL(ddl_operator.alter_table_drop_index(
                  index_table_schema,
                  new_table_schema,
                  trans))) {
                LOG_WARN("failed to alter table drop index", K(*drop_index_arg), K(ret));
              }
            }
          }
        } else if (index_arg->index_action_type_ == ObIndexArg::ALTER_INDEX) {
          ObAlterIndexArg *alter_index_arg = static_cast<ObAlterIndexArg *>(index_arg);
          alter_index_arg->tenant_id_ = origin_table_schema.get_tenant_id();
          const ObString &index_name = alter_index_arg->index_name_;
          ObIndexNameHashWrapper index_key(index_name);
          bool is_exist = false;
          if (OB_FAIL(check_index_table_exist(origin_table_schema.get_tenant_id(),
                                              origin_table_schema.get_database_id(),
                                              origin_table_schema.get_table_id(),
                                              index_name,
                                              schema_guard,
                                              is_exist))) {
            LOG_WARN("failed to check index table", K(ret));
          } else {
            if (is_exist) {
              if (OB_HASH_EXIST == drop_index_name_set.exist_refactored(index_key)) {
                //alter table t1 drop index c1, alter index c1
                is_exist = false;
              } else if ((OB_HASH_EXIST == rename_ori_index_name_set.exist_refactored(index_key))
                  || (OB_HASH_EXIST == rename_new_index_name_set.exist_refactored(index_key))) {
                //FIXME(juxanxue): we don't support alter visible and rename index operations in one statement
                //alter table t1 rename ori_idx to new_idx, alter ori_idx
                //alter table t1 rename ori_idx to new_idx, alter new_idx
                //In current cases, alter index will override the rename index operation
                is_exist = false;
              }
            }
            if (!is_exist) {
              ret = OB_ERR_KEY_DOES_NOT_EXISTS;
              const ObString &data_table_name = origin_table_schema.get_table_name_str();
              LOG_USER_ERROR(OB_ERR_KEY_DOES_NOT_EXISTS, index_name.length(), index_name.ptr(), data_table_name.length(),
                            data_table_name.ptr());
              LOG_WARN("index does not exist in table", K(index_name), K(data_table_name), K(ret));
            }
          }
          if (OB_SUCC(ret)) {
            if (OB_FAIL(ddl_operator.alter_table_alter_index(
                        origin_table_schema.get_tenant_id(),
                        origin_table_schema.get_table_id(),
                        origin_table_schema.get_database_id(),
                        *alter_index_arg,
                        trans))) {
              LOG_WARN("failed to alter table alter index!", K(alter_index_arg), K(ret));
            } else if (OB_HASH_EXIST == alter_index_name_set.exist_refactored(index_key)) {
              ret = OB_ERR_ALTER_TABLE_ALTER_DUPLICATED_INDEX;
              LOG_USER_ERROR(OB_ERR_ALTER_TABLE_ALTER_DUPLICATED_INDEX, index_name.length(), index_name.ptr());
              LOG_WARN("Duplicate alter index operations", K(ret), K(index_name));
            } else if (OB_FAIL(alter_index_name_set.set_refactored(index_key))) {
              LOG_WARN("set index name to hash set failed", K(ret), K(index_name));
            }
          }
        } else if (ObIndexArg::DROP_FOREIGN_KEY == index_arg->index_action_type_) {
          ObDropForeignKeyArg *drop_foreign_key_arg = static_cast<ObDropForeignKeyArg *>(index_arg);
          const ObForeignKeyInfo *parent_table_mock_foreign_key_info = NULL;
          if (OB_FAIL(ddl_operator.alter_table_drop_foreign_key(origin_table_schema, *drop_foreign_key_arg, trans, parent_table_mock_foreign_key_info,
              origin_table_schema.get_in_offline_ddl_white_list()))) {
            LOG_WARN("failed to alter table drop foreign key", K(ret), K(drop_foreign_key_arg));
          } else if (NULL != parent_table_mock_foreign_key_info) {
            if (OB_FAIL(drop_parent_table_mock_foreign_key_infos_array.push_back(parent_table_mock_foreign_key_info))) {
              LOG_WARN("failed to push back to drop_foreign_key_infos_array", K(ret), KPC(parent_table_mock_foreign_key_info));
            }
          }
        } else if (ObIndexArg::ALTER_INDEX_PARALLEL == index_arg->index_action_type_) {
          ObAlterIndexParallelArg *alter_index_parallel_arg =
            static_cast<ObAlterIndexParallelArg *>(index_arg);
          bool is_exist = false;
          if (OB_FAIL(check_index_table_exist(origin_table_schema.get_tenant_id(),
                                      origin_table_schema.get_database_id(),
                                      origin_table_schema.get_table_id(),
                                      alter_index_parallel_arg->index_name_,
                                      schema_guard,
                                      is_exist))) {

          } else if (!is_exist) {
            ret = OB_ERR_KEY_DOES_NOT_EXISTS;
            const ObString &index_name = alter_index_parallel_arg->index_name_;
            const ObString &data_table_name = origin_table_schema.get_table_name_str();
            LOG_WARN("the index is not exist", K(ret), K(alter_index_parallel_arg));
            LOG_USER_ERROR(OB_ERR_KEY_DOES_NOT_EXISTS, index_name.length(), index_name.ptr(),
                           data_table_name.length(), data_table_name.ptr());
          } else if (OB_FAIL(ddl_operator.alter_index_table_parallel(
                     origin_table_schema.get_tenant_id(),
                     origin_table_schema.get_table_id(),
                     origin_table_schema.get_database_id(),
                     *alter_index_parallel_arg,
                     trans))) {
            LOG_WARN("failed to alter index table parallel",
              K(ret), K(origin_table_schema.get_table_name()));
          }
        } else if (ObIndexArg::RENAME_INDEX == index_arg->index_action_type_) {
          ObRenameIndexArg *rename_index_arg = static_cast<ObRenameIndexArg *>(index_arg);
          const ObString &ori_index_name = rename_index_arg->origin_index_name_;
          const ObString &new_index_name = rename_index_arg->new_index_name_;
          bool is_exist_ori = false;
          bool is_exist_new = false;
          ObIndexNameHashWrapper ori_index_key(ori_index_name);
          ObIndexNameHashWrapper new_index_key(new_index_name);
          const ObTableSchema *orig_index_schema = nullptr;

          // 1. check conflicts with origin index name
          // 2. check confilicts with new index name
          // 3. alter table rename index
          if (OB_FAIL(check_index_table_exist(origin_table_schema.get_tenant_id(),
                                              origin_table_schema.get_database_id(),
                                              origin_table_schema.get_table_id(),
                                              ori_index_name,
                                              schema_guard,
                                              is_exist_ori))) {
            LOG_WARN("failed to check index table", K(ret));
          } else if (OB_FAIL(ret) || !is_exist_ori) {
            LOG_WARN("failed to check origin index name exist!", K(ori_index_name));
            const ObString &data_table_name = origin_table_schema.get_table_name_str();
            ret = OB_ERR_KEY_DOES_NOT_EXISTS;
            LOG_USER_ERROR(OB_ERR_KEY_DOES_NOT_EXISTS, ori_index_name.length(), ori_index_name.ptr(), data_table_name.length(), data_table_name.ptr());
          } else if (OB_FAIL(ret) || (is_exist_ori && (OB_HASH_EXIST == add_index_name_set.exist_refactored(ori_index_key)))) {
            // add ori_idx, rename ori_idx to new_idx // ERROR 1176 (42000): Key 'ori_idx' doesn't exist in table
            LOG_WARN("failed to rename index that isn't added yet!", K(ori_index_name));
            const ObString &data_table_name = origin_table_schema.get_table_name_str();
            ret = OB_ERR_KEY_DOES_NOT_EXISTS;
            LOG_USER_ERROR(OB_ERR_KEY_DOES_NOT_EXISTS, ori_index_name.length(), ori_index_name.ptr(), data_table_name.length(), data_table_name.ptr());
          } else if (OB_FAIL(ret) || (OB_HASH_EXIST == alter_index_name_set.exist_refactored(ori_index_key))) {
            // alter ori_idx, rename ori_idx to new_idx
            //FIXME(juxanxue): we don't support alter visible and rename index operations in one statement
            //In current cases, alter index will be overrided by the rename index operation
            LOG_WARN("failed to rename index that is altering index visible attribute!", K(ori_index_name));
            ret = OB_ERR_WRONG_OBJECT;
            LOG_USER_ERROR(OB_ERR_WRONG_OBJECT, to_cstring(ori_index_name), "visible attribute", "updated");
          } else if (OB_FAIL(check_index_table_exist(origin_table_schema.get_tenant_id(),
                                              origin_table_schema.get_database_id(),
                                              origin_table_schema.get_table_id(),
                                              new_index_name,
                                              schema_guard,
                                              is_exist_new))) {
            LOG_WARN("failed to check index table", K(ret));
          } else if (is_exist_new && !(OB_HASH_EXIST == drop_index_name_set.exist_refactored(new_index_key))) {
            // Due to MySQL behavior, attemp to support scenario: drop idx_new, rename idx_ori to idx_new
            bool is_oracle_mode = false;
            if (OB_FAIL(origin_table_schema.check_if_oracle_compat_mode(is_oracle_mode))) {
              LOG_WARN("fail to check if tenant mode is oracle mode", K(ret));
            } else if (is_oracle_mode) {
              ret = OB_ERR_EXIST_OBJECT;
              LOG_WARN("duplicate index name", K(ret), K(new_index_name));
            } else {
              ret = OB_ERR_KEY_NAME_DUPLICATE;
              LOG_USER_ERROR(OB_ERR_KEY_NAME_DUPLICATE, new_index_name.length(), new_index_name.ptr());
              LOG_WARN("duplicate index name", K(ret), K(new_index_name));
            }
          } else if (OB_HASH_EXIST == add_index_name_set.exist_refactored(new_index_key)) {
            // add new_idx, rename ori_idx to new_idx // ERROR 1061 (42000): Duplicate key name 'new_idx'
            ret = OB_ERR_KEY_NAME_DUPLICATE;
            LOG_WARN("duplicate index name", K(new_index_name), K(ret));
            LOG_USER_ERROR(OB_ERR_KEY_NAME_DUPLICATE, new_index_name.length(), new_index_name.ptr());
          } else if (OB_HASH_EXIST == drop_index_name_set.exist_refactored(ori_index_key)) {
              // drop ori_idx, rename ori_idx to new_idx
              const ObString &data_table_name = origin_table_schema.get_table_name_str();
              ret = OB_ERR_KEY_DOES_NOT_EXISTS;
              LOG_USER_ERROR(OB_ERR_KEY_DOES_NOT_EXISTS, ori_index_name.length(), ori_index_name.ptr(), data_table_name.length(), data_table_name.ptr());
          } else {
            SMART_VAR(ObTableSchema, new_index_schema) {
              if (OB_FAIL(ddl_operator.alter_table_rename_index(
                          origin_table_schema.get_tenant_id(),
                          origin_table_schema.get_table_id(),
                          origin_table_schema.get_database_id(),
                          *rename_index_arg,
                          nullptr /* new_index_status */,
                          false, /* is_in_deleting */
                          trans,
                          new_index_schema))) {
                LOG_WARN("failed to rename index", K(*rename_index_arg), K(ret));
              }
            }
          }

          if (OB_FAIL(ret)) {
          } else if (OB_FAIL(rename_ori_index_name_set.set_refactored(ori_index_key))) {
            const ObString &data_table_name = origin_table_schema.get_table_name_str();
            //To be compatible with Mysql
            ret = OB_ERR_KEY_DOES_NOT_EXISTS;
            LOG_USER_ERROR(OB_ERR_KEY_DOES_NOT_EXISTS, ori_index_name.length(), ori_index_name.ptr(), data_table_name.length(), data_table_name.ptr());
            LOG_WARN("set index name to hash set failed", K(ret), K(ori_index_name));
          } else if (OB_FAIL(rename_new_index_name_set.set_refactored(new_index_key))) {
            LOG_WARN("set index name to hash set failed", K(ret), K(new_index_name));
          }
        } else if (ObIndexArg::ALTER_INDEX_TABLESPACE == index_arg->index_action_type_) {
          ObAlterIndexTablespaceArg *alter_index_tablespace_arg =
            static_cast<ObAlterIndexTablespaceArg *>(index_arg);
          bool is_exist = false;
          if (OB_FAIL(check_index_table_exist(origin_table_schema.get_tenant_id(),
                                              origin_table_schema.get_database_id(),
                                              origin_table_schema.get_table_id(),
                                              alter_index_tablespace_arg->index_name_,
                                              schema_guard,
                                              is_exist))) {
            LOG_WARN("failed to check index table", K(ret));
          } else if (!is_exist) {
            ret = OB_ERR_KEY_DOES_NOT_EXISTS;
            const ObString &index_name = alter_index_tablespace_arg->index_name_;
            const ObString &data_table_name = origin_table_schema.get_table_name_str();
            LOG_WARN("the index is not exist", K(index_name), K(ret));
            LOG_USER_ERROR(OB_ERR_KEY_DOES_NOT_EXISTS, index_name.length(), index_name.ptr(),
                           data_table_name.length(), data_table_name.ptr());
          } else if (OB_FAIL(ddl_operator.alter_index_table_tablespace(origin_table_schema.get_table_id(),
                                                            origin_table_schema.get_database_id(),
                                                            *alter_index_tablespace_arg,
                                                            schema_guard,
                                                            trans))) {
            LOG_WARN("failed to alter index table tablespace",
              K(ret), K(origin_table_schema.get_table_name()));
          }
        } else {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("Unknown index action type!", K_(index_arg->index_action_type), K(ret));
        }
      }
    }
    if (OB_SUCC(ret) && !drop_parent_table_mock_foreign_key_infos_array.empty()) {
      ObArray<ObMockFKParentTableSchema> mock_fk_parent_table_schema_array;
      ObArray<const ObMockFKParentTableSchema*> mock_fk_parent_table_schema_ptr_array;
      ObSchemaService *schema_service = schema_service_->get_schema_service();
      if (OB_ISNULL(schema_service)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("schema service is null", K(ret));
      } else if (OB_FAIL(prepare_gen_mock_fk_parent_tables_for_drop_fks(
          schema_guard, origin_table_schema.get_tenant_id(),
          drop_parent_table_mock_foreign_key_infos_array,
          mock_fk_parent_table_schema_ptr_array,
          mock_fk_parent_table_schema_array))) {
        LOG_WARN("failed to prepare_gen_mock_fk_parent_tables_for_drop_fks", K(ret));
      } else if (OB_FAIL(gen_mock_fk_parent_tables_for_drop_fks(
                 schema_guard, mock_fk_parent_table_schema_ptr_array, mock_fk_parent_table_schema_array))) {
        LOG_WARN("failed to gen_mock_fk_parent_tables_for_drop_fks", K(ret));
      } else if (OB_FAIL(ddl_operator.deal_with_mock_fk_parent_tables(trans, schema_guard, mock_fk_parent_table_schema_array))) {
        LOG_WARN("fail to deal_with_mock_fk_parent_tables", K(ret), K(mock_fk_parent_table_schema_array));
      }
    }
  } // end of alter_index_name_set
  } // end of rename_new_index_name_set
  } // end of rename_ori_index_name_set
  } // end of drop_index_name_set
  } // end of add_index_name_set
  return ret;
}

// MockFKParentTable begin

int ObDDLService::gen_mock_fk_parent_table_for_create_fk(
    share::schema::ObSchemaGetterGuard &schema_guard,
    const uint64_t tenant_id,
    const ObCreateForeignKeyArg &foreign_key_arg,
    const ObMockFKParentTableSchema *tmp_mock_fk_parent_table_ptr,
    ObForeignKeyInfo &foreign_key_info,
    ObMockFKParentTableSchema &mock_fk_parent_table_schema)
{
  int ret = OB_SUCCESS;
  if (!foreign_key_arg.is_parent_table_mock_) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("foreign_key_arg is not parent_table_mock", K(ret));
  } else {
    mock_fk_parent_table_schema.reset();
    const ObMockFKParentTableSchema *mock_fk_parent_table_ptr = NULL;
    const ObDatabaseSchema *db_schema = NULL;
    if (OB_FAIL(schema_guard.get_database_schema(tenant_id,
                foreign_key_arg.parent_database_, db_schema))) {
      LOG_WARN("failed to get parent database schema", K(ret), K(foreign_key_arg));
    } else if (NULL == db_schema) {
      ret = OB_ERR_BAD_DATABASE;
      LOG_USER_ERROR(OB_ERR_BAD_DATABASE, foreign_key_arg.parent_database_.length(), foreign_key_arg.parent_database_.ptr());
    } else if (OB_INVALID_ID == db_schema->get_database_id()) {
      ret = OB_ERR_BAD_DATABASE;
      LOG_WARN("database id is invalid", K(ret), K(tenant_id), K(db_schema->get_database_id()));
    } else if (db_schema->is_in_recyclebin()) {
      ret = OB_ERR_OPERATION_ON_RECYCLE_OBJECT;
      LOG_WARN("Can't not create outline of db in recyclebin", K(ret), K(*db_schema));
    } else if (OB_FAIL(schema_guard.get_mock_fk_parent_table_schema_with_name(
               tenant_id,
               db_schema->get_database_id(),
               foreign_key_arg.parent_table_,
               mock_fk_parent_table_ptr))) {
      LOG_WARN("get_mock_fk_parent_table_schema_with_name failed", K(ret), K(tenant_id), K(db_schema->get_database_id()), K(foreign_key_arg.parent_table_));
    } else if (NULL != tmp_mock_fk_parent_table_ptr && NULL != mock_fk_parent_table_ptr) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("mock_fk_parent_table_ptr and tmp_mock_fk_parent_table_ptr are both not null", K(ret), KPC(mock_fk_parent_table_ptr), KPC(tmp_mock_fk_parent_table_ptr));
    } else if (NULL != tmp_mock_fk_parent_table_ptr) {
      mock_fk_parent_table_ptr = tmp_mock_fk_parent_table_ptr;
    }
    if (OB_FAIL(ret)) {
    } else if (OB_ISNULL(mock_fk_parent_table_ptr)) { // mock_fk_parent_table is not exist, need to add a new one
      uint64_t mock_fk_parent_table_id = OB_INVALID_ID;
      ObSchemaService *schema_service = schema_service_->get_schema_service();
      if (OB_ISNULL(schema_service)) {
        ret = OB_ERR_SYS;
        LOG_ERROR("schema_service must not null", K(ret));
      } else if (OB_FAIL(schema_service->fetch_new_table_id(tenant_id, mock_fk_parent_table_id))) {
        LOG_WARN("fail to fetch_new_table_id", K(ret));
      } else {
        mock_fk_parent_table_schema.set_operation_type(ObMockFKParentTableOperationType::MOCK_FK_PARENT_TABLE_OP_CREATE_TABLE_BY_ADD_FK_IN_CHILD_TBALE);
        mock_fk_parent_table_schema.set_tenant_id(tenant_id);
        mock_fk_parent_table_schema.set_database_id(db_schema->get_database_id());
        mock_fk_parent_table_schema.set_mock_fk_parent_table_id(mock_fk_parent_table_id);
        mock_fk_parent_table_schema.set_mock_fk_parent_table_name(foreign_key_arg.parent_table_);
        uint64_t column_id = OB_INVALID_ID;
        uint64_t max_used_column_id = 0;
        bool is_column_exist = false;
        for (int64_t j = 0; OB_SUCC(ret) && j < foreign_key_arg.parent_columns_.count(); ++j) {
          mock_fk_parent_table_schema.get_column_id_by_column_name(foreign_key_arg.parent_columns_.at(j), column_id, is_column_exist);
          if (is_column_exist) {
            ret = OB_ERR_COLUMN_DUPLICATE;
            LOG_USER_ERROR(OB_ERR_COLUMN_DUPLICATE, foreign_key_arg.parent_columns_.at(j).length(), foreign_key_arg.parent_columns_.at(j).ptr());
          } else if (OB_FAIL(mock_fk_parent_table_schema.add_column_info_to_column_array(std::make_pair(++max_used_column_id, foreign_key_arg.parent_columns_.at(j))))) {
            LOG_WARN("fail to add_column_info_to_column_array for mock_fk_parent_table_schema", K(max_used_column_id), K(foreign_key_arg.parent_columns_.at(j)));
          } else if (OB_FAIL(foreign_key_info.parent_column_ids_.push_back(max_used_column_id))) {
            LOG_WARN("failed to push parent column id", K(ret), K(max_used_column_id));
          }
        }
      }
    } else { // mock_fk_parent_table is already exist, need to add new columns or update schema version
      bool is_alter_table = (NULL == tmp_mock_fk_parent_table_ptr);
      mock_fk_parent_table_schema.assign(*mock_fk_parent_table_ptr);
      if (is_alter_table) {
        mock_fk_parent_table_schema.reset_column_array();
      }
      uint64_t max_used_column_id = mock_fk_parent_table_ptr->get_column_array().at(mock_fk_parent_table_ptr->get_column_array().count() - 1).first;
      bool is_column_exist = false;
      uint64_t column_id = OB_INVALID_ID;
      for (int64_t j = 0; OB_SUCC(ret) && j < foreign_key_arg.parent_columns_.count(); ++j) {
        mock_fk_parent_table_ptr->get_column_id_by_column_name(foreign_key_arg.parent_columns_.at(j), column_id, is_column_exist);
        if (is_column_exist) {
          if (OB_FAIL(foreign_key_info.parent_column_ids_.push_back(column_id))) {
            LOG_WARN("failed to push parent column id", K(ret), K(column_id));
          }
        } else if (!is_column_exist) {
          if (OB_FAIL(mock_fk_parent_table_schema.add_column_info_to_column_array(std::make_pair(++max_used_column_id, foreign_key_arg.parent_columns_.at(j))))) {
            LOG_WARN("fail to add_column_info_to_column_array for mock_fk_parent_table_schema", K(max_used_column_id), K(foreign_key_arg.parent_columns_.at(j)));
          } else if (OB_FAIL(foreign_key_info.parent_column_ids_.push_back(max_used_column_id))) {
            LOG_WARN("failed to push parent column id", K(ret), K(max_used_column_id));
          }
        }
      }
      if (OB_SUCC(ret)) {
        if (is_alter_table) { // alter table add fk
          if (mock_fk_parent_table_schema.get_column_array().empty()) {
            mock_fk_parent_table_schema.set_operation_type(ObMockFKParentTableOperationType::MOCK_FK_PARENT_TABLE_OP_UPDATE_SCHEMA_VERSION);
          } else {
            mock_fk_parent_table_schema.set_operation_type(ObMockFKParentTableOperationType::MOCK_FK_PARENT_TABLE_OP_ADD_COLUMN);
          }
        } else { // create table add fk
          mock_fk_parent_table_schema.set_operation_type(ObMockFKParentTableOperationType::MOCK_FK_PARENT_TABLE_OP_CREATE_TABLE_BY_ADD_FK_IN_CHILD_TBALE);
        }
      }
    }
  }
  return ret;
}

int ObDDLService::prepare_gen_mock_fk_parent_tables_for_drop_fks(
      share::schema::ObSchemaGetterGuard &schema_guard,
      const int64_t tenant_id,
      const ObIArray<const ObForeignKeyInfo*> &foreign_key_info_array,
      ObIArray<const ObMockFKParentTableSchema*> &mock_fk_parent_table_schema_ptr_array,
      ObIArray<ObMockFKParentTableSchema> &mock_fk_parent_table_schema_array)
{
  int ret = OB_SUCCESS;
  for (int64_t i = 0; i < foreign_key_info_array.count(); ++i) {
    const ObForeignKeyInfo* foreign_key_info = foreign_key_info_array.at(i);
    if (OB_ISNULL(foreign_key_info)) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("foreign_key_info is null", K(ret));
    } else if (!foreign_key_info->is_parent_table_mock_) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("parent_table is not mock", K(ret), KPC(foreign_key_info));
    } else {
      bool is_in_array = false;
      for (int64_t j = 0; OB_SUCC(ret) && !is_in_array && j < mock_fk_parent_table_schema_array.count(); ++j) {
        if (mock_fk_parent_table_schema_array.at(j).get_mock_fk_parent_table_id() == foreign_key_info->parent_table_id_) {
          is_in_array = true;
          if (OB_FAIL(mock_fk_parent_table_schema_array.at(j).add_foreign_key_info(*foreign_key_info))) {
            LOG_WARN("add_foreign_key_info failed", K(ret), K(foreign_key_info));
          }
        }
      }
      if (OB_SUCC(ret) && !is_in_array) {
        ObMockFKParentTableSchema mock_fk_parent_table_schema;
        const ObMockFKParentTableSchema *mock_fk_parent_table_ptr = NULL;
        if (OB_FAIL(schema_guard.get_mock_fk_parent_table_schema_with_id(
            tenant_id, foreign_key_info->parent_table_id_, mock_fk_parent_table_ptr))) {
          LOG_WARN("get_mock_fk_parent_table_schema_with_id failed", K(ret), K(tenant_id), K(foreign_key_info->parent_table_id_));
        } else if (OB_ISNULL(mock_fk_parent_table_ptr)) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("mock_fk_parent_table_ptr is null", K(ret));
        } else if (OB_FAIL(mock_fk_parent_table_schema.assign(*mock_fk_parent_table_ptr))) {
          LOG_WARN("assign mock_fk_parent_table failed", K(ret));
        } else if (FALSE_IT(mock_fk_parent_table_schema.reset_column_array())){
        } else if (FALSE_IT(mock_fk_parent_table_schema.reset_foreign_key_infos())) {
        } else if (OB_FAIL(mock_fk_parent_table_schema.add_foreign_key_info(*foreign_key_info))) {
          LOG_WARN("add_foreign_key_info failed", K(ret), K(foreign_key_info));
        } else if (OB_FAIL(mock_fk_parent_table_schema_array.push_back(mock_fk_parent_table_schema))) {
          LOG_WARN("push_back mock_fk_parent_table_schema_array failed", K(ret), K(mock_fk_parent_table_schema));
        } else if (OB_FAIL(mock_fk_parent_table_schema_ptr_array.push_back(mock_fk_parent_table_ptr))) {
          LOG_WARN("push_back mock_fk_parent_table_schema_ptr_array failed", K(ret), KPC(mock_fk_parent_table_ptr));
        }
      }
    }
  }
  return ret;
}

int ObDDLService::gen_mock_fk_parent_tables_for_drop_fks(
    share::schema::ObSchemaGetterGuard &schema_guard,
    ObIArray<const ObMockFKParentTableSchema*> &mock_fk_parent_table_schema_ptr_array,
    ObIArray<ObMockFKParentTableSchema> &mock_fk_parent_table_schema_array)
{
  int ret = OB_SUCCESS;
  if (mock_fk_parent_table_schema_ptr_array.count() != mock_fk_parent_table_schema_array.count()) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("counts of mock_fk_parent_table_schema_ptr_array and mock_fk_parent_table_schema_array are not equal",
        K(ret), K(mock_fk_parent_table_schema_ptr_array.count()), K(mock_fk_parent_table_schema_array.count()));
  }
  if (OB_SUCC(ret) && !mock_fk_parent_table_schema_array.empty()) {
    for (int64_t i = 0; OB_SUCC(ret) && i < mock_fk_parent_table_schema_array.count(); ++i) {
      ObMockFKParentTableSchema &tmp_mock_fk_parent_table_schema = mock_fk_parent_table_schema_array.at(i);
      const ObMockFKParentTableSchema *orig_mock_fk_parent_table_ptr = mock_fk_parent_table_schema_ptr_array.at(i);
      ObArray<ObForeignKeyInfo> foreign_key_infos_to_be_deleted;
      if (OB_FAIL(foreign_key_infos_to_be_deleted.assign(tmp_mock_fk_parent_table_schema.get_foreign_key_infos()))) {
        LOG_WARN("failed to assign foreign_key_infos_to_be_deleted", K(ret));
      } else {
        tmp_mock_fk_parent_table_schema.reset_foreign_key_infos();
      }
      for (int64_t j = 0; OB_SUCC(ret) && j < orig_mock_fk_parent_table_ptr->get_foreign_key_infos().count(); ++j) {
        bool is_to_be_remained = true;
        for (int64_t k = 0; is_to_be_remained && k < foreign_key_infos_to_be_deleted.count(); ++k) {
          if (orig_mock_fk_parent_table_ptr->get_foreign_key_infos().at(j).foreign_key_id_ == foreign_key_infos_to_be_deleted.at(k).foreign_key_id_) {
            is_to_be_remained = false;
          }
        }
        if (is_to_be_remained) {
          if (OB_FAIL(tmp_mock_fk_parent_table_schema.add_foreign_key_info(
                      orig_mock_fk_parent_table_ptr->get_foreign_key_infos().at(j)))) {
            LOG_WARN("failed to add_foreign_key_info",
                K(ret), K(orig_mock_fk_parent_table_ptr->get_foreign_key_infos().at(j)));
          }
        }
      }
      if (OB_SUCC(ret)) {
        if (OB_FAIL(tmp_mock_fk_parent_table_schema.reconstruct_column_array_by_foreign_key_infos(orig_mock_fk_parent_table_ptr))) {
          LOG_WARN("reconstruct_column_array_by_foreign_key_infos failed",
              K(ret), K(tmp_mock_fk_parent_table_schema), KPC(orig_mock_fk_parent_table_ptr));
        } else if (tmp_mock_fk_parent_table_schema.get_column_array().count()
                       == orig_mock_fk_parent_table_ptr->get_column_array().count()) {
          // If all columns in mock_fk_parent_table_schema are still referenced by other fk, only update schema version
          tmp_mock_fk_parent_table_schema.set_operation_type(MOCK_FK_PARENT_TABLE_OP_UPDATE_SCHEMA_VERSION);
        } else if (tmp_mock_fk_parent_table_schema.get_column_array().count() == 0) {
          // If all columns in mock_fk_parent_table_schema will be dropped, drop this mock_fk_parent_table
          if (OB_FAIL(tmp_mock_fk_parent_table_schema.set_column_array(orig_mock_fk_parent_table_ptr->get_column_array()))) {
            LOG_WARN("set_column_array failed",
                K(ret), K(tmp_mock_fk_parent_table_schema), KPC(orig_mock_fk_parent_table_ptr));
          } else {
            tmp_mock_fk_parent_table_schema.set_operation_type(MOCK_FK_PARENT_TABLE_OP_DROP_TABLE);
          }
        } else {
          // If part of columns in mock_fk_parent_table_schema will be dropped, drop columns in this mock_fk_parent_table
          tmp_mock_fk_parent_table_schema.set_operation_type(MOCK_FK_PARENT_TABLE_OP_DROP_COLUMN);
          ObMockFKParentTableColumnArray columns_to_be_remained;
          if (OB_FAIL(columns_to_be_remained.assign(tmp_mock_fk_parent_table_schema.get_column_array()))) {
            LOG_WARN("failed to assign columns_to_be_remained", K(ret));
          } else {
            tmp_mock_fk_parent_table_schema.reset_column_array();
          }
          for (int64_t i = 0; OB_SUCC(ret) && i < orig_mock_fk_parent_table_ptr->get_column_array().count(); ++i) {
            bool is_col_to_be_deleted = true;
            for (int64_t j = 0; is_col_to_be_deleted && j < columns_to_be_remained.count(); ++j) {
              if (orig_mock_fk_parent_table_ptr->get_column_array().at(i).first == columns_to_be_remained.at(j).first) {
                is_col_to_be_deleted = false;
              }
            }
            if (is_col_to_be_deleted) {
              if (OB_FAIL(tmp_mock_fk_parent_table_schema.add_column_info_to_column_array(
                  std::make_pair(orig_mock_fk_parent_table_ptr->get_column_array().at(i).first,
                                 orig_mock_fk_parent_table_ptr->get_column_array().at(i).second)))) {
                LOG_WARN("fail to add_column_info_to_column_array", K(ret));
              }
            }
          }
        }
      }
    }
  }
  return ret;
}

int ObDDLService::gen_mock_fk_parent_table_for_drop_table(
    share::schema::ObSchemaService *schema_service,
    share::schema::ObSchemaGetterGuard &schema_guard,
    const obrpc::ObDropTableArg &drop_table_arg,
    const DropTableIdHashSet &drop_table_set,
    const ObIArray<ObForeignKeyInfo> &foreign_key_infos,
    const ObForeignKeyInfo &violated_foreign_key_info,
    const ObTableSchema *table_schema,
    ObMockFKParentTableSchema &mock_fk_parent_table_schema)
{
  int ret = OB_SUCCESS;
  const ObTableSchema *child_table_schema = NULL;
  if (OB_FAIL(schema_guard.get_table_schema(drop_table_arg.tenant_id_, violated_foreign_key_info.child_table_id_, child_table_schema))) {
    LOG_WARN("get_table_schema failed", K(ret), K(violated_foreign_key_info));
  } else if (OB_ISNULL(child_table_schema)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("child_table is null", K(ret), K(violated_foreign_key_info));
  } else if (drop_table_arg.foreign_key_checks_) {
    ret = OB_ERR_TABLE_IS_REFERENCED;
    LOG_USER_ERROR(OB_ERR_TABLE_IS_REFERENCED,
                   table_schema->get_table_name_str().length(), table_schema->get_table_name_str().ptr(),
                   violated_foreign_key_info.foreign_key_name_.length(), violated_foreign_key_info.foreign_key_name_.ptr(),
                   child_table_schema->get_table_name_str().length(), child_table_schema->get_table_name_str().ptr());
  } else if (!drop_table_arg.foreign_key_checks_) {
    ret = OB_SUCCESS;
    uint64_t mock_fk_parent_table_id = OB_INVALID_ID;
    if (drop_table_arg.tables_.count() > 1) {
      ret = OB_NOT_SUPPORTED;
      LOG_USER_ERROR(OB_NOT_SUPPORTED, "drop fk parent table and more tables in one sql");
      LOG_WARN("drop fk parent table and more tables in one sqll", K(ret), KPC(child_table_schema), KPC(table_schema));
    } else if (child_table_schema->get_database_id() != table_schema->get_database_id()) {
      ret = OB_NOT_SUPPORTED;
      LOG_USER_ERROR(OB_NOT_SUPPORTED, "drop fk parent table which has child tables in different database");
      LOG_WARN("drop fk parent table which has child tables in different database not supported", K(ret), KPC(child_table_schema), KPC(table_schema));
    } else if (OB_FAIL(schema_service->fetch_new_table_id(table_schema->get_tenant_id(), mock_fk_parent_table_id))) {
      LOG_WARN("fail to fetch_new_table_id", K(ret));
    } else {
      // mock_fk_parent_table_ptr = &mock_fk_parent_table_schema;
      mock_fk_parent_table_schema.set_tenant_id(table_schema->get_tenant_id());
      mock_fk_parent_table_schema.set_database_id(table_schema->get_database_id());
      mock_fk_parent_table_schema.set_mock_fk_parent_table_id(mock_fk_parent_table_id);
      mock_fk_parent_table_schema.set_mock_fk_parent_table_name(table_schema->get_table_name_str());
    }
    if (OB_SUCC(ret)) {
      for (int64_t i = 0; OB_SUCC(ret) && i < foreign_key_infos.count(); ++i) {
        ObForeignKeyInfo tmp_foreign_key_info = foreign_key_infos.at(i);
        if (OB_HASH_EXIST == drop_table_set.exist_refactored(tmp_foreign_key_info.child_table_id_)) {
          // do-nothing.
          // child table will be dropped in the same drop table SQL. multiple transactions update the schema, but the schema cache is not refreshed.
        } else if (tmp_foreign_key_info.child_table_id_ == tmp_foreign_key_info.parent_table_id_) {
          // do nothing, self referenced fk should be dropped.
        } else if (table_schema->get_table_id() != violated_foreign_key_info.child_table_id_) {
          // The difference of foreign_key_info between orig_parent_table and mock_fk_parent_table is only parent_table_id.
          // parent_column_ids are all the same.
          const int64_t invalid_cst_id = 0;
          tmp_foreign_key_info.set_parent_table_id(mock_fk_parent_table_schema.get_mock_fk_parent_table_id());
          tmp_foreign_key_info.set_is_parent_table_mock(true);
          tmp_foreign_key_info.set_fk_ref_type(FK_REF_TYPE_INVALID);
          tmp_foreign_key_info.set_ref_cst_id(invalid_cst_id);
          if (OB_FAIL(mock_fk_parent_table_schema.add_foreign_key_info(tmp_foreign_key_info))) {
            LOG_WARN("fail to add_foreign_key_info for mock_fk_parent_table_schema", K(ret), K(mock_fk_parent_table_schema), K(tmp_foreign_key_info));
          } else {
            // add column info of foreign_key_infos to mock_fk_parent_table_schema info
            for (int64_t j = 0; OB_SUCC(ret) && j < foreign_key_infos.at(i).parent_column_ids_.count(); ++j) {
              const ObColumnSchemaV2 * parent_column_schema = NULL;
              if (OB_ISNULL(parent_column_schema = table_schema->get_column_schema(foreign_key_infos.at(i).parent_column_ids_.at(j)))){
                ret = OB_ERR_UNEXPECTED;
                LOG_WARN("got null column schema", KPC(table_schema), K(i), K(foreign_key_infos.at(i)), K(j), K(foreign_key_infos.at(i).parent_column_ids_.at(j)));
              } else {
                bool is_existed = false;
                // check if column info is in mock_fk_parent_table_schema already
                for (int64_t k = 0; !is_existed && k < mock_fk_parent_table_schema.get_column_array().count(); ++k) {
                  if (parent_column_schema->get_column_id() == mock_fk_parent_table_schema.get_column_array().at(k).first
                      && 0 == parent_column_schema->get_column_name_str().compare(mock_fk_parent_table_schema.get_column_array().at(k).second)) {
                    is_existed = true;
                  }
                }
                if (!is_existed) {
                  if (OB_FAIL(mock_fk_parent_table_schema.add_column_info_to_column_array(std::make_pair(parent_column_schema->get_column_id(), parent_column_schema->get_column_name_str())))) {
                    LOG_WARN("fail to add_column_info_to_column_array for mock_fk_parent_table_schema", KPC(parent_column_schema));
                  }
                }
              }
            }
          }
        } else {
          // This fk is only a self reference fk, do nothing
        }
      }
    }
  }
  return ret;
}

int ObDDLService::check_fk_columns_type_for_replacing_mock_fk_parent_table(
    share::schema::ObSchemaGetterGuard &schema_guard,
    const ObTableSchema &real_parent_table_schema,
    const ObMockFKParentTableSchema *&mock_parent_table_schema)
{
  int ret = OB_SUCCESS;
  bool is_oracle_mode = false;
  if (OB_ISNULL(mock_parent_table_schema)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("mock_parent_table_schema is not exist", K(ret));
  } else {
    if (OB_ISNULL(mock_parent_table_schema)) {
      ret = OB_ERR_UNEXPECTED;
      SQL_RESV_LOG(WARN, "mock_parent_table_schema is null", K(ret));
    } else {
      for (int64_t i = 0; OB_SUCC(ret) && i < mock_parent_table_schema->get_foreign_key_infos().count(); ++i) {
        const ObTableSchema *child_table_schema = NULL;
        const ObForeignKeyInfo &fk_info = mock_parent_table_schema->get_foreign_key_infos().at(i);
        if (OB_FAIL(schema_guard.get_table_schema(mock_parent_table_schema->get_tenant_id(), fk_info.child_table_id_, child_table_schema))) {
          LOG_WARN("table is not exist", K(ret), K(fk_info.child_table_id_));
        } else if (OB_ISNULL(child_table_schema)) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("parent table schema is null", K(ret));
        } else if (OB_FAIL(child_table_schema->check_if_oracle_compat_mode(is_oracle_mode))) {
          LOG_WARN("check if oracle compat mode failed", K(ret));
        } else {
          // prepare params for check_foreign_key_columns_type
          ObArray<ObString> child_columns;
          ObArray<ObString> parent_columns;
          bool is_column_exist = false;
          for (int64_t j = 0; OB_SUCC(ret) && j < fk_info.child_column_ids_.count(); ++j) {
            ObString child_column_name;
            const ObColumnSchemaV2 *child_col = child_table_schema->get_column_schema(fk_info.child_column_ids_.at(j));
            if (OB_ISNULL(child_col)) {
              ret = OB_ERR_UNEXPECTED;
              LOG_WARN("column is not exist", K(ret), K(fk_info), K(j));
            } else if (OB_FAIL(child_columns.push_back(child_col->get_column_name_str()))) {
              LOG_WARN("fail to push_back to child_columns", K(ret), K(child_col->get_column_name_str()));
            }
          }
          for (int64_t j = 0; OB_SUCC(ret) && j < fk_info.parent_column_ids_.count(); ++j) {
            ObString parent_column_name;
            mock_parent_table_schema->get_column_name_by_column_id(fk_info.parent_column_ids_.at(j), parent_column_name, is_column_exist);
            if (!is_column_exist) {
              ret = OB_ERR_UNEXPECTED;
              LOG_WARN("column is not exist", K(ret), K(fk_info), K(j));
            } else if (OB_FAIL(parent_columns.push_back(parent_column_name))) {
              LOG_WARN("fail to push_back to real_parent_table_schema_columns", K(ret));
            }
          }
          if (OB_FAIL(ret)) {
          } else if (OB_FAIL(ObResolverUtils::check_foreign_key_columns_type(
              !is_oracle_mode/*is_mysql_compat_mode*/,
              *child_table_schema,
              real_parent_table_schema,
              child_columns,
              parent_columns,
              NULL))) {
            ret = OB_ERR_CANNOT_ADD_FOREIGN;
            LOG_WARN("Failed to check_foreign_key_columns_type", K(ret));
          }
        }
      }
    }
  }
  return ret;
}

int ObDDLService::get_uk_cst_id_for_replacing_mock_fk_parent_table(
    const ObIArray<const share::schema::ObTableSchema*> &index_table_schemas,
    share::schema::ObForeignKeyInfo &foreign_key_info)
{
  int ret = OB_SUCCESS;
  bool is_match = false;
  for (int64_t i = 0; OB_SUCC(ret) && !is_match && i < index_table_schemas.count(); ++i) {
    const ObTableSchema *index_table_schema = index_table_schemas.at(i);
    if (index_table_schema->is_unique_index()) {
      const ObColumnSchemaV2 *index_col = NULL;
      const ObIndexInfo &index_info = index_table_schema->get_index_info();
      ObArray<uint64_t> uk_columns;
      for (int64_t j = 0; OB_SUCC(ret) && j < index_info.get_size(); ++j) {
        if (OB_ISNULL(index_col = index_table_schema->get_column_schema(index_info.get_column(j)->column_id_))) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("get index column schema failed", K(ret), K(index_info.get_column(j)->column_id_));
        } else if (index_col->is_hidden() || index_col->is_shadow_column()) { // do nothing
        } else if (OB_FAIL(uk_columns.push_back(index_info.get_column(j)->column_id_))) {
          LOG_WARN("push back index column failed", K(ret), KPC(index_col));
        }
      }
      if (OB_SUCC(ret)) {
        if (OB_FAIL(sql::ObResolverUtils::check_match_columns(foreign_key_info.parent_column_ids_, uk_columns, is_match))) {
          LOG_WARN("Failed to check_match_columns", K(ret));
        } else if (is_match) {
          foreign_key_info.fk_ref_type_ = FK_REF_TYPE_UNIQUE_KEY;
          foreign_key_info.ref_cst_id_ = index_table_schema->get_table_id();
        }
      }
    }
  }
  return ret;
}

// replace mock fk parent table with real fk parent table
int ObDDLService::gen_mock_fk_parent_table_for_replacing_mock_fk_parent_table(
    share::schema::ObSchemaGetterGuard &schema_guard,
    const uint64_t drop_mock_fk_parent_table_id,
    const share::schema::ObTableSchema &real_parent_table,
    const ObIArray<const share::schema::ObTableSchema*> &uk_index_schemas,
    ObMockFKParentTableSchema &mock_fk_parent_table_schema)
{
  int ret = OB_SUCCESS;
  const ObMockFKParentTableSchema *mock_fk_parent_table_ptr = NULL;
  if (OB_FAIL(schema_guard.get_mock_fk_parent_table_schema_with_id(
      real_parent_table.get_tenant_id(),
      drop_mock_fk_parent_table_id,
      mock_fk_parent_table_ptr))) {
    LOG_WARN("check_mock_fk_parent_table_exist_by_id failed", K(ret), K(real_parent_table.get_tenant_id()), K(drop_mock_fk_parent_table_id));
  } else if (OB_ISNULL(mock_fk_parent_table_ptr)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("mock_fk_parent_table is not exist", K(ret), K(mock_fk_parent_table_ptr));
  } else if (OB_FAIL(mock_fk_parent_table_schema.assign(*mock_fk_parent_table_ptr))) {
    LOG_WARN("mock_fk_parent_table_schema assign failed", K(ret), KPC(mock_fk_parent_table_ptr));
  } else if (FALSE_IT(mock_fk_parent_table_schema.set_operation_type(share::schema::ObMockFKParentTableOperationType::MOCK_FK_PARENT_TABLE_OP_REPLACED_BY_REAL_PREANT_TABLE))) {
  } else if (OB_FAIL(check_fk_columns_type_for_replacing_mock_fk_parent_table(schema_guard, real_parent_table, mock_fk_parent_table_ptr))) {
    LOG_WARN("check_fk_columns_type_for_replacing_mock_fk_parent_table failed", K(ret), K(real_parent_table), KPC(mock_fk_parent_table_ptr));
  } else {
    const ObIArray<ObForeignKeyInfo> &ori_mock_fk_infos_array = mock_fk_parent_table_ptr->get_foreign_key_infos();
    // modify the parent column id of fk, make it fit with real parent table
    // mock_column_id -> column_name -> real_column_id
    bool is_column_exist = false;
    for (int64_t i = 0; OB_SUCC(ret) && i < ori_mock_fk_infos_array.count(); ++i) {
      mock_fk_parent_table_schema.get_foreign_key_infos().at(i).parent_column_ids_.reuse();
      mock_fk_parent_table_schema.get_foreign_key_infos().at(i).fk_ref_type_ = FK_REF_TYPE_INVALID;
      mock_fk_parent_table_schema.get_foreign_key_infos().at(i).is_parent_table_mock_ = false;
      mock_fk_parent_table_schema.get_foreign_key_infos().at(i).parent_table_id_ = real_parent_table.get_table_id();
      for (int64_t j = 0;  OB_SUCC(ret) && j < ori_mock_fk_infos_array.at(i).parent_column_ids_.count(); ++j) {
        uint64_t mock_parent_table_column_id = ori_mock_fk_infos_array.at(i).parent_column_ids_.at(j);
        ObString column_name;
        mock_fk_parent_table_ptr->get_column_name_by_column_id(mock_parent_table_column_id, column_name, is_column_exist);
        if (!is_column_exist) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("column is not exist", K(ret), K(mock_parent_table_column_id), KPC(mock_fk_parent_table_ptr));
        } else {
          const ObColumnSchemaV2 *col_schema = NULL;
          if (OB_ISNULL(col_schema = real_parent_table.get_column_schema(column_name))) {
            ret = OB_ERR_UNEXPECTED;
            LOG_WARN("get column schema failed", K(ret), K(column_name));
          } else if (OB_FAIL(mock_fk_parent_table_schema.get_foreign_key_infos().at(i).parent_column_ids_.push_back(col_schema->get_column_id()))) {
            LOG_WARN("push_back to parent_column_ids failed", K(ret), K(col_schema->get_column_id()));
          }
        }
      }
      // check and mofidy ref cst type and ref cst id of fk
      const ObRowkeyInfo &rowkey_info = real_parent_table.get_rowkey_info();
      common::ObArray<uint64_t> pk_column_ids;
      for (int64_t i = 0; OB_SUCC(ret) && i < rowkey_info.get_size(); ++i) {
        uint64_t column_id = 0;
        const ObColumnSchemaV2 *col_schema = NULL;
        if (OB_FAIL(rowkey_info.get_column_id(i, column_id))) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("fail to get rowkey info", K(ret), K(i), K(rowkey_info));
        } else if (OB_ISNULL(col_schema = real_parent_table.get_column_schema(column_id))) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("get index column schema failed", K(ret));
        } else if (col_schema->is_hidden() || col_schema->is_shadow_column()) {
          // do nothing
        } else if(OB_FAIL(pk_column_ids.push_back(col_schema->get_column_id()))) {
          LOG_WARN("push back column_id failed", K(ret), K(col_schema->get_column_id()));
        }
      }
      bool is_match = false;
      if (OB_FAIL(ret)) {
      } else if (OB_FAIL(sql::ObResolverUtils::check_match_columns(pk_column_ids, mock_fk_parent_table_schema.get_foreign_key_infos().at(i).parent_column_ids_, is_match))) {
        LOG_WARN("check_match_columns failed", K(ret));
      } else if (is_match) {
        mock_fk_parent_table_schema.get_foreign_key_infos().at(i).fk_ref_type_ = FK_REF_TYPE_PRIMARY_KEY;
      } else { // pk is not match, check if uk match
        if (OB_FAIL(get_uk_cst_id_for_replacing_mock_fk_parent_table(
            uk_index_schemas, mock_fk_parent_table_schema.get_foreign_key_infos().at(i)))) {
          LOG_WARN("fail to get_uk_cst_id_for_replacing_mock_fk_parent_table", K(ret));
        } else if (FK_REF_TYPE_INVALID == mock_fk_parent_table_schema.get_foreign_key_infos().at(i).fk_ref_type_) {
          ret = OB_ERR_CANNOT_ADD_FOREIGN;
          LOG_WARN("fk_ref_type is invalid", K(ret), KPC(mock_fk_parent_table_ptr));
        }
      }
    }
  }

  return ret;
}

// MockFKParentTable end

int ObDDLService::get_index_schema_by_name(
    const uint64_t data_table_id,
    const uint64_t database_id,
    const ObDropIndexArg &drop_index_arg,
    ObSchemaGetterGuard &schema_guard,
    const ObTableSchema *&index_table_schema)
{
  int ret = OB_SUCCESS;
  index_table_schema = nullptr;
  ObString index_table_name;
  ObArenaAllocator allocator(ObModIds::OB_SCHEMA);
  const uint64_t tenant_id = drop_index_arg.tenant_id_;
  const ObString &index_name = drop_index_arg.index_name_;
  const bool is_mlog = (obrpc::ObIndexArg::DROP_MLOG == drop_index_arg.index_action_type_);
  index_table_schema = nullptr;

  //build index name and get index schema
  if (is_mlog) {
    index_table_name = index_name;
  } else if (OB_FAIL(ObTableSchema::build_index_table_name(allocator,
                                                    data_table_id,
                                                    index_name,
                                                    index_table_name))) {
    LOG_WARN("build_index_table_name failed", K(ret), K(data_table_id), K(index_name));
  }
  if (OB_SUCC(ret)) {
    const bool is_index = !is_mlog;
    if (OB_FAIL(schema_guard.get_table_schema(tenant_id,
                                              database_id,
                                              index_table_name,
                                              is_index,
                                              index_table_schema))) {
      LOG_WARN("fail to get table schema", K(ret), K(tenant_id), K(database_id), K(index_table_schema));
    } else if (nullptr == index_table_schema) {
      ret = OB_ERR_CANT_DROP_FIELD_OR_KEY;
      LOG_USER_ERROR(OB_ERR_CANT_DROP_FIELD_OR_KEY, index_name.length(), index_name.ptr());
      LOG_WARN("get index table schema failed", K(tenant_id),
          K(database_id), K(index_table_name), K(ret));
    } else if (is_index && data_table_id != index_table_schema->get_data_table_id()) {
      ret = OB_ERR_CANT_DROP_FIELD_OR_KEY;
      LOG_USER_ERROR(OB_ERR_CANT_DROP_FIELD_OR_KEY, index_name.length(), index_name.ptr());
      LOG_WARN("get index table schema failed", K(ret), K(tenant_id),
          K(database_id), K(index_table_name), K(index_table_schema->get_table_id()), K(ret));
      index_table_schema = nullptr;
    } else if (index_table_schema->is_in_recyclebin()) {
      ret = OB_ERR_OPERATION_ON_RECYCLE_OBJECT;
      LOG_WARN("index table is in recyclebin", K(ret));
    }
  }
  return ret;
}

int ObDDLService::get_valid_index_schema_by_id_for_drop_index_(
    const uint64_t data_table_id,
    const ObDropIndexArg &drop_index_arg,
    ObSchemaGetterGuard &schema_guard,
    const ObTableSchema *&index_table_schema)
{
  int ret = OB_SUCCESS;
  const uint64_t tenant_id = drop_index_arg.tenant_id_;
  // drop_index_arg.index_name_ is specified by user. drop_index_arg.index_name_ may be
  // not matched with drop_index_arg.index_table_id_. For example: Drop FTS index need
  // to drop all built-in FTS index tables. One index_name correspond to multiple index
  // tables. So, index_name may be not matched with index_table_id
  const uint64_t table_id = drop_index_arg.index_table_id_;
  const ObString index_name = drop_index_arg.index_name_;
  index_table_schema = nullptr;
  if (OB_UNLIKELY(OB_INVALID_ID == data_table_id || !drop_index_arg.is_valid())) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("invalid arguments", K(ret), K(data_table_id), K(drop_index_arg));
  } else if (OB_FAIL(schema_guard.get_table_schema(tenant_id, table_id, index_table_schema))) {
    LOG_WARN("fail to get table schema", K(ret), K(tenant_id), K(table_id), K(index_table_schema));
  } else if (nullptr == index_table_schema) {
    ret = OB_ERR_CANT_DROP_FIELD_OR_KEY;
    LOG_USER_ERROR(OB_ERR_CANT_DROP_FIELD_OR_KEY, index_name.length(), index_name.ptr());
    LOG_WARN("get index table schema failed", K(tenant_id), K(data_table_id), K(ret));
  } else if (index_table_schema->is_in_recyclebin()) {
    ret = OB_ERR_OPERATION_ON_RECYCLE_OBJECT;
    LOG_WARN("index table is in recyclebin", K(ret), K(data_table_id), K(drop_index_arg));
  }
  return ret;
}

// To avoid ddl hung when drop and add index with same index-name in single stmt,
// should rename dropping index firstly, and then push it into ddl scheduler queue.
int ObDDLService::rename_dropping_index_name(
    const share::schema::ObTableSchema &orig_index_schema,
    const bool is_inner_and_domain_index,
    const ObDropIndexArg &drop_index_arg,
    ObSchemaGetterGuard &schema_guard,
    ObDDLOperator &ddl_operator,
    common::ObMySQLTransaction &trans,
    common::ObIArray<share::schema::ObTableSchema> &new_index_schemas)
{
  int ret = OB_SUCCESS;
  int nwrite = 0;
  const int64_t buf_size = number::ObNumber::MAX_PRINTABLE_SIZE;
  char buf[buf_size] = {0};
  ObString index_name = drop_index_arg.index_name_;
  bool need_rename = true;
  // When dropping fts index, one index_name corresponds to multiple index tables. We can not decide
  // right index table by index_name. To handle multiple FTS built-in index tables, inner DDL task
  // will drop these built-in index tables by drop_index_arg.index_table_id_.
  const ObTableSchema *index_table_schema = nullptr;
  const uint64_t tenant_id = orig_index_schema.get_tenant_id();
  const uint64_t data_table_id = orig_index_schema.get_data_table_id();
  const uint64_t database_id = orig_index_schema.get_database_id();

  if (!is_inner_and_domain_index) {
    if (OB_FAIL(get_index_schema_by_name(data_table_id, database_id, drop_index_arg,
                                         schema_guard, index_table_schema))) {
      LOG_WARN("get index schema by name", K(ret), K(data_table_id), K(database_id));
    }
  } else if (OB_FAIL(get_valid_index_schema_by_id_for_drop_index_(data_table_id, drop_index_arg, schema_guard,
          index_table_schema))) {
    LOG_WARN("fail to get valid index schema by id for drop index", K(ret), K(data_table_id), K(drop_index_arg));
  } else if (OB_ISNULL(index_table_schema)) {
    // FTS or multi-value index and inner rpc, if its table schema does not exist, just skip rename and return success.
    need_rename = false;
  } else {
    ObArenaAllocator allocator(ObModIds::OB_SCHEMA);
    ObString cur_index_name;
    if (OB_FAIL(index_table_schema->get_index_name(cur_index_name))) {
      LOG_WARN("build_index_table_name failed", K(ret), K(data_table_id), KPC(index_table_schema));
    } else if (0 != index_name.case_compare(cur_index_name)) {
      // FTS index and inner rpc, it has been renamed, just skip rename and return success.
      need_rename = false;
    }
  }
  if (OB_FAIL(ret) || !need_rename) {
  } else if (OB_ISNULL(index_table_schema)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("unexpected error, index table schema is nullptr", K(ret), KP(index_table_schema), K(data_table_id),
        K(drop_index_arg));
  } else if ((!drop_index_arg.is_inner_ || drop_index_arg.is_parent_task_dropping_fts_index_) && index_table_schema->is_fts_index_aux()) {
    // This task is the parent task of drop fts index, no need to rename.
    if (OB_FAIL(get_dropping_domain_index_invisiable_aux_table_schema(index_table_schema->get_tenant_id(), data_table_id,
       index_table_schema->get_table_id(), true, index_table_schema->get_table_name_str(), schema_guard, ddl_operator,
       trans, new_index_schemas))) {
      LOG_WARN("fail to get dropping fts aux table schema", K(ret), K(data_table_id), K(index_table_schema));
    } else if (OB_FAIL(new_index_schemas.push_back(*index_table_schema))) {
      LOG_WARN("fail to push back index schema", K(ret), KPC(index_table_schema));
    }
  } else if ((!drop_index_arg.is_inner_ || drop_index_arg.is_parent_task_dropping_multivalue_index_) && index_table_schema->is_multivalue_index_aux()) {
    if (OB_FAIL(get_dropping_domain_index_invisiable_aux_table_schema(index_table_schema->get_tenant_id(), data_table_id,
       index_table_schema->get_table_id(), false, index_table_schema->get_table_name_str(), schema_guard, ddl_operator,
       trans, new_index_schemas))) {
      LOG_WARN("fail to get dropping fts aux table schema", K(ret), K(data_table_id), K(index_table_schema));
    } else if (OB_FAIL(new_index_schemas.push_back(*index_table_schema))) {
      LOG_WARN("fail to push back index schema", K(ret), KPC(index_table_schema));
    }
  } else if ((!drop_index_arg.is_inner_ || drop_index_arg.is_vec_inner_drop_) && index_table_schema->is_vec_domain_index()) {
    // This task is the parent task of drop vec index, no need to rename.
    if (OB_FAIL(ObVectorIndexUtil::get_dropping_vec_index_invisiable_table_schema(*index_table_schema, data_table_id,
        drop_index_arg.is_vec_inner_drop_, schema_guard, ddl_operator, trans, new_index_schemas))) {
      LOG_WARN("fail to get dropping vec index table schema", K(ret), K(data_table_id), K(index_table_schema));
    } else if (OB_FAIL(new_index_schemas.push_back(*index_table_schema))) {
      LOG_WARN("fail to push back index schema", K(ret), KPC(index_table_schema));
    }
  } else if ((nwrite = snprintf(buf, buf_size, "%s:%lu",
    "SYS_DELETING_INDEX", ObTimeUtility::current_time())) >= buf_size || nwrite < 0) {
    ret = common::OB_BUF_NOT_ENOUGH;
    LOG_WARN("buf is not large enough", K(ret), K(buf_size));
  } else {
    SMART_VAR(ObTableSchema, new_index_schema) {
      const bool is_in_deleting = true;
      const ObIndexStatus new_index_status = INDEX_STATUS_UNAVAILABLE;
      ObString new_index_name = ObString::make_string(buf);
      obrpc::ObRenameIndexArg rename_index_arg;
      rename_index_arg.tenant_id_         = index_table_schema->get_tenant_id();
      rename_index_arg.origin_index_name_ = index_name;
      rename_index_arg.new_index_name_    = new_index_name;
      if (OB_INVALID_ID != drop_index_arg.index_table_id_) {
        if (OB_FAIL(ddl_operator.alter_table_rename_index_with_origin_index_name(index_table_schema->get_tenant_id(),
                                                          index_table_schema->get_table_id(),
                                                          new_index_name,
                                                          new_index_status,
                                                          is_in_deleting,
                                                          trans,
                                                          new_index_schema))) {
          LOG_WARN("fail to alter table rename index", K(ret));
        }
      } else if (OB_FAIL(ddl_operator.alter_table_rename_index(tenant_id,
                                                               data_table_id,
                                                               database_id,
                                                               rename_index_arg,
                                                               &new_index_status,
                                                               is_in_deleting,
                                                               trans,
                                                               new_index_schema))) {
        LOG_WARN("rename index failed", K(ret));
      }
      if (FAILEDx(new_index_schemas.push_back(new_index_schema))) {
        LOG_WARN("fail to push back new index schemas", K(ret), K(new_index_schema));
      }
    }
  }
  return ret;
}

int ObDDLService::get_dropping_domain_index_invisiable_aux_table_schema(
    const uint64_t tenant_id,
    const uint64_t data_table_id,
    const uint64_t index_table_id,
    const bool is_fts_index,
    const ObString &index_name,
    share::schema::ObSchemaGetterGuard &schema_guard,
    ObDDLOperator &ddl_operator,
    common::ObMySQLTransaction &trans,
    common::ObIArray<share::schema::ObTableSchema> &new_aux_schemas)
{
  int ret = OB_SUCCESS;
  const share::schema::ObTableSchema *data_table_schema = nullptr;
  ObSEArray<const ObSimpleTableSchemaV2 *, OB_MAX_AUX_TABLE_PER_MAIN_TABLE> indexs;
  if (OB_UNLIKELY(OB_INVALID_ID == data_table_id
        || OB_INVALID_ID == index_table_id
        || OB_INVALID_TENANT_ID == tenant_id
        || index_name.empty())) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("invalid arguments", K(ret), K(data_table_id), K(index_table_id), K(tenant_id), K(index_name));
  } else if (OB_FAIL(schema_guard.get_table_schema(tenant_id, data_table_id, data_table_schema))) {
    LOG_WARN("fail to get index schema with data table id", K(ret), K(tenant_id), K(data_table_id));
  } else if (OB_ISNULL(data_table_schema)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("unexpected error, data table schema is nullptr", K(ret), KP(data_table_schema));
  } else {
    SMART_VAR(ObTableSchema, new_aux_schema) {
      const ObIArray<share::schema::ObAuxTableMetaInfo> &indexs = data_table_schema->get_simple_index_infos();
      const share::schema::ObTableSchema *doc_word_schema = nullptr;
      const share::schema::ObTableSchema *rowkey_doc_schema = nullptr;
      const share::schema::ObTableSchema *doc_rowkey_schema = nullptr;
      for (int64_t i = 0; OB_SUCC(ret) && i < indexs.count(); ++i) {
        const share::schema::ObAuxTableMetaInfo &info = indexs.at(i);
        if (share::schema::is_rowkey_doc_aux(info.index_type_)) {
          if (OB_FAIL(schema_guard.get_table_schema(tenant_id, info.table_id_, rowkey_doc_schema))) {
            LOG_WARN("fail to get rowkey doc table schema", K(ret), K(tenant_id), K(info));
          } else if (OB_ISNULL(rowkey_doc_schema)) {
            ret = OB_ERR_UNEXPECTED;
            LOG_WARN("unexpeted error, rowkey doc schema is nullptr", K(ret), K(info));
          } else if (OB_FAIL(new_aux_schemas.push_back(*rowkey_doc_schema))) {
            LOG_WARN("fail to push doc rowkey table schema", K(ret), KPC(rowkey_doc_schema));
          }
        } else if (share::schema::is_doc_rowkey_aux(info.index_type_)) {
          if (OB_FAIL(schema_guard.get_table_schema(tenant_id, info.table_id_, doc_rowkey_schema))) {
            LOG_WARN("fail to get doc rowkey table schema", K(ret), K(tenant_id), K(info));
          } else if (OB_ISNULL(doc_rowkey_schema)) {
            ret = OB_ERR_UNEXPECTED;
            LOG_WARN("unexpected error, doc rowkey schema is nullptr", K(ret), K(info));
          } else if (OB_FAIL(new_aux_schemas.push_back(*doc_rowkey_schema))) {
            LOG_WARN("fail to push doc rowkey table schema", K(ret), KPC(doc_rowkey_schema));
          }
        } else if (is_fts_index && share::schema::is_fts_doc_word_aux(info.index_type_)){
          if (OB_FAIL(schema_guard.get_table_schema(tenant_id, info.table_id_, doc_word_schema))) {
            LOG_WARN("fail to get doc word table schema", K(ret), K(tenant_id), K(info));
          } else if (OB_ISNULL(doc_word_schema)) {
            ret = OB_ERR_UNEXPECTED;
            LOG_WARN("unexpected error, doc word schema is nullptr", K(ret), K(info));
          } else {
            int nwrite = 0;
            const int64_t buf_size = OB_MAX_TABLE_NAME_BUF_LENGTH;
            char buf[buf_size] = {0};
            if (OB_FAIL(databuff_printf(buf, buf_size, "%.*s_fts_doc_word", index_name.length(), index_name.ptr()))) {
              LOG_WARN("fail to printf fts doc word name str", K(ret), K(index_name));
            } else if (0 == doc_word_schema->get_table_name_str().case_compare(buf)) {
              if (OB_FAIL(new_aux_schemas.push_back(*doc_word_schema))) {
                LOG_WARN("fail to push doc word table schema", K(ret), KPC(doc_word_schema));
              }
            }
          }
        }
      }
    }
    STORAGE_FTS_LOG(INFO, "get dropping fts aux table name", K(ret), K(tenant_id), K(data_table_id), K(index_table_id));
  }
  return ret;
}

int ObDDLService::generate_tmp_idx_schemas(
                  const ObTableSchema &new_table_schema,
                  ObIArray<ObTableSchema> &idx_schemas,
                  ObSchemaGetterGuard &schema_guard) {
  int ret = OB_SUCCESS;
  const uint64_t tenant_id = new_table_schema.get_tenant_id();
  ObSEArray<ObAuxTableMetaInfo, 16> simple_index_infos;
  if (OB_FAIL(new_table_schema.get_simple_index_infos(simple_index_infos))) {
    LOG_WARN("get simple_index_infos failed", K(ret));
  } else {
    for (int64_t i = 0; OB_SUCC(ret) && i < simple_index_infos.count(); ++i) {
      const ObTableSchema *index_table_schema = NULL;
      if (OB_FAIL(schema_guard.get_table_schema(tenant_id,
          simple_index_infos.at(i).table_id_, index_table_schema))) {
        RS_LOG(WARN, "get_table_schema failed", K(ret), K(tenant_id),
               "table id", simple_index_infos.at(i).table_id_);
      } else if (OB_ISNULL(index_table_schema)) {
        ret = OB_ERR_UNEXPECTED;
        RS_LOG(WARN, "table schema should not be null", K(ret));
      } else {
        if (OB_FAIL(idx_schemas.push_back(*index_table_schema))) {
          RS_LOG(WARN, "fail to push back to idx_schemas", K(ret));
        }
      }
    }
  }
  return ret;
}

int ObDDLService::fill_column_collation(
    const ObSQLMode sql_mode,
    const bool is_oracle_mode,
    const ObTableSchema &table_schema,
    common::ObIAllocator &allocator,
    ObColumnSchemaV2 &column_schema)
{
  int ret = OB_SUCCESS;
  ObObjTypeClass col_tc = column_schema.get_data_type_class();
  ObCollationType collation_type = table_schema.get_collation_type();
  ObCharsetType charset_type = table_schema.get_charset_type();
  const ObCollationType cur_extended_type_info_collation = ObCharset::get_system_collation();
  if (ObStringTC == col_tc) {
    if (OB_FAIL(ObDDLResolver::check_and_fill_column_charset_info(
                column_schema, charset_type, collation_type))) {
      RS_LOG(WARN, "failed to fill column charset info", K(ret));
    } else if (OB_FAIL(ObDDLResolver::check_string_column_length(
                       column_schema, is_oracle_mode))) {
      RS_LOG(WARN, "failed to check string column length", K(ret));
    }
  } else if (ObRawTC == col_tc) {
    if (OB_FAIL(ObDDLResolver::check_raw_column_length(column_schema))) {
      RS_LOG(WARN, "failed to check raw column length", K(ret), K(column_schema));
    }
  } else if (ob_is_text_tc(column_schema.get_data_type())) {
    if (OB_FAIL(ObDDLResolver::check_and_fill_column_charset_info(
        column_schema, table_schema.get_charset_type(), table_schema.get_collation_type()))) {
      RS_LOG(WARN, "failed to fill column charset info", K(ret));
    } else if (OB_FAIL(ObDDLResolver::check_text_column_length_and_promote(column_schema,
                       table_schema.get_table_id(), true))) {
      RS_LOG(WARN, "failed to check text or blob column length", K(ret));
    }
  } else if (ObEnumSetTC == col_tc) {
    if (OB_FAIL(ObDDLResolver::check_and_fill_column_charset_info(column_schema, charset_type, collation_type))) {
      LOG_WARN("fail to check and fill column charset info", K(ret), K(column_schema));
    } else if (OB_FAIL(ObResolverUtils::check_extended_type_info(
                allocator,
                column_schema.get_extended_type_info(),
                cur_extended_type_info_collation,
                column_schema.get_column_name_str(),
                column_schema.get_data_type(),
                column_schema.get_collation_type(),
                sql_mode))) {
      LOG_WARN("fail to fill extended type info", K(ret), K(column_schema));
    } else if (OB_FAIL(ObDDLResolver::calc_enum_or_set_data_length(column_schema))) {
      LOG_WARN("fail to calc data length", K(ret), K(column_schema));
    }
  } else if (ObGeometryTC == col_tc && !column_schema.is_nullable()) {
    ret = OB_ER_INVALID_USE_OF_NULL;
    LOG_WARN("alter table add geometry column can not has not null constraint", K(ret));
  }
  return ret;
}

int ObDDLService::resolve_orig_default_value(ObColumnSchemaV2 &alter_column_schema,
    const ObTimeZoneInfoWrap &tz_info_wrap, const common::ObString *nls_formats,
    ObIAllocator &allocator)
{
  int ret = OB_SUCCESS;
  const ObObj &cur_default_value = alter_column_schema.get_cur_default_value();
  if (alter_column_schema.is_identity_column()) {
    // When column is identity column or generated column, cur_default_value and orig_default_value is different
    // cur_default_value:"SEQUENCE.NEXTVAL"
    // orig_default_value:pure_sequnece_id
  } else if (!cur_default_value.is_null()) {
    if (OB_FAIL(alter_column_schema.set_orig_default_value(cur_default_value))) {
      LOG_WARN("fail to set orig default value for alter table", K(ret), K(cur_default_value));
    }
  } else if (alter_column_schema.is_nullable()) {
    ObObj null_obj;
    null_obj.set_null();
    if (OB_FAIL(alter_column_schema.set_orig_default_value(null_obj))) {
      LOG_WARN("fail to set origin default value", K(ret));
    }
  } else {
    ObObj default_value;
    default_value.set_type(alter_column_schema.get_data_type());
    if (OB_FAIL(default_value.build_not_strict_default_value(alter_column_schema.get_accuracy().get_precision(), alter_column_schema.get_collation_type()))) {
      LOG_WARN("failed to build not strict default value", K(ret));
    } else if (OB_FAIL(alter_column_schema.set_orig_default_value(default_value))) {
      LOG_WARN("failed to set orig default value", K(ret));
    }
  }
  if (OB_SUCC(ret) && !alter_column_schema.is_identity_column()) {
    ObObj orig_default_value = alter_column_schema.get_orig_default_value();
    if (IS_DEFAULT_NOW_OBJ(orig_default_value) || alter_column_schema.is_default_expr_v2_column()) {
      if (OB_FAIL(ObDDLResolver::calc_default_value(alter_column_schema,
                                                    orig_default_value,
                                                    tz_info_wrap,
                                                    nls_formats,
                                                    allocator))) {
        LOG_WARN("fail to calc default now expr", K(ret));
      } else if (OB_FAIL(alter_column_schema.set_orig_default_value(orig_default_value))) {
        LOG_WARN("fail to set orig default value", K(orig_default_value), K(ret));
      }
    }
  }
  return ret;
}

int ObDDLService::resolve_timestamp_column(AlterColumnSchema *alter_column_schema,
                                            ObTableSchema &new_table_schema,
                                            ObColumnSchemaV2 &new_column_schema,
                                            const common::ObTimeZoneInfoWrap &tz_info_wrap,
                                            const common::ObString *nls_formats,
                                            ObIAllocator &allocator)
{
  int ret = OB_SUCCESS;
  bool is_oracle_mode = false;
  if (OB_ISNULL(alter_column_schema)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("alter_column_schema is NULL", K(ret));
  } else if (OB_FAIL(new_table_schema.check_if_oracle_compat_mode(is_oracle_mode))) {
    LOG_WARN("fail to check if tenant mode is oracle mode", K(ret));
  } else if (ObTimestampType != new_column_schema.get_data_type()
      || new_column_schema.is_generated_column()
      || false == alter_column_schema->check_timestamp_column_order_
      || new_column_schema.is_udt_related_column(is_oracle_mode)) {
    //nothing to do
  } else {
    bool is_first_timestamp = false;
    ObTableSchema::const_column_iterator it_begin = new_table_schema.column_begin();
    ObTableSchema::const_column_iterator it_end = new_table_schema.column_end();
    bool found = false;
    for(; OB_SUCC(ret) && it_begin != it_end && !found; it_begin++) {
      if (OB_ISNULL(it_begin)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("it_begin should not be NULL", K(ret));
      } else if (OB_ISNULL(*it_begin)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("*it_begin should not be NULL", K(ret));
      } else {
        if (ObTimestampType == (*it_begin)->get_data_type()) {
          if (0 == (*it_begin)->get_column_name_str().case_compare(
                  new_column_schema.get_column_name_str())) {
            is_first_timestamp = true;
          }
          found = true;
        }
      }
    }
    if (OB_SUCC(ret) && !found) {
      // 1.add new column will add to the end of table schema
      // 2.new table schema not contain new column now
      if (OB_DDL_ADD_COLUMN == alter_column_schema->alter_type_
          && ObTimestampType == new_column_schema.get_data_type()) {
        is_first_timestamp = true;
      }
    }
    if (OB_SUCC(ret)) {
      if (OB_DDL_ALTER_COLUMN == alter_column_schema->alter_type_) {
        //drop default or set default
        //mysql seem like set default now will couse a parser error;
        if (is_first_timestamp && alter_column_schema->is_drop_default_) {
          //new_column_schema is orig_column_schema;
          //if default value is now(), on update current timestamp is false;
          if (!new_column_schema.is_nullable()
              && !IS_DEFAULT_NOW_OBJ(new_column_schema.get_cur_default_value())) {
            new_column_schema.set_on_update_current_timestamp(true);
          } else {
            //do nothing
          }
        }
      } else {
        bool is_set_default = alter_column_schema->is_set_default_;
        bool is_set_null = alter_column_schema->is_set_nullable_;
        if (is_first_timestamp
            && !is_set_null
            && !is_set_default
            && !new_column_schema.is_on_update_current_timestamp()) {
          new_column_schema.set_nullable(false);
          new_column_schema.get_cur_default_value().set_ext(
              ObActionFlag::OP_DEFAULT_NOW_FLAG);
          new_column_schema.set_on_update_current_timestamp(true);
        } else if (!is_set_null) {
          new_column_schema.set_nullable(false);
          if (!is_set_default) {
            if (alter_column_schema->is_no_zero_date_) {
              ret = OB_INVALID_DEFAULT;
              LOG_USER_ERROR(OB_INVALID_DEFAULT, alter_column_schema->get_column_name_str().length(),
                             alter_column_schema->get_column_name_str().ptr());
            } else {
              int64_t zero_date = ObTimeConverter::ZERO_DATETIME;
              ObTimeConverter::round_datetime(alter_column_schema->get_data_scale(), zero_date);
              new_column_schema.get_cur_default_value().set_timestamp(zero_date);
            }
          } else if (new_column_schema.get_cur_default_value().is_null()) {
            ret = OB_INVALID_DEFAULT;
            LOG_USER_ERROR(OB_INVALID_DEFAULT, new_column_schema.get_column_name_str().length(),
                           new_column_schema.get_column_name_str().ptr());
          }
        } else {
          new_column_schema.set_nullable(true);
          if (!is_set_default) {
            new_column_schema.get_cur_default_value().set_null();
          }
        }
        if(OB_SUCC(ret)
           && OB_DDL_ADD_COLUMN == alter_column_schema->alter_type_) {
          ObObj cur_default_value = new_column_schema.get_cur_default_value();
          if (IS_DEFAULT_NOW_OBJ(cur_default_value) || alter_column_schema->is_default_expr_v2_column()) {
            if (OB_FAIL(ObDDLResolver::calc_default_value(*alter_column_schema,
                                                          cur_default_value,
                                                          tz_info_wrap,
                                                          nls_formats,
                                                          allocator))) {
              LOG_WARN("fail to calc default now expr", K(ret));
            }
          }
          if (OB_SUCC(ret)) {
            if (OB_FAIL(alter_column_schema->set_orig_default_value(cur_default_value))) {
              OB_LOG(WARN, "fail to set orig default value", K(cur_default_value), K(ret));
            }
          }
        }
      }
    }
  }
  return ret;
}

int ObDDLService::deal_default_value_padding(ObColumnSchemaV2 &column_schema, ObIAllocator &allocator)
{
  int ret = OB_SUCCESS;
  ObString str;
  if (column_schema.get_orig_default_value().is_null()
      || column_schema.get_data_type() != ObCharType
      || column_schema.get_collation_type() != CS_TYPE_BINARY) {
    //nothing to do;
  } else if (OB_FAIL(column_schema.get_orig_default_value().get_string(str))) {
    LOG_WARN("fail to get string", K(ret));
  } else {
    int64_t strlen = ObCharset::strlen_char(column_schema.get_collation_type(),
                                            str.ptr(), str.length());
    if (strlen >= column_schema.get_data_length()) {
      //nothing to do
      //check_default_value_length will check length;
    } else {
      char* ptr = NULL;
      int64_t real_size = str.length() + column_schema.get_data_length() - strlen;
      if (NULL == (ptr = static_cast<char *>(allocator.alloc(real_size)))) {
        ret = OB_ALLOCATE_MEMORY_FAILED;
      } else {
        MEMCPY(ptr, str.ptr(), str.length());
        memset(ptr + str.length(), OB_PADDING_BINARY, column_schema.get_data_length() - strlen);
        ObString new_string(real_size, ptr);
        ObObj new_default_value;
        new_default_value.set_binary(new_string);
        if (OB_FAIL(column_schema.set_orig_default_value(new_default_value))) {
          LOG_WARN("fail to set orig default value", K(ret), K(new_default_value));
        }
      }
    }
  }
  return ret;
}

int ObDDLService::pre_check_orig_column_schema(
    const AlterColumnSchema &alter_column_schema,
    const ObTableSchema &origin_table_schema,
    common::hash::ObHashSet<ObColumnNameHashWrapper> &update_column_name_set)
{
  int ret = OB_SUCCESS;
  const ObString &orig_column_name = alter_column_schema.get_origin_column_name();
  const ObColumnSchemaV2 *orig_column_schema = origin_table_schema.get_column_schema(orig_column_name);
  const ObColumnSchemaV2 *column_schema_from_old_table_schema = origin_table_schema.get_column_schema(orig_column_name);
  ObColumnNameHashWrapper orig_column_key(orig_column_name);
  bool is_oracle_mode = false;
  if (NULL == column_schema_from_old_table_schema) {
    ret = OB_ERR_BAD_FIELD_ERROR;
    LOG_USER_ERROR(OB_ERR_BAD_FIELD_ERROR, orig_column_name.length(), orig_column_name.ptr(),
                    origin_table_schema.get_table_name_str().length(),
                    origin_table_schema.get_table_name_str().ptr());
    LOG_WARN("failed to find old column schema", K(ret), K(orig_column_name));
  } else if (OB_HASH_EXIST == update_column_name_set.exist_refactored(orig_column_key)) {
    //column that has been modified, can't not modify again
    ret = OB_ERR_BAD_FIELD_ERROR;
    LOG_USER_ERROR(OB_ERR_BAD_FIELD_ERROR,
                    orig_column_name.length(), orig_column_name.ptr(),
                    origin_table_schema.get_table_name_str().length(),
                    origin_table_schema.get_table_name_str().ptr());
    LOG_WARN("column that has been altered, can't not update again", K(ret));
  } else if (OB_FAIL(origin_table_schema.check_if_oracle_compat_mode(is_oracle_mode))) {
    LOG_WARN("fail to check if tenant mode is oracle mode", K(ret));
  } else if (OB_FAIL(check_generated_column_modify_authority(*orig_column_schema, alter_column_schema, is_oracle_mode))) {
    LOG_WARN("check generated column modify authority", K(ret), KPC(orig_column_schema), K(alter_column_schema));
  }
  return ret;
}

int ObDDLService::check_generated_column_modify_authority(
    const ObColumnSchemaV2 &old_column_schema, const AlterColumnSchema &alter_column_schema, bool is_oracle_mode)
{
  int ret = OB_SUCCESS;
  if (old_column_schema.is_generated_column() && alter_column_schema.is_generated_column()) {
    if ((old_column_schema.is_virtual_generated_column() && alter_column_schema.is_virtual_generated_column())
     || (old_column_schema.is_stored_generated_column() && alter_column_schema.is_stored_generated_column())) {
      ObString old_def;
      ObString alter_def;
      if (OB_FAIL(old_column_schema.get_cur_default_value().get_string(old_def))) {
        LOG_WARN("get old generated column definition failed", K(ret), K(old_column_schema));
      } else if (OB_FAIL(alter_column_schema.get_cur_default_value().get_string(alter_def))) {
        LOG_WARN("get new generated column definition failed", K(ret), K(alter_column_schema));
      } else if (!ObCharset::case_insensitive_equal(old_def, alter_def)) {
        ret = OB_NOT_SUPPORTED;
        LOG_USER_ERROR(OB_NOT_SUPPORTED, "Modify generated column definition");
        LOG_WARN("generated column schema definition changed", K(ret), K(old_column_schema), K(alter_column_schema));
      }
    } else {
      ret = OB_ERR_UNSUPPORTED_ACTION_ON_GENERATED_COLUMN;
      LOG_USER_ERROR(OB_ERR_UNSUPPORTED_ACTION_ON_GENERATED_COLUMN, "Changing the STORED status");
    }
  } else if (old_column_schema.is_generated_column() || alter_column_schema.is_generated_column()) {
    if (is_oracle_mode && old_column_schema.is_generated_column()
        && old_column_schema.get_data_type() != alter_column_schema.get_data_type()) {
      ret = OB_ERR_MODIFY_TYPE_OF_GENCOL;
      LOG_WARN("cannot change the data-type of virtual column without modifying the underlying expression", K(ret), K(old_column_schema), K(alter_column_schema));
    } else if (is_oracle_mode && alter_column_schema.is_generated_column()) {
      ret = OB_ERR_MODIFY_REALCOL_TO_GENCOL;
      LOG_WARN("cannot alter a real column to have an expression", K(ret), K(old_column_schema), K(alter_column_schema));
    } else {
      ret = OB_ERR_UNSUPPORTED_ACTION_ON_GENERATED_COLUMN;
      LOG_USER_ERROR(OB_ERR_UNSUPPORTED_ACTION_ON_GENERATED_COLUMN, "Changing the STORED status");
    }
  }
  return ret;
}

int ObDDLService::update_generated_column_schema(
    const AlterColumnSchema &alter_column_schema,
    const ObColumnSchemaV2 &orig_column_schema,
    const ObTableSchema &origin_table_schema,
    const ObTimeZoneInfoWrap &tz_info_wrap,
    const sql::ObLocalSessionVar *local_session_var,
    ObTableSchema &new_table_schema,
    const bool need_update_default_value,
    const bool need_update_session_var,
    ObDDLOperator *ddl_operator,
    common::ObMySQLTransaction *trans)
{
  int ret = OB_SUCCESS;
  ObTableSchema::const_column_iterator col_iter = new_table_schema.column_begin();
  ObTableSchema::const_column_iterator col_end = new_table_schema.column_end();
  ObColumnSchemaV2 *column = nullptr;
  const bool need_del_stats = false;
  bool for_view = false;
  if (OB_ISNULL(tz_info_wrap.get_time_zone_info())) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("invalid tz info", K(ret));
  }
  for (; OB_SUCC(ret) && col_iter != col_end; col_iter++) {
    if (OB_ISNULL(column = *col_iter)) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("*col_iter is NULL", K(ret));
    } else if (column->has_cascaded_column_id(orig_column_schema.get_column_id())) {
      ObColumnSchemaV2 new_generated_column_schema;
      if (OB_FAIL(new_generated_column_schema.assign(*column))) {
        LOG_WARN("failed to copy new gen column", K(ret));
      } else if (need_update_session_var
                 && OB_FAIL(modify_generated_column_local_vars(new_generated_column_schema,
                                                              orig_column_schema.get_column_name_str(),
                                                              orig_column_schema.get_data_type(),
                                                              alter_column_schema,
                                                              new_table_schema,
                                                              local_session_var))) {
        LOG_WARN("modify local session vars failed", K(ret));
      } else if (need_update_default_value
                 && OB_FAIL(modify_generated_column_default_value(new_generated_column_schema,
                                                              const_cast<ObString&>(orig_column_schema.get_column_name_str()),
                                                               alter_column_schema.get_column_name_str(),
                                                               new_table_schema,
                                                               *tz_info_wrap.get_time_zone_info()))) {
        LOG_WARN("modify generated column value failed", K(ret));
      } else if (OB_FAIL(new_table_schema.alter_column(new_generated_column_schema,
                        ObTableSchema::CHECK_MODE_ONLINE,
                        for_view))) {
        // alter_column must not change column_array_ because we are still looping the column_array_
        LOG_WARN("failed to change column", K(ret));
      } else if (OB_NOT_NULL(ddl_operator) && OB_NOT_NULL(trans)) {
        if (OB_FAIL(ddl_operator->update_single_column(*trans,
                                                       origin_table_schema,
                                                       new_table_schema,
                                                       new_generated_column_schema,
                                                       need_del_stats))) {
          LOG_WARN("generated column failed to alter column", K(ret));
        }
      }
    }
  }
  return ret;
}

int ObDDLService::modify_generated_column_default_value(ObColumnSchemaV2 &generated_column,
                                                         common::ObString &column_name,
                                                         const ObString &new_column_name,
                                                         const ObTableSchema &table_schema,
                                                         const ObTimeZoneInfo &tz_info)
{
  int ret = OB_SUCCESS;
  if (generated_column.is_generated_column()) {
    ObString col_def;
    ObArenaAllocator allocator(ObModIds::OB_SCHEMA);
    ObRawExprFactory expr_factory(allocator);
    SMART_VARS_3((ObSQLSessionInfo, default_session), (ObExecContext, exec_ctx, allocator),
                (ObPhysicalPlanCtx, phy_plan_ctx, allocator)) {
      LinkExecCtxGuard link_guard(default_session, exec_ctx);
      uint64_t tenant_id = table_schema.get_tenant_id();
      const ObTenantSchema *tenant_schema = NULL;
      ObSchemaGetterGuard schema_guard;
      ObRawExpr *expr = NULL;
      lib::Worker::CompatMode compat_mode = lib::Worker::CompatMode::INVALID;

      if (OB_FAIL(default_session.init(0, 0, &allocator))) {
        LOG_WARN("init empty session failed", K(ret));
      } else if (OB_FAIL(schema_service_->get_tenant_schema_guard(tenant_id, schema_guard))) {
        LOG_WARN("get schema guard failed", K(ret));
      } else if (OB_FAIL(schema_guard.get_tenant_info(tenant_id, tenant_schema))) {
        LOG_WARN("get tenant_schema failed", K(ret));
      } else if (OB_FAIL(default_session.init_tenant(tenant_schema->get_tenant_name_str(), tenant_id))) {
        LOG_WARN("init tenant failed", K(ret));
      } else if (OB_FAIL(default_session.load_all_sys_vars(schema_guard))) {
        LOG_WARN("session load system variable failed", K(ret));
      } else if (OB_FAIL(default_session.load_default_configs_in_pc())) {
        LOG_WARN("session load default configs failed", K(ret));
      } else if (OB_FAIL(generated_column.get_cur_default_value().get_string(col_def))) {
        LOG_WARN("get cur default value failed", K(ret));
      } else if (FALSE_IT(exec_ctx.set_physical_plan_ctx(&phy_plan_ctx))) {
      } else if (FALSE_IT(exec_ctx.set_my_session(&default_session))) {
      } else if (OB_FAIL(ObRawExprUtils::build_generated_column_expr(NULL,
                                                                     col_def,
                                                                     expr_factory,
                                                                     default_session,
                                                                     table_schema,
                                                                     expr))) {
        LOG_WARN("build generated column expr failed", K(ret));
      } else if (OB_FAIL(ObCompatModeGetter::get_table_compat_mode(table_schema.get_tenant_id(), table_schema.get_table_id(), compat_mode))) {
        LOG_WARN("failed to get table compat mode", K(ret));
      }
      if (OB_SUCC(ret)) {
        ObRawExprModifyColumnName modifyColumnName(new_column_name, column_name, compat_mode);
        if (OB_FAIL(modifyColumnName.modifyColumnName(*expr))) {
          LOG_WARN("modifyColumnName modify column name failed", K(ret));
        } else {
          SMART_VAR(char[OB_MAX_DEFAULT_VALUE_LENGTH], expr_str_buf) {
            MEMSET(expr_str_buf, 0, sizeof(expr_str_buf));
            ObString expr_def;
            int64_t pos = 0;
            ObObj default_value;
            ObRawExprPrinter expr_printer(expr_str_buf, OB_MAX_DEFAULT_VALUE_LENGTH, &pos, &schema_guard, &tz_info);
            if (OB_FAIL(expr_printer.do_print(expr, T_NONE_SCOPE, true))) {
              LOG_WARN("print expr definition failed", K(ret));
            } else if (FALSE_IT(expr_def.assign_ptr(expr_str_buf, static_cast<int32_t>(pos)))) {
            } else if (FALSE_IT(default_value.set_varchar(expr_def))) {
            } else if (OB_FAIL(generated_column.set_cur_default_value(
                           default_value,
                           generated_column.is_default_expr_v2_column()))) {
              LOG_WARN("set cur default value failed", K(ret));
            } else if (OB_FAIL(generated_column.set_orig_default_value(default_value))) {
              LOG_WARN("set original default value failed", K(ret));
            }
          }
        }
      }
      exec_ctx.set_physical_plan_ctx(NULL);
    }
  }
  return ret;
}

int ObDDLService::modify_generated_column_local_vars(ObColumnSchemaV2 &generated_column,
                                                    const common::ObString &column_name,
                                                    const ObObjType origin_type,
                                                    const AlterColumnSchema &new_column_schema,
                                                    const ObTableSchema &table_schema,
                                                    const sql::ObLocalSessionVar *local_session_var) {
  int ret = OB_SUCCESS;
  if (generated_column.is_generated_column()
      && origin_type != new_column_schema.get_data_type()) {
    ObString col_def;
    ObArenaAllocator allocator(ObModIds::OB_SCHEMA);
    ObRawExprFactory expr_factory(allocator);
    SMART_VARS_3((ObSQLSessionInfo, default_session), (ObExecContext, exec_ctx, allocator),
                 (ObPhysicalPlanCtx, phy_plan_ctx, allocator)) {
      LinkExecCtxGuard link_guard(default_session, exec_ctx);
      uint64_t tenant_id = table_schema.get_tenant_id();
      const ObTenantSchema *tenant_schema = NULL;
      ObSchemaGetterGuard schema_guard;
      ObRawExpr *expr = NULL;
      lib::Worker::CompatMode compat_mode = lib::Worker::CompatMode::INVALID;
      if (OB_FAIL(default_session.init(0, 0, &allocator))) {
        LOG_WARN("init empty session failed", K(ret));
      } else if (OB_FAIL(schema_service_->get_tenant_schema_guard(tenant_id, schema_guard))) {
        LOG_WARN("get schema guard failed", K(ret));
      } else if (OB_FAIL(schema_guard.get_tenant_info(tenant_id, tenant_schema))) {
        LOG_WARN("get tenant_schema failed", K(ret));
      } else if (OB_FAIL(default_session.init_tenant(tenant_schema->get_tenant_name_str(), tenant_id))) {
        LOG_WARN("init tenant failed", K(ret));
      } else if (OB_FAIL(default_session.load_all_sys_vars(schema_guard))) {
        LOG_WARN("session load system variable failed", K(ret));
      } else if (OB_FAIL(default_session.load_default_configs_in_pc())) {
        LOG_WARN("session load default configs failed", K(ret));
      } else if (NULL != local_session_var
                 && OB_FAIL(local_session_var->update_session_vars_with_local(default_session))) {
        LOG_WARN("fail to update session vars", K(ret));
      } else if (FALSE_IT(exec_ctx.set_physical_plan_ctx(&phy_plan_ctx))) {
      } else if (FALSE_IT(exec_ctx.set_my_session(&default_session))) {
      } else if (OB_FAIL(generated_column.get_cur_default_value().get_string(col_def))) {
        LOG_WARN("get cur default value failed", K(ret));
      } else if (OB_FAIL(ObRawExprUtils::build_generated_column_expr(NULL,
                                                                     col_def,
                                                                     expr_factory,
                                                                     default_session,
                                                                     table_schema,
                                                                     expr))) {
        LOG_WARN("build generated column expr failed", K(ret));
      } else if (OB_FAIL(ObCompatModeGetter::get_table_compat_mode(table_schema.get_tenant_id(), table_schema.get_table_id(), compat_mode))) {
        LOG_WARN("failed to get table compat mode", K(ret));
      }
      if (OB_SUCC(ret)) {
        ObRawExpr *expr_with_implicit_cast = NULL;
        ObExprResType dst_type;
        dst_type.set_meta(generated_column.get_meta_type());
        dst_type.set_accuracy(generated_column.get_accuracy());
        dst_type.set_collation_level(CS_LEVEL_IMPLICIT);
        ObSQLMode sql_mode = default_session.get_sql_mode();
        if (NULL != local_session_var) {
          sql::ObSessionSysVar *sys_var = NULL;
          if (OB_FAIL(local_session_var->get_local_var(share::SYS_VAR_SQL_MODE, sys_var))) {
            LOG_WARN("fail to get sys var", K(ret));
          } else if (NULL != sys_var) {
            sql_mode = sys_var->val_.get_uint64();
          }
        }
        if (OB_FAIL(ret)) {
        } else if (OB_FAIL(ObRawExprUtils::erase_operand_implicit_cast(expr, expr))) {
          LOG_WARN("erase implicit cast failed", K(ret));
        } else if (OB_ISNULL(expr)) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("unexpected null", K(ret), KP(expr));
        } else if (OB_FAIL(modify_depend_column_type(expr, column_name, new_column_schema, compat_mode))) {
          LOG_WARN("modify column type failed", K(ret));
        } else if (OB_FAIL(expr->formalize_with_local_vars(&default_session,
                                                          local_session_var,
                                                          OB_INVALID_INDEX_INT64))) {
          LOG_WARN("expr formalize failed", K(ret));
        } else if (OB_FAIL(ObRawExprUtils::try_add_cast_expr_above(&expr_factory, &default_session,
                           *expr, dst_type, expr_with_implicit_cast))) {
          LOG_WARN("try add cast expr above failed", K(ret));
        } else if (OB_ISNULL(expr_with_implicit_cast)) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("unexpected null", K(ret), KP(expr_with_implicit_cast));
        } else if (OB_FAIL(expr_with_implicit_cast->formalize_with_local_vars(&default_session,
                                                                              local_session_var,
                                                                              OB_INVALID_INDEX_INT64))) {
          LOG_WARN("expr formalize failed", K(ret));
        } else if (OB_FAIL(ObRawExprUtils::extract_local_vars_for_gencol(expr_with_implicit_cast,
                                                                         sql_mode,
                                                                         generated_column))) {
          LOG_WARN("extract sysvar from expr failed", K(ret));
        }
      }
      exec_ctx.set_physical_plan_ctx(NULL);
    }
  }
  return ret;
}

int ObDDLService::modify_depend_column_type(sql::ObRawExpr *expr,
                                            const ObString &column_name,
                                            const AlterColumnSchema &column_schema,
                                            lib::Worker::CompatMode compat_mode) {
  int ret = OB_SUCCESS;
  if (OB_ISNULL(expr)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("unexpected null", K(ret));
  } else if (expr->has_flag(IS_COLUMN)) {
    ObColumnRefRawExpr *column_expr = static_cast<ObColumnRefRawExpr *>(expr);
    lib::CompatModeGuard compat_guard(compat_mode);
    if (ObColumnNameHashWrapper(column_expr->get_column_name()) == ObColumnNameHashWrapper(column_name)) {
      column_expr->set_data_type(column_schema.get_data_type());
      column_expr->set_lob_column(is_lob_storage(column_schema.get_data_type()));
      if (ob_is_string_type(column_schema.get_data_type())
        || ob_is_enumset_tc(column_schema.get_data_type())
        || ob_is_json_tc(column_schema.get_data_type())
        || ob_is_geometry_tc(column_schema.get_data_type())) {
        column_expr->set_collation_type(column_schema.get_collation_type());
        column_expr->set_collation_level(CS_LEVEL_IMPLICIT);
      } else {
        column_expr->set_collation_type(CS_TYPE_BINARY);
        column_expr->set_collation_level(CS_LEVEL_NUMERIC);
      }
      if (OB_SUCC(ret)) {
        column_expr->set_accuracy(column_schema.get_accuracy());
        if (column_schema.is_decimal_int()) {
          ObObjMeta data_meta = column_schema.get_meta_type();
          data_meta.set_scale(column_schema.get_accuracy().get_scale());
          column_expr->set_meta_type(data_meta);
        }
        if (OB_FAIL(column_expr->extract_info())) {
          LOG_WARN("extract column expr info failed", K(ret));
        }
      }
      if (OB_SUCC(ret) && (column_schema.is_enum_or_set() || column_schema.is_collection())) {
        if (OB_FAIL(column_expr->set_enum_set_values(column_schema.get_extended_type_info()))) {
          LOG_WARN("failed to set enum set values", K(ret));
        }
      }
      if (OB_SUCC(ret) && column_schema.is_xmltype()) {
      //  column_expr->set_udt_id(column_schema.get_sub_data_type());
      // }
      // ToDo : @gehao, need to conver extend type to udt type?
      //if (OB_SUCC(ret) && column_schema.is_sql_xmltype()) {
        column_expr->set_data_type(ObUserDefinedSQLType);
        column_expr->set_subschema_id(ObXMLSqlType);
        // reset accuracy
      }
    }
  } else if (expr->has_flag(CNT_COLUMN)) {
    for (int64_t i = 0; OB_SUCC(ret) && i < expr->get_param_count(); ++i) {
      if (OB_FAIL(SMART_CALL(modify_depend_column_type(expr->get_param_expr(i), column_name,
                                                       column_schema, compat_mode)))) {
        LOG_WARN("modify depend column type failed", K(ret));
      }
    }
  }
  return ret;
}

int ObDDLService::modify_part_func_expr(const share::schema::ObTableSchema &orig_table_schema,
                                        const share::schema::ObColumnSchemaV2 &orig_column_schema,
                                        const share::schema::AlterColumnSchema &alter_column_schema,
                                        share::schema::ObTableSchema &new_table_schema,
                                        const common::ObTimeZoneInfo &tz_info,
                                        common::ObIAllocator &allocator)
{
  int ret = OB_SUCCESS;
  bool is_partition_key = false;
  if (OB_FAIL(orig_table_schema.is_tbl_partition_key(orig_column_schema,
                                                     is_partition_key,
                                                     false /* ignore_presetting_key */))) {
    LOG_WARN("fail to check is tbl partition key", KR(ret), K(orig_table_schema), K(orig_column_schema));
  } else if (is_partition_key) {
    if (share::schema::PARTITION_FUNC_TYPE_KEY_IMPLICIT != orig_table_schema.get_part_option().get_part_func_type()
        && orig_table_schema.get_part_option().get_part_func_expr_str().empty()) {
      if (orig_table_schema.is_auto_partitioned_table()) {
        // when part_func_expr is empty,
        // auto-partitioned non-partitioned table will choose rowkey for data table and
        // choose index columns for global index as presetting partition key.
        // in this case, it is no need to modify part_func_expr
      } else {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("a partitioned table doesn't have part_fun_expr", KR(ret), K(orig_table_schema));
      }
    } else if (OB_FAIL(modify_part_func_expr(orig_column_schema.get_column_name_str(),
                                             alter_column_schema.get_column_name_str(),
                                             new_table_schema, tz_info, allocator))) {
      LOG_WARN("modify part func expr failed", KR(ret));
    }
  }

  return ret;
}

int ObDDLService::modify_part_func_expr(
    const ObString &orig_column_name,
    const ObString &alter_column_name,
    share::schema::ObTableSchema &table_schema,
    const common::ObTimeZoneInfo &tz_info,
    common::ObIAllocator &allocator)
{
  int ret = OB_SUCCESS;

  if (!table_schema.is_partitioned_table() && !table_schema.is_auto_partitioned_table()) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("invalid part level", KR(ret), K(table_schema.get_part_level()),
                                   K(table_schema.get_part_option()));
  } else if (share::schema::PARTITION_FUNC_TYPE_KEY_IMPLICIT != table_schema.get_part_option().get_part_func_type()
          && table_schema.get_part_option().get_part_func_expr_str().empty()) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("invalid argument", KR(ret), K(table_schema));
  } else if (PARTITION_LEVEL_ZERO == table_schema.get_part_level()
             || PARTITION_LEVEL_ONE == table_schema.get_part_level()) {
    if (OB_FAIL(modify_func_expr_column_name(
                orig_column_name, alter_column_name, table_schema, tz_info, allocator, false))) {
      LOG_WARN("fail to modify func expr column name",
               K(ret), K(orig_column_name), K(alter_column_name), K(table_schema));
    }
  } else if (PARTITION_LEVEL_TWO == table_schema.get_part_level()) {
    if (OB_FAIL(modify_func_expr_column_name(
        orig_column_name, alter_column_name, table_schema, tz_info, allocator, false))) {
      LOG_WARN("fail to modify func expr column name for partition level one",
               K(ret), K(orig_column_name), K(alter_column_name), K(table_schema));
    } else if (OB_FAIL(modify_func_expr_column_name(
               orig_column_name, alter_column_name, table_schema, tz_info, allocator, true))) {
      LOG_WARN("fail to modify func expr column name for partition level two",
               K(ret), K(orig_column_name), K(alter_column_name), K(table_schema));
    }
  }
  return ret;
}

int ObDDLService::modify_func_expr_column_name(
    const ObString &orig_column_name,
    const ObString &alter_column_name,
    ObTableSchema &table_schema,
    const ObTimeZoneInfo &tz_info,
    common::ObIAllocator &allocator,
    bool is_sub_part)
{
  int ret = OB_SUCCESS;
  ObRawExprFactory expr_factory(allocator);
  uint64_t tenant_id = table_schema.get_tenant_id();
  const ObTenantSchema *tenant_schema = NULL;
  ObSchemaGetterGuard schema_guard;
  ObRawExpr *expr = NULL;
  ObArray<ObQualifiedName> columns;
  const ObColumnSchemaV2 *col_schema = NULL;
  ObPartitionOption &part_option = table_schema.get_part_option();
  ObPartitionOption &sub_part_option = table_schema.get_sub_part_option();
  ObString orig_part_expr;
  ObArray<ObString> expr_strs;

  SMART_VAR(ObSQLSessionInfo, default_session) {
    if (!is_sub_part) {
      orig_part_expr = part_option.get_part_func_expr_str();
    } else {
      orig_part_expr = sub_part_option.get_part_func_expr_str();
    }
    if (OB_FAIL(ret)) {
    } else if (part_option.get_part_func_type()
               == share::schema::PARTITION_FUNC_TYPE_KEY_IMPLICIT) {
      // partition by key(), no part func expr need to change
      // do nothing
    } else if (OB_FAIL(default_session.init(0, 0, &allocator))) {
      LOG_WARN("init empty session failed", K(ret));
    } else if (OB_FAIL(schema_service_->get_tenant_schema_guard(tenant_id, schema_guard))) {
      LOG_WARN("get schema guard failed", K(ret));
    } else if (OB_FAIL(schema_guard.get_tenant_info(tenant_id, tenant_schema))) {
      LOG_WARN("get tenant_schema failed", K(ret));
    } else if (OB_FAIL(
                 default_session.init_tenant(tenant_schema->get_tenant_name_str(), tenant_id))) {
      LOG_WARN("init tenant failed", K(ret));
    } else if (OB_FAIL(default_session.load_all_sys_vars(schema_guard))) {
      LOG_WARN("session load system variable failed", K(ret));
    } else if (OB_FAIL(default_session.load_default_configs_in_pc())) {
      LOG_WARN("session load default configs failed", K(ret));
    } else if (orig_part_expr.empty()) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("partition func expr is empty", K(ret));
    } else if (OB_FAIL(split_on(orig_part_expr, ',', expr_strs))) {
      LOG_WARN("fail to split func expr", K(ret), K(orig_part_expr));
    } else {
      char *new_part_func_expr_buf = NULL;
      int64_t outer_pos = 0;
      lib::Worker::CompatMode compat_mode = lib::Worker::CompatMode::INVALID;
      if (OB_ISNULL(new_part_func_expr_buf =
                    static_cast<char *>(allocator.alloc(OB_MAX_SQL_LENGTH)))) {
        ret = OB_ALLOCATE_MEMORY_FAILED;
        LOG_WARN("fail to alloc new_part_func_expr", K(ret));
      } else if (OB_FAIL(ObCompatModeGetter::get_table_compat_mode(table_schema.get_tenant_id(), table_schema.get_table_id(), compat_mode))) {
        LOG_WARN("failed to get table compat mode", K(ret));
      } else {
        ObRawExprModifyColumnName modifyColumnName(
            alter_column_name, orig_column_name, compat_mode);
        for (int64_t i = 0; OB_SUCC(ret) && i < expr_strs.count(); ++i) {
          expr = NULL;
          columns.reset();
          if (OB_FAIL(ObRawExprUtils::build_generated_column_expr(
                      expr_strs.at(i), expr_factory, default_session,
                      expr, columns, &table_schema, false /* allow_sequence */, NULL,
                      NULL, sql::ObResolverUtils::DISABLE_CHECK, false))) {
            LOG_WARN("build generated column expr failed", K(ret));
          }
          for (int64_t i = 0; OB_SUCC(ret) && i <columns.count(); i++) {
            const ObQualifiedName &q_name = columns.at(i);
            if (OB_UNLIKELY(!q_name.database_name_.empty()
                            || OB_UNLIKELY(!q_name.tbl_name_.empty()))) {
              ret = OB_ERR_UNEXPECTED;
              LOG_WARN("invaild generated_column column name", K(q_name));
            } else if (OB_ISNULL(col_schema = table_schema.get_column_schema(q_name.col_name_))) {
              ret = OB_ERR_UNEXPECTED;
              LOG_WARN("column schema is null", K(ret), K(q_name.col_name_));
            } else if (OB_FAIL(ObRawExprUtils::init_column_expr(*col_schema, *q_name.ref_expr_))) {
              LOG_WARN("init column expr failed", K(ret));
            } else {
              q_name.ref_expr_->set_ref_id(table_schema.get_table_id(), col_schema->get_column_id());
            }
          }
          if (OB_FAIL(ret)) {
          } else if (OB_FAIL(modifyColumnName.modifyColumnName(*expr))) {
            LOG_WARN("modifyColumnName modify column name failed", K(ret));
          } else {
            SMART_VAR(char[OB_MAX_SQL_LENGTH], expr_str_buf) {
              MEMSET(expr_str_buf, 0, sizeof(expr_str_buf));
              int64_t inner_pos = 0;
              ObRawExprPrinter expr_printer(expr_str_buf, OB_MAX_SQL_LENGTH, &inner_pos, &schema_guard, &tz_info);
              if (OB_FAIL(expr_printer.do_print(expr, T_NONE_SCOPE, true))) {
                LOG_WARN("print expr definition failed", K(ret));
              } else if (0 == i
                         && OB_FAIL(databuff_printf(new_part_func_expr_buf,
                                                    OB_MAX_SQL_LENGTH,
                                                    outer_pos,
                                                    "%.*s",
                                                    static_cast<int>(inner_pos),
                                                    expr_str_buf))) {
                LOG_WARN("fail to print expr_str_buf", K(ret), K(i), K(expr_str_buf));
              } else if (0 != i
                         && OB_FAIL(databuff_printf(new_part_func_expr_buf,
                                                    OB_MAX_SQL_LENGTH,
                                                    outer_pos,
                                                    ", %.*s",
                                                    static_cast<int>(inner_pos),
                                                    expr_str_buf))) {
                LOG_WARN("fail to print expr_str_buf", K(ret), K(i), K(expr_str_buf));
              }
            }
          }
        }
        if (OB_SUCC(ret)) {
          ObString new_part_func_expr_str;
          if (FALSE_IT(new_part_func_expr_str.assign_ptr(
                      new_part_func_expr_buf, static_cast<int32_t>(outer_pos)))) {
          } else if (!is_sub_part && OB_FAIL(part_option.set_part_expr(new_part_func_expr_str))) {
            LOG_WARN("set part expr failed", K(ret));
          } else if (is_sub_part && OB_FAIL(sub_part_option.set_part_expr(new_part_func_expr_str))) {
            LOG_WARN("set sub part expr failed", K(ret));
          }
        }
      }
    }
  }

  return ret;
}

int ObDDLService::modify_part_func_expr_for_global_index(
    const share::schema::ObColumnSchemaV2 &orig_column,
    const share::schema::ObColumnSchemaV2 &alter_column,
    share::schema::ObTableSchema &new_table_schema,
    const common::ObTimeZoneInfo &tz_info,
    common::ObIAllocator &allocator,
    ObDDLOperator *ddl_operator,
    common::ObMySQLTransaction *trans,
    ObIArray<ObTableSchema> *global_idx_schema_array/*=NULL*/)
{
  int ret = OB_SUCCESS;
  const bool update_inner_table = nullptr != ddl_operator && nullptr != trans;
  ObSchemaGetterGuard schema_guard;
  const uint64_t tenant_id = new_table_schema.get_tenant_id();
  ObSchemaService *schema_service = schema_service_->get_schema_service();
  ObSEArray<ObAuxTableMetaInfo, 16> simple_index_infos;

  if (OB_ISNULL(schema_service)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("schema_service is NULL", K(ret));
  } else if (OB_FAIL(schema_service_->get_tenant_schema_guard(tenant_id, schema_guard))) {
    LOG_WARN("get schema guard failed", K(ret));
  } else if (OB_FAIL(new_table_schema.get_simple_index_infos(simple_index_infos))) {
    LOG_WARN("get simple_index_infos failed", K(ret));
  } else {
    const ObTableSchema *index_schema = NULL;
    for (int64_t i = 0; OB_SUCC(ret) && i < simple_index_infos.count(); ++i) {
      if (OB_FAIL(schema_guard.get_table_schema(
                  tenant_id, simple_index_infos.at(i).table_id_, index_schema))) {
        LOG_WARN("get table_schema failed", K(ret), K(tenant_id), K(simple_index_infos.at(i).table_id_));
      } else if (OB_ISNULL(index_schema)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("index schema should not be null", K(ret));
      } else if (index_schema->is_in_recyclebin()) {
        ret = OB_ERR_OPERATION_ON_RECYCLE_OBJECT;
        LOG_WARN("index table is in recyclebin", K(ret));
      } else if (index_schema->is_global_index_table()
                 && index_schema->is_partitioned_table()) {
        const ObColumnSchemaV2 *origin_column_schema =
              index_schema->get_column_schema(orig_column.get_column_id());
        if (OB_ISNULL(origin_column_schema)) {
          // skip, this column is not in global index
        } else if (origin_column_schema->is_tbl_part_key_column()) {
          ObTableSchema new_index_schema;
          if (OB_FAIL(new_index_schema.assign(*index_schema))) {
            LOG_WARN("assign index_schema failed", K(ret));
          } else if (OB_FAIL(modify_part_func_expr(origin_column_schema->get_column_name_str(),
                      alter_column.get_column_name_str(), new_index_schema, tz_info, allocator))) {
            LOG_WARN("modify part func expr failed", K(ret));
          } else if (update_inner_table && OB_FAIL(ddl_operator->update_partition_option(*trans, new_index_schema))) {
            LOG_WARN("update partition option failed",
                     K(ret), K(new_index_schema), K(*index_schema));
          } else if (nullptr != global_idx_schema_array && OB_FAIL(global_idx_schema_array->push_back(new_index_schema))) {
            LOG_WARN("fail to push_back to global_idx_schema_array",
                     K(ret), K(new_index_schema));
          }
        }
      } else {
        // skip
      }
    } // end of for
  } // end of else
  return ret;
}

int ObDDLService::modify_constraint_check_expr(
    const share::schema::ObColumnSchemaV2 &orig_column,
    const share::schema::ObColumnSchemaV2 &alter_column,
    share::schema::ObTableSchema &table_schema,
    obrpc::ObAlterTableArg &alter_table_arg,
    const bool is_oracle_mode,
    const common::ObTimeZoneInfo &tz_info,
    common::ObIAllocator &allocator,
    ObDDLOperator *ddl_operator,
    common::ObMySQLTransaction *trans)
{
  int ret = OB_SUCCESS;
  const bool update_inner_table = nullptr != ddl_operator && nullptr != trans;
  bool need_modify_check_expr = false;
  ObString new_check_expr_str;
  ObTableSchema::const_constraint_iterator iter = table_schema.constraint_begin();

  for (; OB_SUCC(ret) && iter != table_schema.constraint_end(); ++iter) {
    need_modify_check_expr = false;
    if ((*iter)->get_check_expr_str().empty()) {
      continue;
    }
    // -- mysql mode, should succeed
    // drop table t1;
    // CREATE TABLE t1 (col1 INT primary key, col2 INT);
    // ALTER TABLE t1 ADD CONSTRAINT my_check3 CHECK (col1+col2> col1+1);
    // ALTER TABLE t1 drop CONSTRAINT my_check3, rename column col2 to col3,rename column col1 to col5;
    //
    // if corresponding constraint is being dropped, columns are allowed to be renamed, and there's no
    // need to rebuild constraint expression.
    bool dropped_cst = false;
    if (!is_oracle_mode
        && alter_table_arg.alter_constraint_type_ == obrpc::ObAlterTableArg::DROP_CONSTRAINT) {
      for (auto cst_iter = alter_table_arg.alter_table_schema_.constraint_begin();
           !dropped_cst && cst_iter != alter_table_arg.alter_table_schema_.constraint_end();
           cst_iter++) {
        if ((*cst_iter)->get_constraint_name_str().case_compare((*iter)->get_constraint_name_str())
            == 0) {
          dropped_cst = true;
        }
      }
    }
    if (dropped_cst) {
    } else if (OB_FAIL(rebuild_constraint_check_expr(orig_column, alter_column, **iter, table_schema,
                tz_info, allocator, new_check_expr_str, need_modify_check_expr))) {
      LOG_WARN("fail to gen constraint check expr", K(ret));
    } else if (need_modify_check_expr) {
      (*iter)->set_check_expr(new_check_expr_str);
      (*iter)->set_is_modify_check_expr(true);
      if (update_inner_table && OB_FAIL(ddl_operator->update_check_constraint_state(*trans,
                  table_schema, **iter))) {
        LOG_WARN("update check expr constraint failed", K(ret));
      }
    }
  }
  return ret;
}

// Oracle mode, to refill column id for not null constraint that is referenced to newly-added column.
// Case: alter table table_name add column_name not null.
int ObDDLService::refill_columns_id_for_not_null_constraint(
    const share::schema::ObTableSchema &alter_table_schema,
    const share::schema::ObColumnSchemaV2 &alter_column_schema)
{
  int ret = OB_SUCCESS;
  ObTableSchema::constraint_iterator target_cst_iter = nullptr;
  if (alter_column_schema.has_not_null_constraint()) {
    ObTableSchema::constraint_iterator cst_iter = alter_table_schema.constraint_begin_for_non_const_iter();
    for (; NULL == target_cst_iter && OB_SUCC(ret) &&
          cst_iter != alter_table_schema.constraint_end_for_non_const_iter();
          cst_iter++) {
      if (OB_ISNULL(*cst_iter)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("error unexpected", K(ret));
      } else if (CONSTRAINT_TYPE_NOT_NULL == (*cst_iter)->get_constraint_type()) {
        ObString cst_col_name;
        if (OB_FAIL((*cst_iter)->get_not_null_column_name(cst_col_name))) {
          LOG_WARN("get not null cst column name failed", K(ret));
        } else if (alter_column_schema.get_column_name_str().length() == cst_col_name.length()
            && 0 == alter_column_schema.get_column_name_str().compare(cst_col_name)) {
          target_cst_iter = cst_iter;
        }
        LOG_DEBUG("alter table add column not null", K((*cst_iter)->get_check_expr_str()), K(cst_col_name),
        K(alter_column_schema.get_column_name_str()));
      }
    }
    ObSEArray<uint64_t, 1> column_id_array;
    if (OB_FAIL(ret)) {
    } else if (OB_ISNULL(target_cst_iter)) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("constraint not found for added not null column", K(ret), K(alter_column_schema));
    } else if (OB_FAIL(column_id_array.push_back(alter_column_schema.get_column_id()))) {
      LOG_WARN("push back failed", K(ret));
    } else if (OB_FAIL((*target_cst_iter)->assign_column_ids(column_id_array))) {
      LOG_WARN("assign column ids failed", K(ret));
    }
  }
  return ret;
}

// column_id_array is invalid if newly-added check constraint is referenced to
// newly-added column, due to invalid column id in resolver phase.
// Case: alter table table_name add column_name constraint check(column_name < 10) check(column_name > 0).
int ObDDLService::refill_columns_id_for_check_constraint(
    const share::schema::ObTableSchema &orig_table_schema,
    const share::schema::ObTableSchema &alter_table_schema,
    const share::schema::ObColumnSchemaV2 &alter_column_schema,
    const bool is_oracle_mode,
    common::ObIAllocator &allocator)
{
  int ret = OB_SUCCESS;
  bool need_refill_columns_id = false;
  ObTableSchema::const_constraint_iterator check_cst_iter = nullptr;
  ObTableSchema::constraint_iterator target_cst_iter = nullptr;
  // check whether there are some unexpected case firstly.
  for (check_cst_iter = alter_table_schema.constraint_begin();
      OB_SUCC(ret) && check_cst_iter != alter_table_schema.constraint_end(); check_cst_iter++) {
    if (OB_ISNULL(*check_cst_iter)) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("error unexpected", K(ret));
    } else if (CONSTRAINT_TYPE_CHECK == (*check_cst_iter)->get_constraint_type()) {
      if (is_oracle_mode) {
        // set error code here to avoid forgetting refill columns id
        // under oracle mode if supported in the future.
        ret = OB_NOT_SUPPORTED;
        LOG_WARN("add column and alter check constraint in single stmt is not supported", K(ret), K(alter_table_schema));
      } else if (1 == (*check_cst_iter)->get_column_cnt()
        && nullptr == orig_table_schema.get_column_schema(*((*check_cst_iter)->cst_col_begin()))) {
        // Currently, newly-added check constraint is referenced to newly-added column under Mysql mode.
        need_refill_columns_id = true;
      }
    }
  }
  // refill columns id array.
  if (OB_SUCC(ret) && need_refill_columns_id) {
    ObRawExprFactory expr_factory(allocator);
    uint64_t tenant_id = alter_column_schema.get_tenant_id();
    const ObTenantSchema *tenant_schema = nullptr;
    ObSchemaGetterGuard schema_guard;
    SMART_VAR(ObSQLSessionInfo, default_session) {
      if (OB_FAIL(default_session.init(0, 0, &allocator))) {
        LOG_WARN("init empty session failed", K(ret));
      } else if (OB_FAIL(schema_service_->get_tenant_schema_guard(tenant_id, schema_guard))) {
        LOG_WARN("get schema guard failed", K(ret), K(tenant_id));
      } else if (OB_FAIL(schema_guard.get_tenant_info(tenant_id, tenant_schema))) {
        LOG_WARN("get tenant_schema failed", K(ret), K(tenant_id));
      } else if (OB_FAIL(default_session.init_tenant(tenant_schema->get_tenant_name_str(), tenant_id))) {
        LOG_WARN("init tenant failed", K(ret), K(tenant_id));
      } else if (OB_FAIL(default_session.load_all_sys_vars(schema_guard))) {
        LOG_WARN("session load system variable failed", K(ret));
      } else if (OB_FAIL(default_session.load_default_configs_in_pc())) {
        LOG_WARN("session load default configs failed", K(ret));
      } else {
        for (target_cst_iter = alter_table_schema.constraint_begin_for_non_const_iter();
          OB_SUCC(ret) && target_cst_iter != alter_table_schema.constraint_end_for_non_const_iter(); target_cst_iter++) {
          if (OB_ISNULL(*target_cst_iter)) {
            ret = OB_ERR_UNEXPECTED;
            LOG_WARN("error unexpected", K(ret));
          } else if (CONSTRAINT_TYPE_CHECK == (*target_cst_iter)->get_constraint_type()
              && 1 == (*target_cst_iter)->get_column_cnt()
              && nullptr == orig_table_schema.get_column_schema(*((*target_cst_iter)->cst_col_begin()))) {
            ObString check_expr = (*target_cst_iter)->get_check_expr_str();
            ObArray<ObQualifiedName> columns;
            ObSEArray<uint64_t, 1> column_id_array;
            ObRawExpr *expr = nullptr;
            const ParseNode *node = nullptr;
            if (check_expr.empty()) {
              ret = OB_ERR_UNEXPECTED;
              LOG_WARN("check expr is empty", K(ret));
            } else if (OB_FAIL(ObRawExprUtils::parse_bool_expr_node_from_str(check_expr,
                                                                            expr_factory.get_allocator(), node))) {
              LOG_WARN("parse expr node from string failed", K(ret));
            } else if (OB_ISNULL(node)) {
              ret = OB_ERR_UNEXPECTED;
              LOG_WARN("node is null", K(ret));
            } else if (OB_FAIL(ObRawExprUtils::build_check_constraint_expr(expr_factory,
                                                                          default_session, *node, expr, columns))) {
              LOG_WARN("build generated column expr failed", K(ret), K(check_expr));
            } else if (1 != columns.count()) {
              ret = OB_ERR_UNEXPECTED;
              LOG_WARN("unexpected column count of the new-built check constraint", K(ret), K(check_expr));
            } else if (0 != alter_column_schema.get_column_name_str().case_compare(columns.at(0).col_name_)) {
              // may be newly-added column without check constraint, do nothing.
            } else if (OB_FAIL(column_id_array.push_back(alter_column_schema.get_column_id()))) {
              LOG_WARN("push back failed", K(ret));
            } else if (OB_FAIL((*target_cst_iter)->assign_column_ids(column_id_array))) {
              LOG_WARN("assign column ids failed", K(ret));
            }
          }
        }
      }
    }
  }
  return ret;
}

int ObDDLService::refill_column_id_array_for_constraint(
    const ObAlterTableArg::AlterConstraintType op_type,
    const ObTableSchema &new_table_schema,
    AlterTableSchema &alter_table_schema)
{
  int ret = OB_SUCCESS;
  if (obrpc::ObAlterTableArg::ADD_CONSTRAINT == op_type) {
    ObTableSchema::constraint_iterator cst_iter = alter_table_schema.constraint_begin_for_non_const_iter();
    for (; OB_SUCC(ret) && cst_iter != alter_table_schema.constraint_end_for_non_const_iter(); cst_iter++) {
      if (OB_ISNULL(*cst_iter)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("error unexpected", K(ret));
      } else if (CONSTRAINT_TYPE_NOT_NULL == (*cst_iter)->get_constraint_type()) {
        ObString cst_col_name;
        const ObColumnSchemaV2 *column = nullptr;
        if (OB_FAIL((*cst_iter)->get_not_null_column_name(cst_col_name))) {
          LOG_WARN("get not null cst column name failed", K(ret), KPC(*cst_iter));
        } else if (OB_ISNULL(column = new_table_schema.get_column_schema(cst_col_name))) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("no column in new table schema", K(ret), K(cst_col_name));
        } else if (OB_FAIL((*cst_iter)->assign_not_null_cst_column_id(column->get_column_id()))) {
          LOG_WARN("failed to assign not null cst column id", K(ret), KPC(column));
        }
      }
    }
  }
  return ret;
}

int ObDDLService::rebuild_constraint_check_expr(
    const share::schema::ObColumnSchemaV2 &orig_column,
    const share::schema::ObColumnSchemaV2 &alter_column,
    const share::schema::ObConstraint &cst,
    share::schema::ObTableSchema &table_schema,
    const common::ObTimeZoneInfo &tz_info,
    common::ObIAllocator &allocator,
    ObString &new_check_expr_str,
    bool &need_modify_check_expr)
{
  int ret = OB_SUCCESS;
  ObRawExprFactory expr_factory(allocator);
  uint64_t tenant_id = orig_column.get_tenant_id();
  const ObTenantSchema *tenant_schema = NULL;
  ObSchemaGetterGuard schema_guard;
  ObRawExpr *expr = NULL;
  const ParseNode *node = NULL;
  ObArray<ObQualifiedName> columns;
  const ObColumnSchemaV2 *col_schema = NULL;
  ObString orig_check_expr = cst.get_check_expr_str();
  SMART_VAR(ObSQLSessionInfo, default_session) {
    if (OB_FAIL(default_session.init(0, 0, &allocator))) {
      LOG_WARN("init empty session failed", K(ret));
    } else if (OB_FAIL(schema_service_->get_tenant_schema_guard(tenant_id, schema_guard))) {
      LOG_WARN("get schema guard failed", K(ret));
    } else if (OB_FAIL(schema_guard.get_tenant_info(tenant_id, tenant_schema))) {
      LOG_WARN("get tenant_schema failed", K(ret));
    } else if (OB_FAIL(default_session.init_tenant(tenant_schema->get_tenant_name_str(), tenant_id))) {
      LOG_WARN("init tenant failed", K(ret));
    } else if (OB_FAIL(default_session.load_all_sys_vars(schema_guard))) {
      LOG_WARN("session load system variable failed", K(ret));
    } else if (OB_FAIL(default_session.load_default_configs_in_pc())) {
      LOG_WARN("session load default configs failed", K(ret));
    } else if (orig_check_expr.empty()) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("check expr is empty", K(ret));
    } else {
      char *new_check_expr_buf = NULL;
      int64_t outer_pos = 0;
      if (OB_ISNULL(new_check_expr_buf =
          static_cast<char *>(allocator.alloc(OB_MAX_SQL_LENGTH)))) {
        ret = OB_ALLOCATE_MEMORY_FAILED;
        LOG_WARN("fail to alloc new_check_expr_buf", K(ret));
      } else {
        char *new_check_expr_buf = NULL;
        int64_t outer_pos = 0;
        lib::Worker::CompatMode compat_mode = lib::Worker::CompatMode::INVALID;
        if (OB_ISNULL(new_check_expr_buf =
                      static_cast<char *>(allocator.alloc(OB_MAX_SQL_LENGTH)))) {
          ret = OB_ALLOCATE_MEMORY_FAILED;
          LOG_WARN("fail to alloc new_check_expr_buf", K(ret));
        } else if (OB_FAIL(ObCompatModeGetter::get_table_compat_mode(table_schema.get_tenant_id(), table_schema.get_table_id(), compat_mode))) {
          LOG_WARN("failed to get table compat mode", K(ret));
        } else {
          ObRawExprModifyColumnName modifyColumnName(
              alter_column.get_column_name_str(), orig_column.get_column_name_str(), compat_mode);
          if (OB_FAIL(ObRawExprUtils::parse_bool_expr_node_from_str(orig_check_expr,
                                                                    expr_factory.get_allocator(), node))) {
            LOG_WARN("parse expr node from string failed", K(ret));
          } else if (OB_ISNULL(node)) {
            ret = OB_ERR_UNEXPECTED;
            LOG_WARN("node is null", K(ret));
          } else if (OB_FAIL(ObRawExprUtils::build_check_constraint_expr(expr_factory,
                                                                         default_session, *node, expr, columns))) {
            LOG_WARN("build generated column expr failed", K(ret), K(orig_check_expr));
          }
          for (int64_t i = 0; OB_SUCC(ret) && i <columns.count(); i++) {
            const ObQualifiedName &q_name = columns.at(i);
            if (0 == orig_column.get_column_name_str().
                case_compare(q_name.col_name_)) {
              need_modify_check_expr = true;
            }
            if (OB_UNLIKELY(!q_name.database_name_.empty()
                            || OB_UNLIKELY(!q_name.tbl_name_.empty()))) {
              ret = OB_ERR_UNEXPECTED;
              LOG_WARN("invalid generated_column column name", K(q_name));
            } else if (OB_ISNULL(col_schema = table_schema.get_column_schema(q_name.col_name_))) {
              ret = OB_ERR_UNEXPECTED;
              LOG_WARN("column schema is null", K(ret), K(q_name.col_name_));
            } else if (OB_FAIL(ObRawExprUtils::init_column_expr(*col_schema, *q_name.ref_expr_))) {
              LOG_WARN("init column expr failed", K(ret), K((*col_schema).get_column_name_str()));
            } else {
              q_name.ref_expr_->set_ref_id(table_schema.get_table_id(), col_schema->get_column_id());
            }
          }
          if (OB_FAIL(ret)) {
          } else if (OB_FAIL(modifyColumnName.modifyColumnName(*expr))) {
            LOG_WARN("modifyColumnName modify column name failed", K(ret));
          } else {
            SMART_VAR(char[OB_MAX_SQL_LENGTH], expr_str_buf) {
              MEMSET(expr_str_buf, 0, sizeof(expr_str_buf));
              int64_t inner_pos = 0;
              ObRawExprPrinter expr_printer(expr_str_buf, OB_MAX_SQL_LENGTH, &inner_pos, &schema_guard, &tz_info);
              if (OB_FAIL(expr_printer.do_print(expr, T_NONE_SCOPE, true))) {
                LOG_WARN("print expr definition failed", K(ret));
              } else if (OB_FAIL(databuff_printf(new_check_expr_buf,
                                                 OB_MAX_SQL_LENGTH,
                                                 outer_pos,
                                                 "%.*s",
                                                 static_cast<int>(inner_pos),
                                                 expr_str_buf))) {
                LOG_WARN("fail to print expr_str_buf", K(ret), K(expr_str_buf));
              }
            }
          }
          if (OB_SUCC(ret)) {
            new_check_expr_str.assign_ptr(new_check_expr_buf, static_cast<int32_t>(outer_pos));
          }
        }
      }
    }
  }
  return ret;
}

int ObDDLService::check_can_alter_column_type(
    const share::schema::ObColumnSchemaV2 &src_column,
    const share::schema::ObColumnSchemaV2 &dst_column,
    const share::schema::ObTableSchema &table_schema,
    const bool is_oracle_mode)
{
  int ret = OB_SUCCESS;
  uint64_t data_version = 0;
  bool is_change_column_type = false;
  bool is_in_index = false;
  bool has_generated_depend = src_column.has_generated_column_deps();
  if (OB_FAIL(check_is_change_column_type(src_column, dst_column, is_change_column_type))) {
    LOG_WARN("fail to check is change column type", K(ret), K(src_column), K(dst_column));
  } else if (is_change_column_type) {
    if (OB_FAIL(check_column_in_index(src_column.get_column_id(), table_schema, is_in_index))) {
      LOG_WARN("fail to check column is in index table", K(ret));
    } else if (OB_FAIL(GET_MIN_DATA_VERSION(table_schema.get_tenant_id(), data_version))) {
      LOG_WARN("fail to get data version", KR(ret), K(table_schema.get_tenant_id()));
    } else if (is_in_index || has_generated_depend) {
      // is_in_index==true means : src_column is 'the index column' or 'index create by user'.
      const common::ObObjMeta &src_meta = src_column.get_meta_type();
      const common::ObObjMeta &dst_meta = dst_column.get_meta_type();
      if (is_oracle_mode) {        // in oracle mode
        if (is_in_index
          && data_version >= DATA_VERSION_4_3_0_1
          && common::ObNumberType == src_column.get_meta_type().get_type() && common::ObNumberFloatType == dst_column.get_meta_type().get_type()) {
          // support number -> float in oracle mode in version 4.3
        } else {
          ret = OB_NOT_SUPPORTED;
          LOG_WARN("cannot modify column in index table", K(ret), K(src_column), K(dst_column), K(table_schema));
        }
      } else {
        // in mysql mode
        if (((has_generated_depend && !is_in_index) || data_version >= DATA_VERSION_4_2_2_0)
            && common::is_match_alter_integer_column_online_ddl_rules(src_meta, dst_meta)) {
            // support online ddl with index, generated column depended:
            // smaller integer -> big integer in mysql mode in version 4.2.2
        } else if (((has_generated_depend && !is_in_index) || data_version >= DATA_VERSION_4_3_0_0)
            && common::is_match_alter_string_column_online_ddl_rules(src_meta, dst_meta, src_column.get_data_length(), dst_column.get_data_length())) {
          // generated column type change of: varchar <-> tinytext; varbinary <-> tinyblob; smaller lob -> larger lob; (4.3.0.0)
        } else {
          ret = OB_NOT_SUPPORTED;
          LOG_WARN("cannot modify column in index table", K(ret), K(src_column), K(dst_column), K(table_schema), K(data_version));
        }
      }
    }
  }
  return ret;
}

int ObDDLService::check_is_change_column_type(
    const share::schema::ObColumnSchemaV2 &src_column,
    const share::schema::ObColumnSchemaV2 &dst_column,
    bool &is_change_column_type)
{
  int ret = OB_SUCCESS;
  is_change_column_type = src_column.get_data_type() != dst_column.get_data_type();
  LOG_INFO("check is change column type", K(is_change_column_type), K(src_column), K(dst_column));
  return ret;
}

int ObDDLService::check_column_in_index(
    const uint64_t column_id,
    const share::schema::ObTableSchema &table_schema,
    bool &is_in_index)
{
  int ret = OB_SUCCESS;
  const uint64_t tenant_id = table_schema.get_tenant_id();
  ObSchemaGetterGuard schema_guard;
  ObArray<ObColDesc> column_ids;
  is_in_index = false;
  if (OB_UNLIKELY(!table_schema.is_valid())) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("invalid arguments", K(ret), K(table_schema));
  } else if (OB_FAIL(schema_service_->get_tenant_schema_guard(tenant_id, schema_guard))) {
    LOG_WARN("fail to get schema guard", K(ret));
  } else if (OB_FAIL(table_schema.get_rowkey_column_ids(column_ids))) {
    LOG_WARN("fail to get rowkey column ids", K(ret));
  } else {
    for (int64_t i = 0; OB_SUCC(ret) && i < column_ids.count() && !is_in_index; ++i) {
      if (column_id == column_ids.at(i).col_id_) {
        is_in_index = true;
        LOG_WARN("column in data table rowkey columns", K(column_id));
      }
    }
    if (OB_SUCC(ret) && !is_in_index) {
      ObSEArray<ObAuxTableMetaInfo, 16> simple_index_infos;
      if (OB_FAIL(table_schema.get_simple_index_infos(simple_index_infos))) {
        LOG_WARN("get simple_index_infos failed", K(ret));
      }
      for (int64_t i = 0; OB_SUCC(ret) && i < simple_index_infos.count() && !is_in_index; ++i) {
        const ObTableSchema *index_table_schema = NULL;
        if (OB_FAIL(schema_guard.get_table_schema(tenant_id, simple_index_infos.at(i).table_id_, index_table_schema))) {
          LOG_WARN("fail to get table schema",  K(ret), K(tenant_id), "table id", simple_index_infos.at(i).table_id_);
        } else if (OB_ISNULL(index_table_schema)) {
          ret = OB_TABLE_NOT_EXIST;
          LOG_WARN("index table schema must not be NULL", K(ret));
        } else {
          column_ids.reuse();
          if (OB_FAIL(index_table_schema->get_column_ids(column_ids))) {
            LOG_WARN("fail to get column ids", K(ret));
          }
          for (int64_t j = 0; OB_SUCC(ret) && !is_in_index && j < column_ids.count(); ++j) {
            const ObColumnSchemaV2 *column_schema = NULL;
            if (column_id == column_ids.at(j).col_id_) {
              is_in_index = true;
            } else if (column_ids.at(j).col_id_ > common::OB_MIN_SHADOW_COLUMN_ID) {
              // table schema get column_schema by shadow_column_id, the result is nullptr;
            } else if (OB_ISNULL(column_schema = table_schema.get_column_schema(column_ids.at(j).col_id_))) {
              ret = OB_SCHEMA_ERROR;
              LOG_WARN("column schema must not be NULL", K(ret), K(column_ids.at(j)));
            } else if (column_schema->is_generated_column()) {
              ObArray<uint64_t> ref_column_ids;
              if (OB_FAIL(column_schema->get_cascaded_column_ids(ref_column_ids))) {
                LOG_WARN("fail to get cascade column ids", K(ret));
              } else {
                /*
                  If the column (namely A) is depended by a generated column, and users build an index on the generated column.
                  We consider the column (A) is in index.
                */
                is_in_index = has_exist_in_array(ref_column_ids, column_id);
              }
            }
          }
        }
      }
    }
  }
  return ret;
}

int ObDDLService::fill_new_column_attributes(
    const AlterColumnSchema &alter_column_schema,
    ObColumnSchemaV2 &new_column_schema)
{
  int ret = OB_SUCCESS;
  int64_t new_schema_version = OB_INVALID_VERSION;
  const uint64_t tenant_id = new_column_schema.get_tenant_id();
  if (OB_FAIL(schema_service_->gen_new_schema_version(tenant_id, new_schema_version))) {
    LOG_WARN("fail to gen new schema_version", K(ret), K(tenant_id));
  } else {
    new_column_schema.set_column_name(alter_column_schema.get_column_name_str());
    new_column_schema.set_charset_type(alter_column_schema.get_charset_type());
    new_column_schema.set_collation_type(alter_column_schema.get_collation_type());
    new_column_schema.set_data_type(alter_column_schema.get_data_type());
    new_column_schema.set_data_length(alter_column_schema.get_data_length());
    new_column_schema.set_data_precision(alter_column_schema.get_data_precision());
    new_column_schema.set_data_scale(alter_column_schema.get_data_scale());
    if (!is_oracle_mode() || alter_column_schema.is_set_default_) {
      new_column_schema.set_cur_default_value(
          alter_column_schema.get_cur_default_value(),
          alter_column_schema.is_default_expr_v2_column());
    }
    new_column_schema.set_zero_fill(alter_column_schema.is_zero_fill());
    new_column_schema.set_is_hidden(alter_column_schema.is_hidden());
    new_column_schema.set_nullable(alter_column_schema.is_nullable());
    new_column_schema.set_autoincrement(alter_column_schema.is_autoincrement());
    new_column_schema.set_column_flags(alter_column_schema.get_column_flags());
    new_column_schema.set_comment(alter_column_schema.get_comment_str());
    new_column_schema.set_schema_version(new_schema_version);
    new_column_schema.set_on_update_current_timestamp(
        alter_column_schema.is_on_update_current_timestamp());
    new_column_schema.set_extended_type_info(alter_column_schema.get_extended_type_info());
    new_column_schema.set_srs_id(alter_column_schema.get_srs_id());
    new_column_schema.set_skip_index_attr(alter_column_schema.get_skip_index_attr().get_packed_value());
    new_column_schema.set_lob_chunk_size(alter_column_schema.get_lob_chunk_size());
    if (OB_FAIL(new_column_schema.get_local_session_var().deep_copy(alter_column_schema.get_local_session_var()))) {
      LOG_WARN("deep copy local session vars failed", K(ret));
    }
  }
  return ret;
}

int ObDDLService::check_modify_column_when_upgrade(
    const share::schema::ObColumnSchemaV2 &new_column,
    const share::schema::ObColumnSchemaV2 &orig_column)
{
  int ret = OB_SUCCESS;
  if (obrpc::OB_UPGRADE_STAGE_DBUPGRADE != GCTX.get_upgrade_stage()) {
    // do nothing
  } else {
    ObColumnSchemaV2 tmp_column;
    if (OB_FAIL(tmp_column.assign(new_column))) {
      LOG_WARN("assign failed", K(ret), K(new_column));
    } else {
      tmp_column.set_schema_version(orig_column.get_schema_version());
      tmp_column.set_data_length(orig_column.get_data_length());
      tmp_column.set_data_precision(orig_column.get_data_precision());
      tmp_column.set_data_scale(orig_column.get_data_scale());
      if (tmp_column != orig_column) {
        ret = OB_OP_NOT_ALLOW;
        LOG_WARN("can only modify column's length", K(ret), K(new_column), K(orig_column));
      } else if (new_column.get_data_length() < orig_column.get_data_length()
                 || new_column.get_data_precision() < orig_column.get_data_precision()
                 || new_column.get_data_scale() < orig_column.get_data_scale()) {
        ret = OB_OP_NOT_ALLOW;
        LOG_WARN("can only increase column's length", K(ret), K(new_column), K(orig_column));
      }
    }
  }
  return ret;
}
int ObDDLService::alter_shadow_column_for_index(
    const ObArray<ObTableSchema> &idx_schema_array,
    const AlterColumnSchema *alter_column_schema,
    const ObColumnSchemaV2 &new_column_schema,
    ObDDLOperator &ddl_operator,
    common::ObMySQLTransaction &trans)
{
  int ret = OB_SUCCESS;
  bool need_del_stats = false;
  if (OB_ISNULL(alter_column_schema)) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("invalid argument", K(ret), KP(alter_column_schema));
  } else if (!new_column_schema.is_rowkey_column()) {
    // column is not rowkey column, need not update
  } else {
    const ObColumnSchemaV2 *origin_shadow_column_schema = nullptr;
    for (int64_t i = 0; OB_SUCC(ret) && i < idx_schema_array.count(); ++i) {
      const ObTableSchema& idx_table_schema = idx_schema_array.at(i);
      if (idx_table_schema.get_shadow_rowkey_column_num() > 0) {
        const ObColumnSchemaV2 *origin_shadow_column_schema = nullptr;
        if (OB_ISNULL(origin_shadow_column_schema = idx_table_schema.get_column_schema(alter_column_schema->get_column_id() + common::OB_MIN_SHADOW_COLUMN_ID))) {
          ret = OB_ENTRY_NOT_EXIST;
          LOG_WARN("origin_shadow_column not exist", K(ret), KPC(alter_column_schema), K(idx_table_schema));
        } else if (!origin_shadow_column_schema->is_rowkey_column()) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("origin_shadow_column_schema is not rowkey column", K(ret), K(idx_table_schema), KPC(origin_shadow_column_schema));
        } else {
          SMART_VAR(ObColumnSchemaV2, new_aux_column_schema) {
            new_aux_column_schema.reset();
            if (OB_FAIL(new_aux_column_schema.assign(*origin_shadow_column_schema))){
              LOG_WARN("fail to assgin new_aux_column_schema", K(ret), KPC(origin_shadow_column_schema));
            } else if (OB_FAIL(fill_new_column_attributes(*alter_column_schema, new_aux_column_schema))) {
              LOG_WARN("failed to fill new column attributes", K(ret), KPC(alter_column_schema), K(new_aux_column_schema));
            } else if (FALSE_IT(new_aux_column_schema.set_autoincrement(false))) {
              // shadow column of index table should not be auto_increment
            } else if (OB_FAIL(ObIndexBuilderUtil::set_shadow_column_info(origin_shadow_column_schema->get_column_name(), origin_shadow_column_schema->get_column_id(), new_aux_column_schema))) {
              LOG_WARN("fail to set shadow_column_info", K(ret), K(new_aux_column_schema), K(origin_shadow_column_schema->get_column_name()));
            } else if (OB_FAIL(ddl_operator.update_single_column(trans,
                      idx_table_schema,
                      idx_table_schema,
                      new_aux_column_schema,
                      need_del_stats))) {
              LOG_WARN("schema service update aux column failed", K(ret), K(idx_table_schema), K(new_aux_column_schema));
            } else if (OB_FAIL(ddl_operator.sync_aux_schema_version_for_history(
                    trans,
                    idx_table_schema))) {
              LOG_WARN("fail to update aux schema version for update column", K(ret), K(idx_table_schema));
            }
          } // end SMART_VAR
        }
      }
    } // end for
  }
  return ret;
}


int ObDDLService::check_new_column_for_index(
    ObIArray<ObTableSchema> &idx_schemas,
    const ObColumnSchemaV2 &new_column_schema)
{
  int ret = OB_SUCCESS;
  int idx_cnt = idx_schemas.count();
  ObTableSchema *index_table_schema = NULL;
  ObColumnSchemaV2 copy_index_column_schema;
  bool for_view = false;
  for (int64_t i = 0; OB_SUCC(ret) && i < idx_cnt; ++i) {
    index_table_schema = &idx_schemas.at(i);
    if (OB_ISNULL(index_table_schema)) {
      ret = OB_ERR_UNEXPECTED;
      RS_LOG(WARN, "table schema should not be null", K(ret));
    } else {
      const ObColumnSchemaV2 *origin_idx_column_schema =
        index_table_schema->get_column_schema(new_column_schema.get_column_id());
      if (NULL == origin_idx_column_schema) {
        RS_LOG(INFO, "index table do not contain this column",
               "column_name", new_column_schema.get_column_name_str(),
               "index_table", index_table_schema->get_table_name_str());
        continue;
      } else if (!origin_idx_column_schema->is_rowkey_column()) {
        RS_LOG(INFO, "ingore not rowkey column",
               "column_name", new_column_schema.get_column_name_str(),
               "index_table", index_table_schema->get_table_name_str());
      } else {
        copy_index_column_schema.reset();
        if (OB_FAIL(copy_index_column_schema.assign(new_column_schema))) {
          LOG_WARN("fail to assign column schema", KR(ret), K(new_column_schema));
        } else {
          copy_index_column_schema.set_rowkey_position(origin_idx_column_schema->get_rowkey_position());
          copy_index_column_schema.set_index_position(origin_idx_column_schema->get_index_position());
          copy_index_column_schema.set_tbl_part_key_pos(origin_idx_column_schema->get_tbl_part_key_pos());
          if (OB_FAIL(index_table_schema->alter_column(copy_index_column_schema,
                      ObTableSchema::CHECK_MODE_ONLINE,
                      for_view))) {
            RS_LOG(WARN, "failed to alter index column schema", K(copy_index_column_schema), K(ret));
          } else if (!index_table_schema->is_valid()) {
            ret = OB_SCHEMA_ERROR;
            RS_LOG(WARN, "idx table schema is invalid!", K(ret));
          }
        }
      }
    }
  }
  return ret;
}

// wrapper for alter column effects
// if column is in
// 1. update index if modified column is in index
// 2. update materialized view if modified column is in materialized view
// but 2 is disabled for now
int ObDDLService::alter_table_update_index_and_view_column(
    const ObTableSchema &new_table_schema,
    const ObColumnSchemaV2 &new_column_schema,
    ObDDLOperator &ddl_operator,
    common::ObMySQLTransaction &trans,
    const ObIArray<ObTableSchema> *global_idx_schema_array/*=NULL*/)
{
  int ret = OB_SUCCESS;
  if (OB_FAIL(alter_table_update_aux_column(new_table_schema, new_column_schema,
             ddl_operator, trans, USER_INDEX, global_idx_schema_array))) {
    LOG_WARN("fail to update index column", K(ret), K(new_table_schema), K(new_column_schema));
  } else if (OB_FAIL(alter_table_update_aux_column(new_table_schema, new_column_schema,
            ddl_operator, trans, AUX_VERTIAL_PARTITION_TABLE))) {
    LOG_WARN("fail to update aux vp column", K(ret), K(new_table_schema), K(new_column_schema));
  }
  return ret;
}

// aux schema column
int ObDDLService::alter_table_update_aux_column(
    const ObTableSchema &new_table_schema,
    const ObColumnSchemaV2 &new_column_schema,
    ObDDLOperator &ddl_operator,
    common::ObMySQLTransaction &trans,
    const ObTableType table_type,
    const ObIArray<ObTableSchema> *global_idx_schema_array/*=NULL*/)
{
  int ret = OB_SUCCESS;
  //update column in aux table
  ObSchemaGetterGuard schema_guard;
  ObColumnSchemaV2 new_aux_column_schema;
  const bool is_index = USER_INDEX == table_type;
  ObSEArray<uint64_t, 16> aux_vp_tid_array;
  ObSEArray<ObAuxTableMetaInfo, 16> simple_index_infos;
  const uint64_t tenant_id = new_table_schema.get_tenant_id();
  const bool need_del_stats = false;

  if (OB_FAIL(schema_service_->get_tenant_schema_guard(tenant_id, schema_guard))) {
    RS_LOG(WARN, "get schema guard failed", K(ret));
  } else if (is_index && OB_FAIL(new_table_schema.get_simple_index_infos(simple_index_infos))) {
    LOG_WARN("get simple_index_infos failed", K(ret));
  } else if (!is_index && OB_FAIL(new_table_schema.get_aux_vp_tid_array(aux_vp_tid_array))) {
    LOG_WARN("get_aux_tid_array failed", K(ret), K(is_index));
  } else {
    //update all aux table schema
    const ObTableSchema *aux_table_schema = NULL;
    int64_t N = is_index ? simple_index_infos.count() : aux_vp_tid_array.count();
    for (int64_t i = 0; OB_SUCC(ret) && i < N; ++i) {
      aux_table_schema = NULL;
      if (is_index && OB_NOT_NULL(global_idx_schema_array) && !global_idx_schema_array->empty()) {
        for (int64_t j = 0; OB_SUCC(ret) && j < global_idx_schema_array->count(); ++j) {
          if (simple_index_infos.at(i).table_id_ == global_idx_schema_array->at(j).get_table_id()) {
            aux_table_schema = &(global_idx_schema_array->at(j));
            break;
          }
        }
      }
      uint64_t tid = is_index ? simple_index_infos.at(i).table_id_ : aux_vp_tid_array.at(i);
      if (OB_FAIL(ret)) {
      } else if (OB_ISNULL(aux_table_schema)
                 && OB_FAIL(schema_guard.get_table_schema(
                            tenant_id, tid, aux_table_schema))) {
        RS_LOG(WARN, "get_table_schema failed", K(ret), K(tenant_id), K(tid));
      } else if (OB_ISNULL(aux_table_schema)) {
        ret = OB_ERR_UNEXPECTED;
        RS_LOG(WARN, "aux schema should not be null", K(ret));
      } else if (aux_table_schema->is_in_recyclebin()) {
        ret = OB_ERR_OPERATION_ON_RECYCLE_OBJECT;
        LOG_WARN("aux table is in recyclebin", K(ret));
      } else {
        const ObColumnSchemaV2 *origin_column_schema =
            aux_table_schema->get_column_schema(new_column_schema.get_column_id());
        if (NULL != origin_column_schema) {
          // exist such column in aux schema
          if (OB_FAIL(new_aux_column_schema.assign(new_column_schema))) {
            LOG_WARN("fail to assign column", KR(ret), K(new_column_schema));
          } else {
            new_aux_column_schema.set_table_id(aux_table_schema->get_table_id());
            new_aux_column_schema.set_autoincrement(false);
            //save the rowkey postion and aux postion
            if (is_index) {
              new_aux_column_schema.set_rowkey_position(origin_column_schema->get_rowkey_position());
              new_aux_column_schema.set_index_position(origin_column_schema->get_index_position());
              new_aux_column_schema.set_tbl_part_key_pos(origin_column_schema->get_tbl_part_key_pos());
              ObIndexBuilderUtil::del_column_flags_and_default_value(new_aux_column_schema);
            }
            if (!is_index) {
              // VP column of primary table need not update.
              new_aux_column_schema.set_column_flags(AUX_VP_COLUMN_FLAG);
            }
          }
          //will only update some attribute, not include rowkey postion or aux position
          if (FAILEDx(ddl_operator.update_column_and_column_group(trans, *aux_table_schema, *aux_table_schema,
                                                                  new_aux_column_schema, need_del_stats))) {
            LOG_WARN("fail to update column and column group", KR(ret), K(new_aux_column_schema), K(*aux_table_schema));
          }

          //  update relevant shadow pk when column changed
          const ObColumnSchemaV2 *shadow_column = nullptr;
          if (OB_FAIL(ret)) {
          } else if (!aux_table_schema->is_global_index_table()) {
            /* skip */
          } else if (OB_ISNULL(shadow_column = aux_table_schema->get_column_schema(origin_column_schema->get_column_id() + common::OB_MIN_SHADOW_COLUMN_ID))) {
            /* skip */
          } else {
            const bool need_del_stat = false;
            ObColumnSchemaV2 new_shadow_column;
            new_shadow_column.assign(*shadow_column);
            new_shadow_column.set_data_type(new_column_schema.get_data_type());
            if (OB_FAIL(ddl_operator.update_single_column(trans, *aux_table_schema, *aux_table_schema,
                                                          new_shadow_column, need_del_stat))) {
              LOG_WARN("failed to update global index shadow column", K(ret));
            }
          }

          if (OB_FAIL(ret)) {
          } else if (OB_FAIL(ddl_operator.sync_aux_schema_version_for_history(
                  trans,
                  *aux_table_schema))) {
            RS_LOG(WARN, "fail to update aux schema version for update column");
          }
        }
      }
    } // end of for
  } // end of else
  return ret;
}

int ObDDLService::alter_sequence_in_alter_column(const ObTableSchema &table_schema,
    ObColumnSchemaV2 &column_schema,
    ObMySQLTransaction &trans,
    ObSchemaGetterGuard &schema_guard,
    ObSequenceDDLArg &sequence_ddl_arg)
{
  int ret = OB_SUCCESS;
  // stmt type check add for modify from identity column to normal column and it's still a identity column
  if (column_schema.is_identity_column() && sequence_ddl_arg.get_stmt_type() != common::OB_INVALID_ID) {
    ObSequenceDDLProxy ddl_operator(*schema_service_);
    ObSequenceSchema sequence_schema = sequence_ddl_arg.sequence_schema();
    char temp_sequence_name[OB_MAX_SEQUENCE_NAME_LENGTH + 1] = { 0 };
    int32_t len = snprintf(temp_sequence_name, sizeof(temp_sequence_name), "%s%lu%c%lu",
                           "ISEQ$$_",
                           ObSchemaUtils::get_extract_schema_id(column_schema.get_tenant_id(), column_schema.get_table_id()),
                           '_',
                           column_schema.get_column_id());
    if (OB_UNLIKELY(len < 0)) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("create sequence name fail", K(ret), K(column_schema));
    } else {
      ObString sequence_name = ObString::make_string(temp_sequence_name);
      sequence_schema.set_database_id(table_schema.get_database_id());
      sequence_schema.set_sequence_name(sequence_name);
      if (OB_FAIL(ddl_operator.alter_sequence(sequence_schema,
                                              sequence_ddl_arg.option_bitset_,
                                              trans,
                                              schema_guard,
                                              NULL,
                                              FROM_TABLE_DDL))) {
        LOG_WARN("alter sequence fail", K(ret));
      }
    }
  }
  return ret;
}

int ObDDLService::redistribute_column_ids(
    ObTableSchema &new_table_schema)
{
  int ret = OB_SUCCESS;
  hash::ObHashMap<uint64_t, uint64_t> column_id_map;
  if (OB_FAIL(column_id_map.create(OB_MAX_COLUMN_NUMBER / 2, lib::ObLabel("DDLSrvTmp")))) {
    LOG_WARN("failed to create column id map", K(ret));
  } else if (OB_FAIL(new_table_schema.generate_new_column_id_map(column_id_map))) {
    LOG_WARN("failed to generate new column id map", K(ret));
  } else if (OB_FAIL(new_table_schema.convert_column_ids_for_ddl(column_id_map))) {
    LOG_WARN("failed to convert new table schema column id", K(ret));
  } else if (OB_FAIL(new_table_schema.sort_column_array_by_column_id())) {
    LOG_WARN("failed to sort column", KR(ret), K(new_table_schema));
  } else if (OB_FAIL(adjust_cg_for_offline(new_table_schema))) {
    LOG_WARN("failed to adjust cg after redistribute_column_ids", KR(ret));
  }
  return ret;
}

int ObDDLService::delete_constraint_update_new_table(
    const AlterTableSchema &alter_table_schema,
    ObTableSchema &new_table_schema)
{
  int ret = OB_SUCCESS;
  // remove constraint from new table schema caused by drop column.
  for (ObTableSchema::const_constraint_iterator iter = alter_table_schema.constraint_begin();
      OB_SUCC(ret) && iter != alter_table_schema.constraint_end(); iter ++) {
    if (OB_FAIL(new_table_schema.delete_constraint((*iter)->get_constraint_name_str()))) {
      LOG_WARN("fail to remove constraint from new table schema", K(ret), K(*iter));
    } else {/* do nothing */}
  }
  return ret;
}

// if all rls policies are column level and all sec columns of each policy will be droppped, then
// the rls object flag should be removed
int ObDDLService::update_new_table_rls_flag(
    share::schema::ObSchemaGetterGuard &schema_guard,
    const common::ObIArray<uint64_t> &drop_cols_id_arr,
    ObTableSchema &table_schema)
{
  int ret = OB_SUCCESS;
  if (!table_schema.has_table_flag(CASCADE_RLS_OBJECT_FLAG) ||
      !table_schema.get_rls_group_ids().empty() ||
      !table_schema.get_rls_context_ids().empty() ||
      0 == drop_cols_id_arr.count()) {
    // do nothing
  } else {
    const uint64_t tenant_id = table_schema.get_tenant_id();
    const ObRlsPolicySchema *policy = NULL;
    const int64_t policy_count = table_schema.get_rls_policy_ids().count();
    bool found = false;
    for (int64_t i = 0; OB_SUCC(ret) && !found && i < policy_count; ++i) {
      uint64_t policy_id = table_schema.get_rls_policy_ids().at(i);
      if (OB_FAIL(schema_guard.get_rls_policy_schema_by_id(tenant_id, policy_id, policy))) {
        LOG_WARN("failed to get rls policy schema", K(ret));
      } else if (OB_ISNULL(policy)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("get null rls policy schema", KR(ret), K(policy_id), K(table_schema));
      } else if (!policy->is_column_level_policy()) {
        found = true;
      }
      for (int64_t j = 0; OB_SUCC(ret) && !found && j < policy->get_sec_column_count(); ++j) {
        const ObRlsSecColumnSchema* sec_column = policy->get_sec_column_by_idx(j);
        bool found_col = false;
        if (OB_ISNULL(sec_column)) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("rls column is null", KR(ret));
        }
        for (int64_t k = 0; OB_SUCC(ret) && !found_col && k < drop_cols_id_arr.count(); ++k) {
          if (sec_column->get_column_id() == drop_cols_id_arr.at(k)) {
            found_col = true;
          }
        }
        if (OB_SUCC(ret) && !found_col) {
          found = true;
        }
      }
    }
    if (OB_SUCC(ret) && !found) {
      table_schema.del_table_flag(CASCADE_RLS_OBJECT_FLAG);
    }
  }
  return ret;
}

int ObDDLService::drop_udt_hidden_columns(const ObColumnSchemaV2 &new_origin_col, ObTableSchema &new_table_schema)
{
  int ret = OB_SUCCESS;
  bool is_oracle_mode = false;
  if (OB_FAIL(new_table_schema.check_if_oracle_compat_mode(is_oracle_mode))) {
    LOG_WARN("failed to get oracle mode", K(ret));
  } else if (is_oracle_mode && new_origin_col.is_xmltype()) {
    ObSEArray<ObColumnSchemaV2 *, 1> hidden_cols;
    if (OB_FAIL(new_table_schema.get_column_schema_in_same_col_group(new_origin_col.get_column_id(),
                                                                     new_origin_col.get_udt_set_id(),
                                                                     hidden_cols))) {
      LOG_WARN("failed to get column schema", K(ret));
    } else {
      for (int i = 0; i < hidden_cols.count() && OB_SUCC(ret); i++) {
        ObColumnSchemaV2 *next_col = new_table_schema.get_column_schema_by_prev_next_id(hidden_cols.at(i)->get_next_column_id());
        if (OB_ISNULL(next_col)) {
          // do nothing since local_column is tail column
        } else {
          next_col->set_prev_column_id(hidden_cols.at(i)->get_prev_column_id());
        }
        if (OB_FAIL(new_table_schema.delete_column(hidden_cols.at(i)->get_column_name_str()))) {
          LOG_WARN("fail to delete column", K(ret), K(hidden_cols.at(i)->get_column_name_str()));
        }
      }
    }
  }
  return ret;
}

int ObDDLService::drop_column_update_new_table(
    const ObString &column_name,
    ObTableSchema &new_table_schema)
{
  int ret = OB_SUCCESS;
  const uint64_t tenant_id = new_table_schema.get_tenant_id();
  ObColumnSchemaV2 *new_origin_col = new_table_schema.get_column_schema(column_name);
  ObSchemaService *schema_service = schema_service_->get_schema_service();
  if (OB_ISNULL(schema_service)) {
    ret = OB_ERR_SYS;
    LOG_WARN("schema_service must not null", K(ret));
  } else if (OB_ISNULL(new_origin_col)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("fail to get column from new table schema", K(ret));
  } else {
    ObColumnSchemaV2 *next_col = new_table_schema.get_column_schema_by_prev_next_id(new_origin_col->get_next_column_id());
    if (OB_ISNULL(next_col)) {
      // do nothing since local_column is tail column
    } else {
      next_col->set_prev_column_id(new_origin_col->get_prev_column_id());
    }
    if (OB_FAIL(ret)) {
    } else if (OB_FAIL(new_table_schema.delete_column(new_origin_col->get_column_name_str()))) {
      LOG_WARN("fail to delete column", K(ret), K(new_origin_col->get_column_name_str()));
    } else if (OB_FAIL(drop_udt_hidden_columns(*new_origin_col, new_table_schema))) {
      LOG_WARN("fail to delete udt hidden column", K(ret), K(new_origin_col->get_column_name_str()));
    }
  }
  return ret;
}

// update relevant inner table if all of schema_guard, ddl_operator and trans are not null
int ObDDLService::add_new_column_to_table_schema(
    const ObTableSchema &origin_table_schema,
    const AlterTableSchema &alter_table_schema,
    const common::ObTimeZoneInfoWrap &tz_info_wrap,
    const common::ObString &nls_formats,
    sql::ObLocalSessionVar &local_session_var,
    obrpc::ObSequenceDDLArg &sequence_ddl_arg,
    common::ObIAllocator &allocator,
    ObTableSchema &new_table_schema,
    AlterColumnSchema &alter_column_schema,
    ObIArray<ObString> &gen_col_expr_arr,
    ObSchemaGetterGuard &schema_guard,
    uint64_t &curr_udt_set_id,
    ObDDLOperator *ddl_operator,
    common::ObMySQLTransaction *trans)
{
  int ret = OB_SUCCESS;
  const ObSQLMode sql_mode = alter_table_schema.get_sql_mode();
  const bool update_inner_table = nullptr != ddl_operator && nullptr != trans;
  bool is_oracle_mode = false;
  bool is_contain_part_key = false;
  LOG_DEBUG("check before alter table column", K(origin_table_schema), K(alter_table_schema), K(new_table_schema));
  if (OB_FAIL(origin_table_schema.check_if_oracle_compat_mode(is_oracle_mode))) {
    LOG_WARN("failed to get oracle mode", K(ret));
  } else if (OB_ISNULL(tz_info_wrap.get_time_zone_info())
             || OB_ISNULL(tz_info_wrap.get_time_zone_info()->get_tz_info_map())) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("invalid tz_info_wrap", K(tz_info_wrap), K(ret));
  }
  // fill column collation
  if (OB_FAIL(ret)) {
  } else if (OB_FAIL(fill_column_collation(sql_mode,
                                           is_oracle_mode,
                                           new_table_schema,
                                           allocator,
                                           alter_column_schema))) {
    LOG_WARN("failed to fill column collation", K(ret));
  } else {
    int64_t max_used_column_id = new_table_schema.get_max_used_column_id();
    const uint64_t tenant_id = new_table_schema.get_tenant_id();
    if (is_inner_table(new_table_schema.get_table_id())
        && (OB_INVALID_ID == alter_column_schema.get_column_id()
        || alter_column_schema.get_column_id() != max_used_column_id + 1)) {
      // 225 is barrier version, after this adding column in system table need specify column_id
      ret = OB_OP_NOT_ALLOW;
      LOG_WARN("inner table should add column at last and specify column_id",
               K(ret), K(alter_column_schema), K(max_used_column_id));
      LOG_USER_ERROR(OB_OP_NOT_ALLOW, "inner table add column without column_id");
    } else {
      if (alter_column_schema.is_udt_hidden_column()) {
        // udt hidden column
        char col_name[OB_MAX_COLUMN_NAME_LENGTH] = {0};
        alter_column_schema.set_udt_set_id(curr_udt_set_id);
        databuff_printf(col_name, OB_MAX_COLUMN_NAME_LENGTH, "SYS_NC%05lu$",max_used_column_id + 1);
        if (OB_FAIL(alter_column_schema.set_column_name(col_name))) {
          SQL_RESV_LOG(WARN, "failed to set column name", K(ret));
        }
      }
      if (OB_SUCC(ret)) {
        alter_column_schema.set_column_id(++max_used_column_id);
        alter_column_schema.set_rowkey_position(0);
        alter_column_schema.set_index_position(0);
        alter_column_schema.set_not_part_key();
        alter_column_schema.set_table_id(new_table_schema.get_table_id());
        alter_column_schema.set_tenant_id(new_table_schema.get_tenant_id());
        if (new_table_schema.is_primary_vp_table()) {
          // The last column add in the primary VP
          alter_column_schema.add_column_flag(PRIMARY_VP_COLUMN_FLAG);
        }
        if (alter_column_schema.is_xmltype()) {
          alter_column_schema.set_udt_set_id(alter_column_schema.get_column_id());
          curr_udt_set_id = alter_column_schema.get_udt_set_id();
        }
        new_table_schema.set_max_used_column_id(max_used_column_id);
      }
    }
  }
  if (OB_FAIL(ret)) {
  } else if (OB_FAIL(refill_columns_id_for_check_constraint(origin_table_schema,
                                                            alter_table_schema,
                                                            alter_column_schema,
                                                            is_oracle_mode,
                                                            allocator))) {
    LOG_WARN("fail to refill columns id for check constraint", K(ret));
  } else if (is_oracle_mode
    && OB_FAIL(refill_columns_id_for_not_null_constraint(alter_table_schema,
                                                         alter_column_schema))) {
    LOG_WARN("fail to refill column id to constraints", K(ret));
  } else if (OB_FAIL(update_prev_id_for_add_column(origin_table_schema,
        new_table_schema, alter_column_schema, ddl_operator, trans))) {
    LOG_WARN("failed to update prev id", K(ret));
  } else if (update_inner_table) {
    if (OB_FAIL(ddl_operator->create_sequence_in_add_column(new_table_schema,
            alter_column_schema, *trans, schema_guard, sequence_ddl_arg))) {
      LOG_WARN("alter table add identity column fail", K(alter_column_schema), K(ret));
    }
  }
  if (OB_SUCC(ret)) {
    ObSchemaChecker schema_checker;
    if (OB_FAIL(schema_checker.init(schema_guard))) {
      LOG_WARN("failed to init schema guard", K(ret));
    } else if (alter_column_schema.is_udt_related_column(is_oracle_mode)) {
      // udt column/oracle gis not need to do the flowing else ifs:
      // 1. default values is check and calculated in resolver, only check dependency version on RS
      // 2. udt column and it's hidden columns cannot be primary key
      LOG_INFO("alter table add udt related column", K(alter_column_schema));
    } else if (OB_FAIL(ObDDLResolver::check_default_value(
                alter_column_schema.get_cur_default_value(),
                tz_info_wrap, &nls_formats, &local_session_var, allocator,
                new_table_schema,
                alter_column_schema,
                gen_col_expr_arr,
                alter_table_schema.get_sql_mode(),
                false, /* allow_sequence */
                &schema_checker))) {
      LOG_WARN("fail to check default value", K(alter_column_schema), K(ret));
    } else if (OB_FAIL(resolve_orig_default_value(alter_column_schema,
                                                  tz_info_wrap,
                                                  &nls_formats,
                                                  allocator))) {
      LOG_WARN("fail to resolve default value", K(ret));
    } else if (alter_column_schema.is_primary_key_) {
      if (new_table_schema.get_rowkey_column_num() > 0) {
        if (new_table_schema.is_table_without_pk()) {
          ret = OB_NOT_SUPPORTED;
          LOG_WARN("not support to add primary key!", K(ret));
        } else {
          ret = OB_ERR_MULTIPLE_PRI_KEY;
          LOG_WARN("multiple primary key defined", K(ret));
        }
      }
    }
  }
  if (OB_SUCC(ret)) {
    const ObColumnSchemaV2 *mem_col = NULL;
    if (OB_FAIL(resolve_timestamp_column(&alter_column_schema,
                                         new_table_schema,
                                         alter_column_schema,
                                         tz_info_wrap,
                                         &nls_formats,
                                         allocator))) {
      LOG_WARN("fail to resolve timestamp column", K(ret));
    } else if (OB_FAIL(deal_default_value_padding(alter_column_schema, allocator))) {
      LOG_WARN("fail to deal default value padding", K(alter_column_schema), K(ret));
    } else if (OB_FAIL(new_table_schema.check_primary_key_cover_partition_column())) {
      LOG_WARN("fail to check primary key cover partition column", K(ret));
    } else if (OB_FAIL(new_table_schema.add_column(alter_column_schema))) {
      if (OB_ERR_COLUMN_DUPLICATE == ret) {
        const ObString &column_name = alter_column_schema.get_column_name_str();
        LOG_USER_ERROR(OB_ERR_COLUMN_DUPLICATE, column_name.length(), column_name.ptr());
        LOG_WARN("duplicate column name", K(column_name), K(ret));
      }
      LOG_WARN("failed to add new column", K(ret));
    } else if (OB_ISNULL(mem_col = new_table_schema.get_column_schema(
                                    alter_column_schema.get_column_id()))) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("mem_col is NULL", K(ret));
    } else {
      alter_column_schema.set_prev_column_id(mem_col->get_prev_column_id());
      if (update_inner_table) {
        if (OB_FAIL(ddl_operator->insert_single_column(*trans,
                new_table_schema, alter_column_schema))) {
          LOG_WARN("failed to add column", K(ret), K(alter_column_schema));
        }
      }
    }
  }
  return ret;
}

int ObDDLService::add_column_to_column_group(
    const share::schema::ObTableSchema &origin_table_schema,
    const share::schema::AlterTableSchema &alter_table_schema,
    share::schema::ObTableSchema &new_table_schema,
    ObDDLOperator &ddl_operator,
    common::ObMySQLTransaction &trans)
{
  int ret = OB_SUCCESS;
  uint64_t cur_column_group_id = origin_table_schema.get_next_single_column_group_id();
#ifdef ERRSIM
  int tmp_ret = OB_E(EventTable::EN_DDL_CREATE_OLD_VERSION_COLUMN_GROUP) OB_SUCCESS;
  if (OB_TMP_FAIL(tmp_ret)) {
    cur_column_group_id = origin_table_schema.get_max_used_column_group_id() + 1;
  }
#endif
  ObArray<uint64_t> column_ids;
  ObTableSchema::const_column_iterator it_begin = alter_table_schema.column_begin();
  ObTableSchema::const_column_iterator it_end = alter_table_schema.column_end();
  AlterColumnSchema *alter_column_schema = nullptr;

  if (!origin_table_schema.is_valid() ) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("invalid argument", K(ret), K(origin_table_schema), K(alter_table_schema));
  } else if (!new_table_schema.is_column_store_supported()) {
    /* skip*/
  } else {
    for(; OB_SUCC(ret) && it_begin != it_end; it_begin++) {
      if (OB_ISNULL(alter_column_schema = static_cast<AlterColumnSchema *>(*it_begin))) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("*it_begin is NULL", K(ret));
      } else if (alter_column_schema->alter_type_ == OB_DDL_ADD_COLUMN) {
        const ObColumnSchemaV2 *column_schema = nullptr;
        if (OB_ISNULL(column_schema = new_table_schema.get_column_schema(alter_column_schema->get_column_name_str()))) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("unexpected null column schema", K(ret), KPC(alter_column_schema), K(new_table_schema));
        } else if (column_schema->is_virtual_generated_column()) {
          // skip virtual column
        } else if (OB_FAIL(column_ids.push_back(column_schema->get_column_id()))) {
          LOG_WARN("fali to push back column id", K(ret));
        }
      }
    }

    if (OB_FAIL(ret)) {
      /* skip do nothing*/
    } else if (column_ids.count() == 0){
      /* do not add column, skip */
    } else {
      bool is_all_cg_exist = false;
      bool is_each_cg_exist = false;
      if (OB_FAIL(new_table_schema.is_column_group_exist(OB_ALL_COLUMN_GROUP_NAME, is_all_cg_exist))) {
        LOG_WARN("fail to check whether all cg exist", K(ret), K(new_table_schema));
      } else if (OB_FAIL(new_table_schema.is_column_group_exist(OB_EACH_COLUMN_GROUP_NAME, is_each_cg_exist))) {
        LOG_WARN("fail to check whether each cg exist", K(ret), K(new_table_schema));
      }

      /* update info about each column group*/
      if (OB_FAIL(ret)) {
      } else if (is_each_cg_exist) {
        HEAP_VAR(ObTableSchema, tmp_table) {
        if (OB_FAIL(tmp_table.assign(new_table_schema))) {
          LOG_WARN("fail to assign", K(ret), K(new_table_schema), K(tmp_table));
        }
        tmp_table.reset_column_group_info();
        for (int64_t i = 0; OB_SUCC(ret) && i < column_ids.count(); i++) {
          ObColumnGroupSchema cg_schema;
          if (OB_FAIL(ObSchemaUtils::build_single_column_group(new_table_schema,
                                                   new_table_schema.get_column_schema(column_ids.at(i)),
                                                   new_table_schema.get_tenant_id(),
                                                   cur_column_group_id++,
                                                   cg_schema))) {
            LOG_WARN("fail to build single column group", K(ret), K(new_table_schema), K(column_ids.at(i)));
          } else if (OB_FAIL(new_table_schema.add_column_group(cg_schema))) {
            LOG_WARN("fail to add new column group schema to table", K(ret), K(cg_schema));
          } else if (OB_FAIL(tmp_table.add_column_group(cg_schema))) {
            LOG_WARN("fail to add new column group schema to tmp_cg", K(ret), K(tmp_table), K(cg_schema));
          }
        }
        if (OB_FAIL(ret)) {
        } else if (tmp_table.get_column_group_count() == 0){
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("column_group array should not be empty", K(ret), K(tmp_table));
        } else if (OB_FAIL(ddl_operator.insert_column_groups(trans, tmp_table))) {
          LOG_WARN("fail to insert new table_schema to each column gorup", K(ret), K(tmp_table));
        }
        }
      }
      /* update info about all column group*/
      if (OB_FAIL(ret)) {
      } else if (is_all_cg_exist) {
        ObColumnGroupSchema* all_cg = nullptr;
        if (OB_FAIL(new_table_schema.get_column_group_by_name(OB_ALL_COLUMN_GROUP_NAME, all_cg))) {
          LOG_WARN("fail to get all column group", K(ret), K(new_table_schema));
        } else if (OB_ISNULL(all_cg)) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("column group should not be null", K(ret));
        }
        for (int64_t i = 0; OB_SUCC(ret) && i < column_ids.count(); i++) {
          if (OB_FAIL(all_cg->add_column_id(column_ids.at(i)))) {
            LOG_WARN("fail to add column id", K(ret), K(new_table_schema), K(column_ids.at(i)));
          }
        }
        if (OB_FAIL(ret)){
        } else if (column_ids.count() == 0) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("column_ids should not be empty", K(ret), K(column_ids));
        } else if (OB_FAIL(ddl_operator.insert_column_ids_into_column_group(trans, new_table_schema, column_ids, *all_cg))) {
          LOG_WARN("fail to insert column ids into inner table", K(ret), K(new_table_schema),K(column_ids));
        }
      }

      /* update info about default column group*/
      if (OB_FAIL(ret)) {
      } else if (!is_all_cg_exist && !is_each_cg_exist) {
        ObColumnGroupSchema *default_cg = nullptr;
        if (OB_FAIL(new_table_schema.get_column_group_by_name(OB_DEFAULT_COLUMN_GROUP_NAME, default_cg))) {
          LOG_WARN("fail get default column group", K(ret), K(new_table_schema));
        } else if (OB_ISNULL(default_cg)) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("column group should not be null", K(ret), K(new_table_schema));
        }
        for (int64_t i = 0; OB_SUCC(ret) && i < column_ids.count(); i++) {
          if (OB_FAIL(default_cg->add_column_id(column_ids.at(i)))) {
            LOG_WARN("fail to add column id", K(ret), K(new_table_schema), K(column_ids.at(i)));
          }
        }

        if (OB_FAIL(ret)){
        } else if (column_ids.count() == 0) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("column_ids should not be empty", K(ret), K(column_ids));
        } else if (OB_FAIL(ddl_operator.insert_column_ids_into_column_group(trans, new_table_schema, column_ids, *default_cg))) {
          LOG_WARN("fail to insert column ids into inner table", K(ret), K(new_table_schema));
        }
      }
    }
  }
  return ret;
}

int ObDDLService::gen_alter_column_new_table_schema_offline(
    const ObTableSchema &origin_table_schema,
    AlterTableSchema &alter_table_schema,
    ObTableSchema &new_table_schema,
    obrpc::ObAlterTableArg &alter_table_arg,
    ObSchemaGetterGuard &schema_guard,
    bool &need_redistribute_column_id)
{
  int ret = OB_SUCCESS;
  const common::ObTimeZoneInfoWrap &tz_info_wrap = alter_table_arg.tz_info_wrap_;
  const common::ObString *nls_formats = alter_table_arg.nls_formats_;
  common::ObIAllocator &allocator = alter_table_arg.allocator_;
  need_redistribute_column_id = false;
  bool is_contain_part_key = false;
  // drop column related.
  int64_t new_table_cols_cnt = 0;
  ObArray<uint64_t> drop_cols_id_arr;
  uint64_t curr_udt_set_id = 0;
  bool is_oracle_mode = false;
  LOG_DEBUG("check before alter table column", K(origin_table_schema), K(alter_table_schema), K(new_table_schema));
  ObSchemaChecker schema_checker;
  if (OB_ISNULL(tz_info_wrap.get_time_zone_info())
      || OB_ISNULL(tz_info_wrap.get_time_zone_info()->get_tz_info_map())) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("invalid tz_info_wrap", K(ret), K(tz_info_wrap));
  } else if (OB_FAIL(origin_table_schema.check_if_oracle_compat_mode(is_oracle_mode))) {
    RS_LOG(WARN, "failed to get oracle mode", K(ret));
  } else if (OB_ISNULL(nls_formats)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("invalid nls_formats", K(ret));
  } else if (OB_FAIL(schema_checker.init(schema_guard))) {
    LOG_WARN("init schema checker failed", K(ret));
  } else {
    AlterColumnSchema *alter_column_schema;
    ObTableSchema::const_column_iterator it_begin = alter_table_schema.column_begin();
    ObTableSchema::const_column_iterator it_end = alter_table_schema.column_end();
    common::hash::ObHashSet<ObColumnNameHashWrapper> update_column_name_set;
    lib::Worker::CompatMode compat_mode = (is_oracle_mode ?
    lib::Worker::CompatMode::ORACLE : lib::Worker::CompatMode::MYSQL);
    lib::CompatModeGuard tmpCompatModeGuard(compat_mode);
    ObSEArray<ObString, 4> gen_col_expr_arr;
    if (OB_FAIL(update_column_name_set.create(32))) {
      LOG_WARN("failed to create update column name set", K(ret));
    } else if (OB_FAIL(get_all_dropped_column_ids(alter_table_arg,
                                                  origin_table_schema,
                                                  drop_cols_id_arr,
                                                  &new_table_cols_cnt/*final columns count of new table*/))) {
      LOG_WARN("fail to prefetch all drop columns id", K(ret), K(alter_table_arg));
    } else if (drop_cols_id_arr.size() > 0) {
      if (OB_FAIL(check_drop_column_with_drop_foreign_key(alter_table_arg,
                                                          origin_table_schema,
                                                          drop_cols_id_arr))) {
        LOG_WARN("fail to check drop foreign key caused by drop column", K(ret), K(alter_table_arg));
      } else if (OB_FAIL(check_drop_column_with_drop_constraint(alter_table_arg,
                                                                schema_guard,
                                                                origin_table_schema,
                                                                drop_cols_id_arr))) {
        LOG_WARN("fail to check drop constraint caused by drop column", K(ret), K(alter_table_arg));
      } else if (OB_FAIL(delete_constraint_update_new_table(alter_table_schema,
                                                            new_table_schema))) {
        LOG_WARN("fail to delete constraints from new table", K(ret));
      } else if (OB_FAIL(update_new_table_rls_flag(schema_guard,
                                                   drop_cols_id_arr,
                                                   new_table_schema))) {
        LOG_WARN("fail to update new table flags", K(ret));
      } else {/* do nothing. */}
    }
    bool is_rename_first = false;
    if (OB_FAIL(ret)) {
    } else if (OB_FAIL(check_rename_first(alter_table_schema, origin_table_schema, is_oracle_mode,
                                          is_rename_first))) {
      LOG_WARN("check rename first failed", K(ret));
    }
    if (OB_SUCC(ret) && is_rename_first) {
      if (OB_FAIL(pre_rename_mysql_columns_offline(
                   origin_table_schema, alter_table_schema, is_oracle_mode, alter_table_arg,
                   new_table_schema, schema_checker, schema_guard, update_column_name_set,
                   need_redistribute_column_id, is_contain_part_key))) {
        LOG_WARN("pre rename columns failed", K(ret));
      }
    }
    share::schema::ObTableSchema::const_column_iterator iter = origin_table_schema.column_begin();
    share::schema::ObTableSchema::const_column_iterator end = origin_table_schema.column_end();
    for (; OB_SUCC(ret) && iter != end; ++iter) {
      const share::schema::ObColumnSchemaV2 *column = *iter;
      if (OB_ISNULL(column)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("invalid column schema", K(column));
      } else if (column->is_generated_column()) {
        const common::ObObj* ObObjtmp = &column->get_cur_default_value();
        if (OB_FAIL(gen_col_expr_arr.push_back(ObObjtmp->get_string()))) {
        LOG_WARN("fail to push back ObSEArray gen_col_expr_arr", K(ret));
        }
      }
    }
    for (; OB_SUCC(ret) && it_begin != it_end; it_begin++) {
      if (OB_ISNULL(alter_column_schema = static_cast<AlterColumnSchema *>(*it_begin))) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("*it_begin is NULL", K(ret));
      } else {
        const ObString &orig_column_name = alter_column_schema->get_origin_column_name();
        // column that has been add, alter, change or modify
        const ObColumnSchemaV2 *orig_column_schema = NULL;
        const ObSchemaOperationType op_type = alter_column_schema->alter_type_;
        switch (op_type) {
          case OB_DDL_DROP_COLUMN: {
            if (OB_FAIL(drop_column_offline(origin_table_schema, new_table_schema,
                                                   schema_guard, orig_column_name,
                                                   new_table_cols_cnt))) {
              LOG_WARN("drop column offline failed", K(ret));
            } else {
              need_redistribute_column_id = true;
            }
            break;
          }
          case OB_DDL_ADD_COLUMN: {
            if (OB_FAIL(add_new_column_to_table_schema(origin_table_schema,
                                                       alter_table_schema,
                                                       tz_info_wrap,
                                                       *nls_formats,
                                                       alter_table_arg.local_session_var_,
                                                       alter_table_arg.sequence_ddl_arg_,
                                                       alter_table_arg.allocator_,
                                                       new_table_schema,
                                                       *alter_column_schema,
                                                       gen_col_expr_arr,
                                                       schema_guard,
                                                       curr_udt_set_id,
                                                       nullptr,
                                                       nullptr))) {
              LOG_WARN("failed to add new column to table schema", K(ret));
            } else {
              ObSEArray<ObString, 1> new_pk_column;
              if (alter_column_schema->is_primary_key_) {
                if (new_table_schema.is_table_with_pk()) {
                  ret = OB_ERR_MULTIPLE_PRI_KEY;
                  LOG_WARN("multiple primary key defined", K(ret));
                } else if (OB_FAIL(new_pk_column.push_back(alter_column_schema->get_column_name_str()))){
                  LOG_WARN("failed to push back pk col name", K(ret));
                } else if (OB_FAIL(add_primary_key(new_pk_column, new_table_schema))) {
                  LOG_WARN("failed to add pk to table", K(ret), K(new_pk_column), K(new_table_schema));
                } else if (OB_FAIL(new_table_schema.check_primary_key_cover_partition_column())) {
                  LOG_WARN("failed to check primary key cover partition column", K(ret));
                }
              }
            }
            if (OB_FAIL(ret)) {
            } else {
              const ObString &column_name = alter_column_schema->get_column_name_str();
              ObColumnNameHashWrapper column_key(column_name);
              if (!is_oracle_mode) {
                // mysql mode
                // do not check updated column name for adding column
                // otherwise following alter stmt will report error:
                //
                // CREATE TABLE t1 (col1 varchar(50), col2 varchar(50),c5 varchar(50),c3 varchar(50) ,c4 varchar(50) primary key,c8 int);
                // ALTER TABLE t1 add c5 int,drop c3,rename column c5 to c3;
                //
                // do nothing
              } else if (OB_HASH_EXIST == update_column_name_set.exist_refactored(column_key)) {
                ret = OB_HASH_EXIST;
                LOG_WARN("duplicate column name", K(ret), K(column_name));
              } else if (OB_FAIL(update_column_name_set.set_refactored(column_key))) {
                LOG_WARN("failed to add column name to hash set.", K(column_name), K(ret));
              }
              if (OB_FAIL(ret)) {
              } else {
                need_redistribute_column_id = true;
              }
            }
            break;
          }
          case OB_DDL_CHANGE_COLUMN:
          case OB_DDL_MODIFY_COLUMN: {
            if (is_rename_first && alter_column_schema->alter_type_ == OB_DDL_CHANGE_COLUMN
                && is_rename_column(*alter_column_schema)) {
              // do nothing
            } else {
              ObColumnSchemaV2 new_column_schema;
              ObSEArray<ObString, 1> new_pk_column;
              bool is_change_column_order = false;
              orig_column_schema = new_table_schema.get_column_schema(orig_column_name);
              ObColumnNameHashWrapper orig_column_key(orig_column_name);
              if (OB_FAIL(prepare_change_modify_column_offline(
                                                       *alter_column_schema, origin_table_schema,
                                                       alter_table_schema, is_oracle_mode,
                                                       alter_table_arg, new_table_schema,
                                                       schema_checker, schema_guard,
                                                       update_column_name_set,
                                                       new_column_schema,
                                                       is_contain_part_key))) {
                LOG_WARN("prepare new column schema failed", K(ret));
              } else if (OB_FAIL(new_table_schema.alter_column(
                           new_column_schema, ObTableSchema::CHECK_MODE_OFFLINE, false))) {
                LOG_WARN("alter column failed", K(ret));
              } else if (alter_column_schema->is_primary_key_) {
                if (OB_FAIL(new_pk_column.push_back(alter_column_schema->get_column_name_str()))) {
                  LOG_WARN("push back element failed", K(ret));
                }
              }
              if (OB_FAIL(ret)) {
              } else if (!new_pk_column.empty() && OB_FAIL(add_primary_key(new_pk_column, new_table_schema))) {
                LOG_WARN("failed to add pk to table", K(ret), K(new_pk_column), K(new_table_schema));
              } else if (OB_FAIL(new_table_schema.check_primary_key_cover_partition_column())) {
                RS_LOG(WARN, "fail to check primary key cover partition column", K(ret));
              } else {
                if (OB_HASH_EXIST == update_column_name_set.exist_refactored(orig_column_key)) {
                  ret = OB_HASH_EXIST;
                  RS_LOG(WARN, "duplicate index name", K(ret), K(orig_column_name));
                } else if (OB_FAIL(update_column_name_set.set_refactored(orig_column_key))) {
                  RS_LOG(WARN, "failed to add index_name to hash set.", K(orig_column_name), K(ret));
                } else if (OB_FAIL(check_is_change_column_order(
                             new_table_schema, *alter_column_schema, is_change_column_order))) {
                  LOG_WARN("failed to check is change column order", K(ret));
                } else if (is_change_column_order) {
                  //in add column instant, same column name's column id is not the same
                  //no need to check column schema's column id
                  if (OB_FAIL(new_table_schema.reorder_column(
                               new_column_schema.get_column_name_str(),
                               alter_column_schema->is_first_,
                               alter_column_schema->get_prev_column_name(),
                               alter_column_schema->get_next_column_name()))) {
                    LOG_WARN("failed to reorder column", K(ret));
                  } else {
                    need_redistribute_column_id = true;
                  }
                }
              }
            }
            break;
          }
          case OB_DDL_ALTER_COLUMN: {
            ObSchemaChecker schema_checker;
            orig_column_schema = new_table_schema.get_column_schema(orig_column_name);
            ObColumnNameHashWrapper orig_column_key(orig_column_name);
            if (OB_FAIL(schema_checker.init(schema_guard))) {
              LOG_WARN("failed to init schema guard", K(ret));
            } else if (OB_ISNULL(orig_column_schema)) {
              ret = OB_ERR_BAD_FIELD_ERROR;
              LOG_USER_ERROR(OB_ERR_BAD_FIELD_ERROR, orig_column_name.length(), orig_column_name.ptr(),
                    origin_table_schema.get_table_name_str().length(), origin_table_schema.get_table_name_str().ptr());
              LOG_WARN("unknown column", KR(ret), K(orig_column_name), K(new_table_schema));
            } else if (OB_FAIL(pre_check_orig_column_schema(*alter_column_schema,
                                                     origin_table_schema,
                                                     update_column_name_set))) {
              RS_LOG(WARN, "failed to pre check orig column schema", K(ret));
            }
            //column that has been modified, can't not modify again
            if (OB_SUCC(ret)) {
              ObColumnSchemaV2 new_column_schema;
              bool for_view = false;
              if (OB_FAIL(new_column_schema.assign(*orig_column_schema))) {
                LOG_WARN("fail to assign column schema", KR(ret));
              } else if (OB_FAIL(resolve_timestamp_column(alter_column_schema,
                                                   new_table_schema,
                                                   new_column_schema,
                                                   tz_info_wrap,
                                                   nls_formats,
                                                   allocator))) {
                RS_LOG(WARN, "fail to resolve timestamp column", K(ret));
              } else if (OB_FAIL(new_table_schema.alter_column(new_column_schema,
                         ObTableSchema::CHECK_MODE_OFFLINE,
                         for_view))) {
                RS_LOG(WARN, "failed to change column", K(ret));
              } else {
                ObObj default_value;
                if (alter_column_schema->is_drop_default_) {
                  default_value.set_null();
                  if (OB_FAIL(new_column_schema.set_cur_default_value(default_value, false))) {
                    RS_LOG(WARN, "failed to set current default value", K(ret), K(default_value));
                  } else {
                    new_column_schema.del_column_flag(DEFAULT_EXPR_V2_COLUMN_FLAG);
                  }
                } else {
                  default_value = alter_column_schema->get_cur_default_value();
                  bool is_default_expr_v2 = alter_column_schema->is_default_expr_v2_column();
                  if (!default_value.is_null() && ob_is_text_tc(new_column_schema.get_data_type())) {
                    ret = OB_INVALID_DEFAULT;
                    LOG_USER_ERROR(OB_INVALID_DEFAULT, new_column_schema.get_column_name_str().length(),
                                                       new_column_schema.get_column_name_str().ptr());
                    RS_LOG(WARN, "BLOB, TEXT column can't have a default value!", K(default_value), K(ret));
                  } else if (ob_is_json_tc(new_column_schema.get_data_type())
                             || ob_is_geometry_tc(new_column_schema.get_data_type())) {
                    // cannot alter json column to any default value
                    // text column also cannot be alter to null in mysql
                    ret = OB_ERR_BLOB_CANT_HAVE_DEFAULT;
                    LOG_USER_ERROR(OB_ERR_BLOB_CANT_HAVE_DEFAULT, new_column_schema.get_column_name_str().length(),
                                   new_column_schema.get_column_name_str().ptr());
                    RS_LOG(WARN, "JSON column can't have a default value!", K(default_value), K(ret));
                  } else if (!new_column_schema.is_nullable() && default_value.is_null()) {
                    ret = OB_INVALID_DEFAULT;
                    LOG_USER_ERROR(OB_INVALID_DEFAULT, new_column_schema.get_column_name_str().length(),
                                   new_column_schema.get_column_name_str().ptr());
                    RS_LOG(WARN, "not null column with default value null!", K(ret));
                  } else if (OB_FAIL(new_column_schema.set_cur_default_value(default_value,
                                                                             is_default_expr_v2))) {
                    RS_LOG(WARN, "failed to set current default value", K(ret), K(default_value), K(is_default_expr_v2));
                  // The check_default_value function not only verifies the
                  // default value but also performs type conversion on it.
                  // Therefore, the cur_default_value from the column_schema
                  // should be passed when calling this function.
                  } else if (OB_FAIL(ObDDLResolver::check_default_value(new_column_schema.get_cur_default_value(),
                                                                        tz_info_wrap,
                                                                        nls_formats,
                                                                        &alter_table_arg.local_session_var_,
                                                                        allocator,
                                                                        new_table_schema,
                                                                        new_column_schema,
                                                                        gen_col_expr_arr,
                                                                        alter_table_schema.get_sql_mode(),
                                                                        !alter_column_schema->is_generated_column(), /* allow_sequence */
                                                                        &schema_checker))) {
                    LOG_WARN("fail to check default value", KPC(alter_column_schema),K(ret));
                  }
                }
              }
              if (OB_SUCC(ret)) {
                if (OB_FAIL(new_table_schema.alter_column(new_column_schema,
                            ObTableSchema::CHECK_MODE_OFFLINE,
                            for_view))) {
                  RS_LOG(WARN, "failed to change column", K(ret));
                } else if (OB_FAIL(new_table_schema.check_primary_key_cover_partition_column())) {
                  RS_LOG(WARN, "failed to check primary key cover partition column", K(ret));
                } else {
                  if (OB_HASH_EXIST == update_column_name_set.exist_refactored(orig_column_key)) {
                    ret = OB_HASH_EXIST;
                    RS_LOG(WARN, "duplicate index name", K(ret), K(orig_column_name));
                  } else if (OB_FAIL(update_column_name_set.set_refactored(orig_column_key))) {
                    RS_LOG(WARN, "failed to add index_name to hash set.",
                           K(orig_column_name), K(ret));
                  }
                }
              }
            }
            break;
          }
          default: {
            ret = OB_ERR_UNEXPECTED;
            RS_LOG(WARN, "invalid offline ddl operator type!", K_(alter_column_schema->alter_type));
            break;
          }
        }
      }
    }
    if (OB_SUCC(ret) && OB_FAIL(check_has_multi_autoinc(new_table_schema))) {
      LOG_WARN("failed to check table has multi autoinc", K(ret));
    }
    if (OB_SUCC(ret) && need_redistribute_column_id) {
      if (OB_FAIL(redistribute_column_ids(new_table_schema))) {
        LOG_WARN("failed to redistribute column ids", K(ret));
      } else {
        // do nothing
      }
    }
    if (OB_SUCC(ret)) {
      if (OB_FAIL(new_table_schema.sort_column_array_by_column_id())) {
        LOG_WARN("failed to sort column", K(ret), K(new_table_schema));
      }
    }
  }
  return ret;
}

// update relevant inner table if both ddl_operator and trans are not null
int ObDDLService::update_prev_id_for_add_column(const ObTableSchema &origin_table_schema,
    ObTableSchema &new_table_schema,
    AlterColumnSchema &alter_column_schema,
    ObDDLOperator *ddl_operator,
    common::ObMySQLTransaction *trans)
{
  int ret = OB_SUCCESS;
  const uint64_t tenant_id = origin_table_schema.get_tenant_id();
  const bool is_first = alter_column_schema.is_first_;
  const bool is_after = (!alter_column_schema.get_prev_column_name().empty());
  const bool is_before = (!alter_column_schema.get_next_column_name().empty());
  const bool is_last = !(is_first || is_after || is_before);
  const bool update_inner_table = nullptr != ddl_operator && nullptr != trans;
  const bool need_del_stats = false;
  if (is_last) {
    // do nothing
  } else {
    ObString pos_column_name;
    const uint64_t alter_column_id = alter_column_schema.get_column_id();
    if (is_first) {
      // this first means the first of no hidden/shdow column.
      ObColumnIterByPrevNextID iter(new_table_schema);
      const ObColumnSchemaV2 *head_col = NULL;
      const ObColumnSchemaV2 *col = NULL;
      bool is_first = false;
      while (OB_SUCC(ret) && OB_SUCC(iter.next(col))) {
        if (OB_ISNULL(col)) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("The column is null", K(ret));
        } else if (col->is_shadow_column() || col->is_hidden()) {
          // do nothing
        } else if (!is_first) {
          head_col = col;
          is_first = true;
        }
      }
      if (ret != OB_ITER_END) {
        LOG_WARN("Failed to iterate all table columns. iter quit. ", K(ret));
      } else {
        ret = OB_SUCCESS;
        if (OB_ISNULL(head_col)) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("Failed to get first column", K(ret));
        } else {
          alter_column_schema.set_next_column_name(head_col->get_column_name());
        }
      }
    }
    if (OB_SUCC(ret)) {
      pos_column_name = (is_after ? alter_column_schema.get_prev_column_name()
          : alter_column_schema.get_next_column_name());
      ObColumnSchemaV2 *pos_column_schema = new_table_schema.get_column_schema(pos_column_name);
      ObColumnSchemaV2 *update_column_schema = NULL;
      if (OB_ISNULL(pos_column_schema)) {
        ret = OB_ERR_BAD_FIELD_ERROR;
        LOG_USER_ERROR(OB_ERR_BAD_FIELD_ERROR, pos_column_name.length(), pos_column_name.ptr(),
                      new_table_schema.get_table_name_str().length(),
                      new_table_schema.get_table_name_str().ptr());
        LOG_WARN("pos column is NULL", K(pos_column_name));
      } else {
        if (is_after) {
          // add column after
          alter_column_schema.set_prev_column_id(pos_column_schema->get_column_id());
          update_column_schema = new_table_schema.get_column_schema_by_prev_next_id(pos_column_schema->get_next_column_id());
          if (OB_NOT_NULL(update_column_schema)) {
            update_column_schema->set_prev_column_id(alter_column_id);
          }
        } else {
          // add column before / first
          alter_column_schema.set_prev_column_id(pos_column_schema->get_prev_column_id());
          update_column_schema = pos_column_schema;
          update_column_schema->set_prev_column_id(alter_column_id);
        }
        if (OB_SUCC(ret)) {
          if (OB_ISNULL(update_column_schema)) {
            // alter column is the last column
          } else if (update_inner_table) {
            if (OB_FAIL(ddl_operator->update_single_column(
                        *trans,
                        origin_table_schema,
                        new_table_schema,
                        *update_column_schema,
                        need_del_stats))) {
              LOG_WARN("Failed to update single column", K(ret), K(update_column_schema->get_column_name_str()));
            }
          }
        }
      }
    }
  }
  return ret;
}

int ObDDLService::alter_table_column(const ObTableSchema &origin_table_schema,
                                     const AlterTableSchema &alter_table_schema,
                                     ObTableSchema &new_table_schema,
                                     obrpc::ObAlterTableArg &alter_table_arg,
                                     ObSchemaGetterGuard &schema_guard,
                                     const uint64_t tenant_data_version,
                                     ObDDLOperator &ddl_operator,
                                     common::ObMySQLTransaction &trans,
                                     ObIArray<ObTableSchema> *global_idx_schema_array/*=NULL*/)
{
  int ret = OB_SUCCESS;
  const common::ObTimeZoneInfoWrap &tz_info_wrap = alter_table_arg.tz_info_wrap_;
  const common::ObString *nls_formats = alter_table_arg.nls_formats_;
  common::ObIAllocator &allocator = alter_table_arg.allocator_;
  bool is_oracle_mode = false;
  bool for_view = false;
  const bool need_del_stats = false;
  LOG_INFO("check before alter table column", K(origin_table_schema),
  K(alter_table_schema), K(new_table_schema));
  ObSchemaChecker schema_checker;
  // drop column related.
  int64_t new_table_cols_cnt = 0;
  ObArray<uint64_t> drop_cols_id_arr;
  ObArray<uint64_t> unused_column_ids;
  if (OB_FAIL(schema_checker.init(schema_guard))) {
    LOG_WARN("failed to init schema guard", K(ret));
  } else if (OB_FAIL(origin_table_schema.check_if_oracle_compat_mode(is_oracle_mode))) {
    RS_LOG(WARN, "failed to get oracle mode", K(ret));
  } else if (OB_ISNULL(tz_info_wrap.get_time_zone_info())
             || OB_ISNULL(tz_info_wrap.get_time_zone_info()->get_tz_info_map())) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("invalid tz_info_wrap", K(tz_info_wrap), K(ret));
  } else if (OB_ISNULL(nls_formats)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("invalid nls_formats", K(ret));
  } else if (OB_FAIL(get_all_dropped_column_ids(alter_table_arg, origin_table_schema,
      drop_cols_id_arr, &new_table_cols_cnt/*final columns count of new table*/))) {
    LOG_WARN("fail to prefetch all drop columns id", K(ret), K(alter_table_arg));
  } else if (OB_FAIL(origin_table_schema.get_unused_column_ids(unused_column_ids))) {
    LOG_WARN("get unused columns failed", K(ret), K(origin_table_schema));
  } else {
    AlterColumnSchema *alter_column_schema;
    ObTableSchema::const_column_iterator it_begin = alter_table_schema.column_begin();
    ObTableSchema::const_column_iterator it_end = alter_table_schema.column_end();
    lib::Worker::CompatMode compat_mode = (is_oracle_mode ?
    lib::Worker::CompatMode::ORACLE : lib::Worker::CompatMode::MYSQL);
    lib::CompatModeGuard tmpCompatModeGuard(compat_mode);
    ObArray<ObTableSchema> idx_schema_array;
    common::hash::ObHashSet<ObColumnNameHashWrapper> update_column_name_set;
    ObSEArray<ObString, 4> gen_col_expr_arr;
    uint64_t curr_udt_set_id = 0;
    bool is_origin_table_has_lob_column = false;
    if (OB_FAIL(update_column_name_set.create(32))) {
      LOG_WARN("failed to create update column name set", K(ret));
    } else if (OB_FAIL(generate_tmp_idx_schemas(new_table_schema, idx_schema_array, schema_guard))) {
      LOG_WARN("generate tmp idx schemas failed", K(ret));
    } else {
      ObColumnIterByPrevNextID iter(origin_table_schema);
      while (OB_SUCC(ret)) {
        const share::schema::ObColumnSchemaV2 *column = nullptr;
        if (OB_FAIL(iter.next(column))) {
          if (OB_ITER_END == ret) {
            ret = OB_SUCCESS;
            break;
          } else {
            LOG_WARN("iter failed", KR(ret));
          }
        } else if (OB_ISNULL(column)) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("invalid column", KR(ret));
        } else if (column->is_generated_column()) {
          const common::ObObj* ObObjtmp = &column->get_cur_default_value();
          if (OB_FAIL(gen_col_expr_arr.push_back(ObObjtmp->get_string()))) {
            LOG_WARN("failed to pushback to gen_col_expr_arr", K(ret));
          }
        }
        if (OB_SUCC(ret) && is_lob_storage(column->get_data_type())) {
          is_origin_table_has_lob_column = true;
        }
      }
    }
    bool is_rename_first = false;
    if (OB_FAIL(ret)) {
    } else if (OB_FAIL(check_rename_first(alter_table_schema, origin_table_schema, is_oracle_mode,
                                          is_rename_first))) {
      LOG_WARN("check rename first failed", K(ret));
    }

    if (OB_SUCC(ret) && is_rename_first) {
      if (OB_FAIL(pre_rename_mysql_columns_online(
                   origin_table_schema, alter_table_schema, is_oracle_mode, new_table_schema,
                   alter_table_arg, schema_checker, ddl_operator, update_column_name_set, trans,
                   schema_guard, idx_schema_array, global_idx_schema_array))) {
        LOG_WARN("failed to pre alter change columns", K(ret));
      }
    }

    // Extended type info is resolved in session collation type, then we convert it to
    // system collation in ObDDLResolver::fill_extended_type_info().
    bool is_all_column_exactly_same_type = true;
    bool has_drop_column_op = false;
    const ObCollationType cur_extended_type_info_collation = ObCharset::get_system_collation();
    for(;OB_SUCC(ret) && it_begin != it_end; it_begin++) {
      if (OB_ISNULL(alter_column_schema = static_cast<AlterColumnSchema *>(*it_begin))) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("*it_begin is NULL", K(ret));
      } else if (OB_UNLIKELY(!drop_cols_id_arr.empty() && OB_DDL_DROP_COLUMN != alter_column_schema->alter_type_)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("drop column and other alter columns in single stmt simultaneously in trans", KR(ret), K(alter_table_arg));
      } else {
        const ObString &orig_column_name = alter_column_schema->get_origin_column_name();
        //column that has been alter, change or modify
        const ObColumnSchemaV2 *orig_column_schema = NULL;
        ObColumnSchemaV2 new_column_schema;
        switch (alter_column_schema->alter_type_) {
          case OB_DDL_DROP_COLUMN: {
            const bool overall_check_need =
              (alter_column_schema == static_cast<AlterColumnSchema *>(*alter_table_schema.column_begin()));
            if (overall_check_need) {
              if (OB_FAIL(ObDDLLock::lock_for_online_drop_column_in_trans(origin_table_schema, trans))) {
                // Table: row_exclusive, Online_DDL: exclusive
                // to conflict with all ddl.
                LOG_WARN("lock for online drop column in trans failed", KR(ret));
              } else if (OB_UNLIKELY(unused_column_ids.count() >= OB_MAX_UNUSED_COLUMNS_COUNT)) {
                ret = OB_OP_NOT_ALLOW;
                LOG_WARN("The number of obsolete columns reaches the limit", KR(ret));
                LOG_USER_ERROR(OB_OP_NOT_ALLOW, "The number of obsolete columns reaches the limit. Use \"alter table table_name force\" to defragment first, otherwise dropping column is");
              } else if (OB_UNLIKELY(drop_cols_id_arr.empty())) {
                ret = OB_ERR_UNEXPECTED;
                LOG_WARN("unexpected empty columns info", KR(ret), K(alter_table_arg));
              } else if (OB_FAIL(check_drop_column_with_drop_foreign_key(alter_table_arg,
                  origin_table_schema, drop_cols_id_arr))) {
                LOG_WARN("fail to check drop foreign key caused by drop column", KR(ret), K(alter_table_arg));
              } else if (OB_FAIL(check_drop_column_with_drop_constraint(alter_table_arg,
                  schema_guard, origin_table_schema, drop_cols_id_arr))) {
                LOG_WARN("fail to check drop constraint caused by drop column", KR(ret), K(alter_table_arg));
              } else {/* do nothing. */}
            }
            if (OB_SUCC(ret)) {
              if (OB_ISNULL(orig_column_schema = origin_table_schema.get_column_schema(orig_column_name))) {
                ret = OB_ERR_UNEXPECTED;
                LOG_WARN("invalid null column schema", KR(ret), KPC(orig_column_schema));
              } else if (is_oracle_mode && orig_column_schema->is_xmltype()) {
                ObSEArray<ObColumnSchemaV2 *, 1> hidden_cols;
                if (OB_FAIL(new_table_schema.get_column_schema_in_same_col_group(
                    orig_column_schema->get_column_id(), orig_column_schema->get_udt_set_id(), hidden_cols))) {
                  LOG_WARN("failed to get column schema", KR(ret));
                } else {
                  for (int64_t i = 0; OB_SUCC(ret) && i < hidden_cols.count(); i++) {
                    if (OB_FAIL(drop_column_online(schema_guard, origin_table_schema, hidden_cols.at(i)->get_column_name_str(),
                          new_table_cols_cnt, ddl_operator, trans, new_table_schema, update_column_name_set))) {
                      LOG_WARN("online drop column failed", KR(ret));
                    }
                  }
                }
              }
            }
            if (FAILEDx(drop_column_online(schema_guard, origin_table_schema, orig_column_name,
                  new_table_cols_cnt, ddl_operator, trans, new_table_schema, update_column_name_set))) {
              LOG_WARN("online drop column failed", KR(ret));
            }
            break;
          }
          case OB_DDL_ADD_COLUMN: {
            if (OB_UNLIKELY(unused_column_ids.count() >= OB_MAX_UNUSED_COLUMNS_COUNT)) {
              ret = OB_OP_NOT_ALLOW;
              LOG_WARN("The number of obsolete columns reaches the limit", KR(ret));
              LOG_USER_ERROR(OB_OP_NOT_ALLOW, "The number of obsolete columns reaches the limit. Use \"alter table table_name force\" to defragment first, otherwise adding column instantly is");
            } else if (OB_FAIL(add_new_column_to_table_schema(origin_table_schema,
                                                       alter_table_schema,
                                                       tz_info_wrap,
                                                       *nls_formats,
                                                       alter_table_arg.local_session_var_,
                                                       alter_table_arg.sequence_ddl_arg_,
                                                       alter_table_arg.allocator_,
                                                       new_table_schema,
                                                       *alter_column_schema,
                                                       gen_col_expr_arr,
                                                       schema_guard,
                                                       curr_udt_set_id,
                                                       &ddl_operator,
                                                       &trans))) {
              LOG_WARN("failed to add new column to table schema", K(ret));
            }
            break;
          }
          case OB_DDL_CHANGE_COLUMN: {
            if (is_rename_first && is_rename_column(*alter_column_schema)) { break; }
            orig_column_schema = new_table_schema.get_column_schema(orig_column_name);
            if (OB_FAIL(prepare_change_modify_column_online(
                         *alter_column_schema, origin_table_schema, alter_table_schema,
                         is_oracle_mode, alter_table_arg, new_table_schema, schema_checker,
                         ddl_operator, trans, schema_guard, global_idx_schema_array,
                         update_column_name_set, new_column_schema))) {
              LOG_WARN("prepare alter column failed", K(ret));
            } else if (OB_FAIL(new_table_schema.alter_column(
                         new_column_schema, ObTableSchema::CHECK_MODE_ONLINE, for_view))) {
              LOG_WARN("failed to alter column", K(ret));
            } else if (OB_FAIL(check_new_column_for_index(idx_schema_array, new_column_schema))) {
              RS_LOG(WARN, "failed to check new column for index", K(ret));
            } else if (OB_FAIL(new_table_schema.check_primary_key_cover_partition_column())) {
              RS_LOG(WARN, "fail to check primary key cover partition column", K(ret));
            } else if (OB_FAIL(ddl_operator.update_column_and_column_group(trans, origin_table_schema, new_table_schema,
                                                                           new_column_schema, need_del_stats))) {
              LOG_WARN("failed to update column and column group", KR(ret), K(new_column_schema));
            } else if (OB_FAIL(alter_shadow_column_for_index(idx_schema_array, alter_column_schema, new_column_schema, ddl_operator, trans))) {
              RS_LOG(WARN, "failed to alter shadow column for index", K(ret));
            } else if (OB_FAIL(alter_table_update_index_and_view_column(
                                 new_table_schema,
                                 new_column_schema,
                                 ddl_operator,
                                 trans,
                                 global_idx_schema_array))) {
              RS_LOG(WARN, "failed to update index column", K(ret));
            } else {
              ObColumnNameHashWrapper orig_column_key(orig_column_name);
              if (OB_FAIL(update_column_name_set.exist_refactored(orig_column_key))) {
                if (OB_HASH_EXIST == ret) {
                  RS_LOG(WARN, "duplicate index name", K(ret), K(orig_column_name));
                } else if (OB_HASH_NOT_EXIST == ret) {
                  ret = OB_SUCCESS;
                } else {
                  RS_LOG(WARN, "exist refactored failed", K(ret));
                }
              }
              if (OB_FAIL(ret)) {
              } else if (OB_FAIL(update_column_name_set.set_refactored(orig_column_key))) {
                RS_LOG(WARN, "failed to add index_name to hash set.", K(orig_column_name), K(ret));
              }
            }
            break;
          }
          case OB_DDL_MODIFY_COLUMN: {
            LOG_DEBUG("check alter column schema", KPC(alter_column_schema));
            orig_column_schema = new_table_schema.get_column_schema(orig_column_name);
            ObColumnNameHashWrapper orig_column_key(orig_column_name);
            if (alter_column_schema->is_generated_column()
                && OB_FAIL(ObDDLResolver::reformat_generated_column_expr(
                                                alter_column_schema->get_cur_default_value(),
                                                tz_info_wrap,
                                                nls_formats,
                                                orig_column_schema->get_local_session_var(),
                                                allocator,
                                                new_table_schema,
                                                *alter_column_schema,
                                                alter_table_schema.get_sql_mode(),
                                                &schema_checker))) {
              LOG_WARN("fail to check default value", KPC(alter_column_schema), K(ret));
            } else if (OB_FAIL(pre_check_orig_column_schema(*alter_column_schema,
                                                            origin_table_schema,
                                                            update_column_name_set))) {
              RS_LOG(WARN, "failed to pre check orig column schema", K(ret));
            } else if (!alter_column_schema->is_generated_column()   /* Not support modify to generate columns, so there is no need to check again here */
                      && !alter_column_schema->is_udt_related_column(is_oracle_mode)  /* udt default values are checked in resolver */
                      && OB_FAIL(ObDDLResolver::check_default_value(alter_column_schema->get_cur_default_value(),
                                                                    tz_info_wrap,
                                                                    nls_formats,
                                                                    NULL,
                                                                    allocator,
                                                                    new_table_schema,
                                                                    *alter_column_schema,
                                                                    gen_col_expr_arr,
                                                                    alter_table_schema.get_sql_mode(),
                                                                    !alter_column_schema->is_generated_column(), /* allow_sequence */
                                                                    &schema_checker))) {
              LOG_WARN("fail to check default value", KPC(alter_column_schema), K(ret));
            } else if (OB_FAIL(alter_sequence_in_alter_column(new_table_schema,
                                                              *alter_column_schema,
                                                              trans,
                                                              schema_guard,
                                                              alter_table_arg.sequence_ddl_arg_))) {
              LOG_WARN("alter table modeify identity column fail", K(alter_column_schema), K(ret));
            } else if (alter_column_schema->is_primary_key_) {
              if (new_table_schema.get_rowkey_column_num() > 0) {
                if (new_table_schema.is_table_without_pk()) {
                  ret = OB_NOT_SUPPORTED;
                  RS_LOG(WARN, "not support to add primary key!", K(ret));
                } else {
                  ret = OB_ERR_MULTIPLE_PRI_KEY;
                  RS_LOG(WARN, "multiple primary key defined", K(ret));
                }
              }
            }
            if (OB_SUCC(ret) && alter_column_schema->is_autoincrement_) {
              if (alter_column_schema->is_autoincrement()) {
                if (orig_column_schema->get_column_id() != new_table_schema.get_autoinc_column_id()) {
                  // not supported now; from non-auto-increment column to auto-increment column
                  ret = OB_NOT_SUPPORTED;
                  RS_LOG(WARN, "from non-auto-increment column to auto-increment column", K(ret));
                }
              }
            }

            if (OB_SUCC(ret)) {
              if (OB_FAIL(check_can_alter_column_type(*orig_column_schema, *alter_column_schema, origin_table_schema, is_oracle_mode))) {
                LOG_WARN("fail to check can alter column type", K(ret));
              }
            }

            if (OB_SUCC(ret)) {
              if (OB_FAIL(new_column_schema.assign(*orig_column_schema))) {
                LOG_WARN("fail to assign column schema", KR(ret));
              } else if (OB_FAIL(fill_new_column_attributes(*alter_column_schema,
                                                     new_column_schema))) {
                RS_LOG(WARN, "fail to fill new column attributes", K(ret));
              } else if (OB_FAIL(check_modify_column_when_upgrade(new_column_schema, *orig_column_schema))) {
                LOG_WARN("fail to check modify column when upgrade",
                         K(ret), K(new_column_schema), K(*orig_column_schema));
              } else if (OB_FAIL(resolve_timestamp_column(alter_column_schema,
                                                          new_table_schema,
                                                          new_column_schema,
                                                          tz_info_wrap,
                                                          nls_formats,
                                                          allocator))) {
                RS_LOG(WARN, "fail to resolve timestamp column", K(ret));
              } else if (OB_FAIL(new_table_schema.alter_column(new_column_schema,
                                 ObTableSchema::CHECK_MODE_ONLINE,
                                 for_view))) {
                RS_LOG(WARN, "failed to change column", K(ret));
              } else if (OB_FAIL(check_new_column_for_index(
                                 idx_schema_array,
                                 new_column_schema))) {
                RS_LOG(WARN, "failed to check new column for index", K(ret));
              } else if (OB_FAIL(new_table_schema.check_primary_key_cover_partition_column())) {
                RS_LOG(WARN, "fail to check primary key cover partition column", K(ret));
              } else if (OB_FAIL(ddl_operator.update_single_column(
                                 trans,
                                 origin_table_schema,
                                 new_table_schema,
                                 new_column_schema,
                                 need_del_stats))) {
                RS_LOG(WARN, "failed to alter column", K(alter_column_schema), K(ret));
              } else if (OB_FAIL(alter_shadow_column_for_index(idx_schema_array, alter_column_schema, new_column_schema, ddl_operator, trans))) {
                RS_LOG(WARN, "failed to alter shadow column for index", K(ret));
              } else if (OB_FAIL(alter_table_update_index_and_view_column(new_table_schema,
                                                                          new_column_schema,
                                                                          ddl_operator,
                                                                          trans))) {
                RS_LOG(WARN, "failed to update index column", K(ret));
              } else {
                if (OB_HASH_EXIST == update_column_name_set.exist_refactored(orig_column_key)) {
                  ret = OB_HASH_EXIST;
                  RS_LOG(WARN, "duplicate index name", K(ret), K(orig_column_name));
                } else if (OB_FAIL(update_column_name_set.set_refactored(orig_column_key))) {
                  RS_LOG(WARN, "failed to add index_name to hash set.",
                         K(orig_column_name), K(ret));
                }
              }
            }
            break;
          }
          case OB_DDL_ALTER_COLUMN: {
            orig_column_schema = new_table_schema.get_column_schema(orig_column_name);
            ObColumnNameHashWrapper orig_column_key(orig_column_name);
            if (OB_FAIL(pre_check_orig_column_schema(*alter_column_schema,
                                                     origin_table_schema,
                                                     update_column_name_set))) {
              RS_LOG(WARN, "failed to pre check orig column schema", K(ret));
            }

            //column that has been modified, can't not modify again
            if (OB_SUCC(ret)) {
              if (OB_FAIL(new_column_schema.assign(*orig_column_schema))) {
                LOG_WARN("fail to assign column schema", KR(ret));
              } else if (OB_FAIL(resolve_timestamp_column(alter_column_schema,
                                                   new_table_schema,
                                                   new_column_schema,
                                                   tz_info_wrap,
                                                   nls_formats,
                                                   allocator))) {
                RS_LOG(WARN, "fail to resolve timestamp column", K(ret));
              } else {
                ObObj default_value;
                if (alter_column_schema->is_drop_default_) {
                  default_value.set_null();
                  if (OB_FAIL(new_column_schema.set_cur_default_value(default_value, false))) {
                    RS_LOG(WARN, "failed to set current default value", K(ret), K(default_value));
                  } else {
                    new_column_schema.del_column_flag(DEFAULT_EXPR_V2_COLUMN_FLAG);
                  }
                } else {
                  bool is_default_expr_v2 = alter_column_schema->is_default_expr_v2_column();
                  default_value = alter_column_schema->get_cur_default_value();
                  if (!default_value.is_null() && ob_is_text_tc(new_column_schema.get_data_type())) {
                    ret = OB_INVALID_DEFAULT;
                    LOG_USER_ERROR(OB_INVALID_DEFAULT, new_column_schema.get_column_name_str().length(),
                                                       new_column_schema.get_column_name_str().ptr());
                    RS_LOG(WARN, "BLOB, TEXT column can't have a default value!", K(default_value), K(ret));
                  } else if (ob_is_json_tc(new_column_schema.get_data_type())
                             || ob_is_geometry_tc(new_column_schema.get_data_type())) {
                    // cannot alter json column to any default value
                    // text column also cannot be alter to null in mysql
                    ret = OB_ERR_BLOB_CANT_HAVE_DEFAULT;
                    LOG_USER_ERROR(OB_ERR_BLOB_CANT_HAVE_DEFAULT, new_column_schema.get_column_name_str().length(),
                                   new_column_schema.get_column_name_str().ptr());
                    RS_LOG(WARN, "JSON column can't have a default value!", K(default_value), K(ret));
                  } else if (!new_column_schema.is_nullable() && default_value.is_null()) {
                    ret = OB_INVALID_DEFAULT;
                    LOG_USER_ERROR(OB_INVALID_DEFAULT, new_column_schema.get_column_name_str().length(),
                                   new_column_schema.get_column_name_str().ptr());
                    RS_LOG(WARN, "not null column with default value null!", K(ret));
                  } else if (OB_FAIL(new_column_schema.set_cur_default_value(default_value,
                                                                             is_default_expr_v2))) {
                    RS_LOG(WARN, "failed to set current default value", K(ret), K(default_value), K(is_default_expr_v2));
                    // The check_default_value function not only verifies the
                    // default value but also performs type conversion on it.
                    // Therefore, the cur_default_value from the column_schema
                    // should be passed when calling this function.
                  } else if (OB_FAIL(ObDDLResolver::check_default_value(new_column_schema.get_cur_default_value(),
                                                                        tz_info_wrap,
                                                                        nls_formats,
                                                                        &alter_table_arg.local_session_var_,
                                                                        allocator,
                                                                        new_table_schema,
                                                                        new_column_schema,
                                                                        gen_col_expr_arr,
                                                                        alter_table_schema.get_sql_mode(),
                                                                        !alter_column_schema->is_generated_column(), /* allow_sequence */
                                                                        &schema_checker))) {
                    LOG_WARN("fail to check default value", K(new_column_schema),K(ret));
                  }
                }
              }
              if (OB_SUCC(ret)) {
                if (OB_FAIL(new_table_schema.alter_column(new_column_schema,
                            ObTableSchema::CHECK_MODE_ONLINE,
                            for_view))) {
                  RS_LOG(WARN, "failed to change column", K(ret));
                } else if (OB_FAIL(new_table_schema.check_primary_key_cover_partition_column())) {
                  RS_LOG(WARN, "failed to check primary key cover partition column", K(ret));
                } else if (OB_FAIL(ddl_operator.update_single_column(trans,
                                                                     origin_table_schema,
                                                                     new_table_schema,
                                                                     new_column_schema,
                                                                     need_del_stats))) {
                  RS_LOG(WARN, "failed to alter column", K(alter_column_schema), K(ret));
                } else if (OB_FAIL(alter_table_update_index_and_view_column(new_table_schema,
                                                                            new_column_schema,
                                                                            ddl_operator,
                                                                            trans))) {
                  RS_LOG(WARN, "failed to update index column", K(ret));
                } else {
                  if (OB_HASH_EXIST == update_column_name_set.exist_refactored(orig_column_key)) {
                    ret = OB_HASH_EXIST;
                    RS_LOG(WARN, "duplicate index name", K(ret), K(orig_column_name));
                  } else if (OB_FAIL(update_column_name_set.set_refactored(orig_column_key))) {
                    RS_LOG(WARN, "failed to add index_name to hash set.",
                           K(orig_column_name), K(ret));
                  }
                }
              }
            }
            break;
          }
          default: {
            ret = OB_INVALID_ARGUMENT;
            RS_LOG(WARN, "unhandled operator type!", K_(alter_column_schema->alter_type));
            break;
          }
        }

        if (OB_SUCC(ret) && is_all_column_exactly_same_type && OB_NOT_NULL(orig_column_schema)) {
          if (OB_FAIL(ObTableSchema::check_is_exactly_same_type(*orig_column_schema, new_column_schema, is_all_column_exactly_same_type))) {
            RS_LOG(WARN, "check column type exactly same failed", K(ret), KPC(orig_column_schema), K(new_column_schema));
          }
        }
      }
    }

    bool is_add_lob = false;
    if(OB_FAIL(ret)) {
    } else if (OB_FAIL(add_column_to_column_group(origin_table_schema,
        alter_table_schema, new_table_schema, ddl_operator, trans))) {
      LOG_WARN("fail to add_column_to_column_group", K(ret), K(alter_table_schema), K(new_table_schema));
    } else if (OB_FAIL(new_table_schema.check_skip_index_valid())) {
      LOG_WARN("failed to check new table schema skip index", K(ret));
    } else if (OB_FAIL(new_table_schema.check_row_length(is_oracle_mode))) {
      LOG_WARN("failed to check_row_length", K(ret), K(new_table_schema));
    } else if (!is_origin_table_has_lob_column) {
      if (OB_FAIL(create_aux_lob_table_if_need(
          new_table_schema, schema_guard, ddl_operator, trans,
          false/*need_sync_schema_version*/, is_add_lob))) {
        LOG_WARN("fail to create_aux_lob_table_if_need", K(ret), K(new_table_schema));
      }
    }
    if (OB_SUCC(ret) && !is_add_lob && drop_cols_id_arr.empty()) {
      const bool require_strict_binary_format = share::ObDDLUtil::use_idempotent_mode(tenant_data_version) && !is_all_column_exactly_same_type;
      if (OB_FAIL(ObDDLLock::lock_for_common_ddl_in_trans(new_table_schema, require_strict_binary_format, trans))) {
        LOG_WARN("failed to lock ddl lock", K(ret));
      }
    }
  }

  return ret;
}

int ObDDLService::create_aux_lob_table_if_need(ObTableSchema &data_table_schema,
                                               ObSchemaGetterGuard &schema_guard,
                                               ObDDLOperator &ddl_operator,
                                               common::ObMySQLTransaction &trans,
                                               const bool need_sync_schema_version,
                                               bool &is_add_lob)
{
  int ret = OB_SUCCESS;
  is_add_lob = false;
  ObArray<ObTableSchema> aux_table_schemas;
  const uint64_t tenant_id = data_table_schema.get_tenant_id();
  SCN frozen_scn;
  uint64_t tenant_data_version = 0;
  ObArray<bool> need_create_empty_majors;

  if (OB_FAIL(ObMajorFreezeHelper::get_frozen_scn(tenant_id, frozen_scn))) {
    LOG_WARN("failed to get frozen status for create tablet", KR(ret), K(tenant_id));
  } else if (OB_FAIL(build_aux_lob_table_schema_if_need(data_table_schema, aux_table_schemas))) {
    LOG_WARN("fail to build_aux_lob_table_schema_if_need", K(ret), K(data_table_schema));
  } else if (aux_table_schemas.count() == 0) {
    // no need create aux lob table, do nothing
  } else if (is_sys_table(data_table_schema.get_table_id())) {
    // whatever enable_sys_table_ddl is, sys table do not allow create lob aux tables by alter columns
    char err_msg[number::ObNumber::MAX_PRINTABLE_SIZE];
    ret = OB_OP_NOT_ALLOW;
    (void)snprintf(err_msg, sizeof(err_msg),"%s", "system table add or modify column");
    LOG_USER_ERROR(OB_OP_NOT_ALLOW, err_msg);
  } else if (OB_FAIL(GET_MIN_DATA_VERSION(tenant_id, tenant_data_version))) {
    LOG_WARN("get min data version failed", K(ret), K(tenant_id));
  } else {
    ObTableCreator table_creator(
                   tenant_id,
                   frozen_scn,
                   trans);
    ObNewTableTabletAllocator new_table_tablet_allocator(
                              tenant_id,
                              schema_guard,
                              sql_proxy_);
    int64_t last_schema_version = OB_INVALID_VERSION;
    is_add_lob = true;
    if (OB_FAIL(get_last_schema_version(last_schema_version))) {
      LOG_WARN("fail to get last schema version", KR(ret));
    } else if (OB_FAIL(ObDDLLock::lock_for_add_lob_in_trans(data_table_schema, trans))) {
      LOG_WARN("failed to add lock online ddl lock", K(ret));
    } else if (OB_FAIL(table_creator.init(true/*need_tablet_cnt_check*/))) {
      LOG_WARN("fail to init table creator", KR(ret));
    } else if (OB_FAIL(new_table_tablet_allocator.init())) {
      LOG_WARN("fail to init new table tablet allocator", KR(ret));
    }

    ObSEArray<const ObTableSchema*, 2> schemas;
    for (int64_t i = 0; OB_SUCC(ret) && i < aux_table_schemas.count(); i++) {
      share::schema::ObTableSchema &table_schema = aux_table_schemas.at(i);
      if (OB_FAIL(ddl_operator.create_table(table_schema, trans, NULL,
        need_sync_schema_version && (i == aux_table_schemas.count() - 1)))) {
        LOG_WARN("failed to create table schema", K(ret));
      } else if (OB_FAIL(schemas.push_back(&table_schema))
        || OB_FAIL(need_create_empty_majors.push_back(true))) {
        LOG_WARN("failed to push_back table schema", K(ret), K(table_schema));
      }

      if (OB_SUCC(ret)) {
        if (OB_FAIL(ddl_operator.insert_ori_schema_version(
            trans, tenant_id, table_schema.get_table_id(), last_schema_version))) {
          LOG_WARN("failed to insert_ori_schema_version!", K(ret), K(table_schema), K(last_schema_version));
        }
      }
    }
    if (OB_SUCC(ret)) {
      common::ObArray<share::ObLSID> ls_id_array;
      if(OB_FAIL(new_table_tablet_allocator.prepare_like(data_table_schema))) {
        LOG_WARN("fail to prepare like", KR(ret), K(data_table_schema));
      } else if (OB_FAIL(new_table_tablet_allocator.get_ls_id_array(ls_id_array))) {
        LOG_WARN("fail to get ls id array", KR(ret));
      } else if (OB_FAIL(table_creator.add_create_tablets_of_local_aux_tables_arg(
          schemas, &data_table_schema, ls_id_array, tenant_data_version,
          need_create_empty_majors /*online_ddl, need_create_empty_major_sstable*/))) {
        LOG_WARN("create table partitions failed", KR(ret), K(last_schema_version));
      } else if (OB_FAIL(table_creator.execute())) {
        LOG_WARN("fail to execute crate tablet", KR(ret));
      }
    }
    // finishing is always invoked for new table tablet allocator
    int tmp_ret = OB_SUCCESS;
    if (OB_SUCCESS != (tmp_ret = new_table_tablet_allocator.finish(OB_SUCCESS == ret))) {
      LOG_WARN("fail to finish new table tablet allocator", KR(tmp_ret));
    }
    FLOG_INFO("finish create aux lob table for data table", K(ret), K(data_table_schema));
  }

  return ret;
}

int ObDDLService::alter_table_foreign_keys(const share::schema::ObTableSchema &orig_table_schema,
                                           share::schema::ObTableSchema &inc_table_schema,
                                           ObDDLOperator &ddl_operator,
                                           common::ObMySQLTransaction &trans)
{
  int ret = OB_SUCCESS;
  if (OB_FAIL(ddl_operator.add_table_foreign_keys(
      orig_table_schema,
      inc_table_schema,
      trans))) {
    LOG_WARN("failed to add table foreign_keys", K(ret));
  }
  return ret;
}

int ObDDLService::drop_not_null_cst_in_column_flag(
    const ObTableSchema &orig_table_schema,
    const AlterTableSchema &alter_table_schema,
    ObTableSchema &new_table_schema,
    ObDDLOperator &ddl_operator,
    ObMySQLTransaction &trans)
{
  int ret = OB_SUCCESS;
  const bool need_del_stats = false;
  int64_t col_cnt = new_table_schema.get_column_count();

  common::hash::ObHashSet<uint64_t> drop_col_id;
  ObTableSchema::const_column_iterator it_begin = alter_table_schema.column_begin();
  ObTableSchema::const_column_iterator it_end = alter_table_schema.column_end();
  /* get column in drop */
  int64_t hash_set_size = alter_table_schema.get_column_count() > 0 ? alter_table_schema.get_column_count() : 1;
  if (OB_FAIL(drop_col_id.create(hash_set_size,
                                 lib::ObLabel("ChkDropCol"),
                                 lib::ObLabel("ChkDropCol")))) {
    LOG_WARN("failed to init drop column set", K(ret), K(alter_table_schema));
  }
  for (; OB_SUCC(ret) && it_begin != it_end; it_begin ++) {
    const AlterColumnSchema *alter_col = static_cast<AlterColumnSchema*>(*it_begin);
    if (OB_ISNULL(alter_col)) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("alter col should not be null", K(ret), K(alter_table_schema));
    } else if (OB_DDL_DROP_COLUMN != alter_col->alter_type_) {
    } else {
      const ObColumnSchemaV2 *orig_col = orig_table_schema.get_column_schema(alter_col->get_origin_column_name());
      if (OB_ISNULL(orig_col)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("orig col should not be null", K(ret), K(orig_table_schema), K(alter_table_schema));
      } else if (OB_FAIL(drop_col_id.set_refactored(orig_col->get_column_id()))) {
        LOG_WARN("failed to set refactor", K(ret));
      }
    }
  }

  for (ObTableSchema::const_constraint_iterator iter = alter_table_schema.constraint_begin(); OB_SUCC(ret) &&
      iter != alter_table_schema.constraint_end(); iter ++) {
    if (CONSTRAINT_TYPE_NOT_NULL == (*iter)->get_constraint_type()) {
      const uint64_t column_id = *((*iter)->cst_col_begin());
      ObColumnSchemaV2 *column = new_table_schema.get_column_schema(column_id);
      if (OB_ISNULL(column)) {
        /* if col in drop, skip */
        if (OB_HASH_EXIST != drop_col_id.exist_refactored(column_id)) {
          if (ret == OB_HASH_NOT_EXIST) {
            ret = OB_ERR_UNEXPECTED;
            LOG_WARN("invalid column", K(ret), K(column_id), K(alter_table_schema));
          } else {
            LOG_WARN("failed to get check col in drop col list", K(ret), K(column_id));
          }
        }
      } else if (column->has_not_null_constraint()) {
        column->drop_not_null_cst();
        if (OB_FAIL(ddl_operator.update_single_column(
            trans, orig_table_schema, new_table_schema, *column, need_del_stats))) {
          LOG_WARN("failed to update single column", K(ret));
        }
      }
    }
  }
  return ret;
}

int ObDDLService::alter_not_null_cst_in_column_flag(
    const ObTableSchema &orig_table_schema,
    const AlterTableSchema &alter_table_schema,
    ObTableSchema &new_table_schema,
    ObDDLOperator &ddl_operator,
    ObMySQLTransaction &trans)
{
  int ret = OB_SUCCESS;
  const bool need_del_stats = false;
  for (ObTableSchema::const_constraint_iterator iter = alter_table_schema.constraint_begin(); OB_SUCC(ret) &&
      iter != alter_table_schema.constraint_end(); iter ++) {
    if (CONSTRAINT_TYPE_NOT_NULL == (*iter)->get_constraint_type()) {
      const uint64_t column_id = *((*iter)->cst_col_begin());
      ObColumnSchemaV2 *column = new_table_schema.get_column_schema(column_id);
      if (OB_ISNULL(column)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("invalid column", K(ret));
      } else {
        int64_t column_flags = column->get_column_flags();
        column->add_or_del_column_flag(NOT_NULL_ENABLE_FLAG, (*iter)->get_enable_flag());
        column->add_or_del_column_flag(NOT_NULL_RELY_FLAG, (*iter)->get_rely_flag());
        column->add_or_del_column_flag(NOT_NULL_VALIDATE_FLAG, (*iter)->is_validated());
        if (column_flags != column->get_column_flags()
            && OB_FAIL(ddl_operator.update_single_column(
                       trans, orig_table_schema, new_table_schema, *column, need_del_stats))) {
          LOG_WARN("failed to update single column", K(ret));
        }
      }
    }
  }
  return ret;
}

int ObDDLService::alter_table_auto_increment(
    const ObTableSchema &orig_table_schema,
    const AlterTableSchema &alter_table_schema,
    const obrpc::ObAlterTableArg &alter_table_arg,
    share::schema::ObSchemaGetterGuard &schema_guard,
    ObTableSchema &new_table_schema,
    ObDDLOperator &ddl_operator,
    ObMySQLTransaction &trans)
{
  int ret = OB_SUCCESS;
  ObAutoincrementService &auto_inc_service = ObAutoincrementService::get_instance();
  const uint64_t tenant_id = orig_table_schema.get_tenant_id();
  const uint64_t table_id = orig_table_schema.get_table_id();
  const uint64_t column_id = orig_table_schema.get_autoinc_column_id();
  const bool is_order_mode = orig_table_schema.is_order_auto_increment_mode();
  const int64_t truncate_version = orig_table_schema.get_truncate_version();
  uint64_t current_auto_increment = 0;
  // Step 1: Determine whether the auto-increment value needs to be increased or decreased.
  // If increased, only the table schema needs to be updated. It reduces the complexity of the
  // alter_table operation.
  const bool is_reduced_autoinc =
      (alter_table_schema.get_auto_increment() < orig_table_schema.get_auto_increment());
  if (!is_reduced_autoinc && OB_FAIL(auto_inc_service.get_sequence_value(tenant_id,
                table_id, column_id, is_order_mode, truncate_version, current_auto_increment))) {
    LOG_WARN("fail to get sequence value", K(ret));
  } else if (is_reduced_autoinc ||
      (alter_table_schema.get_auto_increment() < current_auto_increment)) {
    // Step 2: Query the maximum value of the auto-increment column and use it as a base value.
    const ObDatabaseSchema *db_schema = nullptr;
    const ObColumnSchemaV2 *column_schema = nullptr;
    if (OB_FAIL(schema_guard.get_database_schema(tenant_id, orig_table_schema.get_database_id(),
                                                 db_schema))) {
      LOG_WARN("failed to get database schema", K(ret), K(orig_table_schema.get_data_table_id()));
    } else if (OB_ISNULL(db_schema)) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("error unexpected, database schema must not be nullptr", K(ret));
    } else if (OB_ISNULL(column_schema = orig_table_schema.get_column_schema(column_id))) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("failed to get alter column schema", K(ret), K(column_id));
    } else if (OB_FAIL(lock_table(trans, orig_table_schema))) {
      LOG_WARN("failed to lock table for alter auto_increment", K(ret));
    } else {
      uint64_t current_max_value = 0;
      SMART_VAR(ObMySQLProxy::MySQLResult, res) {
        ObTimeoutCtx timeout_ctx;
        ObSqlString sql;
        sqlclient::ObMySQLResult *result = NULL;
        common::ObCommonSqlProxy *user_sql_proxy = GCTX.ddl_sql_proxy_;
        ObSessionParam session_param;
        int64_t sql_mode = alter_table_arg.sql_mode_;
        session_param.sql_mode_ = reinterpret_cast<int64_t *>(&sql_mode);
        session_param.ddl_info_.set_is_ddl(true);
        // if data_table_id != dest_table_id, meaning this is happening in ddl double write
        session_param.ddl_info_.set_source_table_hidden(orig_table_schema.is_user_hidden_table());
        ObObj obj;
        const int64_t DDL_INNER_SQL_EXECUTE_TIMEOUT = ObDDLUtil::calc_inner_sql_execute_timeout();
        const bool is_unsigned_type = ob_is_unsigned_type(column_schema->get_data_type());
        if (OB_FAIL(timeout_ctx.set_trx_timeout_us(DDL_INNER_SQL_EXECUTE_TIMEOUT))) {
          LOG_WARN("set trx timeout failed", K(ret));
        } else if (OB_FAIL(timeout_ctx.set_timeout(DDL_INNER_SQL_EXECUTE_TIMEOUT))) {
          LOG_WARN("set timeout failed", K(ret));
        } else if (OB_FAIL(sql.assign_fmt("SELECT /*+no_rewrite*/ CAST(MAX(`%s`) AS %s) AS MAX_VALUE FROM `%s`.`%s`",
                                    column_schema->get_column_name(),
                                    is_unsigned_type ? "UNSIGNED" : "SIGNED",
                                    db_schema->get_database_name(),
                                    orig_table_schema.get_table_name()))) {
          LOG_WARN("failed to assign fmt", K(ret), K(column_schema->get_column_name_str()),
                                           K(db_schema->get_database_name_str()));
        } else if (OB_FAIL(user_sql_proxy->read(res, tenant_id, sql.ptr(), &session_param))) {
          LOG_WARN("fail to read", KR(ret), K(sql));
        } else if (OB_ISNULL(result = res.get_result())) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("get result failed", K(ret));
        } else if (OB_FAIL(result->next())) {
          if (OB_ITER_END == ret) {
            // empty table
            ret = OB_SUCCESS;
          } else {
            LOG_WARN("fail to get next", KR(ret));
          }
        } else if (OB_FAIL(result->get_obj("MAX_VALUE", obj))) {
          LOG_WARN("fail to get result obj", K(ret));
        } else if (is_unsigned_type) {
          current_max_value = obj.get_uint64();
        } else {
          current_max_value = MAX(0, obj.get_int());
        }
      }

      // Step 3: Clear the auto increment cache and reset the inner table of the auto-increment
      // column. all new auto_inc request will be based on the auto_increment value in the table
      // schema.
      if (OB_FAIL(ret)) {
      } else if (OB_FAIL(auto_inc_service.clear_autoinc_cache_all(tenant_id, table_id,
                                                                  column_id, is_order_mode,
                                                                  false /* ignore_rpc_errors */))) {
        LOG_WARN("fail to clear autoinc cache all", K(ret));
      } else if (OB_FAIL(ddl_operator.reinit_autoinc_row(new_table_schema, trans))) {
        LOG_WARN("fail to reinit autoinc row", K(ret));
      } else {
        // Step 4: Update the table schema with the maximum values of auto_increment by setting and
        // max_value plus 1.
        const uint64_t next_value =
          (current_max_value == UINT64_MAX) ? UINT64_MAX : (current_max_value + 1);
        new_table_schema.set_auto_increment(MAX(next_value, new_table_schema.get_auto_increment()));
      }
    }
  }
  return ret;
}

int ObDDLService::alter_table_constraints(const ObAlterTableArg::AlterConstraintType op_type,
                                          share::schema::ObSchemaGetterGuard &schema_guard,
                                          const ObTableSchema &orig_table_schema,
                                          const AlterTableSchema &inc_table_schema,
                                          ObTableSchema &new_table_schema,
                                          ObDDLOperator &ddl_operator,
                                          ObMySQLTransaction &trans)
{
  int ret = OB_SUCCESS;
  if (OB_FAIL(check_can_alter_table_constraints(op_type, schema_guard, orig_table_schema, inc_table_schema))) {
    LOG_WARN("fail to check can alter constraints", K(ret), K(op_type), K(inc_table_schema));
  }

  if (OB_SUCC(ret)) {
    if (obrpc::ObAlterTableArg::ADD_CONSTRAINT == op_type) {
      if (OB_FAIL(ddl_operator.add_table_constraints(inc_table_schema,
                                                     new_table_schema,
                                                     trans))) {
        LOG_WARN("failed to add table constraints", K(ret));
      }
    } else if (obrpc::ObAlterTableArg::DROP_CONSTRAINT == op_type) {
      if (OB_FAIL(drop_not_null_cst_in_column_flag(orig_table_schema, inc_table_schema, new_table_schema, ddl_operator, trans))) {
        LOG_WARN("failed to drop not null cst in column flag", K(ret));
      } else if (OB_FAIL(ddl_operator.drop_table_constraints(orig_table_schema,
                                                      inc_table_schema,
                                                      new_table_schema,
                                                      trans))) {
        LOG_WARN("failed to drop table constraints", K(ret));
      }
    } else if (obrpc::ObAlterTableArg::ALTER_CONSTRAINT_STATE == op_type) {
      if (OB_FAIL(alter_not_null_cst_in_column_flag(orig_table_schema, inc_table_schema,
                                                    new_table_schema, ddl_operator, trans))) {
        LOG_WARN("failed to drop not null cst in column flag", K(ret));
      } else if (OB_FAIL(ddl_operator.modify_check_constraints_state(
                  orig_table_schema,
                  inc_table_schema,
                  new_table_schema,
                  trans))) {
        LOG_WARN("failed to drop table constraints", K(ret));
      }
    } else {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("unexpected op type", K(op_type), K(ret));
    }
  }
  return ret;
}

int ObDDLService::check_can_alter_table_constraints(
  const ObAlterTableArg::AlterConstraintType op_type,
  share::schema::ObSchemaGetterGuard &schema_guard,
  const ObTableSchema &orig_table_schema,
  const AlterTableSchema &inc_table_schema)
{
  int ret = OB_SUCCESS;
  bool is_oracle_mode = false;

  if (OB_FAIL(orig_table_schema.check_if_oracle_compat_mode(is_oracle_mode))) {
    LOG_WARN("fail to check if tenant mode is oracle mode", K(ret));
  } else {
    for (ObTableSchema::const_constraint_iterator iter = inc_table_schema.constraint_begin(); OB_SUCC(ret) &&
      iter != inc_table_schema.constraint_end(); iter ++) {
      const ObString &cst_name = (*iter)->get_constraint_name_str();
      if (ObAlterTableArg::AlterConstraintType::ADD_CONSTRAINT == op_type) {
        bool is_check_constraint_name_exist = true;
        if (OB_FAIL(check_constraint_name_is_exist(schema_guard,
                                                   orig_table_schema,
                                                   cst_name,
                                                   false,
                                                   is_check_constraint_name_exist))) {
          LOG_WARN("fail to check check constraint name is exist or not", K(ret), K(cst_name));
        } else if (is_check_constraint_name_exist) {
          ret = OB_ERR_CONSTRAINT_NAME_DUPLICATE;
          LOG_WARN("check constraint name is duplicate", K(ret), K(cst_name));
          if (!is_oracle_mode) {
            LOG_USER_ERROR(OB_ERR_CONSTRAINT_NAME_DUPLICATE, cst_name.length(), cst_name.ptr());
          }
        }
      }
      for (ObTableSchema::const_constraint_iterator iter_r = iter + 1; OB_SUCC(ret) &&
        iter_r != inc_table_schema.constraint_end(); iter_r ++) {
        if (is_oracle_mode && (*iter_r)->get_constraint_name_str() == cst_name) {
          ret = OB_ERR_CONSTRAINT_NAME_DUPLICATE;
          LOG_WARN("duplicate constraint name", K(ret), K(cst_name));
        } else if (!is_oracle_mode && 0 == cst_name.case_compare((*iter_r)->get_constraint_name_str())) {
          ret = OB_ERR_CONSTRAINT_NAME_DUPLICATE;
          LOG_USER_ERROR(OB_ERR_CONSTRAINT_NAME_DUPLICATE, cst_name.length(), cst_name.ptr());
          LOG_WARN("duplicate constraint name", K(ret), K(cst_name));
        }
      }
      if (OB_SUCC(ret)) {
        ObConstraint* const* res = nullptr;
        if (is_oracle_mode) {
          res = std::find_if(orig_table_schema.constraint_begin(), orig_table_schema.constraint_end(), [&cst_name](const ObConstraint *cst)
                             { return cst_name == cst->get_constraint_name_str(); });
        } else {
          res = std::find_if(orig_table_schema.constraint_begin(), orig_table_schema.constraint_end(), [&cst_name](const ObConstraint* cst)
                             { return 0 == cst_name.case_compare(cst->get_constraint_name_str()); });
        }
        if (orig_table_schema.constraint_end() == res) {
          if (obrpc::ObAlterTableArg::DROP_CONSTRAINT == op_type
              || obrpc::ObAlterTableArg::ALTER_CONSTRAINT_STATE == op_type) {
            ret = OB_ERR_CONTRAINT_NOT_FOUND;
            LOG_WARN("constraint not exist", K(ret), K(cst_name));
          }
        } else {
          if (obrpc::ObAlterTableArg::ADD_CONSTRAINT == op_type) {
            ret = OB_ERR_CONSTRAINT_NAME_DUPLICATE;
            if (!is_oracle_mode) {
              LOG_USER_ERROR(OB_ERR_CONSTRAINT_NAME_DUPLICATE, cst_name.length(), cst_name.ptr());
            }
            LOG_WARN("constraint duplicate", K(ret), K(cst_name));
          } else if (obrpc::ObAlterTableArg::DROP_CONSTRAINT == op_type
                     || obrpc::ObAlterTableArg::ALTER_CONSTRAINT_STATE == op_type) {
            // drop or modify state
            if ((*iter)->get_constraint_id() != OB_INVALID_ID) {
              if ((*iter)->get_constraint_id() != (*res)->get_constraint_id()) {
                ret = OB_ERR_CONTRAINT_NOT_FOUND;
                LOG_WARN("constraint not exist", K(ret), K((*iter)->get_constraint_id()), K((*res)->get_constraint_id()));
              }
            }
            if (OB_FAIL(ret)) {
            } else if (CONSTRAINT_TYPE_PRIMARY_KEY == (*res)->get_constraint_type()) {
              if (obrpc::ObAlterTableArg::DROP_CONSTRAINT == op_type) {
                ret = OB_ERR_UNEXPECTED;
                LOG_WARN("unexpected err", K(ret), KPC(*res), K(inc_table_schema));
              } else {
                ret = OB_NOT_SUPPORTED;
                LOG_WARN("alter pk constraint state not supported now", K(ret), K((*res)->get_constraint_type()));
                LOG_USER_ERROR(OB_NOT_SUPPORTED, "alter state of primary key constraint");
              }
            } else {
              const_cast<ObConstraint *>(*iter)->set_constraint_id((*res)->get_constraint_id());
              const_cast<ObConstraint *>(*iter)->set_constraint_type((*res)->get_constraint_type());
            }
          }
        }
      }
    }
  }
  return ret;
}

int ObDDLService::get_partition_by_subpart_name(const ObTableSchema &orig_table_schema,
                                                const ObSubPartition &subpart_name,
                                                const ObPartition *&part,
                                                const ObSubPartition *&subpart)
{
  int ret = OB_SUCCESS;
  if (part != NULL) {
    ObSubPartIterator sub_iter(orig_table_schema, *part, CHECK_PARTITION_MODE_NORMAL);
    while (OB_SUCC(ret) && OB_SUCC(sub_iter.next(subpart))) {
      if (OB_ISNULL(subpart)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("get null subpartition", K(ret));
      } else if (ObCharset::case_insensitive_equal(subpart_name.get_part_name(),
                                                   subpart->get_part_name())) {
        break;
      }
    }
  } else {
    ObPartIterator iter(orig_table_schema, CHECK_PARTITION_MODE_NORMAL);
    while (OB_SUCC(ret) && OB_SUCC(iter.next(part))) {
      if (OB_ISNULL(part)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("part is null", KR(ret));
      } else {
        ObSubPartIterator sub_iter(orig_table_schema, *part, CHECK_PARTITION_MODE_NORMAL);
        subpart = NULL;
        while (OB_SUCC(ret) && OB_SUCC(sub_iter.next(subpart))) {
          if (OB_ISNULL(subpart)) {
            ret = OB_ERR_UNEXPECTED;
            LOG_WARN("get null subpartition", K(ret));
          } else if (ObCharset::case_insensitive_equal(subpart_name.get_part_name(),
                                                       subpart->get_part_name())) {
            break;
          }
        }
        if (OB_ITER_END == ret) {
          ret = OB_SUCCESS;
        } else {
          break;
        }
      }
    }
  }
  return ret;
}

int ObDDLService::gen_inc_table_schema_for_add_part(
    const ObTableSchema &orig_table_schema,
    AlterTableSchema &inc_table_schema)
{
  int ret = OB_SUCCESS;
  // TODO: Add partition in interval partitioned table should reorganize part_idx, which will be supported in the later version.
  int64_t max_part_idx = OB_INVALID_INDEX;
  if (orig_table_schema.is_interval_part()) {
    ret = OB_NOT_SUPPORTED;
    LOG_WARN("add interval part will support later", KR(ret));
  } else if (OB_FAIL(orig_table_schema.get_max_part_idx(max_part_idx, orig_table_schema.is_external_table()))) {
    LOG_WARN("fail to get max part idx", KR(ret), K(orig_table_schema));
  } else {
    const int64_t inc_part_num = inc_table_schema.get_part_option().get_part_num();
    ObPartition **inc_part_array = inc_table_schema.get_part_array();
    for (int64_t i = 0; i < inc_part_num && OB_SUCC(ret); ++i) {
      ObPartition *inc_part = inc_part_array[i];
      if (OB_ISNULL(inc_part)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("inc part is null", KR(ret), K(i));
      } else {
        // part_id will be generated by generate_object_id_for_partition_schema()
        inc_part->set_part_idx(max_part_idx + i + 1);
        if (PARTITION_LEVEL_TWO == orig_table_schema.get_part_level()) {
          for (int64_t j = 0; OB_SUCC(ret) && j < inc_part->get_subpartition_num(); j++) {
            ObSubPartition *subpart = inc_part->get_subpart_array()[j];
            if (OB_ISNULL(subpart)) {
              ret = OB_ERR_UNEXPECTED;
              LOG_WARN("subpart is null", KR(ret), K(j));
            } else if (subpart->get_sub_part_idx() >= 0
                       && subpart->get_sub_part_idx() != j) {
              ret = OB_ERR_UNEXPECTED;
              LOG_WARN("subpart_idx should be invalid", KR(ret), KPC(subpart), K(j));
            } else {
              // subpart_id will be generated by generate_object_id_for_partition_schema()
              // For some situations(not hash like), subpart_idx from ddl_resolver is invalid.
              subpart->set_sub_part_idx(j);
            }
          } // end for iterate inc subpart
        }
      }
    } // end for iterate inc part
  }
  return ret;
}

int ObDDLService::gen_inc_table_schema_for_add_subpart(
    const ObTableSchema &orig_table_schema,
    AlterTableSchema &inc_table_schema)
{
  int ret = OB_SUCCESS;
  const int64_t inc_part_num = inc_table_schema.get_part_option().get_part_num();
  ObPartition **inc_part_array = inc_table_schema.get_part_array();
  for (int64_t i = 0; i < inc_part_num && OB_SUCC(ret); ++i) {
    ObPartition *inc_part = inc_part_array[i];
    ObCheckPartitionMode check_partition_mode = CHECK_PARTITION_MODE_NORMAL;
    ObPartIterator iter(orig_table_schema, check_partition_mode);
    const ObPartition *part = NULL;
    while (OB_SUCC(ret) && OB_SUCC(iter.next(part))) {
      if (OB_ISNULL(part) || OB_ISNULL(inc_part)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("NULL ptr", KP(part), KP(inc_part), KR(ret));
      } else if (ObCharset::case_insensitive_equal(inc_part->get_part_name(),
                                                   part->get_part_name())) {
        inc_part->set_part_id(part->get_part_id());
        int64_t max_subpart_idx = OB_INVALID_INDEX;
        if (OB_FAIL(part->get_max_sub_part_idx(max_subpart_idx))) {
          LOG_WARN("fail to get max sub part idx", KR(ret), KPC(part));
        } else {
          for (int64_t j = 0; OB_SUCC(ret) && j < inc_part->get_subpartition_num(); j++) {
            ObSubPartition *subpart = inc_part->get_subpart_array()[j];
            if (OB_ISNULL(subpart)) {
              ret = OB_ERR_UNEXPECTED;
              LOG_WARN("subpart is null", KR(ret), K(j));
            } else {
              // subpart_id will be generated by generate_object_id_for_partition_schema()
              subpart->set_sub_part_idx(max_subpart_idx + j + 1);
            }
          } // end for iterate inc subpart
        }
        break;
      }
    }
    if (OB_ITER_END == ret) {
      ret = OB_PARTITION_NOT_EXIST;
      LOG_WARN("part should exists", KR(ret));
    } else if (OB_FAIL(ret)) {
      LOG_WARN("iter part failed", KR(ret));
    }
  } // end for iterate inc part
  return ret;
}

int ObDDLService::gen_inc_table_schema_for_rename_part_(
    const ObTableSchema &orig_table_schema,
    AlterTableSchema &inc_table_schema)
{
  int ret = OB_SUCCESS;
  ObPartition **inc_part_array = inc_table_schema.get_part_array();
  ObPartition *inc_part = nullptr;
  const int64_t inc_part_num = inc_table_schema.get_partition_num();
  if (OB_ISNULL(inc_part_array)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("inc part array is null", KR(ret), K(inc_table_schema));
  } else if (OB_UNLIKELY(1 != inc_part_num)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("rename multiple partitions at a time not support", KR(ret),K(inc_part_num));
  } else if (OB_ISNULL(inc_part = inc_part_array[0])) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("inc part is null", KR(ret), KP(inc_part_array), K(inc_table_schema));
  } else {
    const ObPartition *part = nullptr;
    const ObString &part_name = inc_part->get_part_name();
    const ObString &new_part_name = inc_table_schema.get_new_part_name();
    if (OB_FAIL(orig_table_schema.get_partition_by_name(part_name, part))) {
      LOG_WARN("should check schema first or should not in this step", KR(ret), K(part_name), K(orig_table_schema));
    } else if (OB_ISNULL(part)) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("get part failed", KR(ret), K(orig_table_schema));
    } else if (OB_FAIL(inc_part->assign(*part))) {
      LOG_WARN("failed to assign partition", KR(ret), KPC(part), KPC(inc_part));
    } else if (OB_FAIL(inc_part->set_part_name(new_part_name))) {
      LOG_WARN("failed to set inc part name", KR(ret), KPC(part), K(new_part_name));
    }
  }
  return ret;
}

int ObDDLService::gen_inc_table_schema_for_rename_subpart_(
    const share::schema::ObTableSchema &orig_table_schema,
    share::schema::AlterTableSchema &inc_table_schema)
{
  int ret = OB_SUCCESS;
  ObPartition **inc_part_array = inc_table_schema.get_part_array();
  ObPartition *inc_part = nullptr;
  const int64_t inc_part_num = inc_table_schema.get_partition_num();
  if (OB_ISNULL(inc_part_array)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("inc part array is null", KR(ret), K(inc_table_schema));
  } else if (OB_UNLIKELY(1 != inc_part_num)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("rename multiple subpartitions at a time not support", KR(ret), K(inc_part_num));
  } else if (OB_ISNULL(inc_part = inc_part_array[0])) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("inc_part is null", KR(ret), KP(inc_part_array), K(inc_table_schema));
  } else {
    const ObPartition *part = nullptr;
    const ObSubPartition *subpart = nullptr;
    ObSubPartition **inc_subpart_array = inc_part->get_subpart_array();
    ObSubPartition *inc_subpart = nullptr;
    ObSubPartition inc_subpart_content;
    const int64_t inc_subpart_num = inc_part->get_subpartition_num();
    const ObString &new_part_name = inc_table_schema.get_new_part_name();
    if (OB_ISNULL(inc_subpart_array)) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("inc subpart array is null", KR(ret), K(inc_table_schema));
    } else if (OB_UNLIKELY(1 != inc_subpart_num)) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("rename multiple subpartitions at a time not support", KR(ret), K(inc_subpart_num));
    } else if (OB_ISNULL(inc_subpart = inc_subpart_array[0])) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("inc subpart is null", KR(ret), KP(inc_subpart_array));
    } else if (OB_FAIL(orig_table_schema.get_subpartition_by_name(inc_subpart->get_part_name(), part, subpart))) {
      LOG_WARN("should check schema first or should not in this step", KR(ret), KP(part), KP(subpart));
    } else if (OB_ISNULL(part) || OB_ISNULL(subpart)) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("get null part/subpart", KR(ret), KP(part), KP(subpart));
    } else if (OB_FAIL(inc_subpart_content.assign(*subpart))) {
      LOG_WARN("failed to assign subpartition", KR(ret), KPC(part), K(inc_subpart_content));
    } else if (OB_FAIL(inc_subpart_content.set_part_name(new_part_name))) {
      LOG_WARN("set inc subpart name faild", KR(ret), K(inc_subpart_content), K(new_part_name));
    } else if (OB_FAIL(inc_part->ObBasePartition::assign(*part))) {
      LOG_WARN("failed to assign inc part", KR(ret), KPC(inc_part), KPC(part));
    } else if (OB_FAIL(inc_part->add_partition(inc_subpart_content))) {
      LOG_WARN("inc_part fail to add inc_subpart", KR(ret), KPC(inc_part), K(inc_subpart_content));
    } else {
      //not used in rename subpartition, just for integrity
      inc_part->set_sub_part_num(part->get_sub_part_num());
    }
  }
  return ret;
}

int ObDDLService::gen_inc_table_schema_for_drop_part(
    const ObTableSchema &orig_table_schema,
    AlterTableSchema &inc_table_schema)
{
  int ret = OB_SUCCESS;
  const int64_t inc_part_num = inc_table_schema.get_part_option().get_part_num();
  ObPartition **inc_part_array = inc_table_schema.get_part_array();
  for (int64_t i = 0; i < inc_part_num && OB_SUCC(ret); ++i) {
    ObPartition *inc_part = inc_part_array[i];
    ObCheckPartitionMode check_partition_mode = CHECK_PARTITION_MODE_NORMAL;
    ObPartIterator iter(orig_table_schema, check_partition_mode);
    const ObPartition *part = NULL;
    while (OB_SUCC(ret) && OB_SUCC(iter.next(part))) {
      if (OB_ISNULL(part) || OB_ISNULL(inc_part)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("NULL ptr", KP(part), KP(inc_part), KR(ret));
      } else if (ObCharset::case_insensitive_equal(inc_part->get_part_name(),
                                                   part->get_part_name())) {
        if (OB_FAIL(inc_part->assign(*part))) {
          LOG_WARN("failed to assign partition", KR(ret), K(part), K(inc_part));
        }
        break;
      }
    }
    if (OB_ITER_END == ret) {
      ret = OB_PARTITION_NOT_EXIST;
      LOG_WARN("part should exists", KR(ret));
    } else if (OB_SUCC(ret) && orig_table_schema.is_interval_part()
               && inc_part->get_high_bound_val()
                  == orig_table_schema.get_transition_point()) {
      // deleting the partition whose high level is transition point not support yet.
      ret = OB_NOT_SUPPORTED;
      LOG_WARN("delete the max range part of interval part table", KR(ret), K(orig_table_schema), K(inc_table_schema));
      LOG_USER_ERROR(OB_NOT_SUPPORTED, "delete the max range part of interval part table");
    }
  }
  return ret;
}


int ObDDLService::gen_inc_table_schema_for_drop_subpart(const ObTableSchema &orig_table_schema,
                                                        AlterTableSchema &inc_table_schema)
{
  int ret = OB_SUCCESS;
  ObPartition *inc_part = NULL;
  if (1 != inc_table_schema.get_partition_num()) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("drop subparts not in a part", KR(ret));
  } else if (OB_ISNULL(inc_part = inc_table_schema.get_part_array()[0])) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("inc_part is null", K(ret));
  } else {
    const ObPartition *part = NULL;
    for (int64_t i = 0; OB_SUCC(ret) && i < inc_part->get_subpartition_num(); i++) {
      ObSubPartition *inc_subpart = inc_part->get_subpart_array()[i];
      if (OB_ISNULL(inc_subpart)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("inc_subpart is null", K(ret));
      } else if (i == 0) {
        const ObSubPartition *subpart = NULL;
        if (OB_FAIL(get_partition_by_subpart_name(orig_table_schema, *inc_subpart, part, subpart))) {
          if (OB_ITER_END == ret) {
            ret = OB_PARTITION_NOT_EXIST;
            LOG_USER_ERROR(OB_PARTITION_NOT_EXIST);
            LOG_WARN("part should exists", K(ret));
          } else {
            LOG_WARN("fail to get partition by subpart name", K(ret));
          }
        } else if (OB_ISNULL(part) || OB_ISNULL(subpart)) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("get null part/subpart", K(ret));
        } else if (part->get_sub_part_num() <= inc_part->get_subpartition_num()) {
          ret = OB_ERR_DROP_LAST_PARTITION;
          LOG_WARN("cannot drop all subpartitions", K(part),
                   "partitions current", part->get_sub_part_num(),
                   "partitions to be dropped", inc_part->get_subpartition_num(),
                   K(ret));
          LOG_USER_ERROR(OB_ERR_DROP_LAST_PARTITION);
        } else if (OB_FAIL(inc_subpart->assign(*subpart))) {
          LOG_WARN("failed to assign partition", K(ret), K(part), K(inc_part));
        } else {
          inc_part->set_part_id(part->get_part_id());
        }
      } else if (OB_ISNULL(part)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("part is null", KR(ret));
      } else {
        const ObSubPartition *subpart = NULL;
        if (OB_FAIL(get_partition_by_subpart_name(orig_table_schema, *inc_subpart, part, subpart))) {
          if (OB_ITER_END == ret) {
            part = NULL;
            if (OB_FAIL(get_partition_by_subpart_name(orig_table_schema, *inc_subpart, part, subpart))) {
              if (OB_ITER_END == ret) {
                ret = OB_PARTITION_NOT_EXIST;
                LOG_USER_ERROR(OB_PARTITION_NOT_EXIST);
                LOG_WARN("part should exists", K(ret));
              } else {
                LOG_WARN("fail to get partition by subpart name", K(ret));
              }
            } else if (OB_ISNULL(part) || OB_ISNULL(subpart)) {
              ret = OB_ERR_UNEXPECTED;
              LOG_WARN("get null part/subpart", K(ret));
            } else {
              ret = OB_INVALID_PARTITION;
              LOG_USER_ERROR(OB_INVALID_PARTITION);
              LOG_WARN("subparts for deleted in different part", K(ret));
            }
          } else {
            LOG_WARN("fail to get partition by subpart name", K(ret));
          }
        } else if (OB_FAIL(inc_subpart->assign(*subpart))) {
          LOG_WARN("failed to assign partition", K(ret), K(part), K(inc_part));
        }
      }
    }
  }
  return ret;
}

int ObDDLService::gen_inc_table_schema_for_trun_part(
    const ObTableSchema &orig_table_schema,
    AlterTableSchema &inc_table_schema,
    AlterTableSchema &del_table_schema)
{
  int ret = OB_SUCCESS;
  const int64_t inc_part_num = inc_table_schema.get_part_option().get_part_num();
  ObPartition **inc_part_array = inc_table_schema.get_part_array();
  const int64_t del_part_num = del_table_schema.get_part_option().get_part_num();
  ObPartition **del_part_array = del_table_schema.get_part_array();
  if (OB_ISNULL(inc_part_array) || OB_ISNULL(del_part_array)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("partition array is null", KR(ret), K(inc_part_array), K(del_part_array), K(inc_table_schema));
  } else if (inc_part_num != del_part_num) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("inc is not equal to del", KR(ret), K(del_table_schema), K(inc_table_schema));
  }
  for (int64_t i = 0; i < inc_part_num && OB_SUCC(ret); ++i) {
    ObPartition *inc_part = inc_part_array[i];
    ObPartition *del_part = del_part_array[i];
    if (OB_ISNULL(inc_part) || OB_ISNULL(del_part)) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("partition is null", KR(ret), K(i), K(inc_part_num));
    } else {
      ObCheckPartitionMode check_partition_mode = CHECK_PARTITION_MODE_NORMAL;
      ObPartIterator iter(orig_table_schema, check_partition_mode);
      const ObPartition *part = NULL;
      while (OB_SUCC(ret) && OB_SUCC(iter.next(part))) {
        if (OB_ISNULL(part)) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("NULL ptr", K(part), KR(ret));
        } else if (ObCharset::case_insensitive_equal(inc_part->get_part_name(),
                                                     part->get_part_name())) {
          if (OB_FAIL(inc_part->assign(*part))) {
            LOG_WARN("failed to assign partition", KR(ret), K(part), K(inc_part));
          } else if (OB_FAIL(del_part->assign(*part))) {
            LOG_WARN("failed to assign partition", KR(ret), K(part), K(del_part));
          }
          break;
        }
      }
      if (OB_ITER_END == ret) {
        ret = OB_PARTITION_NOT_EXIST;
        LOG_WARN("part should exists", KR(ret), KPC(inc_part));
      }
    }
  }
  return ret;
}

// sql puts the subpartitions of multiple partitions that may exist
// into one partition and transfers it
// This function generates the necessary partitions
// and puts all the subpartitions in the right position
int ObDDLService::gen_inc_table_schema_for_trun_subpart(
    const ObTableSchema &orig_table_schema,
    AlterTableSchema &inc_table_schema,
    AlterTableSchema &del_table_schema)
{
  int ret = OB_SUCCESS;
  ObPartition *inc_part_p = NULL;
  ObPartition inc_part_set;
  if (OB_ISNULL(inc_table_schema.get_part_array())) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("part_array is null", KR(ret));
  } else if (OB_ISNULL(inc_part_p = inc_table_schema.get_part_array()[0])) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("inc_part_p is null", KR(ret));
  } else if (OB_FAIL(inc_part_set.assign(*inc_part_p))) {
    LOG_WARN("failed to assign partition", KR(ret), K(inc_part_set), KPC(inc_part_p));
  } else {
    inc_table_schema.reset_partition_array();
    for (int64_t i = 0; OB_SUCC(ret) && i < inc_part_set.get_subpartition_num(); i++) {
      const ObPartition *inc_part = NULL;
      const ObSubPartition *orig_subpart = NULL;
      const ObPartition *orig_part = NULL;
      ObSubPartition *inc_subpart = inc_part_set.get_subpart_array()[i];
      if (OB_ISNULL(inc_subpart)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("inc_subpart is null", KR(ret));
      } else if (OB_FAIL(get_partition_by_subpart_name(orig_table_schema, *inc_subpart, orig_part, orig_subpart))) {
        if (OB_ITER_END == ret) {
          ret = OB_PARTITION_NOT_EXIST;
          LOG_USER_ERROR(OB_PARTITION_NOT_EXIST);
          LOG_WARN("part should exists", KR(ret), KPC(inc_subpart));
        } else {
          LOG_WARN("fail to get partition by subpart name", KR(ret));
        }
      } else if (OB_ISNULL(orig_part) || OB_ISNULL(orig_subpart)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("get null part/subpart", KR(ret));
      } else if (0 != i && OB_FAIL(get_part_by_part_id(
                 inc_table_schema, orig_part->get_part_id(), inc_part))) {
        LOG_WARN("fail to get partition", KR(ret), K(orig_part->get_part_id()));
      } else if (OB_ISNULL(inc_part)) {
        ObPartition new_part;
        new_part.set_part_id(orig_part->get_part_id());
        if (OB_FAIL(new_part.add_partition(*orig_subpart))){
          LOG_WARN("failed to add partition", K(orig_subpart), KR(ret));
        } else if (OB_FAIL(inc_table_schema.add_partition(new_part))) {
          LOG_WARN("failed to add partition", KR(ret));
        }
      } else if (OB_FAIL(const_cast<ObPartition*>(inc_part)->add_partition(*orig_subpart))) {
        LOG_WARN("failed to add partition", K(orig_subpart), KR(ret));
      }
    }
    if (OB_SUCC(ret)) {
      ObPartition **part_array = inc_table_schema.get_part_array();
      const int64_t part_num = inc_table_schema.get_partition_num();
      inc_table_schema.set_part_num(part_num);
      if (OB_ISNULL(part_array)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("part_array is NULL", K(ret));
      }
      for (int64_t i = 0; OB_SUCC(ret) && i < part_num; i++) {
        if (OB_ISNULL(part_array[i])) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("part_array[i] is NULL", K(ret), K(i));
        } else {
          part_array[i]->set_sub_part_num(part_array[i]->get_subpartition_num());
        }
      }
    }
  }
  if (FAILEDx(del_table_schema.assign_partition_schema(inc_table_schema))) {
    LOG_WARN("fail to assign partition schema", K(inc_table_schema), KR(ret));
  }
  return ret;
}


const char* ObDDLService::ddl_type_str(const ObDDLType ddl_type)
{
  const char *str = "";
  if (DDL_CREATE_INDEX == ddl_type) {
    str = "create index";
  } else if (DDL_CREATE_PARTITIONED_LOCAL_INDEX == ddl_type) {
    str = "create partitioned local index";
  } else if (DDL_MODIFY_COLUMN == ddl_type) {
    str = "alter column";
  } else if (DDL_CHECK_CONSTRAINT == ddl_type) {
    str = "add or modify check constraint";
  } else if (DDL_FOREIGN_KEY_CONSTRAINT == ddl_type) {
    str = "alter foreign key constraint";
  } else if (DDL_ADD_PRIMARY_KEY == ddl_type) {
    str = "add primary key";
  } else if (DDL_DROP_PRIMARY_KEY == ddl_type) {
    str = "drop primary key";
  } else if (DDL_ALTER_PRIMARY_KEY == ddl_type) {
    str = "alter primary key";
  } else if (DDL_ALTER_PARTITION_BY == ddl_type) {
    str = "alter partition by";
  } else if (DDL_DROP_COLUMN == ddl_type) {
    str = "drop column";
  } else if (DDL_ADD_NOT_NULL_COLUMN == ddl_type) {
    str = "add not null column";
  } else if (DDL_ADD_COLUMN_OFFLINE == ddl_type) {
    str = "add column offline";
  } else if (DDL_ADD_COLUMN_ONLINE == ddl_type) {
    str = "add column online";
  } else if (DDL_ADD_COLUMN_INSTANT == ddl_type) {
    str = "add column instant";
  } else if (DDL_COMPOUND_INSTANT == ddl_type) {
    str = "ddl compound instant";
  } else if (DDL_COLUMN_REDEFINITION == ddl_type) {
    str = "column redefinition";
  } else if (DDL_TABLE_REDEFINITION == ddl_type) {
    str = "table redefinition";
  } else if (DDL_DIRECT_LOAD == ddl_type) {
    str = "direct load";
  } else if (DDL_DIRECT_LOAD_INSERT == ddl_type) {
    str = "direct load insert";
  } else if (DDL_MODIFY_AUTO_INCREMENT == ddl_type) {
    str = "modify auto_increment";
  } else if (DDL_CONVERT_TO_CHARACTER == ddl_type) {
    str = "convert to character";
  } else if (DDL_CHANGE_COLUMN_NAME == ddl_type) {
    str = "change column name";
  } else if (DDL_TABLE_RESTORE == ddl_type) {
    str = "recover restore table ddl";
  } else if (DDL_MVIEW_COMPLETE_REFRESH == ddl_type) {
    str = "mview complete refresh";
  } else if (DDL_CREATE_MVIEW == ddl_type) {
    str = "create mview";
  } else if (DDL_CREATE_MLOG == ddl_type) {
    str = "create materialized view log";
  } else if (DDL_DROP_MLOG == ddl_type) {
    str = "drop materialized view log";
  } else if (DDL_AUTO_SPLIT_BY_RANGE == ddl_type) {
    str = "auto split by range";
  } else if (DDL_AUTO_SPLIT_NON_RANGE == ddl_type) {
    str = "auto split by non range";
  } else if (DDL_MANUAL_SPLIT_BY_RANGE == ddl_type) {
    str = "manual split by range";
  } else if (DDL_MANUAL_SPLIT_NON_RANGE == ddl_type) {
    str = "manual split non range";
  } else if (DDL_PARTITION_SPLIT_RECOVERY_TABLE_REDEFINITION == ddl_type) {
    str = "partition split recovery table redefinition";
  }

  return str;
}

int ObDDLService::get_part_by_part_id(
    const ObPartitionSchema &partition_schema,
    const int64_t part_id,
    const ObPartition *&part)
{
  int ret = OB_SUCCESS;
  part = NULL;
  for (int64_t i = 0; OB_SUCC(ret) && i < partition_schema.get_partition_num(); i++) {
    const ObPartition *it = partition_schema.get_part_array()[i];
    if (OB_NOT_NULL(it) && it->get_part_id() == part_id) {
      part = it;
      break;
    }
  }
  return ret;
}

int ObDDLService::old_update_global_index_schema_(const obrpc::ObAlterTableArg &arg,
                                                  ObMySQLTransaction &trans,
                                                  ObDDLOperator &ddl_operator,
                                                  ObSchemaGetterGuard &schema_guard,
                                                  const ObTableSchema &orig_table_schema,
                                                  const ObTableSchema &index_table_schema,
                                                  ObTableSchema &new_index_table_schema,
                                                  const int64_t tenant_data_version)
{
  int ret = OB_SUCCESS;
  if (OB_UNLIKELY(!index_table_schema.is_valid()) || OB_UNLIKELY(!orig_table_schema.is_valid())) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("invalid argument", KR(ret), K(index_table_schema), K(orig_table_schema));
  } else if (!arg.is_update_global_indexes_) {
    if (OB_FAIL(ddl_operator.update_index_status(
                             index_table_schema.get_tenant_id(),
                             index_table_schema.get_data_table_id(),
                             index_table_schema.get_table_id(),
                             INDEX_STATUS_UNUSABLE,
                             orig_table_schema.get_in_offline_ddl_white_list(),
                             trans,
                             nullptr /* ddl_stmt_str */))) {
      LOG_WARN("update_index_status failed", KR(ret), K(index_table_schema.get_data_table_id()));
    }
  } else {
    if (OB_FAIL(drop_directly_and_create_index_schema_(schema_guard, orig_table_schema, new_index_table_schema,
                                                       nullptr /* ddl_stmt */, &trans, tenant_data_version))) {
      LOG_WARN("ddl_service_ rebuild_index failed", KR(ret));
    }
  }
  return ret;
}

int ObDDLService::make_index_unusable_(common::ObIAllocator &allocator,
                                       ObMySQLTransaction &trans,
                                       ObDDLOperator &ddl_operator,
                                       const ObTableSchema &orig_table_schema,
                                       const ObTableSchema &index_table_schema,
                                       ObTableSchema &new_index_table_schema,
                                       ObIArray<ObDDLTaskRecord> &ddl_tasks,
                                       ObIArray<obrpc::ObDDLRes> &ddl_res_array)
{
  int ret = OB_SUCCESS;
  if (OB_UNLIKELY(!index_table_schema.is_valid()) || OB_UNLIKELY(!orig_table_schema.is_valid())) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("invalid argument", KR(ret), K(index_table_schema), K(orig_table_schema));
  } else if (OB_FAIL(ddl_operator.update_index_status(
                                  index_table_schema.get_tenant_id(),
                                  index_table_schema.get_data_table_id(),
                                  index_table_schema.get_table_id(),
                                  INDEX_STATUS_UNAVAILABLE,
                                  orig_table_schema.get_in_offline_ddl_white_list(),
                                  trans,
                                  nullptr /* ddl_stmt_str */))) {
    LOG_WARN("update_index_status failed", KR(ret), K(index_table_schema.get_data_table_id()));
  }
  if (OB_SUCC(ret)) {
    bool has_index_task = false;
    obrpc::ObDropIndexArg *drop_index_arg = nullptr;
    typedef common::ObSEArray<share::schema::ObTableSchema, 4> TableSchemaArray;
    SMART_VAR(TableSchemaArray, drop_index_schemas) {
    if (OB_FAIL(ObDDLTaskRecordOperator::check_has_index_or_mlog_task(
                trans, index_table_schema, orig_table_schema.get_tenant_id(), orig_table_schema.get_table_id(),
                has_index_task))) {
      LOG_WARN("fail to check has index task", KR(ret), K(orig_table_schema.get_tenant_id()),
                K(orig_table_schema.get_table_id()), K(index_table_schema.get_table_id()));
    } else if (has_index_task) {
      ret = OB_NOT_SUPPORTED;
      LOG_WARN("not support to drop a building or dropping index", KR(ret));
    } else if (OB_FAIL(drop_index_schemas.push_back(new_index_table_schema))) {
      LOG_WARN("fail to push back new index table schema", KR(ret));
    } else if (OB_FAIL(prepare_drop_index_arg_(allocator, new_index_table_schema, true/*only_set_status*/,
                                              true/*is_add_to_scheduler*/, true/*is_inner*/, drop_index_arg))) {
      LOG_WARN("fail to prepare drop index arg", KR(ret));
    } else if (OB_ISNULL(drop_index_arg)) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("drop index arg is nullptr", KR(ret));
    } else if (OB_FAIL(submit_drop_index_task_and_fill_ddl_result_(allocator, trans, drop_index_schemas, orig_table_schema,
                       drop_index_arg, nullptr /*inc_data_tablet_ids*/, nullptr /*del_data_tablet_ids*/, has_index_task,
                                                                   ddl_tasks, ddl_res_array))) {
      LOG_WARN("fail to submit drop index task", KR(ret));
    }
    } // end smart var
  }
  return ret;
}

int ObDDLService::prepare_drop_index_arg_(common::ObIAllocator &allocator,
                                          ObTableSchema &index_table_schema,
                                          const bool only_set_status,
                                          const bool is_add_to_scheduler,
                                          const bool is_inner,
                                          obrpc::ObDropIndexArg *&drop_index_arg)
{
  int ret = OB_SUCCESS;
  drop_index_arg = nullptr;
  void *tmp_ptr = nullptr;
  if (OB_UNLIKELY(!index_table_schema.is_valid())) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("index table schema's origin index name is nullptr", KR(ret), K(index_table_schema));
  } else if (OB_FAIL(index_table_schema.generate_origin_index_name())) {
    LOG_WARN("fail to generate origin index name", KR(ret));
  } else if (nullptr == (tmp_ptr = (ObDropIndexArg*)allocator.alloc(sizeof(ObDropIndexArg)))) {
    ret = OB_ALLOCATE_MEMORY_FAILED;
    SHARE_LOG(ERROR, "failed to alloc memory!", KR(ret));
  } else {
    drop_index_arg = new (tmp_ptr)ObDropIndexArg();
    drop_index_arg->index_action_type_ = ObIndexArg::DROP_INDEX;
    drop_index_arg->is_add_to_scheduler_ = is_add_to_scheduler;
    drop_index_arg->is_inner_ = is_inner;
    drop_index_arg->tenant_id_ = index_table_schema.get_tenant_id();
    if (OB_FAIL(ob_write_string(allocator, index_table_schema.get_origin_index_name_str(),drop_index_arg->index_name_))) {
      LOG_WARN("fail to write index name", KR(ret));
    }
  }
  return ret;
}

int ObDDLService::submit_drop_index_task_and_fill_ddl_result_(common::ObIAllocator &allocator,
                                                              ObMySQLTransaction &trans,
                                                              const common::ObIArray<share::schema::ObTableSchema> &drop_index_schemas,
                                                              const ObTableSchema &orig_table_schema,
                                                              const obrpc::ObDropIndexArg *drop_index_arg,
                                                              const common::ObIArray<common::ObTabletID> *inc_data_tablet_ids,
                                                              const common::ObIArray<common::ObTabletID> *del_data_tablet_ids,
                                                              bool &has_index_task,
                                                              ObIArray<ObDDLTaskRecord> &ddl_tasks,
                                                              ObIArray<obrpc::ObDDLRes> &ddl_res_array)
{
  int ret = OB_SUCCESS;
  ObIndexBuilder index_builder(*this);
  ObDDLRes ddl_res;
  ObDDLTaskRecord task_record;
  if (0 >= drop_index_schemas.count() || OB_ISNULL(drop_index_arg)) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("drop index shemas count should larger than 0 or drop_index_arg is null",
             KR(ret), K(drop_index_schemas.count()), KP(drop_index_arg));
  } else if (OB_FAIL(index_builder.submit_drop_index_task(trans, orig_table_schema, drop_index_schemas,
                                                          *drop_index_arg, inc_data_tablet_ids, del_data_tablet_ids,
                                                          allocator, has_index_task, task_record))) {
    LOG_WARN("failed to submit drop index task", KR(ret));
  } else {
    const ObTableSchema &drop_index_schema = drop_index_schemas.at(0);
    ddl_res.task_id_ = task_record.task_id_;
    ddl_res.tenant_id_ = drop_index_schema.get_tenant_id();
    ddl_res.schema_id_ = drop_index_schema.get_table_id();
    if (OB_FAIL(ddl_tasks.push_back(task_record))) {
      LOG_WARN("push back ddl task failed", KR(ret));
    } else if (OB_FAIL(ddl_res_array.push_back(ddl_res))) {
      LOG_WARN("push back ddl res array failed", KR(ret));
    }
  }
  return ret;
}
int ObDDLService::drop_and_create_index_schema_(obrpc::ObAlterTableArg &arg,
                                                share::schema::ObSchemaGetterGuard &schema_guard,
                                                ObMySQLTransaction &trans,
                                                ObDDLOperator &ddl_operator,
                                                const ObTableSchema &orig_table_schema,
                                                const ObTableSchema &index_table_schema,
                                                const int64_t tenant_data_version,
                                                ObTableSchema &new_index_table_schema,
                                                ObIArray<ObDDLTaskRecord> &ddl_tasks,
                                                ObIArray<obrpc::ObDDLRes> &ddl_res_array)
{
  int ret = OB_SUCCESS;
  if (OB_FAIL(check_inner_stat())) {
    LOG_WARN("fail to check inner stat", KR(ret));
  } else if (OB_UNLIKELY(!orig_table_schema.is_valid() || !index_table_schema.is_valid())) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("invalid argument", KR(ret), K(orig_table_schema), K(index_table_schema));
  } else if (OB_FAIL(ddl_operator.update_index_status(
                                  index_table_schema.get_tenant_id(),
                                  index_table_schema.get_data_table_id(),
                                  index_table_schema.get_table_id(),
                                  INDEX_STATUS_UNAVAILABLE,
                                  orig_table_schema.get_in_offline_ddl_white_list(),
                                  trans,
                                  nullptr /* ddl_stmt_str */))) {
    LOG_WARN("update_index_status failed", KR(ret), K(index_table_schema.get_data_table_id()));
  }
  if (OB_SUCC(ret)) {
    obrpc::ObDropIndexArg *drop_index_arg = nullptr;
    bool has_index_task = false;
    bool is_inner_and_domain_index = index_table_schema.is_fts_or_multivalue_index();
    ObSArray<obrpc::ObIndexArg *> &index_arg_list = arg.index_arg_list_;
    if (OB_FAIL(ObDDLTaskRecordOperator::check_has_index_or_mlog_task(
                trans, index_table_schema, orig_table_schema.get_tenant_id(), orig_table_schema.get_table_id(), has_index_task))) {
      LOG_WARN("fail to check has index task", KR(ret), K(orig_table_schema.get_tenant_id()),
                K(orig_table_schema.get_table_id()), K(index_table_schema.get_table_id()));
    } else if (has_index_task) {
      ret = OB_NOT_SUPPORTED;
      LOG_WARN("not support to drop a building or dropping index", KR(ret));
    } else if (OB_FAIL(prepare_drop_index_arg_(arg.allocator_, new_index_table_schema, true/*only_set_status*/,
                                              true/*is_add_to_scheduler*/, true/*is_inner*/, drop_index_arg))) {
      LOG_WARN("fail to prepare drop index arg", KR(ret));
    } else if (OB_ISNULL(drop_index_arg)) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("drop index arg is nullptr", KR(ret));
    } else if (OB_FAIL(index_arg_list.push_back(drop_index_arg))) {
      LOG_WARN("push back to index_arg_list failed", KR(ret), KP(drop_index_arg));
    }
  }
  if (OB_SUCC(ret)) {
    uint64_t new_table_id = OB_INVALID_ID;
    ObSchemaService *schema_service = schema_service_->get_schema_service();
    if (OB_ISNULL(schema_service)) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("schema service is null", KR(ret));
    } else if (OB_FAIL(schema_service->fetch_new_table_id(
                       new_index_table_schema.get_tenant_id(), new_table_id))) {
      LOG_WARN("fail to fetch new table id", KR(ret));
    } else {
      new_index_table_schema.set_table_id(new_table_id);
      new_index_table_schema.set_index_status(INDEX_STATUS_UNAVAILABLE);
      if (OB_FAIL(generate_tablet_id(new_index_table_schema))) {
        LOG_WARN("failed to generate tablet id", KR(ret));
      } else if (OB_FAIL(create_index_or_mlog_table_in_trans(new_index_table_schema,
                         nullptr /* ddl_stmt */, &trans, schema_guard,
                         false/*need_check_tablet_cnt*/, tenant_data_version))) {
        LOG_WARN("create_index_in_trans failed", KR(ret), K(new_index_table_schema));
      }
    }
  }
  return ret;
}

int ObDDLService::prepare_create_index_arg_(common::ObIAllocator &allocator,
                                            const ObTableSchema &new_index_table_schema,
                                            const obrpc::ObIndexArg::IndexActionType index_action_type,
                                            obrpc::ObCreateIndexArg *&create_index_arg)
{
  int ret = OB_SUCCESS;
  void *tmp_ptr = nullptr;
  create_index_arg = nullptr;
  if (nullptr == (tmp_ptr = (ObCreateIndexArg*)allocator.alloc(sizeof(ObCreateIndexArg)))) {
    ret = OB_ALLOCATE_MEMORY_FAILED;
    SHARE_LOG(ERROR, "failed to alloc memory!", KR(ret));
  } else {
    create_index_arg = new (tmp_ptr)ObCreateIndexArg();
    create_index_arg->index_action_type_ = index_action_type;
    create_index_arg->index_type_ = new_index_table_schema.get_index_type();
    if (OB_FAIL(create_index_arg->index_schema_.assign(new_index_table_schema))) {
      LOG_WARN("fail to assign index schema", KR(ret), K(new_index_table_schema));
    }
  }
  return ret;
}

int ObDDLService::update_global_index(ObAlterTableArg &arg,
                                      const uint64_t tenant_id,
                                      const ObTableSchema &orig_table_schema,
                                      ObDDLOperator &ddl_operator,
                                      ObMySQLTransaction &trans,
                                      const uint64_t tenant_data_version,
                                      obrpc::ObAlterTableRes &res,
                                      ObIArray<ObDDLTaskRecord> &ddl_tasks)
{
  int ret = OB_SUCCESS;
  ObSEArray<ObAuxTableMetaInfo, 16> simple_index_infos;
  ObSchemaGetterGuard schema_guard;
  bool is_oracle_mode = false;
  if (obrpc::ObAlterTableArg::DROP_PARTITION == arg.alter_part_type_
      || obrpc::ObAlterTableArg::DROP_SUB_PARTITION == arg.alter_part_type_
      || obrpc::ObAlterTableArg::TRUNCATE_SUB_PARTITION == arg.alter_part_type_
      || obrpc::ObAlterTableArg::TRUNCATE_PARTITION == arg.alter_part_type_) {
    if (OB_ISNULL(schema_service_)) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("valid schema service", K(ret));
    } else if (OB_FAIL(schema_service_->get_tenant_schema_guard(tenant_id, schema_guard))) {
      LOG_WARN("fail to get tenant schema guard", K(ret), K(tenant_id));
    } else if (OB_FAIL(orig_table_schema.get_simple_index_infos(simple_index_infos))) {
      LOG_WARN("get_index_tid_array failed", K(ret));
    } else if (OB_FAIL(ObCompatModeGetter::check_is_oracle_mode_with_tenant_id(tenant_id, is_oracle_mode))) {
      LOG_WARN("fail to get compat mode", KR(ret), K(tenant_id));
    } else {
      SMART_VAR(ObTableSchema, new_index_table_schema) {
      for (int64_t i = 0; OB_SUCC(ret) && i < simple_index_infos.count(); ++i) {
        const ObTableSchema *index_table_schema = nullptr;
        new_index_table_schema.reset();
        if (OB_FAIL(schema_guard.get_table_schema(
                    tenant_id, simple_index_infos.at(i).table_id_, index_table_schema))) {
          LOG_WARN("get_table_schema failed", K(tenant_id),
                   "table id", simple_index_infos.at(i).table_id_, K(ret));
        } else if (OB_ISNULL(index_table_schema)) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("table schema should not be null", K(ret));
        } else if (index_table_schema->is_unavailable_index()) {
            ret = OB_NOT_SUPPORTED;
            LOG_WARN("drop/truncate partition before finishing index build not support", KR(ret));
            LOG_USER_ERROR(OB_NOT_SUPPORTED, "drop/truncate partition before finishing index build ");
        } else if (!index_table_schema->can_read_index()) {
          // If the index is not available, the partition operation will not do any intervention
        } else if (!index_table_schema->is_global_index_table()) {
          // do nothing
        } else if (OB_FAIL(new_index_table_schema.assign(*index_table_schema))) {
          LOG_WARN("fail to assign schema", KR(ret));
        } else {
          if (tenant_data_version < DATA_VERSION_4_3_3_0
              || is_oracle_mode) {
            if (OB_FAIL(old_update_global_index_schema_(arg, trans, ddl_operator, schema_guard,
                                                        orig_table_schema, *index_table_schema,
                                                        new_index_table_schema,tenant_data_version))) {
              LOG_WARN("fail to update global index schema", KR(ret), K(arg));
            }
          } else {
            ObIArray<obrpc::ObDDLRes> &ddl_res_array = res.ddl_res_array_;
            if (arg.is_update_global_indexes_) {
              if(OB_FAIL(drop_and_create_index_schema_(arg, schema_guard, trans, ddl_operator, orig_table_schema,
                                                       *index_table_schema, tenant_data_version, new_index_table_schema,
                                                       ddl_tasks, ddl_res_array))) {
                LOG_WARN("fail to drop and create index schema", KR(ret), K(arg));
              }
            } else {
              if (OB_FAIL(make_index_unusable_(arg.allocator_, trans, ddl_operator, orig_table_schema, *index_table_schema,
                                               new_index_table_schema, ddl_tasks, ddl_res_array))) {
                LOG_WARN("fail to make index unusable", KR(ret), K(arg));
              }
            }
          }
          if (OB_FAIL(ret)
              || !arg.is_update_global_indexes_) {
            // do nothing
          } else {
            ObSArray<obrpc::ObIndexArg *> &index_arg_list = arg.index_arg_list_;
            obrpc::ObCreateIndexArg *create_index_arg = nullptr;
            if (OB_FAIL(prepare_create_index_arg_(arg.allocator_, new_index_table_schema,
                                                  ObIndexArg::REBUILD_INDEX, create_index_arg))) {
              LOG_WARN("fail to prepare create index arg", KR(ret));
            } else if (OB_FAIL(index_arg_list.push_back(create_index_arg))) {
              LOG_WARN("push back to index_arg_list failed", KR(ret), K(create_index_arg));
            }
          }
        }
      }
      }// end smart var
    }
  }
  return ret;
}

int ObDDLService::fill_interval_info_for_set_interval(const ObTableSchema &orig_table_schema,
                                                      ObTableSchema &new_table_schema,
                                                      AlterTableSchema &inc_table_schema)
{
  int ret = OB_SUCCESS;
  ObPartition **part_array = orig_table_schema.get_part_array();
  int64_t part_num = orig_table_schema.get_partition_num();
  if (OB_ISNULL(part_array)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("fail to part_array is null", K(orig_table_schema), K(inc_table_schema), KR(ret));
  } else if (part_num < 1) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("part num is less 1", K(orig_table_schema), K(inc_table_schema), KR(ret));
  } else if (OB_ISNULL(part_array[part_num - 1])) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("the last part is null", K(orig_table_schema), K(inc_table_schema), KR(ret));
  } else if (OB_FAIL(new_table_schema.set_transition_point(part_array[part_num - 1]->get_high_bound_val()))) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("fail to set_transition_point", K(orig_table_schema), K(inc_table_schema),KR(ret));
  } else if (OB_FAIL(new_table_schema.set_interval_range(inc_table_schema.get_interval_range()))) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("fail to new_table_schema.set_interval_range", K(orig_table_schema), K(inc_table_schema),KR(ret));
  } else {
    new_table_schema.get_part_option().set_part_func_type(PARTITION_FUNC_TYPE_INTERVAL);
  }
  return ret;
}

int ObDDLService::fill_interval_info_for_offline(const ObTableSchema &orig_table_schema,
                                                 ObTableSchema &new_table_schema)
{
  int ret = OB_SUCCESS;
  if (new_table_schema.is_interval_part()) {
    ObPartition **part_array = orig_table_schema.get_part_array();
    int64_t part_num = orig_table_schema.get_partition_num();
    if (OB_ISNULL(part_array)) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("fail to part_array is null", K(orig_table_schema), KR(ret));
    } else if (part_num < 1) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("part num is less 1", K(orig_table_schema), KR(ret));
    } else if (OB_ISNULL(part_array[part_num - 1])) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("the last part is null", K(orig_table_schema), KR(ret));
    } else if (OB_FAIL(new_table_schema.set_transition_point(part_array[part_num - 1]->get_high_bound_val()))) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("fail to set_transition_point", K(orig_table_schema), KR(ret));
    } else if (OB_FAIL(new_table_schema.set_interval_range(orig_table_schema.get_interval_range()))) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("fail to new_table_schema.set_interval_range", K(orig_table_schema), KR(ret));
    } else {
      new_table_schema.get_part_option().set_part_func_type(PARTITION_FUNC_TYPE_INTERVAL);
    }
  }
  return ret;
}

//For truncate part/subpart, inc_table_schema and del_table_schema should be different in the later process.
//For other situations, del_table_schema is useless and equal to inc_table_schema.
//For split partition, upd_table_schema records the partition information which need to be changed.
//For other situations, upd_table_schema is useless and equal to new_table_schema.
int ObDDLService::generate_tables_array(const obrpc::ObAlterTableArg &alter_table_arg,
                                        ObIArray<const ObTableSchema*> &orig_table_schemas,
                                        ObIArray<ObTableSchema*> &new_table_schemas,
                                        ObIArray<AlterTableSchema*> &inc_table_schemas,
                                        ObIArray<AlterTableSchema*> &del_table_schemas,
                                        common::ObIArray<ObTableSchema*> &upd_table_schemas,
                                        const ObTableSchema &orig_table_schema,
                                        ObTableSchema &new_table_schema,
                                        AlterTableSchema &inc_table_schema,
                                        ObSchemaGetterGuard &schema_guard,
                                        ObArenaAllocator &allocator)
{
  int ret = OB_SUCCESS;
  const ObAlterTableArg::AlterPartitionType op_type = alter_table_arg.alter_part_type_;
  const AlterTableSchema &alter_table_schema = alter_table_arg.alter_table_schema_;
  const uint64_t tenant_id = orig_table_schema.get_tenant_id();
  ObSEArray<uint64_t, 20> aux_table_ids;
  ObSEArray<ObAuxTableMetaInfo, 16> simple_index_infos;
  AlterTableSchema tmp_inc_table_schema;
  bool modify_sub_part_template_flags = false;
  if (PARTITION_LEVEL_TWO == orig_table_schema.get_part_level()
      && orig_table_schema.sub_part_template_def_valid()) {
    // sub_part_template_def_valid() is only used for schema printer.
    // To simplify relate logic, we consider that add partition/subpartition or add subpartition
    // make cause partitions different.
    if (obrpc::ObAlterTableArg::ADD_SUB_PARTITION == op_type
        || obrpc::ObAlterTableArg::DROP_SUB_PARTITION == op_type
        || (obrpc::ObAlterTableArg::ADD_PARTITION == op_type && !alter_table_schema.sub_part_template_def_valid())) {
      modify_sub_part_template_flags = true;
      new_table_schema.unset_sub_part_template_def_valid();
    }
  }
  const ObString new_part_name = inc_table_schema.get_new_part_name();
  if ((!orig_table_schema.has_tablet() && !orig_table_schema.is_external_table())
      || orig_table_schema.is_index_local_storage()
      || orig_table_schema.is_aux_lob_table()
      || orig_table_schema.is_mlog_table()) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("table_schema must be data table or global indexes", KR(ret), K(orig_table_schema));
  } else if (orig_table_schema.is_external_table() && (obrpc::ObAlterTableArg::ADD_PARTITION != op_type
                                                    && obrpc::ObAlterTableArg::DROP_PARTITION != op_type)) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("external should only support add and drop partition ddl", K(ret));
  } else if (OB_FAIL(orig_table_schemas.push_back(&orig_table_schema))) {
    LOG_WARN("failed to push back table_schema", KR(ret), K(orig_table_schema));
  } else if (OB_FAIL(new_table_schemas.push_back(&new_table_schema))) {
    LOG_WARN("failed to push back table_schema", KR(ret), K(new_table_schema));
  } else if (OB_FAIL(tmp_inc_table_schema.assign(new_table_schema))) {
    LOG_WARN("failed to push back table_schema", KR(ret), K(new_table_schema));
  } else if (OB_FAIL(tmp_inc_table_schema.assign_partition_schema(inc_table_schema))) {
    LOG_WARN("fail to assign partition schema", K(inc_table_schema), KR(ret));
  } else if (OB_FAIL(tmp_inc_table_schema.set_transition_point(inc_table_schema.get_transition_point()))) {
    LOG_WARN("fail to set transition point", K(ret));
  } else if (OB_FAIL(tmp_inc_table_schema.set_interval_range(inc_table_schema.get_interval_range()))) {
    LOG_WARN("fail to set interval range", K(ret));
  } else if (OB_FAIL(inc_table_schema.assign(tmp_inc_table_schema))) {
    LOG_WARN("failed to push back table_schema", KR(ret), K(new_table_schema));
  } else if (OB_FAIL(inc_table_schema.set_new_part_name(new_part_name))) {
    LOG_WARN("fail to set new part name", KR(ret), K(inc_table_schema));
  } else if (OB_FAIL(inc_table_schemas.push_back(&inc_table_schema))) {
    LOG_WARN("failed to push back table_schema", KR(ret), K(inc_table_schema));
  } else if (OB_FAIL(orig_table_schema.get_simple_index_infos(simple_index_infos))) {
    LOG_WARN("get_simple_index_infos failed", KR(ret));
  }
  if (obrpc::ObAlterTableArg::RENAME_PARTITION == op_type
   ||obrpc::ObAlterTableArg::RENAME_SUB_PARTITION == op_type) {
    //do nothing
  } else {
    for (int64_t i = 0; OB_SUCC(ret) && i < simple_index_infos.count(); ++i) {
      if (OB_FAIL(aux_table_ids.push_back(simple_index_infos.at(i).table_id_))) {
        LOG_WARN("fail to push back index table id", KR(ret));
      }
    }
    if (OB_SUCC(ret)) {
      uint64_t mtid = orig_table_schema.get_aux_lob_meta_tid();
      uint64_t ptid = orig_table_schema.get_aux_lob_piece_tid();
      if (!((mtid != OB_INVALID_ID && ptid != OB_INVALID_ID) || (mtid == OB_INVALID_ID && ptid == OB_INVALID_ID))) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("Expect meta tid and piece tid both valid or both invalid", KR(ret), K(mtid), K(ptid));
      } else if (OB_INVALID_ID != mtid &&
          OB_FAIL(aux_table_ids.push_back(mtid))) {
        LOG_WARN("fail to push back lob meta tid", KR(ret), K(mtid));
      } else if (OB_INVALID_ID != ptid &&
          OB_FAIL(aux_table_ids.push_back(ptid))) {
        LOG_WARN("fail to push back lob piece tid", KR(ret), K(ptid));
      }
    }
    if (OB_SUCC(ret)) {
      uint64_t mlog_tid = orig_table_schema.get_mlog_tid();
      if ((OB_INVALID_ID != mlog_tid)
          && OB_FAIL(aux_table_ids.push_back(mlog_tid))) {
        LOG_WARN("failed to push back materialized view log tid", KR(ret), K(mlog_tid));
      }
    }
  }

  for (int64_t i = 0; OB_SUCC(ret) && i < aux_table_ids.count(); ++i) {
    const ObTableSchema *orig_aux_table_schema = NULL;
    if (OB_FAIL(schema_guard.get_table_schema(tenant_id,
        aux_table_ids.at(i), orig_aux_table_schema))) {
      LOG_WARN("get_table_schema failed", K(tenant_id), K(aux_table_ids.at(i)), KR(ret));
    } else if (OB_ISNULL(orig_aux_table_schema)) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("table schema should not be null", KR(ret));
    } else if (orig_aux_table_schema->is_index_local_storage()
        || orig_aux_table_schema->is_aux_lob_table()
        || orig_aux_table_schema->is_mlog_table()) {
      ObTableSchema *new_aux_table_schema = NULL;
      AlterTableSchema *inc_aux_table_schema = NULL;
      void *new_schema_ptr = allocator.alloc(sizeof(ObTableSchema));
      void *inc_schema_ptr = allocator.alloc(sizeof(AlterTableSchema));
      if (OB_ISNULL(new_schema_ptr) || OB_ISNULL(inc_schema_ptr)) {
        ret = OB_ALLOCATE_MEMORY_FAILED;
        LOG_WARN("fail alloc memory", KR(ret), KP(new_schema_ptr), KP(inc_schema_ptr));
      } else {
        new_aux_table_schema = new (new_schema_ptr)ObTableSchema(&allocator);
        inc_aux_table_schema = new (inc_schema_ptr)AlterTableSchema(&allocator);
      }
      if (OB_FAIL(ret)) {
      } else if (OB_FAIL(new_aux_table_schema->assign(*orig_aux_table_schema))) {
        LOG_WARN("fail to assign schema", KR(ret), KPC(orig_aux_table_schema));
      } else if (modify_sub_part_template_flags
                 && FALSE_IT(new_aux_table_schema->unset_sub_part_template_def_valid())) {
      } else if (OB_FAIL(inc_aux_table_schema->assign(*new_aux_table_schema))) {
        LOG_WARN("failed to push back table_schema", KR(ret), KPC(new_aux_table_schema));
      } else if (OB_FAIL(inc_aux_table_schema->assign_partition_schema_without_auto_part_attr(inc_table_schema))) {
        LOG_WARN("fail to assign partition schema", K(inc_table_schema), KR(ret));
      } else if (OB_FAIL(inc_aux_table_schema->set_transition_point(inc_table_schema.get_transition_point()))) {
        LOG_WARN("fail to set transition point", K(ret));
      } else if (OB_FAIL(inc_aux_table_schema->set_interval_range(inc_table_schema.get_interval_range()))) {
        LOG_WARN("fail to set interval range", K(ret));
      } else if (OB_FAIL(new_table_schemas.push_back(new_aux_table_schema))) {
        LOG_WARN("failed to push back table_schema", KR(ret), K(new_aux_table_schema));
      } else if (OB_FAIL(orig_table_schemas.push_back(orig_aux_table_schema))) {
        LOG_WARN("failed to push back table_schema", KR(ret), K(orig_aux_table_schema));
      } else if (OB_FAIL(inc_table_schemas.push_back(inc_aux_table_schema))) {
        LOG_WARN("failed to push back table_schema", KR(ret), KPC(inc_aux_table_schema));
      }
    }
  }

  if (OB_FAIL(ret)) {
  } else if (obrpc::ObAlterTableArg::REORGANIZE_PARTITION == op_type ||
             obrpc::ObAlterTableArg::SPLIT_PARTITION == op_type ||
             obrpc::ObAlterTableArg::AUTO_SPLIT_PARTITION == op_type) {
    for (int64_t i = 0; OB_SUCC(ret) && i < new_table_schemas.count(); i++) {
      ObTableSchema *upd_table_schema = NULL;
      void *upd_schema_ptr = allocator.alloc(sizeof(ObTableSchema));
      if (OB_ISNULL(upd_schema_ptr)) {
        ret = OB_ALLOCATE_MEMORY_FAILED;
        LOG_WARN("fail alloc memory", KR(ret));
      } else if (FALSE_IT(upd_table_schema = new (upd_schema_ptr)ObTableSchema(&allocator))) {
      } else if (OB_FAIL(upd_table_schema->assign(*new_table_schemas.at(i)))) {
        LOG_WARN("fail to assign partition schema", K(new_table_schemas), KR(ret));
      } else if (FALSE_IT(upd_table_schema->reset_partition_array())) {
      } else if (FALSE_IT(upd_table_schema->reset_hidden_partition_array())) {
      } else if (OB_FAIL(upd_table_schemas.push_back(upd_table_schema))) {
        LOG_WARN("failed to push back table_schema", KR(ret), KPC(upd_table_schema));
      }
    }

    if (OB_FAIL(ret)) {
    } else if (OB_FAIL(generate_split_info_for_schemas_(op_type,
                                                        orig_table_schemas, inc_table_schemas,
                                                        new_table_schemas, upd_table_schemas))) {
      LOG_WARN("fail to generate split info for schema", KR(ret));
    }
  } else if (OB_FAIL(upd_table_schemas.assign(new_table_schemas))) {
    LOG_WARN("failed to assign new_table_schemas", KR(ret), K(new_table_schemas));
  }

  if (OB_FAIL(ret)) {
  } else if (obrpc::ObAlterTableArg::TRUNCATE_PARTITION == op_type
             || obrpc::ObAlterTableArg::TRUNCATE_SUB_PARTITION == op_type) {
    // truncate part/subpart may cause del_table_schema and inc_table_schema
    for (int i = 0; OB_SUCC(ret) && i < inc_table_schemas.count(); i++) {
      AlterTableSchema *del_table_schema = NULL;
      void *del_schema_ptr = allocator.alloc(sizeof(AlterTableSchema));
      if (OB_ISNULL(del_schema_ptr)) {
        ret = OB_ALLOCATE_MEMORY_FAILED;
        LOG_WARN("fail alloc memory", KR(ret));
      } else if (FALSE_IT(del_table_schema = new (del_schema_ptr)AlterTableSchema(&allocator))) {
      } else if (OB_FAIL(del_table_schema->assign(*inc_table_schemas.at(i)))) {
        LOG_WARN("fail to assign partition schema", K(inc_table_schema), KR(ret));
      } else if (OB_FAIL(del_table_schemas.push_back(del_table_schema))) {
        LOG_WARN("failed to push back table_schema", KR(ret), KPC(del_table_schema));
      }
    }
  }
  // for other situations, del_table_schemas is useless and equal to inc_table_schemas
  else if (OB_FAIL(del_table_schemas.assign(inc_table_schemas))) {
    LOG_WARN("failed to assign inc_table_schemas", KR(ret), K(inc_table_schemas));
  }
  return ret;
}

int ObDDLService::reset_interval_info_for_interval_to_range(ObTableSchema &new_table_schema)
{
  int ret = OB_SUCCESS;
  ObRowkey null_row_key;
  null_row_key.reset();
  if (OB_FAIL(new_table_schema.set_transition_point(null_row_key))) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("fail to set_transition_point", K(new_table_schema), KR(ret));
  } else if (OB_FAIL(new_table_schema.set_interval_range(null_row_key))) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("fail to new_table_schema.set_interval_range", K(new_table_schema), KR(ret));
  } else {
    new_table_schema.get_part_option().set_part_func_type(PARTITION_FUNC_TYPE_RANGE_COLUMNS);
  }
  return ret;
}

int ObDDLService::split_global_index_partitions(obrpc::ObAlterTableArg &arg, obrpc::ObAlterTableRes &res)
{
  int ret = OB_SUCCESS;
  AlterTableSchema &alter_table_schema = arg.alter_table_schema_;
  const uint64_t tenant_id = alter_table_schema.get_tenant_id();
  const ObTableSchema *orig_index_schema =  NULL;
  int64_t refreshed_schema_version = 0;
  uint64_t tenant_data_version = 0;
  ObDDLSQLTransaction trans(schema_service_);
  ObSchemaGetterGuard schema_guard;
  schema_guard.set_session_id(arg.session_id_);

  if (OB_FAIL(check_inner_stat())) {
    LOG_WARN("variable is not init", KR(ret));
  } else if (OB_ISNULL(GCTX.root_service_)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("root service is null", KR(ret));
  } else if (!arg.is_valid()) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("invalid arg", KR(ret), K(arg));
  } else if (OB_UNLIKELY(obrpc::ObAlterTableArg::AlterPartitionType::AUTO_SPLIT_PARTITION != arg.alter_part_type_)) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("invalid alter partition type", KR(ret), K(arg.alter_part_type_), K(arg));
  } else if (OB_FAIL(GET_MIN_DATA_VERSION(tenant_id, tenant_data_version))) {
    LOG_WARN("get min data version failed", K(ret), K(tenant_id));
  } else if (OB_FAIL(get_tenant_schema_guard_with_version_in_inner_table(tenant_id,
                                                                         schema_guard))) {
    LOG_WARN("fail to get schema guard with version in inner table", KR(ret), K(tenant_id));
  } else if (OB_FAIL(schema_guard.get_schema_version(tenant_id, refreshed_schema_version))) {
    LOG_WARN("failed to get tenant schema version", KR(ret), K(tenant_id));
  } else if (OB_FAIL(schema_guard.get_table_schema(tenant_id,
                                                   alter_table_schema.get_origin_database_name(),
                                                   alter_table_schema.get_origin_table_name(),
                                                   true /*is_index*/,
                                                   orig_index_schema))) {
    LOG_WARN("fail to get table schema", KR(ret), K(tenant_id), K(alter_table_schema));
  } else if (OB_ISNULL(orig_index_schema)) {
    ret = OB_TABLE_NOT_EXIST;
    LOG_WARN("NULL ptr", KR(ret), K(alter_table_schema));
  } else if (OB_UNLIKELY(!orig_index_schema->is_global_index_table() ||
                         is_sys_index_table(orig_index_schema->get_table_id()))) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("invalid index type", KR(ret), KPC(orig_index_schema));
  } else if (OB_FAIL(fill_part_name(*orig_index_schema, arg.alter_table_schema_))) {
    LOG_WARN("failed to fill part name", KR(ret), K(orig_index_schema), K(arg));
  } else if (OB_FAIL(check_split_global_index_partition_(schema_guard, arg, *orig_index_schema))) {
    LOG_WARN("fail to check split global idnex partition", KR(ret), KPC(orig_index_schema), K(arg));
  } else if (OB_FAIL(trans.start(sql_proxy_, tenant_id, refreshed_schema_version))) {
    LOG_WARN("start transaction failed", KR(ret), K(tenant_id), K(refreshed_schema_version));
  } else {
    ObDDLOperator ddl_operator(*schema_service_, *sql_proxy_);
    ObArray<const ObTableSchema*> orig_index_schemas;
    ObArray<ObTableSchema*> new_index_schemas;
    ObArray<AlterTableSchema*> inc_index_schemas;
    ObArray<const ObTableSchema *> inc_index_schema_ptrs;
    ObArray<AlterTableSchema*> del_index_schemas;
    ObArray<ObTableSchema*> upd_index_schemas;
    ObArenaAllocator allocator("ModifyPart");
    obrpc::ObAlterTableRes res;

    HEAP_VAR(ObArray<ObDDLTaskRecord>, ddl_tasks) {
    HEAP_VAR(ObTableSchema, new_table_schema) {
    if (OB_FAIL(new_table_schema.assign(*orig_index_schema))) {
      LOG_WARN("fail to assign schema", KR(ret));
    } else if (OB_FAIL(generate_tables_array(arg,
                                             orig_index_schemas,
                                             new_index_schemas,
                                             inc_index_schemas,
                                             del_index_schemas,
                                             upd_index_schemas,
                                             *orig_index_schema,
                                             new_table_schema,
                                             alter_table_schema,
                                             schema_guard,
                                             allocator))) {
      LOG_WARN("generate tables array failed", KR(ret), K(arg));
    } else if (OB_FAIL(alter_tables_partitions(arg,
                                               orig_index_schemas,
                                               new_index_schemas,
                                               inc_index_schemas,
                                               del_index_schemas,
                                               upd_index_schemas,
                                               ddl_operator,
                                               schema_guard,
                                               trans))) {
      LOG_WARN("alter table partitions failed", KR(ret), K(arg));
    } else if (OB_FAIL(ddl_operator.inc_table_schema_version(trans, tenant_id,
                                                             orig_index_schema->get_data_table_id()))) {
      LOG_WARN("fail to inc data table schema version", KR(ret), KPC(orig_index_schema));
    } else if (OB_UNLIKELY(inc_index_schemas.count() != 1) ||
               OB_UNLIKELY(new_index_schemas.count() != 1) ||
               OB_ISNULL(inc_index_schemas.at(0)) ||
               OB_ISNULL(new_index_schemas.at(0))) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("invalid schema arr", KR(ret), K(inc_index_schemas.count()), KP(inc_index_schemas.at(0)),
                                     K(new_index_schemas.count()), KP(new_index_schemas.at(0)));
    } else if (FALSE_IT(inc_index_schemas.at(0)->set_schema_version(
                                          new_index_schemas.at(0)->get_schema_version()))) {
    } else if (OB_FAIL(inc_index_schema_ptrs.push_back(inc_index_schemas.at(0)))) {
      LOG_WARN("failed to push back", K(ret));
    } else {
      const ObDDLType ddl_type = ObDDLType::DDL_AUTO_SPLIT_BY_RANGE;
      ObSplitPartitionHelper split_partition_helper(trans, schema_guard, arg.allocator_, tenant_id,
          tenant_data_version, ddl_type, new_index_schemas, upd_index_schemas, inc_index_schema_ptrs, arg.parallelism_);
      ObDDLTaskRecord task_record;
      if (OB_FAIL(split_partition_helper.execute(task_record))) {
        LOG_WARN("failed to execute split partition helper", KR(ret));
      } else if (OB_FAIL(ddl_tasks.push_back(task_record))) {
        LOG_WARN("failed to push back", KR(ret));
      } else {
        res.ddl_type_ = ddl_type;
        res.task_id_ = task_record.task_id_;
      }
    }
    if (OB_FAIL(ret)) {
    } else if (OB_UNLIKELY(ddl_tasks.count() != 1)) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("invalid task number", KR(ret), K(arg), K(ddl_tasks));
    }

    const bool is_commit = OB_SUCC(ret);
    if (trans.is_started()) {
      int temp_ret = OB_SUCCESS;
      if (OB_SUCCESS != (temp_ret = trans.end(is_commit))) {
        LOG_WARN("trans end failed", K(is_commit), K(temp_ret));
        ret = is_commit ? temp_ret : ret;
      }
    }
    if (OB_SUCC(ret)) {
      int tmp_ret = OB_SUCCESS;
      if (OB_FAIL(publish_schema(tenant_id))) {
        LOG_WARN("publish_schema failed", KR(ret));
      } else if (OB_TMP_FAIL(GCTX.root_service_->get_ddl_scheduler().schedule_ddl_task(ddl_tasks.at(0)))) {
        LOG_WARN("fail to schedule ddl task", K(tmp_ret), K(ddl_tasks.at(0)));
      }
    }
    } // end HEAP_VAR(ObTableSchema, new_table_schema)
    } // end HEAP_VAR(ObArray<ObDDLTaskRecord>, ddl_tasks)
  }

  return ret;
}

int ObDDLService::alter_tables_partitions(const obrpc::ObAlterTableArg &alter_table_arg,
                                          ObIArray<const ObTableSchema*> &orig_table_schemas,
                                          ObIArray<ObTableSchema*> &new_table_schemas,
                                          ObIArray<AlterTableSchema*> &inc_table_schemas,
                                          ObIArray<AlterTableSchema*> &del_table_schemas,
                                          ObIArray<ObTableSchema*> &upd_table_schemas,
                                          ObDDLOperator &ddl_operator,
                                          ObSchemaGetterGuard &schema_guard,
                                          ObMySQLTransaction &trans)
{
  int ret = OB_SUCCESS;
  if (orig_table_schemas.count() != new_table_schemas.count()) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("orig_table_schemas.count() new_table_schemas.count() is not equal", KR(ret),
             K(orig_table_schemas), K(new_table_schemas));
  } else if (orig_table_schemas.count() != inc_table_schemas.count()) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("orig_table_schemas.count() inc_table_schemas.count() is not equal", KR(ret),
             K(orig_table_schemas), K(inc_table_schemas));
  } else if (orig_table_schemas.count() != del_table_schemas.count()) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("orig_table_schemas.count() inc_table_schemas.count() is not equal", KR(ret),
             K(orig_table_schemas), K(del_table_schemas));
  }

  for (int64_t i = 0; OB_SUCC(ret) && i < new_table_schemas.count(); ++i) {
    // todo fill AlterTableSchema for splitting partition with inc_table_schema
    if (OB_ISNULL(new_table_schemas.at(i)) || OB_ISNULL(inc_table_schemas.at(i))
        || OB_ISNULL(orig_table_schemas.at(i)) || OB_ISNULL(del_table_schemas.at(i))) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("schema ptr is NULL", KR(ret), K(new_table_schemas.at(i)),
                K(inc_table_schemas.at(i)), K(orig_table_schemas.at(i)), K(del_table_schemas.at(i)));

    } else if (OB_FAIL(alter_table_partitions(alter_table_arg,
                                              *orig_table_schemas.at(i),
                                              *inc_table_schemas.at(i),
                                              *del_table_schemas.at(i),
                                              *upd_table_schemas.at(i),
                                              *new_table_schemas.at(i),
                                              ddl_operator,
                                              schema_guard,
                                              trans,
                                              *orig_table_schemas.at(0)))) {
      LOG_WARN("alter table partitions failed", KR(ret), K(i), KPC(new_table_schemas.at(i)), KPC(inc_table_schemas.at(i)));
    }
  }
  return ret;
}

bool ObDDLService::is_add_and_drop_partition(const obrpc::ObAlterTableArg::AlterPartitionType &op_type)
{
  return obrpc::ObAlterTableArg::ADD_PARTITION == op_type
        || obrpc::ObAlterTableArg::ADD_SUB_PARTITION == op_type
        || obrpc::ObAlterTableArg::DROP_PARTITION == op_type
        || obrpc::ObAlterTableArg::DROP_SUB_PARTITION == op_type
        || obrpc::ObAlterTableArg::TRUNCATE_PARTITION == op_type
        || obrpc::ObAlterTableArg::TRUNCATE_SUB_PARTITION == op_type;
}

int ObDDLService::alter_table_partitions(const obrpc::ObAlterTableArg &alter_table_arg,
                                         const ObTableSchema &orig_table_schema,
                                         AlterTableSchema &inc_table_schema,
                                         AlterTableSchema &del_table_schema,
                                         ObTableSchema &upd_table_schema,
                                         ObTableSchema &new_table_schema,
                                         ObDDLOperator &ddl_operator,
                                         ObSchemaGetterGuard &schema_guard,
                                         ObMySQLTransaction &trans,
                                         const ObTableSchema &orig_data_table_schema)
{
  DEBUG_SYNC(BEFORE_ALTER_TABLE_PARTITION);
  int ret = OB_SUCCESS;
  const uint64_t tenant_id = orig_table_schema.get_tenant_id();
  const int64_t table_id = orig_table_schema.get_table_id();
  const obrpc::ObAlterTableArg::AlterPartitionType &op_type = alter_table_arg.alter_part_type_;
  int64_t schema_version = OB_INVALID_VERSION;
  if (OB_ISNULL(schema_service_)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("valid schema service", K(ret));
  } else if (obrpc::ObAlterTableArg::PARTITIONED_TABLE == op_type) {
    ret = OB_NOT_SUPPORTED;
    LOG_WARN("split table partitions is not supported", KR(ret), K(orig_table_schema));
    LOG_USER_ERROR(OB_NOT_SUPPORTED, "split table partitions is");
  } else if (alter_table_arg.is_split_partition()) {
    if (OB_FAIL(ddl_operator.split_table_partitions(orig_table_schema,
                                                    inc_table_schema,
                                                    new_table_schema,
                                                    upd_table_schema,
                                                    trans))) {
      LOG_WARN("failed to add table partitions", KR(ret));
    }
  } else if (obrpc::ObAlterTableArg::ADD_PARTITION == op_type) {
    if (OB_FAIL(ObDDLLock::lock_for_add_partition_in_trans(orig_table_schema, trans))) {
      LOG_WARN("failed to lock for add drop partition", K(ret));
    } else if (OB_FAIL(gen_inc_table_schema_for_add_part(orig_table_schema, inc_table_schema))) {
      LOG_WARN("fail to gen inc table schema for add part",
               KR(ret), K(orig_table_schema), K(inc_table_schema));
    } else if (OB_FAIL(generate_object_id_for_partition_schema(inc_table_schema))) {
      LOG_WARN("fail to generate object_id for partition schema", KR(ret), K(inc_table_schema));
    } else if (OB_FAIL(generate_tablet_id(inc_table_schema))) {
      LOG_WARN("fail to fetch new table id", K(inc_table_schema), KR(ret));
    } else if (OB_FAIL(fix_local_idx_part_name_for_add_part_(orig_table_schema, inc_table_schema))) {
      LOG_WARN("fail to fix local idx part name for add part", KR(ret), K(orig_table_schema), K(inc_table_schema));
    } else if (OB_FAIL(ddl_operator.add_table_partitions(orig_table_schema,
                                                  inc_table_schema,
                                                  new_table_schema,
                                                  trans))) {
      LOG_WARN("failed to add table partitions", KR(ret));
    }
  } else if (obrpc::ObAlterTableArg::ADD_SUB_PARTITION == op_type) {
    if (OB_FAIL(ObDDLLock::lock_for_add_partition_in_trans(orig_table_schema, trans))) {
      LOG_WARN("failed to lock for add drop partition", K(ret));
    } else if (OB_FAIL(fix_local_idx_part_name_(orig_data_table_schema, orig_table_schema, inc_table_schema))) {
      LOG_WARN("fail to fix local idx part name", KR(ret), K(orig_data_table_schema), K(orig_table_schema), K(inc_table_schema));
    } else if (OB_FAIL(gen_inc_table_schema_for_add_subpart(orig_table_schema, inc_table_schema))) {
      LOG_WARN("fail to gen inc table schema for add subpart",
               KR(ret), K(orig_table_schema), K(inc_table_schema));
    } else if (OB_FAIL(generate_object_id_for_partition_schema(inc_table_schema, true))) {
      LOG_WARN("fail to generate object_id for partition schema", KR(ret), K(inc_table_schema));
    } else if (OB_FAIL(generate_tablet_id(inc_table_schema))) {
      LOG_WARN("fail to fetch new table id", K(inc_table_schema), KR(ret));
    } else if (OB_FAIL(fix_local_idx_part_name_for_add_subpart_(orig_table_schema, inc_table_schema))) {
      LOG_WARN("fail to fix local idx part name for add subpart", KR(ret), K(orig_table_schema), K(inc_table_schema));
    } else if (OB_FAIL(ddl_operator.add_table_subpartitions(orig_table_schema,
                                                            inc_table_schema,
                                                            new_table_schema,
                                                            trans))) {
      LOG_WARN("failed to add table partitions", KR(ret));
    }

  } else if (obrpc::ObAlterTableArg::RENAME_PARTITION == op_type) {
    if (OB_FAIL(gen_inc_table_schema_for_rename_part_(orig_table_schema, inc_table_schema))) {
      LOG_WARN("fail to gen inc table schema for rename part",
                KR(ret), K(orig_table_schema), K(inc_table_schema));
    } else if (OB_FAIL(lock_partitions(trans, inc_table_schema))) {
      LOG_WARN("failed to get tablet ids", KR(ret), K(orig_table_schema), K(inc_table_schema));
    } else if (OB_FAIL(ddl_operator.rename_table_partitions(orig_table_schema,
                                                            inc_table_schema,
                                                            new_table_schema,
                                                            trans))) {
      LOG_WARN("failed to rename table partitions", KR(ret), K(orig_table_schema), K(inc_table_schema), K(new_table_schema));
      }
  } else if (obrpc::ObAlterTableArg::RENAME_SUB_PARTITION == op_type) {
    if (OB_FAIL(gen_inc_table_schema_for_rename_subpart_(orig_table_schema, inc_table_schema))) {
      LOG_WARN("fail to gen inc table schema for rename subpart",
                KR(ret), K(orig_table_schema), K(inc_table_schema));
    } else if (OB_FAIL(lock_partitions(trans, inc_table_schema))) {
      LOG_WARN("failed to get tablet ids", KR(ret), K(orig_table_schema), K(inc_table_schema));
    } else if (OB_FAIL(ddl_operator.rename_table_subpartitions(orig_table_schema,
                                                               inc_table_schema,
                                                               new_table_schema,
                                                               trans))) {
    LOG_WARN("failed to rename table subpartitions", KR(ret), K(orig_table_schema), K(inc_table_schema));
    }
  } else if (obrpc::ObAlterTableArg::DROP_PARTITION == op_type) {
    if (OB_FAIL(fix_local_idx_part_name_(orig_data_table_schema, orig_table_schema, inc_table_schema))) {
      LOG_WARN("fix local idx part name", KR(ret), K(orig_data_table_schema), K(orig_table_schema), K(inc_table_schema));
    } else if (OB_FAIL(gen_inc_table_schema_for_drop_part(orig_table_schema, inc_table_schema))) {
      LOG_WARN("fail to gen inc table schema for drop part",
               KR(ret), K(orig_table_schema), K(inc_table_schema));
    } else if (OB_FAIL(lock_partitions(trans, inc_table_schema))) {
      LOG_WARN("failed to get tablet ids", KR(ret), K(orig_table_schema), K(inc_table_schema));
    } else if (OB_FAIL(ddl_operator.drop_table_partitions(orig_table_schema,
                                                          inc_table_schema,
                                                          new_table_schema,
                                                          trans))) {
      LOG_WARN("failed to drop table partitions", KR(ret));
    }
  } else if (obrpc::ObAlterTableArg::DROP_SUB_PARTITION == op_type) {
    if (inc_table_schema.get_partition_num() != 1) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("drop subparts not in a part", KR(ret));
    } else if (OB_FAIL(fix_local_idx_subpart_name_(orig_data_table_schema, orig_table_schema, inc_table_schema))) {
      LOG_WARN("fail to fix local idx subpart name", KR(ret), K(orig_data_table_schema), K(orig_table_schema), K(inc_table_schema));
    } else if (OB_FAIL(gen_inc_table_schema_for_drop_subpart(orig_table_schema, inc_table_schema))) {
      LOG_WARN("fail to gen inc table for drop subpart",
               KR(ret), K(orig_table_schema), K(inc_table_schema));
    } else if (OB_FAIL(lock_partitions(trans, inc_table_schema))) {
      LOG_WARN("failed to get tablet ids", KR(ret), K(orig_table_schema), K(inc_table_schema));
    } else if (OB_FAIL(ddl_operator.drop_table_subpartitions(orig_table_schema,
                                                   inc_table_schema,
                                                   new_table_schema,
                                                   trans))) {
      LOG_WARN("failed to drop table partitions", KR(ret));
    }
  } else if (obrpc::ObAlterTableArg::TRUNCATE_PARTITION == op_type) {
    if (OB_FAIL(fix_local_idx_part_name_(orig_data_table_schema, orig_table_schema, inc_table_schema))) {
      LOG_WARN("fix local idx part name", KR(ret), K(orig_data_table_schema), K(orig_table_schema), K(inc_table_schema));
    } else if (OB_FAIL(gen_inc_table_schema_for_trun_part(
                orig_table_schema, inc_table_schema, del_table_schema))) {
      LOG_WARN("fail to generate inc table schema", KR(ret), K(orig_table_schema));
    } else if (OB_FAIL(lock_partitions(trans, del_table_schema))) {
      LOG_WARN("failed to get tablet ids", KR(ret), K(orig_table_schema), K(del_table_schema));
    } else if (OB_FAIL(generate_object_id_for_partition_schema(inc_table_schema))) {
      LOG_WARN("fail to generate object_id for partition schema", KR(ret), K(inc_table_schema));
    } else if (OB_FAIL(generate_tablet_id(inc_table_schema))) {
      LOG_WARN("fail to fetch new table id", K(inc_table_schema), KR(ret));
    } else if (OB_FAIL(ddl_operator.truncate_table_partitions(orig_table_schema,
                                                              inc_table_schema,
                                                              del_table_schema,
                                                              trans))) {
      LOG_WARN("failed to truncate partitions", KR(ret));
    }
  } else if (obrpc::ObAlterTableArg::TRUNCATE_SUB_PARTITION == op_type) {
    if (OB_FAIL(fix_local_idx_subpart_name_(orig_data_table_schema, orig_table_schema, inc_table_schema))) {
      LOG_WARN("fail to fix local idx subpart name", KR(ret), K(orig_data_table_schema), K(orig_table_schema), K(inc_table_schema));
    } else if (OB_FAIL(gen_inc_table_schema_for_trun_subpart(
        orig_table_schema, inc_table_schema, del_table_schema))) {
      LOG_WARN("fail to generate inc table schema", KR(ret), K(orig_table_schema));
    } else if (OB_FAIL(lock_partitions(trans, del_table_schema))) {
      LOG_WARN("failed to get tablet ids", KR(ret), K(orig_table_schema), K(del_table_schema));
    } else if (OB_FAIL(generate_object_id_for_partition_schema(inc_table_schema, true))) {
      LOG_WARN("fail to generate object_id for partition schema", KR(ret), K(inc_table_schema));
    } else if (OB_FAIL(generate_tablet_id(inc_table_schema))) {
      LOG_WARN("fail to fetch new table id", K(inc_table_schema), KR(ret));
    } else if (OB_FAIL(ddl_operator.truncate_table_subpartitions(orig_table_schema,
                                                                 inc_table_schema,
                                                                 del_table_schema,
                                                                 trans))) {
      LOG_WARN("failed to drop table partitions", KR(ret));
    }
  } else if (obrpc::ObAlterTableArg::SET_INTERVAL == op_type) {
    // interval part table: modify interval
    // range part table: switch interval part table
    if (OB_FAIL(fill_interval_info_for_set_interval(orig_table_schema,
                                                    new_table_schema,
                                                    inc_table_schema))) {
      LOG_WARN("failed to fill interval info for set interval", KR(ret),
               K(orig_table_schema), K(inc_table_schema));
    }
  } else if (obrpc::ObAlterTableArg::INTERVAL_TO_RANGE == op_type) {
    if (OB_FAIL(reset_interval_info_for_interval_to_range(new_table_schema))) {
      LOG_WARN("failed to reset interval info for interval to range", KR(ret),
               K(orig_table_schema), K(inc_table_schema));
    }
  }
  if (OB_SUCC(ret) && !is_add_and_drop_partition(op_type) && !alter_table_arg.is_split_partition()) {
    if (OB_FAIL(check_alter_partition_with_tablegroup(&orig_table_schema, new_table_schema, schema_guard))) {
      LOG_WARN("fail to check alter partition with tablegroup", KR(ret), K(orig_table_schema), K(new_table_schema));
    }
  }
  return ret;
}

int ObDDLService::check_enable_sys_table_ddl(const ObTableSchema &table_schema,
                                             const ObSchemaOperationType operation_type)
{
  int ret = OB_SUCCESS;
  if (is_inner_table(table_schema.get_table_id())) {
    const bool enable_sys_table_ddl = common::ObServerConfig::get_instance().enable_sys_table_ddl;
    char err_msg[number::ObNumber::MAX_PRINTABLE_SIZE];
    if (!enable_sys_table_ddl) {
      ret = OB_OP_NOT_ALLOW;
      switch(operation_type) {
        case OB_DDL_CREATE_TABLE: {
          (void)snprintf(err_msg, sizeof(err_msg), "%s", "create system table");
          break;
        }
        case OB_DDL_ALTER_TABLE: {
          (void)snprintf(err_msg, sizeof(err_msg), "%s", "alter system table");
          break;
        }
        case OB_DDL_TABLE_RENAME: {
          (void)snprintf(err_msg, sizeof(err_msg), "%s", "rename system table");
          break;
        }
        case OB_DDL_DROP_TABLE: {
          if (table_schema.is_view_table()) {
            (void)snprintf(err_msg, sizeof(err_msg), "%s", "drop system view");
          } else {
            (void)snprintf(err_msg, sizeof(err_msg), "%s", "drop system table");
          }
          break;
        }
        case OB_DDL_TRUNCATE_TABLE_CREATE: {
          (void)snprintf(err_msg, sizeof(err_msg), "%s", "truncate system table");
          break;
        }
        default : {
          (void)snprintf(err_msg, sizeof(err_msg), "%s", "ddl on system table");
        }
      }
    } else if (OB_DDL_DROP_TABLE == operation_type && table_schema.has_partition()) {
      ret = OB_OP_NOT_ALLOW;
      (void)snprintf(err_msg, sizeof(err_msg),
               "%s", "drop partitioned system table");
    } else if (OB_DDL_DROP_COLUMN == operation_type && is_sys_table(table_schema.get_table_id())) {
      ret = OB_OP_NOT_ALLOW;
      (void)snprintf(err_msg, sizeof(err_msg),
               "%s", "system table drop column");

    }
    if (OB_FAIL(ret)) {
      LOG_USER_ERROR(OB_OP_NOT_ALLOW, err_msg);
    }
  }
  return ret;
}

// FIXME: this function should move to observer
int ObDDLService::alter_table_sess_active_time_in_trans(obrpc::ObAlterTableArg &alter_table_arg,
                                                        obrpc::ObAlterTableRes &res,
                                                        const uint64_t tenant_data_version)
{
  int ret = OB_SUCCESS;
  AlterTableSchema &alter_table_schema = alter_table_arg.alter_table_schema_;
  alter_table_schema.set_sess_active_time(ObTimeUtility::current_time());
  ObArray<uint64_t> tenant_ids;
  ObDDLType &ddl_type = res.ddl_type_;
  if (OB_ISNULL(schema_service_)) {
    ret = OB_ERR_UNEXPECTED;
  } else if (OB_FAIL(alter_table_schema.alter_option_bitset_.add_member(obrpc::ObAlterTableArg::SESSION_ACTIVE_TIME))) {
    LOG_WARN("failed to add member SESSION_ACTIVE_TIME for alter table schema", K(ret));
  } else if (OB_FAIL(schema_service_->get_tenant_ids(tenant_ids))) {
    LOG_WARN("fail to get schema guard", K(ret));
  } else {
    ObSchemaGetterGuard schema_guard;
    ObSEArray<const ObSimpleTableSchemaV2 *, 512> table_schemas;
    bool found = false;
    FOREACH_CNT_X(tenant_id, tenant_ids, OB_SUCC(ret) && !found) {
      if (OB_FAIL(schema_service_->get_tenant_schema_guard(*tenant_id, schema_guard))) {
        LOG_WARN("fail to get tenant schema guard", KR(ret), "tenant_id", *tenant_id);
      } else if (OB_FAIL(schema_guard.get_table_schemas_in_tenant(*tenant_id, table_schemas))) {
        LOG_WARN("fail to get table schema", K(ret), "tenant_id", *tenant_id);
      } else {
        const ObDatabaseSchema *database_schema = NULL;
        for (int64_t i = 0; i < table_schemas.count() && OB_SUCC(ret); i++) {
          const ObSimpleTableSchemaV2 *table_schema = table_schemas.at(i);
          if (OB_ISNULL(table_schema)) {
            ret = OB_ERR_UNEXPECTED;
            LOG_WARN("got invalid schema", K(ret), K(i));
          } else if (table_schema->is_tmp_table() && alter_table_arg.session_id_ == table_schema->get_session_id()) {
            // session_id must belong to one tenant
            found = true;
            database_schema = NULL;
            if (OB_FAIL(schema_guard.get_database_schema(*tenant_id, table_schema->get_database_id(), database_schema))) {
              LOG_WARN("failed to get database schema", K(ret), "tenant_id", *tenant_id);
            } else if (OB_ISNULL(database_schema)) {
              ret = OB_ERR_UNEXPECTED;
              LOG_WARN("database schema is null", K(ret));
            } else if (database_schema->is_in_recyclebin() || table_schema->is_in_recyclebin()) {
              LOG_INFO("skip table schema in recyclebin", K(*table_schema));
            } else {
              const uint64_t tenant_id = table_schema->get_tenant_id();
              alter_table_schema.set_origin_database_name(database_schema->get_database_name());
              alter_table_schema.set_origin_table_name(table_schema->get_table_name());
              alter_table_schema.set_tenant_id(tenant_id);
              if (OB_FAIL(check_is_offline_ddl(alter_table_arg, tenant_data_version, res.ddl_type_, res.ddl_need_retry_at_executor_))) {
                LOG_WARN("failed to to check is offline ddl", K(ret));
              } else {
                // offline ddl cannot appear at the same time with other ddl types
                if (is_long_running_ddl(res.ddl_type_)) {
                  if (OB_FAIL(do_offline_ddl_in_trans(alter_table_arg, tenant_data_version, res))) {
                    LOG_WARN("failed to do offline ddl in trans", K(ret), K(alter_table_arg));;
                  }
                } else {
                  if ((alter_table_arg.is_alter_mlog_attributes_ ||
                       alter_table_arg.is_alter_mview_attributes_) &&
                      OB_FAIL(ObMviewAlterService::alter_mview_or_mlog_in_trans(
                          alter_table_arg, res, schema_guard, schema_service_, sql_proxy_,
                          tenant_data_version))) {
                    LOG_WARN("alter_mview_or_mlog_in_trans failed", KR(ret));
                  } else if (OB_FAIL(
                                 alter_table_in_trans(alter_table_arg, res, tenant_data_version))) {
                    LOG_WARN("alter_table_in_trans failed", K(ret));
                  }
                }
              }
            }
          }
        } // end for
      }
    } // end FOREACH_CNT_X
  }
  return ret;
}

int ObDDLService::update_tables_attribute(ObIArray<ObTableSchema*> &new_table_schemas,
                                          ObDDLOperator &ddl_operator,
                                          common::ObMySQLTransaction &trans,
                                          const ObSchemaOperationType operation_type,
                                          const ObString &ddl_stmt_str)
{
  int ret = OB_SUCCESS;
  int64_t schema_count = new_table_schemas.count();
  if (schema_count < 1) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("local index should exist", K(new_table_schemas), KR(ret));
  }
  for (int64_t i = 0; OB_SUCC(ret) && i < schema_count; ++i) {
    if (OB_ISNULL(new_table_schemas.at(i))) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("schema ptr is NULL", KR(ret));
    } else if (OB_FAIL(ddl_operator.update_table_attribute(*new_table_schemas.at(i),
                                                    trans,
                                                    operation_type,
                                                    0 == i ? &ddl_stmt_str : NULL))) {
      LOG_WARN("failed to update data table schema version and max used column is!", KR(ret), KPC(new_table_schemas.at(i)));
    }
  }
  return ret;
}

int ObDDLService::check_need_add_progressive_round(
  const uint64_t tenant_data_version,
  const ObTableSchema &table_schema,
  const AlterTableSchema &alter_table_schema,
  bool &need_add_progressive_round)
{
  int ret = OB_SUCCESS;
  need_add_progressive_round = true;
  bool is_column_store_schema = false;
  bool has_all_column_group = false;
  if (tenant_data_version < DATA_VERSION_4_3_4_0) {
    // do nothing
  } else if (OB_FAIL(table_schema.get_is_column_store(is_column_store_schema))) {
    LOG_WARN("failed to get is column store", KR(ret));
  } else if (!is_column_store_schema) {
  } else if (OB_FAIL(table_schema.has_all_column_group(has_all_column_group))) {
    LOG_WARN("failed to check has all column group", KR(ret));
  } else if (!has_all_column_group) {
    // for pure column store table
    AlterColumnSchema *alter_column_schema = nullptr;
    ObTableSchema::const_column_iterator it_begin = alter_table_schema.column_begin();
    ObTableSchema::const_column_iterator it_end = alter_table_schema.column_end();
    need_add_progressive_round = false;
    for (;OB_SUCC(ret) && it_begin != it_end; it_begin++) {
      if (OB_ISNULL(alter_column_schema = static_cast<AlterColumnSchema *>(*it_begin))) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("*it_begin is NULL", K(ret));
      } else if (OB_DDL_ADD_COLUMN != alter_column_schema->alter_type_) {
        need_add_progressive_round = true;
        break;
      }
    }
  }
  return ret;
}

//fix me :Check whether the newly added index column covers the partition column --by rongxuan.lc
// It can be repaired after the featrue that add index in alter_table statement
int ObDDLService::alter_table_in_trans(obrpc::ObAlterTableArg &alter_table_arg,
                                       obrpc::ObAlterTableRes &res,
                                       const uint64_t tenant_data_version)
{
  int ret = OB_SUCCESS;
  ObDDLType &ddl_type = res.ddl_type_;

  if (OB_FAIL(check_inner_stat())) {
    LOG_WARN("variable is not init");
  } else {
    uint64_t tenant_data_version = 0;
    bool is_oracle_mode = false;
    ObArray<uint64_t> drop_cols_id_arr; // for defensive check.
    ObDDLOperator ddl_operator(*schema_service_, *sql_proxy_);
    AlterTableSchema &alter_table_schema = alter_table_arg.alter_table_schema_;
    HEAP_VAR(ObArray<ObDDLTaskRecord>, ddl_tasks) {
    HEAP_VAR(ObTableSchema, new_table_schema) {
    HEAP_VAR(ObAlterTableArg, const_alter_table_arg) {
    HEAP_VAR(ObMockFKParentTableSchema, mock_fk_parent_table_schema) {
    HEAP_VAR(ObSchemaGetterGuard, schema_guard) {
    const ObTableSchema *orig_table_schema = NULL;
    const ObTenantSchema *tenant_schema = NULL;
    uint64_t tenant_id = alter_table_schema.get_tenant_id();
    schema_guard.set_session_id(alter_table_arg.session_id_);
    int64_t refreshed_schema_version = 0;
    ObArenaAllocator allocator;
    ObArray<ObMockFKParentTableSchema> mock_fk_parent_table_schema_array;
    bool need_modify_notnull_validate = false;
    bool is_add_not_null_col = false;
    ObString empty_stmt;
    const bool need_deep_copy_arg = alter_table_arg.alter_constraint_type_ == ObAlterTableArg::ADD_CONSTRAINT
                                    || alter_table_arg.alter_constraint_type_ == ObAlterTableArg::ALTER_CONSTRAINT_STATE
                                    || alter_table_arg.foreign_key_arg_list_.count() > 0;
    // because the sql optimizer is using validate flag to generate optimized plan,
    // if add or modify a foregin key to validate state, it should take effect after checking whether the data is satisfied with the foreigin key
    if (need_deep_copy_arg && OB_FAIL(ObConstraintTask::deep_copy_table_arg(allocator,
                                                                            alter_table_arg,
                                                                            const_alter_table_arg))) {
      LOG_WARN("deep copy table arg failed", K(ret));
    } else if (alter_table_arg.foreign_key_checks_ && 1 == alter_table_arg.foreign_key_arg_list_.count()
              && alter_table_arg.foreign_key_arg_list_.at(0).need_validate_data_) {
      if ((!alter_table_arg.foreign_key_arg_list_.at(0).is_modify_fk_state_
            && alter_table_arg.foreign_key_arg_list_.at(0).validate_flag_)
          || (alter_table_arg.foreign_key_arg_list_.at(0).is_modify_validate_flag_
            && alter_table_arg.foreign_key_arg_list_.at(0).validate_flag_)) {
        alter_table_arg.ddl_stmt_str_ = empty_stmt;
        alter_table_arg.foreign_key_arg_list_.at(0).validate_flag_ = CST_FK_NO_VALIDATE;
      }
    } else if (OB_FAIL(need_modify_not_null_constraint_validate(
              alter_table_arg, is_add_not_null_col, need_modify_notnull_validate))) {
      LOG_WARN("check need modify not null constraint validate failed", K(ret));
    } else if (need_modify_notnull_validate) {
      alter_table_arg.ddl_stmt_str_ = empty_stmt;
      ObConstraint *cst = *alter_table_arg.alter_table_schema_.constraint_begin_for_non_const_iter();
      cst->set_validate_flag(CST_FK_NO_VALIDATE);
      cst->set_is_modify_validate_flag(false);
      const uint64_t col_id = *(cst->cst_col_begin());
      ObColumnSchemaV2 *col_schema = NULL;
      for (int64_t i = 0; OB_SUCC(ret) && i < alter_table_arg.alter_table_schema_.get_column_count(); i++) {
        if (alter_table_arg.alter_table_schema_.get_column_schema_by_idx(i)->get_column_id() == col_id) {
          col_schema = alter_table_arg.alter_table_schema_.get_column_schema_by_idx(i);
        }
      }
      if (OB_ISNULL(col_schema)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("column schema not found", K(ret), K(alter_table_arg));
      } else {
        col_schema->del_column_flag(NOT_NULL_VALIDATE_FLAG);
      }
    } else if (is_add_not_null_col) {
      for (int64_t i = 0; i < alter_table_arg.alter_table_schema_.get_column_count() && OB_SUCC(ret); i++) {
        ObColumnSchemaV2 *col_schema = NULL;
        if (OB_ISNULL(col_schema = alter_table_arg.alter_table_schema_.get_column_schema_by_idx(i))) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("column schema is null", K(ret));
        } else if (OB_DDL_ADD_COLUMN == static_cast<AlterColumnSchema *>(col_schema)->alter_type_) {
          col_schema->set_is_hidden(true);
        }
      }
    }

    if (OB_FAIL(ret)) {
    } else if (OB_FAIL(get_tenant_schema_guard_with_version_in_inner_table(tenant_id, schema_guard))) {
      LOG_WARN("fail to get schema guard with version in inner table", K(ret), K(tenant_id));
    } else if (OB_FAIL(get_and_check_table_schema(alter_table_arg,
                                                  schema_guard,
                                                  alter_table_schema,
                                                  orig_table_schema))) {
      LOG_WARN("fail to get and check table schema", K(ret));
    } else if (OB_FAIL(schema_guard.get_tenant_info(tenant_id, tenant_schema))) {
      LOG_WARN("fail to get tenant schema", K(ret), K(tenant_id));
    } else if (OB_UNLIKELY(NULL == tenant_schema)) {
      ret =  OB_ERR_UNEXPECTED;
      LOG_WARN("tenant schema is null", K(ret), KP(tenant_schema), K(tenant_id));
    } else if (orig_table_schema->is_materialized_view()) {
      const ObTableSchema *mv_orig_table_schema = nullptr;
      if (OB_FAIL(schema_guard.get_table_schema(tenant_id,
          orig_table_schema->get_data_table_id(), mv_orig_table_schema))) {
        LOG_WARN("failed to get mv container table schema", KR(ret));
      } else if (OB_ISNULL(mv_orig_table_schema)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("mv container table schema is null", KR(ret));
      } else {
        orig_table_schema = mv_orig_table_schema;
      }
    }

    if (OB_FAIL(ret)) {
    } else if (OB_FAIL(new_table_schema.assign(*orig_table_schema))) {
      LOG_WARN("fail to assign schema", K(ret));
    } else if (OB_FAIL(orig_table_schema->check_if_oracle_compat_mode(is_oracle_mode))) {
      LOG_WARN("check is oracle compat mode failed", KR(ret));
    } else if (OB_FAIL(GET_MIN_DATA_VERSION(tenant_id, tenant_data_version))) {
      LOG_WARN("get min data version failed", KR(ret), K(tenant_id));
    } else if (OB_FAIL(get_all_dropped_column_ids(alter_table_arg, *orig_table_schema, drop_cols_id_arr))) {
      LOG_WARN("fail to prefetch all drop columns id", KR(ret), K(alter_table_arg));
    } else if (!drop_cols_id_arr.empty()) {
      if (OB_UNLIKELY(obrpc::ObAlterTableArg::AlterAlgorithm::INSTANT != alter_table_arg.alter_algorithm_)) {
        ret = OB_NOT_SUPPORTED;
        LOG_WARN("not support to drop column instant under non-instant mode", KR(ret), K(alter_table_arg));
      } else if (OB_UNLIKELY(tenant_data_version >= DATA_VERSION_4_3_3_0
                              && tenant_data_version < DATA_VERSION_4_3_5_1)) {
        ret = OB_NOT_SUPPORTED;
        LOG_WARN("not support to drop column instant under this version", KR(ret), K(tenant_data_version), K(alter_table_arg));
      } else if (OB_UNLIKELY(ObDDLType::DDL_DROP_COLUMN_INSTANT != ddl_type)) {
        ret = OB_NOT_SUPPORTED;
        LOG_WARN("not support to drop column instant under non-instant mode", KR(ret), K(alter_table_arg));
      } else if (!is_oracle_mode) {
        ret = OB_NOT_SUPPORTED;
        LOG_WARN("not support to drop column instant under mysql mode now", KR(ret), K(alter_table_arg));
      }
    }

    if (OB_SUCC(ret)) {
      bool need_update_index_table = false;
      bool need_progressive_merge = false;
      if (alter_table_arg.is_alter_columns_) {
        if (OB_FAIL(check_need_add_progressive_round(
            tenant_data_version,
            *orig_table_schema,
            alter_table_arg.alter_table_schema_,
            need_progressive_merge))) {
          LOG_WARN("failed to check need progressive round", KR(ret));
        }
      } else if (alter_table_arg.is_alter_options_
          && alter_table_arg.need_progressive_merge()) {
        if (alter_table_arg.alter_table_schema_.alter_option_bitset_.
            has_member(ObAlterTableArg::ENCRYPTION) &&
            alter_table_arg.alter_table_schema_.is_equal_encryption(*orig_table_schema)) {
          // If the values before and after changing the encryption algorithm in the table are the same,
          // the merge is not marked
        } else {
          need_progressive_merge = true;
        }
      }
      if (OB_SUCC(ret) && need_progressive_merge) {
        alter_table_arg.is_alter_options_ = true;
        alter_table_arg.alter_table_schema_.set_progressive_merge_round(orig_table_schema->get_progressive_merge_round() + 1);
        if (OB_FAIL(alter_table_arg.alter_table_schema_.alter_option_bitset_.add_member(ObAlterTableArg::PROGRESSIVE_MERGE_ROUND))) {
          LOG_WARN("fail to add member progressive merge round", K(ret));
        }
      }
      if (OB_SUCC(ret)) {
        // check foreign key info and check constraint
        // and put foreign_key_info into table schema
        if (OB_FAIL(deal_with_cst_for_alter_table(schema_guard, tenant_id, orig_table_schema, alter_table_arg, mock_fk_parent_table_schema))) {
          LOG_WARN("deal with cst for alter table failed", K(ret));
        } else if (MOCK_FK_PARENT_TABLE_OP_INVALID != mock_fk_parent_table_schema.get_operation_type()
                  && OB_FAIL(mock_fk_parent_table_schema_array.push_back(mock_fk_parent_table_schema))) {
          // scence : alter table add fk referenced a parent table which is not exist
          LOG_WARN("push back to mock_fk_parent_table_schema_array failed", K(ret), K(mock_fk_parent_table_schema));
        } else if (OB_FAIL(set_new_table_options(alter_table_arg,
                    alter_table_schema, *tenant_schema, new_table_schema,
                    *orig_table_schema, schema_guard,
                    need_update_index_table))) {
          LOG_WARN("failed to set new table options", K(ret), K(new_table_schema),
          K(*orig_table_schema), K(ret));
        } else {
          new_table_schema.set_table_flags(alter_table_schema.get_table_flags());
        }
      }

      if (OB_SUCC(ret)) {
        if (is_oracle_mode && OB_FAIL(check_table_udt_id_is_exist(schema_guard, alter_table_schema, tenant_id))) {
           LOG_WARN("check udt id failed", KR(ret), K(alter_table_schema));
        }
      }

      ObDDLSQLTransaction trans(schema_service_);
      bool is_rename_and_need_table_lock =
          alter_table_schema.alter_option_bitset_.has_member(ObAlterTableArg::TABLE_NAME) && // is rename table;
          orig_table_schema->is_user_table() && // only user table
          ((MOCK_DATA_VERSION_4_2_3_0 <= tenant_data_version && tenant_data_version < DATA_VERSION_4_3_0_0) /* ([4.2.3, 4.3.0)) */
            || DATA_VERSION_4_3_2_0 <= tenant_data_version  /* [4.3.2, ~) */ );  // need table lock and rw defense;
      if (OB_FAIL(ret)) {
        //do nothing
      } else if (OB_FAIL(schema_guard.get_schema_version(tenant_id, refreshed_schema_version))) {
        LOG_WARN("failed to get tenant schema version", KR(ret), K(tenant_id));
      } else if (OB_FAIL(trans.start(sql_proxy_, tenant_id, refreshed_schema_version))) {
        LOG_WARN("start transaction failed", KR(ret), K(tenant_id), K(refreshed_schema_version));
      // All alter table behaviors will cause the status to change, which is not as fine as oracle
      } else if (need_modify_dep_obj_status(alter_table_arg)
                 && OB_FAIL(ObDependencyInfo::modify_dep_obj_status(trans, tenant_id,
                                                                    orig_table_schema->get_table_id(),
                                                                    ddl_operator, *schema_service_))) {
        LOG_WARN("failed to modify obj status", K(ret));
      } else if (is_rename_and_need_table_lock &&
                 OB_FAIL(lock_table(trans, *orig_table_schema))) {
        LOG_WARN("failed to get the table_lock of origin table schema for rename op", K(ret), KPC(orig_table_schema));
      } else {
        ObArray<ObTableSchema> global_idx_schema_array;
        //table columns
        if (OB_SUCC(ret) && alter_table_arg.is_alter_columns_) {
          if (OB_FAIL(check_can_alter_column(tenant_id,
                                             alter_table_schema,
                                             *orig_table_schema))) {
            LOG_WARN("fail to can alter column", K(ret), K(alter_table_arg));
          } else if (OB_FAIL(alter_table_column(*orig_table_schema,
                                                alter_table_schema,
                                                new_table_schema,
                                                alter_table_arg,
                                                schema_guard,
                                                tenant_data_version,
                                                ddl_operator,
                                                trans,
                                                &global_idx_schema_array))) {
            LOG_WARN("failed to alter table column!", K(*orig_table_schema), K(new_table_schema), K(ret));
          }
        }
        //table options
        // if there is no auto-increment column, ignore table option auto_increment
        if (0 == new_table_schema.get_autoinc_column_id()) {
          new_table_schema.set_auto_increment(1);
        }
        ObSArray<std::pair<uint64_t, int64_t>> idx_schema_versions;
        if (OB_FAIL(ret)) {
          // error occur
        } else if (OB_FAIL(ddl_operator.alter_table_options(
                schema_guard,
                new_table_schema,
                *orig_table_schema,
                (need_update_index_table || is_rename_and_need_table_lock),
                trans,
                &global_idx_schema_array,
                &idx_schema_versions))) {
          ObString origin_table_name = alter_table_schema.get_origin_table_name();
          LOG_WARN("failed to alter table options,", K(origin_table_name), K(new_table_schema), K(ret));
        } else if (is_rename_and_need_table_lock &&
            OB_FAIL(build_rw_defense_for_table_(
                tenant_data_version,
                new_table_schema,
                new_table_schema.get_schema_version(),
                idx_schema_versions,
                trans))) {
          LOG_WARN("failed to add rw defense for table", K(ret), K(new_table_schema));
        }
        if (OB_SUCC(ret) && !alter_table_schema.alter_option_bitset_.is_empty()) {
          const bool require_strict_binary_format = share::ObDDLUtil::use_idempotent_mode(tenant_data_version) && alter_table_arg.need_progressive_merge();
          if (OB_FAIL(ObDDLLock::lock_for_common_ddl_in_trans(*orig_table_schema, require_strict_binary_format, trans))) {
            LOG_WARN("failed to lock ddl", K(ret));
          }
        }
        // create duplicate ls if needed when altering normal table to duplicate table
        if (OB_SUCC(ret)
            && alter_table_arg.has_alter_duplicate_scope()
            && ObDuplicateScope::DUPLICATE_SCOPE_CLUSTER == new_table_schema.get_duplicate_scope()) {
          ObLSID dup_ls_id;
          ObDupLSCreateHelper dup_ls_create_helper;
          if (OB_FAIL(dup_ls_create_helper.init(tenant_id, sql_proxy_, rpc_proxy_, GCTX.location_service_))) {
            LOG_WARN("init failed", KR(ret), K(tenant_id));
          } else if (OB_FAIL(dup_ls_create_helper.check_and_create_duplicate_ls_if_needed(dup_ls_id))) {
            LOG_WARN("check and create duplicate ls ready if needed failed", KR(ret), K(tenant_id));
          }
        }

        // table foreign key
        if (OB_SUCC(ret)
            && !alter_table_arg.alter_table_schema_.get_foreign_key_infos().empty()) {
          if (OB_FAIL(alter_table_foreign_keys(
              *orig_table_schema,
              alter_table_schema,
              ddl_operator,
              trans))) {
            LOG_WARN("alter table foreign keys failed", K(ret));
          }
        }

        if (alter_table_arg.has_rename_action()) {
          OZ (ObPLDDLService::rebuild_trigger_on_rename(schema_guard, new_table_schema, ddl_operator, trans),
              new_table_schema.get_table_name_str());
        }

        //table indexs
        int64_t new_fetched_snapshot = 0;
        if (OB_SUCC(ret) && alter_table_arg.is_alter_indexs_) {
          if (OB_FAIL(check_restore_point_allow(tenant_id, *orig_table_schema))) {
            LOG_WARN("check restore point allow failed,", K(ret), K(tenant_id), K(orig_table_schema->get_table_id()));
          } else if (OB_FAIL(alter_table_index(alter_table_arg,
                                               *orig_table_schema,
                                               new_table_schema,
                                               schema_guard,
                                               ddl_operator,
                                               trans,
                                               alter_table_arg.allocator_,
                                               tenant_data_version,
                                               res,
                                               ddl_tasks,
                                               new_fetched_snapshot))) {
            LOG_WARN("failed to alter table index!", K(ret));
          }
        }
        // table constraints
        if (OB_SUCC(ret) && alter_table_arg.alter_constraint_type_ !=
            obrpc::ObAlterTableArg::CONSTRAINT_NO_OPERATION) {
          if (OB_FAIL(alter_table_constraints(
              alter_table_arg.alter_constraint_type_,
              schema_guard,
              *orig_table_schema,
              alter_table_schema,
              new_table_schema,
              ddl_operator,
              trans))) {
            LOG_WARN("alter table constraints failed", K(ret));
          }
        }
        // alter auto table attr
        if (OB_SUCC(ret) && alter_table_arg.alter_auto_partition_attr_) {
          ObAlterAutoPartAttrOp alter_auto_part(*this);
          if (OB_FAIL(check_restore_point_allow(tenant_id, *orig_table_schema))) {
            LOG_WARN("check restore point allow failed,", K(ret), K(tenant_id), K(orig_table_schema->get_table_id()));
          } else if (OB_FAIL(alter_auto_part.alter_table_auto_part_attr_if_need(
              alter_table_arg,
              ddl_type,
              schema_guard,
              new_table_schema,
              ddl_operator,
              trans))) {
            LOG_WARN("fail to alter table part attr.", K(ret));
          }
        }
        //table partitions
        // the first element is data_table_schema
        // the others element are local_index_schemas
        ObArray<const ObTableSchema*> orig_table_schemas;
        ObArray<ObTableSchema*> new_table_schemas;
        // 1. truncate part/subpart: del_table_schemas will be deleted and inc_table_schemas will be added
        // 2. others: inc_table_schemas and del_table_schemas is pointed same partitions
        // 3. split partition: upd_table_schemas record the partition information which need to be changed.
        // 4. other situations, upd_table_schemas are useless and point to same partition with new_table_schemas.
        ObArray<AlterTableSchema*> inc_table_schemas;
        ObArray<AlterTableSchema*> del_table_schemas;
        ObArray<ObTableSchema*> upd_table_schemas;
        ObArenaAllocator allocator("ModifyPart");
        //table partitions
        ObArray<int64_t> new_partition_ids;
        if (OB_SUCC(ret) && alter_table_arg.is_alter_partitions_) {
          if (OB_FAIL(check_restore_point_allow(tenant_id, *orig_table_schema))) {
            LOG_WARN("check restore point allow failed,", K(ret), K(tenant_id), K(orig_table_schema->get_table_id()));
          } else if (OB_FAIL(update_global_index(alter_table_arg,
                                                 tenant_id,
                                                 *orig_table_schema,
                                                 ddl_operator,
                                                 trans,
                                                 tenant_data_version,
                                                 res,
                                                 ddl_tasks))) {
            LOG_WARN("update_global_index failed", K(ret));
          } else if (OB_FAIL(generate_tables_array(alter_table_arg,
                                                   orig_table_schemas,
                                                   new_table_schemas,
                                                   inc_table_schemas,
                                                   del_table_schemas,
                                                   upd_table_schemas,
                                                   *orig_table_schema,
                                                   new_table_schema,
                                                   alter_table_schema,
                                                   schema_guard,
                                                   allocator))) {
            LOG_WARN("failed to generate tables array", KR(ret));
          } else if (OB_FAIL(alter_tables_partitions(alter_table_arg,
                                                     orig_table_schemas,
                                                     new_table_schemas,
                                                     inc_table_schemas,
                                                     del_table_schemas,
                                                     upd_table_schemas,
                                                     ddl_operator,
                                                     schema_guard,
                                                     trans))) {
            LOG_WARN("alter table partitions failed", K(ret));
          } else if (orig_table_schemas.count() != new_table_schemas.count()
                     || inc_table_schemas.count() != orig_table_schemas.count()
                     || del_table_schemas.count() != orig_table_schemas.count()
                     || inc_table_schemas.count() <= 0) {
            ret = OB_ERR_UNEXPECTED;
            LOG_WARN("array count is unexpected" , K(orig_table_schemas), K(new_table_schemas),
                     K(inc_table_schemas), K(del_table_schemas), KR(ret));
          } else if (alter_table_arg.task_id_ > 0 && OB_FAIL(ObDDLRetryTask::update_task_status_wait_child_task_finish(trans, tenant_id, alter_table_arg.task_id_))) {
            LOG_WARN("update ddl task status failed", K(ret));
          } else {
            if (alter_table_schema.is_external_table()) {
              if (alter_table_arg.alter_part_type_ == ObAlterTableArg::ADD_PARTITION
                  || alter_table_arg.alter_part_type_ == ObAlterTableArg::DROP_PARTITION) {
                for (int64_t i = 0; OB_SUCC(ret) && i < alter_table_schema.get_partition_num(); i++) {
                  ObAlterTableResArg arg;
                  CK (OB_NOT_NULL(alter_table_schema.get_part_array()) &&
                       OB_NOT_NULL(alter_table_schema.get_part_array()[i]));
                  if (OB_SUCC(ret)) {
                    arg.schema_id_ = alter_table_schema.get_table_id();
                    arg.schema_type_ = ObSchemaType::TABLE_SCHEMA;
                    arg.schema_version_ = alter_table_schema.get_part_array()[i]->get_schema_version();
                    arg.part_object_id_ = alter_table_schema.get_part_array()[i]->get_part_id();
                    OZ (res.res_arg_array_.push_back(arg));
                  }
                }
              } else {
                ret = OB_ERR_UNEXPECTED;
                LOG_WARN("alter external table op type is not valid", K(ret));
              }
            }
          }
        }

        // alter table auto increment value
        if (OB_SUCC(ret) &&
              alter_table_schema.alter_option_bitset_.has_member(ObAlterTableArg::AUTO_INCREMENT) &&
              0 != new_table_schema.get_autoinc_column_id()) {
          if (OB_FAIL(alter_table_auto_increment(*orig_table_schema,
                                                 alter_table_schema,
                                                 alter_table_arg,
                                                 schema_guard,
                                                 new_table_schema,
                                                 ddl_operator,
                                                 trans))) {
            LOG_WARN("fail to alter table auto increment value", K(ret));
          }
        }

        // alter column group delayed
        if (OB_SUCC(ret) && (ObDDLType::DDL_ALTER_COLUMN_GROUP_DELAYED == ddl_type)) {
          if (tenant_data_version < DATA_VERSION_4_3_5_0) {
            ret = OB_NOT_SUPPORTED;
            LOG_WARN("compat version not support", K(ret), K(tenant_data_version));
            LOG_USER_ERROR(OB_NOT_SUPPORTED, "tenant data version is less than 4.3.5, alter column group delayed");
          } else if (GCTX.is_shared_storage_mode()) {
            ret = OB_NOT_SUPPORTED;
            SQL_RESV_LOG(WARN, "alter column group delayed does not support shared storage mode", K(ret));
          } else if (OB_FAIL(ObSchemaUtils::mock_default_cg(orig_table_schema->get_tenant_id(), new_table_schema))) {
            LOG_WARN("fail to mock default cg", K(ret), K(orig_table_schema), K(new_table_schema));
          } else if (OB_FAIL(alter_column_group(alter_table_arg,
                                                *orig_table_schema,
                                                new_table_schema,
                                                schema_guard,
                                                ddl_operator,
                                                trans))) {
            LOG_WARN("failed to alter table column group", K(ret));
          } else if (OB_FAIL(update_column_group_table_inplace(*orig_table_schema,
                                                      new_table_schema,
                                                      ddl_operator,
                                                      trans))) {
            LOG_WARN("failed to alter table column group table", K(ret));
          } else {
            // only change schemas here, leave data reshaping in major merge
            LOG_INFO("alter column group delayed", K(ret), KPC(orig_table_schema), K(new_table_schema));
          }
        }

        if (OB_SUCC(ret)) {
          ObSchemaOperationType operation_type = OB_DDL_ALTER_TABLE;
          if (obrpc::ObAlterTableArg::PARTITIONED_TABLE == alter_table_arg.alter_part_type_) {
            operation_type = OB_DDL_PARTITIONED_TABLE;
          } else if (alter_table_arg.is_split_partition()) {
            operation_type = OB_DDL_SPLIT_PARTITION;
          } else if (obrpc::ObAlterTableArg::TRUNCATE_PARTITION == alter_table_arg.alter_part_type_) {
            operation_type = OB_DDL_TRUNCATE_PARTITION;
          } else if (obrpc::ObAlterTableArg::ADD_SUB_PARTITION == alter_table_arg.alter_part_type_) {
            operation_type = OB_DDL_ADD_SUB_PARTITION;
          } else if (obrpc::ObAlterTableArg::DROP_SUB_PARTITION == alter_table_arg.alter_part_type_) {
            operation_type = OB_DDL_DROP_SUB_PARTITION;
          } else if (obrpc::ObAlterTableArg::TRUNCATE_SUB_PARTITION == alter_table_arg.alter_part_type_) {
            operation_type = OB_DDL_TRUNCATE_SUB_PARTITION;
          } else if (obrpc::ObAlterTableArg::RENAME_PARTITION == alter_table_arg.alter_part_type_) {
            operation_type = OB_DDL_RENAME_PARTITION;
          } else if (obrpc::ObAlterTableArg::RENAME_SUB_PARTITION == alter_table_arg.alter_part_type_) {
            operation_type = OB_DDL_RENAME_SUB_PARTITION;
          } else if (obrpc::ObAlterTableArg::SET_INTERVAL == alter_table_arg.alter_part_type_) {
            operation_type = OB_DDL_SET_INTERVAL;
          } else if (obrpc::ObAlterTableArg::INTERVAL_TO_RANGE == alter_table_arg.alter_part_type_) {
            operation_type = OB_DDL_INTERVAL_TO_RANGE;
          }

          if (!alter_table_arg.is_alter_partitions_) {
            if (alter_table_arg.alter_auto_partition_attr_) {
              // do nothing
            } else if (OB_FAIL(ddl_operator.update_table_attribute(
                        new_table_schema,
                        trans,
                        operation_type,
                        &alter_table_arg.ddl_stmt_str_))) {
              LOG_WARN("failed to update tablets attribute", K(ret), K(new_table_schema));
            }
          } else {
            if (OB_FAIL(update_tables_attribute(
                        new_table_schemas,
                        ddl_operator,
                        trans,
                        operation_type,
                        alter_table_arg.ddl_stmt_str_))) {
              LOG_WARN("failed to update tablets attribute", K(ret), K(new_table_schema));
            }
          }

          if (OB_FAIL(ret)) {
          } else if (alter_table_schema.alter_option_bitset_.has_member(ObAlterTableArg::SESSION_ID) &&
                       0 == new_table_schema.get_session_id() && !new_table_schema.is_tmp_table() &&
                       OB_FAIL(ddl_operator.delete_temp_table_info(trans, new_table_schema))) {
              LOG_WARN("failed to delete temp table info", K(ret));
          }
        }
        // scence : alter table rename to a mock fk parent table existed, will replace mock fk parent table with real table
        if (OB_SUCC(ret) && alter_table_schema.alter_option_bitset_.has_member(obrpc::ObAlterTableArg::TABLE_NAME)) {
          const ObMockFKParentTableSchema *ori_mock_fk_parent_table_ptr = NULL;
          if (OB_FAIL(schema_guard.get_mock_fk_parent_table_schema_with_name(
              orig_table_schema->get_tenant_id(),
              new_table_schema.get_database_id(),
              new_table_schema.get_table_name_str(),
              ori_mock_fk_parent_table_ptr))) {
            LOG_WARN("get_mock_fk_parent_table_schema_with_name failed", K(ret),
                K(orig_table_schema->get_tenant_id()), K(new_table_schema.get_database_id()), K(new_table_schema.get_table_name_str()));
          } else if (OB_NOT_NULL(ori_mock_fk_parent_table_ptr)) {
            ObArray<ObAuxTableMetaInfo> simple_index_infos;
            ObArray<const share::schema::ObTableSchema*> unique_index_schemas;
            if (OB_FAIL(orig_table_schema->get_simple_index_infos(simple_index_infos))) {
              SERVER_LOG(WARN, "get simple_index_infos without delay_deleted_tid failed", K(ret));
            } else {
              const ObTableSchema *index_schema = NULL;
              for (int64_t j = 0; OB_SUCC(ret) && j < simple_index_infos.count(); ++j) {
                if (OB_FAIL(schema_guard.get_table_schema(alter_table_schema.get_tenant_id(), simple_index_infos.at(j).table_id_, index_schema))) {
                  LOG_WARN("fail to get to_table_schema schema", K(ret));
                } else if (OB_ISNULL(index_schema)) {
                  ret = OB_ERR_UNEXPECTED;
                  LOG_WARN("index_schema is null", K(ret));
                } else if (index_schema->is_unique_index() && OB_FAIL(unique_index_schemas.push_back(index_schema))) {
                  LOG_WARN("fail to push_back index_schema to unique_index_schemas", K(ret));
                }
              }
              if (OB_SUCC(ret)) {
                // alter table rename to mock fk parent table name with other actions is not supported
                // so we can use orig_table_schema to check and gen mock_fk_parent_table_for_replacing
                if (OB_FAIL(gen_mock_fk_parent_table_for_replacing_mock_fk_parent_table(
                    schema_guard, ori_mock_fk_parent_table_ptr->get_mock_fk_parent_table_id(),
                    *orig_table_schema, unique_index_schemas, mock_fk_parent_table_schema))) {
                  LOG_WARN("failed to gen_mock_fk_parent_table_for_replacing_mock_fk_parent_table", K(ret));
                } else if (OB_FAIL(mock_fk_parent_table_schema_array.push_back(mock_fk_parent_table_schema))) {
                  LOG_WARN("failed to push mock_fk_parent_table_schema", K(ret), K(mock_fk_parent_table_schema));
                }
              }
            }
          }
        }
        if (FAILEDx(ddl_operator.deal_with_mock_fk_parent_tables(trans, schema_guard, mock_fk_parent_table_schema_array))) {
          LOG_WARN("failed to deal_with_mock_fk_parent_tables", K(ret), K(mock_fk_parent_table_schema_array.count()));
        }
        int64_t last_schema_version = OB_INVALID_VERSION;
        if (OB_SUCC(ret)) {
          if (OB_FAIL(get_last_schema_version(last_schema_version))) {
            LOG_WARN("fail to get last schema version", KR(ret));
          }
        }
        for (int64_t i = 0; OB_SUCC(ret) && i < alter_table_arg.index_arg_list_.size(); ++i) {
          const ObIndexArg *index_arg = alter_table_arg.index_arg_list_.at(i);
          if (OB_ISNULL(index_arg)) {
            ret = OB_INVALID_ARGUMENT;
            LOG_WARN("index arg should not be null", K(ret));
          } else if (index_arg->index_action_type_ == ObIndexArg::ADD_INDEX) {
            const ObCreateIndexArg *create_index_arg = static_cast<const ObCreateIndexArg *>(index_arg);
            const ObTableSchema &index_schema = create_index_arg->index_schema_;
            if (OB_FAIL(ddl_operator.insert_ori_schema_version(
                trans, tenant_id, index_schema.get_table_id(), last_schema_version))) {
              LOG_WARN("failed to insert_ori_schema_version!", K(ret), K(tenant_id));
            }
          }
        }

        if (OB_SUCC(ret)) {
          const bool need_create_empty_major_sstable = !alter_table_arg.is_split_partition();
          common::ObArray<bool> need_create_empty_majors;
          common::ObArray<const ObTableSchema*> inc_table_schema_ptrs;
          common::ObArray<const ObTableSchema*> del_table_schema_ptrs;
          for (int i = 0; i < inc_table_schemas.count() && OB_SUCC(ret); i++) {
            const ObTableSchema *tmp_table_schema = inc_table_schemas.at(i);
            if (OB_ISNULL(tmp_table_schema) || OB_ISNULL(new_table_schemas.at(i))) {
              ret = OB_ERR_UNEXPECTED;
              LOG_WARN("table_schemas is NULL", KR(ret), K(i), K(tmp_table_schema));
            } else if (FALSE_IT(inc_table_schemas.at(i)->set_schema_version(new_table_schemas.at(i)->get_schema_version()))) {
            } else if (OB_FAIL(inc_table_schema_ptrs.push_back(tmp_table_schema))
              || OB_FAIL(need_create_empty_majors.push_back(need_create_empty_major_sstable))) {
              LOG_WARN("fail to push back", KR(ret), KPC(tmp_table_schema));
            }
          }
          for (int i = 0; i < del_table_schemas.count() && OB_SUCC(ret); i++) {
            const ObTableSchema *tmp_table_schema = del_table_schemas.at(i);
            if (OB_ISNULL(tmp_table_schema) || OB_ISNULL(new_table_schemas.at(i))) {
              ret = OB_ERR_UNEXPECTED;
              LOG_WARN("table_schemas is NULL", KR(ret), K(i), K(tmp_table_schema));
            } else if (FALSE_IT(del_table_schemas.at(i)->set_schema_version(new_table_schemas.at(i)->get_schema_version()))) {
            } else if (OB_FAIL(del_table_schema_ptrs.push_back(tmp_table_schema))) {
              LOG_WARN("fail to push back", KR(ret), KPC(tmp_table_schema));
            }
          }
          if (OB_FAIL(ret)) {
          } else if (obrpc::ObAlterTableArg::DROP_PARTITION == alter_table_arg.alter_part_type_
                     || obrpc::ObAlterTableArg::DROP_SUB_PARTITION == alter_table_arg.alter_part_type_
                     || obrpc::ObAlterTableArg::TRUNCATE_PARTITION == alter_table_arg.alter_part_type_
                     || obrpc::ObAlterTableArg::TRUNCATE_SUB_PARTITION == alter_table_arg.alter_part_type_) {
            int64_t new_schema_version = OB_INVALID_VERSION;
            if (OB_FAIL(schema_service_->gen_new_schema_version(tenant_id, new_schema_version))) {
              LOG_WARN("fail to gen new schema_version", KR(ret), K(tenant_id));
            } else {
              ObTabletDrop tablet_drop(tenant_id, trans, new_schema_version);
              if (alter_table_arg.alter_table_schema_.is_external_table()) {
              } else if (OB_FAIL(tablet_drop.init())) {
                LOG_WARN("fail to init tablet drop", KR(ret), K(del_table_schema_ptrs));
              } else if (OB_FAIL(tablet_drop.add_drop_tablets_of_table_arg(del_table_schema_ptrs))) {
                LOG_WARN("failed to add drop tablets", KR(ret), K(del_table_schema_ptrs));
              } else if (OB_FAIL(tablet_drop.execute())) {
                LOG_WARN("failed to execute", KR(ret), K(del_table_schema_ptrs));
              }
            }
          }

          if (OB_FAIL(ret)) {
          } else if (obrpc::ObAlterTableArg::ADD_PARTITION == alter_table_arg.alter_part_type_
                     || obrpc::ObAlterTableArg::ADD_SUB_PARTITION == alter_table_arg.alter_part_type_
                     || obrpc::ObAlterTableArg::TRUNCATE_PARTITION == alter_table_arg.alter_part_type_
                     || obrpc::ObAlterTableArg::TRUNCATE_SUB_PARTITION == alter_table_arg.alter_part_type_) {
            SCN frozen_scn;
            const bool need_check_tablet_cnt = obrpc::ObAlterTableArg::ADD_PARTITION == alter_table_arg.alter_part_type_
                     || obrpc::ObAlterTableArg::ADD_SUB_PARTITION == alter_table_arg.alter_part_type_;
            if (alter_table_arg.alter_table_schema_.is_external_table()) {
            } else if (OB_ISNULL(GCTX.root_service_)) {
              ret = OB_ERR_UNEXPECTED;
              LOG_WARN("root service is null", KR(ret));
            } else if (OB_FAIL(ObMajorFreezeHelper::get_frozen_scn(tenant_id, frozen_scn))) {
              LOG_WARN("failed to get frozen status for create tablet", KR(ret), K(tenant_id));
            } else {
              ObTableCreator table_creator(tenant_id,
                                           frozen_scn,
                                           trans);
              common::ObArray<share::ObLSID> ls_id_array;
              const ObTableSchema *tmp_table_schema = inc_table_schema_ptrs.at(0);
              const ObTablegroupSchema *tablegroup_schema = NULL; // keep NULL if no tablegroup
              ObNewTableTabletAllocator new_table_tablet_allocator(tenant_id, schema_guard, sql_proxy_);
              if (OB_ISNULL(tmp_table_schema)) {
                ret = OB_ERR_UNEXPECTED;
                LOG_WARN("table schema is null", KR(ret), K(inc_table_schema_ptrs));
              } else if (OB_FAIL(table_creator.init(need_check_tablet_cnt))) {
                LOG_WARN("fail to init table creator", KR(ret));
              } else if (OB_FAIL(new_table_tablet_allocator.init())) {
                LOG_WARN("fail to init new table tablet allocator", KR(ret));
              } else if (OB_INVALID_ID != tmp_table_schema->get_tablegroup_id()) {
                if (OB_FAIL(schema_guard.get_tablegroup_schema(
                    tmp_table_schema->get_tenant_id(),
                    tmp_table_schema->get_tablegroup_id(),
                    tablegroup_schema))) {
                  LOG_WARN("get tablegroup_schema failed", KR(ret), KPC(tmp_table_schema));
                } else if (OB_ISNULL(tablegroup_schema)) {
                  ret = OB_ERR_UNEXPECTED;
                  LOG_WARN("tablegroup_schema is null", KR(ret), KPC(tmp_table_schema));
                }
              }
              if (OB_FAIL(ret)) {
              } else if (OB_FAIL(new_table_tablet_allocator.prepare(trans, *tmp_table_schema, tablegroup_schema, true))) {
                LOG_WARN("failed to prepare tablet allocator", KR(ret), KPC(tmp_table_schema));
              } else if (OB_FAIL(new_table_tablet_allocator.get_ls_id_array(ls_id_array))) {
                LOG_WARN("fail to get ls id array", KR(ret));
              } else if (OB_FAIL(table_creator.add_create_tablets_of_tables_arg(
                      inc_table_schema_ptrs,
                      ls_id_array,
                      tenant_data_version,
                      need_create_empty_majors/*need_create_empty_majors*/))) {
                LOG_WARN("create table partitions failed", KR(ret), K(alter_table_schema),
                                                           K(inc_table_schema_ptrs));
              } else if (OB_FAIL(table_creator.execute())) {
                LOG_WARN("execute create partition failed", KR(ret));
              }
            }
          }

          if (OB_SUCC(ret) && alter_table_arg.is_split_partition()) {
            // TODO(zhenhan.gzh): modify ddl_type based on split type(auto/manual) and part_type
            const ObDDLType ddl_type = alter_table_arg.is_auto_split_partition() ? ObDDLType::DDL_AUTO_SPLIT_BY_RANGE : ObDDLType::DDL_MANUAL_SPLIT_BY_RANGE;
            ObSplitPartitionHelper split_partition_helper(trans, schema_guard, alter_table_arg.allocator_, tenant_id,
                tenant_data_version, ddl_type, new_table_schemas, upd_table_schemas, inc_table_schema_ptrs, alter_table_arg.parallelism_);
            ObDDLTaskRecord task_record;
            if (OB_FAIL(split_partition_helper.execute(task_record))) {
              LOG_WARN("failed to execute split partition helper", KR(ret));
            } else if (OB_FAIL(ddl_tasks.push_back(task_record))) {
              LOG_WARN("failed to push back", KR(ret));
            } else {
              res.ddl_type_ = ddl_type;
              res.task_id_ = task_record.task_id_;
            }
          }
        }

        if (OB_FAIL(ret)) {
        } else if (obrpc::ObAlterTableArg::PARTITIONED_TABLE == alter_table_arg.alter_part_type_) {
          ret = OB_NOT_SUPPORTED;
          LOG_WARN("split partition is not supported", KR(ret), K(alter_table_schema));
          LOG_USER_ERROR(OB_NOT_SUPPORTED, "split partition is");
        }

        if (OB_SUCC(ret) &&
             alter_table_schema.alter_option_bitset_.has_member(ObAlterTableArg::INCREMENT_MODE) &&
             0 != orig_table_schema->get_autoinc_column_id()) {
          ObAutoincrementService &autoinc_service = ObAutoincrementService::get_instance();
          if (OB_FAIL(autoinc_service.clear_autoinc_cache_all(tenant_id,
                        orig_table_schema->get_table_id(),
                        orig_table_schema->get_autoinc_column_id(),
                        orig_table_schema->is_order_auto_increment_mode()))) {
            LOG_WARN("fail to clear autoinc cache", K(ret));
          }
        }
        if (OB_SUCC(ret) && ObDDLType::DDL_DROP_COLUMN_INSTANT == ddl_type) {
          // to drop invalid index here is to avoid index being renamed to normal one when updating table options above.
          ObArray<uint64_t> drop_cols_id_arr;
          if (OB_FAIL(get_all_dropped_column_ids(alter_table_arg, *orig_table_schema, drop_cols_id_arr))) {
            LOG_WARN("fail to prefetch all drop columns id", KR(ret), K(alter_table_arg));
          } else if (OB_FAIL(drop_rls_policy_caused_by_drop_column_online(schema_guard, *orig_table_schema,
              drop_cols_id_arr, new_table_schema, ddl_operator, trans))) {
            LOG_WARN("drop rls policy failed", KR(ret));
          } else if (OB_FAIL(drop_lob_caused_by_drop_column_online_if_need(alter_table_arg, *orig_table_schema,
              new_table_schema, alter_table_arg.allocator_, trans, ddl_tasks, res))) {
            LOG_WARN("drop lob caused by drop column online failed", KR(ret));
          } else if (OB_FAIL(drop_index_caused_by_drop_column_online(schema_guard, *orig_table_schema, drop_cols_id_arr,
              alter_table_arg.allocator_, ddl_operator, trans, ddl_tasks))) {
            LOG_WARN("drop index caused by drop column failed", KR(ret), K(drop_cols_id_arr));
          }
        }

        if (OB_FAIL(ret)) {
        } else if (DDL_CREATE_INDEX == ddl_type || DDL_CREATE_PARTITIONED_LOCAL_INDEX == ddl_type || DDL_NORMAL_TYPE == ddl_type) {
          ObIndexBuilder index_builder(*this);
          const ObSArray<ObIndexArg *> &index_arg_list = alter_table_arg.index_arg_list_;
          int tmp_ret = OB_SUCCESS;
          const int64_t new_table_id = new_table_schema.get_table_id();
          ObArray<ObTabletID> inc_tablet_ids;
          ObArray<ObTabletID> del_tablet_ids;
          if (index_arg_list.count() > 0) {
            if (obrpc::ObAlterTableArg::TRUNCATE_PARTITION == alter_table_arg.alter_part_type_
                || obrpc::ObAlterTableArg::TRUNCATE_SUB_PARTITION == alter_table_arg.alter_part_type_) {
              if (OB_FAIL(get_tablets_with_table_id_(inc_table_schemas, new_table_id, inc_tablet_ids))) {
                LOG_WARN("fail to get tablets with table id", KR(ret), K(inc_table_schemas), K(new_table_id));
              }
            }
            if (obrpc::ObAlterTableArg::DROP_PARTITION == alter_table_arg.alter_part_type_
                || obrpc::ObAlterTableArg::DROP_SUB_PARTITION == alter_table_arg.alter_part_type_
                || obrpc::ObAlterTableArg::TRUNCATE_PARTITION == alter_table_arg.alter_part_type_
                || obrpc::ObAlterTableArg::TRUNCATE_SUB_PARTITION == alter_table_arg.alter_part_type_) {
              if (FAILEDx(get_tablets_with_table_id_(del_table_schemas, new_table_id, del_tablet_ids))) {
                LOG_WARN("fail to get tablets with table id", KR(ret), K(inc_table_schemas), K(new_table_id));
              }
            }

          }
          for (int64_t i = 0; OB_SUCC(ret) && i < index_arg_list.size(); ++i) {
            ObIndexArg *index_arg = const_cast<ObIndexArg *>(index_arg_list.at(i));
            ObDDLTaskRecord task_record;
            if (OB_ISNULL(index_arg)) {
              ret = OB_INVALID_ARGUMENT;
              LOG_WARN("index arg should not be null", KR(ret));
            } else if (ObIndexArg::ADD_INDEX == index_arg->index_action_type_
                       || ObIndexArg::REBUILD_INDEX == index_arg->index_action_type_) {
              ObCreateIndexArg *create_index_arg = static_cast<ObCreateIndexArg *>(index_arg);
              ObTableSchema &index_schema = create_index_arg->index_schema_;
              if (INDEX_TYPE_PRIMARY == create_index_arg->index_type_ ||
                  is_vec_index(create_index_arg->index_type_)) {
              } else if (OB_FAIL(index_builder.submit_build_index_task(trans,
                                                                  *create_index_arg,
                                                                  orig_table_schema,
                                                                  &inc_tablet_ids,
                                                                  &del_tablet_ids,
                                                                  &index_schema,
                                                                  alter_table_arg.parallelism_,
                                                                  const_alter_table_arg.consumer_group_id_,
                                                                  tenant_data_version,
                                                                  alter_table_arg.allocator_,
                                                                  task_record,
                                                                  new_fetched_snapshot))) {
                LOG_WARN("fail to submit build index task", KR(ret), "type", create_index_arg->index_type_);
              } else if (OB_FAIL(ddl_tasks.push_back(task_record))) {
                LOG_WARN("fail to push ddl task", KR(ret), K(task_record));
              } else {
                res.task_id_ = task_record.task_id_;
                ObDDLRes ddl_res;
                ddl_res.tenant_id_ = tenant_id;
                ddl_res.schema_id_ = create_index_arg->index_schema_.get_schema_version();
                ddl_res.task_id_ = task_record.task_id_;
                obrpc::ObAlterTableResArg arg(TABLE_SCHEMA,
                                              create_index_arg->index_schema_.get_table_id(),
                                              create_index_arg->index_schema_.get_schema_version());
                if (OB_FAIL(res.res_arg_array_.push_back(arg))) {
                  LOG_WARN("push back to res_arg_array failed", KR(ret), K(arg));
                } else if (OB_FAIL(res.ddl_res_array_.push_back(ddl_res))) {
                  LOG_WARN("failed to push back ddl res array", KR(ret));
                }
              }
            // TODO @wenyu alter table drop index submit drop index task in alter_table_index() now.
            //             should be unified here
            } else if (ObIndexArg::DROP_INDEX == index_arg->index_action_type_ && !alter_table_arg.is_alter_indexs_) {
              ObDropIndexArg *drop_index_arg = static_cast<ObDropIndexArg *>(index_arg);
              if (OB_ISNULL(drop_index_arg)) {
                ret = OB_ERR_UNEXPECTED;
                LOG_WARN("drop index arg is null", KR(ret));
              } else if (OB_FAIL(drop_index_to_scheduler_(trans, schema_guard, alter_table_arg.allocator_ , *orig_table_schema,
                                                          &inc_tablet_ids, &del_tablet_ids, drop_index_arg,
                                                          ddl_operator, res, ddl_tasks))) {
                LOG_WARN("fail to drop index to scheduler", KR(ret), KPC(drop_index_arg));
              }
            }
          }
        }
        if (OB_FAIL(ret)) {
        } else if (alter_table_arg.alter_constraint_type_ == obrpc::ObAlterTableArg::ADD_CONSTRAINT
                    || alter_table_arg.alter_constraint_type_ == obrpc::ObAlterTableArg::ALTER_CONSTRAINT_STATE) {
          ObTableSchema::const_constraint_iterator iter = alter_table_arg.alter_table_schema_.constraint_begin();
          /*
            * observer need return while create index, create constraint or modify constraint.
            * in the resolver phase, it ensures that no other actions will happen at the same time while alter table.
            * check constraint need return constriant_id_ and schema_version_. other constraint return schema_version_.
            * the schema version is data table after finish alter table.
          */
          const bool need_check = need_check_constraint_validity(const_alter_table_arg);
          res.constriant_id_ = (*iter)->get_constraint_id();
          res.schema_version_ = new_table_schema.get_schema_version();
          if (need_check) {
            bool need_modify_notnull_validate = false;
            bool is_add_not_null_col = false;
            if (OB_FAIL(need_modify_not_null_constraint_validate(
                  const_alter_table_arg, is_add_not_null_col, need_modify_notnull_validate))) {
            } else {
              ObDDLTaskRecord task_record;
              ObTableLockOwnerID owner_id;
              ddl_type = is_add_not_null_col ?
                         ObDDLType::DDL_ADD_NOT_NULL_COLUMN : ObDDLType::DDL_CHECK_CONSTRAINT;
              ObCreateDDLTaskParam param(new_table_schema.get_tenant_id(),
                                        ddl_type,
                                        &new_table_schema,
                                        nullptr,
                                        (*iter)->get_constraint_id(),
                                        new_table_schema.get_schema_version(),
                                        0/*parallelsim*/,
                                        const_alter_table_arg.consumer_group_id_,
                                        &alter_table_arg.allocator_,
                                        &const_alter_table_arg);
              if (OB_FAIL(GCTX.root_service_->get_ddl_scheduler().create_ddl_task(param, trans, task_record))) {
                LOG_WARN("submit constraint task failed", K(ret));
              } else if (OB_FAIL(owner_id.convert_from_value(ObLockOwnerType::DEFAULT_OWNER_TYPE,
                                                             task_record.task_id_))) {
                LOG_WARN("failed to get owner id", K(ret), K(task_record.task_id_));
              } else if (OB_FAIL(ObDDLLock::lock_for_common_ddl(new_table_schema,
                                                                owner_id,
                                                                trans))) {
                LOG_WARN("failed to lock online ddl lock", K(ret));
              } else if (OB_FAIL(ddl_tasks.push_back(task_record))) {
                LOG_WARN("fail to push ddl task", K(ret), K(task_record));
              } else {
                res.task_id_ = task_record.task_id_;
              }
            }
          }
        } else if (const_alter_table_arg.foreign_key_arg_list_.count() > 0) {
          if (1 != const_alter_table_arg.foreign_key_arg_list_.count()) {
            ret = OB_ERR_UNEXPECTED;
            LOG_WARN("error unexpected, can not add more than one foreign key at the same time", K(ret));
          } else {
            const ObCreateForeignKeyArg &foreign_key_arg = const_alter_table_arg.foreign_key_arg_list_.at(0);
            int64_t fk_id = OB_INVALID_ID;
            res.schema_version_ = new_table_schema.get_schema_version();
            if (const_alter_table_arg.foreign_key_checks_
                && foreign_key_arg.need_validate_data_
                && ((!foreign_key_arg.is_modify_fk_state_
                  && foreign_key_arg.validate_flag_)
                || (foreign_key_arg.is_modify_validate_flag_
                  && foreign_key_arg.validate_flag_))) {
              const ObIArray<ObForeignKeyInfo> &fk_infos = alter_table_schema.get_foreign_key_infos();
              const int64_t fk_cnt = fk_infos.count();
              const ObTableSchema *parent_table_schema = nullptr;
              ObDDLTaskRecord task_record;
              for (int64_t i = 0; OB_SUCC(ret) && i < fk_infos.count(); ++i) {
                const ObForeignKeyInfo fk_info = fk_infos.at(i);
                if (0 == foreign_key_arg.foreign_key_name_.compare(fk_info.foreign_key_name_)) {
                  fk_id = fk_info.foreign_key_id_;
                  if (fk_info.parent_table_id_ != fk_info.child_table_id_) {
                    // add fk parent table obj info for ddl task record
                    const_alter_table_arg.based_schema_object_infos_.reset();
                    if (OB_FAIL(schema_guard.get_table_schema(tenant_id, fk_info.parent_table_id_, parent_table_schema))) {
                      LOG_WARN("failed to get fk parent table schema", K(ret));
                    } else if (OB_ISNULL(parent_table_schema)) {
                      ret = OB_TABLE_NOT_EXIST;
                      LOG_WARN("parent table not exist", K(ret), K(fk_info));
                    } else if (OB_FAIL(const_alter_table_arg.based_schema_object_infos_.push_back(ObBasedSchemaObjectInfo(
                            parent_table_schema->get_table_id(), TABLE_SCHEMA, parent_table_schema->get_schema_version())))) {
                      LOG_WARN("failed to add fk info", K(ret));
                    }
                  }
                  break;
                }
              }
              if (OB_FAIL(ret)) {
              } else if (OB_INVALID_ID == fk_id) {
                ret = OB_ERR_UNEXPECTED;
                LOG_WARN("can not find foreign key", K(ret));
              } else {
                ObTableLockOwnerID owner_id;
                ObCreateDDLTaskParam param(new_table_schema.get_tenant_id(),
                                           ObDDLType::DDL_FOREIGN_KEY_CONSTRAINT,
                                           &new_table_schema,
                                           nullptr,
                                           fk_id,
                                           new_table_schema.get_schema_version(),
                                           0/*parallelism*/,
                                           const_alter_table_arg.consumer_group_id_,
                                           &alter_table_arg.allocator_,
                                           &const_alter_table_arg);
                if (OB_FAIL(GCTX.root_service_->get_ddl_scheduler().create_ddl_task(param, trans, task_record))) {
                  LOG_WARN("submit constraint task", K(ret));
                } else if (OB_FAIL(owner_id.convert_from_value(ObLockOwnerType::DEFAULT_OWNER_TYPE,
                                                               task_record.task_id_))) {
                  LOG_WARN("failed to get owner id", K(ret), K(task_record.task_id_));
                } else if (nullptr != parent_table_schema && OB_FAIL(ObDDLLock::lock_for_common_ddl(*parent_table_schema,
                                                                                                    owner_id,
                                                                                                    trans))) {
                  LOG_WARN("failed to lock online ddl lock", K(ret));
                } else if (OB_FAIL(ObDDLLock::lock_for_common_ddl(new_table_schema,
                                                                  owner_id,
                                                                  trans))) {
                  LOG_WARN("failed to lock online ddl lock", K(ret));
                } else if (OB_FAIL(ddl_tasks.push_back(task_record))) {
                  LOG_WARN("fail to push ddl task", K(ret), K(task_record));
                } else {
                  res.task_id_ = task_record.task_id_;
                }
              }
            }
          }
        }
        const bool is_commit = OB_SUCC(ret);
        if (trans.is_started()) {
          int temp_ret = OB_SUCCESS;
          if (OB_SUCCESS != (temp_ret = trans.end(is_commit))) {
            LOG_WARN("trans end failed", K(is_commit), K(temp_ret));
            ret = is_commit ? temp_ret : ret;
          }
        }

        DEBUG_SYNC(AFTER_CREATE_SPLIT_TASK);

        if (OB_SUCC(ret)) {
          int tmp_ret = OB_SUCCESS;
          if (OB_FAIL(publish_schema(tenant_id))) {
            LOG_WARN("publish_schema failed", K(ret));
          } else {
            for (int64_t i = 0; OB_SUCCESS == tmp_ret && i < ddl_tasks.count(); i++) {
              ObDDLTaskRecord &task_record = ddl_tasks.at(i);
              if (OB_TMP_FAIL(GCTX.root_service_->get_ddl_scheduler().schedule_ddl_task(task_record))) {
                LOG_WARN("fail to schedule ddl task", K(tmp_ret), K(task_record));
              }
            }
          }

          if (OB_FAIL(ret)) {
          } else if (OB_FAIL(delete_auto_increment_attribute(orig_table_schema, new_table_schema, alter_table_schema))) {
            LOG_WARN("fail to delete auto-incr attribute", K(ret), KPC(orig_table_schema), K(alter_table_schema));
          }
        }
      }
    }
    }
    }
    }
    }
    }
  }
  DEBUG_SYNC(AFTER_CREATE_SPLIT_TASK);
  return ret;
}

int ObDDLService::check_alter_column_group(const obrpc::ObAlterTableArg &alter_table_arg, ObDDLType &ddl_type) const
{
  int ret = OB_SUCCESS;
  if (OB_DDL_ADD_COLUMN_GROUP == alter_table_arg.alter_table_schema_.alter_type_ ||
      OB_DDL_DROP_COLUMN_GROUP == alter_table_arg.alter_table_schema_.alter_type_) {
    if (true == alter_table_arg.is_alter_column_group_delayed_) {
      ddl_type = ObDDLType::DDL_ALTER_COLUMN_GROUP_DELAYED;
    } else  {
      ddl_type = ObDDLType::DDL_ALTER_COLUMN_GROUP;
    }
    if (alter_table_arg.alter_table_schema_.get_column_group_count() <= 0) {
      ret = OB_INVALID_ARGUMENT;
      LOG_WARN("invalid argument, alter table arg don't have any column group when alter column group",
               K(ret), K(alter_table_arg.alter_table_schema_));
    }
  }
   return ret;
}


int ObDDLService::check_is_offline_ddl(ObAlterTableArg &alter_table_arg,
                                       const uint64_t tenant_data_version,
                                       ObDDLType &ddl_type,
                                       bool &ddl_need_retry_at_executor)
{
  int ret = OB_SUCCESS;
  ddl_type = ObDDLType::DDL_INVALID;
  bool is_oracle_mode = false;
  ObSchemaGetterGuard schema_guard;
  const ObTableSchema *orig_table_schema = nullptr;
  const AlterTableSchema &alter_table_schema = alter_table_arg.alter_table_schema_;
  uint64_t tenant_id = alter_table_schema.get_tenant_id();
  if (OB_FAIL(check_inner_stat())) {
    LOG_WARN("variable is not init", K(ret));
  } else if (OB_UNLIKELY(tenant_data_version <= 0)) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("invalid arguments", K(ret), K(tenant_data_version));
  } else if (OB_FAIL(schema_service_->get_tenant_schema_guard(tenant_id, schema_guard))) {
    LOG_WARN("fail to get schema guard", K(ret));
  } else if (OB_FAIL(get_and_check_table_schema(alter_table_arg,
      schema_guard, alter_table_schema, orig_table_schema))) {
    LOG_WARN("fail to get and check table schema", KR(ret));
  } else if (OB_FAIL(orig_table_schema->check_if_oracle_compat_mode(is_oracle_mode))) {
    LOG_WARN("fail to check is oracle mode", KR(ret), KPC(orig_table_schema));
  } else if (share::ObDDLTaskType::DELETE_COLUMN_FROM_SCHEMA == alter_table_arg.ddl_task_type_) {
    // alter table drop unused column[s].
    ObArray<uint64_t> unused_column_ids;
    if (OB_UNLIKELY(!is_oracle_mode)) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("drop column instant not under oracle mode", KR(ret), K(tenant_id),
        "table_id", orig_table_schema->get_table_id());
    } else if (OB_FAIL(orig_table_schema->get_unused_column_ids(unused_column_ids))) {
      LOG_WARN("get unused column ids failed", KR(ret), KPC(orig_table_schema));
    } else {
      ddl_type = unused_column_ids.empty() ? ObDDLType::DDL_NORMAL_TYPE : ObDDLType::DDL_TABLE_REDEFINITION;
    }
  } else {
    bool has_drop_and_add_index = false;
    char err_msg[number::ObNumber::MAX_PRINTABLE_SIZE] = {0};
    if (OB_SUCC(ret) && orig_table_schema->is_heap_organized_table() && tenant_data_version < DATA_VERSION_4_3_5_1) {
      ret = OB_NOT_SUPPORTED;
      LOG_WARN("unable to support heap table before data version 4.3.5.1", KR(ret), KPC(orig_table_schema));
    }
    if (OB_SUCC(ret) && alter_table_arg.is_alter_columns_
        && OB_FAIL(check_alter_table_column(alter_table_arg,
                                            *orig_table_schema,
                                            schema_guard,
                                            is_oracle_mode,
                                            tenant_data_version,
                                            ddl_type,
                                            ddl_need_retry_at_executor))) {
      LOG_WARN("fail to check alter table column", K(ret));
    }
    const ObDDLType alter_table_ddl_type = ddl_type;
    if (OB_SUCC(ret) && alter_table_arg.is_alter_indexs_
        && OB_FAIL(check_alter_table_index(alter_table_arg, *orig_table_schema,
                                           ddl_type,
                                           schema_guard,
                                           has_drop_and_add_index))) {
      LOG_WARN("fail to check alter table index", K(ret));
    }
    if (OB_SUCC(ret) && alter_table_arg.is_alter_partitions_
        && OB_FAIL(check_alter_table_partition(alter_table_arg,
                                               *orig_table_schema,
                                               is_oracle_mode,
                                               ddl_type))) {
      LOG_WARN("fail to check alter table partition", K(ret));
    }
    if (OB_SUCC(ret) && alter_table_arg.alter_auto_partition_attr_) {
      ObAlterAutoPartAttrOp alter_auto_table(*this);
      if (OB_FAIL(alter_auto_table.check_alter_table_partition_attr(alter_table_arg,
                                                                    *orig_table_schema,
                                                                    is_oracle_mode,
                                                                    ddl_type))) {
        LOG_WARN("fail to check alter table partition", K(ret));
      }
    }
    if (OB_SUCC(ret)) {
      if (OB_FAIL(check_alter_column_group(alter_table_arg, ddl_type))) {
        LOG_WARN("fail to check alter column gorup", K(ret), K(alter_table_arg.alter_table_schema_), K(ddl_type));
      }
    }

    if (OB_SUCC(ret) && alter_table_arg.alter_constraint_type_!= obrpc::ObAlterTableArg::CONSTRAINT_NO_OPERATION
        && OB_FAIL(check_alter_table_constraint(alter_table_arg, *orig_table_schema, ddl_type))) {
      LOG_WARN("fail to check alter table constraint", K(ret), K(alter_table_arg), K(ddl_type));
    }
    if (OB_SUCC(ret) && alter_table_arg.is_convert_to_character_
        && OB_FAIL(check_convert_to_character(alter_table_arg, *orig_table_schema, ddl_type))) {
      LOG_WARN("fail to check convert to character", K(ret));
    }
    if (OB_SUCC(ret) && alter_table_arg.foreign_key_arg_list_.count() > 0 && ddl_type == ObDDLType::DDL_INVALID) {
      ddl_type = ObDDLType::DDL_NORMAL_TYPE;
    }

    bool is_dec_lob_inrow_threshold = false;
    if (OB_SUCC(ret) && OB_FAIL(check_alter_lob_inrow_threshold(
          alter_table_arg, alter_table_schema, orig_table_schema,
          is_dec_lob_inrow_threshold, ddl_type))) {
      LOG_WARN("fail to check alter lob_inrow_threshold", K(ret));
    }

    if (OB_SUCC(ret)) {
      // add column instant except modify column with other ddl will get in offline
      if (ObDDLType::DDL_ADD_COLUMN_INSTANT == alter_table_ddl_type
            || ObDDLType::DDL_COMPOUND_INSTANT == alter_table_ddl_type) {
        if (alter_table_ddl_type != ddl_type
            // add foreign key with add column instant should be offline
            || alter_table_arg.foreign_key_arg_list_.count() > 0) {
          ddl_type = ObDDLType::DDL_TABLE_REDEFINITION;
          FLOG_INFO("add column instant with other ddl, rewrite ddl_type to DDL_TABLE_REDEFINITION",
                    K(alter_table_ddl_type), K(ddl_type));
        }
      }
    }

    if (OB_SUCC(ret) && obrpc::ObAlterTableArg::AlterAlgorithm::INSTANT == alter_table_arg.alter_algorithm_) {
      if (!is_oracle_mode
        && ObDDLType::DDL_DROP_COLUMN == ddl_type) {
        // mysql mode can not do drop column instant
        ret = OB_NOT_SUPPORTED;
        LOG_WARN("in mysql mode not supported to drop column instant", KR(ret), K(tenant_data_version));
      } else if (ObDDLType::DDL_DROP_COLUMN == ddl_type) {
        ddl_type = ObDDLType::DDL_DROP_COLUMN_INSTANT;
        if (OB_FAIL(check_can_drop_column_instant_(is_oracle_mode, tenant_data_version))) {
          LOG_WARN("fail to check can drop column instant", KR(ret), K(tenant_id), K(alter_table_arg));
        } else if (OB_FAIL(check_fk_related_table_ddl(*orig_table_schema, ddl_type))) {
          LOG_WARN("check whether the foreign key related table is executing ddl failed", KR(ret));
        }
      }
    }
    if (OB_SUCC(ret)
        && is_oracle_mode
        && ObDDLType::DDL_DROP_COLUMN_INSTANT != ddl_type
        && ObDDLType::DDL_DROP_COLUMN == alter_table_ddl_type) {
      bool is_column_group_store = false;
      if (OB_FAIL(ObCODDLUtil::need_column_group_store(*orig_table_schema, is_column_group_store))) {
        LOG_WARN("fail to check schema is column group store", KR(ret));
      } else if (is_column_group_store) {
        ddl_type = ObDDLType::DDL_TABLE_REDEFINITION;
      }
    }
    if (OB_SUCC(ret)
        && is_long_running_ddl(ddl_type)
        && ! is_dec_lob_inrow_threshold
        && (alter_table_arg.is_alter_options_
        || !alter_table_arg.alter_table_schema_.get_foreign_key_infos().empty())) {
      if (alter_table_arg.is_alter_options_) {// alter options
        (void)snprintf(err_msg, sizeof(err_msg), "%s and alter options in single statment", ddl_type_str(ddl_type));
      } else if (!alter_table_arg.alter_table_schema_.get_foreign_key_infos().empty()) {// alter foreign key
        (void)snprintf(err_msg, sizeof(err_msg), "%s and alter foreign key in single statment", ddl_type_str(ddl_type));
      }
      ret = OB_NOT_SUPPORTED;
      LOG_USER_ERROR(OB_NOT_SUPPORTED, err_msg);
    }
    if (OB_SUCC(ret) && is_long_running_ddl(ddl_type)) {
      if (orig_table_schema->is_primary_vp_table()) {
        ret = OB_NOT_SUPPORTED;
        (void)snprintf(err_msg, sizeof(err_msg), "%s with primary vp table",
                       ddl_type_str(ddl_type));
        LOG_USER_ERROR(OB_NOT_SUPPORTED, err_msg);
      }
      // offline ddl is allowed on table with trigger(enable/disable).
    }
    if (OB_SUCC(ret) && (is_long_running_ddl(ddl_type)
      || alter_table_arg.is_alter_indexs_
      || alter_table_arg.alter_constraint_type_!= obrpc::ObAlterTableArg::CONSTRAINT_NO_OPERATION)) {
      // For add fk operation, fk occurs in orig_table_schema after publish schema,
      // so the routine here can not refuse the add fk operation if related schema executing offline ddl.
      // Don't worry, add_table_foreign_keys in alter_table_in_trans will refuse it.
      if (OB_FAIL(check_fk_related_table_ddl(*orig_table_schema, ddl_type))) {
        LOG_WARN("check whether the foreign key related table is executing ddl failed", K(ret));
      }
    }
    if (OB_SUCC(ret) && is_double_table_long_running_ddl(ddl_type)) {
      bool has_index_operation = false;
      bool will_be_having_domain_index_operation = false;
      bool has_fts_or_multivalue_index = false;
      bool has_vec_index = false;
      bool is_adding_constraint = false;
      bool is_column_store = false;
      uint64_t table_id = alter_table_arg.alter_table_schema_.get_table_id();
      if (orig_table_schema->has_mlog_table()) {
        ret = OB_NOT_SUPPORTED;
        LOG_WARN("double table long running ddl on table with materialized view log is not supported",
                 KR(ret));
        LOG_USER_ERROR(OB_NOT_SUPPORTED,
                       "double table long running ddl on table with materialized view log is");
      } else if (orig_table_schema->table_referenced_by_fast_lsm_mv()) {
        ret = OB_NOT_SUPPORTED;
        LOG_WARN("double table long running ddl on table required by materialized view is not supported",
                 KR(ret));
        LOG_USER_ERROR(OB_NOT_SUPPORTED,
                       "double table long running ddl on table required by materialized view is");
      } else if (orig_table_schema->is_mlog_table()) {
        ret = OB_NOT_SUPPORTED;
        LOG_WARN("double table long running ddl on materialized view log is not supported", KR(ret));
        LOG_USER_ERROR(OB_NOT_SUPPORTED, "double table long running ddl on materialized view log is");
      } else if (OB_FAIL(check_has_index_operation(schema_guard,
                                          tenant_id,
                                          table_id,
                                          has_index_operation))) {
        LOG_WARN("check has index operation failed", K(ret));
      } else if (OB_FAIL(check_alter_domain_column_allowed(alter_table_arg, orig_table_schema))) {
        LOG_WARN("failed to check domain operate column constraint", K(ret));
      } else if (OB_FAIL(check_has_domain_index(schema_guard,
                                             tenant_id,
                                             table_id,
                                             has_fts_or_multivalue_index))) {
        LOG_WARN("check has fts index failed", K(ret));
      } else if (OB_FAIL(check_has_vec_domain_index(schema_guard,
                                             tenant_id,
                                             table_id,
                                             has_vec_index))) {
        LOG_WARN("fail to check has vec domain index", K(ret));
      } else if (OB_FAIL(check_will_be_having_domain_index_operation(alter_table_arg,
                                                                     will_be_having_domain_index_operation))) {
        LOG_WARN("check will be having domain index operation failed", K(ret));
      } else if (OB_FAIL(check_is_adding_constraint(tenant_id, table_id, is_adding_constraint))) {
        LOG_WARN("failed to call check_is_adding_constraint", K(ret));
      } else if (has_index_operation) {
        ret = OB_NOT_SUPPORTED;
        LOG_USER_ERROR(OB_NOT_SUPPORTED, "The DDL cannot be run concurrently with creating index.");
      } else if (will_be_having_domain_index_operation) {
        ret = OB_NOT_SUPPORTED;
        LOG_USER_ERROR(OB_NOT_SUPPORTED, "The DDL cannot be run, as creating/dropping fulltext/multivalue/vector index.");
      } else if (has_fts_or_multivalue_index || has_vec_index) {
        ret = OB_NOT_SUPPORTED;
        LOG_USER_ERROR(OB_NOT_SUPPORTED, "Run this DDL operation on table with fulltext/multivalue/vector index.");
      } else if (is_adding_constraint) {
        ret = OB_NOT_SUPPORTED;
        LOG_USER_ERROR(OB_NOT_SUPPORTED, "The DDL cannot be run concurrently with adding constraint.");
      }
    }
    if (OB_SUCC(ret) && DDL_NORMAL_TYPE == ddl_type && has_drop_and_add_index) {
      omt::ObTenantConfigGuard tenant_config(TENANT_CONF(tenant_id));
      if (OB_UNLIKELY(!tenant_config.is_valid())) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("tenant config is invalid", K(ret), K(tenant_id));
      } else if (!tenant_config->_enable_drop_and_add_index) {
        ret = OB_OP_NOT_ALLOW;
        LOG_USER_ERROR(OB_NOT_SUPPORTED, "Dropping and adding indexes at the same time is a high-risk operation, which is");
      }
    }
  }
  return ret;
}

int ObDDLService::check_alter_domain_column_allowed(
  obrpc::ObAlterTableArg &alter_table_arg,
  const ObTableSchema *orig_table_schema)
{
  int ret = OB_SUCCESS;
  const AlterTableSchema &alter_table_schema = alter_table_arg.alter_table_schema_;
  AlterColumnSchema *alter_column_schema = NULL;

  ObTableSchema::const_column_iterator it_begin = alter_table_schema.column_begin();
  ObTableSchema::const_column_iterator it_end = alter_table_schema.column_end();
  for (; OB_SUCC(ret) && it_begin != it_end; it_begin++) {
    if (OB_ISNULL(alter_column_schema = static_cast<AlterColumnSchema *>(*it_begin))) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("*it_begin is NULL", K(ret));
    } else {
      const ObString &orig_column_name = alter_column_schema->get_origin_column_name();
      const ObColumnSchemaV2 *orig_column_schema = orig_table_schema->get_column_schema(orig_column_name);
      const ObSchemaOperationType op_type = alter_column_schema->alter_type_;
      if (op_type == OB_DDL_DROP_COLUMN &&
          OB_NOT_NULL(orig_column_schema) &&
          orig_column_schema->is_multivalue_generated_array_column()) {
        ret = OB_NOT_SUPPORTED;
                LOG_USER_ERROR(OB_NOT_SUPPORTED, "user direct multivalue index generated budy column");
      }
    }
  }

  return ret;
}

int ObDDLService::check_has_domain_index(
    ObSchemaGetterGuard &schema_guard,
    const uint64_t tenant_id,
    const uint64_t data_table_id,
    bool &domain_index_exist)
{
  int ret = OB_SUCCESS;
  domain_index_exist = false;
  ObRootService *root_service = GCTX.root_service_;
  const ObTableSchema *table_schema = nullptr;
  if (OB_ISNULL(root_service)) {
    ret = OB_ERR_SYS;
    LOG_WARN("error sys, root service must not be nullptr", K(ret));
  } else if (OB_FAIL(root_service->get_ddl_service().get_tenant_schema_guard_with_version_in_inner_table(tenant_id, schema_guard))) {
  } else if (OB_FAIL(schema_guard.get_table_schema(tenant_id, data_table_id, table_schema))) {
    LOG_WARN("get table schema failed", K(ret), K(tenant_id), K(data_table_id));
  } else if (OB_ISNULL(table_schema)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("error unexpected, table schema must not be nullptr", K(ret), K(data_table_id));
  } else {
    const common::ObIArray<ObAuxTableMetaInfo> &index_infos = table_schema->get_simple_index_infos();
    if (index_infos.count() > 0) {
      // if there is indexes in new tables, if so, the indexes is already rebuilt in new table
      for (int64_t i = 0; OB_SUCC(ret) && i < index_infos.count(); ++i) {
        if (share::schema::is_doc_rowkey_aux(index_infos.at(i).index_type_)) {
          domain_index_exist = true;
          break;
        }
      }
    }
  }
  return ret;
}

int ObDDLService::check_has_vec_domain_index(
    ObSchemaGetterGuard &schema_guard,
    const uint64_t tenant_id,
    const uint64_t data_table_id,
    bool &domain_index_exist)
{
  int ret = OB_SUCCESS;
  domain_index_exist = false;
  ObRootService *root_service = GCTX.root_service_;
  const ObTableSchema *table_schema = nullptr;
  if (OB_ISNULL(root_service)) {
    ret = OB_ERR_SYS;
    LOG_WARN("error sys, root service must not be nullptr", K(ret));
  } else if (OB_FAIL(root_service->get_ddl_service().get_tenant_schema_guard_with_version_in_inner_table(tenant_id, schema_guard))) {
  } else if (OB_FAIL(schema_guard.get_table_schema(tenant_id, data_table_id, table_schema))) {
    LOG_WARN("get table schema failed", K(ret), K(tenant_id), K(data_table_id));
  } else if (OB_ISNULL(table_schema)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("error unexpected, table schema must not be nullptr", K(ret), K(data_table_id));
  } else {
    const common::ObIArray<ObAuxTableMetaInfo> &index_infos = table_schema->get_simple_index_infos();
    if (index_infos.count() > 0) {
      // if there is indexes in new tables, if so, the indexes is already rebuilt in new table
      for (int64_t i = 0; OB_SUCC(ret) && i < index_infos.count(); ++i) {
        if (share::schema::is_vec_vid_rowkey_type(index_infos.at(i).index_type_) ||
            share::schema::is_vec_rowkey_vid_type(index_infos.at(i).index_type_) ||
            share::schema::is_vec_domain_index(index_infos.at(i).index_type_)) {
          domain_index_exist = true;
          break;
        }
      }
    }
  }
  return ret;
}

int ObDDLService::check_will_be_having_domain_index_operation(
    const obrpc::ObAlterTableArg &alter_table_arg,
    bool &will_be_having_domain_index_operation/*false*/)
{
  int ret = OB_SUCCESS;
  will_be_having_domain_index_operation = false;
  const ObSArray<ObIndexArg *> &index_arg_list = alter_table_arg.index_arg_list_;
  for (int64_t i = 0; OB_SUCC(ret) && i < index_arg_list.size(); ++i) {
      ObIndexArg *index_arg = index_arg_list.at(i);
      if (OB_ISNULL(index_arg)) {
        ret = OB_INVALID_ARGUMENT;
        LOG_WARN("index arg should not be null", K(ret));
      } else {
        ObCreateIndexArg *create_index_arg = static_cast<ObCreateIndexArg *>(index_arg);
        if (share::schema::is_fts_or_multivalue_index(create_index_arg->index_type_) ||
            share::schema::is_vec_index(create_index_arg->index_type_)) {
          will_be_having_domain_index_operation = true;
          break;
        }
      }
  }
  return ret;
}

int ObDDLService::check_is_oracle_mode_add_column_not_null_ddl(const obrpc::ObAlterTableArg &alter_table_arg,
                                                               ObSchemaGetterGuard &schema_guard,
                                                               bool &is_oracle_mode_add_column_not_null_ddl,
                                                               bool &is_default_value_null)
{
  int ret = OB_SUCCESS;
  is_oracle_mode_add_column_not_null_ddl = false;
  is_default_value_null = false;
  bool is_oracle_mode = false;
  const AlterTableSchema &alter_table_schema = alter_table_arg.alter_table_schema_;
  const ObTableSchema *table_schema = nullptr;
  if (OB_FAIL(check_inner_stat())) {
    LOG_WARN("variable is not init", K(ret));
  } else if (OB_FAIL(get_and_check_table_schema(alter_table_arg,
                                                schema_guard,
                                                alter_table_schema,
                                                table_schema))) {
    LOG_WARN("fail to get and check table schema", K(ret));
  } else if (OB_FAIL(table_schema->check_if_oracle_compat_mode(is_oracle_mode))) {
    LOG_WARN("fail to check is oracle mode", KR(ret), KPC(table_schema));
  } else if (is_oracle_mode
             && alter_table_arg.is_alter_columns_
             && !alter_table_arg.is_alter_indexs_
             && !alter_table_arg.is_alter_partitions_
             && !alter_table_arg.is_convert_to_character_
             && obrpc::ObAlterTableArg::ADD_CONSTRAINT == alter_table_arg.alter_constraint_type_) {
    is_oracle_mode_add_column_not_null_ddl = true;
    ObTableSchema::const_column_iterator it_begin = alter_table_schema.column_begin();
    ObTableSchema::const_column_iterator it_end = alter_table_schema.column_end();
    AlterColumnSchema *alter_column_schema = nullptr;
    for(; OB_SUCC(ret) && is_oracle_mode_add_column_not_null_ddl && it_begin != it_end; it_begin++) {
      if (OB_ISNULL(*it_begin)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("*it_begin is NULL", K(ret));
      } else {
        alter_column_schema = static_cast<AlterColumnSchema *>(*it_begin);
        if (OB_DDL_ADD_COLUMN == alter_column_schema->alter_type_
            && alter_column_schema->get_orig_default_value().is_null()
            && !alter_column_schema->is_identity_column()
            && !alter_column_schema->is_xmltype()
            && !alter_column_schema->is_first_
            && !alter_column_schema->get_next_column_name()
            && !alter_column_schema->get_prev_column_name()
            && !is_lob_storage(alter_column_schema->get_data_type())) {
          // MDS can not register more than 1 buffer ctx in the same trans(RW defensive and create lob),
          // so that add not null lob will not hit the fast path.
        } else {
          is_oracle_mode_add_column_not_null_ddl = false;
        }
        // Check is not null column default value null
        if (!is_default_value_null && alter_column_schema->has_not_null_constraint()) {
          const ObObj &cur_default_value = alter_column_schema->get_cur_default_value();
          is_default_value_null = (cur_default_value.is_null_oracle() ||
            (cur_default_value.is_character_type()
              && (0 == cur_default_value.get_string().case_compare(N_NULL)
                || 0 == cur_default_value.get_string().case_compare("''"))));
        }
      }
    }
  } else {
    is_oracle_mode_add_column_not_null_ddl = false;
  }
  return ret;
}

// check whether there is index operation, including add index and drop index.
int ObDDLService::check_has_index_operation(ObSchemaGetterGuard &schema_guard,
                                          const uint64_t tenant_id,
                                          const uint64_t table_id,
                                          bool &has_index_operation)
{
  int ret = OB_SUCCESS;
  has_index_operation = false;
  // 1. get table schema
  const ObTableSchema *orig_table = nullptr;
  ObSEArray<ObAuxTableMetaInfo, 16> index_infos;

  if (OB_UNLIKELY(tenant_id == OB_INVALID_ID || table_id == OB_INVALID_ID)) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("invalid argument", K(ret), K(tenant_id), K(table_id));
  } else if (OB_FAIL(schema_guard.get_table_schema(tenant_id, table_id, orig_table))) {
    LOG_WARN("get table schema failed", K(ret));
  } else if (OB_ISNULL(orig_table)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("invalid table id", "table_id", table_id);
  } else if (OB_FAIL(orig_table->get_simple_index_infos(index_infos))) {
    LOG_WARN("get simple_index_infos failed", K(ret));
  } else {
    for (int64_t i = 0; OB_SUCC(ret) && i < index_infos.count(); i++) {
      // 2. get all index schemas
      const ObTableSchema *index_schema = NULL;
      if (OB_FAIL(schema_guard.get_table_schema(tenant_id,
                                                index_infos.at(i).table_id_,
                                                index_schema))) {
        LOG_WARN("get index schema failed", K(ret), K(index_infos.at(i).table_id_));
      } else if (OB_ISNULL(index_schema)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("invalid index table id", "index_table_id", index_infos.at(i).table_id_);
      } else if (index_schema->is_unavailable_index()) {
        // 3. check if index is still constructing
        has_index_operation = true;
        break;
      }
    }
  }
  return ret;
}

// check if is adding check constraint, foreign key, not null constraint
int ObDDLService::check_is_adding_constraint(const uint64_t tenant_id, const uint64_t table_id, bool &is_building)
{
  ObArenaAllocator allocator(lib::ObLabel("DdlTasRecord"));
  return ObDDLTaskRecordOperator::check_is_adding_constraint(sql_proxy_, allocator, tenant_id, table_id, is_building);
}

// check whether the foreign key related table is executing specifing long-running ddl.
// And ddl should be refused if the foreign key related table is executing above ddl.
int ObDDLService::check_fk_related_table_ddl(
    const share::schema::ObTableSchema &data_table_schema,
    const share::ObDDLType &ddl_type)
{
  int ret = OB_SUCCESS;
  const uint64_t tenant_id = data_table_schema.get_tenant_id();
  ObSchemaGetterGuard schema_guard;
  if (OB_UNLIKELY(OB_INVALID_ID == tenant_id || share::ObDDLType::DDL_INVALID == ddl_type)) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("invalid arg", K(ret), K(tenant_id), K(ddl_type));
  } else if (OB_FAIL(check_inner_stat())) {
    LOG_WARN("variable is not init");
  } else if (OB_FAIL(schema_service_->get_tenant_schema_guard(tenant_id, schema_guard))) {
    LOG_WARN("get schema guard failed", K(ret));
  } else {
    const ObIArray<ObForeignKeyInfo> &foreign_key_infos = data_table_schema.get_foreign_key_infos();
    const ObCheckExistedDDLMode check_mode = is_double_table_long_running_ddl(ddl_type) ?
        ObCheckExistedDDLMode::ALL_LONG_RUNNING_DDL : ObCheckExistedDDLMode::DOUBLE_TABLE_RUNNING_DDL;
    for (int64_t i = 0; OB_SUCC(ret) && i < foreign_key_infos.count(); i++) {
      const ObForeignKeyInfo &foreign_key_info = foreign_key_infos.at(i);
      const uint64_t related_table_id = data_table_schema.get_table_id() == foreign_key_info.parent_table_id_
          ? foreign_key_info.child_table_id_
          : foreign_key_info.parent_table_id_;
      bool has_long_running_ddl = false;
      const ObTableSchema *related_schema = nullptr;
      if (foreign_key_info.is_parent_table_mock_
        || data_table_schema.get_table_id() == related_table_id) {
        // mock table and self reference foreign key table, no need to check.
      } else if (OB_FAIL(schema_guard.get_table_schema(tenant_id, related_table_id, related_schema))) {
        LOG_WARN("get schema failed", K(ret), K(tenant_id), K(related_table_id));
      } else if (OB_ISNULL(related_schema)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("unexpected error, related schema is nullptr", K(ret), K(related_table_id), K(foreign_key_info));
      } else if (!related_schema->check_can_do_ddl()) {
        ret = OB_OP_NOT_ALLOW;
        LOG_USER_ERROR(OB_OP_NOT_ALLOW, "execute ddl while foreign key related table is executing long running ddl");
      } else if (OB_FAIL(ObDDLTaskRecordOperator::check_has_long_running_ddl(sql_proxy_,
                                                                            tenant_id,
                                                                            related_table_id,
                                                                            check_mode,
                                                                            has_long_running_ddl))) {
        LOG_WARN("check has long running ddl failed", K(ret), K(tenant_id), K(related_table_id));
      } else if (has_long_running_ddl) {
        ret = OB_OP_NOT_ALLOW;
        LOG_WARN("foreign key related table is executing offline ddl", K(ret), K(check_mode), K(tenant_id),
          "table_id", data_table_schema.get_table_id(), K(related_table_id));
        LOG_USER_ERROR(OB_OP_NOT_ALLOW, "execute ddl while there are some long running ddl on foreign key related table");
      }
    }
  }
  return ret;
}

int ObDDLService::check_can_bind_tablets(const share::ObDDLType ddl_type,
                                         bool &bind_tablets)
{
  int ret = OB_SUCCESS;
  switch (ddl_type) {
  case DDL_ADD_COLUMN_OFFLINE:
  case DDL_DROP_COLUMN:
  case DDL_COLUMN_REDEFINITION: {
    bind_tablets = true;
    break;
  }
  default: {
    bind_tablets = false;
  }
  }
  return ret;
}

// check if the ddl contains primary key operation.
int ObDDLService::check_ddl_with_primary_key_operation(
    const obrpc::ObAlterTableArg &alter_table_arg,
    bool &with_primary_key_operation)
{
  int ret = OB_SUCCESS;
  with_primary_key_operation = false;
  const ObSArray<ObIndexArg *> &index_arg_list = alter_table_arg.index_arg_list_;
  for (int64_t i = 0; OB_SUCC(ret) && !with_primary_key_operation && i < index_arg_list.size(); ++i) {
    ObIndexArg *index_arg = const_cast<ObIndexArg *>(index_arg_list.at(i));
    if (OB_ISNULL(index_arg)) {
      ret = OB_INVALID_ARGUMENT;
      LOG_WARN("index arg should not be null", K(ret));
    } else {
      const ObIndexArg::IndexActionType type = index_arg->index_action_type_;
      with_primary_key_operation = ObIndexArg::DROP_PRIMARY_KEY == type
                                || ObIndexArg::ADD_PRIMARY_KEY == type
                                || ObIndexArg::ALTER_PRIMARY_KEY == type;
    }
  }
  return ret;
}

int ObDDLService::reorder_column_after_add_column_instant_(const ObTableSchema &orig_table_schema,
                                                           ObTableSchema &new_table_schema)
{
  int ret = OB_SUCCESS;
  ObArray<uint64_t> add_instant_column_ids;
  bool add_column_instant = false;
  if (OB_FAIL(check_inner_stat())) {
    LOG_WARN("variable is not init", KR(ret));
  } else if (OB_FAIL(orig_table_schema.has_add_column_instant(add_column_instant))) {
    LOG_WARN("failed to get add instant column ids", KR(ret), K(orig_table_schema));
  } else if (add_column_instant) {
    if (OB_FAIL(redistribute_column_ids(new_table_schema))) {
      LOG_WARN("redistribute column id failed", KR(ret));
    }
  }
  return ret;
}

int ObDDLService::do_offline_ddl_in_trans(obrpc::ObAlterTableArg &alter_table_arg,
                                          const uint64_t tenant_data_version,
                                          obrpc::ObAlterTableRes &res)
{
  int ret = OB_SUCCESS;
  bool bind_tablets = false;
  ObSchemaGetterGuard schema_guard;
  AlterTableSchema &alter_table_schema = alter_table_arg.alter_table_schema_;
  uint64_t tenant_id = alter_table_schema.get_tenant_id();
  const ObDDLType ddl_type = res.ddl_type_;
  ObRootService *root_service = GCTX.root_service_;
  bool need_redistribute_column_id = false;
  if (OB_UNLIKELY(DDL_INVALID == ddl_type || tenant_data_version <= 0)) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("unexpected ddl type", K(ret), K(ddl_type), K(alter_table_arg), K(tenant_data_version));
  } else if (OB_ISNULL(root_service)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("error unexpected, root service must not be nullptr", K(ret));
  } else if (OB_FAIL(check_inner_stat())) {
    LOG_WARN("variable is not init", K(ret));
  } else if (OB_FAIL(get_tenant_schema_guard_with_version_in_inner_table(tenant_id, schema_guard))) {
    LOG_WARN("fail to get schema guard with version in inner table", K(ret), K(tenant_id));
  } else if (OB_FAIL(check_can_bind_tablets(ddl_type, bind_tablets))) {
    LOG_WARN("failed to check can bind tablets", K(ret), K(ddl_type));
  } else {
    ObDDLOperator ddl_operator(*schema_service_, *sql_proxy_);
    ObTableSchema new_table_schema;
    const ObTableSchema *orig_table_schema = NULL;
    if (OB_FAIL(get_and_check_table_schema(alter_table_arg,
                                           schema_guard,
                                           alter_table_schema,
                                           orig_table_schema))) {
      LOG_WARN("fail to get and check table schema", K(ret));
    } else if (OB_FAIL(new_table_schema.assign(*orig_table_schema))) {
      LOG_WARN("fail to assign schema", K(ret));
    } else if (OB_FAIL(ObSchemaUtils::mock_default_cg(orig_table_schema->get_tenant_id(), new_table_schema))) {
      LOG_WARN("fail to mock default cg", K(ret), K(orig_table_schema), K(new_table_schema));
    } else if (OB_FAIL(delete_unused_columns_and_redistribute_schema(*orig_table_schema,
        false/*need_remove_orig_table_unused_column, no need to remove unused columns from inner table since it is a new table.*/,
        true/*need_redistribute_column_id, to redistribute columns id and index tables will be rebuilt with new column_id*/,
        nullptr/*ddl_operator*/, nullptr/*trans*/, new_table_schema))) {
      LOG_WARN("remove all unused columns internally failed", KR(ret));
    } else {
      ObDDLSQLTransaction trans(schema_service_);
      ObDDLTaskRecord task_record;
      int64_t task_id = 0;
      int64_t refreshed_schema_version = 0;
      bool with_primary_key_operation = false;
      ObTableLockOwnerID owner_id;
      if (OB_FAIL(schema_guard.get_schema_version(tenant_id, refreshed_schema_version))) {
        LOG_WARN("failed to get tenant schema version", KR(ret), K(tenant_id));
      } else if (OB_FAIL(trans.start(sql_proxy_, tenant_id, refreshed_schema_version))) {
        LOG_WARN("start transaction failed", KR(ret), K(tenant_id), K(refreshed_schema_version));
      } else if (OB_FAIL(ObDDLTask::fetch_new_task_id(root_service->get_sql_proxy(), tenant_id, task_id))) {
        LOG_WARN("fetch new task id failed", K(ret));
      } else if (OB_FAIL(owner_id.convert_from_value(ObLockOwnerType::DEFAULT_OWNER_TYPE,
                                                     task_id))) {
        LOG_WARN("failed to get owner id", K(ret), K(task_id));
      } else if (OB_FAIL(ObDDLLock::lock_for_offline_ddl(*orig_table_schema,
                                                         nullptr,
                                                         owner_id,
                                                         trans))) {
        LOG_WARN("failed to lock ddl lock", K(ret));
      }
      // TODO yiren, refactor it, create user hidden table after alter index/column/part/cst...
      if (OB_FAIL(ret)) {
      } else if (need_modify_dep_obj_status(alter_table_arg)
                 && OB_FAIL(ObDependencyInfo::modify_dep_obj_status(trans, tenant_id, orig_table_schema->get_table_id(),
                                                      ddl_operator, *schema_service_))) {
        LOG_WARN("failed to modify obj status", K(ret));
      } else if (OB_FAIL(check_ddl_with_primary_key_operation(alter_table_arg,
                                                              with_primary_key_operation))) {
        LOG_WARN("check ddl with primary key operation failed", K(ret));
      } else if (with_primary_key_operation) {
        if (OB_FAIL(check_restore_point_allow(tenant_id, *orig_table_schema))) {
          LOG_WARN("check restore point allow failed,", K(ret), K(tenant_id), KPC(orig_table_schema));
        } else if (OB_FAIL(alter_table_primary_key(alter_table_arg,
                            *orig_table_schema,
                            new_table_schema,
                            schema_guard,
                            ddl_operator,
                            trans,
                            alter_table_arg.allocator_,
                            tenant_data_version))) {
          LOG_WARN("failed to alter table primary key", K(ret));
        }
      }
      if (OB_SUCC(ret) && ObDDLType::DDL_TABLE_REDEFINITION == ddl_type) {
        bool need_modify_notnull_validate = false;
        bool is_add_not_null_col = false;
        if (OB_FAIL(need_modify_not_null_constraint_validate(
            alter_table_arg, is_add_not_null_col, need_modify_notnull_validate))) {
          LOG_WARN("check need modify not null constraint validate failed", K(ret));
        } else if (need_modify_notnull_validate) {
          ObConstraint *cst = *alter_table_schema.constraint_begin_for_non_const_iter();
          const uint64_t col_id = *(cst->cst_col_begin());
          ObColumnSchemaV2 *col_schema = NULL;
          for (int64_t i = 0; OB_SUCC(ret) && i < alter_table_schema.get_column_count(); i++) {
            if (alter_table_schema.get_column_schema_by_idx(i)->get_column_id() == col_id) {
              col_schema = alter_table_schema.get_column_schema_by_idx(i);
            }
          }
          if (OB_ISNULL(col_schema)) {
            ret = OB_ERR_UNEXPECTED;
            LOG_WARN("column schema not found", K(ret), K(alter_table_arg));
          } else {
            col_schema->del_column_flag(NOT_NULL_VALIDATE_FLAG);
          }
        }
      }
      if (OB_SUCC(ret) && (alter_table_arg.is_alter_columns_ ||
          share::ObDDLTaskType::DELETE_COLUMN_FROM_SCHEMA == alter_table_arg.ddl_task_type_)) {
        if (ObDDLType::DDL_MODIFY_AUTO_INCREMENT == ddl_type) {
          new_table_schema.set_in_offline_ddl_white_list(true);
          new_table_schema.set_table_state_flag(ObTableStateFlag::TABLE_STATE_OFFLINE_DDL);
        } else {
          // For ddl type relying on dag to complement data, we need redefine it as table redifition to support column store.
          // Column store replica need take the same strategy to fill column group.
          // However, if there is no cs replica (or ls no leader) when check_alter_table_column, but cs replica arises when create hidden table,
          // cs replica should be ignored to prevent using complement dag to fill column group data.
          const bool ignore_cs_replica = is_complement_data_relying_on_dag(ddl_type);
          if (share::ObDDLTaskType::DELETE_COLUMN_FROM_SCHEMA == alter_table_arg.ddl_task_type_) {
            if (OB_FAIL(check_enable_sys_table_ddl(*orig_table_schema, OB_DDL_DROP_COLUMN))) {
              LOG_WARN("fail to check enable sys table ddl", KR(ret), K(orig_table_schema));
            }
          } else if (OB_FAIL(check_can_alter_column(tenant_id,
                                                    alter_table_schema,
                                                    *orig_table_schema))) {
            LOG_WARN("fail to can alter column", KR(ret), K(alter_table_arg));
          } else if (OB_FAIL(gen_alter_column_new_table_schema_offline(
                              *orig_table_schema,
                              alter_table_schema,
                              new_table_schema,
                              alter_table_arg,
                              schema_guard,
                              need_redistribute_column_id))) {
            LOG_WARN("failed to alter table column!", KR(ret), K(*orig_table_schema), K(new_table_schema));
          }
          if (FAILEDx(adjust_cg_for_offline(new_table_schema))) {
            LOG_WARN("fail to adjust cg  after alter column", KR(ret));
          } else if (FAILEDx(create_user_hidden_table(*orig_table_schema,
                                                      new_table_schema,
                                                      &alter_table_arg.sequence_ddl_arg_,
                                                      bind_tablets,
                                                      schema_guard,
                                                      schema_guard,
                                                      ddl_operator,
                                                      trans,
                                                      alter_table_arg.allocator_,
                                                      tenant_data_version,
                                                      ObString("") /*index_name*/,
                                                      ignore_cs_replica))) {
            LOG_WARN("fail to create user hidden table", KR(ret));
          }
        }
      }
      if (OB_SUCC(ret) && alter_table_arg.is_alter_partitions_) {
        if (OB_FAIL(check_restore_point_allow(tenant_id, *orig_table_schema))) {
          LOG_WARN("check restore point allow failed,", K(ret), K(tenant_id),
          K(orig_table_schema->get_table_id()));
        } else if (OB_FAIL(alter_table_partition_by(alter_table_arg,
                                                    *orig_table_schema,
                                                    new_table_schema,
                                                    schema_guard,
                                                    ddl_operator,
                                                    trans,
                                                    tenant_data_version))) {
          LOG_WARN("failed to alter table partition by", K(ret));
        }
      }
      if (OB_SUCC(ret) && alter_table_arg.alter_auto_partition_attr_) {
        const bool bind_tablets = false;
        ObAlterAutoPartAttrOp alter_auto_part(*this);
        // In the alter table partition by range(xxx) size(xxx) (partitions...) case,
        // it is necessary to modify attributes related to automatic partitioning here.
        if (OB_FAIL(check_restore_point_allow(tenant_id, *orig_table_schema))) {
          LOG_WARN("check restore point allow failed,", K(ret), K(tenant_id),
          K(orig_table_schema->get_table_id()));
        } else if (OB_FAIL(alter_auto_part.alter_table_partition_attr(
            alter_table_arg, *orig_table_schema, new_table_schema))) {
          LOG_WARN("failed to alter table partition by", K(ret));
        } else if (OB_FAIL(create_user_hidden_table(*orig_table_schema,
                                                    new_table_schema,
                                                    &alter_table_arg.sequence_ddl_arg_,
                                                    bind_tablets,
                                                    schema_guard,
                                                    schema_guard,
                                                    ddl_operator,
                                                    trans,
                                                    alter_table_arg.allocator_,
                                                    tenant_data_version))) {
          LOG_WARN("fail to create user_hidden table", K(ret));
        }
      }
      // table constraints
      if (OB_SUCC(ret) &&  alter_table_arg.alter_constraint_type_ !=
          obrpc::ObAlterTableArg::CONSTRAINT_NO_OPERATION) {
        if (ObDDLType::DDL_TABLE_REDEFINITION == ddl_type
            || ObDDLType::DDL_MODIFY_COLUMN == ddl_type) {
          HEAP_VAR(AlterTableSchema, tmp_alter_table_schema) {
            if (OB_FAIL(tmp_alter_table_schema.assign(alter_table_schema))) {
              LOG_WARN("failed to assign", K(ret));
            } else if (OB_FAIL(refill_column_id_array_for_constraint(alter_table_arg.alter_constraint_type_, new_table_schema, tmp_alter_table_schema))) {
              LOG_WARN("failed to refill columns id", K(ret));
            } else if (OB_FAIL(alter_table_constraints(
                alter_table_arg.alter_constraint_type_,
                schema_guard,
                *orig_table_schema,
                tmp_alter_table_schema,
                new_table_schema,
                ddl_operator,
                trans))) {
              LOG_WARN("alter table constraints failed", K(ret));
            }
          }
        }
      }
      if (OB_SUCC(ret) && alter_table_arg.is_convert_to_character_) {
        if (OB_FAIL(convert_to_character(alter_table_arg,
                                         *orig_table_schema,
                                         new_table_schema,
                                         schema_guard,
                                         ddl_operator,
                                         trans,
                                         tenant_data_version))) {
          LOG_WARN("failed to convert to character", K(ret));
        }
      }

      if (OB_SUCC(ret) && ddl_type == ObDDLType::DDL_ALTER_COLUMN_GROUP) {
        if (OB_FAIL(alter_column_group(alter_table_arg,
                                       *orig_table_schema,
                                       new_table_schema,
                                       schema_guard,
                                       ddl_operator,
                                       trans))) {
          LOG_WARN("failed to alter table column group", K(ret));
        } else if (OB_FAIL(create_user_hidden_table(*orig_table_schema,
                                                    new_table_schema,
                                                    &alter_table_arg.sequence_ddl_arg_,
                                                    bind_tablets,
                                                    schema_guard,
                                                    schema_guard,
                                                    ddl_operator,
                                                    trans,
                                                    alter_table_arg.allocator_,
                                                    tenant_data_version))) {
          LOG_WARN("fail to create user_hidden table", K(ret), KPC(orig_table_schema), K(new_table_schema));
        }
      }

      if (OB_SUCC(ret) && ObDDLType::DDL_TABLE_REDEFINITION == ddl_type
          && is_dec_table_lob_inrow_threshold(alter_table_arg, alter_table_schema, orig_table_schema)) {
        new_table_schema.set_lob_inrow_threshold(alter_table_schema.get_lob_inrow_threshold());
        OZ (create_user_hidden_table(*orig_table_schema,
                                    new_table_schema,
                                    &alter_table_arg.sequence_ddl_arg_,
                                    false/*bind_tablets*/,
                                    schema_guard,
                                    schema_guard,
                                    ddl_operator,
                                    trans,
                                    alter_table_arg.allocator_,
                                    tenant_data_version));
      }
      if (OB_SUCC(ret)) {
        if (OB_FAIL(new_table_schema.sort_column_array_by_column_id())) {
          LOG_WARN("failed to sort column", K(ret), K(new_table_schema));
        }
      }
      if (OB_SUCC(ret)) {
        ObSchemaOperationType operation_type = OB_DDL_ALTER_TABLE;
        if (OB_FAIL(ddl_operator.update_table_attribute(new_table_schema,
                                                        trans,
                                                        operation_type))) {
          LOG_WARN("failed to update data table schema attribute", K(ret));
        } else {
          alter_table_arg.alter_table_schema_.set_schema_version(new_table_schema.get_schema_version());
          if (ObDDLType::DDL_MODIFY_AUTO_INCREMENT != ddl_type) {
            alter_table_arg.alter_table_schema_.set_association_table_id(new_table_schema.get_table_id());
          }
        }
      }

      // submit async build index task
      if (OB_FAIL(ret)) {
      } else if (is_double_table_long_running_ddl(ddl_type)) {
        bool has_conflict_ddl = false;
        ObCreateDDLTaskParam param(tenant_id,
                                   ddl_type,
                                   orig_table_schema,
                                   &new_table_schema,
                                   0/*object_id*/,
                                   new_table_schema.get_schema_version(),
                                   alter_table_arg.parallelism_,
                                   alter_table_arg.consumer_group_id_,
                                   &alter_table_arg.allocator_,
                                   &alter_table_arg,
                                   0/*parent_task_id*/,
                                   task_id,
                                   res.ddl_need_retry_at_executor_);
        param.tenant_data_version_ = tenant_data_version;
        if (orig_table_schema->is_external_table()) {
          ret = OB_OP_NOT_ALLOW;
          char err_msg[OB_MAX_ERROR_MSG_LEN] = {0};
          (void)snprintf(err_msg, sizeof(err_msg), "%s on external table is", ddl_type_str(ddl_type));
          LOG_WARN("double table long running ddl on temporary table is disallowed", K(ret), K(ddl_type));
          LOG_USER_ERROR(OB_OP_NOT_ALLOW, err_msg);
        } else if (OB_FAIL(ObDDLTaskRecordOperator::check_has_conflict_ddl(sql_proxy_, tenant_id, orig_table_schema->get_table_id(), 0, ddl_type, has_conflict_ddl))) {
          LOG_WARN("failed to check ddl conflict", K(ret));
        } else if (has_conflict_ddl) {
          ret = OB_EAGAIN;
          LOG_WARN("failed to alter table that has conflict ddl", K(ret), K(orig_table_schema->get_table_id()));
        } else if (OB_FAIL(root_service->get_ddl_scheduler().create_ddl_task(param, trans, task_record))) {
          LOG_WARN("submit ddl task failed", K(ret));
        } else {
          res.task_id_ = task_record.task_id_;
          res.ddl_need_retry_at_executor_ = task_record.ddl_need_retry_at_executor_;
        }
      } else if (is_simple_table_long_running_ddl(ddl_type)) {
        ObCreateDDLTaskParam param(tenant_id,
                                   ddl_type,
                                   &new_table_schema,
                                   nullptr,
                                   0/*object_id*/,
                                   new_table_schema.get_schema_version(),
                                   alter_table_arg.parallelism_,
                                   alter_table_arg.consumer_group_id_,
                                   &alter_table_arg.allocator_,
                                   &alter_table_arg,
                                   0/*parent_task_id*/,
                                   task_id);
        param.tenant_data_version_ = tenant_data_version;
        if (OB_FAIL(root_service->get_ddl_scheduler().create_ddl_task(param, trans, task_record))) {
            LOG_WARN("submit ddl task failed", K(ret));
        } else {
          res.task_id_ = task_record.task_id_;
          res.ddl_need_retry_at_executor_ = task_record.ddl_need_retry_at_executor_;
        }
      } else {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("fail to identify long running ddl type", K(ret), K(ddl_type));
      }

      if (trans.is_started()) {
        int temp_ret = OB_SUCCESS;
        if (OB_SUCCESS != (temp_ret = trans.end(OB_SUCC(ret)))) {
          LOG_WARN_RET(temp_ret, "trans end failed", "is_commit", OB_SUCCESS == ret, K(temp_ret));
          ret = (OB_SUCC(ret)) ? temp_ret : ret;
        }
      }
      if (OB_SUCC(ret)) {
        DEBUG_SYNC(RENAME_TABLE_BEFORE_PUBLISH_SCHEMA);
        int tmp_ret = OB_SUCCESS;
        if (OB_FAIL(publish_schema(tenant_id))) {
          LOG_WARN("publish_schema failed", K(ret));
        } else if (OB_TMP_FAIL(root_service->get_ddl_scheduler().schedule_ddl_task(task_record))) {
          LOG_WARN("fail to schedule ddl task", K(tmp_ret), K(task_record));
        }
      }
    }
  }
  return ret;
}

int ObDDLService::add_not_null_column_to_table_schema(
    obrpc::ObAlterTableArg &alter_table_arg,
    const uint64_t tenant_data_version,
    const ObTableSchema &origin_table_schema,
    ObTableSchema &new_table_schema,
    ObSchemaGetterGuard &schema_guard,
    ObDDLOperator &ddl_operator,
    ObDDLSQLTransaction &trans)
{
  int ret = OB_SUCCESS;
  const uint64_t tenant_id = origin_table_schema.get_tenant_id();
  const AlterTableSchema &alter_table_schema = alter_table_arg.alter_table_schema_;
  AlterColumnSchema *alter_column_schema = nullptr;
  ObSEArray<ObString, 4> gen_col_expr_arr;
  share::schema::ObTableSchema::const_column_iterator iter = origin_table_schema.column_begin();
  share::schema::ObTableSchema::const_column_iterator end = origin_table_schema.column_end();
  int64_t new_schema_version = OB_INVALID_VERSION;
  ObSchemaService *schema_service = schema_service_->get_schema_service();
  for (; OB_SUCC(ret) && iter != end; ++iter) {
    const share::schema::ObColumnSchemaV2 *column = *iter;
    if (OB_ISNULL(column)) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("invalid column schema", K(column));
    } else if (column->is_generated_column()) {
      const common::ObObj* ObObjtmp = &column->get_cur_default_value();
      if (OB_FAIL(gen_col_expr_arr.push_back(ObObjtmp->get_string()))) {
        LOG_WARN("fail to push back ObSEArray gen_col_expr_arr", K(ret));
      }
    }
  }
  const common::ObTimeZoneInfoWrap &tz_info_wrap = alter_table_arg.tz_info_wrap_;
  const common::ObString *nls_formats = alter_table_arg.nls_formats_;
  if (OB_FAIL(ret)) {
  } else if (OB_UNLIKELY(nullptr == schema_service
      || nullptr == tz_info_wrap.get_time_zone_info()
      || nullptr == tz_info_wrap.get_time_zone_info()->get_tz_info_map())) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("invalid tz_info_wrap", K(ret), KP(schema_service), K(tz_info_wrap));
  } else if (OB_ISNULL(nls_formats)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("invalid nls_formats", K(ret));
  } else {
    uint64_t curr_udt_set_id = 0;
    ObTableSchema::const_column_iterator it_begin = alter_table_schema.column_begin();
    ObTableSchema::const_column_iterator it_end = alter_table_schema.column_end();
    for (; OB_SUCC(ret) && it_begin != it_end; it_begin++) {
      if (OB_ISNULL(alter_column_schema = static_cast<AlterColumnSchema *>(*it_begin))) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("*it_begin is NULL", K(ret));
      } else if (OB_FAIL(add_new_column_to_table_schema(origin_table_schema,
                                                        alter_table_schema,
                                                        tz_info_wrap,
                                                        *nls_formats,
                                                        alter_table_arg.local_session_var_,
                                                        alter_table_arg.sequence_ddl_arg_,
                                                        alter_table_arg.allocator_,
                                                        new_table_schema,
                                                        *alter_column_schema,
                                                        gen_col_expr_arr,
                                                        schema_guard,
                                                        curr_udt_set_id,
                                                        &ddl_operator,
                                                        &trans))) {
        LOG_WARN("failed to add new column to table schema", K(ret));
      }
    }
    if (OB_FAIL(ret)) {
    } else if (OB_FAIL(check_has_multi_autoinc(new_table_schema))) {
      LOG_WARN("failed to check table has multi autoinc", K(ret));
    } else if (OB_FAIL(alter_table_constraints(
                      obrpc::ObAlterTableArg::ADD_CONSTRAINT,
                      schema_guard,
                      origin_table_schema,
                      alter_table_schema,
                      new_table_schema,
                      ddl_operator,
                      trans))) {
      LOG_WARN("alter table constraints failed", K(ret));
    } else if (OB_FAIL(add_column_to_column_group(origin_table_schema,
        alter_table_schema, new_table_schema, ddl_operator, trans))) {
      LOG_WARN("fail to add_column_to_column_group", K(ret), K(alter_table_schema), K(new_table_schema));
    } else if (OB_FAIL(new_table_schema.check_skip_index_valid())) {
      LOG_WARN("failed to check new table schema skip index", K(ret));
    } else if (OB_FAIL(schema_service_->gen_new_schema_version(tenant_id, new_schema_version))) {
      LOG_WARN("fail to gen new schema_version", K(ret), K(tenant_id));
    } else if (OB_FALSE_IT(new_table_schema.set_schema_version(new_schema_version))) {
    } else if (OB_FAIL(schema_service->get_table_sql_service().update_table_options(trans,
        origin_table_schema, new_table_schema, OB_DDL_ALTER_TABLE, &alter_table_arg.ddl_stmt_str_))) {
      LOG_WARN("failed to update data table schema attribute", K(ret));
    } else {
      ObArray<ObTabletID> new_tablet_ids;
      if (OB_FAIL(new_table_schema.get_tablet_ids(new_tablet_ids))) {
        LOG_WARN("failed to get tablet ids", K(ret));
      } else if (OB_FAIL(build_single_table_rw_defensive_(tenant_id, tenant_data_version, new_tablet_ids, new_table_schema.get_schema_version(), trans))) {
        LOG_WARN("failed to build rw defensive", K(ret));
      }
    }
  }
  return ret;
}

int ObDDLService::add_not_null_column_default_null_to_table_schema(
    obrpc::ObAlterTableArg &alter_table_arg,
    const uint64_t tenant_data_version,
    const ObTableSchema &origin_table_schema,
    ObTableSchema &new_table_schema,
    ObSchemaGetterGuard &schema_guard,
    ObDDLOperator &ddl_operator,
    ObDDLSQLTransaction &trans)
{
  int ret = OB_SUCCESS;
  bool is_table_empty = false;
  bool is_oracle_mode = false;
  const uint64_t tenant_id = origin_table_schema.get_tenant_id();
  if (OB_FAIL(origin_table_schema.check_if_oracle_compat_mode(is_oracle_mode))) {
    LOG_WARN("fail to check is oracle mode", K(ret), K(origin_table_schema));
  } else if (OB_UNLIKELY(!is_oracle_mode)) {
    ret = OB_NOT_SUPPORTED;
    LOG_WARN("add column not null in mysql mode is online ddl, not offline ddl", K(ret), K(is_oracle_mode));
  } else if (OB_FAIL(lock_table(trans, origin_table_schema))) {
    LOG_WARN("failed to lock ddl lock", K(ret));
  } else if (OB_FAIL(ObDDLUtil::check_table_empty_in_oracle_mode(tenant_id,
                                                                 origin_table_schema.get_table_id(),
                                                                 schema_guard,
                                                                 is_table_empty))) {
    LOG_WARN("failed to check table empty in oracle mode", K(ret));
  } else if (!is_table_empty) {
    ret = OB_ERR_TABLE_ADD_NOT_NULL_COLUMN_NOT_EMPTY;
    LOG_WARN("table add not null column to is not empty", K(ret), K(tenant_id), K(origin_table_schema.get_table_id()));
  } else if (OB_FAIL(add_not_null_column_to_table_schema(alter_table_arg,
                                                         tenant_data_version,
                                                         origin_table_schema,
                                                         new_table_schema,
                                                         schema_guard,
                                                         ddl_operator,
                                                         trans))) {
    LOG_WARN("failed to add new column to table schema");
  }
  return ret;
}

int ObDDLService::do_oracle_add_column_not_null_in_trans(obrpc::ObAlterTableArg &alter_table_arg,
                                                         ObSchemaGetterGuard &schema_guard,
                                                         const uint64_t tenant_data_version,
                                                         const bool is_default_value_null)
{
  int ret = OB_SUCCESS;
  ObArray<uint64_t> unused_column_ids;
  const AlterTableSchema &alter_table_schema = alter_table_arg.alter_table_schema_;
  const uint64_t tenant_id = alter_table_schema.get_tenant_id();
  if (OB_FAIL(check_inner_stat())) {
    LOG_WARN("variable is not init", K(ret));
  } else {
    SMART_VAR(ObTableSchema, new_table_schema) {
      const ObTableSchema *origin_table_schema = nullptr;
      bool is_oracle_mode = false;
      const common::ObTimeZoneInfoWrap &tz_info_wrap = alter_table_arg.tz_info_wrap_;
      const common::ObString *nls_formats = alter_table_arg.nls_formats_;
      if (OB_FAIL(get_and_check_table_schema(alter_table_arg,
                                            schema_guard,
                                            alter_table_schema,
                                            origin_table_schema))) {
        LOG_WARN("fail to get and check table schema", K(ret));
      } else if (OB_FAIL(origin_table_schema->get_unused_column_ids(unused_column_ids))) {
        LOG_WARN("get unused column ids failed", KR(ret), KPC(origin_table_schema));
      } else if (OB_UNLIKELY(unused_column_ids.count() >= OB_MAX_UNUSED_COLUMNS_COUNT)) {
        ret = OB_OP_NOT_ALLOW;
        LOG_WARN("The number of obsolete columns reaches the limit", KR(ret));
        LOG_USER_ERROR(OB_OP_NOT_ALLOW, "The number of obsolete columns reaches the limit. Use \"alter table table_name force\" to defragment first, otherwise adding column instantly is");
      } else if (OB_FAIL(origin_table_schema->check_if_oracle_compat_mode(is_oracle_mode))) {
        LOG_WARN("fail to check is oracle mode", K(ret), K(origin_table_schema));
      } else if (OB_UNLIKELY(!is_oracle_mode)) {
        ret = OB_NOT_SUPPORTED;
        LOG_WARN("add column not null in mysql mode is online ddl, not offline ddl", K(ret), K(is_oracle_mode));
      } else if (OB_ISNULL(tz_info_wrap.get_time_zone_info())
                || OB_ISNULL(tz_info_wrap.get_time_zone_info()->get_tz_info_map())) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("invalid tz_info_wrap", K(tz_info_wrap), K(ret));
      } else if (OB_ISNULL(nls_formats)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("invalid nls_formats", K(ret));
      } else if (OB_FAIL(new_table_schema.assign(*origin_table_schema))) {
        LOG_WARN("fail to assign schema", K(ret));
      } else {
        ObDDLSQLTransaction trans(schema_service_);
        int64_t refreshed_schema_version = 0;
        new_table_schema.set_progressive_merge_round(origin_table_schema->get_progressive_merge_round() + 1);
        if (OB_FAIL(schema_guard.get_schema_version(tenant_id, refreshed_schema_version))) {
          LOG_WARN("failed to get tenant schema version", KR(ret), K(tenant_id));
        } else if (OB_FAIL(trans.start(sql_proxy_, tenant_id, refreshed_schema_version))) {
          LOG_WARN("start transaction failed", KR(ret), K(tenant_id), K(refreshed_schema_version));
        } else {
          ObDDLOperator ddl_operator(*schema_service_, *sql_proxy_);
          if (is_default_value_null) {
            if (OB_FAIL(add_not_null_column_default_null_to_table_schema(
                          alter_table_arg,
                          tenant_data_version,
                          *origin_table_schema,
                          new_table_schema,
                          schema_guard,
                          ddl_operator,
                          trans))) {
              LOG_WARN("failed to add default value null column to table schema", K(ret));
            }
          } else if (OB_FAIL(add_not_null_column_to_table_schema(alter_table_arg,
                                                                tenant_data_version,
                                                                *origin_table_schema,
                                                                new_table_schema,
                                                                schema_guard,
                                                                ddl_operator,
                                                                trans))) {
            LOG_WARN("failed to add column to table schema", K(ret));
          }
          if (OB_SUCC(ret)) {
            // add lob.
            const bool has_lob_in_origin_table = origin_table_schema->has_lob_column(true/*ignore_unused_column*/);
            const bool has_lob_in_new_table = new_table_schema.has_lob_column(true/*ignore_unused_column*/);
            if (!has_lob_in_origin_table && has_lob_in_new_table) {
              // origin table doesnt has lob, fast add not null lob column should create new lobs.
              bool is_add_lob = false;
              if (OB_FAIL(create_aux_lob_table_if_need(new_table_schema, schema_guard, ddl_operator, trans,
                true/*need_sync_table_schema_version*/, is_add_lob))) {
                LOG_WARN("fail to create_aux_lob_table_if_need", KR(ret), K(new_table_schema));
              }
            }
          }
        }
        if (trans.is_started()) {
          int temp_ret = OB_SUCCESS;
          if (OB_SUCCESS != (temp_ret = trans.end(OB_SUCC(ret)))) {
            LOG_WARN_RET(temp_ret, "trans end failed", "is_commit", OB_SUCCESS == ret, K(temp_ret));
            ret = (OB_SUCC(ret)) ? temp_ret : ret;
          }
        }
        if (OB_SUCC(ret)) {
          if (OB_FAIL(publish_schema(tenant_id))) {
            LOG_WARN("publish_schema failed", K(ret));
          }
        }
      }
    }
  }
  return ret;
}

int ObDDLService::create_hidden_table(
    const obrpc::ObCreateHiddenTableArg &create_hidden_table_arg,
    obrpc::ObCreateHiddenTableRes &res)
{
  int ret = OB_SUCCESS;
  uint64_t tenant_data_version = 0;
  const uint64_t tenant_id = create_hidden_table_arg.get_tenant_id();
  const int64_t table_id = create_hidden_table_arg.get_table_id();
  const uint64_t dest_tenant_id = tenant_id;
  ObRootService *root_service = GCTX.root_service_;
  bool bind_tablets = true;
  ObSchemaGetterGuard schema_guard;
  const ObTableSchema *orig_table_schema = NULL;
  const ObDatabaseSchema *orig_database_schema = nullptr;
  common::ObArenaAllocator allocator_for_redef(lib::ObLabel("StartRedefTable"));
  if (OB_UNLIKELY(!create_hidden_table_arg.is_valid())) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("create_hidden_table_arg is invalid", K(ret), K(create_hidden_table_arg));
  } else if (OB_ISNULL(root_service)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("error unexpected, root service must not be nullptr", K(ret));
  } else if (OB_FAIL(check_inner_stat())) {
    LOG_WARN("variable is not init", K(ret));
  } else if (OB_FAIL(get_tenant_schema_guard_with_version_in_inner_table(tenant_id, schema_guard))) {
    LOG_WARN("fail to get schema guard with version in inner table", K(ret), K(tenant_id));
  } else if (OB_FAIL(schema_guard.get_table_schema(tenant_id, table_id, orig_table_schema))) {
    LOG_WARN("fail to get table schema", K(ret));
  } else if (OB_ISNULL(orig_table_schema)) {
    ret = OB_TABLE_NOT_EXIST;
    LOG_WARN("orig table schema is nullptr", K(ret));
  } else if (OB_FAIL(schema_guard.get_database_schema(tenant_id, orig_table_schema->get_database_id(), orig_database_schema))) {
    LOG_WARN("fail to get orig database schema", K(ret));
  } else if (OB_ISNULL(orig_database_schema)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("orig_database_schema is nullptr", K(ret));
  } else if (OB_FAIL(GET_MIN_DATA_VERSION(tenant_id, tenant_data_version))) {
    LOG_WARN("get min data version failed", K(ret), K(tenant_id));
  } else {
    HEAP_VAR(ObTableSchema, new_table_schema) {
      ObDDLOperator ddl_operator(*schema_service_, *sql_proxy_);
      if (OB_FAIL(new_table_schema.assign(*orig_table_schema))) {
        LOG_WARN("fail to assign schema", K(ret));
      } else {
        ObDDLSQLTransaction trans(schema_service_);
        common::ObArenaAllocator allocator;
        ObDDLTaskRecord task_record;
        ObTableLockOwnerID owner_id;
        int64_t task_id = 0;
        int64_t refreshed_schema_version = 0;
        new_table_schema.set_tenant_id(dest_tenant_id);
        new_table_schema.set_table_state_flag(ObTableStateFlag::TABLE_STATE_OFFLINE_DDL);
        new_table_schema.set_table_referenced_by_mv(ObTableReferencedByMVFlag::IS_NOT_REFERENCED_BY_MV);
        if (OB_FAIL(schema_guard.get_schema_version(tenant_id, refreshed_schema_version))) {
          LOG_WARN("failed to get tenant schema version", KR(ret), K(tenant_id));
        } else if (OB_FAIL(trans.start(sql_proxy_, tenant_id, refreshed_schema_version))) {
          LOG_WARN("start transaction failed", KR(ret), K(tenant_id), K(refreshed_schema_version));
        } else if (OB_FAIL(ObDDLTask::fetch_new_task_id(root_service->get_sql_proxy(), tenant_id, task_id))) {
          LOG_WARN("fetch new task id failed", K(ret));
        } else if (OB_FAIL(owner_id.convert_from_value(ObLockOwnerType::DEFAULT_OWNER_TYPE, task_id))) {
          LOG_WARN("failed to get owner id", K(ret), K(task_id));
        } else if (OB_FAIL(ObDDLLock::lock_for_offline_ddl(*orig_table_schema,
                                                           nullptr,
                                                           owner_id,
                                                           trans))) {
          LOG_WARN("failed to lock ddl lock", K(ret));
        } else if (OB_UNLIKELY(orig_table_schema->is_offline_ddl_table())) {
          ret = OB_SCHEMA_EAGAIN;
          LOG_WARN("table in offline ddl or direct load, retry", K(ret), K(orig_table_schema->get_table_id()), K(orig_table_schema->get_table_mode()));
        } else if (OB_FAIL(create_user_hidden_table(
                  *orig_table_schema,
                  new_table_schema,
                  nullptr,
                  bind_tablets,
                  schema_guard,
                  schema_guard,
                  ddl_operator,
                  trans,
                  allocator,
                  tenant_data_version))) {
          LOG_WARN("fail to create hidden table", K(ret));
        } else {
          LOG_INFO("create hidden table success!", K(table_id), K(new_table_schema));
        }
        if (OB_SUCC(ret)) {
          HEAP_VAR(obrpc::ObAlterTableArg, alter_table_arg) {
            ObPrepareAlterTableArgParam param;
            if (OB_FAIL(param.init(create_hidden_table_arg.get_consumer_group_id(),
                                   create_hidden_table_arg.get_session_id(),
                                   create_hidden_table_arg.get_sql_mode(),
                                   create_hidden_table_arg.get_ddl_stmt_str(),
                                   orig_table_schema->get_table_name_str(),
                                   orig_database_schema->get_database_name_str(),
                                   orig_database_schema->get_database_name_str(),
                                   create_hidden_table_arg.get_tz_info(),
                                   create_hidden_table_arg.get_tz_info_wrap(),
                                   create_hidden_table_arg.get_nls_formats(),
                                   create_hidden_table_arg.get_foreign_key_checks()))) {
              LOG_WARN("param init failed", K(ret));
            } else if (OB_FAIL(root_service->get_ddl_scheduler().prepare_alter_table_arg(param, &new_table_schema, alter_table_arg))) {
              LOG_WARN("prepare alter table arg fail", K(ret));
            } else if (!create_hidden_table_arg.get_tablet_ids().empty()){
              const ObIArray<ObTabletID> &tablet_ids = create_hidden_table_arg.get_tablet_ids();
              alter_table_arg.alter_table_schema_.reset_partition_array();
              for (int64_t i = 0; OB_SUCC(ret) && (i < tablet_ids.count()); ++i) {
                ObPartition part;
                if (OB_FALSE_IT(part.set_tablet_id(tablet_ids.at(i)))) {
                } else if (OB_FAIL(alter_table_arg.alter_table_schema_.add_partition(part))) {
                  LOG_WARN("failed to add partition", KR(ret), K(part));
                }
              }
              if (OB_SUCC(ret)) {
                alter_table_arg.is_direct_load_partition_ = true;
              }
            }

            LOG_DEBUG("alter table arg preparation complete!", K(ret), K(alter_table_arg));
            if (OB_SUCC(ret)) {
              ObCreateDDLTaskParam param(tenant_id,
                                        create_hidden_table_arg.get_ddl_type(),
                                        orig_table_schema,
                                        &new_table_schema,
                                        table_id,
                                        orig_table_schema->get_schema_version(),
                                        create_hidden_table_arg.get_parallelism(),
                                        create_hidden_table_arg.get_consumer_group_id(),
                                        &allocator_for_redef,
                                        &alter_table_arg,
                                        0,
                                        task_id);
              param.tenant_data_version_ = tenant_data_version;
              if (OB_FAIL(root_service->get_ddl_scheduler().create_ddl_task(param, trans, task_record))) {
                LOG_WARN("submit ddl task failed", K(ret));
              } else {
                res.tenant_id_ = tenant_id;
                res.table_id_ = table_id;
                res.dest_tenant_id_ = dest_tenant_id;
                res.dest_table_id_ = task_record.target_object_id_;
                res.schema_version_ = task_record.schema_version_;
                res.trace_id_ = task_record.trace_id_;
                res.task_id_ = task_record.task_id_;
              }
            }
          }
        }
        if (trans.is_started()) {
          int temp_ret = OB_SUCCESS;
          if (OB_SUCCESS != (temp_ret = trans.end(OB_SUCC(ret)))) {
            LOG_WARN("trans end failed", "is_commit", OB_SUCCESS == ret, K(temp_ret));
            ret = (OB_SUCC(ret)) ? temp_ret : ret;
          }
        }
        if (OB_SUCC(ret)) {
          int tmp_ret = OB_SUCCESS;
          if (OB_FAIL(publish_schema(tenant_id))) {
            LOG_WARN("publish_schema failed", K(ret));
          } else if (OB_TMP_FAIL(root_service->get_ddl_scheduler().schedule_ddl_task(task_record))) {
            LOG_WARN("fail to schedule ddl task", K(tmp_ret), K(task_record));
          } else {
            LOG_INFO("schedule ddl task success", K(task_record));
          }
        }
      }
    }
  }
  return ret;
}

int ObDDLService::mview_complete_refresh(
    const obrpc::ObMViewCompleteRefreshArg &arg,
    obrpc::ObMViewCompleteRefreshRes &res,
    share::schema::ObSchemaGetterGuard &schema_guard)
{
  int ret = OB_SUCCESS;
  const uint64_t tenant_id = arg.tenant_id_;
  ObRootService *root_service = GCTX.root_service_;
  int64_t refreshed_schema_version = 0;
  common::ObArenaAllocator allocator("MVRef");
  ObDDLTaskRecord task_record;
  uint64_t tenant_data_version = 0;
  if (OB_UNLIKELY(!arg.is_valid())) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("invalid args", KR(ret), K(arg));
  } else if (OB_ISNULL(root_service)) {
    ret = OB_ERR_SYS;
    LOG_WARN("error unexpected, root service must not be nullptr", KR(ret));
  } else if (OB_FAIL(check_inner_stat())) {
    LOG_WARN("variable is not init", KR(ret));
  } else if (OB_FAIL(schema_guard.get_schema_version(tenant_id, refreshed_schema_version))) {
    LOG_WARN("failed to get tenant schema version", KR(ret), K(tenant_id));
  } else if (OB_FAIL(GET_MIN_DATA_VERSION(tenant_id, tenant_data_version))) {
    LOG_WARN("get min data version failed", K(ret), K(tenant_id));
  } else {
    ObDDLSQLTransaction trans(schema_service_);
    if (OB_FAIL(trans.start(sql_proxy_, tenant_id, refreshed_schema_version))) {
      LOG_WARN("start transaction failed", KR(ret), K(tenant_id), K(refreshed_schema_version));
    } else if (OB_FAIL(mview_complete_refresh_in_trans(arg, res, trans, allocator, schema_guard, tenant_data_version, task_record))) {
      LOG_WARN("failed to do mview complete refresh in trans", KR(ret), K(arg));
    }
    if (trans.is_started()) {
      int temp_ret = OB_SUCCESS;
      if (OB_SUCCESS != (temp_ret = trans.end(OB_SUCC(ret)))) {
        LOG_WARN("trans end failed", "is_commit", OB_SUCCESS == ret, KR(temp_ret));
        ret = COVER_SUCC(temp_ret);
      }
    }
  }
  if (OB_SUCC(ret)) {
    int tmp_ret = OB_SUCCESS;
    if (OB_FAIL(publish_schema(tenant_id))) {
      LOG_WARN("publish_schema failed", KR(ret));
    } else if (OB_TMP_FAIL(root_service->get_ddl_scheduler().schedule_ddl_task(task_record))) {
      LOG_WARN("fail to schedule ddl task", KR(tmp_ret), K(task_record));
    } else {
      LOG_INFO("schedule ddl task success", K(task_record));
    }
  }
  return ret;
}

int ObDDLService::mview_complete_refresh_in_trans(
    const obrpc::ObMViewCompleteRefreshArg &arg,
    obrpc::ObMViewCompleteRefreshRes &res,
    ObDDLSQLTransaction &trans,
    common::ObIAllocator &allocator,
    share::schema::ObSchemaGetterGuard &schema_guard,
    const uint64_t tenant_data_version,
    ObDDLTaskRecord &task_record)
{
  int ret = OB_SUCCESS;
  const uint64_t tenant_id = arg.tenant_id_;
  const int64_t mview_table_id = arg.table_id_;
  ObRootService *root_service = GCTX.root_service_;
  const ObTableSchema *mview_table_schema = nullptr;
  const ObDatabaseSchema *database_schema = nullptr;
  const ObTableSchema *container_table_schema = nullptr;
  int64_t refreshed_schema_version = 0;
  int64_t task_id = 0;
  if (OB_UNLIKELY(!arg.is_valid())) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("invalid args", KR(ret), K(arg));
  } else if (OB_ISNULL(root_service)) {
    ret = OB_ERR_SYS;
    LOG_WARN("error unexpected, root service must not be nullptr", KR(ret));
  } else if (OB_FAIL(check_inner_stat())) {
    LOG_WARN("variable is not init", KR(ret));
  } else if (OB_FAIL(schema_guard.get_schema_version(tenant_id, refreshed_schema_version))) {
    LOG_WARN("failed to get tenant schema version", KR(ret), K(tenant_id));
  } else if (OB_FAIL(schema_guard.get_table_schema(tenant_id, mview_table_id, mview_table_schema))) {
    LOG_WARN("fail to get table schema", KR(ret), K(tenant_id), K(mview_table_id));
  } else if (OB_ISNULL(mview_table_schema)) {
    ret = OB_TABLE_NOT_EXIST;
    LOG_WARN("mview table schema is nullptr", KR(ret), K(tenant_id), K(mview_table_id));
  } else if (OB_UNLIKELY(!mview_table_schema->is_materialized_view())) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("unexpected not materialized view", KR(ret), KPC(mview_table_schema));
  } else if (OB_FAIL(schema_guard.get_database_schema(tenant_id, mview_table_schema->get_database_id(), database_schema))) {
    LOG_WARN("fail to get database schema", KR(ret), K(tenant_id), "database_id", mview_table_schema->get_database_id());
  } else if (OB_ISNULL(database_schema)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("database schema is nullptr", KR(ret), K(tenant_id), "database_id", mview_table_schema->get_database_id());
  } else if (OB_FAIL(schema_guard.get_table_schema(tenant_id, mview_table_schema->get_data_table_id(), container_table_schema))) {
    LOG_WARN("fail to get table schema", KR(ret), K(tenant_id), "container_table_id", mview_table_schema->get_data_table_id());
  } else if (OB_ISNULL(container_table_schema)) {
    ret = OB_TABLE_NOT_EXIST;
    LOG_WARN("container table schema is nullptr", KR(ret), K(tenant_id), "container_table_id", mview_table_schema->get_data_table_id());
  } else {
    HEAP_VAR(ObTableSchema, new_container_table_schema) {
      if (OB_FAIL(new_container_table_schema.assign(*container_table_schema))) {
        LOG_WARN("fail to assign schema", KR(ret));
      } else {
        const bool bind_tablets = true;
        ObDDLOperator ddl_operator(*schema_service_, *sql_proxy_);
        new_container_table_schema.set_table_state_flag(ObTableStateFlag::TABLE_STATE_OFFLINE_DDL);
        ObTableLockOwnerID owner_id;
        if (OB_FAIL(ObDDLTask::fetch_new_task_id(root_service->get_sql_proxy(), tenant_id, task_id))) {
          LOG_WARN("fetch new task id failed", KR(ret), K(tenant_id));
        } else if (OB_FAIL(owner_id.convert_from_value(ObLockOwnerType::DEFAULT_OWNER_TYPE,
                                                task_id))) {
          LOG_WARN("failed to get owner id", K(ret), K(task_id));
        } else if (OB_FAIL(ObDDLLock::lock_for_offline_ddl(*container_table_schema,
                                                            nullptr,
                                                            owner_id,
                                                            trans))) {
          LOG_WARN("failed to lock ddl lock", KR(ret));
        } else if (OB_FAIL(create_user_hidden_table(*container_table_schema,
                                                    new_container_table_schema,
                                                    nullptr/*sequence_ddl_arg*/,
                                                    bind_tablets,
                                                    schema_guard,
                                                    schema_guard,
                                                    ddl_operator,
                                                    trans,
                                                    allocator,
                                                    tenant_data_version))) {
          LOG_WARN("fail to create hidden table", KR(ret));
        } else {
          LOG_INFO("create hidden table success!", K(mview_table_id), "container_table_id", container_table_schema->get_table_id(),
                   K(new_container_table_schema));
        }
      }
      if (OB_SUCC(ret)) {
        HEAP_VAR(obrpc::ObAlterTableArg, alter_table_arg) {
          ObString empty_ddl_stmt_str;
          ObPrepareAlterTableArgParam prepare_param;
          ObArray<ObDependencyInfo> dependency_infos;
          if (OB_FAIL(prepare_param.init(arg.consumer_group_id_,
                                          arg.session_id_,
                                          arg.sql_mode_,
                                          empty_ddl_stmt_str,
                                          container_table_schema->get_table_name_str(),
                                          database_schema->get_database_name_str(),
                                          database_schema->get_database_name_str(),
                                          arg.tz_info_,
                                          arg.tz_info_wrap_,
                                          arg.nls_formats_,
                                          true/*foreign key checks*/))) {
            LOG_WARN("prepare param init failed", KR(ret));
          } else if (OB_FAIL(root_service->get_ddl_scheduler().prepare_alter_table_arg(prepare_param, &new_container_table_schema, alter_table_arg))) {
            LOG_WARN("prepare alter table arg fail", KR(ret));
          } else if (OB_FAIL(alter_table_arg.based_schema_object_infos_.assign(arg.based_schema_object_infos_))) {
            LOG_WARN("fail to assign based schema object infos", KR(ret));
          } else {
            alter_table_arg.mview_refresh_info_.is_mview_complete_refresh_ = true;
            alter_table_arg.mview_refresh_info_.mview_table_id_ = mview_table_id;
            alter_table_arg.mview_refresh_info_.last_refresh_scn_ = arg.last_refresh_scn_;
            alter_table_arg.mview_refresh_info_.start_time_ = ObTimeUtil::current_time();
            LOG_DEBUG("alter table arg preparation complete!", K(alter_table_arg));
            ObCreateDDLTaskParam param(tenant_id,
                                        DDL_MVIEW_COMPLETE_REFRESH,
                                        container_table_schema,
                                        &new_container_table_schema,
                                        container_table_schema->get_table_id(),
                                        refreshed_schema_version,
                                        arg.parallelism_,
                                        arg.consumer_group_id_,
                                        &allocator,
                                        &alter_table_arg,
                                        arg.parent_task_id_,
                                        task_id);
            param.tenant_data_version_ = tenant_data_version;
            if (OB_FAIL(root_service->get_ddl_scheduler().create_ddl_task(param, trans, task_record))) {
              LOG_WARN("submit ddl task failed", KR(ret));
            } else {
              res.trace_id_ = task_record.trace_id_;
              res.task_id_ = task_record.task_id_;
            }
          }
        }
      }
    }
  }
  return ret;
}

int ObDDLService::recover_restore_table_ddl_task(
      const obrpc::ObRecoverRestoreTableDDLArg &arg)
{
  int ret = OB_SUCCESS;
  ObDDLTaskRecord task_record;
  common::ObArenaAllocator allocator(lib::ObLabel("CreateDDLParam"));
  ObRootService *root_service = GCTX.root_service_;
  if (OB_UNLIKELY(!arg.is_valid())) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("invalid arg", K(ret), K(arg));
  } else if (OB_ISNULL(root_service)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("root_service is nullptr", K(ret));
  } else if (OB_FAIL(check_inner_stat())) {
    LOG_WARN("variable is not init", K(ret));
  } else {
    // For the restore table operation, the role of the source tenant is backup, there will be no write operation on it,
    // thus we need no lock on it.
    // Same as the offline ddl, we will create a restore dest table, a hidden one with table mode `hidden_offline_ddl`.
    // Different from the offline ddl, we will not change any attribute of the source table.
    uint64_t tenant_data_version = 0;
    int64_t refreshed_dst_tenant_version = 0;
    const uint64_t session_id = arg.target_schema_.get_session_id();
    ObSchemaGetterGuard hold_buf_src_tenant_schema_guard;
    ObSchemaGetterGuard hold_buf_dst_tenant_schema_guard;
    ObSchemaGetterGuard *src_tenant_schema_guard = nullptr;
    ObSchemaGetterGuard *dst_tenant_schema_guard = nullptr;
    hold_buf_src_tenant_schema_guard.set_session_id(session_id);
    hold_buf_dst_tenant_schema_guard.set_session_id(session_id);
    const ObTableSchema *src_table_schema = nullptr;
    const ObDatabaseSchema *src_db_schema = nullptr;
    const ObDatabaseSchema *dst_db_schema = nullptr;
    const uint64_t src_tenant_id = arg.src_tenant_id_;
    const uint64_t dst_tenant_id = arg.target_schema_.get_tenant_id();
    ObDDLSQLTransaction dst_tenant_trans(schema_service_); // for dst tenant only.
    bool has_fts_index = false;
    bool has_mv_index = false;
    HEAP_VARS_2((ObTableSchema, dst_table_schema),
                (obrpc::ObAlterTableArg, alter_table_arg)) {
      if (OB_FAIL(get_tenant_schema_guard_with_version_in_inner_table(
                                      src_tenant_id, dst_tenant_id,
                                      hold_buf_src_tenant_schema_guard, hold_buf_dst_tenant_schema_guard,
                                      src_tenant_schema_guard, dst_tenant_schema_guard))) {
        LOG_WARN("failed to get schema guard with version in inner table", K(ret), K(src_tenant_id), K(dst_tenant_id), K(arg));
      } else if (OB_FAIL(src_tenant_schema_guard->get_table_schema(src_tenant_id, arg.src_table_id_, src_table_schema))) {
        LOG_WARN("fail to get table schema", K(ret), K(session_id), K(arg));
      } else if (OB_ISNULL(src_table_schema)) {
        ret = OB_TABLE_NOT_EXIST;
        LOG_WARN("orig table schema is nullptr", K(ret), K(session_id), K(arg));
      } else if (OB_FAIL(src_table_schema->check_has_fts_index(*src_tenant_schema_guard, has_fts_index))) {
        LOG_WARN("failed to check if data table has fulltext index", K(ret));
      } else if (has_fts_index) {
        ret = OB_NOT_SUPPORTED;
        LOG_WARN("failed to import table when table has fulltext index", K(ret));
        LOG_USER_ERROR(OB_NOT_SUPPORTED, "import table with fulltext index");
      } else if (OB_FAIL(src_table_schema->check_has_multivalue_index(*src_tenant_schema_guard, has_mv_index))) {
        LOG_WARN("failed to check if data table has multivalue index", K(ret));
      } else if (has_mv_index) {
        ret = OB_NOT_SUPPORTED;
        LOG_WARN("failed to import table when table has fulltext index", K(ret));
        LOG_USER_ERROR(OB_NOT_SUPPORTED, "import table with multivalue index");
      } else if (OB_FAIL(src_tenant_schema_guard->get_database_schema(src_tenant_id, src_table_schema->get_database_id(), src_db_schema))) {
        LOG_WARN("fail to get orig database schema", K(ret));
      } else if (OB_ISNULL(src_db_schema)) {
        ret = OB_ERR_BAD_DATABASE;
        LOG_WARN("unknown database", K(ret), K(src_tenant_id), "db_id", src_table_schema->get_database_id());
      } else if (OB_FAIL(dst_tenant_schema_guard->get_schema_version(dst_tenant_id, refreshed_dst_tenant_version))) {
        LOG_WARN("failed to get tenant schema version", K(ret), K(dst_tenant_id));
      } else if (OB_FAIL(dst_tenant_schema_guard->get_database_schema(dst_tenant_id, arg.target_schema_.get_database_id(), dst_db_schema))) {
        LOG_WARN("fail to get orig database schema", K(ret), K(arg));
      } else if (OB_ISNULL(dst_db_schema)) {
        ret = OB_ERR_BAD_DATABASE;
        LOG_WARN("unknown database", K(ret), K(dst_tenant_id), K(session_id), "db_id", arg.target_schema_.get_database_id());
      } else if (OB_FAIL(GET_MIN_DATA_VERSION(dst_tenant_id, tenant_data_version))) {
        LOG_WARN("get min data version failed", K(ret), K(dst_tenant_id));
      } else {
        ObDDLOperator ddl_operator(*schema_service_, *sql_proxy_);
        bool is_dest_table_column_store = false;
        ObString index_name("");
        if (OB_FAIL(dst_tenant_trans.start(sql_proxy_, dst_tenant_id, refreshed_dst_tenant_version))) {
          LOG_WARN("start transaction failed", K(ret), K(dst_tenant_id), K(refreshed_dst_tenant_version));
        } else if (OB_FAIL(dst_table_schema.assign(arg.target_schema_))) {
          LOG_WARN("assign failed", K(ret), K(session_id), K(arg));
        } else if (OB_FAIL(dst_table_schema.get_is_column_store(is_dest_table_column_store))) {
          LOG_WARN("judge if dest table is column store failed", K(ret), K(arg));
        } else if (is_dest_table_column_store) {
          ret = OB_NOT_SUPPORTED;
          LOG_WARN("not supported to retore table with column store", K(ret), K(arg));
        } else if (OB_FAIL(create_user_hidden_table(*src_table_schema, dst_table_schema, nullptr/*sequence_ddl_arg*/,
          false/*bind_tablets*/, *src_tenant_schema_guard, *dst_tenant_schema_guard, ddl_operator,
          dst_tenant_trans, allocator, tenant_data_version, index_name, true /*ignore_cs_replica*/))) {
          LOG_WARN("create user hidden table failed", K(ret), K(arg), K(tenant_data_version));
        } else {
          ObPrepareAlterTableArgParam param;
          if (OB_FAIL(param.init(arg.consumer_group_id_, session_id, 0/*sql_mode, unused*/, arg.ddl_stmt_str_,
              src_table_schema->get_table_name_str(), src_db_schema->get_database_name_str(),
              dst_db_schema->get_database_name_str(), arg.tz_info_, arg.tz_info_wrap_, arg.nls_formats_, true/*foreign_key_checks*/))) {
            LOG_WARN("fail to prepare alter table arg param", K(ret), K(arg));
          } else if (OB_FAIL(root_service->get_ddl_scheduler().prepare_alter_table_arg(param, &dst_table_schema, alter_table_arg))) {
            LOG_WARN("prepare alter table arg failed", K(ret), K(param));
          } else {
            alter_table_arg.alter_table_schema_.set_schema_version(dst_table_schema.get_schema_version());
            alter_table_arg.alter_table_schema_.set_table_name(arg.target_schema_.get_table_name_str());
            alter_table_arg.parallelism_ = arg.parallelism_;
            alter_table_arg.consumer_group_id_ = arg.consumer_group_id_;
            ObCreateDDLTaskParam param(dst_table_schema.get_tenant_id(),
                                      ObDDLType::DDL_TABLE_RESTORE,
                                      src_table_schema,
                                      &dst_table_schema,
                                      src_table_schema->get_table_id()/*object_id*/,
                                      src_table_schema->get_schema_version(),
                                      MAX(1, arg.parallelism_),
                                      arg.consumer_group_id_,
                                      &allocator,
                                      &alter_table_arg,
                                      0,
                                      arg.ddl_task_id_);
            param.tenant_data_version_ = tenant_data_version;
            if (OB_FAIL(root_service->get_ddl_scheduler().create_ddl_task(param, dst_tenant_trans, task_record))) {
              LOG_WARN("submit ddl task failed", K(ret));
            }
          }
        }
      }
    }
    if (dst_tenant_trans.is_started()) {
      int tmp_ret = OB_SUCCESS;
      if (OB_TMP_FAIL(dst_tenant_trans.end(OB_SUCC(ret)))) {
        LOG_WARN("trans end failed", "is_commit", OB_SUCCESS == ret, K(tmp_ret));
        ret = (OB_SUCC(ret)) ? tmp_ret : ret;
      }
    }
    if (OB_SUCC(ret)) {
      int tmp_ret = OB_SUCCESS;
      if (OB_FAIL(publish_schema(dst_table_schema.get_tenant_id()))) {
        LOG_WARN("publish_schema failed", K(ret), K(dst_table_schema));
      } else if (OB_TMP_FAIL(root_service->get_ddl_scheduler().schedule_ddl_task(task_record))) {
        LOG_WARN("fail to schedule ddl task", K(tmp_ret), K(task_record));
      }
    }
  }
  return ret;
}

int ObDDLService::get_and_check_table_schema(
    const obrpc::ObAlterTableArg &alter_table_arg,
    ObSchemaGetterGuard &schema_guard,
    const AlterTableSchema &alter_table_schema,
    const ObTableSchema *&orig_table_schema,
    bool is_offline_ddl/*false*/)
{
  int ret = OB_SUCCESS;
  uint64_t tenant_id = alter_table_schema.get_tenant_id();
  schema_guard.set_session_id(alter_table_arg.session_id_);
  const ObString &origin_database_name = alter_table_schema.get_origin_database_name();
  const ObString &origin_table_name = alter_table_schema.get_origin_table_name();
  bool is_alter_comment = false;
  if (origin_database_name.empty() || origin_table_name.empty()) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("database name or table name is null", K(alter_table_schema),
        K(origin_database_name), K(origin_table_name), K(ret));
  } else {
    bool is_index = false;
    bool is_db_in_recyclebin = false;
    uint64_t compat_version = OB_INVALID_VERSION;
    if (OB_FAIL(schema_guard.get_table_schema(tenant_id,
                origin_database_name,
                origin_table_name,
                is_index,
                orig_table_schema))) {
      LOG_WARN("fail to get table schema", K(ret), K(tenant_id),
      K(origin_database_name), K(origin_table_name));
    } else if (NULL == orig_table_schema) {
      ret = OB_TABLE_NOT_EXIST;
      LOG_USER_ERROR(OB_TABLE_NOT_EXIST, to_cstring(origin_database_name),
      to_cstring(origin_table_name));
    } else if (OB_FAIL(schema_guard.check_database_in_recyclebin(
                       tenant_id,
                       orig_table_schema->get_database_id(),
                       is_db_in_recyclebin))) {
      LOG_WARN("check database in recyclebin failed", K(ret), K(tenant_id), K(*orig_table_schema));
    } else if (!alter_table_arg.is_inner_ && is_db_in_recyclebin) {
      ret = OB_ERR_OPERATION_ON_RECYCLE_OBJECT;
      LOG_WARN("can not alter table in recyclebin", K(ret), K(alter_table_arg));
    } else if (!alter_table_arg.skip_sys_table_check_
      && OB_FAIL(check_enable_sys_table_ddl(*orig_table_schema, OB_DDL_ALTER_TABLE))) {
      LOG_WARN("ddl is not allowed on system table", K(ret));
    } else if (!alter_table_arg.is_inner_
      && (orig_table_schema->is_in_recyclebin() || is_db_in_recyclebin)) {
      ret = OB_ERR_OPERATION_ON_RECYCLE_OBJECT;
      LOG_WARN("can not alter table in recyclebin",
      K(ret), K(alter_table_arg), K(is_db_in_recyclebin));
    } else if (orig_table_schema->is_materialized_view()) {
      bool allow_alter_mview = false;
      if (alter_table_arg.is_alter_indexs_) {
        bool is_alter_pk = false;
        for (int64_t i = 0; OB_SUCC(ret) && (i < alter_table_arg.index_arg_list_.count()); ++i) {
          const ObIndexArg *index_arg = alter_table_arg.index_arg_list_.at(i);
          if (OB_ISNULL(index_arg)) {
            ret = OB_ERR_UNEXPECTED;
            LOG_WARN("index arg is null", KR(ret));
          } else if ((ObIndexArg::ADD_PRIMARY_KEY == index_arg->index_action_type_)
              || (ObIndexArg::DROP_PRIMARY_KEY == index_arg->index_action_type_)
              || (ObIndexArg::ALTER_PRIMARY_KEY == index_arg->index_action_type_)) {
            is_alter_pk = true;
            break;
          }
        }
        if (OB_SUCC(ret) && !is_alter_pk) {
          allow_alter_mview = true;
        }
      } else {
        // only allow for alter tablegroup
        if (!alter_table_arg.is_alter_indexs_ && !alter_table_arg.is_alter_columns_ && !alter_table_arg.is_alter_partitions_ &&
            !alter_table_arg.is_update_global_indexes_ && !alter_table_arg.is_convert_to_character_ && alter_table_arg.is_alter_options_) {
          if (1 == alter_table_arg.alter_table_schema_.alter_option_bitset_.num_members() &&
              alter_table_arg.alter_table_schema_.alter_option_bitset_.has_member(ObAlterTableArg::TABLEGROUP_NAME)) {
            allow_alter_mview = true;
          }
        }
      }
      if (OB_SUCC(ret) && (alter_table_arg.is_alter_mview_attributes_ || alter_table_arg.is_alter_mlog_attributes_)) {
        allow_alter_mview = true;
      }
      if (OB_SUCC(ret) && !allow_alter_mview) {
        ret = OB_NOT_SUPPORTED;
        LOG_WARN("alter materialized view is not supported", KR(ret));
        LOG_USER_ERROR(OB_NOT_SUPPORTED, "alter materialized view is");
      }
    } else if (OB_FAIL(alter_table_arg.is_alter_comment(is_alter_comment))) {
      LOG_WARN("failed to get is alter comment", K(ret));
    } else if (OB_FAIL(GET_MIN_DATA_VERSION(orig_table_schema->get_tenant_id(), compat_version))) {
      LOG_WARN("get min data_version failed", K(ret), K(orig_table_schema->get_tenant_id()));
    } else if (!orig_table_schema->is_user_table()
               && !orig_table_schema->is_sys_table()
               && !(orig_table_schema->is_view_table() && is_alter_comment
                    && ObSQLUtils::is_data_version_ge_422_or_431(compat_version))
               && !orig_table_schema->is_tmp_table()
               && !orig_table_schema->is_external_table()) {
      ret = OB_ERR_WRONG_OBJECT;
      LOG_USER_ERROR(OB_ERR_WRONG_OBJECT,
      to_cstring(origin_database_name), to_cstring(origin_table_name), "BASE TABLE");
    // NOTE: if_offline_ddl is true only in the execution phase of offline ddl, skip check_can_do_ddl check
    } else if (!is_offline_ddl && !orig_table_schema->check_can_do_ddl()) {
      ret = OB_OP_NOT_ALLOW;
      LOG_WARN("offline ddl is being executed, other ddl operations are not allowed", K(ret),
                K(is_offline_ddl), K(*orig_table_schema));
    }
  }
  return ret;
}

/*
 * FIXME:
 * Before minor merge sstable sparse format support, in order to relieve the dependence on gc_snapshot_timestamp,
 * first implement a temporary solution to avoid the operation of modifying columns in the process of
 * merging the standalone cluster.
 * 2.2.x,
 * to primary cluster, Since RS determines the major version first,
 *    and then determine the schema_version used for the merge by ddl, it can be judged
 *    whether the merge has started according to whether the __all_zone version numbers are consistent
 * to standalone, It is impossible to make the DDL of the primary cluster and the merge of standalone library
 *    to be mutually exclusive, so this solution is only to reduce the probability of bad cases.
 *    When the merge checksum is inconsistent in the standalone cluster, it can be corrected by drop replica
 * >= 3.3, allow drop column when doing major freeze
 */
int ObDDLService::check_can_alter_column(
    const int64_t tenant_id,
    const AlterTableSchema &alter_table_schema,
    const ObTableSchema &orig_table_schema)
{
  int ret = OB_SUCCESS;
  const int64_t table_id = orig_table_schema.get_table_id();
  DEBUG_SYNC(BEFORE_ALTER_TABLE_COLUMN);
  if (THIS_WORKER.is_timeout_ts_valid()
      && THIS_WORKER.is_timeout()) {
    ret = OB_TIMEOUT;
    LOG_WARN("already timeout", KR(ret));
  } else if (OB_FAIL(check_restore_point_allow(tenant_id, orig_table_schema))) {
    LOG_WARN("restore point check fail, cannot alter column", K(ret), K(tenant_id), K(table_id));
  } else {
    ObTableSchema::const_column_iterator it_begin = alter_table_schema.column_begin();
    ObTableSchema::const_column_iterator it_end = alter_table_schema.column_end();
    AlterColumnSchema *alter_column_schema = NULL;
    bool need_drop_column = false;
    for(; OB_SUCC(ret) && !need_drop_column && it_begin != it_end; it_begin++) {
      if (OB_ISNULL(*it_begin)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("*it_begin is NULL", K(ret));
      } else {
        alter_column_schema = static_cast<AlterColumnSchema *>(*it_begin);
        if (OB_DDL_DROP_COLUMN == alter_column_schema->alter_type_) {
          need_drop_column = true;
          if (OB_FAIL(check_enable_sys_table_ddl(orig_table_schema, OB_DDL_DROP_COLUMN))) {
            LOG_WARN("fail to check enable sys table ddl", KR(ret), K(orig_table_schema));
          }
        }
      }
    }
  }
  return ret;
}

// If there is a restore point on the table, it is not allowed to create indexes, add or delete columns.
int ObDDLService::check_restore_point_allow(const int64_t tenant_id, const ObTableSchema &table_schema)
{
  int ret = OB_SUCCESS;
  bool is_exist = false;
  ObArray<ObTabletID> tablet_ids;
  if (!is_inner_table(table_schema.get_table_id()) && OB_FAIL(get_snapshot_mgr().check_restore_point(
      get_sql_proxy(), tenant_id, table_schema.get_table_id(), is_exist))) {
    LOG_WARN("failed to check restore point", K(ret), K(tenant_id));
  }
  if (OB_SUCC(ret) && is_exist) {
    ret = OB_OP_NOT_ALLOW;
    LOG_WARN("restore point exist, cannot alter ", K(ret), K(tenant_id), K(table_schema.get_table_id()));
    LOG_USER_ERROR(OB_OP_NOT_ALLOW, "restore point exist, create index/alter");
  }
  return ret;
}


// This code will be used for partition operations of table and tablegroup
// 1. for table, parameter is_drop_truncate_and_alter_index parameter avoids the drop/truncate partition
// of table with global index and the index create in the same alter table statement
// 2. for tablegroup, avoid drop/truncate partition in tablegroup with global index
// (After the tablegroup supports drop/truncate partitions of table with global indexes,
// the behavior will be unified.)
int ObDDLService::check_index_valid_for_alter_partition(
    const share::schema::ObTableSchema &orig_table_schema,
    share::schema::ObSchemaGetterGuard &schema_guard,
    const bool is_drop_truncate_and_alter_index,
    const bool is_split)
{
  int ret = OB_SUCCESS;
  const uint64_t tenant_id = orig_table_schema.get_tenant_id();
  ObSEArray<ObAuxTableMetaInfo, 16> simple_index_infos;
  if (OB_FAIL(orig_table_schema.get_simple_index_infos(simple_index_infos))) {
    LOG_WARN("get_index_tid_array failed", KR(ret));
  } else {
    for (int64_t i = 0; OB_SUCC(ret) && i < simple_index_infos.count(); ++i) {
      const ObTableSchema *index_table_schema = NULL;
      if (OB_FAIL(schema_guard.get_table_schema(
                  tenant_id, simple_index_infos.at(i).table_id_, index_table_schema))) {
        LOG_WARN("get_table_schema failed",
                 "table id", simple_index_infos.at(i).table_id_, KR(ret));
      } else if (OB_ISNULL(index_table_schema)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("table schema should not be null", KR(ret));
      } else if (index_table_schema->is_global_index_table()) {
        if (is_drop_truncate_and_alter_index) {
          ret = OB_NOT_SUPPORTED;
          LOG_WARN("alter index and drop/truncate tables with global index not support", KR(ret),
                   "index_tid", simple_index_infos.at(i).table_id_, K(index_table_schema));
          LOG_USER_ERROR(OB_NOT_SUPPORTED, "alter index and drop/truncate tables with global index");
        }
      } else if (index_table_schema->is_global_local_index_table()) {
        if (is_split) {
          ret = OB_NOT_SUPPORTED;
          LOG_WARN("can not convert a non-partitioned table with non-partitioned global index to a partitioned table", KR(ret),
                   "index_tid", simple_index_infos.at(i).table_id_, K(index_table_schema));
          LOG_USER_ERROR(OB_NOT_SUPPORTED, "converting a non-partitioned table with non-partitioned global index to a partitioned table");
        }
      }
    }
  }
  return ret;
}

int ObDDLService::check_alter_set_interval(const share::schema::ObTableSchema &orig_table_schema,
                                           const obrpc::ObAlterTableArg &alter_table_arg)
{
  int ret = OB_SUCCESS;
  ObPartitionFuncType part_func_type = orig_table_schema.get_part_option().get_part_func_type();
  const AlterTableSchema &alter_table_schema = alter_table_arg.alter_table_schema_;
  ObPartition **part_array = orig_table_schema.get_part_array();
  int64_t part_num = orig_table_schema.get_partition_num();
  if (PARTITION_LEVEL_TWO == orig_table_schema.get_part_level()
      && !orig_table_schema.has_sub_part_template_def()) {
    ret = OB_NOT_SUPPORTED;
    LOG_WARN("interval part of composited-partitioned table not support", K(ret), K(orig_table_schema));
    LOG_USER_ERROR(OB_NOT_SUPPORTED, "interval part of composited-partitioned table");
  } else if (1 != orig_table_schema.get_partition_key_column_num()) {
    ret = OB_OP_NOT_ALLOW;
    LOG_WARN("more than one partition key not support", K(ret), K(orig_table_schema));
    LOG_USER_ERROR(OB_OP_NOT_ALLOW, "more than one partition key");
  } else if (OB_ISNULL(part_array)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("fail to part_array is null", K(orig_table_schema), K(alter_table_schema), KR(ret));
  } else if (PARTITION_FUNC_TYPE_INTERVAL == part_func_type) {
    ret = OB_NOT_SUPPORTED;
    LOG_WARN("change interval range in inteval table not support yet", KR(ret));
    LOG_USER_ERROR(OB_NOT_SUPPORTED, "change interval range in inteval table");
  } else if (PARTITION_FUNC_TYPE_INTERVAL != part_func_type
             && PARTITION_FUNC_TYPE_RANGE_COLUMNS != part_func_type) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("type is unexpected when set interval", K(orig_table_schema), K(alter_table_schema), KR(ret));
  } else if (part_num < 1) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("part num is less 1", K(orig_table_schema), K(alter_table_schema), KR(ret));
  } else if (OB_ISNULL(part_array[part_num - 1])) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("the last part is null", K(orig_table_schema), K(alter_table_schema), KR(ret));
  } else if (OB_FAIL(ObPartitionUtils::check_interval_partition_table(part_array[part_num - 1]->get_high_bound_val(),
                     alter_table_schema.get_interval_range()))) {
    LOG_WARN("fail to check_interval_partition_table", KR(ret));
  }
  return ret;
}
// Check various conditions of partition management
// after 2.0 do not allow separate additions, deletions, and splitting of tables in tablegroup.
int ObDDLService::check_alter_partitions(const ObTableSchema &orig_table_schema,
                                         obrpc::ObAlterTableArg &alter_table_arg)
{
  int ret = OB_SUCCESS;
  const obrpc::ObAlterTableArg::AlterPartitionType alter_part_type = alter_table_arg.alter_part_type_;
  const uint64_t tablegroup_id = orig_table_schema.get_tablegroup_id();
  const uint64_t tenant_id = orig_table_schema.get_tenant_id();
  ObSchemaGetterGuard schema_guard;
  bool is_drop_or_truncate = false;
  bool is_split = false;
  bool is_oracle_mode = false;
  bool has_local_index = false;
  uint64_t compat_version = 0;
  if (OB_FAIL(GET_MIN_DATA_VERSION(tenant_id, compat_version))) {
    LOG_WARN("get min data_version failed", K(ret), K(tenant_id));
  } else if (compat_version < DATA_VERSION_4_2_0_0 && OB_INVALID_ID != tablegroup_id) {
    ret = OB_NOT_SUPPORTED;
    LOG_WARN("can not handle table in tablegroup when observer is upgrading", K(ret), K(tenant_id));
    LOG_USER_ERROR(OB_NOT_SUPPORTED, "Can Not Handle Table In Tablegroup When Observer Is Upgrading");
  } else if (GCONF.in_upgrade_mode()) {
    ret = OB_OP_NOT_ALLOW;
    LOG_WARN("in upgrade, can not do partition maintenance", K(ret));
    LOG_USER_ERROR(OB_OP_NOT_ALLOW, "partition maintenance during upgrade");
  } else if (OB_FAIL(get_tenant_schema_guard_with_version_in_inner_table(tenant_id, schema_guard))) {
    LOG_WARN("fail to get schema guard with version in inner table", K(ret), K(tenant_id));
  } else if (OB_FAIL(ObCompatModeGetter::check_is_oracle_mode_with_tenant_id(tenant_id, is_oracle_mode))) {
    LOG_WARN("fail to check is oracle mode", K(ret));
  } else if (orig_table_schema.is_materialized_view()) {
    ret = OB_NOT_SUPPORTED;
    LOG_WARN("alter partition of materialized view is not supported", KR(ret));
    LOG_USER_ERROR(OB_NOT_SUPPORTED, "alter partition of materialized view is");
  } else if (orig_table_schema.is_interval_part()) {
    if (PARTITION_LEVEL_TWO == orig_table_schema.get_part_level()
        && !orig_table_schema.has_sub_part_template_def()) {
      ret = OB_NOT_SUPPORTED;
      LOG_WARN("interval part of composited-partitioned table not support", K(ret), K(orig_table_schema));
      LOG_USER_ERROR(OB_NOT_SUPPORTED, "interval part of composited-partitioned table");
    }
  } else if (OB_FAIL(orig_table_schema.check_has_local_index(schema_guard, has_local_index))) {
    LOG_WARN("fail to check_has_local_index", K(ret), K(has_local_index));
  } else if (obrpc::ObAlterTableArg::PARTITIONED_TABLE == alter_part_type
             && obrpc::ObAlterTableArg::REPARTITION_TABLE == alter_part_type
             && obrpc::ObAlterTableArg::PARTITIONED_PARTITION == alter_part_type) {
    ret = OB_OP_NOT_ALLOW;
    LOG_WARN("split partition in 4.0 not allowed", K(ret), K(tablegroup_id));
    LOG_USER_ERROR(OB_OP_NOT_ALLOW, "split partition in 4.0");
  }
  bool has_fts_or_multivalue_index = false;
  const int64_t table_id = orig_table_schema.get_table_id();

  if (OB_FAIL(ret)) {
  } else if (obrpc::ObAlterTableArg::TRUNCATE_PARTITION == alter_part_type) {
    bool is_truncate = true;
    if (OB_FAIL(check_alter_drop_partitions(orig_table_schema, alter_table_arg, is_truncate))) {
      LOG_WARN("failed to check truncate partitions", K(ret), K(orig_table_schema), K(alter_table_arg));
    }
    is_drop_or_truncate = true;
  } else if (obrpc::ObAlterTableArg::TRUNCATE_SUB_PARTITION == alter_part_type) {
    if (OB_FAIL(check_alter_drop_subpartitions(orig_table_schema, alter_table_arg))) {
      LOG_WARN("failed to check drop partition", KR(ret), K(orig_table_schema), K(alter_table_arg));
    }
    is_drop_or_truncate = true;
  } else if (obrpc::ObAlterTableArg::DROP_PARTITION == alter_part_type) {
    bool is_truncate = false;
    if (OB_FAIL(check_alter_drop_partitions(orig_table_schema, alter_table_arg, is_truncate))) {
      LOG_WARN("failed to check drop partition", K(ret), K(orig_table_schema), K(alter_table_arg));
    }
    is_drop_or_truncate = true;
  } else if (obrpc::ObAlterTableArg::DROP_SUB_PARTITION == alter_part_type) {
    if (OB_FAIL(check_alter_drop_subpartitions(orig_table_schema, alter_table_arg))) {
      LOG_WARN("failed to check drop partition", K(ret), K(orig_table_schema), K(alter_table_arg));
    }
    is_drop_or_truncate = true;
  } else if (alter_table_arg.is_split_partition()) {
    if (OB_FAIL(check_alter_split_partitions(orig_table_schema, alter_table_arg))) {
      LOG_WARN("failed to check split paritions", K(ret), K(orig_table_schema), K(alter_table_arg));
    }
  } else if (obrpc::ObAlterTableArg::RENAME_PARTITION == alter_part_type) {
    if (OB_FAIL(check_alter_rename_partitions_(orig_table_schema, alter_table_arg))) {
      LOG_WARN("failed to check rename partition", KR(ret), K(orig_table_schema), K(alter_table_arg));
    }
  } else if (obrpc::ObAlterTableArg::RENAME_SUB_PARTITION == alter_part_type) {
    if (OB_FAIL(check_alter_rename_subpartitions_(orig_table_schema, alter_table_arg))) {
      LOG_WARN("failed to check rename subpartition", KR(ret), K(orig_table_schema), K(alter_table_arg));
    }
  } else if (obrpc::ObAlterTableArg::ADD_PARTITION == alter_part_type) {
    if (OB_FAIL(check_alter_add_partitions(orig_table_schema, alter_table_arg))) {
      LOG_WARN("failed to check add paritions", K(ret), K(orig_table_schema), K(alter_table_arg));
    }
  } else if (obrpc::ObAlterTableArg::ADD_SUB_PARTITION == alter_part_type) {
    if (OB_FAIL(check_alter_add_subpartitions(orig_table_schema, alter_table_arg))) {
      LOG_WARN("failed to check add paritions", K(ret), K(orig_table_schema), K(alter_table_arg));
    }
  } else if (is_oracle_mode && obrpc::ObAlterTableArg::SET_INTERVAL == alter_part_type) {
    if (OB_FAIL(check_alter_set_interval(orig_table_schema, alter_table_arg))) {
      LOG_WARN("failed to check set interval", K(ret), K(orig_table_schema), K(alter_table_arg));
    }
  } else if (is_oracle_mode && obrpc::ObAlterTableArg::INTERVAL_TO_RANGE == alter_part_type) {
    if (PARTITION_FUNC_TYPE_INTERVAL != orig_table_schema.get_part_option().get_part_func_type()) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("type is unexpected when interval to range", K(orig_table_schema), K(alter_table_arg), KR(ret));
    }
  } else {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("unexpected alter partition type", K(ret), K(alter_part_type));
  }
  if (OB_SUCC(ret)) {
    if (OB_FAIL(check_index_valid_for_alter_partition(orig_table_schema,
                                                      schema_guard,
                                                      is_drop_or_truncate && alter_table_arg.index_arg_list_.size() != 0,
                                                      is_split))) {
      LOG_WARN("failed to check index valid", K(ret), K(is_split), K(is_drop_or_truncate), K(orig_table_schema));
    }
  }
  return ret;
}

int ObDDLService::check_table_pk(const share::schema::ObTableSchema &orig_table_schema)
{
  int ret = OB_SUCCESS;
    // Check whether table is the parent table of the foreign key constraint
    // check whether it is partition table
  if (orig_table_schema.is_parent_table()) {
    // check whether it is self-referential
    if (orig_table_schema.is_child_table()) {
      const ObIArray<ObForeignKeyInfo> &foreign_key_infos = orig_table_schema.get_foreign_key_infos();
      FOREACH_CNT_X(foreign_key_info, foreign_key_infos, OB_SUCC(ret)) {
        if (orig_table_schema.get_table_id() != foreign_key_info->child_table_id_) {
          // If it is not self-referential, there are constraints
          ret = OB_ERR_ATLER_TABLE_ILLEGAL_FK;
          LOG_USER_ERROR(OB_ERR_ATLER_TABLE_ILLEGAL_FK);
        }
      }
    } else {
      // If it is not self-referential, there are constraints
      const ObIArray<ObForeignKeyInfo> &foreign_key_infos = orig_table_schema.get_foreign_key_infos();
      FOREACH_CNT_X(foreign_key_info, foreign_key_infos, OB_SUCC(ret)) {
        if (orig_table_schema.get_table_id() == foreign_key_info->parent_table_id_) {
          ret = OB_ERR_ATLER_TABLE_ILLEGAL_FK;
          LOG_USER_ERROR(OB_ERR_ATLER_TABLE_ILLEGAL_FK);
        }
      }
    }
  }
  return ret;
}

// rename partition
//1. is partition table
//2. ensure that origin partition name exist and new partition name not exist
//3. currently this partition is not splitting

int ObDDLService::check_alter_rename_partitions_(const share::schema::ObTableSchema &orig_table_schema,
                                                const obrpc::ObAlterTableArg &alter_table_arg)
{
  int ret = OB_SUCCESS;
  uint64_t tenant_data_version = 0;
  const ObPartitionLevel part_level = orig_table_schema.get_part_level();
  const AlterTableSchema &alter_table_schema = alter_table_arg.alter_table_schema_;
  const int64_t part_num = alter_table_schema.get_partition_num();
  ObPartition **part_array = alter_table_schema.get_part_array();
  ObPartition *inc_part = nullptr;
  ObCheckPartitionMode check_partition_mode = CHECK_PARTITION_MODE_NORMAL;
  const ObPartition *part = nullptr;
  if (OB_FAIL(GET_MIN_DATA_VERSION(orig_table_schema.get_tenant_id(), tenant_data_version))) {
    LOG_WARN("get data version failed", KR(ret), K(orig_table_schema.get_tenant_id()));
  } else if (tenant_data_version < DATA_VERSION_4_2_1_0) {
    ret = OB_NOT_SUPPORTED;
    LOG_WARN("cluster version and feature mismatch", KR(ret));
  } else if (!orig_table_schema.is_user_table()) {
    ret = OB_NOT_SUPPORTED;
    LOG_WARN("unsupport behavior on not user table", KR(ret), K(orig_table_schema));
  } else if (PARTITION_LEVEL_ZERO == part_level) {
    ret = OB_ERR_PARTITION_MGMT_ON_NONPARTITIONED;
    LOG_USER_ERROR(OB_ERR_PARTITION_MGMT_ON_NONPARTITIONED);
    LOG_WARN("unsupport management on non_partition table", KR(ret));
  } else if (OB_ISNULL(part_array)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("part_array is null", KR(ret), KP(part_array));
  } else if (OB_UNLIKELY(1 != part_num)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("rename multi part at a time not support", KR(ret), K(part_num));
  } else if (OB_ISNULL(inc_part = part_array[0])) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("inc part is null", KR(ret), KP(part_array), K(alter_table_schema));
  } else {
    const ObString &origin_partition_name = inc_part->get_part_name();
    const ObString &new_partition_name = alter_table_schema.get_new_part_name();
    if (OB_UNLIKELY(ObCharset::case_insensitive_equal(origin_partition_name, new_partition_name))) {
      ret = OB_ERR_RENAME_PARTITION_NAME_DUPLICATE;
      LOG_USER_ERROR(OB_ERR_RENAME_PARTITION_NAME_DUPLICATE, new_partition_name.length(), new_partition_name.ptr());
      LOG_WARN("origin part name equal to new part name", KR(ret), K(origin_partition_name), K(new_partition_name));
    } else if (OB_FAIL(orig_table_schema.check_partition_duplicate_with_name(new_partition_name))) {
      if (OB_DUPLICATE_OBJECT_NAME_EXIST == ret) {
        ret = OB_ERR_RENAME_PARTITION_NAME_DUPLICATE;
        LOG_USER_ERROR(OB_ERR_RENAME_PARTITION_NAME_DUPLICATE, new_partition_name.length(), new_partition_name.ptr());
        LOG_WARN("new part name duplicate with existed partition", KR(ret), K(new_partition_name));
      } else {
        LOG_WARN("check new part name duplicate failed", KR(ret), K(new_partition_name));
      }
    } else if (OB_FAIL(orig_table_schema.get_partition_by_name(origin_partition_name, part))) {
      LOG_WARN("get part by name failed", KR(ret), K(origin_partition_name), K(orig_table_schema));
    }
  }
  return ret;
}

// rename subpartition
//1. is subpartition table
//2. ensure that origin subpartition name exist and new subpartition name not exist
//3. currently this partition is not splitting

int ObDDLService::check_alter_rename_subpartitions_(const share::schema::ObTableSchema &orig_table_schema,
                                                  const obrpc::ObAlterTableArg &alter_table_arg)
{
  int ret = OB_SUCCESS;
  uint64_t tenant_data_version = 0;
  const ObPartitionLevel part_level = orig_table_schema.get_part_level();
  const AlterTableSchema &alter_table_schema = alter_table_arg.alter_table_schema_;
  const int64_t part_num = alter_table_schema.get_partition_num();
  ObPartition **part_array = alter_table_schema.get_part_array();
  ObPartition *inc_part = nullptr;
  const ObPartition *part = nullptr;
  const ObSubPartition *subpart = nullptr;
  if (OB_FAIL(GET_MIN_DATA_VERSION(orig_table_schema.get_tenant_id(), tenant_data_version))) {
    LOG_WARN("get data version failed", KR(ret), K(orig_table_schema.get_tenant_id()));
  } else if (tenant_data_version < DATA_VERSION_4_2_1_0) {
    ret = OB_NOT_SUPPORTED;
    LOG_WARN("cluster version and feature mismatch", KR(ret));
  } else if (!orig_table_schema.is_user_table()) {
    ret = OB_NOT_SUPPORTED;
    LOG_WARN("unsupport behavior on not user table", KR(ret), K(orig_table_schema));
  } else if (PARTITION_LEVEL_ZERO == part_level) {
    ret = OB_ERR_PARTITION_MGMT_ON_NONPARTITIONED;
    LOG_USER_ERROR(OB_ERR_PARTITION_MGMT_ON_NONPARTITIONED);
    LOG_WARN("unsupport management on not partition table", KR(ret));
  } else if (PARTITION_LEVEL_ONE == part_level) {
    ret = OB_ERR_NOT_COMPOSITE_PARTITION;
    LOG_USER_ERROR(OB_ERR_NOT_COMPOSITE_PARTITION);
    LOG_WARN("unsupport management on not composite partition table", KR(ret));
  } else if (OB_ISNULL(part_array)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("part_array is null", KR(ret), K(alter_table_schema));
  } else if (OB_UNLIKELY(1 != part_num)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("rename subpart on multi part at a time not support", KR(ret), K(part_num));
  } else if (OB_ISNULL(inc_part = part_array[0])) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("inc part is null", KR(ret), KP(part_array));
  } else {
    ObSubPartition **subpart_array = inc_part->get_subpart_array();
    ObSubPartition *inc_subpart = nullptr;
    const int64_t subpart_num = inc_part->get_subpartition_num();
    if (OB_ISNULL(subpart_array)) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("subpart_array is null", KR(ret));
    } else if (OB_UNLIKELY(1 != subpart_num)) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("rename multi subpart at a time not support", KR(ret), K(part_num));
    } else if (OB_ISNULL(inc_subpart = subpart_array[0])) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("inc_subpart is null", KR(ret), KP(subpart_array));
    } else {
      const ObString &origin_partition_name = inc_subpart->get_part_name();
      const ObString &new_partition_name = alter_table_schema.get_new_part_name();
      if (OB_UNLIKELY(ObCharset::case_insensitive_equal(origin_partition_name, new_partition_name))) {
        ret = OB_ERR_RENAME_SUBPARTITION_NAME_DUPLICATE;
        LOG_USER_ERROR(OB_ERR_RENAME_SUBPARTITION_NAME_DUPLICATE, new_partition_name.length(), new_partition_name.ptr());
        LOG_WARN("origin subpart name equal to new subpart name", KR(ret), K(origin_partition_name), K(new_partition_name));
      } else if (OB_FAIL(orig_table_schema.check_partition_duplicate_with_name(new_partition_name))) {
        if (OB_DUPLICATE_OBJECT_NAME_EXIST == ret) {
          ret = OB_ERR_RENAME_SUBPARTITION_NAME_DUPLICATE;
          LOG_USER_ERROR(OB_ERR_RENAME_SUBPARTITION_NAME_DUPLICATE, new_partition_name.length(), new_partition_name.ptr());
          LOG_WARN("new subpart name duplicate with existed partition", KR(ret), K(new_partition_name));
        } else {
          LOG_WARN("check new subpart name duplicate failed", KR(ret), K(new_partition_name));
        }
      } else if (OB_FAIL(orig_table_schema.get_subpartition_by_name(origin_partition_name, part, subpart))) {
        LOG_WARN("get subpart by name failed", KR(ret), K(origin_partition_name), K(orig_table_schema));
      }
    }
  }
  return ret;
}
// drop or truncate partition
//1. is partition table
//2. Cannot drop all partitions, but truncate does not have this restriction
//3. Ensure that all operating partitions exist1
//4. Currently this partition is not splitting
//5. The partition type can only be list or range
int ObDDLService::check_alter_drop_partitions(const share::schema::ObTableSchema &orig_table_schema,
                                              const obrpc::ObAlterTableArg &alter_table_arg,
                                              const bool is_truncate)
{
  int ret = OB_SUCCESS;
  const AlterTableSchema &alter_table_schema = alter_table_arg.alter_table_schema_;
  const int64_t part_num = alter_table_schema.get_part_option().get_part_num();
  ObPartition **part_array = alter_table_schema.get_part_array();
  const ObPartitionLevel part_level = orig_table_schema.get_part_level();
  const ObPartitionOption &part_option = orig_table_schema.get_part_option();
  if (OB_ISNULL(part_array)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("part_array is null", K(ret), K(part_array));
  } else if (PARTITION_LEVEL_ZERO == part_level) {
    ret = OB_ERR_PARTITION_MGMT_ON_NONPARTITIONED;
    LOG_WARN("unsupport management on non-partition table", K(ret));
  } else if (!part_option.is_range_part() && !part_option.is_list_part()) {
    ret = OB_ERR_ONLY_ON_RANGE_LIST_PARTITION;
    LOG_WARN("drop partition can only be used on RANGE/LIST partitions", K(ret), K(alter_table_arg));
  } else if (OB_FAIL(check_table_pk(orig_table_schema))) {
    LOG_WARN("cannot drop/truncate partition with foreign keys", K(ret), K(alter_table_arg));
  } else if (is_truncate) {
  } else if (alter_table_schema.get_part_option().get_part_num() >=
              orig_table_schema.get_part_option().get_part_num()) {
    ret = OB_ERR_DROP_LAST_PARTITION;
    LOG_WARN("cannot drop all partitions",
            "partitions current", orig_table_schema.get_part_option().get_part_num(),
            "partitions to be dropped", alter_table_schema.get_part_option().get_part_num(),
            K(ret));
    LOG_USER_ERROR(OB_ERR_DROP_LAST_PARTITION);
  }
  if (OB_SUCC(ret)) {
    for (int64_t i = 0; OB_SUCC(ret) && i < part_num; ++i) {
      ObCheckPartitionMode check_partition_mode = CHECK_PARTITION_MODE_NORMAL;
      ObPartIterator iter(orig_table_schema, check_partition_mode);
      const ObPartition *part = NULL;
      while (OB_SUCC(ret) && OB_SUCC(iter.next(part))) {
        if (OB_ISNULL(part)) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("NULL ptr", K(part), K(ret));
        } else if (ObCharset::case_insensitive_equal(part->get_part_name(),
                                                     part_array[i]->get_part_name())) {
          break;
        }
      }
      if (OB_FAIL(ret)) {
        if (OB_ITER_END != ret) {
          LOG_WARN("iter failed", K(ret));
        } else {
          ret = OB_ERR_DROP_PARTITION_NON_EXISTENT;
          LOG_WARN("partition to be dropped not exist", K(ret), "partition name", part_array[i]->get_part_name());
          LOG_USER_ERROR(OB_ERR_DROP_PARTITION_NON_EXISTENT);
        }
      }
    }
  }
  return ret;
}

int ObDDLService::check_alter_drop_subpartitions(const share::schema::ObTableSchema &orig_table_schema,
                                                 const obrpc::ObAlterTableArg &alter_table_arg)
{
  int ret = OB_SUCCESS;
  const ObPartitionLevel part_level = orig_table_schema.get_part_level();
  const ObPartitionOption &subpart_option = orig_table_schema.get_sub_part_option();
  if (PARTITION_LEVEL_ZERO == part_level || PARTITION_LEVEL_ONE == part_level) {
    ret = OB_ERR_PARTITION_MGMT_ON_NONPARTITIONED;
    LOG_WARN("unsupport management on non-partition table", K(ret));
  } else if (!subpart_option.is_range_part() && !subpart_option.is_list_part()) {
    ret = OB_ERR_ONLY_ON_RANGE_LIST_PARTITION;
    LOG_WARN("drop partition can only be used on RANGE/LIST partitions", K(ret), K(alter_table_arg));
  } else if (OB_FAIL(check_table_pk(orig_table_schema))) {
    LOG_WARN("cannot drop/truncate partition with foreign keys", K(ret), K(alter_table_arg));
  }
  return ret;
}

// filter out the partition which is same to orig_table_schema in alter_table_arg
int ObDDLService::filter_out_duplicate_interval_part(const share::schema::ObTableSchema &orig_table_schema,
                                                     share::schema::ObTableSchema &alter_table_schema)
{
  int ret = OB_SUCCESS;
  int64_t j = 0;
  int64_t inc_num = 0;
  const ObRowkey *rowkey_orig= NULL;
  const int64_t inc_part_num = alter_table_schema.get_part_option().get_part_num();
  const int64_t orig_part_num = orig_table_schema.get_part_option().get_part_num();
  ObPartition **inc_part_array = alter_table_schema.get_part_array();
  ObPartition **orig_part_array = orig_table_schema.get_part_array();
  if (!orig_table_schema.is_interval_part()) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("orig_table_schema is not interval part", K(ret), K(orig_table_schema), K(alter_table_schema));
  } else if (OB_ISNULL(inc_part_array)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("inc_part_array is null", K(ret), K(orig_table_schema), K(alter_table_schema));
  } else if (OB_ISNULL(orig_part_array)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("orig_part_array is null", K(ret), K(orig_table_schema), K(alter_table_schema));
  } else if (orig_table_schema.get_interval_range() != alter_table_schema.get_interval_range()
      || orig_table_schema.get_transition_point() != alter_table_schema.get_transition_point()) {
    ret = OB_ERR_INTERVAL_PARTITION_ERROR;
    LOG_WARN("interval_range or transition_point is changed", KR(ret), K(orig_table_schema), K(alter_table_schema));
  }
  for (int64_t i = 0; OB_SUCC(ret) && i < inc_part_num; ++i) {
    ObString empty_str;
    const ObRowkey *rowkey_cur = NULL;
    if (OB_ISNULL(inc_part_array[i])) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("ptr is null", K(ret), K(orig_table_schema), K(alter_table_schema));
    }
    // interval part name is generated in rs, so reset.
    else if (OB_FAIL(inc_part_array[i]->set_part_name(empty_str))) {
      LOG_WARN("fail to set_part_name", KR(ret), K(orig_table_schema), K(alter_table_schema));
    } else if (NULL == (rowkey_cur = &inc_part_array[i]->get_high_bound_val())) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("ptr is null", K(ret), K(orig_table_schema), K(alter_table_schema));
    }
    while (OB_SUCC(ret) && j < orig_part_num) {
      if (OB_ISNULL(orig_part_array[j])) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("ptr is null", K(ret), K(orig_table_schema), K(alter_table_schema));
      } else if (NULL == (rowkey_orig = &orig_part_array[j]->get_high_bound_val())) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("ptr is null", K(ret), K(orig_table_schema), K(alter_table_schema));
      } else if (*rowkey_orig < *rowkey_cur) {
        j++;
      } else {
        break;
      }
    }
    if (OB_FAIL(ret)) {
    } else if (*rowkey_orig != *rowkey_cur) {
      if (inc_num != i) {
        inc_part_array[inc_num] = inc_part_array[i];
      }
      inc_num++;
    }
  }
  if (OB_FAIL(ret)) {
  } else if (0 == inc_num) {
    LOG_INFO("all interval part for add is exist", K(alter_table_schema), K(orig_table_schema));
    ret = OB_ERR_INTERVAL_PARTITION_EXIST;
  } else if (inc_num != inc_part_num) {
    alter_table_schema.set_part_num(inc_num);
    alter_table_schema.set_partition_num(inc_num);
  }
  return ret;
}

// add partition
//1. ensure it is partition table
//2. The number of new partitions cannot exceed the limit of the number of partitions
//3. The partition name of the newly added partition cannot conflict with the previous partition
//4. Partition type, currently only supports list or range partition
//5. The value of the newly added partition is consistent with the previous one
int ObDDLService::check_alter_add_partitions(const share::schema::ObTableSchema &orig_table_schema,
                                             obrpc::ObAlterTableArg &alter_table_arg)
{
  int ret = OB_SUCCESS;
  AlterTableSchema &alter_table_schema = alter_table_arg.alter_table_schema_;
  const int64_t inc_part_num = alter_table_schema.get_part_option().get_part_num();
  const int64_t orig_part_num = orig_table_schema.get_part_option().get_part_num();
  ObPartition **inc_part_array = alter_table_schema.get_part_array();
  ObPartition **orig_part_array = orig_table_schema.get_part_array();
  const ObPartitionLevel part_level = orig_table_schema.get_part_level();
  const ObPartitionOption &part_option = orig_table_schema.get_part_option();
  bool is_oracle_mode = false;
  if (OB_ISNULL(inc_part_array)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("inc_part_array is null", K(ret), K(orig_table_schema), K(alter_table_schema));
  } else if (OB_ISNULL(orig_part_array)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("orig_part_array is null", K(ret), K(orig_table_schema), K(alter_table_schema));
  } else if (OB_FAIL(orig_table_schema.check_if_oracle_compat_mode(is_oracle_mode))) {
    LOG_WARN("fail to check is oracle mode", K(ret));
  } else if (PARTITION_LEVEL_ZERO == part_level) {
    ret = OB_ERR_PARTITION_MGMT_ON_NONPARTITIONED;
    LOG_WARN("unsupport management on non-partition table", K(ret));
  } else if (!part_option.is_range_part() && !part_option.is_list_part()) {
    ret = OB_ERR_ONLY_ON_RANGE_LIST_PARTITION;
    LOG_WARN("add partition can only be used on RANGE/LIST partitions", K(ret), K(alter_table_arg));
  } else if ((is_oracle_mode && OB_MAX_PARTITION_NUM_ORACLE < orig_table_schema.get_all_part_num() + inc_part_num)
             || (!is_oracle_mode
                   && ObResolverUtils::get_mysql_max_partition_num(orig_table_schema.get_tenant_id())
                       < orig_table_schema.get_all_part_num() + inc_part_num)) {
    ret = OB_TOO_MANY_PARTITIONS_ERROR;
    LOG_WARN("too many partitions", K(ret),
             "partition cnt current", orig_table_schema.get_all_part_num(),
             "partition cnt to be added", inc_part_num);
  }
  if (!orig_table_schema.is_interval_part()) {
    for (int64_t i = 0; OB_SUCC(ret) && i < inc_part_num; ++i) {
      ObCheckPartitionMode check_partition_mode = CHECK_PARTITION_MODE_NORMAL;
      ObPartIterator iter(orig_table_schema, check_partition_mode);
      const ObPartition *part = NULL;
      ObPartition *inc_part = inc_part_array[i];
      while (OB_SUCC(ret) && OB_SUCC(iter.next(part))) {
        if (OB_ISNULL(part) || OB_ISNULL(inc_part)) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("NULL ptr", K(part), K(ret));
        } else if (ObCharset::case_insensitive_equal(part->get_part_name(),
                                                     inc_part->get_part_name())) {
          ret = OB_ERR_SAME_NAME_PARTITION;
          LOG_WARN("duplicate partition name", K(ret), K(inc_part_array[i]->get_part_name()));
          LOG_USER_ERROR(OB_ERR_SAME_NAME_PARTITION, inc_part_array[i]->get_part_name().length(),
                         inc_part_array[i]->get_part_name().ptr());
        } else if (PARTITION_LEVEL_TWO == orig_table_schema.get_part_level()) {
          if (0 == inc_part->get_subpartition_num() || OB_ISNULL(inc_part->get_subpart_array())) {
            ret = OB_ERR_UNEXPECTED;
            LOG_WARN("subpart_array is NULL", K(part), K(ret), K(i));
          } else {
            for (int j = 0; OB_SUCC(ret) && j < inc_part->get_subpartition_num(); j++) {
              ObSubPartition *subpart = NULL;
              ObSubPartition *inc_subpart = inc_part->get_subpart_array()[j];
              int64_t k = 0, subpart_num = part->get_subpartition_num();
              for (k = 0; OB_SUCC(ret) && k < subpart_num; k++) {
                subpart = part->get_subpart_array()[k];
                if (OB_ISNULL(subpart) || OB_ISNULL(inc_subpart)) {
                  ret = OB_ERR_UNEXPECTED;
                  LOG_WARN("NULL ptr", K(part), K(inc_subpart), K(ret));
                } else if (ObCharset::case_insensitive_equal(subpart->get_part_name(),
                                                             inc_subpart->get_part_name())) {
                  ret = OB_ERR_SAME_NAME_SUBPARTITION;
                  LOG_WARN("duplicate subpartition name", K(ret), K(subpart->get_part_name()));
                  LOG_USER_ERROR(OB_ERR_SAME_NAME_SUBPARTITION, subpart->get_part_name().length(),
                                 subpart->get_part_name().ptr());
                }
              }
            }
          }
        }
      }
      if (OB_ITER_END == ret) {
        ret = OB_SUCCESS;
      }
    }// end for
  }

  // check the part of inc part is increased
  if (OB_FAIL(ret)) {
  } else if (orig_table_schema.is_range_part()) {
    const ObRowkey *rowkey_last = NULL;
    if (orig_table_schema.is_interval_part()) {
      rowkey_last = &orig_table_schema.get_transition_point();
    } else if (OB_ISNULL(orig_part_array[orig_part_num - 1])) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("ptr is null", K(ret), K(orig_part_array), K(alter_table_schema));
    } else {
      rowkey_last = &orig_part_array[orig_part_num - 1]->get_high_bound_val();
    }
    if (OB_FAIL(ret)) {
    } else if (OB_ISNULL(rowkey_last)) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("ptr is null", K(ret), K(orig_part_array), K(alter_table_schema));
    }
    for (int64_t i = 0; OB_SUCC(ret) && i < inc_part_num; ++i) {
      const ObRowkey *rowkey_cur = &inc_part_array[i]->get_high_bound_val();

      if (*rowkey_cur <= *rowkey_last) {
        ret = OB_ERR_ADD_PART_BOUN_NOT_INC;
        LOG_WARN("range values should increasing", K(ret), K(rowkey_cur), K(rowkey_last));
        LOG_USER_ERROR(OB_ERR_ADD_PART_BOUN_NOT_INC);
      } else {
        rowkey_last = rowkey_cur;
      }
    }

    if (OB_SUCC(ret) && orig_table_schema.is_interval_part()
        && filter_out_duplicate_interval_part(orig_table_schema, alter_table_schema)) {
      LOG_WARN("fail to filter out duplicate interval part", KR(ret), K(orig_table_schema), K(alter_table_schema));
    }
  } else if (orig_table_schema.is_list_part()) {
    if (OB_FAIL(check_add_list_partition(orig_table_schema, alter_table_schema))) {
      LOG_WARN("failed to check add list partition", K(ret), K(orig_table_schema), K(alter_table_schema));
    }
  }
  return ret;
}

int ObDDLService::check_alter_split_partitions(const share::schema::ObTableSchema &orig_table_schema,
                                               obrpc::ObAlterTableArg &alter_table_arg)
{
  int ret = OB_SUCCESS;
  AlterTableSchema &alter_table_schema = alter_table_arg.alter_table_schema_;
  const uint64_t tenant_id = orig_table_schema.get_tenant_id();
  ObPartitionLevel target_part_level = alter_table_arg.is_auto_split_partition() ?
                                       orig_table_schema.get_target_part_level_for_auto_partitioned_table() :
                                       orig_table_schema.get_part_level();

  if (alter_table_arg.is_auto_split_partition() && !orig_table_schema.is_auto_partitioned_table()) {
    // the table might be disabled auto-partition after trigger auto splitting partition
    ret = OB_OP_NOT_ALLOW;
    LOG_WARN("attempt to auto split partition for non-auto-partitioned table", KR(ret),
                                                                               K(alter_table_arg),
                                                                               K(orig_table_schema));
  } else if (OB_FAIL(orig_table_schema.check_enable_split_partition(alter_table_arg.is_auto_split_partition()))) {
    LOG_WARN("fail to check enable split partition", KR(ret), K(orig_table_schema));
  } else if (OB_FAIL(check_split_partition_val_(orig_table_schema, alter_table_schema,
                                                target_part_level, alter_table_arg.alter_part_type_))) {
    LOG_WARN("fail to check split partition val", KR(ret), K(orig_table_schema), K(alter_table_arg));
  } else if (OB_FAIL(check_split_partition_name_(orig_table_schema, alter_table_schema, target_part_level))) {
    LOG_WARN("fail to check split partition name", KR(ret), K(orig_table_schema), K(alter_table_arg));
  }

  return ret;
}

int ObDDLService::check_split_partition_val_(const share::schema::ObTableSchema &orig_table_schema,
                                             const AlterTableSchema &alter_table_schema,
                                             const ObPartitionLevel target_part_level,
                                             const obrpc::ObAlterTableArg::AlterPartitionType type)
{
  int ret = OB_SUCCESS;
  const ObPartitionLevel ori_part_level = orig_table_schema.get_part_level();
  ObPartition **split_part_array = alter_table_schema.get_part_array();
  const int64_t split_part_num = alter_table_schema.get_partition_num();

  if (OB_ISNULL(split_part_array)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("split_part_array is null", KR(ret), K(orig_table_schema), K(alter_table_schema));
  } else if (PARTITION_LEVEL_ONE == target_part_level) {
    if (PARTITION_LEVEL_ZERO == ori_part_level) {
      // only auto partitioning could run here, user could not manual split non-partitioned table
      if (obrpc::ObAlterTableArg::AUTO_SPLIT_PARTITION != type) {
        ret = OB_OP_NOT_ALLOW;
        LOG_WARN("invalid operate", KR(ret), K(orig_table_schema), K(alter_table_schema), K(type));
      } else if (OB_FAIL(check_split_partitions_from_same_source_(split_part_array, split_part_num,
                                                                  0, split_part_num,
                                                                  orig_table_schema,
                                                                  target_part_level,
                                                                  type))) {
        LOG_WARN("fail to check split partitions from same source partition", KR(ret),
                                                                              K(target_part_level),
                                                                              K(orig_table_schema),
                                                                              K(alter_table_schema));
      }
    } else if (PARTITION_LEVEL_ONE == ori_part_level) {
      // in future, we need to support to split multiple partitions, which means the split_part_array
      // might include various set of split partitions from different source partitions.
      // thus, we need to:
      // 1. divide split partitions into different sets based on split_source_tablet_id
      // 2. check whether each set is valid.
      // moreover, we assume that the split partitions from same source is continued in the split_part_array.
      common::hash::ObHashSet<uint64_t> traversed_source_ids;

      if (OB_ISNULL(split_part_array[0])) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("NULL ptr", KR(ret));
      } else if (OB_FAIL(traversed_source_ids.create(hash::cal_next_prime(split_part_num)))) {
        LOG_WARN("fail to create hashset", KR(ret));
      } else if (OB_FAIL(traversed_source_ids.set_refactored(split_part_array[0]->get_split_source_tablet_id().id(), 0 /*flag*/))) {
        LOG_WARN("fail to set tablet_id", KR(ret), KPC(split_part_array[0]));
      } else {
        int64_t start_idx = 0; // the first idx of a set of same source splitting partitions
        bool traverse_same_source_over = false; // traverse a set of same source splitting partitions over

        for (int64_t i = 0; OB_SUCC(ret) && i < split_part_num; ++i) {
          ObPartition *split_part = split_part_array[i];
          if (!split_part->get_split_source_tablet_id().is_valid()) {
            ret = OB_ERR_UNEXPECTED;
            LOG_WARN("invalid source_tablet_id", KR(ret), KPC(split_part));
          } else if (i == split_part_num - 1) {
            traverse_same_source_over = true;
          } else if (OB_ISNULL(split_part_array[i+1])) {
            ret = OB_ERR_UNEXPECTED;
            LOG_WARN("NULL ptr", KR(ret), K(alter_table_schema));
          } else if (split_part_array[i+1]->get_split_source_tablet_id() != split_part->get_split_source_tablet_id()) {
            traverse_same_source_over = true;
            // TODO:
            // we don't support to split multiple source partitions from same table in the meanwhile now.
            // when we support the online ddl lock, we need to delete the following error code
            ret = OB_NOT_SUPPORTED;
            LOG_WARN("not support to split multiple source partitions", KR(ret), K(alter_table_schema));
          }

          if (OB_FAIL(ret)) {
          } else if (traverse_same_source_over)  {
            if (OB_FAIL(check_split_partitions_from_same_source_(split_part_array, split_part_num,
                                                                 start_idx, i+1,
                                                                 orig_table_schema,
                                                                 target_part_level,
                                                                 type))) {
              LOG_WARN("fail to check split partitions from same source partition", KR(ret),
                                                                                    K(target_part_level),
                                                                                    K(orig_table_schema),
                                                                                    K(alter_table_schema));
            } else if (i != split_part_num - 1) {
              ObTabletID curr_source_tablet_id = split_part_array[i+1]->get_split_source_tablet_id();
              start_idx = i + 1;
              traverse_same_source_over = false;
              if (OB_FAIL(traversed_source_ids.set_refactored(curr_source_tablet_id.id(), 0 /*flag*/))) {
                // it means the split parts from same source part are not continuous in array
                if (ret == OB_HASH_EXIST) {
                  LOG_WARN("split part array should be sorted by source_tablet_id", KR(ret), K(curr_source_tablet_id.id()),
                                                                                    K(alter_table_schema));
                } else {
                  LOG_WARN("fail to set tablet_id", KR(ret), K(curr_source_tablet_id));
                }
              }
            }
          } // end if (traverse_same_source_over)
        } // end for
      }
    } else {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("invalid part level", KR(ret), K(ori_part_level), K(target_part_level),
                                     K(alter_table_schema), K(orig_table_schema));
    }
  } else if (PARTITION_LEVEL_TWO == target_part_level) {
    ret = OB_NOT_SUPPORTED;
    LOG_WARN("not support to split subpartition", KR(ret), K(alter_table_schema), K(orig_table_schema));
  } else {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("invalid part level", KR(ret), K(ori_part_level),
                                   K(alter_table_schema), K(orig_table_schema));
  }

  return ret;
}

// 1. the number of a set of split partition must be between [2, SPLIT_NUM_LIMIT]
// 2. source_tablet_id must exist in origin table schema
// 3. split partition array should be sorted by high_bound_val
// 4. if the table is range partition, the high_bound_val range of split partitions should be equal to that of source partition
int ObDDLService::check_split_partitions_from_same_source_(ObPartition **split_part_array,
                                                           const int64_t part_array_size,
                                                           const int64_t start, const int64_t end,
                                                           const share::schema::ObTableSchema &orig_table_schema,
                                                           const ObPartitionLevel target_part_level,
                                                           const obrpc::ObAlterTableArg::AlterPartitionType type)
{
  int ret = OB_SUCCESS;
  const ObPartitionLevel ori_part_level = orig_table_schema.get_part_level();
  const ObPartitionOption &part_option = orig_table_schema.get_part_option();
  ObPartition **orig_part_array = orig_table_schema.get_part_array();
  const int64_t orig_part_num = orig_table_schema.get_partition_num();
  const int64_t split_part_num = end - start;
  static const int64_t MANUAL_SPLIT_NUM_LIMIT = INT64_MAX; // not limit the number of manual splitting partition
  static const int64_t AUTO_SPLIT_NUM_LIMIT = 2;
  const int64_t SPLIT_NUM_LIMIT = obrpc::ObAlterTableArg::AUTO_SPLIT_PARTITION == type ?
                                  AUTO_SPLIT_NUM_LIMIT : MANUAL_SPLIT_NUM_LIMIT;

  if (end > part_array_size || end <= start || start < 0) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("invalid interval", KR(ret), K(start), K(end), K(part_array_size));
  } else if (split_part_num < 2) {
    ret = OB_ERR_SPLIT_INTO_ONE_PARTITION;
    LOG_WARN("split partition number is invalid", KR(ret), K(split_part_num));
  } else if (split_part_num > SPLIT_NUM_LIMIT) {
    ret = OB_NOT_SUPPORTED;
    LOG_WARN("invalid splitting partition number", KR(ret), K(split_part_num), K(SPLIT_NUM_LIMIT), K(type));
    char buffer[number::ObNumber::MAX_PRINTABLE_SIZE] = {0};
    (void)snprintf(buffer, sizeof(buffer), "%s splitting partition number more than %ld is",
                                            obrpc::ObAlterTableArg::AUTO_SPLIT_PARTITION == type ? "auto" : "manual",
                                            SPLIT_NUM_LIMIT);
    LOG_USER_ERROR(OB_NOT_SUPPORTED, buffer);
  } else if (orig_table_schema.is_partitioned_table()
              && !orig_table_schema.is_valid_split_part_type()) {
    ret = OB_NOT_SUPPORTED;
    LOG_WARN("only support to split range part", KR(ret), K(orig_table_schema));
  } else {
    // check sorting by value
    for (int64_t i = start; OB_SUCC(ret) && i < end; ++i) {
      ObPartition *split_part = split_part_array[i];
      if (OB_ISNULL(split_part)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("NULL ptr", KR(ret));
      } else if (i > 0 && !ObBasePartition::range_like_func_less_than(split_part_array[i-1], split_part)) {
        ret = OB_ERR_RANGE_NOT_INCREASING_ERROR;
        LOG_WARN("range values should increasing", KR(ret), K(split_part_array[i-1]->get_high_bound_val()),
                                                   K(split_part_array[i]->get_high_bound_val()));
      }
    }
  }

  // check existence of source_tablet_id and value validity of split partitions
  if (OB_FAIL(ret)) {
  } else if (PARTITION_LEVEL_ONE == target_part_level) {
    if (PARTITION_LEVEL_ZERO == ori_part_level) {
      for (int64_t i = start; OB_SUCC(ret) && i < end; ++i) {
        const ObPartition *split_part = split_part_array[i];
        if (split_part->get_split_source_tablet_id() != orig_table_schema.get_tablet_id()) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("source_tablet_id is invalid", KR(ret), K(split_part->get_split_source_tablet_id()),
                                                  K(orig_table_schema.get_table_id()));
        }
      }
    } else if (PARTITION_LEVEL_ONE == ori_part_level) {
      const ObTabletID source_tablet_id = split_part_array[start]->get_split_source_tablet_id();
      const ObPartition *part = NULL;
      bool find = false;

      for (int64_t j = 0; !find && OB_SUCC(ret) && j < orig_part_num; ++j) {
        part = orig_part_array[j];
        if (OB_ISNULL(part)) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("NULL ptr", KR(ret));
        } else if (part->get_tablet_id() == source_tablet_id) { // check existence
          find = true;
          // check boundary
          const ObRowkey& min_val = split_part_array[start]->get_high_bound_val();
          const ObRowkey& max_val = split_part_array[end - 1]->get_high_bound_val();
          if (max_val != part->get_high_bound_val()) {
            ret = OB_ERR_RANGE_NOT_INCREASING_ERROR;
            LOG_WARN("the value of last partition is not equal the origin range", KR(ret), K(max_val), KPC(part));
          } else if (j != 0 && min_val <= orig_part_array[j - 1]->get_high_bound_val()) {
            ret = OB_ERR_RANGE_NOT_INCREASING_ERROR;
            LOG_WARN("the value of split partition conflicts with existent partition", KR(ret), K(min_val),
                                                                                       KPC(orig_part_array[j - 1]));
          }
        }
      } // end for

      if (OB_FAIL(ret)) {
      } else if (OB_UNLIKELY(!find)) {
        ret = OB_UNKNOWN_PARTITION;
        LOG_WARN("source_tablet_id not exists in origin table", KR(ret), K(source_tablet_id), K(orig_table_schema));
      }
    } else {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("invalid part level", KR(ret), K(target_part_level), K(ori_part_level));
    }
  } else if (PARTITION_LEVEL_TWO == target_part_level) {
    ret = OB_NOT_SUPPORTED;
    LOG_WARN("not supported the part level", KR(ret), K(target_part_level), K(ori_part_level));
  } else {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("invalid part level", KR(ret), K(target_part_level), K(ori_part_level));
  }

  return ret;
}

// the name of split patitions should be different with other partitions' name
// except that of its source partition
int ObDDLService::check_split_partition_name_(const share::schema::ObTableSchema &orig_table_schema,
                                              const AlterTableSchema &alter_table_schema,
                                              const ObPartitionLevel target_part_level)
{
  int ret = OB_SUCCESS;
  const ObPartitionLevel ori_part_level = orig_table_schema.get_part_level();
  ObPartition **split_part_array = alter_table_schema.get_part_array();
  const int64_t split_part_num = alter_table_schema.get_partition_num();

  if (OB_ISNULL(split_part_array)) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("invalid argument", KR(ret), K(alter_table_schema));
  } else if (split_part_num < 0) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("invalid argument", KR(ret), K(split_part_num));
  } else {
    ObArenaAllocator allocator;
    ObPartition** sorted_part_array;
    if (OB_ISNULL(sorted_part_array = static_cast<ObPartition**>(allocator.alloc(
                                            sizeof(ObPartition*) * (split_part_num))))) {
      ret = OB_ALLOCATE_MEMORY_FAILED;
      LOG_WARN("fail to alloc", KR(ret), K(split_part_num));
    } else {
      for (int64_t i = 0; OB_SUCC(ret) && i < split_part_num; ++i) {
        sorted_part_array[i] = split_part_array[i];
      }
      ob_sort(sorted_part_array, sorted_part_array + split_part_num, ObPartitionNameCmp(CS_TYPE_UTF8MB4_GENERAL_CI));
    }

    ObString last_split_part_name;
    const ObPartition *ori_part = NULL;
    for (int64_t i = 0; OB_SUCC(ret) && i < split_part_num; ++i) {
      ObPartition *split_part = sorted_part_array[i];
      if (OB_ISNULL(split_part)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("NULL ptr", KR(ret));
      } else if (split_part->is_empty_partition_name()) {
        // will be generated part name in fill_part_name() of ObDDLService
      } else if (!last_split_part_name.empty()
                  && ObCharset::case_insensitive_equal(split_part->get_part_name(),
                                                       last_split_part_name)) {
        ret = OB_ERR_SAME_NAME_PARTITION;
        LOG_WARN("duplicate partition name in split part array", KR(ret), "part name", split_part->get_part_name());
        LOG_USER_ERROR(OB_ERR_SAME_NAME_PARTITION, split_part->get_part_name().length(),
                                                   split_part->get_part_name().ptr());
      } else if (PARTITION_LEVEL_ONE == target_part_level) {
        if (PARTITION_LEVEL_ZERO == ori_part_level) {
        } else if (PARTITION_LEVEL_ONE != ori_part_level) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("invalid part level", KR(ret), K(ori_part_level), K(target_part_level));
        } else if (OB_FAIL(orig_table_schema.get_partition_by_name(split_part->get_part_name(),
                                                                   ori_part))) {
          if (OB_UNLIKELY(ret != OB_UNKNOWN_PARTITION)) {
            LOG_WARN("fail to get partition by name", KR(ret), KPC(split_part));
          } else {
            ret = OB_SUCCESS;
          }
        } else if (OB_ISNULL(ori_part)) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("NULL ptr", KR(ret));
        } else if (ori_part->get_tablet_id() != split_part->get_split_source_tablet_id()) {
          ret = OB_ERR_SAME_NAME_PARTITION;
          LOG_WARN("duplicate partition name with orign part array", KR(ret),
                   "part name", split_part->get_part_name());
          LOG_USER_ERROR(OB_ERR_SAME_NAME_PARTITION, split_part->get_part_name().length(),
                                                     split_part->get_part_name().ptr());
        }
        if (OB_SUCC(ret)) {
          last_split_part_name = split_part->get_part_name();
        }
      } else if (PARTITION_LEVEL_TWO == target_part_level) {
        ret = OB_NOT_SUPPORTED;
        LOG_WARN("not support to split subpartition", KR(ret), K(ori_part_level), K(target_part_level),
                                                      K(orig_table_schema), K(alter_table_schema));
      } else {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("invalid part level", KR(ret), K(ori_part_level), K(target_part_level),
                                       K(orig_table_schema), K(alter_table_schema));
      }
    } // end for
  }

  return ret;
}

int ObDDLService::check_split_global_index_partition_(ObSchemaGetterGuard &schema_guard,
																					            obrpc::ObAlterTableArg &arg,
																					            const share::schema::ObTableSchema &orig_index_schema)
{
  int ret = OB_SUCCESS;
  bool is_db_in_recyclebin = false;
  uint64_t tenant_id = orig_index_schema.get_tenant_id();

  if (OB_FAIL(schema_guard.check_database_in_recyclebin(
                     tenant_id,
                     orig_index_schema.get_database_id(),
                     is_db_in_recyclebin))) {
    LOG_WARN("check database in recyclebin failed", KR(ret), K(tenant_id), K(orig_index_schema));
  } else if (OB_UNLIKELY(is_db_in_recyclebin)) {
    ret = OB_ERR_OPERATION_ON_RECYCLE_OBJECT;
    LOG_WARN("database is in recyclebin", KR(ret), K(orig_index_schema));
  } else if (OB_UNLIKELY(orig_index_schema.is_in_recyclebin())) {
    // when drop data table, the related index of it will be moved to recyclebin;
    // when drop index, the index will be removed directly
    ret = OB_ERR_OPERATION_ON_RECYCLE_OBJECT;
    LOG_WARN("index table is in recyclebin", KR(ret), K(orig_index_schema));
  } else if (!orig_index_schema.check_can_do_ddl()) {
    ret = OB_OP_NOT_ALLOW;
    LOG_WARN("offline ddl is being executed, other ddl operations are not allowed", KR(ret), K(orig_index_schema));
  } else if (OB_FAIL(check_restore_point_allow(tenant_id, orig_index_schema))) {
    LOG_WARN("check restore point allow failed,", KR(ret), K(tenant_id), K(orig_index_schema.get_table_id()));
  } else if (OB_FAIL(check_alter_partitions(orig_index_schema, arg))) {
    LOG_WARN("check alter partitions failed", KR(ret), K(orig_index_schema), K(arg));
  }
  return ret;
}

int ObDDLService::check_alter_add_subpartitions(const share::schema::ObTableSchema &orig_table_schema,
                                             const obrpc::ObAlterTableArg &alter_table_arg)
{
  int ret = OB_SUCCESS;
  const AlterTableSchema &alter_table_schema = alter_table_arg.alter_table_schema_;
  int64_t part_num = 0;
  ObArray<const ObPartition*> orig_parts;

  // get count of all partitions
  for (int64_t i = 0; OB_SUCC(ret) && i < alter_table_schema.get_partition_num(); i++) {
    if (OB_ISNULL(alter_table_schema.get_part_array()[i])) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("part_array[i] is null", K(ret), K(i));
    } else {
      part_num += alter_table_schema.get_part_array()[i]->get_subpartition_num();
    }
  }

  ObPartition **part_array = alter_table_schema.get_part_array();
  const ObPartitionLevel part_level = orig_table_schema.get_part_level();
  const ObPartitionOption &subpart_option = orig_table_schema.get_sub_part_option();
  bool is_oracle_mode = false;
  if (OB_SUCC(ret) && OB_ISNULL(part_array)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("part_array is null", K(ret), K(part_array));
  } else if (OB_FAIL(orig_table_schema.check_if_oracle_compat_mode(is_oracle_mode))) {
    LOG_WARN("fail to check is oracle mode", K(ret));
  } else if (PARTITION_LEVEL_ZERO == part_level) {
    ret = OB_ERR_PARTITION_MGMT_ON_NONPARTITIONED;
    LOG_WARN("unsupport management on non-partition table", K(ret));
  } else if (!subpart_option.is_range_part() && !subpart_option.is_list_part()) {
    ret = OB_ERR_ONLY_ON_RANGE_LIST_PARTITION;
    LOG_WARN("add partition can only be used on RANGE/LIST partitions", K(ret), K(alter_table_arg));
  } else if ((is_oracle_mode && OB_MAX_PARTITION_NUM_ORACLE < orig_table_schema.get_all_part_num() + part_num)
             || (!is_oracle_mode
                  && ObResolverUtils::get_mysql_max_partition_num(orig_table_schema.get_tenant_id())
                        < orig_table_schema.get_all_part_num() + part_num)) {
    ret = OB_TOO_MANY_PARTITIONS_ERROR;
    LOG_WARN("too many partitions", K(ret),
             "partition cnt current", orig_table_schema.get_all_part_num(),
             "partition cnt to be added", part_num);
  }

  // To add a subpartition separately, verify whether the partition exists
  // and whether the subpartition does not exist
  for (int64_t i = 0; OB_SUCC(ret) && i < alter_table_schema.get_partition_num(); ++i) {
    ObCheckPartitionMode check_partition_mode = CHECK_PARTITION_MODE_NORMAL;
    ObPartIterator iter(orig_table_schema, check_partition_mode);
    const ObPartition *part = NULL;
    const ObPartition *inc_part = part_array[i];
    if (OB_ISNULL(inc_part)
        || 0 == inc_part->get_subpartition_num()
        || OB_ISNULL(inc_part->get_subpart_array())) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("nontemplate subpart_array is NULL", KR(ret), K(i));
    }
    while (OB_SUCC(ret) && OB_SUCC(iter.next(part))) {
      if (OB_ISNULL(part) || OB_ISNULL(inc_part)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("NULL ptr", KR(ret));
      } else if (ObCharset::case_insensitive_equal(part->get_part_name(),
                                                   inc_part->get_part_name())) {
        if (OB_FAIL(orig_parts.push_back(part))) {
          LOG_WARN("fail to push back orig parts", KR(ret), KPC(part));
        }
      }

      for (int j = 0; OB_SUCC(ret) && j < inc_part->get_subpartition_num(); j++) {
        ObSubPartition *subpart = NULL;
        ObSubPartition *inc_subpart = inc_part->get_subpart_array()[j];
        int64_t k = 0, subpart_num = part->get_subpartition_num();
        for (k = 0; OB_SUCC(ret) && k < subpart_num; k++) {
          subpart = part->get_subpart_array()[k];
          if (OB_ISNULL(subpart) || OB_ISNULL(inc_subpart)) {
            ret = OB_ERR_UNEXPECTED;
            LOG_WARN("NULL ptr", KR(ret));
          } else if (ObCharset::case_insensitive_equal(subpart->get_part_name(),
                                                       inc_subpart->get_part_name())) {
            ret = OB_ERR_SAME_NAME_SUBPARTITION;
            LOG_WARN("duplicate subpartition name", KR(ret), K(subpart->get_part_name()));
            LOG_USER_ERROR(OB_ERR_SAME_NAME_SUBPARTITION, subpart->get_part_name().length(),
                            subpart->get_part_name().ptr());
          }
        }
      }
    }
    if (OB_ITER_END == ret) {
      if (OB_UNLIKELY(orig_parts.empty())) {
        ret = OB_PARTITION_NOT_EXIST;
        LOG_WARN("partition does not exist", KR(ret), K(inc_part->get_part_name()));
        LOG_USER_ERROR(OB_PARTITION_NOT_EXIST);
      } else {
        ret = OB_SUCCESS;
      }
    }
  }// end for

  // check whether the value of list and range is valid
  for (int64_t i = 0; OB_SUCC(ret) && i < alter_table_schema.get_partition_num(); ++i) {
    ObPartition *inc_part = part_array[i];
    int64_t inc_subpart_num = inc_part->get_subpartition_num();
    ObSubPartition ** inc_subpart_array = inc_part->get_subpart_array();

    if (OB_ISNULL(inc_part) || OB_ISNULL(inc_subpart_array) || 0 == inc_subpart_num) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("NULL ptr", K(inc_part), K(inc_subpart_num), K(ret));
    } else if (alter_table_schema.get_partition_num() != orig_parts.count()) {
      ret = OB_ERR_UNEXPECTED;
      LOG_WARN("orig_parts count not equal inc__part count", K(orig_parts.count()), K(inc_subpart_num), K(ret));
    } else if (orig_table_schema.is_range_subpart()) {
      const int64_t orig_subpart_num = orig_parts.at(i)->get_subpartition_num();
      ObSubPartition **orig_subpart_array = orig_parts.at(i)->get_subpart_array();
      if (OB_ISNULL(orig_subpart_array) || OB_ISNULL(orig_subpart_array[orig_subpart_num - 1])) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("NULL ptr", K(ret), K(orig_subpart_array), K(orig_subpart_array[orig_subpart_num - 1]));
      } else {
        const ObRowkey *rowkey_last =
          &(orig_subpart_array[orig_subpart_num - 1]->get_high_bound_val());
        for (int64_t j = 0; OB_SUCC(ret) && j < inc_subpart_num; ++j) {
          const ObRowkey *rowkey_cur =
            &inc_subpart_array[j]->get_high_bound_val();
          if (*rowkey_cur <= *rowkey_last) {
            ret = OB_ERR_ADD_PART_BOUN_NOT_INC;
            LOG_WARN("range values should increasing", K(ret), K(rowkey_cur), K(rowkey_last));
          } else {
            rowkey_last = rowkey_cur;
          }
        }
      }
    } else if (orig_table_schema.is_list_subpart()) {
      if (OB_FAIL(check_add_list_subpartition(*orig_parts.at(i), *inc_part))) {
        LOG_WARN("failed to check add list partition", K(ret), K(orig_table_schema), K(alter_table_schema));
      }
    }

  }
  return ret;
}

// Check whether the newly added partition has a duplicate value with the old partition
int ObDDLService::check_add_list_partition(const share::schema::ObPartitionSchema &orig_part,
                                           const share::schema::ObPartitionSchema &new_part,
                                           const int64_t split_part_id)
{
  int ret = OB_SUCCESS;
  int64_t part_num = new_part.get_partition_num();
  ObPartition **part_array = new_part.get_part_array();
  int64_t orig_part_num = orig_part.get_partition_num();
  ObPartition **orig_part_array = orig_part.get_part_array();
  common::ObRowkey row_key;
  if (OB_ISNULL(part_array) || OB_ISNULL(orig_part_array)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("part array is null", K(ret), KP(part_array), KP(orig_part_array));
  } else {
    common::hash::ObHashSet<common::ObRowkey> list_row_map;
    if (OB_FAIL(list_row_map.create(hash::cal_next_prime(part_num),
                                    ObModIds::OB_PARTITION_SPLIT,
                                    ObModIds::OB_PARTITION_SPLIT))) {
      LOG_WARN("failed to create list value", K(ret));
    } else {
      for (int64_t i = 0; OB_SUCC(ret) && i < orig_part_num; ++i) {
        if (OB_ISNULL(orig_part_array[i])) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("part is null", K(ret), K(i), K(orig_part));
        } else if (split_part_id == orig_part_array[i]->get_part_id()) {
          // Ensure that the split partition, except for the split partition,
          // cannot conflict with other partitions
          continue;
        } else {
          const ObIArray<common::ObNewRow>* orig_list_value = &(orig_part_array[i]->get_list_row_values());
          if (OB_ISNULL(orig_list_value)) {
            ret = OB_ERR_UNEXPECTED;
            LOG_WARN("list row value is null", K(ret), K(orig_list_value));
          }
          for (int j = 0; OB_SUCC(ret) && j < orig_list_value->count(); ++j) {
            const common::ObNewRow *new_row = &(orig_list_value->at(j));
            if (!orig_part.is_external_table() && 1 <= new_row->get_count() && new_row->get_cell(0).is_max_value()) {
              ret = OB_ERR_ADD_PARTITION_TO_DEFAULT_LIST;
              LOG_WARN("can add a table has default partition", K(ret), K(orig_part_array));
              LOG_USER_ERROR(OB_ERR_ADD_PARTITION_TO_DEFAULT_LIST);
            } else {
              row_key.reset();
              row_key.assign(new_row->cells_, new_row->get_count());
              if (OB_FAIL(list_row_map.set_refactored(row_key))) {
                LOG_WARN("failed to insert hash map", K(ret), K(row_key));
              }
            }
          }
        }
      }// end for
      for (int64_t i = 0; OB_SUCC(ret) && i < part_num; ++i) {
        if (OB_ISNULL(part_array[i])) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("part is null", K(ret), K(i), K(new_part));
        } else {
          const ObIArray<common::ObNewRow>* list_value = &(part_array[i]->get_list_row_values());
          if (OB_ISNULL(list_value)) {
            ret = OB_ERR_UNEXPECTED;
            LOG_WARN("list row value is null", K(ret), K(list_value));
          }
          for (int j = 0; OB_SUCC(ret) && j < list_value->count(); ++j) {
            row_key.reset();
            row_key.assign(list_value->at(j).cells_, list_value->at(j).get_count());
            if (OB_HASH_NOT_EXIST != list_row_map.exist_refactored(row_key)) {
              ret = OB_ERR_MULTIPLE_DEF_CONST_IN_LIST_PART;
              LOG_WARN("add partition values duplicate to orig table", K(ret), K(row_key));
              LOG_USER_ERROR(OB_ERR_MULTIPLE_DEF_CONST_IN_LIST_PART);
            }
          }//end for
        }
      }// end for
    }
  }
  return ret;
}

// Check whether the newly added partition has a duplicate value with the old partition
int ObDDLService::check_add_list_subpartition(const ObPartition &orig_part, const ObPartition &new_part)
{
  int ret = OB_SUCCESS;
  int64_t subpart_num = new_part.get_subpartition_num();
  ObSubPartition **subpart_array = new_part.get_subpart_array();
  int64_t orig_subpart_num = orig_part.get_subpartition_num();
  ObSubPartition **orig_subpart_array = orig_part.get_subpart_array();
  common::ObRowkey row_key;
  if (OB_ISNULL(subpart_array) || OB_ISNULL(orig_subpart_array)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("part array is null", K(ret), KP(subpart_array), KP(orig_subpart_array));
  } else {
    common::hash::ObHashSet<common::ObRowkey> list_row_map;
    if (OB_FAIL(list_row_map.create(hash::cal_next_prime(subpart_num),
                                    "ChkAddLstSPar",
                                    "ChkAddLstSPar"))) {
      LOG_WARN("failed to create list value", K(ret));
    } else {
      for (int64_t i = 0; OB_SUCC(ret) && i < orig_subpart_num; ++i) {
        if (OB_ISNULL(orig_subpart_array[i])) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("part is null", K(ret), K(i), K(orig_part));
        } else {
          const ObIArray<common::ObNewRow>* orig_list_value = &(orig_subpart_array[i]->get_list_row_values());
          if (OB_ISNULL(orig_list_value)) {
            ret = OB_ERR_UNEXPECTED;
            LOG_WARN("list row value is null", K(ret), K(orig_list_value));
          }
          for (int j = 0; OB_SUCC(ret) && j < orig_list_value->count(); ++j) {
            const common::ObNewRow *new_row = &(orig_list_value->at(j));
            if (1 == new_row->get_count() && new_row->get_cell(0).is_max_value()) {
              ret = OB_ERR_ADD_PARTITION_TO_DEFAULT_LIST;
              LOG_WARN("can add a table has default partition", K(ret), K(orig_subpart_array));
              LOG_USER_ERROR(OB_ERR_ADD_PARTITION_TO_DEFAULT_LIST);
            } else {
              row_key.reset();
              row_key.assign(new_row->cells_, new_row->get_count());
              if (OB_FAIL(list_row_map.set_refactored(row_key))) {
                LOG_WARN("failed to insert hash map", K(ret), K(row_key));
              }
            }
          }
        }
      }// end for
      for (int64_t i = 0; OB_SUCC(ret) && i < subpart_num; ++i) {
        if (OB_ISNULL(subpart_array[i])) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("part is null", K(ret), K(i), K(new_part));
        } else {
          const ObIArray<common::ObNewRow>* list_value = &(subpart_array[i]->get_list_row_values());
          if (OB_ISNULL(list_value)) {
            ret = OB_ERR_UNEXPECTED;
            LOG_WARN("list row value is null", K(ret), K(list_value));
          }
          for (int j = 0; OB_SUCC(ret) && j < list_value->count(); ++j) {
            row_key.reset();
            row_key.assign(list_value->at(j).cells_, list_value->at(j).get_count());
            if (OB_HASH_NOT_EXIST != list_row_map.exist_refactored(row_key)) {
              ret = OB_ERR_MULTIPLE_DEF_CONST_IN_LIST_PART;
              LOG_WARN("add partition values duplicate to orig table", K(ret), K(row_key));
              LOG_USER_ERROR(OB_ERR_MULTIPLE_DEF_CONST_IN_LIST_PART);
            }
          }//end for
        }
      }// end for
    }
  }
  return ret;
}
int ObDDLService::alter_table(obrpc::ObAlterTableArg &alter_table_arg,
                              obrpc::ObAlterTableRes &res)
{
  int ret = OB_SUCCESS;
  int64_t start_usec = 0;
  int64_t end_usec = 0;
  int64_t cost_usec = 0;
  start_usec = ObTimeUtility::current_time();
  bool is_alter_sess_active_time = false;
  bool is_alter_comment = false;
  const AlterTableSchema &alter_table_schema = alter_table_arg.alter_table_schema_;
  const uint64_t tenant_id = alter_table_schema.get_tenant_id();
  int64_t &task_id = res.task_id_;
  ObDDLType &ddl_type = res.ddl_type_;
  bool &ddl_need_retry_at_executor = res.ddl_need_retry_at_executor_;
  ddl_type = DDL_INVALID;

  if (OB_FAIL(check_inner_stat())) {
    LOG_WARN("variable is not init", K(ret));
  }

  ObSimpleTableSchemaV2 orig_table;
  if (OB_SUCC(ret)) {
    AlterTableSchema &alter_table_schema = alter_table_arg.alter_table_schema_;
    ObSchemaGetterGuard schema_guard;
    schema_guard.set_session_id(alter_table_arg.session_id_);
    const ObTableSchema *orig_table_schema =  NULL;
    is_alter_sess_active_time = alter_table_schema.alter_option_bitset_.has_member(obrpc::ObAlterTableArg::SESSION_ACTIVE_TIME);
    is_alter_comment = alter_table_schema.alter_option_bitset_.has_member(obrpc::ObAlterTableArg::COMMENT);
    LOG_DEBUG("debug view comment", K(is_alter_comment), K(alter_table_schema));
    ObTZMapWrap tz_map_wrap;
    uint64_t tenant_data_version = 0;
    if (OB_FAIL(ret)) {
    } else if (OB_FAIL(GET_MIN_DATA_VERSION(tenant_id, tenant_data_version))) {
      LOG_WARN("get tenant data version failed", KR(ret), K(tenant_id));
    } else if (alter_table_arg.has_alter_duplicate_scope() && (tenant_data_version < MOCK_DATA_VERSION_4_2_3_0
        || (tenant_data_version >= DATA_VERSION_4_3_0_0 && tenant_data_version < DATA_VERSION_4_3_5_1))) {
      ret = OB_NOT_SUPPORTED;
      LOG_WARN("alter table duplicate scope not supported", KR(ret));
      LOG_USER_ERROR(OB_NOT_SUPPORTED, "alter table duplicate scope");
    } else if (alter_table_arg.has_alter_duplicate_scope() && !is_user_tenant(tenant_id)) {
      ret = OB_NOT_SUPPORTED;
      LOG_WARN("not user tenant, alter table duplicate scope not supported", KR(ret), K(tenant_id));
      LOG_USER_ERROR(OB_NOT_SUPPORTED, "not user tenant, alter table duplicate scope");
    } else if (OB_FAIL(OTTZ_MGR.get_tenant_tz(tenant_id, tz_map_wrap))) {
      LOG_WARN("get tenant timezone map failed", K(ret), K(tenant_id));
    } else if (FALSE_IT(alter_table_arg.set_tz_info_map(tz_map_wrap.get_tz_map()))) {
    } else if (OB_FAIL(get_tenant_schema_guard_with_version_in_inner_table(tenant_id,
                                                                          schema_guard))) {
      LOG_WARN("fail to get schema guard with version in inner table", K(ret), K(tenant_id));
    } else if (OB_FAIL(check_parallel_ddl_conflict(schema_guard, alter_table_arg))) {
      LOG_WARN("check parallel ddl conflict failed", KR(ret));
    } else if (false == is_alter_sess_active_time) {
      const ObString &origin_database_name = alter_table_schema.get_origin_database_name();
      const ObString &origin_table_name = alter_table_schema.get_origin_table_name();
      if (origin_database_name.empty() || origin_table_name.empty()) {
        ret = OB_INVALID_ARGUMENT;
        LOG_WARN("database name or table name is null", K(alter_table_schema),
            K(origin_database_name), K(origin_table_name), K(ret));
      } else {
        if (OB_FAIL(schema_guard.get_table_schema(tenant_id,
                                                  origin_database_name,
                                                  origin_table_name,
                                                  false,
                                                  orig_table_schema))) {
          LOG_WARN("fail to get table schema", K(ret), K(tenant_id), K(origin_database_name),
                  K(origin_table_name));
        } else if (NULL == orig_table_schema) {
          ret = OB_ERR_UNEXPECTED;
          LOG_WARN("NULL ptr", KR(ret), KP(orig_table_schema));
        } else if (OB_FAIL(orig_table.assign(*orig_table_schema))) {
          LOG_WARN("fail to assign schema", K(ret));
        }
      }
    }
    // check schema
    if (OB_SUCC(ret)
        && alter_table_arg.is_alter_partitions_
        && obrpc::ObAlterTableArg::REPARTITION_TABLE != alter_table_arg.alter_part_type_) {
      if (OB_ISNULL(orig_table_schema)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("table schema is null", KR(ret), K(alter_table_arg));
      } else {
        if (obrpc::ObAlterTableArg::DROP_PARTITION != alter_table_arg.alter_part_type_) {
          if ((alter_table_arg.alter_table_schema_.is_range_part()
              && !alter_table_arg.alter_table_schema_.is_interval_part())
              || alter_table_arg.alter_table_schema_.is_list_part()) {
            if (OB_FAIL(fill_part_name(*orig_table_schema, alter_table_arg.alter_table_schema_))) {
              LOG_WARN("failed to fill part name", K(ret));
            }
          }
        }
        if (OB_FAIL(ret)) {
        } else if (OB_FAIL(check_alter_partitions(*orig_table_schema, alter_table_arg))) {
          LOG_WARN("check alter partitions failed", K(ret), K(orig_table_schema), K(alter_table_arg));
        }
      }
    }

    //do alter table in transaction
    if (OB_SUCC(ret)) {
      uint64_t data_version = 0;
      bool is_oracle_mode_add_column_not_null_ddl = false;
      bool is_default_value_null = false;
      if (OB_FAIL(GET_MIN_DATA_VERSION(tenant_id, data_version))) {
        LOG_WARN("failed to get tenant data version", K(ret), K(tenant_id), K(data_version));
      } else if (is_alter_sess_active_time) {
        if (OB_FAIL(alter_table_sess_active_time_in_trans(alter_table_arg,
                                                          res,
                                                          data_version))) {
          LOG_WARN("alter_table_in_trans failed", K(ret));
        } else {
          LOG_INFO("refresh session active time of temp tables succeed!", K(ret));
        }
      } else if (OB_FAIL(check_is_offline_ddl(alter_table_arg, data_version, ddl_type, ddl_need_retry_at_executor))) {
        LOG_WARN("failed to check is offline ddl", K(ret), K(alter_table_arg));
      } else if (((MOCK_DATA_VERSION_4_2_1_3 <= data_version && DATA_VERSION_4_3_0_0 > data_version)
               || (DATA_VERSION_4_3_2_0 <= data_version)) // [4213, 430) & [4320, )
                 && OB_FAIL(check_is_oracle_mode_add_column_not_null_ddl(alter_table_arg,
                                                                         schema_guard,
                                                                         is_oracle_mode_add_column_not_null_ddl,
                                                                         is_default_value_null))) {
        LOG_WARN("falied to check is oracle add column not null", K(ret));
      } else if (is_oracle_mode_add_column_not_null_ddl) {
        if (OB_FAIL(do_oracle_add_column_not_null_in_trans(alter_table_arg,
                                                           schema_guard,
                                                           data_version,
                                                           is_default_value_null))) {
          LOG_WARN("add column not null failed", K(ret));
        } else {
          ddl_type = ObDDLType::DDL_NORMAL_TYPE;
        }
      } else {
        // offline ddl cannot appear at the same time with other ddl types
        if (is_long_running_ddl(ddl_type)) {
          if (OB_FAIL(do_offline_ddl_in_trans(alter_table_arg, data_version, res))) {
            LOG_WARN("failed to do offline ddl in trans", K(ret), K(alter_table_arg), K(ddl_type));
          }
        } else {
          if ((alter_table_arg.is_alter_mlog_attributes_ ||
               alter_table_arg.is_alter_mview_attributes_) &&
              OB_FAIL(ObMviewAlterService::alter_mview_or_mlog_in_trans(alter_table_arg, res, schema_guard, schema_service_, sql_proxy_, data_version))) {
            LOG_WARN("alter_mview_or_mlog_in_trans failed", KR(ret));
          } else if (OB_FAIL(alter_table_in_trans(alter_table_arg, res, data_version))) {
            LOG_WARN("alter_table_in_trans failed", K(ret));
          }
        }
      }
      end_usec = ObTimeUtility::current_time();
      cost_usec = end_usec - start_usec;
      start_usec = end_usec;
      LOG_INFO("alter_table_in_trans cost: ", KR(ret), K(cost_usec), K(ddl_type), "ddl_stmt", alter_table_arg.ddl_stmt_str_,
        K(alter_table_arg));
    }
  }

  // just for debug
  if (OB_SUCC(ret) && false == is_alter_sess_active_time) {
    int tmp_ret = OB_SUCCESS;
    const uint64_t table_id = (OB_INVALID_ID == orig_table.get_table_id()) ?
        alter_table_arg.alter_table_schema_.get_table_id() :
        orig_table.get_table_id();
    const ObTableSchema *new_table_schema = NULL;
    HEAP_VAR(ObSchemaGetterGuard, new_schema_guard) {
      if (OB_FAIL(get_tenant_schema_guard_with_version_in_inner_table(tenant_id, new_schema_guard))) {
        LOG_WARN("fail to get schema guard with version in inner table", K(ret), K(tenant_id));
      } else if (OB_FAIL(new_schema_guard.get_table_schema(tenant_id, table_id, new_table_schema))) {
        LOG_WARN("fail to get table schema", K(tmp_ret), K(tenant_id), K(table_id));
      } else if (NULL == new_table_schema) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("NULL ptr", K(tmp_ret));
      }
    }
		// check const_column_iterator and ObColumnIterByPrevNextID
    if (OB_SUCC(ret)) {
      // unused_columns + prev_next_iter_columns = columns_arry_in_schema.
      ObArray<uint64_t> unused_column_ids; // caused by drop column online.
      const int64_t column_cnt_in_column_arr = new_table_schema->get_column_count();
      int64_t column_cnt_in_prev_next_iter = 0;
      const ObColumnSchemaV2 *col = nullptr;
      ObColumnIterByPrevNextID iter(*new_table_schema);
      while (OB_SUCC(ret) && OB_SUCC(iter.next(col))) {
        column_cnt_in_prev_next_iter++;
      }
      if (ret != OB_ITER_END) {
        LOG_WARN("Failed to iterate all table columns. iter quit. ", KR(ret), KPC(new_table_schema));
      } else if (OB_FAIL(new_table_schema->get_unused_column_ids(unused_column_ids))) {
        LOG_WARN("get unused column ids failed", KR(ret), KPC(new_table_schema));
      } else if (OB_UNLIKELY(column_cnt_in_column_arr != column_cnt_in_prev_next_iter + unused_column_ids.count())) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("Inconsistent iterators: const_column_iterator is less than ObColumnIterByPrevNextID", KR(ret),
          K(column_cnt_in_column_arr), K(column_cnt_in_prev_next_iter), K(unused_column_ids), KPC(new_table_schema));
      }
    }
  }

  return ret;
}

int ObDDLService::rename_table(const obrpc::ObRenameTableArg &rename_table_arg)
{
  int ret = OB_SUCCESS;
  if (OB_FAIL(check_inner_stat())) {
    LOG_WARN("variable is not init");
  } else {
    ObDDLOperator ddl_operator(*schema_service_, *sql_proxy_);
    ObSchemaGetterGuard schema_guard;
    const uint64_t tenant_id = rename_table_arg.tenant_id_;
    bool is_oracle_mode = false;
    bool sequence_exist = false;
    RenameOracleObjectType rename_oracle_obj_type = RENAME_TYPE_INVALID;
    if (OB_FAIL(get_tenant_schema_guard_with_version_in_inner_table(tenant_id, schema_guard))) {
      LOG_WARN("fail to get schema guard with version in inner table", K(ret), K(tenant_id));
    } else {
      ObSchemaService *schema_service = schema_service_->get_schema_service();
      lib::Worker::CompatMode compat_mode = lib::Worker::CompatMode::MYSQL;
      if (OB_ISNULL(schema_service)) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("schema_guard or schema service is null",
            K(schema_service), K(ret));
      } else if (OB_INVALID_ID == tenant_id) {
        ret = OB_INVALID_ARGUMENT;
        LOG_WARN("tenant_id is invalid", K(tenant_id), K(ret));
      } else if (is_virtual_tenant_id(tenant_id) || OB_SYS_TENANT_ID == tenant_id) {
        compat_mode = lib::Worker::CompatMode::MYSQL;
        is_oracle_mode = false;
      } else if (OB_FAIL(ObCompatModeGetter::get_tenant_mode(tenant_id, compat_mode))) {
        LOG_WARN("failed to get compat mode", K(ret), K(tenant_id));
      } else {
        if (lib::Worker::CompatMode::ORACLE == compat_mode) {
          is_oracle_mode = true;
        } else {
          is_oracle_mode = false;
        }
        if (is_oracle_mode && rename_table_arg.rename_table_items_.size() > 1) {
          ret = OB_ERR_ALTER_TABLE_RENAME_WITH_OPTION;
          LOG_WARN("alter table rename can't be combined with other operations in oracle mode",
                   K(ret), K(rename_table_arg));
        }
      }
      ObDDLSQLTransaction trans(schema_service_);
      uint64_t tenant_data_version = OB_INVALID_VERSION;
      int64_t refreshed_schema_version = 0;
      if (OB_FAIL(ret)) {
      } else if (OB_FAIL(schema_guard.get_schema_version(tenant_id, refreshed_schema_version))) {
        LOG_WARN("failed to get tenant schema version", KR(ret), K(tenant_id));
      } else if (OB_FAIL(trans.start(sql_proxy_, tenant_id, refreshed_schema_version))) {
        LOG_WARN("start transaction failed", KR(ret), K(tenant_id), K(refreshed_schema_version));
      } else if (OB_FAIL(GET_MIN_DATA_VERSION(tenant_id, tenant_data_version))) {
        LOG_WARN("fail to get data version", K(ret), K(tenant_id), K(tenant_data_version));
      } else {
        //todo use array to replace hashmap and hashset @hualong
        //record table already be renamed in the schema mgr
        common::hash::ObHashSet<ObTableItem> delete_table_set;
        //record new table name set
        //table_item -> table_id
        common::hash::ObHashMap<ObTableItem, uint64_t> new_table_map;
        ObArray<std::pair<uint64_t, share::schema::ObObjectType>> all_dep_objs;
        const int64_t rename_items_count = rename_table_arg.rename_table_items_.size();

        if (OB_FAIL(delete_table_set.create(rename_items_count))) {
          LOG_WARN("failed to add create ObHashSet", KR(ret));
        } else if (OB_FAIL(new_table_map.create(rename_items_count, ObModIds::OB_HASH_BUCKET_RENAME_TABLE_MAP))) {
          LOG_WARN("failed to add create ObHashMap", K(ret));
        } else {
          for (int64_t i = 0; OB_SUCC(ret) && i < rename_table_arg.rename_table_items_.size(); ++i) {
            const ObRenameTableItem &rename_item = rename_table_arg.rename_table_items_.at(i);
            const ObTableSchema *table_schema = nullptr;
            if (OB_FAIL(schema_guard.get_table_schema(tenant_id,
                                                      rename_item.origin_db_name_,
                                                      rename_item.origin_table_name_,
                                                      false,
                                                      table_schema))) {
              LOG_WARN("fail to get table schema", K(ret));
            } else if (nullptr == table_schema) {
              // skip
            } else if (table_schema->is_materialized_view()) {
              ret = OB_NOT_SUPPORTED;
              LOG_WARN("rename materialized view is not supported",
                  KR(ret), K(table_schema->get_table_name()));
              LOG_USER_ERROR(OB_NOT_SUPPORTED, "rename materialized view is");
            } else if (table_schema->is_mlog_table()) {
              ret = OB_NOT_SUPPORTED;
              LOG_WARN("rename materialized view log is not supported",
                  KR(ret), K(table_schema->get_table_name()));
              LOG_USER_ERROR(OB_NOT_SUPPORTED, "rename materialized view log is");
            } else if (table_schema->has_mlog_table()) {
              ret = OB_NOT_SUPPORTED;
   