/**
 * Copyright (c) 2021 OceanBase
 * OceanBase CE is licensed under Mulan PubL v2.
 * You can use this software according to the terms and conditions of the Mulan PubL v2.
 * You may obtain a copy of Mulan PubL v2 at:
 *          http://license.coscl.org.cn/MulanPubL-2.0
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PubL v2 for more details.
 */

#include "lib/charset/ob_ctype.h"

/**
  The following PINYIN/STROKE Chinese character weight arrays are
  based upon the PINYIN/STROKE collation in zh.xml file of CLDR42
  (http://cldr.unicode.org/)
  (http://unicode.org/Public/cldr/42/)

  The following RADICAL Chinese character weight arrays are based
  upon the kRSUnicode of UNICODE HAN DATABASE. We choose the first
  radical-stroke count of kRSUnicode. The radical with apostrophe (')
  which indicates a simplified version of the given radical will be
  placed before the traditional version.
  e.g.
    The weight of 闯(GB+B4B3) is 84216 while the weight of 門(GB+E954) is 84390.
  If two characters have the same the radical and remaining strokes,
  they will be sorted by the code point in GB18030-2022.
  e.g.
    The weight of 闯(GB+B4B3) is 84216 while the weight of 问(GB+CECA) is 84217
  (https://www.unicode.org/reports/tr38/tr38-33.html#kRSUnicode)
  (https://www.unicode.org/Public/UCD/latest/ucd/Unihan.zip)

  The arrays include the sort order of the code points accordingly.
  The weight here just mean the order of the corresponding gb18030_2022.
  The arrays do not contain code points deleted by gb18030_2022, such as
  GB+FE51, GB+FD9C.

  There are 44085/97093/98051 Chinese characters being taken into
  account in PINYIN/STROKE/RADICAL arrays.
  The weight starts from 1 to 44085/97093/98051 and there are no
  duplicate weights in the collation.

  The weight of each code point shall be the corresponding weight in
  these arrays plus CHINESE_WEIGHT_BASE.
*/

static const unsigned int PINYIN_2_BYTE_START_2022 = 0x8140;
static const unsigned int PINYIN_2_BYTE_END_2022 = 0xFE9F;

static const unsigned int PINYIN_4_BYTE_1_START_2022 = 0x8138FD38;
static const unsigned int PINYIN_4_1_DIFF_2022 = 11328;
static const unsigned int PINYIN_4_BYTE_1_END_2022 = 0x82359737;

static const unsigned int PINYIN_4_BYTE_2_START_2022 = 0x95328236;
static const unsigned int PINYIN_4_2_DIFF_2022 = 254536;
static const unsigned int PINYIN_4_BYTE_2_END_2022 = 0x9A37F738;

static const unsigned int STROKE_2_BYTE_START_2022 = 0x8140;
static const unsigned int STROKE_2_BYTE_END_2022 = 0xFE9F;

static const unsigned int STROKE_4_BYTE_1_START_2022 = 0x8138FD38;
static const unsigned int STROKE_4_1_DIFF_2022 = 11328;
static const unsigned int STROKE_4_BYTE_1_END_2022 = 0x82359832;

static const unsigned int STROKE_4_BYTE_2_START_2022 = 0x95328236;
static const unsigned int STROKE_4_2_DIFF_2022 = 254536;
static const unsigned int STROKE_4_BYTE_2_END_2022 = 0x9B31A337;

static const unsigned int RADICAL_2_BYTE_START_2022 = 0x8140;
static const unsigned int RADICAL_2_BYTE_END_2022 = 0xFEA0;

static const unsigned int RADICAL_4_BYTE_1_START_2022 = 0x8139EE39;
static const unsigned int RADICAL_4_1_DIFF_2022 = 12439;
static const unsigned int RADICAL_4_BYTE_1_END_2022 = 0x8430AE33;

static const unsigned int RADICAL_4_BYTE_2_START_2022 = 0x95328236;
static const unsigned int RADICAL_4_2_DIFF_2022 = 254536;
static const unsigned int RADICAL_4_BYTE_2_END_2022 = 0x9B31A337;

static const uint16 gb18030_2022_2_pinyin_weight_py[] = {
  16323, 28217, 34164, 10708, 21648, 4104,  28850, 6524,  26203, 18824, 39282, 1913,  15200, 13279, 14725, 10029,   /*[GB+8140, GB+814F]*/
  2929,  43325, 414,   15041, 8444,  37743, 38011, 33335, 15255, 37666, 38537, 11770, 3515,  38618, 36561, 8896,    /*[GB+8150, GB+815F]*/
  15256, 34166, 11790, 6668,  28818, 22511, 13551, 15561, 28749, 20996, 11567, 43430, 36163, 36762, 8645,  27857,   /*[GB+8160, GB+816F]*/
  22319, 8948,  30236, 39439, 5120,  41632, 42313, 10148, 8958,  20363, 19181, 38237, 15799, 20694, 28865,          /*[GB+8170, GB+817E]*/
  39415, 29587, 30071, 9380,  36504, 24834, 36508, 5307,  37664, 34675, 14985, 37538, 39372, 5493,  18619, 7203,    /*[GB+8180, GB+818F]*/
  21232, 13099, 13100, 32486, 416,   1853,  8341,  27624, 1854,  34199, 32041, 11337, 8981,  1996,  3225,  3068,    /*[GB+8190, GB+819F]*/
  3069,  27028, 7787,  3226,  25366, 33167, 37350, 8216,  2466,  32489, 5881,  42542, 23813, 39418, 6284,  7082,    /*[GB+81A0, GB+81AF]*/
  33411, 38029, 35368, 13103, 70,    14882, 23968, 5476,  8366,  30785, 5059,  40344, 3239,  34478, 36361, 35372,   /*[GB+81B0, GB+81BF]*/
  40882, 34297, 23322, 22749, 35140, 35820, 42740, 26367, 1055,  23969, 38044, 29593, 37941, 41863, 10610, 20724,   /*[GB+81C0, GB+81CF]*/
  42969, 2230,  26368, 1178,  41428, 4709,  37146, 41283, 1225,  34205, 26461, 29001, 765,   12454, 11171, 34652,   /*[GB+81D0, GB+81DF]*/
  9258,  7667,  21945, 1914,  11241, 12779, 10214, 26631, 4727,  35465, 28400, 32052, 8859,  17333, 37764, 10082,   /*[GB+81E0, GB+81EF]*/
  38474, 21498, 35830, 153,   20022, 7538,  20402, 2741,  3622,  36364, 42652, 27280, 5132,  29598, 5344,           /*[GB+81F0, GB+81FE]*/
  30460, 23185, 14845, 15171, 3598,  41808, 44057, 4110,  20210, 15431, 29211, 31508, 28528, 31959, 2010,  22453,   /*[GB+8240, GB+824F]*/
  34764, 31946, 39623, 33922, 35775, 10121, 16725, 33355, 38102, 8651,  18795, 12736, 15084, 25481, 27655, 23811,   /*[GB+8250, GB+825F]*/
  33539, 8652,  34080, 35587, 6029,  3243,  42272, 17334, 29500, 12813, 34943, 14007, 1708,  4252,  8000,  36517,   /*[GB+8260, GB+826F]*/
  941,   39726, 35327, 11774, 3040,  42024, 1918,  15292, 37265, 5064,  18603, 32369, 17335, 2472,  43727,          /*[GB+8270, GB+827E]*/
  9362,  31264, 29081, 21252, 30569, 4467,  35510, 23893, 24743, 9023,  15091, 30018, 14582, 7896,  42094, 43630,   /*[GB+8280, GB+828F]*/
  20409, 29669, 18017, 12737, 6545,  43598, 1035,  33358, 15588, 22416, 2330,  13877, 37471, 27923, 23187, 38742,   /*[GB+8290, GB+829F]*/
  3461,  25383, 27387, 42625, 4565,  32746, 39639, 1922,  31023, 36915, 8238,  30789, 33178, 7381,  3256,  28615,   /*[GB+82A0, GB+82AF]*/
  16213, 6039,  2657,  948,   35169, 12260, 37373, 41291, 4113,  36573, 39222, 35793, 40945, 4635,  8731,  1141,    /*[GB+82B0, GB+82BF]*/
  42314, 21663, 13146, 37965, 35170, 36311, 2310,  6911,  2564,  41694, 31657, 949,   40515, 14593, 32747, 29805,   /*[GB+82C0, GB+82CF]*/
  42350, 29887, 34058, 40156, 27207, 23336, 40441, 14180, 20670, 6112,  30717, 10968, 14607, 33597, 28114, 25490,   /*[GB+82D0, GB+82DF]*/
  15868, 2474,  4267,  27627, 960,   34952, 39098, 37275, 30670, 30204, 37162, 7654,  1924,  13585, 40581, 9772,    /*[GB+82E0, GB+82EF]*/
  2372,  17909, 39005, 40703, 43705, 1076,  29712, 4361,  39656, 41143, 43815, 28181, 4439,  15114, 10776,          /*[GB+82F0, GB+82FE]*/
  41348, 26031, 36936, 6063,  35197, 19795, 962,   24316, 14854, 18749, 20061, 20761, 25297, 34240, 30674, 38862,   /*[GB+8340, GB+834F]*/
  6626,  43137, 28122, 25709, 15189, 31952, 43994, 24668, 17717, 3123,  10097, 24772, 3471,  2855,  32782, 12999,   /*[GB+8350, GB+835F]*/
  1977,  12577, 4381,  31412, 5546,  14375, 8192,  34548, 15468, 7435,  19209, 2066,  9857,  34241, 34549, 21891,   /*[GB+8360, GB+836F]*/
  37577, 14937, 13709, 25772, 8260,  42780, 106,   27600, 37877, 16080, 23247, 2862,  38297, 5674,  16817,          /*[GB+8370, GB+837E]*/
  13921, 4289,  14383, 27917, 40587, 2451,  1811,  199,   30492, 4071,  2788,  17467, 22769, 14870, 25509, 21278,   /*[GB+8380, GB+838F]*/
  33391, 23155, 26174, 22770, 3100,  19059, 20244, 16906, 870,   39880, 1655,  40615, 42477, 29656, 39161, 10925,   /*[GB+8390, GB+839F]*/
  26109, 3439,  18463, 30974, 32812, 19744, 4224,  2884,  26918, 29129, 12661, 18528, 20486, 40621, 23391, 30804,   /*[GB+83A0, GB+83AF]*/
  36879, 17881, 22534, 40592, 35535, 6990,  6992,  31830, 3071,  7537,  7539,  14796, 36769, 28827, 0,     25274,   /*[GB+83B0, GB+83BF]*/
  8084,  20959, 28433, 6646,  0,     18235, 12359, 32487, 22666, 18828, 39441, 10838, 31188, 8080,  13420, 38238,   /*[GB+83C0, GB+83CF]*/
  21010, 26889, 40031, 20993, 2552,  15147, 9906,  15164, 21016, 42750, 9657,  35790, 38535, 16206, 37753, 21411,   /*[GB+83D0, GB+83DF]*/
  28882, 21257, 15590, 16635, 19114, 8733,  35132, 21578, 30508, 9055,  11802, 23633, 8452,  21860, 25207, 13721,   /*[GB+83E0, GB+83EF]*/
  29746, 21151, 31750, 9841,  15092, 26095, 6617,  9024,  38490, 33,    18367, 4453,  21950, 43278, 5021,           /*[GB+83F0, GB+83FE]*/
  29526, 7155,  14938, 19184, 33688, 6797,  7711,  7712,  15341, 4208,  41861, 8218,  22248, 42165, 8219,  8225,    /*[GB+8440, GB+844F]*/
  16151, 8959,  24465, 16793, 10649, 27025, 4402,  12916, 25477, 5137,  12921, 5478,  32335, 15808, 18025, 8453,    /*[GB+8450, GB+845F]*/
  26585, 6233,  21861, 19245, 42545, 1748,  14553, 14554, 27991, 2925,  9390,  4450,  16959, 7215,  7610,  14562,   /*[GB+8460, GB+846F]*/
  4729,  15051, 19776, 20518, 40786, 39981, 5186,  16501, 27859, 4451,  5188,  18030, 8001,  2935,  24855,          /*[GB+8470, GB+847E]*/
  4452,  43599, 9074,  1952,  7122,  26052, 6742,  14010, 13373, 36567, 12766, 7142,  6912,  33244, 9914,  28636,   /*[GB+8480, GB+848F]*/
  14016, 5260,  16152, 4454,  4348,  2952,  31865, 20062, 18059, 23956, 28020, 16616, 9889,  11995, 40957, 43370,   /*[GB+8490, GB+849F]*/
  18638, 15631, 19500, 10303, 14384, 10304, 14057, 14058, 30697, 13468, 14073, 14074, 42479, 2885,  13972, 22008,   /*[GB+84A0, GB+84AF]*/
  18180, 42960, 36495, 14883, 16526, 35014, 42166, 38046, 42809, 22028, 17694, 15733, 16625, 37080, 32210, 34940,   /*[GB+84B0, GB+84BF]*/
  22160, 16860, 14563, 19003, 16446, 14892, 9182,  21866, 3810,  17608, 39084, 16447, 15736, 26053, 20031, 2247,    /*[GB+84C0, GB+84CF]*/
  3819,  16177, 21665, 6620,  35863, 33439, 38163, 36248, 33099, 28641, 17710, 20063, 24317, 28941, 12984, 14218,   /*[GB+84D0, GB+84DF]*/
  3170,  26734, 34727, 38273, 15896, 7688,  15698, 31602, 15632, 5457,  20738, 36254, 36256, 20285, 18424, 3307,    /*[GB+84E0, GB+84EF]*/
  26919, 26738, 40341, 15206, 838,   32990, 35532, 8897,  8898,  38020, 23857, 15343, 30871, 9259,  7374,           /*[GB+84F0, GB+84FE]*/
  9505,  5261,  15297, 9520,  22592, 15277, 37746, 14202, 16307, 11568, 26369, 7716,  25810, 40667, 16863, 11588,   /*[GB+8540, GB+854F]*/
  39640, 10236, 12561, 5451,  10300, 18630, 18639, 30004, 6799,  15332, 15965, 33903, 16353, 36770, 26394, 36358,   /*[GB+8550, GB+855F]*/
  22912, 43799, 28858, 33904, 32430, 32431, 35020, 5425,  13147, 29319, 16908, 33529, 35160, 14543, 267,   26120,   /*[GB+8560, GB+856F]*/
  28280, 3750,  35832, 26755, 33430, 13126, 7383,  33623, 27613, 32712, 41526, 10709, 36480, 5982,  41029,          /*[GB+8570, GB+857E]*/
  23678, 36449, 25805, 36454, 42261, 23681, 31017, 28340, 31441, 43910, 8002,  2565,  40055, 18370, 6113,  14855,   /*[GB+8580, GB+858F]*/
  311,   10255, 36946, 18385, 3124,  17459, 18401, 36703, 36866, 40127, 9473,  19110, 27235, 26582, 7576,  17896,   /*[GB+8590, GB+859F]*/
  6738,  43083, 2364,  2365,  2369,  97,    5393,  29006, 9952,  463,   27483, 28866, 43319, 34082, 29801, 13128,   /*[GB+85A0, GB+85AF]*/
  32857, 27459, 4873,  9907,  26250, 3744,  18945, 11286, 21751, 20866, 35961, 43548, 5140,  37509, 23483, 12764,   /*[GB+85B0, GB+85BF]*/
  37673, 4105,  25992, 14728, 2231,  7279,  11341, 33276, 22330, 36202, 6640,  11288, 9180,  16047, 20207, 9256,    /*[GB+85C0, GB+85CF]*/
  21086, 25022, 33279, 14434, 11289, 3747,  28563, 31978, 33075, 7357,  28667, 15711, 23852, 33036, 20614, 26849,   /*[GB+85D0, GB+85DF]*/
  5987,  28873, 42742, 31422, 33913, 38074, 25079, 24425, 43563, 35786, 38075, 34760, 5244,  26586, 10646, 32053,   /*[GB+85E0, GB+85EF]*/
  24619, 1750,  8459,  37040, 11035, 10537, 5212,  32250, 44085, 2233,  30461, 34482, 12809, 25086, 7611,           /*[GB+85F0, GB+85FE]*/
  7363,  33923, 42751, 29212, 34942, 16766, 12042, 30873, 7364,  36068, 10397, 17753, 31565, 35542, 39352, 6447,    /*[GB+8640, GB+864F]*/
  22863, 35529, 8076,  10957, 40047, 19760, 24621, 20884, 43805, 11060, 34506, 17700, 1954,  18853, 442,   21753,   /*[GB+8650, GB+865F]*/
  19016, 30019, 8498,  10614, 29445, 39364, 9705,  9766,  1011,  10658, 38138, 13629, 31716, 34306, 32370, 18320,   /*[GB+8660, GB+866F]*/
  26281, 3245,  40749, 36484, 6643,  25031, 5925,  25999, 22051, 9577,  6677,  26591, 18831, 17689, 27863,          /*[GB+8670, GB+867E]*/
  43860, 33131, 14908, 25106, 7552,  25632, 205,   40791, 22917, 39402, 31268, 17395, 33572, 30,    23294, 43335,   /*[GB+8680, GB+868F]*/
  6041,  11252, 2331,  34617, 31982, 33362, 33064, 27926, 9773,  25036, 25037, 5527,  37553, 43577, 1196,  5073,    /*[GB+8690, GB+869F]*/
  11075, 36486, 25040, 41557, 18835, 34329, 38746, 9926,  41633, 10405, 20451, 36656, 5894,  26646, 2939,  15443,   /*[GB+86A0, GB+86AF]*/
  30719, 196,   5535,  39764, 41565, 42677, 2782,  32283, 39739, 40518, 37233, 21668, 11670, 40399, 12187, 12188,   /*[GB+86B0, GB+86BF]*/
  12968, 42598, 32648, 27933, 35873, 12266, 7145,  23019, 18857, 39765, 27699, 3638,  25706, 36924, 5441,  23854,   /*[GB+86C0, GB+86CF]*/
  2370,  38998, 32572, 38849, 2209,  33798, 14503, 40760, 11851, 31041, 37286, 11184, 25552, 27772, 39006, 11440,   /*[GB+86D0, GB+86DF]*/
  35055, 20696, 2661,  10592, 27355, 32222, 12985, 23700, 33252, 25454, 43460, 17788, 32269, 0,     30255, 42116,   /*[GB+86E0, GB+86EF]*/
  27597, 9356,  22313, 25545, 1334,  315,   18621, 43916, 41519, 22080, 30646, 14456, 3926,  14278, 16153,          /*[GB+86F0, GB+86FE]*/
  30676, 28028, 36,    34963, 19801, 34816, 11614, 12578, 10415, 23491, 34242, 40811, 3090,  24523, 5789,  20677,   /*[GB+8740, GB+874F]*/
  11692, 8846,  30695, 37516, 27583, 14457, 34964, 11905, 20718, 26869, 17720, 34968, 13009, 17083, 34829, 10919,   /*[GB+8750, GB+875F]*/
  8681,  25713, 4290,  2965,  5552,  36392, 7338,  43997, 7689,  3656,  12424, 40612, 4426,  4955,  5548,  39838,   /*[GB+8760, GB+876F]*/
  31986, 33669, 25144, 10913, 18640, 35736, 12428, 38593, 36331, 23050, 29535, 36846, 38981, 5228,  41187,          /*[GB+8770, GB+877E]*/
  42787, 23250, 12606, 35226, 14468, 40170, 25,    39024, 16822, 39675, 23843, 5766,  11285, 7062,  5604,  35352,   /*[GB+8780, GB+878F]*/
  24149, 38529, 43940, 23160, 5955,  17588, 30327, 12772, 34184, 37617, 7134,  24191, 4075,  13485, 14955, 3101,    /*[GB+8790, GB+879F]*/
  36259, 21080, 31176, 19997, 12642, 2118,  39164, 23075, 38603, 11876, 21084, 11326, 41610, 18134, 19525, 10609,   /*[GB+87A0, GB+87AF]*/
  22514, 21997, 36996, 18484, 19929, 19674, 22002, 5581,  3441,  24380, 24154, 34743, 22003, 34027, 7211,  16748,   /*[GB+87B0, GB+87BF]*/
  36716, 2891,  38811, 6169,  17264, 30441, 34872, 4620,  12069, 12893, 43161, 23094, 34873, 2307,  18163, 3001,    /*[GB+87C0, GB+87CF]*/
  2803,  18543, 38431, 20454, 40537, 29852, 33895, 40902, 13853, 40538, 42961, 17515, 23107, 17570, 19616, 12450,   /*[GB+87D0, GB+87DF]*/
  38467, 23122, 31852, 16309, 39978, 15166, 24214, 40345, 4763,  12453, 7280,  10436, 31538, 31747, 10438, 26825,   /*[GB+87E0, GB+87EF]*/
  27103, 10439, 30462, 10442, 38555, 12738, 10659, 20413, 10444, 4359,  32649, 40058, 26616, 16687, 24707,          /*[GB+87F0, GB+87FE]*/
  40072, 40073, 36545, 31776, 31791, 31792, 31866, 20380, 12593, 38329, 20322, 20340, 36497, 31825, 31503, 24654,   /*[GB+8840, GB+884F]*/
  20017, 16794, 36406, 39701, 25248, 25249, 43311, 5647,  25206, 34168, 27988, 22992, 22029, 13617, 42173, 36242,   /*[GB+8850, GB+885F]*/
  38048, 25964, 21087, 27197, 31998, 1032,  1228,  15045, 6018,  14977, 13335, 5983,  14978, 666,   30712, 26375,   /*[GB+8860, GB+886F]*/
  42003, 9922,  30471, 15149, 37147, 8342,  20023, 26206, 9649,  36205, 466,   19249, 42977, 8708,  11574,          /*[GB+8870, GB+887E]*/
  42262, 379,   5346,  23723, 21908, 13353, 11173, 20255, 4656,  3753,  38475, 11542, 6960,  41471, 8479,  10087,   /*[GB+8880, GB+888F]*/
  37259, 7172,  10220, 2687,  37085, 35023, 29018, 1916,  11359, 5317,  16118, 5654,  10854, 202,   35417, 34507,   /*[GB+8890, GB+889F]*/
  694,   8387,  507,   38140, 38691, 10747, 35850, 4468,  25913, 1094,  7878,  26757, 39087, 16053, 13570, 6032,    /*[GB+88A0, GB+88AF]*/
  17672, 15714, 14807, 41545, 19018, 11899, 2249,  15153, 37513, 5895,  36426, 26647, 30083, 26025, 32374, 15600,   /*[GB+88B0, GB+88BF]*/
  20438, 41871, 16574, 3982,  6547,  30668, 2348,  4259,  1095,  16215, 7177,  38164, 42101, 43276, 24869, 15601,   /*[GB+88C0, GB+88CF]*/
  22710, 16460, 17122, 22811, 13737, 9079,  39742, 7384,  9844,  18018, 7910,  36461, 17123, 233,   28428, 22597,   /*[GB+88D0, GB+88DF]*/
  31727, 3543,  12709, 1308,  18750, 10406, 43129, 813,   39475, 20965, 14505, 27371, 37559, 9453,  43639, 39476,   /*[GB+88E0, GB+88EF]*/
  12267, 7399,  37278, 854,   4665,  21106, 3120,  32122, 38700, 8296,  42629, 14761, 14812, 11284, 9087,           /*[GB+88F0, GB+88FE]*/
  4504,  13890, 24452, 17905, 34721, 12245, 6935,  32313, 35976, 13408, 13177, 16810, 38863, 39105, 16156, 29898,   /*[GB+8940, GB+894F]*/
  28785, 21558, 33101, 3558,  31777, 26774, 42601, 18387, 42602, 742,   40696, 6966,  31233, 33460, 41969, 36250,   /*[GB+8950, GB+895F]*/
  41828, 88,    9509,  36676, 16222, 32965, 35198, 19592, 10567, 17662, 3091,  23904, 1120,  3386,  20074, 19980,   /*[GB+8960, GB+896F]*/
  23452, 25501, 21124, 22081, 43094, 29371, 19846, 1635,  2583,  42407, 41418, 16249, 6200,  3413,  42122,          /*[GB+8970, GB+897E]*/
  33968, 10411, 25623, 14930, 6075,  5024,  36947, 30339, 40904, 25402, 32884, 43245, 25667, 28126, 28127, 491,     /*[GB+8980, GB+898F]*/
  24671, 16818, 6585,  7742,  26791, 7057,  43977, 6076,  28648, 7247,  7252,  30590, 5859,  22206, 8141,  12282,   /*[GB+8990, GB+899F]*/
  30591, 5321,  37566, 43032, 14059, 395,   25602, 25672, 16515, 38330, 24103, 6208,  14157, 37519, 36180, 30599,   /*[GB+89A0, GB+89AF]*/
  17548, 15641, 12045, 5683,  26935, 25517, 36260, 34576, 33885, 111,   36443, 5733,  27378, 14956, 17921, 16940,   /*[GB+89B0, GB+89BF]*/
  19930, 36717, 30617, 32822, 12047, 19746, 19747, 27464, 18486, 19163, 2892,  36271, 36734, 17882, 534,   32327,   /*[GB+89C0, GB+89CF]*/
  27667, 43186, 37675, 20725, 42986, 43190, 11661, 17171, 11672, 35875, 17177, 29045, 20924, 43978, 29046, 37716,   /*[GB+89D0, GB+89DF]*/
  9748,  14204, 8291,  41093, 1567,  30013, 26824, 19250, 5190,  35573, 35263, 22576, 34193, 40142, 20995, 7117,    /*[GB+89E0, GB+89EF]*/
  26049, 32297, 9659,  25115, 21389, 21390, 3416,  40860, 9951,  9150,  30821, 30509, 36755, 1176,  11970,          /*[GB+89F0, GB+89FE]*/
  40389, 13554, 424,   7516,  5884,  23752, 15732, 35024, 6147,  26608, 12170, 16125, 40940, 1026,  12346, 7965,    /*[GB+8A40, GB+8A4F]*/
  6245,  36379, 1065,  384,   33100, 10533, 33461, 386,   14183, 18631, 7152,  40330, 14184, 28965, 8197,  12857,   /*[GB+8A50, GB+8A5F]*/
  1439,  20323, 7213,  6472,  25250, 15257, 23358, 2736,  34202, 7800,  13284, 11339, 5419,  8369,  15079, 10577,   /*[GB+8A60, GB+8A6F]*/
  33059, 42547, 23513, 16530, 42616, 40346, 8353,  3620,  40207, 43167, 36898, 22332, 35317, 8107,  8220,           /*[GB+8A70, GB+8A7E]*/
  32435, 6864,  468,   43320, 41431, 22413, 40145, 31683, 34301, 42090, 7267,  22031, 1239,  28402, 7268,  11038,   /*[GB+8A80, GB+8A8F]*/
  35787, 7638,  41864, 21809, 668,   8371,  19251, 43553, 26892, 27995, 37041, 20756, 43437, 15566, 27050, 6605,    /*[GB+8A90, GB+8A9F]*/
  13129, 8864,  34710, 11922, 15682, 9654,  13728, 13729, 22916, 42093, 12942, 34495, 11258, 10088, 16018, 16755,   /*[GB+8AA0, GB+8AAF]*/
  36903, 21946, 19007, 23822, 7368,  39354, 36641, 28410, 28765, 10221, 26635, 29665, 11360, 34508, 28287, 3517,    /*[GB+8AB0, GB+8ABF]*/
  35162, 20885, 8388,  21097, 16454, 4591,  4592,  31511, 35476, 22456, 39454, 24602, 22655, 42292, 10660, 43171,   /*[GB+8AC0, GB+8ACF]*/
  24365, 31961, 34509, 33284, 36642, 33364, 14,    36650, 39455, 29609, 18322, 15366, 43229, 24750, 34315, 43336,   /*[GB+8AD0, GB+8ADF]*/
  6548,  20037, 52,    7270,  19782, 21616, 4850,  24622, 15444, 2332,  19282, 34788, 29140, 25038, 12399, 7823,    /*[GB+8AE0, GB+8AEF]*/
  33153, 27419, 30572, 7916,  7917,  31193, 22711, 26649, 15087, 14993, 25282, 6183,  35515, 11839, 32310,          /*[GB+8AF0, GB+8AFE]*/
  17346, 38502, 4009,  22628, 8735,  20414, 234,   17124, 38563, 36523, 15367, 18346, 6152,  34316, 11900, 11983,   /*[GB+8B40, GB+8B4F]*/
  5661,  37375, 22505, 4593,  13680, 31658, 39767, 32650, 6053,  27241, 21157, 5442,  27390, 25884, 12400, 33183,   /*[GB+8B50, GB+8B5F]*/
  25390, 4505,  21710, 8740,  14719, 6913,  37851, 42630, 12268, 21618, 164,   38748, 35977, 14506, 32575, 41897,   /*[GB+8B60, GB+8B6F]*/
  26216, 28927, 35185, 32173, 18751, 21041, 26904, 29512, 24217, 32865, 32220, 4941,  11673, 14608, 814,            /*[GB+8B70, GB+8B7E]*/
  35703, 31659, 10160, 4169,  37287, 40075, 27130, 27356, 19487, 21161, 23603, 363,   20617, 17070, 25929, 41703,   /*[GB+8B80, GB+8B8F]*/
  40076, 14507, 27138, 21924, 38759, 29902, 22946, 30841, 23686, 17619, 22602, 786,   24175, 38275, 39814, 36047,   /*[GB+8B90, GB+8B9F]*/
  20754, 37706, 16273, 39013, 22820, 10174, 14858, 43095, 40812, 10626, 17791, 41520, 11857, 11858, 387,   22682,   /*[GB+8BA0, GB+8BAF]*/
  20698, 24344, 9729,  33380, 25714, 32180, 41252, 21715, 34349, 34350, 22101, 18898, 18641, 12003, 10180, 5827,    /*[GB+8BB0, GB+8BBF]*/
  42130, 35738, 37711, 11989, 33670, 17084, 26955, 36956, 2863,  14283, 21164, 7843,  7690,  34255, 38331, 12608,   /*[GB+8BC0, GB+8BCF]*/
  14469, 8799,  28975, 30105, 25603, 18725, 12118, 35353, 22955, 6586,  38332, 2376,  112,   22928, 23161, 20619,   /*[GB+8BD0, GB+8BDF]*/
  31387, 4076,  14957, 4691,  39549, 24377, 27171, 27315, 22421, 36608, 30497, 38796, 25518, 40269, 21630, 1170,    /*[GB+8BE0, GB+8BEF]*/
  28519, 35527, 22843, 6806,  19583, 40021, 17557, 36997, 19375, 14419, 22929, 17559, 25348, 38812, 29351,          /*[GB+8BF0, GB+8BFE]*/
  12665, 26692, 21523, 18172, 20324, 36735, 42918, 17565, 15802, 20616, 35376, 29601, 30463, 39347, 43111, 30168,   /*[GB+8C40, GB+8C4F]*/
  21754, 4851,  25283, 36427, 22765, 43475, 18085, 36181, 2175,  22405, 23081, 38813, 20325, 27195, 26122, 31709,   /*[GB+8C50, GB+8C5F]*/
  27276, 37352, 37755, 15047, 43266, 42932, 11349, 28760, 37805, 31375, 28828, 3518,  26834, 9494,  34772, 41070,   /*[GB+8C60, GB+8C6F]*/
  806,   28493, 12343, 21542, 16636, 2333,  40606, 39988, 26026, 11178, 41699, 14026, 23144, 1925,  12102,          /*[GB+8C70, GB+8C7E]*/
  25972, 28781, 23145, 14920, 23147, 42383, 39658, 816,   16836, 23148, 15621, 25978, 11615, 28792, 23150, 28508,   /*[GB+8C80, GB+8C8F]*/
  32791, 35134, 16837, 12594, 16088, 38361, 37901, 830,   25900, 3989,  832,   8275,  20253, 6995,  24537, 43086,   /*[GB+8C90, GB+8C9F]*/
  8389,  16455, 14142, 43089, 36315, 29235, 7014,  5730,  14545, 7577,  7578,  29067, 28219, 21971, 18975, 34426,   /*[GB+8CA0, GB+8CAF]*/
  34427, 17197, 32473, 18969, 20872, 32478, 32475, 32476, 37255, 7221,  17063, 42599, 8960,  9780,  31920, 8974,    /*[GB+8CB0, GB+8CBF]*/
  8975,  33909, 1137,  14739, 31209, 6277,  28673, 41760, 35164, 22705, 33556, 32737, 20807, 19865, 24446, 31133,   /*[GB+8CC0, GB+8CCF]*/
  8009,  35186, 20235, 2627,  35227, 15657, 15797, 18922, 15955, 29194, 34041, 22793, 32208, 34200, 18264, 7347,    /*[GB+8CD0, GB+8CDF]*/
  12509, 12510, 19618, 27032, 33412, 10734, 28395, 4134,  27033, 40208, 641,   37353, 268,   33416, 14558, 7350,    /*[GB+8CE0, GB+8CEF]*/
  8108,  32342, 25367, 24831, 14732, 44053, 44054, 37148, 15567, 9063,  36165, 24495, 18286, 31336, 26378,          /*[GB+8CF0, GB+8CFE]*/
  8461,  32092, 24047, 380,   16914, 26379, 24553, 31337, 19252, 3687,  24428, 10742, 35652, 31558, 36166, 1561,    /*[GB+8D40, GB+8D4F]*/
  11047, 437,   20552, 7369,  8717,  6342,  20026, 7524,  7542,  8865,  26609, 37787, 22218, 28766, 154,   32631,   /*[GB+8D50, GB+8D5F]*/
  12090, 21540, 18211, 13356, 31566, 32632, 39356, 25201, 18212, 37261, 14252, 7287,  35025, 2186,  26826, 8234,    /*[GB+8D60, GB+8D6F]*/
  22555, 18215, 39212, 34510, 27122, 5717,  28419, 3519,  31752, 9423,  9187,  34089, 38557, 39635, 17673,          /*[GB+8D70, GB+8D7E]*/
  16247, 34410, 26758, 3961,  30366, 19117, 11979, 15370, 17347, 21823, 17125, 17126, 43808, 9845,  36462, 9098,    /*[GB+8D80, GB+8D8F]*/
  20415, 20416, 18001, 7125,  41886, 38565, 6549,  10665, 41887, 32748, 24070, 14596, 43809, 16685, 6550,  38566,   /*[GB+8D90, GB+8D9F]*/
  43449, 40877, 12269, 39477, 37105, 8248,  26311, 37106, 31027, 37969, 42356, 28928, 7400,  43016, 16198, 20263,   /*[GB+8DA0, GB+8DAF]*/
  36795, 21158, 10675, 12971, 12189, 31473, 33452, 39478, 43641, 17448, 16432, 36666, 36667, 32756, 43642, 2699,    /*[GB+8DB0, GB+8DBF]*/
  30087, 16373, 25885, 24892, 31778, 6968,  33628, 33103, 2476,  5666,  27139, 14622, 16157, 19488, 33462, 25652,   /*[GB+8DC0, GB+8DCF]*/
  1077,  6114,  25455, 39106, 23032, 5161,  28786, 27492, 29717, 43657, 14222, 18888, 16275, 2959,  6392,  2602,    /*[GB+8DD0, GB+8DDF]*/
  6481,  31733, 19848, 41243, 41244, 316,   2514,  26416, 25557, 5026,  43931, 5725,  5726,  33808, 39815, 23836,   /*[GB+8DE0, GB+8DEF]*/
  19655, 34731, 2628,  1979,  25893, 14284, 36598, 17723, 41311, 18900, 18901, 14819, 7248,  43979, 14463,          /*[GB+8DF0, GB+8DFE]*/
  10305, 37312, 14285, 37313, 15909, 41188, 38333, 36182, 22569, 37604, 37605, 37890, 23062, 34440, 13205, 35238,   /*[GB+8E40, GB+8E4F]*/
  16437, 33673, 6091,  396,   43938, 32605, 27172, 5734,  19443, 14673, 39676, 40273, 38663, 27368, 14684, 18446,   /*[GB+8E50, GB+8E5F]*/
  10192, 19676, 19677, 27183, 33751, 15506, 2893,  38959, 16979, 36719, 22580, 26693, 3223,  5001,  20326, 6136,    /*[GB+8E60, GB+8E6F]*/
  6137,  23097, 36737, 36736, 36886, 17037, 36887, 16792, 12228, 14979, 3181,  19055, 15564, 24733, 24734,          /*[GB+8E70, GB+8E7E]*/
  26331, 41998, 41432, 37807, 14847, 36365, 15734, 443,   8446,  40484, 6455,  22991, 8069,  23539, 42174, 11808,   /*[GB+8E80, GB+8E8F]*/
  5515,  5348,  22032, 1945,  22043, 38107, 38108, 24434, 25211, 15712, 27283, 29307, 29404, 25758, 41683, 28680,   /*[GB+8E90, GB+8E9F]*/
  26835, 5363,  10142, 24447, 41411, 27656, 32311, 5369,  27928, 24874, 6845,  11489, 41967, 35704, 32652, 38200,   /*[GB+8EA0, GB+8EAF]*/
  704,   24453, 6370,  9510,  23604, 30843, 21561, 13707, 30957, 12415, 42573, 28029, 1636,  10454, 40813, 22274,   /*[GB+8EB0, GB+8EBF]*/
  708,   15061, 2966,  42430, 11623, 1361,  1383,  41391, 21586, 25673, 2827,  8142,  21288, 713,   4077,  21778,   /*[GB+8EC0, GB+8ECF]*/
  4194,  14664, 34453, 17495, 13717, 1915,  9032,  13305, 31433, 40862, 10111, 21969, 25904, 7085,  36481, 573,     /*[GB+8ED0, GB+8EDF]*/
  14735, 43168, 36412, 4731,  31560, 37954, 42274, 31320, 42275, 34776, 31753, 10231, 16727, 20887, 39297, 2187,    /*[GB+8EE0, GB+8EEF]*/
  1917,  3603,  17348, 13149, 29672, 29228, 26079, 39770, 21746, 29751, 2570,  34620, 8012,  15301, 7422,           /*[GB+8EF0, GB+8EFE]*/
  10161, 19593, 27936, 29752, 42384, 2265,  26082, 15312, 15302, 19804, 38707, 5389,  20076, 38276, 4182,  31793,   /*[GB+8F40, GB+8F4F]*/
  29537, 35348, 21748, 3125,  33381, 8032,  10113, 16742, 16824, 1407,  25604, 19186, 18923, 19932, 13506, 38960,   /*[GB+8F50, GB+8F5F]*/
  34275, 31459, 39048, 18174, 31461, 36279, 5918,  23565, 12455, 22414, 12458, 8178,  15523, 36772, 40692, 37667,   /*[GB+8F60, GB+8F6F]*/
  7607,  27607, 7608,  28943, 6283,  11793, 33228, 31906, 14205, 504,   28486, 15814, 30825, 8648,  5989,           /*[GB+8F70, GB+8F7E]*/
  3160,  41756, 37808, 26610, 28228, 35968, 15094, 6246,  41346, 14210, 25590, 23896, 1309,  28350, 5536,  13904,   /*[GB+8F80, GB+8F8F]*/
  9234,  7634,  1335,  16618, 13914, 1781,  34841, 5553,  10416, 14225, 11424, 21448, 10427, 32326, 15980, 13326,   /*[GB+8F90, GB+8F9F]*/
  20024, 12565, 12566, 37891, 37892, 37913, 40193, 40195, 32992, 36904, 39730, 3627,  6247,  39107, 24271, 3676,    /*[GB+8FA0, GB+8FAF]*/
  43312, 32165, 13109, 42549, 32490, 5886,  19254, 8464,  32505, 33542, 32509, 3599,  42294, 15095, 4847,  42540,   /*[GB+8FB0, GB+8FBF]*/
  4853,  6557,  13393, 42318, 4854,  42666, 17349, 35172, 14755, 14017, 28931, 13681, 1577,  8742,  32757, 23688,   /*[GB+8FC0, GB+8FCF]*/
  33631, 41904, 24319, 31062, 5807,  41911, 1763,  3928,  3292,  14386, 12609, 21136, 19770, 34637, 877,   26534,   /*[GB+8FD0, GB+8FDF]*/
  17812, 26987, 6493,  8883,  26990, 8930,  25067, 28861, 33517, 7684,  38739, 21811, 33034, 33426, 13110, 33914,   /*[GB+8FE0, GB+8FEF]*/
  13618, 32436, 4769,  39710, 24832, 5060,  30510, 31999, 25369, 12701, 35538, 16888, 34206, 16263, 15568,          /*[GB+8FF0, GB+8FFE]*/
  38082, 801,   21812, 23776, 15575, 669,   22547, 31402, 11824, 15576, 4029,  6025,  31713, 39205, 8377,  33915,   /*[GB+9040, GB+904F]*/
  1243,  39146, 36090, 4773,  1871,  12338, 35833, 1244,  29215, 33543, 30537, 39080, 22154, 42183, 28887, 33930,   /*[GB+9050, GB+905F]*/
  17755, 11259, 16865, 22162, 42190, 35028, 31322, 6344,  10958, 10222, 33544, 14337, 29505, 11583, 26211, 12459,   /*[GB+9060, GB+906F]*/
  3814,  37790, 35543, 9956,  19203, 3757,  28221, 26611, 40218, 23862, 5792,  12536, 34990, 20880, 39088,          /*[GB+9070, GB+907E]*/
  39985, 23993, 40222, 39745, 31755, 14753, 41558, 19204, 10959, 25818, 31135, 2250,  25485, 33558, 20757, 37690,   /*[GB+9080, GB+908F]*/
  11250, 3600,  33440, 18039, 18854, 4776,  38168, 18324, 22589, 7385,  33442, 4855,  5798,  3142,  21243, 18347,   /*[GB+9090, GB+909F]*/
  10039, 10034, 24751, 16577, 31271, 20417, 33598, 16217, 10351, 22816, 7052,  10489, 41169, 39996, 14811, 13409,   /*[GB+90A0, GB+90AF]*/
  17445, 39746, 11095, 26650, 30573, 31141, 23011, 4594,  12686, 35334, 7403,  30250, 43701, 13738, 39099, 5799,    /*[GB+90B0, GB+90BF]*/
  22600, 40444, 6372,  4569,  26144, 2571,  1531,  21878, 43644, 31028, 972,   35979, 32577, 9278,  32758, 1311,    /*[GB+90C0, GB+90CF]*/
  35980, 12190, 21670, 39100, 16178, 5662,  38504, 3373,  21051, 25227, 16472, 90,    25831, 36797, 23411, 43702,   /*[GB+90D0, GB+90DF]*/
  27578, 38749, 26783, 9511,  40371, 29917, 24902, 37291, 29713, 12362, 13179, 9782,  15622, 4456,  22821, 35057,   /*[GB+90E0, GB+90EF]*/
  16158, 41950, 39108, 2539,  36389, 28542, 2056,  16181, 33961, 12749, 39112, 37167, 18389, 27704, 30844,          /*[GB+90F0, GB+90FE]*/
  38515, 35895, 30525, 40448, 21952, 9594,  4861,  24272, 22275, 10456, 3863,  2428,  2406,  2401,  5028,  21892,   /*[GB+9140, GB+914F]*/
  41355, 31653, 388,   29372, 10046, 26792, 40761, 15316, 18624, 23505, 29719, 25936, 38715, 20276, 28184, 31867,   /*[GB+9150, GB+915F]*/
  20765, 25308, 28364, 26104, 6085,  42131, 19805, 15767, 24785, 24776, 39841, 24468, 18890, 4890,  39158, 3929,    /*[GB+9160, GB+916F]*/
  42432, 31654, 3472,  25145, 26417, 23916, 995,   26175, 14286, 3852,  18644, 24473, 17085, 12596, 3574,           /*[GB+9170, GB+917E]*/
  38720, 38721, 33876, 33690, 5554,  30593, 7249,  7019,  29928, 15929, 2590,  34831, 7692,  8195,  17729, 17793,   /*[GB+9180, GB+918F]*/
  34351, 21893, 18957, 33382, 2431,  15910, 4956,  34566, 30649, 28608, 24034, 38335, 34256, 22570, 5556,  30652,   /*[GB+9190, GB+919F]*/
  15064, 29687, 14398, 32902, 36007, 25953, 25944, 15642, 2542,  16518, 38797, 38336, 19188, 27794, 16089, 12028,   /*[GB+91A0, GB+91AF]*/
  113,   36613, 17235, 34185, 3889,  39677, 38727, 5328,  21373, 122,   21289, 7030,  24958, 22018, 17473, 21250,   /*[GB+91B0, GB+91BF]*/
  4078,  42480, 23428, 36609, 37175, 2119,  42499, 16943, 16907, 39332, 8436,  19526, 21779, 3590,  12678, 3019,    /*[GB+91C0, GB+91CF]*/
  17558, 12032, 36062, 26947, 3024,  13507, 15676, 12070, 28385, 18788, 22501, 21477, 30806, 15982, 9111,  43204,   /*[GB+91D0, GB+91DF]*/
  40201, 9208,  13722, 6618,  6362,  40884, 13640, 42041, 17005, 41304, 25558, 9237,  13915, 39816, 36840, 11624,   /*[GB+91E0, GB+91EF]*/
  41316, 33993, 34002, 26537, 11794, 11796, 7348,  28868, 31122, 20998, 6178,  28207, 37809, 37961, 36798,          /*[GB+91F0, GB+91FE]*/
  26253, 17813, 7786,  27349, 40576, 41402, 6274,  39393, 16666, 8982,  28396, 2646,  32044, 9751,  33414, 5818,    /*[GB+9240, GB+924F]*/
  27035, 33518, 25371, 13619, 39424, 377,   33526, 42177, 42256, 5819,  36760, 16889, 1554,  15342, 33035, 11644,   /*[GB+9250, GB+925F]*/
  40209, 25995, 5480,  41941, 40390, 32344, 22641, 24606, 24607, 465,   357,   40778, 40482, 1872,  42184, 22451,   /*[GB+9260, GB+926F]*/
  2239,  37357, 41751, 40939, 37042, 11010, 22753, 37535, 3807,  23978, 13555, 21173, 26380, 4243,  13559,          /*[GB+9270, GB+927E]*/
  41030, 23712, 22303, 25372, 32055, 574,   5991,  15577, 17200, 25199, 25270, 11236, 23128, 9548,  5134,  37791,   /*[GB+9280, GB+928F]*/
  35029, 2558,  12382, 546,   42276, 17202, 7175,  7176,  23275, 41809, 9261,  14438, 6609,  22307, 19008, 40941,   /*[GB+9290, GB+929F]*/
  20211, 6345,  15829, 18993, 15530, 36514, 34307, 39627, 32059, 2021,  9425,  31111, 3527,  27509, 16537, 21153,   /*[GB+92A0, GB+92AF]*/
  23276, 15355, 23891, 13879, 28002, 27369, 35034, 2654,  8293,  14343, 15294, 12814, 31756, 24608, 40873, 29222,   /*[GB+92B0, GB+92BF]*/
  37487, 15843, 32350, 2251,  44003, 37548, 41095, 20212, 29747, 32060, 698,   32739, 32439, 25919, 28333, 21824,   /*[GB+92C0, GB+92CF]*/
  21218, 8661,  5712,  33154, 15716, 40230, 43699, 3320,  31723, 1037,  27482, 44020, 33179, 24754, 3533,  27733,   /*[GB+92D0, GB+92DF]*/
  20393, 26096, 9080,  6000,  12740, 24875, 29090, 25292, 19285, 36528, 41888, 18836, 9927,  38169, 12825, 28110,   /*[GB+92E0, GB+92EF]*/
  41945, 2336,  1861,  31137, 42667, 16522, 23531, 14496, 21964, 14599, 21407, 27702, 23959, 23407, 13891,          /*[GB+92F0, GB+92FE]*/
  41898, 15231, 13892, 36677, 22486, 11299, 24168, 32578, 27579, 22981, 28933, 43703, 41829, 36317, 39064, 1490,    /*[GB+9340, GB+934F]*/
  37107, 12191, 36799, 40608, 35708, 16420, 13137, 31144, 17262, 17278, 3374,  15236, 31770, 12401, 9386,  3995,    /*[GB+9350, GB+935F]*/
  34808, 34996, 25385, 2665,  892,   37279, 32579, 1136,  33067, 43704, 9387,  9669,  26007, 27140, 26775, 4015,    /*[GB+9360, GB+936F]*/
  30185, 30170, 27132, 744,   16381, 37292, 5723,  42049, 23337, 17251, 26354, 34428, 14922, 21769, 7424,           /*[GB+9370, GB+937E]*/
  4473,  28021, 30378, 41038, 23607, 18390, 30845, 11676, 42386, 32225, 11938, 33069, 25619, 31235, 41705, 7425,    /*[GB+9380, GB+938F]*/
  23417, 26667, 41079, 33371, 28355, 16299, 29761, 27541, 2961,  30215, 15238, 3927,  4409,  9943,  8300,  6077,    /*[GB+9390, GB+939F]*/
  25135, 29768, 18722, 3473,  3644,  10047, 20079, 19849, 43711, 8908,  11860, 40959, 4442,  30815, 11997, 22403,   /*[GB+93A0, GB+93AF]*/
  14149, 10175, 38940, 325,   42434, 23051, 20836, 3015,  16619, 4144,  28365, 31868, 14376, 2449,  16552, 1637,    /*[GB+93B0, GB+93BF]*/
  14223, 37299, 9680,  25311, 13011, 24333, 17692, 7063,  34559, 27375, 10307, 40613, 38302, 34352, 3480,  22567,   /*[GB+93C0, GB+93CF]*/
  11461, 10128, 5231,  22895, 19185, 41951, 12425, 14387, 13313, 2503,  5495,  5496,  1980,  3259,  15795, 8682,    /*[GB+93D0, GB+93DF]*/
  1049,  8683,  43363, 32783, 24639, 26011, 7059,  11939, 14388, 30402, 25777, 33142, 13928, 39027, 22527, 19992,   /*[GB+93E0, GB+93EF]*/
  43364, 40831, 24642, 4612,  13046, 5632,  27795, 26062, 14526, 16789, 5461,  35078, 16114, 402,   15643,          /*[GB+93F0, GB+93FE]*/
  37592, 7466,  21253, 21449, 13315, 30498, 5735,  17555, 15550, 37502, 27344, 37628, 37621, 22775, 33204, 14675,   /*[GB+9440, GB+944F]*/
  1842,  23162, 9250,  42481, 42459, 17228, 21999, 14090, 35103, 19074, 30559, 570,   29820, 19999, 20389, 26968,   /*[GB+9450, GB+945F]*/
  31007, 37176, 17982, 2297,  29130, 40618, 22902, 34454, 16101, 18491, 17304, 12160, 38896, 19933, 19749, 25343,   /*[GB+9460, GB+946F]*/
  25344, 25349, 17497, 34276, 38814, 21142, 2832,  33113, 4993,  35115, 28386, 20492, 16108, 21478, 3674,           /*[GB+9470, GB+947E]*/
  40622, 20329, 30561, 18546, 6138,  32241, 5636,  14427, 17567, 18574, 10266, 10224, 24760, 36337, 16324, 37944,   /*[GB+9480, GB+948F]*/
  8933,  605,   24555, 6108,  16633, 21867, 33433, 11062, 2562,  4254,  9267,  35861, 14447, 3367,  28429, 7143,    /*[GB+9490, GB+949F]*/
  39645, 581,   35862, 7144,  23022, 6157,  7128,  38203, 16433, 36927, 12973, 31688, 34953, 7129,  37116, 34115,   /*[GB+94A0, GB+94AF]*/
  21885, 37,    38,    5947,  41835, 29259, 18904, 26428, 35575, 37991, 14399, 28134, 14400, 43389, 38362, 18726,   /*[GB+94B0, GB+94BF]*/
  1424,  18149, 34982, 34983, 36162, 24839, 24858, 15855, 41099, 17622, 617,   39689, 17506, 32842, 28565, 13675,   /*[GB+94C0, GB+94CF]*/
  13682, 39650, 14458, 31385, 6722,  38545, 25534, 26465, 41231, 5200,  43346, 43365, 25940, 19144, 43390, 4299,    /*[GB+94D0, GB+94DF]*/
  6951,  42962, 2930,  10852, 23824, 39206, 21176, 24859, 24001, 19475, 8392,  7897,  14997, 41481, 37970, 28239,   /*[GB+94E0, GB+94EF]*/
  14027, 39505, 42460, 7695,  24279, 7697,  41204, 16828, 30140, 39579, 13327, 13396, 12834, 35016, 31316,          /*[GB+94F0, GB+94FE]*/
  30469, 6020,  35681, 2928,  37075, 28761, 21813, 21814, 31979, 4499,  33349, 930,   642,   14561, 28566, 11820,   /*[GB+9540, GB+954F]*/
  7891,  10270, 35963, 8073,  25965, 11577, 36635, 40868, 7892,  28529, 15578, 37089, 1876,  36094, 24556, 41757,   /*[GB+9550, GB+955F]*/
  19260, 10955, 22044, 1568,  35841, 12680, 43725, 28891, 7997,  6346,  32963, 6541,  51,    1877,  270,   19626,   /*[GB+9560, GB+956F]*/
  16920, 31376, 28773, 12345, 36071, 16987, 35694, 14345, 14899, 42298, 31572, 11437, 8868,  34679, 10760,          /*[GB+9570, GB+957E]*/
  16060, 34519, 17127, 42763, 33580, 28635, 41559, 41560, 32380, 10964, 32390, 31276, 43347, 43949, 42731, 28072,   /*[GB+9580, GB+958F]*/
  22762, 33600, 25044, 41946, 38204, 240,   32391, 19120, 3046,  32521, 34920, 7924,  9456,  37853, 40323, 12408,   /*[GB+9590, GB+959F]*/
  35798, 21886, 37475, 38974, 29168, 32771, 26061, 21053, 22467, 13905, 37117, 4506,  37301, 30259, 24709, 14377,   /*[GB+95A0, GB+95AF]*/
  16161, 9161,  33105, 3147,  25136, 10979, 36948, 18402, 13459, 21246, 40632, 35228, 10983, 22095, 4801,  22829,   /*[GB+95B0, GB+95BF]*/
  41362, 12597, 10800, 36050, 4365,  18908, 30653, 15065, 24334, 19145, 33880, 38337, 13470, 12367, 5394,  37579,   /*[GB+95C0, GB+95CF]*/
  37580, 18428, 30600, 31616, 34927, 8037,  28514, 41505, 10990, 38363, 34693, 35447, 28467, 14404, 872,   15130,   /*[GB+95D0, GB+95DF]*/
  36984, 125,   37622, 27322, 21297, 18470, 3402,  16944, 6424,  18961, 37006, 12894, 19940, 27186, 19687, 20544,   /*[GB+95E0, GB+95EF]*/
  20330, 27984, 30807, 36888, 42920, 40179, 37528, 11579, 29082, 2508,  28581, 2621,  2426,  35878, 12569,          /*[GB+95F0, GB+95FE]*/
  38649, 25840, 8100,  24118, 8111,  19261, 7960,  31127, 23350, 31377, 17659, 43912, 21917, 12242, 31983, 12976,   /*[GB+9640, GB+964F]*/
  38755, 43655, 32554, 31617, 17671, 17733, 19688, 5847,  29201, 22246, 26988, 422,   7167,  5703,  18270, 10211,   /*[GB+9650, GB+965F]*/
  15210, 1177,  3506,  4723,  27351, 25258, 39426, 33236, 36285, 7717,  43555, 27041, 32049, 6021,  20873, 3801,    /*[GB+9660, GB+966F]*/
  38056, 8902,  9481,  18027, 14242, 21613, 32437, 15262, 20874, 29602, 40040, 7993,  6538,  34208, 4500,           /*[GB+9670, GB+967E]*/
  29074, 11973, 35323, 4107,  4108,  29663, 33172, 14889, 12936, 20950, 1247,  284,   8112,  38086, 8469,  22452,   /*[GB+9680, GB+968F]*/
  11821, 36414, 6673,  35362, 37223, 7331,  41470, 40430, 43974, 28567, 6026,  19894, 29664, 41157, 34215, 30475,   /*[GB+9690, GB+969F]*/
  33847, 9947,  770,   22045, 38116, 37543, 37550, 28892, 22997, 1185,  37792, 22757, 17247, 11050, 673,   7718,    /*[GB+96A0, GB+96AF]*/
  42558, 5358,  4659,  37152, 8378,  21022, 28260, 29668, 28411, 22455, 39089, 42278, 40942, 5522,  9702,  2244,    /*[GB+96B0, GB+96BF]*/
  295,   8481,  438,   7223,  1261,  28003, 29610, 4247,  23779, 28875, 9948,  40943, 37363, 15289, 19565, 21094,   /*[GB+96C0, GB+96CF]*/
  18301, 27115, 40726, 28572, 39085, 36302, 25098, 41684, 37810, 34659, 41818, 18327, 27761, 16194, 1016,  27013,   /*[GB+96D0, GB+96DF]*/
  562,   27014, 1919,  43442, 4036,  38145, 4735,  14011, 43944, 7549,  7586,  39298, 7649,  9550,  41164, 19020,   /*[GB+96E0, GB+96EF]*/
  38486, 38146, 28902, 8503,  27285, 37811, 16681, 26128, 27669, 36213, 7225,  43226, 39457, 40605, 0,              /*[GB+96F0, GB+96FE]*/
  38743, 14587, 19566, 41289, 36303, 27873, 41695, 890,   31448, 17205, 15100, 24600, 1038,  27420, 33581, 17623,   /*[GB+9740, GB+974F]*/
  19570, 8239,  24755, 33039, 16024, 8991,  29880, 26299, 39302, 19767, 23871, 43174, 6043,  35972, 31764, 40757,   /*[GB+9750, GB+975F]*/
  296,   1292,  5929,  10668, 42036, 27060, 952,   13885, 12182, 32381, 23381, 13577, 31343, 13397, 34790, 20221,   /*[GB+9760, GB+976F]*/
  12707, 2608,  8120,  29674, 18043, 6701,  25969, 38928, 15374, 17172, 43339, 2815,  32749, 15101, 18044,          /*[GB+9770, GB+977E]*/
  1795,  34175, 8343,  30885, 42323, 17350, 18613, 43294, 18053, 25120, 1892,  20421, 4781,  25492, 24893, 2350,    /*[GB+9780, GB+978F]*/
  2845,  5801,  7969,  23552, 739,   12691, 3546,  40732, 18372, 39774, 39775, 9850,  16066, 6622,  9084,  32522,   /*[GB+9790, GB+979F]*/
  5201,  7731,  3456,  41301, 25045, 40003, 36801, 39776, 26618, 38205, 27018, 24762, 43298, 8543,  17357, 43756,   /*[GB+97A0, GB+97AF]*/
  43757, 8085,  28435, 26058, 22718, 32392, 20054, 10908, 14510, 4048,  15533, 15453, 3547,  44022, 25548, 36433,   /*[GB+97B0, GB+97BF]*/
  893,   43348, 43464, 1798,  23900, 6506,  4212,  3047,  21210, 11902, 13893, 10162, 33950, 5754,  10287, 6158,    /*[GB+97C0, GB+97CF]*/
  42042, 26740, 21959, 8665,  9402,  23960, 28116, 37836, 27839, 7190,  8758,  32772, 32587, 13683, 43653, 13769,   /*[GB+97D0, GB+97DF]*/
  37854, 33802, 36937, 36816, 13770, 39659, 11106, 33191, 24237, 1146,  37239, 12835, 35722, 27491, 37118, 17279,   /*[GB+97E0, GB+97EF]*/
  36678, 1029,  12409, 17007, 14763, 17008, 29518, 8254,  32177, 42324, 22276, 21054, 11849, 11686, 31480,          /*[GB+97F0, GB+97FE]*/
  39508, 39237, 29716, 36113, 37164, 41707, 24238, 37569, 14624, 37570, 7102,  39485, 32588, 21117, 31148, 13168,   /*[GB+9840, GB+984F]*/
  14625, 27271, 12278, 17823, 26668, 34627, 28838, 30580, 32969, 28480, 4311,  1147,  13688, 38277, 41245, 8580,    /*[GB+9850, GB+985F]*/
  23419, 21570, 9799,  14049, 30323, 37387, 27893, 40088, 43585, 21931, 29918, 13708, 37302, 14643, 12366, 9034,    /*[GB+9860, GB+986F]*/
  25403, 20679, 40089, 27148, 28793, 42057, 32974, 31481, 27141, 30735, 41104, 29528, 28650, 16388, 33649,          /*[GB+9870, GB+987E]*/
  9790,  24778, 9162,  30174, 23608, 30849, 9302,  4508,  6118,  23302, 13191, 9674,  25559, 25457, 21125, 35908,   /*[GB+9880, GB+988F]*/
  9109,  43301, 32137, 25710, 37198, 6119,  13689, 43955, 5729,  19656, 27674, 33816, 13012, 18634, 12598, 39019,   /*[GB+9890, GB+989F]*/
  25510, 10498, 8917,  8922,  31870, 12005, 27841, 5032,  23911, 39321, 14187, 11865, 12211, 10308, 23052, 38303,   /*[GB+98A0, GB+98AF]*/
  9131,  16279, 10181, 10182, 20844, 5949,  43180, 17825, 17667, 3387,  4802,  18089, 35610, 26063, 29375, 31549,   /*[GB+98B0, GB+98BF]*/
  10051, 40821, 29929, 17913, 19987, 18811, 21579, 19812, 3191,  29930, 16396, 1643,  20096, 15240, 41654, 40966,   /*[GB+98C0, GB+98CF]*/
  29120, 20769, 22953, 37205, 31357, 23912, 43039, 27897, 26680, 13785, 4803,  13013, 36599, 36195, 7627,  36338,   /*[GB+98D0, GB+98DF]*/
  42136, 43920, 4869,  24713, 29265, 11998, 17094, 41730, 28136, 33694, 4514,  6210,  7664,  21989, 33393, 26957,   /*[GB+98E0, GB+98EF]*/
  19214, 19511, 25721, 34568, 27479, 7747,  41255, 17771, 40382, 29438, 7064,  3485,  30756, 21280, 29957,          /*[GB+98F0, GB+98FE]*/
  15932, 6211,  12611, 13029, 23400, 32183, 23179, 27433, 31618, 40911, 8150,  26177, 26912, 11270, 25416, 9732,    /*[GB+9940, GB+994F]*/
  19580, 17796, 9134,  33881, 28653, 43586, 27668, 13222, 6661,  15017, 14128, 4312,  30411, 29126, 14162, 19192,   /*[GB+9950, GB+995F]*/
  23254, 38659, 12630, 28041, 43964, 36056, 3489,  9039,  15488, 43968, 38364, 24716, 8265,  12496, 5685,  13488,   /*[GB+9960, GB+996F]*/
  30117, 24477, 3581,  4225,  43070, 10318, 13224, 14724, 13694, 41125, 13934, 25607, 5773,  37997, 1703,           /*[GB+9970, GB+997E]*/
  29691, 28310, 18447, 38604, 30912, 30501, 21635, 24969, 31878, 1819,  12867, 13499, 25340, 22779, 23165, 37724,   /*[GB+9980, GB+998F]*/
  9176,  16238, 38732, 23308, 26090, 36859, 24970, 21599, 41510, 10324, 4520,  13058, 17024, 24530, 5868,  4195,    /*[GB+9990, GB+999F]*/
  9333,  21636, 39169, 42510, 12358, 25345, 17931, 17862, 27566, 20004, 18471, 4999,  20293, 21785, 12654, 23470,   /*[GB+99A0, GB+99AF]*/
  20199, 42482, 9141,  6810,  40116, 18492, 8050,  43397, 29822, 18670, 14236, 4201,  26115, 42862, 19941, 36726,   /*[GB+99B0, GB+99BF]*/
  18509, 42861, 3443,  14687, 7490,  29849, 12035, 23090, 39918, 19689, 17424, 14491, 34457, 10200, 15553, 34870,   /*[GB+99C0, GB+99CF]*/
  19382, 38820, 13843, 38676, 39279, 38904, 34642, 23265, 2142,  2901,  17507, 15555, 29353, 28389, 32706, 4881,    /*[GB+99D0, GB+99DF]*/
  26697, 26538, 2494,  15337, 39942, 20497, 18564, 5003,  20331, 5640,  15989, 36740, 17571, 17519, 42921, 17889,   /*[GB+99E0, GB+99EF]*/
  18254, 537,   22522, 39955, 19413, 10132, 38087, 25264, 35690, 3162,  4248,  25111, 16175, 38147, 15844,          /*[GB+99F0, GB+99FE]*/
  33938, 35852, 11016, 17066, 17614, 16843, 29455, 16,    4313,  3771,  25886, 16218, 16846, 16223, 4380,  9882,    /*[GB+9A40, GB+9A4F]*/
  38510, 39510, 34819, 37581, 33256, 30680, 14943, 23458, 11625, 31155, 12060, 35741, 23845, 33882, 34970, 4315,    /*[GB+9A50, GB+9A5F]*/
  28137, 10635, 4300,  38365, 7480,  39560, 4619,  12075, 2240,  2241,  15579, 25377, 3697,  27770, 3772,  27776,   /*[GB+9A60, GB+9A6F]*/
  42604, 30093, 30094, 18405, 40823, 39535, 18430, 10193, 7349,  41535, 22034, 22035, 37224, 28574, 35853,          /*[GB+9A70, GB+9A7E]*/
  15857, 25221, 26300, 29881, 26055, 38206, 8360,  37562, 2398,  12693, 6386,  41718, 40405, 32970, 4130,  31157,   /*[GB+9A80, GB+9A8F]*/
  14944, 28189, 38596, 6259,  15322, 12612, 5007,  5462,  6901,  14163, 18776, 1844,  6811,  13828, 13844, 42989,   /*[GB+9A90, GB+9A9F]*/
  26097, 38170, 27866, 25762, 16421, 34903, 36383, 12485, 25663, 13003, 23459, 12491, 6948,  37725, 34881, 10028,   /*[GB+9AA0, GB+9AAF]*/
  21149, 50,    14717, 24056, 2873,  10901, 2327,  24051, 18991, 13567, 27577, 22257, 32191, 36305, 7587,  27124,   /*[GB+9AB0, GB+9ABF]*/
  34414, 15439, 26301, 6702,  27874, 23796, 15455, 7130,  1142,  27612, 21056, 27581, 29108, 32198, 11108, 30390,   /*[GB+9AC0, GB+9ACF]*/
  27616, 20194, 22205, 20976, 31619, 27222, 41199, 27747, 41200, 21300, 20005, 6439,  15810, 25069, 37081, 22670,   /*[GB+9AD0, GB+9ADF]*/
  1851,  28412, 25099, 33560, 26027, 36538, 6558,  37168, 40329, 41939, 1855,  5645,  29394, 17814, 22794, 7713,    /*[GB+9AE0, GB+9AEF]*/
  10209, 1784,  40863, 21649, 6232,  43314, 4384,  32337, 7801,  5309,  32046, 20869, 26258, 24399, 25252,          /*[GB+9AF0, GB+9AFE]*/
  33229, 33230, 31840, 15260, 41999, 42178, 25259, 15048, 15049, 27440, 16015, 11345, 26713, 1555,  9019,  42544,   /*[GB+9B40, GB+9B4F]*/
  7858,  15816, 11771, 23198, 35824, 35835, 40347, 36761, 5517,  39192, 11810, 12804, 23345, 21088, 21533, 3909,    /*[GB+9B50, GB+9B5F]*/
  23661, 1182,  42179, 23635, 43222, 40488, 19464, 40779, 8224,  21174, 40511, 32057, 18289, 21499, 37768, 7630,    /*[GB+9B60, GB+9B6F]*/
  31211, 29497, 16916, 15168, 15349, 38088, 13560, 42553, 12516, 1013,  40781, 42872, 39147, 9700,  11580,          /*[GB+9B70, GB+9B7E]*/
  24429, 4712,  42006, 29863, 3452,  29596, 15430, 27532, 28568, 15824, 8470,  13287, 10033, 5518,  38552, 23599,   /*[GB+9B80, GB+9B8F]*/
  37544, 12384, 40578, 3520,  32633, 11544, 14002, 29617, 7544,  35471, 8484,  27760, 42193, 38689, 33288, 16329,   /*[GB+9B90, GB+9B9F]*/
  20559, 225,   31126, 22163, 17958, 37684, 21503, 26638, 24557, 34891, 35157, 29866, 16867, 25820, 15570, 24436,   /*[GB+9BA0, GB+9BAF]*/
  35544, 42280, 10089, 21912, 33239, 26385, 38118, 15089, 28678, 7643,  2933,  14895, 19022, 26288, 32740, 10663,   /*[GB+9BB0, GB+9BBF]*/
  20889, 39214, 2022,  6694,  11374, 38148, 38924, 17534, 10713, 18216, 9394,  18744, 4532,  8294,  38149, 31575,   /*[GB+9BC0, GB+9BCF]*/
  17707, 13573, 3914,  15176, 30021, 3457,  23827, 34515, 28532, 17190, 24406, 10752, 14990, 22884, 31720, 34948,   /*[GB+9BD0, GB+9BDF]*/
  31513, 7288,  31981, 18329, 29405, 29618, 17961, 29406, 6765,  40360, 11251, 35517, 29039, 4595,  41412, 16578,   /*[GB+9BE0, GB+9BEF]*/
  33180, 32065, 6553,  33132, 15375, 28345, 18798, 12687, 30370, 25825, 5797,  15745, 9775,  10490, 36577,          /*[GB+9BF0, GB+9BFE]*/
  42669, 23953, 10966, 3042,  24756, 7864,  42106, 20041, 15447, 17964, 41647, 24448, 31273, 39750, 22712, 20042,   /*[GB+9C40, GB+9C4F]*/
  21547, 15102, 19287, 20418, 26475, 39751, 22886, 1627,  11594, 39997, 17351, 26028, 25447, 42198, 38640, 2034,    /*[GB+9C50, GB+9C5F]*/
  1039,  39990, 33065, 27489, 7918,  39991, 16431, 13399, 28346, 27767, 37691, 23574, 33584, 39993, 27243, 12192,   /*[GB+9C60, GB+9C6F]*/
  13894, 23359, 26314, 31474, 7732,  33133, 6056,  32581, 27297, 2572,  11081, 36803, 11393, 21511, 20961,          /*[GB+9C70, GB+9C7E]*/
  38750, 36804, 11300, 28620, 21159, 40556, 12710, 22435, 10246, 3833,  7405,  18755, 25122, 25123, 21110, 31229,   /*[GB+9C80, GB+9C8F]*/
  4898,  32653, 2371,  12552, 22074, 35709, 13170, 13887, 8325,  38208, 38702, 28783, 14511, 3466,  30672, 39487,   /*[GB+9C90, GB+9C9F]*/
  1315,  21880, 28690, 31729, 28588, 39101, 15456, 6623,  31896, 14369, 26315, 30693, 19650, 12836, 40060, 22487,   /*[GB+9CA0, GB+9CAF]*/
  677,   39308, 26661, 43175, 18860, 2846,  34332, 4541,  23024, 43466, 20811, 38858, 17275, 8297,  19790, 32654,   /*[GB+9CB0, GB+9CBF]*/
  1967,  38859, 24506, 14924, 36939, 30260, 40080, 18625, 37382, 21267, 43279, 3560,  30524, 30336, 32226, 9614,    /*[GB+9CC0, GB+9CCF]*/
  27706, 41075, 28788, 20445, 20649, 32589, 40557, 33464, 32971, 25549, 40805, 28697, 34,    40373, 38511, 29906,   /*[GB+9CD0, GB+9CDF]*/
  30304, 28698, 20,    23651, 4277,  33082, 2477,  21770, 9293,  10972, 12364, 33962, 5936,  42388, 35488, 14627,   /*[GB+9CE0, GB+9CEF]*/
  9235,  30029, 14452, 12571, 38578, 9124,  19653, 4543,  37294, 39799, 35645, 11862, 1339,  1638,  42417,          /*[GB+9CF0, GB+9CFE]*/
  14220, 16647, 28552, 28185, 21571, 318,   19983, 11784, 39157, 2962,  7841,  39016, 10362, 20816, 26084, 39524,   /*[GB+9D40, GB+9D4F]*/
  13441, 36470, 3188,  33870, 13442, 19807, 19657, 14860, 10458, 4862,  42132, 8920,  25600, 14459, 4793,  4548,    /*[GB+9D50, GB+9D5F]*/
  31869, 23454, 30961, 37517, 33810, 21572, 30737, 28186, 10789, 18626, 32228, 3723,  8968,  8301,  43618, 16277,   /*[GB+9D60, GB+9D6F]*/
  23880, 41389, 43198, 35909, 12797, 14051, 36595, 29373, 18891, 5045,  31063, 14154, 4863,  38945, 11414,          /*[GB+9D70, GB+9D7E]*/
  35646, 29253, 10048, 43747, 17147, 35910, 42418, 24181, 39843, 14466, 24514, 5672,  12599, 14653, 33384, 23535,   /*[GB+9D80, GB+9D8F]*/
  13206, 32668, 29944, 25408, 33675, 33982, 36393, 7109,  21894, 27477, 29932, 41727, 4864,  38279, 41655, 32319,   /*[GB+9D90, GB+9D9F]*/
  28128, 36332, 17086, 37477, 31794, 27522, 11229, 1390,  2865,  29175, 24640, 19136, 32669, 27808, 27786, 3575,    /*[GB+9DA0, GB+9DAF]*/
  15196, 12006, 14287, 17794, 5128,  11462, 10691, 40379, 19597, 11463, 8593,  10984, 11109, 34353, 14060, 28037,   /*[GB+9DB0, GB+9DBF]*/
  33983, 39977, 19988, 23188, 39551, 19190, 21683, 5605,  12216, 40832, 35243, 39864, 28976, 36183, 19347, 32454,   /*[GB+9DC0, GB+9DCF]*/
  43508, 39028, 12613, 2455,  18771, 6212,  37608, 12120, 41731, 20849, 5499,  38341, 30106, 24188, 15644, 30403,   /*[GB+9DD0, GB+9DDF]*/
  25947, 43380, 23251, 10417, 14951, 8145,  27797, 13207, 12614, 4221,  30419, 29689, 6484,  27798, 42952, 18660,   /*[GB+9DE0, GB+9DEF]*/
  21519, 28725, 29266, 21587, 23191, 38886, 38887, 21290, 14960, 24959, 13489, 5103,  33208, 30860, 38664,          /*[GB+9DF0, GB+9DFE]*/
  38665, 7026,  4693,  12858, 26110, 17590, 16095, 62,    32678, 1814,  9819,  25424, 38888, 1815,  17229, 8045,    /*[GB+9E40, GB+9E4F]*/
  2481,  21083, 14092, 32813, 20594, 40640, 20291, 18448, 39166, 37207, 20000, 29657, 42461, 38894, 6808,  32540,   /*[GB+9E50, GB+9E5F]*/
  12438, 35265, 23620, 28521, 1662,  2880,  22122, 19529, 13823, 27813, 3587,  9825,  1826,  12883, 34588, 19934,   /*[GB+9E60, GB+9E6F]*/
  26013, 27184, 18493, 15134, 34863, 38898, 30064, 32824, 12033, 36235, 42856, 19680, 17419, 7034,  7767,           /*[GB+9E70, GB+9E7E]*/
  11754, 17420, 29132, 19456, 38900, 21469, 13516, 18782, 14099, 38905, 19164, 38421, 13838, 2896,  5411,  26921,   /*[GB+9E80, GB+9E8F]*/
  13966, 17498, 7768,  29389, 40026, 43410, 8277,  28387, 17940, 17499, 4878,  26535, 39049, 25429, 7673,  15974,   /*[GB+9E90, GB+9E9F]*/
  37028, 38908, 27524, 40656, 20333, 37013, 18164, 21525, 28168, 30562, 5637,  14428, 3002,  38915, 11006, 42919,   /*[GB+9EA0, GB+9EAF]*/
  17568, 17516, 22532, 32330, 20341, 36357, 34475, 37031, 9051,  37036, 39968, 10080, 34756, 34757, 12374, 11290,   /*[GB+9EB0, GB+9EBF]*/
  43126, 41060, 35142, 3802,  40547, 25082, 42554, 8113,  23211, 32996, 24631, 38090, 23982, 16173, 23639, 16174,   /*[GB+9EC0, GB+9ECF]*/
  23705, 22253, 18971, 10083, 35325, 10084, 10085, 1615,  5250,  34085, 41876, 42876, 16429, 41474, 8485,  474,     /*[GB+9ED0, GB+9EDF]*/
  35158, 35159, 19449, 43289, 23727, 30477, 37153, 31539, 28064, 32854, 12342, 42658, 14346, 35854, 11262, 26716,   /*[GB+9EE0, GB+9EEF]*/
  12178, 35594, 35595, 34416, 38546, 33243, 42661, 37267, 28903, 32565, 31576, 21761, 40553, 16176, 34093,          /*[GB+9EF0, GB+9EFE]*/
  36072, 41880, 24561, 12464, 10092, 3275,  12392, 15593, 0,     19024, 8523,  15177, 35175, 24705, 31449, 43328,   /*[GB+9F40, GB+9F4F]*/
  31515, 10566, 17660, 36910, 35864, 3825,  27126, 11664, 33586, 29093, 11179, 36247, 16732, 15689, 34792, 33587,   /*[GB+9F50, GB+9F5F]*/
  43193, 16061, 6044,  35176, 13152, 36580, 20222, 36919, 15448, 19207, 17134, 12742, 31984, 33604, 5080,  33299,   /*[GB+9F60, GB+9F6F]*/
  11301, 3212,  8666,  33188, 4782,  8326,  24454, 26146, 27490, 33605, 26147, 35389, 38210, 15875, 39778,          /*[GB+9F70, GB+9F7E]*/
  9106,  24080, 35570, 9107,  28589, 3143,  28236, 35562, 22887, 9403,  32962, 3382,  11185, 17049, 42606, 34060,   /*[GB+9F80, GB+9F8F]*/
  12410, 8327,  18760, 35427, 14370, 1340,  38756, 42944, 32774, 31843, 28078, 33652, 23363, 23364, 2852,  36590,   /*[GB+9F90, GB+9F9F]*/
  15180, 15190, 21193, 32879, 41000, 14917, 26158, 27246, 12201, 7736,  26323, 30095, 37119, 19046, 14516, 40762,   /*[GB+9FA0, GB+9FAF]*/
  9883,  40325, 22490, 28944, 18870, 9675,  30738, 3215,  7522,  2069,  12351, 35071, 33970, 33471, 40452, 11124,   /*[GB+9FB0, GB+9FBF]*/
  11188, 33654, 40377, 22400, 28081, 26159, 37434, 21573, 18635, 38871, 33377, 9516,  36949, 25633, 1364,  1644,    /*[GB+9FC0, GB+9FCF]*/
  4804,  20098, 13789, 19813, 23913, 30035, 38306, 30948, 15912, 43667, 11866, 37880, 42437, 32892, 19577, 10801,   /*[GB+9FD0, GB+9FDF]*/
  23460, 26978, 15194, 17151, 5008,  40912, 13806, 33697, 33698, 38307, 34972, 3877,  12293, 2975,  37582, 30601,   /*[GB+9FE0, GB+9FEF]*/
  36972, 36339, 25678, 16091, 5830,  7111,  28460, 14295, 8151,  29549, 39866, 19146, 31620, 28249, 8152,           /*[GB+9FF0, GB+9FFE]*/
  36340, 17586, 21166, 30817, 38345, 15197, 21248, 15144, 14401, 38882, 38366, 36184, 17474, 30527, 2459,  33714,   /*[GB+A040, GB+A04F]*/
  26804, 43715, 18671, 12497, 42904, 19359, 32609, 38367, 35094, 41507, 12631, 5289,  23255, 17475, 27323, 11209,   /*[GB+A050, GB+A05F]*/
  36266, 14964, 4087,  5776,  37448, 11210, 17592, 1669,  27179, 18473, 22131, 27501, 20294, 17305, 303,   36272,   /*[GB+A060, GB+A06F]*/
  16945, 29479, 18933, 18512, 19943, 15957, 18965, 37008, 33759, 35277, 19691, 37631, 2385,  26940, 40290,          /*[GB+A070, GB+A07E]*/
  17597, 4882,  3979,  10065, 15681, 3311,  21481, 30809, 17603, 42922, 17572, 19414, 39971, 41456, 41868, 23728,   /*[GB+A080, GB+A08F]*/
  3454,  32872, 10839, 15913, 37492, 43857, 7594,  4422,  16360, 40669, 6387,  25560, 39020, 25608, 23643, 3178,    /*[GB+A090, GB+A09F]*/
  13759, 4408,  39512, 41001, 1495,  724,   2070,  4411,  39325, 6813,  3467,  23197, 15211, 30313, 27042, 20860,   /*[GB+A0A0, GB+A0AF]*/
  7861,  9065,  36901, 9211,  933,   29612, 39359, 9216,  22219, 5995,  26731, 26643, 43603, 20892, 16542, 25286,   /*[GB+A0B0, GB+A0BF]*/
  18056, 24624, 9085,  42111, 1018,  26662, 4542,  13595, 8255,  16382, 15461, 4178,  33972, 982,   20578, 14767,   /*[GB+A0C0, GB+A0CF]*/
  20626, 27618, 32893, 20977, 7065,  31621, 25723, 33715, 18474, 6814,  19083, 23563, 24283, 2134,  33760, 4025,    /*[GB+A0D0, GB+A0DF]*/
  32700, 17040, 4027,  464,   13285, 2774,  43315, 9205,  16311, 2000,  11530, 36505, 38541, 12055, 43189, 16891,   /*[GB+A0E0, GB+A0EF]*/
  42617, 22251, 23983, 15433, 37769, 28569, 32095, 19256, 24053, 22754, 37359, 39349, 36208, 5519,  2006,           /*[GB+A0F0, GB+A0FE]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A140, GB+A14F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A150, GB+A15F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A160, GB+A16F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,              /*[GB+A170, GB+A17E]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A180, GB+A18F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A190, GB+A19F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A1A0, GB+A1AF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A1B0, GB+A1BF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A1C0, GB+A1CF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A1D0, GB+A1DF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A1E0, GB+A1EF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,              /*[GB+A1F0, GB+A1FE]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A240, GB+A24F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A250, GB+A25F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A260, GB+A26F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,              /*[GB+A270, GB+A27E]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A280, GB+A28F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A290, GB+A29F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A2A0, GB+A2AF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A2B0, GB+A2BF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A2C0, GB+A2CF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A2D0, GB+A2DF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A2E0, GB+A2EF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,              /*[GB+A2F0, GB+A2FE]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A340, GB+A34F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A350, GB+A35F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A360, GB+A36F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,              /*[GB+A370, GB+A37E]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A380, GB+A38F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A390, GB+A39F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A3A0, GB+A3AF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A3B0, GB+A3BF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A3C0, GB+A3CF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A3D0, GB+A3DF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A3E0, GB+A3EF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,              /*[GB+A3F0, GB+A3FE]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A440, GB+A44F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A450, GB+A45F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A460, GB+A46F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,              /*[GB+A470, GB+A47E]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A480, GB+A48F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A490, GB+A49F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A4A0, GB+A4AF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A4B0, GB+A4BF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A4C0, GB+A4CF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A4D0, GB+A4DF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A4E0, GB+A4EF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,              /*[GB+A4F0, GB+A4FE]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A540, GB+A54F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A550, GB+A55F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A560, GB+A56F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,              /*[GB+A570, GB+A57E]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A580, GB+A58F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A590, GB+A59F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A5A0, GB+A5AF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A5B0, GB+A5BF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A5C0, GB+A5CF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A5D0, GB+A5DF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A5E0, GB+A5EF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,              /*[GB+A5F0, GB+A5FE]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A640, GB+A64F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A650, GB+A65F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A660, GB+A66F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,              /*[GB+A670, GB+A67E]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A680, GB+A68F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A690, GB+A69F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A6A0, GB+A6AF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A6B0, GB+A6BF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A6C0, GB+A6CF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A6D0, GB+A6DF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A6E0, GB+A6EF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,              /*[GB+A6F0, GB+A6FE]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A740, GB+A74F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A750, GB+A75F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A760, GB+A76F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,              /*[GB+A770, GB+A77E]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A780, GB+A78F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A790, GB+A79F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A7A0, GB+A7AF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A7B0, GB+A7BF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A7C0, GB+A7CF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A7D0, GB+A7DF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A7E0, GB+A7EF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,              /*[GB+A7F0, GB+A7FE]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A840, GB+A84F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A850, GB+A85F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A860, GB+A86F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,              /*[GB+A870, GB+A87E]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A880, GB+A88F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A890, GB+A89F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A8A0, GB+A8AF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A8B0, GB+A8BF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A8C0, GB+A8CF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A8D0, GB+A8DF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A8E0, GB+A8EF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,              /*[GB+A8F0, GB+A8FE]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A940, GB+A94F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A950, GB+A95F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A960, GB+A96F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,              /*[GB+A970, GB+A97E]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A980, GB+A98F]*/
  0,     0,     0,     0,     0,     0,     19242, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A990, GB+A99F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A9A0, GB+A9AF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A9B0, GB+A9BF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A9C0, GB+A9CF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A9D0, GB+A9DF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+A9E0, GB+A9EF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,              /*[GB+A9F0, GB+A9FE]*/
  16717, 34405, 12092, 11052, 41475, 13132, 36368, 28004, 30358, 31569, 17757, 20890, 33563, 34094, 946,   42301,   /*[GB+AA40, GB+AA4F]*/
  36644, 27867, 10753, 14998, 23551, 7926,  34793, 582,   24879, 1630,  38703, 17352, 13742, 25539, 17128, 36931,   /*[GB+AA50, GB+AA5F]*/
  43729, 42326, 41890, 36465, 19035, 6154,  43647, 10288, 8249,  33135, 40061, 4349,  31842, 36539, 34997, 13588,   /*[GB+AA60, GB+AA6F]*/
  1491,  39231, 37295, 30171, 2058,  21926, 29755, 20674, 40082, 5327,  39818, 28699, 10911, 25550, 38246,          /*[GB+AA70, GB+AA7E]*/
  41709, 2478,  10922, 20838, 14189, 22085, 2856,  319,   10915, 5031,  1050,  1391,  1367,  12284, 24672, 19137,   /*[GB+AA80, GB+AA8F]*/
  35922, 31614, 37438, 29465, 34833, 29052, 7060,  14470, 9310,  15775, 6792,  12615, 16825, 34442, 34446, 23163,   /*[GB+AA90, GB+AA9F]*/
  1503,  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+AAA0, GB+AAAF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+AAB0, GB+AABF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+AAC0, GB+AACF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+AAD0, GB+AADF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+AAE0, GB+AAEF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,              /*[GB+AAF0, GB+AAFE]*/
  12860, 23307, 21374, 19078, 22620, 10115, 29057, 19935, 30347, 34594, 21470, 26922, 22625, 20494, 34464, 25003,   /*[GB+AB40, GB+AB4F]*/
  15984, 21743, 43444, 19911, 29859, 26255, 8854,  17817, 423,   11346, 6022,  4386,  8936,  39429, 25025, 39430,   /*[GB+AB50, GB+AB5F]*/
  3134,  11352, 33353, 8373,  14743, 36456, 1564,  731,   40214, 15825, 21214, 15826, 13724, 5483,  24363, 25536,   /*[GB+AB60, GB+AB6F]*/
  38121, 194,   24437, 8486,  36036, 4713,  9635,  13673, 28262, 4654,  26900, 28577, 28413, 37797, 43858,          /*[GB+AB70, GB+AB7E]*/
  19567, 1266,  41679, 14808, 14441, 14013, 18331, 10093, 34221, 42662, 36571, 35654, 37096, 35792, 29871, 25914,   /*[GB+AB80, GB+AB8F]*/
  38559, 36307, 808,   34716, 34095, 10145, 3915,  35856, 23828, 17760, 5597,  38744, 33005, 7290,  3534,  6045,    /*[GB+AB90, GB+AB9F]*/
  33369, 0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+ABA0, GB+ABAF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+ABB0, GB+ABBF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+ABC0, GB+ABCF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+ABD0, GB+ABDF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+ABE0, GB+ABEF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,              /*[GB+ABF0, GB+ABFE]*/
  33296, 3529,  16062, 21102, 954,   31516, 34521, 4262,  10672, 36044, 36660, 36104, 35658, 8524,  33588, 19289,   /*[GB+AC40, GB+AC4F]*/
  18046, 14912, 30247, 8242,  32353, 6190,  24367, 41235, 27771, 21832, 39779, 17358, 21833, 28645, 32655, 31277,   /*[GB+AC50, GB+AC5F]*/
  4269,  1097,  3458,  7428,  3048,  1098,  20055, 10010, 6250,  966,   26765, 43349, 25922, 7677,  14813, 6847,    /*[GB+AC60, GB+AC6F]*/
  14764, 12713, 39660, 21120, 4509,  35989, 31048, 35428, 5387,  27247, 21837, 13773, 32775, 27393, 12202,          /*[GB+AC70, GB+AC7E]*/
  35062, 4351,  13906, 43135, 3144,  18762, 26669, 6942,  36466, 12281, 19498, 28706, 27155, 30264, 37305, 32972,   /*[GB+AC80, GB+AC8F]*/
  33378, 41719, 14933, 38874, 20652, 30850, 19501, 18406, 17628, 41837, 25561, 5146,  15902, 41463, 1806,  29114,   /*[GB+AC90, GB+AC9F]*/
  3096,  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+ACA0, GB+ACAF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+ACB0, GB+ACBF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+ACC0, GB+ACCF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+ACD0, GB+ACDF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+ACE0, GB+ACEF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,              /*[GB+ACF0, GB+ACFE]*/
  17152, 43097, 4805,  37718, 24939, 15063, 30269, 26336, 328,   18723, 21224, 38598, 37583, 32903, 33316, 5831,    /*[GB+AD40, GB+AD4F]*/
  35665, 40914, 36342, 26498, 5686,  19147, 18910, 26178, 29959, 10184, 15066, 7749,  14945, 19512, 13032, 12616,   /*[GB+AD50, GB+AD5F]*/
  15067, 127,   40738, 5609,  14405, 10389, 30658, 12632, 12127, 27810, 30120, 31251, 4226,  39562, 14965, 19936,   /*[GB+AD60, GB+AD6F]*/
  1822,  29155, 43941, 17484, 33891, 43524, 36059, 27409, 33209, 8926,  17863, 6815,  18496, 42511, 27264,          /*[GB+AD70, GB+AD7E]*/
  18152, 40647, 26188, 31177, 10198, 30056, 17311, 19692, 19944, 18515, 17598, 38822, 21479, 34643, 26197, 10066,   /*[GB+AD80, GB+AD8F]*/
  5787,  40659, 12149, 2007,  2023,  42119, 18576, 0,     34651, 25260, 645,   23846, 7893,  5484,  0,     0,       /*[GB+AD90, GB+AD9F]*/
  32279, 0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+ADA0, GB+ADAF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+ADB0, GB+ADBF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+ADC0, GB+ADCF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+ADD0, GB+ADDF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+ADE0, GB+ADEF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,              /*[GB+ADF0, GB+ADFE]*/
  11650, 37798, 564,   15690, 3082,  3633,  0,     5663,  21364, 43235, 24456, 1496,  42772, 0,     4678,  38773,   /*[GB+AE40, GB+AE4F]*/
  25137, 34347, 19853, 6087,  23462, 43098, 19216, 33394, 5465,  33121, 38803, 36873, 31223, 10683, 3088,  26050,   /*[GB+AE50, GB+AE5F]*/
  28421, 2942,  2943,  27422, 29833, 28452, 20018, 8323,  40983, 8450,  24402, 31507, 41806, 21254, 1249,  19586,   /*[GB+AE60, GB+AE6F]*/
  36286, 19465, 22220, 40352, 8487,  9392,  14746, 8488,  31217, 22221, 7120,  14139, 32217, 5257,  22457,          /*[GB+AE70, GB+AE7E]*/
  22224, 22225, 2563,  31220, 8871,  1280,  5263,  42327, 20368, 23587, 37693, 11987, 28304, 39488, 22230, 16068,   /*[GB+AE80, GB+AE8F]*/
  38174, 19480, 6377,  11990, 5603,  43239, 14148, 3566,  3148,  32033, 17847, 2674,  19513, 6408,  19220, 14167,   /*[GB+AE90, GB+AE9F]*/
  4091,  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+AEA0, GB+AEAF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+AEB0, GB+AEBF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+AEC0, GB+AECF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+AED0, GB+AEDF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+AEE0, GB+AEEF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,              /*[GB+AEF0, GB+AEFE]*/
  24204, 6426,  6431,  14568, 5524,  29099, 42419, 22411, 1183,  9067,  42725, 34171, 35686, 3427,  15831, 24850,   /*[GB+AF40, GB+AF4F]*/
  34049, 5063,  1267,  43726, 36149, 42028, 7644,  42199, 8003,  15357, 28008, 23005, 30956, 31541, 14498, 32750,   /*[GB+AF50, GB+AF5F]*/
  12468, 30543, 3826,  21188, 15111, 34797, 31730, 20897, 34798, 29994, 8399,  7147,  29042, 38248, 1078,  41416,   /*[GB+AF60, GB+AF6F]*/
  10013, 20622, 19124, 13423, 31283, 169,   3844,  1343,  21838, 6970,  7275,  36489, 4945,  28545, 42608,          /*[GB+AF70, GB+AF7E]*/
  3854,  39819, 8258,  37124, 3388,  31783, 10413, 33011, 12208, 16740, 38516, 15915, 33679, 9984,  38308, 4412,    /*[GB+AF80, GB+AF8F]*/
  12584, 23373, 6120,  41423, 28556, 4964,  38346, 43739, 29829, 27800, 6315,  19876, 19877, 25948, 8617,  18916,   /*[GB+AF90, GB+AF9F]*/
  25725, 0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+AFA0, GB+AFAF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+AFB0, GB+AFBF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+AFC0, GB+AFCF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+AFD0, GB+AFDF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+AFE0, GB+AFEF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,              /*[GB+AFF0, GB+AFFE]*/
  19520, 17734, 34372, 8041,  5466,  38717, 11203, 34373, 9988,  10326, 23278, 39901, 32684, 39039, 17928, 18453,   /*[GB+B040, GB+B04F]*/
  29184, 5579,  19194, 19195, 1777,  13508, 3671,  37177, 14531, 41935, 21085, 18530, 12899, 17426, 13080, 36086,   /*[GB+B050, GB+B05F]*/
  38966, 38678, 39057, 30563, 6142,  20546, 20347, 20351, 1942,  1949,  475,   7631,  7633,  25795, 13116, 40746,   /*[GB+B060, GB+B06F]*/
  21013, 23517, 4714,  19273, 22052, 13134, 23954, 9118,  7295,  10771, 1316,  4056,  25498, 42,    34926,          /*[GB+B070, GB+B07E]*/
  10985, 12368, 10986, 40835, 5056,  10993, 34931, 37594, 10996, 14408, 135,   35453, 12372, 18516, 24311, 11161,   /*[GB+B080, GB+B08F]*/
  14484, 8985,  23757, 26316, 26777, 40961, 9802,  16031, 16032, 42781, 40970, 9820,  41452, 6818,  25026, 38843,   /*[GB+B090, GB+B09F]*/
  886,   7,     1,     13,    15,    9,     11,    10,    32,    44,    58,    54,    69,    93,    78,    84,      /*[GB+B0A0, GB+B0AF]*/
  179,   158,   150,   203,   224,   243,   222,   230,   228,   263,   269,   286,   293,   308,   326,   334,     /*[GB+B0B0, GB+B0BF]*/
  360,   383,   385,   398,   399,   428,   444,   420,   418,   541,   445,   415,   440,   417,   470,   484,     /*[GB+B0C0, GB+B0CF]*/
  501,   497,   509,   503,   532,   508,   505,   552,   561,   559,   568,   560,   577,   579,   586,   613,     /*[GB+B0D0, GB+B0DF]*/
  609,   616,   604,   610,   611,   643,   644,   665,   670,   663,   688,   662,   661,   671,   692,   695,     /*[GB+B0E0, GB+B0EF]*/
  699,   723,   725,   719,   738,   751,   733,   752,   737,   740,   769,   771,   763,   778,   1953,           /*[GB+B0F0, GB+B0FE]*/
  41435, 11053, 11066, 1955,  32373, 39404, 7826,  8668,  26317, 5755,  20066, 41240, 18091, 14861, 35912, 13783,   /*[GB+B140, GB+B14F]*/
  23610, 181,   19923, 35814, 42705, 5687,  185,   9826,  18517, 9021,  32549, 25083, 40130, 31218, 35326, 33933,   /*[GB+B150, GB+B15F]*/
  8229,  21819, 21913, 28896, 40353, 23586, 7895,  21025, 34498, 28906, 37048, 41881, 37366, 28422, 12815, 5311,    /*[GB+B160, GB+B16F]*/
  41763, 16924, 15358, 28533, 21184, 22054, 43001, 41686, 28907, 6357,  22807, 43606, 3210,  1883,  21506,          /*[GB+B170, GB+B17E]*/
  19633, 31580, 6449,  6046,  22643, 21918, 36105, 41824, 37100, 22064, 42626, 22055, 21112, 28292, 10773, 12194,   /*[GB+B180, GB+B18F]*/
  3607,  5163,  15752, 7296,  20812, 34529, 33608, 17192, 28080, 31284, 10385, 32400, 18020, 28945, 26160, 19047,   /*[GB+B190, GB+B19F]*/
  41906, 792,   788,   803,   812,   802,   799,   841,   840,   865,   848,   859,   876,   885,   900,   891,     /*[GB+B1A0, GB+B1AF]*/
  884,   912,   968,   938,   918,   940,   942,   923,   927,   963,   964,   947,   1012,  1025,  1023,  1040,    /*[GB+B1B0, GB+B1BF]*/
  1066,  1069,  1090,  1112,  1130,  1113,  1143,  1168,  1174,  1203,  1192,  1184,  1369,  1347,  1373,  1222,    /*[GB+B1C0, GB+B1CF]*/
  1277,  1264,  1218,  1231,  1342,  1224,  1291,  1362,  1219,  24179, 1405,  1431,  1419,  1272,  1510,  1485,    /*[GB+B1D0, GB+B1DF]*/
  1492,  1520,  1522,  1566,  1560,  1550,  1598,  1599,  1604,  1583,  1622,  1629,  1645,  1693,  1739,  1730,    /*[GB+B1E0, GB+B1EF]*/
  1749,  1776,  1794,  1797,  1812,  1800,  1792,  1838,  1858,  1856,  1878,  1867,  1873,  1880,  1879,           /*[GB+B1F0, GB+B1FE]*/
  18065, 17398, 30096, 15760, 1344,  12696, 20067, 38250, 42733, 39526, 12697, 20675, 34182, 35497, 12420, 10388,   /*[GB+B240, GB+B24F]*/
  40560, 4573,  13774, 21194, 11505, 35993, 31313, 35913, 7660,  33197, 17080, 21410, 33109, 16653, 5678,  3337,    /*[GB+B250, GB+B25F]*/
  26169, 22106, 29417, 40836, 41425, 38347, 6261,  16620, 22107, 29439, 4815,  19774, 3658,  20774, 10185, 32458,   /*[GB+B260, GB+B26F]*/
  27467, 33717, 24680, 28518, 18984, 3308,  34374, 37595, 35939, 22183, 10319, 14083, 37626, 136,   12645,          /*[GB+B270, GB+B27E]*/
  13945, 41511, 21139, 4121,  27752, 23182, 36269, 37455, 12875, 21310, 21638, 24381, 21639, 17932, 16946, 36018,   /*[GB+B280, GB+B28F]*/
  21642, 12900, 19950, 21326, 19694, 10068, 20825, 33900, 30811, 16258, 42965, 14809, 39780, 29459, 40807, 15991,   /*[GB+B290, GB+B29F]*/
  28489, 1920,  1911,  1951,  1964,  1981,  1947,  1959,  1948,  2046,  2011,  2057,  2029,  2072,  1998,  2004,    /*[GB+B2A0, GB+B2AF]*/
  2041,  2040,  2073,  2049,  24496, 2003,  2208,  2182,  2207,  2206,  2256,  2228,  2229,  2232,  2288,  2254,    /*[GB+B2B0, GB+B2BF]*/
  2238,  2296,  2311,  2323,  2318,  2314,  2320,  2342,  2346,  2329,  2334,  2349,  2352,  2377,  2361,  2393,    /*[GB+B2C0, GB+B2CF]*/
  2391,  2394,  2425,  2440,  2470,  2473,  2465,  2468,  2492,  2504,  2505,  2521,  2510,  2538,  2555,  2573,    /*[GB+B2D0, GB+B2DF]*/
  2554,  2553,  2560,  2624,  2642,  2664,  2643,  2689,  2690,  2688,  2710,  2700,  2708,  2740,  2744,  2742,    /*[GB+B2E0, GB+B2EF]*/
  2766,  2778,  2779,  2820,  2444,  2858,  2849,  2842,  2853,  2946,  2924,  2947,  3022,  3039,  3043,           /*[GB+B2F0, GB+B2FE]*/
  11481, 38922, 42670, 14407, 40916, 40192, 528,   6496,  25075, 43559, 33428, 41537, 16675, 9068,  5427,  15833,   /*[GB+B340, GB+B34F]*/
  18306, 8380,  21820, 7333,  16265, 42194, 14748, 1790,  7360,  41539, 30078, 32255, 14801, 22056, 15359, 39733,   /*[GB+B350, GB+B35F]*/
  16367, 32113, 7376,  22227, 43002, 23329, 24407, 19274, 23758, 17821, 1956,  24564, 39132, 16925, 16541, 26655,   /*[GB+B360, GB+B36F]*/
  42824, 16874, 7386,  25225, 20045, 32751, 80,    34522, 36661, 23873, 17762, 11385, 26117, 32646, 25703,          /*[GB+B370, GB+B37E]*/
  38175, 16546, 26771, 3012,  17630, 11303, 39489, 34801, 34110, 20921, 20567, 39102, 3250,  3285,  36932, 27882,   /*[GB+B380, GB+B38F]*/
  17174, 39781, 3835,  11904, 19975, 3410,  23372, 29681, 43356, 16551, 36591, 43240, 16582, 3536,  24904, 43730,   /*[GB+B390, GB+B39F]*/
  26099, 3114,  3076,  3080,  41383, 3078,  3073,  3113,  3121,  3135,  3141,  3138,  3168,  3157,  3163,  3182,    /*[GB+B3A0, GB+B3AF]*/
  3189,  3192,  3180,  3204,  3206,  3238,  3255,  3293,  3283,  3267,  3294,  3319,  3346,  3352,  3345,  3368,    /*[GB+B3B0, GB+B3BF]*/
  3350,  3351,  3353,  3432,  3426,  3481,  3459,  3516,  3580,  3505,  3507,  3523,  3552,  3544,  3576,  3513,    /*[GB+B3C0, GB+B3CF]*/
  3511,  3601,  3602,  3612,  3619,  3646,  3690,  29003, 3679,  3685,  3678,  3681,  3761,  3752,  3749,  3743,    /*[GB+B3D0, GB+B3DF]*/
  3804,  3822,  3798,  3815,  3906,  3907,  3956,  3960,  3981,  4005,  4062,  4049,  4066,  4057,  4054,  4058,    /*[GB+B3E0, GB+B3EF]*/
  4028,  4045,  4115,  4103,  4126,  4135,  4133,  4188,  4162,  4199,  4167,  4176,  4163,  4156,  4216,           /*[GB+B3F0, GB+B3FE]*/
  19125, 16030, 1973,  21839, 13775, 11187, 26779, 18006, 38580, 33373, 24905, 20440, 22563, 27399, 36688, 6511,    /*[GB+B440, GB+B44F]*/
  32132, 38762, 16480, 32886, 29466, 41720, 6945,  34125, 5673,  31054, 22611, 31239, 25138, 7106,  8259,  26795,   /*[GB+B450, GB+B45F]*/
  26796, 20653, 9521,  29936, 7305,  19504, 29541, 30748, 11940, 24023, 32785, 31243, 34130, 33682, 18649, 32670,   /*[GB+B460, GB+B46F]*/
  40411, 6974,  17263, 43099, 37315, 43102, 2870,  25158, 331,   23886, 19600, 19994, 16254, 4445,  3418,           /*[GB+B470, GB+B47E]*/
  38654, 17917, 1647,  25159, 25160, 5035,  43675, 4614,  20434, 13047, 28146, 17735, 40917, 14084, 34005, 6520,    /*[GB+B480, GB+B48F]*/
  30608, 23616, 40529, 25683, 5913,  18454, 14085, 33719, 41394, 25728, 13946, 39887, 43259, 11145, 16493, 40846,   /*[GB+B490, GB+B49F]*/
  17859, 4211,  4214,  4309,  4276,  4283,  4235,  4324,  4343,  4347,  4362,  4354,  4358,  4378,  4385,  4405,    /*[GB+B4A0, GB+B4AF]*/
  4407,  4427,  4420,  4438,  4448,  4459,  4460,  4470,  4479,  4465,  4501,  4507,  4552,  4531,  4536,  4529,    /*[GB+B4B0, GB+B4BF]*/
  4578,  4581,  4596,  4632,  4663,  4680,  4682,  4679,  4677,  4666,  4652,  4708,  4728,  4743,  4724,  4810,    /*[GB+B4C0, GB+B4CF]*/
  4785,  4764,  4762,  4835,  4836,  4897,  4918,  4962,  4967,  4933,  4992,  5009,  5005,  5029,  5019,  5020,    /*[GB+B4D0, GB+B4DF]*/
  5071,  5083,  5088,  5075,  5090,  5123,  5131,  5139,  5149,  5150,  5144,  5197,  5194,  5205,  5219,  5240,    /*[GB+B4E0, GB+B4EF]*/
  5270,  5320,  5304,  5306,  5323,  5330,  5333,  5402,  5355,  5359,  5338,  5362,  5375,  5356,  5332,           /*[GB+B4F0, GB+B4FE]*/
  14676, 4229,  37632, 26809, 5689,  37999, 24037, 24038, 39902, 24355, 7486,  141,   16412, 13830, 39903, 27414,   /*[GB+B540, GB+B54F]*/
  23770, 4699,  2122,  37178, 20690, 34389, 16947, 17871, 17937, 42521, 18498, 18519, 7770,  26817, 23773, 38823,   /*[GB+B550, GB+B55F]*/
  18533, 19695, 19696, 22141, 29354, 10069, 17512, 2303,  36741, 27084, 40210, 30314, 20665, 35149, 37225, 24841,   /*[GB+B560, GB+B56F]*/
  1060,  6997,  42619, 27051, 37678, 28769, 39361, 42285, 31342, 8718,  21541, 30005, 21180, 2780,  20223,          /*[GB+B570, GB+B57E]*/
  39637, 33298, 43003, 10234, 34096, 42029, 9190,  9191,  29410, 14913, 28536, 8873,  17173, 6047,  30892, 9851,    /*[GB+B580, GB+B58F]*/
  10040, 43950, 19305, 20057, 5722,  42112, 1318,  41634, 12412, 39313, 38512, 43479, 12837, 41711, 40164, 33303,   /*[GB+B590, GB+B59F]*/
  34431, 5357,  5432,  5423,  5418,  5422,  5434,  5491,  5487,  5512,  5537,  5514,  5529,  5532,  5520,  5528,    /*[GB+B5A0, GB+B5AF]*/
  5534,  5594,  5625,  5626,  5656,  5657,  5697,  5719,  5736,  5745,  5714,  5721,  5713,  5742,  5767,  5749,    /*[GB+B5B0, GB+B5BF]*/
  5756,  5752,  5808,  5796,  5815,  5843,  5821,  5822,  5849,  5865,  5855,  5853,  5900,  5882,  5906,  5924,    /*[GB+B5C0, GB+B5CF]*/
  5926,  5931,  5919,  5927,  5942,  5940,  5990,  5988,  5814,  6061,  6048,  6030,  6019,  6037,  6059,  6122,    /*[GB+B5D0, GB+B5DF]*/
  6110,  6115,  6161,  6148,  6146,  6213,  6177,  6170,  6171,  6106,  6175,  6184,  6188,  6186,  6192,  6252,    /*[GB+B5E0, GB+B5EF]*/
  6231,  6262,  6240,  6227,  6295,  6285,  6288,  6294,  6330,  6328,  6397,  6404,  6338,  6364,  6385,           /*[GB+B5F0, GB+B5FE]*/
  37120, 42052, 37704, 21121, 29521, 6067,  975,   41722, 39115, 13446, 9195,  30741, 29529, 20680, 30392, 8771,    /*[GB+B640, GB+B64F]*/
  35998, 24941, 39888, 13034, 29647, 2871,  5559,  10320, 30121, 18240, 23256, 21457, 5738,  18476, 40291, 31106,   /*[GB+B650, GB+B65F]*/
  40660, 17995, 27236, 35177, 31711, 26991, 43560, 2686,  38093, 34211, 22859, 26213, 8185,  10956, 40355, 42016,   /*[GB+B660, GB+B66F]*/
  14986, 42017, 39723, 16728, 675,   23995, 22707, 18335, 39215, 23996, 2025,  19275, 22865, 44017, 42030,          /*[GB+B670, GB+B67E]*/
  15584, 12782, 31582, 42333, 12784, 11084, 38494, 43452, 42334, 27016, 6875,  42825, 23241, 8744,  9156,  36384,   /*[GB+B680, GB+B68F]*/
  29411, 17175, 8993,  15003, 31034, 31773, 36547, 20441, 20069, 9854,  44027, 43284, 745,   12989, 42053, 17178,   /*[GB+B690, GB+B69F]*/
  18007, 6452,  6460,  6454,  6461,  6473,  6479,  6510,  6497,  6492,  6525,  6530,  6531,  6584,  6587,  6598,    /*[GB+B6A0, GB+B6AF]*/
  6610,  6601,  6608,  6600,  6611,  6645,  6672,  6686,  6676,  6687,  6697,  6706,  6644,  6749,  6760,  6776,    /*[GB+B6B0, GB+B6BF]*/
  6761,  6768,  6844,  6854,  6851,  6861,  6893,  6863,  6869,  6882,  6860,  6916,  6925,  6946,  6930,  6932,    /*[GB+B6C0, GB+B6CF]*/
  6937,  6963,  6993,  6988,  6989,  7056,  7051,  7072,  7053,  7095,  7084,  7092,  7088,  7099,  7126,  7119,    /*[GB+B6D0, GB+B6DF]*/
  7116,  7135,  7171,  7193,  7166,  7243,  7233,  7216,  7235,  7228,  7303,  7286,  7302,  7282,  7309,  7277,    /*[GB+B6E0, GB+B6EF]*/
  7285,  7375,  7344,  7353,  7417,  7391,  7380,  7517,  7536,  7535,  7580,  7579,  7584,  7583,  7605,           /*[GB+B6F0, GB+B6FE]*/
  23906, 1895,  43951, 39802, 20375, 34628, 37709, 33974, 1535,  13447, 8582,  24183, 23420, 14522, 42609, 43661,   /*[GB+B740, GB+B74F]*/
  35805, 3474,  5762,  34354, 43490, 39845, 35926, 42439, 9166,  9810,  27156, 30108, 27234, 13472, 16286, 2432,    /*[GB+B750, GB+B75F]*/
  21132, 42485, 13448, 20122, 29840, 13035, 38948, 33053, 26229, 11198, 38349, 12303, 25850, 13316, 34850, 24682,   /*[GB+B760, GB+B76F]*/
  14300, 43304, 42613, 43975, 20245, 30131, 23258, 27815, 12646, 26917, 23431, 39904, 24357, 13509, 31937,          /*[GB+B770, GB+B77E]*/
  33055, 3493,  12876, 16948, 20248, 1674,  27825, 43309, 18186, 5004,  32209, 39397, 28414, 15054, 37416, 17698,   /*[GB+B780, GB+B78F]*/
  37367, 783,   1884,  32218, 42882, 14442, 6291,  33246, 10157, 4406,  37424, 34802, 3538,  16642, 5543,  43357,   /*[GB+B790, GB+B79F]*/
  35723, 7613,  7629,  7646,  7655,  7637,  7636,  7648,  7668,  7675,  7701,  7683,  7686,  7699,  7743,  7719,    /*[GB+B7A0, GB+B7AF]*/
  7722,  7754,  7710,  7724,  7785,  7790,  7814,  7815,  7798,  7806,  7804,  7857,  7859,  7855,  7877,  7876,    /*[GB+B7B0, GB+B7BF]*/
  7873,  7874,  7884,  7885,  7890,  7907,  7920,  7913,  7915,  7911,  7948,  7963,  7964,  7988,  7996,  7991,    /*[GB+B7C0, GB+B7CF]*/
  7994,  7999,  8072,  8089,  8068,  8075,  8065,  8071,  8106,  8128,  8109,  8172,  8181,  8177,  8182,  8187,    /*[GB+B7D0, GB+B7DF]*/
  8188,  8214,  8227,  8226,  8256,  8235,  8252,  8215,  8228,  8240,  8295,  8290,  8328,  8312,  8322,  8321,    /*[GB+B7E0, GB+B7EF]*/
  8451,  8352,  8365,  8417,  8375,  8412,  8456,  8466,  8570,  8541,  8483,  8525,  8448,  8478,  8468,           /*[GB+B7F0, GB+B7FE]*/
  29837, 9981,  43428, 33140, 32229, 36550, 26170, 37316, 37317, 31386, 3193,  31244, 6316,  15646, 18983, 33700,   /*[GB+B840, GB+B84F]*/
  33480, 16970, 4415,  41447, 16852, 16853, 3490,  5104,  18917, 40771, 5012,  25783, 26189, 6731,  40776, 19750,   /*[GB+B850, GB+B85F]*/
  25868, 4240,  28755, 8712,  25266, 4249,  11365, 24851, 10904, 28578, 8077,  21744, 26569, 43004, 19277, 19636,   /*[GB+B860, GB+B86F]*/
  1921,  565,   29631, 11395, 15112, 6308,  38253, 29247, 15104, 26575, 24415, 18092, 43117, 2635,  5834,           /*[GB+B870, GB+B87E]*/
  5129,  32287, 15140, 16240, 15142, 17819, 23892, 3691,  20881, 42877, 32348, 14347, 30006, 25968, 36459, 43227,   /*[GB+B880, GB+B88F]*/
  40049, 10858, 2607,  1284,  37962, 6554,  28015, 5215,  42885, 22357, 9710,  25827, 21876, 772,   2566,  7653,    /*[GB+B890, GB+B89F]*/
  19291, 8504,  8522,  8561,  8537,  8447,  8644,  8642,  8664,  8657,  8659,  8650,  24662, 8670,  8647,  8678,    /*[GB+B8A0, GB+B8AF]*/
  8721,  8734,  8821,  8753,  8715,  8738,  8700,  8713,  8698,  8761,  8702,  8741,  8699,  8705,  8701,  8760,    /*[GB+B8B0, GB+B8BF]*/
  8707,  8849,  8845,  8860,  8885,  8915,  8904,  8907,  8909,  9017,  8928,  8934,  8950,  8951,  8937,  8989,    /*[GB+B8C0, GB+B8CF]*/
  8994,  8986,  8990,  9046,  9053,  9056,  9073,  9072,  9062,  9061,  9096,  9102,  9060,  9135,  9114,  9117,    /*[GB+B8D0, GB+B8DF]*/
  9127,  9115,  9136,  9158,  9169,  9167,  9181,  9217,  9238,  9229,  9201,  9225,  9218,  9212,  9226,  9264,    /*[GB+B8E0, GB+B8EF]*/
  9283,  9269,  10531, 9263,  9288,  9364,  9358,  9359,  9367,  9370,  9371,  9396,  9449,  9391,  9414,           /*[GB+B8F0, GB+B8FE]*/
  23319, 8526,  7728,  8992,  7827,  28835, 21007, 31186, 26138, 21877, 20568, 10291, 26404, 3705,  38505, 37427,   /*[GB+B940, GB+B94F]*/
  1201,  17209, 34904, 30183, 27298, 12978, 10867, 42633, 17362, 5601,  2580,  8407,  31784, 18066, 17635, 15535,   /*[GB+B950, GB+B95F]*/
  10014, 13907, 10684, 31594, 34119, 42393, 3553,  30007, 43029, 44028, 31476, 2581,  9194,  8963,  16878, 40375,   /*[GB+B960, GB+B96F]*/
  23451, 34923, 24613, 17370, 43763, 1207,  1371,  9366,  30486, 9950,  39409, 13784, 5763,  3720,  41913,          /*[GB+B970, GB+B97E]*/
  27944, 42734, 20085, 19130, 16081, 8584,  41005, 9727,  25314, 16082, 4480,  2587,  43821, 2167,  36000, 1497,    /*[GB+B980, GB+B98F]*/
  30191, 34437, 24470, 35439, 11707, 37881, 40190, 4512,  20278, 33261, 6588,  29470, 13211, 14631, 35440, 21199,   /*[GB+B990, GB+B99F]*/
  7844,  9422,  9429,  9432,  9470,  9480,  9475,  9495,  9504,  9482,  9498,  9472,  9488,  9471,  9541,  9544,    /*[GB+B9A0, GB+B9AF]*/
  9547,  9574,  9573,  9601,  9595,  9598,  9632,  9631,  9653,  9650,  9652,  9658,  9720,  9713,  9695,  9726,    /*[GB+B9B0, GB+B9BF]*/
  9692,  9699,  9698,  9697,  9795,  9749,  9778,  9765,  9755,  9757,  9840,  9843,  9839,  9852,  9868,  9865,    /*[GB+B9C0, GB+B9CF]*/
  9912,  9916,  9924,  9932,  9935,  9946,  9953,  9978,  9967,  9969,  9970,  9968,  10016, 10009, 10072, 10035,   /*[GB+B9D0, GB+B9DF]*/
  10064, 10032, 10079, 10110, 10124, 10168, 10137, 10134, 10155, 10133, 10136, 10141, 10212, 10229, 10217, 10225,   /*[GB+B9E0, GB+B9EF]*/
  10278, 10271, 10298, 10276, 10268, 10356, 10357, 10383, 10410, 10404, 10440, 10485, 10502, 10517, 10528,          /*[GB+B9F0, GB+B9FE]*/
  4368,  8261,  42902, 12296, 25847, 26226, 21732, 25512, 9731,  29004, 40383, 11135, 30757, 40271, 4021,  27499,   /*[GB+BA40, GB+BA4F]*/
  41921, 23063, 25515, 34928, 19743, 23921, 6858,  43382, 4184,  27966, 43046, 25566, 19668, 13808, 2279,  12634,   /*[GB+BA50, GB+BA5F]*/
  1414,  5952,  4822,  36610, 2434,  43148, 24107, 24306, 39905, 31874, 40844, 27967, 10322, 37903, 11871, 2976,    /*[GB+BA60, GB+BA6F]*/
  16657, 24478, 40769, 13048, 19859, 2596,  22900, 30223, 5010,  6266,  30227, 17827, 18871, 34851, 2124,           /*[GB+BA70, GB+BA7E]*/
  21600, 27968, 5690,  18925, 5470,  8690,  13947, 21899, 17100, 5408,  14310, 30194, 17738, 34864, 20162, 28990,   /*[GB+BA80, GB+BA8F]*/
  40598, 25187, 23560, 24972, 23561, 9006,  15664, 20163, 20150, 36721, 5615,  27602, 43072, 9624,  25346, 18688,   /*[GB+BA90, GB+BA9F]*/
  29005, 10553, 10548, 10562, 10585, 10574, 10584, 10583, 10624, 10632, 10644, 10682, 10643, 10669, 10674, 10645,   /*[GB+BAA0, GB+BAAF]*/
  10714, 10710, 10809, 10806, 10750, 10740, 10805, 10749, 10762, 10737, 10732, 10840, 10853, 10860, 10929, 10928,   /*[GB+BAB0, GB+BABF]*/
  10916, 10907, 10943, 10941, 10963, 10945, 10961, 11009, 11018, 11070, 11089, 11064, 11029, 11036, 11032, 11031,   /*[GB+BAC0, GB+BACF]*/
  11083, 10912, 11057, 11040, 11080, 11193, 11190, 11196, 11175, 11227, 11226, 11238, 11243, 11244, 11247, 11249,   /*[GB+BAD0, GB+BADF]*/
  11248, 11269, 11271, 11260, 11291, 11292, 11294, 11370, 11392, 11363, 11342, 11338, 11340, 11484, 11479, 11491,   /*[GB+BAE0, GB+BAEF]*/
  11529, 11541, 11546, 11538, 11573, 11566, 11576, 11689, 11653, 11681, 11652, 11709, 11648, 11708, 11677,          /*[GB+BAF0, GB+BAFE]*/
  17500, 17107, 39580, 40282, 10938, 41739, 30502, 31179, 23091, 4093,  37921, 24990, 30980, 43162, 42800, 7706,    /*[GB+BB40, GB+BB4F]*/
  29826, 42799, 25529, 43416, 30986, 20177, 19951, 13977, 32207, 38913, 39948, 17430, 19702, 25869, 18704, 17517,   /*[GB+BB50, GB+BB5F]*/
  25360, 40297, 42593, 26542, 18705, 1519,  6956,  43900, 18187, 29579, 20506, 38920, 40309, 43425, 39969, 7720,    /*[GB+BB60, GB+BB6F]*/
  28405, 41541, 28415, 23343, 11055, 22703, 5141,  41410, 25272, 41094, 1193,  649,   33444, 27868, 16266,          /*[GB+BB70, GB+BB7E]*/
  27238, 1285,  5067,  38736, 41553, 21507, 30530, 11836, 8957,  15595, 22067, 42764, 3636,  29885, 11396, 31588,   /*[GB+BB80, GB+BB8F]*/
  2567,  19123, 43176, 566,   17691, 7557,  26405, 11097, 34541, 8563,  18226, 40249, 20086, 15471, 24926, 589,     /*[GB+BB90, GB+BB9F]*/
  41358, 11646, 11772, 11777, 11809, 11791, 11811, 11795, 11895, 11897, 11917, 11935, 11934, 11974, 11968, 11966,   /*[GB+BBA0, GB+BBAF]*/
  11975, 12025, 12022, 12021, 12024, 12041, 12054, 12085, 12096, 10546, 12154, 12175, 12181, 12173, 12193, 12205,   /*[GB+BBB0, GB+BBBF]*/
  12183, 12177, 12171, 12164, 12238, 12246, 12265, 12302, 12299, 12310, 12258, 12261, 12271, 12279, 12344, 12348,   /*[GB+BBC0, GB+BBCF]*/
  12340, 12347, 12373, 12383, 12405, 12433, 12381, 12469, 12451, 12486, 12480, 12595, 12508, 12550, 12543, 12540,   /*[GB+BBD0, GB+BBDF]*/
  12544, 12513, 12538, 12512, 12514, 12533, 12529, 12681, 12679, 12684, 12715, 12704, 12741, 12774, 12781, 12792,   /*[GB+BBE0, GB+BBEF]*/
  12791, 12817, 12807, 12832, 12863, 12808, 12830, 12915, 12923, 12958, 12924, 12988, 13016, 12952, 13005,          /*[GB+BBF0, GB+BBFE]*/
  10499, 11906, 27636, 30750, 1538,  21697, 43735, 14065, 18103, 35245, 8427,  23425, 998,   9816,  35648, 13606,   /*[GB+BC40, GB+BC4F]*/
  43181, 30763, 27640, 8204,  14228, 21993, 27643, 27644, 23430, 33732, 18818, 16829, 2177,  12137, 29191, 43745,   /*[GB+BC50, GB+BC5F]*/
  34596, 31882, 23102, 18520, 44082, 5973,  23103, 31399, 17606, 29490, 15218, 9483,  41943, 15221, 39351, 13363,   /*[GB+BC60, GB+BC6F]*/
  2746,  42754, 36296, 40182, 11367, 39399, 11056, 32349, 27052, 33002, 26290, 22351, 43446, 31685, 23218,          /*[GB+BC70, GB+BC7E]*/
  8348,  13381, 29086, 4533,  23997, 41819, 27869, 11376, 42201, 13136, 8081,  40362, 27056, 5490,  14810, 7898,    /*[GB+BC80, GB+BC8F]*/
  5068,  15270, 40492, 544,   8396,  42337, 24766, 4044,  11386, 40494, 33944, 8527,  35178, 28432, 1962,  43011,   /*[GB+BC90, GB+BC9F]*/
  26396, 12928, 12919, 12944, 13031, 12914, 12933, 12948, 12961, 12980, 13106, 13115, 13167, 13190, 13260, 13174,   /*[GB+BCA0, GB+BCAF]*/
  13101, 13131, 13138, 13111, 13114, 13178, 13112, 13293, 13277, 13139, 13278, 13428, 13339, 13467, 13347, 13330,   /*[GB+BCB0, GB+BCBF]*/
  13402, 13346, 13394, 13360, 13390, 13389, 13328, 13329, 13358, 13337, 13345, 13336, 13382, 13334, 13599, 13565,   /*[GB+BCC0, GB+BCCF]*/
  13552, 13556, 13571, 13549, 13626, 13645, 13677, 13670, 13678, 13679, 13710, 13697, 13699, 13698, 13706, 13719,   /*[GB+BCD0, GB+BCDF]*/
  13736, 13718, 13716, 13743, 13720, 13772, 13733, 13725, 13726, 13715, 13761, 13876, 13886, 13872, 13919, 13896,   /*[GB+BCE0, GB+BCEF]*/
  13866, 13880, 13908, 13869, 13884, 13883, 14005, 16226, 14044, 14038, 14006, 13995, 14045, 14063, 13996,          /*[GB+BCF0, GB+BCFE]*/
  19292, 43012, 28290, 9028,  37158, 8528,  32119, 41768, 5370,  4263,  28684, 42567, 34322, 43863, 15155, 678,     /*[GB+BD40, GB+BD4F]*/
  26477, 22068, 29229, 43946, 16930, 14999, 27064, 10868, 35190, 14617, 42831, 4051,  9930,  567,   15878, 16931,   /*[GB+BD50, GB+BD5F]*/
  11679, 4739,  12104, 9404,  30837, 14618, 16736, 14362, 26664, 8891,  20569, 36111, 1073,  34530, 8549,  9368,    /*[GB+BD60, GB+BD6F]*/
  6624,  27135, 31393, 38506, 17906, 35191, 15754, 8874,  6378,  31644, 29514, 14215, 34664, 12554, 15879,          /*[GB+BD70, GB+BD7E]*/
  13910, 15761, 3648,  21674, 41834, 20231, 3554,  26326, 29109, 722,   31645, 34811, 12108, 25889, 9438,  35647,   /*[GB+BD80, GB+BD8F]*/
  31049, 31704, 35063, 11407, 33964, 8564,  31451, 30031, 7012,  17179, 8408,  15010, 11853, 42054, 36673, 15186,   /*[GB+BD90, GB+BD9F]*/
  8302,  14009, 14015, 14001, 14000, 14019, 14030, 14012, 13999, 14152, 14134, 14135, 14138, 14129, 14169, 14181,   /*[GB+BDA0, GB+BDAF]*/
  14178, 14176, 14175, 14203, 14217, 14206, 14289, 14267, 14299, 14268, 14260, 14245, 14249, 14254, 14258, 14251,   /*[GB+BDB0, GB+BDBF]*/
  15966, 14357, 14354, 14352, 14334, 14353, 14340, 14332, 14342, 14402, 14341, 14366, 14448, 14460, 14445, 14446,   /*[GB+BDC0, GB+BDCF]*/
  14431, 14450, 14508, 14495, 14494, 14499, 14514, 14492, 14642, 14555, 14548, 15438, 14566, 14598, 14628, 14647,   /*[GB+BDD0, GB+BDDF]*/
  14575, 14579, 14720, 14716, 14736, 13240, 14737, 14745, 14752, 14726, 14747, 14750, 14740, 14786, 14814, 14788,   /*[GB+BDE0, GB+BDEF]*/
  14797, 14787, 14799, 14832, 14850, 14857, 14842, 14856, 14888, 14929, 14900, 14925, 14887, 14902, 14901,          /*[GB+BDF0, GB+BDFE]*/
  13425, 35900, 27020, 43662, 3332,  7194,  18407, 20271, 18808, 4065,  26724, 28294, 24927, 43280, 24929, 32404,   /*[GB+BE40, GB+BE4F]*/
  25506, 34551, 29048, 32662, 30899, 32405, 9090,  32533, 1084,  43246, 2344,  10500, 5089,  20428, 19575, 25053,   /*[GB+BE50, GB+BE5F]*/
  41308, 1372,  4607,  19333, 21622, 24780, 25843, 31240, 43663, 10363, 43764, 33658, 43491, 35521, 18843, 14862,   /*[GB+BE60, GB+BE6F]*/
  7932,  27425, 21846, 39823, 43713, 7741,  20272, 35914, 38776, 28225, 25186, 35927, 34632, 13793, 16484,          /*[GB+BE70, GB+BE7E]*/
  34561, 27402, 21624, 13018, 6947,  3969,  6088,  21849, 21716, 40094, 35231, 822,   29544, 26227, 1498,  12156,   /*[GB+BE80, GB+BE8F]*/
  9408,  4808,  21680, 32896, 8785,  32797, 31672, 9619,  21733, 35079, 18769, 43669, 1592,  40456, 38520, 31065,   /*[GB+BE90, GB+BE9F]*/
  9892,  14843, 14884, 14987, 15011, 14984, 15008, 15001, 15019, 14981, 14995, 15014, 15009, 14983, 15040, 15071,   /*[GB+BEA0, GB+BEAF]*/
  15057, 15056, 15122, 15118, 15110, 15132, 15082, 15096, 15117, 15107, 15097, 15080, 15173, 15181, 15232, 15217,   /*[GB+BEB0, GB+BEBF]*/
  15209, 15264, 15268, 15254, 15263, 15253, 15271, 15298, 15299, 15279, 15282, 15308, 15284, 15300, 15285, 15412,   /*[GB+BEC0, GB+BECF]*/
  15348, 15350, 15356, 15345, 15353, 15451, 15426, 15522, 15528, 15526, 15524, 15627, 15569, 15603, 15559, 15573,   /*[GB+BED0, GB+BEDF]*/
  15607, 15639, 15619, 15586, 15560, 15602, 15580, 15591, 15687, 15697, 15686, 15735, 15747, 15710, 15742, 15794,   /*[GB+BEE0, GB+BEEF]*/
  15983, 15815, 15854, 15840, 15939, 15837, 15807, 15811, 15834, 16013, 16026, 16021, 16010, 16012, 16054,          /*[GB+BEF0, GB+BEFE]*/
  42465, 40457, 3482,  2867,  5396,  34132, 40095, 43714, 35744, 28606, 32598, 9409,  36001, 38883, 14953, 38351,   /*[GB+BF40, GB+BF4F]*/
  43250, 22833, 712,   9812,  23614, 42790, 13809, 4640,  26683, 29378, 40462, 34138, 5036,  33702, 27159, 30857,   /*[GB+BF50, GB+BF5F]*/
  8800,  40384, 3340,  9171,  27364, 11726, 40590, 30967, 34571, 29947, 41789, 43740, 30858, 12356, 2353,  1415,    /*[GB+BF60, GB+BF6F]*/
  8336,  4968,  18114, 30228, 36852, 33886, 43742, 17856, 15779, 25519, 20851, 42141, 20246, 22289, 24307,          /*[GB+BF70, GB+BF7E]*/
  18662, 36132, 43718, 13049, 28042, 30132, 20287, 1107,  37722, 27717, 22184, 25626, 12723, 34261, 13490, 27961,   /*[GB+BF80, GB+BF8F]*/
  35671, 26880, 36133, 30143, 25687, 40919, 44047, 42074, 28152, 27645, 19156, 39906, 7698,  18926, 4618,  43986,   /*[GB+BF90, GB+BF9F]*/
  14093, 16048, 16069, 16056, 16050, 16058, 16112, 16111, 16116, 9213,  16123, 16130, 16159, 16141, 16154, 16192,   /*[GB+BFA0, GB+BFAF]*/
  16197, 16195, 16207, 16210, 16245, 16267, 16276, 16288, 16296, 16310, 16305, 16312, 16325, 16327, 16331, 16345,   /*[GB+BFB0, GB+BFBF]*/
  16424, 16355, 16358, 16374, 16398, 16394, 16362, 16419, 10538, 16423, 16435, 16441, 16442, 16449, 16458, 16505,   /*[GB+BFC0, GB+BFCF]*/
  16511, 16507, 16510, 16529, 16528, 16571, 16599, 16594, 16603, 16610, 16622, 16630, 16639, 16677, 16680, 16692,   /*[GB+BFD0, GB+BFDF]*/
  16718, 16741, 16724, 16738, 16751, 16767, 16776, 16779, 16777, 16795, 16815, 16798, 16796, 16834, 16845, 16858,   /*[GB+BFE0, GB+BFEF]*/
  16875, 16890, 16872, 16917, 16929, 16913, 16912, 16958, 16964, 16960, 16967, 16999, 16985, 17011, 10149,          /*[GB+BFF0, GB+BFFE]*/
  26973, 2981,  27436, 35673, 12647, 12014, 43890, 33733, 25628, 40483, 5322,  28610, 12656, 34023, 27819, 13952,   /*[GB+C040, GB+C04F]*/
  14170, 12138, 40741, 4827,  35281, 14413, 1457,  5582,  38406, 23272, 30144, 38407, 27971, 35766, 13518, 1828,    /*[GB+C050, GB+C05F]*/
  25471, 17501, 24692, 36273, 24991, 23939, 37460, 22143, 17991, 35117, 43897, 16950, 39177, 35952, 17874, 34280,   /*[GB+C060, GB+C06F]*/
  2902,  14423, 19952, 2906,  38831, 2326,  26945, 34290, 43926, 43901, 20609, 18192, 17575, 17894, 18792,          /*[GB+C070, GB+C07E]*/
  11347, 27043, 42973, 41803, 6347,  33939, 38751, 30888, 40321, 34531, 5022,  35151, 8357,  8358,  25185, 1963,    /*[GB+C080, GB+C08F]*/
  24457, 34722, 41515, 9091,  38787, 10058, 43987, 30614, 3492,  25176, 33122, 38816, 17875, 30629, 19953, 32499,   /*[GB+C090, GB+C09F]*/
  9054,  17077, 17072, 17073, 17118, 17119, 17169, 17187, 17201, 17199, 17215, 17213, 17244, 17246, 17265, 17286,   /*[GB+C0A0, GB+C0AF]*/
  17277, 17289, 17328, 17342, 17332, 17399, 17454, 17444, 17441, 17438, 17469, 17451, 17434, 17462, 17458, 17537,   /*[GB+C0B0, GB+C0BF]*/
  17532, 17551, 17539, 17577, 17583, 17625, 17627, 17615, 17620, 17609, 17658, 17674, 17690, 17693, 17695, 17750,   /*[GB+C0C0, GB+C0CF]*/
  17751, 17754, 17790, 17785, 17784, 17836, 17810, 17837, 17861, 17918, 17914, 17967, 17919, 17901, 17849, 17805,   /*[GB+C0D0, GB+C0DF]*/
  17959, 17957, 18003, 18005, 18016, 18028, 18042, 18045, 18108, 18112, 18033, 18034, 18061, 18221, 18207, 18208,   /*[GB+C0E0, GB+C0EF]*/
  18234, 18203, 18339, 18309, 18268, 18326, 18271, 18262, 18275, 18334, 18260, 18272, 18366, 18284, 18295,          /*[GB+C0F0, GB+C0FE]*/
  20455, 28407, 9707,  42936, 15360, 20958, 21822, 9928,  31036, 15755, 8550,  28547, 9931,  43358, 39803, 213,     /*[GB+C140, GB+C14F]*/
  7661,  29530, 20685, 19578, 521,   7663,  3197,  32904, 1395,  13491, 3939,  19584, 15782, 21601, 41512, 20483,   /*[GB+C150, GB+C15F]*/
  24115, 13087, 13096, 20337, 21500, 5252,  37091, 39300, 39295, 8117,  447,   37188, 9769,  25537, 40670, 9116,    /*[GB+C160, GB+C16F]*/
  38178, 43014, 37838, 27136, 26838, 25622, 12109, 34543, 38256, 37212, 25562, 25413, 39538, 9410,  40103,          /*[GB+C170, GB+C17E]*/
  7764,  28046, 8158,  28048, 18733, 23288, 25636, 9576,  3917,  8082,  11378, 3821,  5069,  8529,  34099, 1028,    /*[GB+C180, GB+C18F]*/
  17248, 24002, 19590, 42338, 26478, 33795, 35053, 33612, 16422, 12556, 12404, 34803, 27940, 11408, 14151, 5764,    /*[GB+C190, GB+C19F]*/
  27946, 18373, 18266, 18356, 18280, 18294, 18258, 18090, 18023, 18602, 18617, 18611, 18605, 18684, 18623, 18608,   /*[GB+C1A0, GB+C1AF]*/
  18610, 18607, 18715, 18717, 18757, 18743, 18742, 18740, 18805, 18796, 18797, 18806, 18794, 18825, 18856, 18863,   /*[GB+C1B0, GB+C1BF]*/
  18859, 18852, 18855, 18872, 18881, 18885, 18876, 18909, 18887, 18874, 17769, 17833, 18979, 18986, 18978, 18972,   /*[GB+C1C0, GB+C1CF]*/
  18996, 19043, 19023, 18997, 19034, 19122, 19112, 19151, 19149, 19113, 19111, 19167, 19119, 19183, 19205, 19198,   /*[GB+C1D0, GB+C1DF]*/
  19109, 19265, 19300, 19324, 19328, 19278, 19244, 19295, 19268, 19246, 19279, 19438, 19441, 19447, 19446, 19458,   /*[GB+C1E0, GB+C1EF]*/
  19476, 19497, 19482, 19495, 19471, 19462, 19502, 19470, 19564, 19585, 19617, 19643, 19619, 19642, 19667,          /*[GB+C1F0, GB+C1FE]*/
  3856,  36002, 3869,  43670, 32362, 12429, 11512, 11199, 24280, 18664, 11518, 337,   19157, 23853, 25784, 341,     /*[GB+C240, GB+C24F]*/
  12657, 36019, 5784,  17749, 9636,  9639,  9637,  6359,  27384, 22426, 6910,  31440, 24003, 37831, 26592, 13581,   /*[GB+C250, GB+C25F]*/
  15597, 4171,  20442, 13196, 41015, 19822, 13061, 17798, 12886, 39178, 12037, 38062, 25916, 40365, 5431,  11380,   /*[GB+C260, GB+C26F]*/
  42098, 23646, 41769, 3281,  41893, 39303, 32267, 31328, 7559,  36540, 31405, 35885, 10973, 28646, 19048,          /*[GB+C270, GB+C27E]*/
  15015, 1208,  6007,  10459, 33012, 35916, 24410, 4795,  6512,  22728, 31488, 15546, 4809,  16969, 18650, 18651,   /*[GB+C280, GB+C28F]*/
  33110, 17089, 18666, 18674, 4823,  28600, 29720, 31455, 17101, 23076, 42144, 5472,  23170, 25800, 22782, 31458,   /*[GB+C290, GB+C29F]*/
  31460, 19649, 19733, 19736, 19732, 19797, 19777, 19843, 19854, 19870, 19863, 19889, 19886, 19918, 19888, 19896,   /*[GB+C2A0, GB+C2AF]*/
  19973, 19967, 19969, 19976, 20170, 20068, 20175, 20072, 20033, 20050, 20117, 20056, 20025, 20021, 20095, 20184,   /*[GB+C2B0, GB+C2BF]*/
  20206, 20227, 20208, 20219, 20239, 20229, 20232, 20258, 20265, 20256, 20259, 20268, 20260, 20308, 20309, 20307,   /*[GB+C2C0, GB+C2CF]*/
  20315, 20355, 20357, 20366, 20367, 20392, 20406, 20395, 20394, 20400, 20401, 20444, 20462, 20477, 20449, 20463,   /*[GB+C2D0, GB+C2DF]*/
  20470, 20471, 20472, 20528, 20571, 20553, 20556, 20554, 20615, 20621, 20644, 20645, 20647, 20641, 20668, 20697,   /*[GB+C2E0, GB+C2EF]*/
  20695, 20701, 20712, 20726, 20727, 20723, 20728, 20770, 20768, 20760, 20813, 20841, 20828, 20835, 20837,          /*[GB+C2F0, GB+C2FE]*/
  19703, 41479, 29619, 29910, 41496, 38025, 26260, 16503, 2544,  9209,  2001,  12166, 3748,  27044, 27336, 40144,   /*[GB+C340, GB+C34F]*/
  2650,  10647, 8116,  23777, 23706, 39203, 26851, 33037, 24843, 24054, 33536, 33537, 16506, 6348,  7998,  476,     /*[GB+C350, GB+C35F]*/
  2014,  26467, 31219, 43561, 7962,  42019, 22803, 24438, 43613, 8654,  34309, 22222, 26388, 16364, 3755,  37047,   /*[GB+C360, GB+C36F]*/
  23662, 3630,  35546, 7550,  11263, 43614, 10146, 3531,  31379, 5070,  21098, 35038, 35039, 20730, 35040,          /*[GB+C370, GB+C37E]*/
  23129, 32441, 33041, 35600, 15105, 39304, 11254, 19039, 28009, 31517, 21103, 4537,  28537, 25448, 5817,  15692,   /*[GB+C380, GB+C38F]*/
  4940,  35598, 35386, 32066, 3539,  22657, 6703,  22596, 24136, 9777,  18359, 41413, 5081,  14513, 18838, 29390,   /*[GB+C390, GB+C39F]*/
  1631,  20762, 20870, 20882, 20877, 20875, 20868, 20919, 20943, 20953, 20971, 20946, 20947, 21006, 20994, 21019,   /*[GB+C3A0, GB+C3AF]*/
  21020, 21042, 21065, 21026, 21081, 21091, 21090, 21101, 21126, 21131, 21118, 21089, 21095, 21107, 21163, 21150,   /*[GB+C3B0, GB+C3BF]*/
  21152, 21177, 21187, 21172, 21186, 21211, 21237, 21251, 21262, 21273, 21299, 21268, 21368, 21360, 21388, 21386,   /*[GB+C3C0, GB+C3CF]*/
  21409, 21445, 21462, 21450, 21417, 21425, 21413, 21496, 21544, 21539, 21537, 21575, 21546, 21553, 21619, 21615,   /*[GB+C3D0, GB+C3DF]*/
  21617, 21664, 21650, 21657, 21659, 21672, 21694, 21707, 21711, 21713, 21736, 21725, 21728, 21742, 21741, 21773,   /*[GB+C3E0, GB+C3EF]*/
  21757, 21808, 21864, 21859, 21875, 21872, 21870, 21910, 21937, 21911, 21920, 21909, 21956, 21966, 21968,          /*[GB+C3F0, GB+C3FE]*/
  20423, 24235, 17907, 17074, 4477,  5538,  22659, 22398, 27066, 28538, 4601,  15379, 6625,  24083, 10451, 33193,   /*[GB+C440, GB+C44F]*/
  21123, 22726, 43136, 3845,  20468, 23490, 6068,  171,   22607, 29330, 40449, 42607, 27248, 31785, 32778, 14372,   /*[GB+C450, GB+C45F]*/
  13597, 6943,  1346,  3089,  25458, 32272, 29043, 30742, 29923, 43243, 38284, 18892, 13197, 24094, 35072, 20242,   /*[GB+C460, GB+C46F]*/
  23464, 3097,  20102, 10462, 23670, 4328,  14193, 8419,  43100, 20279, 14288, 38979, 20196, 36194, 5130,           /*[GB+C470, GB+C47E]*/
  31622, 22839, 18911, 5101,  10186, 34844, 30949, 7751,  42137, 14298, 11631, 5096,  27480, 34633, 30065, 8202,    /*[GB+C480, GB+C48F]*/
  43071, 5503,  16826, 23257, 15941, 4301,  15724, 17296, 18728, 32027, 24974, 5112,  1845,  36267, 22635, 33210,   /*[GB+C490, GB+C49F]*/
  40700, 21980, 22004, 21981, 21983, 21992, 21985, 22007, 22016, 22026, 22061, 22113, 22121, 22037, 22084, 22082,   /*[GB+C4A0, GB+C4AF]*/
  22041, 22167, 22158, 22193, 22216, 22212, 22211, 22214, 22210, 22271, 22280, 22273, 22264, 22279, 22243, 22247,   /*[GB+C4B0, GB+C4BF]*/
  22277, 22254, 22286, 22306, 22321, 22331, 22347, 22328, 22343, 22333, 22410, 22407, 22408, 22427, 22423, 22454,   /*[GB+C4C0, GB+C4CF]*/
  22449, 22460, 22518, 22552, 22591, 22586, 22626, 22629, 22649, 22656, 22667, 22681, 22686, 22694, 22737, 22704,   /*[GB+C4D0, GB+C4DF]*/
  22700, 22695, 22751, 22750, 22805, 22825, 22804, 22822, 22857, 22855, 22858, 22897, 22896, 22885, 22915, 22927,   /*[GB+C4E0, GB+C4EF]*/
  22931, 22936, 22974, 22980, 23006, 23082, 23008, 23057, 23058, 23004, 23124, 23138, 23134, 23156, 23130,          /*[GB+C4F0, GB+C4FE]*/
  34590, 1670,  35528, 16839, 17307, 36616, 19945, 12896, 40503, 20539, 26543, 40701, 20339, 22748, 40508, 25267,   /*[GB+C540, GB+C54F]*/
  33174, 10127, 19152, 10116, 14331, 13415, 9123,  4128,  21625, 23064, 9282,  14034, 6380,  42044, 35604, 30487,   /*[GB+C550, GB+C55F]*/
  34490, 2653,  33953, 39663, 35524, 15548, 15327, 35408, 28339, 15267, 30544, 28958, 30678, 24726, 24727, 10021,   /*[GB+C560, GB+C56F]*/
  12015, 31315, 34121, 5705,  7723,  23521, 30516, 7727,  4357,  19908, 19298, 34100, 26150, 23684, 16758,          /*[GB+C570, GB+C57E]*/
  8565,  40763, 8303,  18069, 39517, 17636, 39804, 2074,  22924, 15396, 12298, 16485, 1593,  22283, 6403,  5768,    /*[GB+C580, GB+C58F]*/
  755,   2673,  38354, 2482,  19823, 5404,  36196, 37450, 5838,  5617,  25609, 20006, 38001, 13254, 14107, 12887,   /*[GB+C590, GB+C59F]*/
  24992, 23133, 23186, 23196, 23206, 23217, 23208, 23238, 23234, 23231, 23274, 23314, 23323, 23333, 23341, 23361,   /*[GB+C5A0, GB+C5AF]*/
  23369, 23367, 23380, 23427, 23432, 23403, 23436, 23441, 23446, 23442, 23502, 23484, 23488, 23439, 23522, 23519,   /*[GB+C5B0, GB+C5BF]*/
  23530, 23540, 23541, 23532, 23544, 23550, 23553, 23549, 23576, 23570, 23595, 23590, 23600, 23613, 23644, 23645,   /*[GB+C5C0, GB+C5CF]*/
  23634, 23642, 23660, 23679, 23683, 23701, 23707, 23711, 23722, 23721, 23755, 23731, 23749, 23754, 23774, 23780,   /*[GB+C5D0, GB+C5DF]*/
  23793, 23801, 23798, 23792, 23829, 23819, 23817, 23840, 23847, 23867, 23860, 23872, 23885, 23898, 23907, 23899,   /*[GB+C5E0, GB+C5EF]*/
  23905, 23922, 23923, 23890, 23910, 23952, 23961, 23974, 23990, 24040, 23971, 23977, 24022, 24081, 24055,          /*[GB+C5F0, GB+C5FE]*/
  20010, 19954, 2910,  29360, 13811, 24411, 36998, 37025, 2536,  17815, 31435, 31333, 14244, 23889, 38043, 2765,    /*[GB+C640, GB+C64F]*/
  21612, 8931,  39707, 11804, 43593, 12530, 30014, 42252, 32001, 32721, 25077, 32994, 26992, 8458,  35826, 11896,   /*[GB+C650, GB+C65F]*/
  15818, 9629,  358,   4158,  21015, 27096, 10850, 38543, 39286, 38064, 25790, 24133, 31015, 5923,  18293, 39129,   /*[GB+C660, GB+C66F]*/
  936,   37952, 23135, 22755, 2008,  1874,  35640, 37360, 34212, 11354, 41031, 6540,  6337,  22982, 11582,          /*[GB+C670, GB+C67E]*/
  21093, 28571, 9701,  1251,  32852, 8477,  43434, 42187, 13364, 4772,  40064, 36167, 9262,  18308, 27118, 2790,    /*[GB+C680, GB+C68F]*/
  3357,  39724, 10888, 43596, 19013, 33292, 13365, 10140, 4734,  9655,  10090, 14257, 39453, 43441, 7647,  20923,   /*[GB+C690, GB+C69F]*/
  42621, 24067, 24082, 24062, 24042, 24123, 24138, 24185, 24159, 24205, 24224, 24215, 24260, 24263, 24276, 24320,   /*[GB+C6A0, GB+C6AF]*/
  24292, 24315, 24330, 24335, 24347, 24371, 24364, 24394, 24400, 24401, 24426, 24431, 24450, 24422, 24424, 24443,   /*[GB+C6B0, GB+C6BF]*/
  24423, 24433, 24493, 24497, 24546, 24520, 24563, 24581, 24554, 24568, 24601, 24629, 24724, 24626, 24661, 24665,   /*[GB+C6C0, GB+C6CF]*/
  24663, 24667, 2248,  24700, 24703, 24706, 24704, 24710, 24732, 24731, 24761, 24763, 24746, 24753, 24740, 24735,   /*[GB+C6D0, GB+C6DF]*/
  24744, 24779, 24742, 24738, 24835, 24894, 24836, 24840, 24880, 24870, 24862, 24828, 24924, 24842, 24826, 24891,   /*[GB+C6E0, GB+C6EF]*/
  25035, 25019, 25013, 25017, 25021, 25087, 25092, 25152, 25065, 25072, 25073, 25074, 25081, 25066, 25191,          /*[GB+C6F0, GB+C6FE]*/
  25261, 42879, 12094, 8383,  8867,  5214,  4377,  14988, 7548,  226,   25698, 3693,  24440, 14803, 17758, 29220,   /*[GB+C740, GB+C74F]*/
  43170, 5254,  2561,  43597, 6873,  32743, 18036, 6696,  8390,  27015, 38560, 2253,  40396, 30023, 3521,  3363,    /*[GB+C750, GB+C75F]*/
  33293, 1755,  33568, 9430,  43006, 22060, 43173, 44018, 32061, 26292, 30203, 3364,  23894, 21259, 35513, 14991,   /*[GB+C760, GB+C76F]*/
  3247,  16022, 36647, 5195,  10754, 5196,  13632, 32492, 29874, 23219, 28231, 34517, 33063, 14590, 22459,          /*[GB+C770, GB+C77E]*/
  22260, 31833, 9585,  20920, 4737,  6050,  26470, 6555,  43755, 17249, 20047, 32860, 23017, 17129, 24664, 40554,   /*[GB+C780, GB+C78F]*/
  9155,  10492, 20419, 4046,  4471,  41296, 21219, 18047, 851,   26002, 15717, 25917, 6001,  14500, 14852, 25704,   /*[GB+C790, GB+C79F]*/
  30479, 25212, 25213, 25271, 25251, 25269, 25281, 25244, 25255, 25302, 25246, 25294, 25382, 25419, 25380, 25381,   /*[GB+C7A0, GB+C7AF]*/
  25374, 25409, 25456, 25439, 25462, 25486, 25491, 25476, 25505, 25535, 25530, 25551, 25531, 25594, 25598, 25593,   /*[GB+C7B0, GB+C7BF]*/
  25615, 25676, 25665, 25661, 25642, 25700, 25727, 25694, 25695, 25740, 25780, 25775, 25766, 25757, 25752, 25759,   /*[GB+C7C0, GB+C7CF]*/
  25803, 13562, 25802, 25807, 25813, 25877, 25875, 25874, 25915, 25921, 25927, 25906, 25939, 25923, 25975, 25996,   /*[GB+C7D0, GB+C7DF]*/
  26018, 26020, 26019, 26021, 26022, 26029, 26066, 26057, 26059, 26054, 26075, 26077, 26094, 26148, 26123, 26212,   /*[GB+C7E0, GB+C7EF]*/
  26201, 26204, 26304, 26261, 26252, 26279, 26266, 26410, 26364, 26400, 26365, 26402, 26376, 26374, 26476,          /*[GB+C7F0, GB+C7FE]*/
  9399,  11929, 25920, 162,   32520, 1096,  42729, 13747, 19179, 30640, 31226, 5753,  11778, 11090, 4503,  3083,    /*[GB+C840, GB+C84F]*/
  12098, 8010,  17353, 27931, 36789, 37832, 31347, 24882, 32385, 2568,  41771, 32194, 15228, 1303,  38181, 23588,   /*[GB+C850, GB+C85F]*/
  2183,  23715, 6503,  26219, 16376, 34723, 32446, 39654, 39492, 8748,  18756, 35985, 22488, 2578,  33136, 4567,    /*[GB+C860, GB+C86F]*/
  34804, 1532,  21047, 167,   38861, 17210, 17211, 14146, 21673, 44072, 44073, 43796, 27244, 33860, 37564,          /*[GB+C870, GB+C87E]*/
  168,   26481, 13763, 8553,  20267, 15006, 23850, 11397, 11398, 11493, 36933, 31731, 43470, 26060, 21514, 12273,   /*[GB+C880, GB+C88F]*/
  28441, 8910,  42768, 14036, 32764, 2051,  32583, 13416, 6938,  37428, 30027, 26666, 41700, 12695, 28692, 17450,   /*[GB+C890, GB+C89F]*/
  43650, 26567, 26570, 26578, 26597, 26583, 26614, 26704, 26630, 26684, 26637, 26629, 26653, 26642, 26712, 26730,   /*[GB+C8A0, GB+C8AF]*/
  26729, 26741, 10273, 26750, 26752, 26782, 26786, 26772, 26767, 26837, 26839, 26862, 26874, 26890, 26899, 26926,   /*[GB+C8B0, GB+C8BF]*/
  26937, 26938, 26936, 26946, 26952, 26962, 26970, 26974, 26976, 26985, 26984, 26982, 27009, 27046, 27030, 27026,   /*[GB+C8C0, GB+C8CF]*/
  27024, 27039, 27036, 27078, 27079, 27098, 27112, 27111, 27144, 27119, 27165, 27152, 27143, 27121, 27117, 27194,   /*[GB+C8D0, GB+C8DF]*/
  27242, 27237, 27275, 27284, 27326, 27310, 27316, 27279, 27339, 27337, 27335, 27348, 27363, 27383, 27379, 27431,   /*[GB+C8E0, GB+C8EF]*/
  27452, 27450, 27470, 27471, 27485, 27487, 27512, 27516, 27538, 27582, 27588, 27580, 27599, 27604, 27608,          /*[GB+C8F0, GB+C8FE]*/
  37237, 39999, 21114, 40322, 29230, 43130, 9979,  26905, 36150, 11909, 32765, 39390, 29757, 38513, 28700, 4545,    /*[GB+C940, GB+C94F]*/
  28790, 40326, 41713, 27301, 18395, 26743, 40083, 18396, 15540, 33638, 4165,  35780, 31787, 19492, 12844, 6162,    /*[GB+C950, GB+C95F]*/
  25499, 43797, 24579, 5165,  40005, 4172,  39805, 23606, 24666, 22379, 33803, 8135,  40376, 27493, 2479,  21430,   /*[GB+C960, GB+C96F]*/
  30172, 21928, 29758, 19493, 33804, 9785,  17626, 8918,  5204,  30730, 20527, 35991, 37384, 10296, 43708,          /*[GB+C970, GB+C97E]*/
  10359, 44074, 31348, 2582,  23832, 5561,  28445, 17665, 19318, 6309,  31352, 21009, 31544, 4282,  173,   18628,   /*[GB+C980, GB+C98F]*/
  4797,  24458, 26223, 14863, 4549,  14649, 32663, 31902, 2517,  39847, 38286, 43545, 1348,  19984, 2269,  41359,   /*[GB+C990, GB+C99F]*/
  17843, 27625, 27658, 27670, 27681, 27698, 27705, 27707, 27732, 27735, 27773, 27758, 27762, 27838, 27847, 27872,   /*[GB+C9A0, GB+C9AF]*/
  27861, 27851, 27858, 27853, 27856, 27915, 27927, 27887, 27963, 27972, 28005, 27998, 27992, 27985, 27990, 28030,   /*[GB+C9B0, GB+C9BF]*/
  27999, 28054, 28060, 28133, 28148, 28140, 28115, 28095, 28103, 28130, 28183, 28175, 28178, 28209, 28208, 28218,   /*[GB+C9C0, GB+C9CF]*/
  28220, 28227, 28233, 28229, 28237, 28230, 28258, 28257, 28266, 28268, 28286, 28281, 28282, 28300, 28303, 28320,   /*[GB+C9D0, GB+C9DF]*/
  28314, 28331, 28348, 28356, 28341, 28354, 28342, 28338, 28336, 28423, 28394, 28401, 28397, 28399, 28431, 28418,   /*[GB+C9E0, GB+C9EF]*/
  28406, 28478, 28487, 28488, 28498, 28477, 28527, 28543, 28535, 28564, 28559, 28590, 28576, 28558, 28605,          /*[GB+C9F0, GB+C9FE]*/
  25601, 20840, 36689, 19334, 13417, 1639,  10366, 10716, 5944,  20089, 28371, 28188, 5945,  36253, 2075,  6080,    /*[GB+CA40, GB+CA4F]*/
  5168,  28453, 36120, 11697, 320,   21516, 19809, 42574, 24525, 14186, 21574, 4798,  22950, 12586, 15769, 38589,   /*[GB+CA50, GB+CA5F]*/
  14052, 38514, 10460, 3383,  11864, 27889, 16650, 25508, 20624, 40677, 40818, 18104, 32799, 13214, 25414, 28652,   /*[GB+CA60, GB+CA6F]*/
  21282, 23498, 2971,  6165,  27432, 17915, 39541, 25718, 4180,  11947, 13796, 20720, 40380, 777,   39255,          /*[GB+CA70, GB+CA7E]*/
  20125, 26956, 7453,  31066, 7978,  15917, 7679,  27308, 8147,  17090, 29437, 36493, 35745, 8788,  5681,  33312,   /*[GB+CA80, GB+CA8F]*/
  6589,  29546, 34845, 33987, 19659, 32977, 28247, 13797, 30175, 19348, 39869, 34133, 29654, 23253, 36008, 40463,   /*[GB+CA90, GB+CA9F]*/
  39848, 28613, 28638, 28640, 28634, 28632, 28663, 28662, 28679, 28677, 28691, 28670, 28660, 28669, 28742, 28746,   /*[GB+CAA0, GB+CAAF]*/
  28767, 28753, 28476, 28771, 28770, 28759, 28756, 28815, 28817, 28822, 28829, 28826, 28823, 28860, 28855, 28847,   /*[GB+CAB0, GB+CABF]*/
  28851, 28893, 28870, 28889, 28959, 28913, 28872, 28890, 28942, 28974, 28899, 28853, 28871, 28939, 28881, 28848,   /*[GB+CAC0, GB+CACF]*/
  28854, 28888, 28884, 28878, 28879, 29008, 29014, 29020, 29016, 29028, 29037, 29034, 29029, 29047, 29033, 29121,   /*[GB+CAD0, GB+CADF]*/
  29075, 29091, 29083, 29069, 29111, 29073, 29103, 29092, 29100, 29065, 29144, 29141, 29148, 29180, 29167, 29182,   /*[GB+CAE0, GB+CAEF]*/
  29171, 29172, 29170, 29173, 29166, 29202, 29210, 29216, 29205, 29203, 29218, 29250, 29227, 29245, 29252,          /*[GB+CAF0, GB+CAFE]*/
  33821, 10994, 118,   12618, 12619, 13476, 4694,  34630, 32459, 21775, 18012, 14160, 2457,  28463, 25606, 18667,   /*[GB+CB40, GB+CB4F]*/
  16392, 40104, 5286,  31165, 30703, 41190, 30176, 34257, 7752,  6486,  35247, 9817,  29179, 14077, 10895, 27558,   /*[GB+CB50, GB+CB5F]*/
  36257, 37441, 593,   7027,  24378, 32806, 23164, 4083,  20706, 24294, 13493, 40739, 3399,  41735, 7599,  22777,   /*[GB+CB60, GB+CB6F]*/
  38889, 4871,  34852, 24960, 7665,  13936, 35934, 17025, 1544,  6320,  21594, 17476, 14963, 26184, 25853,          /*[GB+CB70, GB+CB7E]*/
  18941, 34379, 29965, 20201, 38387, 35944, 35136, 38388, 17267, 17929, 14472, 5957,  42240, 907,   37451, 22128,   /*[GB+CB80, GB+CB8F]*/
  12221, 1665,  29821, 30616, 31905, 26193, 25731, 32926, 19530, 12636, 23471, 9177,  40470, 828,   18501, 29187,   /*[GB+CB90, GB+CB9F]*/
  4197,  29221, 29288, 29291, 29300, 29298, 29305, 29306, 29323, 29321, 29346, 29337, 29369, 28392, 29393, 29415,   /*[GB+CBA0, GB+CBAF]*/
  29412, 29428, 29441, 29430, 29434, 29444, 29457, 29452, 29449, 29513, 29539, 29534, 29501, 29493, 29488, 29486,   /*[GB+CBB0, GB+CBBF]*/
  29584, 29635, 29591, 29634, 29588, 4725,  28857, 29606, 29586, 29662, 29708, 29705, 29733, 29731, 29728, 29727,   /*[GB+CBC0, GB+CBCF]*/
  29730, 29753, 29774, 29814, 29827, 29831, 29835, 29854, 29873, 29875, 29892, 29914, 29899, 29905, 29878, 29861,   /*[GB+CBD0, GB+CBDF]*/
  29864, 29998, 30008, 30009, 30017, 30049, 30050, 30048, 30070, 30097, 30072, 30129, 30091, 30104, 30081, 30164,   /*[GB+CBE0, GB+CBEF]*/
  30179, 30180, 30214, 30206, 30200, 30217, 30246, 30243, 30251, 30235, 30321, 30310, 30311, 30312, 30335,          /*[GB+CBF0, GB+CBFE]*/
  65,    19227, 36020, 26014, 17423, 32203, 33497, 27437, 27438, 24981, 19937, 29847, 31938, 21324, 40477, 24481,   /*[GB+CC40, GB+CC4F]*/
  39694, 36270, 13510, 15162, 36021, 26240, 29848, 15163, 23941, 2179,  26923, 38423, 34459, 39588, 15507, 18735,   /*[GB+CC50, GB+CC5F]*/
  18736, 38672, 25610, 38817, 19682, 31698, 11914, 40286, 19379, 37343, 21143, 10829, 16980, 17502, 13521, 5693,    /*[GB+CC60, GB+CC6F]*/
  20858, 17994, 17876, 12446, 8278,  42079, 32939, 17034, 12038, 18166, 13529, 17945, 12052, 20495, 13088,          /*[GB+CC70, GB+CC7E]*/
  17038, 20180, 13854, 27575, 30989, 17886, 26726, 34883, 38457, 20342, 21235, 1746,  29504, 34784, 4264,  11598,   /*[GB+CC80, GB+CC8F]*/
  35699, 5159,  8536,  35716, 19978, 11782, 10975, 14281, 15617, 868,   36704, 41318, 41319, 16973, 1816,  34015,   /*[GB+CC90, GB+CC9F]*/
  29286, 30342, 30356, 30422, 30407, 30464, 30474, 30473, 30468, 30515, 30519, 30507, 30512, 30511, 30531, 30549,   /*[GB+CCA0, GB+CCAF]*/
  30535, 30555, 30550, 30564, 30605, 30582, 30594, 30586, 30570, 30634, 30642, 30637, 30677, 30669, 30664, 30666,   /*[GB+CCB0, GB+CCBF]*/
  30688, 30725, 30727, 30716, 30720, 30751, 30714, 30758, 30788, 30800, 30790, 30816, 30813, 30834, 30832, 30846,   /*[GB+CCC0, GB+CCCF]*/
  30833, 30889, 30878, 30876, 30886, 30881, 30920, 30925, 30933, 30976, 30958, 30955, 30959, 30997, 30996, 31005,   /*[GB+CCD0, GB+CCDF]*/
  31001, 31029, 31071, 31078, 31026, 31109, 31145, 31168, 31136, 31130, 31125, 31124, 31187, 31194, 31234, 31208,   /*[GB+CCE0, GB+CCEF]*/
  31224, 31213, 31287, 31278, 31323, 31334, 31339, 31390, 31395, 31403, 31409, 31401, 31432, 31437, 31442,          /*[GB+CCF0, GB+CCFE]*/
  6237,  26269, 6464,  34045, 15836, 41542, 28316, 39400, 10657, 43568, 34677, 7866,  7379,  448,   3762,  25278,   /*[GB+CD40, GB+CD4F]*/
  33003, 40225, 40226, 16023, 24864, 31579, 24865, 40050, 15847, 12465, 25918, 24866, 42624, 22261, 32493, 8118,    /*[GB+CD50, GB+CD5F]*/
  8119,  10861, 8727,  26857, 852,   22715, 5373,  24451, 3701,  15598, 18360, 8736,  40993, 9640,  24072, 24073,   /*[GB+CD60, GB+CD6F]*/
  34325, 6249,  1757,  1888,  41167, 31411, 9779,  38854, 9284,  4741,  12470, 20902, 8749,  19042, 37702,          /*[GB+CD70, GB+CD7E]*/
  34337, 18377, 38221, 24459, 24767, 28324, 37839, 32527, 22075, 25833, 10249, 26153, 17766, 35430, 14765, 26327,   /*[GB+CD80, GB+CD8F]*/
  13648, 31967, 3251,  977,   10780, 36046, 28550, 8674,  34545, 17637, 1349,  40006, 39314, 14634, 5539,  6197,    /*[GB+CD90, GB+CD9F]*/
  12471, 31434, 31463, 31471, 31466, 31467, 31512, 31522, 31540, 31573, 31599, 31624, 31554, 31584, 31556, 31587,   /*[GB+CDA0, GB+CDAF]*/
  31642, 31640, 31643, 31638, 31650, 31656, 31665, 31664, 31699, 31708, 31712, 31715, 31749, 31754, 31759, 31757,   /*[GB+CDB0, GB+CDBF]*/
  31763, 31824, 31827, 31832, 31841, 31851, 31899, 31914, 31951, 31968, 31971, 31960, 31977, 31995, 32025, 32056,   /*[GB+CDC0, GB+CDCF]*/
  32043, 32067, 32116, 32089, 32087, 32100, 32174, 32166, 30353, 32192, 32212, 32280, 32223, 32213, 32243, 32247,   /*[GB+CDD0, GB+CDDF]*/
  32264, 32281, 32294, 32320, 32307, 32314, 32345, 32351, 32334, 32352, 32340, 32401, 32371, 32379, 32395, 32377,   /*[GB+CDE0, GB+CDEF]*/
  32366, 32375, 32429, 32447, 32477, 32488, 32485, 32506, 32501, 32504, 32548, 32551, 32545, 32544, 32562,          /*[GB+CDF0, GB+CDFE]*/
  29684, 15397, 24933, 39829, 16084, 29531, 20430, 18409, 6399,  30900, 17148, 10689, 10793, 749,   7955,  32595,   /*[GB+CE40, GB+CE4F]*/
  7055,  40014, 30306, 25459, 27454, 22730, 32888, 18845, 6566,  7438,  656,   6081,  32535, 2403,  37169, 6514,    /*[GB+CE50, GB+CE5F]*/
  17287, 14635, 31489, 21072, 35746, 21626, 14527, 28795, 36004, 36843, 27257, 32601, 8789,  40096, 21200, 32898,   /*[GB+CE60, GB+CE6F]*/
  27309, 35081, 34066, 38781, 28715, 3970,  30700, 42843, 43671, 31067, 40097, 21283, 17293, 6789,  26228,          /*[GB+CE70, GB+CE7E]*/
  18417, 33143, 40332, 26576, 38878, 14156, 691,   29550, 4751,  33707, 33091, 18668, 629,   27166, 13228, 33263,   /*[GB+CE80, GB+CE8F]*/
  35669, 10812, 37893, 1159,  11948, 37993, 6900,  22444, 11139, 11730, 10183, 20655, 38356, 33021, 38885, 30943,   /*[GB+CE90, GB+CE9F]*/
  42579, 32613, 32586, 32560, 32621, 32628, 32639, 32623, 32642, 32644, 32845, 32661, 32647, 32722, 32568, 32724,   /*[GB+CEA0, GB+CEAF]*/
  32713, 32714, 32717, 32720, 32846, 32887, 32850, 32855, 32856, 32863, 32922, 32847, 32870, 32861, 32859, 32890,   /*[GB+CEB0, GB+CEBF]*/
  32844, 32975, 32966, 33004, 32991, 32999, 32993, 33033, 33049, 33038, 33057, 33080, 33077, 33117, 33125, 33141,   /*[GB+CEC0, GB+CECF]*/
  33128, 33139, 33147, 33195, 33171, 33185, 33168, 33235, 33234, 33242, 33224, 33231, 33241, 33238, 33273, 33280,   /*[GB+CED0, GB+CEDF]*/
  33295, 33278, 33277, 33274, 33351, 33337, 33360, 33338, 33379, 33340, 33354, 33415, 33409, 33467, 33446, 33427,   /*[GB+CEE0, GB+CEEF]*/
  33405, 33407, 33441, 33438, 33530, 33655, 33531, 33520, 33590, 33919, 33599, 33668, 33515, 33642, 33562,          /*[GB+CEF0, GB+CEFE]*/
  2483,  27716, 24952, 31400, 28196, 6097,  20144, 32916, 42488, 3395,  26506, 24108, 39564, 14086, 19824, 25982,   /*[GB+CF40, GB+CF4F]*/
  38667, 14165, 33025, 34853, 32462, 41606, 41659, 20631, 10429, 19521, 4825,  18129, 20822, 34854, 3155,  22129,   /*[GB+CF50, GB+CF5F]*/
  43923, 29559, 26237, 30945, 42145, 23931, 14410, 26517, 1735,  18928, 10263, 33889, 13318, 43104, 7957,  17739,   /*[GB+CF60, GB+CF6F]*/
  15951, 15952, 38605, 2881,  14312, 22575, 34865, 33326, 3973,  36349, 29560, 4198,  3494,  5618,  18246,          /*[GB+CF70, GB+CF7E]*/
  28157, 38002, 15070, 5292,  25177, 4649,  34696, 28380, 25956, 38901, 2800,  18478, 40892, 36022, 42909, 40851,   /*[GB+CF80, GB+CF8F]*/
  35009, 20932, 35285, 24993, 27189, 13967, 10939, 43413, 14702, 24383, 11025, 7775,  17942, 14699, 17320, 21904,   /*[GB+CF90, GB+CF9F]*/
  18521, 33609, 33557, 33524, 33577, 33683, 33506, 33578, 33653, 33585, 33632, 33519, 33606, 33823, 33796, 33789,   /*[GB+CFA0, GB+CFAF]*/
  33785, 33801, 33856, 33855, 33848, 33910, 33958, 33906, 33920, 34065, 34051, 34073, 34148, 34127, 34116, 34083,   /*[GB+CFB0, GB+CFBF]*/
  34075, 34087, 34165, 27932, 34172, 34167, 34224, 34238, 34201, 34198, 34247, 34204, 34305, 34304, 34326, 34323,   /*[GB+CFC0, GB+CFCF]*/
  34300, 34308, 34302, 34342, 34407, 34409, 34488, 34536, 34480, 34544, 34526, 34532, 34496, 34518, 34491, 34489,   /*[GB+CFD0, GB+CFDF]*/
  34613, 34616, 34645, 34614, 34631, 34634, 34621, 34609, 34665, 34660, 34654, 34684, 34676, 34674, 34713, 34711,   /*[GB+CFE0, GB+CFEF]*/
  34736, 34726, 34707, 34719, 34795, 34800, 34838, 36147, 34771, 34857, 34805, 34779, 34773, 34899, 34918,          /*[GB+CFF0, GB+CFFE]*/
  26247, 23104, 19955, 6907,  34878, 42869, 19706, 18168, 19707, 8280,  37484, 24120, 22519, 9833,  38832, 29196,   /*[GB+D040, GB+D04F]*/
  33776, 2422,  26554, 26703, 2423,  20803, 14713, 42925, 43310, 12239, 23348, 23781, 42634, 20733, 7618,  16113,   /*[GB+D050, GB+D05F]*/
  21795, 34042, 16246, 40153, 26471, 19302, 36107, 29231, 31652, 34724, 34345, 11711, 32900, 5769,  3932,  32906,   /*[GB+D060, GB+D06F]*/
  5774,  43274, 8987,  39437, 37772, 8384,  8078,  14571, 42561, 5488,  38129, 42622, 14749, 42195, 35042,          /*[GB+D070, GB+D07E]*/
  26853, 42022, 16019, 2794,  12388, 26854, 32115, 19440, 853,   37423, 44066, 1286,  28288, 15599, 11177, 36169,   /*[GB+D080, GB+D08F]*/
  41766, 37822, 23542, 1957,  5892,  8728,  10353, 42306, 42343, 26858, 38158, 32062, 22354, 9605,  36100, 41554,   /*[GB+D090, GB+D09F]*/
  26391, 34915, 34936, 34947, 34939, 34950, 34949, 34946, 35000, 34991, 35001, 35004, 35082, 35015, 35030, 35058,   /*[GB+D0A0, GB+D0AF]*/
  35017, 35046, 35021, 35049, 35128, 35173, 35153, 35284, 35242, 35147, 35148, 35193, 35163, 35356, 35319, 35343,   /*[GB+D0B0, GB+D0BF]*/
  35324, 35320, 35344, 35318, 35313, 35380, 35387, 35416, 35429, 35422, 35421, 35503, 35457, 35468, 35462, 35460,   /*[GB+D0C0, GB+D0CF]*/
  35459, 35499, 35506, 35504, 35507, 35505, 35533, 35531, 35547, 35536, 35539, 35558, 35560, 35585, 35588, 35596,   /*[GB+D0D0, GB+D0DF]*/
  35641, 35661, 35660, 35650, 35656, 35655, 35726, 35680, 35731, 35700, 35725, 35693, 35776, 35782, 35902, 35870,   /*[GB+D0E0, GB+D0EF]*/
  35838, 35819, 35823, 4256,  35840, 35883, 35876, 35865, 35866, 35962, 35975, 35966, 36042, 36043, 36031,          /*[GB+D0F0, GB+D0FE]*/
  39758, 21419, 2027,  1965,  3774,  3768,  16733, 27062, 14212, 14021, 2043,  14603, 7555,  27293, 42826, 10158,   /*[GB+D140, GB+D14F]*/
  38495, 16121, 35530, 35717, 17130, 22948, 29240, 13643, 17176, 18229, 15696, 28442, 9286,  41773, 19484, 13429,   /*[GB+D150, GB+D15F]*/
  38262, 2212,  43178, 29413, 26840, 18224, 8350,  2823,  17142, 30895, 40167, 19320, 3778,  3050,  4060,  18841,   /*[GB+D160, GB+D16F]*/
  23802, 7930,  40007, 36820, 6784,  42423, 37980, 24908, 10495, 16524, 24770, 31153, 31056, 8774,  3966,           /*[GB+D170, GB+D17E]*/
  35223, 6400,  17149, 6917,  35666, 35667, 40168, 8775,  39531, 31897, 36838, 12421, 23740, 5453,  40406, 30326,   /*[GB+D180, GB+D18F]*/
  9620,  12026, 27161, 40169, 22443, 15198, 30266, 630,   27688, 22957, 38788, 14772, 12029, 16743, 18652, 18105,   /*[GB+D190, GB+D19F]*/
  28722, 36070, 36084, 36098, 36097, 36152, 36155, 36164, 36161, 36193, 36201, 36244, 36251, 36316, 36281, 36287,   /*[GB+D1A0, GB+D1AF]*/
  36280, 36284, 36283, 36373, 36362, 36359, 36360, 36370, 36363, 36407, 36413, 36419, 36424, 36562, 36405, 36453,   /*[GB+D1B0, GB+D1BF]*/
  36447, 36460, 36463, 36469, 36464, 36488, 36483, 36500, 36503, 36579, 36902, 36584, 36569, 36578, 36654, 36626,   /*[GB+D1C0, GB+D1CF]*/
  36646, 36674, 36634, 36624, 36630, 36701, 36665, 36638, 36637, 36764, 36784, 36788, 36775, 36828, 36911, 36925,   /*[GB+D1D0, GB+D1DF]*/
  36973, 36897, 36906, 36935, 36907, 36905, 36929, 36908, 36921, 36914, 37046, 37038, 37052, 37050, 37076, 37071,   /*[GB+D1E0, GB+D1EF]*/
  37079, 37082, 37072, 37090, 37074, 37155, 37145, 37157, 37150, 37185, 37200, 37247, 37240, 37220, 37306,          /*[GB+D1F0, GB+D1FE]*/
  20243, 38357, 6334,  35257, 34262, 32918, 1701,  2527,  13040, 27843, 779,   1398,  8619,  13937, 43150, 13938,   /*[GB+D240, GB+D24F]*/
  5106,  13241, 5467,  40530, 7756,  2113,  34742, 35365, 1765,  26958, 20819, 17477, 368,   40847, 10327, 2547,    /*[GB+D250, GB+D25F]*/
  30134, 23259, 2830,  18730, 5614,  29189, 30659, 1460,  17490, 8629,  42242, 7038,  29193, 32276, 28995, 571,     /*[GB+D260, GB+D26F]*/
  35113, 2146,  3445,  17428, 19699, 33838, 34285, 17513, 41640, 5413,  15557, 40662, 28738, 13984, 38451,          /*[GB+D270, GB+D27E]*/
  17523, 36502, 33521, 8314,  8751,  8340,  525,   11154, 13090, 13098, 13997, 9971,  1573,  36912, 10159, 15860,   /*[GB+D280, GB+D28F]*/
  24256, 21040, 21551, 21552, 21766, 28917, 29632, 2821,  20469, 15881, 21567, 31394, 18636, 37435, 42424, 16033,   /*[GB+D290, GB+D29F]*/
  33812, 37293, 37254, 37297, 37271, 37282, 37260, 37361, 37370, 37418, 37419, 37459, 37474, 37476, 37486, 37485,   /*[GB+D2A0, GB+D2AF]*/
  37514, 37510, 37507, 37524, 37472, 37521, 37522, 37523, 37563, 37532, 37555, 37665, 37697, 37672, 37699, 37696,   /*[GB+D2B0, GB+D2BF]*/
  37677, 37670, 37671, 37864, 37749, 37845, 37829, 37745, 37820, 37868, 37757, 37766, 37783, 37912, 37971, 37958,   /*[GB+D2C0, GB+D2CF]*/
  37960, 37935, 37934, 37947, 37936, 38018, 38055, 38085, 38068, 38033, 38014, 38053, 38371, 38191, 38257, 38122,   /*[GB+D2D0, GB+D2DF]*/
  38028, 38261, 38242, 38304, 38017, 38013, 38151, 38245, 38097, 38026, 38159, 38065, 38036, 38370, 38180, 38095,   /*[GB+D2E0, GB+D2EF]*/
  38482, 38483, 38469, 38487, 38484, 38471, 38477, 38540, 38571, 38568, 38564, 38626, 38621, 38622, 38639,          /*[GB+D2F0, GB+D2FE]*/
  28086, 32603, 33988, 31292, 39553, 17552, 7468,  6856,  25894, 23703, 13494, 21939, 38890, 9687,  26443, 41324,   /*[GB+D340, GB+D34F]*/
  14967, 9989,  5841,  14102, 20484, 26604, 13832, 32697, 15969, 26450, 20487, 17563, 28524, 5974,  10002, 36900,   /*[GB+D350, GB+D35F]*/
  31280, 26275, 14805, 4919,  42345, 3231,  13142, 5540,  6004,  11991, 26672, 9296,  28947, 10252, 14721, 12752,   /*[GB+D360, GB+D36F]*/
  26332, 35437, 22733, 13020, 20105, 40968, 1418,  35449, 28197, 9533,  42515, 36189, 4306,  33743, 37923,          /*[GB+D370, GB+D37E]*/
  18558, 15976, 33771, 37017, 33779, 36631, 6499,  8719,  26276, 26277, 14439, 13367, 7821,  36374, 6293,  11450,   /*[GB+D380, GB+D38F]*/
  2795,  30922, 35697, 14591, 37823, 27057, 36375, 38561, 28104, 25102, 32063, 13385, 36376, 38562, 7292,  8086,    /*[GB+D390, GB+D39F]*/
  36533, 38685, 38740, 38783, 38747, 38805, 38738, 38777, 38848, 38851, 38852, 38847, 38872, 38844, 38892, 38846,   /*[GB+D3A0, GB+D3AF]*/
  38944, 38937, 38975, 38970, 38996, 39000, 38999, 39036, 39001, 39003, 39011, 39114, 39110, 39079, 39081, 39090,   /*[GB+D3B0, GB+D3BF]*/
  39075, 39092, 39083, 39126, 39148, 39139, 39150, 39142, 39184, 39185, 39197, 39221, 39193, 39201, 39230, 39287,   /*[GB+D3C0, GB+D3CF]*/
  39281, 39280, 39340, 39343, 39375, 39362, 39338, 39342, 39395, 39403, 39416, 39434, 39509, 39518, 39503, 39530,   /*[GB+D3D0, GB+D3DF]*/
  39421, 39440, 39494, 39438, 39481, 39486, 39465, 39471, 39611, 39456, 39622, 39610, 39615, 39629, 39614, 39630,   /*[GB+D3E0, GB+D3EF]*/
  39618, 39698, 39741, 39706, 39717, 35678, 39789, 39766, 39729, 39771, 39797, 39749, 39722, 39716, 39808,          /*[GB+D3F0, GB+D3FE]*/
  37230, 29735, 28502, 38569, 35336, 15862, 34901, 22644, 3370,  39225, 42211, 35548, 7900,  35388, 3164,  28347,   /*[GB+D440, GB+D44F]*/
  36663, 27517, 43282, 35795, 38185, 38186, 29893, 3640,  11019, 28443, 11099, 35888, 41775, 43020, 41968, 9661,    /*[GB+D450, GB+D45F]*/
  43583, 41171, 9786,  8752,  13899, 6381,  19306, 6005,  37191, 22562, 23650, 42769, 9030,  38224, 15606, 37429,   /*[GB+D460, GB+D46F]*/
  41077, 37841, 37842, 26574, 41484, 24460, 1322,  35571, 26409, 482,   5272,  43865, 30839, 42945, 4673,           /*[GB+D470, GB+D47E]*/
  41568, 39103, 35800, 36324, 38263, 12349, 11112, 28948, 2707,  34961, 28703, 11245, 2751,  9670,  10253, 26673,   /*[GB+D480, GB+D48F]*/
  12573, 14636, 11992, 8876,  34667, 32592, 28446, 42773, 31597, 21435, 21960, 7430,  12414, 36683, 35551, 9933,    /*[GB+D490, GB+D49F]*/
  7621,  39731, 39768, 39786, 39736, 39849, 39699, 39987, 39992, 39983, 40030, 40042, 40052, 40046, 40059, 40092,   /*[GB+D4A0, GB+D4AF]*/
  40036, 40035, 40048, 40081, 40079, 40062, 40129, 40147, 40166, 40148, 40149, 40178, 40181, 40244, 40233, 37420,   /*[GB+D4B0, GB+D4BF]*/
  40213, 40243, 40198, 40223, 40229, 40364, 40340, 40356, 40342, 40395, 40387, 40427, 40458, 40440, 40319, 40451,   /*[GB+D4C0, GB+D4CF]*/
  40425, 40486, 40513, 40509, 40552, 40550, 40548, 40564, 40580, 40574, 40575, 40603, 40619, 40627, 40635, 40671,   /*[GB+D4D0, GB+D4DF]*/
  40694, 40697, 40707, 40712, 40719, 40742, 40723, 40722, 40734, 40729, 40775, 40768, 40755, 40747, 40745, 40770,   /*[GB+D4E0, GB+D4EF]*/
  40784, 40780, 40777, 40782, 40883, 40898, 40907, 40909, 2625,  40930, 40936, 40947, 40950, 40982, 36499,          /*[GB+D4F0, GB+D4FE]*/
  1927,  31382, 37857, 17910, 42838, 16879, 16759, 33254, 13452, 42426, 27073, 4952,  17668, 7306,  16900, 7515,    /*[GB+D540, GB+D54F]*/
  31527, 5544,  985,   2859,  39381, 16553, 25770, 25891, 29297, 174,   39668, 34965, 3563,  14768, 34552, 33259,   /*[GB+D550, GB+D55F]*/
  33472, 9197,  29741, 2193,  12587, 15120, 29446, 41838, 29447, 6786,  11908, 3151,  29391, 14659, 16487, 26431,   /*[GB+D560, GB+D56F]*/
  4865,  34910, 30111, 32537, 34356, 7979,  3659,  30405, 38314, 22834, 38600, 6313,  24144, 43373, 2972,           /*[GB+D570, GB+D57E]*/
  3338,  43273, 13464, 24792, 30596, 43251, 32800, 15400, 26086, 6590,  41971, 40827, 43765, 25321, 43374, 18867,   /*[GB+D580, GB+D58F]*/
  14066, 4292,  10923, 20447, 28511, 1698,  12007, 24246, 39544, 6410,  35751, 24259, 28978, 36010, 28979, 12756,   /*[GB+D590, GB+D59F]*/
  12008, 40995, 40989, 41032, 41067, 41081, 40543, 41059, 41068, 41061, 41103, 41096, 41120, 41129, 41138, 41145,   /*[GB+D5A0, GB+D5AF]*/
  41201, 41160, 41178, 41166, 41158, 41229, 41225, 22892, 41230, 41228, 41336, 41286, 41282, 41285, 41290, 41302,   /*[GB+D5B0, GB+D5BF]*/
  41295, 41363, 41347, 41353, 41356, 41343, 41388, 41387, 41404, 41399, 41403, 41408, 41400, 41407, 41424, 41417,   /*[GB+D5C0, GB+D5CF]*/
  41433, 41434, 41458, 41460, 41476, 41487, 41489, 41468, 41495, 41467, 41521, 41530, 41543, 41567, 41599, 41631,   /*[GB+D5D0, GB+D5DF]*/
  41635, 41653, 41641, 41644, 41678, 41706, 41687, 41710, 41688, 41734, 41670, 41672, 41673, 41752, 41762, 41749,   /*[GB+D5E0, GB+D5EF]*/
  41845, 41816, 41844, 41801, 41905, 41875, 41891, 41866, 41877, 41862, 41867, 41954, 41944, 41955, 41961,          /*[GB+D5F0, GB+D5FE]*/
  7469,  42642, 6092,  35089, 8608,  24712, 31490, 14078, 25061, 39871, 43511, 43103, 33884, 12621, 38523, 182,     /*[GB+D640, GB+D64F]*/
  34367, 22474, 3396,  8317,  42849, 37127, 36976, 12304, 36011, 9323,  23423, 24797, 22182, 37611, 32907, 35445,   /*[GB+D650, GB+D65F]*/
  30972, 42706, 28147, 24529, 17096, 12357, 12869, 9251,  38891, 21444, 34930, 21596, 33887, 25570, 3342,  36232,   /*[GB+D660, GB+D66F]*/
  31082, 29969, 758,   3732,  25332, 28984, 14198, 40108, 35260, 11206, 30861, 37327, 37328, 19885, 39572,          /*[GB+D670, GB+D67E]*/
  1666,  4893,  18130, 21994, 21995, 28198, 41613, 21967, 13954, 40848, 14532, 18677, 19827, 2461,  23472, 10391,   /*[GB+D680, GB+D68F]*/
  33826, 43400, 339,   340,   14877, 41614, 37915, 11638, 14237, 20786, 3203,  10824, 11954, 2989,  35763, 40922,   /*[GB+D690, GB+D69F]*/
  27829, 41960, 41963, 41959, 41957, 41995, 42004, 41992, 41996, 42060, 42007, 42010, 42033, 41994, 41991, 42009,   /*[GB+D6A0, GB+D6AF]*/
  42108, 42092, 42109, 42110, 42086, 42095, 42089, 42172, 42191, 42162, 42212, 42163, 42168, 42180, 42254, 42297,   /*[GB+D6B0, GB+D6BF]*/
  42321, 42251, 42304, 42396, 42264, 42273, 42260, 42361, 42303, 42392, 42268, 42266, 42328, 42363, 42265, 42336,   /*[GB+D6C0, GB+D6CF]*/
  42541, 42560, 42551, 42562, 42565, 42556, 42595, 42594, 42623, 42614, 42615, 42650, 42654, 42649, 42657, 42651,   /*[GB+D6D0, GB+D6DF]*/
  42679, 42719, 42723, 42724, 42743, 42760, 42744, 42753, 42794, 42820, 42819, 42833, 42808, 42822, 42821, 42811,   /*[GB+D6E0, GB+D6EF]*/
  42883, 42870, 42880, 42943, 42934, 42956, 42951, 42929, 41667, 42988, 42971, 43015, 42983, 43026, 43018,          /*[GB+D6F0, GB+D6FE]*/
  33744, 40973, 7035,  41984, 22577, 17491, 7321,  38818, 15959, 13071, 44001, 14416, 2178,  12659, 43156, 33330,   /*[GB+D740, GB+D74F]*/
  40900, 41021, 28806, 25786, 30620, 40901, 24718, 28611, 36024, 40773, 30625, 5633,  30149, 34461, 13076, 14479,   /*[GB+D750, GB+D75F]*/
  41331, 22517, 37732, 66,    41209, 12505, 12019, 38424, 38425, 28159, 26948, 23311, 25472, 7045,  30447, 11882,   /*[GB+D760, GB+D76F]*/
  42713, 10940, 144,   38825, 14113, 39930, 13978, 12671, 6826,  41620, 36143, 40657, 17946, 28525, 32949,          /*[GB+D770, GB+D77E]*/
  3003,  18567, 37926, 1612,  41626, 37018, 7503,  4098,  32953, 4099,  37469, 2915,  26949, 38680, 17520, 3448,    /*[GB+D780, GB+D78F]*/
  35125, 23112, 12079, 40664, 38461, 5643,  41221, 37032, 6842,  32042, 27029, 35537, 35563, 1233,  42659, 28496,   /*[GB+D790, GB+D79F]*/
  35702, 42970, 42980, 42992, 42985, 43065, 41454, 43076, 43079, 43088, 43110, 43142, 43140, 43144, 43172, 43166,   /*[GB+D7A0, GB+D7AF]*/
  43177, 43165, 43199, 43185, 43187, 4476,  43212, 43206, 43248, 43224, 43230, 43270, 43275, 43291, 43288, 43287,   /*[GB+D7B0, GB+D7BF]*/
  43292, 44023, 43321, 43332, 43334, 41666, 43316, 43324, 43435, 43436, 43448, 43438, 43467, 43451, 43431, 43580,   /*[GB+D7C0, GB+D7CF]*/
  43547, 43566, 43582, 43624, 43591, 43605, 43590, 43687, 43646, 43673, 43628, 43633, 43697, 43724, 43749, 43781,   /*[GB+D7D0, GB+D7DF]*/
  43786, 43787, 43795, 43801, 43802, 43810, 43859, 43853, 43854, 43855, 43878, 43894, 43936, 43961, 43948, 43952,   /*[GB+D7E0, GB+D7EF]*/
  43976, 43981, 44015, 44040, 44041, 41066, 44067, 44048, 44050, 44063, 0,     0,     0,     0,     0,              /*[GB+D7F0, GB+D7FE]*/
  25279, 11387, 34057, 13156, 11409, 10629, 11317, 33722, 33723, 18929, 10723, 6827,  19710, 25034, 18230, 5823,    /*[GB+D840, GB+D84F]*/
  1155,  29263, 34578, 8268,  42524, 42538, 37033, 37035, 4246,  12397, 32008, 38187, 38188, 13749, 453,   11551,   /*[GB+D850, GB+D85F]*/
  7413,  4166,  13778, 16512, 8877,  15618, 8413,  33663, 1807,  42850, 13607, 8154,  33724, 2114,  32978, 12129,   /*[GB+D860, GB+D86F]*/
  5963,  43683, 8161,  38412, 232,   24074, 22363, 24005, 9644,  22368, 39374, 22076, 29637, 12110, 17143,          /*[GB+D870, GB+D87E]*/
  11114, 247,   18095, 22734, 1211,  39681, 13608, 31845, 20945, 33725, 38374, 15662, 4148,  30621, 12080, 15997,   /*[GB+D880, GB+D88F]*/
  925,   41681, 40045, 8720,  2321,  9579,  30934, 37824, 10866, 32356, 24366, 12831, 7831,  30545, 10038, 40802,   /*[GB+D890, GB+D89F]*/
  42347, 4234,  12912, 33404, 8895,  22913, 27527, 23967, 9379,  3504,  9273,  22543, 7465,  10347, 39446, 24339,   /*[GB+D8A0, GB+D8AF]*/
  1173,  32040, 37214, 37252, 41993, 5880,  35373, 38690, 16995, 39817, 9126,  30916, 42928, 13117, 22441, 21750,   /*[GB+D8B0, GB+D8BF]*/
  12920, 24822, 21497, 921,   27764, 9797,  40861, 28335, 5189,  36782, 15869, 29527, 37601, 36978, 7854,  24535,   /*[GB+D8C0, GB+D8CF]*/
  10215, 17065, 1525,  40828, 9921,  39283, 5699,  38016, 33032, 15044, 16670, 10269, 16142, 17260, 12946, 28100,   /*[GB+D8D0, GB+D8DF]*/
  32308, 16787, 24270, 15894, 25660, 12770, 38327, 15146, 32507, 26983, 6453,  41381, 17807, 27507, 9202,  22244,   /*[GB+D8E0, GB+D8EF]*/
  27027, 39612, 24126, 36448, 32248, 33339, 3038,  2467,  16306, 42967, 23184, 16117, 39143, 38045, 9596,           /*[GB+D8F0, GB+D8FE]*/
  7617,  43021, 28937, 1323,  43484, 7619,  10292, 24258, 1529,  20717, 5382,  28643, 16932, 8019,  31407, 37844,   /*[GB+D940, GB+D94F]*/
  3707,  21048, 11183, 1324,  20071, 19208, 12575, 8878,  24240, 43485, 13686, 35905, 40886, 14374, 8879,  40673,   /*[GB+D950, GB+D95F]*/
  14043, 38782, 36390, 41839, 28307, 1808,  1809,  26333, 28308, 4394,  40678, 42695, 17403, 40633, 4752,  3339,    /*[GB+D960, GB+D96F]*/
  28210, 31294, 23808, 9411,  34357, 20742, 14067, 30113, 8791,  30681, 4866,  4867,  42446, 13022, 41422,          /*[GB+D970, GB+D97E]*/
  6857,  14954, 35554, 4574,  40416, 825,   40561, 17406, 8337,  2501,  13053, 28657, 38375, 43151, 8810,  9688,    /*[GB+D980, GB+D98F]*/
  27601, 40849, 18930, 38392, 597,   3422,  32465, 42504, 43260, 1671,  40337, 40932, 5583,  40648, 36999, 24691,   /*[GB+D990, GB+D99F]*/
  28160, 31555, 32088, 22796, 13553, 13118, 7609,  39346, 16765, 16208, 42810, 38070, 31317, 2775,  14333, 23232,   /*[GB+D9A0, GB+D9AF]*/
  22159, 4031,  36768, 18296, 26271, 18210, 39624, 24403, 39082, 24846, 8324,  25483, 27486, 23548, 43290, 29080,   /*[GB+D9B0, GB+D9BF]*/
  20519, 33127, 1191,  31128, 9972,  16573, 15587, 8186,  36781, 35043, 13387, 32566, 43698, 19780, 30791, 1796,    /*[GB+D9C0, GB+D9CF]*/
  23382, 3841,  33644, 15059, 14047, 14277, 15310, 31601, 35995, 5456,  31553, 32028, 28315, 25256, 43856, 40275,   /*[GB+D9D0, GB+D9DF]*/
  4987,  5921,  33507, 36382, 11427, 10470, 2979,  17033, 762,   24657, 11293, 8517,  8449,  29860, 29594, 31663,   /*[GB+D9E0, GB+D9EF]*/
  36763, 2018,  10349, 21039, 35181, 20313, 24611, 1894,  38880, 20537, 17864, 1852,  11801, 19001, 34404,          /*[GB+D9F0, GB+D9FE]*/
  32468, 38907, 14970, 9045,  34395, 40684, 1472,  6828,  29158, 37014, 28215, 36145, 19772, 9049,  40686, 40033,   /*[GB+DA40, GB+DA4F]*/
  1325,  42682, 8329,  40334, 33594, 31598, 33966, 3470,  3487,  34140, 30766, 43780, 18312, 28106, 26140, 38635,   /*[GB+DA50, GB+DA5F]*/
  34423, 43456, 15863, 25971, 3702,  4634,  3433,  6382,  5901,  33957, 41172, 15882, 13189, 3717,  4174,  9888,    /*[GB+DA60, GB+DA6F]*/
  36227, 31355, 7191,  8997,  30219, 4954,  33813, 41498, 29926, 38655, 15478, 14068, 26797, 4609,  5050,           /*[GB+DA70, GB+DA7E]*/
  20107, 5682,  26231, 43512, 31077, 26440, 3891,  12319, 25732, 25688, 14474, 40774, 31181, 7603,  40624, 16729,   /*[GB+DA80, GB+DA8F]*/
  16368, 15864, 3408,  7908,  40232, 512,   24884, 32196, 38226, 22888, 19308, 21189, 16689, 4638,  25202, 43023,   /*[GB+DA90, GB+DA9F]*/
  15385, 29670, 21529, 42597, 21915, 36623, 14549, 11447, 28093, 23437, 15565, 22638, 9754,  11008, 5984,  41469,   /*[GB+DAA0, GB+DAAF]*/
  26372, 37759, 16861, 17899, 9923,  13123, 12377, 28408, 9651,  26633, 41869, 12735, 35788, 25754, 9184,  16893,   /*[GB+DAB0, GB+DABF]*/
  7514,  43751, 43329, 32744, 39460, 28495, 2937,  30082, 3371,  14023, 36220, 37557, 7389,  39759, 35973, 163,     /*[GB+DAC0, GB+DACF]*/
  6051,  43455, 24232, 21972, 5628,  29894, 28936, 21557, 41574, 13911, 40899, 25711, 15905, 36961, 41184, 3442,    /*[GB+DAD0, GB+DADF]*/
  14542, 14846, 8697,  33410, 25247, 15043, 638,   6173,  44051, 883,   35463, 8862,  42287, 22999, 43753, 4469,    /*[GB+DAE0, GB+DAEF]*/
  24065, 32571, 12264, 16992, 33822, 10640, 26121, 16909, 20865, 33233, 7856,  1869,  23973, 926,   37530,          /*[GB+DAF0, GB+DAFE]*/
  6159,  23602, 15612, 22722, 37860, 12993, 7241,  7242,  42839, 26674, 43077, 34669, 3846,  24241, 42401, 13649,   /*[GB+DB40, GB+DB4F]*/
  2343,  5278,  15474, 29116, 31797, 4287,  15121, 23046, 34826, 2273,  5127,  22237, 29117, 14378, 25664, 22175,   /*[GB+DB50, GB+DB5F]*/
  14069, 24946, 33144, 32801, 14660, 13216, 23055, 15402, 23056, 20432, 20108, 18021, 32414, 26682, 43825, 25848,   /*[GB+DB60, GB+DB6F]*/
  37987, 4961,  43672, 23962, 41908, 39555, 7112,  4383,  39120, 6093,  41048, 3419,  9893,  30762, 8610,           /*[GB+DB70, GB+DB7E]*/
  43828, 23424, 2732,  31955, 25572, 22901, 6124,  31083, 13244, 19460, 40644, 1433,  3945,  20155, 18931, 30706,   /*[GB+DB80, GB+DB8F]*/
  5410,  29975, 33890, 17058, 13065, 42146, 25577, 5964,  23622, 43685, 18678, 40714, 22905, 31939, 15502, 34272,   /*[GB+DB90, GB+DB9F]*/
  42587, 5985,  30470, 13624, 42271, 42812, 16799, 25809, 12087, 40431, 18315, 38923, 9185,  33550, 8495,  24064,   /*[GB+DBA0, GB+DBAF]*/
  30571, 36791, 15749, 36593, 38582, 41350, 24526, 28124, 43767, 19373, 8273,  4152,  12165, 20722, 26462, 28278,   /*[GB+DBB0, GB+DBBF]*/
  11034, 9354,  21359, 35860, 35075, 29802, 35255, 15968, 38620, 25435, 5646,  3140,  29484, 1551,  1027,  26312,   /*[GB+DBC0, GB+DBCF]*/
  1033,  7367,  7640,  29145, 13027, 39025, 11202, 32622, 33226, 9204,  41800, 16910, 24127, 37748, 18277, 24829,   /*[GB+DBD0, GB+DBDF]*/
  639,   8938,  19734, 6174,  19891, 3268,  3686,  32091, 22697, 22213, 378,   36416, 6341,  6604,  16144, 28001,   /*[GB+DBE0, GB+DBEF]*/
  28206, 22584, 8863,  38550, 3525,  28772, 10403, 36246, 19017, 40150, 42100, 206,   38165, 24069, 22918,          /*[GB+DBF0, GB+DBFE]*/
  1990,  4976,  15962, 19229, 30333, 25689, 15798, 18873, 7073,  10062, 5297,  1467,  1468,  15670, 25788, 7081,    /*[GB+DC40, GB+DC4F]*/
  4095,  13083, 33403, 40292, 22908, 42156, 18560, 42532, 4206,  6955,  32950, 19711, 19237, 34286, 32955, 17521,   /*[GB+DC50, GB+DC5F]*/
  26931, 27525, 23113, 30457, 26558, 13274, 4994,  5183,  33901, 17042, 15999, 19239, 5438,  8088,  31113, 7192,    /*[GB+DC60, GB+DC6F]*/
  9514,  17641, 27021, 20531, 61,    13023, 15479, 16589, 17795, 36850, 21167, 16291, 26445, 19831, 17801,          /*[GB+DC70, GB+DC7E]*/
  7210,  42150, 37000, 31119, 5786,  38836, 39959, 3241,  36512, 10227, 16020, 40227, 35385, 5365,  35970, 7822,    /*[GB+DC80, GB+DC8F]*/
  27059, 28010, 16895, 29097, 32010, 3372,  5376,  7390,  22365, 24886, 20963, 27389, 16896, 25386, 43114, 11296,   /*[GB+DC90, GB+DC9F]*/
  11604, 23895, 31834, 27740, 5368,  16684, 6366,  38501, 18002, 11550, 9292,  40074, 20832, 39012, 18807, 3722,    /*[GB+DCA0, GB+DCAF]*/
  35357, 24119, 38444, 2549,  14243, 22409, 6859,  25254, 12929, 32339, 26118, 25020, 34610, 8457,  36629, 40349,   /*[GB+DCB0, GB+DCBF]*/
  7989,  13343, 18282, 7354,  15571, 24046, 42181, 27441, 34485, 3074,  4766,  25905, 33421, 25479, 24833, 27993,   /*[GB+DCC0, GB+DCCF]*/
  1557,  42974, 16612, 37948, 22039, 8945,  24340, 19622, 471,   22255, 15351, 26893, 26074, 3689,  8476,  19257,   /*[GB+DCD0, GB+DCDF]*/
  22938, 38688, 20954, 38845, 26126, 21815, 28259, 25482, 31018, 26951, 1270,  4662,  15529, 31571, 12461, 42817,   /*[GB+DCE0, GB+DCEF]*/
  31468, 25699, 8494,  27012, 35509, 26640, 12531, 36297, 21914, 13366, 14256, 3913,  14137, 20555, 35473,          /*[GB+DCF0, GB+DCFE]*/
  26485, 16933, 6006,  19310, 5383,  297,   41777, 7834,  16882, 37166, 23877, 967,   9718,  9719,  23735, 43025,   /*[GB+DD40, GB+DD4F]*/
  27213, 7415,  486,   42721, 42218, 37283, 16378, 38227, 42377, 28951, 24463, 7560,  9562,  15466, 14454, 10098,   /*[GB+DD50, GB+DD5F]*/
  11116, 16160, 26675, 42687, 40585, 42402, 28306, 18865, 39832, 28243, 39249, 32452, 38651, 41581, 32409, 8680,    /*[GB+DD60, GB+DD6F]*/
  26035, 42697, 22735, 18010, 41590, 41314, 18868, 43504, 12431, 32538, 4610,  10503, 16229, 37988, 23919,          /*[GB+DD70, GB+DD7E]*/
  25514, 10370, 22898, 24471, 10019, 991,   20433, 23557, 18814, 27407, 27259, 13233, 37128, 34368, 4371,  4908,    /*[GB+DD80, GB+DD8F]*/
  4518,  9324,  39260, 11323, 29124, 8802,  43513, 8611,  32979, 1054,  41261, 39566, 32981, 30862, 9822,  41737,   /*[GB+DD90, GB+DD9F]*/
  34146, 36298, 9384,  14896, 20714, 30167, 11369, 42756, 16212, 1167,  28776, 33130, 39301, 7289,  21099, 39218,   /*[GB+DDA0, GB+DDAF]*/
  18338, 31758, 34223, 8510,  30022, 39219, 5928,  28425, 10007, 17675, 38745, 4534,  15000, 24881, 33593, 29676,   /*[GB+DDB0, GB+DDBF]*/
  14851, 22431, 24759, 480,   29096, 3044,  31404, 39468, 12099, 1302,  8535,  31768, 5533,  5077,  36582, 15376,   /*[GB+DDC0, GB+DDCF]*/
  5660,  13746, 32384, 38853, 9714,  10764, 25194, 8251,  27067, 34624, 32584, 2950,  16155, 25124, 17075, 33861,   /*[GB+DDD0, GB+DDDF]*/
  7409,  815,   23524, 31477, 19792, 23578, 35984, 13589, 41714, 28701, 27359, 22086, 7521,  976,   33104, 10891,   /*[GB+DDE0, GB+DDEF]*/
  13187, 18070, 746,   13777, 29461, 17679, 38867, 39806, 29924, 21275, 6654,  33873, 18724, 4951,  19211,          /*[GB+DDF0, GB+DDFE]*/
  40109, 20156, 14313, 3201,  43120, 32927, 12724, 36197, 41618, 14475, 41327, 2200,  17778, 8162,  7702,  19162,   /*[GB+DE40, GB+DE4F]*/
  9337,  27827, 16241, 12141, 38005, 13262, 43264, 7574,  39931, 14114, 11333, 17880, 23839, 18561, 18572, 19957,   /*[GB+DE50, GB+DE5F]*/
  19240, 5343,  42664, 18804, 32973, 4675,  43953, 1589,  17288, 4687,  36156, 685,   1597,  36157, 633,   4702,    /*[GB+DE60, GB+DE6F]*/
  1609,  1611,  23244, 23249, 2991,  4588,  37668, 27082, 1486,  28747, 39391, 2810,  8932,  39396, 37752,          /*[GB+DE70, GB+DE7E]*/
  24737, 40489, 32547, 31831, 43267, 5241,  10851, 5242,  7281,  5243,  3270,  22799, 42269, 3355,  42984, 30872,   /*[GB+DE80, GB+DE8F]*/
  32098, 26606, 12462, 6615,  39363, 22415, 37802, 14580, 19014, 36369, 7091,  15174, 39149, 26293, 6038,  15098,   /*[GB+DE90, GB+DE9F]*/
  37963, 26493, 16651, 35960, 18956, 12602, 36394, 15918, 27426, 43960, 13215, 21281, 7694,  24944, 11425, 35248,   /*[GB+DEA0, GB+DEAF]*/
  11322, 32606, 38355, 33120, 29817, 1417,  10894, 30500, 27318, 36263, 34447, 9174,  18138, 12888, 26527, 11280,   /*[GB+DEB0, GB+DEBF]*/
  7771,  23092, 21480, 9540,  38111, 16904, 18604, 5213,  38106, 33553, 40691, 23733, 39183, 18968, 8855,  8961,    /*[GB+DEC0, GB+DECF]*/
  29062, 21213, 31854, 3313,  8649,  23637, 359,   14574, 14338, 40491, 38144, 20218, 16055, 31309, 37490, 31,      /*[GB+DED0, GB+DEDF]*/
  22356, 13299, 9940,  563,   15373, 24610, 19029, 25384, 10036, 6373,  40949, 36536, 26003, 39484, 209,   35981,   /*[GB+DEE0, GB+DEEF]*/
  1891,  16998, 40160, 29106, 7529,  4325,  13768, 29460, 41239, 23414, 27682, 20576, 38770, 42124, 10787,          /*[GB+DEF0, GB+DEFE]*/
  41646, 33445, 18612, 30302, 12466, 9770,  17354, 1042,  5198,  15867, 1114,  39227, 42673, 14914, 39760, 4579,    /*[GB+DF40, GB+DF4F]*/
  31138, 5267,  40133, 23409, 5665,  30051, 36811, 3708,  31037, 28784, 41701, 39234, 40447, 7416,  10522, 5273,    /*[GB+DF50, GB+DF5F]*/
  32656, 22465, 37296, 4129,  36388, 30385, 6072,  3718,  40136, 29912, 20650, 10050, 41361, 28961, 2305,  3858,    /*[GB+DF60, GB+DF6F]*/
  29927, 7107,  6082,  19810, 3727,  5148,  26972, 25322, 36079, 39853, 37883, 7458,  18907, 28980, 20743,          /*[GB+DF70, GB+DF7E]*/
  10554, 41192, 30965, 7600,  1504,  1509,  18142, 40117, 37346, 20496, 18250, 31462, 25015, 27987, 39417, 27278,   /*[GB+DF80, GB+DF8F]*/
  33511, 8367,  16308, 1785,  35316, 28485, 40034, 5122,  12794, 703,   15572, 1234,  32249, 16862, 10138, 28671,   /*[GB+DF90, GB+DF9F]*/
  16676, 41577, 35076, 19884, 43998, 4989,  9001,  12215, 24147, 35500, 43388, 12776, 43908, 22531, 38015, 30929,   /*[GB+DFA0, GB+DFAF]*/
  5349,  28930, 2202,  3795,  12917, 16628, 5700,  17811, 40937, 36408, 37217, 8368,  8639,  38047, 5324,  18276,   /*[GB+DFB0, GB+DFBF]*/
  7352,  1180,  1010,  10396, 25993, 38625, 40490, 16110, 8844,  9694,  19448, 6532,  23131, 7118,  22546, 39145,   /*[GB+DFC0, GB+DFCF]*/
  29494, 16864, 12938, 28490, 12378, 5249,  18992, 37782, 34765, 1256,  4628,  10086, 12522, 35589, 37679, 23569,   /*[GB+DFD0, GB+DFDF]*/
  16801, 7170,  13352, 21752, 21408, 41065, 23233, 9372,  22156, 20669, 3626,  17701, 9421,  33283, 40944, 30239,   /*[GB+DFE0, GB+DFEF]*/
  40750, 33552, 44061, 12947, 8315,  40792, 23406, 21706, 19115, 43128, 41439, 30882, 11590, 5072,  27925,          /*[GB+DFF0, GB+DFFE]*/
  8861,  11044, 13124, 11540, 35467, 33786, 10172, 23378, 17638, 36639, 3514,  6690,  20209, 8655,  33282, 17610,   /*[GB+E040, GB+E04F]*/
  13627, 9420,  2015,  33934, 939,   34895, 24749, 26023, 42665, 43752, 24432, 17344, 22708, 39220, 2275,  34615,   /*[GB+E050, GB+E05F]*/
  15452, 39031, 25648, 37695, 21104, 27488, 958,   29098, 39647, 40439, 11483, 16991, 34618, 34619, 29749, 30722,   /*[GB+E060, GB+E06F]*/
  21923, 33619, 27340, 4275,  43472, 43760, 37565, 33251, 34625, 40368, 10971, 39010, 21050, 3183,  8411,           /*[GB+E070, GB+E07E]*/
  18954, 43093, 11854, 25656, 20831, 25657, 35807, 5857,  1450,  36328, 1374,  40906, 32664, 41973, 21067, 19132,   /*[GB+E080, GB+E08F]*/
  5454,  21284, 37599, 2545,  16820, 8262,  21285, 16938, 18729, 40639, 2875,  39163, 13055, 36988, 2888,  5176,    /*[GB+E090, GB+E09F]*/
  12068, 38997, 5526,  1961,  6502,  17607, 18345, 29290, 4331,  6365,  5217,  22463, 18051, 16996, 14502, 39063,   /*[GB+E0A0, GB+E0AF]*/
  17067, 15230, 29750, 38500, 3830,  14504, 19784, 16734, 23434, 12548, 25928, 314,   29897, 6748,  16380, 23029,   /*[GB+E0B0, GB+E0BF]*/
  11020, 3329,  30309, 9290,  6,     22298, 10910, 6331,  21,    86,    30212, 10543, 31542, 3642,  23833, 17998,   /*[GB+E0C0, GB+E0CF]*/
  2512,  24318, 24774, 38767, 1081,  29811, 5939,  21569, 23882, 15793, 18896, 24638, 4332,  14462, 23435, 25938,   /*[GB+E0D0, GB+E0DF]*/
  19883, 2622,  5826,  10893, 14948, 15928, 37714, 27585, 24027, 27314, 2295,  12881, 22510, 32620, 13863, 22447,   /*[GB+E0E0, GB+E0EF]*/
  20397, 11643, 19248, 39353, 39634, 26024, 39642, 12117, 32625, 42263, 23820, 30786, 4033,  40796, 10445,          /*[GB+E0F0, GB+E0FE]*/
  33763, 40654, 18539, 39463, 36792, 43947, 20964, 41826, 6705,  41698, 40056, 8841,  31198, 25203, 24577, 4061,    /*[GB+E140, GB+E14F]*/
  40586, 4605,  39378, 38713, 24669, 43957, 10569, 32672, 20114, 17547, 36605, 30903, 41253, 30598, 43252, 34369,   /*[GB+E150, GB+E15F]*/
  6750,  30657, 26343, 4555,  40465, 24587, 16408, 29781, 4118,  5153,  40468, 39136, 292,   41091, 14229, 24309,   /*[GB+E160, GB+E16F]*/
  3423,  39910, 18131, 40715, 37729, 14234, 30623, 7635,  23263, 38426, 37009, 38427, 22932, 27330, 4096,           /*[GB+E170, GB+E17E]*/
  37020, 19395, 21489, 21490, 22934, 35412, 14488, 27969, 21495, 37029, 1559,  28924, 28996, 33734, 14798, 26280,   /*[GB+E180, GB+E18F]*/
  37959, 18953, 5706,  41437, 6466,  24567, 26296, 449,   41692, 42099, 450,   20361, 8660,  22417, 6299,  28113,   /*[GB+E190, GB+E19F]*/
  25745, 32643, 33184, 20834, 41421, 8592,  7691,  13108, 25018, 25257, 24830, 26370, 36450, 34484, 376,   2604,    /*[GB+E1A0, GB+E1AF]*/
  17436, 425,   11817, 16425, 6535,  13671, 35651, 5347,  9630,  21000, 21810, 38078, 6606,  14437, 36290, 41873,   /*[GB+E1B0, GB+E1BF]*/
  17703, 17337, 29671, 36575, 9846,  34897, 10407, 16575, 15852, 27131, 37376, 32291, 40566, 32651, 39479, 5160,    /*[GB+E1C0, GB+E1CF]*/
  19842, 43462, 21108, 28644, 29680, 13309, 41420, 19135, 5860,  1817,  37902, 6127,  3794,  7888,  4925,  36367,   /*[GB+E1D0, GB+E1DF]*/
  37088, 11543, 17339, 33851, 3081,  12270, 37290, 42232, 14397, 26555, 27986, 26711, 26254, 220,   10112, 20662,   /*[GB+E1E0, GB+E1EF]*/
  23207, 40391, 34079, 23726, 7995,  27116, 16804, 29031, 30166, 1279,  15740, 18330, 39458, 34417, 38558,          /*[GB+E1F0, GB+E1FE]*/
  16641, 4390,  43576, 7795,  11930, 11931, 10765, 9082,  24888, 20895, 27107, 6052,  29629, 33946, 38192, 2769,    /*[GB+E240, GB+E24F]*/
  28686, 31829, 33596, 23344, 25289, 26308, 14024, 24171, 37494, 14815, 500,   7867,  3378,  35486, 6681,  40246,   /*[GB+E250, GB+E25F]*/
  25295, 8404,  24006, 22369, 35342, 7299,  15884, 7101,  9611,  38641, 25393, 653,   27539, 26999, 3169,  23221,   /*[GB+E260, GB+E26F]*/
  8091,  40401, 37974, 25924, 24007, 10409, 11401, 38576, 16029, 6306,  38230, 42570, 33863, 8911,  27108,          /*[GB+E270, GB+E27E]*/
  12796, 30522, 16315, 40067, 19919, 7418,  25925, 7148,  43495, 22766, 31789, 28952, 21841, 9725,  16385, 19321,   /*[GB+E280, GB+E28F]*/
  1896,  29638, 9794,  2062,  24014, 39810, 29639, 44029, 2192,  39243, 31237, 13687, 41715, 28840, 28953, 42121,   /*[GB+E290, GB+E29F]*/
  15614, 29993, 37692, 10491, 20460, 22713, 28301, 4939,  21422, 11678, 2701,  32762, 32873, 21111, 22560, 41357,   /*[GB+E2A0, GB+E2AF]*/
  15119, 15914, 18905, 35244, 36261, 12071, 4374,  12799, 30169, 38587, 42161, 28743, 30711, 32002, 33911, 27047,   /*[GB+E2B0, GB+E2BF]*/
  39711, 3805,  37775, 34678, 1960,  39464, 12708, 2666,  29759, 21976, 35608, 14864, 27638, 43145, 22521, 24124,   /*[GB+E2C0, GB+E2CF]*/
  33344, 10213, 23725, 35591, 34656, 32171, 160,   39643, 1290,  9406,  317,   14859, 2864,  35240, 19187, 38798,   /*[GB+E2D0, GB+E2DF]*/
  35413, 5701,  5138,  3009,  33346, 42255, 23503, 3908,  33345, 16172, 3132,  4449,  29661, 1553,  23205, 11818,   /*[GB+E2E0, GB+E2EF]*/
  4241,  23859, 5246,  37183, 44055, 22699, 8465,  3161,  38083, 37767, 31649, 36566, 2557,  16145, 36292,          /*[GB+E2F0, GB+E2FE]*/
  2824,  28704, 28705, 36115, 41441, 858,   11117, 1354,  28594, 4175,  28791, 2063,  43030, 3847,  40496, 24508,   /*[GB+E340, GB+E34F]*/
  31600, 25401, 8571,  41134, 19574, 25304, 8572,  18398, 40251, 24015, 37055, 681,   1974,  14638, 9615,  29249,   /*[GB+E350, GB+E35F]*/
  41909, 22235, 33864, 33865, 6073,  13598, 22278, 30644, 12112, 37975, 29523, 16935, 16122, 916,   31607, 35489,   /*[GB+E360, GB+E36F]*/
  11417, 14380, 3782,  7624,  20579, 1899,  28962, 22176, 13600, 38590, 16034, 42693, 3998,  34690, 31608,          /*[GB+E370, GB+E37E]*/
  22096, 17975, 13008, 39833, 35918, 27002, 44005, 42427, 28119, 3859,  34436, 35490, 26678, 24020, 31413, 42844,   /*[GB+E380, GB+E38F]*/
  34733, 21933, 16770, 37308, 34245, 34348, 35614, 16035, 2671,  17768, 13200, 24142, 27304, 21517, 37710, 38518,   /*[GB+E390, GB+E39F]*/
  10100, 16451, 40432, 945,   29707, 25275, 16963, 17168, 38143, 31129, 26613, 25824, 35516, 7966,  3119,  32516,   /*[GB+E3A0, GB+E3AF]*/
  4040,  11592, 5074,  40445, 17071, 7404,  18019, 43233, 25747, 1312,  29903, 25496, 39015, 15062, 25717, 3930,    /*[GB+E3B0, GB+E3BF]*/
  4295,  19189, 21376, 31261, 12434, 29320, 36625, 32630, 11348, 21863, 16264, 30361, 20185, 17170, 15226, 17616,   /*[GB+E3C0, GB+E3CF]*/
  39762, 3045,  33947, 33007, 12689, 7392,  26593, 26773, 11119, 31238, 26744, 10717, 25587, 23597, 25532, 29425,   /*[GB+E3D0, GB+E3DF]*/
  25070, 29592, 2738,  8223,  40038, 22250, 21651, 7086,  21532, 9753,  1556,  33060, 10880, 32626, 17818, 8941,    /*[GB+E3E0, GB+E3EF]*/
  29208, 19621, 19895, 37043, 29603, 32094, 19255, 21002, 20549, 36091, 23638, 32093, 11351, 21865, 14982,          /*[GB+E3F0, GB+E3FE]*/
  214,   6529,  39319, 27779, 16343, 25407, 29532, 23,    6310,  10803, 27457, 28967, 16558, 26339, 34836, 41591,   /*[GB+E440, GB+E44F]*/
  35668, 40699, 31069, 5207,  9894,  11465, 42578, 31661, 20238, 21130, 17645, 32457, 35349, 40381, 992,   29938,   /*[GB+E450, GB+E45F]*/
  39856, 2861,  6515,  2089,  10804, 13655, 11423, 4990,  8263,  2826,  32415, 42451, 29547, 36006, 11949, 39671,   /*[GB+E460, GB+E46F]*/
  31360, 16937, 43377, 20382, 35492, 25980, 28555, 10693, 20383, 37498, 4186,  40928, 15404, 34563, 31414,          /*[GB+E470, GB+E47E]*/
  20905, 24728, 18106, 23654, 27458, 3572,  9198,  18232, 30942, 1862,  43042, 41843, 31742, 19579, 43966, 15651,   /*[GB+E480, GB+E48F]*/
  3128,  40138, 14079, 9093,  6317,  30907, 3099,  20435, 10506, 19349, 24028, 20127, 18119, 25568, 24615, 15725,   /*[GB+E490, GB+E49F]*/
  21850, 12091, 32733, 19009, 13568, 41677, 38480, 12460, 42814, 13359, 35842, 12526, 30874, 36294, 14208, 19468,   /*[GB+E4A0, GB+E4AF]*/
  11773, 36295, 27352, 29870, 33289, 17340, 32640, 43327, 15688, 2606,  700,   33561, 21154, 12176, 42938, 24878,   /*[GB+E4B0, GB+E4BF]*/
  33583, 29675, 6770,  43295, 24165, 21666, 9026,  7949,  4856,  28500, 10037, 20043, 29329, 35188, 36586, 21671,   /*[GB+E4C0, GB+E4CF]*/
  14359, 29754, 12272, 35880, 23849, 13757, 36110, 33187, 21109, 36940, 25931, 16473, 28358, 20925, 38865, 24708,   /*[GB+E4D0, GB+E4DF]*/
  18392, 27357, 30324, 12745, 1338,  35663, 8672,  30728, 23668, 21925, 12283, 38869, 34820, 17544, 2515,  11617,   /*[GB+E4E0, GB+E4EF]*/
  20577, 12206, 18764, 42842, 37708, 20081, 36048, 8996,  29260, 29540, 28036, 28297, 31613, 2866,  17405,          /*[GB+E4F0, GB+E4FE]*/
  43967, 23924, 250,   24029, 34573, 36439, 43215, 17979, 16404, 16590, 30417, 17154, 6796,  22676, 4485,  43515,   /*[GB+E540, GB+E54F]*/
  41918, 1020,  23065, 43741, 4556,  30603, 6518,  24954, 25418, 43255, 13041, 39873, 14871, 10022, 20983, 3056,    /*[GB+E550, GB+E55F]*/
  31297, 33708, 18775, 30908, 9860,  5208,  29271, 41736, 20128, 21375, 20129, 11910, 1702,  8851,  17381, 16517,   /*[GB+E560, GB+E56F]*/
  7870,  33505, 22445, 32460, 11787, 5812,  34258, 24227, 12773, 18869, 7672,  21227, 16167, 38791, 5914,           /*[GB+E570, GB+E57E]*/
  18777, 10425, 34448, 6903,  31813, 32681, 43680, 8811,  27261, 13247, 7475,  16039, 3421,  31084, 41016, 11875,   /*[GB+E580, GB+E58F]*/
  37131, 6950,  34147, 39567, 16564, 28601, 12312, 32819, 8812,  41449, 2677,  25854, 28727, 11328, 17026, 31298,   /*[GB+E590, GB+E59F]*/
  25685, 30038, 18236, 5558,  2869,  18659, 27317, 24679, 1428,  10932, 43392, 10827, 35276, 38897, 40285, 8211,    /*[GB+E5A0, GB+E5AF]*/
  11002, 535,   21611, 10208, 5652,  21535, 39355, 3360,  23146, 25305, 25310, 33469, 18899, 25326, 12119, 13942,   /*[GB+E5B0, GB+E5BF]*/
  13940, 4586,  36507, 33422, 15170, 40785, 37774, 7582,  13563, 15083, 5352,  11545, 23682, 2188,  18217, 26294,   /*[GB+E5C0, GB+E5CF]*/
  34785, 31132, 26828, 16990, 32569, 12186, 20048, 4360,  12274, 26321, 34112, 321,   9672,  30386, 19494, 34250,   /*[GB+E5D0, GB+E5DF]*/
  19141, 15650, 35250, 21737, 30136, 17256, 13324, 12542, 31895, 42358, 16319, 42188, 12949, 7272,  2446,  33867,   /*[GB+E5E0, GB+E5EF]*/
  15633, 3027,  15081, 23325, 21501, 8533,  1310,  39946, 3265,  29448, 7912,  36627, 33343, 39708, 1181,           /*[GB+E5F0, GB+E5FE]*/
  25686, 11520, 31662, 4826,  12132, 37627, 21852, 14087, 6920,  14080, 29695, 17027, 11739, 36013, 7204,  14679,   /*[GB+E640, GB+E64F]*/
  41738, 1505,  42583, 43520, 35624, 37501, 21169, 23583, 26,    14781, 25527, 21138, 30283, 5293,  759,   34150,   /*[GB+E650, GB+E65F]*/
  18683, 30287, 16189, 37337, 37634, 23309, 33092, 27182, 30773, 30288, 25579, 18481, 29477, 4486,  2125,  23623,   /*[GB+E660, GB+E66F]*/
  5236,  1162,  27691, 9094,  43522, 33268, 38895, 12371, 31367, 19533, 16168, 30195, 27907, 29786, 32466,          /*[GB+E670, GB+E67E]*/
  10998, 41855, 41856, 17649, 38394, 40115, 30803, 23077, 33827, 13611, 9253,  20657, 15705, 29744, 43871, 30289,   /*[GB+E680, GB+E68F]*/
  34191, 8269,  32982, 22317, 20008, 30290, 23473, 43844, 31881, 35630, 10063, 36140, 18780, 29058, 20824, 22130,   /*[GB+E690, GB+E69F]*/
  20485, 14886, 43554, 10135, 23194, 39422, 29595, 5245,  42717, 27996, 25806, 36513, 26950, 29078, 20306, 14250,   /*[GB+E6A0, GB+E6AF]*/
  24346, 2743,  18214, 24405, 32216, 34310, 30201, 6033,  32736, 7271,  15099, 1694,  14594, 3041,  1289,  2840,    /*[GB+E6B0, GB+E6BF]*/
  23330, 362,   40157, 31472, 33451, 9664,  21974, 24174, 89,    24368, 3643,  18084, 36594, 25595, 24291, 3092,    /*[GB+E6C0, GB+E6CF]*/
  17839, 41352, 33671, 28131, 1406,  22959, 20620, 29348, 8853,  8848,  8454,  23316, 43461, 14544, 15803, 767,     /*[GB+E6D0, GB+E6DF]*/
  40688, 29607, 8714,  43750, 38100, 23318, 5353,  34770, 11926, 24230, 18038, 24890, 16465, 43208, 2368,  42379,   /*[GB+E6E0, GB+E6EF]*/
  33459, 389,   19496, 28120, 1641,  4800,  2974,  13513, 34639, 29485, 39394, 42739, 9206,  32338, 16911,          /*[GB+E6F0, GB+E6FE]*/
  1451,  32931, 19542, 5969,  27646, 43722, 37917, 20164, 16568, 25583, 5044,  24812, 3105,  30707, 20855, 39044,   /*[GB+E740, GB+E74F]*/
  2992,  8270,  15139, 1672,  29282, 19881, 35674, 4831,  19683, 40649, 14104, 2533,  18154, 34194, 33746, 16293,   /*[GB+E750, GB+E75F]*/
  29385, 1133,  41426, 25528, 3497,  20174, 11959, 13263, 24693, 12663, 25629, 24518, 19165, 27828, 35675, 27649,   /*[GB+E760, GB+E76F]*/
  3498,  17110, 29565, 19545, 22579, 12321, 24342, 30151, 7773,  25734, 26625, 37138, 30708, 34747, 15970,          /*[GB+E770, GB+E77E]*/
  14316, 43989, 18938, 25865, 17745, 7039,  35366, 40600, 13077, 13969, 42588, 5871,  36446, 38962, 6982,  15971,   /*[GB+E780, GB+E78F]*/
  23310, 40601, 24720, 31418, 7774,  3499,  6491,  28161, 16140, 13841, 8055,  30156, 20011, 15708, 12667, 39932,   /*[GB+E790, GB+E79F]*/
  18700, 40348, 23972, 29070, 9022,  35150, 8472,  42747, 8473,  4245,  5351,  16726, 10856, 14209, 9428,  34782,   /*[GB+E7A0, GB+E7AF]*/
  31021, 19293, 25041, 7919,  28223, 10352, 7178,  29041, 19571, 26719, 32382, 43453, 16470, 34623, 31035, 21729,   /*[GB+E7B0, GB+E7BF]*/
  12555, 29515, 1581,  9610,  43236, 21834, 14926, 41780, 27358, 9160,  18068, 38255, 13776, 1802,  24273, 20839,   /*[GB+E7C0, GB+E7CF]*/
  17842, 22174, 27712, 35080, 18906, 40910, 14158, 25465, 25680, 12121, 43893, 37213, 13014, 4342,  40549, 39002,   /*[GB+E7D0, GB+E7DF]*/
  6458,  12926, 32729, 1788,  32997, 15830, 16361, 19629, 6180,  5360,  24558, 21818, 13569, 7588,  9552,  35698,   /*[GB+E7E0, GB+E7EF]*/
  36458, 10857, 37268, 20560, 33850, 12394, 18716, 24896, 38757, 24895, 11781, 17135, 36806, 4857,  32394,          /*[GB+E7F0, GB+E7FE]*/
  43406, 25691, 14117, 43415, 17877, 1473,  31419, 12143, 37646, 7164,  10509, 5620,  15678, 8164,  5299,  38436,   /*[GB+E840, GB+E84F]*/
  145,   43694, 36403, 6324,  43063, 11283, 43265, 13089, 23105, 11163, 12901, 26043, 1830,  38830, 17113, 23174,   /*[GB+E850, GB+E85F]*/
  35771, 14119, 14120, 25473, 2735,  42536, 21799, 18183, 17883, 13091, 43909, 16954, 28216, 23950, 17325, 6834,    /*[GB+E860, GB+E86F]*/
  29482, 4624,  20304, 1686,  882,   20013, 34608, 16841, 19716, 7507,  19958, 14122, 17604, 2159,  13857,          /*[GB+E870, GB+E87E]*/
  37470, 2917,  34649, 14124, 33781, 10074, 2497,  23115, 17952, 4995,  26560, 23659, 20509, 43886, 20346, 40721,   /*[GB+E880, GB+E88F]*/
  23118, 16001, 30812, 42927, 17529, 27533, 37084, 35474, 850,   35481, 20896, 36431, 42348, 34229, 33301, 24910,   /*[GB+E890, GB+E89F]*/
  3051,  3323,  15378, 21058, 39514, 40163, 34118, 22606, 103,   30740, 14868, 12294, 38784, 5048,  4806,  36051,   /*[GB+E8A0, GB+E8AF]*/
  41365, 24676, 2460,  26515, 20134, 1443,  40651, 33074, 32780, 40450, 30852, 33419, 1620,  25024, 2648,  20664,   /*[GB+E8B0, GB+E8BF]*/
  18288, 24050, 21723, 37358, 27443, 13867, 4210,  3512,  4770,  34763, 7860,  23529, 42979, 22425, 42279, 41642,   /*[GB+E8C0, GB+E8CF]*/
  19627, 15288, 24435, 19901, 34084, 34767, 39358, 42192, 32189, 42012, 19262, 9634,  5994,  18302, 7222,  3453,    /*[GB+E8D0, GB+E8DF]*/
  16330, 17759, 36421, 26953, 42299, 41685, 10091, 24747, 31514, 9874,  15295, 11980, 11261, 10279, 14586, 20310,   /*[GB+E8E0, GB+E8EF]*/
  15739, 157,   35791, 7824,  9848,  8520,  15856, 43573, 30205, 19286, 4215,  8126,  6773,  25493, 41482,          /*[GB+E8F0, GB+E8FE]*/
  42403, 38764, 12290, 43496, 2090,  30801, 27162, 1126,  19519, 12134, 17315, 2909,  41386, 3075,  15272, 361,     /*[GB+E940, GB+E94F]*/
  6383,  26432, 18964, 21474, 21216, 20667, 29322, 28070, 12818, 21217, 36664, 1305,  10766, 1306,  28017, 16133,   /*[GB+E950, GB+E95F]*/
  16316, 1075,  11403, 27474, 27662, 34338, 34339, 13765, 21883, 34059, 29426, 6709,  41002, 22631, 41179, 23879,   /*[GB+E960, GB+E96F]*/
  34163, 19323, 1588,  1355,  27475, 105,   9983,  9304,  9305,  7662,  4288,  11460, 10177, 21890, 27757,          /*[GB+E970, GB+E97E]*/
  17183, 17684, 20191, 31485, 27953, 15475, 40266, 40267, 2973,  26598, 19219, 3057,  27979, 17184, 36607, 33022,   /*[GB+E980, GB+E98F]*/
  36705, 7471,  12699, 39874, 33023, 11471, 780,   11467, 26603, 37403, 33026, 660,   251,   32682, 38530, 17231,   /*[GB+E990, GB+E99F]*/
  26807, 20466, 4475,  18802, 10494, 14028, 6055,  15377, 43788, 28544, 22468, 40952, 18759, 17542, 13181, 24396,   /*[GB+E9A0, GB+E9AF]*/
  15538, 26221, 6939,  4478,  3434,  20186, 2705,  15537, 36114, 21116, 38864, 41717, 7974,  30388, 30187, 35210,   /*[GB+E9B0, GB+E9BF]*/
  9125,  5030,  9163,  29462, 1804,  27147, 42841, 35209, 14866, 25599, 24787, 4146,  30745, 42847, 11705, 8999,    /*[GB+E9C0, GB+E9CF]*/
  40270, 26067, 32146, 15931, 25720, 25946, 19993, 43982, 33693, 15483, 40101, 17854, 36706, 19193, 2176,  2722,    /*[GB+E9D0, GB+E9DF]*/
  39238, 327,   22036, 4926,  28177, 31262, 40398, 18747, 24301, 5444,  13169, 1839,  38340, 27045, 7362,  9703,    /*[GB+E9E0, GB+E9EF]*/
  16365, 19905, 42196, 38132, 41759, 11587, 18314, 37264, 28912, 42309, 26645, 20034, 41563, 22890, 32528,          /*[GB+E9F0, GB+E9FE]*/
  17480, 6753,  26686, 8267,  31255, 23079, 30432, 16169, 11148, 26810, 4446,  9991,  6728,  25064, 16976, 30777,   /*[GB+EA40, GB+EA4F]*/
  9993,  24296, 16256, 34030, 12664, 2996,  24207, 5694,  12144, 30443, 33030, 30331, 10738, 16146, 6745,  12557,   /*[GB+EA50, GB+EA5F]*/
  5279,  12125, 17809, 35371, 8899,  42248, 38470, 37073, 6670,  7351,  28561, 23787, 16527, 40388, 42171, 24044,   /*[GB+EA60, GB+EA6F]*/
  3450,  7355,  26373, 5986,  19437, 32090, 1870,  7217,  29077, 7218,  7540,  10218, 39398, 16052, 25755,          /*[GB+EA70, GB+EA7E]*/
  35475, 4530,  8729,  34088, 28067, 28579, 24634, 41814, 34505, 5715,  8007,  38161, 7006,  20411, 38489, 15363,   /*[GB+EA80, GB+EA8F]*/
  3366,  20035, 28584, 34422, 38498, 42937, 37102, 27092, 34105, 3962,  36922, 38499, 29233, 5899,  32570, 23018,   /*[GB+EA90, GB+EA9F]*/
  7008,  4602,  43471, 4902,  20113, 19148, 32858, 13714, 25533, 13633, 13307, 13165, 16200, 5850,  8914,  13916,   /*[GB+EAA0, GB+EAAF]*/
  40676, 23443, 19266, 2260,  1123,  40929, 24193, 24627, 8843,  17259, 9018,  10953, 30568, 9153,  40867, 35322,   /*[GB+EAB0, GB+EABF]*/
  40351, 15169, 11049, 40604, 21004, 39721, 3115,  22802, 24861, 3528,  37549, 3179,  36909, 12390, 2190,  10667,   /*[GB+EAC0, GB+EACF]*/
  10245, 35987, 17006, 102,   21930, 31988, 36265, 37447, 33755, 22529, 1015,  28898, 16923, 37801, 42308, 43447,   /*[GB+EAD0, GB+EADF]*/
  8870,  14907, 41825, 17396, 26305, 12981, 5492,  8776,  2814,  13384, 33797, 5933,  39519, 9679,  14947, 26596,   /*[GB+EAE0, GB+EAEF]*/
  14004, 14227, 24398, 20952, 9760,  33366, 9849,  12977, 15610, 13760, 24220, 16341, 25819, 27510, 548,            /*[GB+EAF0, GB+EAFE]*/
  208,   14501, 8912,  36812, 12406, 40403, 33457, 30733, 13434, 5724,  313,   33967, 38645, 27542, 26964, 19133,   /*[GB+EB40, GB+EB4F]*/
  31919, 5858,  14395, 30053, 394,   34438, 8140,  22772, 7566,  13042, 5732,  38662, 42469, 19745, 33747, 18436,   /*[GB+EB50, GB+EB5F]*/
  18459, 11658, 42035, 38193, 25926, 25296, 12105, 9673,  15757, 11621, 40524, 39545, 4074,  30041, 10825, 33211,   /*[GB+EB60, GB+EB6F]*/
  29347, 10060, 4200,  40531, 39042, 13067, 33735, 19609, 18143, 22477, 36191, 40541, 13275, 13276, 22323,          /*[GB+EB70, GB+EB7E]*/
  8362,  27769, 22269, 8093,  23667, 40369, 3848,  37056, 197,   33466, 6198,  5670,  11856, 6311,  22284, 3394,    /*[GB+EB80, GB+EB8F]*/
  41088, 19341, 24799, 42702, 11426, 41193, 38526, 43047, 32024, 19350, 6634,  38793, 33488, 19362, 19363, 11429,   /*[GB+EB90, GB+EB9F]*/
  38531, 9301,  551,   21032, 22202, 5079,  14029, 27611, 29107, 3127,  20183, 24714, 26539, 24326, 5704,  34207,   /*[GB+EBA0, GB+EBAF]*/
  4346,  6542,  36521, 38488, 16463, 40324, 24628, 3824,  14368, 6775,  6388,  39324, 40043, 10464, 40202, 33170,   /*[GB+EBB0, GB+EBBF]*/
  27114, 12233, 15050, 27381, 30513, 9484,  43268, 22339, 37257, 25440, 19630, 6613,  16119, 19904, 13674, 28530,   /*[GB+EBC0, GB+EBCF]*/
  42755, 44060, 9872,  41680, 26466, 42020, 15090, 10094, 6619,  36572, 16807, 27537, 10565, 24231, 41817, 21508,   /*[GB+EBD0, GB+EBDF]*/
  32009, 20461, 5184,  23714, 32383, 22977, 15005, 36588, 7950,  39490, 43649, 6507,  14033, 4900,  22491, 21675,   /*[GB+EBE0, GB+EBEF]*/
  32271, 7429,  29248, 3557,  38977, 9303,  20236, 1840,  30962, 42441, 4336,  9821,  21298, 27718, 28044,          /*[GB+EBF0, GB+EBFE]*/
  20744, 20745, 40418, 19606, 21397, 1827,  33495, 32932, 17237, 33828, 38413, 5585,  30995, 39933, 19684, 5412,    /*[GB+EC40, GB+EC4F]*/
  13264, 23676, 37141, 32704, 8279,  34037, 13530, 21341, 21342, 17884, 18582, 12907, 68,    8064,  5415,  19717,   /*[GB+EC50, GB+EC5F]*/
  19396, 149,   8284,  18594, 835,   11221, 11222, 11224, 1941,  26017, 31199, 41716, 3478,  26103, 15127, 15133,   /*[GB+EC60, GB+EC6F]*/
  31206, 7928,  21691, 861,   31299, 12669, 37654, 6467,  2704,  25396, 27068, 5934,  6885,  33458, 27069,          /*[GB+EC70, GB+EC7E]*/
  25933, 23225, 38646, 27518, 22382, 22097, 43867, 682,   38292, 37436, 30901, 988,   14524, 11418, 23736, 1900,    /*[GB+EC80, GB+EC8F]*/
  38522, 9316,  30905, 14662, 178,   11240, 9566,  25204, 31454, 38984, 30040, 31361, 36130, 16607, 1108,  30424,   /*[GB+EC90, GB+EC9F]*/
  28226, 18675, 19217, 39427, 33582, 37700, 27937, 35345, 28377, 1623,  27536, 15615, 29765, 1652,  1653,  29066,   /*[GB+ECA0, GB+ECAF]*/
  9667,  9792,  11736, 7968,  13025, 17468, 39432, 23826, 20956, 41163, 14996, 22761, 19489, 37982, 37077, 32728,   /*[GB+ECB0, GB+ECBF]*/
  7087,  25631, 28768, 11585, 42990, 36096, 30476, 37551, 37094, 33448, 10671, 21241, 3166,  36930, 11688, 39800,   /*[GB+ECC0, GB+ECCF]*/
  32590, 6941,  775,   35988, 1494,  31966, 19459, 20845, 28190, 40454, 38305, 39867, 7748,  30119, 34451, 15975,   /*[GB+ECD0, GB+ECDF]*/
  5017,  1619,  5750,  35896, 33696, 18287, 11819, 15151, 11840, 7923,  28849, 29597, 34210, 42186, 26387, 11825,   /*[GB+ECE0, GB+ECEF]*/
  8490,  44059, 21416, 42015, 4661,  41690, 31324, 24897, 2847,  33963, 43366, 33883, 26925, 30928, 30633,          /*[GB+ECF0, GB+ECFE]*/
  1906,  17232, 17330, 35270, 716,   7534,  26238, 26239, 11149, 25785, 22294, 13816, 5915,  32985, 30865, 30433,   /*[GB+ED40, GB+ED4F]*/
  35111, 23624, 9338,  1461,  17239, 30709, 19835, 10339, 25736, 36158, 13084, 13850, 14173, 3030,  5300,  11886,   /*[GB+ED50, GB+ED5F]*/
  34472, 25362, 6836,  32277, 13859, 17530, 32635, 27070, 8586,  21196, 26736, 9317,  32802, 25781, 10701, 3153,    /*[GB+ED60, GB+ED6F]*/
  17233, 27272, 40472, 28381, 32828, 9335,  599,   30866, 9627,  40475, 9143,  1462,  32834, 30160, 6831,           /*[GB+ED70, GB+ED7E]*/
  32278, 6837,  8496,  15276, 34253, 35253, 34263, 13074, 40523, 17824, 23925, 12320, 38801, 40478, 23944, 190,     /*[GB+ED80, GB+ED8F]*/
  38534, 34700, 11887, 37546, 6476,  26080, 16994, 34725, 29435, 10625, 35720, 37863, 35724, 7337,  29737, 17052,   /*[GB+ED90, GB+ED9F]*/
  24913, 7001,  13630, 12535, 23349, 22678, 37184, 43600, 26764, 25299, 21884, 30939, 25154, 7021,  21076, 21249,   /*[GB+EDA0, GB+EDAF]*/
  9112,  39697, 39705, 5248,  36168, 21727, 12931, 8944,  5653,  12811, 3244,  7089,  36515, 43323, 1488,  8230,    /*[GB+EDB0, GB+EDBF]*/
  7651,  79,    18333, 19634, 41033, 31578, 5998,  17261, 32112, 8505,  35480, 20894, 34098, 25644, 41140, 6621,    /*[GB+EDC0, GB+EDCF]*/
  22557, 9363,  33189, 25129, 7009,  974,   6509,  3411,  6783,  14645, 6079,  36118, 1534,  41587, 10369, 27687,   /*[GB+EDD0, GB+EDDF]*/
  26106, 26504, 7067,  5866,  14168, 2302,  21311, 2148,  16196, 42220, 8622,  8694,  35688, 21658, 16614, 7078,    /*[GB+EDE0, GB+EDEF]*/
  21724, 5426,  28614, 39986, 37817, 30026, 43607, 3634,  22166, 17397, 13895, 6057,  30209, 36468, 22823,          /*[GB+EDF0, GB+EDFE]*/
  10873, 39812, 32359, 619,   7104,  5946,  5455,  23653, 24511, 19442, 3291,  15060, 17976, 11132, 25670, 7462,    /*[GB+EE40, GB+EE4F]*/
  7308,  32803, 35083, 17222, 28512, 37897, 37898, 10551, 6987,  39672, 24418, 17977, 8684,  13656, 31674, 12626,   /*[GB+EE50, GB+EE5F]*/
  17022, 13662, 20452, 31532, 3488,  38950, 40335, 11733, 10813, 15068, 31927, 31928, 24376, 17407, 31929, 43530,   /*[GB+EE60, GB+EE6F]*/
  43529, 4487,  6521,  17412, 30609, 10822, 25337, 16409, 5107,  36082, 25897, 37908, 27589, 31094, 7314,           /*[GB+EE70, GB+EE7E]*/
  7483,  36713, 33072, 16239, 39069, 43105, 36714, 34452, 35405, 38003, 40175, 27695, 6129,  6130,  14199, 16974,   /*[GB+EE80, GB+EE8F]*/
  17986, 17746, 24312, 32302, 20792, 4979,  37347, 11003, 25737, 9863,  36401, 36883, 12672, 3031,  27332, 21347,   /*[GB+EE90, GB+EE9F]*/
  1832,  30030, 24178, 27455, 29813, 17013, 21063, 16397, 21935, 24303, 3486,  16253, 19150, 9827,  31438, 1248,    /*[GB+EEA0, GB+EEAF]*/
  26715, 31216, 7816,  41761, 28305, 32399, 31892, 8493,  9077,  9768,  18379, 36819, 24093, 17545, 18115, 13050,   /*[GB+EEB0, GB+EEBF]*/
  40918, 11067, 10056, 15709, 14790, 8847,  37939, 24489, 41429, 18950, 31828, 4388,  28000, 21215, 2768,  23342,   /*[GB+EEC0, GB+EECF]*/
  2246,  30514, 15585, 648,   25375, 7863,  16313, 31684, 12795, 498,   39735, 41885, 9771,  16369, 24542, 2189,    /*[GB+EED0, GB+EEDF]*/
  2028,  40228, 22262, 30638, 6182,  29454, 28914, 36101, 30317, 1287,  22709, 23999, 7141,  16338, 17763, 7589,    /*[GB+EEE0, GB+EEEF]*/
  39305, 3541,  13637, 22558, 37491, 5600,  6300,  38497, 16150, 42827, 6504,  6528,  11932, 26657, 10529,          /*[GB+EEF0, GB+EEFE]*/
  34473, 24391, 19960, 17573, 23117, 26708, 31525, 15182, 12576, 38936, 39067, 17984, 8231,  1632,  9887,  8575,    /*[GB+EF40, GB+EF4F]*/
  34061, 41250, 1640,  27545, 488,   30488, 19057, 9896,  36131, 28251, 15659, 1660,  29561, 32830, 37136, 37339,   /*[GB+EF50, GB+EF5F]*/
  29791, 16170, 29792, 7703,  19543, 33833, 19610, 24284, 24285, 19550, 1679,  1680,  1681,  18942, 1682,  27834,   /*[GB+EF60, GB+EF6F]*/
  8285,  35639, 37078, 41227, 29524, 37309, 19499, 7914,  7708,  7946,  28764, 2367,  12956, 6501,  29621,          /*[GB+EF70, GB+EF7E]*/
  32068, 41168, 32014, 27071, 39791, 15765, 3840,  38647, 7837,  7838,  30173, 38642, 31692, 37865, 44076, 1357,    /*[GB+EF80, GB+EF8F]*/
  14722, 30853, 817,   4688,  31425, 29640, 818,   28955, 7245,  10595, 27074, 31291, 14381, 13653, 1901,  37310,   /*[GB+EF90, GB+EF9F]*/
  31609, 27878, 6301,  41895, 27768, 3994,  30692, 207,   27294, 17716, 17363, 30938, 16550, 40927, 18225, 9192,    /*[GB+EFA0, GB+EFAF]*/
  7300,  5203,  20373, 19573, 16132, 13900, 17633, 25974, 15386, 3,     25556, 23416, 1019,  5805,  16475, 17144,   /*[GB+EFB0, GB+EFBF]*/
  9856,  12769, 23799, 15719, 30583, 43488, 25845, 16163, 29533, 7441,  2672,  29771, 12114, 24,    19873, 25563,   /*[GB+EFC0, GB+EFCF]*/
  8030,  21127, 22108, 9315,  15702, 22315, 19507, 38317, 13604, 1810,  1651,  30704, 20850, 20475, 39032, 43830,   /*[GB+EFD0, GB+EFDF]*/
  36128, 5912,  2876,  15943, 24686, 19996, 7029,  17478, 24717, 4991,  25574, 5867,  12878, 43402, 38395, 2734,    /*[GB+EFE0, GB+EFEF]*/
  1675,  42575, 28492, 5164,  42405, 1188,  43564, 22058, 29139, 20225, 12972, 8402,  17631, 16383, 27019,          /*[GB+EFF0, GB+EFFE]*/
  4689,  34691, 37170, 15772, 7596,  17835, 33667, 1983,  22660, 7464,  2276,  16090, 6720,  29939, 39546, 28971,   /*[GB+F040, GB+F04F]*/
  37324, 12722, 10507, 28972, 14081, 43256, 1903,  34574, 2282,  37616, 30604, 7938,  41367, 32911, 10023, 7472,    /*[GB+F050, GB+F05F]*/
  23365, 40469, 11741, 12314, 12641, 13817, 11522, 130,   30769, 8102,  32921, 9823,  2678,  29745, 30775, 2126,    /*[GB+F060, GB+F06F]*/
  9140,  34018, 17099, 19607, 29787, 30913, 37638, 32983, 22001, 30779, 20789, 1453,  39922, 35631, 14880,          /*[GB+F070, GB+F07E]*/
  27652, 17111, 43160, 28163, 3895,  5586,  38429, 13079, 26960, 3501,  32946, 34702, 41214, 8103,  10608, 21343,   /*[GB+F080, GB+F08F]*/
  37021, 22013, 2918,  34704, 20511, 40665, 22525, 6494,  32047, 7219,  11059, 17834, 33551, 6698,  16057, 10493,   /*[GB+F090, GB+F09F]*/
  9788,  41785, 13461, 27801, 22873, 8822,  26928, 10139, 14350, 10969, 33636, 24528, 6356,  11841, 39077, 15219,   /*[GB+F0A0, GB+F0AF]*/
  39980, 804,   41815, 9709,  6546,  19910, 26472, 3632,  29507, 7556,  42349, 9881,  35601, 20312, 2054,  18057,   /*[GB+F0B0, GB+F0BF]*/
  9789,  39793, 34340, 31040, 33374, 21714, 172,   902,   4547,  11703, 7444,  4684,  21128, 33473, 37439, 13802,   /*[GB+F0C0, GB+F0CF]*/
  38795, 41658, 19602, 18921, 14307, 15330, 39899, 11879, 20159, 10070, 22637, 6459,  14493, 18291, 28099, 18305,   /*[GB+F0D0, GB+F0DF]*/
  39208, 8952,  16366, 5489,  41071, 23756, 42999, 36039, 13574, 36487, 37828, 42368, 17714, 33455, 5162,  34333,   /*[GB+F0E0, GB+F0EF]*/
  27881, 42889, 8024,  9853,  32776, 39661, 39408, 5448,  17283, 38281, 11504, 2797,  19871, 13690, 27745,          /*[GB+F0F0, GB+F0FE]*/
  37575, 2065,  8138,  2078,  22768, 1378,  2105,  31811, 10634, 7939,  13818, 184,   139,   34273, 40336, 8167,    /*[GB+F140, GB+F14F]*/
  24360, 35360, 20648, 39792, 8299,  10782, 5937,  32134, 41575, 3719,  36326, 43031, 42064, 23835, 35398, 27109,   /*[GB+F150, GB+F15F]*/
  27543, 40408, 33014, 42129, 5550,  20192, 39250, 2080,  820,   15907, 32141, 38293, 26425, 33015, 26433, 15159,   /*[GB+F160, GB+F16F]*/
  24548, 41446, 40017, 23809, 42786, 15630, 43043, 23320, 15406, 24024, 40689, 13711, 19342, 41786, 30491,          /*[GB+F170, GB+F17E]*/
  8794,  37173, 28842, 1403,  32144, 32145, 29649, 19508, 20687, 24252, 30911, 42473, 27168, 30969, 6632,  36258,   /*[GB+F180, GB+F18F]*/
  26621, 28465, 15160, 7598,  10599, 2106,  42853, 38528, 20589, 42703, 5565,  10600, 19524, 15492, 29721, 25898,   /*[GB+F190, GB+F19F]*/
  20911, 3865,  22105, 622,   13209, 12301, 1646,  20532, 38947, 41147, 19666, 38660, 4020,  632,   17416, 38383,   /*[GB+F1A0, GB+F1AF]*/
  6217,  24152, 6135,  26545, 38179, 29710, 33535, 26125, 43269, 1523,  37368, 31380, 6711,  16384, 39529, 36252,   /*[GB+F1B0, GB+F1BF]*/
  15626, 39670, 37669, 2727,  22346, 27054, 14804, 21181, 23647, 5599,  25193, 9223,  16523, 18618, 3555,  18721,   /*[GB+F1C0, GB+F1CF]*/
  13898, 1695,  4217,  31150, 1350,  15391, 7131,  5226,  984,   819,   20237, 1536,  17464, 3783,  41638, 25627,   /*[GB+F1D0, GB+F1DF]*/
  27324, 23658, 24125, 35691, 16025, 5124,  14800, 17897, 43567, 3228,  29627, 12767, 17789, 30696, 23497, 19815,   /*[GB+F1E0, GB+F1EF]*/
  14192, 23305, 22146, 6379,  6462,  5436,  19297, 23143, 9884,  17088, 335,   30579, 10613, 24867, 10863,          /*[GB+F1F0, GB+F1FE]*/
  17651, 10823, 31815, 36014, 31976, 16096, 7342,  3609,  35451, 46,    20157, 43217, 42710, 28382, 24254, 17158,   /*[GB+F240, GB+F24F]*/
  30914, 17382, 43686, 16494, 24976, 24977, 36991, 7940,  27722, 36992, 9340,  37406, 33499, 24267, 4832,  24268,   /*[GB+F250, GB+F25F]*/
  25426, 7941,  12322, 25422, 12777, 39584, 31099, 26691, 34156, 43688, 17030, 27265, 29564, 9902,  32156, 10197,   /*[GB+F260, GB+F26F]*/
  29795, 25351, 3593,  42518, 19546, 23948, 30979, 33834, 2543,  6823,  37010, 40119, 43775, 27724, 28164,          /*[GB+F270, GB+F27E]*/
  24997, 42528, 29352, 20176, 33835, 20488, 41376, 22138, 410,   2387,  1683,  4834,  26453, 1470,  42529, 39945,   /*[GB+F280, GB+F28F]*/
  35770, 11961, 1992,  29989, 34879, 19169, 41337, 7076,  19556, 32160, 2639,  6226,  14322, 31421, 37022, 20501,   /*[GB+F290, GB+F29F]*/
  41216, 14619, 11103, 38934, 16379, 10781, 7463,  43101, 23066, 20755, 27689, 10999, 27327, 24385, 11569, 25378,   /*[GB+F2A0, GB+F2AF]*/
  26263, 13291, 2793,  12386, 9360,  21256, 8511,  24063, 27445, 34420, 10905, 14754, 9497,  6679,  38632, 3631,    /*[GB+F2B0, GB+F2BF]*/
  10620, 9715,  11091, 18361, 39224, 26860, 41074, 26215, 19301, 3463,  39371, 26152, 13642, 22561, 42375, 29516,   /*[GB+F2C0, GB+F2CF]*/
  26407, 31478, 17212, 24898, 14272, 37111, 22172, 28549, 41518, 28242, 33308, 18071, 4173,  8567,  25555, 26034,   /*[GB+F2D0, GB+F2DF]*/
  24934, 33660, 39830, 7933,  10501, 10418, 38287, 24095, 31358, 26677, 32318, 17640, 21372, 4513,  27160, 22498,   /*[GB+F2E0, GB+F2EF]*/
  8598,  17018, 16401, 8790,  29776, 39542, 39256, 19817, 26338, 1499,  20979, 25951, 336,   20818, 20930,          /*[GB+F2F0, GB+F2FE]*/
  15037, 38452, 37655, 32084, 24361, 42805, 37030, 19724, 20205, 30993, 34650, 13545, 29366, 15518, 33844, 12081,   /*[GB+F340, GB+F34F]*/
  18200, 1689,  27102, 38485, 1624,  35418, 25881, 43638, 29886, 17003, 29350, 32752, 32768, 32769, 39498, 9031,    /*[GB+F350, GB+F35F]*/
  38235, 265,   863,   989,   4641,  31115, 25671, 16781, 31956, 9445,  24248, 39575, 30067, 2140,  34869, 727,     /*[GB+F360, GB+F36F]*/
  2141,  4646,  22006, 18940, 19838, 34875, 40685, 31121, 1849,  16842, 19961, 9121,  25771, 16321, 25750,          /*[GB+F370, GB+F37E]*/
  17747, 27755, 17141, 5938,  7903,  35615, 26868, 5551,  17150, 1841,  7678,  24025, 7680,  31161, 866,   1404,    /*[GB+F380, GB+F38F]*/
  20981, 8602,  7567,  27169, 26436, 9529,  17225, 23928, 43073, 28254, 30229, 31175, 18462, 1843,  5965,  23936,   /*[GB+F390, GB+F39F]*/
  29698, 20630, 40105, 33706, 3661,  30761, 23693, 28982, 12311, 2531,  24281, 30764, 33720, 34744, 42582, 41370,   /*[GB+F3A0, GB+F3AF]*/
  29316, 20984, 23932, 12649, 23621, 28153, 12889, 21380, 2886,  18690, 21794, 18167, 6909,  26559, 8351,  38778,   /*[GB+F3B0, GB+F3BF]*/
  26111, 34186, 28911, 42873, 39448, 13135, 6843,  12954, 13882, 41478, 43572, 11834, 26139, 24544, 5266,  28586,   /*[GB+F3C0, GB+F3CF]*/
  40798, 9638,  18355, 29625, 31346, 13579, 1489,  3635,  16643, 1319,  34425, 36672, 26663, 41899, 40374, 28946,   /*[GB+F3D0, GB+F3DF]*/
  9108,  23533, 28240, 34922, 26102, 40817, 25842, 43034, 27495, 25405, 32200, 1370,  5452,  16587, 40010, 34823,   /*[GB+F3E0, GB+F3EF]*/
  41728, 17087, 12295, 11510, 9621,  7981,  18431, 1413,  3728,  29961, 21782, 6662,  20135, 6949,  10260,          /*[GB+F3F0, GB+F3FE]*/
  41923, 43689, 29442, 13961, 32187, 11757, 17309, 24996, 18699, 41794, 23949, 20691, 27622, 20794, 20795, 27849,   /*[GB+F440, GB+F44F]*/
  35772, 19097, 25356, 25355, 22520, 17241, 23177, 1850,  26933, 6699,  6719,  22632, 11470, 34019, 6732,  10727,   /*[GB+F450, GB+F45F]*/
  6735,  6734,  15251, 39960, 39964, 36980, 8689,  25957, 10195, 43691, 19611, 10199, 28201, 21190, 13435, 24917,   /*[GB+F460, GB+F46F]*/
  8858,  35735, 39897, 31934, 24804, 18851, 8977,  24286, 1464,  13322, 35768, 4125,  36889, 41217, 39474,          /*[GB+F470, GB+F47E]*/
  5708,  27001, 14639, 459,   11419, 32075, 6312,  13312, 35930, 7310,  7443,  27899, 10875, 32020, 22111, 14774,   /*[GB+F480, GB+F48F]*/
  28513, 659,   40098, 24099, 19989, 33018, 11717, 19922, 40500, 7880,  8148,  22387, 39257, 24265, 21987, 11142,   /*[GB+F490, GB+F49F]*/
  34142, 6218,  40597, 5837,  2169,  17421, 42797, 39436, 39452, 3919,  33954, 23007, 23352, 4356,  28006, 37957,   /*[GB+F4A0, GB+F4AF]*/
  1194,  42564, 650,   7899,  9355,  19914, 42886, 40800, 33615, 28241, 32779, 21371, 29024, 2530,  3943,  21322,   /*[GB+F4B0, GB+F4BF]*/
  25880, 22944, 13583, 26328, 27891, 1459,  5897,  25617, 30213, 14656, 30699, 33703, 34217, 21504, 446,   18357,   /*[GB+F4C0, GB+F4CF]*/
  31392, 33610, 43469, 2448,  19131, 43733, 27633, 11511, 40594, 4686,  35808, 27256, 26357, 14233, 9373,  13437,   /*[GB+F4D0, GB+F4DF]*/
  38125, 19296, 33611, 43035, 7975,  13926, 24225, 11137, 38369, 21531, 42113, 24928, 25052, 37325, 5788,  8398,    /*[GB+F4E0, GB+F4EF]*/
  26448, 15222, 15384, 19049, 43486, 40623, 22485, 41637, 14141, 28922, 6465,  8954,  42762, 37964, 9721,           /*[GB+F4F0, GB+F4FE]*/
  26437, 10698, 24032, 19353, 32150, 1989,  26346, 24476, 8616,  1421,  4721,  32912, 15409, 6263,  524,   39264,   /*[GB+F540, GB+F54F]*/
  10375, 24033, 22870, 35446, 30494, 869,   8806,  41051, 15640, 9736,  28801, 6570,  5417,  30427, 14680, 29127,   /*[GB+F550, GB+F55F]*/
  11560, 34697, 7569,  253,   32683, 41509, 42854, 38724, 19068, 20591, 31625, 31117, 38379, 1934,  32820, 14303,   /*[GB+F560, GB+F56F]*/
  16710, 10189, 34265, 9331,  12475, 17773, 8621,  16347, 35626, 7162,  16040, 31095, 21686, 28256, 41052,          /*[GB+F570, GB+F57E]*/
  30230, 25899, 39576, 22663, 41616, 10378, 9447,  29843, 33327, 26349, 28045, 24648, 12223, 31366, 18244, 27908,   /*[GB+F580, GB+F58F]*/
  27909, 16350, 21317, 3586,  18144, 43784, 33737, 39122, 28469, 43533, 24983, 41933, 34698, 22664, 4560,  13514,   /*[GB+F590, GB+F59F]*/
  6269,  4944,  32131, 34237, 21953, 42224, 36953, 27964, 3571,  31798, 17974, 17153, 23785, 11732, 31079, 35813,   /*[GB+F5A0, GB+F5AF]*/
  10572, 30768, 17740, 2201,  14477, 33745, 15671, 18247, 36276, 28821, 5174,  7079,  26163, 36179, 4971,  1770,    /*[GB+F5B0, GB+F5BF]*/
  849,   30316, 13889, 8397,  25543, 42115, 8403,  28019, 18381, 32127, 13591, 2166,  30480, 17051, 25655, 1352,    /*[GB+F5C0, GB+F5CF]*/
  34433, 34236, 12994, 14275, 18809, 13453, 4580,  12027, 3726,  42447, 6166,  2088,  42134, 14070, 6411,  4337,    /*[GB+F5D0, GB+F5DF]*/
  42611, 15549, 7159,  5152,  24250, 27258, 23070, 20785, 24802, 4202,  15961, 24719, 7766,  4975,  42914, 19232,   /*[GB+F5E0, GB+F5EF]*/
  2908,  19095, 43884, 35305, 42258, 6251,  22087, 35607, 22125, 24109, 11663, 15861, 28180, 9717,  43481,          /*[GB+F5F0, GB+F5FE]*/
  25863, 9862,  42738, 6576,  17383, 8053,  22743, 38415, 17159, 20166, 15334, 3062,  15028, 20436, 19374, 43773,   /*[GB+F640, GB+F64F]*/
  18155, 21381, 43692, 42519, 22874, 11788, 39586, 6013,  28730, 28471, 12224, 31101, 11525, 35455, 42864, 17316,   /*[GB+F650, GB+F65F]*/
  43693, 40891, 1511,  1512,  12225, 26694, 40895, 32614, 32615, 39589, 4521,  27267, 6428,  12327, 18785, 36874,   /*[GB+F660, GB+F66F]*/
  26241, 26242, 13962, 1165,  7492,  37140, 8830,  27591, 9009,  34072, 32188, 11760, 28997, 27503, 36026,          /*[GB+F670, GB+F67E]*/
  32987, 25522, 11004, 33270, 7883,  27727, 19553, 20658, 28808, 28734, 9996,  43537, 30983, 30350, 37348, 7329,    /*[GB+F680, GB+F68F]*/
  39073, 25431, 24999, 32988, 27505, 28482, 18702, 350,   17832, 12447, 21905, 13531, 31373, 26456, 13851, 28473,   /*[GB+F690, GB+F69F]*/
  20800, 9513,  29925, 42443, 43483, 26114, 15113, 39473, 18383, 33008, 31486, 13454, 23837, 7936,  27956, 38610,   /*[GB+F6A0, GB+F6AF]*/
  67,    34599, 20711, 3429,  15541, 776,   31356, 43498, 16513, 39673, 4611,  33213, 21656, 40071, 32158, 43205,   /*[GB+F6B0, GB+F6BF]*/
  30181, 15743, 15392, 20580, 26521, 4090,  26167, 20317, 33478, 40636, 22185, 406,   19532, 1008,  35359, 39235,   /*[GB+F6C0, GB+F6CF]*/
  7879,  517,   24464, 22867, 19920, 29836, 8765,  11554, 30484, 10179, 14650, 32789, 7563,  13456, 14282, 34692,   /*[GB+F6D0, GB+F6DF]*/
  36330, 9444,  18107, 18656, 13800, 28799, 31363, 10371, 27901, 12214, 13466, 26038, 19354, 43769, 7937,  17155,   /*[GB+F6E0, GB+F6EF]*/
  3058,  9861,  22738, 22871, 6264,  28724, 43518, 8206,  6417,  7478,  26236, 8814,  12315, 1507,  27720,          /*[GB+F6F0, GB+F6FE]*/
  33839, 26352, 1716,  13532, 13524, 42916, 14174, 35637, 43108, 39055, 41377, 16295, 36200, 1743,  39951, 26457,   /*[GB+F740, GB+F74F]*/
  34752, 1994,  14429, 36355, 29990, 12333, 43991, 28171, 28172, 7709,  10344, 19170, 36356, 21722, 33899, 40925,   /*[GB+F750, GB+F75F]*/
  34646, 8213,  10003, 11564, 16833, 40896, 27730, 41219, 9014,  10346, 38994, 18253, 3110,  17888, 29197, 147,     /*[GB+F760, GB+F76F]*/
  27334, 13543, 35958, 11891, 29198, 18589, 19108, 18593, 21805, 41742, 34705, 7512,  19963, 10076, 18201,          /*[GB+F770, GB+F77E]*/
  34294, 5707,  13303, 32015, 458,   11120, 39244, 41039, 518,   24090, 16344, 31610, 40887, 11557, 16816, 41043,   /*[GB+F780, GB+F78F]*/
  16036, 39117, 24956, 28466, 1161,  3103,  31086, 32984, 32611, 26688, 14166, 23695, 13827, 32934, 33830, 11884,   /*[GB+F790, GB+F79F]*/
  9011,  343,   24978, 30346, 9992,  37338, 17830, 1715,  36199, 20791, 21903, 39045, 10337, 28165, 43990, 18249,   /*[GB+F7A0, GB+F7AF]*/
  5281,  37060, 5288,  25719, 20781, 13826, 15415, 27263, 9626,  1005,  14766, 31671, 16699, 9806,  6008,  11516,   /*[GB+F7B0, GB+F7BF]*/
  9319,  16411, 1437,  19833, 25236, 16840, 1847,  6833,  21198, 490,   36845, 18848, 34846, 32541, 3672,  34683,   /*[GB+F7C0, GB+F7CF]*/
  36966, 31429, 30869, 39054, 1626,  17145, 20975, 26873, 31362, 13481, 43517, 35621, 26690, 15245, 1848,  12148,   /*[GB+F7D0, GB+F7DF]*/
  19106, 21984, 12432, 21451, 13311, 16037, 42955, 21455, 24989, 344,   28388, 19172, 5407,  4303,  39331, 34154,   /*[GB+F7E0, GB+F7EF]*/
  37731, 26449, 6824,  18162, 26070, 2438,  261,   8156,  39385, 33332, 36890, 33774, 26347, 10636, 40979,          /*[GB+F7F0, GB+F7FE]*/
  41215, 10026, 22945, 37977, 8577,  18399, 15237, 2197,  36945, 8656,  6253,  12998, 8330,  27360, 8969,  28712,   /*[GB+F840, GB+F84F]*/
  8332,  21934, 821,   40012, 42069, 11868, 25942, 8423,  624,   33019, 13801, 28719, 39861, 8364,  37244, 15926,   /*[GB+F850, GB+F85F]*/
  15927, 24146, 12062, 41849, 824,   36967, 36437, 41980, 7871,  8333,  33020, 23465, 5401,  9246,  27313, 19355,   /*[GB+F860, GB+F86F]*/
  21776, 8603,  32152, 21851, 18443, 1540,  42476, 9247,  40018, 4698,  26509, 34847, 3664,  5566,  15410,          /*[GB+F870, GB+F87E]*/
  37400, 9737,  42580, 39877, 37065, 39878, 36442, 31417, 39879, 31250, 38794, 6978,  33265, 7570,  9900,  120,     /*[GB+F880, GB+F88F]*/
  42073, 36987, 11276, 34856, 13664, 19069, 42855, 37132, 31088, 11430, 20592, 27321, 22187, 9252,  27005, 14304,   /*[GB+F890, GB+F89F]*/
  35627, 0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+F8A0, GB+F8AF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+F8B0, GB+F8BF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+F8C0, GB+F8CF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+F8D0, GB+F8DF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+F8E0, GB+F8EF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,              /*[GB+F8F0, GB+F8FE]*/
  42708, 3666,  20593, 11277, 22872, 7343,  20320, 13665, 13504, 31817, 12066, 32185, 2220,  33328, 15706, 39914,   /*[GB+F940, GB+F94F]*/
  2128,  16098, 16099, 1160,  33738, 16100, 15498, 31744, 15027, 31096, 7315,  7316,  16903, 11751, 33401, 28470,   /*[GB+F950, GB+F95F]*/
  17417, 14490, 23655, 20168, 24112, 29134, 8632,  189,   43408, 23946, 25958, 25347, 908,   6270,  20169, 26813,   /*[GB+F960, GB+F96F]*/
  13834, 15505, 31839, 36445, 40023, 24985, 18156, 37643, 43219, 16592, 7261,  17160, 28603, 24986, 15029,          /*[GB+F970, GB+F97E]*/
  38416, 38417, 15030, 43534, 17384, 6577,  24814, 4561,  9415,  15418, 15963, 38418, 43988, 13075, 29283, 38809,   /*[GB+F980, GB+F98F]*/
  3896,  21720, 27268, 191,   26244, 31103, 11761, 31104, 7493,  14537, 20988, 8634,  4523,  31820, 36875, 11160,   /*[GB+F990, GB+F99F]*/
  40121, 0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+F9A0, GB+F9AF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+F9B0, GB+F9BF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+F9C0, GB+F9CF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+F9D0, GB+F9DF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+F9E0, GB+F9EF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,              /*[GB+F9F0, GB+F9FE]*/
  24229, 17163, 21144, 11762, 38810, 4401,  33501, 15508, 6579,  2487,  7906,  11217, 38826, 40122, 34282, 33097,   /*[GB+FA40, GB+FA4F]*/
  28735, 11218, 4204,  30783, 34158, 27506, 19554, 13269, 9747,  13848, 30196, 10834, 4703,  4704,  38441, 37461,   /*[GB+FA50, GB+FA5F]*/
  37012, 13085, 18541, 31258, 16661, 31011, 31012, 38442, 31822, 20659, 34876, 9147,  31259, 3406,  13270, 31885,   /*[GB+FA60, GB+FA6F]*/
  41665, 351,   37408, 37737, 23476, 3899,  42534, 19613, 39056, 20204, 1474,  29356, 43418, 39599, 33334,          /*[GB+FA70, GB+FA7E]*/
  15988, 38615, 31108, 29573, 14323, 38453, 11963, 1478,  38834, 29991, 12334, 7781,  14324, 18944, 37023, 9149,    /*[GB+FA80, GB+FA8F]*/
  15339, 34398, 34399, 31823, 20721, 43992, 39952, 38837, 20181, 31886, 34400, 36192, 38459, 24209, 29199, 20505,   /*[GB+FA90, GB+FA9F]*/
  33778, 0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+FAA0, GB+FAAF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+FAB0, GB+FABF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+FAC0, GB+FACF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+FAD0, GB+FADF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+FAE0, GB+FAEF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,              /*[GB+FAF0, GB+FAFE]*/
  37929, 13097, 40858, 39606, 41220, 37658, 37144, 24210, 23178, 11892, 21494, 38839, 21355, 5978,  40308, 39954,   /*[GB+FB40, GB+FB4F]*/
  17953, 2226,  19964, 11223, 19726, 29367, 40313, 38841, 10077, 26564, 18202, 20353, 28672, 2471,  34787, 19280,   /*[GB+FB50, GB+FB5F]*/
  36170, 11266, 42676, 33302, 17146, 25306, 9407,  40009, 29913, 11127, 26224, 38324, 33089, 19510, 13219, 38325,   /*[GB+FB60, GB+FB6F]*/
  37721, 12135, 41203, 21318, 29357, 19974, 14823, 19356, 13963, 34393, 5179,  13975, 13988, 36742, 39156,          /*[GB+FB70, GB+FB7E]*/
  4921,  23741, 4922,  23743, 43049, 13819, 21454, 39690, 19538, 3401,  16041, 19159, 22746, 15335, 16042, 15033,   /*[GB+FB80, GB+FB8F]*/
  18506, 34641, 34394, 13613, 21472, 18542, 41379, 15038, 25008, 19418, 36749, 4923,  20732, 11128, 3219,  8425,    /*[GB+FB90, GB+FB9F]*/
  21698, 0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+FBA0, GB+FBAF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+FBB0, GB+FBBF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+FBC0, GB+FBCF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+FBD0, GB+FBDF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+FBE0, GB+FBEF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,              /*[GB+FBF0, GB+FBFE]*/
  21699, 8431,  23768, 26602, 26441, 22188, 8434,  34586, 17385, 21705, 3905,  8287,  26434, 21702, 21082, 22155,   /*[GB+FC40, GB+FC4F]*/
  43774, 23375, 8169,  12277, 14824, 10104, 31203, 31695, 12013, 16941, 11434, 3677,  11225, 38326, 5501,  34001,   /*[GB+FC50, GB+FC5F]*/
  31991, 22103, 25421, 6168,  36864, 21171, 36876, 40289, 5634,  36619, 36732, 36880, 5509,  41797, 5414,  2439,    /*[GB+FC60, GB+FC6F]*/
  37739, 21148, 41280, 36896, 6841,  19965, 8176,  21677, 40111, 4974,  26605, 3202,  32238, 42865, 42080,          /*[GB+FC70, GB+FC7E]*/
  21351, 353,   1744,  32164, 1483,  3184,  21577, 6483,  43505, 9796,  6575,  8160,  40027, 3067,  9148,  25181,   /*[GB+FC80, GB+FC8F]*/
  40028, 30710, 30953, 29163, 8044,  33028, 495,   6268,  32155, 42585, 26526, 28602, 28803, 28807, 31497, 15510,   /*[GB+FC90, GB+FC9F]*/
  15036, 0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+FCA0, GB+FCAF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+FCB0, GB+FCBF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+FCC0, GB+FCCF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+FCD0, GB+FCDF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+FCE0, GB+FCEF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,              /*[GB+FCF0, GB+FCFE]*/
  12731, 15516, 36885, 31745, 29576, 34604, 17895, 37444, 33492, 33496, 11476, 35301, 7506,  35676, 33123, 40980,   /*[GB+FD40, GB+FD4F]*/
  23280, 22542, 24937, 41108, 13505, 43535, 13086, 13094, 3785,  3437,  3438,  11152, 36474, 38611, 35287, 781,     /*[GB+FD50, GB+FD5F]*/
  40855, 35293, 2789,  3673,  36877, 15554, 31371, 19380, 19364, 4150,  26698, 35297, 16520, 23096, 15338, 37407,   /*[GB+FD60, GB+FD6F]*/
  4622,  40421, 39695, 4232,  38009, 22747, 40857, 43778, 26581, 40424, 36894, 23480, 7510,  33222, 38463,          /*[GB+FD70, GB+FD7E]*/
  4650,  43779, 6143,  4233,  14975, 36559, 11093, 38577, 19669, 23696, 9534,  23697, 36878, 19701, 19751, 9538,    /*[GB+FD80, GB+FD8F]*/
  16204, 5302,  19433, 5303,  10191, 26249, 1747,  4462,  11165, 15994, 35127, 39958, 0,     0,     0,     0,       /*[GB+FD90, GB+FD9F]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+FDA0, GB+FDAF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+FDB0, GB+FDBF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+FDC0, GB+FDCF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+FDD0, GB+FDDF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+FDE0, GB+FDEF]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,              /*[GB+FDF0, GB+FDFE]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+FE40, GB+FE4F]*/
  3130,  0,     0,     0,     36560, 42766, 42741, 5698,  14715, 0,     32282, 10728, 10722, 34913, 35415, 42771,   /*[GB+FE50, GB+FE5F]*/
  42745, 0,     9059,  16783, 29726, 29704, 0,     0,     9064,  16814, 30357, 23195, 0,     0,     0,     19773,   /*[GB+FE60, GB+FE6F]*/
  2427,  28863, 4013,  0,     27037, 27077, 0,     531,   522,   39976, 43085, 26161, 17103, 17092, 0,              /*[GB+FE70, GB+FE7E]*/
  35337, 37019, 15125, 26065, 0,     28155, 37495, 24509, 28156, 43308, 28162, 16002, 33052, 41974, 4335,  41975,   /*[GB+FE80, GB+FE8F]*/
  0,     0,     39598, 38725, 4522,  26243, 39597, 30984, 28720, 14306, 19070, 15032, 15509, 31013, 24212, 36776,   /*[GB+FE90, GB+FE9F]*/
};


static const uint16 gb18030_2022_4_pinyin_weight_py_p1[] = {
  39337, 36563,   /*[GB+8138FD38, GB+8138FD39]*/
  38684, 27006, 15145, 0,     5711,  2181,  34914, 33503, 32483, 32484,   /*[GB+8138FE30, GB+8138FE39]*/
  33504, 29660, 37251, 13548, 13325, 35414, 29063, 24653, 0,     27110,   /*[GB+81398130, GB+81398139]*/
  27097, 40197, 5337,  0,     21858, 29427, 29401, 1691,  41466, 41455,   /*[GB+81398230, GB+81398239]*/
  25614, 26728, 0,     22297, 28856, 29000, 29582, 29583, 9095,  32543,   /*[GB+81398330, GB+81398339]*/
  32542, 0,     0,     17782, 39975, 27274, 15519, 2548,  2550,  2551,    /*[GB+81398430, GB+81398439]*/
  11789, 37742, 33784, 33783, 14127, 0,     0,     36752, 1009,  3253,    /*[GB+81398530, GB+81398539]*/
  4585,  4587,  4626,  8695,  14841, 3111,  3112,  41398, 21236, 0,       /*[GB+81398630, GB+81398639]*/
  8696,  39621, 26048, 32711, 37663, 8289,  7947,  28811, 0,     28812,   /*[GB+81398730, GB+81398739]*/
  28813, 0,     20660, 9764,  10264, 39609, 22973, 20016, 20753, 12336,   /*[GB+81398830, GB+81398839]*/
  21687, 25011, 25012, 3792,  3793,  19729, 19730, 10207, 10206, 0,       /*[GB+81398930, GB+81398939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+81398A30, GB+81398A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+81398B30, GB+81398B39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+81398C30, GB+81398C39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+81398D30, GB+81398D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+81398E30, GB+81398E39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+81398F30, GB+81398F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+81399030, GB+81399039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+81399130, GB+81399139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+81399230, GB+81399239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+81399330, GB+81399339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+81399430, GB+81399439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+81399530, GB+81399539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+81399630, GB+81399639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+81399730, GB+81399739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+81399830, GB+81399839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+81399930, GB+81399939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+81399A30, GB+81399A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+81399B30, GB+81399B39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+81399C30, GB+81399C39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+81399D30, GB+81399D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+81399E30, GB+81399E39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+81399F30, GB+81399F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139A030, GB+8139A039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139A130, GB+8139A139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139A230, GB+8139A239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139A330, GB+8139A339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139A430, GB+8139A439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139A530, GB+8139A539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139A630, GB+8139A639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139A730, GB+8139A739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139A830, GB+8139A839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139A930, GB+8139A939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139AA30, GB+8139AA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139AB30, GB+8139AB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139AC30, GB+8139AC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139AD30, GB+8139AD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139AE30, GB+8139AE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139AF30, GB+8139AF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139B030, GB+8139B039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139B130, GB+8139B139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139B230, GB+8139B239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139B330, GB+8139B339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139B430, GB+8139B439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139B530, GB+8139B539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139B630, GB+8139B639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139B730, GB+8139B739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139B830, GB+8139B839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139B930, GB+8139B939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139BA30, GB+8139BA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139BB30, GB+8139BB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139BC30, GB+8139BC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139BD30, GB+8139BD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139BE30, GB+8139BE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139BF30, GB+8139BF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139C030, GB+8139C039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139C130, GB+8139C139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139C230, GB+8139C239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139C330, GB+8139C339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139C430, GB+8139C439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139C530, GB+8139C539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139C630, GB+8139C639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139C730, GB+8139C739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139C830, GB+8139C839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139C930, GB+8139C939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139CA30, GB+8139CA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139CB30, GB+8139CB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139CC30, GB+8139CC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139CD30, GB+8139CD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139CE30, GB+8139CE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139CF30, GB+8139CF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139D030, GB+8139D039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139D130, GB+8139D139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139D230, GB+8139D239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139D330, GB+8139D339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139D430, GB+8139D439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139D530, GB+8139D539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139D630, GB+8139D639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139D730, GB+8139D739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139D830, GB+8139D839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139D930, GB+8139D939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139DA30, GB+8139DA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139DB30, GB+8139DB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139DC30, GB+8139DC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139DD30, GB+8139DD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139DE30, GB+8139DE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139DF30, GB+8139DF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139E030, GB+8139E039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139E130, GB+8139E139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139E230, GB+8139E239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139E330, GB+8139E339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139E430, GB+8139E439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139E530, GB+8139E539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139E630, GB+8139E639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139E730, GB+8139E739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139E830, GB+8139E839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139E930, GB+8139E939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139EA30, GB+8139EA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139EB30, GB+8139EB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139EC30, GB+8139EC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139ED30, GB+8139ED39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     26202,   /*[GB+8139EE30, GB+8139EE39]*/
  31305, 0,     0,     16773, 33336, 38623, 0,     0,     0,     0,       /*[GB+8139EF30, GB+8139EF39]*/
  0,     37744, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+8139F030, GB+8139F039]*/
  0,     35018, 0,     0,     0,     0,     0,     4030,  0,     0,       /*[GB+8139F130, GB+8139F139]*/
  0,     0,     23404, 0,     0,     5440,  0,     0,     0,     35829,   /*[GB+8139F230, GB+8139F239]*/
  35464, 0,     35534, 19463, 19176, 34626, 39035, 35370, 41745, 5339,    /*[GB+8139F330, GB+8139F339]*/
  33406, 23585, 27277, 0,     20642, 25478, 38030, 38538, 22668, 3611,    /*[GB+8139F430, GB+8139F439]*/
  8217,  0,     0,     0,     43286, 7886,  354,   33342, 44049, 0,       /*[GB+8139F530, GB+8139F539]*/
  6599,  29862, 38071, 26124, 16859, 17956, 22583, 42976, 29071, 0,       /*[GB+8139F630, GB+8139F639]*/
  0,     0,     35785, 0,     0,     28409, 14744, 6339,  23379, 29865,   /*[GB+8139F730, GB+8139F739]*/
  38103, 19756, 38969, 1091,  0,     0,     0,     17443, 21709, 38136,   /*[GB+8139F830, GB+8139F839]*/
  18317, 13372, 39632, 20457, 2777,  0,     0,     0,     12706, 35794,   /*[GB+8139F930, GB+8139F939]*/
  12541, 26963, 0,     0,     10768, 33948, 30521, 37276, 12547, 16065,   /*[GB+8139FA30, GB+8139FA39]*/
  20672, 20374, 30723, 37285, 41486, 41101, 39657, 43361, 7623,  26909,   /*[GB+8139FB30, GB+8139FB39]*/
  25050, 3850,  33375, 10784, 30793, 27774, 29525, 26165, 17838, 27544,   /*[GB+8139FC30, GB+8139FC39]*/
  0,     0,     17053, 24670, 30401, 29146, 37062, 23495, 30489, 0,       /*[GB+8139FD30, GB+8139FD39]*/
  21627, 38719, 6314,  39674, 21777, 16094, 22962, 35261, 39272, 0,       /*[GB+8139FE30, GB+8139FE39]*/
  0,     3310,  8272,  17934, 18529, 0,     20543, 0,     13331, 0,       /*[GB+82308130, GB+82308139]*/
  0,     0,     0,     26628, 0,     2316,  18826, 9752,  21011, 0,       /*[GB+82308230, GB+82308239]*/
  9910,  30077, 0,     0,     21034, 20759, 26615, 28940, 18171, 0,       /*[GB+82308330, GB+82308339]*/
  32498, 16632, 6878,  41842, 31431, 0,     0,     1912,  12805, 6603,    /*[GB+82308430, GB+82308439]*/
  9575,  3451,  0,     25876, 15172, 20030, 35511, 0,     22461, 35182,   /*[GB+82308530, GB+82308539]*/
  0,     1332,  14641, 29877, 0,     9474,  0,     39339, 35458, 25205,   /*[GB+82308630, GB+82308639]*/
  24045, 6172,  8646,  20548, 25208, 25209, 30690, 545,   8949,  4655,    /*[GB+82308730, GB+82308739]*/
  35964, 17656, 0,     0,     28317, 6241,  18031, 11978, 31667, 24216,   /*[GB+82308830, GB+82308839]*/
  5893,  27388, 7382,  25822, 38195, 43297, 27448, 13751, 0,     3842,    /*[GB+82308930, GB+82308939]*/
  3965,  33645, 0,     20378, 5824,  19126, 15895, 29915, 34962, 40629,   /*[GB+82308A30, GB+82308A39]*/
  0,     0,     42947, 41251, 13803, 43789, 4314,  35254, 18444, 0,       /*[GB+82308B30, GB+82308B39]*/
  3894,  33831, 13862, 0,     13105, 0,     7987,  4238,  1058,  14564,   /*[GB+82308C30, GB+82308C39]*/
  0,     473,   18830, 16806, 0,     34054, 1719,  15897, 17852, 35404,   /*[GB+82308D30, GB+82308D39]*/
  594,   37174, 20289, 950,   7395,  19977, 0,     0,     3277,  23392,   /*[GB+82308E30, GB+82308E39]*/
  36033, 11256, 39641, 0,     10250, 38239, 36075, 9592,  19862, 31000,   /*[GB+82308F30, GB+82308F39]*/
  17808, 28859, 0,     30184, 37412, 34197, 43785, 0,     26753, 38539,   /*[GB+82309030, GB+82309039]*/
  33512, 42167, 13616, 11805, 17242, 37950, 16445, 8376,  25912, 76,      /*[GB+82309130, GB+82309139]*/
  0,     16466, 4168,  35133, 4177,  32597, 0,     0,     12163, 29868,   /*[GB+82309230, GB+82309239]*/
  39367, 0,     16087, 41453, 35818, 28819, 0,     29289, 17062, 29343,   /*[GB+82309330, GB+82309339]*/
  11121, 8921,  36753, 26251, 28393, 11969, 33514, 7799,  23704, 5479,    /*[GB+82309430, GB+82309439]*/
  7887,  9479,  294,   8640,  22639, 36203, 39188, 11918, 0,     3348,    /*[GB+82309530, GB+82309539]*/
  10435, 22299, 11971, 18285, 7639,  34761, 24620, 0,     29599, 0,       /*[GB+82309630, GB+82309639]*/
  0,     17820, 19201, 38104, 11531, 0,     35839, 26464, 7541,  0,       /*[GB+82309730, GB+82309739]*/
  0,     36289, 0,     0,     0,     0,     23000, 32735, 35161, 31019,   /*[GB+82309830, GB+82309839]*/
  11373, 32029, 23001, 23002, 38554, 41682, 0,     0,     0,     0,       /*[GB+82309930, GB+82309939]*/
  0,     29035, 23405, 37554, 24868, 31700, 10664, 16059, 6582,  12683,   /*[GB+82309A30, GB+82309A39]*/
  20036, 15365, 12820, 19281, 0,     31269, 20412, 0,     0,     0,       /*[GB+82309B30, GB+82309B39]*/
  0,     0,     0,     9274,  36585, 28780, 36174, 23841, 4566,  23200,   /*[GB+82309C30, GB+82309C39]*/
  7181,  40803, 7397,  35050, 39151, 7398,  28617, 33043, 16686, 11671,   /*[GB+82309D30, GB+82309D39]*/
  9276,  34107, 20829, 20372, 13160, 11485, 42351, 0,     0,     32284,   /*[GB+82309E30, GB+82309E39]*/
  0,     601,   87,    43211, 25298, 9663,  5542,  896,   2055,  4142,    /*[GB+82309F30, GB+82309F39]*/
  18386, 34960, 35662, 0,     0,     0,     0,     0,     11413, 31151,   /*[GB+8230A030, GB+8230A039]*/
  4947,  17218, 17718, 42406, 35003, 33647, 0,     25838, 40958, 33648,   /*[GB+8230A130, GB+8230A139]*/
  0,     0,     4860,  13201, 12851, 30340, 36693, 35921, 24513, 27593,   /*[GB+8230A230, GB+8230A239]*/
  0,     0,     0,     10422, 37603, 34694, 36154, 11133, 44078, 38328,   /*[GB+8230A330, GB+8230A339]*/
  4690,  0,     17999, 34371, 30505, 27170, 38360, 42496, 33727, 34378,   /*[GB+8230A430, GB+8230A439]*/
  15660, 13249, 0,     23769, 18485, 0,     17494, 27596, 10726, 36727,   /*[GB+8230A530, GB+8230A539]*/
  26452, 0,     36739, 16205, 3791,  22988, 12800, 0,     1238,  34077,   /*[GB+8230A630, GB+8230A639]*/
  33098, 36037, 32316, 39273, 25901, 35821, 22993, 1229,  10947, 15046,   /*[GB+8230A730, GB+8230A739]*/
  374,   375,   0,     0,     41671, 30532, 15428, 0,     44052, 2234,    /*[GB+8230A830, GB+8230A839]*/
  14572, 75,    40690, 4657,  7641,  0,     0,     0,     0,     23003,   /*[GB+8230A930, GB+8230A939]*/
  19588, 21096, 7004,  697,   1275,  805,   0,     4260,  34174, 31270,   /*[GB+8230AA30, GB+8230AA39]*/
  3079,  0,     0,     7123,  32573, 8739,  7182,  39648, 37515, 16997,   /*[GB+8230AB30, GB+8230AB39]*/
  32753, 16809, 0,     32574, 37234, 19740, 35425, 2214,  3710,  35056,   /*[GB+8230AC30, GB+8230AC39]*/
  23030, 17663, 37705, 43656, 20763, 41419, 34181, 10386, 35068, 0,       /*[GB+8230AD30, GB+8230AD39]*/
  13457, 18897, 38298, 13202, 38594, 0,     5229,  38299, 35237, 10988,   /*[GB+8230AE30, GB+8230AE39]*/
  39118, 16231, 3016,  30493, 30755, 42140, 871,   21294, 17023, 2879,    /*[GB+8230AF30, GB+8230AF39]*/
  17922, 0,     34021, 0,     33750, 25789, 22535, 40320, 0,     19625,   /*[GB+8230B030, GB+8230B039]*/
  8716,  43629, 0,     9756,  16127, 6256,  11996, 17057, 0,     9151,    /*[GB+8230B130, GB+8230B139]*/
  30924, 0,     28055, 17389, 22994, 8460,  9152,  25797, 667,   13557,   /*[GB+8230B230, GB+8230B239]*/
  16572, 33925, 39743, 43207, 28497, 4597,  34807, 13314, 23315, 34888,   /*[GB+8230B330, GB+8230B339]*/
  38032, 39419, 37750, 36756, 28484, 26891, 10948, 27529, 16014, 39190,   /*[GB+8230B430, GB+8230B439]*/
  0,     35361, 23775, 26208, 2811,  0,     2235,  6534,  29600, 7581,    /*[GB+8230B530, GB+8230B539]*/
  0,     20999, 40428, 12934, 0,     25742, 35540, 23724, 4153,  23858,   /*[GB+8230B630, GB+8230B639]*/
  23395, 14573, 37682, 7612,  7220,  0,     0,     0,     7173,  0,       /*[GB+8230B730, GB+8230B739]*/
  0,     25815, 20215, 26282, 29803, 2442,  6692,  33555, 8232,  38141,   /*[GB+8230B830, GB+8230B839]*/
  30202, 25792, 24560, 35329, 31639, 35383, 39211, 944,   19761, 0,       /*[GB+8230B930, GB+8230B939]*/
  0,     0,     0,     40359, 18032, 30365, 17535, 20809, 25616, 42720,   /*[GB+8230BA30, GB+8230BA39]*/
  36916, 33575, 20038, 33576, 27734, 7832,  0,     32754, 7676,  38197,   /*[GB+8230BB30, GB+8230BB39]*/
  22598, 3464,  30671, 12970, 29234, 24234, 165,   16757, 2662,  0,       /*[GB+8230BC30, GB+8230BC39]*/
  34330, 42355, 0,     0,     8247,  18752, 36314, 35893, 21559, 12563,   /*[GB+8230BD30, GB+8230BD39]*/
  22272, 39007, 41238, 38240, 23299, 30726, 33626, 40370, 29244, 8557,    /*[GB+8230BE30, GB+8230BE39]*/
  38241, 5276,  0,     18632, 2513,  2373,  15620, 20075, 29916, 22683,   /*[GB+8230BF30, GB+8230BF39]*/
  390,   212,   25503, 0,     5025,  4790,  0,     26870, 22894, 20703,   /*[GB+8230C030, GB+8230C039]*/
  35364, 40264, 22402, 400,   28459, 20686, 0,     0,     17589, 33712,   /*[GB+8230C130, GB+8230C139]*/
  40276, 42498, 33112, 12030, 21398, 22964, 32428, 21467, 23088, 26531,   /*[GB+8230C230, GB+8230C239]*/
  40655, 18791, 42159, 43549, 10545, 35822, 10951, 35965, 42293, 21660,   /*[GB+8230C330, GB+8230C339]*/
  4539,  9665,  0,     4553,  20338, 42966, 29015, 18946, 15280, 35129,   /*[GB+8230C430, GB+8230C439]*/
  6495,  14731, 27113, 20871, 0,     16450, 37362, 23137, 37784, 17611,   /*[GB+8230C530, GB+8230C539]*/
  39060, 38556, 36651, 29879, 0,     19116, 36428, 20960, 21916, 43945,   /*[GB+8230C630, GB+8230C639]*/
  39649, 38198, 9666,  21515, 16075, 33047, 0,     16274, 6201,  19654,   /*[GB+8230C730, GB+8230C739]*/
  0,     35498, 5092,  25715, 21634, 21400, 25987, 0,     32341, 5790,    /*[GB+8230C830, GB+8230C839]*/
  77,    0,     1571,  23283, 18633, 14865, 39392, 4464,  44042, 2164,    /*[GB+8230C930, GB+8230C939]*/
  12380, 37414, 31947, 13391, 159,   20562, 13298, 32755, 1966,  40495,   /*[GB+8230CA30, GB+8230CA39]*/
  35894, 22877, 40426, 0,     496,   41534, 15346, 32725, 35155, 25088,   /*[GB+8230CB30, GB+8230CB39]*/
  37785, 35156, 4658,  26283, 6744,  22976, 25116, 13304, 31901, 0,       /*[GB+8230CC30, GB+8230CC39]*/
  29700, 6203,  17722, 41269, 0,     0,     38536, 2603,  13280, 12511,   /*[GB+8230CD30, GB+8230CD39]*/
  43551, 17435, 22544, 15563, 25991, 5341,  0,     14551, 35783, 4765,    /*[GB+8230CE30, GB+8230CE39]*/
  39127, 6671,  3683,  0,     21862, 12256, 30073, 16426, 43803, 10952,   /*[GB+8230CF30, GB+8230CF39]*/
  3510,  36206, 22698, 3806,  18606, 223,   22215, 29495, 34655, 37086,   /*[GB+8230D030, GB+8230D039]*/
  11923, 5191,  26284, 17704, 8500,  7005,  20886, 17612, 32168, 10661,   /*[GB+8230D130, GB+8230D139]*/
  20918, 2020,  26827, 24856, 10662, 0,     19763, 1793,  31344, 40794,   /*[GB+8230D230, GB+8230D239]*/
  24871, 40626, 21421, 23794, 41292, 34717, 9099,  0,     24872, 0,       /*[GB+8230D330, GB+8230D339]*/
  20039, 2601,  40443, 7401,  6914,  21829, 32576, 26659, 29809, 21830,   /*[GB+8230D430, GB+8230D439]*/
  31728, 0,     21951, 37381, 15885, 18388, 16812, 9103,  40077, 5319,    /*[GB+8230D530, GB+8230D539]*/
  0,     17719, 19802, 25504, 323,   1697,  39014, 20926, 5727,  0,       /*[GB+8230D630, GB+8230D639]*/
  324,   0,     33809, 8591,  5458,  15315, 27476, 31612, 26426, 7446,    /*[GB+8230D730, GB+8230D739]*/
  24784, 13203, 13204, 11943, 14464, 43962, 1699,  21287, 592,   32804,   /*[GB+8230D830, GB+8230D839]*/
  37990, 401,   39685, 10930, 7025,  33207, 22844, 4996,  0,     18135,   /*[GB+8230D930, GB+8230D939]*/
  19931, 22965, 12031, 18489, 0,     20298, 8276,  21524, 39961, 0,       /*[GB+8230DA30, GB+8230DA39]*/
  15558, 0,     0,     41234, 23884, 37938, 0,     13392, 1175,  0,       /*[GB+8230DB30, GB+8230DB39]*/
  27034, 12229, 7715,  9257,  16723, 14734, 27852, 0,     29491, 31559,   /*[GB+8230DC30, GB+8230DC39]*/
  39979, 43433, 1240,  16768, 18297, 12237, 36291, 23396, 0,     41549,   /*[GB+8230DD30, GB+8230DD39]*/
  33062, 34317, 25220, 37489, 21035, 0,     28012, 29236, 0,     25651,   /*[GB+8230DE30, GB+8230DE39]*/
  43271, 17133, 33453, 38754, 4424,  31042, 18622, 1145,  9612,  20901,   /*[GB+8230DF30, GB+8230DF39]*/
  35207, 8331,  19803, 40705, 41976, 4181,  20847, 19665, 0,     38726,   /*[GB+8230E030, GB+8230E039]*/
  24353, 41986, 13837, 20327, 22986, 38202, 0,     13436, 13221, 41121,   /*[GB+8230E130, GB+8230E139]*/
  39616, 15259, 12084, 42175, 17245, 19253, 42182, 1024,  41063, 15347,   /*[GB+8230E230, GB+8230E239]*/
  5516,  18970, 38110, 41473, 34497, 3813,  4732,  3754,  36771, 17613,   /*[GB+8230E330, GB+8230E339]*/
  6693,  19764, 2839,  0,     31908, 2695,  53,    3765,  0,     38929,   /*[GB+8230E430, GB+8230E439]*/
  41566, 31669, 0,     31913, 2709,  37386, 43710, 0,     23589, 25774,   /*[GB+8230E530, GB+8230E539]*/
  18642, 25943, 19991, 36970, 16318, 29844, 38419, 2809,  15167, 14185,   /*[GB+8230E630, GB+8230E639]*/
  0,     15129, 0,     6607,  0,     15729, 10736, 6017,  0,     0,       /*[GB+8230E730, GB+8230E739]*/
  11344, 0,     3688,  6235,  1242,  0,     36366, 19899, 0,     35027,   /*[GB+8230E830, GB+8230E839]*/
  1258,  0,     1276,  0,     34318, 27446, 1779,  7590,  15751, 0,       /*[GB+8230E930, GB+8230E939]*/
  41832, 971,   7423,  39667, 26497, 40631, 21473, 38081, 29650, 0,       /*[GB+8230EA30, GB+8230EA39]*/
  0,     0,     28097, 30472, 22263, 15109, 1576,  27146, 2641,  2454,    /*[GB+8230EB30, GB+8230EB39]*/
  6456,  0,     0,     0,     0,     5920,  31637, 30355, 35470, 29666,   /*[GB+8230EC30, GB+8230EC39]*/
  7137,  33929, 30826, 0,     31020, 28102, 14018, 42319, 32567, 38695,   /*[GB+8230ED30, GB+8230ED39]*/
  0,     0,     12153, 42600, 25117, 43643, 0,     35200, 35208, 40814,   /*[GB+8230EE30, GB+8230EE39]*/
  32660, 0,     0,     30409, 41189, 23190, 0,     0,     35314, 38038,   /*[GB+8230EF30, GB+8230EF39]*/
  27008, 29204, 2737,  43313, 0,     21654, 13119, 7875,  23816, 74,      /*[GB+8230F030, GB+8230F039]*/
  7803,  356,   25994, 25188, 34938, 8070,  8940,  25639, 9214,  31562,   /*[GB+8230F130, GB+8230F139]*/
  2812,  39348, 9113,  1034,  8710,  4242,  42978, 0,     0,     10855,   /*[GB+8230F230, GB+8230F239]*/
  23123, 15828, 3910,  2745,  16600, 19019, 18298, 39720, 0,     39442,   /*[GB+8230F330, GB+8230F339]*/
  10580, 18299, 11480, 9546,  16456, 40152, 5793,  12525, 14253, 10126,   /*[GB+8230F430, GB+8230F439]*/
  15175, 44069, 8723,  25823, 913,   3273,  4669,  20888, 10615, 33935,   /*[GB+8230F530, GB+8230F539]*/
  26286, 12341, 0,     0,     4034,  27657, 36576, 42105, 5800,  30935,   /*[GB+8230F630, GB+8230F639]*/
  21240, 19284, 29036, 31963, 2395,  6371,  3278,  23897, 37698, 15446,   /*[GB+8230F730, GB+8230F739]*/
  13395, 17356, 31272, 40159, 0,     2335,  24752, 39747, 18616, 4780,    /*[GB+8230F830, GB+8230F839]*/
  0,     0,     0,     39482, 13164, 32868, 21510, 30088, 35051, 35706,   /*[GB+8230F930, GB+8230F939]*/
  3831,  26313, 12567, 0,     39483, 25834, 29432, 29414, 7184,  19789,   /*[GB+8230FA30, GB+8230FA39]*/
  0,     23689, 30523, 38648, 27709, 7970,  3331,  40078, 37852, 12743,   /*[GB+8230FB30, GB+8230FB39]*/
  27775, 37578, 21889, 8174,  11105, 0,     38709, 2584,  22828, 397,     /*[GB+8230FC30, GB+8230FC39]*/
  8304,  18643, 3095,  2960,  20625, 6332,  11616, 20078, 99,    38300,   /*[GB+8230FD30, GB+8230FD39]*/
  11944, 40964, 11629, 7447,  12852, 30189, 22831, 34557, 18097, 34558,   /*[GB+8230FE30, GB+8230FE39]*/
  36971, 19662, 21245, 14825, 13045, 0,     1545,  39688, 12866, 21738,   /*[GB+82318130, GB+82318139]*/
  4084,  20707, 0,     17828, 14685, 32937, 38420, 36017, 34022, 2435,    /*[GB+82318230, GB+82318239]*/
  17496, 38675, 35290, 40506, 20542, 19410, 25370, 12803, 13723, 33148,   /*[GB+82318330, GB+82318339]*/
  0,     0,     9266,  42818, 6353,  39093, 13308, 37115, 27361, 33815,   /*[GB+82318430, GB+82318439]*/
  29387, 39882, 37756, 25438, 13120, 26587, 31214, 29007, 25436, 22245,   /*[GB+82318530, GB+82318539]*/
  14792, 20997, 38624, 8901,  24488, 36067, 21012, 7889,  36451, 12376,   /*[GB+82318630, GB+82318639]*/
  39714, 9870,  9954,  19563, 7359,  43558, 43595, 1246,  32252, 17531,   /*[GB+82318730, GB+82318739]*/
  19004, 0,     0,     0,     10581, 38478, 42813, 3990,  34406, 36093,   /*[GB+82318830, GB+82318839]*/
  0,     26273, 23823, 10223, 7543,  9549,  26130, 11584, 17756, 18300,   /*[GB+82318930, GB+82318939]*/
  3428,  27626, 43426, 33151, 24609, 16538, 32036, 23865, 30936, 30368,   /*[GB+82318A30, GB+82318A39]*/
  43338, 1709,  9847,  11593, 0,     1887,  42322, 6583,  6984,  42668,   /*[GB+82318B30, GB+82318B39]*/
  22672, 19178, 24521, 13300, 21831, 32759, 3257,  9668,  706,   27296,   /*[GB+82318C30, GB+82318C39]*/
  30546, 2210,  43645, 16966, 17711, 10770, 38857, 42360, 14609, 35496,   /*[GB+82318D30, GB+82318D39]*/
  35059, 36321, 28076, 25399, 34999, 29904, 10544, 21563, 12711, 24011,   /*[GB+82318E30, GB+82318E39]*/
  0,     12568, 22373, 29714, 1045,  4016,  14623, 12363, 17541, 0,       /*[GB+82318F30, GB+82318F39]*/
  11861, 6652,  12846, 10361, 37300, 2585,  10256, 14048, 13917, 5728,    /*[GB+82319030, GB+82319039]*/
  14931, 20678, 12581, 21678, 2407,  20381, 24180, 37197, 15623, 15624,   /*[GB+82319130, GB+82319139]*/
  26785, 0,     25319, 27965, 0,     15317, 12853, 40410, 5282,  35997,   /*[GB+82319230, GB+82319239]*/
  34832, 8033,  2591,  37591, 0,     5820,  0,     25945, 12495, 32023,   /*[GB+82319330, GB+82319339]*/
  0,     25605, 33819, 22771, 27586, 21296, 31873, 17554, 10931, 4755,    /*[GB+82319430, GB+82319439]*/
  41149, 301,   20535, 21784, 0,     8432,  0,     35104, 2120,  12644,   /*[GB+82319530, GB+82319539]*/
  26089, 35105, 0,     0,     2132,  25427, 24532, 14420, 15973, 17186,   /*[GB+82319630, GB+82319639]*/
  15511, 7502,  23106, 25359, 6436,  6437,  0,     24745, 42026, 24876,   /*[GB+82319730, GB+82319739]*/
  43234, 16668, 39425, 25907, 16673, 11048, 8480,  9450,  5993,  34513,   /*[GB+82319830, GB+82319839]*/
  10277, 11063, 26836, 10759, 31641, 2032,  28071, 1199,  20051, 37561,   /*[GB+82319930, GB+82319939]*/
  22717, 4327,  27659, 6304,  20052, 31689, 18719, 16434, 27660, 41778,   /*[GB+82319A30, GB+82319A39]*/
  4330,  18758, 21052, 0,     25312, 16183, 28273, 34818, 1385,  40965,   /*[GB+82319B30, GB+82319B39]*/
  38716, 33691, 28135, 29941, 27553, 27462, 4582,  19939, 19381, 2685,    /*[GB+82319C30, GB+82319C39]*/
  0,     12198, 0,     0,     13622, 672,   11654, 6680,  0,     19852,   /*[GB+82319D30, GB+82319D39]*/
  15411, 15786, 16428, 30238, 20557, 41551, 6478,  6944,  43038, 36758,   /*[GB+82319E30, GB+82319E39]*/
  23680, 2694,  0,     0,     0,     0,     37966, 0,     0,     39228,   /*[GB+82319F30, GB+82319F39]*/
  12426, 37393, 37351, 42169, 9543,  25023, 9382,  0,     0,     11539,   /*[GB+8231A030, GB+8231A039]*/
  21536, 8467,  11578, 10122, 30636, 5888,  0,     36643, 0,     0,       /*[GB+8231A130, GB+8231A139]*/
  26588, 0,     3116,  21947, 30831, 846,   156,   0,     0,     34412,   /*[GB+8231A230, GB+8231A239]*/
  0,     0,     0,     21036, 17676, 22484, 951,   3369,  0,     7925,    /*[GB+8231A330, GB+8231A339]*/
  42732, 12974, 14509, 29237, 0,     17191, 6374,  20053, 0,     0,       /*[GB+8231A430, GB+8231A439]*/
  0,     0,     39507, 30481, 3014,  20830, 21887, 12199, 32968, 23362,   /*[GB+8231A530, GB+8231A539]*/
  12200, 11495, 15115, 2067,  34435, 18403, 14932, 0,     20928, 24321,   /*[GB+8231A630, GB+8231A639]*/
  10920, 37126, 0,     34567, 29942, 32805, 3298,  33692, 14950, 2631,    /*[GB+8231A730, GB+8231A739]*/
  11197, 8099,  27980, 19357, 0,     7031,  24807, 24730, 40277, 2121,    /*[GB+8231A830, GB+8231A839]*/
  0,     12653, 6425,  37007, 15679, 14483, 22508, 19104, 39420, 31162,   /*[GB+8231A930, GB+8231A939]*/
  31204, 33350, 11438, 34892, 10965, 0,     31327, 41901, 0,     12252,   /*[GB+8231AA30, GB+8231AA39]*/
  8784,  0,     0,     31990, 0,     27083, 14326, 0,     35377, 0,       /*[GB+8231AB30, GB+8231AB39]*/
  0,     40146, 15823, 11919, 0,     730,   22161, 0,     32727, 0,       /*[GB+8231AC30, GB+8231AC39]*/
  21178, 29611, 1569,  19902, 26384, 0,     0,     9270,  41552, 20220,   /*[GB+8231AD30, GB+8231AD39]*/
  23572, 27123, 26287, 19021, 9551,  34413, 33937, 35332, 0,     22943,   /*[GB+8231AE30, GB+8231AE39]*/
  0,     0,     0,     35047, 19032, 8393,  5158,  43340, 452,   44070,   /*[GB+8231AF30, GB+8231AF39]*/
  41561, 43913, 11078, 13151, 0,     13740, 0,     0,     0,     31771,   /*[GB+8231B030, GB+8231B039]*/
  34331, 36802, 30721, 30374, 6003,  15874, 273,   10676, 34902, 15454,   /*[GB+8231B130, GB+8231B139]*/
  32580, 720,   43209, 23023, 31311, 22433, 0,     0,     39312, 21620,   /*[GB+8231B230, GB+8231B239]*/
  0,     0,     22437, 28621, 2667,  36589, 9388,  3997,  27392, 13646,   /*[GB+8231B330, GB+8231B339]*/
  25930, 20967, 7426,  18391, 3712,  40672, 11107, 14626, 22891, 0,       /*[GB+8231B430, GB+8231B439]*/
  10043, 11496, 8916,  0,     1046,  30263, 33255, 13438, 33650, 26166,   /*[GB+8231B530, GB+8231B539]*/
  11123, 33084, 34346, 14644, 12716, 24091, 28451, 4017,  41836, 0,       /*[GB+8231B630, GB+8231B639]*/
  41181, 29463, 13002, 29739, 42227, 1031,  0,     0,     0,     17669,   /*[GB+8231B730, GB+8231B739]*/
  1386,  36124, 23805, 5395,  24788, 42065, 24096, 2967,  1387,  29931,   /*[GB+8231B830, GB+8231B839]*/
  12854, 11239, 15193, 4366,  14188, 22684, 9807,  7905,  0,     0,       /*[GB+8231B930, GB+8231B939]*/
  30410, 5098,  33695, 5811,  34361, 16851, 41594, 30328, 11721, 5099,    /*[GB+8231BA30, GB+8231BA39]*/
  20115, 15773, 20116, 25511, 23765, 41853, 0,     18429, 2523,  24948,   /*[GB+8231BB30, GB+8231BB39]*/
  0,     0,     31163, 19358, 26513, 18727, 19995, 29154, 9589,  41602,   /*[GB+8231BC30, GB+8231BC39]*/
  23717, 14958, 26068, 0,     0,     43677, 24687, 14874, 1705,  14091,   /*[GB+8231BD30, GB+8231BD39]*/
  10376, 0,     1820,  40717, 19081, 18150, 20536, 28522, 21637, 14098,   /*[GB+8231BE30, GB+8231BE39]*/
  5968,  1004,  0,     18731, 0,     34388, 24382, 26815, 19690, 43972,   /*[GB+8231BF30, GB+8231BF39]*/
  0,     15967, 28049, 36188, 0,     35294, 0,     17560, 25002, 37925,   /*[GB+8231C030, GB+8231C039]*/
  23394, 18181, 40296, 0,     37946, 3621,  13349, 10842, 35146, 16533,   /*[GB+8231C130, GB+8231C139]*/
  43439, 11013, 33931, 26589, 10540, 34047, 10541, 10144, 2816,  36304,   /*[GB+8231C230, GB+8231C239]*/
  35695, 28534, 16640, 34055, 27929, 39405, 36537, 24623, 43812, 39306,   /*[GB+8231C330, GB+8231C339]*/
  43602, 18720, 34232, 34179, 37978, 27938, 36938, 14451, 33651, 3779,    /*[GB+8231C430, GB+8231C439]*/
  28956, 16280, 38653, 11230, 38339, 33713, 27807, 14959, 37623, 39170,   /*[GB+8231C530, GB+8231C539]*/
  26816, 37506, 20332, 17166, 41956, 0,     0,     0,     0,     34993,   /*[GB+8231C630, GB+8231C639]*/
  0,     5078,  35611, 255,   35642, 2389,  4375,  40986, 0,     38117,   /*[GB+8231C730, GB+8231C739]*/
  23985, 16678, 28582, 17624, 31950, 33601, 19304, 24764, 33186, 18754,   /*[GB+8231C830, GB+8231C839]*/
  6774,  21244, 17582, 32773, 6940,  39,    40568, 12582, 38278, 22102,   /*[GB+8231C930, GB+8231C939]*/
  43619, 8198,  23920, 0,     1426,  18511, 19942, 20604, 10536, 41753,   /*[GB+8231CA30, GB+8231CA39]*/
  8869,  26759, 41697, 16580, 3548,  15304, 15886, 13439, 19346, 0,       /*[GB+8231CB30, GB+8231CB39]*/
  28267, 26803, 27463, 4590,  22693, 42005, 19775, 23713, 0,     0,       /*[GB+8231CC30, GB+8231CC39]*/
  844,   27125, 34225, 17963, 34791, 8394,  26474, 0,     27875, 42216,   /*[GB+8231CD30, GB+8231CD39]*/
  30576, 27212, 29832, 38935, 20968, 22438, 1586,  0,     29301, 30736,   /*[GB+8231CE30, GB+8231CE39]*/
  10788, 27744, 27149, 0,     5829,  24674, 14294, 30654, 0,     26877,   /*[GB+8231CF30, GB+8231CF39]*/
  23166, 19082, 6440,  6355,  42618, 0,     20277, 5557,  33509, 10210,   /*[GB+8231D030, GB+8231D039]*/
  13104, 22795, 38024, 22914, 39620, 32500, 10518, 40866, 36628, 5061,    /*[GB+8231D130, GB+8231D139]*/
  34299, 14328, 31682, 8704,  23818, 0,     39144, 26209, 36415, 2242,    /*[GB+8231D230, GB+8231D239]*/
  1562,  28876, 40988, 38089, 1563,  0,     6996,  17439, 37685, 2792,    /*[GB+8231D330, GB+8231D339]*/
  3911,  36095, 35843, 39444, 35592, 0,     0,     0,     10401, 0,       /*[GB+8231D430, GB+8231D439]*/
  0,     0,     19766, 35166, 3274,  13881, 30538, 24163, 40607, 36038,   /*[GB+8231D530, GB+8231D539]*/
  34312, 22975, 0,     0,     0,     0,     0,     21548, 13400, 23298,   /*[GB+8231D630, GB+8231D639]*/
  11595, 11903, 32517, 39223, 40797, 1314,  21509, 25541, 35174, 7825,    /*[GB+8231D730, GB+8231D739]*/
  38171, 30542, 17965, 39094, 0,     14911, 28351, 38696, 13301, 0,       /*[GB+8231D830, GB+8231D839]*/
  29882, 0,     0,     22430, 32523, 21695, 29890, 38209, 27962, 33603,   /*[GB+8231D930, GB+8231D939]*/
  13171, 20566, 39154, 21046, 41036, 30089, 42364, 1579,  18055, 0,       /*[GB+8231DA30, GB+8231DA39]*/
  0,     0,     0,     0,     0,     0,     25767, 10042, 33633, 41833,   /*[GB+8231DB30, GB+8231DB39]*/
  39009, 23033, 16076, 35202, 37383, 35203, 42050, 22688, 0,     29519,   /*[GB+8231DC30, GB+8231DC39]*/
  19741, 3381,  21565, 26776, 5447,  28077, 0,     0,     0,     29919,   /*[GB+8231DD30, GB+8231DD39]*/
  35212, 2068,  6482,  43820, 0,     29254, 28328, 10790, 30551, 9164,    /*[GB+8231DE30, GB+8231DE39]*/
  0,     0,     0,     22385, 21580, 36333, 21247, 14061, 5047,  15911,   /*[GB+8231DF30, GB+8231DF39]*/
  11194, 8034,  28796, 3261,  28554, 23356, 24469, 20848, 0,     0,       /*[GB+8231E030, GB+8231E039]*/
  0,     0,     38342, 4072,  0,     16707, 791,   17851, 16438, 27954,   /*[GB+8231E130, GB+8231E139]*/
  1410,  30054, 9320,  24192, 38343, 34362, 22837, 38881, 42954, 4554,    /*[GB+8231E230, GB+8231E239]*/
  8308,  35938, 24305, 33396, 18915, 2491,  43791, 44012, 1601,  37445,   /*[GB+8231E330, GB+8231E339]*/
  12126, 23582, 35622, 0,     17926, 26113, 34975, 14309, 10472, 0,       /*[GB+8231E430, GB+8231E439]*/
  0,     36720, 36187, 42863, 11279, 38899, 33741, 0,     0,     18696,   /*[GB+8231E530, GB+8231E539]*/
  34458, 12140, 38533, 0,     18783, 28089, 2495,  1007,  13973, 29285,   /*[GB+8231E630, GB+8231E639]*/
  7853,  6223,  0,     533,   39601, 0,     0,     22530, 17951, 38464,   /*[GB+8231E730, GB+8231E739]*/
  5342,  0,     2837,  3205,  8935,  14890, 22677, 0,     0,     0,       /*[GB+8231E830, GB+8231E839]*/
  18952, 22047, 39294, 0,     19587, 10655, 0,     39130, 14903, 3758,    /*[GB+8231E930, GB+8231E939]*/
  27055, 23237, 0,     0,     11451, 31310, 0,     17,    9880,  1628,    /*[GB+8231EA30, GB+8231EA39]*/
  2035,  26136, 0,     29238, 4492,  12484, 3213,  8743,  12402, 7406,    /*[GB+8231EB30, GB+8231EB39]*/
  32871, 8129,  30577, 0,     20422, 11186, 39109, 12487, 0,     39511,   /*[GB+8231EC30, GB+8231EC39]*/
  43706, 36941, 26324, 41488, 15185, 30482, 0,     0,     0,     0,       /*[GB+8231ED30, GB+8231ED39]*/
  0,     0,     31970, 19129, 15191, 41040, 35432, 11863, 0,     35911,   /*[GB+8231EE30, GB+8231EE39]*/
  0,     0,     0,     5087,  26085, 22104, 0,     40710, 1122,  3657,    /*[GB+8231EF30, GB+8231EF39]*/
  0,     0,     36974, 9321,  22114, 996,   15722, 6407,  41506, 0,       /*[GB+8231F030, GB+8231F039]*/
  33315, 36975, 0,     15938, 34260, 30499, 10721, 0,     6167,  13500,   /*[GB+8231F130, GB+8231F139]*/
  14690, 16351, 43891, 0,     35278, 17425, 7769,  12895, 34032, 23101,   /*[GB+8231F230, GB+8231F239]*/
  21482, 26884, 5016,  38547, 21538, 0,     15901, 26386, 31577, 32440,   /*[GB+8231F330, GB+8231F339]*/
  41517, 18209, 28261, 16604, 34209, 41536, 42027, 31378, 29084, 981,     /*[GB+8231F430, GB+8231F439]*/
  37625, 24266, 3026,  11797, 16521, 15212, 152,   4527,  25373, 934,     /*[GB+8231F530, GB+8231F539]*/
  433,   8114,  16354, 32103, 32104, 44016, 19264, 0,     10233, 36570,   /*[GB+8231F630, GB+8231F639]*/
  28905, 11534, 19033, 27877, 29623, 0,     953,   27061, 6771,  2036,    /*[GB+8231F730, GB+8231F739]*/
  18803, 25452, 8016,  13443, 43707, 12411, 11110, 18062, 40090, 40253,   /*[GB+8231F830, GB+8231F839]*/
  35612, 2968,  5950,  17846, 14867, 3968,  29653, 24715, 37399, 14164,   /*[GB+8231F930, GB+8231F939]*/
  12065, 12217, 30863, 27366, 33114, 38909, 26961, 38548, 28869, 38542,   /*[GB+8231FA30, GB+8231FA39]*/
  15817, 32000, 36034, 13561, 42555, 25808, 42981, 6236,  0,     39350,   /*[GB+8231FB30, GB+8231FB39]*/
  0,     0,     37795, 28830, 38120, 22048, 0,     0,     26760, 34780,   /*[GB+8231FC30, GB+8231FC39]*/
  33290, 9395,  38925, 31469, 28832, 22706, 9398,  30371, 33134, 15449,   /*[GB+8231FD30, GB+8231FD39]*/
  2941,  24302, 43342, 11596, 22601, 36668, 9642,  39652, 11492, 0,       /*[GB+8231FE30, GB+8231FE39]*/
  29520, 3645,  11852, 37193, 33083, 34537, 24369, 27154, 19808, 17767,   /*[GB+82328130, GB+82328139]*/
  28031, 34821, 40816, 10598, 7744,  10719, 2825,  41312, 0,     30343,   /*[GB+82328230, GB+82328239]*/
  43045, 23252, 10817, 39561, 43394, 39384, 18495, 12884, 33761, 34278,   /*[GB+82328330, GB+82328339]*/
  2897,  18697, 0,     29489, 15281, 24655, 26259, 9545,  43556, 39431,   /*[GB+82328430, GB+82328439]*/
  0,     0,     27090, 23212, 21092, 434,   15252, 0,     35844, 24441,   /*[GB+82328530, GB+82328539]*/
  1570,  21023, 0,     0,     0,     0,     37816, 39459, 0,     24442,   /*[GB+82328630, GB+82328639]*/
  26390, 809,   12539, 0,     0,     0,     2257,  20893, 17272, 31767,   /*[GB+82328730, GB+82328739]*/
  33297, 18350, 19290, 0,     13410, 16067, 43758, 7185,  15876, 5381,    /*[GB+82328830, GB+82328839]*/
  973,   0,     0,     0,     0,     0,     17281, 1801,  30052, 31782,   /*[GB+82328930, GB+82328939]*/
  36151, 0,     0,     0,     0,     0,     7250,  0,     0,     30101,   /*[GB+82328A30, GB+82328A39]*/
  1392,  31739, 27788, 2452,  31800, 21679, 14820, 20240, 0,     0,       /*[GB+82328B30, GB+82328B39]*/
  41317, 1212,  13225, 40897, 36012, 18449, 0,     0,     30126, 39037,   /*[GB+82328C30, GB+82328C39]*/
  29183, 0,     0,     7312,  0,     0,     0,     0,     26196, 20498,   /*[GB+82328D30, GB+82328D39]*/
  41810, 32004, 9706,  39636, 17903, 2037,  22658, 24243, 18765, 30796,   /*[GB+82328E30, GB+82328E39]*/
  18646, 32976, 5611,  18590, 31464, 32253, 42746, 9071,  35472, 285,     /*[GB+82328F30, GB+82328F39]*/
  7818,  23958, 2024,  32110, 29094, 37827, 2038,  25826, 31686, 9556,    /*[GB+82329030, GB+82329039]*/
  31586, 10677, 3550,  14613, 12203, 35518, 6194,  2771,  6627,  24087,   /*[GB+82329130, GB+82329139]*/
  27395, 19052, 28623, 23492, 6078,  39525, 4364,  27157, 16281, 30747,   /*[GB+82329230, GB+82329239]*/
  4870,  24293, 4444,  20118, 31623, 41981, 18451, 27562, 23591, 29551,   /*[GB+82329330, GB+82329339]*/
  0,     5612,  11746, 38404, 34589, 35298, 20502, 19614, 0,     30581,   /*[GB+82329430, GB+82329439]*/
  9036,  0,     30607, 0,     0,     0,     39194, 22450, 0,     9097,    /*[GB+82329530, GB+82329539]*/
  16049, 3816,  9603,  32372, 18351, 19478, 19040, 34109, 898,   211,     /*[GB+82329630, GB+82329639]*/
  39801, 15462, 27252, 36334, 43499, 5170,  2456,  40926, 39038, 8816,    /*[GB+82329730, GB+82329739]*/
  27412, 0,     33842, 29246, 14329, 14330, 35687, 41406, 0,     0,       /*[GB+82329830, GB+82329839]*/
  29403, 3356,  7792,  13133, 42013, 0,     9842,  33443, 0,     25821,   /*[GB+82329930, GB+82329939]*/
  29223, 10542, 32111, 6766,  43570, 26856, 22258, 8724,  19272, 13153,   /*[GB+82329A30, GB+82329A39]*/
  35599, 36073, 22397, 36429, 14757, 18352, 5264,  27290, 39998, 20230,   /*[GB+82329B30, GB+82329B39]*/
  28504, 18223, 18861, 9435,  35390, 34998, 25973, 25832, 3284,  39232,   /*[GB+82329C30, GB+82329C39]*/
  2261,  16897, 26769, 92,    25888, 25553, 4278,  23831, 17216, 37703,   /*[GB+82329D30, GB+82329D39]*/
  9941,  28624, 24222, 0,     42775, 12285, 12418, 11695, 983,   0,       /*[GB+82329E30, GB+82329E39]*/
  0,     40960, 13445, 9809,  33680, 9165,  2786,  20683, 43040, 31954,   /*[GB+82329F30, GB+82329F39]*/
  43249, 34251, 17643, 0,     0,     0,     42464, 114,   34443, 10423,   /*[GB+8232A030, GB+8232A039]*/
  33820, 0,     31957, 2433,  27751, 34267, 14782, 8207,  26846, 0,       /*[GB+8232A130, GB+8232A139]*/
  37453, 5739,  13666, 17936, 36729, 19948, 31942, 38912, 24208, 20606,   /*[GB+8232A230, GB+8232A239]*/
  18577, 1778,  0,     21024, 555,   12361, 0,     37432, 11022, 4572,    /*[GB+8232A330, GB+8232A339]*/
  11129, 23189, 4092,  18531, 30810, 12086, 1268,  441,   3276,  37186,   /*[GB+8232A430, GB+8232A439]*/
  5265,  310,   36176, 0,     43478, 5225,  26910, 710,   5171,  32416,   /*[GB+8232A530, GB+8232A539]*/
  30429, 795,   8943,  36645, 33546, 43000, 36485, 7819,  39368, 161,     /*[GB+8232A630, GB+8232A639]*/
  31911, 21269, 28360, 14935, 9815,  13471, 25726, 14412, 36730, 34035,   /*[GB+8232A730, GB+8232A739]*/
  16261, 21652, 36088, 27997, 33173, 25265, 12169, 27048, 41811, 31191,   /*[GB+8232A830, GB+8232A839]*/
  15832, 35031, 25090, 271,   21179, 9761,  0,     30829, 7725,  15594,   /*[GB+8232A930, GB+8232A939]*/
  3010,  29431, 1282,  21030, 29453, 9767,  11439, 11985, 20564, 10864,   /*[GB+8232AA30, GB+8232AA39]*/
  13634, 26654, 8872,  12244, 2211,  9783,  8250,  22266, 83,    38930,   /*[GB+8232AB30, GB+8232AB39]*/
  29433, 18862, 14614, 3834,  14615, 4014,  24487, 3324,  36669, 6848,    /*[GB+8232AC30, GB+8232AC39]*/
  6058,  0,     18864, 34538, 1634,  35519, 21366, 37571, 21566, 25127,   /*[GB+8232AD30, GB+8232AD39]*/
  25128, 33198, 35213, 39821, 25231, 3567,  37388, 38774, 37125, 13193,   /*[GB+8232AE30, GB+8232AE39]*/
  43658, 35994, 21844, 16165, 37394, 36842, 30190, 10310, 12369, 38876,   /*[GB+8232AF30, GB+8232AF39]*/
  28625, 2716,  18648, 0,     36053, 4370,  3301,  22838, 26599, 21718,   /*[GB+8232B030, GB+8232B039]*/
  12861, 39552, 41256, 11734, 2634,  1656,  25420, 33718, 14195, 16621,   /*[GB+8232B130, GB+8232B139]*/
  20705, 20931, 41262, 1546,  13059, 15948, 22515, 1444,  28989, 29480,   /*[GB+8232B230, GB+8232B239]*/
  22134, 19086, 21791, 22140, 33767, 2905,  26546, 14486, 12908, 34293,   /*[GB+8232B330, GB+8232B339]*/
  35837, 23214, 31581, 11507, 39885, 0,     3937,  2145,  43899, 6242,    /*[GB+8232B430, GB+8232B439]*/
  43293, 12951, 25224, 0,     35520, 12585, 28750, 16669, 0,     6959,    /*[GB+8232B530, GB+8232B539]*/
  37263, 39445, 732,   14576, 41643, 13575, 28833, 5999,  6581,  4667,    /*[GB+8232B630, GB+8232B639]*/
  8725,  21821, 41689, 41765, 0,     36920, 25744, 10844, 9557,  25645,   /*[GB+8232B730, GB+8232B739]*/
  20369, 9962,  17276, 27449, 7671,  5185,  36670, 9506,  14616, 9942,    /*[GB+8232B830, GB+8232B839]*/
  10453, 30261, 33157, 41970, 23034, 6307,  17390, 30382, 5084,  36435,   /*[GB+8232B930, GB+8232B939]*/
  10358, 0,     0,     5903,  0,     21621, 14521, 21845, 15543, 39527,   /*[GB+8232BA30, GB+8232BA39]*/
  41721, 41493, 41082, 35433, 0,     623,   11130, 9682,  11422, 17725,   /*[GB+8232BB30, GB+8232BB39]*/
  33475, 1982,  16561, 20121, 4965,  18661, 37719, 25779, 29150, 0,       /*[GB+8232BC30, GB+8232BC39]*/
  36126, 14828, 25896, 12498, 29960, 4430,  7068,  19670, 0,     22574,   /*[GB+8232BD30, GB+8232BD39]*/
  30610, 5506,  32825, 9004,  5291,  18475, 2298,  34595, 23626, 17313,   /*[GB+8232BE30, GB+8232BE39]*/
  42868, 22968, 12036, 38910, 34600, 17601, 22012, 538,   0,     10226,   /*[GB+8232BF30, GB+8232BF39]*/
  1187,  8381,  12812, 38153, 19589, 37049, 38493, 15748, 12783, 3551,    /*[GB+8232C030, GB+8232C039]*/
  6707,  7297,  0,     36818, 43241, 41080, 25048, 39516, 26735, 12786,   /*[GB+8232C130, GB+8232C139]*/
  23035, 12287, 15544, 28369, 0,     0,     23917, 21936, 2525,  19819,   /*[GB+8232C230, GB+8232C239]*/
  18110, 4413,  0,     5040,  28151, 5469,  24980, 0,     17427, 19420,   /*[GB+8232C330, GB+8232C339]*/
  18948, 27087, 39435, 38094, 6275,  25027, 37771, 22860, 8382,  13874,   /*[GB+8232C430, GB+8232C439]*/
  36457, 7862,  27444, 34216, 0,     0,     1283,  28775, 24565, 22866,   /*[GB+8232C530, GB+8232C539]*/
  42335, 30887, 31274, 31275, 27354, 38176, 19036, 236,   11085, 26137,   /*[GB+8232C630, GB+8232C639]*/
  18354, 10156, 43608, 29884, 40154, 36532, 2696,  32396, 15695, 31521,   /*[GB+8232C730, GB+8232C739]*/
  39310, 12553, 13897, 27423, 20899, 15534, 43480, 15388, 170,   30098,   /*[GB+8232C830, GB+8232C839]*/
  17367, 12746, 26721, 3049,  7239,  16583, 22645, 31055, 35806, 15313,   /*[GB+8232C930, GB+8232C939]*/
  12288, 25140, 14646, 20972, 36597, 0,     42230, 31918, 0,     108,     /*[GB+8232CA30, GB+8232CA39]*/
  23691, 2500,  30749, 7526,  12753, 24942, 4183,  30272, 43371, 23303,   /*[GB+8232CB30, GB+8232CB39]*/
  31807, 28461, 19856, 1648,  18111, 20776, 35355, 2614,  12309, 21168,   /*[GB+8232CC30, GB+8232CC39]*/
  9139,  18681, 5778,  41263, 43525, 0,     0,     42522, 529,   5119,    /*[GB+8232CD30, GB+8232CD39]*/
  26246, 0,     19697, 34284, 8060,  10484, 3509,  15287, 7372,  3912,    /*[GB+8232CE30, GB+8232CE39]*/
  40221, 11364, 37369, 36423, 37269, 31583, 41073, 39370, 36218, 37374,   /*[GB+8232CF30, GB+8232CF39]*/
  16464, 12185, 17632, 40242, 3375,  0,     0,     28546, 0,     23149,   /*[GB+8232D030, GB+8232D039]*/
  21927, 11313, 4410,  40412, 35999, 14946, 43381, 39413, 30557, 16287,   /*[GB+8232D130, GB+8232D139]*/
  26179, 0,     3585,  15242, 36234, 41924, 3938,  23594, 25782, 0,       /*[GB+8232D230, GB+8232D239]*/
  26522, 17489, 38405, 27187, 29569, 25253, 29604, 0,     7645,  0,       /*[GB+8232D330, GB+8232D339]*/
  21258, 11981, 0,     0,     10589, 25765, 4272,  26780, 7010,  18394,   /*[GB+8232D430, GB+8232D439]*/
  514,   14777, 35754, 20595, 0,     40393, 42563, 11835, 38630, 24566,   /*[GB+8232D530, GB+8232D539]*/
  42200, 25442, 0,     8956,  14020, 43008, 43009, 16719, 23013, 27447,   /*[GB+8232D630, GB+8232D639]*/
  40799, 279,   42370, 9583,  38215, 3639,  12979, 42830, 17765, 27063,   /*[GB+8232D730, GB+8232D739]*/
  27133, 41900, 22366, 2574,  0,     0,     37856, 15888, 1760,  3561,    /*[GB+8232D830, GB+8232D839]*/
  16078, 6712,  32777, 38254, 41571, 36681, 0,     27614, 20427, 24466,   /*[GB+8232D930, GB+8232D939]*/
  41464, 10686, 39822, 5391,  41494, 7954,  27945, 19331, 30393, 26418,   /*[GB+8232DA30, GB+8232DA39]*/
  20904, 37596, 790,   10311, 9917,  22496, 9312,  0,     28797, 16399,   /*[GB+8232DB30, GB+8232DB39]*/
  30273, 4685,  42782, 30490, 16819, 26012, 35743, 6855,  2592,  12155,   /*[GB+8232DC30, GB+8232DC39]*/
  4807,  27594, 41977, 25411, 14821, 43668, 32794, 0,     0,     33995,   /*[GB+8232DD30, GB+8232DD39]*/
  22390, 24677, 29778, 15647, 41733, 28250, 30856, 627,   30413, 25516,   /*[GB+8232DE30, GB+8232DE39]*/
  33090, 27163, 20588, 11735, 29819, 42581, 24683, 21783, 14829, 28252,   /*[GB+8232DF30, GB+8232DF39]*/
  21591, 29275, 19360, 17920, 14196, 18013, 42486, 6281,  0,     27642,   /*[GB+8232E030, GB+8232E039]*/
  9740,  7849,  21204, 30142, 13948, 30771, 35280, 16712, 33325, 7761,    /*[GB+8232E130, GB+8232E139]*/
  20596, 2380,  2637,  19365, 37726, 4874,  40385, 21312, 39919, 42512,   /*[GB+8232E230, GB+8232E239]*/
  38000, 5508,  12885, 32692, 30618, 27818, 35267, 29823, 29724, 25352,   /*[GB+8232E330, GB+8232E339]*/
  19548, 38434, 0,     17990, 18165, 8057,  19089, 19235, 34601, 34987,   /*[GB+8232E430, GB+8232E439]*/
  23475, 21483, 34288, 26930, 43163, 29359, 36744, 1614,  19425, 11366,   /*[GB+8232E530, GB+8232E539]*/
  24860, 18973, 651,   1294,  11665, 11666, 0,     2576,  23830, 26149,   /*[GB+8232E630, GB+8232E639]*/
  21921, 15306, 2266,  21122, 27630, 32880, 0,     0,     18093, 26723,   /*[GB+8232E730, GB+8232E739]*/
  0,     12717, 34688, 0,     28966, 38877, 0,     22497, 12297, 15318,   /*[GB+8232E830, GB+8232E839]*/
  36602, 0,     27554, 31875, 35256, 41603, 21228, 34006, 20783, 0,       /*[GB+8232E930, GB+8232E939]*/
  12316, 30611, 34976, 37633, 1446,  20478, 7762,  18499, 5057,  4317,    /*[GB+8232EA30, GB+8232EA39]*/
  5782,  5972,  16951, 4205,  34289, 3033,  21492, 25525, 26267, 41812,   /*[GB+8232EB30, GB+8232EB39]*/
  0,     0,     0,     11837, 8953,  3760,  9960,  22259, 2026,  11982,   /*[GB+8232EC30, GB+8232EC39]*/
  9426,  37270, 21031, 32512, 0,     0,     0,     27292, 36172, 41892,   /*[GB+8232ED30, GB+8232ED39]*/
  21826, 14179, 0,     41294, 44021, 40231, 19038, 0,     42765, 1295,    /*[GB+8232EE30, GB+8232EE39]*/
  27065, 39782, 0,     4600,  7591,  38216, 21513, 26098, 0,     32524,   /*[GB+8232EF30, GB+8232EF39]*/
  13426, 2215,  0,     1725,  7737,  40250, 18067, 7738,  26492, 8677,    /*[GB+8232F030, GB+8232F039]*/
  7562,  7276,  41914, 31201, 39824, 14936, 25054, 15472, 17371, 3260,    /*[GB+8232F130, GB+8232F139]*/
  917,   23229, 38311, 35809, 22178, 36335, 8594,  0,     23127, 31453,   /*[GB+8232F230, GB+8232F239]*/
  1105,  41044, 32599, 16400, 37241, 23508, 34834, 9442,  30759, 10315,   /*[GB+8232F330, GB+8232F339]*/
  12617, 30329, 0,     37322, 5233,  25166, 14873, 20385, 21592, 21593,   /*[GB+8232F430, GB+8232F439]*/
  13810, 20137, 7755,  23467, 21452, 14692, 8691,  1782,  12370, 29842,   /*[GB+8232F530, GB+8232F539]*/
  37334, 23084, 14833, 18732, 2136,  13831, 31118, 19370, 43895, 28731,   /*[GB+8232F630, GB+8232F639]*/
  38677, 5783,  4094,  2300,  21797, 36881, 17564, 3980,  14321, 29358,   /*[GB+8232F730, GB+8232F739]*/
  26626, 23110, 20607, 0,     28739, 20610, 42924, 0,     15741, 15178,   /*[GB+8232F830, GB+8232F839]*/
  7592,  38218, 27453, 2345,  27003, 8595,  17470, 30148, 39449, 39216,   /*[GB+8232F930, GB+8232F939]*/
  6155,  19294, 43013, 30372, 24449, 41131, 14271, 4481,  2271,  16656,   /*[GB+8232FA30, GB+8232FA39]*/
  5142,  0,     10711, 10712, 22194, 11826, 9490,  5891,  8508,  36099,   /*[GB+8232FB30, GB+8232FB39]*/
  21424, 21113, 17677, 9855,  41490, 30383, 39826, 43734, 18116, 20124,   /*[GB+8232FC30, GB+8232FC39]*/
  33321, 17860, 13320, 18537, 18188, 0,     24501, 37156, 32266, 32120,   /*[GB+8232FD30, GB+8232FD39]*/
  23874, 0,     41483, 10248, 0,     35779, 22401, 26789, 32786, 41915,   /*[GB+8232FE30, GB+8232FE39]*/
  6565,  32798, 2098,  0,     12218, 36134, 40599, 18552, 36780, 12259,   /*[GB+82338130, GB+82338139]*/
  36219, 11667, 811,   26903, 34794, 24569, 18974, 42681, 38219, 35884,   /*[GB+82338230, GB+82338239]*/
  20570, 16339, 4280,  0,     22378, 10687, 3216,  20087, 41247, 30394,   /*[GB+82338330, GB+82338339]*/
  8428,  11321, 40920, 25729, 29973, 24358, 10067, 0,     12694, 4164,    /*[GB+82338430, GB+82338439]*/
  0,     7546,  7547,  27386, 25030, 29628, 15464, 0,     36835, 748,     /*[GB+82338530, GB+82338539]*/
  37586, 43492, 22646, 4458,  2506,  31170, 10819, 44034, 41531, 32263,   /*[GB+82338630, GB+82338639]*/
  9397,  1298,  7616,  43019, 33465, 33009, 42422, 42777, 20088, 33013,   /*[GB+82338730, GB+82338739]*/
  10365, 26337, 17292, 40571, 29815, 21629, 6011,  25167, 2529,  24322,   /*[GB+82338830, GB+82338839]*/
  18682, 28732, 19704, 29911, 25071, 40143, 8292,  35682, 15827, 6035,    /*[GB+82338930, GB+82338939]*/
  24262, 10006, 23213, 27049, 41807, 8903,  24161, 30635, 3207,  4563,    /*[GB+82338A30, GB+82338A39]*/
  11011, 22050, 1752,  25093, 28897, 1189,  15835, 29614, 0,     9877,    /*[GB+82338B30, GB+82338B39]*/
  22352, 12481, 33565, 7615,  35597, 22164, 0,     33790, 42339, 27473,   /*[GB+82338C30, GB+82338C39]*/
  15450, 6363,  41649, 28291, 21362, 1299,  10763, 39466, 34523, 23664,   /*[GB+82338D30, GB+82338D39]*/
  22687, 2399,  2262,  0,     25046, 13413, 43350, 20059, 16070, 34533,   /*[GB+82338E30, GB+82338E39]*/
  33613, 2340,  33046, 42114, 43615, 17138, 4784,  31279, 4281,  5904,    /*[GB+82338F30, GB+82338F39]*/
  4571,  26222, 41572, 40953, 27249, 1836,  13186, 33637, 42837, 15889,   /*[GB+82339030, GB+82339039]*/
  9295,  12990, 5222,  3334,  30305, 27496, 34689, 12352, 24951, 43036,   /*[GB+82339130, GB+82339139]*/
  30188, 2772,  33107, 16391, 16342, 9804,  8880,  7842,  4811,  2522,    /*[GB+82339230, GB+82339239]*/
  42442, 2969,  17848, 35616, 41146, 41589, 39539, 10312, 9525,  40595,   /*[GB+82339330, GB+82339339]*/
  5463,  12862, 29779, 30682, 9733,  33999, 20778, 7161,  404,   24194,   /*[GB+82339430, GB+82339439]*/
  33484, 64,    21313, 24201, 21314, 37179, 42513, 2137,  38906, 32696,   /*[GB+82339530, GB+82339539]*/
  26941, 17508, 36618, 2999,  26700, 41770, 24689, 0,     30483, 8006,    /*[GB+82339630, GB+82339639]*/
  29165, 0,     5680,  5173,  30540, 31222, 3767,  30404, 13691, 29440,   /*[GB+82339730, GB+82339739]*/
  12329, 18951, 0,     0,     3315,  14906, 7378,  9607,  8531,  7234,    /*[GB+82339830, GB+82339839]*/
  0,     7408,  1074,  30838, 6060,  0,     6069,  2268,  32406, 41497,   /*[GB+82339930, GB+82339939]*/
  20429, 24930, 22282, 25507, 0,     43664, 29773, 0,     39252, 42700,   /*[GB+82339A30, GB+82339A39]*/
  30416, 0,     29963, 2283,  33824, 14197, 2546,  8809,  30970, 3309,    /*[GB+82339B30, GB+82339B39]*/
  8818,  8046,  33402, 33549, 37159, 21961, 23702, 20933, 27848, 21346,   /*[GB+82339C30, GB+82339C39]*/
  2537,  31338, 16143, 576,   34917, 35382, 25095, 0,     0,     28270,   /*[GB+82339D30, GB+82339D39]*/
  12179, 23199, 34894, 3362,  5433,  8237,  38631, 272,   26901, 27106,   /*[GB+82339E30, GB+82339E39]*/
  20758, 7829,  26399, 28836, 11088, 1574,  5372,  22071, 5848,  0,       /*[GB+82339F30, GB+82339F39]*/
  0,     16876, 0,     2750,  7187,  39311, 10970, 0,     9885,  36221,   /*[GB+8233A030, GB+8233A039]*/
  17968, 14853, 25047, 26406, 32526, 37701, 18882, 0,     0,     36682,   /*[GB+8233A130, GB+8233A139]*/
  38258, 38581, 24906, 41573, 33965, 38259, 37496, 33307, 42055, 42398,   /*[GB+8233A230, GB+8233A239]*/
  10715, 4604,  8410,  4544,  24462, 16785, 4059,  0,     32181, 4796,    /*[GB+8233A330, GB+8233A339]*/
  9128,  16762, 26420, 26421, 42058, 21391, 18408, 42691, 30398, 42059,   /*[GB+8233A430, GB+8233A439]*/
  9858,  18844, 11620, 17285, 6164,  4748,  38779, 0,     0,     24932,   /*[GB+8233A530, GB+8233A539]*/
  43368, 2752,  21071, 6788,  38521, 2787,  27456, 11246, 27403, 8420,    /*[GB+8233A630, GB+8233A639]*/
  17401, 35522, 13795, 38312, 21165, 0,     20909, 13474, 30224, 10810,   /*[GB+8233A730, GB+8233A739]*/
  0,     18434, 43587, 43870, 37323, 9244,  18117, 25060, 9588,  18435,   /*[GB+8233A830, GB+8233A839]*/
  1863,  30225, 0,     0,     29964, 4132,  13812, 35096, 1000,  35815,   /*[GB+8233A930, GB+8233A939]*/
  15135, 24685, 19361, 34673, 44079, 6319,  4558,  26088, 22475, 41107,   /*[GB+8233AA30, GB+8233AA39]*/
  20288, 37906, 28275, 39563, 11952, 18127, 23528, 0,     0,     18128,   /*[GB+8233AB30, GB+8233AB39]*/
  0,     0,     29381, 0,     38372, 23192, 29557, 16747, 8819,  37727,   /*[GB+8233AC30, GB+8233AC39]*/
  5839,  26881, 2598,  0,     31091, 25955, 1706,  30145, 32685, 7071,    /*[GB+8233AD30, GB+8233AD39]*/
  27816, 137,   25171, 43999, 16855, 7983,  0,     38734, 0,     27738,   /*[GB+8233AE30, GB+8233AE39]*/
  6729,  12658, 35282, 40850, 30619, 30776, 42514, 38409, 8628,  7319,    /*[GB+8233AF30, GB+8233AF39]*/
  0,     16102, 13612, 2724,  34390, 20857, 0,     1466,  19378, 14701,   /*[GB+8233B030, GB+8233B039]*/
  17108, 13667, 0,     3496,  17688, 35456, 8058,  20203, 41055, 11162,   /*[GB+8233B130, GB+8233B139]*/
  13082, 22784, 38914, 34988, 30988, 17779, 40856, 17036, 0,     25432,   /*[GB+8233B230, GB+8233B239]*/
  15512, 24297, 7778,  31678, 19196, 21485, 43420, 35121, 11889, 21486,   /*[GB+8233B330, GB+8233B339]*/
  14539, 40540, 4886,  18584, 26886, 42923, 38616, 10837, 0,     38466,   /*[GB+8233B430, GB+8233B439]*/
  20349, 40315, 26888, 19428, 22935, 39973, 23368, 0,     38155, 23370,   /*[GB+8233B530, GB+8233B539]*/
  38156, 25379, 34101, 4213,  38574, 21555, 33639, 22380, 16224, 43866,   /*[GB+8233B630, GB+8233B639]*/
  34122, 36698, 31808, 31008, 33266, 30286, 38609, 3957,  42727, 20917,   /*[GB+8233B730, GB+8233B739]*/
  40044, 23351, 21708, 40730, 32386, 18048, 26401, 22362, 28779, 1304,    /*[GB+8233B830, GB+8233B839]*/
  43454, 736,   0,     15756, 34682, 17000, 23579, 16877, 36318, 41076,   /*[GB+8233B930, GB+8233B939]*/
  37281, 17139, 12413, 33640, 7304,  37121, 31353, 39241, 15891, 18072,   /*[GB+8233BA30, GB+8233BA39]*/
  0,     18094, 3476,  13450, 11785, 41309, 8679,  3093,  10017, 15473,   /*[GB+8233BB30, GB+8233BB39]*/
  21277, 3052,  30679, 22179, 35444, 18231, 36604, 29775, 28716, 38313,   /*[GB+8233BC30, GB+8233BC39]*/
  1931,  4812,  11514, 32407, 6089,  13039, 9322,  10696, 2099,  35619,   /*[GB+8233BD30, GB+8233BD39]*/
  19517, 2410,  2411,  38373, 36057, 36711, 40740, 10820, 39068, 43679,   /*[GB+8233BE30, GB+8233BE39]*/
  0,     16289, 39565, 24801, 41660, 20633, 0,     0,     29382, 14966,   /*[GB+8233BF30, GB+8233BF39]*/
  10061, 24690, 19223, 0,     31496, 14171, 17308, 38410, 39043, 4758,    /*[GB+8233C030, GB+8233C039]*/
  36870, 14698, 36274, 32940, 34460, 23171, 8828,  9342,  0,     22144,   /*[GB+8233C130, GB+8233C139]*/
  43059, 22406, 34465, 33031, 18555, 2419,  21802, 13855, 22851, 2807,    /*[GB+8233C230, GB+8233C239]*/
  32332, 35847, 23353, 20731, 43914, 16250, 16115, 10862, 0,     0,       /*[GB+8233C330, GB+8233C339]*/
  39870, 32941, 42875, 0,     0,     38128, 0,     6243,  8512,  1195,    /*[GB+8233C430, GB+8233C439]*/
  42940, 43575, 29232, 34103, 22716, 0,     14364, 36320, 3921,  23301,   /*[GB+8233C530, GB+8233C539]*/
  27137, 42376, 27672, 0,     28024, 39807, 0,     14818, 0,     20092,   /*[GB+8233C630, GB+8233C639]*/
  10628, 1727,  38288, 43956, 41310, 39831, 32408, 22731, 10018, 15904,   /*[GB+8233C730, GB+8233C739]*/
  1103,  2404,  0,     7254,  25148, 37243, 17091, 27377, 11515, 36336,   /*[GB+8233C830, GB+8233C839]*/
  35232, 0,     0,     35088, 2100,  16489, 5037,  35935, 569,   23468,   /*[GB+8233C930, GB+8233C939]*/
  43719, 0,     31171, 4227,  3731,  22960, 10059, 8309,  35258, 5840,    /*[GB+8233CA30, GB+8233CA39]*/
  32686, 15953, 40931, 27568, 7208,  19371, 21332, 0,     10514, 21333,   /*[GB+8233CB30, GB+8233CB39]*/
  19698, 0,     38993, 0,     10073, 4985,  18195, 6840,  0,     1625,    /*[GB+8233CC30, GB+8233CC39]*/
  25287, 33616, 0,     5794,  5795,  34524, 18615, 0,     28293, 35054,   /*[GB+8233CD30, GB+8233CD39]*/
  28693, 32874, 0,     0,     11191, 39248, 20104, 17402, 38949, 28629,   /*[GB+8233CE30, GB+8233CE39]*/
  15776, 25163, 13939, 40467, 0,     25172, 0,     19228, 13259, 20709,   /*[GB+8233CF30, GB+8233CF39]*/
  4433,  22904, 1829,  18591, 19430, 9081,  3562,  35992, 34432, 11699,   /*[GB+8233D030, GB+8233D039]*/
  1153,  43824, 5335,  5336,  12754, 27587, 3302,  31076, 0,     23429,   /*[GB+8233D130, GB+8233D139]*/
  42489, 19531, 8051,  14415, 9994,  33832, 19160, 36030, 27091, 30921,   /*[GB+8233D230, GB+8233D239]*/
  24135, 35384, 28105, 42346, 32268, 31687, 31196, 37694, 35179, 24137,   /*[GB+8233D330, GB+8233D339]*/
  37272, 37273, 23354, 10967, 23125, 38697, 7793,  22462, 37231, 32442,   /*[GB+8233D430, GB+8233D439]*/
  40134, 34111, 42770, 40135, 28935, 21696, 33617, 13418, 30893, 8018,    /*[GB+8233D530, GB+8233D539]*/
  36223, 22720, 4674,  21556, 1582,  14037, 22314, 39809, 7431,  42223,   /*[GB+8233D630, GB+8233D639]*/
  27000, 35903, 20376, 12574, 36387, 22566, 10794, 13652, 6716,  12001,   /*[GB+8233D730, GB+8233D739]*/
  31735, 24416, 4953,  33661, 29742, 21440, 35397, 33477, 26171, 41366,   /*[GB+8233D830, GB+8233D839]*/
  30902, 35523, 15320, 15638, 12755, 31068, 20772, 36959, 13021, 29051,   /*[GB+8233D930, GB+8233D939]*/
  17916, 32412, 3296,  2458,  14778, 39382, 12493, 41049, 29970, 9328,    /*[GB+8233DA30, GB+8233DA39]*/
  22618, 34008, 0,     6980,  3737,  32687, 41615, 10377, 3176,  3667,    /*[GB+8233DB30, GB+8233DB39]*/
  40713, 12660, 20321, 18934, 17742, 32081, 12442, 33498, 407,   28384,   /*[GB+8233DC30, GB+8233DC39]*/
  30057, 20748, 30687, 35409, 15073, 201,   30448, 2903,  32945, 31893,   /*[GB+8233DD30, GB+8233DD39]*/
  13526, 3404,  3312,  39943, 34470, 0,     0,     0,     22619, 0,       /*[GB+8233DE30, GB+8233DE39]*/
  26278, 14140, 29711, 16071, 18894, 15477, 0,     20823, 19093, 0,       /*[GB+8233DF30, GB+8233DF39]*/
  4265,  3775,  34668, 25892, 21162, 29256, 2791,  3786,  9744,  39554,   /*[GB+8233E030, GB+8233E039]*/
  38524, 0,     19522, 17737, 29276, 41621, 29363, 12398, 0,     0,       /*[GB+8233E130, GB+8233E139]*/
  7414,  0,     27950, 43737, 15921, 16093, 31844, 19829, 32693, 3946,    /*[GB+8233E230, GB+8233E239]*/
  43060, 19094, 0,     41628, 41462, 0,     38189, 4140,  22721, 1969,    /*[GB+8233E330, GB+8233E339]*/
  29997, 37986, 10987, 36551, 12130, 20852, 20853, 26528, 17777, 10937,   /*[GB+8233E430, GB+8233E439]*/
  42586, 21828, 34328, 41830, 29169, 44024, 43022, 9662,  36112, 38225,   /*[GB+8233E530, GB+8233E539]*/
  42400, 35066, 14927, 2405,  0,     2272,  18813, 42067, 13465, 32413,   /*[GB+8233E630, GB+8233E639]*/
  10054, 15401, 109,   8792,  10313, 11559, 36977, 27406, 42467, 1712,    /*[GB+8233E730, GB+8233E739]*/
  37907, 30281, 6419,  10332, 28658, 36402, 3447,  28327, 0,     0,       /*[GB+8233E830, GB+8233E839]*/
  4575,  11382, 6628,  3477,  32787, 27329, 29164, 2309,  13145, 40517,   /*[GB+8233E930, GB+8233E939]*/
  24883, 36583, 8737,  39787, 8554,  24575, 42045, 30643, 44025, 3258,    /*[GB+8233EA30, GB+8233EA39]*/
  26484, 39377, 11115, 11553, 10254, 7432,  14216, 40407, 31705, 5126,    /*[GB+8233EB30, GB+8233EB39]*/
  31737, 8777,  44031, 11700, 0,     2087,  41445, 16007, 30701, 15923,   /*[GB+8233EC30, GB+8233EC39]*/
  8793,  12305, 4516,  39119, 4493,  30282, 3733,  25333, 2313,  34912,   /*[GB+8233ED30, GB+8233ED39]*/
  20788, 2382,  25173, 14095, 1449,  13072, 42152, 42913, 26529, 41273,   /*[GB+8233EE30, GB+8233EE39]*/
  13267, 1517,  0,     18559, 18569, 40302, 26705, 3455,  8722,  2749,    /*[GB+8233EF30, GB+8233EF39]*/
  30814, 28923, 10883, 25830, 24885, 2044,  22364, 31702, 4161,  4943,    /*[GB+8233F030, GB+8233F039]*/
  40245, 42046, 3377,  4274,  1326,  21265, 485,   31230, 21835, 18995,   /*[GB+8233F130, GB+8233F139]*/
  8316,  3469,  26360, 31349, 8569,  17217, 13912, 0,     0,     0,       /*[GB+8233F230, GB+8233F239]*/
  41840, 26335, 44077, 3857,  17014, 19053, 987,   6892,  33386, 0,       /*[GB+8233F330, GB+8233F339]*/
  43376, 20109, 30702, 0,     4185,  18846, 31295, 17182, 3098,  15924,   /*[GB+8233F430, GB+8233F439]*/
  31804, 12213, 8035,  1400,  0,     34068, 33206, 13480, 26600, 17054,   /*[GB+8233F530, GB+8233F539]*/
  11731, 26232, 30123, 2312,  0,     26362, 24196, 23694, 32275, 37330,   /*[GB+8233F630, GB+8233F639]*/
  27173, 36264, 4972,  6420,  3892,  5175,  21395, 36085, 7209,  1771,    /*[GB+8233F730, GB+8233F739]*/
  41662, 4203,  3025,  10336, 6954,  43793, 5875,  17388, 30990, 40303,   /*[GB+8233F830, GB+8233F839]*/
  26706, 42926, 19309, 3326,  41776, 8754,  28353, 31383, 16760, 35,      /*[GB+8233F930, GB+8233F939]*/
  0,     26172, 29270, 10559, 28088, 32301, 41274, 19752, 15223, 18313,   /*[GB+8233FA30, GB+8233FA39]*/
  0,     4502,  27128, 40234, 15865, 16301, 7794,  24887, 11388, 8539,    /*[GB+8233FB30, GB+8233FB39]*/
  19917, 11400, 32130, 21836, 31231, 15764, 25049, 41948, 26100, 9563,    /*[GB+8233FC30, GB+8233FC39]*/
  31241, 17642, 21066, 38712, 20111, 40015, 15480, 24187, 0,     35090,   /*[GB+8233FD30, GB+8233FD39]*/
  1596,  12698, 42852, 27176, 27690, 33264, 2756,  16563, 28149, 23934,   /*[GB+8233FE30, GB+8233FE39]*/
  20854, 35629, 0,     4829,  16567, 43121, 2887,  29562, 3947,  30150,   /*[GB+82348130, GB+82348139]*/
  1006,  16191, 0,     42527, 32951, 21856, 19394, 43885, 23116, 19431,   /*[GB+82348230, GB+82348239]*/
  25104, 40236, 0,     38315, 33892, 3384,  0,     27220, 3403,  23262,   /*[GB+82348330, GB+82348339]*/
  39195, 13344, 2009,  7894,  0,     0,     4927,  5997,  14261, 39462,   /*[GB+82348430, GB+82348439]*/
  11071, 35859, 39734, 26393, 0,     583,   9401,  15179, 0,     36542,   /*[GB+82348530, GB+82348539]*/
  29241, 39233, 29736, 37574, 2499,  37307, 29257, 36691, 29314, 18980,   /*[GB+82348630, GB+82348639]*/
  4813,  39852, 2101,  30055, 0,     36979, 17980, 19158, 31009, 6817,    /*[GB+82348730, GB+82348739]*/
  40304, 13282, 0,     40350, 0,     0,     15352, 15525, 4138,  3358,    /*[GB+82348830, GB+82348839]*/
  9493,  34730, 34408, 155,   10228, 39631, 17902, 0,     31760, 3365,    /*[GB+82348930, GB+82348939]*/
  35479, 26295, 10885, 0,     5627,  2338,  6002,  36790, 43458, 0,       /*[GB+82348A30, GB+82348A39]*/
  38758, 2848,  0,     18076, 30262, 20651, 20656, 0,     30743, 23804,   /*[GB+82348B30, GB+82348B39]*/
  19811, 24782, 5169,  31805, 7459,  2408,  14661, 37895, 13246, 5639,    /*[GB+82348C30, GB+82348C39]*/
  15986, 1190,  17960, 38160, 4535,  4538,  24570, 18049, 40565, 30520,   /*[GB+82348D30, GB+82348D39]*/
  24576, 4929,  15613, 35906, 7835,  0,     35907, 7622,  12788, 42840,   /*[GB+82348E30, GB+82348E39]*/
  26907, 7659,  15699, 10631, 18815, 42068, 21585, 39412, 0,     2613,    /*[GB+82348F30, GB+82348F39]*/
  21135, 38525, 21684, 31809, 17021, 0,     0,     21597, 27177, 39892,   /*[GB+82349030, GB+82349039]*/
  25573, 21458, 15503, 24155, 14879, 32557, 13511, 21340, 14116, 36239,   /*[GB+82349130, GB+82349139]*/
  881,   9013,  3005,  18596, 18222, 26297, 7096,  38976, 40402, 3379,    /*[GB+82349230, GB+82349239]*/
  42219, 26908, 0,     20377, 16136, 10257, 40257, 12591, 24184, 2711,    /*[GB+82349330, GB+82349339]*/
  7196,  2868,  27898, 28968, 28376, 35493, 38879, 28969, 3870,  37612,   /*[GB+82349430, GB+82349439]*/
  10697, 8040,  37613, 36855, 43907, 29785, 14830, 7258,  34581, 9990,    /*[GB+82349530, GB+82349539]*/
  30864, 25860, 2983,  10704, 21396, 40278, 4973,  25520, 14878, 22241,   /*[GB+82349630, GB+82349639]*/
  40022, 0,     23888, 41985, 42516, 4559,  39691, 22190, 32469, 14238,   /*[GB+82349730, GB+82349739]*/
  24813, 29980, 24343, 31257, 16856, 4978,  30155, 0,     14536, 14110,   /*[GB+82349830, GB+82349839]*/
  347,   14424, 37647, 0,     37648, 19700, 40720, 796,   18701, 0,       /*[GB+82349930, GB+82349939]*/
  12145, 20303, 32708, 34466, 31420, 2152,  41989, 42917, 911,   21349,   /*[GB+82349A30, GB+82349A39]*/
  35139, 23477, 39180, 0,     34934, 18580, 41028, 21493, 0,     0,       /*[GB+82349B30, GB+82349B39]*/
  0,     35131, 0,     0,     0,     0,     13407, 14519, 0,     22951,   /*[GB+82349C30, GB+82349C39]*/
  345,   4266,  33450, 10012, 35194, 31523, 36224, 5669,  41180, 30645,   /*[GB+82349D30, GB+82349D39]*/
  23881, 35074, 35919, 34555, 29643, 16780, 33313, 12771, 27478, 6752,    /*[GB+82349E30, GB+82349E39]*/
  12124, 17224, 8803,  4338,  34370, 25952, 25799, 17471, 0,     36552,   /*[GB+82349F30, GB+82349F39]*/
  38800, 26808, 10848, 4576,  42494, 0,     32835, 36731, 34750, 38437,   /*[GB+8234A030, GB+8234A039]*/
  22788, 0,     28748, 6457,  43552, 15820, 35828, 40053, 0,     0,       /*[GB+8234A130, GB+8234A139]*/
  35789, 5743,  31212, 9361,  37803, 11372, 37686, 0,     18219, 16682,   /*[GB+8234A230, GB+8234A239]*/
  34421, 30024, 33941, 36102, 0,     0,     5898,  17355, 42675, 22920,   /*[GB+8234A330, GB+8234A339]*/
  3542,  14025, 1330,  43131, 19312, 10977, 747,   30732, 3650,  20676,   /*[GB+8234A430, GB+8234A439]*/
  34547, 29331, 39017, 26423, 0,     24673, 12603, 32673, 37994, 37614,   /*[GB+8234A530, GB+8234A539]*/
  0,     3304,  10934, 1818,  0,     34591, 2900,  12762, 0,     10767,   /*[GB+8234A630, GB+8234A639]*/
  4671,  42048, 24899, 17001, 27251, 0,     38765, 35559, 0,     11714,   /*[GB+8234A730, GB+8234A739]*/
  5051,  0,     26801, 5953,  33487, 26235, 0,     37001, 18939, 1171,    /*[GB+8234A830, GB+8234A839]*/
  0,     1831,  0,     40029, 23371, 784,   38933, 11411, 4683,  25232,   /*[GB+8234A930, GB+8234A939]*/
  31057, 39834, 17844, 789,   0,     13455, 8601,  34564, 2612,  11632,   /*[GB+8234AA30, GB+8234AA39]*/
  27805, 1088,  26037, 39686, 32236, 63,    10700, 5575,  9330,  5958,    /*[GB+8234AB30, GB+8234AB39]*/
  12879, 23672, 0,     43216, 19366, 20708, 20746, 18692, 34867, 36198,   /*[GB+8234AC30, GB+8234AC39]*/
  41859, 24596, 8829,  23291, 34038, 7047,  5591,  40422, 34606, 38682,   /*[GB+8234AD30, GB+8234AD39]*/
  29138, 7050,  1135,  11885, 7967,  8011,  40521, 993,   7943,  34239,   /*[GB+8234AE30, GB+8234AE39]*/
  28954, 21676, 41248, 41249, 41182, 12625, 8686,  32422, 22022, 25735,   /*[GB+8234AF30, GB+8234AF39]*/
  18967, 0,     21804, 11607, 11404, 39496, 24911, 7244,  275,   0,       /*[GB+8234B030, GB+8234B039]*/
  515,   6074,  36122, 6083,  1375,  42778, 23737, 31408, 37884, 0,       /*[GB+8234B130, GB+8234B139]*/
  13661, 42470, 31810, 35092, 5564,  31365, 35252, 3152,  40139, 10024,   /*[GB+8234B230, GB+8234B239]*/
  18849, 1109,  0,     20147, 13256, 36135, 29279, 6756,  29789, 11748,   /*[GB+8234B330, GB+8234B339]*/
  40471, 2993,  717,   27185, 7323,  33094, 526,   8311,  39414, 41663,   /*[GB+8234B430, GB+8234B439]*/
  8166,  10025, 2222,  9345,  7075,  12330, 6830,  31120, 2154,  25523,   /*[GB+8234B530, GB+8234B539]*/
  19102, 19720, 32958, 41341, 17527, 30046, 22383, 1376,  32140, 43037,   /*[GB+8234B630, GB+8234B639]*/
  6333,  2218,  15491, 24592, 34151, 32829, 24594, 5239,  7707,  2834,    /*[GB+8234B730, GB+8234B739]*/
  11890, 40522, 0,     0,     0,     0,     0,     7776,  35303, 11405,   /*[GB+8234B830, GB+8234B839]*/
  3709,  785,   38583, 0,     15018, 2102,  27411, 4123,  38808, 37736,   /*[GB+8234B930, GB+8234B939]*/
  8892,  17140, 40404, 41781, 36491, 15398, 11556, 21848, 549,   9307,    /*[GB+8234BA30, GB+8234BA39]*/
  1590,  43302, 10981, 41782, 28628, 9376,  1402,  7202,  4557,  4323,    /*[GB+8234BB30, GB+8234BB39]*/
  27663, 3582,  26875, 3420,  21075, 23810, 32610, 24150, 8688,  43305,   /*[GB+8234BC30, GB+8234BC39]*/
  24803, 19154, 37723, 21230, 33322, 25174, 6422,  3424,  34152, 11150,   /*[GB+8234BD30, GB+8234BD39]*/
  27692, 9901,  11524, 302,   8692,  25690, 12761, 24039, 36724, 29563,   /*[GB+8234BE30, GB+8234BE39]*/
  33829, 21942, 17059, 9339,  0,     408,   27651, 29386, 19837, 41795,   /*[GB+8234BF30, GB+8234BF39]*/
  12670, 2904,  0,     19233, 22318, 10835, 6832,  14974, 21645, 7782,    /*[GB+8234C030, GB+8234C039]*/
  7511,  3165,  11406, 11412, 39813, 36228, 23716, 1151,  3173,  39320,   /*[GB+8234C130, GB+8234C139]*/
  37873, 36230, 27546, 35929, 18419, 18437, 40172, 7023,  12864, 27957,   /*[GB+8234C230, GB+8234C239]*/
  18014, 24605, 11636, 10469, 2284,  27428, 32928, 29790, 257,   39574,   /*[GB+8234C330, GB+8234C339]*/
  34635, 11278, 37137, 34868, 37340, 0,     1469,  0,     11282, 30919,   /*[GB+8234C430, GB+8234C439]*/
  19558, 0,     43050, 0,     33959, 40628, 38234, 6714,  40086, 15309,   /*[GB+8234C530, GB+8234C539]*/
  0,     2064,  31058, 38943, 0,     37876, 22869, 28296, 1048,  9617,    /*[GB+8234C630, GB+8234C639]*/
  658,   22099, 8881,  7530,  28334, 0,     42454, 37206, 14072, 40173,   /*[GB+8234C730, GB+8234C739]*/
  29419, 31080, 32810, 36397, 42472, 38359, 27022, 28986, 11740, 22648,   /*[GB+8234C830, GB+8234C839]*/
  37480, 14096, 30066, 38957, 829,   11749, 11750, 37637, 0,     37210,   /*[GB+8234C930, GB+8234C939]*/
  18693, 33749, 7532,  6981,  40620, 43058, 38963, 38965, 14881, 4436,    /*[GB+8234CA30, GB+8234CA39]*/
  5589,  0,     16831, 38447, 37653, 13979, 7533,  23175, 4707,  25474,   /*[GB+8234CB30, GB+8234CB39]*/
  36240, 1995,  21528, 29424, 22015, 18822, 25058, 25059, 29026, 8631,    /*[GB+8234CC30, GB+8234CC39]*/
  2079,  1128,  1769,  37998, 32961, 12107, 7740,  24914, 20974, 8781,    /*[GB+8234CD30, GB+8234CD39]*/
  277,   280,   8782,  24935, 26841, 32142, 38322, 2092,  24247, 493,     /*[GB+8234CE30, GB+8234CE39]*/
  0,     36052, 0,     0,     39876, 3730,  19924, 37899, 18438, 0,       /*[GB+8234CF30, GB+8234CF39]*/
  22958, 34013, 33323, 0,     17981, 24645, 43307, 43921, 43403, 3060,    /*[GB+8234D030, GB+8234D039]*/
  258,   7571,  39913, 18022, 8823,  41022, 12727, 4577,  29794, 1164,    /*[GB+8234D130, GB+8234D139]*/
  1463,  41023, 0,     11159, 18525, 0,     10830, 40572, 9746,  3595,    /*[GB+8234D230, GB+8234D239]*/
  19839, 22145, 21605, 20750, 412,   41622, 42915, 12332, 7780,  5877,    /*[GB+8234D330, GB+8234D339]*/
  31633, 0,     6835,  33221, 32956, 13539, 3903,  19173, 1687,  19725,   /*[GB+8234D430, GB+8234D439]*/
  13993, 23119, 20514, 28475, 0,     9899,  23093, 38194, 16691, 32360,   /*[GB+8234D530, GB+8234D539]*/
  32231, 25234, 2093,  16320, 19343, 9037,  9898,  10555, 16885, 11273,   /*[GB+8234D630, GB+8234D639]*/
  16972, 40839, 31456, 17652, 1436,  12219, 24589, 37404, 32467, 31178,   /*[GB+8234D730, GB+8234D739]*/
  30068, 16764, 7037,  371,   14111, 22010, 17114, 16832, 262,   20693,   /*[GB+8234D830, GB+8234D839]*/
  26081, 25659, 0,     16336, 11005, 7214,  34231, 22399, 30220, 14771,   /*[GB+8234D930, GB+8234D939]*/
  24021, 23527, 29686, 3094,  23060, 20773, 29693, 4754,  34264, 17234,   /*[GB+8234DA30, GB+8234DA39]*/
  0,     5959,  24616, 31368, 43843, 33165, 8048,  2356,  23937, 27590,   /*[GB+8234DB30, GB+8234DB39]*/
  0,     27266, 24982, 5178,  23627, 2147,  20793, 43723, 4761,  17115,   /*[GB+8234DC30, GB+8234DC39]*/
  13540, 17522, 0,     21353, 21647, 23631, 19962, 43905, 15246, 19552,   /*[GB+8234DD30, GB+8234DD39]*/
  37983, 33017, 18421, 18439, 40933, 42963, 12718, 28481, 3872,  35526,   /*[GB+8234DE30, GB+8234DE39]*/
  32539, 6574,  12880, 24153, 11639, 21205, 3262,  21207, 3177,  15515,   /*[GB+8234DF30, GB+8234DF39]*/
  23312, 0,     38458, 27333, 19435, 36544, 0,     25143, 43497, 0,       /*[GB+8234E030, GB+8234E039]*/
  753,   9523,  40829, 14775, 39547, 25941, 994,   461,   32151, 37064,   /*[GB+8234E130, GB+8234E139]*/
  25724, 39329, 42474, 14780, 22118, 28609, 28142, 24955, 28143, 21520,   /*[GB+8234E230, GB+8234E239]*/
  9569,  37910, 9464,  9465,  31697, 8433,  36186, 37639, 31495, 31369,   /*[GB+8234E330, GB+8234E339]*/
  22189, 19536, 2383,  18145, 29133, 20167, 12892, 5210,  23564, 19544,   /*[GB+8234E430, GB+8234E439]*/
  15667, 41328, 15504, 41934, 43845, 34592, 42520, 0,     0,     17310,   /*[GB+8234E530, GB+8234E539]*/
  0,     0,     17317, 35769, 9468,  7326,  22209, 42647, 31102, 40120,   /*[GB+8234E630, GB+8234E639]*/
  41211, 9469,  33096, 17655, 29799, 41056, 10560, 11962, 41278, 3066,    /*[GB+8234E730, GB+8234E739]*/
  19840, 3032,  42533, 32952, 36065, 40744, 21857, 10202, 29853, 0,       /*[GB+8234E830, GB+8234E839]*/
  0,     29572, 7265,  2618,  16857, 30991, 22665, 17748, 20015, 37928,   /*[GB+8234E930, GB+8234E939]*/
  35307, 36891, 26072, 24651, 4100,  34401, 10027, 14711, 17432, 21358,   /*[GB+8234EA30, GB+8234EA39]*/
  37662, 3054,  18442, 0,     0,     5392,  6895,  11420, 0,     33980,   /*[GB+8234EB30, GB+8234EB39]*/
  0,     24947, 0,     40099, 13218, 40459, 7904,  9524,  10876, 41850,   /*[GB+8234EC30, GB+8234EC39]*/
  26798, 0,     0,     14776, 24106, 9038,  36055, 28598, 28802, 25749,   /*[GB+8234ED30, GB+8234ED39]*/
  4695,  6412,  2107,  6265,  32417, 4696,  42237, 557,   33395, 827,     /*[GB+8234EE30, GB+8234EE39]*/
  5567,  494,   31626, 0,     9532,  15329, 10323, 4756,  39330, 40110,   /*[GB+8234EF30, GB+8234EF39]*/
  17774, 15494, 8626,  23080, 7317,  7318,  35502, 16255, 36994, 31818,   /*[GB+8234F030, GB+8234F039]*/
  24625, 1111,  21943, 29422, 37004, 24987, 40118, 1740,  0,     36027,   /*[GB+8234F130, GB+8234F139]*/
  11526, 12328, 37250, 15784, 17032, 7494,  13265, 22139, 3977,  833,     /*[GB+8234F230, GB+8234F239]*/
  33502, 41860, 35954, 30450, 3898,  33768, 4884,  20638, 16662, 37015,   /*[GB+8234F330, GB+8234F339]*/
  2420,  0,     11220, 5846,  26885, 31635, 39953, 34753, 22582, 29443,   /*[GB+8234F430, GB+8234F439]*/
  8168,  24696, 19398, 373,   12150, 37930, 12151, 21356, 38840, 17954,   /*[GB+8234F530, GB+8234F539]*/
  37034, 836,   6445,  19436, 9104,  34860, 32289, 4341,  5979,  12115,   /*[GB+8234F630, GB+8234F639]*/
  37398, 18423, 21442, 11634, 28599, 13610, 38607, 32612, 0,     24298,   /*[GB+8234F730, GB+8234F739]*/
  20182, 19399, 38296, 2325,  28125, 11628, 29149, 32079, 22120, 11950,   /*[GB+8234F830, GB+8234F839]*/
  31428, 1908,  23938, 12726, 8435,  10513, 2291,  18157, 3023,  24116,   /*[GB+8234F930, GB+8234F939]*/
  5180,  21344, 30298, 25637, 42158, 16955, 1172,  352,   21357, 34565,   /*[GB+8234FA30, GB+8234FA39]*/
  16749, 31676, 31846, 32831, 34274, 0,     31849, 17781, 3006,  22835,   /*[GB+8234FB30, GB+8234FB39]*/
  22841, 18121, 6595,  15674, 25521, 2151,  27983, 40978, 30923, 25524,   /*[GB+8234FC30, GB+8234FC39]*/
  23273, 38270, 15123, 9000,  5960,  13933, 21203, 5290,  13955, 39916,   /*[GB+8234FD30, GB+8234FD39]*/
  35271, 40591, 20914, 18158, 10393, 36275, 30444, 41664, 37211, 31894,   /*[GB+8234FE30, GB+8234FE39]*/
  28203, 34043, 25693, 32959, 38995, 4318,  26525, 32239, 0,     42084,   /*[GB+82358130, GB+82358139]*/
  31533, 9836,  28204, 2306,  8627,  31430, 30439, 30451, 43387, 10702,   /*[GB+82358230, GB+82358239]*/
  24480, 11157, 43221, 42803, 2157,  19557, 23357, 33739, 23771, 6104,    /*[GB+82358330, GB+82358339]*/
  11026, 31183, 32306, 31185, 24966, 13498, 3738,  527,   14969, 16498,   /*[GB+82358430, GB+82358439]*/
  18527, 15675, 26580, 17322, 9832,  25242, 25007, 34605, 13989, 28810,   /*[GB+82358530, GB+82358539]*/
  13856, 49,    11964, 40981, 40859, 37409, 41224, 13546, 2764,  37037,   /*[GB+82358630, GB+82358639]*/
  13852, 0,     0,     14318, 31632, 22479, 40293, 0,     3742,  0,       /*[GB+82358730, GB+82358739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+82358830, GB+82358839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+82358930, GB+82358939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+82358A30, GB+82358A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+82358B30, GB+82358B39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+82358C30, GB+82358C39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+82358D30, GB+82358D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+82358E30, GB+82358E39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+82358F30, GB+82358F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+82359030, GB+82359039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+82359130, GB+82359139]*/
  0,     0,     28074, 0,     0,     0,     0,     0,     0,     0,       /*[GB+82359230, GB+82359239]*/
  0,     0,     9105,  30334, 20734, 0,     0,     0,     0,     9241,    /*[GB+82359330, GB+82359339]*/
  5445,  0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+82359430, GB+82359439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+82359530, GB+82359539]*/
  0,     0,     403,   31221, 22760, 0,     0,     0,     0,     0,       /*[GB+82359630, GB+82359639]*/
  0,     6564,  42455, 17648, 254,   0,     0,     20735,   /*[GB+82359730, GB+82359737]*/
};


static const uint16 gb18030_2022_4_pinyin_weight_py_p2[] = {
  11007, 24736, 0,     25801,   /*[GB+95328236, GB+95328239]*/
  0,     10573, 0,     0,     0,     26200, 2502,  0,     0,     28852,   /*[GB+95328330, GB+95328339]*/
  0,     0,     0,     0,     0,     29487, 15805, 0,     0,     0,       /*[GB+95328430, GB+95328439]*/
  0,     0,     0,     39702, 0,     16570, 0,     0,     0,     0,       /*[GB+95328530, GB+95328539]*/
  43429, 0,     0,     0,     35461, 0,     0,     0,     0,     0,       /*[GB+95328630, GB+95328639]*/
  0,     0,     0,     0,     0,     22192, 0,     0,     0,     0,       /*[GB+95328730, GB+95328739]*/
  0,     12939, 37537, 16017, 0,     0,     25376, 20029, 0,     0,       /*[GB+95328830, GB+95328839]*/
  0,     6700,  0,     0,     0,     0,     0,     0,     0,     4141,    /*[GB+95328930, GB+95328939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+95328A30, GB+95328A39]*/
  0,     0,     0,     28973, 0,     0,     0,     0,     0,     0,       /*[GB+95328B30, GB+95328B39]*/
  0,     0,     25864, 0,     0,     0,     0,     8852,  0,     0,       /*[GB+95328C30, GB+95328C39]*/
  0,     0,     0,     0,     0,     24845, 0,     0,     0,     0,       /*[GB+95328D30, GB+95328D39]*/
  0,     0,     0,     0,     0,     2958,  0,     0,     0,     0,       /*[GB+95328E30, GB+95328E39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     12059, 0,       /*[GB+95328F30, GB+95328F39]*/
  38012, 0,     14540, 0,     0,     0,     0,     0,     0,     0,       /*[GB+95329030, GB+95329039]*/
  0,     43748, 0,     0,     43546, 0,     0,     0,     0,     0,       /*[GB+95329130, GB+95329139]*/
  0,     0,     0,     40485, 0,     14789, 0,     0,     23566, 0,       /*[GB+95329230, GB+95329239]*/
  6957,  4838,  0,     28526, 0,     0,     0,     0,     0,     0,       /*[GB+95329330, GB+95329339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+95329430, GB+95329439]*/
  12257, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+95329530, GB+95329539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     38619, 0,       /*[GB+95329630, GB+95329639]*/
  0,     0,     0,     0,     0,     0,     0,     37068, 0,     0,       /*[GB+95329730, GB+95329739]*/
  15202, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+95329830, GB+95329839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+95329930, GB+95329939]*/
  0,     28398, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+95329A30, GB+95329A39]*/
  0,     0,     0,     0,     0,     0,     15291, 0,     0,     0,       /*[GB+95329B30, GB+95329B39]*/
  0,     0,     0,     0,     0,     0,     0,     37576, 0,     0,       /*[GB+95329C30, GB+95329C39]*/
  0,     6633,  0,     0,     15800, 14541, 0,     6271,  0,     15801,   /*[GB+95329D30, GB+95329D39]*/
  4463,  0,     0,     0,     19243, 0,     0,     0,     31447, 0,       /*[GB+95329E30, GB+95329E39]*/
  0,     0,     0,     0,     0,     0,     0,     9381,  0,     0,       /*[GB+95329F30, GB+95329F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     36509, 0,       /*[GB+9532A030, GB+9532A039]*/
  0,     37777, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9532A130, GB+9532A139]*/
  0,     0,     0,     0,     0,     32659, 0,     0,     14547, 0,       /*[GB+9532A230, GB+9532A239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     37763, 0,       /*[GB+9532A330, GB+9532A339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     6448,    /*[GB+9532A430, GB+9532A439]*/
  0,     0,     24854, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9532A530, GB+9532A539]*/
  0,     0,     33793, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9532A630, GB+9532A639]*/
  773,   0,     0,     0,     0,     35184, 0,     0,     0,     0,       /*[GB+9532A730, GB+9532A739]*/
  0,     0,     0,     41415, 0,     0,     0,     0,     0,     0,       /*[GB+9532A830, GB+9532A839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9532A930, GB+9532A939]*/
  0,     0,     39023, 0,     0,     0,     35931, 0,     0,     0,       /*[GB+9532AA30, GB+9532AA39]*/
  0,     0,     0,     0,     0,     6450,  0,     5468,  0,     0,       /*[GB+9532AB30, GB+9532AB39]*/
  0,     32841, 0,     0,     0,     9905,  0,     0,     0,     0,       /*[GB+9532AC30, GB+9532AC39]*/
  0,     7797,  0,     0,     0,     0,     22025, 0,     0,     33510,   /*[GB+9532AD30, GB+9532AD39]*/
  36754, 0,     0,     22696, 5513,  0,     0,     0,     0,     0,       /*[GB+9532AE30, GB+9532AE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9532AF30, GB+9532AF39]*/
  0,     0,     0,     0,     0,     5477,  0,     0,     0,     30822,   /*[GB+9532B030, GB+9532B039]*/
  39613, 0,     9476,  0,     0,     0,     0,     16753, 4237,  0,       /*[GB+9532B130, GB+9532B139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9532B230, GB+9532B239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9532B330, GB+9532B339]*/
  0,     26584, 0,     22027, 0,     28664, 0,     8983,  0,     28562,   /*[GB+9532B430, GB+9532B439]*/
  0,     15078, 0,     0,     0,     0,     0,     0,     0,     32048,   /*[GB+9532B530, GB+9532B539]*/
  0,     0,     0,     29009, 0,     0,     0,     0,     22987, 0,       /*[GB+9532B630, GB+9532B639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9532B730, GB+9532B739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9532B830, GB+9532B839]*/
  0,     0,     0,     0,     40429, 9911,  0,     0,     0,     0,       /*[GB+9532B930, GB+9532B939]*/
  0,     0,     34759, 17696, 0,     0,     5521,  30197, 0,     0,       /*[GB+9532BA30, GB+9532BA39]*/
  0,     20916, 37778, 0,     30932, 0,     1255,  0,     0,     0,       /*[GB+9532BB30, GB+9532BB39]*/
  0,     0,     0,     0,     30354, 0,     0,     0,     0,     0,       /*[GB+9532BC30, GB+9532BC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9532BD30, GB+9532BD39]*/
  0,     0,     0,     0,     0,     20551, 0,     0,     0,     0,       /*[GB+9532BE30, GB+9532BE39]*/
  0,     0,     0,     0,     0,     0,     33846, 12682, 5255,  0,       /*[GB+9532BF30, GB+9532BF39]*/
  0,     15589, 0,     6762,  0,     0,     204,   0,     0,     0,       /*[GB+9532C030, GB+9532C039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9532C130, GB+9532C139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9532C230, GB+9532C239]*/
  0,     0,     0,     0,     0,     21183, 0,     0,     26855, 0,       /*[GB+9532C330, GB+9532C339]*/
  28,    39727, 0,     0,     14008, 0,     25097, 0,     0,     0,       /*[GB+9532C430, GB+9532C439]*/
  0,     0,     0,     0,     0,     0,     0,     21871, 0,     0,       /*[GB+9532C530, GB+9532C539]*/
  0,     42757, 42291, 42596, 22587, 1923,  0,     43127, 29225, 36378,   /*[GB+9532C630, GB+9532C639]*/
  15849, 25445, 0,     0,     9915,  0,     31722, 0,     0,     0,       /*[GB+9532C730, GB+9532C739]*/
  38972, 42315, 0,     0,     0,     0,     0,     0,     16989, 0,       /*[GB+9532C830, GB+9532C839]*/
  0,     0,     0,     0,     0,     0,     3430,  0,     0,     0,       /*[GB+9532C930, GB+9532C939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9532CA30, GB+9532CA39]*/
  0,     0,     18746, 36425, 0,     0,     0,     0,     10520, 21726,   /*[GB+9532CB30, GB+9532CB39]*/
  28318, 39633, 0,     29622, 29804, 0,     42316, 0,     0,     25793,   /*[GB+9532CC30, GB+9532CC39]*/
  0,     8732,  0,     0,     15442, 961,   0,     1307,  0,     0,       /*[GB+9532CD30, GB+9532CD39]*/
  30248, 0,     0,     25450, 21949, 2948,  0,     0,     27703, 12967,   /*[GB+9532CE30, GB+9532CE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9532CF30, GB+9532CF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9532D030, GB+9532D039]*/
  0,     0,     0,     0,     0,     9581,  26142, 0,     0,     0,       /*[GB+9532D130, GB+9532D139]*/
  27233, 0,     0,     0,     29808, 29810, 37277, 0,     0,     0,       /*[GB+9532D230, GB+9532D239]*/
  0,     0,     0,     0,     0,     0,     4010,  0,     0,     29308,   /*[GB+9532D330, GB+9532D339]*/
  41516, 18384, 8913,  30028, 41174, 0,     43196, 0,     0,     0,       /*[GB+9532D430, GB+9532D439]*/
  0,     0,     0,     0,     0,     8757,  0,     0,     0,     0,       /*[GB+9532D530, GB+9532D539]*/
  0,     12983, 6650,  0,     0,     0,     0,     0,     0,     0,       /*[GB+9532D630, GB+9532D639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9532D730, GB+9532D739]*/
  0,     12560, 0,     0,     13901, 36813, 42380, 0,     0,     0,       /*[GB+9532D830, GB+9532D839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     21160, 37430,   /*[GB+9532D930, GB+9532D939]*/
  5902,  37848, 0,     0,     0,     1762,  0,     0,     26489, 38272,   /*[GB+9532DA30, GB+9532DA39]*/
  0,     37194, 0,     0,     0,     41003, 0,     0,     0,     27941,   /*[GB+9532DB30, GB+9532DB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9532DC30, GB+9532DC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9532DD30, GB+9532DD39]*/
  0,     0,     0,     0,     0,     0,     0,     17368, 0,     0,       /*[GB+9532DE30, GB+9532DE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9532DF30, GB+9532DF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     15893, 0,       /*[GB+9532E030, GB+9532E039]*/
  24773, 0,     0,     39522, 0,     0,     40567, 27547, 27780, 0,       /*[GB+9532E130, GB+9532E139]*/
  0,     7108,  0,     0,     0,     14723, 16395, 0,     0,     40188,   /*[GB+9532E230, GB+9532E239]*/
  0,     0,     0,     13922, 37311, 0,     0,     0,     0,     0,       /*[GB+9532E330, GB+9532E339]*/
  0,     0,     0,     0,     0,     34249, 0,     34967, 25666, 0,       /*[GB+9532E430, GB+9532E439]*/
  0,     0,     39837, 26495, 0,     0,     0,     0,     0,     34248,   /*[GB+9532E530, GB+9532E539]*/
  20583, 0,     10114, 0,     0,     3479,  4443,  37888, 0,     41953,   /*[GB+9532E630, GB+9532E639]*/
  0,     43738, 7020,  0,     41136, 0,     0,     0,     0,     0,       /*[GB+9532E730, GB+9532E739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     7980,    /*[GB+9532E830, GB+9532E839]*/
  37889, 21286, 0,     0,     0,     0,     0,     0,     1502,  14671,   /*[GB+9532E930, GB+9532E939]*/
  29273, 18914, 1214,  29857, 0,     0,     0,     6095,  0,     0,       /*[GB+9532EA30, GB+9532EA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9532EB30, GB+9532EB39]*/
  0,     0,     0,     999,   33071, 0,     0,     0,     0,     21307,   /*[GB+9532EC30, GB+9532EC39]*/
  0,     2978,  0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9532ED30, GB+9532ED39]*/
  0,     0,     0,     5737,  0,     0,     0,     0,     24379, 13956,   /*[GB+9532EE30, GB+9532EE39]*/
  19224, 10330, 24809, 11331, 0,     0,     0,     13258, 35272, 41932,   /*[GB+9532EF30, GB+9532EF39]*/
  2986,  0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9532F030, GB+9532F039]*/
  37341, 2994,  0,     0,     0,     0,     0,     0,     6133,  4003,    /*[GB+9532F130, GB+9532F139]*/
  22652, 22653, 0,     0,     41152, 1514,  0,     2998,  0,     0,       /*[GB+9532F230, GB+9532F239]*/
  0,     0,     0,     0,     0,     0,     34880, 0,     0,     0,       /*[GB+9532F330, GB+9532F339]*/
  0,     4984,  35358, 15076, 25365, 0,     26047, 0,     0,     0,       /*[GB+9532F430, GB+9532F439]*/
  0,     9750,  0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9532F530, GB+9532F539]*/
  0,     0,     33425, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9532F630, GB+9532F639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9532F730, GB+9532F739]*/
  0,     0,     0,     0,     0,     0,     40132, 1886,  0,     0,       /*[GB+9532F830, GB+9532F839]*/
  0,     0,     32358, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9532F930, GB+9532F939]*/
  0,     0,     0,     0,     0,     0,     22952, 0,     0,     0,       /*[GB+9532FA30, GB+9532FA39]*/
  0,     18767, 0,     0,     26966, 0,     0,     0,     0,     0,       /*[GB+9532FB30, GB+9532FB39]*/
  7846,  5954,  0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9532FC30, GB+9532FC39]*/
  0,     0,     12436, 38401, 34382, 0,     0,     0,     0,     0,       /*[GB+9532FD30, GB+9532FD39]*/
  0,     0,     0,     0,     17514, 8838,  0,     35583, 0,     0,       /*[GB+9532FE30, GB+9532FE39]*/
  18823, 30819, 13102, 0,     0,     0,     14727, 40984, 28665, 0,       /*[GB+95338130, GB+95338139]*/
  0,     0,     0,     0,     24847, 1521,  0,     17533, 19177, 0,       /*[GB+95338230, GB+95338239]*/
  0,     0,     0,     0,     0,     0,     42381, 1331,  28647, 0,       /*[GB+95338330, GB+95338339]*/
  0,     0,     0,     28655, 0,     25959, 0,     0,     1618,  33508,   /*[GB+95338430, GB+95338439]*/
  0,     0,     0,     0,     0,     15731, 0,     12930, 0,     33522,   /*[GB+95338530, GB+95338539]*/
  25963, 0,     0,     10578, 0,     0,     0,     20407, 0,     0,       /*[GB+95338630, GB+95338639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     40238, 0,       /*[GB+95338730, GB+95338739]*/
  0,     0,     0,     0,     0,     0,     18620, 0,     0,     0,       /*[GB+95338830, GB+95338839]*/
  0,     0,     0,     620,   0,     0,     11267, 0,     0,     0,       /*[GB+95338930, GB+95338939]*/
  24806, 0,     0,     0,     25342, 41990, 20992, 0,     0,     0,       /*[GB+95338A30, GB+95338A39]*/
  0,     4837,  0,     0,     22327, 0,     0,     0,     0,     0,       /*[GB+95338B30, GB+95338B39]*/
  31502, 0,     43625, 0,     0,     0,     0,     0,     0,     0,       /*[GB+95338C30, GB+95338C39]*/
  0,     15148, 41459, 0,     0,     0,     0,     0,     0,     0,       /*[GB+95338D30, GB+95338D39]*/
  0,     22883, 3532,  0,     0,     25226, 0,     0,     39794, 14365,   /*[GB+95338E30, GB+95338E39]*/
  0,     0,     0,     0,     0,     41503, 0,     0,     0,     0,       /*[GB+95338F30, GB+95338F39]*/
  0,     5917,  15203, 0,     0,     0,     30061, 0,     0,     37215,   /*[GB+95339030, GB+95339039]*/
  0,     0,     0,     32474, 0,     0,     18875, 0,     31557, 0,       /*[GB+95339130, GB+95339139]*/
  21385, 0,     0,     0,     0,     39290, 0,     0,     0,     0,       /*[GB+95339230, GB+95339239]*/
  0,     0,     0,     29508, 0,     0,     0,     0,     0,     0,       /*[GB+95339330, GB+95339339]*/
  0,     19866, 0,     0,     0,     38517, 0,     0,     0,     0,       /*[GB+95339430, GB+95339439]*/
  0,     3988,  0,     0,     0,     0,     0,     9012,  15201, 0,       /*[GB+95339530, GB+95339539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     25989, 15165,   /*[GB+95339630, GB+95339639]*/
  0,     35019, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+95339730, GB+95339739]*/
  11172, 0,     0,     0,     30824, 0,     26272, 35022, 15085, 22879,   /*[GB+95339830, GB+95339839]*/
  15086, 0,     0,     13125, 0,     0,     0,     0,     0,     0,       /*[GB+95339930, GB+95339939]*/
  0,     0,     31265, 0,     5065,  6350,  0,     26076, 0,     0,       /*[GB+95339A30, GB+95339A39]*/
  0,     0,     0,     0,     0,     24486, 24445, 0,     6361,  19864,   /*[GB+95339B30, GB+95339B39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     18718,   /*[GB+95339C30, GB+95339C39]*/
  10673, 23665, 30718, 0,     0,     0,     37849, 36045, 30303, 19486,   /*[GB+95339D30, GB+95339D39]*/
  29370, 28541, 0,     2264,  29760, 0,     0,     25935, 28507, 0,       /*[GB+95339E30, GB+95339E39]*/
  0,     0,     23277, 31530, 14153, 0,     0,     0,     0,     0,       /*[GB+95339F30, GB+95339F39]*/
  0,     0,     0,     33689, 42478, 0,     0,     0,     0,     0,       /*[GB+9533A030, GB+9533A039]*/
  0,     17413, 18464, 18465, 0,     0,     11153, 14473, 0,     36733,   /*[GB+9533A130, GB+9533A139]*/
  0,     29064, 0,     0,     28814, 0,     0,     0,     0,     0,       /*[GB+9533A230, GB+9533A239]*/
  0,     41746, 0,     39138, 0,     0,     0,     0,     0,     0,       /*[GB+9533A330, GB+9533A339]*/
  30301, 33275, 0,     0,     0,     0,     0,     3072,  4841,  0,       /*[GB+9533A430, GB+9533A439]*/
  0,     0,     15574, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9533A530, GB+9533A539]*/
  29072, 0,     0,     0,     0,     0,     15293, 32636, 0,     0,       /*[GB+9533A630, GB+9533A639]*/
  0,     0,     0,     0,     0,     0,     12819, 0,     0,     0,       /*[GB+9533A730, GB+9533A739]*/
  0,     0,     14515, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9533A830, GB+9533A839]*/
  40733, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9533A930, GB+9533A939]*/
  23482, 0,     0,     0,     0,     0,     9908,  0,     0,     0,       /*[GB+9533AA30, GB+9533AA39]*/
  0,     0,     0,     10900, 18204, 42249, 34479, 0,     0,     2184,    /*[GB+9533AB30, GB+9533AB39]*/
  3131,  0,     0,     0,     0,     0,     0,     0,     0,     40318,   /*[GB+9533AC30, GB+9533AC39]*/
  11074, 0,     0,     0,     0,     0,     0,     0,     30854, 0,       /*[GB+9533AD30, GB+9533AD39]*/
  0,     0,     1673,  0,     0,     0,     0,     6228,  0,     7606,    /*[GB+9533AE30, GB+9533AE39]*/
  15204, 0,     0,     0,     0,     6014,  38019, 17116, 0,     41525,   /*[GB+9533AF30, GB+9533AF39]*/
  0,     17198, 42648, 15562, 0,     0,     0,     28094, 27919, 6230,    /*[GB+9533B030, GB+9533B039]*/
  602,   12922, 0,     0,     42543, 0,     0,     37747, 0,     16608,   /*[GB+9533B130, GB+9533B139]*/
  33225, 0,     0,     0,     9203,  421,   0,     0,     9597,  0,       /*[GB+9533B230, GB+9533B239]*/
  0,     34298, 9866,  19562, 3746,  9934,  4345,  0,     18026, 4931,    /*[GB+9533B330, GB+9533B339]*/
  29287, 0,     0,     0,     0,     0,     0,     1179,  0,     0,       /*[GB+9533B430, GB+9533B439]*/
  0,     1868,  18273, 0,     0,     15265, 31318, 7168,  0,     36564,   /*[GB+9533B530, GB+9533B539]*/
  26607, 0,     0,     19002, 0,     16443, 0,     9369,  41674, 0,       /*[GB+9533B630, GB+9533B639]*/
  8110,  0,     0,     0,     0,     0,     0,     0,     0,     37765,   /*[GB+9533B730, GB+9533B739]*/
  0,     15283, 35831, 14335, 0,     0,     20254, 15266, 0,     4106,    /*[GB+9533B830, GB+9533B839]*/
  0,     0,     34486, 0,     16800, 6999,  0,     0,     0,     0,       /*[GB+9533B930, GB+9533B939]*/
  20450, 33540, 25997, 2243,  0,     0,     0,     0,     0,     0,       /*[GB+9533BA30, GB+9533BA39]*/
  0,     0,     0,     0,     25210, 0,     0,     0,     0,     0,       /*[GB+9533BB30, GB+9533BB39]*/
  0,     0,     0,     0,     0,     0,     0,     23992, 36420, 1062,    /*[GB+9533BC30, GB+9533BC39]*/
  10487, 9873,  0,     0,     0,     15436, 0,     0,     25189, 0,       /*[GB+9533BD30, GB+9533BD39]*/
  15841, 0,     0,     0,     0,     0,     18318, 0,     0,     0,       /*[GB+9533BE30, GB+9533BE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     11898, 14259,   /*[GB+9533BF30, GB+9533BF39]*/
  0,     0,     0,     0,     0,     0,     25214, 0,     40990, 25190,   /*[GB+9533C030, GB+9533C039]*/
  0,     41556, 2658,  38927, 0,     0,     36574, 0,     3918,  0,       /*[GB+9533C130, GB+9533C139]*/
  0,     0,     3764,  0,     32309, 0,     29748, 0,     0,     0,       /*[GB+9533C230, GB+9533C239]*/
  0,     0,     16214, 40054, 0,     0,     0,     0,     0,     0,       /*[GB+9533C330, GB+9533C339]*/
  0,     0,     0,     4038,  0,     30249, 31725, 0,     0,     41564,   /*[GB+9533C430, GB+9533C439]*/
  30999, 0,     33248, 0,     5216,  18368, 2660,  0,     0,     0,       /*[GB+9533C530, GB+9533C539]*/
  0,     0,     0,     0,     0,     0,     0,     27129, 9582,  26768,   /*[GB+9533C630, GB+9533C639]*/
  0,     18050, 0,     0,     0,     0,     30836, 0,     0,     0,       /*[GB+9533C730, GB+9533C739]*/
  0,     0,     18369, 0,     0,     21432, 0,     3843,  0,     0,       /*[GB+9533C830, GB+9533C839]*/
  10384, 19796, 4440,  30254, 14367, 14918, 0,     0,     0,     7656,    /*[GB+9533C930, GB+9533C939]*/
  41102, 0,     0,     0,     0,     0,     0,     0,     14919, 5082,    /*[GB+9533CA30, GB+9533CA39]*/
  0,     0,     2640,  43995, 0,     41491, 0,     0,     24328, 41242,   /*[GB+9533CB30, GB+9533CB39]*/
  33646, 37433, 8676,  3925,  0,     0,     0,     0,     0,     5086,    /*[GB+9533CC30, GB+9533CC39]*/
  0,     0,     0,     9890,  0,     0,     0,     0,     0,     0,       /*[GB+9533CD30, GB+9533CD39]*/
  0,     0,     0,     0,     0,     13000, 0,     0,     27781, 41183,   /*[GB+9533CE30, GB+9533CE39]*/
  19452, 27782, 37590, 0,     0,     0,     0,     0,     15399, 0,       /*[GB+9533CF30, GB+9533CF39]*/
  0,     0,     0,     0,     31738, 0,     0,     0,     27311, 40830,   /*[GB+9533D030, GB+9533D039]*/
  12116, 0,     0,     0,     0,     34439, 0,     25323, 41504, 0,       /*[GB+9533D130, GB+9533D139]*/
  0,     0,     0,     0,     0,     2412,  0,     0,     17227, 18445,   /*[GB+9533D230, GB+9533D239]*/
  27260, 0,     0,     0,     6798,  0,     0,     19060, 0,     0,       /*[GB+9533D330, GB+9533D339]*/
  0,     0,     38802, 18466, 0,     0,     6805,  0,     19369, 0,       /*[GB+9533D430, GB+9533D439]*/
  0,     0,     0,     0,     0,     0,     32325, 0,     0,     0,       /*[GB+9533D530, GB+9533D539]*/
  0,     6435,  0,     0,     0,     15205, 0,     18259, 16665, 16525,   /*[GB+9533D630, GB+9533D639]*/
  0,     41748, 0,     0,     0,     0,     0,     0,     11170, 0,       /*[GB+9533D730, GB+9533D739]*/
  1226,  0,     23970, 0,     0,     0,     0,     0,     10841, 0,       /*[GB+9533D830, GB+9533D839]*/
  0,     0,     0,     0,     0,     43318, 6983,  0,     38072, 0,       /*[GB+9533D930, GB+9533D939]*/
  0,     0,     0,     0,     0,     0,     16444, 38073, 22042, 0,       /*[GB+9533DA30, GB+9533DA39]*/
  3811,  2390,  0,     9419,  16448, 28883, 0,     0,     0,     0,       /*[GB+9533DB30, GB+9533DB39]*/
  0,     0,     0,     19269, 1063,  0,     0,     6931,  0,     0,       /*[GB+9533DC30, GB+9533DC39]*/
  0,     0,     15685, 22588, 43569, 0,     0,     43728, 0,     0,       /*[GB+9533DD30, GB+9533DD39]*/
  0,     0,     0,     0,     0,     30715, 0,     0,     34106, 10769,   /*[GB+9533DE30, GB+9533DE39]*/
  0,     0,     0,     0,     20371, 25388, 0,     0,     0,     0,       /*[GB+9533DF30, GB+9533DF39]*/
  0,     22079, 23450, 10909, 0,     0,     0,     40998, 15758, 4859,    /*[GB+9533E030, GB+9533E039]*/
  0,     0,     0,     0,     18400, 41004, 39317, 6199,  15898, 979,     /*[GB+9533E130, GB+9533E139]*/
  0,     0,     0,     37385, 24329, 0,     0,     0,     0,     0,       /*[GB+9533E230, GB+9533E239]*/
  0,     14939, 16164, 27783, 37171, 14940, 0,     0,     0,     16491,   /*[GB+9533E330, GB+9533E339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9533E430, GB+9533E439]*/
  2831,  0,     0,     22907, 0,     32470, 20305, 0,     0,     0,       /*[GB+9533E530, GB+9533E539]*/
  0,     0,     40343, 37216, 764,   0,     0,     16011, 36032, 0,       /*[GB+9533E630, GB+9533E639]*/
  42653, 0,     0,     0,     0,     0,     0,     0,     17061, 0,       /*[GB+9533E730, GB+9533E739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     26469, 28285,   /*[GB+9533E830, GB+9533E839]*/
  30178, 0,     0,     0,     6743,  0,     16784, 23759, 0,     0,       /*[GB+9533E930, GB+9533E939]*/
  0,     0,     0,     0,     856,   0,     0,     0,     8766,  15311,   /*[GB+9533EA30, GB+9533EA39]*/
  26865, 0,     0,     0,     15407, 0,     0,     0,     0,     0,       /*[GB+9533EB30, GB+9533EB39]*/
  26182, 0,     0,     42712, 11965, 797,   0,     0,     0,     0,       /*[GB+9533EC30, GB+9533EC39]*/
  0,     37754, 0,     37779, 37780, 0,     0,     0,     0,     21021,   /*[GB+9533ED30, GB+9533ED39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     27397, 0,       /*[GB+9533EE30, GB+9533EE39]*/
  0,     0,     0,     4701,  0,     0,     10641, 0,     4839,  0,       /*[GB+9533EF30, GB+9533EF39]*/
  0,     0,     33908, 0,     0,     0,     0,     26632, 31335, 0,       /*[GB+9533F030, GB+9533F039]*/
  6289,  0,     10650, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9533F130, GB+9533F139]*/
  0,     37512, 0,     0,     0,     0,     0,     7269,  32641, 0,       /*[GB+9533F230, GB+9533F239]*/
  2475,  6303,  0,     0,     0,     7396,  6064,  0,     30003, 26676,   /*[GB+9533F330, GB+9533F339]*/
  0,     0,     7445,  23456, 36049, 0,     0,     0,     33392, 0,       /*[GB+9533F430, GB+9533F439]*/
  0,     0,     38402, 0,     22191, 0,     0,     0,     0,     0,       /*[GB+9533F530, GB+9533F539]*/
  0,     0,     11570, 0,     0,     0,     10651, 0,     0,     0,       /*[GB+9533F630, GB+9533F639]*/
  0,     0,     0,     0,     0,     0,     0,     28744, 0,     0,       /*[GB+9533F730, GB+9533F739]*/
  0,     27528, 0,     0,     0,     0,     1221,  0,     10642, 15077,   /*[GB+9533F830, GB+9533F839]*/
  33905, 0,     25990, 5157,  4722,  0,     603,   0,     0,     0,       /*[GB+9533F930, GB+9533F939]*/
  0,     6958,  0,     0,     0,     0,     33912, 0,     0,     0,       /*[GB+9533FA30, GB+9533FA39]*/
  0,     0,     0,     0,     0,     0,     0,     42025, 20358, 0,       /*[GB+9533FB30, GB+9533FB39]*/
  11589, 13148, 9938,  0,     0,     0,     0,     0,     23666, 0,       /*[GB+9533FC30, GB+9533FC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9533FD30, GB+9533FD39]*/
  0,     0,     42846, 0,     0,     0,     0,     1215,  0,     39557,   /*[GB+9533FE30, GB+9533FE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+95348130, GB+95348139]*/
  25014, 0,     0,     11030, 4209,  0,     0,     28279, 3800,  1999,    /*[GB+95348230, GB+95348239]*/
  0,     0,     0,     27088, 39198, 0,     0,     0,     22412, 0,       /*[GB+95348330, GB+95348339]*/
  0,     0,     0,     12521, 31340, 647,   0,     0,     0,     0,       /*[GB+95348430, GB+95348439]*/
  35689, 0,     0,     0,     39210, 3820,  0,     0,     0,     0,       /*[GB+95348530, GB+95348539]*/
  0,     0,     0,     0,     0,     11264, 0,     0,     0,     32298,   /*[GB+95348630, GB+95348639]*/
  0,     0,     35206, 0,     0,     0,     15908, 0,     30047, 26016,   /*[GB+95348730, GB+95348739]*/
  43125, 0,     0,     0,     0,     0,     0,     13332, 0,     0,       /*[GB+95348830, GB+95348839]*/
  1227,  0,     33541, 0,     0,     0,     0,     0,     13159, 0,       /*[GB+95348930, GB+95348939]*/
  16074, 0,     0,     18886, 39162, 0,     0,     0,     0,     0,       /*[GB+95348A30, GB+95348A39]*/
  0,     15423, 0,     0,     0,     0,     40199, 0,     0,     729,     /*[GB+95348B30, GB+95348B39]*/
  0,     0,     24043, 0,     0,     40864, 0,     0,     38031, 5981,    /*[GB+95348C30, GB+95348C39]*/
  0,     0,     25804, 0,     30233, 0,     4726,  0,     42972, 40204,   /*[GB+95348D30, GB+95348D39]*/
  0,     0,     0,     0,     0,     14248, 0,     0,     0,     0,       /*[GB+95348E30, GB+95348E39]*/
  28757, 0,     0,     37781, 34081, 0,     0,     0,     0,     0,       /*[GB+95348F30, GB+95348F39]*/
  0,     0,     40041, 0,     0,     0,     0,     10441, 0,     16452,   /*[GB+95349030, GB+95349039]*/
  0,     0,     5066,  38137, 0,     0,     0,     0,     0,     0,       /*[GB+95349130, GB+95349139]*/
  0,     0,     0,     18319, 0,     6149,  0,     0,     33571, 0,       /*[GB+95349230, GB+95349239]*/
  0,     0,     0,     1333,  0,     0,     1526,  21100, 18344, 0,       /*[GB+95349330, GB+95349339]*/
  0,     29807, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+95349430, GB+95349439]*/
  19479, 10295, 16471, 0,     0,     0,     0,     37850, 0,     33866,   /*[GB+95349530, GB+95349539]*/
  38584, 0,     0,     0,     0,     16477, 0,     0,     0,     28363,   /*[GB+95349630, GB+95349639]*/
  0,     0,     0,     33158, 0,     0,     0,     0,     0,     0,       /*[GB+95349730, GB+95349739]*/
  24186, 0,     0,     0,     0,     0,     0,     0,     40262, 11421,   /*[GB+95349830, GB+95349839]*/
  0,     0,     18425, 8798,  0,     0,     0,     0,     0,     0,       /*[GB+95349930, GB+95349939]*/
  0,     15947, 34266, 0,     0,     0,     0,     6134,  0,     0,       /*[GB+95349A30, GB+95349A39]*/
  18563, 0,     0,     0,     0,     0,     0,     31707, 0,     0,       /*[GB+95349B30, GB+95349B39]*/
  0,     0,     13713, 0,     0,     558,   6015,  41384, 0,     0,       /*[GB+95349C30, GB+95349C39]*/
  0,     0,     0,     39700, 0,     0,     0,     0,     6994,  0,       /*[GB+95349D30, GB+95349D39]*/
  0,     0,     0,     2362,  31748, 0,     0,     0,     0,     0,       /*[GB+95349E30, GB+95349E39]*/
  0,     0,     30536, 13127, 24848, 28101, 22864, 0,     0,     0,       /*[GB+95349F30, GB+95349F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     10041, 0,       /*[GB+9534A030, GB+9534A039]*/
  1202,  0,     0,     35424, 0,     0,     0,     0,     0,     0,       /*[GB+9534A130, GB+9534A139]*/
  0,     41788, 0,     0,     0,     0,     0,     27514, 0,     22024,   /*[GB+9534A230, GB+9534A239]*/
  0,     8445,  0,     0,     0,     0,     30820, 728,   38021, 0,       /*[GB+9534A330, GB+9534A339]*/
  0,     0,     0,     0,     1707,  0,     33513, 0,     14550, 0,       /*[GB+9534A430, GB+9534A439]*/
  0,     0,     0,     0,     0,     0,     14885, 0,     0,     0,       /*[GB+9534A530, GB+9534A539]*/
  0,     0,     0,     0,     25263, 0,     0,     0,     0,     0,       /*[GB+9534A630, GB+9534A639]*/
  0,     0,     0,     0,     29608, 15052, 0,     3756,  0,     0,       /*[GB+9534A730, GB+9534A739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     15055,   /*[GB+9534A830, GB+9534A839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9534A930, GB+9534A939]*/
  0,     0,     0,     30086, 0,     0,     0,     0,     0,     0,       /*[GB+9534AA30, GB+9534AA39]*/
  0,     0,     0,     40951, 18060, 0,     0,     0,     43300, 0,       /*[GB+9534AB30, GB+9534AB39]*/
  0,     0,     0,     1585,  0,     0,     0,     0,     0,     32018,   /*[GB+9534AC30, GB+9534AC39]*/
  0,     0,     0,     1380,  0,     0,     8031,  0,     0,     0,       /*[GB+9534AD30, GB+9534AD39]*/
  5810,  0,     42899, 0,     0,     0,     0,     15416, 0,     0,       /*[GB+9534AE30, GB+9534AE39]*/
  0,     0,     0,     0,     0,     37937, 0,     0,     36496, 0,       /*[GB+9534AF30, GB+9534AF39]*/
  0,     3796,  9909,  42164, 0,     0,     0,     0,     0,     0,       /*[GB+9534B030, GB+9534B039]*/
  27081, 0,     0,     39140, 0,     1997,  0,     13283, 24393, 0,       /*[GB+9534B130, GB+9534B139]*/
  0,     38737, 37039, 20940, 0,     0,     0,     0,     0,     0,       /*[GB+9534B230, GB+9534B239]*/
  0,     19754, 22300, 27576, 4344,  0,     4651,  33341, 27031, 0,       /*[GB+9534B330, GB+9534B339]*/
  0,     0,     5340,  13107, 0,     37942, 0,     26848, 0,     0,       /*[GB+9534B430, GB+9534B439]*/
  12802, 9867,  0,     41529, 24157, 0,     0,     40487, 664,   14556,   /*[GB+9534B530, GB+9534B539]*/
  0,     0,     11474, 0,     0,     34481, 12375, 0,     0,     0,       /*[GB+9534B630, GB+9534B639]*/
  0,     0,     0,     0,     0,     40938, 5325,  9207,  0,     24158,   /*[GB+9534B730, GB+9534B739]*/
  0,     24261, 28751, 18827, 40205, 11806, 1552,  0,     0,     27085,   /*[GB+9534B830, GB+9534B839]*/
  0,     27086, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9534B930, GB+9534B939]*/
  0,     0,     37674, 41997, 0,     14793, 33076, 3158,  0,     26205,   /*[GB+9534BA30, GB+9534BA39]*/
  0,     42931, 0,     40985, 24536, 221,   0,     11037, 0,     4136,    /*[GB+9534BB30, GB+9534BB39]*/
  36633, 0,     0,     0,     28874, 11815, 7358,  0,     0,     0,       /*[GB+9534BC30, GB+9534BC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9534BD30, GB+9534BD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9534BE30, GB+9534BE39]*/
  28758, 0,     0,     19890, 0,     32054, 5345,  32295, 24491, 27198,   /*[GB+9534BF30, GB+9534BF39]*/
  15344, 0,     1944,  0,     0,     0,     0,     0,     0,     0,       /*[GB+9534C030, GB+9534C039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     39625, 6642,    /*[GB+9534C130, GB+9534C139]*/
  0,     10219, 29030, 0,     0,     30198, 22800, 42655, 19757, 1875,    /*[GB+9534C230, GB+9534C239]*/
  44002, 37539, 26896, 0,     19259, 27534, 0,     0,     17900, 7365,    /*[GB+9534C330, GB+9534C339]*/
  0,     42620, 13292, 0,     0,     7366,  0,     0,     0,     44058,   /*[GB+9534C430, GB+9534C439]*/
  0,     0,     22342, 40392, 0,     0,     0,     0,     0,     0,       /*[GB+9534C530, GB+9534C539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9534C630, GB+9534C639]*/
  0,     0,     0,     0,     0,     0,     35154, 43928, 29213, 6526,    /*[GB+9534C730, GB+9534C739]*/
  7682,  27011, 0,     693,   10652, 11357, 37680, 0,     37681, 0,       /*[GB+9534C830, GB+9534C839]*/
  0,     16356, 38105, 12456, 41870, 0,     0,     0,     0,     0,       /*[GB+9534C930, GB+9534C939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9534CA30, GB+9534CA39]*/
  0,     0,     15093, 0,     0,     9265,  0,     0,     23281, 25814,   /*[GB+9534CB30, GB+9534CB39]*/
  0,     6351,  0,     13375, 38139, 37804, 0,     8499,  29450, 29451,   /*[GB+9534CC30, GB+9534CC39]*/
  39086, 16509, 11927, 11449, 0,     0,     0,     11061, 0,     0,       /*[GB+9534CD30, GB+9534CD39]*/
  11015, 25441, 25215, 0,     29616, 0,     696,   0,     0,     0,       /*[GB+9534CE30, GB+9534CE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9534CF30, GB+9534CF39]*/
  15842, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9534D030, GB+9534D039]*/
  0,     0,     0,     0,     14989, 16453, 0,     0,     0,     0,       /*[GB+9534D130, GB+9534D139]*/
  0,     12,    19778, 0,     31717, 0,     0,     4423,  0,     0,       /*[GB+9534D230, GB+9534D239]*/
  29732, 3524,  0,     32563, 0,     0,     23327, 0,     15269, 0,       /*[GB+9534D330, GB+9534D339]*/
  0,     1791,  0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9534D430, GB+9534D439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9534D530, GB+9534D539]*/
  0,     0,     0,     0,     0,     0,     0,     34945, 28580, 11533,   /*[GB+9534D630, GB+9534D639]*/
  0,     0,     42998, 0,     9974,  12957, 0,     13388, 0,     33573,   /*[GB+9534D730, GB+9534D739]*/
  0,     28344, 23489, 11659, 30364, 34896, 0,     40756, 0,     0,       /*[GB+9534D830, GB+9534D839]*/
  2171,  25107, 32219, 32064, 5748,  0,     22355, 0,     0,     0,       /*[GB+9534D930, GB+9534D939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9534DA30, GB+9534DA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9534DB30, GB+9534DB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9534DC30, GB+9534DC39]*/
  41098, 0,     0,     36522, 0,     0,     33363, 41744, 5816,  11017,   /*[GB+9534DD30, GB+9534DD39]*/
  0,     264,   24068, 27765, 8173,  9878,  0,     0,     0,     24543,   /*[GB+9534DE30, GB+9534DE39]*/
  0,     0,     0,     36103, 10617, 9078,  451,   43632, 21387, 0,       /*[GB+9534DF30, GB+9534DF39]*/
  12821, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9534E030, GB+9534E039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9534E130, GB+9534E139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9534E230, GB+9534E239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9534E330, GB+9534E339]*/
  0,     6109,  33574, 0,     0,     5312,  22533, 0,     0,     0,       /*[GB+9534E430, GB+9534E439]*/
  6244,  20561, 16459, 0,     0,     0,     0,     38196, 15870, 11094,   /*[GB+9534E530, GB+9534E539]*/
  0,     13161, 0,     0,     11180, 23020, 27469, 25389, 5378,  28235,   /*[GB+9534E630, GB+9534E639]*/
  16467, 42887, 0,     28687, 20262, 13586, 24233, 11549, 12969, 30373,   /*[GB+9534E730, GB+9534E739]*/
  4047,  33137, 15108, 24505, 41100, 35339, 0,     0,     1575,  0,       /*[GB+9534E830, GB+9534E839]*/
  0,     35874, 0,     0,     0,     0,     9716,  14760, 0,     0,       /*[GB+9534E930, GB+9534E939]*/
  34341, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9534EA30, GB+9534EA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9534EB30, GB+9534EB39]*/
  0,     0,     7294,  0,     2047,  24269, 0,     0,     0,     40544,   /*[GB+9534EC30, GB+9534EC39]*/
  0,     23575, 31726, 0,     38752, 0,     0,     0,     0,     0,       /*[GB+9534ED30, GB+9534ED39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9534EE30, GB+9534EE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9534EF30, GB+9534EF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9534F030, GB+9534F039]*/
  0,     0,     0,     0,     0,     0,     34680, 0,     0,     23410,   /*[GB+9534F130, GB+9534F139]*/
  9227,  2048,  35183, 0,     0,     0,     41702, 39502, 22818, 0,       /*[GB+9534F230, GB+9534F239]*/
  0,     0,     0,     0,     36385, 32270, 0,     287,   10777, 11306,   /*[GB+9534F330, GB+9534F339]*/
  5446,  0,     23385, 0,     2541,  13176, 22691, 39104, 34806, 0,       /*[GB+9534F430, GB+9534F439]*/
  4319,  37431, 0,     9291,  30724, 857,   2953,  0,     35892, 0,       /*[GB+9534F530, GB+9534F539]*/
  0,     10549, 0,     4053,  0,     13902, 44010, 0,     0,     0,       /*[GB+9534F630, GB+9534F639]*/
  32878, 5220,  24008, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9534F730, GB+9534F739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9534F830, GB+9534F839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9534F930, GB+9534F939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9534FA30, GB+9534FA39]*/
  0,     0,     0,     0,     12196, 0,     33624, 0,     23855, 19457,   /*[GB+9534FB30, GB+9534FB39]*/
  22234, 21755, 17678, 31965, 615,   0,     0,     9233,  0,     16739,   /*[GB+9534FC30, GB+9534FC39]*/
  0,     0,     13593, 1972,  0,     0,     0,     0,     0,     0,       /*[GB+9534FD30, GB+9534FD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9534FE30, GB+9534FE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+95358130, GB+95358139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+95358230, GB+95358239]*/
  0,     0,     0,     0,     0,     12197, 0,     43816, 20572, 0,       /*[GB+95358330, GB+95358339]*/
  0,     0,     0,     0,     0,     18082, 0,     11122, 21977, 0,       /*[GB+95358430, GB+95358439]*/
  29416, 28450, 16303, 3851,  19329, 20529, 0,     0,     36824, 41492,   /*[GB+95358530, GB+95358539]*/
  4321,  9798,  25977, 0,     30585, 8193,  31790, 0,     0,     0,       /*[GB+95358630, GB+95358639]*/
  0,     19330, 0,     0,     17666, 0,     0,     0,     0,     0,       /*[GB+95358730, GB+95358739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+95358830, GB+95358839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+95358930, GB+95358939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     17455, 40630,   /*[GB+95358A30, GB+95358A39]*/
  33468, 0,     0,     0,     0,     18083, 4,     20379, 42225, 4063,    /*[GB+95358B30, GB+95358B39]*/
  14219, 0,     13781, 0,     0,     0,     0,     20426, 37866, 0,       /*[GB+95358C30, GB+95358C39]*/
  28182, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+95358D30, GB+95358D39]*/
  0,     0,     0,     0,     0,     13001, 0,     0,     0,     0,       /*[GB+95358E30, GB+95358E39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+95358F30, GB+95358F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+95359030, GB+95359039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     38274, 23126,   /*[GB+95359130, GB+95359139]*/
  0,     0,     0,     12592, 0,     40963, 0,     0,     10718, 0,       /*[GB+95359230, GB+95359239]*/
  38652, 1381,  177,   34063, 22732, 0,     0,     0,     5909,  13923,   /*[GB+95359330, GB+95359339]*/
  23609, 0,     0,     39839, 4333,  40499, 0,     2713,  0,     41583,   /*[GB+95359430, GB+95359439]*/
  27784, 0,     23844, 9728,  41584, 0,     0,     0,     0,     0,       /*[GB+95359530, GB+95359539]*/
  18096, 6657,  0,     4067,  0,     43935, 24582, 0,     0,     28309,   /*[GB+95359630, GB+95359639]*/
  19661, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+95359730, GB+95359739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     29258, 0,       /*[GB+95359830, GB+95359839]*/
  14941, 19339, 0,     0,     16278, 17290, 0,     35737, 14942, 0,       /*[GB+95359930, GB+95359939]*/
  4352,  0,     0,     0,     40263, 0,     0,     0,     0,     0,       /*[GB+95359A30, GB+95359A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+95359B30, GB+95359B39]*/
  0,     0,     0,     0,     20719, 35236, 15241, 13469, 0,     40268,   /*[GB+95359C30, GB+95359C39]*/
  0,     0,     0,     13804, 0,     10694, 0,     27551, 12607, 25776,   /*[GB+95359D30, GB+95359D39]*/
  0,     27793, 43937, 0,     0,     19990, 12010, 4147,  28087, 33203,   /*[GB+95359E30, GB+95359E39]*/
  13220, 43378, 34359, 37715, 10466, 17093, 0,     0,     0,     0,       /*[GB+95359F30, GB+95359F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9535A030, GB+9535A039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9535A130, GB+9535A139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9535A230, GB+9535A239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     42698,   /*[GB+9535A330, GB+9535A339]*/
  0,     0,     20094, 0,     1984,  28800, 38982, 16704, 0,     0,       /*[GB+9535A430, GB+9535A439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9535A530, GB+9535A539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9535A630, GB+9535A639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     42456,   /*[GB+9535A730, GB+9535A739]*/
  35084, 0,     0,     37618, 7473,  20286, 10815, 37619, 0,     0,       /*[GB+9535A830, GB+9535A839]*/
  0,     0,     20590, 4615,  7848,  42139, 38986, 33054, 32234, 121,     /*[GB+9535A930, GB+9535A939]*/
  39558, 0,     0,     11911, 0,     19061, 15020, 40528, 0,     0,       /*[GB+9535AA30, GB+9535AA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9535AB30, GB+9535AB39]*/
  0,     0,     0,     0,     0,     40679, 7024,  0,     13486, 0,       /*[GB+9535AC30, GB+9535AC39]*/
  0,     0,     33146, 0,     13235, 33711, 0,     41321, 31872, 0,       /*[GB+9535AD30, GB+9535AD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9535AE30, GB+9535AE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9535AF30, GB+9535AF39]*/
  39559, 0,     0,     0,     0,     19071, 0,     0,     42497, 28729,   /*[GB+9535B030, GB+9535B039]*/
  0,     17776, 17414, 32821, 23746, 3736,  38952, 6725,  0,     6726,    /*[GB+9535B130, GB+9535B139]*/
  0,     875,   25855, 29277, 0,     42143, 0,     0,     0,     0,       /*[GB+9535B230, GB+9535B239]*/
  0,     19072, 0,     23930, 0,     41522, 0,     0,     0,     0,       /*[GB+9535B330, GB+9535B339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9535B430, GB+9535B439]*/
  0,     0,     0,     23469, 0,     0,     35262, 13250, 17415, 38893,   /*[GB+9535B530, GB+9535B539]*/
  2623,  0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9535B630, GB+9535B639]*/
  0,     0,     0,     0,     0,     0,     17806, 0,     0,     0,       /*[GB+9535B730, GB+9535B739]*/
  0,     0,     0,     20448, 0,     0,     0,     19675, 34020, 0,       /*[GB+9535B830, GB+9535B839]*/
  0,     0,     19225, 0,     0,     10196, 0,     0,     0,     0,       /*[GB+9535B930, GB+9535B939]*/
  0,     0,     0,     0,     0,     35454, 0,     0,     0,     18148,   /*[GB+9535BA30, GB+9535BA39]*/
  4648,  0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9535BB30, GB+9535BB39]*/
  0,     0,     0,     0,     0,     26091, 0,     0,     0,     0,       /*[GB+9535BC30, GB+9535BC39]*/
  0,     0,     0,     0,     0,     13836, 5781,  13964, 26116, 35288,   /*[GB+9535BD30, GB+9535BD39]*/
  38991, 0,     0,     0,     0,     0,     0,     0,     0,     10532,   /*[GB+9535BE30, GB+9535BE39]*/
  0,     41668, 28313, 21466, 12139, 0,     0,     0,     0,     0,       /*[GB+9535BF30, GB+9535BF39]*/
  0,     0,     0,     0,     0,     0,     0,     4982,  27328, 27523,   /*[GB+9535C030, GB+9535C039]*/
  12898, 37733, 0,     5916,  0,     20365, 0,     38430, 0,     0,       /*[GB+9535C130, GB+9535C139]*/
  0,     0,     0,     0,     2180,  23698, 30626, 7327,  40683, 4877,    /*[GB+9535C230, GB+9535C239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9535C330, GB+9535C339]*/
  0,     41113, 0,     33896, 20938, 0,     17323, 40481, 0,     0,       /*[GB+9535C430, GB+9535C439]*/
  0,     0,     0,     0,     0,     7497,  0,     0,     0,     6430,    /*[GB+9535C530, GB+9535C539]*/
  0,     0,     0,     0,     0,     0,     0,     9998,  0,     0,       /*[GB+9535C630, GB+9535C639]*/
  0,     12226, 0,     0,     0,     28999, 13981, 0,     0,     41218,   /*[GB+9535C730, GB+9535C739]*/
  13273, 12227, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9535C830, GB+9535C839]*/
  0,     32471, 20545, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9535C930, GB+9535C939]*/
  0,     6736,  0,     0,     0,     0,     0,     18790, 0,     0,       /*[GB+9535CA30, GB+9535CA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9535CB30, GB+9535CB39]*/
  0,     22989, 22481, 15278, 40200, 0,     0,     37218, 4403,  0,       /*[GB+9535CC30, GB+9535CC39]*/
  0,     0,     2424,  18205, 7083,  22482, 22483, 0,     0,     0,       /*[GB+9535CD30, GB+9535CD39]*/
  0,     0,     27099, 0,     0,     0,     0,     40206, 0,     0,       /*[GB+9535CE30, GB+9535CE39]*/
  39189, 0,     38472, 0,     10437, 0,     0,     0,     5651,  0,       /*[GB+9535CF30, GB+9535CF39]*/
  0,     0,     0,     0,     0,     0,     0,     41675, 21415, 6340,    /*[GB+9535D030, GB+9535D039]*/
  0,     0,     41676, 0,     0,     0,     16754, 0,     10653, 29729,   /*[GB+9535D130, GB+9535D139]*/
  11046, 12941, 41544, 0,     0,     0,     1881,  0,     32637, 31655,   /*[GB+9535D230, GB+9535D239]*/
  0,     31751, 0,     0,     9075,  19779, 26641, 12739, 43113, 26763,   /*[GB+9535D330, GB+9535D339]*/
  0,     11384, 0,     5658,  11077, 30518, 9939,  0,     39740, 0,       /*[GB+9535D430, GB+9535D439]*/
  36535, 0,     0,     32312, 26829, 0,     0,     0,     0,     15871,   /*[GB+9535D530, GB+9535D539]*/
  23449, 0,     0,     26617, 42117, 0,     0,     19313, 33253, 35392,   /*[GB+9535D630, GB+9535D639]*/
  5275,  0,     40001, 40161, 0,     0,     0,     22091, 0,     39245,   /*[GB+9535D730, GB+9535D739]*/
  0,     32317, 0,     0,     33376, 0,     41351, 0,     0,     35996,   /*[GB+9535D830, GB+9535D839]*/
  0,     0,     26965, 10368, 39840, 0,     0,     0,     0,     0,       /*[GB+9535D930, GB+9535D939]*/
  34143, 1541,  39266, 0,     38532, 0,     36060, 39274, 17868, 0,       /*[GB+9535DA30, GB+9535DA39]*/
  0,     0,     0,     0,     31501, 0,     0,     41669, 0,     0,       /*[GB+9535DB30, GB+9535DB39]*/
  0,     0,     40573, 8842,  17258, 0,     0,     26751, 0,     0,       /*[GB+9535DC30, GB+9535DC39]*/
  0,     0,     15424, 0,     4497,  5211,  31997, 8,     0,     0,       /*[GB+9535DD30, GB+9535DD39]*/
  0,     43550, 0,     0,     12255, 38049, 0,     0,     0,     0,       /*[GB+9535DE30, GB+9535DE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     839,     /*[GB+9535DF30, GB+9535DF39]*/
  3682,  0,     0,     27100, 0,     0,     0,     0,     0,     0,       /*[GB+9535E030, GB+9535E039]*/
  19887, 0,     0,     14557, 28867, 0,     43877, 0,     0,     0,       /*[GB+9535E130, GB+9535E139]*/
  0,     0,     0,     38050, 0,     0,     8179,  8180,  0,     0,       /*[GB+9535E230, GB+9535E239]*/
  0,     22030, 0,     0,     0,     29206, 0,     0,     0,     33845,   /*[GB+9535E330, GB+9535E339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     307,     /*[GB+9535E430, GB+9535E439]*/
  0,     24134, 24427, 24494, 13620, 42716, 0,     26207, 0,     0,       /*[GB+9535E530, GB+9535E539]*/
  0,     39288, 30567, 0,     0,     27199, 0,     21534, 0,     0,       /*[GB+9535E630, GB+9535E639]*/
  0,     0,     0,     0,     0,     0,     38076, 0,     27202, 0,       /*[GB+9535E730, GB+9535E739]*/
  0,     19006, 26127, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9535E830, GB+9535E839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9535E930, GB+9535E939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9535EA30, GB+9535EA39]*/
  0,     12457, 13354, 0,     0,     0,     0,     9183,  0,     0,       /*[GB+9535EB30, GB+9535EB39]*/
  0,     0,     14894, 0,     0,     39204, 2651,  5791,  38476, 0,       /*[GB+9535EC30, GB+9535EC39]*/
  39719, 943,   0,     2019,  0,     0,     0,     0,     0,     0,       /*[GB+9535ED30, GB+9535ED39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9535EE30, GB+9535EE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9535EF30, GB+9535EF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9535F030, GB+9535F039]*/
  25641, 0,     0,     0,     0,     0,     2729,  0,     35328, 0,       /*[GB+9535F130, GB+9535F139]*/
  3696,  0,     0,     0,     0,     40751, 16986, 0,     8008,  0,       /*[GB+9535F230, GB+9535F239]*/
  0,     30318, 9961,  0,     0,     7124,  0,     0,     0,     0,       /*[GB+9535F330, GB+9535F339]*/
  10150, 0,     42102, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9535F430, GB+9535F439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9535F530, GB+9535F539]*/
  0,     0,     0,     0,     0,     0,     2940,  22593, 0,     0,       /*[GB+9535F630, GB+9535F639]*/
  11660, 0,     30883, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9535F730, GB+9535F739]*/
  0,     0,     0,     0,     0,     0,     0,     38166, 0,     0,       /*[GB+9535F830, GB+9535F839]*/
  23021, 41141, 12101, 0,     6879,  0,     25114, 2569,  0,     0,       /*[GB+9535F930, GB+9535F939]*/
  4472,  0,     0,     0,     5218,  0,     0,     0,     42352, 9454,    /*[GB+9535FA30, GB+9535FA39]*/
  0,     0,     0,     33118, 0,     0,     0,     0,     0,     0,       /*[GB+9535FB30, GB+9535FB39]*/
  0,     0,     0,     0,     0,     0,     0,     6880,  0,     0,       /*[GB+9535FC30, GB+9535FC39]*/
  0,     3704,  0,     0,     0,     238,   17208, 0,     33182, 0,       /*[GB+9535FD30, GB+9535FD39]*/
  0,     0,     38753, 0,     24257, 0,     0,     0,     0,     0,       /*[GB+9535FE30, GB+9535FE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+95368130, GB+95368139]*/
  0,     40996, 43075, 0,     29900, 0,     0,     0,     0,     22819,   /*[GB+95368230, GB+95368239]*/
  0,     0,     0,     0,     0,     0,     42888, 2850,  0,     0,       /*[GB+95368330, GB+95368339]*/
  1116,  22725, 42118, 12562, 0,     0,     0,     0,     0,     0,       /*[GB+95368430, GB+95368439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+95368530, GB+95368539]*/
  0,     0,     0,     0,     0,     0,     34177, 0,     42382, 33625,   /*[GB+95368630, GB+95368639]*/
  0,     0,     14182, 0,     0,     0,     0,     0,     0,     0,       /*[GB+95368730, GB+95368739]*/
  0,     0,     0,     6967,  8406,  0,     0,     14279, 3185,  588,     /*[GB+95368830, GB+95368839]*/
  0,     0,     0,     0,     0,     19050, 0,     0,     0,     0,       /*[GB+95368930, GB+95368939]*/
  0,     0,     322,   0,     0,     0,     0,     0,     0,     0,       /*[GB+95368A30, GB+95368A39]*/
  0,     0,     0,     0,     0,     0,     0,     40704, 4285,  0,       /*[GB+95368B30, GB+95368B39]*/
  0,     32179, 0,     0,     10914, 16272, 38585, 0,     34550, 0,       /*[GB+95368C30, GB+95368C39]*/
  0,     0,     0,     0,     0,     8767,  1729,  0,     16968, 0,       /*[GB+95368D30, GB+95368D39]*/
  0,     0,     25846, 27548, 0,     0,     0,     0,     0,     0,       /*[GB+95368E30, GB+95368E39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+95368F30, GB+95368F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     5230,    /*[GB+95369030, GB+95369039]*/
  37518, 0,     0,     0,     41390, 0,     18810, 0,     6986,  0,       /*[GB+95369130, GB+95369139]*/
  0,     0,     0,     17721, 36255, 0,     0,     0,     0,     0,       /*[GB+95369230, GB+95369239]*/
  0,     0,     0,     0,     42429, 0,     16702, 0,     0,     0,       /*[GB+95369330, GB+95369339]*/
  30114, 33145, 0,     0,     0,     16705, 0,     0,     0,     0,       /*[GB+95369430, GB+95369439]*/
  0,     0,     0,     0,     0,     0,     0,     13927, 0,     0,       /*[GB+95369530, GB+95369539]*/
  0,     0,     0,     0,     14190, 0,     0,     0,     0,     43254,   /*[GB+95369630, GB+95369639]*/
  0,     29376, 39548, 0,     0,     27552, 0,     39862, 17549, 0,       /*[GB+95369730, GB+95369739]*/
  0,     0,     0,     0,     39881, 0,     25467, 15656, 0,     19062,   /*[GB+95369830, GB+95369839]*/
  0,     0,     29153, 34577, 0,     0,     8924,  0,     0,     0,       /*[GB+95369930, GB+95369939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     30496, 0,       /*[GB+95369A30, GB+95369A39]*/
  0,     0,     0,     31300, 0,     0,     0,     0,     0,     0,       /*[GB+95369B30, GB+95369B39]*/
  0,     21394, 0,     5961,  0,     21633, 0,     0,     0,     0,       /*[GB+95369C30, GB+95369C39]*/
  0,     0,     0,     0,     0,     0,     12437, 0,     0,     0,       /*[GB+95369D30, GB+95369D39]*/
  0,     0,     0,     0,     0,     0,     0,     7259,  0,     0,       /*[GB+95369E30, GB+95369E39]*/
  0,     19073, 0,     0,     0,     0,     17418, 38608, 17556, 0,       /*[GB+95369F30, GB+95369F39]*/
  14314, 0,     12882, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9536A030, GB+9536A039]*/
  0,     0,     0,     10431, 0,     0,     41329, 0,     0,     0,       /*[GB+9536A130, GB+9536A139]*/
  0,     0,     0,     21522, 0,     0,     16977, 0,     0,     0,       /*[GB+9536A230, GB+9536A239]*/
  0,     0,     0,     7264,  0,     0,     0,     0,     0,     0,       /*[GB+9536A330, GB+9536A339]*/
  0,     38612, 0,     0,     0,     0,     0,     0,     0,     17993,   /*[GB+9536A430, GB+9536A439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9536A530, GB+9536A539]*/
  0,     0,     0,     9591,  0,     0,     0,     0,     0,     31504,   /*[GB+9536A630, GB+9536A639]*/
  37253, 0,     32502, 0,     0,     0,     0,     14583, 0,     0,       /*[GB+9536A730, GB+9536A739]*/
  0,     0,     35593, 0,     29226, 0,     0,     0,     0,     0,       /*[GB+9536A830, GB+9536A839]*/
  0,     32864, 0,     0,     39795, 0,     0,     0,     0,     0,       /*[GB+9536A930, GB+9536A939]*/
  0,     0,     0,     0,     0,     0,     0,     41198, 0,     0,       /*[GB+9536AA30, GB+9536AA39]*/
  0,     0,     0,     0,     0,     266,   0,     0,     0,     0,       /*[GB+9536AB30, GB+9536AB39]*/
  0,     27697, 4097,  0,     16774, 0,     0,     0,     15520, 10576,   /*[GB+9536AC30, GB+9536AC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9536AD30, GB+9536AD39]*/
  21653, 0,     0,     0,     0,     10882, 0,     0,     4039,  0,       /*[GB+9536AE30, GB+9536AE39]*/
  0,     0,     19314, 0,     43652, 0,     0,     0,     0,     0,       /*[GB+9536AF30, GB+9536AF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9536B030, GB+9536B039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     17117,   /*[GB+9536B130, GB+9536B139]*/
  0,     0,     42546, 0,     41430, 0,     6446,  9633,  40358, 5429,    /*[GB+9536B230, GB+9536B239]*/
  23397, 0,     0,     0,     0,     0,     0,     1882,  0,     26859,   /*[GB+9536B330, GB+9536B339]*/
  2817,  0,     0,     0,     27208, 38491, 2818,  0,     0,     42354,   /*[GB+9536B430, GB+9536B439]*/
  0,     0,     9963,  23383, 28434, 0,     0,     29901, 0,     0,       /*[GB+9536B530, GB+9536B539]*/
  33156, 3777,  0,     0,     0,     0,     0,     0,     21772, 42123,   /*[GB+9536B630, GB+9536B639]*/
  0,     0,     24775, 0,     0,     9618,  0,     0,     0,     0,       /*[GB+9536B730, GB+9536B739]*/
  19858, 0,     43519, 0,     0,     0,     0,     5635,  0,     34474,   /*[GB+9536B830, GB+9536B839]*/
  0,     27273, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9536B930, GB+9536B939]*/
  0,     0,     0,     0,     33516, 0,     0,     0,     16752, 0,       /*[GB+9536BA30, GB+9536BA39]*/
  10265, 4526,  14729, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9536BB30, GB+9536BB39]*/
  0,     0,     0,     0,     14730, 33523, 0,     16667, 0,     9696,    /*[GB+9536BC30, GB+9536BC39]*/
  41062, 7808,  0,     0,     35144, 0,     0,     0,     0,     0,       /*[GB+9536BD30, GB+9536BD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9536BE30, GB+9536BE39]*/
  0,     12088, 0,     22939, 33926, 0,     0,     0,     0,     0,       /*[GB+9536BF30, GB+9536BF39]*/
  0,     0,     0,     0,     0,     4914,  0,     10348, 0,     0,       /*[GB+9536C030, GB+9536C039]*/
  0,     33554, 0,     0,     0,     0,     0,     25200, 0,     0,       /*[GB+9536C130, GB+9536C139]*/
  20862, 0,     0,     41546, 0,     0,     15737, 0,     0,     0,       /*[GB+9536C230, GB+9536C239]*/
  1717,  0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9536C330, GB+9536C339]*/
  0,     0,     1718,  0,     0,     0,     0,     26648, 0,     0,       /*[GB+9536C430, GB+9536C439]*/
  0,     0,     0,     33949, 0,     0,     14356, 0,     26658, 42213,   /*[GB+9536C530, GB+9536C539]*/
  31197, 16128, 0,     0,     0,     0,     27628, 0,     0,     43474,   /*[GB+9536C630, GB+9536C639]*/
  0,     0,     0,     0,     0,     0,     0,     14620, 0,     0,       /*[GB+9536C730, GB+9536C739]*/
  0,     0,     0,     0,     1759,  0,     6715,  43917, 0,     0,       /*[GB+9536C830, GB+9536C839]*/
  0,     0,     0,     0,     0,     0,     36825, 0,     0,     0,       /*[GB+9536C930, GB+9536C939]*/
  0,     0,     0,     0,     0,     0,     0,     1382,  0,     0,       /*[GB+9536CA30, GB+9536CA39]*/
  0,     16788, 0,     36969, 32674, 0,     12064, 0,     10989, 0,       /*[GB+9536CB30, GB+9536CB39]*/
  0,     0,     0,     9530,  0,     0,     21295, 0,     0,     17923,   /*[GB+9536CC30, GB+9536CC39]*/
  0,     6101,  0,     1910,  12074, 0,     0,     32240, 15979, 0,       /*[GB+9536CD30, GB+9536CD39]*/
  0,     0,     0,     0,     0,     0,     3797,  0,     0,     0,       /*[GB+9536CE30, GB+9536CE39]*/
  0,     419,   15258, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9536CF30, GB+9536CF39]*/
  0,     6016,  0,     41401, 0,     5308,  28752, 10949, 0,     0,       /*[GB+9536D030, GB+9536D039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9536D130, GB+9536D139]*/
  0,     0,     37526, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9536D230, GB+9536D239]*/
  0,     0,     0,     1230,  24131, 37355, 0,     0,     5885,  2441,    /*[GB+9536D330, GB+9536D339]*/
  24362, 40212, 25791, 23975, 0,     0,     0,     0,     0,     0,       /*[GB+9536D430, GB+9536D439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9536D530, GB+9536D539]*/
  0,     0,     0,     32167, 35145, 0,     0,     0,     0,     0,       /*[GB+9536D630, GB+9536D639]*/
  0,     37533, 0,     0,     7809,  9869,  11823, 27338, 0,     0,       /*[GB+9536D730, GB+9536D739]*/
  0,     0,     0,     26897, 8354,  12236, 0,     0,     0,     0,       /*[GB+9536D830, GB+9536D839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9536D930, GB+9536D939]*/
  27282, 0,     0,     0,     0,     0,     0,     0,     21003, 0,       /*[GB+9536DA30, GB+9536DA39]*/
  0,     6961,  12523, 33927, 35590, 0,     0,     26898, 37683, 0,       /*[GB+9536DB30, GB+9536DB39]*/
  0,     41547, 0,     13376, 9186,  39365, 0,     24635, 0,     0,       /*[GB+9536DC30, GB+9536DC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9536DD30, GB+9536DD39]*/
  0,     0,     0,     0,     0,     0,     4253,  4915,  41548, 22941,   /*[GB+9536DE30, GB+9536DE39]*/
  0,     25816, 0,     0,     2691,  0,     22942, 30020, 0,     0,       /*[GB+9536DF30, GB+9536DF39]*/
  0,     0,     0,     2692,  3526,  37266, 6871,  32479, 0,     22919,   /*[GB+9536E030, GB+9536E039]*/
  21420, 0,     0,     0,     0,     0,     23295, 33943, 0,     37228,   /*[GB+9536E130, GB+9536E139]*/
  0,     2819,  0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9536E230, GB+9536E239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9536E330, GB+9536E339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9536E430, GB+9536E439]*/
  0,     0,     0,     2366,  0,     0,     0,     0,     0,     0,       /*[GB+9536E530, GB+9536E539]*/
  0,     0,     0,     0,     0,     0,     35171, 21765, 16569, 0,       /*[GB+9536E630, GB+9536E639]*/
  4942,  0,     28618, 23649, 11848, 0,     16468, 34528, 0,     11486,   /*[GB+9536E730, GB+9536E739]*/
  26143, 43640, 0,     0,     8540,  22432, 0,     22815, 0,     16720,   /*[GB+9536E830, GB+9536E839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9536E930, GB+9536E939]*/
  0,     0,     0,     0,     22680, 0,     0,     0,     0,     0,       /*[GB+9536EA30, GB+9536EA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     9228,    /*[GB+9536EB30, GB+9536EB39]*/
  0,     0,     0,     11487, 0,     18,    0,     28688, 0,     0,       /*[GB+9536EC30, GB+9536EC39]*/
  0,     0,     0,     0,     0,     0,     35602, 4777,  14266, 0,       /*[GB+9536ED30, GB+9536ED39]*/
  40997, 34796, 18753, 26573, 0,     0,     28075, 35199, 0,     9584,    /*[GB+9536EE30, GB+9536EE39]*/
  21767, 2783,  0,     7520,  0,     0,     0,     6710,  0,     0,       /*[GB+9536EF30, GB+9536EF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9536F030, GB+9536F039]*/
  0,     0,     0,     0,     0,     0,     16644, 0,     0,     0,       /*[GB+9536F130, GB+9536F139]*/
  31350, 28696, 0,     0,     0,     27673, 0,     0,     9980,  0,       /*[GB+9536F230, GB+9536F239]*/
  0,     0,     10982, 42408, 37203, 31547, 1359,  0,     21978, 0,       /*[GB+9536F330, GB+9536F339]*/
  8579,  42891, 0,     0,     0,     0,     0,     25596, 0,     0,       /*[GB+9536F430, GB+9536F439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9536F530, GB+9536F539]*/
  0,     0,     0,     0,     0,     0,     0,     42409, 0,     0,       /*[GB+9536F630, GB+9536F639]*/
  29767, 0,     0,     22949, 15766, 0,     37195, 0,     12251, 0,       /*[GB+9536F730, GB+9536F739]*/
  0,     0,     1082,  21979, 3186,  0,     0,     0,     20234, 28246,   /*[GB+9536F830, GB+9536F839]*/
  2216,  40908, 29536, 0,     43958, 40189, 40593, 20356, 0,     0,       /*[GB+9536F930, GB+9536F939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9536FA30, GB+9536FA39]*/
  0,     26496, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9536FB30, GB+9536FB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9536FC30, GB+9536FC39]*/
  0,     0,     21731, 0,     0,     0,     0,     0,     43141, 0,       /*[GB+9536FD30, GB+9536FD39]*/
  0,     0,     0,     0,     0,     0,     5675,  0,     40013, 0,       /*[GB+9536FE30, GB+9536FE39]*/
  0,     0,     0,     0,     0,     0,     15547, 0,     0,     12490,   /*[GB+95378130, GB+95378139]*/
  25153, 0,     40460, 0,     20846, 0,     22017, 0,     0,     0,       /*[GB+95378230, GB+95378239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+95378330, GB+95378339]*/
  0,     0,     0,     0,     0,     0,     0,     24277, 0,     14949,   /*[GB+95378430, GB+95378439]*/
  0,     0,     0,     0,     0,     37245, 0,     0,     0,     0,       /*[GB+95378530, GB+95378539]*/
  0,     0,     3873,  22836, 29777, 0,     0,     0,     0,     0,       /*[GB+95378630, GB+95378639]*/
  29274, 0,     0,     24278, 0,     0,     0,     0,     0,     0,       /*[GB+95378730, GB+95378739]*/
  0,     0,     10816, 0,     0,     0,     0,     0,     0,     0,       /*[GB+95378830, GB+95378839]*/
  0,     0,     0,     0,     37248, 0,     22651, 0,     0,     0,       /*[GB+95378930, GB+95378939]*/
  0,     0,     0,     0,     28987, 0,     40019, 0,     2354,  14683,   /*[GB+95378A30, GB+95378A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     35264,   /*[GB+95378B30, GB+95378B39]*/
  0,     0,     0,     36718, 34861, 0,     0,     0,     0,     0,       /*[GB+95378C30, GB+95378C39]*/
  0,     0,     0,     0,     0,     0,     0,     35273, 18487, 0,       /*[GB+95378D30, GB+95378D39]*/
  7851,  0,     0,     0,     0,     0,     0,     0,     0,     43054,   /*[GB+95378E30, GB+95378E39]*/
  0,     22393, 0,     43122, 0,     0,     16975, 0,     0,     0,       /*[GB+95378F30, GB+95378F39]*/
  20490, 0,     0,     0,     0,     0,     0,     0,     0,     25198,   /*[GB+95379030, GB+95379039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+95379130, GB+95379139]*/
  32328, 0,     0,     0,     0,     0,     0,     29160, 0,     3610,    /*[GB+95379230, GB+95379239]*/
  0,     38022, 0,     0,     0,     0,     10942, 0,     14435, 0,       /*[GB+95379330, GB+95379339]*/
  0,     12515, 0,     34937, 4653,  0,     0,     0,     0,     0,       /*[GB+95379430, GB+95379439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+95379530, GB+95379539]*/
  13355, 0,     0,     0,     0,     0,     0,     0,     22756, 22396,   /*[GB+95379630, GB+95379639]*/
  22759, 31110, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+95379730, GB+95379739]*/
  0,     0,     0,     0,     15608, 0,     21958, 0,     0,     0,       /*[GB+95379830, GB+95379839]*/
  0,     18052, 0,     42631, 0,     35877, 0,     26155, 8558,  0,       /*[GB+95379930, GB+95379939]*/
  1837,  0,     0,     0,     13421, 0,     0,     24900, 33960, 0,       /*[GB+95379A30, GB+95379A39]*/
  0,     0,     0,     0,     5856,  7561,  0,     0,     0,     0,       /*[GB+95379B30, GB+95379B39]*/
  0,     29147, 31611, 34909, 0,     24104, 0,     0,     0,     0,       /*[GB+95379C30, GB+95379C39]*/
  0,     0,     0,     0,     5497,  0,     13251, 0,     0,     0,       /*[GB+95379D30, GB+95379D39]*/
  0,     0,     0,     0,     0,     34981, 0,     0,     0,     4880,    /*[GB+95379E30, GB+95379E39]*/
  0,     0,     0,     1783,  27196, 0,     0,     0,     0,     0,       /*[GB+95379F30, GB+95379F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9537A030, GB+9537A039]*/
  0,     21688, 0,     0,     0,     0,     21689, 0,     29068, 34889,   /*[GB+9537A130, GB+9537A139]*/
  800,   32259, 0,     23753, 0,     0,     0,     0,     0,     0,       /*[GB+9537A230, GB+9537A239]*/
  0,     0,     0,     8886,  0,     11575, 11257, 0,     42871, 9936,    /*[GB+9537A330, GB+9537A339]*/
  0,     0,     0,     10275, 0,     0,     0,     0,     0,     0,       /*[GB+9537A430, GB+9537A439]*/
  0,     0,     0,     0,     0,     5354,  0,     0,     1789,  12339,   /*[GB+9537A530, GB+9537A539]*/
  0,     0,     2693,  0,     0,     0,     34173, 15437, 0,     37364,   /*[GB+9537A630, GB+9537A639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9537A730, GB+9537A739]*/
  0,     0,     0,     0,     8170,  40752, 0,     0,     0,     8233,    /*[GB+9537A830, GB+9537A839]*/
  0,     0,     0,     0,     0,     0,     15368, 39744, 0,     0,       /*[GB+9537A930, GB+9537A939]*/
  0,     0,     0,     12685, 0,     0,     0,     0,     0,     0,       /*[GB+9537AA30, GB+9537AA39]*/
  0,     0,     14595, 35568, 0,     22429, 0,     0,     0,     0,       /*[GB+9537AB30, GB+9537AB39]*/
  0,     23296, 0,     28619, 0,     39769, 0,     0,     12103, 9434,    /*[GB+9537AC30, GB+9537AC39]*/
  32389, 0,     32123, 25650, 0,     0,     0,     0,     0,     0,       /*[GB+9537AD30, GB+9537AD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9537AE30, GB+9537AE39]*/
  38701, 0,     13587, 0,     0,     0,     0,     0,     0,     30256,   /*[GB+9537AF30, GB+9537AF39]*/
  0,     14621, 33627, 33102, 0,     0,     0,     20900, 0,     0,       /*[GB+9537B030, GB+9537B039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9537B130, GB+9537B139]*/
  37114, 0,     37288, 0,     0,     0,     0,     20903, 23453, 0,       /*[GB+9537B230, GB+9537B239]*/
  0,     198,   0,     0,     0,     19869, 0,     0,     0,     0,       /*[GB+9537B330, GB+9537B339]*/
  0,     0,     0,     0,     0,     0,     0,     7436,  43584, 0,       /*[GB+9537B430, GB+9537B439]*/
  0,     0,     0,     7437,  0,     245,   0,     0,     0,     0,       /*[GB+9537B530, GB+9537B539]*/
  12845, 0,     2626,  0,     0,     0,     0,     0,     0,     0,       /*[GB+9537B630, GB+9537B639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     35574, 13458,   /*[GB+9537B730, GB+9537B739]*/
  0,     44032, 0,     24938, 0,     0,     0,     0,     41919, 0,       /*[GB+9537B830, GB+9537B839]*/
  0,     0,     0,     0,     13028, 24795, 15720, 23157, 0,     0,       /*[GB+9537B930, GB+9537B939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9537BA30, GB+9537BA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9537BB30, GB+9537BB39]*/
  0,     0,     0,     0,     0,     27812, 0,     0,     0,     0,       /*[GB+9537BC30, GB+9537BC39]*/
  0,     11208, 27227, 25983, 0,     0,     0,     0,     15417, 0,       /*[GB+9537BD30, GB+9537BD39]*/
  0,     18488, 0,     0,     0,     0,     0,     28804, 0,     0,       /*[GB+9537BE30, GB+9537BE39]*/
  22847, 34387, 8441,  0,     0,     27345, 0,     0,     0,     35582,   /*[GB+9537BF30, GB+9537BF39]*/
  10338, 0,     13528, 0,     21383, 8443,  0,     27603, 39941, 14482,   /*[GB+9537C030, GB+9537C039]*/
  21403, 19715, 25586, 0,     21488, 0,     0,     37931, 0,     19719,   /*[GB+9537C130, GB+9537C139]*/
  10639, 22853, 17803, 27850, 0,     0,     19197, 0,     39703, 0,       /*[GB+9537C230, GB+9537C239]*/
  0,     0,     0,     0,     0,     23376, 0,     0,     0,     0,       /*[GB+9537C330, GB+9537C339]*/
  0,     33431, 0,     0,     0,     1527,  0,     0,     1528,  35978,   /*[GB+9537C430, GB+9537C439]*/
  0,     13753, 0,     0,     1530,  0,     0,     0,     0,     0,       /*[GB+9537C530, GB+9537C539]*/
  0,     0,     0,     0,     5806,  0,     0,     0,     0,     43096,   /*[GB+9537C630, GB+9537C639]*/
  0,     0,     0,     27221, 0,     0,     0,     0,     29333, 0,       /*[GB+9537C730, GB+9537C739]*/
  0,     0,     0,     0,     0,     0,     13550, 0,     0,     12478,   /*[GB+9537C830, GB+9537C839]*/
  0,     0,     41155, 0,     0,     0,     572,   19000, 0,     34992,   /*[GB+9537C930, GB+9537C939]*/
  0,     0,     0,     0,     0,     0,     0,     13870, 29019, 0,       /*[GB+9537CA30, GB+9537CA39]*/
  0,     0,     0,     16337, 0,     0,     0,     9973,  20359, 0,       /*[GB+9537CB30, GB+9537CB39]*/
  0,     0,     0,     0,     23297, 3118,  0,     0,     0,     0,       /*[GB+9537CC30, GB+9537CC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9537CD30, GB+9537CD39]*/
  18801, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9537CE30, GB+9537CE39]*/
  0,     22436, 27341, 0,     0,     0,     42410, 0,     0,     0,       /*[GB+9537CF30, GB+9537CF39]*/
  0,     0,     0,     0,     2519,  0,     0,     0,     0,     0,       /*[GB+9537D030, GB+9537D039]*/
  0,     0,     0,     0,     18426, 0,     0,     0,     0,     0,       /*[GB+9537D130, GB+9537D139]*/
  0,     0,     0,     0,     0,     17481, 0,     0,     0,     2833,    /*[GB+9537D230, GB+9537D239]*/
  0,     32472, 0,     0,     18263, 0,     0,     33408, 23720, 39341,   /*[GB+9537D330, GB+9537D339]*/
  0,     8929,  0,     0,     0,     151,   35586, 29395, 27430, 0,       /*[GB+9537D430, GB+9537D439]*/
  0,     0,     0,     0,     640,   39191, 0,     0,     0,     0,       /*[GB+9537D530, GB+9537D539]*/
  0,     0,     0,     0,     12780, 0,     0,     12387, 0,     0,       /*[GB+9537D630, GB+9537D639]*/
  44062, 34774, 0,     21614, 0,     0,     0,     0,     8856,  40131,   /*[GB+9537D730, GB+9537D739]*/
  0,     2172,  3230,  31949, 0,     2173,  0,     0,     0,     0,       /*[GB+9537D830, GB+9537D839]*/
  0,     8857,  0,     31325, 22309, 0,     0,     0,     0,     11685,   /*[GB+9537D930, GB+9537D939]*/
  23031, 0,     0,     0,     0,     12472, 19847, 0,     31052, 0,       /*[GB+9537DA30, GB+9537DA39]*/
  25773, 25716, 42610, 0,     0,     0,     5910,  0,     0,     0,       /*[GB+9537DB30, GB+9537DB39]*/
  19143, 0,     0,     26695, 43107, 0,     17878, 0,     35124, 0,       /*[GB+9537DC30, GB+9537DC39]*/
  0,     26986, 0,     0,     5593,  0,     6739,  22878, 0,     0,       /*[GB+9537DD30, GB+9537DD39]*/
  0,     28820, 0,     0,     34296, 0,     0,     0,     0,     0,       /*[GB+9537DE30, GB+9537DE39]*/
  0,     42087, 0,     0,     0,     71,    4627,  24656, 0,     28824,   /*[GB+9537DF30, GB+9537DF39]*/
  0,     0,     0,     26377, 29162, 6107,  0,     34916, 29398, 0,       /*[GB+9537E030, GB+9537E039]*/
  12089, 0,     0,     0,     37786, 15683, 0,     0,     42189, 0,       /*[GB+9537E130, GB+9537E139]*/
  0,     0,     0,     0,     0,     0,     41472, 0,     0,     0,       /*[GB+9537E230, GB+9537E239]*/
  0,     0,     0,     35851, 0,     0,     0,     0,     0,     0,       /*[GB+9537E330, GB+9537E339]*/
  0,     0,     0,     0,     0,     19762, 0,     42995, 0,     30240,   /*[GB+9537E430, GB+9537E439]*/
  0,     0,     0,     6352,  0,     0,     26473, 0,     16461, 11591,   /*[GB+9537E530, GB+9537E539]*/
  15369, 0,     26078, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9537E630, GB+9537E639]*/
  0,     0,     0,     0,     0,     1860,  0,     0,     0,     0,       /*[GB+9537E730, GB+9537E739]*/
  0,     0,     0,     31140, 0,     15872, 0,     0,     26310, 0,       /*[GB+9537E830, GB+9537E839]*/
  0,     0,     0,     0,     0,     0,     0,     14213, 0,     0,       /*[GB+9537E930, GB+9537E939]*/
  0,     0,     0,     0,     40442, 0,     0,     21192, 24009, 0,       /*[GB+9537EA30, GB+9537EA39]*/
  26487, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9537EB30, GB+9537EB39]*/
  0,     0,     21560, 0,     0,     31146, 0,     0,     16180, 0,       /*[GB+9537EC30, GB+9537EC39]*/
  1204,  0,     26415, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9537ED30, GB+9537ED39]*/
  0,     31332, 0,     4291,  0,     0,     0,     0,     0,     0,       /*[GB+9537EE30, GB+9537EE39]*/
  15476, 0,     33672, 0,     0,     0,     19213, 0,     0,     0,       /*[GB+9537EF30, GB+9537EF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9537F030, GB+9537F039]*/
  0,     3789,  13069, 0,     0,     0,     0,     0,     19938, 0,       /*[GB+9537F130, GB+9537F139]*/
  0,     0,     18544, 0,     0,     0,     0,     0,     15804, 0,       /*[GB+9537F230, GB+9537F239]*/
  0,     0,     0,     0,     0,     42807, 20019, 0,     0,     0,       /*[GB+9537F330, GB+9537F339]*/
  0,     0,     0,     0,     22995, 0,     0,     0,     0,     0,       /*[GB+9537F430, GB+9537F439]*/
  26636, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9537F530, GB+9537F539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9537F630, GB+9537F639]*/
  0,     0,     0,     0,     0,     36498, 0,     7345,  0,     11798,   /*[GB+9537F730, GB+9537F739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9537F830, GB+9537F839]*/
  0,     0,     0,     0,     20867, 0,     0,     0,     0,     0,       /*[GB+9537F930, GB+9537F939]*/
  0,     0,     0,     33413, 0,     0,     2645,  0,     0,     0,       /*[GB+9537FA30, GB+9537FA39]*/
  0,     25872, 14552, 11343, 0,     5420,  7523,  40865, 11807, 282,     /*[GB+9537FB30, GB+9537FB39]*/
  14733, 8703,  39128, 43627, 8222,  0,     0,     0,     0,     0,       /*[GB+9537FC30, GB+9537FC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     22249, 0,       /*[GB+9537FD30, GB+9537FD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     27759, 4843,    /*[GB+9537FE30, GB+9537FE39]*/
  0,     0,     0,     16262, 0,     0,     0,     0,     0,     0,       /*[GB+95388130, GB+95388139]*/
  37410, 73,    768,   0,     24538, 0,     28825, 7810,  0,     15429,   /*[GB+95388230, GB+95388239]*/
  24048, 0,     32851, 16671, 25798, 8939,  0,     0,     0,     0,       /*[GB+95388330, GB+95388339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+95388430, GB+95388439]*/
  0,     0,     16915, 30075, 1059,  13558, 36510, 0,     0,     0,       /*[GB+95388530, GB+95388539]*/
  16242, 22996, 0,     35469, 0,     33928, 0,     19202, 7174,  0,       /*[GB+95388630, GB+95388639]*/
  2932,  0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+95388730, GB+95388739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+95388830, GB+95388839]*/
  28885, 0,     0,     7000,  0,     14136, 39626, 20027, 7525,  0,       /*[GB+95388930, GB+95388939]*/
  0,     9759,  0,     32738, 3246,  12174, 1754,  0,     0,     10748,   /*[GB+95388A30, GB+95388A39]*/
  31907, 22348, 25032, 0,     31666, 39984, 32491, 0,     33285, 9188,    /*[GB+95388B30, GB+95388B39]*/
  0,     0,     16536, 0,     37806, 0,     0,     0,     0,     0,       /*[GB+95388C30, GB+95388C39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     34775, 0,       /*[GB+95388D30, GB+95388D39]*/
  10230, 36518, 30080, 29706, 0,     43326, 0,     0,     31718, 34411,   /*[GB+95388E30, GB+95388E39]*/
  0,     0,     0,     0,     40872, 18323, 0,     0,     42996, 0,       /*[GB+95388F30, GB+95388F39]*/
  14584, 0,     0,     31134, 0,     0,     35044, 26135, 0,     36524,   /*[GB+95389030, GB+95389039]*/
  15371, 0,     0,     38567, 42103, 0,     16216, 43450, 0,     16370,   /*[GB+95389130, GB+95389139]*/
  0,     23009, 25589, 32376, 40795, 0,     15372, 0,     43604, 0,       /*[GB+95389230, GB+95389239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+95389330, GB+95389339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+95389430, GB+95389439]*/
  0,     0,     0,     0,     36525, 0,     0,     19118, 0,     24873,   /*[GB+95389530, GB+95389539]*/
  0,     0,     0,     0,     12467, 0,     0,     0,     0,     25108,   /*[GB+95389630, GB+95389639]*/
  0,     37098, 30084, 0,     25039, 10152, 0,     0,     0,     0,       /*[GB+95389730, GB+95389739]*/
  0,     0,     0,     0,     26001, 7273,  0,     44068, 0,     0,       /*[GB+95389830, GB+95389839]*/
  40874, 25109, 13162, 0,     32124, 6367,  0,     12549, 20966, 0,       /*[GB+95389930, GB+95389939]*/
  35796, 0,     0,     11488, 36796, 34663, 4778,  11674, 0,     0,       /*[GB+95389A30, GB+95389A39]*/
  239,   0,     1890,  0,     0,     0,     0,     25546, 0,     0,       /*[GB+95389B30, GB+95389B39]*/
  0,     7183,  0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+95389C30, GB+95389C39]*/
  42941, 6368,  39152, 25043, 28782, 36249, 39153, 16199, 25746, 0,       /*[GB+95389D30, GB+95389D39]*/
  0,     25547, 23848, 0,     0,     26660, 0,     38856, 0,     0,       /*[GB+95389E30, GB+95389E39]*/
  0,     0,     0,     27880, 0,     0,     0,     30842, 0,     11456,   /*[GB+95389F30, GB+95389F39]*/
  24140, 37289, 0,     0,     0,     0,     31779, 2784,  0,     34178,   /*[GB+9538A030, GB+9538A039]*/
  24901, 0,     26156, 0,     0,     14921, 0,     0,     0,     0,       /*[GB+9538A130, GB+9538A139]*/
  0,     0,     0,     0,     0,     0,     41704, 0,     0,     0,       /*[GB+9538A230, GB+9538A239]*/
  42834, 0,     33629, 0,     33081, 42603, 0,     0,     0,     30092,   /*[GB+9538A330, GB+9538A339]*/
  0,     0,     16386, 17219, 16304, 0,     0,     3187,  1360,  22092,   /*[GB+9538A430, GB+9538A439]*/
  43033, 10785, 39666, 37867, 20623, 0,     0,     25133, 10387, 20833,   /*[GB+9538A530, GB+9538A539]*/
  18889, 19127, 43818, 17911, 11859, 4441,  25134, 17840, 0,     0,       /*[GB+9538A630, GB+9538A639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9538A730, GB+9538A739]*/
  0,     0,     0,     3652,  0,     24524, 6393,  0,     0,     0,       /*[GB+9538A830, GB+9538A839]*/
  0,     0,     17912, 0,     0,     0,     37985, 0,     0,     0,       /*[GB+9538A930, GB+9538A939]*/
  0,     6204,  0,     0,     7058,  9130,  11622, 0,     34830, 8850,    /*[GB+9538AA30, GB+9538AA39]*/
  23883, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9538AB30, GB+9538AB39]*/
  0,     0,     0,     0,     0,     0,     28509, 0,     0,     0,       /*[GB+9538AC30, GB+9538AC39]*/
  0,     32667, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9538AD30, GB+9538AD39]*/
  0,     7018,  3190,  38657, 16823, 16706, 0,     43963, 9813,  0,       /*[GB+9538AE30, GB+9538AE39]*/
  0,     40461, 42457, 0,     0,     13487, 3484,  0,     0,     0,       /*[GB+9538AF30, GB+9538AF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     35239, 34441,   /*[GB+9538B030, GB+9538B039]*/
  0,     0,     0,     43939, 200,   10896, 0,     0,     24550, 0,       /*[GB+9538B130, GB+9538B139]*/
  5956,  37620, 0,     0,     0,     22572, 0,     0,     0,     0,       /*[GB+9538B230, GB+9538B239]*/
  0,     0,     0,     0,     0,     14672, 757,   17553, 2489,  0,       /*[GB+9538B330, GB+9538B339]*/
  1423,  0,     0,     0,     0,     41258, 25164, 0,     0,     0,       /*[GB+9538B430, GB+9538B439]*/
  0,     0,     22573, 0,     0,     20290, 0,     16942, 0,     21998,   /*[GB+9538B530, GB+9538B539]*/
  0,     17924, 23747, 0,     0,     0,     0,     0,     18467, 2636,    /*[GB+9538B630, GB+9538B639]*/
  0,     5688,  17925, 0,     0,     7484,  0,     1132,  15956, 0,       /*[GB+9538B730, GB+9538B739]*/
  0,     0,     0,     0,     0,     0,     0,     36061, 23083, 0,       /*[GB+9538B830, GB+9538B839]*/
  10601, 0,     0,     0,     0,     0,     34455, 0,     13965, 21468,   /*[GB+9538B930, GB+9538B939]*/
  23089, 0,     0,     0,     0,     0,     0,     0,     0,     2490,    /*[GB+9538BA30, GB+9538BA39]*/
  29725, 40924, 38432, 0,     0,     0,     3978,  0,     2493,  0,       /*[GB+9538BB30, GB+9538BB39]*/
  0,     0,     0,     17943, 23393, 18545, 0,     0,     18173, 20491,   /*[GB+9538BC30, GB+9538BC39]*/
  0,     0,     0,     30805, 0,     0,     23108, 23109, 0,     13095,   /*[GB+9538BD30, GB+9538BD39]*/
  0,     17950, 0,     22536, 0,     0,     19174, 19408, 0,     0,       /*[GB+9538BE30, GB+9538BE39]*/
  34402, 39966, 0,     40545, 26710, 18998, 0,     0,     0,     0,       /*[GB+9538BF30, GB+9538BF39]*/
  0,     39709, 12231, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9538C030, GB+9538C039]*/
  0,     0,     22585, 0,     36371, 0,     15445, 12822, 0,     38199,   /*[GB+9538C130, GB+9538C139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     33740, 27824,   /*[GB+9538C230, GB+9538C239]*/
  14425, 39052, 0,     0,     0,     0,     0,     0,     0,     28661,   /*[GB+9538C330, GB+9538C339]*/
  14976, 32432, 37508, 15208, 0,     0,     9542,  0,     0,     0,       /*[GB+9538C430, GB+9538C439]*/
  0,     12389, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9538C530, GB+9538C539]*/
  7597,  0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9538C630, GB+9538C639]*/
  0,     10731, 0,     0,     0,     0,     0,     0,     8501,  0,       /*[GB+9538C730, GB+9538C739]*/
  0,     0,     8518,  43337, 12959, 0,     0,     0,     0,     0,       /*[GB+9538C830, GB+9538C839]*/
  0,     0,     0,     0,     0,     0,     0,     711,   0,     0,       /*[GB+9538C930, GB+9538C939]*/
  24825, 28816, 0,     6273,  23814, 34403, 27606, 0,     0,     0,       /*[GB+9538CA30, GB+9538CA39]*/
  0,     3070,  40180, 0,     9477,  0,     33227, 0,     8066,  0,       /*[GB+9538CB30, GB+9538CB39]*/
  0,     2926,  0,     22669, 15812, 0,     41457, 0,     25368, 0,       /*[GB+9538CC30, GB+9538CC39]*/
  0,     355,   0,     0,     0,     0,     32503, 42548, 0,     12232,   /*[GB+9538CD30, GB+9538CD39]*/
  0,     2236,  42933, 1241,  3159,  41865, 8462,  16613, 0,     44013,   /*[GB+9538CE30, GB+9538CE39]*/
  36089, 0,     8709,  0,     0,     0,     37356, 0,     0,     1946,    /*[GB+9538CF30, GB+9538CF39]*/
  0,     928,   35026, 28886, 37788, 0,     11361, 5062,  0,     38109,   /*[GB+9538D030, GB+9538D039]*/
  43087, 0,     0,     0,     0,     3812,  0,     0,     0,     0,       /*[GB+9538D130, GB+9538D139]*/
  0,     0,     24498, 0,     0,     0,     38551, 0,     0,     0,       /*[GB+9538D230, GB+9538D239]*/
  0,     0,     0,     0,     0,     40151, 0,     0,     0,     0,       /*[GB+9538D330, GB+9538D339]*/
  15154, 0,     0,     21027, 25484, 0,     38142, 0,     0,     0,       /*[GB+9538D430, GB+9538D439]*/
  33286, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9538D530, GB+9538D539]*/
  0,     0,     0,     889,   12823, 4852,  16576, 0,     0,     0,       /*[GB+9538D630, GB+9538D639]*/
  0,     30367, 0,     10757, 25488, 0,     0,     0,     42104, 0,       /*[GB+9538D730, GB+9538D739]*/
  0,     0,     0,     0,     27766, 0,     0,     25284, 10488, 0,       /*[GB+9538D830, GB+9538D839]*/
  0,     10350, 0,     0,     13754, 42568, 21669, 10244, 28929, 22170,   /*[GB+9538D930, GB+9538D939]*/
  7402,  499,   17273, 0,     0,     0,     42767, 0,     13163, 0,       /*[GB+9538DA30, GB+9538DA39]*/
  0,     0,     0,     0,     40731, 0,     0,     0,     40948, 38201,   /*[GB+9538DB30, GB+9538DB39]*/
  0,     9641,  0,     0,     10166, 38760, 27970, 11104, 743,   22083,   /*[GB+9538DC30, GB+9538DC39]*/
  21266, 0,     0,     33463, 5384,  0,     0,     15183, 0,     0,       /*[GB+9538DD30, GB+9538DD39]*/
  0,     0,     10778, 0,     0,     31548, 16617, 18086, 42411, 12579,   /*[GB+9538DE30, GB+9538DE39]*/
  40610, 0,     6279,  4948,  0,     0,     0,     0,     0,     0,       /*[GB+9538DF30, GB+9538DF39]*/
  0,     0,     0,     42412, 0,     16771, 0,     34734, 12004, 18902,   /*[GB+9538E030, GB+9538E039]*/
  5093,  25668, 14385, 0,     35739, 7625,  0,     32077, 30592, 42431,   /*[GB+9538E130, GB+9538E139]*/
  0,     0,     0,     0,     0,     0,     22613, 21069, 6084,  2629,    /*[GB+9538E230, GB+9538E239]*/
  0,     0,     14396, 18657, 0,     27902, 5555,  0,     0,     30115,   /*[GB+9538E330, GB+9538E339]*/
  18658, 10525, 0,     0,     1700,  0,     4753,  6209,  20282, 22773,   /*[GB+9538E430, GB+9538E439]*/
  36851, 17472, 0,     0,     8925,  4191,  0,     0,     0,     1440,    /*[GB+9538E530, GB+9538E539]*/
  43838, 12643, 0,     17393, 34381, 8208,  11213, 0,     0,     0,       /*[GB+9538E630, GB+9538E639]*/
  0,     0,     0,     0,     0,     37457, 41272, 0,     22654, 0,       /*[GB+9538E730, GB+9538E739]*/
  20541, 0,     40124, 0,     22690, 0,     0,     0,     0,     0,       /*[GB+9538E830, GB+9538E839]*/
  0,     27007, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9538E930, GB+9538E939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9538EA30, GB+9538EA39]*/
  9277,  0,     13903, 24461, 0,     0,     0,     1780,  0,     0,       /*[GB+9538EB30, GB+9538EB39]*/
  14075, 0,     0,     1933,  0,     0,     0,     0,     0,     21530,   /*[GB+9538EC30, GB+9538EC39]*/
  11792, 0,     0,     0,     6272,  0,     39141, 37221, 1056,  0,       /*[GB+9538ED30, GB+9538ED39]*/
  3349,  12935, 0,     37222, 0,     0,     0,     0,     0,     0,       /*[GB+9538EE30, GB+9538EE39]*/
  0,     0,     0,     9975,  36917, 0,     0,     0,     0,     0,       /*[GB+9538EF30, GB+9538EF39]*/
  0,     0,     0,     0,     0,     0,     0,     3788,  0,     27920,   /*[GB+9538F030, GB+9538F039]*/
  36757, 38034, 38035, 3266,  0,     0,     10735, 12230, 0,     0,       /*[GB+9538F130, GB+9538F139]*/
  0,     0,     29402, 30074, 26989, 30565, 42176, 0,     7802,  8313,    /*[GB+9538F230, GB+9538F239]*/
  0,     0,     0,     0,     30566, 0,     21412, 24049, 2237,  22336,   /*[GB+9538F330, GB+9538F339]*/
  31210, 469,   38079, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9538F430, GB+9538F439]*/
  0,     0,     36765, 0,     31319, 0,     37256, 28491, 16357, 31567,   /*[GB+9538F530, GB+9538F539]*/
  0,     36069, 0,     0,     0,     0,     0,     0,     0,     39357,   /*[GB+9538F630, GB+9538F639]*/
  0,     578,   0,     0,     0,     34090, 20216, 17189, 40668, 26285,   /*[GB+9538F730, GB+9538F739]*/
  0,     0,     4936,  43911, 19841, 0,     34091, 0,     0,     0,       /*[GB+9538F830, GB+9538F839]*/
  0,     0,     0,     0,     0,     0,     0,     28420, 0,     0,       /*[GB+9538F930, GB+9538F939]*/
  24659, 0,     14994, 25540, 38167, 0,     23010, 6964,  0,     14597,   /*[GB+9538FA30, GB+9538FA39]*/
  30085, 41293, 4894,  0,     0,     1067,  9976,  28332, 0,     14909,   /*[GB+9538FB30, GB+9538FB39]*/
  6042,  0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9538FC30, GB+9538FC39]*/
  0,     5435,  0,     22418, 0,     23282, 0,     13150, 36657, 0,       /*[GB+9538FD30, GB+9538FD39]*/
  23300, 0,     6881,  32866, 24218, 0,     0,     0,     11675, 0,       /*[GB+9538FE30, GB+9538FE39]*/
  13702, 37560, 16045, 17449, 17274, 38503, 0,     0,     0,     31910,   /*[GB+95398130, GB+95398139]*/
  0,     0,     0,     0,     0,     0,     0,     22599, 0,     0,       /*[GB+95398230, GB+95398239]*/
  0,     0,     43864, 0,     0,     0,     0,     20673, 29517, 42385,   /*[GB+95398330, GB+95398339]*/
  0,     0,     12407, 43238, 0,     12564, 0,     0,     0,     0,       /*[GB+95398430, GB+95398439]*/
  0,     4170,  0,     3286,  0,     0,     35606, 17453, 0,     4791,    /*[GB+95398530, GB+95398539]*/
  28551, 22093, 37707, 37298, 33868, 43932, 1929,  0,     0,     0,       /*[GB+95398630, GB+95398639]*/
  0,     0,     0,     0,     0,     0,     0,     39523, 0,     20077,   /*[GB+95398730, GB+95398739]*/
  0,     0,     0,     0,     31921, 32792, 0,     8143,  28510, 0,       /*[GB+95398830, GB+95398839]*/
  0,     0,     0,     0,     0,     0,     0,     18903, 0,     0,       /*[GB+95398930, GB+95398939]*/
  0,     0,     0,     0,     29176, 5498,  15721, 39550, 35400, 37320,   /*[GB+95398A30, GB+95398A39]*/
  29839, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+95398B30, GB+95398B39]*/
  12789, 0,     25329, 0,     0,     0,     0,     0,     20632, 0,       /*[GB+95398C30, GB+95398C39]*/
  0,     16166, 0,     0,     0,     19998, 0,     39165, 0,     0,       /*[GB+95398D30, GB+95398D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     34587, 0,       /*[GB+95398E30, GB+95398E39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     33329,   /*[GB+95398F30, GB+95398F39]*/
  0,     38674, 33752, 0,     0,     41112, 35289, 0,     0,     0,       /*[GB+95399030, GB+95399039]*/
  26532, 0,     0,     0,     18175, 0,     0,     0,     0,     25358,   /*[GB+95399130, GB+95399139]*/
  0,     0,     0,     0,     0,     0,     19409, 20345, 0,     0,       /*[GB+95399230, GB+95399239]*/
  0,     0,     2808,  0,     0,     0,     0,     0,     0,     0,       /*[GB+95399330, GB+95399339]*/
  0,     0,     0,     0,     41962, 0,     36658, 0,     0,     0,       /*[GB+95399430, GB+95399439]*/
  0,     0,     0,     0,     0,     0,     38686, 16983, 0,     0,       /*[GB+95399530, GB+95399539]*/
  0,     26371, 0,     8455,  0,     39713, 0,     0,     0,     0,       /*[GB+95399630, GB+95399639]*/
  0,     24837, 0,     0,     0,     0,     25089, 13357, 39982, 0,       /*[GB+95399730, GB+95399739]*/
  0,     0,     0,     0,     9189,  15738, 0,     24857, 0,     8888,    /*[GB+95399830, GB+95399839]*/
  0,     26733, 0,     0,     0,     0,     32867, 0,     0,     0,       /*[GB+95399930, GB+95399939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     42413,   /*[GB+95399A30, GB+95399A39]*/
  0,     0,     0,     13913, 0,     29642, 0,     0,     38334, 25324,   /*[GB+95399B30, GB+95399B39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+95399C30, GB+95399C39]*/
  18261, 0,     0,     40666, 38051, 0,     2317,  38052, 9210,  0,       /*[GB+95399D30, GB+95399D39]*/
  6343,  0,     42011, 38112, 0,     40551, 5364,  0,     29888, 0,       /*[GB+95399E30, GB+95399E39]*/
  0,     0,     0,     0,     14674, 3440,  26533, 0,     10730, 34295,   /*[GB+95399F30, GB+95399F39]*/
  0,     0,     0,     0,     0,     0,     26627, 14546, 0,     0,       /*[GB+9539A030, GB+9539A039]*/
  0,     15730, 0,     0,     0,     0,     5511,  0,     0,     14791,   /*[GB+9539A130, GB+9539A139]*/
  0,     0,     0,     0,     0,     0,     1857,  11642, 0,     0,       /*[GB+9539A230, GB+9539A239]*/
  0,     15813, 0,     39423, 0,     0,     0,     0,     0,     0,       /*[GB+9539A330, GB+9539A339]*/
  0,     18206, 25588, 29396, 16672, 0,     41750, 0,     0,     0,       /*[GB+9539A430, GB+9539A439]*/
  0,     8463,  28403, 0,     0,     0,     4466,  0,     0,     31568,   /*[GB+9539A530, GB+9539A539]*/
  0,     38113, 0,     37087, 0,     0,     32102, 42656, 13130, 0,       /*[GB+9539A630, GB+9539A639]*/
  0,     0,     0,     0,     36372, 0,     28494, 35969, 0,     0,       /*[GB+9539A730, GB+9539A739]*/
  0,     0,     0,     19469, 39989, 11662, 41965, 0,     0,     23870,   /*[GB+9539A830, GB+9539A839]*/
  0,     0,     0,     15853, 0,     0,     0,     0,     0,     0,       /*[GB+9539A930, GB+9539A939]*/
  0,     0,     0,     0,     42357, 24219, 40158, 0,     13755, 0,       /*[GB+9539AA30, GB+9539AA39]*/
  0,     0,     23687, 0,     0,     0,     43133, 0,     34343, 0,       /*[GB+9539AB30, GB+9539AB39]*/
  1083,  0,     4792,  0,     22094, 0,     0,     0,     10455, 0,       /*[GB+9539AC30, GB+9539AC39]*/
  0,     0,     3573,  25669, 0,     0,     0,     0,     0,     0,       /*[GB+9539AD30, GB+9539AD39]*/
  1408,  0,     0,     25625, 0,     42699, 0,     0,     0,     0,       /*[GB+9539AE30, GB+9539AE39]*/
  0,     0,     7760,  1736,  0,     0,     0,     0,     0,     0,       /*[GB+9539AF30, GB+9539AF39]*/
  0,     0,     0,     0,     2131,  27230, 0,     0,     0,     0,       /*[GB+9539B030, GB+9539B039]*/
  0,     6490,  26696, 15336, 37345, 0,     0,     0,     0,     0,       /*[GB+9539B130, GB+9539B139]*/
  0,     0,     0,     0,     0,     34074, 0,     0,     40724, 0,       /*[GB+9539B230, GB+9539B239]*/
  0,     0,     0,     0,     0,     5430,  0,     33365, 32117, 0,       /*[GB+9539B330, GB+9539B339]*/
  11599, 0,     0,     0,     0,     33630, 0,     0,     0,     0,       /*[GB+9539B430, GB+9539B439]*/
  17369, 0,     7934,  0,     0,     0,     0,     0,     0,     0,       /*[GB+9539B530, GB+9539B539]*/
  0,     0,     0,     11753, 0,     0,     0,     0,     0,     0,       /*[GB+9539B630, GB+9539B639]*/
  0,     0,     0,     0,     0,     0,     34203, 0,     0,     28056,   /*[GB+9539B730, GB+9539B739]*/
  0,     0,     0,     7992,  0,     0,     5192,  0,     8502,  0,       /*[GB+9539B830, GB+9539B839]*/
  4261,  0,     0,     0,     0,     0,     0,     0,     0,     6527,    /*[GB+9539B930, GB+9539B939]*/
  17579, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9539BA30, GB+9539BA39]*/
  0,     33869, 0,     0,     0,     0,     0,     1642,  39863, 30116,   /*[GB+9539BB30, GB+9539BB39]*/
  33879, 0,     0,     0,     0,     24617, 0,     28090, 0,     0,       /*[GB+9539BC30, GB+9539BC39]*/
  0,     0,     14432, 0,     38037, 0,     0,     32336, 13286, 0,       /*[GB+9539BD30, GB+9539BD39]*/
  32343, 31958, 0,     0,     0,     283,   0,     31189, 3684,  0,       /*[GB+9539BE30, GB+9539BE39]*/
  0,     0,     26850, 0,     27531, 38544, 23976, 4710,  31561, 38627,   /*[GB+9539BF30, GB+9539BF39]*/
  0,     0,     9260,  31321, 41874, 42752, 0,     37789, 16775, 29667,   /*[GB+9539C030, GB+9539C039]*/
  0,     0,     0,     6031,  0,     0,     0,     0,     35033, 0,       /*[GB+9539C130, GB+9539C139]*/
  34777, 10123, 32169, 8236,  33287, 0,     0,     35653, 0,     0,       /*[GB+9539C230, GB+9539C239]*/
  0,     0,     0,     0,     0,     0,     0,     39213, 0,     19283,   /*[GB+9539C330, GB+9539C339]*/
  36918, 0,     0,     6551,  25110, 30884, 10666, 0,     3700,  29673,   /*[GB+9539C430, GB+9539C439]*/
  0,     0,     0,     0,     0,     26718, 0,     0,     10758, 0,       /*[GB+9539C530, GB+9539C539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     27270,   /*[GB+9539C630, GB+9539C639]*/
  25118, 16129, 39480, 2663,  3613,  39772, 0,     1926,  0,     4779,    /*[GB+9539C730, GB+9539C739]*/
  42829, 0,     39785, 0,     0,     0,     0,     15873, 19594, 27708,   /*[GB+9539C830, GB+9539C839]*/
  39796, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9539C930, GB+9539C939]*/
  0,     0,     0,     0,     0,     0,     0,     29309, 0,     0,       /*[GB+9539CA30, GB+9539CA39]*/
  0,     0,     0,     40162, 0,     0,     41349, 0,     0,     29310,   /*[GB+9539CB30, GB+9539CB39]*/
  0,     4218,  41354, 27632, 34243, 0,     5027,  21369, 5941,  0,       /*[GB+9539CC30, GB+9539CC39]*/
  0,     0,     42414, 391,   0,     0,     0,     0,     0,     0,       /*[GB+9539CD30, GB+9539CD39]*/
  35609, 0,     24242, 0,     14465, 16848, 27549, 34556, 41086, 6205,    /*[GB+9539CE30, GB+9539CE39]*/
  0,     0,     0,     0,     0,     0,     0,     37878, 0,     0,       /*[GB+9539CF30, GB+9539CF39]*/
  12610, 28132, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9539D030, GB+9539D039]*/
  3971,  37894, 35241, 42458, 31397, 0,     24421, 34376, 0,     0,       /*[GB+9539D130, GB+9539D139]*/
  34259, 29971, 0,     4997,  0,     0,     0,     0,     0,     29722,   /*[GB+9539D230, GB+9539D239]*/
  0,     0,     0,     11233, 0,     34593, 0,     39275, 0,     39929,   /*[GB+9539D330, GB+9539D339]*/
  0,     0,     30503, 0,     15981, 22537, 0,     6105,  0,     38023,   /*[GB+9539D430, GB+9539D439]*/
  1220,  0,     0,     0,     0,     0,     0,     35679, 38039, 27350,   /*[GB+9539D530, GB+9539D539]*/
  0,     9478,  0,     0,     38040, 0,     0,     0,     0,     42250,   /*[GB+9539D630, GB+9539D639]*/
  35315, 0,     13333, 0,     34169, 0,     0,     0,     41427, 22640,   /*[GB+9539D730, GB+9539D739]*/
  35141, 0,     0,     0,     38054, 0,     0,     0,     0,     39619,   /*[GB+9539D830, GB+9539D839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9539D930, GB+9539D939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9539DA30, GB+9539DA39]*/
  0,     0,     0,     8641,  0,     28337, 0,     40037, 7789,  0,       /*[GB+9539DB30, GB+9539DB39]*/
  8370,  33418, 33525, 11435, 0,     0,     0,     13338, 3133,  0,       /*[GB+9539DC30, GB+9539DC39]*/
  0,     0,     0,     22033, 23821, 0,     0,     22201, 26265, 21017,   /*[GB+9539DD30, GB+9539DD39]*/
  0,     5247,  0,     34078, 28404, 30931, 11350, 1245,  0,     0,       /*[GB+9539DE30, GB+9539DE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     19735,   /*[GB+9539DF30, GB+9539DF39]*/
  0,     0,     0,     22752, 0,     25696, 0,     0,     0,     0,       /*[GB+9539E030, GB+9539E039]*/
  0,     0,     0,     27380, 0,     0,     0,     0,     0,     0,       /*[GB+9539E130, GB+9539E139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9539E230, GB+9539E239]*/
  14207, 2652,  21502, 0,     0,     38114, 0,     30199, 0,     33432,   /*[GB+9539E330, GB+9539E339]*/
  35967, 0,     0,     33787, 0,     37955, 0,     0,     0,     0,       /*[GB+9539E430, GB+9539E439]*/
  0,     0,     0,     0,     22551, 20700, 34766, 32853, 0,     0,       /*[GB+9539E530, GB+9539E539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9539E630, GB+9539E639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9539E730, GB+9539E739]*/
  0,     0,     0,     0,     16244, 0,     0,     19765, 20217, 43184,   /*[GB+9539E830, GB+9539E839]*/
  0,     0,     0,     0,     0,     0,     42295, 0,     35512, 0,       /*[GB+9539E930, GB+9539E939]*/
  9424,  14897, 34511, 13377, 5193,  0,     17705, 8171,  15592, 0,       /*[GB+9539EA30, GB+9539EA39]*/
  0,     0,     0,     21745, 34092, 0,     0,     0,     0,     29869,   /*[GB+9539EB30, GB+9539EB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9539EC30, GB+9539EC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9539ED30, GB+9539ED39]*/
  0,     0,     42296, 0,     11832, 16637, 0,     30241, 22806, 0,       /*[GB+9539EE30, GB+9539EE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9539EF30, GB+9539EF39]*/
  30947, 42997, 0,     0,     0,     0,     0,     5262,  0,     26298,   /*[GB+9539F030, GB+9539F039]*/
  36526, 0,     34319, 0,     0,     22671, 0,     0,     0,     42204,   /*[GB+9539F130, GB+9539F139]*/
  1756,  0,     0,     0,     3983,  17446, 6552,  26830, 34718, 0,       /*[GB+9539F230, GB+9539F239]*/
  34898, 32378, 27353, 32550, 22812, 0,     547,   36527, 0,     0,       /*[GB+9539F330, GB+9539F339]*/
  0,     0,     0,     29509, 38633, 0,     39748, 0,     0,     0,       /*[GB+9539F430, GB+9539F439]*/
  0,     0,     18040, 12824, 0,     0,     0,     0,     0,     0,       /*[GB+9539F530, GB+9539F539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9539F630, GB+9539F639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9539F730, GB+9539F739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9539F830, GB+9539F839]*/
  0,     0,     0,     734,   0,     0,     0,     0,     0,     0,       /*[GB+9539F930, GB+9539F939]*/
  0,     0,     0,     0,     0,     33579, 0,     15227, 0,     0,       /*[GB+9539FA30, GB+9539FA39]*/
  35187, 25291, 23412, 35495, 7146,  13306, 33368, 22203, 36926, 25119,   /*[GB+9539FB30, GB+9539FB39]*/
  22310, 3832,  11490, 0,     27742, 0,     22559, 0,     0,     3604,    /*[GB+9539FC30, GB+9539FC39]*/
  3605,  17048, 0,     13703, 31769, 11297, 6772,  0,     0,     34108,   /*[GB+9539FD30, GB+9539FD39]*/
  42632, 12833, 3963,  5268,  0,     0,     21043, 37426, 0,     0,       /*[GB+9539FE30, GB+9539FE39]*/
  0,     0,     0,     15694, 0,     0,     0,     0,     0,     0,       /*[GB+96308130, GB+96308139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96308230, GB+96308239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     28932, 0,       /*[GB+96308330, GB+96308339]*/
  0,     38572, 0,     0,     0,     9781,  33454, 0,     0,     0,       /*[GB+96308430, GB+96308439]*/
  10523, 31142, 0,     11298, 0,     0,     0,     0,     0,     0,       /*[GB+96308530, GB+96308539]*/
  0,     0,     0,     0,     0,     26975, 0,     37834, 0,     32031,   /*[GB+96308630, GB+96308639]*/
  0,     0,     0,     26157, 10593, 0,     25126, 0,     0,     12840,   /*[GB+96308730, GB+96308739]*/
  31147, 24010, 0,     0,     9436,  0,     35201, 0,     21562, 9193,    /*[GB+96308830, GB+96308839]*/
  30322, 34685, 0,     29105, 0,     0,     8559,  0,     0,     0,       /*[GB+96308930, GB+96308939]*/
  0,     0,     43092, 19595, 0,     0,     0,     0,     0,     0,       /*[GB+96308A30, GB+96308A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96308B30, GB+96308B39]*/
  0,     0,     0,     0,     0,     0,     0,     39236, 0,     0,       /*[GB+96308C30, GB+96308C39]*/
  0,     0,     3608,  6969,  0,     0,     0,     0,     0,     0,       /*[GB+96308D30, GB+96308D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96308E30, GB+96308E39]*/
  0,     0,     0,     0,     0,     0,     18087, 37196, 18088, 0,       /*[GB+96308F30, GB+96308F39]*/
  0,     19981, 22236, 30100, 100,   0,     0,     16646, 0,     41576,   /*[GB+96309030, GB+96309039]*/
  101,   30960, 0,     20270, 31916, 1150,  0,     0,     0,     0,       /*[GB+96309130, GB+96309139]*/
  0,     0,     21370, 0,     12580, 12113, 0,     0,     0,     0,       /*[GB+96309230, GB+96309239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96309330, GB+96309339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96309430, GB+96309439]*/
  0,     0,     0,     17220, 0,     35346, 0,     0,     0,     27743,   /*[GB+96309530, GB+96309539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     29251,   /*[GB+96309630, GB+96309639]*/
  26784, 460,   31969, 0,     0,     0,     8783,  1731,  0,     30798,   /*[GB+96309730, GB+96309739]*/
  0,     34735, 0,     29538, 2082,  0,     20704, 5676,  0,     10306,   /*[GB+96309830, GB+96309839]*/
  11228, 33674, 5677,  38301, 0,     1152,  0,     9730,  5094,  27785,   /*[GB+96309930, GB+96309939]*/
  0,     0,     0,     9309,  39842, 22324, 0,     18410, 42433, 0,       /*[GB+96309A30, GB+96309A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96309B30, GB+96309B39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96309C30, GB+96309C39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     41500, 0,       /*[GB+96309D30, GB+96309D39]*/
  0,     0,     13010, 27400, 0,     0,     0,     3999,  0,     0,       /*[GB+96309E30, GB+96309E39]*/
  0,     0,     0,     0,     0,     0,     14651, 0,     0,     0,       /*[GB+96309F30, GB+96309F39]*/
  0,     0,     0,     0,     0,     0,     3149,  41585, 0,     0,       /*[GB+9630A030, GB+9630A039]*/
  0,     0,     29940, 39026, 0,     0,     25155, 43379, 0,     0,       /*[GB+9630A130, GB+9630A139]*/
  16185, 0,     37606, 0,     25156, 0,     0,     0,     0,     0,       /*[GB+9630A230, GB+9630A239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9630A330, GB+9630A339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9630A430, GB+9630A439]*/
  0,     35578, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9630A530, GB+9630A539]*/
  0,     0,     0,     0,     0,     0,     0,     37717, 4117,  0,       /*[GB+9630A630, GB+9630A639]*/
  0,     0,     31891, 123,   24354, 0,     0,     19063, 21631, 124,     /*[GB+9630A730, GB+9630A739]*/
  0,     22019, 32913, 38987, 22774, 0,     0,     0,     2108,  0,       /*[GB+9630A830, GB+9630A839]*/
  19603, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9630A930, GB+9630A939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     27461,   /*[GB+9630AA30, GB+9630AA39]*/
  0,     4192,  0,     0,     0,     0,     0,     20198, 2721,  0,       /*[GB+9630AB30, GB+9630AB39]*/
  0,     4304,  0,     27750, 18136, 0,     29697, 0,     18468, 0,       /*[GB+9630AC30, GB+9630AC39]*/
  0,     0,     0,     34014, 0,     36614, 5154,  0,     19527, 0,       /*[GB+9630AD30, GB+9630AD39]*/
  0,     0,     0,     0,     0,     0,     21308, 0,     41323, 0,       /*[GB+9630AE30, GB+9630AE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     43201, 0,       /*[GB+9630AF30, GB+9630AF39]*/
  0,     21739, 0,     18490, 0,     15551, 0,     0,     0,     35274,   /*[GB+9630B030, GB+9630B039]*/
  35275, 19748, 19678, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9630B130, GB+9630B139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     30978, 43052,   /*[GB+9630B230, GB+9630B239]*/
  0,     0,     0,     0,     0,     0,     0,     2894,  34456, 0,       /*[GB+9630B330, GB+9630B339]*/
  0,     38903, 23838, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9630B430, GB+9630B439]*/
  35114, 0,     14481, 0,     0,     0,     3948,  0,     0,     0,       /*[GB+9630B530, GB+9630B539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9630B630, GB+9630B639]*/
  0,     0,     0,     0,     0,     0,     0,     11027, 0,     0,       /*[GB+9630B730, GB+9630B739]*/
  0,     0,     0,     0,     0,     0,     0,     32035, 0,     0,       /*[GB+9630B830, GB+9630B839]*/
  0,     0,     0,     0,     0,     11446, 0,     0,     20804, 0,       /*[GB+9630B930, GB+9630B939]*/
  14840, 0,     26460, 6669,  26257, 0,     0,     28560, 40546, 0,       /*[GB+9630BA30, GB+9630BA39]*/
  0,     37943, 0,     0,     0,     0,     11972, 0,     0,     0,       /*[GB+9630BB30, GB+9630BB39]*/
  0,     16193, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9630BC30, GB+9630BC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     40219, 22801,   /*[GB+9630BD30, GB+9630BD39]*/
  29502, 0,     33149, 0,     0,     0,     2392,  0,     13735, 0,       /*[GB+9630BE30, GB+9630BE39]*/
  21760, 28263, 0,     27209, 8955,  0,     0,     0,     0,     25591,   /*[GB+9630BF30, GB+9630BF39]*/
  0,     29142, 43344, 0,     0,     0,     0,     0,     0,     28689,   /*[GB+9630C030, GB+9630C039]*/
  41300, 31143, 0,     0,     0,     0,     40999, 41175, 0,     0,       /*[GB+9630C130, GB+9630C139]*/
  0,     0,     0,     8189,  21768, 0,     40879, 0,     0,     0,       /*[GB+9630C230, GB+9630C239]*/
  42415, 25309, 10786, 9300,  0,     0,     0,     0,     0,     0,       /*[GB+9630C330, GB+9630C339]*/
  2402,  0,     10457, 14280, 0,     39018, 329,   0,     0,     0,       /*[GB+9630C430, GB+9630C439]*/
  0,     0,     0,     41007, 0,     33981, 0,     0,     0,     35740,   /*[GB+9630C530, GB+9630C539]*/
  33383, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9630C630, GB+9630C639]*/
  0,     0,     0,     15930, 13026, 0,     3875,  0,     32419, 0,       /*[GB+9630C730, GB+9630C739]*/
  21780, 40889, 0,     0,     0,     0,     14697, 28805, 0,     33753,   /*[GB+9630C830, GB+9630C839]*/
  0,     7495,  0,     0,     0,     11799, 11803, 0,     18269, 0,       /*[GB+9630C930, GB+9630C939]*/
  0,     4239,  0,     0,     37676, 21001, 35684, 42552, 0,     38084,   /*[GB+9630CA30, GB+9630CA39]*/
  0,     0,     0,     0,     0,     0,     18878, 0,     0,     0,       /*[GB+9630CB30, GB+9630CB39]*/
  0,     13730, 13871, 15432, 0,     0,     42987, 0,     0,     0,       /*[GB+9630CC30, GB+9630CC39]*/
  33359, 0,     0,     0,     0,     0,     0,     16462, 16430, 18371,   /*[GB+9630CD30, GB+9630CD39]*/
  1198,  9279,  0,     35707, 27886, 19315, 16387, 0,     0,     0,       /*[GB+9630CE30, GB+9630CE39]*/
  0,     0,     2083,  1501,  29325, 24968, 28166, 0,     0,     0,       /*[GB+9630CF30, GB+9630CF39]*/
  12913, 0,     25741, 0,     0,     0,     0,     0,     38041, 15809,   /*[GB+9630D030, GB+9630D039]*/
  41385, 0,     35374, 0,     0,     0,     0,     32045, 10575, 34170,   /*[GB+9630D130, GB+9630D139]*/
  0,     32086, 37751, 0,     0,     0,     0,     0,     0,     4932,    /*[GB+9630D230, GB+9630D239]*/
  0,     0,     0,     14131, 22448, 0,     0,     23951, 0,     14559,   /*[GB+9630D330, GB+9630D339]*/
  36146, 11645, 0,     0,     0,     15521, 0,     0,     0,     0,       /*[GB+9630D430, GB+9630D439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9630D530, GB+9630D539]*/
  0,     0,     0,     39284, 23324, 37527, 0,     0,     38687, 0,       /*[GB+9630D630, GB+9630D639]*/
  16595, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9630D730, GB+9630D739]*/
  34762, 34612, 0,     0,     0,     0,     22548, 0,     41405, 0,       /*[GB+9630D830, GB+9630D839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9630D930, GB+9630D939]*/
  0,     0,     0,     0,     0,     0,     14565, 0,     0,     23326,   /*[GB+9630DA30, GB+9630DA39]*/
  28098, 0,     0,     0,     19893, 0,     0,     0,     0,     0,       /*[GB+9630DB30, GB+9630DB39]*/
  0,     0,     0,     0,     13621, 0,     0,     0,     0,     42726,   /*[GB+9630DC30, GB+9630DC39]*/
  27203, 0,     0,     20028, 27535, 23334, 0,     2012,  41538, 0,       /*[GB+9630DD30, GB+9630DD39]*/
  25966, 0,     4629,  43804, 0,     33150, 33356, 0,     0,     22983,   /*[GB+9630DE30, GB+9630DE39]*/
  0,     0,     0,     34214, 11362, 0,     0,     0,     0,     0,       /*[GB+9630DF30, GB+9630DF39]*/
  0,     0,     0,     0,     0,     0,     0,     37542, 0,     0,       /*[GB+9630E030, GB+9630E039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9630E130, GB+9630E139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9630E230, GB+9630E239]*/
  0,     0,     0,     0,     0,     0,     0,     31535, 14848, 0,       /*[GB+9630E330, GB+9630E339]*/
  0,     0,     0,     14585, 11176, 31719, 41550, 24348, 14898, 22503,   /*[GB+9630E430, GB+9630E439]*/
  0,     0,     0,     0,     7093,  0,     33559, 35165, 0,     33936,   /*[GB+9630E530, GB+9630E539]*/
  17706, 6924,  13378, 2655,  4008,  0,     9076,  0,     0,     0,       /*[GB+9630E630, GB+9630E639]*/
  0,     0,     34658, 5720,  0,     0,     0,     0,     0,     0,       /*[GB+9630E730, GB+9630E739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9630E830, GB+9630E839]*/
  0,     0,     0,     0,     0,     1572,  34778, 35330, 0,     0,       /*[GB+9630E930, GB+9630E939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9630EA30, GB+9630EA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9630EB30, GB+9630EB39]*/
  0,     0,     0,     39644, 34320, 18041, 25446, 0,     0,     21155,   /*[GB+9630EC30, GB+9630EC39]*/
  0,     25643, 36529, 0,     25192, 26198, 0,     735,   41889, 0,       /*[GB+9630ED30, GB+9630ED39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     40875, 29328,   /*[GB+9630EE30, GB+9630EE39]*/
  0,     0,     27741, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9630EF30, GB+9630EF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9630F030, GB+9630F039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9630F130, GB+9630F139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9630F230, GB+9630F239]*/
  0,     20040, 0,     0,     0,     35045, 0,     8662,  41139, 0,       /*[GB+9630F330, GB+9630F339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9630F430, GB+9630F439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9630F530, GB+9630F539]*/
  0,     0,     0,     0,     0,     0,     0,     40878, 0,     6936,    /*[GB+9630F630, GB+9630F639]*/
  0,     5854,  39773, 0,     20264, 0,     32444, 36175, 14358, 40196,   /*[GB+9630F730, GB+9630F739]*/
  42359, 32760, 0,     9280,  15532, 0,     36800, 5199,  21044, 0,       /*[GB+9630F830, GB+9630F839]*/
  0,     0,     0,     0,     0,     0,     8542,  19,    0,     0,       /*[GB+9630F930, GB+9630F939]*/
  35982, 0,     9083,  166,   0,     0,     0,     0,     13166, 0,       /*[GB+9630FA30, GB+9630FA39]*/
  0,     0,     0,     0,     24077, 42214, 0,     0,     23384, 0,       /*[GB+9630FB30, GB+9630FB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9630FC30, GB+9630FC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9630FD30, GB+9630FD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9630FE30, GB+9630FE39]*/
  0,     0,     0,     23652, 0,     37835, 0,     0,     14610, 0,       /*[GB+96318130, GB+96318139]*/
  43463, 0,     13704, 32292, 0,     0,     13705, 0,     0,     0,       /*[GB+96318230, GB+96318239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96318330, GB+96318339]*/
  0,     0,     0,     0,     0,     2954,  0,     30257, 30258, 13180,   /*[GB+96318430, GB+96318439]*/
  29715, 0,     31002, 24012, 24522, 0,     0,     0,     0,     0,       /*[GB+96318530, GB+96318539]*/
  21564, 0,     0,     0,     0,     0,     37568, 0,     26008, 14923,   /*[GB+96318630, GB+96318639]*/
  0,     0,     15791, 0,     0,     40002, 27372, 0,     0,     0,       /*[GB+96318730, GB+96318739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96318830, GB+96318839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     679,   0,       /*[GB+96318930, GB+96318939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96318A30, GB+96318A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96318B30, GB+96318B39]*/
  0,     0,     0,     0,     0,     0,     1799,  0,     0,     0,       /*[GB+96318C30, GB+96318C39]*/
  32885, 40764, 25839, 29769, 19982, 0,     0,     0,     6394,  4326,    /*[GB+96318D30, GB+96318D39]*/
  1363,  42892, 20618, 8027,  24331, 38710, 36117, 0,     392,   43362,   /*[GB+96318E30, GB+96318E39]*/
  43819, 0,     0,     1205,  0,     0,     0,     0,     0,     17681,   /*[GB+96318F30, GB+96318F39]*/
  0,     31152, 0,     0,     0,     0,     0,     31384, 13782, 0,       /*[GB+96319030, GB+96319039]*/
  0,     0,     0,     31647, 0,     0,     0,     0,     0,     0,       /*[GB+96319130, GB+96319139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96319230, GB+96319239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96319330, GB+96319339]*/
  0,     0,     0,     7132,  6629,  0,     0,     0,     1533,  0,       /*[GB+96319430, GB+96319439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96319530, GB+96319539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96319630, GB+96319639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     42435, 0,       /*[GB+96319730, GB+96319739]*/
  8144,  0,     0,     0,     16300, 42436, 41105, 1384,  16849, 0,       /*[GB+96319830, GB+96319839]*/
  683,   15796, 26427, 0,     24786, 17845, 35077, 30698, 0,     0,       /*[GB+96319930, GB+96319939]*/
  0,     0,     0,     0,     0,     0,     29772, 0,     990,   0,       /*[GB+96319A30, GB+96319A39]*/
  0,     0,     0,     0,     0,     0,     0,     37204, 13924, 0,       /*[GB+96319B30, GB+96319B39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96319C30, GB+96319C39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96319D30, GB+96319D39]*/
  0,     0,     0,     0,     0,     0,     0,     40765, 0,     0,       /*[GB+96319E30, GB+96319E39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96319F30, GB+96319F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9631A030, GB+9631A039]*/
  0,     0,     0,     0,     43078, 0,     0,     7746,  0,     28329,   /*[GB+9631A130, GB+9631A139]*/
  0,     26176, 0,     24584, 0,     31415, 27903, 40527, 0,     0,       /*[GB+9631A230, GB+9631A239]*/
  0,     22956, 9964,  5053,  0,     0,     0,     0,     0,     0,       /*[GB+9631A330, GB+9631A339]*/
  0,     0,     0,     0,     0,     0,     0,     25778, 0,     6406,    /*[GB+9631A430, GB+9631A439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9631A530, GB+9631A539]*/
  0,     0,     0,     0,     0,     24351, 4692,  0,     754,   0,       /*[GB+9631A630, GB+9631A639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9631A730, GB+9631A739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9631A830, GB+9631A839]*/
  0,     38728, 0,     0,     0,     34444, 0,     0,     37996, 21735,   /*[GB+9631A930, GB+9631A939]*/
  6928,  42791, 0,     16591, 0,     0,     0,     0,     0,     0,       /*[GB+9631AA30, GB+9631AA39]*/
  0,     0,     41369, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9631AB30, GB+9631AB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9631AC30, GB+9631AC39]*/
  0,     0,     19528, 0,     42239, 2980,  6807,  40113, 0,     0,       /*[GB+9631AD30, GB+9631AD39]*/
  30307, 14686, 18469, 9570,  0,     0,     0,     0,     0,     0,       /*[GB+9631AE30, GB+9631AE39]*/
  0,     0,     0,     0,     715,   0,     0,     0,     0,     0,       /*[GB+9631AF30, GB+9631AF39]*/
  0,     0,     0,     0,     0,     10471, 18924, 28520, 0,     0,       /*[GB+9631B030, GB+9631B039]*/
  0,     0,     0,     0,     0,     0,     0,     22966, 0,     5013,    /*[GB+9631B130, GB+9631B139]*/
  32832, 0,     32080, 0,     0,     29845, 0,     19679, 0,     0,       /*[GB+9631B230, GB+9631B239]*/
  0,     0,     0,     34862, 36867, 0,     0,     0,     0,     0,       /*[GB+9631B330, GB+9631B339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     26092,   /*[GB+9631B430, GB+9631B439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     33754,   /*[GB+9631B530, GB+9631B539]*/
  0,     39587, 0,     41987, 35291, 2773,  8210,  0,     10476, 0,       /*[GB+9631B630, GB+9631B639]*/
  15072, 17594, 34277, 0,     0,     19376, 0,     0,     0,     0,       /*[GB+9631B730, GB+9631B739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9631B830, GB+9631B839]*/
  0,     0,     17939, 0,     0,     0,     16105, 34986, 0,     0,       /*[GB+9631B930, GB+9631B939]*/
  0,     0,     0,     0,     0,     0,     40535, 0,     0,     0,       /*[GB+9631BA30, GB+9631BA39]*/
  0,     0,     0,     0,     9999,  25867, 20600, 37463, 20328, 30453,   /*[GB+9631BB30, GB+9631BB39]*/
  0,     0,     0,     0,     0,     0,     0,     20601, 0,     0,       /*[GB+9631BC30, GB+9631BC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9631BD30, GB+9631BD39]*/
  502,   3034,  0,     43422, 0,     0,     0,     0,     0,     0,       /*[GB+9631BE30, GB+9631BE39]*/
  0,     0,     0,     31388, 0,     0,     0,     32329, 19411, 0,       /*[GB+9631BF30, GB+9631BF39]*/
  0,     0,     39967, 25068, 0,     24838, 0,     0,     0,     0,       /*[GB+9631C030, GB+9631C039]*/
  13348, 2005,  0,     28674, 8653,  0,     10143, 0,     0,     6150,    /*[GB+9631C130, GB+9631C139]*/
  0,     10889, 0,     8889,  0,     24877, 0,     0,     0,     0,       /*[GB+9631C230, GB+9631C239]*/
  0,     0,     0,     3545,  12551, 0,     0,     34114, 28787, 42387,   /*[GB+9631C330, GB+9631C339]*/
  24921, 0,     10596, 0,     0,     14389, 18427, 0,     18959, 0,       /*[GB+9631C430, GB+9631C439]*/
  25674, 0,     0,     0,     27560, 0,     24815, 28666, 0,     0,       /*[GB+9631C530, GB+9631C539]*/
  14560, 0,     0,     0,     0,     0,     0,     929,   1487,  432,     /*[GB+9631C630, GB+9631C639]*/
  16016, 23979, 0,     0,     5482,  0,     0,     30713, 17044, 16674,   /*[GB+9631C730, GB+9631C739]*/
  0,     16626, 0,     0,     0,     0,     0,     28675, 28676, 12943,   /*[GB+9631C830, GB+9631C839]*/
  843,   0,     0,     0,     16427, 16866, 0,     0,     0,     0,       /*[GB+9631C930, GB+9631C939]*/
  21868, 0,     0,     18879, 7370,  9268,  0,     0,     0,     32510,   /*[GB+9631CA30, GB+9631CA39]*/
  7139,  0,     0,     25216, 11928, 0,     11436, 0,     0,     23866,   /*[GB+9631CB30, GB+9631CB39]*/
  0,     14440, 0,     0,     0,     0,     26389, 43601, 42758, 16871,   /*[GB+9631CC30, GB+9631CC39]*/
  0,     27864, 0,     13379, 32564, 24636, 36171, 0,     28232, 0,       /*[GB+9631CD30, GB+9631CD39]*/
  0,     0,     0,     0,     17621, 42205, 31518, 0,     0,     5313,    /*[GB+9631CE30, GB+9631CE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9631CF30, GB+9631CF39]*/
  0,     0,     0,     37099, 14910, 42206, 0,     0,     43345, 0,       /*[GB+9631D030, GB+9631D039]*/
  40516, 2844,  0,     0,     0,     0,     21045, 0,     0,     0,       /*[GB+9631D130, GB+9631D139]*/
  16579, 42678, 11600, 23876, 0,     0,     0,     15303, 0,     0,       /*[GB+9631D230, GB+9631D239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     4598,  21879,   /*[GB+9631D330, GB+9631D339]*/
  0,     0,     0,     0,     34954, 0,     6852,  32657, 0,     2400,    /*[GB+9631D430, GB+9631D439]*/
  39504, 6886,  16134, 24176, 0,     0,     3559,  0,     0,     0,       /*[GB+9631D530, GB+9631D539]*/
  4570,  0,     28272, 36814, 16813, 0,     40186, 0,     0,     0,       /*[GB+9631D630, GB+9631D639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9631D730, GB+9631D739]*/
  0,     0,     0,     0,     24922, 41912, 0,     16478, 24923, 42226,   /*[GB+9631D830, GB+9631D839]*/
  20080, 0,     0,     0,     0,     24018, 23418, 23751, 0,     0,       /*[GB+9631D930, GB+9631D939]*/
  0,     0,     0,     0,     7973,  0,     0,     0,     0,     33010,   /*[GB+9631DA30, GB+9631DA39]*/
  0,     0,     21274, 0,     0,     0,     0,     0,     28084, 0,       /*[GB+9631DB30, GB+9631DB39]*/
  0,     0,     35576, 0,     7253,  1711,  0,     0,     0,     0,       /*[GB+9631DC30, GB+9631DC39]*/
  0,     0,     0,     0,     0,     0,     39159, 0,     20842, 0,       /*[GB+9631DD30, GB+9631DD39]*/
  18960, 0,     0,     35085, 20364, 25675, 5863,  0,     3578,  3579,    /*[GB+9631DE30, GB+9631DE39]*/
  0,     0,     0,     0,     0,     4613,  0,     0,     0,     0,       /*[GB+9631DF30, GB+9631DF39]*/
  0,     0,     0,     0,     0,     0,     2593,  0,     0,     0,       /*[GB+9631E030, GB+9631E039]*/
  0,     0,     0,     0,     17853, 41259, 18239, 18669, 26845, 0,       /*[GB+9631E130, GB+9631E139]*/
  0,     0,     0,     0,     0,     0,     0,     3398,  0,     3583,    /*[GB+9631E230, GB+9631E239]*/
  9739,  0,     43743, 4085,  4399,  0,     0,     0,     0,     0,       /*[GB+9631E330, GB+9631E339]*/
  0,     0,     17985, 29478, 20292, 0,     0,     0,     0,     8633,    /*[GB+9631E430, GB+9631E439]*/
  0,     18507, 0,     27666, 0,     0,     0,     27621, 0,     0,       /*[GB+9631E530, GB+9631E539]*/
  0,     27571, 0,     0,     0,     23098, 0,     0,     43882, 18252,   /*[GB+9631E630, GB+9631E639]*/
  0,     0,     0,     29200, 0,     0,     8636,  0,     0,     0,       /*[GB+9631E730, GB+9631E739]*/
  0,     0,     0,     0,     0,     0,     0,     1260,  0,     0,       /*[GB+9631E830, GB+9631E839]*/
  0,     5746,  0,     0,     0,     0,     28682, 0,     0,     0,       /*[GB+9631E930, GB+9631E939]*/
  9029,  30675, 0,     0,     0,     0,     20843, 0,     0,     18098,   /*[GB+9631EA30, GB+9631EA39]*/
  0,     0,     1409,  0,     0,     0,     23619, 0,     39173, 0,       /*[GB+9631EB30, GB+9631EB39]*/
  0,     0,     0,     15214, 0,     10398, 18877, 0,     0,     33177,   /*[GB+9631EC30, GB+9631EC39]*/
  25217, 6678,  0,     19030, 0,     14348, 0,     19031, 0,     0,       /*[GB+9631ED30, GB+9631ED39]*/
  0,     0,     0,     31329, 0,     0,     10412, 0,     23669, 25662,   /*[GB+9631EE30, GB+9631EE39]*/
  0,     5948,  40453, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9631EF30, GB+9631EF39]*/
  17816, 0,     0,     0,     29496, 35321, 0,     0,     0,     0,       /*[GB+9631F030, GB+9631F039]*/
  35331, 34714, 20522, 0,     0,     0,     0,     0,     1070,  31326,   /*[GB+9631F130, GB+9631F139]*/
  0,     0,     0,     0,     0,     0,     34955, 0,     6651,  0,       /*[GB+9631F230, GB+9631F239]*/
  0,     0,     0,     5667,  31479, 43134, 0,     0,     0,     0,       /*[GB+9631F330, GB+9631F339]*/
  0,     23457, 0,     33201, 0,     0,     0,     42953, 0,     0,       /*[GB+9631F430, GB+9631F439]*/
  35354, 27404, 0,     0,     43391, 5684,  0,     0,     0,     5108,    /*[GB+9631F530, GB+9631F539]*/
  0,     0,     0,     43404, 0,     0,     0,     0,     0,     4840,    /*[GB+9631F630, GB+9631F639]*/
  2927,  0,     0,     0,     0,     37151, 0,     0,     0,     0,       /*[GB+9631F730, GB+9631F739]*/
  0,     0,     0,     0,     0,     36778, 0,     0,     0,     0,       /*[GB+9631F830, GB+9631F839]*/
  0,     0,     0,     0,     0,     0,     0,     36785, 0,     41822,   /*[GB+9631F930, GB+9631F939]*/
  0,     0,     0,     0,     0,     0,     0,     23399, 36928, 0,       /*[GB+9631FA30, GB+9631FA39]*/
  0,     0,     0,     0,     0,     7901,  0,     0,     0,     36815,   /*[GB+9631FB30, GB+9631FB39]*/
  39506, 0,     0,     31043, 8768,  1030,  0,     36827, 0,     12416,   /*[GB+9631FC30, GB+9631FC39]*/
  0,     0,     0,     0,     0,     12355, 0,     0,     10314, 36983,   /*[GB+9631FD30, GB+9631FD39]*/
  0,     11744, 1661,  0,     0,     0,     0,     0,     0,     30161,   /*[GB+9631FE30, GB+9631FE39]*/
  0,     0,     0,     0,     0,     0,     43594, 13340, 7332,  13380,   /*[GB+96328130, GB+96328139]*/
  17045, 0,     18858, 0,     0,     0,     12841, 0,     32675, 43285,   /*[GB+96328230, GB+96328239]*/
  0,     0,     0,     31505, 0,     0,     0,     40562, 39344, 0,       /*[GB+96328330, GB+96328339]*/
  0,     0,     0,     27040, 0,     0,     0,     21690, 0,     0,       /*[GB+96328430, GB+96328439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96328530, GB+96328539]*/
  22337, 0,     0,     31714, 0,     5424,  0,     15822, 0,     0,       /*[GB+96328630, GB+96328639]*/
  35685, 5887,  0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96328730, GB+96328739]*/
  0,     0,     34709, 0,     0,     0,     0,     0,     0,     35565,   /*[GB+96328830, GB+96328839]*/
  0,     0,     39291, 9913,  0,     0,     33545, 0,     0,     0,       /*[GB+96328930, GB+96328939]*/
  0,     0,     0,     0,     0,     0,     11174, 0,     0,     0,       /*[GB+96328A30, GB+96328A39]*/
  0,     6474,  0,     0,     19900, 0,     35777, 0,     42759, 34514,   /*[GB+96328B30, GB+96328B39]*/
  12240, 2656,  28831, 9025,  23398, 227,   0,     0,     0,     34994,   /*[GB+96328C30, GB+96328C39]*/
  0,     0,     0,     0,     0,     0,     0,     10960, 0,     0,       /*[GB+96328D30, GB+96328D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     25882, 9431,    /*[GB+96328E30, GB+96328E39]*/
  28013, 8519,  0,     0,     0,     0,     0,     0,     0,     40876,   /*[GB+96328F30, GB+96328F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     5530,    /*[GB+96329030, GB+96329039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96329130, GB+96329139]*/
  0,     0,     0,     0,     6153,  28430, 0,     43862, 0,     0,       /*[GB+96329230, GB+96329239]*/
  0,     0,     0,     0,     0,     0,     1720,  0,     0,     0,       /*[GB+96329330, GB+96329339]*/
  4474,  41651, 5380,  0,     0,     33155, 26145, 0,     0,     0,       /*[GB+96329430, GB+96329439]*/
  19121, 0,     12690, 12975, 0,     0,     0,     0,     0,     0,       /*[GB+96329530, GB+96329539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96329630, GB+96329639]*/
  0,     2511,  0,     0,     0,     0,     22265, 0,     0,     6375,    /*[GB+96329730, GB+96329739]*/
  32869, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96329830, GB+96329839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     1578,  31112,   /*[GB+96329930, GB+96329939]*/
  0,     38973, 0,     31780, 0,     0,     0,     0,     0,     0,       /*[GB+96329A30, GB+96329A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96329B30, GB+96329B39]*/
  9457,  0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96329C30, GB+96329C39]*/
  0,     0,     0,     0,     3711,  4899,  31114, 0,     0,     0,       /*[GB+96329D30, GB+96329D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     12847, 24777,   /*[GB+96329E30, GB+96329E39]*/
  27711, 27700, 36076, 289,   22440, 0,     37123, 29113, 27892, 0,       /*[GB+96329F30, GB+96329F39]*/
  0,     0,     0,     31526, 0,     0,     0,     0,     0,     0,       /*[GB+9632A030, GB+9632A039]*/
  0,     36549, 12350, 0,     0,     0,     1803,  0,     0,     0,       /*[GB+9632A130, GB+9632A139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9632A230, GB+9632A239]*/
  107,   0,     23506, 2520,  0,     330,   0,     21070, 0,     0,       /*[GB+9632A330, GB+9632A339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9632A430, GB+9632A439]*/
  0,     0,     0,     0,     21279, 0,     31202, 0,     0,     0,       /*[GB+9632A530, GB+9632A539]*/
  0,     0,     0,     27701, 35932, 16252, 0,     0,     0,     0,       /*[GB+9632A630, GB+9632A639]*/
  0,     0,     0,     17670, 0,     0,     0,     0,     0,     0,       /*[GB+9632A730, GB+9632A739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     1734,  4868,    /*[GB+9632A830, GB+9632A839]*/
  0,     0,     34360, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9632A930, GB+9632A939]*/
  0,     0,     31989, 0,     0,     0,     0,     39883, 5568,  38988,   /*[GB+9632AA30, GB+9632AA39]*/
  0,     41450, 0,     24729, 0,     0,     0,     0,     0,     0,       /*[GB+9632AB30, GB+9632AB39]*/
  0,     0,     12628, 0,     0,     0,     0,     0,     131,   22021,   /*[GB+9632AC30, GB+9632AC39]*/
  0,     0,     15024, 17482, 0,     0,     0,     0,     0,     0,       /*[GB+9632AD30, GB+9632AD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     19080, 24310,   /*[GB+9632AE30, GB+9632AE39]*/
  0,     2133,  26194, 0,     0,     1454,  0,     0,     0,     0,       /*[GB+9632AF30, GB+9632AF39]*/
  0,     39046, 0,     0,     0,     0,     0,     18508, 0,     0,       /*[GB+9632B030, GB+9632B039]*/
  0,     0,     0,     0,     0,     23099, 0,     5813,  0,     0,       /*[GB+9632B130, GB+9632B139]*/
  0,     12076, 0,     0,     0,     40299, 0,     0,     4525,  0,       /*[GB+9632B230, GB+9632B239]*/
  18583, 0,     41380, 19412, 4562,  0,     0,     0,     0,     0,       /*[GB+9632B330, GB+9632B339]*/
  0,     2559,  36293, 0,     0,     0,     15527, 12677, 0,     0,       /*[GB+9632B430, GB+9632B439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9632B530, GB+9632B539]*/
  0,     0,     0,     0,     31543, 0,     0,     0,     0,     0,       /*[GB+9632B630, GB+9632B639]*/
  0,     0,     23151, 15634, 0,     0,     0,     0,     0,     0,       /*[GB+9632B730, GB+9632B739]*/
  0,     2755,  0,     0,     0,     0,     0,     0,     40716, 0,       /*[GB+9632B830, GB+9632B839]*/
  0,     0,     0,     39962, 0,     0,     0,     16504, 0,     0,       /*[GB+9632B930, GB+9632B939]*/
  0,     0,     0,     0,     16921, 7961,  0,     0,     0,     0,       /*[GB+9632BA30, GB+9632BA39]*/
  0,     0,     0,     40436, 25443, 0,     0,     0,     26651, 0,       /*[GB+9632BB30, GB+9632BB39]*/
  0,     0,     24571, 0,     23812, 0,     0,     0,     0,     0,       /*[GB+9632BC30, GB+9632BC39]*/
  0,     0,     0,     0,     9458,  38243, 20573, 0,     0,     0,       /*[GB+9632BD30, GB+9632BD39]*/
  0,     0,     0,     0,     0,     0,     0,     16838, 0,     36077,   /*[GB+9632BE30, GB+9632BE39]*/
  22925, 0,     0,     0,     0,     0,     11720, 15482, 0,     0,       /*[GB+9632BF30, GB+9632BF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9632C030, GB+9632C039]*/
  0,     37629, 0,     0,     0,     0,     33756, 0,     0,     40295,   /*[GB+9632C130, GB+9632C139]*/
  30808, 24397, 7077,  919,   0,     0,     18947, 0,     0,     0,       /*[GB+9632C230, GB+9632C239]*/
  0,     0,     0,     0,     39076, 0,     0,     0,     0,     0,       /*[GB+9632C330, GB+9632C339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     36409, 0,       /*[GB+9632C430, GB+9632C439]*/
  0,     14327, 0,     0,     17188, 0,     41805, 29207, 0,     0,       /*[GB+9632C530, GB+9632C539]*/
  28762, 0,     0,     0,     39199, 23568, 34890, 13121, 31856, 0,       /*[GB+9632C630, GB+9632C639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9632C730, GB+9632C739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     24741, 11039,   /*[GB+9632C830, GB+9632C839]*/
  0,     0,     16596, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9632C930, GB+9632C939]*/
  37536, 3808,  0,     0,     0,     0,     0,     0,     16328, 40220,   /*[GB+9632CA30, GB+9632CA39]*/
  0,     0,     32254, 22880, 0,     4660,  0,     37793, 0,     15088,   /*[GB+9632CB30, GB+9632CB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9632CC30, GB+9632CC39]*/
  0,     0,     0,     0,     15340, 0,     0,     0,     0,     0,       /*[GB+9632CD30, GB+9632CD39]*/
  0,     37045, 18029, 0,     5326,  3958,  0,     0,     0,     0,       /*[GB+9632CE30, GB+9632CE39]*/
  0,     37812, 0,     0,     0,     0,     10751, 0,     0,     0,       /*[GB+9632CF30, GB+9632CF39]*/
  0,     37687, 0,     3992,  11833, 43074, 0,     0,     0,     0,       /*[GB+9632D030, GB+9632D039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9632D130, GB+9632D139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9632D230, GB+9632D239]*/
  0,     31858, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9632D330, GB+9632D339]*/
  26131, 7224,  0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9632D430, GB+9632D439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     31574, 34220,   /*[GB+9632D530, GB+9632D539]*/
  0,     0,     0,     0,     0,     8521,  0,     0,     6185,  33794,   /*[GB+9632D630, GB+9632D639]*/
  34995, 41823, 25761, 31724, 0,     0,     0,     0,     12828, 0,       /*[GB+9632D730, GB+9632D739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     11079,   /*[GB+9632D830, GB+9632D839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9632D930, GB+9632D939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9632DA30, GB+9632DA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9632DB30, GB+9632DB39]*/
  0,     10761, 16928, 30207, 0,     29038, 31345, 0,     0,     0,       /*[GB+9632DC30, GB+9632DC39]*/
  41696, 0,     0,     22673, 0,     25451, 38573, 0,     18837, 27934,   /*[GB+9632DD30, GB+9632DD39]*/
  43609, 24078, 9122,  0,     0,     14916, 39229, 0,     28117, 0,       /*[GB+9632DE30, GB+9632DE39]*/
  21554, 23504, 0,     11601, 0,     0,     0,     39373, 0,     21363,   /*[GB+9632DF30, GB+9632DF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9632E030, GB+9632E039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9632E130, GB+9632E139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9632E230, GB+9632E239]*/
  0,     0,     0,     17787, 0,     0,     0,     0,     0,     0,       /*[GB+9632E330, GB+9632E339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9632E430, GB+9632E439]*/
  42215, 0,     0,     1200,  0,     0,     0,     28436, 25121, 34230,   /*[GB+9632E530, GB+9632E539]*/
  23601, 16302, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9632E630, GB+9632E639]*/
  0,     0,     0,     0,     0,     0,     0,     29324, 24079, 0,       /*[GB+9632E730, GB+9632E739]*/
  40555, 42942, 0,     29762, 15184, 0,     0,     2851,  7735,  34905,   /*[GB+9632E830, GB+9632E839]*/
  38644, 11497, 21055, 31781, 8962,  13422, 0,     0,     0,     37855,   /*[GB+9632E930, GB+9632E939]*/
  0,     39798, 15156, 23762, 0,     34809, 0,     9645,  0,     0,       /*[GB+9632EA30, GB+9632EA39]*/
  9613,  30186, 34429, 43116, 0,     0,     0,     0,     0,     0,       /*[GB+9632EB30, GB+9632EB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9632EC30, GB+9632EC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9632ED30, GB+9632ED39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     9389,  0,       /*[GB+9632EE30, GB+9632EE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9632EF30, GB+9632EF39]*/
  4055,  0,     0,     0,     0,     0,     25653, 31044, 40372, 0,       /*[GB+9632F030, GB+9632F039]*/
  0,     28022, 19045, 0,     42221, 0,     0,     0,     23546, 0,       /*[GB+9632F130, GB+9632F139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9632F230, GB+9632F239]*/
  0,     0,     0,     0,     0,     0,     0,     15460, 17365, 0,       /*[GB+9632F330, GB+9632F339]*/
  0,     0,     43581, 0,     26490, 9800,  15899, 42125, 290,   26010,   /*[GB+9632F430, GB+9632F439]*/
  24092, 43918, 0,     25404, 0,     5166,  0,     13192, 31053, 27302,   /*[GB+9632F530, GB+9632F539]*/
  0,     10568, 36327, 0,     980,   42126, 0,     7105,  0,     0,       /*[GB+9632F630, GB+9632F639]*/
  0,     0,     0,     0,     0,     0,     20767, 5631,  0,     0,       /*[GB+9632F730, GB+9632F739]*/
  0,     0,     27093, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9632F830, GB+9632F839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9632F930, GB+9632F939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9632FA30, GB+9632FA39]*/
  0,     0,     0,     0,     0,     0,     8967,  0,     0,     9110,    /*[GB+9632FB30, GB+9632FB39]*/
  30389, 0,     32199, 37199, 16695, 42416, 0,     0,     0,     0,       /*[GB+9632FC30, GB+9632FC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9632FD30, GB+9632FD39]*/
  0,     0,     0,     0,     0,     0,     18404, 0,     0,     0,       /*[GB+9632FE30, GB+9632FE39]*/
  0,     0,     13786, 22832, 28455, 750,   29312, 6658,  0,     25320,   /*[GB+96338130, GB+96338139]*/
  10690, 25196, 8998,  0,     0,     4550,  2714,  1388,  37712, 8418,    /*[GB+96338230, GB+96338239]*/
  7339,  0,     17724, 10921, 18100, 0,     0,     0,     0,     30940,   /*[GB+96338330, GB+96338339]*/
  28456, 0,     38595, 0,     0,     13787, 0,     0,     0,     2715,    /*[GB+96338430, GB+96338439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96338530, GB+96338539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96338630, GB+96338639]*/
  0,     0,     0,     0,     0,     0,     0,     23053, 4904,  0,       /*[GB+96338730, GB+96338739]*/
  0,     37879, 0,     0,     0,     30746, 0,     0,     15770, 0,       /*[GB+96338830, GB+96338839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96338930, GB+96338939]*/
  0,     0,     3864,  9646,  0,     0,     14652, 41586, 11706, 20906,   /*[GB+96338A30, GB+96338A39]*/
  0,     0,     0,     43766, 29651, 0,     0,     8038,  43506, 43507,   /*[GB+96338B30, GB+96338B39]*/
  0,     14665, 29548, 0,     4515,  23766, 0,     0,     0,     37500,   /*[GB+96338C30, GB+96338C39]*/
  5911,  0,     17850, 35749, 27312, 0,     23536, 15774, 33994, 37607,   /*[GB+96338D30, GB+96338D39]*/
  180,   0,     38338, 13805, 42900, 0,     29690, 33162, 0,     27796,   /*[GB+96338E30, GB+96338E39]*/
  42235, 1158,  43146, 0,     0,     0,     14226, 10991, 0,     3876,    /*[GB+96338F30, GB+96338F39]*/
  7110,  0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96339030, GB+96339039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96339130, GB+96339139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96339230, GB+96339239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96339330, GB+96339339]*/
  0,     2094,  13223, 4322,  0,     20587, 0,     0,     27434, 0,       /*[GB+96339430, GB+96339439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96339530, GB+96339539]*/
  0,     0,     0,     0,     0,     11722, 0,     0,     0,     0,       /*[GB+96339630, GB+96339639]*/
  0,     5569,  0,     0,     10720, 26802, 27905, 41260, 40843, 4372,    /*[GB+96339730, GB+96339739]*/
  24800, 6413,  0,     41090, 31706, 0,     0,     4645,  27561, 0,       /*[GB+96339830, GB+96339839]*/
  20476, 0,     0,     13236, 0,     0,     0,     0,     0,     0,       /*[GB+96339930, GB+96339939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96339A30, GB+96339A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96339B30, GB+96339B39]*/
  20533, 25954, 0,     0,     0,     26183, 15777, 0,     0,     0,       /*[GB+96339C30, GB+96339C39]*/
  126,   13935, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96339D30, GB+96339D39]*/
  0,     0,     0,     31169, 33024, 0,     0,     25682, 0,     0,       /*[GB+96339E30, GB+96339E39]*/
  0,     23559, 12725, 0,     0,     132,   0,     29475, 18679, 7032,    /*[GB+96339F30, GB+96339F39]*/
  0,     30428, 14875, 1441,  36860, 9199,  24295, 39900, 28379, 0,       /*[GB+9633A030, GB+9633A039]*/
  0,     13822, 0,     11745, 0,     0,     19075, 0,     1605,  29972,   /*[GB+9633A130, GB+9633A139]*/
  14308, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9633A230, GB+9633A239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9633A330, GB+9633A339]*/
  0,     0,     0,     0,     0,     0,     43261, 0,     0,     0,       /*[GB+9633A430, GB+9633A439]*/
  0,     10637, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9633A530, GB+9633A539]*/
  0,     7113,  0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9633A630, GB+9633A639]*/
  35137, 21320, 8437,  20160, 30685, 0,     0,     19540, 34269, 27694,   /*[GB+9633A730, GB+9633A739]*/
  0,     0,     4911,  43182, 0,     3343,  0,     0,     0,     0,       /*[GB+9633A830, GB+9633A839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9633A930, GB+9633A939]*/
  0,     0,     18781, 0,     0,     0,     18151, 0,     0,     0,       /*[GB+9633AA30, GB+9633AA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     23964, 32186,   /*[GB+9633AB30, GB+9633AB39]*/
  0,     0,     32204, 0,     18936, 34984, 4496,  12048, 22967, 25350,   /*[GB+9633AC30, GB+9633AC39]*/
  18510, 0,     0,     23718, 31370, 19541, 33331, 0,     0,     0,       /*[GB+9633AD30, GB+9633AD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9633AE30, GB+9633AE39]*/
  0,     0,     0,     0,     38961, 0,     41024, 0,     0,     0,       /*[GB+9633AF30, GB+9633AF39]*/
  0,     0,     0,     0,     0,     0,     39593, 0,     34462, 36063,   /*[GB+9633B030, GB+9633B039]*/
  29326, 33764, 0,     0,     21146, 27845, 18786, 0,     15247, 17802,   /*[GB+9633B130, GB+9633B139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9633B230, GB+9633B239]*/
  0,     0,     0,     0,     0,     0,     34874, 43776, 0,     0,       /*[GB+9633B330, GB+9633B339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     19555, 0,       /*[GB+9633B430, GB+9633B439]*/
  41514, 0,     41623, 0,     17944, 0,     0,     0,     0,     0,       /*[GB+9633B530, GB+9633B539]*/
  0,     0,     0,     0,     0,     0,     0,     6929,  0,     0,       /*[GB+9633B630, GB+9633B639]*/
  0,     0,     0,     0,     0,     0,     0,     13982, 29327, 44036,   /*[GB+9633B730, GB+9633B739]*/
  25870, 0,     17780, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9633B830, GB+9633B839]*/
  0,     0,     0,     0,     0,     39947, 38450, 22790, 0,     0,       /*[GB+9633B930, GB+9633B939]*/
  2620,  0,     0,     0,     0,     0,     0,     37024, 0,     27415,   /*[GB+9633BA30, GB+9633BA39]*/
  0,     0,     0,     0,     0,     0,     36743, 6438,  21646, 0,       /*[GB+9633BB30, GB+9633BB39]*/
  0,     0,     0,     0,     0,     17890, 0,     32331, 0,     0,       /*[GB+9633BC30, GB+9633BC39]*/
  0,     0,     0,     0,     22326, 0,     0,     0,     0,     0,       /*[GB+9633BD30, GB+9633BD39]*/
  36747, 0,     0,     0,     17955, 0,     0,     27914, 11640, 0,       /*[GB+9633BE30, GB+9633BE39]*/
  0,     33527, 33528, 0,     39285, 10611, 0,     10535, 0,     32211,   /*[GB+9633BF30, GB+9633BF39]*/
  35834, 23980, 30533, 33532, 33533, 1787,  25873, 33534, 39433, 33916,   /*[GB+9633C030, GB+9633C039]*/
  0,     4730,  25480, 34048, 0,     0,     32244, 7371,  39292, 35508,   /*[GB+9633C130, GB+9633C139]*/
  22701, 10654, 1262,  28573, 0,     0,     41159, 6179,  39628, 0,       /*[GB+9633C230, GB+9633C239]*/
  23487, 0,     10232, 32511, 25276, 37813, 0,     0,     43806, 0,       /*[GB+9633C330, GB+9633C339]*/
  25277, 6500,  16540, 0,     4255,  37688, 0,     0,     10618, 16844,   /*[GB+9633C430, GB+9633C439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9633C530, GB+9633C539]*/
  0,     0,     6189,  33951, 43465, 19303, 43610, 0,     39777, 12692,   /*[GB+9633C630, GB+9633C639]*/
  0,     29585, 16219, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9633C730, GB+9633C739]*/
  241,   0,     39307, 13182, 12744, 25195, 11498, 11499, 0,     6191,    /*[GB+9633C830, GB+9633C839]*/
  0,     0,     0,     0,     0,     35002, 0,     0,     0,     28357,   /*[GB+9633C930, GB+9633C939]*/
  27939, 0,     0,     0,     35069, 37303, 5315,  0,     35211, 3653,    /*[GB+9633CA30, GB+9633CA39]*/
  39318, 11021, 27942, 0,     0,     0,     0,     30485, 0,     42893,   /*[GB+9633CB30, GB+9633CB39]*/
  0,     56,    0,     0,     0,     26787, 40822, 0,     17252, 19874,   /*[GB+9633CC30, GB+9633CC39]*/
  4334,  0,     39322, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9633CD30, GB+9633CD39]*/
  31156, 0,     28721, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9633CE30, GB+9633CE39]*/
  0,     34971, 34003, 0,     0,     0,     0,     0,     12872, 3890,    /*[GB+9633CF30, GB+9633CF39]*/
  38382, 0,     0,     0,     0,     29156, 40280, 2895,  7491,  33757,   /*[GB+9633D030, GB+9633D039]*/
  33765, 38964, 43849, 40504, 0,     0,     40507, 0,     0,     0,       /*[GB+9633D130, GB+9633D139]*/
  0,     0,     0,     0,     30352, 32433, 0,     0,     0,     35378,   /*[GB+9633D230, GB+9633D239]*/
  0,     0,     32546, 8643,  0,     0,     0,     0,     19968, 0,       /*[GB+9633D330, GB+9633D339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9633D430, GB+9633D439]*/
  0,     0,     13868, 0,     0,     36636, 0,     1263,  16508, 10031,   /*[GB+9633D530, GB+9633D539]*/
  0,     0,     43443, 0,     0,     0,     0,     0,     17046, 42728,   /*[GB+9633D630, GB+9633D639]*/
  42300, 0,     0,     31765, 0,     0,     0,     30369, 0,     4268,    /*[GB+9633D730, GB+9633D739]*/
  3465,  3606,  43017, 0,     5314,  0,     0,     0,     0,     0,       /*[GB+9633D830, GB+9633D839]*/
  0,     0,     0,     1337,  0,     13684, 0,     0,     38244, 0,       /*[GB+9633D930, GB+9633D939]*/
  0,     40248, 9089,  0,     0,     0,     0,     0,     8970,  0,       /*[GB+9633DA30, GB+9633DA39]*/
  0,     0,     0,     0,     25677, 0,     0,     0,     4189,  4193,    /*[GB+9633DB30, GB+9633DB39]*/
  1425,  0,     0,     0,     10325, 0,     0,     9824,  1909,  38733,   /*[GB+9633DC30, GB+9633DC39]*/
  43263, 9830,  0,     18547, 0,     0,     0,     0,     0,     7346,    /*[GB+9633DD30, GB+9633DD39]*/
  5331,  0,     0,     0,     0,     0,     2388,  0,     0,     0,       /*[GB+9633DE30, GB+9633DE39]*/
  0,     0,     31123, 6862,  38057, 0,     0,     0,     6335,  0,       /*[GB+9633DF30, GB+9633DF39]*/
  23209, 0,     36207, 22642, 10272, 16326, 0,     0,     4376,  0,       /*[GB+9633E030, GB+9633E039]*/
  0,     0,     40987, 39200, 0,     575,   28763, 6176,  23515, 26274,   /*[GB+9633E130, GB+9633E139]*/
  0,     0,     0,     36210, 0,     22046, 16359, 39293, 14339, 2013,    /*[GB+9633E230, GB+9633E239]*/
  0,     0,     0,     0,     35643, 0,     0,     0,     0,     0,       /*[GB+9633E330, GB+9633E339]*/
  21505, 20558, 0,     36214, 0,     7226,  0,     7614,  28007, 0,       /*[GB+9633E430, GB+9633E439]*/
  17064, 22627, 21661, 18328, 20360, 0,     6354,  0,     25218, 17962,   /*[GB+9633E530, GB+9633E539]*/
  0,     21037, 0,     11253, 3279,  42325, 0,     9774,  5143,  0,       /*[GB+9633E630, GB+9633E639]*/
  0,     0,     0,     33447, 30926, 0,     0,     33602, 37235, 32761,   /*[GB+9633E730, GB+9633E739]*/
  0,     43813, 20671, 39651, 23955, 38207, 26004, 40240, 16008, 14214,   /*[GB+9633E830, GB+9633E839]*/
  35879, 1071,  0,     0,     0,     0,     20524, 43210, 0,     0,       /*[GB+9633E930, GB+9633E939]*/
  0,     0,     0,     0,     6887,  34728, 0,     0,     12570, 0,       /*[GB+9633EA30, GB+9633EA39]*/
  0,     0,     9791,  16334, 31149, 0,     35426, 12712, 1493,  0,       /*[GB+9633EB30, GB+9633EB39]*/
  0,     0,     16479, 16335, 0,     0,     5167,  0,     0,     0,       /*[GB+9633EC30, GB+9633EC39]*/
  0,     0,     0,     20097, 0,     43959, 40709, 14390, 10052, 0,       /*[GB+9633ED30, GB+9633ED39]*/
  0,     0,     0,     36600, 7564,  0,     26064, 0,     0,     5864,    /*[GB+9633EE30, GB+9633EE39]*/
  29652, 30118, 18981, 0,     0,     0,     0,     28138, 0,     1427,    /*[GB+9633EF30, GB+9633EF39]*/
  32914, 37624, 0,     41150, 0,     37503, 6323,  133,   0,     0,       /*[GB+9633F030, GB+9633F039]*/
  14232, 0,     0,     29846, 0,     12049, 39927, 0,     0,     26939,   /*[GB+9633F130, GB+9633F139]*/
  0,     0,     6139,  43883, 606,   0,     25908, 0,     0,     13566,   /*[GB+9633F230, GB+9633F239]*/
  0,     24060, 0,     0,     31668, 0,     0,     0,     4041,  0,       /*[GB+9633F330, GB+9633F339]*/
  0,     0,     0,     10237, 0,     0,     0,     0,     0,     0,       /*[GB+9633F430, GB+9633F439]*/
  0,     0,     0,     0,     12986, 0,     0,     0,     0,     0,       /*[GB+9633F530, GB+9633F539]*/
  0,     0,     36225, 0,     6193,  0,     0,     1587,  40569, 31603,   /*[GB+9633F630, GB+9633F639]*/
  0,     0,     0,     0,     0,     0,     28085, 0,     9859,  26793,   /*[GB+9633F730, GB+9633F739]*/
  0,     0,     0,     0,     0,     0,     9814,  0,     0,     0,       /*[GB+9633F830, GB+9633F839]*/
  0,     0,     0,     0,     11723, 16790, 0,     0,     9685,  0,       /*[GB+9633F930, GB+9633F939]*/
  29958, 0,     4086,  0,     16565, 0,     6754,  0,     0,     0,       /*[GB+9633FA30, GB+9633FA39]*/
  0,     38403, 0,     0,     5780,  25582, 0,     0,     0,     0,       /*[GB+9633FB30, GB+9633FB39]*/
  0,     19709, 0,     18182, 0,     18575, 26045, 0,     32619, 0,       /*[GB+9633FC30, GB+9633FC39]*/
  22157, 0,     0,     0,     0,     25080, 0,     14177, 35035, 0,       /*[GB+9633FD30, GB+9633FD39]*/
  0,     0,     0,     5386,  0,     19806, 0,     0,     0,     0,       /*[GB+9633FE30, GB+9633FE39]*/
  0,     0,     10030, 0,     0,     0,     23788, 0,     0,     24061,   /*[GB+96348130, GB+96348139]*/
  0,     15744, 0,     897,   15900, 15768, 28957, 0,     0,     0,       /*[GB+96348230, GB+96348239]*/
  0,     35135, 0,     0,     27429, 15141, 24599, 27605, 0,     0,       /*[GB+96348330, GB+96348339]*/
  0,     0,     12925, 0,     0,     0,     0,     0,     0,     0,       /*[GB+96348430, GB+96348439]*/
  0,     8074,  932,   14741, 27508, 0,     23981, 0,     0,     0,       /*[GB+96348530, GB+96348539]*/
  0,     0,     6027,  20951, 542,   543,   31341, 19263, 28575, 41758,   /*[GB+96348630, GB+96348639]*/
  23986, 33434, 0,     40871, 845,   0,     0,     0,     0,     0,       /*[GB+96348730, GB+96348739]*/
  0,     20213, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96348830, GB+96348839]*/
  0,     0,     0,     0,     0,     0,     10887, 6674,  8482,  22702,   /*[GB+96348930, GB+96348939]*/
  0,     0,     0,     9271,  0,     0,     27286, 34415, 0,     0,       /*[GB+96348A30, GB+96348A39]*/
  1278,  0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96348B30, GB+96348B39]*/
  20957, 0,     0,     0,     27210, 26302, 0,     0,     0,     2033,    /*[GB+96348C30, GB+96348C39]*/
  0,     10890, 22594, 36659, 0,     0,     0,     0,     0,     0,       /*[GB+96348D30, GB+96348D39]*/
  0,     23734, 30025, 0,     32193, 0,     26403, 18054, 5802,  0,       /*[GB+96348E30, GB+96348E39]*/
  14611, 14612, 10354, 13756, 1313,  0,     0,     0,     0,     0,       /*[GB+96348F30, GB+96348F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96349030, GB+96349039]*/
  27661, 705,   4540,  0,     0,     0,     22434, 721,   0,     0,       /*[GB+96349130, GB+96349139]*/
  27142, 13594, 29763, 0,     0,     0,     5803,  0,     0,     0,       /*[GB+96349230, GB+96349239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96349330, GB+96349339]*/
  34233, 41176, 21057, 0,     0,     43476, 0,     13440, 24925, 0,       /*[GB+96349430, GB+96349439]*/
  0,     0,     0,     27362, 30216, 27216, 33470, 27150, 27151, 0,       /*[GB+96349530, GB+96349539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     30391, 0,       /*[GB+96349630, GB+96349639]*/
  29770, 0,     0,     0,     0,     0,     0,     21222, 18101, 0,       /*[GB+96349730, GB+96349739]*/
  0,     5046,  43666, 21223, 33877, 0,     20929, 23054, 0,     30034,   /*[GB+96349830, GB+96349839]*/
  0,     23806, 0,     0,     1389,  6086,  27617, 0,     26503, 25722,   /*[GB+96349930, GB+96349939]*/
  0,     8101,  29943, 0,     0,     0,     0,     0,     0,     35750,   /*[GB+96349A30, GB+96349A39]*/
  0,     0,     0,     27223, 13030, 0,     0,     26514, 19064, 0,       /*[GB+96349B30, GB+96349B39]*/
  0,     0,     0,     0,     0,     0,     0,     27748, 0,     0,       /*[GB+96349C30, GB+96349C39]*/
  17194, 0,     5109,  37630, 1935,  0,     14688, 0,     26519, 26520,   /*[GB+96349D30, GB+96349D39]*/
  0,     0,     0,     21309, 26878, 0,     1821,  3199,  0,     0,       /*[GB+96349E30, GB+96349E39]*/
  6812,  0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96349F30, GB+96349F39]*/
  26924, 34279, 0,     0,     30917, 26540, 23100, 0,     0,     29135,   /*[GB+9634A030, GB+9634A039]*/
  20012, 0,     17196, 0,     0,     0,     0,     0,     21816, 21869,   /*[GB+9634A130, GB+9634A139]*/
  0,     0,     0,     5531,  0,     0,     38711, 0,     0,     34974,   /*[GB+9634A230, GB+9634A239]*/
  0,     0,     0,     13341, 0,     0,     0,     0,     38479, 6543,    /*[GB+9634A330, GB+9634A339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     8095,  42635,   /*[GB+9634A430, GB+9634A439]*/
  0,     0,     0,     9808,  0,     0,     0,     0,     0,     2684,    /*[GB+9634A530, GB+9634A539]*/
  0,     19461, 0,     0,     2203,  0,     0,     0,     23512, 29589,   /*[GB+9634A630, GB+9634A639]*/
  5702,  41747, 0,     0,     27989, 0,     4329,  0,     15261, 0,       /*[GB+9634A730, GB+9634A739]*/
  0,     0,     0,     0,     16440, 3680,  0,     0,     0,     0,       /*[GB+9634A830, GB+9634A839]*/
  0,     11812, 18281, 27854, 0,     0,     23567, 32627, 33348, 0,       /*[GB+9634A930, GB+9634A939]*/
  0,     0,     38842, 0,     0,     0,     0,     27855, 5883,  0,       /*[GB+9634AA30, GB+9634AA39]*/
  0,     5421,  0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9634AB30, GB+9634AB39]*/
  0,     0,     0,     31710, 11041, 24539, 0,     42185, 23210, 22798,   /*[GB+9634AC30, GB+9634AC39]*/
  0,     0,     0,     0,     0,     27200, 9955,  0,     42091, 0,       /*[GB+9634AD30, GB+9634AD39]*/
  0,     13122, 0,     0,     24430, 0,     0,     0,     0,     0,       /*[GB+9634AE30, GB+9634AE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9634AF30, GB+9634AF39]*/
  0,     0,     0,     0,     0,     0,     7811,  0,     0,     14567,   /*[GB+9634B030, GB+9634B039]*/
  10561, 0,     0,     0,     41287, 0,     33932, 0,     0,     43440,   /*[GB+9634B130, GB+9634B139]*/
  0,     0,     33788, 24314, 0,     0,     1014,  0,     13873, 0,       /*[GB+9634B230, GB+9634B239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9634B330, GB+9634B339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9634B430, GB+9634B439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9634B530, GB+9634B539]*/
  0,     14003, 0,     0,     40512, 0,     0,     0,     0,     0,       /*[GB+9634B630, GB+9634B639]*/
  0,     0,     1036,  21028, 0,     0,     40753, 43191, 0,     16894,   /*[GB+9634B730, GB+9634B739]*/
  0,     0,     1166,  0,     23573, 0,     0,     0,     0,     0,       /*[GB+9634B830, GB+9634B839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9634B930, GB+9634B939]*/
  0,     0,     21029, 30667, 0,     0,     0,     0,     0,     0,       /*[GB+9634BA30, GB+9634BA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9634BB30, GB+9634BB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9634BC30, GB+9634BC39]*/
  0,     0,     0,     0,     0,     0,     32030, 20520, 0,     0,       /*[GB+9634BD30, GB+9634BD39]*/
  30539, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9634BE30, GB+9634BE39]*/
  0,     0,     0,     0,     0,     195,   0,     0,     0,     0,       /*[GB+9634BF30, GB+9634BF39]*/
  0,     10670, 42884, 0,     7229,  7230,  9027,  0,     0,     0,       /*[GB+9634C030, GB+9634C039]*/
  0,     0,     0,     0,     0,     0,     26199, 0,     32518, 0,       /*[GB+9634C130, GB+9634C139]*/
  22063, 41648, 33040, 43192, 0,     14497, 23760, 0,     0,     0,       /*[GB+9634C230, GB+9634C239]*/
  0,     0,     0,     0,     29883, 0,     0,     0,     0,     15604,   /*[GB+9634C330, GB+9634C339]*/
  0,     0,     24757, 2445,  0,     31861, 27876, 0,     32118, 0,       /*[GB+9634C430, GB+9634C439]*/
  0,     11984, 0,     38172, 0,     0,     0,     0,     0,     0,       /*[GB+9634C530, GB+9634C539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9634C630, GB+9634C639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9634C730, GB+9634C739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9634C830, GB+9634C839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9634C930, GB+9634C939]*/
  0,     0,     0,     0,     0,     0,     21825, 42566, 0,     0,       /*[GB+9634CA30, GB+9634CA39]*/
  0,     29456, 0,     0,     0,     38173, 32519, 309,   0,     0,       /*[GB+9634CB30, GB+9634CB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     29858, 0,       /*[GB+9634CC30, GB+9634CC39]*/
  0,     0,     0,     0,     0,     0,     10247, 32175, 12483, 0,       /*[GB+9634CD30, GB+9634CD39]*/
  0,     35881, 40609, 0,     43578, 1580,  0,     5269,  38492, 26720,   /*[GB+9634CE30, GB+9634CE39]*/
  0,     0,     12043, 22311, 40519, 0,     31030, 0,     0,     0,       /*[GB+9634CF30, GB+9634CF39]*/
  0,     0,     37837, 30547, 28323, 29458, 0,     35482, 0,     0,       /*[GB+9634D030, GB+9634D039]*/
  39309, 0,     0,     8127,  0,     0,     0,     0,     0,     0,       /*[GB+9634D130, GB+9634D139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9634D230, GB+9634D239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9634D330, GB+9634D339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     16469,   /*[GB+9634D430, GB+9634D439]*/
  0,     0,     0,     8544,  0,     0,     0,     0,     0,     0,       /*[GB+9634D530, GB+9634D539]*/
  21881, 0,     0,     0,     0,     0,     0,     0,     24169, 0,       /*[GB+9634D630, GB+9634D639]*/
  13172, 25768, 42605, 9033,  40004, 3713,  0,     0,     0,     25497,   /*[GB+9634D730, GB+9634D739]*/
  0,     44026, 0,     35061, 20969, 0,     11687, 0,     24177, 36386,   /*[GB+9634D830, GB+9634D839]*/
  0,     34117, 31045, 0,     0,     22374, 4320,  0,     0,     0,       /*[GB+9634D930, GB+9634D939]*/
  0,     0,     0,     0,     0,     0,     33372, 0,     0,     0,       /*[GB+9634DA30, GB+9634DA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9634DB30, GB+9634DB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9634DC30, GB+9634DC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9634DD30, GB+9634DD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9634DE30, GB+9634DE39]*/
  12248, 36226, 30927, 0,     25769, 0,     0,     14273, 0,     0,       /*[GB+9634DF30, GB+9634DF39]*/
  0,     0,     0,     0,     0,     0,     5630,  584,   0,     0,       /*[GB+9634E030, GB+9634E039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9634E130, GB+9634E139]*/
  0,     0,     0,     5671,  16648, 0,     15395, 27894, 15012, 0,       /*[GB+9634E230, GB+9634E239]*/
  0,     21982, 23285, 0,     29464, 0,     29255, 43179, 8581,  0,       /*[GB+9634E330, GB+9634E339]*/
  0,     40674, 35070, 17682, 31550, 0,     0,     0,     0,     0,       /*[GB+9634E430, GB+9634E439]*/
  0,     41578, 0,     0,     2450,  0,     40252, 0,     0,     42774,   /*[GB+9634E530, GB+9634E539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9634E630, GB+9634E639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9634E730, GB+9634E739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9634E830, GB+9634E839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9634E930, GB+9634E939]*/
  30552, 0,     0,     0,     36687, 20082, 36829, 0,     0,     0,       /*[GB+9634EA30, GB+9634EA39]*/
  0,     0,     40815, 29311, 0,     0,     0,     0,     0,     0,       /*[GB+9634EB30, GB+9634EB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9634EC30, GB+9634EC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9634ED30, GB+9634ED39]*/
  0,     0,     0,     10420, 42895, 0,     27306, 27307, 0,     0,       /*[GB+9634EE30, GB+9634EE39]*/
  16228, 13460, 9132,  0,     0,     0,     35229, 0,     0,     23507,   /*[GB+9634EF30, GB+9634EF39]*/
  13788, 0,     0,     0,     42133, 41008, 0,     11441, 0,     16850,   /*[GB+9634F030, GB+9634F039]*/
  0,     2084,  0,     0,     27787, 248,   14062, 0,     30963, 0,       /*[GB+9634F130, GB+9634F139]*/
  0,     29688, 0,     21392, 38597, 30554, 10421, 0,     0,     27376,   /*[GB+9634F230, GB+9634F239]*/
  32891, 0,     0,     29646, 0,     0,     0,     0,     0,     0,       /*[GB+9634F330, GB+9634F339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9634F430, GB+9634F439]*/
  0,     0,     0,     0,     0,     18645, 0,     0,     0,     0,       /*[GB+9634F530, GB+9634F539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9634F630, GB+9634F639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     25146, 0,       /*[GB+9634F730, GB+9634F739]*/
  41392, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9634F830, GB+9634F839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9634F930, GB+9634F939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9634FA30, GB+9634FA39]*/
  0,     6592,  8604,  28515, 29945, 38344, 18772, 0,     11134, 0,       /*[GB+9634FB30, GB+9634FB39]*/
  41732, 0,     40833, 0,     5054,  5055,  0,     0,     0,     0,       /*[GB+9634FC30, GB+9634FC39]*/
  0,     0,     0,     0,     0,     8334,  18237, 16654, 0,     0,       /*[GB+9634FD30, GB+9634FD39]*/
  0,     34969, 39327, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9634FE30, GB+9634FE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96358130, GB+96358139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96358230, GB+96358239]*/
  0,     0,     0,     10926, 0,     0,     0,     0,     0,     10807,   /*[GB+96358330, GB+96358339]*/
  16516, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96358430, GB+96358439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     39865,   /*[GB+96358530, GB+96358539]*/
  0,     0,     0,     0,     0,     12159, 30226, 0,     17298, 0,       /*[GB+96358630, GB+96358639]*/
  6723,  14082, 24516, 1542,  0,     0,     0,     0,     36233, 0,       /*[GB+96358730, GB+96358739]*/
  1602,  0,     0,     0,     0,     32915, 0,     0,     0,     0,       /*[GB+96358830, GB+96358839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96358930, GB+96358939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96358A30, GB+96358A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96358B30, GB+96358B39]*/
  0,     0,     0,     0,     0,     0,     0,     5578,  14689, 598,     /*[GB+96358C30, GB+96358C39]*/
  0,     22903, 34583, 27814, 0,     0,     11957, 4316,  0,     0,       /*[GB+96358D30, GB+96358D39]*/
  23509, 19076, 5962,  2355,  0,     41019, 0,     20200, 0,     0,       /*[GB+96358E30, GB+96358E39]*/
  0,     12873, 0,     0,     18472, 38953, 0,     32823, 1442,  10473,   /*[GB+96358F30, GB+96358F39]*/
  0,     24199, 0,     0,     1663,  0,     0,     0,     0,     0,       /*[GB+96359030, GB+96359039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96359130, GB+96359139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96359230, GB+96359239]*/
  36861, 0,     0,     0,     43153, 0,     0,     0,     0,     0,       /*[GB+96359330, GB+96359339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     11431, 0,       /*[GB+96359430, GB+96359439]*/
  0,     0,     19226, 7485,  0,     38669, 17593, 0,     37452, 0,       /*[GB+96359530, GB+96359539]*/
  0,     36138, 18494, 0,     0,     0,     0,     0,     0,     0,       /*[GB+96359630, GB+96359639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96359730, GB+96359739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96359830, GB+96359839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96359930, GB+96359939]*/
  0,     0,     17595, 19377, 33758, 11332, 0,     14421, 43409, 0,       /*[GB+96359A30, GB+96359A39]*/
  0,     0,     42151, 0,     0,     2143,  30952, 218,   0,     0,       /*[GB+96359B30, GB+96359B39]*/
  36350, 17935, 40682, 12504, 0,     0,     0,     0,     0,     0,       /*[GB+96359C30, GB+96359C39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96359D30, GB+96359D39]*/
  34031, 11433, 0,     7852,  13974, 4879,  40536, 0,     2299,  0,       /*[GB+96359E30, GB+96359E39]*/
  39175, 0,     0,     7043,  23596, 0,     0,     0,     0,     0,       /*[GB+96359F30, GB+96359F39]*/
  0,     0,     0,     30445, 0,     23657, 0,     0,     0,     7705,    /*[GB+9635A030, GB+9635A039]*/
  33766, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9635A130, GB+9635A139]*/
  37464, 20493, 0,     0,     1516,  0,     14972, 18548, 0,     0,       /*[GB+9635A230, GB+9635A239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9635A330, GB+9635A339]*/
  37016, 6733,  0,     0,     20859, 0,     9539,  26944, 2462,  0,       /*[GB+9635A430, GB+9635A439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9635A530, GB+9635A539]*/
  0,     0,     0,     0,     0,     21234, 0,     0,     0,     0,       /*[GB+9635A630, GB+9635A639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     9838,    /*[GB+9635A730, GB+9635A739]*/
  29336, 0,     0,     0,     0,     0,     36745, 1484,  0,     0,       /*[GB+9635A830, GB+9635A839]*/
  0,     0,     0,     0,     1690,  3597,  17060, 0,     12793, 0,       /*[GB+9635A930, GB+9635A939]*/
  0,     0,     0,     0,     0,     0,     0,     3803,  0,     33169,   /*[GB+9635AA30, GB+9635AA39]*/
  0,     4896,  42257, 0,     0,     0,     0,     0,     0,     29397,   /*[GB+9635AB30, GB+9635AB39]*/
  0,     0,     9919,  24632, 35836, 29498, 0,     33352, 20404, 0,       /*[GB+9635AC30, GB+9635AC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9635AD30, GB+9635AD39]*/
  8372,  0,     28877, 0,     0,     12517, 12234, 23516, 0,     0,       /*[GB+9635AE30, GB+9635AE39]*/
  0,     0,     0,     0,     42935, 0,     37794, 0,     0,     0,       /*[GB+9635AF30, GB+9635AF39]*/
  0,     18303, 28065, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9635B030, GB+9635B039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9635B130, GB+9635B139]*/
  0,     0,     0,     0,     0,     0,     21817, 0,     9215,  0,       /*[GB+9635B230, GB+9635B239]*/
  11586, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9635B330, GB+9635B339]*/
  0,     0,     0,     19628, 0,     7518,  7650,  0,     0,     35855,   /*[GB+9635B430, GB+9635B439]*/
  37814, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9635B530, GB+9635B539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9635B630, GB+9635B639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9635B730, GB+9635B739]*/
  0,     0,     3698,  0,     0,     0,     0,     37815, 0,     0,       /*[GB+9635B830, GB+9635B839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     5930,    /*[GB+9635B930, GB+9635B939]*/
  12482, 11082, 0,     41034, 0,     0,     0,     0,     0,     0,       /*[GB+9635BA30, GB+9635BA39]*/
  0,     0,     0,     0,     0,     0,     40366, 34226, 0,     0,       /*[GB+9635BB30, GB+9635BB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9635BC30, GB+9635BC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     34321, 17786,   /*[GB+9635BD30, GB+9635BD39]*/
  28289, 28915, 43341, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9635BE30, GB+9635BE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9635BF30, GB+9635BF39]*/
  0,     0,     1721,  15274, 33138, 14360, 8545,  0,     34622, 16179,   /*[GB+9635C030, GB+9635C039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9635C130, GB+9635C139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9635C230, GB+9635C239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9635C330, GB+9635C339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9635C430, GB+9635C439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9635C530, GB+9635C539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9635C630, GB+9635C639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9635C730, GB+9635C739]*/
  22604, 0,     12842, 13183, 17280, 0,     0,     0,     0,     8346,    /*[GB+9635C830, GB+9635C839]*/
  28079, 18976, 21771, 3287,  0,     0,     21975, 0,     0,     0,       /*[GB+9635C930, GB+9635C939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     19798,   /*[GB+9635CA30, GB+9635CA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9635CB30, GB+9635CB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9635CC30, GB+9635CC39]*/
  0,     0,     0,     0,     7237,  0,     0,     22605, 0,     12987,   /*[GB+9635CD30, GB+9635CD39]*/
  0,     0,     43027, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9635CE30, GB+9635CE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9635CF30, GB+9635CF39]*/
  29907, 7238,  0,     0,     0,     15192, 0,     0,     40570, 12488,   /*[GB+9635D030, GB+9635D039]*/
  38941, 11693, 19210, 33106, 10791, 0,     0,     0,     22470, 0,       /*[GB+9635D130, GB+9635D139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9635D230, GB+9635D239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9635D330, GB+9635D339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9635D430, GB+9635D439]*/
  0,     0,     0,     33971, 0,     9035,  0,     0,     0,     0,       /*[GB+9635D530, GB+9635D539]*/
  11189, 13004, 34687, 27895, 0,     0,     0,     0,     0,     0,       /*[GB+9635D630, GB+9635D639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     31972, 0,       /*[GB+9635D730, GB+9635D739]*/
  41444, 29261, 0,     39323, 13790, 0,     0,     0,     0,     0,       /*[GB+9635D830, GB+9635D839]*/
  40766, 0,     0,     0,     26979, 0,     0,     0,     41364, 0,       /*[GB+9635D930, GB+9635D939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9635DA30, GB+9635DA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9635DB30, GB+9635DB39]*/
  0,     0,     0,     27498, 0,     0,     0,     0,     0,     0,       /*[GB+9635DC30, GB+9635DC39]*/
  36601, 0,     0,     0,     0,     0,     0,     5095,  0,     0,       /*[GB+9635DD30, GB+9635DD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     13208,   /*[GB+9635DE30, GB+9635DE39]*/
  28191, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9635DF30, GB+9635DF39]*/
  0,     7467,  17730, 30655, 0,     43044, 0,     0,     0,     0,       /*[GB+9635E030, GB+9635E039]*/
  0,     19191, 0,     40913, 0,     15723, 11630, 0,     0,     0,       /*[GB+9635E130, GB+9635E139]*/
  0,     0,     0,     0,     0,     34843, 0,     0,     0,     0,       /*[GB+9635E230, GB+9635E239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9635E330, GB+9635E339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9635E430, GB+9635E439]*/
  0,     0,     0,     28516, 12859, 0,     0,     0,     17095, 0,       /*[GB+9635E530, GB+9635E539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9635E630, GB+9635E639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     4296,    /*[GB+9635E730, GB+9635E739]*/
  42788, 0,     0,     0,     300,   0,     43393, 0,     0,     0,       /*[GB+9635E830, GB+9635E839]*/
  0,     35448, 0,     21781, 11635, 0,     0,     0,     0,     0,       /*[GB+9635E930, GB+9635E939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9635EA30, GB+9635EA39]*/
  0,     0,     0,     0,     30606, 0,     0,     0,     0,     1429,    /*[GB+9635EB30, GB+9635EB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     6487,    /*[GB+9635EC30, GB+9635EC39]*/
  0,     0,     0,     0,     0,     16187, 0,     1664,  0,     0,       /*[GB+9635ED30, GB+9635ED39]*/
  0,     0,     12874, 19077, 5011,  0,     0,     0,     0,     0,       /*[GB+9635EE30, GB+9635EE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9635EF30, GB+9635EF39]*/
  0,     34584, 26980, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9635F030, GB+9635F039]*/
  0,     0,     0,     0,     0,     0,     0,     40281, 0,     36268,   /*[GB+9635F130, GB+9635F139]*/
  0,     18962, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9635F230, GB+9635F239]*/
  0,     0,     0,     27911, 0,     0,     28993, 0,     0,     0,       /*[GB+9635F330, GB+9635F339]*/
  35279, 0,     0,     0,     0,     0,     0,     0,     0,     34871,   /*[GB+9635F430, GB+9635F439]*/
  0,     0,     0,     37504, 17561, 38422, 0,     0,     0,     0,       /*[GB+9635F530, GB+9635F539]*/
  0,     18734, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9635F630, GB+9635F639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9635F730, GB+9635F739]*/
  0,     0,     2144,  2507,  0,     0,     0,     0,     0,     0,       /*[GB+9635F830, GB+9635F839]*/
  0,     37458, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9635F930, GB+9635F939]*/
  18787, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9635FA30, GB+9635FA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9635FB30, GB+9635FB39]*/
  0,     30454, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9635FC30, GB+9635FC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9635FD30, GB+9635FD39]*/
  0,     0,     0,     13534, 0,     0,     33772, 42535, 0,     0,       /*[GB+9635FE30, GB+9635FE39]*/
  0,     0,     33775, 0,     0,     40306, 0,     0,     0,     0,       /*[GB+96368130, GB+96368139]*/
  0,     0,     34476, 0,     43427, 0,     0,     0,     0,     0,       /*[GB+96368230, GB+96368239]*/
  0,     0,     0,     41382, 0,     0,     0,     0,     0,     43852,   /*[GB+96368330, GB+96368339]*/
  0,     22302, 0,     0,     0,     0,     0,     0,     5744,  19010,   /*[GB+96368430, GB+96368439]*/
  22304, 0,     0,     0,     0,     0,     0,     0,     0,     23729,   /*[GB+96368530, GB+96368539]*/
  0,     15582, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96368630, GB+96368639]*/
  20362, 0,     20525, 0,     0,     29292, 28224, 0,     0,     20526,   /*[GB+96368730, GB+96368739]*/
  0,     8096,  0,     0,     0,     860,   0,     0,     0,     0,       /*[GB+96368830, GB+96368839]*/
  18411, 0,     0,     35577, 0,     0,     0,     0,     0,     0,       /*[GB+96368930, GB+96368939]*/
  0,     0,     0,     0,     5607,  0,     0,     0,     0,     0,       /*[GB+96368A30, GB+96368A39]*/
  0,     0,     0,     0,     3618,  0,     0,     0,     41397, 0,       /*[GB+96368B30, GB+96368B39]*/
  0,     29825, 0,     0,     28483, 0,     0,     0,     0,     0,       /*[GB+96368C30, GB+96368C39]*/
  0,     0,     9353,  0,     0,     0,     0,     0,     39407, 0,       /*[GB+96368D30, GB+96368D39]*/
  12417, 3299,  0,     14436, 42991, 29079, 0,     0,     34900, 0,       /*[GB+96368E30, GB+96368E39]*/
  0,     0,     23152, 0,     0,     0,     0,     0,     0,     14130,   /*[GB+96368F30, GB+96368F39]*/
  0,     14132, 0,     0,     0,     0,     0,     0,     0,     6290,    /*[GB+96369030, GB+96369039]*/
  0,     0,     0,     0,     0,     25592, 26303, 0,     8241,  0,       /*[GB+96369130, GB+96369139]*/
  0,     0,     0,     0,     41303, 16375, 0,     0,     0,     0,       /*[GB+96369230, GB+96369239]*/
  0,     0,     0,     0,     0,     0,     6395,  40824, 0,     0,       /*[GB+96369330, GB+96369339]*/
  10105, 27799, 8199,  0,     0,     14191, 0,     36707, 42483, 0,       /*[GB+96369430, GB+96369439]*/
  0,     0,     18513, 0,     0,     0,     19415, 0,     0,     0,       /*[GB+96369530, GB+96369539]*/
  37758, 0,     26568, 23598, 9599,  0,     13672, 11051, 0,     23957,   /*[GB+96369630, GB+96369639]*/
  0,     15583, 0,     3271,  0,     0,     19025, 28904, 24562, 34715,   /*[GB+96369730, GB+96369739]*/
  0,     24164, 20523, 4938,  0,     39646, 0,     0,     0,     16605,   /*[GB+96369830, GB+96369839]*/
  35189, 0,     0,     0,     0,     32393, 36805, 23797, 0,     0,       /*[GB+96369930, GB+96369939]*/
  0,     3549,  0,     0,     0,     0,     31046, 3288,  1341,  18761,   /*[GB+96369A30, GB+96369A39]*/
  13685, 0,     31482, 0,     0,     0,     31003, 0,     0,     0,       /*[GB+96369B30, GB+96369B39]*/
  0,     0,     6402,  0,     29262, 18102, 20195, 34064, 0,     5033,    /*[GB+96369C30, GB+96369C39]*/
  0,     0,     0,     1985,  31923, 24675, 0,     19215, 8200,  0,       /*[GB+96369D30, GB+96369D39]*/
  2109,  3018,  0,     0,     5608,  30506, 5777,  0,     0,     18514,   /*[GB+96369E30, GB+96369E39]*/
  0,     36452, 36455, 41161, 0,     0,     37826, 0,     24765, 0,       /*[GB+96369F30, GB+96369F39]*/
  0,     0,     0,     0,     0,     0,     11800, 0,     31436, 0,       /*[GB+9636A030, GB+9636A039]*/
  16624, 0,     0,     4528,  39202, 8183,  0,     23377, 31307, 14891,   /*[GB+9636A130, GB+9636A139]*/
  24052, 3354,  24160, 0,     14742, 10216, 0,     0,     0,     0,       /*[GB+9636A230, GB+9636A239]*/
  0,     0,     0,     0,     0,     0,     43188, 0,     0,     11647,   /*[GB+9636A330, GB+9636A339]*/
  4109,  29209, 30827, 24057, 27204, 27205, 0,     11532, 23863, 0,       /*[GB+9636A430, GB+9636A439]*/
  0,     0,     0,     0,     0,     580,   0,     34086, 0,     0,       /*[GB+9636A530, GB+9636A539]*/
  0,     25967, 22758, 0,     30828, 26590, 0,     0,     35036, 0,       /*[GB+9636A630, GB+9636A639]*/
  41477, 11901, 35333, 0,     29010, 0,     0,     31766, 0,     18799,   /*[GB+9636A730, GB+9636A739]*/
  1293,  4139,  0,     35419, 0,     35335, 8395,  0,     0,     0,       /*[GB+9636A830, GB+9636A839]*/
  0,     14756, 0,     0,     0,     8663,  0,     17712, 30937, 28352,   /*[GB+9636A930, GB+9636A939]*/
  0,     0,     3167,  4461,  0,     0,     0,     0,     0,     0,       /*[GB+9636AA30, GB+9636AA39]*/
  26863, 11536, 1072,  0,     2339,  0,     0,     0,     0,     22204,   /*[GB+9636AB30, GB+9636AB39]*/
  0,     0,     0,     35710, 6376,  0,     0,     2955,  39513, 42636,   /*[GB+9636AC30, GB+9636AC39]*/
  0,     0,     0,     18063, 29013, 0,     0,     0,     0,     0,       /*[GB+9636AD30, GB+9636AD39]*/
  6779,  0,     20944, 12280, 0,     30894, 0,     6888,  31047, 28591,   /*[GB+9636AE30, GB+9636AE39]*/
  21119, 0,     0,     0,     41708, 25937, 24182, 30739, 2480,  37304,   /*[GB+9636AF30, GB+9636AF39]*/
  0,     35664, 709,   9801,  0,     0,     0,     2270,  0,     0,       /*[GB+9636B030, GB+9636B039]*/
  0,     0,     0,     0,     9676,  2071,  0,     0,     0,     33070,   /*[GB+9636B130, GB+9636B139]*/
  0,     0,     13444, 0,     0,     0,     0,     0,     17253, 0,       /*[GB+9636B230, GB+9636B239]*/
  0,     5034,  21895, 4930,  23461, 39021, 0,     0,     0,     0,       /*[GB+9636B330, GB+9636B339]*/
  20978, 16481, 20863, 6485,  12061, 7198,  14155, 29933, 0,     0,       /*[GB+9636B430, GB+9636B439]*/
  0,     0,     2632,  30412, 0,     12300, 15934, 36341, 0,     0,       /*[GB+9636B530, GB+9636B539]*/
  35579, 0,     21588, 26844, 17731, 0,     0,     42484, 32814, 0,       /*[GB+9636B630, GB+9636B639]*/
  0,     0,     0,     27809, 0,     0,     0,     40681, 0,     0,       /*[GB+9636B730, GB+9636B739]*/
  0,     0,     0,     217,   32938, 0,     12050, 0,     0,     41334,   /*[GB+9636B830, GB+9636B839]*/
  0,     38821, 9254,  12666, 0,     26699, 0,     0,     0,     0,       /*[GB+9636B930, GB+9636B939]*/
  0,     19098, 15517, 536,   17891, 0,     20802, 19416, 0,     0,       /*[GB+9636BA30, GB+9636BA39]*/
  18265, 13281, 0,     0,     0,     12452, 35375, 28862, 41527, 0,       /*[GB+9636BB30, GB+9636BB39]*/
  0,     1943,  0,     0,     0,     2647,  0,     0,     0,     2649,    /*[GB+9636BC30, GB+9636BC39]*/
  14980, 426,   924,   0,     0,     36899, 0,     11813, 0,     39428,   /*[GB+9636BD30, GB+9636BD39]*/
  0,     1232,  4355,  0,     13288, 0,     0,     0,     22252, 0,       /*[GB+9636BE30, GB+9636BE39]*/
  20949, 42550, 0,     37529, 6641,  37511, 0,     0,     0,     27101,   /*[GB+9636BF30, GB+9636BF39]*/
  38473, 0,     10954, 0,     22338, 31423, 8711,  22217, 40563, 0,       /*[GB+9636C030, GB+9636C039]*/
  11649, 0,     3314,  32096, 0,     0,     4244,  8471,  0,     0,       /*[GB+9636C130, GB+9636C139]*/
  40783, 0,     0,     0,     0,     842,   0,     0,     0,     0,       /*[GB+9636C230, GB+9636C239]*/
  5992,  2328,  20020, 24541, 5251,  37545, 0,     37956, 0,     0,       /*[GB+9636C330, GB+9636C339]*/
  0,     34657, 1138,  42815, 0,     37796, 0,     20257, 0,     16868,   /*[GB+9636C430, GB+9636C439]*/
  0,     0,     42283, 12528, 0,     0,     0,     32245, 5889,  29217,   /*[GB+9636C530, GB+9636C539]*/
  19011, 40727, 42284, 22553, 0,     0,     0,     0,     0,     0,       /*[GB+9636C630, GB+9636C639]*/
  0,     0,     0,     2776,  0,     0,     34781, 40693, 0,     0,       /*[GB+9636C730, GB+9636C739]*/
  39732, 6695,  2748,  35037, 37095, 0,     34418, 807,   0,     0,       /*[GB+9636C830, GB+9636C839]*/
  0,     0,     0,     0,     0,     0,     41097, 0,     26289, 0,       /*[GB+9636C930, GB+9636C939]*/
  11655, 40579, 15845, 0,     10616, 0,     0,     0,     0,     0,       /*[GB+9636CA30, GB+9636CA39]*/
  0,     0,     0,     229,   40754, 0,     0,     27930, 0,     34520,   /*[GB+9636CB30, GB+9636CB39]*/
  3766,  36787, 0,     235,   0,     0,     0,     41562, 15858, 0,       /*[GB+9636CC30, GB+9636CC39]*/
  0,     18349, 0,     17822, 0,     0,     2337,  0,     20044, 0,       /*[GB+9636CD30, GB+9636CD39]*/
  13578, 0,     0,     34176, 34957, 36587, 35711, 0,     7097,  38860,   /*[GB+9636CE30, GB+9636CE39]*/
  12403, 31031, 23284, 33857, 0,     0,     31772, 0,     0,     0,       /*[GB+9636CF30, GB+9636CF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     32285,   /*[GB+9636D030, GB+9636D039]*/
  3322,  0,     0,     0,     11302, 0,     31032, 35983, 40520, 0,       /*[GB+9636D130, GB+9636D139]*/
  0,     0,     0,     0,     0,     9281,  0,     0,     0,     19791,   /*[GB+9636D230, GB+9636D239]*/
  2781,  23605, 13184, 0,     30379, 0,     0,     33634, 0,     0,       /*[GB+9636D330, GB+9636D339]*/
  34810, 0,     27710, 13596, 29908, 12249, 0,     5145,  0,     30380,   /*[GB+9636D430, GB+9636D439]*/
  29299, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9636D530, GB+9636D539]*/
  8560,  18393, 0,     28359, 0,     30729, 0,     0,     0,     0,       /*[GB+9636D630, GB+9636D639]*/
  0,     0,     6116,  0,     0,     0,     1365,  0,     9677,  4949,    /*[GB+9636D730, GB+9636D739]*/
  0,     25313, 0,     0,     17841, 29920, 0,     0,     43732, 10892,   /*[GB+9636D830, GB+9636D839]*/
  0,     0,     0,     0,     0,     0,     0,     3853,  2516,  0,       /*[GB+9636D930, GB+9636D939]*/
  0,     33196, 34822, 19051, 36596, 0,     0,     0,     0,     0,       /*[GB+9636DA30, GB+9636DA39]*/
  0,     1366,  0,     12207, 0,     33656, 3654,  35742, 22568, 36695,   /*[GB+9636DB30, GB+9636DB39]*/
  0,     35230, 41009, 0,     30107, 0,     33984, 1085,  26871, 29469,   /*[GB+9636DC30, GB+9636DC39]*/
  621,   10309, 16137, 3336,  0,     0,     35923, 0,     0,     0,       /*[GB+9636DD30, GB+9636DD39]*/
  0,     0,     0,     0,     7448,  18412, 33676, 12212, 29934, 0,       /*[GB+9636DE30, GB+9636DE39]*/
  3126,  0,     0,     0,     0,     0,     20099, 36696, 0,     0,       /*[GB+9636DF30, GB+9636DF39]*/
  5606,  5500,  37063, 0,     41137, 15645, 0,     7157,  27714, 17380,   /*[GB+9636E030, GB+9636E039]*/
  29946, 0,     0,     0,     0,     0,     0,     40834, 0,     0,       /*[GB+9636E130, GB+9636E139]*/
  0,     1411,  0,     0,     38729, 0,     10897, 0,     19065, 0,       /*[GB+9636E230, GB+9636E239]*/
  0,     10933, 37129, 0,     0,     0,     0,     0,     29476, 134,     /*[GB+9636E330, GB+9636E339]*/
  26187, 0,     0,     17927, 35106, 0,     28988, 0,     0,     0,       /*[GB+9636E430, GB+9636E439]*/
  0,     0,     0,     20001, 0,     26812, 18686, 0,     0,     0,       /*[GB+9636E530, GB+9636E539]*/
  0,     0,     34978, 0,     38815, 0,     0,     35116, 0,     0,       /*[GB+9636E630, GB+9636E639]*/
  0,     0,     0,     0,     19384, 39179, 0,     0,     0,     0,       /*[GB+9636E730, GB+9636E739]*/
  5638,  17569, 34882, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9636E830, GB+9636E839]*/
  38280, 0,     0,     0,     33232, 0,     38058, 32050, 2204,  0,       /*[GB+9636E930, GB+9636E939]*/
  35379, 0,     0,     29492, 14794, 0,     427,   7669,  0,     22038,   /*[GB+9636EA30, GB+9636EA39]*/
  27484, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9636EB30, GB+9636EB39]*/
  0,     0,     0,     0,     5310,  13350, 0,     0,     0,     0,       /*[GB+9636EC30, GB+9636EC39]*/
  29867, 26129, 439,   0,     0,     0,     0,     0,     0,     0,       /*[GB+9636ED30, GB+9636ED39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9636EE30, GB+9636EE39]*/
  0,     0,     31215, 39207, 0,     32105, 0,     32296, 39360, 0,       /*[GB+9636EF30, GB+9636EF39]*/
  6544,  0,     0,     33849, 16597, 0,     0,     0,     26132, 6962,    /*[GB+9636F030, GB+9636F039]*/
  7227,  0,     38150, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9636F130, GB+9636F139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9636F230, GB+9636F239]*/
  0,     0,     0,     0,     0,     0,     33564, 25878, 29872, 0,       /*[GB+9636F330, GB+9636F339]*/
  0,     19477, 0,     32354, 0,     0,     0,     0,     0,     0,       /*[GB+9636F430, GB+9636F439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9636F530, GB+9636F539]*/
  0,     0,     0,     3249,  42823, 0,     21038, 0,     0,     0,       /*[GB+9636F630, GB+9636F639]*/
  0,     0,     0,     26652, 0,     0,     0,     0,     0,     39406,   /*[GB+9636F730, GB+9636F739]*/
  0,     38212, 21426, 0,     0,     17359, 42365, 0,     0,     0,       /*[GB+9636F830, GB+9636F839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9636F930, GB+9636F939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9636FA30, GB+9636FA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     22719, 0,       /*[GB+9636FB30, GB+9636FB39]*/
  614,   0,     0,     0,     6559,  0,     0,     0,     42366, 0,       /*[GB+9636FC30, GB+9636FC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9636FD30, GB+9636FD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9636FE30, GB+9636FE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96378130, GB+96378139]*/
  0,     0,     0,     0,     0,     0,     0,     38247, 0,     0,       /*[GB+96378230, GB+96378239]*/
  19316, 39515, 4787,  6066,  42389, 0,     0,     0,     27394, 0,       /*[GB+96378330, GB+96378339]*/
  0,     14041, 0,     0,     0,     0,     0,     32449, 0,     14934,   /*[GB+96378430, GB+96378439]*/
  0,     23690, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96378530, GB+96378539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96378630, GB+96378639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96378730, GB+96378739]*/
  0,     0,     0,     20083, 26491, 0,     33871, 5280,  0,     0,       /*[GB+96378830, GB+96378839]*/
  0,     0,     11867, 20530, 0,     17826, 0,     0,     0,     0,       /*[GB+96378930, GB+96378939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96378A30, GB+96378A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96378B30, GB+96378B39]*/
  0,     0,     0,     0,     9565,  0,     0,     0,     0,     19453,   /*[GB+96378C30, GB+96378C39]*/
  0,     0,     0,     0,     0,     0,     17732, 0,     43147, 0,       /*[GB+96378D30, GB+96378D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96378E30, GB+96378E39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96378F30, GB+96378F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96379030, GB+96379039]*/
  0,     0,     0,     0,     40735, 10992, 34739, 0,     0,     10995,   /*[GB+96379130, GB+96379139]*/
  18450, 0,     0,     6214,  9327,  0,     0,     0,     0,     0,       /*[GB+96379230, GB+96379239]*/
  0,     0,     0,     0,     0,     12128, 0,     0,     0,     0,       /*[GB+96379330, GB+96379339]*/
  0,     0,     7474,  0,     34144, 0,     0,     0,     0,     13824,   /*[GB+96379430, GB+96379439]*/
  24971, 34149, 39333, 0,     0,     0,     0,     0,     0,     0,       /*[GB+96379530, GB+96379539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96379630, GB+96379639]*/
  0,     41930, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96379730, GB+96379739]*/
  43155, 0,     0,     0,     3021,  0,     0,     0,     0,     0,       /*[GB+96379830, GB+96379839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96379930, GB+96379939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96379A30, GB+96379A39]*/
  0,     35295, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96379B30, GB+96379B39]*/
  0,     0,     0,     22581, 0,     0,     0,     0,     0,     0,       /*[GB+96379C30, GB+96379C39]*/
  0,     13535, 31260, 0,     0,     0,     0,     36895, 0,     0,       /*[GB+96379D30, GB+96379D39]*/
  0,     10944, 35367, 19417, 0,     607,   1092,  0,     0,     0,       /*[GB+96379E30, GB+96379E39]*/
  0,     9602,  19271, 0,     0,     17206, 25222, 14449, 0,     7519,    /*[GB+96379F30, GB+96379F39]*/
  37280, 6746,  0,     0,     0,     0,     0,     12798, 6853,  23783,   /*[GB+9637A030, GB+9637A039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     40091, 0,       /*[GB+9637A130, GB+9637A139]*/
  0,     19818, 35491, 0,     0,     18647, 37314, 33677, 37321, 0,       /*[GB+9637A230, GB+9637A239]*/
  33728, 0,     0,     19946, 0,     37011, 0,     0,     26701, 0,       /*[GB+9637A330, GB+9637A339]*/
  0,     0,     0,     26934, 32258, 43800, 7812,  38091, 6865,  30076,   /*[GB+9637A430, GB+9637A439]*/
  0,     23987, 0,     10656, 0,     35845, 0,     9553,  0,     6034,    /*[GB+9637A530, GB+9637A539]*/
  0,     22350, 0,     0,     0,     0,     0,     0,     7231,  32221,   /*[GB+9637A630, GB+9637A639]*/
  0,     0,     23012, 0,     0,     0,     0,     0,     6296,  12243,   /*[GB+9637A730, GB+9637A739]*/
  0,     0,     31033, 7833,  0,     0,     0,     33300, 274,   0,       /*[GB+9637A830, GB+9637A839]*/
  24455, 0,     0,     0,     0,     10678, 0,     9088,  18064, 0,       /*[GB+9637A930, GB+9637A939]*/
  7054,  8759,  22375, 0,     2600,  0,     0,     0,     0,     14520,   /*[GB+9637AA30, GB+9637AA39]*/
  0,     26101, 0,     38772, 34671, 0,     0,     0,     0,     11694,   /*[GB+9637AB30, GB+9637AB39]*/
  0,     0,     29921, 0,     0,     0,     0,     0,     0,     9240,    /*[GB+9637AC30, GB+9637AC39]*/
  7449,  35924, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9637AD30, GB+9637AD39]*/
  33699, 0,     0,     0,     16283, 10467, 14529, 4369,  17855, 11274,   /*[GB+9637AE30, GB+9637AE39]*/
  43983, 0,     0,     0,     0,     24345, 0,     0,     5836,  33729,   /*[GB+9637AF30, GB+9637AF39]*/
  17858, 0,     28150, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9637B030, GB+9637B039]*/
  0,     0,     0,     19947, 0,     7044,  16106, 0,     0,     3028,    /*[GB+9637B130, GB+9637B139]*/
  0,     35119, 32242, 41624, 0,     43109, 0,     0,     0,     19615,   /*[GB+9637B230, GB+9637B239]*/
  17893, 0,     0,     0,     5350,  8942,  0,     0,     0,     0,       /*[GB+9637B330, GB+9637B339]*/
  0,     0,     28894, 0,     0,     36817, 0,     0,     0,     0,       /*[GB+9637B430, GB+9637B439]*/
  8973,  0,     0,     0,     36728, 0,     0,     0,     0,     0,       /*[GB+9637B530, GB+9637B539]*/
  30015, 0,     0,     0,     42559, 0,     28895, 0,     0,     0,       /*[GB+9637B630, GB+9637B639]*/
  0,     28637, 0,     0,     0,     2944,  0,     0,     0,     0,       /*[GB+9637B730, GB+9637B739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9637B830, GB+9637B839]*/
  0,     0,     0,     12308, 38730, 0,     0,     21378, 0,     0,       /*[GB+9637B930, GB+9637B939]*/
  0,     0,     0,     0,     26927, 0,     0,     34653, 0,     0,       /*[GB+9637BA30, GB+9637BA39]*/
  955,   0,     0,     0,     4367,  0,     0,     0,     0,     24695,   /*[GB+9637BB30, GB+9637BB39]*/
  0,     0,     0,     0,     0,     0,     0,     16352, 17243, 0,       /*[GB+9637BC30, GB+9637BC39]*/
  0,     26714, 0,     10881, 3809,  20876, 0,     0,     0,     0,       /*[GB+9637BD30, GB+9637BD39]*/
  41064, 0,     0,     0,     8184,  0,     3227,  0,     0,     0,       /*[GB+9637BE30, GB+9637BE39]*/
  0,     0,     0,     15053, 0,     0,     0,     0,     0,     0,       /*[GB+9637BF30, GB+9637BF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     19012,   /*[GB+9637C030, GB+9637C039]*/
  0,     22344, 22345, 31570, 0,     0,     0,     26852, 43861, 23994,   /*[GB+9637C130, GB+9637C139]*/
  39299, 0,     29085, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9637C230, GB+9637C239]*/
  0,     0,     0,     19026, 29011, 31889, 0,     9154,  28264, 32108,   /*[GB+9637C330, GB+9637C339]*/
  0,     22458, 0,     0,     0,     32172, 9501,  6297,  0,     0,       /*[GB+9637C430, GB+9637C439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     21361, 701,     /*[GB+9637C530, GB+9637C539]*/
  0,     35048, 29624, 31519, 10280, 0,     0,     8546,  10289, 0,       /*[GB+9637C630, GB+9637C639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     10290,   /*[GB+9637C730, GB+9637C739]*/
  0,     0,     0,     0,     0,     0,     0,     42946, 0,     17366,   /*[GB+9637C830, GB+9637C839]*/
  0,     20439, 31282, 26906, 0,     0,     6561,  0,     0,     0,       /*[GB+9637C930, GB+9637C939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9637CA30, GB+9637CA39]*/
  15759, 36679, 0,     0,     27373, 5494,  0,     0,     21062, 0,       /*[GB+9637CB30, GB+9637CB39]*/
  0,     0,     0,     0,     20316, 0,     35925, 0,     33678, 0,       /*[GB+9637CC30, GB+9637CC39]*/
  0,     0,     0,     0,     0,     0,     20627, 24789, 0,     2753,    /*[GB+9637CD30, GB+9637CD39]*/
  0,     0,     28194, 10808, 24474, 0,     0,     0,     13033, 0,       /*[GB+9637CE30, GB+9637CE39]*/
  0,     0,     0,     18452, 0,     39884, 631,   0,     30950, 0,       /*[GB+9637CF30, GB+9637CF39]*/
  0,     0,     0,     4079,  0,     0,     4080,  0,     0,     0,       /*[GB+9637D030, GB+9637D039]*/
  24808, 33730, 1003,  0,     0,     0,     37640, 0,     0,     10117,   /*[GB+9637D130, GB+9637D139]*/
  0,     43053, 0,     0,     0,     17869, 17870, 2681,  0,     0,       /*[GB+9637D230, GB+9637D239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     10119, 0,       /*[GB+9637D330, GB+9637D339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9637D430, GB+9637D439]*/
  0,     6441,  0,     0,     0,     0,     0,     36479, 0,     0,       /*[GB+9637D530, GB+9637D539]*/
  0,     0,     22998, 29101, 0,     42367, 0,     0,     0,     42462,   /*[GB+9637D630, GB+9637D639]*/
  0,     0,     42463, 24151, 0,     15213, 15215, 38059, 39345, 0,       /*[GB+9637D730, GB+9637D739]*/
  15216, 0,     0,     0,     0,     12167, 0,     6866,  0,     0,       /*[GB+9637D830, GB+9637D839]*/
  0,     0,     0,     0,     0,     0,     0,     30875, 25812, 25910,   /*[GB+9637D930, GB+9637D939]*/
  35381, 2813,  13361, 0,     25998, 0,     0,     0,     0,     0,       /*[GB+9637DA30, GB+9637DA39]*/
  0,     0,     6870,  42014, 0,     0,     23486, 0,     33435, 0,       /*[GB+9637DB30, GB+9637DB39]*/
  32998, 0,     0,     0,     0,     0,     1281,  0,     0,     887,     /*[GB+9637DC30, GB+9637DC39]*/
  0,     22226, 14904, 30879, 18880, 0,     0,     0,     0,     2509,    /*[GB+9637DD30, GB+9637DD39]*/
  40991, 0,     0,     0,     0,     0,     3759,  36422, 16988, 38694,   /*[GB+9637DE30, GB+9637DE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     19641, 25223,   /*[GB+9637DF30, GB+9637DF39]*/
  0,     10843, 28222, 10588, 2659,  0,     14349, 17761, 0,     0,       /*[GB+9637E030, GB+9637E039]*/
  0,     0,     0,     0,     33589, 0,     10281, 2039,  0,     0,       /*[GB+9637E130, GB+9637E139]*/
  0,     0,     0,     0,     0,     42207, 0,     32037, 8547,  0,       /*[GB+9637E230, GB+9637E239]*/
  11602, 0,     23025, 38213, 43195, 0,     0,     0,     2702,  0,       /*[GB+9637E330, GB+9637E339]*/
  0,     0,     29995, 0,     0,     40446, 0,     0,     0,     0,       /*[GB+9637E430, GB+9637E439]*/
  0,     0,     6883,  0,     33607, 4391,  35483, 14361, 28437, 0,       /*[GB+9637E530, GB+9637E539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9637E630, GB+9637E639]*/
  32480, 894,   7952,  14042, 26670, 38249, 6562,  35897, 22376, 13185,   /*[GB+9637E730, GB+9637E739]*/
  0,     0,     41779, 24903, 6971,  38579, 0,     15307, 24088, 35393,   /*[GB+9637E830, GB+9637E839]*/
  20425, 2341,  19450, 1724,  5758,  5804,  0,     0,     0,     0,       /*[GB+9637E930, GB+9637E939]*/
  0,     17329, 0,     33635, 15616, 0,     34906, 0,     15007, 0,       /*[GB+9637EA30, GB+9637EA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9637EB30, GB+9637EB39]*/
  0,     0,     0,     0,     0,     0,     0,     32299, 0,     22610,   /*[GB+9637EC30, GB+9637EC39]*/
  34629, 26788, 25841, 31734, 35803, 12583, 0,     0,     0,     21843,   /*[GB+9637ED30, GB+9637ED39]*/
  32784, 0,     39246, 31917, 5390,  0,     0,     0,     16436, 22384,   /*[GB+9637EE30, GB+9637EE39]*/
  0,     8769,  39820, 42228, 0,     10627, 22,    8770,  0,     0,       /*[GB+9637EF30, GB+9637EF39]*/
  0,     0,     0,     0,     0,     0,     0,     37059, 0,     0,       /*[GB+9637F030, GB+9637F039]*/
  28794, 0,     2857,  0,     0,     0,     3866,  40455, 29302, 0,       /*[GB+9637F130, GB+9637F139]*/
  29935, 27686, 0,     7450,  41952, 41,    30270, 2274,  0,     26842,   /*[GB+9637F230, GB+9637F239]*/
  38309, 36841, 0,     22386, 33385, 0,     0,     0,     0,     0,       /*[GB+9637F330, GB+9637F339]*/
  0,     0,     0,     0,     0,     18413, 18414, 0,     33681, 15916,   /*[GB+9637F430, GB+9637F439]*/
  28713, 0,     36492, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9637F530, GB+9637F539]*/
  0,     0,     0,     0,     0,     3392,  38875, 1412,  3300,  0,       /*[GB+9637F630, GB+9637F639]*/
  0,     40969, 32184, 11869, 30966, 38983, 1213,  23158, 18773, 35401,   /*[GB+9637F730, GB+9637F739]*/
  39678, 0,     0,     0,     0,     0,     0,     0,     997,   0,       /*[GB+9637F830, GB+9637F839]*/
  21990, 6977,  0,     5731,  24949, 0,     16186, 0,     0,     0,       /*[GB+9637F930, GB+9637F939]*/
  0,     0,     29303, 0,     0,     34849, 42612, 43258, 0,     1603,    /*[GB+9637FA30, GB+9637FA39]*/
  0,     32815, 33716, 0,     5833,  0,     35007, 23592, 23068, 0,       /*[GB+9637FB30, GB+9637FB39]*/
  0,     1766,  28378, 8042,  21897, 25165, 0,     0,     0,     0,       /*[GB+9637FC30, GB+9637FC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9637FD30, GB+9637FD39]*/
  0,     0,     0,     0,     28145, 30280, 0,     0,     0,     0,       /*[GB+9637FE30, GB+9637FE39]*/
  0,     0,     0,     0,     0,     0,     0,     13237, 0,     0,       /*[GB+96388130, GB+96388139]*/
  5505,  15781, 20149, 0,     405,   0,     0,     0,     38384, 29185,   /*[GB+96388230, GB+96388239]*/
  30141, 32924, 32365, 4228,  0,     0,     31935, 0,     33214, 0,       /*[GB+96388330, GB+96388339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     1455,  0,       /*[GB+96388430, GB+96388439]*/
  38670, 12790, 0,     0,     16190, 23168, 0,     0,     0,     0,       /*[GB+96388530, GB+96388539]*/
  140,   0,     18497, 0,     41109, 0,     0,     0,     0,     0,       /*[GB+96388630, GB+96388639]*/
  0,     0,     0,     0,     0,     20172, 0,     0,     0,     0,       /*[GB+96388730, GB+96388739]*/
  1608,  23625, 0,     0,     0,     0,     0,     0,     0,     10335,   /*[GB+96388830, GB+96388839]*/
  29850, 21335, 34463, 19771, 0,     25178, 0,     0,     0,     0,       /*[GB+96388930, GB+96388939]*/
  0,     3029,  38445, 10879, 0,     18738, 10071, 0,     32840, 0,       /*[GB+96388A30, GB+96388A39]*/
  0,     0,     0,     15985, 17885, 20336, 18578, 0,     24122, 0,       /*[GB+96388B30, GB+96388B39]*/
  0,     0,     0,     0,     12162, 0,     0,     0,     0,     0,       /*[GB+96388C30, GB+96388C39]*/
  0,     0,     0,     0,     0,     0,     10154, 0,     0,     0,       /*[GB+96388D30, GB+96388D39]*/
  0,     15463, 0,     0,     5825,  0,     0,     0,     8039,  0,       /*[GB+96388E30, GB+96388E39]*/
  0,     0,     0,     0,     0,     42000, 0,     21175, 0,     12168,   /*[GB+96388F30, GB+96388F39]*/
  0,     0,     0,     23518, 1186,  0,     24499, 0,     0,     0,       /*[GB+96389030, GB+96389039]*/
  0,     0,     0,     7545,  0,     12172, 0,     0,     0,     0,       /*[GB+96389130, GB+96389139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     3139,    /*[GB+96389230, GB+96389239]*/
  0,     20563, 8359,  0,     0,     0,     0,     0,     0,     0,       /*[GB+96389330, GB+96389339]*/
  0,     4050,  0,     43814, 22464, 34921, 0,     0,     0,     0,       /*[GB+96389430, GB+96389439]*/
  0,     585,   20064, 0,     20574, 0,     0,     22921, 40806, 0,       /*[GB+96389530, GB+96389539]*/
  0,     0,     43028, 11690, 0,     0,     12419, 30797, 4064,  0,       /*[GB+96389630, GB+96389639]*/
  0,     0,     0,     0,     0,     12286, 6653,  0,     0,     0,       /*[GB+96389730, GB+96389739]*/
  0,     0,     0,     0,     0,     21749, 0,     2095,  0,     0,       /*[GB+96389830, GB+96389839]*/
  6096,  0,     5851,  24643, 0,     29694, 4081,  0,     0,     0,       /*[GB+96389930, GB+96389939]*/
  0,     37454, 21379, 19681, 0,     0,     0,     0,     18687, 0,       /*[GB+96389A30, GB+96389A39]*/
  0,     1772,  0,     0,     0,     0,     20249, 0,     0,     0,       /*[GB+96389B30, GB+96389B39]*/
  0,     27836, 44037, 0,     0,     0,     5136,  19419, 41942, 24132,   /*[GB+96389C30, GB+96389C39]*/
  782,   0,     0,     26754, 0,     0,     23988, 22502, 23989, 2165,    /*[GB+96389D30, GB+96389D39]*/
  937,   7632,  0,     21874, 22053, 32261, 41436, 42302, 4916,  0,       /*[GB+96389E30, GB+96389E39]*/
  0,     0,     0,     36312, 13154, 10282, 0,     3535,  0,     0,       /*[GB+96389F30, GB+96389F39]*/
  0,     10772, 34958, 26770, 0,     4599,  0,     8667,  0,     0,       /*[GB+9638A030, GB+9638A039]*/
  0,     0,     8400,  25976, 20065, 26778, 6160,  25300, 0,     0,       /*[GB+9638A130, GB+9638A139]*/
  0,     0,     3122,  30381, 899,   0,     0,     6891,  1102,  11555,   /*[GB+9638A230, GB+9638A239]*/
  0,     0,     0,     0,     41041, 41042, 0,     0,     0,     0,       /*[GB+9638A330, GB+9638A339]*/
  26794, 20628, 10692, 0,     0,     19503, 20119, 0,     43509, 0,       /*[GB+9638A430, GB+9638A439]*/
  24148, 42789, 0,     40711, 0,     23226, 0,     0,     12633, 0,       /*[GB+9638A530, GB+9638A539]*/
  0,     36185, 0,     17301, 0,     0,     0,     0,     0,     23287,   /*[GB+9638A630, GB+9638A639]*/
  36868, 26915, 22622, 0,     17312, 10118, 6819,  0,     0,     19949,   /*[GB+9638A730, GB+9638A739]*/
  0,     0,     0,     13986, 35304, 25184, 0,     0,     34755, 0,       /*[GB+9638A830, GB+9638A839]*/
  0,     10486, 14569, 20941, 0,     0,     34050, 16961, 0,     0,       /*[GB+9638A930, GB+9638A939]*/
  0,     0,     0,     0,     39131, 0,     10563, 21543, 37421, 0,       /*[GB+9638AA30, GB+9638AA39]*/
  0,     32964, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9638AB30, GB+9638AB39]*/
  0,     18353, 15746, 33245, 25702, 0,     0,     0,     0,     0,       /*[GB+9638AC30, GB+9638AC39]*/
  0,     0,     0,     0,     0,     0,     6305,  4270,  0,     0,       /*[GB+9638AD30, GB+9638AD39]*/
  30208, 0,     0,     3920,  0,     0,     26619, 28361, 0,     0,       /*[GB+9638AE30, GB+9638AE39]*/
  0,     0,     0,     0,     0,     36680, 21365, 15625, 0,     0,       /*[GB+9638AF30, GB+9638AF39]*/
  0,     0,     0,     0,     0,     31801, 0,     0,     0,     0,       /*[GB+9638B030, GB+9638B039]*/
  0,     0,     23248, 21991, 0,     0,     0,     0,     0,     8201,    /*[GB+9638B130, GB+9638B139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     333,   10424,   /*[GB+9638B230, GB+9638B239]*/
  11724, 2414,  7066,  10571, 14406, 0,     0,     0,     0,     0,       /*[GB+9638B330, GB+9638B339]*/
  0,     0,     9738,  0,     0,     0,     0,     14827, 0,     0,       /*[GB+9638B430, GB+9638B439]*/
  37133, 0,     0,     0,     0,     0,     0,     0,     2757,  0,       /*[GB+9638B530, GB+9638B539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9638B630, GB+9638B639]*/
  12445, 0,     0,     0,     0,     0,     0,     0,     26541, 16416,   /*[GB+9638B730, GB+9638B739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     26046,   /*[GB+9638B830, GB+9638B839]*/
  38460, 0,     0,     16171, 14325, 0,     0,     4004,  2205,  9383,    /*[GB+9638B930, GB+9638B939]*/
  14247, 42001, 0,     0,     33061, 0,     1786,  0,     0,     0,       /*[GB+9638BA30, GB+9638BA39]*/
  35564, 7813,  0,     0,     37770, 4387,  37413, 0,     0,     37044,   /*[GB+9638BB30, GB+9638BB39]*/
  6867,  36766, 0,     21255, 0,     0,     0,     0,     0,     3623,    /*[GB+9638BC30, GB+9638BC39]*/
  22256, 14255, 0,     23347, 0,     10443, 36211, 0,     0,     8489,    /*[GB+9638BD30, GB+9638BD39]*/
  36148, 8379,  23825, 22049, 33547, 33176, 28066, 0,     0,     33548,   /*[GB+9638BE30, GB+9638BE39]*/
  25091, 21692, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9638BF30, GB+9638BF39]*/
  5485,  0,     4111,  0,     0,     0,     0,     0,     0,     0,       /*[GB+9638C030, GB+9638C039]*/
  0,     8005,  21756, 0,     36215, 35857, 29506, 15531, 21005, 847,     /*[GB+9638C130, GB+9638C139]*/
  0,     37818, 9875,  22808, 0,     37819, 0,     44064, 0,     0,       /*[GB+9638C230, GB+9638C239]*/
  23328, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9638C330, GB+9638C339]*/
  0,     0,     0,     6181,  7820,  38152, 28908, 0,     0,     4917,    /*[GB+9638C430, GB+9638C439]*/
  41764, 0,     0,     0,     0,     0,     28916, 14351, 11547, 7553,    /*[GB+9638C530, GB+9638C539]*/
  0,     0,     0,     0,     17966, 36217, 9452,  0,     29012, 0,       /*[GB+9638C630, GB+9638C639]*/
  15691, 0,     0,     0,     0,     0,     0,     0,     14600, 32645,   /*[GB+9638C730, GB+9638C739]*/
  0,     29022, 15103, 0,     35778, 3993,  0,     0,     0,     0,       /*[GB+9638C830, GB+9638C839]*/
  0,     0,     0,     0,     0,     14143, 22199, 0,     0,     39752,   /*[GB+9638C930, GB+9638C939]*/
  0,     0,     15877, 0,     0,     0,     0,     31536, 0,     0,       /*[GB+9638CA30, GB+9638CA39]*/
  34799, 0,     6647,  0,     10450, 20898, 32481, 35882, 32552, 30210,   /*[GB+9638CB30, GB+9638CB39]*/
  15753, 40241, 0,     10679, 0,     28438, 0,     35052, 19481, 27466,   /*[GB+9638CC30, GB+9638CC39]*/
  0,     0,     0,     0,     0,     0,     0,     1317,  0,     0,       /*[GB+9638CD30, GB+9638CD39]*/
  22630, 0,     0,     0,     32445, 15305, 26742, 0,     0,     0,       /*[GB+9638CE30, GB+9638CE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     22817, 0,       /*[GB+9638CF30, GB+9638CF39]*/
  21427, 30211, 0,     25620, 0,     0,     10779, 43355, 21433, 35898,   /*[GB+9638D030, GB+9638D039]*/
  0,     17664, 14629, 6508,  3145,  42390, 7929,  13647, 16077, 12843,   /*[GB+9638D130, GB+9638D139]*/
  24768, 15387, 43272, 0,     6195,  14371, 36434, 0,     41241, 0,       /*[GB+9638D230, GB+9638D239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     42051,   /*[GB+9638D330, GB+9638D339]*/
  0,     20736, 11609, 35204, 28789, 10169, 0,     0,     0,     0,       /*[GB+9638D430, GB+9638D439]*/
  0,     0,     0,     0,     0,     0,     0,     35899, 0,     0,       /*[GB+9638D530, GB+9638D539]*/
  13194, 0,     4457,  0,     21064, 27374, 35727, 12209, 27943, 15542,   /*[GB+9638D630, GB+9638D639]*/
  0,     0,     0,     16936, 0,     11506, 9982,  9891,  0,     21436,   /*[GB+9638D730, GB+9638D739]*/
  6396,  1368,  18842, 17284, 28082, 20084, 33973, 0,     0,     29812,   /*[GB+9638D830, GB+9638D839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9638D930, GB+9638D939]*/
  0,     0,     23455, 0,     18008, 0,     0,     0,     0,     0,       /*[GB+9638DA30, GB+9638DA39]*/
  0,     0,     0,     16696, 10176, 0,     0,     33657, 23611, 27789,   /*[GB+9638DB30, GB+9638DB39]*/
  16009, 11464, 10053, 15635, 0,     22442, 11946, 9311,  18415, 9681,    /*[GB+9638DC30, GB+9638DC39]*/
  31158, 0,     20684, 30964, 5283,  0,     0,     0,     24790, 39844,   /*[GB+9638DD30, GB+9638DD39]*/
  14391, 21774, 9441,  21393, 32894, 0,     31064, 24940, 0,     0,       /*[GB+9638DE30, GB+9638DE39]*/
  3393,  6659,  0,     23612, 0,     0,     0,     0,     0,     0,       /*[GB+9638DF30, GB+9638DF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9638E030, GB+9638E039]*/
  10802, 0,     0,     0,     21589, 20629, 20120, 24796, 16560, 0,       /*[GB+9638E130, GB+9638E139]*/
  6409,  25157, 14296, 16284, 25679, 21590, 28038, 0,     0,     0,       /*[GB+9638E230, GB+9638E239]*/
  0,     0,     0,     13791, 18109, 16488, 35933, 0,     0,     0,       /*[GB+9638E330, GB+9638E339]*/
  0,     0,     0,     20775, 8335,  3017,  12492, 0,     0,     0,       /*[GB+9638E430, GB+9638E439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9638E530, GB+9638E539]*/
  0,     0,     0,     0,     0,     16655, 0,     0,     32816, 10057,   /*[GB+9638E630, GB+9638E639]*/
  13238, 44006, 12868, 35095, 0,     0,     0,     0,     30127, 0,       /*[GB+9638E730, GB+9638E739]*/
  27408, 0,     30944, 0,     0,     0,     41982, 17156, 34695, 21632,   /*[GB+9638E830, GB+9638E839]*/
  0,     34004, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9638E930, GB+9638E939]*/
  0,     0,     27513, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9638EA30, GB+9638EA39]*/
  0,     0,     0,     0,     0,     7481,  21786, 42958, 43772, 21377,   /*[GB+9638EB30, GB+9638EB39]*/
  0,     33731, 0,     30770, 0,     13712, 3104,  13252, 0,     0,       /*[GB+9638EC30, GB+9638EC39]*/
  0,     0,     0,     0,     0,     0,     43398, 0,     0,     0,       /*[GB+9638ED30, GB+9638ED39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9638EE30, GB+9638EE39]*/
  0,     0,     0,     11214, 2723,  25175, 21640, 41793, 16713, 37641,   /*[GB+9638EF30, GB+9638EF39]*/
  42711, 0,     13829, 0,     23656, 0,     0,     12441, 0,     21941,   /*[GB+9638F030, GB+9638F039]*/
  19608, 0,     0,     0,     0,     0,     0,     0,     29421, 0,       /*[GB+9638F130, GB+9638F139]*/
  20747, 18159, 0,     0,     29481, 37920, 0,     0,     0,     0,       /*[GB+9638F230, GB+9638F239]*/
  18518, 0,     0,     0,     35011, 30946, 35635, 0,     0,     36142,   /*[GB+9638F330, GB+9638F339]*/
  18532, 21336, 32701, 21337, 0,     0,     0,     0,     0,     0,       /*[GB+9638F430, GB+9638F439]*/
  0,     0,     37465, 17511, 19385, 38827, 38828, 18549, 13976, 10201,   /*[GB+9638F530, GB+9638F539]*/
  0,     0,     0,     10000, 35299, 0,     0,     28390, 0,     43925,   /*[GB+9638F630, GB+9638F639]*/
  0,     0,     0,     0,     0,     0,     0,     16259, 17892, 0,       /*[GB+9638F730, GB+9638F739]*/
  0,     0,     0,     0,     1613,  0,     0,     29161, 23346, 35846,   /*[GB+9638F830, GB+9638F839]*/
  0,     0,     0,     10962, 0,     0,     0,     0,     10238, 0,       /*[GB+9638F930, GB+9638F939]*/
  41142, 17629, 4988,  42369, 8298,  25887, 0,     40804, 22377, 23223,   /*[GB+9638FA30, GB+9638FA39]*/
  38251, 0,     0,     4794,  28707, 13918, 43659, 36830, 38775, 0,       /*[GB+9638FB30, GB+9638FB39]*/
  0,     0,     27398, 0,     27158, 33985, 0,     9985,  16184, 0,       /*[GB+9638FC30, GB+9638FC39]*/
  33474, 0,     25949, 4816,  0,     40837, 35266, 0,     39886, 40638,   /*[GB+9638FD30, GB+9638FD39]*/
  4417,  18242, 18243, 35943, 21461, 35945, 27413, 0,     0,     10331,   /*[GB+9638FE30, GB+9638FE39]*/
  27229, 43896, 0,     20989, 0,     25960, 5014,  5015,  5018,  0,       /*[GB+96398130, GB+96398139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     33240, 0,       /*[GB+96398230, GB+96398239]*/
  7670,  479,   0,     0,     0,     0,     0,     0,     25219, 42329,   /*[GB+96398330, GB+96398339]*/
  31391, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96398430, GB+96398439]*/
  42330, 42107, 0,     12184, 4043,  0,     42331, 0,     0,     0,       /*[GB+96398530, GB+96398539]*/
  38931, 0,     0,     0,     33456, 895,   0,     11394, 28505, 0,       /*[GB+96398630, GB+96398639]*/
  15887, 17079, 0,     0,     37979, 36546, 0,     0,     1148,  0,       /*[GB+96398730, GB+96398739]*/
  0,     0,     16778, 25301, 0,     0,     41440, 0,     16162, 28187,   /*[GB+96398830, GB+96398839]*/
  0,     0,     246,   41579, 42420, 0,     0,     0,     0,     0,       /*[GB+96398930, GB+96398939]*/
  0,     42438, 0,     14392, 0,     0,     0,     0,     0,     0,       /*[GB+96398A30, GB+96398A39]*/
  29552, 24681, 23500, 0,     0,     0,     0,     0,     0,     0,       /*[GB+96398B30, GB+96398B39]*/
  43399, 0,     0,     0,     0,     0,     0,     38829, 0,     12078,   /*[GB+96398C30, GB+96398C39]*/
  0,     36501, 0,     0,     28754, 23514, 24701, 0,     0,     0,       /*[GB+96398D30, GB+96398D39]*/
  20878, 2767,  0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96398E30, GB+96398E39]*/
  0,     40354, 0,     0,     9762,  0,     0,     0,     0,     0,       /*[GB+96398F30, GB+96398F39]*/
  0,     0,     0,     0,     0,     0,     0,     5486,  0,     22554,   /*[GB+96399030, GB+96399039]*/
  0,     41540, 0,     11651, 0,     0,     0,     0,     0,     16535,   /*[GB+96399130, GB+96399139]*/
  0,     6358,  31446, 0,     0,     9958,  0,     0,     26119, 28834,   /*[GB+96399230, GB+96399239]*/
  13676, 381,   22322, 24395, 13631, 0,     0,     0,     0,     0,       /*[GB+96399330, GB+96399339]*/
  0,     0,     0,     0,     0,     0,     0,     41645, 2252,  33152,   /*[GB+96399430, GB+96399439]*/
  0,     2730,  0,     0,     0,     0,     22556, 16211, 0,     0,       /*[GB+96399530, GB+96399539]*/
  0,     6767,  9959,  26133, 0,     27127, 37967, 6965,  17904, 0,       /*[GB+96399630, GB+96399639]*/
  42671, 16756, 7274,  34227, 6187,  23408, 7387,  39004, 33449, 16543,   /*[GB+96399730, GB+96399739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+96399830, GB+96399839]*/
  0,     0,     0,     0,     0,     0,     0,     42332, 0,     0,       /*[GB+96399930, GB+96399939]*/
  0,     42208, 36313, 0,     0,     41966, 0,     0,     37101, 0,       /*[GB+96399A30, GB+96399A39]*/
  12829, 13155, 22595, 0,     0,     0,     0,     36531, 20046, 0,       /*[GB+96399B30, GB+96399B39]*/
  0,     8401,  27629, 4271,  32763, 0,     8669,  16547, 29630, 16314,   /*[GB+96399C30, GB+96399C39]*/
  0,     38214, 11988, 0,     0,     0,     0,     0,     0,     0,       /*[GB+96399D30, GB+96399D39]*/
  39653, 0,     0,     0,     0,     18374, 0,     0,     19180, 6849,    /*[GB+96399E30, GB+96399E39]*/
  7407,  0,     0,     0,     25621, 6782,  0,     0,     14630, 4603,    /*[GB+96399F30, GB+96399F39]*/
  34539, 0,     0,     0,     9159,  0,     0,     0,     0,     0,       /*[GB+9639A030, GB+9639A039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9639A130, GB+9639A139]*/
  0,     0,     0,     0,     0,     0,     5759,  0,     0,     0,       /*[GB+9639A230, GB+9639A239]*/
  11307, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9639A330, GB+9639A339]*/
  0,     43660, 0,     0,     25139, 32138, 11312, 24141, 9461,  0,       /*[GB+9639A430, GB+9639A439]*/
  23043, 0,     0,     16586, 0,     0,     42229, 0,     0,     0,       /*[GB+9639A530, GB+9639A539]*/
  0,     0,     0,     34924, 0,     0,     0,     0,     0,     0,       /*[GB+9639A630, GB+9639A639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     28368,   /*[GB+9639A730, GB+9639A739]*/
  39528, 14150, 0,     0,     0,     0,     0,     25051, 3414,  27683,   /*[GB+9639A830, GB+9639A839]*/
  0,     30271, 25410, 12600, 0,     28129, 7451,  0,     0,     0,       /*[GB+9639A930, GB+9639A939]*/
  0,     0,     0,     0,     0,     26225, 0,     16482, 0,     0,       /*[GB+9639AA30, GB+9639AA39]*/
  33086, 43500, 13210, 0,     0,     0,     0,     5305,  0,     5206,    /*[GB+9639AB30, GB+9639AB39]*/
  0,     0,     0,     19855, 16285, 17223, 5951,  25794, 0,     22115,   /*[GB+9639AC30, GB+9639AC39]*/
  0,     0,     10505, 11320, 3196,  11231, 0,     0,     0,     0,       /*[GB+9639AD30, GB+9639AD39]*/
  0,     0,     0,     0,     2524,  41595, 16406, 0,     10372, 0,       /*[GB+9639AE30, GB+9639AE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     35753, 23926,   /*[GB+9639AF30, GB+9639AF39]*/
  15940, 0,     0,     9002,  29553, 0,     30128, 26805, 0,     33319,   /*[GB+9639B030, GB+9639B039]*/
  36708, 23963, 34932, 23593, 0,     0,     0,     0,     0,     0,       /*[GB+9639B130, GB+9639B139]*/
  0,     0,     0,     0,     0,     0,     0,     17299, 0,     0,       /*[GB+9639B230, GB+9639B239]*/
  0,     0,     0,     0,     0,     0,     0,     1129,  41791, 13253,   /*[GB+9639B330, GB+9639B339]*/
  0,     0,     14876, 18680, 16519, 0,     42722, 0,     0,     0,       /*[GB+9639B430, GB+9639B439]*/
  0,     0,     0,     0,     40772, 0,     17829, 24810, 1938,  0,       /*[GB+9639B530, GB+9639B539]*/
  0,     0,     0,     0,     0,     0,     0,     38671, 24356, 0,       /*[GB+9639B630, GB+9639B639]*/
  0,     0,     0,     29824, 20295, 0,     5971,  6820,  18966, 43411,   /*[GB+9639B730, GB+9639B739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     3129,  0,       /*[GB+9639B830, GB+9639B839]*/
  0,     0,     0,     0,     0,     0,     3444,  30446, 0,     0,       /*[GB+9639B930, GB+9639B939]*/
  0,     0,     0,     26818, 5785,  0,     0,     26942, 0,     24597,   /*[GB+9639BA30, GB+9639BA39]*/
  0,     0,     0,     0,     0,     0,     42591, 35012, 0,     0,       /*[GB+9639BB30, GB+9639BB39]*/
  14172, 26547, 17948, 2308,  26749, 0,     0,     0,     0,     0,       /*[GB+9639BC30, GB+9639BC39]*/
  0,     34754, 17997, 0,     0,     0,     17605, 0,     0,     17526,   /*[GB+9639BD30, GB+9639BD39]*/
  0,     17268, 0,     17269, 0,     0,     39972, 0,     0,     0,       /*[GB+9639BE30, GB+9639BE39]*/
  0,     0,     14433, 25902, 12927, 0,     0,     8984,  0,     0,       /*[GB+9639BF30, GB+9639BF39]*/
  38060, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9639C030, GB+9639C039]*/
  0,     0,     0,     0,     38092, 42008, 0,     0,     1692,  28570,   /*[GB+9639C130, GB+9639C139]*/
  15286, 0,     0,     0,     0,     11014, 8491,  0,     15354, 0,       /*[GB+9639C230, GB+9639C239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9639C330, GB+9639C339]*/
  0,     0,     0,     0,     0,     0,     44043, 37799, 0,     0,       /*[GB+9639C430, GB+9639C439]*/
  0,     0,     34500, 37800, 0,     29613, 0,     4494,  0,     0,       /*[GB+9639C530, GB+9639C539]*/
  22057, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9639C630, GB+9639C639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     41165,   /*[GB+9639C730, GB+9639C739]*/
  0,     36309, 0,     0,     27287, 0,     12816, 0,     0,     0,       /*[GB+9639C830, GB+9639C839]*/
  28234, 0,     0,     0,     29040, 0,     0,     0,     0,     0,       /*[GB+9639C930, GB+9639C939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     39369, 39753,   /*[GB+9639CA30, GB+9639CA39]*/
  0,     0,     16063, 0,     0,     0,     0,     0,     0,     43468,   /*[GB+9639CB30, GB+9639CB39]*/
  20058, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9639CC30, GB+9639CC39]*/
  0,     0,     0,     0,     0,     0,     3773,  17137, 0,     0,       /*[GB+9639CD30, GB+9639CD39]*/
  0,     0,     43283, 0,     0,     0,     0,     0,     11500, 0,       /*[GB+9639CE30, GB+9639CE39]*/
  0,     35799, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9639CF30, GB+9639CF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9639D030, GB+9639D039]*/
  0,     0,     43654, 38978, 0,     0,     42836, 0,     0,     19596,   /*[GB+9639D130, GB+9639D139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9639D230, GB+9639D239]*/
  0,     23338, 0,     0,     0,     0,     0,     0,     1393,  0,       /*[GB+9639D330, GB+9639D339]*/
  3867,  0,     43868, 8306,  20100, 24711, 0,     0,     0,     0,       /*[GB+9639D430, GB+9639D439]*/
  0,     43143, 0,     41588, 28714, 39669, 20101, 18812, 0,     0,       /*[GB+9639D530, GB+9639D539]*/
  0,     15935, 18982, 1087,  0,     0,     0,     0,     0,     0,       /*[GB+9639D630, GB+9639D639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9639D730, GB+9639D739]*/
  0,     38368, 9987,  0,     0,     0,     0,     0,     0,     0,       /*[GB+9639D830, GB+9639D839]*/
  366,   0,     0,     10321, 21898, 0,     36862, 17485, 0,     0,       /*[GB+9639D930, GB+9639D939]*/
  2123,  0,     0,     40641, 39335, 0,     0,     0,     0,     0,       /*[GB+9639DA30, GB+9639DA39]*/
  0,     0,     0,     0,     0,     38433, 22783, 0,     0,     0,       /*[GB+9639DB30, GB+9639DB39]*/
  0,     0,     22786, 10515, 16109, 0,     28736, 0,     34468, 0,       /*[GB+9639DC30, GB+9639DC39]*/
  25361, 26819, 17041, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9639DD30, GB+9639DD39]*/
  0,     0,     28322, 0,     12838, 0,     32451, 0,     0,     0,       /*[GB+9639DE30, GB+9639DE39]*/
  0,     0,     8056,  8063,  12918, 39704, 0,     0,     0,     42002,   /*[GB+9639DF30, GB+9639DF39]*/
  9920,  0,     14570, 20879, 11042, 0,     39289, 0,     0,     0,       /*[GB+9639E030, GB+9639E039]*/
  0,     0,     0,     6868,  29499, 0,     18292, 0,     0,     14577,   /*[GB+9639E130, GB+9639E139]*/
  23215, 506,   39447, 0,     0,     0,     0,     0,     42018, 0,       /*[GB+9639E230, GB+9639E239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     11054, 16363,   /*[GB+9639E330, GB+9639E339]*/
  0,     0,     0,     0,     6872,  13576, 0,     3316,  0,     4495,    /*[GB+9639E430, GB+9639E439]*/
  11068, 41128, 0,     0,     0,     0,     21185, 0,     0,     11069,   /*[GB+9639E530, GB+9639E539]*/
  43571, 42881, 0,     0,     32114, 0,     0,     0,     0,     0,       /*[GB+9639E630, GB+9639E639]*/
  44004, 0,     27291, 7232,  14211, 0,     0,     0,     0,     0,       /*[GB+9639E730, GB+9639E739]*/
  0,     0,     0,     0,     13635, 11265, 0,     1068,  22065, 0,       /*[GB+9639E830, GB+9639E839]*/
  0,     0,     0,     43811, 0,     0,     1722,  0,     16735, 13641,   /*[GB+9639E930, GB+9639E939]*/
  0,     0,     0,     0,     43299, 0,     35603, 0,     0,     0,       /*[GB+9639EA30, GB+9639EA39]*/
  0,     0,     17361, 11096, 0,     25646, 0,     0,     0,     0,       /*[GB+9639EB30, GB+9639EB39]*/
  0,     0,     0,     7971,  28592, 0,     0,     0,     43242, 16847,   /*[GB+9639EC30, GB+9639EC39]*/
  40880, 34234, 0,     1345,  38252, 0,     3146,  0,     0,     0,       /*[GB+9639ED30, GB+9639ED39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9639EE30, GB+9639EE39]*/
  0,     0,     21059, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9639EF30, GB+9639EF39]*/
  0,     0,     0,     0,     32402, 0,     0,     0,     0,     0,       /*[GB+9639F030, GB+9639F039]*/
  0,     33257, 16697, 33159, 35434, 16389, 0,     15239, 6915,  12210,   /*[GB+9639F130, GB+9639F139]*/
  0,     0,     42421, 2586,  27253, 13195, 0,     37584, 0,     0,       /*[GB+9639F230, GB+9639F239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9639F330, GB+9639F339]*/
  0,     15013, 37201, 0,     0,     0,     0,     43712, 0,     0,       /*[GB+9639F430, GB+9639F439]*/
  0,     0,     0,     0,     0,     2429,  0,     0,     0,     0,       /*[GB+9639F530, GB+9639F539]*/
  0,     0,     0,     29542, 18416, 9811,  3150,  0,     7977,  19505,   /*[GB+9639F630, GB+9639F639]*/
  0,     14654, 40331, 0,     0,     42440, 0,     0,     4068,  1732,    /*[GB+9639F730, GB+9639F739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9639F830, GB+9639F839]*/
  0,     0,     0,     0,     0,     0,     13017, 0,     0,     0,       /*[GB+9639F930, GB+9639F939]*/
  0,     0,     0,     0,     0,     0,     20473, 13807, 0,     4414,    /*[GB+9639FA30, GB+9639FA39]*/
  29377, 0,     20283, 16092, 14471, 0,     31164, 40971, 0,     0,       /*[GB+9639FB30, GB+9639FB39]*/
  38348, 0,     4817,  22661, 13605, 0,     0,     0,     0,     0,       /*[GB+9639FC30, GB+9639FC39]*/
  13473, 0,     0,     0,     0,     0,     0,     0,     0,     115,     /*[GB+9639FD30, GB+9639FD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     13930,   /*[GB+9639FE30, GB+9639FE39]*/
  0,     0,     1053,  0,     7753,  30130, 44007, 6215,  9138,  9173,    /*[GB+97308130, GB+97308139]*/
  17736, 0,     43395, 0,     0,     0,     0,     0,     0,     0,       /*[GB+97308230, GB+97308239]*/
  0,     0,     0,     0,     0,     0,     31930, 0,     0,     0,       /*[GB+97308330, GB+97308339]*/
  1430,  15489, 0,     0,     0,     0,     0,     0,     11951, 0,       /*[GB+97308430, GB+97308439]*/
  0,     0,     3584,  0,     0,     0,     16827, 5613,  9334,  35107,   /*[GB+97308530, GB+97308539]*/
  0,     14691, 0,     2379,  0,     0,     0,     0,     0,     0,       /*[GB+97308630, GB+97308639]*/
  0,     0,     43839, 0,     24688, 0,     0,     29186, 2219,  0,       /*[GB+97308730, GB+97308739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     23169,   /*[GB+97308830, GB+97308839]*/
  36869, 42796, 0,     21321, 0,     1547,  0,     34746, 0,     0,       /*[GB+97308930, GB+97308939]*/
  0,     0,     20161, 18153, 0,     0,     0,     13517, 0,     21792,   /*[GB+97308A30, GB+97308A39]*/
  17987, 0,     42523, 39174, 1548,  0,     0,     0,     0,     0,       /*[GB+97308B30, GB+97308B39]*/
  0,     0,     22295, 0,     26916, 0,     0,     0,     0,     0,       /*[GB+97308C30, GB+97308C39]*/
  0,     0,     22969, 0,     0,     0,     0,     0,     0,     0,       /*[GB+97308D30, GB+97308D39]*/
  26702, 41625, 0,     0,     0,     0,     17996, 0,     0,     0,       /*[GB+97308E30, GB+97308E39]*/
  0,     0,     0,     5642,  15992, 0,     0,     0,     19403, 0,       /*[GB+97308F30, GB+97308F39]*/
  19421, 36748, 0,     0,     0,     37354, 41802, 24739, 72,    0,       /*[GB+97309030, GB+97309039]*/
  23317, 20915, 0,     0,     0,     0,     0,     0,     0,     16209,   /*[GB+97309130, GB+97309139]*/
  0,     15220, 36773, 21693, 0,     38553, 32347, 37417, 32214, 24058,   /*[GB+97309230, GB+97309239]*/
  30079, 0,     0,     0,     0,     0,     0,     0,     0,     16598,   /*[GB+97309330, GB+97309339]*/
  0,     0,     11375, 0,     21948, 19276, 38154, 28424, 0,     44065,   /*[GB+97309430, GB+97309439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+97309530, GB+97309539]*/
  0,     31721, 0,     39133, 0,     32262, 10239, 11452, 0,     0,       /*[GB+97309630, GB+97309639]*/
  0,     28918, 0,     35569, 0,     2,     0,     0,     0,     0,       /*[GB+97309730, GB+97309739]*/
  0,     0,     0,     3537,  0,     16544, 38177, 37187, 31475, 6708,    /*[GB+97309830, GB+97309839]*/
  2703,  19591, 0,     0,     0,     26318, 36074, 28439, 16835, 31651,   /*[GB+97309930, GB+97309939]*/
  0,     25453, 0,     4127,  0,     0,     0,     0,     33044, 0,       /*[GB+97309A30, GB+97309A39]*/
  19768, 210,   0,     0,     0,     0,     0,     0,     16220, 0,       /*[GB+97309B30, GB+97309B39]*/
  37377, 0,     8548,  0,     0,     0,     1117,  17543, 25229, 6196,    /*[GB+97309C30, GB+97309C39]*/
  0,     0,     14453, 10171, 0,     35572, 0,     0,     16474, 0,       /*[GB+97309D30, GB+97309D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+97309E30, GB+97309E39]*/
  0,     0,     34540, 33223, 0,     0,     0,     0,     0,     0,       /*[GB+97309F30, GB+97309F39]*/
  0,     0,     0,     0,     9564,  0,     0,     0,     23493, 16390,   /*[GB+9730A030, GB+9730A039]*/
  0,     0,     0,     16698, 0,     0,     0,     0,     0,     31245,   /*[GB+9730A130, GB+9730A139]*/
  9683,  20654, 0,     19598, 0,     0,     0,     32895, 33051, 0,       /*[GB+9730A230, GB+9730A239]*/
  0,     0,     0,     0,     0,     9586,  0,     31802, 23153, 0,       /*[GB+9730A330, GB+9730A339]*/
  0,     21581, 0,     0,     0,     17644, 25464, 20777, 41596, 0,       /*[GB+9730A430, GB+9730A439]*/
  12011, 39029, 14952, 0,     21202, 0,     0,     8605,  0,     0,       /*[GB+9730A530, GB+9730A539]*/
  0,     26505, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9730A630, GB+9730A639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     19604, 8807,    /*[GB+9730A730, GB+9730A739]*/
  5570,  0,     9568,  0,     5105,  0,     0,     35501, 0,     0,       /*[GB+9730A830, GB+9730A839]*/
  0,     0,     0,     0,     31743, 29055, 0,     0,     0,     0,       /*[GB+9730A930, GB+9730A939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     26195, 0,       /*[GB+9730AA30, GB+9730AA39]*/
  0,     0,     0,     0,     0,     0,     0,     27188, 0,     0,       /*[GB+9730AB30, GB+9730AB39]*/
  0,     0,     0,     0,     0,     18534, 0,     0,     0,     13081,   /*[GB+9730AC30, GB+9730AC39]*/
  32206, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9730AD30, GB+9730AD39]*/
  0,     0,     31636, 0,     0,     0,     0,     0,     30631, 0,       /*[GB+9730AE30, GB+9730AE39]*/
  19422, 0,     38061, 27382, 0,     23538, 0,     0,     0,     2304,    /*[GB+9730AF30, GB+9730AF39]*/
  0,     0,     0,     40224, 26762, 43005, 10586, 0,     0,     0,       /*[GB+9730B030, GB+9730B039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     7652,    /*[GB+9730B130, GB+9730B139]*/
  10590, 0,     0,     0,     0,     0,     0,     0,     19037, 0,       /*[GB+9730B230, GB+9730B239]*/
  0,     0,     0,     0,     2191,  24409, 19041, 0,     0,     0,       /*[GB+9730B330, GB+9730B339]*/
  0,     0,     0,     0,     17050, 8562,  31285, 33192, 0,     15389,   /*[GB+9730B430, GB+9730B439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     41723, 0,       /*[GB+9730B530, GB+9730B539]*/
  8583,  0,     0,     0,     0,     0,     0,     0,     19658, 0,       /*[GB+9730B630, GB+9730B639]*/
  0,     0,     33986, 31246, 0,     0,     0,     13462, 0,     0,       /*[GB+9730B730, GB+9730B739]*/
  0,     37446, 0,     4966,  0,     0,     23710, 25851, 0,     0,       /*[GB+9730B830, GB+9730B839]*/
  0,     0,     0,     19671, 13317, 0,     0,     0,     0,     0,       /*[GB+9730B930, GB+9730B939]*/
  31628, 37914, 0,     3061,  0,     0,     0,     0,     0,     9535,    /*[GB+9730BA30, GB+9730BA39]*/
  0,     0,     6638,  0,     0,     0,     0,     0,     0,     0,       /*[GB+9730BB30, GB+9730BB39]*/
  34648, 0,     0,     31509, 0,     43084, 37951, 38123, 43565, 25029,   /*[GB+9730BC30, GB+9730BC39]*/
  0,     0,     2728,  0,     0,     0,     0,     0,     0,     0,       /*[GB+9730BD30, GB+9730BD39]*/
  0,     0,     7090,  0,     0,     3916,  20032, 7094,  0,     7865,    /*[GB+9730BE30, GB+9730BE39]*/
  28909, 31131, 12953, 26214, 29400, 3361,  0,     0,     12360, 29002,   /*[GB+9730BF30, GB+9730BF39]*/
  0,     0,     40361, 0,     0,     0,     0,     0,     19637, 0,       /*[GB+9730C030, GB+9730C039]*/
  20808, 9604,  0,     0,     0,     34228, 22066, 0,     28501, 0,       /*[GB+9730C130, GB+9730C139]*/
  24503, 37425, 26395, 26902, 0,     0,     0,     15605, 0,     0,       /*[GB+9730C230, GB+9730C239]*/
  38634, 556,   23014, 0,     4011,  0,     0,     0,     0,     0,       /*[GB+9730C330, GB+9730C339]*/
  0,     0,     0,     0,     27134, 4379,  23026, 18375, 14145, 16332,   /*[GB+9730C430, GB+9730C439]*/
  2575,  3996,  43067, 43579, 37109, 0,     0,     0,     0,     0,       /*[GB+9730C530, GB+9730C539]*/
  0,     0,     33045, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9730C630, GB+9730C639]*/
  0,     0,     0,     0,     0,     0,     0,     13411, 13412, 0,       /*[GB+9730C730, GB+9730C739]*/
  0,     0,     20269, 26325, 7103,  787,   2822,  0,     2059,  0,       /*[GB+9730C830, GB+9730C839]*/
  3647,  41652, 20942, 0,     13424, 21747, 40165, 0,     33304, 42394,   /*[GB+9730C930, GB+9730C939]*/
  24413, 0,     0,     3924,  0,     0,     0,     0,     0,     21434,   /*[GB+9730CA30, GB+9730CA39]*/
  7953,  5147,  21270, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9730CB30, GB+9730CB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9730CC30, GB+9730CC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9730CD30, GB+9730CD39]*/
  0,     0,     0,     38588, 20927, 6163,  6257,  0,     25406, 0,       /*[GB+9730CE30, GB+9730CE39]*/
  0,     10884, 42127, 15470, 22922, 0,     0,     0,     21437, 9803,    /*[GB+9730CF30, GB+9730CF39]*/
  0,     0,     0,     0,     0,     43069, 23044, 43367, 0,     37585,   /*[GB+9730D030, GB+9730D039]*/
  4891,  0,     35729, 0,     33975, 0,     0,     1978,  0,     0,       /*[GB+9730D130, GB+9730D139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9730D230, GB+9730D239]*/
  0,     0,     2430,  0,     0,     0,     0,     36831, 0,     0,       /*[GB+9730D330, GB+9730D339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9730D430, GB+9730D439]*/
  0,     14869, 0,     0,     15545, 5679,  6896,  0,     37499, 15124,   /*[GB+9730D530, GB+9730D539]*/
  16483, 20584, 32795, 31740, 39251, 23581, 0,     24097, 6517,  0,       /*[GB+9730D630, GB+9730D639]*/
  32796, 3295,  14064, 0,     29543, 43372, 29828, 0,     27401, 0,       /*[GB+9730D730, GB+9730D739]*/
  39536, 0,     0,     0,     0,     7452,  0,     0,     16722, 0,       /*[GB+9730D830, GB+9730D839]*/
  43041, 0,     0,     0,     0,     0,     34131, 0,     0,     0,       /*[GB+9730D930, GB+9730D939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9730DA30, GB+9730DA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9730DB30, GB+9730DB39]*/
  0,     0,     0,     0,     0,     8606,  30906, 33701, 4022,  40102,   /*[GB+9730DC30, GB+9730DC39]*/
  20197, 2594,  28139, 19514, 0,     33996, 13036, 37992, 30602, 0,       /*[GB+9730DD30, GB+9730DD39]*/
  11725, 0,     0,     5151,  9357,  0,     28977, 27715, 11466, 34569,   /*[GB+9730DE30, GB+9730DE39]*/
  0,     0,     34137, 0,     0,     0,     0,     22287, 30276, 0,       /*[GB+9730DF30, GB+9730DF39]*/
  41148, 0,     8426,  27802, 23321, 38350, 0,     0,     0,     0,       /*[GB+9730E030, GB+9730E039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9730E130, GB+9730E139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9730E230, GB+9730E239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9730E330, GB+9730E339]*/
  0,     25950, 26107, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9730E430, GB+9730E439]*/
  0,     0,     0,     0,     0,     12635, 29379, 5502,  23466, 22123,   /*[GB+9730E530, GB+9730E539]*/
  25330, 3887,  23558, 15778, 0,     0,     3198,  19066, 1864,  16658,   /*[GB+9730E630, GB+9730E639]*/
  5571,  4082,  31551, 5572,  20821, 11872, 18918, 34375, 0,     2528,    /*[GB+9730E730, GB+9730E739]*/
  20136, 4398,  33320, 20782, 0,     0,     0,     43588, 0,     6902,    /*[GB+9730E830, GB+9730E839]*/
  0,     0,     29388, 8808,  15658, 42792, 0,     6318,  32555, 4418,    /*[GB+9730E930, GB+9730E939]*/
  25331, 31975, 0,     18672, 1657,  18122, 0,     0,     18123, 0,       /*[GB+9730EA30, GB+9730EA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9730EB30, GB+9730EB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9730EC30, GB+9730EC39]*/
  0,     0,     0,     0,     0,     0,     1445,  8817,  5110,  6755,    /*[GB+9730ED30, GB+9730ED39]*/
  0,     40642, 19672, 36346, 26190, 13060, 25469, 0,     13949, 28255,   /*[GB+9730EE30, GB+9730EE39]*/
  7260,  29131, 2286,  35759, 6594,  0,     26879, 0,     37134, 27435,   /*[GB+9730EF30, GB+9730EF39]*/
  19222, 13950, 6098,  8157,  6219,  43970, 0,     23180, 0,     0,       /*[GB+9730F030, GB+9730F039]*/
  0,     0,     30010, 31301, 256,   0,     0,     2597,  6522,  28468,   /*[GB+9730F130, GB+9730F139]*/
  6904,  31089, 14409, 43971, 41395, 13951, 5580,  5507,  29723, 0,       /*[GB+9730F230, GB+9730F239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9730F330, GB+9730F339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9730F430, GB+9730F439]*/
  41264, 31493, 42500, 0,     0,     39269, 23562, 0,     0,     0,       /*[GB+9730F530, GB+9730F539]*/
  0,     0,     0,     0,     0,     0,     0,     18245, 0,     0,       /*[GB+9730F630, GB+9730F639]*/
  25425, 0,     30146, 15552, 142,   9336,  15665, 32026, 1456,  25238,   /*[GB+9730F730, GB+9730F739]*/
  2135,  12655, 0,     14100, 0,     0,     9625,  30012, 0,     0,       /*[GB+9730F830, GB+9730F839]*/
  0,     0,     4700,  25635, 0,     0,     0,     36722, 0,     0,       /*[GB+9730F930, GB+9730F939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9730FA30, GB+9730FA39]*/
  0,     0,     0,     6221,  0,     0,     21789, 0,     0,     0,       /*[GB+9730FB30, GB+9730FB39]*/
  0,     0,     0,     0,     0,     0,     24595, 19445, 14700, 43055,   /*[GB+9730FC30, GB+9730FC39]*/
  9831,  0,     0,     6923,  41513, 0,     28277, 25985, 21471, 0,       /*[GB+9730FD30, GB+9730FD39]*/
  24484, 4875,  4026,  0,     0,     27569, 0,     0,     0,     0,       /*[GB+9730FE30, GB+9730FE39]*/
  0,     0,     31302, 0,     0,     0,     0,     0,     0,     0,       /*[GB+97318130, GB+97318139]*/
  0,     0,     0,     0,     0,     0,     0,     19549, 20202, 20009,   /*[GB+97318230, GB+97318239]*/
  43777, 0,     0,     0,     20299, 12161, 0,     31372, 31943, 25630,   /*[GB+97318330, GB+97318339]*/
  19231, 910,   23748, 41213, 0,     18535, 0,     0,     0,     31105,   /*[GB+97318430, GB+97318439]*/
  11763, 0,     0,     0,     0,     0,     19090, 0,     0,     0,       /*[GB+97318530, GB+97318539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+97318630, GB+97318639]*/
  0,     0,     0,     0,     0,     12506, 26454, 36087, 0,     15143,   /*[GB+97318730, GB+97318739]*/
  6432,  30060, 0,     32947, 0,     36738, 36620, 689,   0,     14201,   /*[GB+97318830, GB+97318839]*/
  22787, 18550, 11766, 25179, 42592, 0,     0,     0,     0,     0,       /*[GB+97318930, GB+97318939]*/
  0,     0,     0,     1476,  0,     0,     39602, 6433,  19236, 18565,   /*[GB+97318A30, GB+97318A39]*/
  43419, 13536, 15421, 0,     8281,  0,     39949, 0,     0,     0,       /*[GB+97318B30, GB+97318B39]*/
  0,     0,     0,     0,     0,     0,     19100, 40539, 25433, 14538,   /*[GB+97318C30, GB+97318C39]*/
  10001, 0,     43423, 0,     0,     8839,  0,     0,     0,     0,       /*[GB+97318D30, GB+97318D39]*/
  0,     0,     0,     27837, 0,     0,     4986,  0,     0,     0,       /*[GB+97318E30, GB+97318E39]*/
  0,     0,     0,     12507, 0,     0,     0,     0,     5695,  19723,   /*[GB+97318F30, GB+97318F39]*/
  38465, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+97319030, GB+97319039]*/
  0,     0,     0,     27526, 40314, 0,     5980,  0,     0,     0,       /*[GB+97319130, GB+97319139]*/
  0,     0,     0,     9016,  40602, 28174, 39970, 2163,  0,     6498,    /*[GB+97319230, GB+97319239]*/
  7721,  0,     39715, 0,     28416, 0,     0,     0,     0,     0,       /*[GB+97319330, GB+97319339]*/
  9489,  0,     21762, 32006, 0,     0,     19027, 0,     0,     0,       /*[GB+97319430, GB+97319439]*/
  0,     0,     0,     0,     0,     40946, 23782, 0,     21423, 0,       /*[GB+97319530, GB+97319539]*/
  21919, 7828,  0,     22358, 29626, 37830, 13580, 43010, 0,     0,       /*[GB+97319630, GB+97319639]*/
  0,     0,     0,     612,   39754, 0,     24545, 0,     0,     0,       /*[GB+97319730, GB+97319739]*/
  12058, 2447,  14270, 0,     0,     0,     30578, 0,     0,     0,       /*[GB+97319830, GB+97319839]*/
  0,     0,     0,     0,     0,     42371, 0,     21512, 16333, 0,       /*[GB+97319930, GB+97319939]*/
  0,     0,     0,     37229, 7011,  26722, 2267,  4279,  25748, 19490,   /*[GB+97319A30, GB+97319A39]*/
  2060,  0,     16268, 8191,  0,     0,     0,     0,     0,     0,       /*[GB+97319B30, GB+97319B39]*/
  0,     0,     0,     5760,  0,     0,     0,     6713,  9294,  0,       /*[GB+97319C30, GB+97319C39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+97319D30, GB+97319D39]*/
  0,     0,     0,     19332, 33811, 0,     22830, 42688, 42689, 0,       /*[GB+97319E30, GB+97319E39]*/
  0,     0,     0,     4018,  0,     0,     0,     0,     0,     0,       /*[GB+97319F30, GB+97319F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9731A030, GB+9731A039]*/
  22856, 13006, 0,     26419, 0,     0,     0,     0,     0,     0,       /*[GB+9731A130, GB+9731A139]*/
  0,     0,     0,     0,     0,     0,     16138, 0,     0,     34560,   /*[GB+9731A230, GB+9731A239]*/
  0,     11131, 0,     19138, 0,     43502, 0,     0,     0,     23496,   /*[GB+9731A330, GB+9731A339]*/
  4957,  0,     0,     0,     0,     2717,  0,     0,     0,     0,       /*[GB+9731A430, GB+9731A439]*/
  0,     42638, 2198,  0,     0,     0,     0,     0,     4023,  33997,   /*[GB+9731A530, GB+9731A539]*/
  27555, 34363, 27803, 21700, 0,     7750,  42070, 0,     5100,  0,       /*[GB+9731A630, GB+9731A639]*/
  0,     0,     0,     0,     34183, 0,     0,     0,     0,     0,       /*[GB+9731A730, GB+9731A739]*/
  0,     0,     0,     0,     23426, 18124, 43832, 0,     5041,  40845,   /*[GB+9731A830, GB+9731A839]*/
  0,     18125, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9731A930, GB+9731A939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9731AA30, GB+9731AA39]*/
  24973, 0,     43306, 5111,  24647, 0,     7763,  30612, 0,     0,       /*[GB+9731AB30, GB+9731AB39]*/
  0,     0,     0,     0,     0,     0,     0,     43526, 43872, 42709,   /*[GB+9731AC30, GB+9731AC39]*/
  27180, 19155, 30613, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9731AD30, GB+9731AD39]*/
  28994, 0,     0,     0,     5058,  43532, 8438,  0,     0,     0,       /*[GB+9731AE30, GB+9731AE39]*/
  0,     34979, 0,     0,     0,     0,     0,     0,     8271,  0,       /*[GB+9731AF30, GB+9731AF39]*/
  0,     0,     0,     0,     0,     34597, 14106, 0,     8212,  0,       /*[GB+9731B030, GB+9731B039]*/
  0,     0,     0,     18536, 22142, 0,     0,     0,     0,     0,       /*[GB+9731B130, GB+9731B139]*/
  0,     39176, 0,     0,     0,     0,     0,     12903, 0,     26455,   /*[GB+9731B230, GB+9731B239]*/
  0,     0,     0,     0,     22933, 0,     0,     0,     21484, 0,       /*[GB+9731B330, GB+9731B339]*/
  0,     25182, 0,     0,     11167, 0,     18789, 0,     0,     0,       /*[GB+9731B430, GB+9731B439]*/
  0,     0,     0,     44083, 0,     0,     19426, 0,     0,     42874,   /*[GB+9731B530, GB+9731B539]*/
  0,     22937, 0,     0,     13290, 27089, 14578, 8988,  0,     0,       /*[GB+9731B630, GB+9731B639]*/
  38124, 0,     0,     42718, 0,     33436, 0,     0,     0,     0,       /*[GB+9731B730, GB+9731B739]*/
  9427,  4937,  0,     21763, 0,     0,     0,     36310, 0,     42031,   /*[GB+9731B830, GB+9731B839]*/
  34893, 0,     0,     8506,  11656, 0,     0,     0,     5890,  0,       /*[GB+9731B930, GB+9731B939]*/
  15846, 6292,  0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9731BA30, GB+9731BA39]*/
  0,     29021, 0,     0,     32513, 0,     0,     0,     0,     0,       /*[GB+9731BB30, GB+9731BB39]*/
  0,     22359, 5896,  28920, 4670,  29095, 0,     32265, 3280,  7729,    /*[GB+9731BC30, GB+9731BC39]*/
  0,     9711,  39994, 0,     0,     9977,  0,     0,     0,     0,       /*[GB+9731BD30, GB+9731BD39]*/
  0,     0,     0,     0,     25828, 0,     41232, 5371,  28302, 0,       /*[GB+9731BE30, GB+9731BE39]*/
  0,     0,     0,     0,     0,     0,     42680, 34681, 21922, 43611,   /*[GB+9731BF30, GB+9731BF39]*/
  12247, 21428, 0,     33952, 42372, 23577, 7186,  0,     0,     0,       /*[GB+9731C030, GB+9731C039]*/
  4740,  22171, 0,     3232,  0,     38217, 9609,  0,     0,     0,       /*[GB+9731C130, GB+9731C139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     15004,   /*[GB+9731C230, GB+9731C239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9731C330, GB+9731C339]*/
  0,     40903, 24414, 37572, 14632, 0,     24013, 0,     0,     27888,   /*[GB+9731C430, GB+9731C439]*/
  43197, 15187, 0,     0,     19491, 39662, 0,     15390, 0,     0,       /*[GB+9731C530, GB+9731C539]*/
  0,     0,     23415, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9731C630, GB+9731C639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     21060, 0,       /*[GB+9731C730, GB+9731C739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9731C830, GB+9731C839]*/
  3330,  0,     43138, 22923, 16606, 14523, 11999, 0,     0,     35347,   /*[GB+9731C930, GB+9731C939]*/
  44030, 36950, 15903, 0,     0,     0,     0,     11618, 42776, 28370,   /*[GB+9731CA30, GB+9731CA39]*/
  0,     36832, 0,     35214, 6398,  0,     0,     3333,  0,     0,       /*[GB+9731CB30, GB+9731CB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9731CC30, GB+9731CC39]*/
  0,     0,     0,     0,     0,     0,     13920, 13449, 0,     0,       /*[GB+9731CD30, GB+9731CD39]*/
  4608,  11415, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9731CE30, GB+9731CE39]*/
  5284,  0,     0,     0,     16139, 35441, 12601, 13925, 42783, 41045,   /*[GB+9731CF30, GB+9731CF39]*/
  8786,  3868,  1106,  23421, 0,     0,     0,     13463, 25412, 0,       /*[GB+9731D030, GB+9731D039]*/
  32455, 23478, 1394,  29471, 0,     0,     0,     0,     0,     0,       /*[GB+9731D130, GB+9731D139]*/
  0,     0,     15016, 37597, 0,     0,     0,     0,     0,     0,       /*[GB+9731D230, GB+9731D239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9731D330, GB+9731D339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9731D430, GB+9731D439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     7982,  0,       /*[GB+9731D530, GB+9731D539]*/
  0,     18113, 0,     0,     18432, 24105, 0,     0,     0,     0,       /*[GB+9731D630, GB+9731D639]*/
  0,     0,     30122, 19515, 11136, 12733, 30656, 29473, 42466, 2096,    /*[GB+9731D730, GB+9731D739]*/
  0,     0,     0,     33998, 0,     0,     0,     24527, 26847, 0,       /*[GB+9731D830, GB+9731D839]*/
  22288, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9731D930, GB+9731D939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9731DA30, GB+9731DA39]*/
  0,     0,     0,     0,     0,     39030, 0,     0,     0,     0,       /*[GB+9731DB30, GB+9731DB39]*/
  5403,  0,     0,     0,     0,     0,     0,     0,     25063, 6282,    /*[GB+9731DC30, GB+9731DC39]*/
  23069, 29380, 0,     28253, 17185, 30133, 0,     6663,  6415,  0,       /*[GB+9731DD30, GB+9731DD39]*/
  0,     0,     0,     0,     0,     0,     9531,  0,     0,     0,       /*[GB+9731DE30, GB+9731DE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9731DF30, GB+9731DF39]*/
  0,     0,     0,     0,     0,     43119, 10468, 0,     0,     0,       /*[GB+9731E030, GB+9731E039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     35760, 26523,   /*[GB+9731E130, GB+9731E139]*/
  0,     0,     36347, 0,     0,     14311, 41611, 0,     6220,  27677,   /*[GB+9731E230, GB+9731E239]*/
  1089,  0,     30285, 25470, 0,     0,     0,     35761, 0,     36348,   /*[GB+9731E330, GB+9731E339]*/
  0,     0,     22127, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9731E430, GB+9731E439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9731E530, GB+9731E539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9731E630, GB+9731E639]*/
  0,     0,     0,     0,     0,     30147, 17306, 42959, 42798, 0,       /*[GB+9731E730, GB+9731E739]*/
  18500, 0,     5471,  15499, 0,     40476, 2987,  20482, 0,     0,       /*[GB+9731E830, GB+9731E839]*/
  27820, 0,     18689, 0,     43892, 0,     0,     17422, 29384, 25862,   /*[GB+9731E930, GB+9731E939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9731EA30, GB+9731EA39]*/
  6665,  0,     0,     0,     0,     0,     33500, 21327, 0,     13519,   /*[GB+9731EB30, GB+9731EB39]*/
  0,     0,     3675,  0,     22781, 0,     0,     0,     0,     0,       /*[GB+9731EC30, GB+9731EC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9731ED30, GB+9731ED39]*/
  0,     0,     37342, 0,     0,     17318, 0,     0,     20300, 0,       /*[GB+9731EE30, GB+9731EE39]*/
  30158, 8442,  0,     0,     17941, 32836, 0,     0,     0,     0,       /*[GB+9731EF30, GB+9731EF39]*/
  0,     0,     0,     0,     4833,  0,     0,     0,     0,     0,       /*[GB+9731F030, GB+9731F039]*/
  18551, 0,     24386, 0,     16044, 15556, 0,     17324, 0,     0,       /*[GB+9731F130, GB+9731F139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     13537,   /*[GB+9731F230, GB+9731F239]*/
  0,     0,     21801, 0,     37466, 1518,  0,     0,     0,     4885,    /*[GB+9731F330, GB+9731F339]*/
  29577, 0,     0,     29578, 0,     0,     11168, 0,     0,     0,       /*[GB+9731F430, GB+9731F439]*/
  0,     0,     0,     0,     0,     0,     0,     22540, 0,     6442,    /*[GB+9731F530, GB+9731F539]*/
  0,     0,     3269,  40433, 0,     35644, 29087, 0,     2945,  21827,   /*[GB+9731F630, GB+9731F639]*/
  18614, 38507, 35423, 32582, 9784,  31670, 30325, 7972,  5221,  23036,   /*[GB+9731F730, GB+9731F739]*/
  4950,  44046, 14655, 36125, 2097,  14826, 38666, 35755, 0,     39450,   /*[GB+9731F830, GB+9731F839]*/
  35566, 0,     25100, 888,   35477, 9558,  0,     0,     43929, 0,       /*[GB+9731F930, GB+9731F939]*/
  0,     0,     14512, 0,     16131, 0,     0,     35487, 901,   29110,   /*[GB+9731FA30, GB+9731FA39]*/
  39825, 0,     42735, 41246, 0,     0,     0,     0,     0,     0,       /*[GB+9731FB30, GB+9731FB39]*/
  42577, 0,     0,     0,     2718,  0,     4484,  19601, 0,     0,       /*[GB+9731FC30, GB+9731FC39]*/
  0,     0,     30039, 0,     42957, 0,     0,     0,     0,     0,       /*[GB+9731FD30, GB+9731FD39]*/
  0,     0,     0,     1607,  0,     0,     0,     35406, 0,     36555,   /*[GB+9731FE30, GB+9731FE39]*/
  0,     0,     19390, 0,     0,     0,     0,     36558, 0,     0,       /*[GB+97328130, GB+97328139]*/
  0,     0,     31439, 0,     0,     0,     0,     0,     0,     0,       /*[GB+97328230, GB+97328239]*/
  0,     0,     0,     0,     0,     5922,  0,     0,     0,     0,       /*[GB+97328330, GB+97328339]*/
  0,     0,     0,     24059, 11827, 2605,  0,     0,     0,     0,       /*[GB+97328430, GB+97328439]*/
  0,     0,     31192, 22195, 15713, 0,     22196, 0,     15596, 19568,   /*[GB+97328530, GB+97328539]*/
  0,     19439, 0,     0,     0,     19569, 11838, 0,     0,     0,       /*[GB+97328630, GB+97328639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+97328730, GB+97328739]*/
  8509,  11657, 0,     0,     7388,  9502,  9712,  0,     0,     0,       /*[GB+97328830, GB+97328839]*/
  0,     9929,  0,     0,     0,     0,     0,     0,     0,     20370,   /*[GB+97328930, GB+97328939]*/
  0,     7730,  20266, 21263, 8551,  19483, 0,     0,     0,     0,       /*[GB+97328A30, GB+97328A39]*/
  0,     35064, 9723,  0,     34542, 2061,  0,     13427, 0,     0,       /*[GB+97328B30, GB+97328B39]*/
  0,     0,     0,     0,     0,     26620, 20070, 0,     0,     0,       /*[GB+97328C30, GB+97328C39]*/
  0,     0,     0,     0,     0,     0,     29467, 0,     0,     22198,   /*[GB+97328D30, GB+97328D39]*/
  39827, 10792, 0,     0,     0,     0,     0,     40254, 5549,  0,       /*[GB+97328E30, GB+97328E39]*/
  0,     0,     0,     39537, 13794, 0,     0,     9092,  0,     0,       /*[GB+97328F30, GB+97328F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     2526,    /*[GB+97329030, GB+97329039]*/
  28557, 19581, 0,     0,     0,     0,     14297, 0,     0,     29948,   /*[GB+97329130, GB+97329139]*/
  29949, 42641, 0,     0,     0,     0,     0,     0,     18985, 0,       /*[GB+97329230, GB+97329239]*/
  36081, 20138, 0,     13492, 0,     0,     36709, 0,     0,     0,       /*[GB+97329330, GB+97329339]*/
  0,     20151, 0,     21900, 31090, 0,     0,     0,     12220, 0,       /*[GB+97329430, GB+97329439]*/
  0,     38385, 30660, 0,     33400, 0,     0,     0,     13070, 0,       /*[GB+97329530, GB+97329539]*/
  0,     0,     0,     0,     0,     6821,  17162, 0,     16043, 0,       /*[GB+97329630, GB+97329639]*/
  0,     0,     0,     28737, 22509, 24598, 0,     0,     29137, 26739,   /*[GB+97329730, GB+97329739]*/
  0,     0,     0,     0,     0,     0,     27038, 0,     0,     8115,    /*[GB+97329830, GB+97329839]*/
  0,     0,     30359, 32005, 0,     37092, 0,     0,     0,     0,       /*[GB+97329930, GB+97329939]*/
  0,     0,     0,     0,     0,     0,     18337, 0,     0,     0,       /*[GB+97329A30, GB+97329A39]*/
  0,     0,     7127,  4633,  0,     9776,  8121,  0,     0,     27240,   /*[GB+97329B30, GB+97329B39]*/
  0,     0,     0,     9119,  34661, 0,     34662, 11535, 0,     30835,   /*[GB+97329C30, GB+97329C39]*/
  28112, 37110, 43612, 0,     40063, 0,     0,     0,     0,     0,       /*[GB+97329D30, GB+97329D39]*/
  0,     0,     29855, 0,     0,     4392,  34666, 0,     618,   0,       /*[GB+97329E30, GB+97329E39]*/
  20814, 0,     8673,  17266, 18227, 0,     14633, 39155, 0,     0,       /*[GB+97329F30, GB+97329F39]*/
  0,     0,     39828, 0,     3855,  0,     4393,  38282, 28032, 0,       /*[GB+9732A030, GB+9732A039]*/
  0,     0,     13212, 36603, 0,     0,     33476, 4551,  20907, 0,       /*[GB+9732A130, GB+9732A139]*/
  0,     0,     0,     8596,  13602, 9684,  9743,  13658, 0,     0,       /*[GB+9732A230, GB+9732A239]*/
  0,     34364, 0,     14961, 43622, 19821, 0,     0,     9686,  0,       /*[GB+9732A330, GB+9732A339]*/
  0,     0,     27004, 0,     28043, 0,     0,     15949, 31629, 39334,   /*[GB+9732A430, GB+9732A439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9732A530, GB+9732A539]*/
  0,     0,     13825, 6816,  0,     11755, 0,     0,     0,     27723,   /*[GB+9732A630, GB+9732A639]*/
  39928, 0,     0,     0,     0,     0,     20749, 0,     42078, 36617,   /*[GB+9732A730, GB+9732A739]*/
  9145,  0,     12051, 0,     0,     0,     0,     0,     26707, 0,       /*[GB+9732A830, GB+9732A839]*/
  0,     37181, 0,     43927, 0,     0,     0,     34769, 38126, 36774,   /*[GB+9732A930, GB+9732A939]*/
  11368, 39451, 0,     0,     0,     3817,  0,     3692,  0,     0,       /*[GB+9732AA30, GB+9732AA39]*/
  10859, 27763, 23520, 30362, 8083,  3629,  0,     0,     11379, 36216,   /*[GB+9732AB30, GB+9732AB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     42209, 0,       /*[GB+9732AC30, GB+9732AC39]*/
  0,     0,     0,     26479, 0,     0,     0,     0,     33592, 8530,    /*[GB+9732AD30, GB+9732AD39]*/
  0,     29102, 10591, 0,     24572, 0,     4715,  0,     0,     0,       /*[GB+9732AE30, GB+9732AE39]*/
  0,     0,     0,     0,     2796,  0,     0,     11304, 0,     0,       /*[GB+9732AF30, GB+9732AF39]*/
  0,     0,     23750, 28444, 34812, 0,     0,     35990, 4717,  31483,   /*[GB+9732B030, GB+9732B039]*/
  24578, 0,     0,     0,     0,     0,     0,     30395, 2670,  0,       /*[GB+9732B130, GB+9732B139]*/
  0,     43822, 12848, 35915, 36951, 2798,  0,     32136, 0,     0,       /*[GB+9732B230, GB+9732B239]*/
  0,     0,     0,     0,     34355, 36003, 11513, 12157, 9313,  3984,    /*[GB+9732B330, GB+9732B339]*/
  1396,  11315, 11316, 3724,  0,     2719,  0,     0,     0,     0,       /*[GB+9732B430, GB+9732B439]*/
  0,     0,     0,     0,     0,     0,     0,     41046, 0,     41124,   /*[GB+9732B530, GB+9732B539]*/
  30414, 0,     0,     24585, 30415, 0,     39258, 8607,  4642,  5285,    /*[GB+9732B630, GB+9732B639]*/
  30344, 0,     19516, 0,     0,     4643,  0,     11327, 0,     10818,   /*[GB+9732B730, GB+9732B739]*/
  17255, 0,     28726, 0,     0,     0,     0,     31630, 12648, 11146,   /*[GB+9732B830, GB+9732B839]*/
  24337, 39907, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9732B930, GB+9732B939]*/
  0,     0,     34270, 10724, 0,     24590, 0,     0,     0,     0,       /*[GB+9732BA30, GB+9732BA39]*/
  0,     0,     17319, 12904, 0,     0,     0,     0,     0,     0,       /*[GB+9732BB30, GB+9732BB39]*/
  0,     0,     30529, 0,     0,     0,     17752, 0,     29219, 0,       /*[GB+9732BC30, GB+9732BC39]*/
  0,     0,     5747,  6151,  0,     0,     0,     0,     0,     0,       /*[GB+9732BD30, GB+9732BD39]*/
  0,     0,     0,     0,     0,     0,     35580, 0,     0,     32558,   /*[GB+9732BE30, GB+9732BE39]*/
  0,     3264,  22424, 0,     15806, 0,     0,     7554,  7558,  23340,   /*[GB+9732BF30, GB+9732BF39]*/
  23355, 0,     0,     0,     0,     0,     0,     43118, 0,     0,       /*[GB+9732C030, GB+9732C039]*/
  0,     0,     23433, 0,     18999, 17898, 0,     435,   0,     0,       /*[GB+9732C130, GB+9732C139]*/
  0,     0,     3462,  0,     0,     10164, 26665, 9365,  0,     9559,    /*[GB+9732C230, GB+9732C239]*/
  28295, 0,     0,     0,     0,     17372, 41916, 38283, 10364, 32594,   /*[GB+9732C330, GB+9732C339]*/
  20443, 0,     0,     0,     28798, 38785, 28626, 31803, 1397,  0,       /*[GB+9732C430, GB+9732C439]*/
  40825, 42639, 0,     27224, 24950, 8801,  2595,  0,     0,     0,       /*[GB+9732C530, GB+9732C539]*/
  0,     18126, 20784, 0,     18673, 1658,  0,     0,     0,     4431,    /*[GB+9732C630, GB+9732C639]*/
  38386, 0,     0,     0,     23584, 0,     0,     0,     0,     38408,   /*[GB+9732C730, GB+9732C739]*/
  16830, 0,     0,     1677,  0,     3902,  26552, 22153, 41532, 27921,   /*[GB+9732C830, GB+9732C839]*/
  27922, 0,     0,     0,     0,     0,     0,     37226, 9491,  22428,   /*[GB+9732C930, GB+9732C939]*/
  0,     0,     35167, 0,     0,     31308, 0,     0,     0,     0,       /*[GB+9732CA30, GB+9732CA39]*/
  0,     0,     37488, 0,     0,     27870, 0,     0,     0,     0,       /*[GB+9732CB30, GB+9732CB39]*/
  0,     27739, 0,     0,     6111,  35867, 0,     0,     0,     0,       /*[GB+9732CC30, GB+9732CC39]*/
  0,     26480, 0,     0,     0,     0,     0,     0,     11305, 28642,   /*[GB+9732CD30, GB+9732CD39]*/
  31537, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9732CE30, GB+9732CE39]*/
  0,     0,     0,     0,     7266,  0,     0,     0,     0,     18884,   /*[GB+9732CF30, GB+9732CF39]*/
  0,     11457, 18228, 0,     34686, 0,     0,     28548, 0,     0,       /*[GB+9732D030, GB+9732D039]*/
  8409,  0,     0,     0,     0,     0,     0,     0,     36836, 32534,   /*[GB+9732D130, GB+9732D139]*/
  24781, 7195,  0,     12000, 25315, 0,     35215, 0,     0,     0,       /*[GB+9732D230, GB+9732D239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     4747,    /*[GB+9732D330, GB+9732D339]*/
  28595, 0,     0,     0,     7626,  0,     35442, 0,     31973, 36958,   /*[GB+9732D430, GB+9732D439]*/
  0,     19056, 0,     0,     21441, 0,     0,     0,     0,     0,       /*[GB+9732D530, GB+9732D539]*/
  0,     0,     0,     0,     0,     0,     43736, 0,     43503, 0,       /*[GB+9732D630, GB+9732D639]*/
  11727, 38884, 18665, 5234,  31249, 31314, 0,     0,     0,     0,       /*[GB+9732D730, GB+9732D739]*/
  0,     0,     0,     0,     0,     27164, 0,     117,   0,     0,       /*[GB+9732D830, GB+9732D839]*/
  128,   41604, 10428, 20139, 41451, 21453, 18919, 41605, 0,     0,       /*[GB+9732D930, GB+9732D939]*/
  0,     26577, 4824,  0,     0,     31457, 0,     30615, 41265, 11747,   /*[GB+9732DA30, GB+9732DA39]*/
  0,     24338, 0,     5237,  27181, 0,     0,     0,     0,     0,       /*[GB+9732DB30, GB+9732DB39]*/
  22621, 0,     0,     0,     0,     22516, 5616,  14414, 0,     0,       /*[GB+9732DC30, GB+9732DC39]*/
  0,     0,     0,     15666, 7601,  0,     0,     0,     0,     0,       /*[GB+9732DD30, GB+9732DD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     18553, 0,       /*[GB+9732DE30, GB+9732DE39]*/
  10432, 32305, 0,     0,     0,     0,     22990, 0,     14802, 0,       /*[GB+9732DF30, GB+9732DF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9732E030, GB+9732E039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     24128,   /*[GB+9732E130, GB+9732E139]*/
  3799,  0,     0,     0,     0,     0,     0,     0,     24129, 38042,   /*[GB+9732E230, GB+9732E239]*/
  6740,  32251, 36243, 0,     25076, 28096, 0,     0,     35683, 0,       /*[GB+9732E330, GB+9732E339]*/
  0,     11012, 23640, 0,     23778, 0,     35541, 0,     3751,  30534,   /*[GB+9732E430, GB+9732E439]*/
  43943, 43888, 25084, 6741,  0,     0,     0,     0,     0,     0,       /*[GB+9732E530, GB+9732E539]*/
  0,     0,     0,     0,     0,     0,     0,     29399, 0,     0,       /*[GB+9732E630, GB+9732E639]*/
  22320, 33538, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9732E730, GB+9732E739]*/
  0,     3208,  38096, 0,     0,     41879, 0,     0,     15434, 5361,    /*[GB+9732E830, GB+9732E839]*/
  0,     27609, 0,     0,     42993, 32438, 9763,  0,     27610, 674,     /*[GB+9732E930, GB+9732E939]*/
  13700, 20729, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9732EA30, GB+9732EA39]*/
  0,     0,     0,     0,     32190, 0,     25094, 0,     0,     0,       /*[GB+9732EB30, GB+9732EB39]*/
  0,     43169, 32106, 0,     0,     24439, 0,     0,     0,     0,       /*[GB+9732EC30, GB+9732EC39]*/
  0,     0,     0,     0,     0,     23868, 16873, 37821, 0,     35168,   /*[GB+9732ED30, GB+9732ED39]*/
  40183, 11237, 0,     11482, 41882, 4564,  28910, 32256, 0,     35041,   /*[GB+9732EE30, GB+9732EE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9732EF30, GB+9732EF39]*/
  0,     0,     9451,  0,     0,     0,     0,     0,     0,     0,       /*[GB+9732F030, GB+9732F039]*/
  0,     0,     0,     0,     0,     7377,  0,     0,     0,     0,       /*[GB+9732F130, GB+9732F139]*/
  0,     0,     0,     0,     0,     16716, 22353, 0,     0,     15361,   /*[GB+9732F230, GB+9732F239]*/
  36106, 26397, 3282,  20226, 11086, 28639, 22504, 0,     11087, 2697,    /*[GB+9732F330, GB+9732F339]*/
  36581, 9433,  23015, 0,     10447, 36662, 10008, 0,     0,     42340,   /*[GB+9732F430, GB+9732F439]*/
  17804, 0,     0,     0,     0,     0,     0,     6846,  25112, 26398,   /*[GB+9732F530, GB+9732F539]*/
  15859, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9732F630, GB+9732F639]*/
  0,     0,     0,     0,     0,     8243,  0,     35868, 31964, 0,       /*[GB+9732F730, GB+9732F739]*/
  10680, 16688, 0,     0,     28440, 42374, 0,     23708, 41902, 18376,   /*[GB+9732F830, GB+9732F839]*/
  32397, 0,     7796,  35391, 0,     0,     36541, 0,     0,     0,       /*[GB+9732F930, GB+9732F939]*/
  0,     15380, 28539, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9732FA30, GB+9732FA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     20922,   /*[GB+9732FB30, GB+9732FB39]*/
  0,     32032, 43351, 33614, 38704, 15002, 32011, 0,     0,     9455,    /*[GB+9732FC30, GB+9732FC39]*/
  13414, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9732FD30, GB+9732FD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9732FE30, GB+9732FE39]*/
  0,     0,     0,     43115, 0,     0,     31406, 0,     42043, 0,       /*[GB+97338130, GB+97338139]*/
  13173, 0,     0,     0,     38866, 32881, 0,     12204, 31484, 2854,    /*[GB+97338230, GB+97338239]*/
  0,     0,     17009, 25230, 680,   2668,  32178, 22492, 14517, 0,       /*[GB+97338330, GB+97338339]*/
  36592, 0,     31786, 0,     33048, 0,     4788,  0,     0,     35901,   /*[GB+97338430, GB+97338439]*/
  38708, 0,     0,     1118,  0,     0,     0,     0,     20188, 0,       /*[GB+97338530, GB+97338539]*/
  0,     0,     0,     40558, 5223,  0,     0,     0,     23037, 15539,   /*[GB+97338630, GB+97338639]*/
  11501, 0,     0,     0,     9459,  0,     0,     0,     0,     0,       /*[GB+97338730, GB+97338739]*/
  0,     0,     0,     11502, 16201, 9512,  0,     12489, 35216, 0,       /*[GB+97338830, GB+97338839]*/
  0,     33976, 10688, 21438, 0,     33108, 12750, 27713, 35396, 41580,   /*[GB+97338930, GB+97338939]*/
  12849, 0,     9517,  0,     0,     27598, 14816, 32230, 0,     0,       /*[GB+97338A30, GB+97338A39]*/
  0,     6985,  3655,  0,     0,     0,     0,     0,     0,     0,       /*[GB+97338B30, GB+97338B39]*/
  0,     0,     0,     33659, 0,     0,     0,     0,     21439, 40675,   /*[GB+97338C30, GB+97338C39]*/
  27684, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+97338D30, GB+97338D39]*/
  0,     0,     0,     0,     0,     32016, 42397, 33050, 0,     0,       /*[GB+97338E30, GB+97338E39]*/
  38599, 32034, 0,     3931,  40826, 0,     34835, 21986, 4958,  0,       /*[GB+97338F30, GB+97338F39]*/
  0,     1539,  35617, 0,     0,     37882, 0,     0,     0,     0,       /*[GB+97339030, GB+97339039]*/
  0,     0,     12354, 0,     40967, 30221, 12720, 15636, 0,     0,       /*[GB+97339130, GB+97339139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+97339230, GB+97339239]*/
  0,     4959,  0,     0,     13037, 36343, 30193, 2633,  0,     38353,   /*[GB+97339330, GB+97339339]*/
  0,     0,     0,     0,     1649,  0,     0,     0,     15936, 18433,   /*[GB+97339430, GB+97339439]*/
  0,     0,     23767, 0,     0,     0,     0,     40501, 37609, 0,       /*[GB+97339530, GB+97339539]*/
  1416,  41656, 41657, 0,     15323, 41597, 0,     0,     29267, 0,       /*[GB+97339630, GB+97339639]*/
  0,     0,     33704, 0,     0,     0,     0,     0,     0,     0,       /*[GB+97339730, GB+97339739]*/
  0,     0,     0,     0,     0,     35811, 22420, 34365, 10373, 32905,   /*[GB+97339830, GB+97339839]*/
  0,     0,     13239, 27563, 0,     0,     6593,  23388, 6899,  41925,   /*[GB+97339930, GB+97339939]*/
  16708, 0,     0,     21938, 0,     0,     0,     0,     0,     0,       /*[GB+97339A30, GB+97339A39]*/
  0,     0,     0,     0,     0,     793,   12629, 0,     0,     43678,   /*[GB+97339B30, GB+97339B39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+97339C30, GB+97339C39]*/
  0,     0,     0,     0,     27664, 0,     30951, 37905, 0,     39889,   /*[GB+97339D30, GB+97339D39]*/
  0,     0,     0,     37449, 23181, 0,     4088,  12759, 0,     7033,    /*[GB+97339E30, GB+97339E39]*/
  0,     25170, 38954, 1936,  23167, 12317, 0,     0,     0,     0,       /*[GB+97339F30, GB+97339F39]*/
  0,     0,     0,     0,     38955, 0,     0,     0,     794,   0,       /*[GB+9733A030, GB+9733A039]*/
  0,     0,     10129, 17933, 44000, 0,     0,     0,     0,     0,       /*[GB+9733A130, GB+9733A139]*/
  0,     0,     0,     2988,  0,     0,     0,     0,     0,     0,       /*[GB+9733A230, GB+9733A239]*/
  0,     0,     0,     13957, 0,     0,     0,     21328, 0,     34985,   /*[GB+9733A330, GB+9733A339]*/
  0,     0,     0,     0,     0,     35407, 0,     18160, 0,     0,       /*[GB+9733A430, GB+9733A439]*/
  0,     0,     0,     0,     0,     0,     25751, 0,     0,     0,       /*[GB+9733A530, GB+9733A539]*/
  0,     32838, 22395, 0,     23677, 0,     17879, 0,     0,     20499,   /*[GB+9733A630, GB+9733A639]*/
  0,     0,     0,     20334, 0,     9417,  0,     20335, 0,     0,       /*[GB+9733A730, GB+9733A739]*/
  26553, 0,     0,     0,     20507, 0,     22523, 0,     0,     20508,   /*[GB+9733A830, GB+9733A839]*/
  40312, 0,     0,     0,     0,     0,     29408, 0,     0,     21550,   /*[GB+9733A930, GB+9733A939]*/
  32495, 2577,  13762, 32525, 0,     0,     0,     0,     0,     13603,   /*[GB+9733AA30, GB+9733AA39]*/
  0,     0,     0,     0,     12142, 0,     0,     0,     18793, 43592,   /*[GB+9733AB30, GB+9733AB39]*/
  553,   29017, 0,     0,     32368, 0,     0,     0,     29088, 0,       /*[GB+9733AC30, GB+9733AC39]*/
  0,     0,     0,     10147, 33566, 0,     27288, 37422, 0,     0,       /*[GB+9733AD30, GB+9733AD39]*/
  9120,  0,     0,     0,     0,     0,     0,     40255, 0,     0,       /*[GB+9733AE30, GB+9733AE39]*/
  39022, 32273, 2085,  0,     0,     0,     0,     35402, 0,     0,       /*[GB+9733AF30, GB+9733AF39]*/
  24195, 2110,  0,     0,     10602, 41151, 33218, 0,     37642, 1465,    /*[GB+9733B030, GB+9733B039]*/
  10606, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9733B130, GB+9733B139]*/
  0,     0,     3818,  0,     0,     42341, 0,     22714, 0,     0,       /*[GB+9733B230, GB+9733B239]*/
  0,     33305, 57,    0,     0,     0,     0,     0,     55,    39679,   /*[GB+9733B330, GB+9733B339]*/
  3878,  0,     0,     15131, 42487, 42501, 42502, 15425, 0,     0,       /*[GB+9733B430, GB+9733B439]*/
  0,     0,     10648, 0,     0,     0,     24404, 0,     0,     37371,   /*[GB+9733B530, GB+9733B539]*/
  0,     0,     0,     0,     0,     39217, 24408, 0,     22069, 0,       /*[GB+9733B630, GB+9733B639]*/
  0,     0,     0,     0,     44071, 24573, 0,     36177, 16898, 38220,   /*[GB+9733B730, GB+9733B739]*/
  24574, 0,     0,     0,     0,     0,     0,     0,     0,     43247,   /*[GB+9733B830, GB+9733B839]*/
  0,     0,     0,     0,     0,     0,     0,     22729, 26356, 4905,    /*[GB+9733B930, GB+9733B939]*/
  0,     0,     0,     0,     0,     0,     37395, 0,     0,     0,       /*[GB+9733BA30, GB+9733BA39]*/
  0,     8153,  0,     0,     0,     34139, 0,     14159, 2675,  0,       /*[GB+9733BB30, GB+9733BB39]*/
  0,     34973, 2676,  0,     0,     0,     0,     0,     3588,  5113,    /*[GB+9733BC30, GB+9733BC39]*/
  0,     0,     0,     26191, 0,     39920, 0,     39581, 0,     0,       /*[GB+9733BD30, GB+9733BD39]*/
  0,     33073, 0,     2682,  39693, 0,     0,     0,     0,     0,       /*[GB+9733BE30, GB+9733BE39]*/
  0,     44035, 5740,  0,     0,     15787, 5741,  38838, 0,     8320,    /*[GB+9733BF30, GB+9733BF39]*/
  0,     0,     0,     33124, 0,     0,     14905, 25101, 0,     26000,   /*[GB+9733C030, GB+9733C039]*/
  0,     17203, 0,     30541, 34222, 0,     31195, 0,     17207, 0,       /*[GB+9733C130, GB+9733C139]*/
  31312, 0,     11680, 42832, 41170, 30376, 0,     31200, 30384, 30396,   /*[GB+9733C230, GB+9733C239]*/
  11941, 36837, 31426, 0,     31427, 30397, 0,     0,     0,     0,       /*[GB+9733C330, GB+9733C339]*/
  0,     0,     12044, 0,     13659, 25161, 0,     30421, 0,     0,       /*[GB+9733C430, GB+9733C439]*/
  30560, 12016, 0,     0,     43157, 11916, 0,     0,     17518, 0,       /*[GB+9733C530, GB+9733C539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     43993, 38063,   /*[GB+9733C630, GB+9733C639]*/
  8474,  33429, 0,     8475,  0,     6463,  30360, 0,     0,     0,       /*[GB+9733C730, GB+9733C739]*/
  0,     0,     0,     0,     3229,  0,     0,     27105, 26717, 0,       /*[GB+9733C830, GB+9733C839]*/
  9219,  0,     0,     0,     0,     8532,  6049,  6248,  39095, 0,       /*[GB+9733C930, GB+9733C939]*/
  13701, 0,     0,     19645, 0,     0,     39096, 24071, 0,     12785,   /*[GB+9733CA30, GB+9733CA39]*/
  26151, 0,     7733,  33306, 31589, 10869, 0,     0,     30548, 0,       /*[GB+9733CB30, GB+9733CB39]*/
  0,     0,     0,     0,     11255, 0,     0,     0,     0,     0,       /*[GB+9733CC30, GB+9733CC39]*/
  31330, 0,     0,     0,     42684, 0,     0,     590,   35217, 5709,    /*[GB+9733CD30, GB+9733CD39]*/
  0,     14817, 0,     0,     0,     0,     0,     11619, 903,   0,       /*[GB+9733CE30, GB+9733CE39]*/
  6513,  0,     0,     0,     23386, 32897, 39540, 0,     35443, 8597,    /*[GB+9733CF30, GB+9733CF39]*/
  34562, 25147, 31741, 0,     0,     13213, 0,     38980, 0,     5861,    /*[GB+9733D030, GB+9733D039]*/
  32600, 33684, 0,     23556, 0,     28607, 39326, 0,     43,    14076,   /*[GB+9733D130, GB+9733D139]*/
  0,     9622,  27500, 0,     0,     0,     9587,  0,     0,     27949,   /*[GB+9733D230, GB+9733D239]*/
  30760, 0,     0,     0,     0,     0,     0,     20140, 338,   0,       /*[GB+9733D330, GB+9733D339]*/
  25168, 35623, 0,     5329,  0,     0,     0,     7666,  32925, 0,       /*[GB+9733D430, GB+9733D439]*/
  7114,  18927, 7700,  12318, 15950, 30430, 44008, 26959, 2381,  30975,   /*[GB+9733D530, GB+9733D539]*/
  0,     0,     12017, 35762, 0,     41202, 0,     0,     0,     9005,    /*[GB+9733D630, GB+9733D639]*/
  0,     0,     23940, 2384,  35008, 5296,  0,     0,     0,     13520,   /*[GB+9733D730, GB+9733D739]*/
  0,     0,     0,     0,     18248, 0,     0,     23628, 0,     0,       /*[GB+9733D830, GB+9733D839]*/
  0,     19705, 18554, 33837, 30987, 0,     0,     19391, 0,     0,       /*[GB+9733D930, GB+9733D939]*/
  0,     0,     18256, 26887, 19427, 0,     0,     0,     10355, 0,       /*[GB+9733DA30, GB+9733DA39]*/
  0,     0,     0,     0,     24504, 22070, 23545, 0,     0,     511,     /*[GB+9733DB30, GB+9733DB39]*/
  0,     0,     0,     0,     0,     0,     0,     24931, 0,     0,       /*[GB+9733DC30, GB+9733DC39]*/
  36697, 0,     0,     0,     0,     0,     32274, 281,   0,     21962,   /*[GB+9733DD30, GB+9733DD39]*/
  21896, 36399, 21329, 0,     0,     9945,  0,     0,     14246, 0,       /*[GB+9733DE30, GB+9733DE39]*/
  0,     0,     0,     8884,  0,     0,     0,     0,     0,     2319,    /*[GB+9733DF30, GB+9733DF39]*/
  33420, 41533, 27010, 16611, 0,     0,     0,     0,     0,     0,       /*[GB+9733E030, GB+9733E039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     41461, 42557,   /*[GB+9733E130, GB+9733E139]*/
  26268, 10399, 9485,  24633, 11822, 21655, 0,     0,     31190, 0,       /*[GB+9733E230, GB+9733E239]*/
  0,     0,     0,     32508, 0,     0,     0,     0,     0,     0,       /*[GB+9733E330, GB+9733E339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9733E430, GB+9733E439]*/
  0,     0,     0,     0,     42878, 5253,  35567, 22305, 0,     0,       /*[GB+9733E530, GB+9733E539]*/
  15684, 0,     0,     22881, 0,     0,     0,     0,     0,     0,       /*[GB+9733E630, GB+9733E639]*/
  11828, 27862, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9733E730, GB+9733E739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9733E830, GB+9733E839]*/
  42021, 0,     0,     30315, 0,     29503, 0,     0,     0,     38127,   /*[GB+9733E930, GB+9733E939]*/
  0,     0,     0,     0,     0,     0,     0,     26134, 42305, 20224,   /*[GB+9733EA30, GB+9733EA39]*/
  27289, 0,     24863, 39638, 42663, 37097, 34419, 22165, 4037,  12395,   /*[GB+9733EB30, GB+9733EB39]*/
  15224, 15296, 24300, 0,     0,     0,     0,     14443, 0,     9937,    /*[GB+9733EC30, GB+9733EC39]*/
  0,     22059, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9733ED30, GB+9733ED39]*/
  0,     0,     33567, 24660, 0,     0,     0,     0,     0,     0,       /*[GB+9733EE30, GB+9733EE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9733EF30, GB+9733EF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9733F030, GB+9733F039]*/
  0,     0,     0,     13383, 0,     0,     0,     0,     0,     0,       /*[GB+9733F130, GB+9733F139]*/
  33042, 956,   37968, 8534,  29510, 15693, 13404, 0,     22813, 0,       /*[GB+9733F230, GB+9733F239]*/
  1041,  0,     0,     0,     0,     35869, 0,     0,     25970, 2042,    /*[GB+9733F330, GB+9733F339]*/
  0,     0,     32494, 41650, 0,     33181, 28265, 40758, 37160, 0,       /*[GB+9733F430, GB+9733F439]*/
  0,     29734, 23016, 0,     0,     0,     0,     1300,  0,     0,       /*[GB+9733F530, GB+9733F539]*/
  0,     0,     0,     0,     0,     4928,  25542, 0,     0,     0,       /*[GB+9733F630, GB+9733F639]*/
  0,     0,     34951, 42037, 28319, 0,     0,     42342, 23875, 0,       /*[GB+9733F730, GB+9733F739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9733F830, GB+9733F839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9733F930, GB+9733F939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     6298,    /*[GB+9733FA30, GB+9733FA39]*/
  0,     0,     0,     0,     0,     0,     33190, 0,     42210, 1301,    /*[GB+9733FB30, GB+9733FB39]*/
  0,     8122,  0,     0,     0,     0,     0,     22360, 0,     0,       /*[GB+9733FC30, GB+9733FC39]*/
  0,     702,   0,     0,     0,     0,     26319, 22763, 2050,  7098,    /*[GB+9733FD30, GB+9733FD39]*/
  0,     28837, 35713, 3086,  35714, 37493, 21429, 0,     0,     0,       /*[GB+9733FE30, GB+9733FE39]*/
  35341, 43352, 8746,  0,     0,     24139, 36222, 0,     39783, 34336,   /*[GB+97348130, GB+97348139]*/
  39784, 39491, 0,     15381, 30319, 16581, 0,     0,     41903, 21264,   /*[GB+97348230, GB+97348239]*/
  9086,  0,     0,     0,     0,     0,     22267, 33859, 1320,  0,       /*[GB+97348330, GB+97348339]*/
  8747,  0,     0,     0,     0,     0,     34959, 0,     0,     0,       /*[GB+97348430, GB+97348439]*/
  15233, 0,     0,     9643,  0,     0,     0,     0,     0,     0,       /*[GB+97348530, GB+97348539]*/
  0,     0,     0,     0,     0,     0,     3706,  15234, 15235, 0,       /*[GB+97348630, GB+97348639]*/
  0,     27883, 0,     7927,  0,     0,     0,     0,     0,     0,       /*[GB+97348730, GB+97348739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+97348830, GB+97348839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+97348930, GB+97348939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+97348A30, GB+97348A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+97348B30, GB+97348B39]*/
  0,     0,     0,     0,     0,     8552,  0,     0,     0,     32448,   /*[GB+97348C30, GB+97348C39]*/
  35715, 1968,  0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+97348D30, GB+97348D39]*/
  0,     0,     0,     0,     0,     0,     0,     10974, 0,     35065,   /*[GB+97348E30, GB+97348E39]*/
  24239, 39664, 0,     31236, 24089, 0,     28839, 16786, 12991, 0,       /*[GB+97348F30, GB+97348F39]*/
  0,     40954, 22439, 22197, 0,     8566,  6889,  0,     0,     28622,   /*[GB+97349030, GB+97349039]*/
  2706,  0,     3714,  10251, 21840, 30694, 587,   25554, 0,     43359,   /*[GB+97349130, GB+97349139]*/
  32882, 36323, 0,     21730, 40559, 39239, 0,     39376, 0,     28023,   /*[GB+97349230, GB+97349239]*/
  11111, 20233, 42120, 0,     0,     0,     15116, 41712, 0,     0,       /*[GB+97349330, GB+97349339]*/
  21271, 39240, 0,     33194, 487,   0,     0,     15762, 27300, 4901,    /*[GB+97349430, GB+97349439]*/
  42056, 0,     0,     0,     0,     0,     0,     26162, 0,     11691,   /*[GB+97349530, GB+97349539]*/
  37054, 0,     16079, 28325, 16645, 0,     0,     25400, 0,     0,       /*[GB+97349630, GB+97349639]*/
  21272, 0,     0,     0,     0,     0,     31351, 0,     0,     0,       /*[GB+97349730, GB+97349739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+97349830, GB+97349839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+97349930, GB+97349939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+97349A30, GB+97349A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+97349B30, GB+97349B39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+97349C30, GB+97349C39]*/
  23038, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+97349D30, GB+97349D39]*/
  0,     0,     0,     0,     0,     3715,  0,     35553, 0,     12751,   /*[GB+97349E30, GB+97349E39]*/
  0,     0,     5943,  17639, 0,     40706, 2588,  30265, 43873, 30033,   /*[GB+97349F30, GB+97349F39]*/
  0,     34126, 0,     35218, 0,     0,     14648, 39247, 0,     9678,    /*[GB+9734A030, GB+9734A039]*/
  9440,  0,     0,     0,     16083, 12353, 13198, 24603, 33258, 0,       /*[GB+9734A130, GB+9734A139]*/
  38285, 0,     0,     22419, 0,     27217, 22471, 0,     24467, 28123,   /*[GB+9734A230, GB+9734A239]*/
  6258,  13199, 11907, 7015,  16601, 30399, 0,     11459, 0,     29115,   /*[GB+9734A330, GB+9734A339]*/
  0,     0,     0,     11268, 8175,  0,     0,     0,     0,     0,       /*[GB+9734A430, GB+9734A439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9734A530, GB+9734A539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     16649, 0,       /*[GB+9734A630, GB+9734A639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9734A730, GB+9734A739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9734A830, GB+9734A839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9734A930, GB+9734A939]*/
  0,     0,     0,     0,     0,     0,     0,     22868, 0,     0,       /*[GB+9734AA30, GB+9734AA39]*/
  0,     4179,  0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9734AB30, GB+9734AB39]*/
  25634, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9734AC30, GB+9734AC39]*/
  0,     0,     33977, 11696, 29740, 33199, 0,     10597, 27303, 21276,   /*[GB+9734AD30, GB+9734AD39]*/
  0,     27634, 0,     33309, 0,     39253, 0,     30556, 28457, 0,       /*[GB+9734AE30, GB+9734AE39]*/
  0,     0,     25055, 0,     10463, 25233, 34252, 0,     0,     0,       /*[GB+9734AF30, GB+9734AF39]*/
  0,     30036, 20103, 0,     0,     24791, 6260,  0,     0,     24943,   /*[GB+9734B030, GB+9734B039]*/
  13654, 39254, 33817, 3194,  0,     0,     0,     0,     0,     21582,   /*[GB+9734B130, GB+9734B139]*/
  19875, 1210,  0,     0,     0,     0,     0,     0,     23807, 0,       /*[GB+9734B230, GB+9734B239]*/
  0,     0,     41729, 28458, 2970,  8787,  0,     0,     0,     0,       /*[GB+9734B330, GB+9734B339]*/
  26429, 29545, 0,     0,     43919, 0,     0,     0,     0,     0,       /*[GB+9734B430, GB+9734B439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9734B530, GB+9734B539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9734B630, GB+9734B639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9734B730, GB+9734B739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9734B830, GB+9734B839]*/
  0,     0,     0,     41501, 0,     0,     0,     0,     0,     0,       /*[GB+9734B930, GB+9734B939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9734BA30, GB+9734BA39]*/
  0,     24098, 0,     0,     4906,  0,     0,     0,     0,     0,       /*[GB+9734BB30, GB+9734BB39]*/
  9137,  6793,  0,     8429,  9986,  27736, 29816, 13931, 24614, 0,       /*[GB+9734BC30, GB+9734BC39]*/
  2409,  1124,  22200, 41448, 34911, 0,     15484, 29123, 13932, 18118,   /*[GB+9734BD30, GB+9734BD39]*/
  0,     4396,  17797, 0,     11200, 11728, 9734,  41393, 14666, 34740,   /*[GB+9734BE30, GB+9734BE39]*/
  0,     6751,  10695, 13660, 34741, 13226, 29178, 17685, 13038, 28039,   /*[GB+9734BF30, GB+9734BF39]*/
  0,     0,     30859, 43510, 29332, 0,     13227, 4297,  13475, 28462,   /*[GB+9734C030, GB+9734C039]*/
  19218, 18912, 0,     27639, 0,     215,   27405, 0,     31074, 5562,    /*[GB+9734C130, GB+9734C139]*/
  0,     12122, 0,     27556, 0,     0,     0,     0,     0,     0,       /*[GB+9734C230, GB+9734C239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     38661,   /*[GB+9734C330, GB+9734C339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9734C430, GB+9734C439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9734C530, GB+9734C539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9734C630, GB+9734C639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9734C730, GB+9734C739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     27427, 33262,   /*[GB+9734C830, GB+9734C839]*/
  15648, 12123, 18011, 20126, 0,     0,     30558, 40915, 0,     0,       /*[GB+9734C930, GB+9734C939]*/
  0,     25417, 0,     0,     0,     33705, 0,     0,     0,     0,       /*[GB+9734CA30, GB+9734CA39]*/
  0,     0,     0,     0,     0,     4720,  28330, 0,     0,     0,       /*[GB+9734CB30, GB+9734CB39]*/
  0,     27557, 0,     0,     21077, 26516, 0,     2111,  9003,  0,       /*[GB+9734CC30, GB+9734CC39]*/
  25852, 15780, 5610,  3102,  37130, 11143, 0,     13051, 0,     1905,    /*[GB+9734CD30, GB+9734CD39]*/
  0,     21137, 0,     0,     0,     7069,  367,   15021, 20141, 21703,   /*[GB+9734CE30, GB+9734CE39]*/
  6216,  11204, 0,     13813, 0,     0,     11953, 9623,  0,     0,       /*[GB+9734CF30, GB+9734CF39]*/
  20142, 8618,  12499, 0,     40890, 0,     14962, 29554, 26832, 0,       /*[GB+9734D030, GB+9734D039]*/
  0,     0,     0,     0,     5573,  0,     32461, 1543,  0,     0,       /*[GB+9734D130, GB+9734D139]*/
  0,     0,     13663, 0,     0,     0,     0,     0,     0,     5504,    /*[GB+9734D230, GB+9734D239]*/
  15243, 34366, 2112,  0,     0,     0,     0,     0,     0,     0,       /*[GB+9734D330, GB+9734D339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9734D430, GB+9734D439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9734D530, GB+9734D539]*/
  0,     0,     0,     0,     0,     34145, 0,     1659,  0,     0,       /*[GB+9734D630, GB+9734D639]*/
  0,     24588, 0,     0,     27737, 1001,  27958, 32817, 0,     2484,    /*[GB+9734D730, GB+9734D739]*/
  20143, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9734D830, GB+9734D839]*/
  0,     5574,  0,     9828,  40502, 726,   9041,  0,     0,     3175,    /*[GB+9734D930, GB+9734D939]*/
  13501, 18990, 0,     26192, 14094, 20152, 6921,  29999, 37335, 38668,   /*[GB+9734DA30, GB+9734DA39]*/
  0,     30431, 37336, 15025, 4196,  8623,  40114, 28276, 0,     1937,    /*[GB+9734DB30, GB+9734DB39]*/
  5691,  28991, 0,     0,     19927, 25856, 20479, 24591, 0,     21315,   /*[GB+9734DC30, GB+9734DC39]*/
  14693, 0,     0,     13062, 0,     0,     20153, 0,     0,     0,       /*[GB+9734DD30, GB+9734DD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9734DE30, GB+9734DE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9734DF30, GB+9734DF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9734E030, GB+9734E039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9734E130, GB+9734E139]*/
  0,     0,     3154,  21787, 21316, 13953, 0,     0,     2347,  0,       /*[GB+9734E230, GB+9734E239]*/
  29974, 0,     0,     0,     0,     0,     0,     0,     11211, 27567,   /*[GB+9734E330, GB+9734E339]*/
  0,     43527, 16566, 9446,  29558, 0,     0,     0,     0,     0,       /*[GB+9734E430, GB+9734E439]*/
  31097, 41326, 35283, 29392, 3790,  39171, 20007, 21399, 19084, 29658,   /*[GB+9734E530, GB+9734E539]*/
  0,     0,     33742, 7765,  8439,  28523, 31098, 2801,  40283, 0,       /*[GB+9734E630, GB+9734E639]*/
  8440,  14101, 6102,  0,     0,     41661, 35110, 5473,  0,     0,       /*[GB+9734E730, GB+9734E739]*/
  0,     42148, 0,     0,     0,     35946, 0,     0,     0,     0,       /*[GB+9734E830, GB+9734E839]*/
  23085, 7850,  21323, 21901, 0,     0,     0,     0,     0,     0,       /*[GB+9734E930, GB+9734E939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9734EA30, GB+9734EA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9734EB30, GB+9734EB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9734EC30, GB+9734EC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9734ED30, GB+9734ED39]*/
  0,     0,     0,     0,     19834, 6822,  0,     41330, 14108, 10828,   /*[GB+9734EE30, GB+9734EE39]*/
  5584,  27844, 14109, 30624, 14422, 24533, 0,     24485, 0,     43158,   /*[GB+9734EF30, GB+9734EF39]*/
  0,     18937, 43623, 0,     43412, 0,     11881, 0,     0,     0,       /*[GB+9734F030, GB+9734F039]*/
  0,     34028, 0,     21330, 15668, 21793, 34391, 0,     17109, 21331,   /*[GB+9734F130, GB+9734F139]*/
  13839, 0,     0,     0,     23289, 0,     6103,  27725, 0,     0,       /*[GB+9734F230, GB+9734F239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9734F330, GB+9734F339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9734F430, GB+9734F439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9734F530, GB+9734F539]*/
  0,     0,     0,     0,     0,     4307,  42154, 25430, 20250, 0,       /*[GB+9734F630, GB+9734F639]*/
  43414, 0,     0,     0,     44081, 10706, 30069, 9690,  0,     4124,    /*[GB+9734F730, GB+9734F739]*/
  13525, 38435, 39594, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9734F830, GB+9734F839]*/
  23290, 22785, 27504, 0,     0,     0,     19168, 0,     0,     23172,   /*[GB+9734F930, GB+9734F939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9734FA30, GB+9734FA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9734FB30, GB+9734FB39]*/
  0,     0,     0,     0,     0,     0,     0,     25738, 37349, 8836,    /*[GB+9734FC30, GB+9734FC39]*/
  29355, 17112, 26544, 6597,  29195, 0,     0,     0,     0,     0,       /*[GB+9734FD30, GB+9734FD39]*/
  18176, 15513, 27439, 0,     0,     0,     41027, 0,     0,     0,       /*[GB+9734FE30, GB+9734FE39]*/
  34877, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+97358130, GB+97358139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+97358230, GB+97358239]*/
  21233, 28809, 6140,  18566, 5878,  40661, 0,     20500, 2421,  0,       /*[GB+97358330, GB+97358339]*/
  0,     306,   0,     0,     13983, 0,     6325,  0,     0,     38917,   /*[GB+97358430, GB+97358439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+97358530, GB+97358539]*/
  38456, 5641,  23293, 0,     40300, 0,     0,     0,     0,     0,       /*[GB+97358630, GB+97358639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+97358730, GB+97358739]*/
  0,     0,     0,     0,     18255, 18193, 11893, 0,     39388, 0,       /*[GB+97358830, GB+97358839]*/
  0,     0,     0,     0,     0,     0,     22538, 0,     0,     0,       /*[GB+97358930, GB+97358939]*/
  0,     0,     0,     0,     3449,  0,     0,     0,     0,     0,       /*[GB+97358A30, GB+97358A39]*/
  0,     8286,  1745,  0,     0,     0,     0,     20826, 9052,  12911,   /*[GB+97358B30, GB+97358B39]*/
  0,     4924,  0,     39336, 0,     0,     39389, 0,     0,     0,       /*[GB+97358C30, GB+97358C39]*/
  35692, 0,     0,     0,     0,     35858, 11775, 19915, 0,     34102,   /*[GB+97358D30, GB+97358D39]*/
  38182, 0,     0,     0,     0,     0,     0,     0,     11779, 11842,   /*[GB+97358E30, GB+97358E39]*/
  43574, 0,     0,     0,     0,     0,     0,     9507,  31900, 33249,   /*[GB+97358F30, GB+97358F39]*/
  19319, 9724,  42572, 0,     0,     0,     0,     0,     0,     0,       /*[GB+97359030, GB+97359039]*/
  19921, 0,     0,     0,     43874, 0,     0,     0,     31604, 34062,   /*[GB+97359130, GB+97359139]*/
  11125, 0,     0,     0,     0,     40265, 0,     0,     0,     0,       /*[GB+97359230, GB+97359239]*/
  0,     22473, 2086,  11626, 25162, 29151, 25567, 42701, 37442, 9735,    /*[GB+97359330, GB+97359339]*/
  0,     0,     0,     628,   16232, 0,     0,     0,     0,     0,       /*[GB+97359430, GB+97359439]*/
  0,     0,     11138, 13477, 11729, 36710, 0,     0,     0,     0,       /*[GB+97359530, GB+97359539]*/
  4519,  6488,  26234, 11519, 0,     0,     10997, 0,     0,     43876,   /*[GB+97359630, GB+97359639]*/
  0,     34380, 0,     0,     34189, 34024, 0,     0,     27821, 0,       /*[GB+97359730, GB+97359739]*/
  0,     0,     9341,  34029, 0,     0,     0,     9343,  0,     0,       /*[GB+97359830, GB+97359839]*/
  20251, 0,     9344,  16499, 0,     29060, 43064, 0,     30992, 36506,   /*[GB+97359930, GB+97359939]*/
  22797, 0,     0,     0,     0,     0,     0,     0,     20550, 30016,   /*[GB+97359A30, GB+97359A39]*/
  0,     0,     2934,  0,     0,     33437, 0,     39401, 0,     0,       /*[GB+97359B30, GB+97359B39]*/
  0,     0,     0,     0,     0,     0,     0,     40728, 0,     38157,   /*[GB+97359C30, GB+97359C39]*/
  33569, 11381, 26644, 32514, 3763,  33940, 31266, 40397, 0,     37689,   /*[GB+97359D30, GB+97359D39]*/
  13140, 12396, 8349,  0,     8658,  0,     0,     13141, 36040, 0,       /*[GB+97359E30, GB+97359E39]*/
  29341, 30517, 0,     6874,  0,     0,     0,     40051, 0,     0,       /*[GB+97359F30, GB+97359F39]*/
  0,     6036,  0,     0,     42939, 30465, 0,     27211, 36173, 39756,   /*[GB+9735A030, GB+9735A039]*/
  7830,  914,   0,     26571, 0,     2258,  13582, 40994, 0,     23331,   /*[GB+9735A130, GB+9735A139]*/
  28321, 0,     0,     0,     18362, 0,     0,     0,     0,     0,       /*[GB+9735A230, GB+9735A239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9735A330, GB+9735A339]*/
  0,     0,     10240, 9949,  0,     5374,  0,     0,     0,     0,       /*[GB+9735A430, GB+9735A439]*/
  0,     0,     0,     8875,  0,     0,     4742,  0,     36807, 29679,   /*[GB+9735A530, GB+9735A539]*/
  28934, 0,     16737, 42217, 31590, 26482, 7410,  0,     35484, 27299,   /*[GB+9735A630, GB+9735A639]*/
  39493, 0,     0,     38222, 38223, 35887, 8361,  9285,  0,     0,       /*[GB+9735A730, GB+9735A739]*/
  0,     0,     11098, 38508, 0,     11454, 0,     7188,  0,     0,       /*[GB+9735A830, GB+9735A839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     35485,   /*[GB+9735A930, GB+9735A939]*/
  7734,  0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9735AA30, GB+9735AA39]*/
  0,     24769, 27890, 0,     6890,  6070,  18073, 38260, 33805, 9439,    /*[GB+9735AB30, GB+9735AB39]*/
  31595, 16340, 11458, 17193, 23039, 3716,  31050, 0,     31596, 0,       /*[GB+9735AC30, GB+9735AC39]*/
  0,     0,     0,     0,     18074, 22381, 0,     0,     0,     0,       /*[GB+9735AD30, GB+9735AD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9735AE30, GB+9735AE39]*/
  0,     41177, 915,   0,     0,     0,     0,     0,     0,     0,       /*[GB+9735AF30, GB+9735AF39]*/
  0,     0,     0,     0,     0,     0,     0,     31354, 0,     40497,   /*[GB+9735B030, GB+9735B039]*/
  7439,  29049, 16588, 23914, 8772,  20090, 35219, 35220, 35613, 20091,   /*[GB+9735B130, GB+9735B139]*/
  31288, 30400, 4749,  26422, 0,     8773,  42061, 0,     35221, 43783,   /*[GB+9735B230, GB+9735B239]*/
  8029,  21847, 35435, 0,     0,     31605, 24907, 0,     24274, 0,       /*[GB+9735B330, GB+9735B339]*/
  30103, 7595,  0,     0,     0,     11786, 0,     0,     0,     0,       /*[GB+9735B430, GB+9735B439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9735B530, GB+9735B539]*/
  0,     0,     0,     0,     0,     29685, 0,     1726,  6471,  657,     /*[GB+9735B630, GB+9735B639]*/
  28708, 35222, 34908, 7976,  0,     0,     0,     0,     0,     0,       /*[GB+9735B730, GB+9735B739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     4382,  29313,   /*[GB+9735B830, GB+9735B839]*/
  37242, 15919, 28627, 0,     39160, 7845,  0,     0,     17019, 6090,    /*[GB+9735B930, GB+9735B939]*/
  0,     20980, 14657, 0,     36699, 0,     0,     32602, 0,     0,       /*[GB+9735BA30, GB+9735BA39]*/
  27675, 14658, 39543, 32899, 7454,  26681, 15195, 8307,  19663, 6405,    /*[GB+9735BB30, GB+9735BB39]*/
  24245, 0,     18770, 11710, 20280, 0,     0,     0,     0,     0,       /*[GB+9735BC30, GB+9735BC39]*/
  0,     0,     0,     6206,  0,     0,     0,     5097,  22180, 0,       /*[GB+9735BD30, GB+9735BD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9735BE30, GB+9735BE39]*/
  0,     0,     0,     0,     0,     32496, 15700, 16402, 36700, 14393,   /*[GB+9735BF30, GB+9735BF39]*/
  0,     0,     0,     0,     0,     0,     0,     9526,  0,     27167,   /*[GB+9735C030, GB+9735C039]*/
  30177, 28141, 0,     0,     3729,  0,     24953, 30277, 0,     37610,   /*[GB+9735C130, GB+9735C139]*/
  40736, 26747, 41257, 462,   43827, 30278, 41598, 34000, 0,     4222,    /*[GB+9735C230, GB+9735C239]*/
  14403, 43965, 9245,  33481, 0,     0,     20384, 13229, 0,     0,       /*[GB+9735C330, GB+9735C339]*/
  35086, 35087, 0,     0,     6685,  0,     0,     0,     0,     26230,   /*[GB+9735C430, GB+9735C439]*/
  0,     0,     0,     0,     0,     24475, 0,     19518, 0,     0,       /*[GB+9735C530, GB+9735C539]*/
  24375, 4644,  0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9735C630, GB+9735C639]*/
  0,     0,     0,     0,     0,     14667, 0,     12620, 0,     0,       /*[GB+9735C730, GB+9735C739]*/
  0,     27955, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9735C830, GB+9735C839]*/
  0,     0,     0,     0,     42142, 129,   35940, 1432,  0,     37479,   /*[GB+9735C930, GB+9735C939]*/
  22845, 42905, 0,     29966, 0,     35097, 39890, 26438, 0,     0,       /*[GB+9735CA30, GB+9735CA39]*/
  43833, 42072, 41371, 20386, 32818, 3940,  21595, 0,     13052, 0,       /*[GB+9735CB30, GB+9735CB39]*/
  29967, 37520, 33825, 31876, 18676, 36058, 0,     33485, 0,     0,       /*[GB+9735CC30, GB+9735CC39]*/
  0,     0,     0,     20985, 0,     0,     0,     0,     0,     0,       /*[GB+9735CD30, GB+9735CD39]*/
  0,     0,     0,     0,     0,     0,     11428, 0,     0,     20387,   /*[GB+9735CE30, GB+9735CE39]*/
  6809,  4872,  2874,  20145, 29968, 0,     0,     0,     0,     0,       /*[GB+9735CF30, GB+9735CF39]*/
  0,     0,     0,     0,     0,     0,     20388, 0,     0,     0,       /*[GB+9735D030, GB+9735D039]*/
  0,     0,     42584, 18139, 8047,  0,     15069, 17102, 38390, 11958,   /*[GB+9735D130, GB+9735D139]*/
  5114,  42908, 39908, 1110,  31992, 29188, 5409,  33267, 4757,  23193,   /*[GB+9735D230, GB+9735D239]*/
  5692,  43840, 10703, 0,     24113, 4400,  0,     0,     6905,  23537,   /*[GB+9735D330, GB+9735D339]*/
  0,     0,     42857, 0,     35108, 41612, 25857, 36015, 0,     27753,   /*[GB+9735D430, GB+9735D439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9735D530, GB+9735D539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9735D630, GB+9735D639]*/
  1447,  0,     8820,  0,     0,     0,     0,     0,     18477, 0,       /*[GB+9735D730, GB+9735D739]*/
  0,     0,     0,     0,     7320,  0,     37482, 29192, 0,     27822,   /*[GB+9735D830, GB+9735D839]*/
  0,     24811, 10526, 27823, 0,     8825,  20986, 0,     17865, 41205,   /*[GB+9735D930, GB+9735D939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9735DA30, GB+9735DA39]*/
  2802,  0,     0,     0,     0,     32930, 0,     0,     0,     0,       /*[GB+9735DB30, GB+9735DB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9735DC30, GB+9735DC39]*/
  0,     17866, 0,     40893, 38806, 143,   35010, 0,     1458,  0,       /*[GB+9735DD30, GB+9735DD39]*/
  0,     0,     0,     0,     0,     2898,  0,     0,     24117, 4876,    /*[GB+9735DE30, GB+9735DE39]*/
  19087, 24994, 0,     13522, 15034, 6578,  7959,  37922, 31883, 0,       /*[GB+9735DF30, GB+9735DF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9735E030, GB+9735E039]*/
  0,     0,     0,     0,     21382, 2417,  36475, 0,     0,     0,       /*[GB+9735E130, GB+9735E139]*/
  0,     0,     0,     0,     37180, 0,     31499, 0,     0,     0,       /*[GB+9735E230, GB+9735E239]*/
  42155, 0,     35296, 20301, 0,     18538, 0,     20990, 0,     0,       /*[GB+9735E330, GB+9735E339]*/
  34157, 0,     0,     29830, 0,     0,     0,     0,     0,     0,       /*[GB+9735E430, GB+9735E439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9735E530, GB+9735E539]*/
  29851, 36238, 0,     0,     0,     0,     0,     18556, 40125, 0,       /*[GB+9735E630, GB+9735E639]*/
  0,     41279, 0,     30455, 36064, 32948, 37651, 23699, 20991, 31107,   /*[GB+9735E730, GB+9735E739]*/
  24387, 0,     6908,  26351, 38007, 0,     0,     0,     0,     32159,   /*[GB+9735E830, GB+9735E839]*/
  2804,  0,     0,     14973, 0,     0,     0,     0,     7330,  0,       /*[GB+9735E930, GB+9735E939]*/
  0,     0,     0,     0,     0,     2911,  38833, 19392, 0,     34469,   /*[GB+9735EA30, GB+9735EA39]*/
  0,     24819, 0,     40298, 20391, 38918, 26548, 0,     0,     0,       /*[GB+9735EB30, GB+9735EB39]*/
  7985,  43541, 0,     0,     0,     0,     0,     26044, 0,     0,       /*[GB+9735EC30, GB+9735EC39]*/
  0,     23176, 32957, 29361, 0,     8840,  0,     0,     22151, 22152,   /*[GB+9735ED30, GB+9735ED39]*/
  32163, 2806,  40702, 0,     0,     0,     0,     0,     18189, 18194,   /*[GB+9735EE30, GB+9735EE39]*/
  0,     34161, 15728, 0,     22480, 21610, 0,     0,     12335, 0,       /*[GB+9735EF30, GB+9735EF39]*/
  29362, 0,     35817, 0,     0,     7986,  0,     35308, 0,     0,       /*[GB+9735F030, GB+9735F039]*/
  0,     0,     30459, 39125, 0,     41281, 0,     0,     0,     0,       /*[GB+9735F130, GB+9735F139]*/
  0,     0,     0,     25613, 22524, 0,     19241, 0,     0,     0,       /*[GB+9735F230, GB+9735F239]*/
  20350, 34477, 8637,  0,     19429, 0,     0,     0,     27731, 0,       /*[GB+9735F330, GB+9735F339]*/
  12676, 0,     0,     0,     0,     0,     31465, 0,     26051, 0,       /*[GB+9735F430, GB+9735F439]*/
  12241, 0,     231,   0,     0,     0,     0,     0,     0,     20815,   /*[GB+9735F530, GB+9735F539]*/
  0,     22826, 0,     0,     0,     10461, 23494, 0,     0,     34732,   /*[GB+9735F630, GB+9735F639]*/
  0,     14822, 0,     0,     0,     0,     41922, 0,     22301, 0,       /*[GB+9735F730, GB+9735F739]*/
  0,     27665, 11873, 0,     43841, 12501, 0,     0,     13063, 0,       /*[GB+9735F830, GB+9735F839]*/
  0,     0,     37645, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9735F930, GB+9735F939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     35478, 0,       /*[GB+9735FA30, GB+9735FA39]*/
  0,     17270, 39757, 15880, 0,     0,     0,     0,     0,     29239,   /*[GB+9735FB30, GB+9735FB39]*/
  41907, 0,     39111, 0,     9236,  0,     14055, 35399, 0,     0,       /*[GB+9735FC30, GB+9735FC39]*/
  12430, 0,     0,     29317, 0,     0,     3944,  10877, 0,     0,       /*[GB+9735FD30, GB+9735FD39]*/
  0,     0,     18949, 0,     0,     0,     0,     14133, 0,     9486,    /*[GB+9735FE30, GB+9735FE39]*/
  0,     43322, 0,     0,     0,     0,     0,     25028, 0,     0,       /*[GB+97368130, GB+97368139]*/
  0,     0,     25268, 0,     6675,  24500, 0,     0,     11829, 0,       /*[GB+97368230, GB+97368239]*/
  0,     23216, 0,     25096, 6238,  6239,  0,     18310, 0,     0,       /*[GB+97368330, GB+97368339]*/
  35545, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+97368430, GB+97368439]*/
  0,     0,     0,     0,     0,     22308, 0,     41883, 17271, 42307,   /*[GB+97368530, GB+97368539]*/
  0,     7334,  1958,  24502, 35696, 39134, 4668,  18340, 0,     0,       /*[GB+97368630, GB+97368639]*/
  23732, 0,     0,     35657, 0,     0,     0,     0,     0,     0,       /*[GB+97368730, GB+97368739]*/
  0,     0,     0,     0,     0,     24722, 0,     3254,  25113, 0,       /*[GB+97368830, GB+97368839]*/
  0,     38184, 0,     31024, 7179,  19648, 0,     14022, 0,     0,       /*[GB+97368930, GB+97368939]*/
  0,     0,     0,     41298, 40155, 0,     0,     0,     0,     0,       /*[GB+97368A30, GB+97368A39]*/
  0,     0,     39469, 0,     9400,  0,     11548, 0,     0,     0,       /*[GB+97368B30, GB+97368B39]*/
  25042, 0,     22268, 12195, 19769, 33956, 7298,  17661, 8017,  32398,   /*[GB+97368C30, GB+97368C39]*/
  0,     5125,  0,     23901, 0,     0,     0,     5202,  33078, 0,       /*[GB+97368D30, GB+97368D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+97368E30, GB+97368E39]*/
  0,     0,     0,     0,     0,     9157,  0,     0,     0,     5085,    /*[GB+97368F30, GB+97368F39]*/
  0,     0,     25130, 18075, 25835, 25500, 16584, 1100,  0,     29044,   /*[GB+97369030, GB+97369039]*/
  0,     0,     0,     0,     0,     0,     0,     32591, 0,     0,       /*[GB+97369130, GB+97369139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+97369230, GB+97369239]*/
  28025, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+97369330, GB+97369339]*/
  0,     43493, 0,     0,     31154, 25316, 6785,  0,     0,     31795,   /*[GB+97369430, GB+97369439]*/
  0,     0,     32596, 0,     0,     0,     11698, 35436, 0,     28033,   /*[GB+97369530, GB+97369539]*/
  42231, 0,     0,     0,     0,     3780,  0,     0,     0,     0,       /*[GB+97369630, GB+97369639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+97369730, GB+97369739]*/
  0,     0,     42784, 33087, 3725,  30274, 35233, 0,     16486, 0,       /*[GB+97369830, GB+97369839]*/
  0,     27977, 28717, 29050, 0,     14773, 0,     0,     0,     9168,    /*[GB+97369930, GB+97369939]*/
  20241, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+97369A30, GB+97369A39]*/
  35234, 0,     0,     0,     0,     0,     42233, 0,     0,     0,       /*[GB+97369B30, GB+97369B39]*/
  20779, 0,     29315, 16490, 0,     6280,  37720, 0,     29950, 4416,    /*[GB+97369C30, GB+97369C39]*/
  0,     0,     0,     0,     0,     6800,  0,     0,     0,     5102,    /*[GB+97369D30, GB+97369D39]*/
  32201, 0,     0,     35249, 0,     0,     0,     0,     0,     0,       /*[GB+97369E30, GB+97369E39]*/
  0,     36054, 0,     0,     0,     0,     11205, 15942, 0,     0,       /*[GB+97369F30, GB+97369F39]*/
  31172, 8043,  0,     42238, 28983, 31932, 0,     0,     3941,  0,       /*[GB+9736A030, GB+9736A039]*/
  31173, 41322, 11275, 0,     26518, 32679, 0,     7070,  873,   0,       /*[GB+9736A130, GB+9736A139]*/
  0,     0,     18920, 17411, 0,     0,     0,     0,     0,     0,       /*[GB+9736A230, GB+9736A239]*/
  29555, 0,     0,     0,     0,     0,     1704,  35259, 1767,  0,       /*[GB+9736A330, GB+9736A339]*/
  4887,  0,     0,     0,     15661, 11147, 0,     0,     0,     17104,   /*[GB+9736A430, GB+9736A439]*/
  39040, 0,     0,     0,     0,     0,     0,     0,     29278, 0,       /*[GB+9736A530, GB+9736A539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9736A630, GB+9736A639]*/
  0,     23086, 0,     39582, 43405, 21325, 11756, 0,     16495, 19085,   /*[GB+9736A730, GB+9736A739]*/
  0,     0,     0,     0,     0,     0,     0,     14535, 35561, 0,       /*[GB+9736A830, GB+9736A839]*/
  0,     0,     0,     36871, 0,     0,     0,     0,     0,     14703,   /*[GB+9736A930, GB+9736A939]*/
  17314, 29284, 14704, 17872, 0,     0,     43847, 0,     28998, 0,       /*[GB+9736AA30, GB+9736AA39]*/
  0,     0,     0,     0,     32702, 6758,  29983, 0,     0,     0,       /*[GB+9736AB30, GB+9736AB39]*/
  0,     0,     0,     0,     0,     35120, 26929, 0,     0,     0,       /*[GB+9736AC30, GB+9736AC39]*/
  0,     0,     0,     0,     20608, 0,     0,     0,     0,     25364,   /*[GB+9736AD30, GB+9736AD39]*/
  0,     0,     0,     0,     0,     0,     22539, 19423, 0,     0,       /*[GB+9736AE30, GB+9736AE39]*/
  13547, 0,     0,     0,     21955, 0,     0,     9758,  0,     0,       /*[GB+9736AF30, GB+9736AF39]*/
  0,     0,     36041, 0,     0,     0,     35701, 0,     0,     0,       /*[GB+9736B030, GB+9736B039]*/
  0,     2076,  0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9736B130, GB+9736B139]*/
  0,     0,     32607, 0,     0,     0,     0,     0,     16714, 0,       /*[GB+9736B230, GB+9736B239]*/
  0,     0,     32367, 0,     2747,  0,     21033, 16457, 0,     0,       /*[GB+9736B330, GB+9736B339]*/
  4738,  0,     0,     0,     34525, 22072, 0,     0,     0,     0,       /*[GB+9736B430, GB+9736B439]*/
  0,     0,     12688, 3013,  28694, 41774, 7412,  21431, 0,     28695,   /*[GB+9736B530, GB+9736B539]*/
  26408, 29104, 0,     4636,  36809, 0,     0,     11610, 24771, 42399,   /*[GB+9736B630, GB+9736B639]*/
  12250, 0,     0,     0,     0,     0,     0,     0,     42222, 0,       /*[GB+9736B730, GB+9736B739]*/
  39315, 0,     0,     0,     0,     0,     9196,  37389, 24604, 0,       /*[GB+9736B830, GB+9736B839]*/
  0,     0,     0,     0,     0,     0,     0,     37869, 3614,  13451,   /*[GB+9736B930, GB+9736B939]*/
  0,     60,    0,     6630,  0,     30110, 0,     15319, 0,     0,       /*[GB+9736BA30, GB+9736BA39]*/
  0,     0,     0,     0,     25149, 18653, 36078, 0,     18958, 0,       /*[GB+9736BB30, GB+9736BB39]*/
  0,     0,     0,     40464, 36080, 4895,  24226, 0,     17020, 0,       /*[GB+9736BC30, GB+9736BC39]*/
  31075, 12158, 5464,  10316, 3341,  0,     28212, 13478, 0,     0,       /*[GB+9736BD30, GB+9736BD39]*/
  0,     0,     18774, 16203, 28654, 0,     6664,  39267, 24961, 0,       /*[GB+9736BE30, GB+9736BE39]*/
  34933, 0,     0,     0,     0,     0,     38391, 19826, 0,     0,       /*[GB+9736BF30, GB+9736BF39]*/
  4419,  0,     0,     0,     0,     17799, 9142,  0,     0,     0,       /*[GB+9736C030, GB+9736C039]*/
  40921, 0,     32698, 0,     0,     0,     13840, 0,     0,     0,       /*[GB+9736C130, GB+9736C139]*/
  0,     38824, 7779,  18557, 25353, 0,     0,     0,     37468, 0,       /*[GB+9736C230, GB+9736C239]*/
  0,     0,     16982, 32709, 0,     26822, 0,     0,     34935, 26823,   /*[GB+9736C330, GB+9736C339]*/
  0,     0,     11641, 0,     0,     0,     0,     7121,  4250,  0,       /*[GB+9736C430, GB+9736C439]*/
  0,     28426, 0,     0,     43343, 7291,  13405, 0,     0,     30574,   /*[GB+9736C530, GB+9736C539]*/
  0,     23523, 0,     0,     0,     0,     0,     0,     0,     14758,   /*[GB+9736C630, GB+9736C639]*/
  25763, 0,     0,     0,     0,     25391, 15611, 0,     0,     26320,   /*[GB+9736C730, GB+9736C739]*/
  32126, 0,     0,     0,     23413, 29633, 0,     0,     0,     37840,   /*[GB+9736C830, GB+9736C839]*/
  0,     9793,  12747, 23534, 43482, 0,     14274, 0,     0,     33874,   /*[GB+9736C930, GB+9736C939]*/
  28274, 0,     37870, 42425, 0,     0,     0,     0,     0,     0,       /*[GB+9736CA30, GB+9736CA39]*/
  0,     20446, 0,     42694, 15920, 30595, 23422, 15637, 11712, 0,       /*[GB+9736CB30, GB+9736CB39]*/
  42444, 0,     0,     0,     0,     1154,  0,     0,     0,     0,       /*[GB+9736CC30, GB+9736CC39]*/
  0,     0,     0,     0,     0,     3879,  36009, 13230, 9918,  15485,   /*[GB+9736CD30, GB+9736CD39]*/
  33205, 32147, 0,     26342, 32608, 6919,  0,     29053, 0,     41790,   /*[GB+9736CE30, GB+9736CE39]*/
  22647, 0,     0,     34007, 41607, 42490, 0,     22316, 33721, 0,       /*[GB+9736CF30, GB+9736CF39]*/
  0,     0,     13814, 0,     0,     0,     0,     0,     37326, 10474,   /*[GB+9736D030, GB+9736D039]*/
  0,     0,     6012,  0,     12877, 15026, 0,     0,     0,     0,       /*[GB+9736D130, GB+9736D139]*/
  0,     0,     15958, 40284, 0,     0,     0,     0,     0,     0,       /*[GB+9736D230, GB+9736D239]*/
  13261, 33762, 29979, 0,     13845, 0,     17164, 33219, 16952, 1685,    /*[GB+9736D330, GB+9736D339]*/
  15978, 0,     0,     1481,  0,     2914,  0,     43542, 18198, 0,       /*[GB+9736D430, GB+9736D439]*/
  0,     0,     8345,  25437, 36759, 0,     30665, 22040, 0,     0,       /*[GB+9736D530, GB+9736D539]*/
  0,     16638, 33570, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9736D630, GB+9736D639]*/
  0,     0,     11843, 11844, 0,     8538,  0,     0,     37189, 10521,   /*[GB+9736D730, GB+9736D739]*/
  0,     26996, 38698, 8244,  16064, 0,     40367, 0,     0,     36380,   /*[GB+9736D830, GB+9736D839]*/
  0,     33945, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9736D930, GB+9736D939]*/
  0,     0,     0,     0,     34056, 0,     0,     10865, 0,     0,       /*[GB+9736DA30, GB+9736DA39]*/
  0,     0,     0,     0,     0,     0,     11845, 0,     0,     11603,   /*[GB+9736DB30, GB+9736DB39]*/
  24723, 7687,  0,     0,     0,     0,     13590, 0,     0,     37843,   /*[GB+9736DC30, GB+9736DC39]*/
  0,     0,     0,     0,     0,     32070, 22312, 0,     0,     0,       /*[GB+9736DD30, GB+9736DD39]*/
  0,     0,     0,     0,     0,     0,     38575, 38705, 0,     0,       /*[GB+9736DE30, GB+9736DE39]*/
  0,     0,     0,     0,     0,     0,     0,     13430, 32553, 28949,   /*[GB+9736DF30, GB+9736DF39]*/
  6972,  7240,  0,     32133, 32227, 18397, 0,     0,     0,     26977,   /*[GB+9736E030, GB+9736E039]*/
  0,     0,     4744,  35904, 42685, 43616, 0,     0,     0,     0,       /*[GB+9736E130, GB+9736E139]*/
  0,     0,     32530, 36490, 0,     13431, 3214,  0,     0,     0,       /*[GB+9736E230, GB+9736E239]*/
  0,     0,     0,     0,     0,     13188, 0,     0,     0,     0,       /*[GB+9736E330, GB+9736E339]*/
  0,     0,     0,     0,     0,     0,     0,     28083, 31796, 0,       /*[GB+9736E430, GB+9736E439]*/
  1764,  33978, 24019, 41083, 0,     0,     12588, 0,     30218, 0,       /*[GB+9736E530, GB+9736E539]*/
  11192, 0,     40187, 0,     33260, 0,     19335, 0,     41084, 11942,   /*[GB+9736E630, GB+9736E639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9736E730, GB+9736E739]*/
  0,     2860,  0,     0,     0,     0,     0,     0,     0,     7440,    /*[GB+9736E830, GB+9736E839]*/
  0,     3390,  0,     0,     0,     0,     0,     30112, 0,     31293,   /*[GB+9736E930, GB+9736E939]*/
  0,     0,     0,     0,     0,     0,     42445, 31159, 298,   43375,   /*[GB+9736EA30, GB+9736EA39]*/
  43620, 16403, 0,     27790, 31247, 20106, 0,     0,     0,     0,       /*[GB+9736EB30, GB+9736EB39]*/
  28053, 41047, 0,     0,     0,     3933,  0,     36960, 0,     0,       /*[GB+9736EC30, GB+9736EC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9736ED30, GB+9736ED39]*/
  22239, 11627, 0,     0,     0,     0,     0,     0,     3660,  0,       /*[GB+9736EE30, GB+9736EE39]*/
  0,     29937, 0,     0,     0,     0,     0,     22615, 0,     0,       /*[GB+9736EF30, GB+9736EF39]*/
  13231, 7158,  11558, 0,     4892,  40972, 38601, 0,     34929, 0,       /*[GB+9736F030, GB+9736F039]*/
  1595,  1125,  17297, 0,     3662,  0,     25235, 0,     183,   28723,   /*[GB+9736F130, GB+9736F139]*/
  0,     0,     3880,  0,     0,     0,     0,     0,     0,     0,       /*[GB+9736F230, GB+9736F239]*/
  0,     23339, 0,     13479, 0,     0,     0,     0,     0,     0,       /*[GB+9736F330, GB+9736F339]*/
  0,     0,     0,     0,     0,     23499, 0,     34067, 0,     0,       /*[GB+9736F430, GB+9736F439]*/
  2799,  0,     45,    0,     0,     28656, 11144, 0,     13242, 3665,    /*[GB+9736F530, GB+9736F539]*/
  34009, 41926, 0,     0,     30330, 0,     20689, 0,     0,     24035,   /*[GB+9736F630, GB+9736F639]*/
  0,     0,     35756, 25468, 0,     0,     0,     0,     0,     0,       /*[GB+9736F730, GB+9736F739]*/
  0,     0,     0,     34187, 0,     0,     0,     0,     0,     0,       /*[GB+9736F830, GB+9736F839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9736F930, GB+9736F939]*/
  39891, 0,     0,     0,     0,     0,     0,     14677, 34190, 20002,   /*[GB+9736FA30, GB+9736FA39]*/
  0,     25858, 0,     2758,  0,     0,     0,     37208, 13502, 27918,   /*[GB+9736FB30, GB+9736FB39]*/
  19879, 0,     13064, 42503, 32556, 0,     1448,  186,   37728, 187,     /*[GB+9736FC30, GB+9736FC39]*/
  0,     0,     0,     0,     18140, 0,     0,     0,     0,     0,       /*[GB+9736FD30, GB+9736FD39]*/
  0,     0,     0,     0,     0,     0,     0,     34268, 0,     0,       /*[GB+9736FE30, GB+9736FE39]*/
  0,     0,     15333, 30661, 0,     11001, 11215, 0,     0,     40974,   /*[GB+97378130, GB+97378139]*/
  41270, 38411, 34025, 0,     34026, 7681,  36723, 0,     0,     22240,   /*[GB+97378230, GB+97378239]*/
  0,     0,     0,     0,     0,     9741,  0,     0,     0,     0,       /*[GB+97378330, GB+97378339]*/
  0,     0,     0,     0,     40386, 0,     0,     0,     0,     0,       /*[GB+97378430, GB+97378439]*/
  42646, 0,     2990,  4435,  12662, 40534, 10392, 13968, 36476, 0,       /*[GB+97378530, GB+97378539]*/
  0,     0,     34748, 11216, 0,     0,     0,     0,     0,     0,       /*[GB+97378630, GB+97378639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+97378730, GB+97378739]*/
  0,     5587,  0,     0,     0,     21643, 23173, 0,     21338, 0,       /*[GB+97378830, GB+97378839]*/
  19091, 42801, 24650, 30467, 0,     0,     0,     38911, 30981, 10479,   /*[GB+97378930, GB+97378939]*/
  0,     0,     0,     0,     25611, 0,     20302, 27574, 19092, 3741,    /*[GB+97378A30, GB+97378A39]*/
  35138, 0,     0,     10482, 879,   20602, 15785, 0,     0,     0,       /*[GB+97378B30, GB+97378B39]*/
  7498,  0,     0,     0,     0,     0,     0,     0,     0,     11164,   /*[GB+97378C30, GB+97378C39]*/
  0,     21208, 0,     0,     35302, 24388, 0,     10638, 3446,  28170,   /*[GB+97378D30, GB+97378D39]*/
  12673, 0,     0,     0,     0,     0,     0,     0,     38835, 0,       /*[GB+97378E30, GB+97378E39]*/
  13987, 0,     0,     0,     0,     193,   0,     0,     0,     30456,   /*[GB+97378F30, GB+97378F39]*/
  37740, 31945, 0,     0,     0,     19560, 0,     44038, 0,     18199,   /*[GB+97379030, GB+97379039]*/
  0,     24392, 36241, 0,     22685, 6737,  0,     0,     17576, 0,       /*[GB+97379130, GB+97379139]*/
  0,     0,     0,     0,     41156, 15819, 41691, 13144, 25280, 0,       /*[GB+97379230, GB+97379239]*/
  10621, 8124,  0,     10622, 11399, 11113, 11503, 0,     0,     0,       /*[GB+97379330, GB+97379339]*/
  41313, 4069,  30526, 25513, 0,     28375, 38789, 0,     0,     25895,   /*[GB+97379430, GB+97379439]*/
  0,     0,     12870, 0,     34010, 11207, 34016, 34070, 10898, 17800,   /*[GB+97379530, GB+97379539]*/
  0,     18502, 0,     0,     3495,  0,     0,     0,     16107, 33780,   /*[GB+97379630, GB+97379639]*/
  10729, 0,     0,     0,     0,     0,     6704,  0,     6648,  32315,   /*[GB+97379730, GB+97379739]*/
  0,     0,     6649,  40583, 15763, 0,     19845, 4286,  0,     41917,   /*[GB+97379830, GB+97379839]*/
  0,     0,     0,     24945, 16251, 12855, 17376, 0,     0,     0,       /*[GB+97379930, GB+97379939]*/
  0,     0,     0,     0,     8882,  0,     29056, 0,     6571,  0,       /*[GB+97379A30, GB+97379A39]*/
  0,     0,     0,     19828, 31847, 0,     0,     39573, 33486, 0,       /*[GB+97379B30, GB+97379B39]*/
  31256, 0,     0,     0,     0,     0,     0,     0,     10477, 0,       /*[GB+97379C30, GB+97379C39]*/
  0,     0,     0,     0,     30627, 25004, 0,     0,     0,     0,       /*[GB+97379D30, GB+97379D39]*/
  0,     0,     19101, 18585, 0,     36278, 0,     0,     0,     0,       /*[GB+97379E30, GB+97379E39]*/
  9448,  31444, 10755, 4258,  0,     32007, 0,     35557, 39226, 22073,   /*[GB+97379F30, GB+97379F39]*/
  3769,  0,     11780, 6747,  0,     22231, 0,     22367, 0,     19307,   /*[GB+9737A030, GB+9737A039]*/
  0,     0,     0,     94,    34235, 0,     0,     0,     16225, 29636,   /*[GB+9737A130, GB+9737A139]*/
  27615, 0,     0,     0,     38264, 0,     0,     0,     0,     38290,   /*[GB+9737A230, GB+9737A239]*/
  34966, 0,     42062, 6717,  0,     0,     0,     0,     20741, 0,       /*[GB+9737A330, GB+9737A339]*/
  0,     0,     20431, 15922, 0,     0,     0,     25564, 19340, 0,       /*[GB+9737A430, GB+9737A439]*/
  0,     0,     0,     0,     0,     24249, 4907,  7257,  39680, 0,       /*[GB+9737A530, GB+9737A539]*/
  0,     0,     43303, 0,     34011, 12046, 21940, 30765, 0,     0,       /*[GB+9737A630, GB+9737A639]*/
  0,     24644, 0,     21602, 20787, 0,     9944,  0,     25341, 0,       /*[GB+9737A730, GB+9737A739]*/
  19161, 21902, 32833, 2638,  0,     11883, 30058, 0,     0,     15669,   /*[GB+9737A830, GB+9737A839]*/
  27960, 21339, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9737A930, GB+9737A939]*/
  0,     32707, 33777, 19455, 0,     0,     1482,  32960, 0,     0,       /*[GB+9737AA30, GB+9737AA39]*/
  0,     18311, 41555, 0,     39061, 11668, 32355, 454,   13750, 0,       /*[GB+9737AB30, GB+9737AB39]*/
  0,     0,     0,     44044, 41236, 1970,  26220, 37053, 0,     0,       /*[GB+9737AC30, GB+9737AC39]*/
  6560,  26483, 0,     0,     0,     0,     24084, 41132, 0,     0,       /*[GB+9737AD30, GB+9737AD39]*/
  28026, 9671,  1710,  37859, 8568,  0,     35394, 28950, 31545, 0,       /*[GB+9737AE30, GB+9737AE39]*/
  0,     6469,  0,     0,     31736, 34825, 33310, 23803, 12422, 0,       /*[GB+9737AF30, GB+9737AF39]*/
  0,     0,     0,     17373, 0,     0,     0,     29648, 5049,  27951,   /*[GB+9737B030, GB+9737B039]*/
  42736, 41502, 32671, 17377, 1399,  0,     0,     6591,  0,     0,       /*[GB+9737B130, GB+9737B139]*/
  22616, 35006, 26967, 31898, 32908, 39259, 21133, 40106, 42643, 0,       /*[GB+9737B230, GB+9737B239]*/
  0,     0,     0,     0,     0,     0,     29783, 0,     9745,  28298,   /*[GB+9737B330, GB+9737B339]*/
  0,     41465, 24110, 0,     0,     31552, 0,     3668,  23933, 2882,    /*[GB+9737B430, GB+9737B439]*/
  39041, 29383, 0,     33398, 0,     24114, 12222, 0,     8630,  0,       /*[GB+9737B530, GB+9737B539]*/
  1713,  0,     0,     0,     0,     22578, 0,     1714,  32942, 39047,   /*[GB+9737B630, GB+9737B639]*/
  0,     22624, 9966,  0,     0,     0,     0,     0,     18568, 0,       /*[GB+9737B730, GB+9737B739]*/
  35411, 36746, 24559, 23790, 0,     0,     0,     0,     30244, 0,       /*[GB+9737B830, GB+9737B839]*/
  27058, 28069, 0,     0,     0,     0,     30245, 0,     0,     0,       /*[GB+9737B930, GB+9737B939]*/
  0,     0,     5437,  0,     21242, 0,     0,     0,     0,     0,       /*[GB+9737BA30, GB+9737BA39]*/
  0,     0,     0,     29023, 0,     0,     0,     0,     9660,  0,       /*[GB+9737BB30, GB+9737BB39]*/
  10623, 28938, 37165, 0,     9787,  0,     0,     0,     0,     0,       /*[GB+9737BC30, GB+9737BC39]*/
  0,     0,     0,     0,     0,     0,     0,     16377, 0,     0,       /*[GB+9737BD30, GB+9737BD39]*/
  15382, 0,     23580, 2579,  774,   35552, 2324,  0,     0,     19182,   /*[GB+9737BE30, GB+9737BE39]*/
  95,    0,     0,     0,     21568, 17391, 0,     0,     0,     34813,   /*[GB+9737BF30, GB+9737BF39]*/
  0,     28326, 0,     0,     0,     0,     0,     0,     0,     12787,   /*[GB+9737C030, GB+9737C039]*/
  22827, 0,     0,     0,     0,     0,     0,     0,     41972, 0,       /*[GB+9737C130, GB+9737C139]*/
  19212, 41006, 0,     0,     40378, 0,     0,     35917, 0,     0,       /*[GB+9737C230, GB+9737C239]*/
  0,     0,     0,     0,     3568,  33160, 33664, 0,     0,     986,     /*[GB+9737C330, GB+9737C339]*/
  0,     0,     28192, 0,     0,     0,     39850, 21584, 0,     0,       /*[GB+9737C430, GB+9737C439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9737C530, GB+9737C539]*/
  0,     0,     0,     0,     6927,  0,     0,     2754,  7847,  40838,   /*[GB+9737C630, GB+9737C639]*/
  3616,  0,     31491, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9737C730, GB+9737C739]*/
  0,     0,     0,     37904, 0,     0,     0,     0,     0,     37246,   /*[GB+9737C830, GB+9737C839]*/
  0,     0,     16709, 0,     8155,  35099, 3617,  0,     0,     0,       /*[GB+9737C930, GB+9737C939]*/
  0,     0,     0,     0,     0,     17098, 0,     0,     0,     1823,    /*[GB+9737CA30, GB+9737CA39]*/
  0,     19830, 0,     0,     0,     38393, 21603, 35268, 0,     0,       /*[GB+9737CB30, GB+9737CB39]*/
  0,     0,     0,     0,     0,     0,     0,     10194, 0,     20480,   /*[GB+9737CC30, GB+9737CC39]*/
  0,     0,     28154, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9737CD30, GB+9737CD39]*/
  15500, 6757,  0,     0,     34271, 0,     0,     42243, 0,     0,       /*[GB+9737CE30, GB+9737CE39]*/
  1846,  0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9737CF30, GB+9737CF39]*/
  0,     0,     0,     42244, 43159, 36190, 18784, 30152, 0,     0,       /*[GB+9737D030, GB+9737D039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9737D130, GB+9737D139]*/
  17599, 15677, 21644, 36400, 41335, 10394, 0,     0,     0,     0,       /*[GB+9737D230, GB+9737D239]*/
  0,     0,     42531, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9737D330, GB+9737D339]*/
  0,     0,     0,     32954, 26727, 2805,  0,     0,     0,     0,       /*[GB+9737D430, GB+9737D439]*/
  0,     0,     0,     0,     27095, 0,     40301, 0,     43543, 0,       /*[GB+9737D530, GB+9737D539]*/
  0,     0,     20611, 10274, 20715, 3540,  41414, 15383, 31281, 32456,   /*[GB+9737D630, GB+9737D639]*/
  0,     0,     0,     42038, 0,     0,     22489, 0,     0,     0,       /*[GB+9737D730, GB+9737D739]*/
  0,     10630, 0,     0,     0,     0,     33685, 19140, 0,     0,       /*[GB+9737D830, GB+9737D839]*/
  36606, 35936, 0,     0,     0,     0,     11874, 9040,  35941, 0,       /*[GB+9737D930, GB+9737D939]*/
  34017, 0,     0,     0,     5116,  0,     0,     34033, 11764, 0,       /*[GB+9737DA30, GB+9737DA39]*/
  0,     0,     0,     0,     0,     36621, 0,     0,     0,     0,       /*[GB+9737DB30, GB+9737DB39]*/
  0,     0,     0,     0,     38131, 3699,  15848, 0,     43807, 0,       /*[GB+9737DC30, GB+9737DC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     14444, 38190,   /*[GB+9737DD30, GB+9737DD39]*/
  0,     30641, 3827,  481,   31701, 43635, 26306, 0,     0,     3828,    /*[GB+9737DE30, GB+9737DE39]*/
  33853, 0,     0,     0,     0,     0,     0,     0,     22814, 0,       /*[GB+9737DF30, GB+9737DF39]*/
  4920,  0,     33367, 0,     4273,  29834, 39065, 15536, 483,   0,       /*[GB+9737E030, GB+9737E039]*/
  4716,  6062,  23632, 3838,  0,     26353, 0,     36675, 0,     0,       /*[GB+9737E130, GB+9737E139]*/
  0,     0,     0,     0,     0,     0,     0,     41133, 0,     0,       /*[GB+9737E230, GB+9737E239]*/
  0,     0,     34546, 1119,  16880, 25132, 42686, 15465, 25836, 22088,   /*[GB+9737E330, GB+9737E339]*/
  40084, 0,     10297, 43915, 0,     0,     0,     0,     0,     0,       /*[GB+9737E430, GB+9737E439]*/
  0,     0,     0,     25837, 0,     0,     0,     0,     0,     0,       /*[GB+9737E530, GB+9737E539]*/
  0,     0,     11701, 26334, 10550, 8778,  17683, 27947, 33665, 2194,    /*[GB+9737E630, GB+9737E639]*/
  28960, 39113, 10099, 0,     23045, 0,     0,     11537, 0,     0,       /*[GB+9737E730, GB+9737E739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     21576, 0,       /*[GB+9737E830, GB+9737E839]*/
  0,     0,     7455,  34358, 40413, 35928, 25979, 6567,  18009, 25150,   /*[GB+9737E930, GB+9737E939]*/
  17465, 8599,  25056, 3985,  0,     0,     4960,  0,     0,     22116,   /*[GB+9737EA30, GB+9737EA39]*/
  905,   0,     0,     0,     5770,  0,     0,     0,     14668, 4001,    /*[GB+9737EB30, GB+9737EB39]*/
  3881,  39872, 5038,  29951, 31166, 29269, 41010, 8609,  0,     3303,    /*[GB+9737EC30, GB+9737EC39]*/
  8344,  11517, 41011, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9737ED30, GB+9737ED39]*/
  0,     0,     0,     0,     0,     0,     14669, 41012, 41191, 0,       /*[GB+9737EE30, GB+9737EE39]*/
  0,     36853, 10557, 33397, 11955, 6123,  37329, 29784, 25334, 13243,   /*[GB+9737EF30, GB+9737EF39]*/
  35581, 25169, 16038, 0,     10558, 0,     0,     0,     0,     0,       /*[GB+9737F030, GB+9737F039]*/
  0,     0,     36863, 14694, 5042,  0,     31879, 41372, 24282, 20154,   /*[GB+9737F130, GB+9737F139]*/
  42075, 4305,  21604, 25576, 0,     18779, 0,     0,     0,     0,       /*[GB+9737F230, GB+9737F239]*/
  0,     0,     18479, 0,     0,     43684, 7322,  29977, 15960, 0,       /*[GB+9737F330, GB+9737F339]*/
  0,     15501, 30622, 18935, 27620, 6636,  0,     40533, 42149, 0,       /*[GB+9737F430, GB+9737F439]*/
  0,     0,     36139, 19372, 0,     0,     5842,  0,     0,     41210,   /*[GB+9737F530, GB+9737F539]*/
  36025, 25986, 14480, 24206, 0,     0,     10725, 0,     0,     0,       /*[GB+9737F630, GB+9737F639]*/
  0,     0,     39595, 10480, 0,     36352, 0,     0,     36353, 2907,    /*[GB+9737F730, GB+9737F739]*/
  14706, 15514, 36882, 6829,  0,     11473, 34602, 36354, 0,     0,       /*[GB+9737F830, GB+9737F839]*/
  0,     0,     19393, 14709, 38462, 26557, 8980,  8283,  0,     15998,   /*[GB+9737F930, GB+9737F939]*/
  26459, 0,     0,     0,     0,     15290, 0,     13369, 13294, 0,       /*[GB+9737FA30, GB+9737FA39]*/
  0,     0,     0,     0,     0,     33791, 23663, 0,     16926, 16730,   /*[GB+9737FB30, GB+9737FB39]*/
  0,     16731, 41072, 0,     0,     510,   0,     0,     3409,  11846,   /*[GB+9737FC30, GB+9737FC39]*/
  23335, 7293,  35549, 7080,  28587, 32357, 8087,  43636, 0,     33595,   /*[GB+9737FD30, GB+9737FD39]*/
  43457, 0,     11847, 0,     0,     0,     0,     1758,  0,     32195,   /*[GB+9737FE30, GB+9737FE39]*/
  652,   9275,  0,     16372, 0,     0,     0,     0,     0,     0,       /*[GB+97388130, GB+97388139]*/
  43237, 8555,  22077, 13644, 32128, 39788, 0,     22232, 15883, 15457,   /*[GB+97388230, GB+97388239]*/
  9886,  24547, 0,     0,     22764, 19652, 0,     0,     32257, 36810,   /*[GB+97388330, GB+97388339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+97388430, GB+97388439]*/
  0,     0,     0,     0,     4114,  16881, 10594, 0,     34670, 33641,   /*[GB+97388530, GB+97388539]*/
  0,     5135,  31546, 27494, 0,     7150,  3289,  0,     0,     0,       /*[GB+97388630, GB+97388639]*/
  17971, 43487, 0,     41947, 44045, 0,     0,     16269, 40584, 0,       /*[GB+97388730, GB+97388739]*/
  40008, 26164, 0,     0,     0,     7657,  36325, 0,     13432, 0,       /*[GB+97388830, GB+97388839]*/
  2669,  0,     0,     0,     0,     0,     0,     0,     29118, 36121,   /*[GB+97388930, GB+97388939]*/
  35073, 31004, 10796, 34244, 28034, 32038, 10874, 17181, 2611,  6656,    /*[GB+97388A30, GB+97388A39]*/
  23387, 36952, 3569,  24637, 25142, 40256, 8414,  0,     0,     0,       /*[GB+97388B30, GB+97388B39]*/
  0,     31529, 0,     0,     0,     0,     0,     0,     0,     33161,   /*[GB+97388C30, GB+97388C39]*/
  28596, 32182, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+97388D30, GB+97388D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+97388E30, GB+97388E39]*/
  30647, 0,     36494, 42448, 20110, 36844, 15403, 0,     0,     5809,    /*[GB+97388F30, GB+97388F39]*/
  0,     4293,  43869, 7456,  42135, 23918, 0,     1733,  0,     6009,    /*[GB+97389030, GB+97389039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     17378, 0,       /*[GB+97389130, GB+97389139]*/
  37598, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+97389230, GB+97389239]*/
  10924, 23615, 30683, 16282, 35752, 43790, 13232, 33482, 0,     0,       /*[GB+97389330, GB+97389339]*/
  4397,  0,     0,     24586, 36849, 32202, 0,     6794,  0,     24251,   /*[GB+97389430, GB+97389439]*/
  3882,  12757, 24419, 0,     4818,  41050, 0,     0,     0,     0,       /*[GB+97389530, GB+97389539]*/
  32322, 0,     0,     0,     0,     32293, 0,     0,     0,     7470,    /*[GB+97389630, GB+97389639]*/
  32909, 550,   0,     14161, 0,     0,     0,     0,     0,     0,       /*[GB+97389730, GB+97389739]*/
  0,     0,     0,     0,     0,     2720,  0,     4302,  16782, 30973,   /*[GB+97389830, GB+97389839]*/
  43771, 18455, 30423, 27564, 0,     0,     23617, 23618, 0,     0,       /*[GB+97389930, GB+97389939]*/
  0,     27749, 25684, 0,     0,     0,     0,     0,     0,     0,       /*[GB+97389A30, GB+97389A39]*/
  0,     43834, 0,     42491, 36854, 0,     14678, 22689, 0,     0,       /*[GB+97389B30, GB+97389B39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+97389C30, GB+97389C39]*/
  0,     0,     0,     0,     20319, 26444, 0,     5869,  18819, 2982,    /*[GB+97389D30, GB+97389D39]*/
  25859, 19880, 6421,  5043,  0,     0,     13319, 0,     0,     3200,    /*[GB+97389E30, GB+97389E39]*/
  29334, 43842, 0,     16290, 0,     0,     25578, 18141, 0,     0,       /*[GB+97389F30, GB+97389F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9738A030, GB+9738A039]*/
  0,     0,     0,     0,     0,     0,     29304, 39909, 41373, 17930,   /*[GB+9738A130, GB+9738A139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9738A230, GB+9738A239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     24531,   /*[GB+9738A330, GB+9738A339]*/
  0,     0,     0,     0,     41617, 34980, 0,     30662, 5117,  17492,   /*[GB+9738A430, GB+9738A439]*/
  0,     35764, 29281, 41054, 2415,  0,     0,     1216,  23965, 0,       /*[GB+9738A530, GB+9738A539]*/
  0,     42910, 0,     3591,  0,     0,     0,     0,     0,     25733,   /*[GB+9738A630, GB+9738A639]*/
  13073, 0,     0,     0,     0,     0,     41110, 0,     17493, 0,       /*[GB+9738A730, GB+9738A739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9738A830, GB+9738A839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     31303,   /*[GB+9738A930, GB+9738A939]*/
  27570, 14834, 43056, 7262,  0,     2760,  15783, 30780, 1134,  0,       /*[GB+9738AA30, GB+9738AA39]*/
  7772,  19088, 40894, 30153, 0,     0,     0,     0,     0,     0,       /*[GB+9738AB30, GB+9738AB39]*/
  0,     0,     0,     27826, 0,     0,     0,     0,     0,     0,       /*[GB+9738AC30, GB+9738AC39]*/
  0,     0,     0,     0,     0,     0,     0,     42525, 31941, 0,       /*[GB+9738AD30, GB+9738AD39]*/
  26042, 0,     4621,  0,     0,     0,     30449, 1940,  33056, 0,       /*[GB+9738AE30, GB+9738AE39]*/
  0,     24551, 0,     0,     0,     0,     0,     0,     0,     22009,   /*[GB+9738AF30, GB+9738AF39]*/
  2301,  0,     0,     16953, 0,     5181,  26969, 880,   17431, 0,       /*[GB+9738B030, GB+9738B039]*/
  0,     0,     0,     0,     0,     22911, 18177, 0,     0,     0,       /*[GB+9738B130, GB+9738B139]*/
  0,     0,     0,     14426, 19956, 18570, 19712, 10342, 0,     0,       /*[GB+9738B230, GB+9738B239]*/
  0,     3004,  0,     0,     0,     0,     0,     0,     34291, 0,       /*[GB+9738B330, GB+9738B339]*/
  3035,  0,     35306, 41339, 0,     0,     0,     0,     0,     29364,   /*[GB+9738B430, GB+9738B439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9738B530, GB+9738B539]*/
  0,     21527, 20343, 20612, 0,     0,     6144,  0,     0,     0,       /*[GB+9738B630, GB+9738B639]*/
  0,     0,     0,     0,     6443,  0,     32333, 40316, 20352, 0,       /*[GB+9738B730, GB+9738B739]*/
  20354, 0,     0,     0,     0,     18000, 0,     32290, 6451,  22679,   /*[GB+9738B830, GB+9738B839]*/
  28271, 35180, 24075, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9738B930, GB+9738B939]*/
  0,     0,     0,     20962, 0,     38636, 0,     2052,  0,     43024,   /*[GB+9738BA30, GB+9738BA39]*/
  0,     0,     3922,  0,     0,     0,     0,     0,     0,     0,       /*[GB+9738BB30, GB+9738BB39]*/
  22233, 32129, 0,     31646, 37497, 0,     0,     0,     0,     0,       /*[GB+9738BC30, GB+9738BC39]*/
  0,     12365, 0,     27072, 0,     37573, 0,     0,     0,     0,       /*[GB+9738BD30, GB+9738BD39]*/
  0,     32073, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9738BE30, GB+9738BE39]*/
  0,     0,     43880, 39851, 0,     0,     5,     0,     42696, 32321,   /*[GB+9738BF30, GB+9738BF39]*/
  0,     0,     0,     7199,  42644, 10530, 12306, 21701, 0,     0,       /*[GB+9738C030, GB+9738C039]*/
  0,     4517,  25849, 9527,  31492, 21134, 0,     0,     0,     30818,   /*[GB+9738C130, GB+9738C139]*/
  0,     0,     27174, 0,     0,     27175, 24962, 10475, 0,     0,       /*[GB+9738C230, GB+9738C239]*/
  0,     34745, 31254, 0,     0,     0,     0,     0,     0,     34866,   /*[GB+9738C330, GB+9738C339]*/
  0,     0,     41206, 5118,  0,     0,     0,     0,     0,     0,       /*[GB+9738C430, GB+9738C439]*/
  0,     0,     0,     0,     17509, 0,     0,     0,     28472, 0,       /*[GB+9738C530, GB+9738C539]*/
  17947, 18571, 0,     2835,  23114, 20344, 0,     31445, 12534, 0,       /*[GB+9738C630, GB+9738C639]*/
  0,     0,     0,     9499,  0,     0,     0,     0,     0,     0,       /*[GB+9738C730, GB+9738C739]*/
  0,     0,     25103, 39461, 0,     35338, 0,     0,     0,     0,       /*[GB+9738C830, GB+9738C839]*/
  40235, 455,   5377,  12962, 36108, 0,     0,     15866, 23220, 0,       /*[GB+9738C930, GB+9738C939]*/
  0,     0,     0,     0,     0,     0,     6876,  13157, 0,     0,       /*[GB+9738CA30, GB+9738CA39]*/
  0,     0,     0,     0,     23525, 9560,  1044,  0,     16549, 37192,   /*[GB+9738CB30, GB+9738CB39]*/
  19572, 22723, 41078, 38706, 22889, 23761, 0,     9508,  2263,  11101,   /*[GB+9738CC30, GB+9738CC39]*/
  27214, 10293, 0,     0,     0,     1327,  33618, 15458, 12714, 1353,    /*[GB+9738CD30, GB+9738CD39]*/
  0,     31331, 41949, 0,     11308, 38265, 4745,  0,     1928,  0,       /*[GB+9738CE30, GB+9738CE39]*/
  0,     0,     0,     9515,  0,     0,     7658,  0,     0,     37122,   /*[GB+9738CF30, GB+9738CF39]*/
  35801, 0,     0,     11309, 0,     0,     40698, 2785,  11416, 0,       /*[GB+9738D030, GB+9738D039]*/
  31242, 0,     0,     0,     42063, 35438, 35781, 0,     0,     41841,   /*[GB+9738D130, GB+9738D139]*/
  0,     0,     32410, 0,     0,     0,     16085, 0,     0,     0,       /*[GB+9738D230, GB+9738D239]*/
  0,     33200, 0,     0,     20112, 0,     41920, 27218, 3577,  8600,    /*[GB+9738D330, GB+9738D339]*/
  0,     7457,  30855, 30752, 0,     15701, 3233,  30408, 6010,  0,       /*[GB+9738D430, GB+9738D439]*/
  43674, 0,     0,     16555, 31904, 0,     16556, 0,     0,     0,       /*[GB+9738D530, GB+9738D539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9738D630, GB+9738D639]*/
  0,     27219, 40333, 11140, 43716, 4819,  26233, 0,     0,     0,       /*[GB+9738D730, GB+9738D739]*/
  22290, 7160,  35812, 16562, 34572, 0,     0,     0,     0,     0,       /*[GB+9738D830, GB+9738D839]*/
  0,     0,     0,     6795,  16233, 0,     38790, 0,     0,     0,       /*[GB+9738D930, GB+9738D939]*/
  43514, 0,     0,     0,     0,     12307, 0,     23927, 0,     18456,   /*[GB+9738DA30, GB+9738DA39]*/
  0,     2115,  9329,  15490, 16407, 0,     11737, 37331, 30767, 0,       /*[GB+9738DB30, GB+9738DB39]*/
  26185, 27225, 19582, 12638, 13054, 0,     0,     0,     0,     0,       /*[GB+9738DC30, GB+9738DC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     42492,   /*[GB+9738DD30, GB+9738DD39]*/
  0,     30772, 42241, 16292, 0,     0,     0,     0,     0,     0,       /*[GB+9738DE30, GB+9738DE39]*/
  37209, 0,     30802, 11329, 0,     0,     0,     18820, 0,     2532,    /*[GB+9738DF30, GB+9738DF39]*/
  0,     0,     0,     22422, 43720, 0,     5870,  0,     14315, 23942,   /*[GB+9738E030, GB+9738E039]*/
  0,     10106, 7572,  14103, 14476, 23389, 40743, 0,     0,     0,       /*[GB+9738E130, GB+9738E139]*/
  0,     23947, 5619,  0,     26530, 18698, 22296, 17562, 0,     0,       /*[GB+9738E230, GB+9738E239]*/
  0,     0,     8163,  0,     0,     0,     12730, 0,     0,     0,       /*[GB+9738E330, GB+9738E339]*/
  16887, 0,     38679, 29335, 14115, 0,     0,     0,     0,     0,       /*[GB+9738E430, GB+9738E439]*/
  0,     0,     20603, 0,     20178, 0,     0,     0,     0,     0,       /*[GB+9738E530, GB+9738E539]*/
  9349,  26943, 0,     0,     24389, 0,     19713, 0,     0,     0,       /*[GB+9738E630, GB+9738E639]*/
  41798, 34603, 0,     0,     19238, 18708, 28051, 2158,  18579, 0,       /*[GB+9738E730, GB+9738E739]*/
  0,     0,     0,     0,     0,     35126, 9351,  21907, 18710, 0,       /*[GB+9738E830, GB+9738E839]*/
  0,     16000, 42715, 0,     0,     0,     0,     16418, 0,     6444,    /*[GB+9738E930, GB+9738E939]*/
  0,     41629, 0,     29076, 12937, 19631, 10095, 40737, 34598, 25262,   /*[GB+9738EA30, GB+9738EA39]*/
  0,     28427, 0,     0,     38638, 14759, 0,     0,     28447, 28448,   /*[GB+9738EB30, GB+9738EB39]*/
  27519, 0,     0,     0,     0,     33979, 0,     0,     0,     0,       /*[GB+9738EC30, GB+9738EC39]*/
  0,     16745, 0,     26507, 0,     9326,  686,   0,     1434,  25335,   /*[GB+9738ED30, GB+9738ED39]*/
  0,     0,     0,     0,     0,     0,     1824,  687,   0,     44080,   /*[GB+9738EE30, GB+9738EE39]*/
  24200, 0,     12890, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9738EF30, GB+9738EF39]*/
  690,   0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9738F030, GB+9738F039]*/
  0,     0,     23261, 0,     3405,  0,     23856, 0,     0,     8638,    /*[GB+9738F130, GB+9738F139]*/
  31746, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9738F230, GB+9738F239]*/
  24130, 24552, 0,     0,     3745,  0,     0,     36204, 25078, 33423,   /*[GB+9738F330, GB+9738F339]*/
  0,     0,     42259, 6023,  4767,  39196, 0,     0,     0,     0,       /*[GB+9738F430, GB+9738F439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     4768,    /*[GB+9738F530, GB+9738F539]*/
  0,     0,     6024,  43317, 0,     43782, 4845,  0,     0,     23641,   /*[GB+9738F630, GB+9738F639]*/
  36767, 25085, 27201, 13623, 0,     42270, 26270, 40217, 0,     28880,   /*[GB+9738F730, GB+9738F739]*/
  0,     0,     0,     10903, 0,     0,     0,     0,     0,     0,       /*[GB+9738F830, GB+9738F839]*/
  0,     32058, 0,     0,     1751,  0,     16243, 0,     0,     0,       /*[GB+9738F930, GB+9738F939]*/
  4589,  0,     4711,  0,     38628, 28900, 10582, 27385, 0,     37093,   /*[GB+9738FA30, GB+9738FA39]*/
  3625,  0,     4631,  0,     0,     9492,  21418, 0,     13295, 0,       /*[GB+9738FB30, GB+9738FB39]*/
  0,     0,     0,     0,     0,     0,     9385,  40748, 0,     0,       /*[GB+9738FC30, GB+9738FC39]*/
  0,     1093,  0,     0,     0,     0,     0,     35369, 17204, 0,       /*[GB+9738FD30, GB+9738FD39]*/
  6360,  0,     0,     31470, 0,     0,     0,     0,     0,     0,       /*[GB+9738FE30, GB+9738FE39]*/
  0,     0,     0,     0,     0,     0,     29409, 0,     0,     0,       /*[GB+97398130, GB+97398139]*/
  5366,  0,     0,     0,     0,     0,     0,     0,     18218, 0,       /*[GB+97398230, GB+97398239]*/
  39091, 14262, 0,     0,     30332, 26572, 38570, 39995, 14604, 0,       /*[GB+97398330, GB+97398339]*/
  25288, 37232, 36534, 0,     0,     26030, 0,     0,     0,     0,       /*[GB+97398430, GB+97398439]*/
  0,     0,     0,     23795, 0,     0,     0,     0,     0,     0,       /*[GB+97398530, GB+97398539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+97398630, GB+97398639]*/
  0,     0,     0,     0,     0,     0,     0,     13584, 0,     31703,   /*[GB+97398730, GB+97398739]*/
  0,     30998, 0,     0,     0,     0,     0,     7100,  2951,  13592,   /*[GB+97398830, GB+97398839]*/
  3839,  13764, 29242, 0,     0,     0,     0,     0,     0,     0,       /*[GB+97398930, GB+97398939]*/
  0,     30377, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+97398A30, GB+97398A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+97398B30, GB+97398B39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+97398C30, GB+97398C39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     42047,   /*[GB+97398D30, GB+97398D39]*/
  0,     40066, 0,     0,     11605, 0,     19044, 0,     0,     0,       /*[GB+97398E30, GB+97398E39]*/
  40808, 0,     4284,  0,     0,     0,     26361, 1079,  0,     0,       /*[GB+97398F30, GB+97398F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+97399030, GB+97399039]*/
  0,     0,     0,     0,     0,     12111, 16761, 28593, 0,     14637,   /*[GB+97399130, GB+97399139]*/
  0,     32531, 0,     0,     0,     11612, 0,     0,     0,     0,       /*[GB+97399230, GB+97399239]*/
  0,     0,     40819, 40611, 37202, 0,     3781,  15314, 0,     0,       /*[GB+97399330, GB+97399339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     18895, 39410,   /*[GB+97399430, GB+97399439]*/
  0,     0,     0,     0,     1537,  0,     16901, 0,     0,     0,       /*[GB+97399530, GB+97399539]*/
  0,     0,     0,     0,     0,     0,     4131,  36471, 43369, 0,       /*[GB+97399630, GB+97399639]*/
  25844, 34553, 0,     40011, 33387, 14394, 34737, 27952, 0,     42449,   /*[GB+97399730, GB+97399739]*/
  0,     0,     4000,  0,     1500,  32604, 0,     0,     0,     0,       /*[GB+97399830, GB+97399839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+97399930, GB+97399939]*/
  0,     0,     0,     0,     0,     5772,  0,     0,     0,     0,       /*[GB+97399A30, GB+97399A39]*/
  0,     0,     0,     0,     0,     39854, 31922, 0,     0,     3234,    /*[GB+97399B30, GB+97399B39]*/
  0,     0,     0,     12622, 25466, 0,     32807, 0,     0,     0,       /*[GB+97399C30, GB+97399C39]*/
  0,     0,     0,     0,     39261, 0,     0,     0,     0,     0,       /*[GB+97399D30, GB+97399D39]*/
  0,     0,     0,     0,     0,     0,     6094,  0,     5316,  0,       /*[GB+97399E30, GB+97399E39]*/
  0,     39262, 15324, 31926, 40614, 0,     0,     12639, 0,     27959,   /*[GB+97399F30, GB+97399F39]*/
  0,     0,     12871, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9739A030, GB+9739A039]*/
  37332, 0,     0,     0,     0,     34579, 0,     0,     0,     0,       /*[GB+9739A130, GB+9739A139]*/
  34580, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9739A230, GB+9739A239]*/
  0,     0,     0,     0,     6099,  0,     15331, 0,     0,     0,       /*[GB+9739A330, GB+9739A339]*/
  12650, 0,     16348, 39270, 0,     0,     18503, 0,     0,     0,       /*[GB+9739A430, GB+9739A439]*/
  4373,  0,     3739,  0,     12891, 0,     39276, 0,     40287, 0,       /*[GB+9739A530, GB+9739A539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     30442, 40652,   /*[GB+9739A630, GB+9739A639]*/
  0,     0,     0,     23095, 43061, 0,     0,     0,     0,     0,       /*[GB+9739A730, GB+9739A739]*/
  0,     0,     0,     0,     0,     0,     0,     34471, 0,     0,       /*[GB+9739A830, GB+9739A839]*/
  0,     0,     0,     0,     0,     28745, 0,     16623, 25016, 31826,   /*[GB+9739A930, GB+9739A939]*/
  7714,  5121,  0,     0,     31996, 2644,  2315,  34706, 23815, 15042,   /*[GB+9739AA30, GB+9739AA39]*/
  24827, 28269, 23204, 22329, 0,     25903, 0,     0,     0,     0,       /*[GB+9739AB30, GB+9739AB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9739AC30, GB+9739AC39]*/
  0,     1223,  0,     0,     0,     0,     0,     1235,  766,   1558,    /*[GB+9739AD30, GB+9739AD39]*/
  43432, 22334, 32849, 10902, 0,     0,     0,     0,     0,     0,       /*[GB+9739AE30, GB+9739AE39]*/
  0,     14844, 0,     41958, 0,     0,     0,     25796, 0,     0,       /*[GB+9739AF30, GB+9739AF39]*/
  0,     0,     0,     0,     10950, 31564, 40725, 28633, 5133,  12235,   /*[GB+9739B030, GB+9739B039]*/
  27281, 40577, 22861, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9739B130, GB+9739B139]*/
  34213, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9739B230, GB+9739B239]*/
  26634, 13351, 38549, 18213, 20883, 28284, 10746, 5187,  16051, 13370,   /*[GB+9739B330, GB+9739B339]*/
  2245,  19759, 8355,  39209, 16805, 0,     0,     0,     0,     0,       /*[GB+9739B430, GB+9739B439]*/
  34712, 0,     0,     0,     0,     40357, 0,     25911, 12463, 24658,   /*[GB+9739B530, GB+9739B539]*/
  0,     0,     0,     0,     0,     18037, 23791, 29089, 15362, 37825,   /*[GB+9739B630, GB+9739B639]*/
  41884, 3959,  0,     33792, 0,     11776, 27239, 0,     0,     0,       /*[GB+9739B730, GB+9739B739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9739B830, GB+9739B839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     12180, 25760,   /*[GB+9739B930, GB+9739B939]*/
  42034, 38855, 33854, 25649, 13406, 41894, 12263, 0,     39470, 43761,   /*[GB+9739BA30, GB+9739BA39]*/
  21105, 0,     0,     0,     28616, 0,     0,     0,     0,     0,       /*[GB+9739BB30, GB+9739BB39]*/
  0,     0,     0,     0,     0,     0,     0,     26656, 0,     0,       /*[GB+9739BC30, GB+9739BC39]*/
  0,     0,     0,     0,     14144, 11092, 0,     31591, 11102, 32967,   /*[GB+9739BD30, GB+9739BD39]*/
  38228, 23685, 0,     0,     33079, 25392, 18382, 37846, 4455,  35889,   /*[GB+9739BE30, GB+9739BE39]*/
  32766, 0,     0,     0,     0,     0,     9231,  0,     39655, 0,       /*[GB+9739BF30, GB+9739BF39]*/
  0,     41144, 8964,  25303, 16270, 18077, 28449, 10044, 0,     24290,   /*[GB+9739C030, GB+9739C039]*/
  0,     8919,  18078, 0,     11783, 0,     0,     31788, 29436, 0,       /*[GB+9739C130, GB+9739C139]*/
  11855, 18079, 0,     0,     19868, 0,     0,     0,     5668,  0,       /*[GB+9739C230, GB+9739C239]*/
  44075, 28027, 0,     28372, 0,     8305,  15628, 31606, 14379, 25712,   /*[GB+9739C330, GB+9739C339]*/
  9129,  43494, 12289, 28035, 0,     0,     30587, 0,     0,     0,       /*[GB+9739C430, GB+9739C439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9739C530, GB+9739C539]*/
  0,     0,     0,     0,     0,     0,     32074, 0,     19451, 0,       /*[GB+9739C630, GB+9739C639]*/
  3570,  33119, 44033, 39855, 0,     42897, 0,     26843, 33878, 26499,   /*[GB+9739C730, GB+9739C739]*/
  0,     9314,  0,     0,     0,     0,     0,     0,     24793, 35747,   /*[GB+9739C830, GB+9739C839]*/
  0,     0,     0,     0,     8923,  26799, 4073,  21291, 0,     0,       /*[GB+9739C930, GB+9739C939]*/
  0,     0,     0,     0,     28464, 0,     0,     0,     25681, 2413,    /*[GB+9739CA30, GB+9739CA39]*/
  0,     0,     18457, 0,     32463, 17857, 35450, 17650, 0,     0,       /*[GB+9739CB30, GB+9739CB39]*/
  28985, 41927, 7757,  0,     0,     0,     0,     0,     42493, 0,       /*[GB+9739CC30, GB+9739CC39]*/
  0,     0,     0,     38606, 0,     18480, 0,     0,     0,     0,       /*[GB+9739CD30, GB+9739CD39]*/
  22000, 32826, 0,     38807, 26920, 0,     0,     0,     0,     0,       /*[GB+9739CE30, GB+9739CE39]*/
  26624, 0,     0,     0,     0,     20540, 0,     0,     0,     0,       /*[GB+9739CF30, GB+9739CF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     5367,  0,       /*[GB+9739D030, GB+9739D039]*/
  38699, 1197,  9224,  0,     33066, 36793, 21667, 0,     9100,  26307,   /*[GB+9739D130, GB+9739D139]*/
  42039, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9739D230, GB+9739D239]*/
  0,     0,     0,     9722,  31592, 0,     19311, 31038, 4676,  37847,   /*[GB+9739D330, GB+9739D339]*/
  7836,  24507, 1329,  0,     855,   0,     0,     0,     0,     0,       /*[GB+9739D430, GB+9739D439]*/
  0,     0,     0,     23878, 0,     29996, 0,     0,     29682, 32658,   /*[GB+9739D530, GB+9739D539]*/
  34907, 0,     13433, 0,     0,     0,     10976, 36821, 0,     0,       /*[GB+9739D630, GB+9739D639]*/
  0,     0,     0,     0,     0,     0,     37861, 40708, 38942, 22494,   /*[GB+9739D730, GB+9739D739]*/
  0,     0,     0,     0,     0,     40498, 0,     31289, 33666, 14461,   /*[GB+9739D830, GB+9739D839]*/
  36692, 0,     0,     0,     0,     0,     0,     0,     22650, 30650,   /*[GB+9739D930, GB+9739D939]*/
  36962, 31296, 42450, 4019,  30904, 0,     0,     0,     0,     41087,   /*[GB+9739DA30, GB+9739DA39]*/
  0,     5459,  0,     0,     0,     0,     21681, 0,     0,     33388,   /*[GB+9739DB30, GB+9739DB39]*/
  38656, 36963, 17770, 0,     0,     0,     0,     24515, 0,     12758,   /*[GB+9739DC30, GB+9739DC39]*/
  10570, 22207, 4820,  0,     0,     16746, 4024,  0,     39328, 0,       /*[GB+9739DD30, GB+9739DD39]*/
  0,     0,     43383, 0,     17097, 29780, 0,     0,     0,     0,       /*[GB+9739DE30, GB+9739DE39]*/
  0,     0,     38722, 0,     0,     43969, 27676, 19605, 10821, 32920,   /*[GB+9739DF30, GB+9739DF39]*/
  21301, 11738, 18458, 0,     21598, 0,     714,   13943, 0,     0,       /*[GB+9739E030, GB+9739E039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     26806, 0,       /*[GB+9739E130, GB+9739E139]*/
  0,     0,     21302, 0,     22208, 11443, 11877, 21459, 27981, 0,       /*[GB+9739E230, GB+9739E239]*/
  0,     28199, 3236,  0,     43401, 0,     42076, 22926, 0,     0,       /*[GB+9739E330, GB+9739E339]*/
  0,     0,     0,     13503, 0,     0,     16413, 41931, 0,     0,       /*[GB+9739E430, GB+9739E439]*/
  0,     0,     0,     5474,  18963, 41271, 9571,  17743, 11913, 4339,    /*[GB+9739E530, GB+9739E539]*/
  0,     13959, 17105, 0,     0,     0,     0,     28312, 0,     0,       /*[GB+9739E630, GB+9739E639]*/
  0,     0,     0,     0,     3425,  30663, 0,     11759, 21334, 23772,   /*[GB+9739E730, GB+9739E739]*/
  41275, 3107,  0,     9008,  0,     0,     38428, 0,     30154, 0,       /*[GB+9739E830, GB+9739E839]*/
  0,     0,     35953, 13527, 17600, 0,     0,     0,     37924, 0,       /*[GB+9739E930, GB+9739E939]*/
  0,     21607, 0,     21798, 0,     0,     0,     5002,  0,     0,       /*[GB+9739EA30, GB+9739EA39]*/
  17566, 0,     0,     36622, 0,     0,     21491, 0,     0,     0,       /*[GB+9739EB30, GB+9739EB39]*/
  39124, 2496,  13990, 0,     0,     29800, 0,     0,     0,     0,       /*[GB+9739EC30, GB+9739EC39]*/
  0,     0,     36750, 0,     0,     15750, 0,     0,     0,     7460,    /*[GB+9739ED30, GB+9739ED39]*/
  0,     0,     8203,  0,     8205,  0,     0,     0,     0,     0,       /*[GB+9739EE30, GB+9739EE39]*/
  0,     10130, 20702, 0,     18994, 0,     0,     0,     0,     3923,    /*[GB+9739EF30, GB+9739EF39]*/
  0,     18080, 0,     0,     0,     0,     0,     42138, 0,     0,       /*[GB+9739F030, GB+9739F039]*/
  35269, 0,     0,     4089,  0,     13268, 0,     0,     0,     23998,   /*[GB+9739F130, GB+9739F139]*/
  0,     0,     0,     0,     14605, 0,     0,     0,     0,     42730,   /*[GB+9739F230, GB+9739F239]*/
  0,     0,     0,     0,     0,     35550, 0,     0,     0,     16934,   /*[GB+9739F330, GB+9739F339]*/
  0,     0,     0,     0,     0,     0,     0,     15058, 0,     11850,   /*[GB+9739F430, GB+9739F439]*/
  0,     0,     25394, 0,     0,     0,     0,     2609,  0,     0,       /*[GB+9739F530, GB+9739F539]*/
  24909, 32403, 20970, 0,     6682,  0,     0,     0,     0,     0,       /*[GB+9739F630, GB+9739F639]*/
  0,     0,     0,     0,     16627, 0,     5388,  0,     22564, 0,       /*[GB+9739F730, GB+9739F739]*/
  11410, 0,     0,     0,     0,     0,     0,     0,     17392, 7197,    /*[GB+9739F830, GB+9739F839]*/
  25317, 0,     38591, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9739F930, GB+9739F939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     19872, 12423,   /*[GB+9739FA30, GB+9739FA39]*/
  0,     0,     0,     8779,  20973, 0,     42690, 0,     0,     39066,   /*[GB+9739FB30, GB+9739FB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9739FC30, GB+9739FC39]*/
  0,     17727, 13217, 38316, 19506, 4814,  0,     22499, 0,     0,       /*[GB+9739FD30, GB+9739FD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9739FE30, GB+9739FE39]*/
  6897,  0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98308130, GB+98308139]*/
  0,     0,     0,     0,     0,     0,     31987, 34738, 0,     0,       /*[GB+98308230, GB+98308239]*/
  0,     1594,  4428,  33479, 0,     15405, 0,     0,     0,     0,       /*[GB+98308330, GB+98308339]*/
  0,     0,     0,     0,     0,     0,     0,     35005, 24030, 43384,   /*[GB+98308430, GB+98308439]*/
  27460, 0,     27746, 43621, 0,     41978, 21446, 0,     43829, 26435,   /*[GB+98308530, GB+98308539]*/
  0,     3884,  0,     42468, 0,     0,     0,     0,     0,     0,       /*[GB+98308630, GB+98308639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98308730, GB+98308739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     1986,    /*[GB+98308830, GB+98308839]*/
  0,     0,     0,     0,     0,     0,     0,     26737, 25327, 36440,   /*[GB+98308930, GB+98308939]*/
  3235,  11141, 27343, 0,     0,     0,     15408, 33483, 0,     0,       /*[GB+98308A30, GB+98308A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     3885,  16939,   /*[GB+98308B30, GB+98308B39]*/
  0,     4910,  27416, 17230, 3734,  43835, 14301, 0,     39568, 31814,   /*[GB+98308C30, GB+98308C39]*/
  21303, 5235,  29474, 0,     0,     0,     0,     0,     0,     0,       /*[GB+98308D30, GB+98308D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98308E30, GB+98308E39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98308F30, GB+98308F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98309030, GB+98309039]*/
  0,     0,     0,     0,     0,     8266,  9647,  6572,  2733,  21078,   /*[GB+98309130, GB+98309139]*/
  2977,  1506,  39893, 0,     0,     32364, 43836, 0,     43521, 0,       /*[GB+98309230, GB+98309239]*/
  4353,  32420, 32235, 0,     26622, 0,     0,     32421, 0,     34188,   /*[GB+98309330, GB+98309339]*/
  0,     0,     0,     0,     0,     0,     38989, 14088, 0,     0,       /*[GB+98309430, GB+98309439]*/
  32827, 31092, 27721, 0,     24975, 27906, 39911, 13255, 6727,  2984,    /*[GB+98309530, GB+98309539]*/
  31880, 0,     0,     19534, 0,     43262, 0,     0,     0,     0,       /*[GB+98309630, GB+98309639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98309730, GB+98309739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98309830, GB+98309839]*/
  0,     0,     0,     27417, 0,     0,     36989, 9829,  0,     18482,   /*[GB+98309930, GB+98309939]*/
  2679,  0,     0,     0,     6100,  19535, 41266, 24517, 0,     0,       /*[GB+98309A30, GB+98309A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98309B30, GB+98309B39]*/
  0,     0,     0,     0,     19882, 0,     42517, 0,     0,     0,       /*[GB+98309C30, GB+98309C39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98309D30, GB+98309D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98309E30, GB+98309E39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98309F30, GB+98309F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9830A030, GB+9830A039]*/
  0,     18691, 0,     0,     0,     20538, 0,     0,     0,     0,       /*[GB+9830A130, GB+9830A139]*/
  0,     0,     0,     7263,  0,     0,     15972, 18522, 17503, 0,       /*[GB+9830A230, GB+9830A239]*/
  27418, 9742,  2899,  35767, 0,     0,     42245, 0,     0,     0,       /*[GB+9830A330, GB+9830A339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9830A430, GB+9830A439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9830A530, GB+9830A539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9830A630, GB+9830A639]*/
  0,     0,     0,     0,     0,     36237, 1991,  3500,  0,     43057,   /*[GB+9830A730, GB+9830A739]*/
  11234, 0,     0,     635,   0,     0,     0,     0,     33893, 0,       /*[GB+9830A830, GB+9830A839]*/
  0,     0,     0,     6427,  0,     0,     41276, 10481, 0,     0,       /*[GB+9830A930, GB+9830A939]*/
  1678,  17321, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9830AA30, GB+9830AA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9830AB30, GB+9830AB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9830AC30, GB+9830AC39]*/
  0,     0,     14838, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9830AD30, GB+9830AD39]*/
  8894,  0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9830AE30, GB+9830AE39]*/
  0,     0,     0,     0,     0,     0,     21401, 0,     39944, 0,       /*[GB+9830AF30, GB+9830AF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9830B030, GB+9830B039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9830B130, GB+9830B139]*/
  33898, 0,     24288, 29570, 0,     0,     0,     0,     0,     0,       /*[GB+9830B230, GB+9830B239]*/
  5876,  0,     0,     0,     4583,  5976,  13092, 2912,  0,     0,       /*[GB+9830B330, GB+9830B339]*/
  0,     43421, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9830B430, GB+9830B439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9830B530, GB+9830B539]*/
  0,     2357,  0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9830B630, GB+9830B639]*/
  0,     0,     14241, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9830B730, GB+9830B739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9830B830, GB+9830B839]*/
  0,     0,     31679, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9830B930, GB+9830B939]*/
  0,     0,     0,     18196, 0,     0,     0,     0,     25526, 0,       /*[GB+9830BA30, GB+9830BA39]*/
  0,     0,     4584,  0,     0,     0,     0,     0,     0,     0,       /*[GB+9830BB30, GB+9830BB39]*/
  0,     30458, 0,     6326,  0,     13992, 0,     0,     0,     0,       /*[GB+9830BC30, GB+9830BC39]*/
  0,     0,     0,     42247, 15995, 0,     22014, 0,     20510, 0,       /*[GB+9830BD30, GB+9830BD39]*/
  0,     0,     0,     0,     834,   0,     0,     0,     0,     0,       /*[GB+9830BE30, GB+9830BE39]*/
  0,     43904, 0,     0,     0,     0,     0,     0,     0,     41524,   /*[GB+9830BF30, GB+9830BF39]*/
  0,     0,     39608, 0,     0,     837,   0,     0,     20646, 33921,   /*[GB+9830C030, GB+9830C039]*/
  11830, 38133, 7284,  9708,  31761, 41693, 0,     26322, 29895, 18866,   /*[GB+9830C130, GB+9830C139]*/
  26411, 19322, 10045, 17646, 31660, 5232,  19878, 12313, 29054, 14302,   /*[GB+9830C230, GB+9830C239]*/
  43984, 8893,  32688, 0,     0,     17121, 6933,  29677, 24889, 37161,   /*[GB+9830C330, GB+9830C339]*/
  0,     0,     0,     28926, 0,     8890,  0,     0,     5757,  37378,   /*[GB+9830C430, GB+9830C439]*/
  0,     0,     0,     25395, 0,     28238, 3087,  21965, 0,     21973,   /*[GB+9830C530, GB+9830C539]*/
  0,     0,     0,     22608, 0,     0,     4789,  0,     23041, 41443,   /*[GB+9830C630, GB+9830C639]*/
  2610,  0,     0,     29683, 23047, 4750,  0,     0,     16086, 0,       /*[GB+9830C730, GB+9830C739]*/
  28244, 0,     42898, 7201,  249,   1157,  0,     0,     31160, 0,       /*[GB+9830C830, GB+9830C839]*/
  24145, 34135, 26340, 28630, 0,     0,     0,     30705, 0,     0,       /*[GB+9830C930, GB+9830C939]*/
  0,     20780, 24228, 0,     31174, 27178, 0,     0,     0,     0,       /*[GB+9830CA30, GB+9830CA39]*/
  0,     0,     0,     4830,  0,     0,     13066, 8310,  33493, 14478,   /*[GB+9830CB30, GB+9830CB39]*/
  17744, 40923, 23943, 2436,  0,     23264, 0,     2995,  0,     0,       /*[GB+9830CC30, GB+9830CC39]*/
  0,     0,     0,     0,     0,     0,     20798, 10341, 22978, 3953,    /*[GB+9830CD30, GB+9830CD39]*/
  3036,  0,     0,     0,     22541, 0,     0,     34052, 15225, 13297,   /*[GB+9830CE30, GB+9830CE39]*/
  41821, 0,     0,     0,     0,     31520, 0,     0,     21220, 40237,   /*[GB+9830CF30, GB+9830CF39]*/
  0,     42571, 32012, 27451, 35195, 33620, 0,     31524, 23222, 0,       /*[GB+9830D030, GB+9830D039]*/
  32529, 13766, 0,     8092,  0,     0,     0,     0,     0,     0,       /*[GB+9830D130, GB+9830D139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     1584,  0,       /*[GB+9830D230, GB+9830D239]*/
  0,     0,     0,     37862, 0,     0,     6390,  12996, 8995,  0,       /*[GB+9830D330, GB+9830D339]*/
  0,     13779, 15157, 0,     0,     0,     0,     0,     16135, 0,       /*[GB+9830D430, GB+9830D439]*/
  0,     0,     26781, 0,     22469, 22177, 35920, 29718, 28553, 16883,   /*[GB+9830D530, GB+9830D539]*/
  26790, 32665, 0,     0,     0,     6401,  22472, 0,     27497, 9518,    /*[GB+9830D630, GB+9830D639]*/
  6718,  0,     22893, 0,     0,     3172,  11126, 36955, 0,     0,       /*[GB+9830D730, GB+9830D739]*/
  0,     0,     0,     31806, 2217,  0,     11713, 39116, 0,     28841,   /*[GB+9830D830, GB+9830D839]*/
  4294,  0,     0,     0,     0,     0,     0,     0,     0,     34837,   /*[GB+9830D930, GB+9830D939]*/
  21226, 18233, 31070, 0,     13798, 0,     0,     0,     42234, 9895,    /*[GB+9830DA30, GB+9830DA39]*/
  10020, 0,     25151, 0,     7935,  39682, 41600, 32808, 7340,  2828,    /*[GB+9830DB30, GB+9830DB39]*/
  33709, 0,     9818,  0,     0,     0,     0,     0,     0,     26800,   /*[GB+9830DC30, GB+9830DC39]*/
  12624, 0,     35091, 38792, 0,     30418, 32286, 8612,  14779, 24189,   /*[GB+9830DD30, GB+9830DD39]*/
  0,     0,     0,     28631, 39569, 16763, 0,     24197, 35100, 23374,   /*[GB+9830DE30, GB+9830DE39]*/
  34582, 14089, 35942, 0,     1435,  0,     0,     0,     22476, 0,       /*[GB+9830DF30, GB+9830DF39]*/
  18816, 0,     24253, 0,     0,     0,     15136, 0,     0,     0,       /*[GB+9830E030, GB+9830E039]*/
  30345, 36990, 138,   0,     0,     34858, 25580, 33399, 30774, 0,       /*[GB+9830E130, GB+9830E139]*/
  16097, 0,     0,     0,     0,     0,     17236, 0,     0,     0,       /*[GB+9830E230, GB+9830E239]*/
  0,     0,     0,     17687, 0,     22692, 0,     0,     6730,  29157,   /*[GB+9830E330, GB+9830E339]*/
  0,     14417, 23087, 0,     39583, 0,     0,     0,     0,     0,       /*[GB+9830E430, GB+9830E439]*/
  2599,  0,     14418, 0,     12020, 33029, 37483, 7325,  10108, 11915,   /*[GB+9830E530, GB+9830E539]*/
  14317, 0,     0,     0,     0,     0,     0,     0,     17989, 0,       /*[GB+9830E630, GB+9830E639]*/
  28200, 39137, 0,     5844,  9995,  23203, 0,     30159, 34751, 0,       /*[GB+9830E730, GB+9830E739]*/
  27572, 3065,  0,     0,     0,     0,     0,     0,     27481, 0,       /*[GB+9830E830, GB+9830E839]*/
  40338, 0,     8104,  14118, 35956, 0,     0,     0,     34036, 29159,   /*[GB+9830E930, GB+9830E939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9830EA30, GB+9830EA39]*/
  0,     35122, 18581, 0,     0,     31680, 0,     0,     21526, 2916,    /*[GB+9830EB30, GB+9830EB39]*/
  12778, 0,     0,     43123, 40307, 0,     0,     0,     0,     0,       /*[GB+9830EC30, GB+9830EC39]*/
  0,     0,     0,     17528, 0,     36751, 5696,  34708, 40203, 31506,   /*[GB+9830ED30, GB+9830ED39]*/
  1057,  27653, 34504, 6349,  24172, 24236, 0,     30337, 0,     14291,   /*[GB+9830EE30, GB+9830EE39]*/
  37478, 0,     40288, 0,     27080, 25739, 24823, 6229,  24824, 0,       /*[GB+9830EF30, GB+9830EF39]*/
  0,     10733, 40032, 39186, 13113, 8900,  10534, 28864, 0,     26366,   /*[GB+9830F030, GB+9830F039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     33058,   /*[GB+9830F130, GB+9830F139]*/
  0,     0,     41804, 24490, 36632, 9693,  15427, 31306, 0,     0,       /*[GB+9830F230, GB+9830F239]*/
  0,     0,     0,     7356,  0,     0,     36410, 19200, 1236,  0,       /*[GB+9830F330, GB+9830F339]*/
  0,     0,     43557, 11355, 0,     7169,  0,     6998,  36092, 0,       /*[GB+9830F430, GB+9830F439]*/
  28061, 0,     28062, 37258, 26894, 0,     0,     0,     0,     0,       /*[GB+9830F530, GB+9830F539]*/
  0,     0,     32099, 0,     0,     1859,  35848, 31980, 3522,  0,       /*[GB+9830F630, GB+9830F639]*/
  6691,  38134, 0,     0,     0,     3272,  0,     0,     0,     0,       /*[GB+9830F730, GB+9830F739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9830F830, GB+9830F839]*/
  0,     0,     0,     0,     0,     15715, 12955, 0,     41480, 1064,    /*[GB+9830F930, GB+9830F939]*/
  0,     31267, 0,     0,     0,     0,     23869, 0,     0,     0,       /*[GB+9830FA30, GB+9830FA39]*/
  0,     8730,  0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9830FB30, GB+9830FB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     32170, 0,       /*[GB+9830FC30, GB+9830FC39]*/
  34314, 22810, 19639, 0,     0,     43333, 0,     41896, 29429, 43637,   /*[GB+9830FD30, GB+9830FD39]*/
  8245,  6934,  24167, 36794, 29806, 26309, 7393,  25387, 0,     25449,   /*[GB+9830FE30, GB+9830FE39]*/
  0,     2293,  36381, 0,     0,     0,     0,     0,     0,     43231,   /*[GB+98318130, GB+98318139]*/
  0,     0,     21008, 14355, 0,     0,     0,     0,     0,     41233,   /*[GB+98318230, GB+98318239]*/
  24085, 33621, 36934, 8020,  23028, 0,     42378, 0,     30252, 0,       /*[GB+98318330, GB+98318339]*/
  38231, 0,     17908, 35890, 0,     38232, 0,     0,     32585, 0,       /*[GB+98318430, GB+98318439]*/
  0,     12982, 3327,  6384,  0,     0,     0,     0,     0,     0,       /*[GB+98318530, GB+98318539]*/
  0,     0,     0,     0,     0,     40068, 0,     33799, 0,     19485,   /*[GB+98318630, GB+98318639]*/
  30253, 0,     0,     0,     0,     0,     0,     0,     0,     1080,    /*[GB+98318730, GB+98318739]*/
  34180, 36943, 0,     5023,  0,     16271, 0,     0,     26032, 19800,   /*[GB+98318830, GB+98318839]*/
  1149,  0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98318930, GB+98318939]*/
  0,     0,     41306, 5006,  33311, 35730, 3335,  10917, 15906, 0,       /*[GB+98318A30, GB+98318A39]*/
  3436,  2712,  2963,  42128, 36329, 0,     0,     0,     0,     0,       /*[GB+98318B30, GB+98318B39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     9306,    /*[GB+98318C30, GB+98318C39]*/
  3391,  37588, 0,     0,     0,     0,     4220,  26500, 35235, 0,       /*[GB+98318D30, GB+98318D39]*/
  41315, 16514, 0,     15925, 0,     0,     0,     0,     0,     0,       /*[GB+98318E30, GB+98318E39]*/
  0,     0,     0,     0,     0,     26501, 0,     21292, 37615, 43768,   /*[GB+98318F30, GB+98318F39]*/
  24678, 0,     28981, 0,     0,     0,     0,     29181, 2872,  0,       /*[GB+98319030, GB+98319039]*/
  0,     6802,  0,     10430, 20146, 36611, 0,     0,     0,     0,       /*[GB+98319130, GB+98319139]*/
  22963, 1825,  0,     0,     0,     0,     0,     0,     0,     31936,   /*[GB+98319230, GB+98319239]*/
  0,     0,     0,     0,     0,     0,     22848, 12072, 25428, 0,       /*[GB+98319330, GB+98319339]*/
  0,     0,     0,     0,     0,     34196, 0,     0,     19404, 0,       /*[GB+98319430, GB+98319439]*/
  0,     0,     0,     18637, 0,     38318, 0,     18418, 29655, 0,       /*[GB+98319530, GB+98319539]*/
  0,     5405,  0,     0,     32943, 0,     0,     4736,  0,     0,       /*[GB+98319630, GB+98319639]*/
  0,     15273, 11390, 0,     39472, 0,     16993, 0,     0,     0,       /*[GB+98319730, GB+98319739]*/
  10870, 9232,  7909,  0,     0,     17002, 0,     0,     10165, 3776,    /*[GB+98319830, GB+98319839]*/
  0,     0,     15275, 0,     0,     30032, 0,     0,     6391,  0,       /*[GB+98319930, GB+98319939]*/
  0,     0,     0,     0,     0,     0,     30063, 0,     0,     0,       /*[GB+98319A30, GB+98319A39]*/
  25932, 0,     0,     0,     10178, 0,     0,     0,     0,     0,       /*[GB+98319B30, GB+98319B39]*/
  0,     43214, 0,     0,     31396, 0,     0,     40258, 0,     0,       /*[GB+98319C30, GB+98319C39]*/
  0,     0,     0,     43933, 0,     0,     0,     0,     0,     0,       /*[GB+98319D30, GB+98319D39]*/
  0,     33314, 5052,  0,     0,     0,     0,     0,     0,     0,       /*[GB+98319E30, GB+98319E39]*/
  0,     0,     0,     42452, 0,     0,     0,     0,     29418, 0,       /*[GB+98319F30, GB+98319F39]*/
  6568,  0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9831A030, GB+9831A039]*/
  0,     32676, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9831A130, GB+9831A139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     3987,    /*[GB+9831A230, GB+9831A239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9831A330, GB+9831A339]*/
  0,     27468, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9831A430, GB+9831A439]*/
  0,     0,     13248, 0,     0,     0,     0,     0,     6267,  0,       /*[GB+9831A530, GB+9831A539]*/
  2485,  0,     16659, 0,     0,     32689, 0,     0,     0,     2416,    /*[GB+9831A630, GB+9831A639]*/
  0,     0,     20634, 23510, 0,     0,     0,     0,     0,     0,       /*[GB+9831A730, GB+9831A739]*/
  27647, 0,     0,     0,     32699, 0,     0,     0,     0,     0,       /*[GB+9831A830, GB+9831A839]*/
  27650, 0,     0,     14835, 0,     0,     0,     0,     0,     0,       /*[GB+9831A930, GB+9831A939]*/
  0,     0,     0,     0,     0,     0,     32703, 0,     0,     0,       /*[GB+9831AA30, GB+9831AA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9831AB30, GB+9831AB39]*/
  0,     0,     0,     0,     2358,  18184, 0,     0,     0,     0,       /*[GB+9831AC30, GB+9831AC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9831AD30, GB+9831AD39]*/
  0,     40310, 0,     0,     0,     0,     40339, 0,     0,     3460,    /*[GB+9831AE30, GB+9831AE39]*/
  0,     0,     28018, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9831AF30, GB+9831AF39]*/
  11606, 27975, 32013, 0,     8363,  0,     26006, 35719, 0,     0,       /*[GB+9831B030, GB+9831B039]*/
  0,     4350,  8755,  0,     0,     0,     38266, 6563,  8573,  8574,    /*[GB+9831B130, GB+9831B139]*/
  40809, 24725, 0,     19325, 0,     0,     0,     27976, 23763, 0,       /*[GB+9831B230, GB+9831B239]*/
  0,     0,     38592, 20581, 12002, 38714, 1121,  39411, 28373, 0,       /*[GB+9831B330, GB+9831B339]*/
  35224, 4219,  0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9831B430, GB+9831B439]*/
  28374, 6202,  0,     0,     0,     38319, 0,     3297,  9443,  19664,   /*[GB+9831B530, GB+9831B539]*/
  24472, 40414, 36964, 22109, 0,     30037, 0,     0,     0,     0,       /*[GB+9831B630, GB+9831B639]*/
  0,     0,     0,     15126, 0,     29743, 23692, 0,     36472, 27792,   /*[GB+9831B730, GB+9831B739]*/
  7205,  0,     0,     4429,  35251, 0,     31871, 9528,  36129, 0,       /*[GB+9831B830, GB+9831B839]*/
  17254, 0,     19351, 0,     5400,  41013, 0,     0,     0,     0,       /*[GB+9831B930, GB+9831B939]*/
  0,     0,     0,     0,     0,     0,     38527, 29692, 0,     39683,   /*[GB+9831BA30, GB+9831BA39]*/
  32148, 32149, 0,     0,     523,   26914, 2116,  5406,  0,     41017,   /*[GB+9831BB30, GB+9831BB39]*/
  11521, 0,     0,     0,     12502, 0,     0,     0,     0,     0,       /*[GB+9831BC30, GB+9831BC39]*/
  0,     19925, 0,     0,     0,     0,     19454, 27319, 0,     0,       /*[GB+9831BD30, GB+9831BD39]*/
  0,     0,     0,     0,     0,     0,     0,     5576,  21304, 34192,   /*[GB+9831BE30, GB+9831BE39]*/
  33111, 10705, 43528, 41857, 27817, 5177,  18483, 0,     6125,  18685,   /*[GB+9831BF30, GB+9831BF39]*/
  9689,  0,     0,     0,     23935, 0,     0,     0,     38804, 0,       /*[GB+9831C030, GB+9831C039]*/
  11561, 0,     7028,  33494, 0,     0,     0,     0,     0,     0,       /*[GB+9831C130, GB+9831C139]*/
  0,     24323, 11212, 0,     19673, 22132, 7984,  20247, 40852, 2138,    /*[GB+9831C230, GB+9831C239]*/
  6222,  20935, 0,     43202, 20165, 23674, 7036,  2289,  0,     0,       /*[GB+9831C330, GB+9831C339]*/
  0,     0,     3344,  20856, 0,     0,     0,     0,     0,     0,       /*[GB+9831C430, GB+9831C439]*/
  0,     0,     33748, 0,     0,     0,     0,     0,     0,     219,     /*[GB+9831C530, GB+9831C539]*/
  42589, 0,     22507, 32205, 11158, 0,     0,     7040,  32324, 42590,   /*[GB+9831C630, GB+9831C639]*/
  2615,  18523, 29349, 0,     0,     0,     2616,  0,     29566, 0,       /*[GB+9831C730, GB+9831C739]*/
  7041,  0,     12700, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9831C830, GB+9831C839]*/
  13846, 23266, 5588,  8834,  12902, 12668, 4705,  0,     39123, 27573,   /*[GB+9831C930, GB+9831C939]*/
  31500, 0,     0,     0,     0,     0,     0,     0,     19612, 0,       /*[GB+9831CA30, GB+9831CA39]*/
  0,     30000, 19386, 20799, 6225,  0,     0,     0,     23719, 0,       /*[GB+9831CB30, GB+9831CB39]*/
  19387, 0,     0,     18562, 0,     23292, 0,     0,     0,     19096,   /*[GB+9831CC30, GB+9831CC39]*/
  28091, 0,     8062,  0,     0,     0,     0,     28092, 0,     0,       /*[GB+9831CD30, GB+9831CD39]*/
  19397, 41340, 0,     1834,  18190, 0,     0,     29580, 26932, 13858,   /*[GB+9831CE30, GB+9831CE39]*/
  43424, 0,     0,     19400, 19401, 21404, 0,     29365, 0,     0,       /*[GB+9831CF30, GB+9831CF39]*/
  0,     0,     19405, 0,     0,     12763, 0,     0,     0,     0,       /*[GB+9831D030, GB+9831D039]*/
  0,     0,     19424, 13861, 26565, 0,     0,     0,     23270, 15106,   /*[GB+9831D130, GB+9831D139]*/
  3328,  0,     0,     0,     0,     0,     41726, 26108, 0,     26112,   /*[GB+9831D230, GB+9831D239]*/
  7482,  0,     0,     27830, 0,     0,     0,     0,     0,     959,     /*[GB+9831D330, GB+9831D339]*/
  0,     7921,  0,     0,     8021,  7951,  0,     0,     0,     0,       /*[GB+9831D430, GB+9831D439]*/
  7868,  16721, 0,     0,     0,     0,     40525, 12604, 0,     7958,    /*[GB+9831D530, GB+9831D539]*/
  0,     0,     0,     7042,  0,     0,     0,     0,     23526, 23224,   /*[GB+9831D630, GB+9831D639]*/
  23709, 5545,  5450,  96,    0,     31290, 3217,  365,   21195, 22495,   /*[GB+9831D730, GB+9831D739]*/
  0,     0,     2174,  39835, 34246, 20739, 0,     0,     24417, 0,       /*[GB+9831D830, GB+9831D839]*/
  6975,  0,     5771,  0,     0,     35525, 22840, 10633, 4298,  29293,   /*[GB+9831D930, GB+9831D939]*/
  20820, 0,     0,     0,     0,     0,     32464, 38376, 6321,  36612,   /*[GB+9831DA30, GB+9831DA39]*/
  0,     33212, 30011, 0,     216,   17487, 22500, 0,     26359, 21704,   /*[GB+9831DB30, GB+9831DB39]*/
  23402, 2418,  2437,  0,     0,     0,     17596, 31304, 37649, 0,       /*[GB+9831DC30, GB+9831DC39]*/
  22909, 0,     29295, 0,     0,     0,     0,     4706,  0,     13985,   /*[GB+9831DD30, GB+9831DD39]*/
  0,     0,     9047,  0,     0,     0,     14125, 10483, 0,     41223,   /*[GB+9831DE30, GB+9831DE39]*/
  0,     20547, 0,     0,     12965, 10294, 0,     0,     0,     13650,   /*[GB+9831DF30, GB+9831DF39]*/
  13310, 0,     0,     36116, 0,     8257,  0,     0,     0,     1356,    /*[GB+9831E030, GB+9831E039]*/
  24912, 0,     0,     40085, 288,   5905,  0,     0,     7433,  8137,    /*[GB+9831E130, GB+9831E139]*/
  0,     0,     15629, 22767, 32139, 0,     28454, 8585,  34129, 26494,   /*[GB+9831E230, GB+9831E239]*/
  24580, 32411, 19336, 20681, 42779, 862,   0,     39836, 0,     0,       /*[GB+9831E330, GB+9831E339]*/
  0,     0,     1104,  20740, 0,     13601, 0,     37172, 0,     16772,   /*[GB+9831E430, GB+9831E439]*/
  14467, 0,     1902,  0,     0,     0,     20586, 10259, 7255,  42453,   /*[GB+9831E530, GB+9831E539]*/
  0,     0,     0,     41846, 7461,  42851, 492,   0,     0,     0,       /*[GB+9831E630, GB+9831E639]*/
  41847, 8264,  6721,  22899, 2280,  7022,  27904, 27806, 1420,  0,       /*[GB+9831E730, GB+9831E739]*/
  0,     0,     42471, 41601, 2281,  0,     0,     0,     15937, 36396,   /*[GB+9831E830, GB+9831E839]*/
  0,     0,     0,     34012, 0,     43396, 595,   37333, 4119,  30425,   /*[GB+9831E930, GB+9831E939]*/
  25336, 0,     22634, 39894, 7476,  13815, 38377, 34855, 0,     23072,   /*[GB+9831EA30, GB+9831EA39]*/
  0,     0,     1865,  0,     0,     0,     0,     10508, 35101, 6322,    /*[GB+9831EB30, GB+9831EB39]*/
  0,     0,     15413, 30284, 6416,  8624,  21685, 28992, 36136, 31093,   /*[GB+9831EC30, GB+9831EC39]*/
  39912, 0,     0,     35109, 8625,  42505, 22780, 36137, 37135, 0,       /*[GB+9831ED30, GB+9831ED39]*/
  8318,  43744, 42795, 36016, 0,     0,     0,     42858, 0,     17331,   /*[GB+9831EE30, GB+9831EE39]*/
  0,     38990, 9200,  17238, 0,     7313,  32690, 21140, 0,     0,       /*[GB+9831EF30, GB+9831EF39]*/
  0,     12034, 4122,  0,     30291, 30434, 30292, 30435, 36236, 0,       /*[GB+9831F030, GB+9831F039]*/
  9572,  13695, 0,     2139,  30436, 40141, 0,     0,     0,     0,       /*[GB+9831F130, GB+9831F139]*/
  0,     0,     30437, 0,     0,     0,     0,     4490,  0,     0,       /*[GB+9831F230, GB+9831F239]*/
  35556, 11156, 30231, 0,     0,     0,     0,     22135, 3976,  30043,   /*[GB+9831F330, GB+9831F339]*/
  40853, 20173, 41375, 20599, 35947, 13842, 28050, 0,     35948, 0,       /*[GB+9831F430, GB+9831F439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9831F530, GB+9831F539]*/
  14200, 0,     0,     0,     878,   20710, 0,     31631, 34034, 0,       /*[GB+9831F630, GB+9831F639]*/
  0,     27191, 0,     28612, 42802, 0,     13847, 8835,  5873,  0,       /*[GB+9831F730, GB+9831F739]*/
  0,     39053, 15420, 0,     38446, 718,   0,     27831, 30162, 0,       /*[GB+9831F830, GB+9831F839]*/
  7165,  35300, 0,     0,     0,     12146, 0,     0,     0,     27347,   /*[GB+9831F930, GB+9831F939]*/
  22789, 42804, 10343, 0,     20605, 0,     0,     0,     0,     0,       /*[GB+9831FA30, GB+9831FA39]*/
  0,     0,     42082, 35957, 0,     42083, 0,     15990, 15422, 0,       /*[GB+9831FB30, GB+9831FB39]*/
  0,     40126, 19959, 0,     0,     2161,  0,     0,     27193, 35309,   /*[GB+9831FC30, GB+9831FC39]*/
  0,     0,     0,     0,     0,     33902, 20512, 0,     0,     0,       /*[GB+9831FD30, GB+9831FD39]*/
  9352,  0,     0,     43887, 10775, 0,     14276, 27520, 25934, 26831,   /*[GB+9831FE30, GB+9831FE39]*/
  23738, 40259, 3290,  8587,  23784, 0,     0,     21197, 0,     0,       /*[GB+98328130, GB+98328139]*/
  30851, 0,     16502, 33990, 0,     0,     0,     0,     0,     7256,    /*[GB+98328230, GB+98328239]*/
  0,     38320, 0,     0,     30124, 0,     34141, 15703, 0,     32677,   /*[GB+98328330, GB+98328339]*/
  0,     38358, 0,     39895, 0,     596,   32154, 30426, 23744, 0,       /*[GB+98328430, GB+98328439]*/
  0,     0,     1907,  0,     0,     40473, 40474, 6952,  27410, 32691,   /*[GB+98328530, GB+98328539]*/
  0,     0,     0,     0,     0,     32694, 10333, 0,     5298,  34155,   /*[GB+98328630, GB+98328639]*/
  0,     0,     12760, 15727, 30044, 0,     30157, 0,     0,     19836,   /*[GB+98328730, GB+98328739]*/
  600,   39934, 41988, 10340, 0,     16981, 9146,  5475,  0,     0,       /*[GB+98328830, GB+98328839]*/
  0,     34467, 41127, 27832, 0,     16417, 2224,  2155,  0,     0,       /*[GB+98328930, GB+98328939]*/
  30163, 0,     39956, 2225,  15249, 15250, 0,     15788, 16003, 0,       /*[GB+98328A30, GB+98328A39]*/
  22341, 41123, 30830, 32781, 34123, 35205, 0,     0,     0,     27565,   /*[GB+98328B30, GB+98328B39]*/
  13056, 0,     0,     35286, 0,     0,     7046,  43589, 0,     0,       /*[GB+98328C30, GB+98328C39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     40137, 26009,   /*[GB+98328D30, GB+98328D39]*/
  8588,  23915, 23739, 0,     38718, 0,     11318, 43826, 0,     9522,    /*[GB+98328E30, GB+98328E39]*/
  6631,  11024, 33202, 0,     23671, 0,     0,     29952, 16234, 23073,   /*[GB+98328F30, GB+98328F39]*/
  10935, 8338,  7477,  37635, 0,     0,     31494, 6637,  41619, 27678,   /*[GB+98329030, GB+98329039]*/
  0,     0,     0,     22136, 29981, 0,     17831, 0,     24721, 7328,    /*[GB+98329130, GB+98329139]*/
  43417, 37652, 0,     0,     0,     34647, 10131, 27023, 19432, 0,       /*[GB+98329230, GB+98329239]*/
  0,     382,   0,     0,     2770,  0,     7149,  26154, 16690, 35721,   /*[GB+98329330, GB+98329339]*/
  12106, 37238, 41831, 31528, 1101,  0,     0,     276,   0,     16202,   /*[GB+98329430, GB+98329439]*/
  0,     16693, 23800, 39379, 364,   21221, 22089, 0,     0,     0,       /*[GB+98329530, GB+98329539]*/
  0,     0,     8675,  26033, 17282, 6683,  30648, 0,     0,     25460,   /*[GB+98329630, GB+98329639]*/
  37437, 32889, 11702, 22098, 11023, 36123, 0,     1377,  24512, 0,       /*[GB+98329730, GB+98329739]*/
  5908,  0,     41783, 0,     28710, 16227, 2589,  0,     0,     35732,   /*[GB+98329830, GB+98329839]*/
  41784, 0,     42949, 0,     0,     0,     0,     12605, 3784,  0,       /*[GB+98329930, GB+98329939]*/
  0,     11442, 23286, 23059, 36965, 0,     3986,  8685,  10103, 24794,   /*[GB+98329A30, GB+98329A39]*/
  0,     9377,  31531, 0,     0,     0,     30651, 25463, 0,     0,       /*[GB+98329B30, GB+98329B39]*/
  15321, 35748, 25057, 0,     41848, 0,     0,     0,     26344, 0,       /*[GB+98329C30, GB+98329C39]*/
  7341,  0,     0,     12627, 11468, 26087, 0,     3252,  0,     0,       /*[GB+98329D30, GB+98329D39]*/
  8804,  0,     11324, 33710, 33317, 20910, 0,     0,     31006, 0,       /*[GB+98329E30, GB+98329E39]*/
  0,     11325, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98329F30, GB+98329F39]*/
  0,     0,     2103,  0,     25981, 9378,  0,     0,     8613,  17055,   /*[GB+9832A030, GB+9832A039]*/
  0,     0,     0,     0,     0,     1768,  15137, 16236, 10188, 0,       /*[GB+9832A130, GB+9832A139]*/
  9175,  35757, 252,   40274, 33489, 37909, 15022, 0,     20148, 26687,   /*[GB+9832A230, GB+9832A239]*/
  31933, 0,     13495, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9832A330, GB+9832A339]*/
  0,     0,     0,     0,     15199, 15945, 24336, 17157, 0,     0,       /*[GB+9832A430, GB+9832A439]*/
  32300, 12652, 7115,  40140, 14695, 0,     10329, 9178,  24593, 21231,   /*[GB+9832A530, GB+9832A539]*/
  43154, 10886, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9832A630, GB+9832A639]*/
  39070, 26348, 0,     17983, 278,   24156, 33095, 0,     0,     26015,   /*[GB+9832A730, GB+9832A739]*/
  0,     25984, 21790, 6666,  21463, 41207, 0,     26093, 37918, 0,       /*[GB+9832A830, GB+9832A839]*/
  0,     0,     0,     0,     0,     0,     634,   0,     0,     15707,   /*[GB+9832A930, GB+9832A939]*/
  0,     40854, 35949, 17504, 20636, 20637, 23474, 909,   0,     24618,   /*[GB+9832AA30, GB+9832AA39]*/
  35950, 0,     0,     0,     409,   0,     0,     0,     0,     0,       /*[GB+9832AB30, GB+9832AB39]*/
  11444, 0,     0,     11445, 41277, 0,     27846, 9179,  0,     24534,   /*[GB+9832AC30, GB+9832AC39]*/
  18987, 0,     0,     0,     0,     32303, 36028, 0,     0,     0,       /*[GB+9832AD30, GB+9832AD39]*/
  0,     0,     17035, 0,     0,     7499,  10836, 27833, 0,     0,       /*[GB+9832AE30, GB+9832AE39]*/
  5590,  0,     0,     0,     0,     0,     36029, 0,     7504,  8927,    /*[GB+9832AF30, GB+9832AF39]*/
  0,     5710,  0,     21384, 37738, 23183, 0,     24390, 0,     0,       /*[GB+9832B030, GB+9832B039]*/
  0,     2488,  0,     0,     0,     0,     40177, 0,     7508,  23111,   /*[GB+9832B130, GB+9832B139]*/
  0,     0,     38681, 0,     0,     25692, 0,     11334, 19406, 0,       /*[GB+9832B230, GB+9832B239]*/
  2836,  38967, 0,     0,     0,     0,     10004, 0,     22971, 35733,   /*[GB+9832B330, GB+9832B339]*/
  30588, 14968, 0,     0,     0,     23903, 0,     18883, 0,     0,       /*[GB+9832B430, GB+9832B439]*/
  969,   0,     0,     35363, 32017, 3171,  8965,  0,     11613, 32532,   /*[GB+9832B530, GB+9832B539]*/
  0,     0,     8576,  23834, 0,     22565, 36322, 36229, 0,     0,       /*[GB+9832B630, GB+9832B639]*/
  19576, 19337, 36231, 26424, 10918, 37874, 10798, 0,     8589,  489,     /*[GB+9832B730, GB+9832B739]*/
  37875, 0,     2077,  0,     0,     0,     11319, 18420, 0,     0,       /*[GB+9832B830, GB+9832B839]*/
  0,     27550, 33686, 0,     0,     0,     28970, 24275, 12009, 37885,   /*[GB+9832B930, GB+9832B939]*/
  2091,  2104,  22662, 26345, 0,     0,     32809, 3305,  39263, 0,       /*[GB+9832BA30, GB+9832BA39]*/
  32910, 12494, 27559, 0,     0,     30909, 11469, 29782, 10814, 23742,   /*[GB+9832BB30, GB+9832BB39]*/
  0,     7881,  0,     19523, 42793, 24111, 0,     18460, 0,     0,       /*[GB+9832BC30, GB+9832BC39]*/
  4488,  33726, 41928, 0,     1127,  41929, 30138, 36856, 0,     0,       /*[GB+9832BD30, GB+9832BD39]*/
  0,     0,     26040, 33490, 18850, 0,     41508, 18817, 0,     0,       /*[GB+9832BE30, GB+9832BE39]*/
  0,     14533, 0,     11330, 39168, 0,     17303, 11562, 0,     40174,   /*[GB+9832BF30, GB+9832BF39]*/
  11432, 37636, 0,     38956, 36083, 39271, 0,     0,     0,     0,       /*[GB+9832C030, GB+9832C039]*/
  26689, 0,     0,     0,     30778, 30293, 0,     18504, 29793, 18505,   /*[GB+9832C130, GB+9832C139]*/
  0,     0,     39921, 0,     0,     38414, 0,     0,     0,     0,       /*[GB+9832C230, GB+9832C239]*/
  0,     35636, 346,   31884, 29982, 29318, 0,     39936, 0,     8274,    /*[GB+9832C330, GB+9832C339]*/
  0,     0,     0,     29984, 31944, 39938, 41936, 41937, 0,     30918,   /*[GB+9832C430, GB+9832C439]*/
  0,     0,     0,     0,     19551, 0,     3594,  30059, 27729, 0,       /*[GB+9832C530, GB+9832C539]*/
  0,     0,     0,     0,     0,     9834,  8282,  19103, 24289, 0,       /*[GB+9832C630, GB+9832C639]*/
  0,     0,     18586, 0,     19721, 4151,  34884, 11335, 35310, 28391,   /*[GB+9832C730, GB+9832C739]*/
  0,     0,     19728, 11478, 36066, 8288,  0,     478,   2016,  30890,   /*[GB+9832C830, GB+9832C839]*/
  29896, 41485, 1633,  29764, 31915, 30268, 34827, 11272, 27719, 0,       /*[GB+9832C930, GB+9832C939]*/
  7931,  0,     0,     0,     0,     23230, 20936, 0,     0,     0,       /*[GB+9832CA30, GB+9832CA39]*/
  0,     12147, 42085, 0,     0,     0,     38135, 0,     39737, 0,       /*[GB+9832CB30, GB+9832CB39]*/
  0,     37833, 40184, 3703,  0,     0,     0,     0,     0,     0,       /*[GB+9832CC30, GB+9832CC39]*/
  0,     0,     0,     0,     0,     38643, 23228, 27215, 0,     0,       /*[GB+9832CD30, GB+9832CD39]*/
  0,     22370, 0,     0,     0,     0,     0,     0,     0,     31232,   /*[GB+9832CE30, GB+9832CE39]*/
  0,     456,   0,     0,     0,     0,     7593,  41910, 7434,  24612,   /*[GB+9832CF30, GB+9832CF39]*/
  12997, 22727, 0,     15188, 13651, 0,     0,     8966,  0,     0,       /*[GB+9832D030, GB+9832D039]*/
  0,     19326, 0,     43954, 0,     0,     978,   0,     0,     0,       /*[GB+9832D130, GB+9832D139]*/
  0,     0,     0,     29112, 37981, 23547, 0,     0,     0,     22612,   /*[GB+9832D230, GB+9832D239]*/
  28963, 0,     20817, 28964, 0,     31059, 0,     0,     0,     0,       /*[GB+9832D330, GB+9832D339]*/
  0,     0,     9519,  0,     0,     0,     0,     17978, 823,   40016,   /*[GB+9832D430, GB+9832D439]*/
  44011, 17647, 35618, 0,     0,     40588, 3615,  13799, 21073, 13657,   /*[GB+9832D530, GB+9832D539]*/
  39857, 0,     0,     39858, 37886, 0,     0,     0,     20864, 40589,   /*[GB+9832D630, GB+9832D639]*/
  0,     43253, 31072, 0,     0,     33991, 15486, 40637, 20131, 30910,   /*[GB+9832D730, GB+9832D739]*/
  0,     0,     43257, 19352, 0,     15652, 0,     0,     13043, 15726,   /*[GB+9832D830, GB+9832D839]*/
  0,     0,     43516, 0,     40191, 6569,  0,     0,     0,     0,       /*[GB+9832D930, GB+9832D939]*/
  22528, 0,     0,     0,     3972,  0,     0,     0,     0,     0,       /*[GB+9832DA30, GB+9832DA39]*/
  0,     0,     0,     291,   0,     0,     0,     9412,  0,     1987,    /*[GB+9832DB30, GB+9832DB39]*/
  6519,  32811, 0,     0,     0,     0,     26685, 16410, 0,     0,       /*[GB+9832DC30, GB+9832DC39]*/
  24198, 16237, 8620,  39121, 0,     31877, 31696, 39383, 37249, 0,       /*[GB+9832DD30, GB+9832DD39]*/
  37481, 0,     0,     36985, 0,     0,     0,     0,     0,     0,       /*[GB+9832DE30, GB+9832DE39]*/
  0,     0,     0,     0,     34377, 0,     31085, 0,     30139, 0,       /*[GB+9832DF30, GB+9832DF39]*/
  0,     0,     4697,  0,     0,     0,     35816, 33491, 2378,  39896,   /*[GB+9832E030, GB+9832E039]*/
  0,     0,     2985,  34153, 0,     16349, 2486,  2680,  26358, 0,       /*[GB+9832E130, GB+9832E139]*/
  0,     5238,  0,     29976, 0,     0,     11912, 0,     0,     0,       /*[GB+9832E230, GB+9832E239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9832E330, GB+9832E339]*/
  0,     33269, 40020, 0,     0,     0,     0,     14239, 34699, 41111,   /*[GB+9832E430, GB+9832E439]*/
  27648, 22005, 0,     28213, 2534,  30042, 4434,  21464, 42911, 3974,    /*[GB+9832E530, GB+9832E539]*/
  13512, 3975,  0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9832E630, GB+9832E639]*/
  0,     0,     0,     0,     0,     18694, 0,     0,     0,     0,       /*[GB+9832E730, GB+9832E739]*/
  10603, 0,     0,     0,     0,     0,     7074,  34701, 3502,  28214,   /*[GB+9832E830, GB+9832E839]*/
  18524, 12326, 0,     0,     5872,  0,     0,     18821, 0,     0,       /*[GB+9832E930, GB+9832E939]*/
  0,     0,     0,     0,     40505, 0,     0,     0,     12897, 19166,   /*[GB+9832EA30, GB+9832EA39]*/
  0,     0,     6825,  10832, 39050, 40176, 10527, 19383, 0,     18737,   /*[GB+9832EB30, GB+9832EB39]*/
  0,     411,   5621,  38438, 23267, 28167, 0,     35410, 0,     0,       /*[GB+9832EC30, GB+9832EC39]*/
  5301,  39939, 2386,  33220, 2725,  2153,  0,     13980, 0,     0,       /*[GB+9832ED30, GB+9832ED39]*/
  0,     0,     0,     0,     21348, 32839, 22011, 0,     0,     0,       /*[GB+9832EE30, GB+9832EE39]*/
  0,     29423, 14707, 29483, 12905, 0,     0,     4625,  0,     19718,   /*[GB+9832EF30, GB+9832EF39]*/
  12053, 0,     32083, 0,     0,     39605, 0,     0,     3037,  39059,   /*[GB+9832F030, GB+9832F039]*/
  12910, 0,     17574, 0,     0,     0,     0,     22335, 431,   8946,    /*[GB+9832F130, GB+9832F139]*/
  37953, 13628, 0,     5258,  6505,  36377, 27017, 15718, 31864, 35797,   /*[GB+9832F230, GB+9832F239]*/
  29738, 0,     2518,  3483,  0,     6477,  0,     0,     0,     0,       /*[GB+9832F330, GB+9832F339]*/
  0,     0,     0,     0,     10552, 0,     0,     0,     0,     33389,   /*[GB+9832F430, GB+9832F439]*/
  0,     0,     0,     0,     0,     0,     25062, 0,     13057, 0,       /*[GB+9832F530, GB+9832F539]*/
  0,     0,     0,     0,     17029, 32695, 0,     0,     0,     0,       /*[GB+9832F630, GB+9832F639]*/
  0,     0,     29027, 8635,  0,     31887, 0,     1761,  0,     30584,   /*[GB+9832F730, GB+9832F739]*/
  10846, 24332, 0,     0,     0,     39533, 30589, 0,     0,     0,       /*[GB+9832F830, GB+9832F839]*/
  0,     0,     0,     0,     34636, 0,     35632, 0,     0,     0,       /*[GB+9832F930, GB+9832F939]*/
  0,     33115, 10604, 23945, 38004, 0,     0,     0,     0,     0,       /*[GB+9832FA30, GB+9832FA39]*/
  0,     30628, 0,     1774,  34644, 0,     0,     38008, 0,     0,       /*[GB+9832FB30, GB+9832FB39]*/
  24299, 12100, 22229, 457,   0,     7839,  0,     0,     0,     6468,    /*[GB+9832FC30, GB+9832FC39]*/
  0,     0,     0,     0,     0,     0,     0,     8097,  0,     0,       /*[GB+9832FD30, GB+9832FD39]*/
  14770, 0,     0,     0,     30232, 0,     0,     0,     0,     0,       /*[GB+9832FE30, GB+9832FE39]*/
  32453, 9239,  0,     0,     8098,  0,     32143, 0,     33016, 9897,    /*[GB+98338130, GB+98338139]*/
  7133,  0,     41592, 4719,  37396, 0,     684,   2277,  22110, 0,       /*[GB+98338230, GB+98338239]*/
  24549, 0,     0,     9318,  0,     0,     19509, 0,     0,     26911,   /*[GB+98338330, GB+98338339]*/
  0,     0,     0,     0,     0,     0,     8971,  0,     11715, 22181,   /*[GB+98338430, GB+98338439]*/
  0,     0,     35620, 12253, 8614,  12474, 0,     26508, 14670, 32078,   /*[GB+98338530, GB+98338539]*/
  39556, 22117, 42704, 15325, 0,     29152, 16884, 26180, 19058, 8805,    /*[GB+98338630, GB+98338639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98338730, GB+98338739]*/
  35937, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98338830, GB+98338839]*/
  0,     0,     19221, 0,     23074, 0,     24036, 0,     8813,  2285,    /*[GB+98338930, GB+98338939]*/
  38378, 0,     0,     2117,  0,     7311,  0,     0,     0,     0,       /*[GB+98338A30, GB+98338A39]*/
  0,     41609, 0,     18461, 0,     0,     31837, 5294,  0,     20158,   /*[GB+98338B30, GB+98338B39]*/
  36615, 6573,  25861, 32423, 21954, 43922, 8824,  26446, 1021,  370,     /*[GB+98338C30, GB+98338C39]*/
  25581, 0,     0,     0,     0,     26833, 0,     0,     0,     0,       /*[GB+98338D30, GB+98338D39]*/
  0,     0,     26811, 11960, 34585, 17195, 0,     0,     0,     5115,    /*[GB+98338E30, GB+98338E39]*/
  0,     0,     37916, 0,     0,     0,     3669,  43746, 22623, 3592,    /*[GB+98338F30, GB+98338F39]*/
  6922,  39072, 41639, 0,     30684, 37139, 35112, 36023, 0,     6953,    /*[GB+98339030, GB+98339039]*/
  29294, 34383, 34384, 0,     0,     7324,  0,     0,     0,     0,       /*[GB+98339130, GB+98339139]*/
  0,     0,     0,     0,     17257, 0,     0,     0,     0,     0,       /*[GB+98339230, GB+98339239]*/
  32933, 39172, 39585, 0,     0,     31010, 0,     14836, 0,     30781,   /*[GB+98339330, GB+98339339]*/
  24995, 0,     6132,  30867, 20296, 41332, 32986, 13523, 304,   23511,   /*[GB+98339430, GB+98339439]*/
  25241, 0,     0,     0,     28733, 30349, 0,     0,     22137, 0,       /*[GB+98339530, GB+98339539]*/
  0,     0,     39277, 0,     0,     0,     0,     0,     0,     0,       /*[GB+98339630, GB+98339639]*/
  41025, 0,     0,     37344, 0,     0,     0,     0,     0,     0,       /*[GB+98339730, GB+98339739]*/
  0,     3949,  18169, 39596, 3000,  37734, 0,     0,     3897,  0,       /*[GB+98339830, GB+98339839]*/
  18170, 0,     0,     0,     0,     0,     0,     0,     0,     31821,   /*[GB+98339930, GB+98339939]*/
  0,     43848, 0,     0,     34396, 0,     0,     0,     0,     34039,   /*[GB+98339A30, GB+98339A39]*/
  0,     1775,  10707, 25005, 27679, 0,     7945,  0,     28169, 0,       /*[GB+98339B30, GB+98339B39]*/
  0,     0,     0,     0,     0,     0,     12077, 0,     0,     0,       /*[GB+98339C30, GB+98339C39]*/
  0,     0,     0,     0,     761,   39603, 39604, 0,     13272, 0,       /*[GB+98339D30, GB+98339D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98339E30, GB+98339E39]*/
  0,     16791, 43695, 0,     0,     0,     0,     0,     0,     34607,   /*[GB+98339F30, GB+98339F39]*/
  21352, 0,     0,     0,     0,     0,     0,     0,     0,     18587,   /*[GB+9833A030, GB+9833A039]*/
  42539, 7783,  19105, 2359,  6839,  10109, 35584, 18197, 25183, 0,       /*[GB+9833A130, GB+9833A139]*/
  0,     15996, 32085, 0,     15680, 34885, 0,     0,     0,     0,       /*[GB+9833A230, GB+9833A239]*/
  26562, 0,     0,     0,     43124, 0,     0,     0,     0,     16004,   /*[GB+9833A330, GB+9833A339]*/
  0,     0,     0,     0,     14738, 0,     42749, 34492, 19623, 37149,   /*[GB+9833A430, GB+9833A439]*/
  26895, 38101, 19015, 1950,  12705, 13371, 6616,  42660, 26612, 14581,   /*[GB+9833A530, GB+9833A539]*/
  0,     0,     18343, 0,     15441, 32388, 1017,  0,     0,     1144,    /*[GB+9833A630, GB+9833A639]*/
  9289,  4568,  0,     25397, 29766, 32876, 3556,  19794, 39497, 17250,   /*[GB+9833A730, GB+9833A739]*/
  25398, 6117,  30338, 41307, 0,     7745,  19079, 31450, 12966, 25290,   /*[GB+9833A830, GB+9833A839]*/
  11683, 0,     0,     39499, 25125, 39500, 32224, 0,     516,   24916,   /*[GB+9833A930, GB+9833A939]*/
  27521, 25658, 36548, 34434, 0,     0,     0,     0,     0,     0,       /*[GB+9833AA30, GB+9833AA39]*/
  4639,  7840,  0,     17180, 10367, 26745, 7442,  26168, 0,     0,       /*[GB+9833AB30, GB+9833AB39]*/
  20682, 16700, 37390, 0,     0,     26746, 4145,  13692, 0,     42950,   /*[GB+9833AC30, GB+9833AC39]*/
  0,     6976,  32246, 0,     22614, 0,     0,     0,     36702, 31615,   /*[GB+9833AD30, GB+9833AD39]*/
  0,     0,     16821, 0,     0,     35494, 10374, 24420, 0,     0,       /*[GB+9833AE30, GB+9833AE39]*/
  0,     39684, 11201, 0,     43385, 0,     0,     28311, 39687, 0,       /*[GB+9833AF30, GB+9833AF39]*/
  0,     13496, 0,     25575, 29420, 4617,  43837, 18015, 22741, 0,       /*[GB+9833B030, GB+9833B039]*/
  32237, 41018, 0,     5577,  0,     0,     0,     0,     0,     0,       /*[GB+9833B130, GB+9833B139]*/
  6906,  1606,  14534, 25237, 11151, 4002,  36715, 0,     36993, 0,       /*[GB+9833B230, GB+9833B239]*/
  0,     0,     29701, 30977, 37405, 0,     16322, 0,     43218, 10334,   /*[GB+9833B330, GB+9833B339]*/
  47,    10605, 0,     0,     0,     0,     30294, 35951, 1676,  0,       /*[GB+9833B430, GB+9833B439]*/
  8339,  26451, 20937, 0,     10478, 0,     0,     0,     0,     0,       /*[GB+9833B530, GB+9833B539]*/
  1471,  15977, 4437,  0,     0,     24694, 0,     0,     0,     38439,   /*[GB+9833B630, GB+9833B639]*/
  0,     0,     25354, 38448, 7500,  19388, 0,     1477,  0,     0,       /*[GB+9833B730, GB+9833B739]*/
  0,     0,     0,     12906, 22023, 0,     0,     36277, 0,     0,       /*[GB+9833B830, GB+9833B839]*/
  37026, 0,     0,     0,     18588, 0,     30632, 0,     0,     0,       /*[GB+9833B930, GB+9833B939]*/
  20348, 0,     16182, 21068, 34828, 0,     0,     0,     0,     59,      /*[GB+9833BA30, GB+9833BA39]*/
  0,     0,     30341, 0,     0,     21201, 0,     10426, 0,     0,       /*[GB+9833BB30, GB+9833BB39]*/
  0,     9172,  22571, 10936, 0,     0,     0,     0,     0,     0,       /*[GB+9833BC30, GB+9833BC39]*/
  0,     0,     0,     0,     26748, 0,     0,     0,     0,     2535,    /*[GB+9833BD30, GB+9833BD39]*/
  27754, 0,     0,     0,     0,     24121, 0,     0,     0,     0,       /*[GB+9833BE30, GB+9833BE39]*/
  0,     0,     35013, 34886, 15459, 0,     0,     0,     0,     3565,    /*[GB+9833BF30, GB+9833BF39]*/
  22609, 0,     0,     0,     0,     0,     22675, 0,     0,     0,       /*[GB+9833C030, GB+9833C039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     22238,   /*[GB+9833C130, GB+9833C139]*/
  0,     28245, 0,     6121,  0,     0,     19344, 0,     41787, 37397,   /*[GB+9833C230, GB+9833C239]*/
  0,     8795,  25415, 26173, 15481, 1932,  20982, 41089, 30466, 0,       /*[GB+9833C330, GB+9833C339]*/
  0,     0,     3934,  0,     0,     0,     0,     0,     0,     41135,   /*[GB+9833C430, GB+9833C439]*/
  0,     28718, 39135, 0,     26181, 5775,  31167, 43223, 0,     38723,   /*[GB+9833C530, GB+9833C539]*/
  0,     22617, 1988,  16886, 42236, 7206,  4821,  826,   36441, 0,       /*[GB+9833C630, GB+9833C639]*/
  0,     0,     0,     0,     0,     0,     0,     18132, 0,     0,       /*[GB+9833C730, GB+9833C739]*/
  15493, 33027, 19067, 0,     0,     33163, 28845, 22961, 20912, 15244,   /*[GB+9833C830, GB+9833C839]*/
  0,     0,     0,     0,     35625, 0,     0,     0,     0,     33164,   /*[GB+9833C930, GB+9833C939]*/
  0,     5779,  0,     33736, 259,   5295,  43721, 10826, 4489,  1163,    /*[GB+9833CA30, GB+9833CA39]*/
  0,     6635,  0,     41396, 0,     0,     0,     36444, 0,     0,       /*[GB+9833CB30, GB+9833CB39]*/
  5966,  12775, 0,     0,     0,     21853, 0,     27619, 8826,  0,       /*[GB+9833CC30, GB+9833CC39]*/
  831,   16496, 20987, 26981, 43690, 25239, 34071, 29796, 35634, 22391,   /*[GB+9833CD30, GB+9833CD39]*/
  0,     0,     0,     20790, 0,     0,     0,     0,     40975, 2889,    /*[GB+9833CE30, GB+9833CE39]*/
  28383, 33166, 0,     0,     0,     0,     48,    760,   10899, 0,       /*[GB+9833CF30, GB+9833CF39]*/
  27726, 30295, 31180, 36554, 0,     1939,  27190, 0,     0,     0,       /*[GB+9833D030, GB+9833D039]*/
  0,     0,     0,     0,     0,     0,     0,     27912, 33116, 0,       /*[GB+9833D130, GB+9833D139]*/
  0,     348,   0,     43183, 0,     24324, 30045, 37735, 29797, 6667,    /*[GB+9833D230, GB+9833D239]*/
  29798, 20489, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9833D330, GB+9833D339]*/
  0,     8061,  44009, 0,     22636, 5845,  42157, 5155,  18943, 13323,   /*[GB+9833D430, GB+9833D439]*/
  1993,  4883,  3596,  2223,  0,     27623, 40658, 0,     0,     0,       /*[GB+9833D530, GB+9833D539]*/
  0,     0,     14485, 0,     0,     37467, 20014, 0,     2463,  0,       /*[GB+9833D630, GB+9833D639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     22791, 0,       /*[GB+9833D730, GB+9833D739]*/
  0,     0,     0,     0,     0,     0,     14710, 24698, 43203, 40663,   /*[GB+9833D830, GB+9833D839]*/
  0,     0,     0,     0,     0,     0,     18592, 0,     0,     17326,   /*[GB+9833D930, GB+9833D939]*/
  0,     0,     3955,  0,     0,     41342, 0,     0,     0,     0,       /*[GB+9833DA30, GB+9833DA39]*/
  0,     0,     0,     0,     0,     1591,  33085, 0,     0,     0,       /*[GB+9833DB30, GB+9833DB39]*/
  0,     11472, 0,     0,     0,     24359, 0,     27835, 0,     0,       /*[GB+9833DC30, GB+9833DC39]*/
  0,     0,     22792, 8105,  35810, 0,     28843, 0,     15672, 0,       /*[GB+9833DD30, GB+9833DD39]*/
  0,     0,     15987, 0,     39963, 0,     10414, 10419, 0,     11716,   /*[GB+9833DE30, GB+9833DE39]*/
  15128, 0,     18440, 35093, 7568,  40107, 10556, 0,     0,     15138,   /*[GB+9833DF30, GB+9833DF39]*/
  0,     16492, 0,     43681, 8049,  0,     23887, 9413,  0,     13833,   /*[GB+9833E030, GB+9833E039]*/
  22742, 0,     34385, 18526, 3221,  0,     7573,  9416,  39940, 11765,   /*[GB+9833E130, GB+9833E139]*/
  8059,  349,   0,     0,     0,     7604,  0,     0,     0,     0,       /*[GB+9833E230, GB+9833E239]*/
  16500, 16750, 2156,  0,     0,     37656, 14487, 0,     0,     0,       /*[GB+9833E330, GB+9833E339]*/
  0,     0,     0,     0,     3224,  9418,  27367, 0,     40311, 0,       /*[GB+9833E430, GB+9833E439]*/
  19175, 0,     0,     0,     0,     39965, 40317, 41117, 34887, 0,       /*[GB+9833E530, GB+9833E539]*/
  0,     21807, 0,     0,     0,     10242, 15229, 0,     32197, 0,       /*[GB+9833E630, GB+9833E639]*/
  0,     33800, 32770, 43354, 32877, 17004, 0,     0,     21191, 0,       /*[GB+9833E730, GB+9833E739]*/
  10978, 10845, 7869,  23201, 39380, 11993, 0,     0,     17680, 0,       /*[GB+9833E830, GB+9833E839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9833E930, GB+9833E939]*/
  0,     0,     42890, 10258, 1379,  13693, 31359, 0,     20274, 16602,   /*[GB+9833EA30, GB+9833EA39]*/
  43934, 19338, 24936, 0,     42894, 0,     0,     0,     0,     9805,    /*[GB+9833EB30, GB+9833EB39]*/
  43875, 0,     37061, 29838, 0,     17016, 0,     3053,  0,     37318,   /*[GB+9833EC30, GB+9833EC39]*/
  0,     0,     39860, 0,     0,     0,     0,     0,     0,     29122,   /*[GB+9833ED30, GB+9833ED39]*/
  17404, 38323, 6660,  0,     0,     0,     33318, 38951, 8615,  43149,   /*[GB+9833EE30, GB+9833EE39]*/
  8687,  0,     29953, 18238, 37443, 31974, 0,     0,     0,     0,       /*[GB+9833EF30, GB+9833EF39]*/
  0,     0,     0,     10317, 0,     0,     0,     20284, 36986, 24963,   /*[GB+9833F030, GB+9833F039]*/
  17686, 42907, 0,     10261, 11637, 0,     0,     0,     0,     0,       /*[GB+9833F130, GB+9833F139]*/
  0,     15023, 0,     0,     3787,  0,     0,     15496, 41020, 0,       /*[GB+9833F230, GB+9833F239]*/
  21719, 0,     0,     0,     0,     0,     0,     0,     42859, 8976,    /*[GB+9833F330, GB+9833F339]*/
  35555, 13257, 0,     0,     0,     27982, 21206, 40479, 0,     0,       /*[GB+9833F430, GB+9833F439]*/
  8978,  29059, 0,     0,     20297, 39386, 14240, 23390, 0,     0,       /*[GB+9833F530, GB+9833F539]*/
  0,     0,     15673, 39387, 0,     0,     38440, 30982, 32705, 3263,    /*[GB+9833F630, GB+9833F639]*/
  19234, 9864,  0,     18540, 18988, 0,     0,     14320, 37142, 1684,    /*[GB+9833F730, GB+9833F739]*/
  24998, 0,     0,     0,     38449, 0,     0,     1833,  21350, 2763,    /*[GB+9833F830, GB+9833F839]*/
  0,     8979,  0,     0,     0,     26549, 5977,  17887, 0,     0,       /*[GB+9833F930, GB+9833F939]*/
  0,     0,     19407, 0,     0,     0,     12082, 26561, 0,     20513,   /*[GB+9833FA30, GB+9833FA39]*/
  0,     17043, 0,     0,     0,     26329, 39665, 11994, 0,     0,       /*[GB+9833FB30, GB+9833FB39]*/
  0,     17973, 0,     27075, 34925, 29644, 0,     0,     6894,  1728,    /*[GB+9833FC30, GB+9833FC39]*/
  0,     0,     0,     0,     23202, 0,     11195, 23786, 0,     8036,    /*[GB+9833FD30, GB+9833FD39]*/
  22285, 0,     0,     8421,  0,     0,     11718, 32497, 27900, 0,       /*[GB+9833FE30, GB+9833FE39]*/
  14292, 33390, 0,     0,     0,     0,     0,     0,     0,     8796,    /*[GB+98348130, GB+98348139]*/
  0,     0,     0,     0,     0,     0,     0,     1904,  43048, 0,       /*[GB+98348230, GB+98348239]*/
  42903, 3663,  0,     28517, 11633, 2195,  0,     0,     0,     0,       /*[GB+98348330, GB+98348339]*/
  26913, 0,     0,     0,     0,     0,     0,     0,     22292, 0,       /*[GB+98348430, GB+98348439]*/
  18441, 0,     0,     13609, 0,     0,     20688, 0,     0,     21306,   /*[GB+98348530, GB+98348539]*/
  22186, 42707, 34449, 12500, 9965,  15328, 0,     22293, 0,     27365,   /*[GB+98348630, GB+98348639]*/
  0,     33324, 0,     27320, 0,     41092, 0,     0,     0,     0,       /*[GB+98348730, GB+98348739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98348830, GB+98348839]*/
  0,     23401, 35102, 0,     14230, 0,     0,     0,     0,     0,       /*[GB+98348930, GB+98348939]*/
  0,     18241, 29128, 38396, 5967,  26069, 15497, 0,     0,     42506,   /*[GB+98348A30, GB+98348A39]*/
  0,     17653, 2287,  16902, 38397, 0,     2127,  0,     0,     0,       /*[GB+98348B30, GB+98348B39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     3893,    /*[GB+98348C30, GB+98348C39]*/
  0,     0,     0,     0,     0,     14235, 0,     33215, 36398, 0,       /*[GB+98348D30, GB+98348D39]*/
  0,     0,     0,     31993, 20913, 0,     7882,  43407, 0,     25240,   /*[GB+98348E30, GB+98348E39]*/
  0,     30348, 24984, 0,     23966, 1738,  8209,  31838, 12018, 0,       /*[GB+98348F30, GB+98348F39]*/
  0,     7487,  0,     0,     0,     7488,  0,     0,     6523,  0,       /*[GB+98349030, GB+98349039]*/
  27325, 0,     0,     0,     0,     0,     7489,  0,     0,     0,       /*[GB+98349130, GB+98349139]*/
  0,     0,     0,     0,     37003, 29659, 0,     0,     0,     0,       /*[GB+98349230, GB+98349239]*/
  0,     38902, 22744, 22745, 37919, 0,     0,     0,     0,     0,       /*[GB+98349330, GB+98349339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98349430, GB+98349439]*/
  0,     21465, 0,     0,     0,     0,     37505, 24519, 4912,  0,       /*[GB+98349530, GB+98349539]*/
  32944, 0,     10607, 38819, 0,     31498, 42526, 7944,  39278, 0,       /*[GB+98349630, GB+98349639]*/
  0,     17031, 260,   530,   0,     10831, 0,     0,     0,     0,       /*[GB+98349730, GB+98349739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     22478, 22446,   /*[GB+98349830, GB+98349839]*/
  0,     0,     15035, 0,     0,     32616, 0,     0,     0,     0,       /*[GB+98349930, GB+98349939]*/
  0,     0,     0,     0,     0,     0,     0,     4308,  0,     30297,   /*[GB+98349A30, GB+98349A39]*/
  30868, 25000, 30782, 32837, 9010,  0,     9346,  0,     10833, 0,       /*[GB+98349B30, GB+98349B39]*/
  22394, 9347,  0,     0,     0,     0,     41938, 0,     0,     0,       /*[GB+98349C30, GB+98349C39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98349D30, GB+98349D39]*/
  0,     0,     0,     0,     0,     30351, 0,     0,     0,     29568,   /*[GB+98349E30, GB+98349E39]*/
  0,     22849, 27696, 0,     0,     0,     0,     0,     0,     0,       /*[GB+98349F30, GB+98349F39]*/
  0,     0,     0,     0,     0,     35118, 0,     0,     0,     43850,   /*[GB+9834A030, GB+9834A039]*/
  39600, 22850, 24816, 0,     0,     28474, 0,     0,     0,     0,       /*[GB+9834A130, GB+9834A139]*/
  0,     0,     2196,  0,     0,     0,     0,     0,     0,     0,       /*[GB+9834A230, GB+9834A239]*/
  0,     0,     0,     0,     0,     0,     0,     17165, 18178, 0,       /*[GB+9834A330, GB+9834A339]*/
  9903,  0,     0,     0,     0,     0,     0,     0,     36884, 2292,    /*[GB+9834A430, GB+9834A439]*/
  14121, 0,     33333, 2619,  19171, 43164, 0,     12448, 0,     31634,   /*[GB+9834A530, GB+9834A539]*/
  41057, 0,     11235, 0,     0,     10516, 0,     0,     0,     0,       /*[GB+9834A630, GB+9834A639]*/
  0,     0,     0,     0,     0,     15075, 0,     0,     0,     6434,    /*[GB+9834A730, GB+9834A739]*/
  0,     38919, 0,     0,     0,     0,     42537, 0,     0,     0,       /*[GB+9834A830, GB+9834A839]*/
  0,     0,     32843, 0,     13541, 27232, 0,     0,     413,   5622,    /*[GB+9834A930, GB+9834A939]*/
  20504, 37657, 32618, 0,     0,     0,     0,     0,     25612, 0,       /*[GB+9834AA30, GB+9834AA39]*/
  0,     0,     0,     0,     0,     9350,  13542, 0,     0,     0,       /*[GB+9834AB30, GB+9834AB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     43794, 0,       /*[GB+9834AC30, GB+9834AC39]*/
  37932, 0,     0,     41058, 0,     19107, 0,     0,     0,     0,       /*[GB+9834AD30, GB+9834AD39]*/
  0,     0,     37660, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9834AE30, GB+9834AE39]*/
  41222, 0,     0,     0,     4102,  1688,  0,     0,     0,     0,       /*[GB+9834AF30, GB+9834AF39]*/
  35959, 39182, 0,     0,     0,     0,     0,     35311, 32710, 18595,   /*[GB+9834B030, GB+9834B039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9834B130, GB+9834B139]*/
  0,     2162,  13994, 2919,  17167, 0,     0,     26073, 0,     0,       /*[GB+9834B230, GB+9834B239]*/
  0,     0,     0,     29368, 33782, 26563, 0,     0,     0,     0,       /*[GB+9834B330, GB+9834B339]*/
  0,     0,     20515, 0,     0,     5644,  22876, 18257, 0,     540,     /*[GB+9834B430, GB+9834B439]*/
  0,     7420,  8405,  8756,  12732, 41085, 175,   0,     0,     26341,   /*[GB+9834B530, GB+9834B539]*/
  4070,  21682, 36395, 31836, 22739, 11894, 29125, 0,     35758, 42645,   /*[GB+9834B630, GB+9834B639]*/
  16294, 0,     39181, 0,     0,     0,     34814, 34815, 4746,  3849,    /*[GB+9834B730, GB+9834B739]*/
  0,     6254,  38267, 0,     6470,  0,     0,     10799, 32361, 0,       /*[GB+9834B830, GB+9834B839]*/
  37984, 864,   38295, 0,     0,     0,     0,     36391, 0,     0,       /*[GB+9834B930, GB+9834B939]*/
  0,     0,     34672, 0,     0,     0,     0,     0,     0,     1169,    /*[GB+9834BA30, GB+9834BA39]*/
  0,     0,     14663, 9242,  40881, 0,     41851, 11719, 33687, 35350,   /*[GB+9834BB30, GB+9834BB39]*/
  34840, 8797,  42640, 0,     21074, 35351, 25565, 0,     0,     0,       /*[GB+9834BC30, GB+9834BC39]*/
  8149,  625,   12063, 0,     0,     0,     0,     0,     0,     14293,   /*[GB+9834BD30, GB+9834BD39]*/
  0,     867,   36438, 37319, 0,     0,     0,     0,     0,     33992,   /*[GB+9834BE30, GB+9834BE39]*/
  0,     15654, 0,     26601, 40272, 30495, 31693, 22119, 41014, 26510,   /*[GB+9834BF30, GB+9834BF39]*/
  0,     8430,  0,     26511, 3886,  0,     39265, 0,     0,     0,       /*[GB+9834C030, GB+9834C039]*/
  0,     0,     0,     0,     0,     0,     0,     31081, 0,     0,       /*[GB+9834C130, GB+9834C139]*/
  32233, 0,     0,     32153, 0,     4190,  0,     9248,  0,     0,       /*[GB+9834C230, GB+9834C239]*/
  0,     0,     0,     0,     0,     9249,  26512, 0,     0,     0,       /*[GB+9834C330, GB+9834C339]*/
  0,     15655, 0,     0,     6418,  37911, 28728, 38380, 0,     10262,   /*[GB+9834C430, GB+9834C439]*/
  14231, 0,     29696, 26186, 0,     7479,  12254, 12476, 36344, 0,       /*[GB+9834C530, GB+9834C539]*/
  0,     15495, 0,     41126, 3888,  17775, 0,     24964, 35628, 0,       /*[GB+9834C630, GB+9834C639]*/
  12435, 31627, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9834C730, GB+9834C739]*/
  0,     0,     0,     0,     8815,  0,     0,     36345, 14681, 21521,   /*[GB+9834C830, GB+9834C839]*/
  39898, 0,     0,     0,     0,     0,     0,     0,     43200, 14305,   /*[GB+9834C930, GB+9834C939]*/
  42507, 3589,  0,     14696, 34859, 3400,  18146, 40279, 0,     42508,   /*[GB+9834CA30, GB+9834CA39]*/
  17741, 33216, 26524, 0,     32482, 0,     37730, 38398, 17654, 0,       /*[GB+9834CB30, GB+9834CB39]*/
  31675, 188,   15954, 36995, 0,     0,     0,     15663, 0,     41858,   /*[GB+9834CC30, GB+9834CC39]*/
  0,     42509, 20934, 0,     0,     0,     35672, 0,     0,     4432,    /*[GB+9834CD30, GB+9834CD39]*/
  4149,  0,     0,     0,     0,     0,     25584, 7942,  3106,  0,       /*[GB+9834CE30, GB+9834CE39]*/
  21641, 29978, 372,   0,     8693,  0,     0,     0,     32935, 42077,   /*[GB+9834CF30, GB+9834CF39]*/
  21855, 3063,  36725, 39923, 0,     8827,  30438, 13321, 0,     8054,    /*[GB+9834D030, GB+9834D039]*/
  0,     0,     11758, 15419, 0,     39692, 0,     0,     0,     0,       /*[GB+9834D130, GB+9834D139]*/
  0,     24988, 21141, 0,     0,     1741,  10511, 0,     0,     0,       /*[GB+9834D230, GB+9834D239]*/
  21963, 0,     32424, 32425, 0,     0,     0,     0,     0,     0,       /*[GB+9834D330, GB+9834D339]*/
  0,     0,     0,     0,     0,     0,     15031, 39924, 34392, 0,       /*[GB+9834D430, GB+9834D439]*/
  0,     4524,  13266, 0,     34640, 23851, 8831,  0,     0,     0,       /*[GB+9834D530, GB+9834D539]*/
  19547, 0,     27592, 36159, 43792, 0,     14705, 0,     0,     41212,   /*[GB+9834D630, GB+9834D639]*/
  0,     39590, 39591, 21145, 21740, 21079, 7163,  8832,  0,     0,       /*[GB+9834D730, GB+9834D739]*/
  0,     0,     0,     0,     0,     14112, 0,     0,     0,     0,       /*[GB+9834D830, GB+9834D839]*/
  0,     0,     0,     0,     0,     0,     21721, 0,     305,   0,       /*[GB+9834D930, GB+9834D939]*/
  0,     0,     0,     16497, 0,     0,     0,     0,     0,     0,       /*[GB+9834DA30, GB+9834DA39]*/
  0,     0,     11527, 0,     0,     0,     9691,  0,     33769, 0,       /*[GB+9834DB30, GB+9834DB39]*/
  27192, 9255,  23629, 40123, 34195, 0,     0,     27913, 24041, 0,       /*[GB+9834DC30, GB+9834DC39]*/
  26071, 39051, 26536, 0,     9590,  0,     9348,  34283, 0,     29985,   /*[GB+9834DD30, GB+9834DD39]*/
  0,     0,     0,     636,   25001, 11563, 0,     0,     0,     33770,   /*[GB+9834DE30, GB+9834DE39]*/
  0,     33271, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9834DF30, GB+9834DF39]*/
  0,     0,     0,     0,     0,     0,     0,     24817, 11888, 10203,   /*[GB+9834E030, GB+9834E039]*/
  0,     5975,  28202, 20751, 21906, 13533, 33840, 34287, 13271, 3109,    /*[GB+9834E130, GB+9834E139]*/
  16663, 3950,  0,     0,     0,     0,     0,     0,     41378, 24313,   /*[GB+9834E230, GB+9834E239]*/
  29987, 20390, 18179, 21402, 3951,  31207, 0,     19389, 0,     3900,    /*[GB+9834E330, GB+9834E339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     3952,  0,       /*[GB+9834E430, GB+9834E439]*/
  0,     3901,  0,     0,     0,     22970, 0,     39074, 0,     0,       /*[GB+9834E530, GB+9834E539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9834E630, GB+9834E639]*/
  0,     0,     21608, 0,     29136, 0,     34040, 0,     7505,  43540,   /*[GB+9834E730, GB+9834E739]*/
  0,     0,     14708, 13093, 11477, 28659, 18573, 0,     24820, 0,       /*[GB+9834E830, GB+9834E839]*/
  42714, 29574, 26550, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9834E930, GB+9834E939]*/
  0,     35123, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9834EA30, GB+9834EA39]*/
  0,     0,     0,     29575, 0,     0,     0,     35773, 0,     0,       /*[GB+9834EB30, GB+9834EB39]*/
  0,     0,     8837,  0,     0,     0,     0,     0,     0,     0,       /*[GB+9834EC30, GB+9834EC39]*/
  0,     0,     0,     0,     0,     0,     0,     23268, 36557, 19559,   /*[GB+9834ED30, GB+9834ED39]*/
  13696, 10204, 17039, 3904,  2464,  4207,  0,     10433, 0,     5510,    /*[GB+9834EE30, GB+9834EE39]*/
  0,     0,     0,     14123, 0,     5623,  11565, 0,     16664, 0,       /*[GB+9834EF30, GB+9834EF39]*/
  4310,  25363, 148,   0,     24211, 0,     0,     0,     0,     0,       /*[GB+9834F030, GB+9834F039]*/
  0,     36404, 15039, 21405, 0,     1835,  17525, 9837,  4101,  0,       /*[GB+9834F130, GB+9834F139]*/
  0,     39058, 9904,  39607, 42806, 0,     0,     0,     0,     0,       /*[GB+9834F230, GB+9834F239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     2360,    /*[GB+9834F330, GB+9834F339]*/
  0,     19561, 2227,  20613, 14712, 41743, 21806, 10120, 0,     13668,   /*[GB+9834F430, GB+9834F439]*/
  0,     17327, 0,     0,     0,     0,     0,     0,     29061, 0,       /*[GB+9834F530, GB+9834F539]*/
  0,     10434, 0,     0,     21406, 25434, 17433, 0,     11169, 31888,   /*[GB+9834F630, GB+9834F639]*/
  0,     0,     0,     0,     0,     12449, 0,     0,     0,     0,       /*[GB+9834F730, GB+9834F739]*/
  0,     0,     11336, 0,     0,     0,     20252, 0,     0,     13669,   /*[GB+9834F830, GB+9834F839]*/
  0,     0,     0,     0,     0,     10205, 0,     0,     37741, 12083,   /*[GB+9834F930, GB+9834F939]*/
  0,     0,     0,     0,     0,     0,     20516, 0,     0,     0,       /*[GB+9834FA30, GB+9834FA39]*/
  16005, 0,     0,     0,     10078, 0,     0,     26709, 22972, 0,       /*[GB+9834FB30, GB+9834FB39]*/
  0,     20805, 0,     0,     40435, 33000, 3823,  3829,  42040, 0,       /*[GB+9834FC30, GB+9834FC39]*/
  4672,  43194, 11933, 14606, 26486, 31732, 21842, 21129, 39534, 332,     /*[GB+9834FD30, GB+9834FD39]*/
  626,   0,     24026, 41740, 19985, 3862,  31673, 0,     14528, 0,       /*[GB+9834FE30, GB+9834FE39]*/
  41197, 0,     0,     0,     0,     0,     14831, 20003, 0,     0,       /*[GB+98358130, GB+98358139]*/
  14105, 30686, 3064,  0,     4759,  0,     0,     32288, 4913,  16257,   /*[GB+98358230, GB+98358239]*/
  0,     1610,  0,     0,     0,     0,     0,     0,     32989, 0,       /*[GB+98358330, GB+98358339]*/
  0,     25357, 0,     0,     0,     9048,  0,     0,     12674, 0,       /*[GB+98358430, GB+98358439]*/
  9015,  0,     13544, 9050,  0,     12040, 0,     0,     0,     29645,   /*[GB+98358530, GB+98358539]*/
  0,     0,     8424,  0,     0,     0,     0,     24100, 0,     2294,    /*[GB+98358630, GB+98358639]*/
  0,     0,     0,     0,     1051,  0,     0,     0,     0,     0,       /*[GB+98358730, GB+98358739]*/
  28844, 0,     0,     12133, 0,     10190, 0,     0,     23501, 0,       /*[GB+98358830, GB+98358839]*/
  0,     0,     0,     0,     0,     0,     0,     23745, 0,     38958,   /*[GB+98358930, GB+98358939]*/
  31534, 34977, 0,     43051, 0,     39578, 0,     0,     0,     0,       /*[GB+98358A30, GB+98358A39]*/
  0,     14097, 0,     0,     26579, 32426, 17161, 43220, 0,     39925,   /*[GB+98358B30, GB+98358B39]*/
  10512, 24482, 43942, 43846, 0,     42866, 23366, 42867, 0,     0,       /*[GB+98358C30, GB+98358C39]*/
  0,     0,     24287, 21475, 0,     0,     0,     0,     1479,  29988,   /*[GB+98358D30, GB+98358D39]*/
  0,     0,     0,     24697, 21487, 0,     0,     0,     0,     0,       /*[GB+98358E30, GB+98358E39]*/
  0,     0,     0,     37659, 39696, 0,     39957, 0,     42964, 0,       /*[GB+98358F30, GB+98358F39]*/
  0,     19402, 0,     0,     0,     0,     0,     0,     23313, 0,       /*[GB+98359030, GB+98359039]*/
  0,     0,     19327, 0,     18955, 0,     32076, 0,     1209,  22389,   /*[GB+98359130, GB+98359139]*/
  26502, 0,     24101, 6684,  23061, 32021, 0,     13024, 0,     19345,   /*[GB+98359230, GB+98359239]*/
  0,     0,     0,     16744, 29955, 0,     0,     0,     31694, 0,       /*[GB+98359330, GB+98359339]*/
  0,     0,     0,     0,     22404, 40840, 0,     0,     31648, 9332,    /*[GB+98359430, GB+98359439]*/
  6979,  0,     0,     14682, 0,     31253, 31398, 3735,  26442, 0,       /*[GB+98359530, GB+98359539]*/
  27910, 0,     2129,  18147, 0,     20597, 0,     18932, 29280, 5852,    /*[GB+98359630, GB+98359639]*/
  0,     3670,  21788, 0,     30915, 12728, 0,     22985, 0,     16103,   /*[GB+98359730, GB+98359739]*/
  11880, 0,     20171, 37644, 0,     22133, 3237,  0,     0,     0,       /*[GB+98359830, GB+98359839]*/
  30308, 0,     16415, 8833,  0,     3222,  0,     0,     30296, 0,       /*[GB+98359930, GB+98359939]*/
  0,     26245, 0,     0,     0,     29986, 0,     40480, 0,     30299,   /*[GB+98359A30, GB+98359A39]*/
  16715, 2150,  0,     19860, 22147, 0,     18739, 36144, 30300, 20801,   /*[GB+98359B30, GB+98359B39]*/
  1480,  0,     0,     0,     0,     0,     0,     0,     31184, 0,       /*[GB+98359C30, GB+98359C39]*/
  18706, 30630, 28173, 0,     26556, 6838,  12152, 27756, 0,     0,       /*[GB+98359D30, GB+98359D39]*/
  0,     16956, 0,     0,     0,     23120, 0,     23121, 20517, 44039,   /*[GB+98359E30, GB+98359E39]*/
  38162, 34527, 3211,  31424, 17375, 0,     0,     0,     0,     0,       /*[GB+98359F30, GB+98359F39]*/
  29468, 0,     21518, 0,     0,     21443, 0,     0,     0,     32418,   /*[GB+9835A030, GB+9835A039]*/
  0,     1052,  25569, 0,     22020, 0,     0,     19539, 33217, 0,       /*[GB+9835A130, GB+9835A139]*/
  21170, 0,     31677, 0,     0,     22242, 0,     21147, 0,     0,       /*[GB+9835A230, GB+9835A239]*/
  0,     0,     44084, 0,     32019, 16317, 32022, 0,     0,     0,       /*[GB+9835A330, GB+9835A339]*/
  3306,  41983, 0,     3942,  31205, 0,     42495, 2878,  2883,  0,       /*[GB+9835A430, GB+9835A439]*/
  0,     26041, 31994, 12503, 26814, 41208, 13835, 2890,  0,     12324,   /*[GB+9835A530, GB+9835A539]*/
  0,     12444, 3740,  0,     12325, 11281, 40419, 0,     31848, 1513,    /*[GB+9835A630, GB+9835A639]*/
  0,     12331, 40420, 0,     0,     0,     0,     22148, 9536,  0,       /*[GB+9835A730, GB+9835A739]*/
  9537,  0,     10345, 0,     2913,  0,     26820, 27465, 16957, 24325,   /*[GB+9835A830, GB+9835A839]*/
  0,     0,     27342, 0,     0,     0,     23227, 11870, 14872, 22842,   /*[GB+9835A930, GB+9835A939]*/
  874,   0,     22778, 0,     1438,  11743, 18133, 0,     0,     42860,   /*[GB+9835AA30, GB+9835AA39]*/
  22325, 0,     26725, 8319,  1217,  18161, 1773,  22875, 6596,  0,       /*[GB+9835AB30, GB+9835AB39]*/
  0,     18703, 22852, 18707, 20639, 41627, 0,     0,     0,     13615,   /*[GB+9835AC30, GB+9835AC39]*/
  37933, 0,     19753, 0,     38271, 0,     0,     0,     0,     5398,    /*[GB+9835AD30, GB+9835AD39]*/
  6898,  0,     0,     0,     0,     37989, 0,     30528, 10847, 29272,   /*[GB+9835AE30, GB+9835AE39]*/
  0,     0,     0,     0,     32363, 0,     29956, 37402, 7628,  0,       /*[GB+9835AF30, GB+9835AF39]*/
  41854, 0,     0,     0,     0,     0,     0,     0,     6724,  13820,   /*[GB+9835B030, GB+9835B039]*/
  0,     0,     0,     23673, 12439, 0,     2759,  28047, 19928, 32929,   /*[GB+9835B130, GB+9835B139]*/
  39917, 0,     37005, 32427, 25787, 20453, 39926, 0,     0,     31819,   /*[GB+9835B230, GB+9835B239]*/
  32936, 0,     32039, 0,     0,     12734, 1022,  25866, 0,     14837,   /*[GB+9835B330, GB+9835B339]*/
  17387, 0,     42246, 39592, 0,     4760,  0,     0,     0,     0,       /*[GB+9835B430, GB+9835B439]*/
  0,     0,     37650, 6429,  2761,  6224,  20797, 0,     5874,  32617,   /*[GB+9835B530, GB+9835B539]*/
  22910, 17992, 1866,  33272, 0,     41796, 0,     0,     27269, 32304,   /*[GB+9835B630, GB+9835B639]*/
  21606, 14784, 0,     11528, 0,     41153, 27346, 43539, 23630, 0,       /*[GB+9835B730, GB+9835B739]*/
  22149, 0,     21609, 0,     24818, 22150, 0,     0,     42081, 637,     /*[GB+9835B830, GB+9835B839]*/
  0,     21800, 0,     20179, 0,     24821, 3954,  0,     18185, 38454,   /*[GB+9835B930, GB+9835B939]*/
  0,     0,     5879,  5156,  0,     7048,  20692, 36892, 0,     40934,   /*[GB+9835BA30, GB+9835BA39]*/
  36893, 7049,  24652, 0,     0,     40305, 0,     0,     0,     12909,   /*[GB+9835BB30, GB+9835BB39]*/
  20752, 13991, 23269, 25961, 0,     25962, 0,     0,     37661, 0,       /*[GB+9835BC30, GB+9835BC39]*/
  30504, 0,     0,     0,     0,     13860, 0,     0,     2726,  0,       /*[GB+9835BD30, GB+9835BD39]*/
  5592,  30994, 18597, 0,     0,     22930, 2920,  40687, 0,     0,       /*[GB+9835BE30, GB+9835BE39]*/
  0,     0,     39974, 0,     43973, 1515,  0,     0,     4231,  0,       /*[GB+9835BF30, GB+9835BF39]*/
  0,     0,     0,     0,     0,     0,     26876, 0,     26882, 37066,   /*[GB+9835C030, GB+9835C039]*/
  2168,  0,     0,     0,     0,     4977,  0,     0,     0,     0,       /*[GB+9835C130, GB+9835C139]*/
  0,     0,     0,     0,     0,     0,     21476, 0,     16439, 0,       /*[GB+9835C230, GB+9835C239]*/
  4983,  0,     0,     0,     0,     0,     0,     33843, 0,     20640,   /*[GB+9835C330, GB+9835C339]*/
  0,     28740, 6145,  0,     0,     28741, 0,     0,     6480,  15158,   /*[GB+9835C430, GB+9835C439]*/
  0,     40100, 8972,  0,     0,     0,     12675, 13044, 0,     23929,   /*[GB+9835C530, GB+9835C539]*/
  0,     5835,  0,     1131,  0,     0,     23675, 30440, 0,     40024,   /*[GB+9835C630, GB+9835C639]*/
  9144,  40025, 0,     0,     0,     0,     0,     13614, 0,     0,       /*[GB+9835C730, GB+9835C739]*/
  0,     16593, 0,     0,     6639,  0,     0,     34397, 25180, 0,       /*[GB+9835C830, GB+9835C839]*/
  27680, 0,     0,     0,     38735, 0,     0,     19722, 0,     0,       /*[GB+9835C930, GB+9835C939]*/
  30954, 19727, 0,     0,     27076, 0,     0,     38731, 24479, 24646,   /*[GB+9835CA30, GB+9835CA39]*/
  40112, 27226, 0,     7872,  0,     0,     0,     10849, 21456, 11752,   /*[GB+9835CB30, GB+9835CB39]*/
  43531, 0,     19368, 15161, 27228, 0,     0,     0,     24483, 10107,   /*[GB+9835CC30, GB+9835CC39]*/
  7602,  0,     0,     0,     0,     0,     0,     0,     0,     4980,    /*[GB+9835CD30, GB+9835CD39]*/
  16104, 0,     0,     0,     0,     0,     0,     0,     35649, 0,       /*[GB+9835CE30, GB+9835CE39]*/
  7575,  31182, 0,     37143, 0,     146,   11767, 33841, 0,     11768,   /*[GB+9835CF30, GB+9835CF39]*/
  0,     29571, 18251, 0,     38455, 9835,  0,     30784, 0,     0,       /*[GB+9835D030, GB+9835D039]*/
  0,     0,     0,     0,     26821, 43696, 18191, 0,     14489, 0,       /*[GB+9835D130, GB+9835D139]*/
  0,     7784,  24699, 29581, 0,     14714, 19966, 18598, 2921,  0,       /*[GB+9835D230, GB+9835D239]*/
  37440, 0,     0,     0,     0,     12440, 0,     0,     0,     11475,   /*[GB+9835D330, GB+9835D339]*/
  6131,  26363, 15964, 0,     24203, 0,     0,     0,     16978, 0,       /*[GB+9835D430, GB+9835D439]*/
  0,     33897, 31014, 0,     0,     35955, 0,     0,     0,     0,       /*[GB+9835D530, GB+9835D539]*/
  0,     1549,  0,     0,     11028, 18709, 0,     0,     29992, 18989,   /*[GB+9835D630, GB+9835D639]*/
  0,     0,     0,     0,     14839, 0,     0,     0,     0,     18599,   /*[GB+9835D730, GB+9835D739]*/
  2922,  0,     0,     24957, 24967, 0,     0,     43536, 0,     43538,   /*[GB+9835D830, GB+9835D839]*/
  0,     25006, 0,     25009, 43544, 0,     41116, 41118, 23543, 0,       /*[GB+9835D930, GB+9835D939]*/
  15414, 0,     0,     36865, 0,     0,     10878, 22392, 0,     0,       /*[GB+9835DA30, GB+9835DA39]*/
  0,     0,     0,     0,     36872, 0,     41333, 28846, 42153, 0,       /*[GB+9835DB30, GB+9835DB39]*/
  0,     0,     0,     40976, 0,     0,     0,     0,     0,     0,       /*[GB+9835DC30, GB+9835DC39]*/
  27231, 40977, 0,     38443, 21944, 36477, 0,     42530, 0,     17240,   /*[GB+9835DD30, GB+9835DD39]*/
  34159, 0,     24255, 30452, 0,     38006, 0,     0,     35638, 41114,   /*[GB+9835DE30, GB+9835DE39]*/
  0,     7212,  7496,  0,     0,     0,     38614, 0,     7501,  30001,   /*[GB+9835DF30, GB+9835DF39]*/
  192,   5182,  0,     32161, 0,     32162, 34160, 0,     0,     4623,    /*[GB+9835E030, GB+9835E039]*/
  0,     30002, 0,     0,     0,     0,     0,     0,     0,     13538,   /*[GB+9835E130, GB+9835E139]*/
  25475, 43851, 41115, 40423, 41338, 0,     37927, 0,     0,     0,       /*[GB+9835E230, GB+9835E239]*/
  0,     0,     36478, 40194, 0,     0,     0,     0,     0,     11166,   /*[GB+9835E330, GB+9835E339]*/
  25243, 0,     0,     0,     2683,  0,     0,     0,     0,     23481,   /*[GB+9835E430, GB+9835E439]*/
  0,     0,     0,     0,     11769, 0,     37027, 0,     25871, 2160,    /*[GB+9835E530, GB+9835E539]*/
  25585, 14785, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9835E630, GB+9835E639]*/
  0,     0,     0,     22854, 0,     0,     3007,  25988, 0,     40718,   /*[GB+9835E730, GB+9835E739]*/
  0,     0,     38683, 35312, 0,     25010, 14126, 0,     0,     35774,   /*[GB+9835E830, GB+9835E839]*/
  0,     40935, 0,     7513,  0,     0,     0,     43798, 38010, 0,       /*[GB+9835E930, GB+9835E939]*/
  0,     0,     0,     42160, 18600, 0,     0,     18601, 38617, 0,       /*[GB+9835EA30, GB+9835EA39]*/
  0,     18712, 0,     2923,  0,     16006, 0,     40542, 0,     0,       /*[GB+9835EB30, GB+9835EB39]*/
  0,     0,     0,     0,     41119, 24244, 0,     19685, 0,     19686,   /*[GB+9835EC30, GB+9835EC39]*/
  0,     0,     0,     0,     19693, 0,     0,     0,     0,     19708,   /*[GB+9835ED30, GB+9835ED39]*/
  0,     0,     19714, 0,     20939, 0,     0,     41630, 0,     0,       /*[GB+9835EE30, GB+9835EE39]*/
  0,     0,     0,     0,     9044,  9628,  26883, 4981,  14319, 0,       /*[GB+9835EF30, GB+9835EF39]*/
  0,     0,     0,     0,     0,     2170,  0,     43062, 26248, 37067,   /*[GB+9835F030, GB+9835F039]*/
  0,     0,     0,     0,     34989, 0,     12477, 26458, 0,     0,       /*[GB+9835F130, GB+9835F139]*/
  0,     0,     19434, 0,     38613, 0,     0,     0,     24213, 0,       /*[GB+9835F230, GB+9835F239]*/
  0,     0,     18711, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9835F330, GB+9835F339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9835F430, GB+9835F439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9835F530, GB+9835F539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9835F630, GB+9835F639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9835F730, GB+9835F739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9835F830, GB+9835F839]*/
  0,     0,     25756, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9835F930, GB+9835F939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9835FA30, GB+9835FA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9835FB30, GB+9835FB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9835FC30, GB+9835FC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9835FD30, GB+9835FD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9835FE30, GB+9835FE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98368130, GB+98368139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98368230, GB+98368239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98368330, GB+98368339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98368430, GB+98368439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98368530, GB+98368539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98368630, GB+98368639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98368730, GB+98368739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98368830, GB+98368839]*/
  0,     0,     0,     0,     0,     7136,  0,     0,     0,     0,       /*[GB+98368930, GB+98368939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98368A30, GB+98368A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98368B30, GB+98368B39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98368C30, GB+98368C39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98368D30, GB+98368D39]*/
  0,     0,     0,     0,     13752, 0,     0,     0,     0,     0,       /*[GB+98368E30, GB+98368E39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     13374,   /*[GB+98368F30, GB+98368F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98369030, GB+98369039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98369130, GB+98369139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98369230, GB+98369239]*/
  0,     0,     0,     0,     16694, 0,     0,     0,     0,     0,       /*[GB+98369330, GB+98369339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     36783,   /*[GB+98369430, GB+98369439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98369530, GB+98369539]*/
  0,     0,     0,     0,     0,     41827, 0,     27511, 0,     0,       /*[GB+98369630, GB+98369639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98369730, GB+98369739]*/
  0,     0,     0,     0,     0,     3240,  0,     20396, 0,     0,       /*[GB+98369830, GB+98369839]*/
  11816, 0,     0,     0,     0,     5595,  0,     0,     25697, 0,       /*[GB+98369930, GB+98369939]*/
  0,     20713, 0,     0,     0,     0,     554,   0,     0,     36777,   /*[GB+98369A30, GB+98369A39]*/
  0,     0,     0,     41186, 0,     0,     0,     0,     0,     0,       /*[GB+98369B30, GB+98369B39]*/
  0,     0,     0,     0,     0,     11076, 0,     0,     17068, 0,       /*[GB+98369C30, GB+98369C39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98369D30, GB+98369D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98369E30, GB+98369E39]*/
  0,     0,     0,     0,     0,     0,     17400, 0,     0,     0,       /*[GB+98369F30, GB+98369F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836A030, GB+9836A039]*/
  0,     0,     0,     0,     17460, 0,     0,     0,     0,     0,       /*[GB+9836A130, GB+9836A139]*/
  29264, 0,     0,     4616,  0,     0,     0,     0,     0,     0,       /*[GB+9836A230, GB+9836A239]*/
  0,     0,     0,     0,     17488, 0,     0,     0,     0,     0,       /*[GB+9836A330, GB+9836A339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     20314, 0,       /*[GB+9836A430, GB+9836A439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836A530, GB+9836A539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836A630, GB+9836A639]*/
  0,     0,     6533,  0,     0,     0,     0,     0,     0,     0,       /*[GB+9836A730, GB+9836A739]*/
  0,     0,     0,     0,     12703, 0,     0,     0,     0,     0,       /*[GB+9836A830, GB+9836A839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836A930, GB+9836A939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836AA30, GB+9836AA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836AB30, GB+9836AB39]*/
  0,     0,     0,     0,     0,     19785, 0,     0,     0,     0,       /*[GB+9836AC30, GB+9836AC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836AD30, GB+9836AD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836AE30, GB+9836AE39]*/
  0,     0,     0,     18977, 0,     0,     19134, 0,     0,     0,       /*[GB+9836AF30, GB+9836AF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836B030, GB+9836B039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836B130, GB+9836B139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836B230, GB+9836B239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836B330, GB+9836B339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836B430, GB+9836B439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836B530, GB+9836B539]*/
  0,     0,     0,     0,     0,     0,     27094, 0,     0,     0,       /*[GB+9836B630, GB+9836B639]*/
  0,     0,     0,     0,     0,     0,     0,     29338, 0,     0,       /*[GB+9836B730, GB+9836B739]*/
  0,     0,     23132, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9836B830, GB+9836B839]*/
  0,     0,     0,     0,     0,     0,     0,     6769,  0,     0,       /*[GB+9836B930, GB+9836B939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836BA30, GB+9836BA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836BB30, GB+9836BB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836BC30, GB+9836BC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836BD30, GB+9836BD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836BE30, GB+9836BE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836BF30, GB+9836BF39]*/
  0,     0,     38769, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9836C030, GB+9836C039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836C130, GB+9836C139]*/
  0,     0,     0,     0,     0,     0,     15609, 0,     0,     0,       /*[GB+9836C230, GB+9836C239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836C330, GB+9836C339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836C430, GB+9836C439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836C530, GB+9836C539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836C630, GB+9836C639]*/
  0,     0,     0,     31909, 0,     0,     29699, 0,     0,     0,       /*[GB+9836C730, GB+9836C739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     15789,   /*[GB+9836C830, GB+9836C839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836C930, GB+9836C939]*/
  0,     0,     0,     0,     0,     20398, 0,     36511, 0,     0,       /*[GB+9836CA30, GB+9836CA39]*/
  0,     0,     0,     0,     26383, 0,     0,     0,     0,     0,       /*[GB+9836CB30, GB+9836CB39]*/
  28299, 36652, 0,     32109, 0,     0,     0,     0,     0,     0,       /*[GB+9836CC30, GB+9836CC39]*/
  0,     0,     0,     0,     0,     0,     0,     19786, 0,     0,       /*[GB+9836CD30, GB+9836CD39]*/
  0,     0,     0,     0,     38938, 0,     0,     0,     0,     0,       /*[GB+9836CE30, GB+9836CE39]*/
  0,     0,     0,     0,     36847, 0,     0,     23067, 0,     0,       /*[GB+9836CF30, GB+9836CF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836D030, GB+9836D039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836D130, GB+9836D139]*/
  0,     0,     0,     0,     0,     0,     8125,  0,     0,     0,       /*[GB+9836D230, GB+9836D239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836D330, GB+9836D339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836D430, GB+9836D439]*/
  0,     25753, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836D530, GB+9836D539]*/
  0,     0,     0,     39008, 25597, 0,     0,     0,     0,     0,       /*[GB+9836D630, GB+9836D639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836D730, GB+9836D739]*/
  0,     0,     0,     0,     0,     0,     38080, 0,     0,     0,       /*[GB+9836D830, GB+9836D839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836D930, GB+9836D939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836DA30, GB+9836DA39]*/
  19755, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836DB30, GB+9836DB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836DC30, GB+9836DC39]*/
  0,     18279, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836DD30, GB+9836DD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836DE30, GB+9836DE39]*/
  0,     0,     0,     4035,  13296, 0,     34219, 0,     0,     0,       /*[GB+9836DF30, GB+9836DF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836E030, GB+9836E039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836E130, GB+9836E139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     24349, 0,       /*[GB+9836E230, GB+9836E239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836E330, GB+9836E339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836E430, GB+9836E439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836E530, GB+9836E539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836E630, GB+9836E639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836E730, GB+9836E739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836E830, GB+9836E839]*/
  0,     0,     0,     0,     0,     32719, 0,     0,     0,     0,       /*[GB+9836E930, GB+9836E939]*/
  4404,  0,     0,     0,     0,     21414, 0,     0,     0,     0,       /*[GB+9836EA30, GB+9836EA39]*/
  0,     0,     0,     9957,  0,     0,     0,     0,     0,     18832,   /*[GB+9836EB30, GB+9836EB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     34512, 0,       /*[GB+9836EC30, GB+9836EC39]*/
  0,     0,     0,     2396,  0,     0,     0,     0,     0,     0,       /*[GB+9836ED30, GB+9836ED39]*/
  0,     34789, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836EE30, GB+9836EE39]*/
  0,     0,     15790, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9836EF30, GB+9836EF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836F030, GB+9836F039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836F130, GB+9836F139]*/
  0,     0,     0,     0,     17294, 0,     0,     0,     0,     0,       /*[GB+9836F230, GB+9836F239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836F330, GB+9836F339]*/
  0,     0,     0,     0,     0,     28068, 0,     0,     0,     0,       /*[GB+9836F430, GB+9836F439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836F530, GB+9836F539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836F630, GB+9836F639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836F730, GB+9836F739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836F830, GB+9836F839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836F930, GB+9836F939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836FA30, GB+9836FA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836FB30, GB+9836FB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836FC30, GB+9836FC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836FD30, GB+9836FD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9836FE30, GB+9836FE39]*/
  0,     0,     8022,  0,     0,     0,     0,     0,     0,     0,       /*[GB+98378130, GB+98378139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98378230, GB+98378239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98378330, GB+98378339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98378430, GB+98378439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     8190,    /*[GB+98378530, GB+98378539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98378630, GB+98378639]*/
  0,     0,     0,     0,     0,     0,     0,     931,   0,     0,       /*[GB+98378730, GB+98378739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98378830, GB+98378839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98378930, GB+98378939]*/
  12950, 0,     0,     0,     0,     0,     18348, 0,     0,     0,       /*[GB+98378A30, GB+98378A39]*/
  0,     0,     0,     13398, 0,     0,     0,     0,     0,     0,       /*[GB+98378B30, GB+98378B39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98378C30, GB+98378C39]*/
  0,     0,     8014,  0,     0,     0,     0,     0,     0,     0,       /*[GB+98378D30, GB+98378D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98378E30, GB+98378E39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98378F30, GB+98378F39]*/
  0,     0,     0,     28366, 0,     0,     0,     0,     0,     0,       /*[GB+98379030, GB+98379039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98379130, GB+98379139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98379230, GB+98379239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98379330, GB+98379339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98379430, GB+98379439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98379530, GB+98379539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98379630, GB+98379639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     5000,    /*[GB+98379730, GB+98379739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98379830, GB+98379839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98379930, GB+98379939]*/
  0,     7427,  0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98379A30, GB+98379A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98379B30, GB+98379B39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98379C30, GB+98379C39]*/
  0,     0,     0,     0,     0,     34848, 0,     0,     0,     0,       /*[GB+98379D30, GB+98379D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98379E30, GB+98379E39]*/
  0,     0,     0,     33917, 0,     0,     0,     0,     0,     0,       /*[GB+98379F30, GB+98379F39]*/
  0,     0,     0,     0,     0,     0,     0,     18290, 0,     0,       /*[GB+9837A030, GB+9837A039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837A130, GB+9837A139]*/
  0,     0,     0,     0,     0,     0,     28774, 0,     0,     0,       /*[GB+9837A230, GB+9837A239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837A330, GB+9837A339]*/
  20459, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837A430, GB+9837A439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837A530, GB+9837A539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837A630, GB+9837A639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837A730, GB+9837A739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837A830, GB+9837A839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     26105,   /*[GB+9837A930, GB+9837A939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837AA30, GB+9837AA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837AB30, GB+9837AB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837AC30, GB+9837AC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837AD30, GB+9837AD39]*/
  0,     0,     0,     0,     0,     0,     0,     40646, 0,     0,       /*[GB+9837AE30, GB+9837AE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837AF30, GB+9837AF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837B030, GB+9837B039]*/
  0,     33918, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837B130, GB+9837B139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837B230, GB+9837B239]*/
  0,     12393, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837B330, GB+9837B339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837B430, GB+9837B439]*/
  0,     0,     0,     0,     38211, 0,     0,     0,     0,     0,       /*[GB+9837B530, GB+9837B539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837B630, GB+9837B639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837B730, GB+9837B739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837B830, GB+9837B839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837B930, GB+9837B939]*/
  0,     0,     0,     0,     0,     42901, 0,     0,     0,     0,       /*[GB+9837BA30, GB+9837BA39]*/
  0,     30420, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837BB30, GB+9837BB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837BC30, GB+9837BC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837BD30, GB+9837BD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837BE30, GB+9837BE39]*/
  0,     0,     0,     5751,  0,     0,     0,     0,     0,     0,       /*[GB+9837BF30, GB+9837BF39]*/
  0,     0,     0,     25701, 0,     0,     0,     0,     0,     0,       /*[GB+9837C030, GB+9837C039]*/
  0,     0,     0,     2469,  0,     23332, 0,     23235, 0,     38693,   /*[GB+9837C130, GB+9837C139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837C230, GB+9837C239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     4771,  0,       /*[GB+9837C330, GB+9837C339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837C430, GB+9837C439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837C530, GB+9837C539]*/
  0,     0,     6777,  0,     0,     0,     0,     0,     0,     0,       /*[GB+9837C630, GB+9837C639]*/
  0,     0,     17136, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9837C730, GB+9837C739]*/
  10167, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837C830, GB+9837C839]*/
  0,     0,     0,     0,     0,     0,     0,     1206,  0,     0,       /*[GB+9837C930, GB+9837C939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837CA30, GB+9837CA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837CB30, GB+9837CB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837CC30, GB+9837CC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837CD30, GB+9837CD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837CE30, GB+9837CE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837CF30, GB+9837CF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837D030, GB+9837D039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837D130, GB+9837D139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837D230, GB+9837D239]*/
  0,     0,     41409, 0,     0,     14014, 19906, 0,     0,     0,       /*[GB+9837D330, GB+9837D339]*/
  0,     0,     28367, 0,     0,     38310, 0,     0,     0,     0,       /*[GB+9837D430, GB+9837D439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837D530, GB+9837D539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837D630, GB+9837D639]*/
  0,     0,     0,     0,     35804, 0,     0,     0,     0,     0,       /*[GB+9837D730, GB+9837D739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     34499, 0,       /*[GB+9837D830, GB+9837D839]*/
  0,     0,     19907, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9837D930, GB+9837D939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837DA30, GB+9837DA39]*/
  24372, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837DB30, GB+9837DB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     36530, 0,       /*[GB+9837DC30, GB+9837DC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837DD30, GB+9837DD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837DE30, GB+9837DE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837DF30, GB+9837DF39]*/
  0,     23764, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837E030, GB+9837E039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837E130, GB+9837E139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837E230, GB+9837E239]*/
  0,     0,     0,     0,     28683, 0,     0,     0,     0,     0,       /*[GB+9837E330, GB+9837E339]*/
  0,     21882, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837E430, GB+9837E439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837E530, GB+9837E539]*/
  0,     0,     13875, 0,     0,     0,     4157,  0,     5598,  0,       /*[GB+9837E630, GB+9837E639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837E730, GB+9837E739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     520,   0,       /*[GB+9837E830, GB+9837E839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837E930, GB+9837E939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837EA30, GB+9837EA39]*/
  0,     0,     0,     0,     0,     0,     0,     43112, 0,     18336,   /*[GB+9837EB30, GB+9837EB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837EC30, GB+9837EC39]*/
  0,     0,     0,     0,     0,     0,     33291, 32741, 0,     0,       /*[GB+9837ED30, GB+9837ED39]*/
  0,     29339, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837EE30, GB+9837EE39]*/
  0,     0,     0,     21888, 0,     0,     10301, 0,     8028,  0,       /*[GB+9837EF30, GB+9837EF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837F030, GB+9837F039]*/
  0,     0,     116,   0,     0,     29962, 0,     0,     0,     0,       /*[GB+9837F130, GB+9837F139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837F230, GB+9837F239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837F330, GB+9837F339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837F430, GB+9837F439]*/
  0,     0,     0,     27540, 0,     0,     0,     0,     0,     0,       /*[GB+9837F530, GB+9837F539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837F630, GB+9837F639]*/
  0,     0,     0,     0,     0,     0,     0,     15248, 0,     0,       /*[GB+9837F730, GB+9837F739]*/
  0,     0,     16634, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9837F830, GB+9837F839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837F930, GB+9837F939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837FA30, GB+9837FA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837FB30, GB+9837FB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837FC30, GB+9837FC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9837FD30, GB+9837FD39]*/
  0,     0,     0,     15207, 11814, 14795, 21014, 6287,  1250,  28668,   /*[GB+9837FE30, GB+9837FE39]*/
  12093, 6612,  8492,  23271, 5318,  18035, 14589, 36655, 3628,  7726,    /*[GB+98388130, GB+98388139]*/
  27671, 18358, 35192, 8745,  31452, 0,     707,   27778, 22281, 33872,   /*[GB+98388230, GB+98388239]*/
  20273, 1086,  25624, 38352, 36262, 43924, 0,     0,     0,     0,       /*[GB+98388330, GB+98388339]*/
  0,     0,     0,     0,     0,     0,     0,     13302, 0,     0,       /*[GB+98388430, GB+98388439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98388530, GB+98388539]*/
  0,     0,     0,     0,     0,     8146,  0,     0,     0,     0,       /*[GB+98388630, GB+98388639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98388730, GB+98388739]*/
  0,     0,     0,     0,     0,     0,     0,     24352, 0,     0,       /*[GB+98388830, GB+98388839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98388930, GB+98388939]*/
  0,     0,     18627, 0,     0,     0,     0,     0,     0,     0,       /*[GB+98388A30, GB+98388A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98388B30, GB+98388B39]*/
  0,     22776, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98388C30, GB+98388C39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98388D30, GB+98388D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98388E30, GB+98388E39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98388F30, GB+98388F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98389030, GB+98389039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98389130, GB+98389139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98389230, GB+98389239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98389330, GB+98389339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     38968, 0,       /*[GB+98389430, GB+98389439]*/
  0,     0,     9069,  0,     0,     32730, 0,     0,     0,     0,       /*[GB+98389530, GB+98389539]*/
  0,     0,     4155,  0,     0,     0,     0,     0,     0,     0,       /*[GB+98389630, GB+98389639]*/
  0,     23239, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98389730, GB+98389739]*/
  0,     0,     0,     13744, 0,     0,     0,     0,     26056, 0,       /*[GB+98389830, GB+98389839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98389930, GB+98389939]*/
  0,     0,     0,     0,     0,     3376,  0,     0,     0,     0,       /*[GB+98389A30, GB+98389A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98389B30, GB+98389B39]*/
  0,     0,     25228, 0,     0,     0,     0,     15890, 0,     0,       /*[GB+98389C30, GB+98389C39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     104,     /*[GB+98389D30, GB+98389D39]*/
  0,     0,     29174, 0,     0,     0,     0,     0,     0,     0,       /*[GB+98389E30, GB+98389E39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98389F30, GB+98389F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838A030, GB+9838A039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838A130, GB+9838A139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838A230, GB+9838A239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838A330, GB+9838A339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838A430, GB+9838A439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838A530, GB+9838A539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838A630, GB+9838A639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838A730, GB+9838A739]*/
  0,     0,     0,     0,     3077,  0,     0,     0,     19916, 0,       /*[GB+9838A830, GB+9838A839]*/
  0,     13888, 0,     0,     0,     0,     0,     0,     14363, 0,       /*[GB+9838A930, GB+9838A939]*/
  0,     36319, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838AA30, GB+9838AA39]*/
  0,     0,     0,     34824, 0,     0,     0,     0,     30941, 0,       /*[GB+9838AB30, GB+9838AB39]*/
  0,     0,     9567,  0,     0,     0,     0,     0,     0,     0,       /*[GB+9838AC30, GB+9838AC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838AD30, GB+9838AD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838AE30, GB+9838AE39]*/
  0,     0,     0,     0,     0,     0,     23444, 0,     0,     0,       /*[GB+9838AF30, GB+9838AF39]*/
  0,     40788, 0,     26954, 0,     10284, 0,     0,     0,     0,       /*[GB+9838B030, GB+9838B039]*/
  12992, 0,     0,     0,     0,     0,     0,     41636, 0,     0,       /*[GB+9838B130, GB+9838B139]*/
  0,     29268, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838B230, GB+9838B239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838B330, GB+9838B339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838B430, GB+9838B439]*/
  20456, 0,     21545, 18763, 0,     32680, 0,     14373, 0,     0,       /*[GB+9838B530, GB+9838B539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838B630, GB+9838B639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838B730, GB+9838B739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838B830, GB+9838B839]*/
  24384, 0,     0,     0,     0,     0,     0,     0,     0,     38027,   /*[GB+9838B930, GB+9838B939]*/
  37219, 8067,  26566, 0,     11572, 37760, 40128, 37761, 22549, 31563,   /*[GB+9838BA30, GB+9838BA39]*/
  14336, 31374, 22512, 3624,  41813, 11977, 13368, 36913, 32515, 0,       /*[GB+9838BB30, GB+9838BB39]*/
  26392, 34327, 33852, 43090, 34919, 42628, 19783, 312,   3837,  17076,   /*[GB+9838BC30, GB+9838BC39]*/
  28118, 14518, 27916, 33662, 42637, 36005, 23159, 13941, 0,     0,       /*[GB+9838BD30, GB+9838BD39]*/
  0,     0,     0,     0,     43634, 0,     0,     0,     0,     0,       /*[GB+9838BE30, GB+9838BE39]*/
  15649, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838BF30, GB+9838BF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838C030, GB+9838C039]*/
  0,     0,     0,     0,     0,     0,     19758, 9656,  18748, 3325,    /*[GB+9838C130, GB+9838C139]*/
  3435,  5563,  9042,  3468,  0,     0,     0,     18364, 0,     1351,    /*[GB+9838C230, GB+9838C239]*/
  0,     0,     0,     0,     0,     0,     2374,  0,     0,     0,       /*[GB+9838C330, GB+9838C339]*/
  0,     0,     0,     0,     0,     3084,  0,     0,     1971,  0,       /*[GB+9838C430, GB+9838C439]*/
  0,     0,     0,     15792, 17374, 0,     0,     0,     40634, 20585,   /*[GB+9838C530, GB+9838C539]*/
  0,     0,     0,     26036, 0,     0,     0,     0,     0,     0,       /*[GB+9838C630, GB+9838C639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838C730, GB+9838C739]*/
  0,     0,     34281, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9838C830, GB+9838C839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838C930, GB+9838C939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838CA30, GB+9838CA39]*/
  40211, 27994, 16797, 22340, 477,   19267, 7817,  42202, 24444, 31227,   /*[GB+9838CB30, GB+9838CB39]*/
  37973, 42683, 22724, 10011, 11310, 27250, 16393, 32901, 14290, 2199,    /*[GB+9838CC30, GB+9838CC39]*/
  16235, 17867, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838CD30, GB+9838CD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838CE30, GB+9838CE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838CF30, GB+9838CF39]*/
  0,     19793, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838D030, GB+9838D039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838D130, GB+9838D139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838D230, GB+9838D239]*/
  0,     0,     0,     0,     43080, 0,     0,     0,     0,     0,       /*[GB+9838D330, GB+9838D339]*/
  0,     0,     0,     21260, 0,     0,     0,     0,     0,     0,       /*[GB+9838D430, GB+9838D439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838D530, GB+9838D539]*/
  0,     0,     0,     0,     0,     36954, 0,     0,     0,     0,       /*[GB+9838D630, GB+9838D639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838D730, GB+9838D739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838D830, GB+9838D839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838D930, GB+9838D939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838DA30, GB+9838DA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838DB30, GB+9838DB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     10699, 0,       /*[GB+9838DC30, GB+9838DC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838DD30, GB+9838DD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838DE30, GB+9838DE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838DF30, GB+9838DF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838E030, GB+9838E039]*/
  0,     0,     0,     0,     0,     429,   43562, 8385,  10402, 4775,    /*[GB+9838E130, GB+9838E139]*/
  25273, 0,     40493, 37051, 19473, 12963, 35871, 25705, 16028, 0,       /*[GB+9838E230, GB+9838E239]*/
  22168, 28540, 35986, 0,     32450, 12995, 19979, 23040, 11118, 43731,   /*[GB+9838E330, GB+9838E339]*/
  39811, 0,     18766, 8139,  5907,  0,     27254, 30275, 906,   1156,    /*[GB+9838E430, GB+9838E439]*/
  29472, 0,     35670, 16854, 27641, 7758,  15944, 2877,  0,     41267,   /*[GB+9838E530, GB+9838E539]*/
  0,     0,     33773, 0,     23246, 0,     0,     0,     0,     0,       /*[GB+9838E630, GB+9838E639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838E730, GB+9838E739]*/
  0,     0,     0,     0,     0,     0,     0,     16124, 0,     6688,    /*[GB+9838E830, GB+9838E839]*/
  1565,  17442, 34786, 0,     19206, 42404, 23042, 0,     0,     0,       /*[GB+9838E930, GB+9838E939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838EA30, GB+9838EA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838EB30, GB+9838EB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838EC30, GB+9838EC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838ED30, GB+9838ED39]*/
  0,     0,     0,     40415, 0,     0,     0,     0,     0,     0,       /*[GB+9838EE30, GB+9838EE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838EF30, GB+9838EF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838F030, GB+9838F039]*/
  0,     0,     0,     12651, 0,     19660, 0,     36153, 0,     0,       /*[GB+9838F130, GB+9838F139]*/
  0,     0,     0,     0,     0,     0,     0,     25909, 1288,  1893,    /*[GB+9838F230, GB+9838F239]*/
  591,   9616,  41979, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9838F330, GB+9838F339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838F430, GB+9838F439]*/
  0,     0,     16679, 41767, 41772, 0,     37976, 8671,  7189,  9374,    /*[GB+9838F530, GB+9838F539]*/
  12748, 40327, 40820, 40260, 25318, 0,     40171, 4969,  7777,  0,       /*[GB+9838F630, GB+9838F639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838F730, GB+9838F739]*/
  0,     39718, 2045,  1115,  244,   27811, 0,     0,     0,     0,       /*[GB+9838F830, GB+9838F839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838F930, GB+9838F939]*/
  13960, 0,     0,     0,     41154, 40039, 41344, 0,     44056, 1253,    /*[GB+9838FA30, GB+9838FA39]*/
  28901, 40434, 2255,  29876, 17617, 20464, 32862, 11682, 23360, 32875,   /*[GB+9838FB30, GB+9838FB39]*/
  12275, 11494, 6973,  18629, 0,     14224, 41195, 34703, 0,     0,       /*[GB+9838FC30, GB+9838FC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838FD30, GB+9838FD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9838FE30, GB+9838FE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98398130, GB+98398139]*/
  0,     0,     0,     0,     0,     10739, 32995, 15821, 32101, 24540,   /*[GB+98398230, GB+98398239]*/
  42289, 15152, 0,     10756, 29,    17131, 30891, 20049, 31039, 12276,   /*[GB+98398330, GB+98398339]*/
  40087, 36944, 33807, 29344, 41582, 2630,  41196, 33836, 0,     0,       /*[GB+98398430, GB+98398439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     17226,   /*[GB+98398530, GB+98398539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98398630, GB+98398639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98398730, GB+98398739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98398830, GB+98398839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98398930, GB+98398939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98398A30, GB+98398A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98398B30, GB+98398B39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98398C30, GB+98398C39]*/
  0,     0,     0,     0,     35872, 11391, 37103, 43091, 27885, 8131,    /*[GB+98398D30, GB+98398D39]*/
  1897,  25708, 37057, 1358,  8515,  19054, 12473, 28711, 4718,  9308,    /*[GB+98398E30, GB+98398E39]*/
  0,     24641, 41593, 7156,  10187, 12012, 18120, 0,     0,     42737,   /*[GB+98398F30, GB+98398F39]*/
  36857, 1508,  43523, 34069, 39071, 26350, 2290,  0,     39937, 27728,   /*[GB+98399030, GB+98399039]*/
  0,     19099, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98399130, GB+98399139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98399230, GB+98399239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98399330, GB+98399339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98399430, GB+98399439]*/
  0,     0,     0,     0,     0,     0,     13732, 42023, 8356,  12056,   /*[GB+98399530, GB+98399539]*/
  15838, 19640, 40992, 8516,  23142, 39738, 9220,  13638, 33247, 0,       /*[GB+98399630, GB+98399639]*/
  24076, 36923, 27295, 40070, 31774, 16899, 1723,  7902,  24918, 43360,   /*[GB+98399730, GB+98399739]*/
  6255,  8578,  31051, 15892, 3861,  11704, 31060, 0,     11509, 7154,    /*[GB+98399830, GB+98399839]*/
  17017, 18422, 4187,  0,     3397,  1422,  41368, 38602, 43985, 12136,   /*[GB+98399930, GB+98399939]*/
  0,     0,     0,     0,     32980, 0,     0,     0,     0,     0,       /*[GB+98399A30, GB+98399A39]*/
  0,     8246,  32071, 0,     24102, 16711, 0,     0,     0,     38992,   /*[GB+98399B30, GB+98399B39]*/
  0,     0,     0,     0,     0,     4970,  0,     0,     0,     0,       /*[GB+98399C30, GB+98399C39]*/
  0,     0,     0,     0,     513,   23048, 37391, 0,     40841, 4223,    /*[GB+98399D30, GB+98399D39]*/
  36858, 14783, 36556, 0,     19738, 22466, 0,     0,     0,     29032,   /*[GB+98399E30, GB+98399E39]*/
  11967, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+98399F30, GB+98399F39]*/
  0,     0,     0,     0,     0,     0,     12702, 0,     0,     29243,   /*[GB+9839A030, GB+9839A039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839A130, GB+9839A139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839A230, GB+9839A239]*/
  0,     0,     0,     0,     0,     0,     920,   29214, 12379, 4032,    /*[GB+9839A330, GB+9839A339]*/
  9578,  17336, 17069, 38768, 17546, 0,     0,     0,     0,     0,       /*[GB+9839A430, GB+9839A439]*/
  0,     5718,  0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839A530, GB+9839A539]*/
  0,     0,     0,     0,     0,     0,     0,     36301, 0,     0,       /*[GB+9839A630, GB+9839A639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839A730, GB+9839A739]*/
  0,     0,     0,     0,     0,     21873, 0,     0,     43477, 42835,   /*[GB+9839A830, GB+9839A839]*/
  0,     15933, 0,     0,     19288, 27153, 42282, 2931,  0,     935,     /*[GB+9839A930, GB+9839A939]*/
  43445, 17713, 17456, 21225, 0,     0,     0,     0,     0,     0,       /*[GB+9839AA30, GB+9839AA39]*/
  0,     0,     0,     0,     0,     0,     0,     16808, 0,     0,       /*[GB+9839AB30, GB+9839AB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     27053,   /*[GB+9839AC30, GB+9839AC39]*/
  11377, 18800, 3964,  0,     0,     0,     0,     0,     0,     0,       /*[GB+9839AD30, GB+9839AD39]*/
  0,     0,     0,     19299, 0,     0,     0,     13019, 0,     0,       /*[GB+9839AE30, GB+9839AE39]*/
  0,     0,     0,     0,     0,     0,     42170, 23636, 2938,  4849,    /*[GB+9839AF30, GB+9839AF39]*/
  30673, 0,     32107, 16554, 30135, 0,     0,     0,     0,     18283,   /*[GB+9839B030, GB+9839B039]*/
  42310, 0,     0,     0,     0,     0,     26264, 29605, 8079,  40394,   /*[GB+9839B130, GB+9839B139]*/
  28107, 18341, 28925, 11389, 16126, 42674, 5274,  43817, 43823, 30267,   /*[GB+9839B230, GB+9839B239]*/
  4963,  11155, 19258, 0,     0,     9298,  17047, 36288, 1139,  0,       /*[GB+9839B330, GB+9839B339]*/
  36035, 20093, 756,   0,     0,     22270, 10685, 29788, 41374, 0,       /*[GB+9839B430, GB+9839B439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     22740, 0,       /*[GB+9839B530, GB+9839B539]*/
  0,     0,     0,     0,     0,     0,     0,     19861, 0,     0,       /*[GB+9839B630, GB+9839B639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839B730, GB+9839B739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839B830, GB+9839B839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839B930, GB+9839B939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839BA30, GB+9839BA39]*/
  0,     38468, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839BB30, GB+9839BB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839BC30, GB+9839BC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     33126, 0,       /*[GB+9839BD30, GB+9839BD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     5541,  0,       /*[GB+9839BE30, GB+9839BE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839BF30, GB+9839BF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839C030, GB+9839C039]*/
  0,     0,     0,     0,     0,     0,     19731, 0,     10267, 0,       /*[GB+9839C130, GB+9839C139]*/
  0,     0,     6991,  6602,  0,     0,     0,     0,     0,     0,       /*[GB+9839C230, GB+9839C239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839C330, GB+9839C339]*/
  18741, 0,     17697, 32732, 0,     0,     0,     0,     0,     0,       /*[GB+9839C430, GB+9839C439]*/
  0,     0,     0,     0,     5523,  3991,  2363,  0,     0,     0,       /*[GB+9839C530, GB+9839C539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839C630, GB+9839C639]*/
  0,     17447, 82,    0,     0,     0,     0,     0,     0,     0,       /*[GB+9839C730, GB+9839C739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     25307,   /*[GB+9839C830, GB+9839C839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839C930, GB+9839C939]*/
  0,     0,     0,     0,     0,     29856, 0,     0,     0,     0,       /*[GB+9839CA30, GB+9839CA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839CB30, GB+9839CB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839CC30, GB+9839CC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839CD30, GB+9839CD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839CE30, GB+9839CE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839CF30, GB+9839CF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839D030, GB+9839D039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839D130, GB+9839D139]*/
  0,     0,     20948, 0,     0,     0,     0,     9007,  0,     0,       /*[GB+9839D230, GB+9839D239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839D330, GB+9839D339]*/
  0,     0,     0,     36640, 0,     0,     0,     0,     0,     0,       /*[GB+9839D430, GB+9839D439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839D530, GB+9839D539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839D630, GB+9839D639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839D730, GB+9839D739]*/
  0,     18267, 0,     18274, 0,     0,     0,     0,     37525, 0,       /*[GB+9839D830, GB+9839D839]*/
  36411, 0,     0,     0,     0,     30237, 0,     0,     0,     34944,   /*[GB+9839D930, GB+9839D939]*/
  0,     0,     0,     19781, 0,     0,     15850, 0,     0,     0,       /*[GB+9839DA30, GB+9839DA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839DB30, GB+9839DB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839DC30, GB+9839DC39]*/
  0,     0,     0,     0,     0,     18829, 0,     0,     0,     0,       /*[GB+9839DD30, GB+9839DD39]*/
  0,     0,     0,     0,     0,     15851, 0,     0,     0,     0,       /*[GB+9839DE30, GB+9839DE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839DF30, GB+9839DF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839E030, GB+9839E039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839E130, GB+9839E139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839E230, GB+9839E239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839E330, GB+9839E339]*/
  0,     0,     0,     0,     0,     35152, 0,     0,     0,     0,       /*[GB+9839E430, GB+9839E439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839E530, GB+9839E539]*/
  0,     6787,  0,     0,     0,     0,     0,     0,     34076, 0,       /*[GB+9839E630, GB+9839E639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839E730, GB+9839E739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839E830, GB+9839E839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839E930, GB+9839E939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839EA30, GB+9839EA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839EB30, GB+9839EB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839EC30, GB+9839EC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839ED30, GB+9839ED39]*/
  0,     0,     0,     0,     0,     0,     0,     11287, 0,     0,       /*[GB+9839EE30, GB+9839EE39]*/
  0,     0,     9057,  4842,  0,     0,     0,     0,     0,     0,       /*[GB+9839EF30, GB+9839EF39]*/
  0,     0,     37531, 12806, 0,     0,     0,     4137,  0,     37540,   /*[GB+9839F030, GB+9839F039]*/
  0,     0,     0,     10547, 0,     0,     0,     0,     0,     0,       /*[GB+9839F130, GB+9839F139]*/
  0,     0,     0,     0,     14893, 0,     0,     0,     0,     0,       /*[GB+9839F230, GB+9839F239]*/
  10400, 17394, 36648, 18321, 13734, 0,     0,     0,     0,     0,       /*[GB+9839F330, GB+9839F339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839F430, GB+9839F439]*/
  0,     0,     0,     0,     23448, 0,     40625, 0,     0,     0,       /*[GB+9839F530, GB+9839F539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     14263,   /*[GB+9839F630, GB+9839F639]*/
  0,     0,     42317, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9839F730, GB+9839F739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     31139, 0,       /*[GB+9839F830, GB+9839F839]*/
  0,     14915, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839F930, GB+9839F939]*/
  0,     7421,  0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839FA30, GB+9839FA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839FB30, GB+9839FB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9839FC30, GB+9839FC39]*/
  0,     0,     5547,  0,     0,     0,     0,     0,     0,     0,       /*[GB+9839FD30, GB+9839FD39]*/
  0,     0,     0,     0,     0,     0,     0,     20275, 0,     0,       /*[GB+9839FE30, GB+9839FE39]*/
  0,     0,     0,     0,     0,     0,     24373, 0,     0,     0,       /*[GB+99308130, GB+99308139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99308230, GB+99308239]*/
  0,     0,     0,     0,     27595, 0,     0,     0,     0,     0,       /*[GB+99308330, GB+99308339]*/
  0,     0,     0,     10390, 0,     0,     0,     0,     0,     6423,    /*[GB+99308430, GB+99308439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99308530, GB+99308539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99308630, GB+99308639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99308730, GB+99308739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99308830, GB+99308839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99308930, GB+99308939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99308A30, GB+99308A39]*/
  0,     0,     43081, 23438, 0,     0,     20437, 0,     0,     0,       /*[GB+99308B30, GB+99308B39]*/
  0,     0,     467,   0,     25638, 0,     0,     0,     2556,  36245,   /*[GB+99308C30, GB+99308C39]*/
  0,     0,     36649, 0,     0,     0,     0,     0,     0,     0,       /*[GB+99308D30, GB+99308D39]*/
  0,     0,     17702, 0,     0,     0,     0,     0,     0,     28109,   /*[GB+99308E30, GB+99308E39]*/
  0,     3407,  0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99308F30, GB+99308F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99309030, GB+99309039]*/
  0,     0,     0,     0,     38586, 0,     0,     0,     0,     0,       /*[GB+99309130, GB+99309139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99309230, GB+99309239]*/
  14221, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99309330, GB+99309339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99309430, GB+99309439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99309530, GB+99309539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     23242, 0,       /*[GB+99309630, GB+99309639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99309730, GB+99309739]*/
  0,     0,     0,     0,     0,     0,     0,     12810, 0,     0,       /*[GB+99309830, GB+99309839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99309930, GB+99309939]*/
  0,     0,     0,     0,     40328, 0,     0,     0,     0,     0,       /*[GB+99309A30, GB+99309A39]*/
  0,     0,     0,     0,     0,     0,     7207,  0,     0,     0,       /*[GB+99309B30, GB+99309B39]*/
  0,     0,     0,     0,     0,     0,     0,     32624, 0,     0,       /*[GB+99309C30, GB+99309C39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99309D30, GB+99309D39]*/
  0,     35466, 0,     0,     0,     0,     0,     11976, 12524, 0,       /*[GB+99309E30, GB+99309E39]*/
  0,     0,     0,     0,     20699, 0,     0,     0,     0,     0,       /*[GB+99309F30, GB+99309F39]*/
  0,     0,     34218, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9930A030, GB+9930A039]*/
  0,     10151, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930A130, GB+9930A139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930A230, GB+9930A239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     17581,   /*[GB+9930A330, GB+9930A339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930A430, GB+9930A439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930A530, GB+9930A539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930A630, GB+9930A639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930A730, GB+9930A739]*/
  0,     25197, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930A830, GB+9930A839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930A930, GB+9930A939]*/
  0,     0,     0,     0,     0,     40645, 0,     0,     0,     0,       /*[GB+9930AA30, GB+9930AA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930AB30, GB+9930AB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930AC30, GB+9930AC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     8221,  0,       /*[GB+9930AD30, GB+9930AD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     36418,   /*[GB+9930AE30, GB+9930AE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930AF30, GB+9930AF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930B030, GB+9930B039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930B130, GB+9930B139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930B230, GB+9930B239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930B330, GB+9930B339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930B430, GB+9930B439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930B530, GB+9930B539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930B630, GB+9930B639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930B730, GB+9930B739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930B830, GB+9930B839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930B930, GB+9930B939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930BA30, GB+9930BA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930BB30, GB+9930BB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930BC30, GB+9930BC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930BD30, GB+9930BD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930BE30, GB+9930BE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930BF30, GB+9930BF39]*/
  0,     0,     0,     0,     0,     0,     0,     37070, 0,     18278,   /*[GB+9930C030, GB+9930C039]*/
  16532, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930C130, GB+9930C139]*/
  0,     0,     0,     0,     0,     6369,  0,     0,     17540, 0,       /*[GB+9930C230, GB+9930C239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     27145,   /*[GB+9930C330, GB+9930C339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930C430, GB+9930C439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930C530, GB+9930C539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     16559, 0,       /*[GB+9930C630, GB+9930C639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930C730, GB+9930C739]*/
  0,     0,     0,     0,     0,     0,     0,     17120, 0,     0,       /*[GB+9930C830, GB+9930C839]*/
  35705, 34424, 0,     0,     17452, 0,     0,     0,     0,     0,       /*[GB+9930C930, GB+9930C939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930CA30, GB+9930CA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930CB30, GB+9930CB39]*/
  0,     0,     0,     0,     0,     0,     17338, 0,     38692, 0,       /*[GB+9930CC30, GB+9930CC39]*/
  0,     0,     0,     0,     0,     0,     0,     35340, 0,     19788,   /*[GB+9930CD30, GB+9930CD39]*/
  0,     0,     0,     0,     38939, 0,     0,     0,     0,     0,       /*[GB+9930CE30, GB+9930CE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930CF30, GB+9930CF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930D030, GB+9930D039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930D130, GB+9930D139]*/
  0,     0,     0,     0,     0,     0,     0,     16609, 0,     0,       /*[GB+9930D230, GB+9930D239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930D330, GB+9930D339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930D430, GB+9930D439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930D530, GB+9930D539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930D630, GB+9930D639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930D730, GB+9930D739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930D830, GB+9930D839]*/
  0,     0,     0,     0,     0,     0,     39617, 0,     0,     0,       /*[GB+9930D930, GB+9930D939]*/
  0,     0,     32718, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9930DA30, GB+9930DA39]*/
  6537,  28176, 0,     0,     0,     25817, 25811, 0,     0,     0,       /*[GB+9930DB30, GB+9930DB39]*/
  11924, 0,     0,     0,     24849, 0,     9496,  0,     0,     0,       /*[GB+9930DC30, GB+9930DC39]*/
  0,     0,     0,     21238, 0,     0,     0,     0,     17783, 0,       /*[GB+9930DD30, GB+9930DD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930DE30, GB+9930DE39]*/
  0,     0,     0,     0,     0,     0,     37163, 0,     0,     0,       /*[GB+9930DF30, GB+9930DF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     22493,   /*[GB+9930E030, GB+9930E039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     33969,   /*[GB+9930E130, GB+9930E139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930E230, GB+9930E239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930E330, GB+9930E339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930E430, GB+9930E439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930E530, GB+9930E539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930E630, GB+9930E639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930E730, GB+9930E739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930E830, GB+9930E839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930E930, GB+9930E939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     6286,    /*[GB+9930EA30, GB+9930EA39]*/
  0,     4006,  0,     0,     0,     0,     0,     24327, 40870, 10539,   /*[GB+9930EB30, GB+9930EB39]*/
  0,     11925, 0,     0,     0,     1259,  0,     0,     22526, 0,       /*[GB+9930EC30, GB+9930EC39]*/
  0,     16297, 0,     0,     0,     19970, 30242, 0,     0,     0,       /*[GB+9930ED30, GB+9930ED39]*/
  0,     0,     0,     0,     0,     0,     22349, 18713, 0,     0,       /*[GB+9930EE30, GB+9930EE39]*/
  0,     0,     0,     12826, 0,     0,     2031,  20458, 0,     0,       /*[GB+9930EF30, GB+9930EF39]*/
  0,     0,     13739, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9930F030, GB+9930F039]*/
  0,     16298, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930F130, GB+9930F139]*/
  0,     14147, 0,     25293, 0,     8013,  0,     0,     0,     0,       /*[GB+9930F230, GB+9930F239]*/
  0,     0,     5629,  0,     0,     0,     22947, 1336,  0,     0,       /*[GB+9930F330, GB+9930F339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930F430, GB+9930F439]*/
  0,     36826, 0,     34817, 0,     0,     0,     0,     0,     0,       /*[GB+9930F530, GB+9930F539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930F630, GB+9930F639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930F730, GB+9930F739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930F830, GB+9930F839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930F930, GB+9930F939]*/
  18220, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930FA30, GB+9930FA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930FB30, GB+9930FB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930FC30, GB+9930FC39]*/
  0,     0,     0,     0,     0,     0,     0,     29177, 0,     0,       /*[GB+9930FD30, GB+9930FD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9930FE30, GB+9930FE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99318130, GB+99318139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99318230, GB+99318239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99318330, GB+99318339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99318430, GB+99318439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99318530, GB+99318539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99318630, GB+99318639]*/
  0,     0,     0,     0,     0,     12827, 0,     0,     0,     32726,   /*[GB+99318730, GB+99318739]*/
  34487, 0,     0,     0,     7002,  0,     0,     16147, 0,     0,       /*[GB+99318830, GB+99318839]*/
  0,     0,     0,     28179, 0,     0,     0,     0,     0,     0,       /*[GB+99318930, GB+99318939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99318A30, GB+99318A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99318B30, GB+99318B39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99318C30, GB+99318C39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99318D30, GB+99318D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99318E30, GB+99318E39]*/
  0,     0,     0,     0,     0,     23440, 0,     0,     0,     0,       /*[GB+99318F30, GB+99318F39]*/
  36519, 0,     0,     30182, 0,     0,     27472, 0,     20465, 0,       /*[GB+99319030, GB+99319039]*/
  0,     5443,  0,     0,     0,     28649, 0,     0,     0,     0,       /*[GB+99319130, GB+99319139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99319230, GB+99319239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     21970, 0,       /*[GB+99319330, GB+99319339]*/
  0,     0,     0,     0,     0,     0,     0,     13572, 0,     0,       /*[GB+99319430, GB+99319439]*/
  0,     0,     27865, 0,     23236, 0,     0,     0,     0,     0,       /*[GB+99319530, GB+99319539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99319630, GB+99319639]*/
  10163, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99319730, GB+99319739]*/
  42362, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99319830, GB+99319839]*/
  0,     0,     0,     0,     7620,  0,     0,     0,     0,     0,       /*[GB+99319930, GB+99319939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     13771, 0,       /*[GB+99319A30, GB+99319A39]*/
  0,     0,     0,     34430, 0,     0,     8023,  0,     0,     0,       /*[GB+99319B30, GB+99319B39]*/
  19128, 36686, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99319C30, GB+99319C39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99319D30, GB+99319D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99319E30, GB+99319E39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99319F30, GB+99319F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931A030, GB+9931A039]*/
  0,     0,     0,     0,     0,     38658, 0,     0,     0,     0,       /*[GB+9931A130, GB+9931A139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931A230, GB+9931A239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931A330, GB+9931A339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931A430, GB+9931A439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931A530, GB+9931A539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931A630, GB+9931A639]*/
  0,     0,     34053, 0,     0,     9879,  0,     0,     0,     0,       /*[GB+9931A730, GB+9931A739]*/
  0,     23842, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931A830, GB+9931A839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931A930, GB+9931A939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931AA30, GB+9931AA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931AB30, GB+9931AB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     22603,   /*[GB+9931AC30, GB+9931AC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931AD30, GB+9931AD39]*/
  12960, 0,     0,     0,     0,     0,     0,     0,     16652, 0,       /*[GB+9931AE30, GB+9931AE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931AF30, GB+9931AF39]*/
  0,     0,     0,     0,     21018, 0,     0,     0,     0,     14050,   /*[GB+9931B030, GB+9931B039]*/
  1805,  0,     0,     6414,  0,     0,     0,     0,     0,     0,       /*[GB+9931B130, GB+9931B139]*/
  0,     0,     0,     0,     0,     0,     0,     32434, 0,     0,       /*[GB+9931B230, GB+9931B239]*/
  31855, 922,   0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931B330, GB+9931B339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931B430, GB+9931B439]*/
  0,     0,     0,     0,     21239, 0,     0,     0,     0,     0,       /*[GB+9931B530, GB+9931B539]*/
  0,     0,     0,     10446, 0,     0,     0,     0,     0,     0,       /*[GB+9931B630, GB+9931B639]*/
  0,     28111, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931B730, GB+9931B739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931B830, GB+9931B839]*/
  0,     8015,  0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931B930, GB+9931B939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     27777,   /*[GB+9931BA30, GB+9931BA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     24510,   /*[GB+9931BB30, GB+9931BB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931BC30, GB+9931BC39]*/
  0,     0,     11945, 0,     0,     0,     8196,  0,     0,     0,       /*[GB+9931BD30, GB+9931BD39]*/
  0,     27948, 0,     0,     0,     32793, 0,     0,     0,     0,       /*[GB+9931BE30, GB+9931BE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931BF30, GB+9931BF39]*/
  0,     0,     0,     0,     0,     0,     17483, 0,     0,     0,       /*[GB+9931C030, GB+9931C039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931C130, GB+9931C139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931C230, GB+9931C239]*/
  0,     0,     0,     0,     23485, 0,     0,     0,     0,     0,       /*[GB+9931C330, GB+9931C339]*/
  3209,  0,     0,     0,     0,     0,     36306, 5256,  0,     0,       /*[GB+9931C430, GB+9931C439]*/
  0,     4042,  9500,  0,     0,     0,     0,     0,     0,     0,       /*[GB+9931C530, GB+9931C539]*/
  21156, 13758, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931C630, GB+9931C639]*/
  0,     0,     0,     0,     2949,  0,     28073, 0,     0,     0,       /*[GB+9931C730, GB+9931C739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931C830, GB+9931C839]*/
  0,     30795, 17584, 0,     0,     0,     32593, 0,     0,     0,       /*[GB+9931C930, GB+9931C939]*/
  26867, 0,     0,     0,     0,     18768, 0,     0,     0,     0,       /*[GB+9931CA30, GB+9931CA39]*/
  0,     0,     0,     0,     0,     17409, 0,     0,     0,     0,       /*[GB+9931CB30, GB+9931CB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931CC30, GB+9931CC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931CD30, GB+9931CD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931CE30, GB+9931CE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931CF30, GB+9931CF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931D030, GB+9931D039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931D130, GB+9931D139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     11065,   /*[GB+9931D230, GB+9931D239]*/
  0,     0,     0,     0,     0,     4154,  0,     0,     0,     0,       /*[GB+9931D330, GB+9931D339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931D430, GB+9931D439]*/
  0,     0,     37415, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9931D530, GB+9931D539]*/
  0,     28014, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931D630, GB+9931D639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931D730, GB+9931D739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931D830, GB+9931D839]*/
  0,     0,     0,     0,     0,     41226, 0,     18304, 0,     0,       /*[GB+9931D930, GB+9931D939]*/
  0,     0,     0,     5659,  0,     0,     36308, 0,     0,     0,       /*[GB+9931DA30, GB+9931DA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931DB30, GB+9931DB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931DC30, GB+9931DC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931DD30, GB+9931DD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931DE30, GB+9931DE39]*/
  0,     17587, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931DF30, GB+9931DF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931E030, GB+9931E039]*/
  0,     0,     0,     4846,  0,     0,     0,     0,     0,     0,       /*[GB+9931E130, GB+9931E139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931E230, GB+9931E239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931E330, GB+9931E339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931E430, GB+9931E439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931E530, GB+9931E539]*/
  0,     0,     0,     0,     0,     0,     20666, 0,     0,     13362,   /*[GB+9931E630, GB+9931E639]*/
  0,     0,     0,     0,     0,     0,     8004,  0,     0,     0,       /*[GB+9931E730, GB+9931E739]*/
  0,     33006, 0,     0,     10408, 0,     0,     0,     0,     0,       /*[GB+9931E830, GB+9931E839]*/
  0,     0,     0,     8194,  0,     0,     0,     0,     0,     0,       /*[GB+9931E930, GB+9931E939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931EA30, GB+9931EA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931EB30, GB+9931EB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     10564, 0,       /*[GB+9931EC30, GB+9931EC39]*/
  0,     0,     0,     0,     37108, 0,     0,     0,     0,     0,       /*[GB+9931ED30, GB+9931ED39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931EE30, GB+9931EE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931EF30, GB+9931EF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931F030, GB+9931F039]*/
  0,     0,     10379, 3137,  0,     0,     0,     0,     0,     4112,    /*[GB+9931F130, GB+9931F139]*/
  0,     38926, 0,     0,     16615, 37556, 42627, 10381, 0,     14031,   /*[GB+9931F230, GB+9931F239]*/
  0,     20737, 0,     0,     0,     0,     0,     0,     0,     4116,    /*[GB+9931F330, GB+9931F339]*/
  0,     0,     0,     0,     0,     21628, 0,     0,     0,     0,       /*[GB+9931F430, GB+9931F439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931F530, GB+9931F539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931F630, GB+9931F639]*/
  0,     0,     0,     0,     0,     0,     37552, 17578, 26761, 0,       /*[GB+9931F730, GB+9931F739]*/
  0,     0,     0,     40400, 16548, 0,     0,     0,     0,     0,       /*[GB+9931F830, GB+9931F839]*/
  9437,  29909, 0,     17457, 35395, 0,     0,     0,     0,     0,       /*[GB+9931F930, GB+9931F939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931FA30, GB+9931FA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931FB30, GB+9931FB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931FC30, GB+9931FC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931FD30, GB+9931FD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9931FE30, GB+9931FE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99328130, GB+99328139]*/
  0,     0,     0,     0,     0,     0,     13289, 0,     0,     0,       /*[GB+99328230, GB+99328239]*/
  0,     0,     0,     0,     0,     0,     0,     26217, 0,     0,       /*[GB+99328330, GB+99328339]*/
  0,     0,     0,     1047,  0,     0,     0,     0,     0,     0,       /*[GB+99328430, GB+99328439]*/
  27978, 0,     0,     0,     31924, 0,     0,     0,     0,     0,       /*[GB+99328530, GB+99328539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99328630, GB+99328639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99328730, GB+99328739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99328830, GB+99328839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99328930, GB+99328939]*/
  0,     0,     0,     0,     0,     23447, 0,     0,     0,     0,       /*[GB+99328A30, GB+99328A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99328B30, GB+99328B39]*/
  0,     0,     0,     0,     19742, 0,     29522, 0,     0,     0,       /*[GB+99328C30, GB+99328C39]*/
  0,     0,     13909, 0,     0,     0,     0,     0,     0,     0,       /*[GB+99328D30, GB+99328D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     16703,   /*[GB+99328E30, GB+99328E39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99328F30, GB+99328F39]*/
  23463, 0,     0,     0,     0,     0,     19444, 0,     0,     0,       /*[GB+99329030, GB+99329039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99329130, GB+99329139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     43898,   /*[GB+99329230, GB+99329239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99329330, GB+99329339]*/
  0,     0,     0,     34516, 0,     0,     0,     0,     4012,  0,       /*[GB+99329430, GB+99329439]*/
  28919, 0,     0,     0,     0,     0,     0,     0,     19844, 0,       /*[GB+99329530, GB+99329539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99329630, GB+99329639]*/
  0,     0,     34570, 0,     0,     0,     0,     0,     0,     0,       /*[GB+99329730, GB+99329739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99329830, GB+99329839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99329930, GB+99329939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99329A30, GB+99329A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99329B30, GB+99329B39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99329C30, GB+99329C39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99329D30, GB+99329D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99329E30, GB+99329E39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99329F30, GB+99329F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9932A030, GB+9932A039]*/
  0,     0,     0,     36282, 21758, 26262, 13864, 0,     5481,  40215,   /*[GB+9932A130, GB+9932A139]*/
  32260, 26382, 41284, 41754, 6276,  34501, 8887,  38481, 16922, 0,       /*[GB+9932A230, GB+9932A239]*/
  20214, 26639, 23571, 24852, 42032, 31443, 18714, 12097, 25489, 3321,    /*[GB+9932A330, GB+9932A339]*/
  21549, 43277, 27391, 37236, 41037, 35712, 0,     9405,  26218, 4637,    /*[GB+9932A430, GB+9932A439]*/
  3836,  0,     30847, 34120, 30099, 42395, 29374, 0,     0,     36833,   /*[GB+9932A530, GB+9932A539]*/
  36834, 25461, 0,     42948, 7693,  26872, 19139, 5560,  22124, 34638,   /*[GB+9932A630, GB+9932A639]*/
  0,     0,     19913, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9932A730, GB+9932A739]*/
  0,     0,     0,     0,     0,     0,     0,     1297,  0,     0,       /*[GB+9932A830, GB+9932A839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9932A930, GB+9932A939]*/
  0,     0,     0,     32742, 0,     0,     0,     0,     0,     0,       /*[GB+9932AA30, GB+9932AA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9932AB30, GB+9932AB39]*/
  0,     0,     0,     0,     0,     0,     29922, 0,     0,     0,       /*[GB+9932AC30, GB+9932AC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9932AD30, GB+9932AD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9932AE30, GB+9932AE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9932AF30, GB+9932AF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9932B030, GB+9932B039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9932B130, GB+9932B139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9932B230, GB+9932B239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9932B330, GB+9932B339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9932B430, GB+9932B439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9932B530, GB+9932B539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9932B630, GB+9932B639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9932B730, GB+9932B739]*/
  0,     0,     22979, 5655,  0,     0,     39868, 0,     0,     0,       /*[GB+9932B830, GB+9932B839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9932B930, GB+9932B939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9932BA30, GB+9932BA39]*/
  5271,  0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9932BB30, GB+9932BB39]*/
  0,     0,     0,     0,     0,     0,     0,     17437, 0,     7674,    /*[GB+9932BC30, GB+9932BC39]*/
  12532, 0,     0,     0,     20827, 0,     0,     18834, 0,     0,       /*[GB+9932BD30, GB+9932BD39]*/
  0,     0,     0,     0,     0,     0,     12023, 0,     0,     0,       /*[GB+9932BE30, GB+9932BE39]*/
  0,     0,     34324, 10448, 0,     39467, 0,     17709, 0,     0,       /*[GB+9932BF30, GB+9932BF39]*/
  0,     0,     0,     0,     29511, 14601, 0,     0,     0,     0,       /*[GB+9932C030, GB+9932C039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9932C130, GB+9932C139]*/
  0,     0,     0,     0,     0,     0,     40885, 0,     0,     0,       /*[GB+9932C230, GB+9932C239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9932C330, GB+9932C339]*/
  0,     24370, 0,     0,     0,     0,     26864, 0,     0,     0,       /*[GB+9932C430, GB+9932C439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9932C530, GB+9932C539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     14053,   /*[GB+9932C630, GB+9932C639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     17463,   /*[GB+9932C730, GB+9932C739]*/
  0,     0,     31903, 0,     0,     0,     0,     0,     0,     24374,   /*[GB+9932C830, GB+9932C839]*/
  0,     0,     0,     27842, 0,     0,     0,     0,     0,     0,       /*[GB+9932C930, GB+9932C939]*/
  0,     10811, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9932CA30, GB+9932CA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9932CB30, GB+9932CB39]*/
  0,     0,     0,     38389, 0,     0,     0,     0,     0,     0,       /*[GB+9932CC30, GB+9932CC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9932CD30, GB+9932CD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9932CE30, GB+9932CE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9932CF30, GB+9932CF39]*/
  0,     0,     0,     0,     33955, 0,     0,     0,     0,     0,       /*[GB+9932D030, GB+9932D039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9932D130, GB+9932D139]*/
  4848,  0,     0,     6556,  0,     0,     0,     0,     0,     1321,    /*[GB+9932D230, GB+9932D239]*/
  0,     5602,  0,     0,     9561,  0,     0,     0,     0,     0,       /*[GB+9932D330, GB+9932D339]*/
  0,     0,     32536, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9932D430, GB+9932D439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     40888, 0,       /*[GB+9932D530, GB+9932D539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9932D630, GB+9932D639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9932D730, GB+9932D739]*/
  0,     32157, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9932D830, GB+9932D839]*/
  0,     0,     0,     0,     0,     0,     0,     22940, 0,     0,       /*[GB+9932D930, GB+9932D939]*/
  0,     0,     0,     6278,  42761, 0,     0,     0,     13636, 6329,    /*[GB+9932DA30, GB+9932DA39]*/
  28685, 27973, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9932DB30, GB+9932DB39]*/
  0,     0,     20575, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9932DC30, GB+9932DC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9932DD30, GB+9932DD39]*/
  0,     0,     0,     0,     0,     40653, 0,     0,     0,     3020,    /*[GB+9932DE30, GB+9932DE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9932DF30, GB+9932DF39]*/
  0,     2739,  6614,  4630,  29615, 31389, 42344, 0,     6850,  5449,    /*[GB+9932E030, GB+9932E039]*/
  25131, 38932, 21932, 0,     5832,  0,     0,     0,     0,     0,       /*[GB+9932E130, GB+9932E139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9932E230, GB+9932E239]*/
  0,     0,     0,     0,     0,     0,     42692, 0,     0,     0,       /*[GB+9932E330, GB+9932E339]*/
  0,     38786, 0,     0,     0,     2453,  0,     0,     0,     0,       /*[GB+9932E430, GB+9932E439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     32919, 0,       /*[GB+9932E530, GB+9932E539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9932E630, GB+9932E639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9932E730, GB+9932E739]*/
  0,     0,     0,     35677, 16629, 42253, 10946, 16046, 42930, 31681,   /*[GB+9932E830, GB+9932E839]*/
  9020,  13998, 32051, 42975, 13865, 30823, 37411, 7990,  36482, 32561,   /*[GB+9932E930, GB+9932E939]*/
  12518, 0,     12519, 5624,  21957, 26993, 12520, 42748, 24702, 11242,   /*[GB+9932EA30, GB+9932EA39]*/
  0,     0,     41069, 4934,  10745, 4935,  0,     31510, 17657, 18609,   /*[GB+9932EB30, GB+9932EB39]*/
  40789, 17699, 10906, 30880, 28052, 0,     2443,  13143, 10380, 38496,   /*[GB+9932EC30, GB+9932EC39]*/
  0,     12545, 28921, 4888,  41035, 12262, 35974, 7140,  9230,  9287,    /*[GB+9932ED30, GB+9932ED39]*/
  11100, 3641,  0,     30320, 10382, 0,     3649,  11611, 38763, 2956,    /*[GB+9932EE30, GB+9932EE39]*/
  40955, 12589, 38289, 18893, 36690, 393,   43996, 7016,  40905, 43139,   /*[GB+9932EF30, GB+9932EF39]*/
  37871, 0,     40767, 119,   30406, 10927, 31890, 12637, 40643, 13958,   /*[GB+9932F030, GB+9932F039]*/
  12067, 31940, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9932F130, GB+9932F139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9932F230, GB+9932F239]*/
  41608, 0,     0,     0,     21061, 0,     0,     0,     0,     0,       /*[GB+9932F330, GB+9932F339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9932F430, GB+9932F439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9932F530, GB+9932F539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9932F630, GB+9932F639]*/
  0,     0,     0,     0,     0,     0,     0,     2498,  10612, 38130,   /*[GB+9932F730, GB+9932F739]*/
  1271,  41964, 4389,  0,     36942, 27396, 6926,  11552, 17585, 41325,   /*[GB+9932F830, GB+9932F839]*/
  0,     0,     0,     0,     0,     30691, 31862, 0,     0,     0,       /*[GB+9932F930, GB+9932F939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9932FA30, GB+9932FA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9932FB30, GB+9932FB39]*/
  0,     0,     0,     0,     0,     18365, 0,     957,   0,     20420,   /*[GB+9932FC30, GB+9932FC39]*/
  0,     43759, 0,     7236,  0,     0,     0,     35067, 0,     0,       /*[GB+9932FD30, GB+9932FD39]*/
  0,     0,     1930,  41360, 0,     0,     0,     0,     0,     10055,   /*[GB+9932FE30, GB+9932FE39]*/
  0,     0,     0,     0,     25571, 0,     0,     30968, 0,     0,       /*[GB+99338130, GB+99338139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99338230, GB+99338239]*/
  0,     22906, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99338330, GB+99338339]*/
  6141,  0,     0,     0,     0,     0,     0,     0,     19739, 0,       /*[GB+99338430, GB+99338439]*/
  17792, 0,     0,     0,     0,     10465, 0,     0,     0,     0,       /*[GB+99338530, GB+99338539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99338630, GB+99338639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     20955,   /*[GB+99338730, GB+99338739]*/
  24844, 0,     37154, 22882, 9554,  16148, 32387, 31228, 0,     40695,   /*[GB+99338830, GB+99338839]*/
  24170, 9297,  39242, 35802, 12590, 27685, 10102, 3195,  1002,  0,       /*[GB+99338930, GB+99338939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99338A30, GB+99338A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99338B30, GB+99338B39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99338C30, GB+99338C39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99338D30, GB+99338D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99338E30, GB+99338E39]*/
  0,     0,     0,     0,     0,     0,     12850, 0,     0,     0,       /*[GB+99338F30, GB+99338F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99339030, GB+99339039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99339130, GB+99339139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99339230, GB+99339239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99339330, GB+99339339]*/
  0,     0,     0,     36299, 17343, 0,     0,     0,     0,     0,       /*[GB+99339430, GB+99339439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     21049,   /*[GB+99339530, GB+99339539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99339630, GB+99339639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99339730, GB+99339739]*/
  0,     0,     0,     0,     0,     39761, 41237, 23245, 0,     0,       /*[GB+99339830, GB+99339839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99339930, GB+99339939]*/
  0,     0,     0,     0,     10125, 0,     0,     0,     0,     0,       /*[GB+99339A30, GB+99339A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99339B30, GB+99339B39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99339C30, GB+99339C39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99339D30, GB+99339D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99339E30, GB+99339E39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99339F30, GB+99339F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9933A030, GB+9933A039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9933A130, GB+9933A139]*/
  0,     41523, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9933A230, GB+9933A239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9933A330, GB+9933A339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9933A430, GB+9933A439]*/
  0,     0,     0,     10743, 11920, 38098, 16631, 23991, 32634, 20408,   /*[GB+9933A530, GB+9933A539]*/
  3117,  26994, 25538, 14806, 24162, 24853, 0,     3359,  0,     28777,   /*[GB+9933A630, GB+9933A639]*/
  29224, 28479, 4159,  19909, 0,     44019, 23141, 41438, 29620, 22228,   /*[GB+9933A730, GB+9933A739]*/
  11383, 24004, 0,     34720, 7180,  10449, 2698,  12964, 6884,  0,       /*[GB+9933A830, GB+9933A839]*/
  11936, 10774, 11402, 37112, 0,     43617, 0,     31286, 4606,  30896,   /*[GB+9933A930, GB+9933A939]*/
  23908, 22674, 0,     0,     4546,  1975,  17972, 27631, 5227,  4903,    /*[GB+9933AA30, GB+9933AA39]*/
  2731,  41442, 17082, 34554, 11508, 4799,  23304, 0,     4482,  33088,   /*[GB+9933AB30, GB+9933AB39]*/
  34134, 41254, 12623, 24798, 0,     24341, 37896, 0,     30279, 33888,   /*[GB+9933AC30, GB+9933AC39]*/
  14530, 0,     11232, 3491,  19153, 0,     1667,  14411, 30137, 1452,    /*[GB+9933AD30, GB+9933AD39]*/
  0,     13078, 0,     2149,  0,     43902, 14455, 0,     0,     0,       /*[GB+9933AE30, GB+9933AE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9933AF30, GB+9933AF39]*/
  0,     0,     0,     0,     0,     0,     12801, 0,     0,     0,       /*[GB+9933B030, GB+9933B039]*/
  0,     0,     0,     19199, 34162, 13727, 11448, 16869, 11058, 4251,    /*[GB+9933B130, GB+9933B139]*/
  0,     0,     12765, 25105, 27924, 0,     0,     16965, 0,     26766,   /*[GB+9933B230, GB+9933B239]*/
  7336,  38509, 37379, 654,   242,   34534, 0,     0,     0,     5828,    /*[GB+9933B330, GB+9933B339]*/
  20663, 33424, 0,     0,     0,     8905,  12945, 0,     0,     0,       /*[GB+9933B430, GB+9933B439]*/
  0,     34104, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9933B530, GB+9933B539]*/
  31912, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9933B630, GB+9933B639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9933B730, GB+9933B739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9933B830, GB+9933B839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9933B930, GB+9933B939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9933BA30, GB+9933BA39]*/
  0,     0,     0,     22291, 0,     0,     0,     0,     0,     0,       /*[GB+9933BB30, GB+9933BB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9933BC30, GB+9933BC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9933BD30, GB+9933BD39]*/
  0,     0,     0,     20534, 0,     13970, 42785, 18847, 0,     0,       /*[GB+9933BE30, GB+9933BE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     10741,   /*[GB+9933BF30, GB+9933BF39]*/
  0,     6327,  0,     0,     30125, 2829,  39935, 0,     37589, 0,       /*[GB+9933C030, GB+9933C039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9933C130, GB+9933C139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9933C230, GB+9933C239]*/
  0,     0,     0,     16984, 39366, 22062, 5932,  5439,  0,     23648,   /*[GB+9933C330, GB+9933C339]*/
  0,     43132, 17969, 9375,  0,     32767, 17214, 0,     33068, 3412,    /*[GB+9933C430, GB+9933C439]*/
  0,     11311, 24783, 0,     0,     0,     37872, 6516,  27635, 24583,   /*[GB+9933C530, GB+9933C539]*/
  299,   29025, 24031, 41268, 0,     39950, 0,     32731, 0,     36209,   /*[GB+9933C630, GB+9933C639]*/
  36212, 0,     0,     0,     16149, 36109, 0,     10452, 11608, 0,       /*[GB+9933C730, GB+9933C739]*/
  0,     0,     0,     18913, 39875, 0,     0,     0,     0,     0,       /*[GB+9933C830, GB+9933C839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     40616,   /*[GB+9933C930, GB+9933C939]*/
  0,     0,     0,     0,     0,     0,     3503,  0,     0,     0,       /*[GB+9933CA30, GB+9933CA39]*/
  0,     0,     0,     0,     0,     0,     29590, 0,     0,     0,       /*[GB+9933CB30, GB+9933CB39]*/
  4421,  22862, 0,     14718, 31016, 9600,  28205, 0,     23136, 38099,   /*[GB+9933CC30, GB+9933CC39]*/
  20806, 0,     4664,  7527,  10519, 20861, 810,   31022, 37558, 0,       /*[GB+9933CD30, GB+9933CD39]*/
  11669, 0,     37473, 37274, 43232, 24173, 0,     30062, 13767, 0,       /*[GB+9933CE30, GB+9933CE39]*/
  0,     5224,  30898, 33643, 26355, 30734, 0,     0,     3967,  38946,   /*[GB+9933CF30, GB+9933CF39]*/
  5862,  39033, 21996, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9933D030, GB+9933D039]*/
  0,     0,     0,     8159,  0,     0,     0,     0,     0,     0,       /*[GB+9933D130, GB+9933D139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9933D230, GB+9933D239]*/
  0,     0,     3008,  7685,  23789, 23984, 15150, 9871,  33175, 0,       /*[GB+9933D330, GB+9933D339]*/
  0,     16870, 0,     28417, 3695,  31762, 17618, 237,   7922,  12768,   /*[GB+9933D430, GB+9933D439]*/
  0,     23909, 24915, 3860,  0,     0,     19142, 31416, 21293, 1668,    /*[GB+9933D530, GB+9933D539]*/
  32082, 13971, 12073, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9933D630, GB+9933D639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9933D730, GB+9933D739]*/
  0,     0,     0,     0,     0,     0,     22633, 22513, 0,     0,       /*[GB+9933D830, GB+9933D839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9933D930, GB+9933D939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9933DA30, GB+9933DA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9933DB30, GB+9933DB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9933DC30, GB+9933DC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9933DD30, GB+9933DD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9933DE30, GB+9933DE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9933DF30, GB+9933DF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9933E030, GB+9933E039]*/
  0,     0,     0,     0,     14592, 26995, 6877,  0,     6302,  0,       /*[GB+9933E130, GB+9933E139]*/
  741,   14762, 0,     33370, 9460,  15393, 5416,  32135, 14640, 32666,   /*[GB+9933E230, GB+9933E239]*/
  41499, 0,     0,     31487, 16346, 31073, 0,     17728, 0,     0,       /*[GB+9933E330, GB+9933E339]*/
  0,     28144, 0,     0,     11742, 17300, 18778, 0,     35452, 41053,   /*[GB+9933E430, GB+9933E439]*/
  31087, 0,     39268, 27262, 13497, 0,     22846, 12323, 26447, 13515,   /*[GB+9933E530, GB+9933E539]*/
  33894, 10510, 0,     15074, 34749, 0,     24649, 9997,  0,     0,       /*[GB+9933E630, GB+9933E639]*/
  0,     0,     10075, 539,   0,     0,     0,     0,     0,     0,       /*[GB+9933E730, GB+9933E739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9933E830, GB+9933E839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     2213,    /*[GB+9933E930, GB+9933E939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9933EA30, GB+9933EA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9933EB30, GB+9933EB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9933EC30, GB+9933EC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     38069, 34758,   /*[GB+9933ED30, GB+9933ED39]*/
  11356, 0,     33001, 32215, 9221,  0,     0,     4160,  28585, 0,       /*[GB+9933EE30, GB+9933EE39]*/
  3637,  26141, 26998, 0,     27879, 4052,  18058, 17634, 4425,  40247,   /*[GB+9933EF30, GB+9933EF39]*/
  24919, 0,     0,     38766, 36684, 0,     36839, 21717, 40261, 12292,   /*[GB+9933F030, GB+9933F039]*/
  24223, 176,   20193, 0,     9243,  17466, 9133,  34254, 24304, 3936,    /*[GB+9933F130, GB+9933F139]*/
  43386, 36982, 24805, 369,   7759,  0,     16660, 36553, 17028, 24202,   /*[GB+9933F230, GB+9933F239]*/
  11219, 26551, 0,     0,     0,     41173, 3055,  4909,  1600,  9043,    /*[GB+9933F330, GB+9933F339]*/
  0,     0,     0,     0,     24965, 0,     0,     0,     0,     0,       /*[GB+9933F430, GB+9933F439]*/
  0,     2322,  0,     2081,  0,     12719, 0,     12721, 22984, 22126,   /*[GB+9933F530, GB+9933F539]*/
  28158, 0,     0,     0,     0,     0,     0,     41792, 23260, 17386,   /*[GB+9933F630, GB+9933F639]*/
  30985, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9933F730, GB+9933F739]*/
  0,     42845, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9933F830, GB+9933F839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9933F930, GB+9933F939]*/
  0,     0,     0,     0,     36467, 22372, 35196, 10872, 0,     0,       /*[GB+9933FA30, GB+9933FA39]*/
  40956, 42428, 23049, 40,    17221, 2964,  0,     5172,  37995, 40842,   /*[GB+9933FB30, GB+9933FB39]*/
  40417, 43831, 36473, 13821, 4647,  2130,  0,     0,     38673, 4230,    /*[GB+9933FC30, GB+9933FC39]*/
  14971, 2762,  15993, 0,     0,     0,     0,     0,     19632, 0,       /*[GB+9933FD30, GB+9933FD39]*/
  0,     37462, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9933FE30, GB+9933FE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99348130, GB+99348139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99348230, GB+99348239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99348330, GB+99348339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99348430, GB+99348439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99348530, GB+99348539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99348630, GB+99348639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99348730, GB+99348739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99348830, GB+99348839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99348930, GB+99348939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99348A30, GB+99348A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99348B30, GB+99348B39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99348C30, GB+99348C39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99348D30, GB+99348D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99348E30, GB+99348E39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99348F30, GB+99348F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99349030, GB+99349039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99349130, GB+99349139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99349230, GB+99349239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99349330, GB+99349339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99349430, GB+99349439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99349530, GB+99349539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99349630, GB+99349639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99349730, GB+99349739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99349830, GB+99349839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99349930, GB+99349939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99349A30, GB+99349A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99349B30, GB+99349B39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99349C30, GB+99349C39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99349D30, GB+99349D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99349E30, GB+99349E39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99349F30, GB+99349F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934A030, GB+9934A039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934A130, GB+9934A139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934A230, GB+9934A239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934A330, GB+9934A339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934A430, GB+9934A439]*/
  0,     0,     0,     0,     0,     0,     18024, 0,     0,     0,       /*[GB+9934A530, GB+9934A539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934A630, GB+9934A639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934A730, GB+9934A739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934A830, GB+9934A839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934A930, GB+9934A939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934AA30, GB+9934AA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934AB30, GB+9934AB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934AC30, GB+9934AC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934AD30, GB+9934AD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934AE30, GB+9934AE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934AF30, GB+9934AF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934B030, GB+9934B039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934B130, GB+9934B139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934B230, GB+9934B239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934B330, GB+9934B339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934B430, GB+9934B439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934B530, GB+9934B539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934B630, GB+9934B639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934B730, GB+9934B739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934B830, GB+9934B839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934B930, GB+9934B939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934BA30, GB+9934BA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934BB30, GB+9934BB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934BC30, GB+9934BC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934BD30, GB+9934BD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934BE30, GB+9934BE39]*/
  0,     0,     0,     0,     0,     0,     0,     13944, 0,     0,       /*[GB+9934BF30, GB+9934BF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934C030, GB+9934C039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934C130, GB+9934C139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934C230, GB+9934C239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934C330, GB+9934C339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934C430, GB+9934C439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934C530, GB+9934C539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934C630, GB+9934C639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934C730, GB+9934C739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934C830, GB+9934C839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934C930, GB+9934C939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934CA30, GB+9934CA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934CB30, GB+9934CB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934CC30, GB+9934CC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934CD30, GB+9934CD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934CE30, GB+9934CE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934CF30, GB+9934CF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934D030, GB+9934D039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934D130, GB+9934D139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934D230, GB+9934D239]*/
  24492, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934D330, GB+9934D339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     19466,   /*[GB+9934D430, GB+9934D439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934D530, GB+9934D539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934D630, GB+9934D639]*/
  0,     0,     0,     0,     22223, 38971, 0,     0,     0,     0,       /*[GB+9934D730, GB+9934D739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934D830, GB+9934D839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934D930, GB+9934D939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934DA30, GB+9934DA39]*/
  0,     0,     12095, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9934DB30, GB+9934DB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934DC30, GB+9934DC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934DD30, GB+9934DD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934DE30, GB+9934DE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934DF30, GB+9934DF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934E030, GB+9934E039]*/
  43930, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934E130, GB+9934E139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934E230, GB+9934E239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934E330, GB+9934E339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934E430, GB+9934E439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934E530, GB+9934E539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934E630, GB+9934E639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934E730, GB+9934E739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934E830, GB+9934E839]*/
  17291, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934E930, GB+9934E939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934EA30, GB+9934EA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934EB30, GB+9934EB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934EC30, GB+9934EC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934ED30, GB+9934ED39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934EE30, GB+9934EE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934EF30, GB+9934EF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934F030, GB+9934F039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934F130, GB+9934F139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934F230, GB+9934F239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934F330, GB+9934F339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934F430, GB+9934F439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934F530, GB+9934F539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934F630, GB+9934F639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934F730, GB+9934F739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934F830, GB+9934F839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934F930, GB+9934F939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934FA30, GB+9934FA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934FB30, GB+9934FB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934FC30, GB+9934FC39]*/
  0,     0,     43626, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9934FD30, GB+9934FD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9934FE30, GB+9934FE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     5716,  0,       /*[GB+99358130, GB+99358139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99358230, GB+99358239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99358330, GB+99358339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99358430, GB+99358439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99358530, GB+99358539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99358630, GB+99358639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99358730, GB+99358739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99358830, GB+99358839]*/
  0,     0,     98,    0,     0,     0,     0,     0,     0,     0,       /*[GB+99358930, GB+99358939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99358A30, GB+99358A39]*/
  32790, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99358B30, GB+99358B39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99358C30, GB+99358C39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99358D30, GB+99358D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99358E30, GB+99358E39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99358F30, GB+99358F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99359030, GB+99359039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99359130, GB+99359139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99359230, GB+99359239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99359330, GB+99359339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99359430, GB+99359439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99359530, GB+99359539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99359630, GB+99359639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99359730, GB+99359739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99359830, GB+99359839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99359930, GB+99359939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99359A30, GB+99359A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99359B30, GB+99359B39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99359C30, GB+99359C39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99359D30, GB+99359D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99359E30, GB+99359E39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99359F30, GB+99359F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935A030, GB+9935A039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935A130, GB+9935A139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935A230, GB+9935A239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935A330, GB+9935A339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935A430, GB+9935A439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935A530, GB+9935A539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935A630, GB+9935A639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935A730, GB+9935A739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935A830, GB+9935A839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935A930, GB+9935A939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935AA30, GB+9935AA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935AB30, GB+9935AB39]*/
  0,     0,     0,     0,     30234, 0,     0,     0,     0,     0,       /*[GB+9935AC30, GB+9935AC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935AD30, GB+9935AD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935AE30, GB+9935AE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935AF30, GB+9935AF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935B030, GB+9935B039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935B130, GB+9935B139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935B230, GB+9935B239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935B330, GB+9935B339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935B430, GB+9935B439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935B530, GB+9935B539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935B630, GB+9935B639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935B730, GB+9935B739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935B830, GB+9935B839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935B930, GB+9935B939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935BA30, GB+9935BA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     16405,   /*[GB+9935BB30, GB+9935BB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935BC30, GB+9935BC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935BD30, GB+9935BD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935BE30, GB+9935BE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935BF30, GB+9935BF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     11571,   /*[GB+9935C030, GB+9935C039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935C130, GB+9935C139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935C230, GB+9935C239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935C330, GB+9935C339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935C430, GB+9935C439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935C530, GB+9935C539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935C630, GB+9935C639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935C730, GB+9935C739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935C830, GB+9935C839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935C930, GB+9935C939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935CA30, GB+9935CA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935CB30, GB+9935CB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935CC30, GB+9935CC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935CD30, GB+9935CD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935CE30, GB+9935CE39]*/
  0,     0,     0,     0,     0,     0,     10395, 0,     0,     0,       /*[GB+9935CF30, GB+9935CF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935D030, GB+9935D039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935D130, GB+9935D139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935D230, GB+9935D239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935D330, GB+9935D339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935D430, GB+9935D439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935D530, GB+9935D539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935D630, GB+9935D639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935D730, GB+9935D739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935D830, GB+9935D839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935D930, GB+9935D939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935DA30, GB+9935DA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935DB30, GB+9935DB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935DC30, GB+9935DC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935DD30, GB+9935DD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935DE30, GB+9935DE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935DF30, GB+9935DF39]*/
  0,     19270, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935E030, GB+9935E039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935E130, GB+9935E139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935E230, GB+9935E239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935E330, GB+9935E339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935E430, GB+9935E439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     91,    0,       /*[GB+9935E530, GB+9935E539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935E630, GB+9935E639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935E730, GB+9935E739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935E830, GB+9935E839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935E930, GB+9935E939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935EA30, GB+9935EA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935EB30, GB+9935EB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935EC30, GB+9935EC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935ED30, GB+9935ED39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935EE30, GB+9935EE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935EF30, GB+9935EF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935F030, GB+9935F039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935F130, GB+9935F139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935F230, GB+9935F239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935F330, GB+9935F339]*/
  0,     0,     0,     0,     0,     0,     0,     24630, 0,     0,       /*[GB+9935F430, GB+9935F439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935F530, GB+9935F539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935F630, GB+9935F639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935F730, GB+9935F739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935F830, GB+9935F839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935F930, GB+9935F939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935FA30, GB+9935FA39]*/
  0,     0,     0,     0,     28499, 0,     0,     0,     0,     0,       /*[GB+9935FB30, GB+9935FB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935FC30, GB+9935FC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935FD30, GB+9935FD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9935FE30, GB+9935FE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99368130, GB+99368139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99368230, GB+99368239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99368330, GB+99368339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99368430, GB+99368439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99368530, GB+99368539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99368630, GB+99368639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99368730, GB+99368739]*/
  0,     6128,  0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99368830, GB+99368839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99368930, GB+99368939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99368A30, GB+99368A39]*/
  0,     0,     0,     38115, 0,     0,     0,     0,     0,     0,       /*[GB+99368B30, GB+99368B39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99368C30, GB+99368C39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99368D30, GB+99368D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99368E30, GB+99368E39]*/
  0,     0,     18137, 0,     0,     0,     0,     0,     0,     0,       /*[GB+99368F30, GB+99368F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99369030, GB+99369039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99369130, GB+99369139]*/
  0,     0,     0,     0,     0,     0,     39443, 0,     0,     0,       /*[GB+99369230, GB+99369239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99369330, GB+99369339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99369430, GB+99369439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99369530, GB+99369539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99369630, GB+99369639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99369730, GB+99369739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99369830, GB+99369839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99369930, GB+99369939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99369A30, GB+99369A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99369B30, GB+99369B39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99369C30, GB+99369C39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99369D30, GB+99369D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99369E30, GB+99369E39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99369F30, GB+99369F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936A030, GB+9936A039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936A130, GB+9936A139]*/
  0,     30794, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936A230, GB+9936A239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936A330, GB+9936A339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936A430, GB+9936A439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936A530, GB+9936A539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936A630, GB+9936A639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936A730, GB+9936A739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936A830, GB+9936A839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936A930, GB+9936A939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936AA30, GB+9936AA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936AB30, GB+9936AB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936AC30, GB+9936AC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936AD30, GB+9936AD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936AE30, GB+9936AE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936AF30, GB+9936AF39]*/
  0,     0,     0,     0,     37945, 21209, 0,     0,     0,     0,       /*[GB+9936B030, GB+9936B039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936B130, GB+9936B139]*/
  0,     0,     0,     0,     0,     0,     7642,  0,     0,     0,       /*[GB+9936B230, GB+9936B239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936B330, GB+9936B339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     43066, 0,       /*[GB+9936B430, GB+9936B439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936B530, GB+9936B539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936B630, GB+9936B639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936B730, GB+9936B739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936B830, GB+9936B839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936B930, GB+9936B939]*/
  0,     0,     0,     0,     0,     0,     17078, 0,     0,     0,       /*[GB+9936BA30, GB+9936BA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936BB30, GB+9936BB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936BC30, GB+9936BC39]*/
  0,     5935,  0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936BD30, GB+9936BD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936BE30, GB+9936BE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936BF30, GB+9936BF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936C030, GB+9936C039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936C130, GB+9936C139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936C230, GB+9936C239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936C330, GB+9936C339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936C430, GB+9936C439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936C530, GB+9936C539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936C630, GB+9936C639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936C730, GB+9936C739]*/
  0,     0,     36520, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9936C830, GB+9936C839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936C930, GB+9936C939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936CA30, GB+9936CA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936CB30, GB+9936CB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936CC30, GB+9936CC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936CD30, GB+9936CD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936CE30, GB+9936CE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936CF30, GB+9936CF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936D030, GB+9936D039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936D130, GB+9936D139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936D230, GB+9936D239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936D330, GB+9936D339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936D430, GB+9936D439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936D530, GB+9936D539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936D630, GB+9936D639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936D730, GB+9936D739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936D830, GB+9936D839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936D930, GB+9936D939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936DA30, GB+9936DA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936DB30, GB+9936DB39]*/
  5596,  0,     0,     0,     27860, 0,     0,     0,     0,     0,       /*[GB+9936DC30, GB+9936DC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936DD30, GB+9936DD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936DE30, GB+9936DE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936DF30, GB+9936DF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936E030, GB+9936E039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936E130, GB+9936E139]*/
  11453, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936E230, GB+9936E239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936E330, GB+9936E339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936E430, GB+9936E439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936E530, GB+9936E539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936E630, GB+9936E639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936E730, GB+9936E739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936E830, GB+9936E839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936E930, GB+9936E939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936EA30, GB+9936EA39]*/
  0,     0,     0,     0,     0,     0,     0,     42896, 0,     0,       /*[GB+9936EB30, GB+9936EB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936EC30, GB+9936EC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936ED30, GB+9936ED39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936EE30, GB+9936EE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936EF30, GB+9936EF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936F030, GB+9936F039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936F130, GB+9936F139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936F230, GB+9936F239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936F330, GB+9936F339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936F430, GB+9936F439]*/
  0,     0,     33281, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9936F530, GB+9936F539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936F630, GB+9936F639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936F730, GB+9936F739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936F830, GB+9936F839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936F930, GB+9936F939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936FA30, GB+9936FA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936FB30, GB+9936FB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936FC30, GB+9936FC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936FD30, GB+9936FD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9936FE30, GB+9936FE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99378130, GB+99378139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99378230, GB+99378239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99378330, GB+99378339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99378430, GB+99378439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99378530, GB+99378539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99378630, GB+99378639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99378730, GB+99378739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99378830, GB+99378839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99378930, GB+99378939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99378A30, GB+99378A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99378B30, GB+99378B39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99378C30, GB+99378C39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99378D30, GB+99378D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99378E30, GB+99378E39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99378F30, GB+99378F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99379030, GB+99379039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99379130, GB+99379139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99379230, GB+99379239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99379330, GB+99379339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99379430, GB+99379439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99379530, GB+99379539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99379630, GB+99379639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99379730, GB+99379739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99379830, GB+99379839]*/
  20467, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99379930, GB+99379939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99379A30, GB+99379A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99379B30, GB+99379B39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99379C30, GB+99379C39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99379D30, GB+99379D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99379E30, GB+99379E39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99379F30, GB+99379F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937A030, GB+9937A039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937A130, GB+9937A139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937A230, GB+9937A239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937A330, GB+9937A339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937A430, GB+9937A439]*/
  17360, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937A530, GB+9937A539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937A630, GB+9937A639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937A730, GB+9937A739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937A830, GB+9937A839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937A930, GB+9937A939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937AA30, GB+9937AA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937AB30, GB+9937AB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937AC30, GB+9937AC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937AD30, GB+9937AD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937AE30, GB+9937AE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937AF30, GB+9937AF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937B030, GB+9937B039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937B130, GB+9937B139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937B230, GB+9937B239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937B330, GB+9937B339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937B430, GB+9937B439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937B530, GB+9937B539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937B630, GB+9937B639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937B730, GB+9937B739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937B830, GB+9937B839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937B930, GB+9937B939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937BA30, GB+9937BA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937BB30, GB+9937BB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937BC30, GB+9937BC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937BD30, GB+9937BD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937BE30, GB+9937BE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937BF30, GB+9937BF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937C030, GB+9937C039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937C130, GB+9937C139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937C230, GB+9937C239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937C330, GB+9937C339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937C430, GB+9937C439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937C530, GB+9937C539]*/
  0,     0,     0,     0,     0,     0,     17715, 0,     0,     0,       /*[GB+9937C630, GB+9937C639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937C730, GB+9937C739]*/
  9462,  0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937C830, GB+9937C839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937C930, GB+9937C939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937CA30, GB+9937CA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937CB30, GB+9937CB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937CC30, GB+9937CC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937CD30, GB+9937CD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937CE30, GB+9937CE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937CF30, GB+9937CF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937D030, GB+9937D039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937D130, GB+9937D139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937D230, GB+9937D239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937D330, GB+9937D339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937D430, GB+9937D439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937D530, GB+9937D539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937D630, GB+9937D639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937D730, GB+9937D739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937D830, GB+9937D839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937D930, GB+9937D939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937DA30, GB+9937DA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937DB30, GB+9937DB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937DC30, GB+9937DC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937DD30, GB+9937DD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937DE30, GB+9937DE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937DF30, GB+9937DF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937E030, GB+9937E039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937E130, GB+9937E139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937E230, GB+9937E239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937E330, GB+9937E339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937E430, GB+9937E439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937E530, GB+9937E539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937E630, GB+9937E639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937E730, GB+9937E739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937E830, GB+9937E839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937E930, GB+9937E939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     17486,   /*[GB+9937EA30, GB+9937EA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937EB30, GB+9937EB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937EC30, GB+9937EC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937ED30, GB+9937ED39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937EE30, GB+9937EE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937EF30, GB+9937EF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937F030, GB+9937F039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937F130, GB+9937F139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937F230, GB+9937F239]*/
  0,     0,     0,     0,     19317, 0,     0,     0,     0,     0,       /*[GB+9937F330, GB+9937F339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937F430, GB+9937F439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937F530, GB+9937F539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937F630, GB+9937F639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937F730, GB+9937F739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937F830, GB+9937F839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937F930, GB+9937F939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937FA30, GB+9937FA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937FB30, GB+9937FB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937FC30, GB+9937FC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937FD30, GB+9937FD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9937FE30, GB+9937FE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     26256, 19898,   /*[GB+99388130, GB+99388139]*/
  42267, 44014, 9925,  0,     0,     18833, 9876,  0,     43754, 27421,   /*[GB+99388230, GB+99388239]*/
  5076,  0,     30102, 2351,  0,     0,     0,     0,     0,     0,       /*[GB+99388330, GB+99388339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99388430, GB+99388439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99388530, GB+99388539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99388630, GB+99388639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99388730, GB+99388739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99388830, GB+99388839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99388930, GB+99388939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99388A30, GB+99388A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99388B30, GB+99388B39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99388C30, GB+99388C39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99388D30, GB+99388D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99388E30, GB+99388E39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99388F30, GB+99388F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99389030, GB+99389039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99389130, GB+99389139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99389230, GB+99389239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99389330, GB+99389339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99389430, GB+99389439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99389530, GB+99389539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99389630, GB+99389639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99389730, GB+99389739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99389830, GB+99389839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99389930, GB+99389939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99389A30, GB+99389A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99389B30, GB+99389B39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99389C30, GB+99389C39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99389D30, GB+99389D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99389E30, GB+99389E39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99389F30, GB+99389F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938A030, GB+9938A039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938A130, GB+9938A139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938A230, GB+9938A239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938A330, GB+9938A339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938A430, GB+9938A439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938A530, GB+9938A539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938A630, GB+9938A639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938A730, GB+9938A739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938A830, GB+9938A839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938A930, GB+9938A939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     6778,  0,       /*[GB+9938AA30, GB+9938AA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938AB30, GB+9938AB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938AC30, GB+9938AC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938AD30, GB+9938AD39]*/
  7739,  0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938AE30, GB+9938AE39]*/
  0,     0,     0,     0,     38780, 0,     0,     0,     0,     0,       /*[GB+9938AF30, GB+9938AF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938B030, GB+9938B039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938B130, GB+9938B139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938B230, GB+9938B239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938B330, GB+9938B339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938B430, GB+9938B439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938B530, GB+9938B539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938B630, GB+9938B639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938B730, GB+9938B739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938B830, GB+9938B839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938B930, GB+9938B939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938BA30, GB+9938BA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938BB30, GB+9938BB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938BC30, GB+9938BC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938BD30, GB+9938BD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938BE30, GB+9938BE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938BF30, GB+9938BF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938C030, GB+9938C039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938C130, GB+9938C139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938C230, GB+9938C239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938C330, GB+9938C339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938C430, GB+9938C439]*/
  0,     0,     14602, 0,     0,     8750,  0,     0,     0,     0,       /*[GB+9938C530, GB+9938C539]*/
  18840, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938C630, GB+9938C639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938C730, GB+9938C739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938C830, GB+9938C839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938C930, GB+9938C939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938CA30, GB+9938CA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938CB30, GB+9938CB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938CC30, GB+9938CC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938CD30, GB+9938CD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938CE30, GB+9938CE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938CF30, GB+9938CF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938D030, GB+9938D039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     18839,   /*[GB+9938D130, GB+9938D139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938D230, GB+9938D239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938D330, GB+9938D339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938D430, GB+9938D439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938D530, GB+9938D539]*/
  0,     0,     0,     0,     0,     0,     0,     35098, 0,     0,       /*[GB+9938D630, GB+9938D639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938D730, GB+9938D739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938D830, GB+9938D839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938D930, GB+9938D939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938DA30, GB+9938DA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938DB30, GB+9938DB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938DC30, GB+9938DC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938DD30, GB+9938DD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938DE30, GB+9938DE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938DF30, GB+9938DF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938E030, GB+9938E039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938E130, GB+9938E139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938E230, GB+9938E239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938E330, GB+9938E339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938E430, GB+9938E439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938E530, GB+9938E539]*/
  38650, 6801,  0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938E630, GB+9938E639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938E730, GB+9938E739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938E830, GB+9938E839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938E930, GB+9938E939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938EA30, GB+9938EA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938EB30, GB+9938EB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938EC30, GB+9938EC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938ED30, GB+9938ED39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938EE30, GB+9938EE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938EF30, GB+9938EF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938F030, GB+9938F039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938F130, GB+9938F139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938F230, GB+9938F239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938F330, GB+9938F339]*/
  0,     30731, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938F430, GB+9938F439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938F530, GB+9938F539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938F630, GB+9938F639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938F730, GB+9938F739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938F830, GB+9938F839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938F930, GB+9938F939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938FA30, GB+9938FA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938FB30, GB+9938FB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938FC30, GB+9938FC39]*/
  0,     0,     0,     0,     0,     0,     5765,  0,     0,     0,       /*[GB+9938FD30, GB+9938FD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9938FE30, GB+9938FE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99398130, GB+99398139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99398230, GB+99398239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99398330, GB+99398339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99398430, GB+99398439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     9704,  7551,    /*[GB+99398530, GB+99398539]*/
  0,     8136,  4510,  12131, 9325,  0,     0,     0,     0,     0,       /*[GB+99398630, GB+99398639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99398730, GB+99398739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99398830, GB+99398839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99398930, GB+99398939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99398A30, GB+99398A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99398B30, GB+99398B39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99398C30, GB+99398C39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99398D30, GB+99398D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99398E30, GB+99398E39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99398F30, GB+99398F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99399030, GB+99399039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99399130, GB+99399139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99399230, GB+99399239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99399330, GB+99399339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99399430, GB+99399439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99399530, GB+99399539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     25544, 0,       /*[GB+99399630, GB+99399639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99399730, GB+99399739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     3417,    /*[GB+99399830, GB+99399839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99399930, GB+99399939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99399A30, GB+99399A39]*/
  16414, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99399B30, GB+99399B39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99399C30, GB+99399C39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99399D30, GB+99399D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99399E30, GB+99399E39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+99399F30, GB+99399F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939A030, GB+9939A039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939A130, GB+9939A139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939A230, GB+9939A239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939A330, GB+9939A339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939A430, GB+9939A439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939A530, GB+9939A539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939A630, GB+9939A639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939A730, GB+9939A739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939A830, GB+9939A839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939A930, GB+9939A939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939AA30, GB+9939AA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939AB30, GB+9939AB39]*/
  9070,  11371, 2843,  43244, 20281, 0,     0,     0,     0,     0,       /*[GB+9939AC30, GB+9939AC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939AD30, GB+9939AD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939AE30, GB+9939AE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939AF30, GB+9939AF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939B030, GB+9939B039]*/
  0,     0,     0,     0,     0,     0,     0,     21212, 4236,  0,       /*[GB+9939B130, GB+9939B139]*/
  0,     27442, 0,     0,     1252,  0,     0,     0,     0,     0,       /*[GB+9939B230, GB+9939B239]*/
  7307,  0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939B330, GB+9939B339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939B430, GB+9939B439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939B530, GB+9939B539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939B630, GB+9939B639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939B730, GB+9939B739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939B830, GB+9939B839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939B930, GB+9939B939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939BA30, GB+9939BA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939BB30, GB+9939BB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939BC30, GB+9939BC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939BD30, GB+9939BD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939BE30, GB+9939BE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939BF30, GB+9939BF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939C030, GB+9939C039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939C130, GB+9939C139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939C230, GB+9939C239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939C330, GB+9939C339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939C430, GB+9939C439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     3136,  32745,   /*[GB+9939C530, GB+9939C539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939C630, GB+9939C639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939C730, GB+9939C739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939C830, GB+9939C839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939C930, GB+9939C939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939CA30, GB+9939CA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939CB30, GB+9939CB39]*/
  0,     0,     8253,  19599, 0,     0,     0,     0,     0,     0,       /*[GB+9939CC30, GB+9939CC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939CD30, GB+9939CD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939CE30, GB+9939CE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939CF30, GB+9939CF39]*/
  0,     1274,  10579, 85,    38321, 0,     0,     0,     0,     0,       /*[GB+9939D030, GB+9939D039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939D130, GB+9939D139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939D230, GB+9939D239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939D330, GB+9939D339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939D430, GB+9939D439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939D530, GB+9939D539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939D630, GB+9939D639]*/
  0,     20891, 10587, 43651, 2540,  7061,  0,     0,     0,     0,       /*[GB+9939D730, GB+9939D739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939D830, GB+9939D839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939D930, GB+9939D939]*/
  0,     0,     0,     0,     23154, 0,     0,     0,     0,     0,       /*[GB+9939DA30, GB+9939DA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939DB30, GB+9939DB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939DC30, GB+9939DC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939DD30, GB+9939DD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939DE30, GB+9939DE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939DF30, GB+9939DF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939E030, GB+9939E039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939E130, GB+9939E139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939E230, GB+9939E239]*/
  0,     0,     0,     35891, 38294, 10302, 2375,  12865, 20132, 11956,   /*[GB+9939E330, GB+9939E339]*/
  33093, 34386, 41741, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9939E430, GB+9939E439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939E530, GB+9939E539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939E630, GB+9939E639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939E730, GB+9939E739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939E830, GB+9939E839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939E930, GB+9939E939]*/
  0,     0,     0,     14751, 37227, 0,     21764, 9503,  3380,  0,       /*[GB+9939EA30, GB+9939EA39]*/
  36968, 0,     29345, 0,     0,     0,     0,     0,     9101,  0,       /*[GB+9939EB30, GB+9939EB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939EC30, GB+9939EC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939ED30, GB+9939ED39]*/
  0,     22173, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939EE30, GB+9939EE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939EF30, GB+9939EF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939F030, GB+9939F039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939F130, GB+9939F139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939F230, GB+9939F239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939F330, GB+9939F339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939F430, GB+9939F439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939F530, GB+9939F539]*/
  0,     0,     0,     16476, 0,     0,     0,     0,     0,     0,       /*[GB+9939F630, GB+9939F639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939F730, GB+9939F739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939F830, GB+9939F839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939F930, GB+9939F939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939FA30, GB+9939FA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939FB30, GB+9939FB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939FC30, GB+9939FC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939FD30, GB+9939FD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9939FE30, GB+9939FE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A308130, GB+9A308139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A308230, GB+9A308239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A308330, GB+9A308339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A308430, GB+9A308439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A308530, GB+9A308539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A308630, GB+9A308639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A308730, GB+9A308739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A308830, GB+9A308839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A308930, GB+9A308939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A308A30, GB+9A308A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A308B30, GB+9A308B39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A308C30, GB+9A308C39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A308D30, GB+9A308D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A308E30, GB+9A308E39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A308F30, GB+9A308F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A309030, GB+9A309039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A309130, GB+9A309139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A309230, GB+9A309239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A309330, GB+9A309339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A309430, GB+9A309439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A309530, GB+9A309539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A309630, GB+9A309639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A309730, GB+9A309739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A309830, GB+9A309839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A309930, GB+9A309939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A309A30, GB+9A309A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A309B30, GB+9A309B39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A309C30, GB+9A309C39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A309D30, GB+9A309D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A309E30, GB+9A309E39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A309F30, GB+9A309F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30A030, GB+9A30A039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30A130, GB+9A30A139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30A230, GB+9A30A239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30A330, GB+9A30A339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30A430, GB+9A30A439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30A530, GB+9A30A539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30A630, GB+9A30A639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30A730, GB+9A30A739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30A830, GB+9A30A839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30A930, GB+9A30A939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30AA30, GB+9A30AA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30AB30, GB+9A30AB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30AC30, GB+9A30AC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30AD30, GB+9A30AD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30AE30, GB+9A30AE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30AF30, GB+9A30AF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30B030, GB+9A30B039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30B130, GB+9A30B139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30B230, GB+9A30B239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30B330, GB+9A30B339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30B430, GB+9A30B439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30B530, GB+9A30B539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30B630, GB+9A30B639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30B730, GB+9A30B739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30B830, GB+9A30B839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30B930, GB+9A30B939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30BA30, GB+9A30BA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30BB30, GB+9A30BB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30BC30, GB+9A30BC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30BD30, GB+9A30BD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30BE30, GB+9A30BE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30BF30, GB+9A30BF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30C030, GB+9A30C039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30C130, GB+9A30C139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30C230, GB+9A30C239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30C330, GB+9A30C339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30C430, GB+9A30C439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30C530, GB+9A30C539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30C630, GB+9A30C639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30C730, GB+9A30C739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30C830, GB+9A30C839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30C930, GB+9A30C939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30CA30, GB+9A30CA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30CB30, GB+9A30CB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30CC30, GB+9A30CC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30CD30, GB+9A30CD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30CE30, GB+9A30CE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30CF30, GB+9A30CF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30D030, GB+9A30D039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30D130, GB+9A30D139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30D230, GB+9A30D239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30D330, GB+9A30D339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30D430, GB+9A30D439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30D530, GB+9A30D539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30D630, GB+9A30D639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30D730, GB+9A30D739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30D830, GB+9A30D839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30D930, GB+9A30D939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30DA30, GB+9A30DA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30DB30, GB+9A30DB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30DC30, GB+9A30DC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30DD30, GB+9A30DD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30DE30, GB+9A30DE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30DF30, GB+9A30DF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30E030, GB+9A30E039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30E130, GB+9A30E139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30E230, GB+9A30E239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30E330, GB+9A30E339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30E430, GB+9A30E439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30E530, GB+9A30E539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30E630, GB+9A30E639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30E730, GB+9A30E739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30E830, GB+9A30E839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30E930, GB+9A30E939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30EA30, GB+9A30EA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30EB30, GB+9A30EB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30EC30, GB+9A30EC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30ED30, GB+9A30ED39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30EE30, GB+9A30EE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30EF30, GB+9A30EF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30F030, GB+9A30F039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30F130, GB+9A30F139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30F230, GB+9A30F239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30F330, GB+9A30F339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30F430, GB+9A30F439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30F530, GB+9A30F539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30F630, GB+9A30F639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30F730, GB+9A30F739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30F830, GB+9A30F839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30F930, GB+9A30F939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30FA30, GB+9A30FA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30FB30, GB+9A30FB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30FC30, GB+9A30FC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30FD30, GB+9A30FD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A30FE30, GB+9A30FE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A318130, GB+9A318139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A318230, GB+9A318239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A318330, GB+9A318339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A318430, GB+9A318439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A318530, GB+9A318539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A318630, GB+9A318639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A318730, GB+9A318739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A318830, GB+9A318839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A318930, GB+9A318939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A318A30, GB+9A318A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A318B30, GB+9A318B39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A318C30, GB+9A318C39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A318D30, GB+9A318D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A318E30, GB+9A318E39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A318F30, GB+9A318F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A319030, GB+9A319039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A319130, GB+9A319139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A319230, GB+9A319239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A319330, GB+9A319339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A319430, GB+9A319439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A319530, GB+9A319539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A319630, GB+9A319639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A319730, GB+9A319739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A319830, GB+9A319839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A319930, GB+9A319939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A319A30, GB+9A319A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A319B30, GB+9A319B39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A319C30, GB+9A319C39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A319D30, GB+9A319D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A319E30, GB+9A319E39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A319F30, GB+9A319F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31A030, GB+9A31A039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31A130, GB+9A31A139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31A230, GB+9A31A239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31A330, GB+9A31A339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31A430, GB+9A31A439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31A530, GB+9A31A539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31A630, GB+9A31A639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31A730, GB+9A31A739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31A830, GB+9A31A839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31A930, GB+9A31A939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31AA30, GB+9A31AA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31AB30, GB+9A31AB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31AC30, GB+9A31AC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31AD30, GB+9A31AD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31AE30, GB+9A31AE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31AF30, GB+9A31AF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31B030, GB+9A31B039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31B130, GB+9A31B139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31B230, GB+9A31B239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31B330, GB+9A31B339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31B430, GB+9A31B439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31B530, GB+9A31B539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31B630, GB+9A31B639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31B730, GB+9A31B739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31B830, GB+9A31B839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31B930, GB+9A31B939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31BA30, GB+9A31BA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31BB30, GB+9A31BB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31BC30, GB+9A31BC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31BD30, GB+9A31BD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31BE30, GB+9A31BE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31BF30, GB+9A31BF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31C030, GB+9A31C039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31C130, GB+9A31C139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31C230, GB+9A31C239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31C330, GB+9A31C339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31C430, GB+9A31C439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31C530, GB+9A31C539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31C630, GB+9A31C639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31C730, GB+9A31C739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31C830, GB+9A31C839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31C930, GB+9A31C939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31CA30, GB+9A31CA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31CB30, GB+9A31CB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31CC30, GB+9A31CC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31CD30, GB+9A31CD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31CE30, GB+9A31CE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31CF30, GB+9A31CF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31D030, GB+9A31D039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31D130, GB+9A31D139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31D230, GB+9A31D239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31D330, GB+9A31D339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31D430, GB+9A31D439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31D530, GB+9A31D539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31D630, GB+9A31D639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31D730, GB+9A31D739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31D830, GB+9A31D839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31D930, GB+9A31D939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31DA30, GB+9A31DA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31DB30, GB+9A31DB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31DC30, GB+9A31DC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31DD30, GB+9A31DD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31DE30, GB+9A31DE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31DF30, GB+9A31DF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31E030, GB+9A31E039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31E130, GB+9A31E139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31E230, GB+9A31E239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31E330, GB+9A31E339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31E430, GB+9A31E439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31E530, GB+9A31E539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31E630, GB+9A31E639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31E730, GB+9A31E739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31E830, GB+9A31E839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31E930, GB+9A31E939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31EA30, GB+9A31EA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31EB30, GB+9A31EB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31EC30, GB+9A31EC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31ED30, GB+9A31ED39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31EE30, GB+9A31EE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31EF30, GB+9A31EF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31F030, GB+9A31F039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31F130, GB+9A31F139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31F230, GB+9A31F239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31F330, GB+9A31F339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31F430, GB+9A31F439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31F530, GB+9A31F539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31F630, GB+9A31F639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31F730, GB+9A31F739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31F830, GB+9A31F839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31F930, GB+9A31F939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31FA30, GB+9A31FA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31FB30, GB+9A31FB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31FC30, GB+9A31FC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31FD30, GB+9A31FD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A31FE30, GB+9A31FE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A328130, GB+9A328139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A328230, GB+9A328239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A328330, GB+9A328339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A328430, GB+9A328439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A328530, GB+9A328539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A328630, GB+9A328639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A328730, GB+9A328739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A328830, GB+9A328839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A328930, GB+9A328939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A328A30, GB+9A328A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A328B30, GB+9A328B39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A328C30, GB+9A328C39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A328D30, GB+9A328D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A328E30, GB+9A328E39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A328F30, GB+9A328F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A329030, GB+9A329039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A329130, GB+9A329139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A329230, GB+9A329239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A329330, GB+9A329339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A329430, GB+9A329439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A329530, GB+9A329539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A329630, GB+9A329639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A329730, GB+9A329739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A329830, GB+9A329839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A329930, GB+9A329939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A329A30, GB+9A329A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A329B30, GB+9A329B39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A329C30, GB+9A329C39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A329D30, GB+9A329D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A329E30, GB+9A329E39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A329F30, GB+9A329F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32A030, GB+9A32A039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32A130, GB+9A32A139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32A230, GB+9A32A239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32A330, GB+9A32A339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32A430, GB+9A32A439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32A530, GB+9A32A539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32A630, GB+9A32A639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32A730, GB+9A32A739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32A830, GB+9A32A839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32A930, GB+9A32A939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32AA30, GB+9A32AA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32AB30, GB+9A32AB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32AC30, GB+9A32AC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32AD30, GB+9A32AD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32AE30, GB+9A32AE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32AF30, GB+9A32AF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32B030, GB+9A32B039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32B130, GB+9A32B139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32B230, GB+9A32B239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32B330, GB+9A32B339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32B430, GB+9A32B439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32B530, GB+9A32B539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32B630, GB+9A32B639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32B730, GB+9A32B739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32B830, GB+9A32B839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32B930, GB+9A32B939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32BA30, GB+9A32BA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32BB30, GB+9A32BB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32BC30, GB+9A32BC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32BD30, GB+9A32BD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32BE30, GB+9A32BE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32BF30, GB+9A32BF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32C030, GB+9A32C039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32C130, GB+9A32C139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32C230, GB+9A32C239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32C330, GB+9A32C339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32C430, GB+9A32C439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32C530, GB+9A32C539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32C630, GB+9A32C639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32C730, GB+9A32C739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32C830, GB+9A32C839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32C930, GB+9A32C939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32CA30, GB+9A32CA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32CB30, GB+9A32CB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32CC30, GB+9A32CC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32CD30, GB+9A32CD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32CE30, GB+9A32CE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32CF30, GB+9A32CF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32D030, GB+9A32D039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32D130, GB+9A32D139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32D230, GB+9A32D239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32D330, GB+9A32D339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32D430, GB+9A32D439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32D530, GB+9A32D539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32D630, GB+9A32D639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32D730, GB+9A32D739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32D830, GB+9A32D839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32D930, GB+9A32D939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32DA30, GB+9A32DA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32DB30, GB+9A32DB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32DC30, GB+9A32DC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32DD30, GB+9A32DD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32DE30, GB+9A32DE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32DF30, GB+9A32DF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32E030, GB+9A32E039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32E130, GB+9A32E139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32E230, GB+9A32E239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32E330, GB+9A32E339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32E430, GB+9A32E439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32E530, GB+9A32E539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32E630, GB+9A32E639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32E730, GB+9A32E739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32E830, GB+9A32E839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32E930, GB+9A32E939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32EA30, GB+9A32EA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32EB30, GB+9A32EB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32EC30, GB+9A32EC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32ED30, GB+9A32ED39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32EE30, GB+9A32EE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32EF30, GB+9A32EF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32F030, GB+9A32F039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32F130, GB+9A32F139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32F230, GB+9A32F239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32F330, GB+9A32F339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32F430, GB+9A32F439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32F530, GB+9A32F539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32F630, GB+9A32F639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32F730, GB+9A32F739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32F830, GB+9A32F839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32F930, GB+9A32F939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32FA30, GB+9A32FA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32FB30, GB+9A32FB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32FC30, GB+9A32FC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32FD30, GB+9A32FD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A32FE30, GB+9A32FE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A338130, GB+9A338139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A338230, GB+9A338239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A338330, GB+9A338339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A338430, GB+9A338439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A338530, GB+9A338539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A338630, GB+9A338639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A338730, GB+9A338739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A338830, GB+9A338839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A338930, GB+9A338939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A338A30, GB+9A338A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A338B30, GB+9A338B39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A338C30, GB+9A338C39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A338D30, GB+9A338D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A338E30, GB+9A338E39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A338F30, GB+9A338F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A339030, GB+9A339039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A339130, GB+9A339139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A339230, GB+9A339239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A339330, GB+9A339339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A339430, GB+9A339439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A339530, GB+9A339539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A339630, GB+9A339639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A339730, GB+9A339739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A339830, GB+9A339839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A339930, GB+9A339939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A339A30, GB+9A339A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A339B30, GB+9A339B39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A339C30, GB+9A339C39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A339D30, GB+9A339D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A339E30, GB+9A339E39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A339F30, GB+9A339F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33A030, GB+9A33A039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33A130, GB+9A33A139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33A230, GB+9A33A239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33A330, GB+9A33A339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33A430, GB+9A33A439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33A530, GB+9A33A539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33A630, GB+9A33A639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33A730, GB+9A33A739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33A830, GB+9A33A839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33A930, GB+9A33A939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33AA30, GB+9A33AA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33AB30, GB+9A33AB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33AC30, GB+9A33AC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33AD30, GB+9A33AD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33AE30, GB+9A33AE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33AF30, GB+9A33AF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33B030, GB+9A33B039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33B130, GB+9A33B139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33B230, GB+9A33B239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33B330, GB+9A33B339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33B430, GB+9A33B439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33B530, GB+9A33B539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33B630, GB+9A33B639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33B730, GB+9A33B739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33B830, GB+9A33B839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33B930, GB+9A33B939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33BA30, GB+9A33BA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33BB30, GB+9A33BB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33BC30, GB+9A33BC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33BD30, GB+9A33BD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33BE30, GB+9A33BE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33BF30, GB+9A33BF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33C030, GB+9A33C039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33C130, GB+9A33C139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33C230, GB+9A33C239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33C330, GB+9A33C339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33C430, GB+9A33C439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33C530, GB+9A33C539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33C630, GB+9A33C639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33C730, GB+9A33C739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33C830, GB+9A33C839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33C930, GB+9A33C939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33CA30, GB+9A33CA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33CB30, GB+9A33CB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33CC30, GB+9A33CC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33CD30, GB+9A33CD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33CE30, GB+9A33CE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33CF30, GB+9A33CF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33D030, GB+9A33D039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33D130, GB+9A33D139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33D230, GB+9A33D239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33D330, GB+9A33D339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33D430, GB+9A33D439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33D530, GB+9A33D539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33D630, GB+9A33D639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33D730, GB+9A33D739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33D830, GB+9A33D839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33D930, GB+9A33D939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33DA30, GB+9A33DA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33DB30, GB+9A33DB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33DC30, GB+9A33DC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33DD30, GB+9A33DD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33DE30, GB+9A33DE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33DF30, GB+9A33DF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33E030, GB+9A33E039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33E130, GB+9A33E139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33E230, GB+9A33E239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33E330, GB+9A33E339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33E430, GB+9A33E439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33E530, GB+9A33E539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33E630, GB+9A33E639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33E730, GB+9A33E739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33E830, GB+9A33E839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33E930, GB+9A33E939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33EA30, GB+9A33EA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33EB30, GB+9A33EB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33EC30, GB+9A33EC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33ED30, GB+9A33ED39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33EE30, GB+9A33EE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33EF30, GB+9A33EF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33F030, GB+9A33F039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33F130, GB+9A33F139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33F230, GB+9A33F239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33F330, GB+9A33F339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33F430, GB+9A33F439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33F530, GB+9A33F539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33F630, GB+9A33F639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33F730, GB+9A33F739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33F830, GB+9A33F839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33F930, GB+9A33F939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33FA30, GB+9A33FA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33FB30, GB+9A33FB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33FC30, GB+9A33FC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33FD30, GB+9A33FD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A33FE30, GB+9A33FE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A348130, GB+9A348139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A348230, GB+9A348239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A348330, GB+9A348339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A348430, GB+9A348439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A348530, GB+9A348539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A348630, GB+9A348639]*/
  0,     0,     0,     0,     0,     25245, 0,     0,     0,     0,       /*[GB+9A348730, GB+9A348739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A348830, GB+9A348839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A348930, GB+9A348939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A348A30, GB+9A348A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A348B30, GB+9A348B39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A348C30, GB+9A348C39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A348D30, GB+9A348D39]*/
  0,     0,     0,     0,     0,     29702, 0,     0,     0,     0,       /*[GB+9A348E30, GB+9A348E39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A348F30, GB+9A348F39]*/
  0,     0,     5650,  0,     0,     0,     0,     0,     8706,  0,       /*[GB+9A349030, GB+9A349039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A349130, GB+9A349139]*/
  0,     34493, 0,     0,     34494, 39728, 0,     0,     0,     0,       /*[GB+9A349230, GB+9A349239]*/
  0,     0,     0,     6040,  0,     0,     0,     0,     31948, 0,       /*[GB+9A349330, GB+9A349339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     20228, 0,       /*[GB+9A349430, GB+9A349439]*/
  0,     0,     0,     0,     0,     29641, 0,     0,     0,     0,       /*[GB+9A349530, GB+9A349539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A349630, GB+9A349639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A349730, GB+9A349739]*/
  40617, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A349830, GB+9A349839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A349930, GB+9A349939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A349A30, GB+9A349A39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A349B30, GB+9A349B39]*/
  36160, 0,     0,     0,     0,     0,     0,     0,     0,     4447,    /*[GB+9A349C30, GB+9A349C39]*/
  0,     0,     0,     1237,  0,     0,     0,     6763,  0,     28531,   /*[GB+9A349D30, GB+9A349D39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A349E30, GB+9A349E39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A349F30, GB+9A349F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     31850, 0,       /*[GB+9A34A030, GB+9A34A039]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34A130, GB+9A34A139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34A230, GB+9A34A239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34A330, GB+9A34A339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34A430, GB+9A34A439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34A530, GB+9A34A539]*/
  0,     0,     20261, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34A630, GB+9A34A639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     6764,    /*[GB+9A34A730, GB+9A34A739]*/
  5428,  0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34A830, GB+9A34A839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34A930, GB+9A34A939]*/
  0,     34097, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34AA30, GB+9A34AA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34AB30, GB+9A34AB39]*/
  0,     0,     0,     0,     0,     0,     0,     32559, 0,     0,       /*[GB+9A34AC30, GB+9A34AC39]*/
  0,     0,     17440, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34AD30, GB+9A34AD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34AE30, GB+9A34AE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34AF30, GB+9A34AF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34B030, GB+9A34B039]*/
  8906,  0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34B130, GB+9A34B139]*/
  0,     0,     0,     0,     6580,  0,     0,     0,     0,     0,       /*[GB+9A34B230, GB+9A34B239]*/
  0,     13625, 11358, 0,     12940, 0,     0,     0,     0,     0,       /*[GB+9A34B330, GB+9A34B339]*/
  0,     0,     0,     11045, 0,     33924, 0,     0,     0,     0,       /*[GB+9A34B430, GB+9A34B439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34B530, GB+9A34B539]*/
  30639, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34B630, GB+9A34B639]*/
  28011, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34B730, GB+9A34B739]*/
  17536, 3085,  0,     0,     6156,  0,     0,     3431,  0,     0,       /*[GB+9A34B830, GB+9A34B839]*/
  0,     0,     0,     17580, 0,     0,     0,     0,     0,     0,       /*[GB+9A34B930, GB+9A34B939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34BA30, GB+9A34BA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34BB30, GB+9A34BB39]*/
  0,     40526, 0,     0,     0,     0,     0,     0,     21988, 0,       /*[GB+9A34BC30, GB+9A34BC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     36712,   /*[GB+9A34BD30, GB+9A34BD39]*/
  0,     0,     40532, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34BE30, GB+9A34BE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34BF30, GB+9A34BF39]*/
  0,     0,     32715, 9058,  34483, 0,     0,     0,     0,     0,       /*[GB+9A34C030, GB+9A34C039]*/
  0,     0,     0,     38077, 0,     13564, 0,     0,     0,     26756,   /*[GB+9A34C130, GB+9A34C139]*/
  37541, 0,     0,     0,     0,     0,     0,     13878, 0,     0,       /*[GB+9A34C230, GB+9A34C239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     33942,   /*[GB+9A34C330, GB+9A34C339]*/
  0,     0,     0,     0,     42353, 0,     0,     41299, 0,     0,       /*[GB+9A34C430, GB+9A34C439]*/
  25618, 0,     0,     0,     0,     0,     0,     0,     0,     34535,   /*[GB+9A34C530, GB+9A34C539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     17012,   /*[GB+9A34C630, GB+9A34C639]*/
  0,     0,     0,     20764, 0,     36685, 25502, 0,     0,     0,       /*[GB+9A34C730, GB+9A34C739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34C830, GB+9A34C839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34C930, GB+9A34C939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34CA30, GB+9A34CA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34CB30, GB+9A34CB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     19851, 0,       /*[GB+9A34CC30, GB+9A34CC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34CD30, GB+9A34CD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34CE30, GB+9A34CE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34CF30, GB+9A34CF39]*/
  0,     0,     5648,  43082, 0,     0,     0,     36565, 1257,  0,       /*[GB+9A34D030, GB+9A34D039]*/
  38741, 0,     41872, 0,     0,     0,     25487, 40793, 0,     40438,   /*[GB+9A34D130, GB+9A34D139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34D230, GB+9A34D239]*/
  0,     0,     0,     0,     0,     17550, 0,     0,     0,     0,       /*[GB+9A34D330, GB+9A34D339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     36417, 0,       /*[GB+9A34D430, GB+9A34D439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34D530, GB+9A34D539]*/
  0,     0,     0,     0,     0,     32716, 0,     0,     0,     0,       /*[GB+9A34D630, GB+9A34D639]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34D730, GB+9A34D739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34D830, GB+9A34D839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34D930, GB+9A34D939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34DA30, GB+9A34DA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34DB30, GB+9A34DB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34DC30, GB+9A34DC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34DD30, GB+9A34DD39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34DE30, GB+9A34DE39]*/
  29703, 0,     19620, 6536,  19892, 37534, 0,     0,     37262, 0,       /*[GB+9A34DF30, GB+9A34DF39]*/
  40869, 0,     0,     0,     0,     16802, 0,     0,     22550, 0,       /*[GB+9A34E030, GB+9A34E039]*/
  36653, 0,     32638, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34E130, GB+9A34E139]*/
  0,     40239, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34E230, GB+9A34E239]*/
  0,     0,     0,     0,     0,     36848, 0,     0,     0,     0,       /*[GB+9A34E330, GB+9A34E339]*/
  0,     0,     0,     4998,  0,     0,     0,     0,     0,     0,       /*[GB+9A34E430, GB+9A34E439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34E530, GB+9A34E539]*/
  0,     33417, 0,     27654, 0,     0,     0,     0,     0,     0,       /*[GB+9A34E630, GB+9A34E639]*/
  19787, 0,     0,     17461, 0,     0,     0,     0,     0,     0,       /*[GB+9A34E730, GB+9A34E739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34E830, GB+9A34E839]*/
  0,     0,     20661, 0,     0,     0,     16803, 0,     0,     0,       /*[GB+9A34E930, GB+9A34E939]*/
  0,     0,     5379,  0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34EA30, GB+9A34EA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34EB30, GB+9A34EB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34EC30, GB+9A34EC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34ED30, GB+9A34ED39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34EE30, GB+9A34EE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34EF30, GB+9A34EF39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     16905, 0,       /*[GB+9A34F030, GB+9A34F039]*/
  0,     31853, 0,     0,     20399, 0,     0,     0,     0,     0,       /*[GB+9A34F130, GB+9A34F139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34F230, GB+9A34F239]*/
  42096, 36568, 0,     0,     0,     19467, 0,     0,     0,     0,       /*[GB+9A34F330, GB+9A34F339]*/
  0,     9593,  0,     0,     0,     0,     0,     34311, 0,     2397,    /*[GB+9A34F430, GB+9A34F439]*/
  0,     0,     28604, 20521, 42320, 0,     0,     0,     0,     0,       /*[GB+9A34F530, GB+9A34F539]*/
  0,     0,     0,     6065,  0,     0,     37567, 0,     0,     0,       /*[GB+9A34F630, GB+9A34F639]*/
  0,     0,     0,     30792, 0,     0,     0,     0,     0,     0,       /*[GB+9A34F730, GB+9A34F739]*/
  20766, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34F830, GB+9A34F839]*/
  0,     0,     3874,  0,     0,     0,     21734, 0,     0,     0,       /*[GB+9A34F930, GB+9A34F939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34FA30, GB+9A34FA39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34FB30, GB+9A34FB39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34FC30, GB+9A34FC39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A34FD30, GB+9A34FD39]*/
  0,     0,     3508,  0,     33347, 0,     0,     0,     0,     0,       /*[GB+9A34FE30, GB+9A34FE39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     28057, 0,       /*[GB+9A358130, GB+9A358139]*/
  0,     0,     0,     37547, 0,     0,     42277, 0,     0,     40510,   /*[GB+9A358230, GB+9A358239]*/
  0,     0,     0,     0,     0,     0,     0,     30165, 0,     0,       /*[GB+9A358330, GB+9A358339]*/
  0,     0,     18325, 0,     0,     0,     0,     16539, 0,     0,       /*[GB+9A358430, GB+9A358439]*/
  0,     0,     0,     27370, 0,     0,     0,     10153, 3011,  0,       /*[GB+9A358530, GB+9A358539]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     6054,    /*[GB+9A358630, GB+9A358639]*/
  0,     0,     0,     10286, 0,     0,     0,     0,     0,     0,       /*[GB+9A358730, GB+9A358739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A358830, GB+9A358839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A358930, GB+9A358939]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A358A30, GB+9A358A39]*/
  0,     0,     25325, 0,     0,     0,     39571, 0,     0,     0,       /*[GB+9A358B30, GB+9A358B39]*/
  0,     0,     0,     0,     0,     0,     0,     36300, 0,     0,       /*[GB+9A358C30, GB+9A358C39]*/
  0,     26381, 0,     0,     14344, 0,     0,     0,     0,     0,       /*[GB+9A358D30, GB+9A358D39]*/
  0,     0,     35060, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9A358E30, GB+9A358E39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A358F30, GB+9A358F39]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A359030, GB+9A359039]*/
  0,     16811, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A359130, GB+9A359139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A359230, GB+9A359239]*/
  0,     0,     0,     0,     0,     0,     0,     0,     36786, 0,       /*[GB+9A359330, GB+9A359339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A359430, GB+9A359439]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A359530, GB+9A359539]*/
  0,     0,     22506, 0,     0,     0,     0,     0,     0,     0,       /*[GB+9A359630, GB+9A359639]*/
  0,     18099, 0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A359730, GB+9A359739]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A359830, GB+9A359839]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A359930, GB+9A359939]*/
  12537, 0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A359A30, GB+9A359A39]*/
  0,     0,     0,     12337, 0,     0,     20403, 0,     0,     4007,    /*[GB+9A359B30, GB+9A359B39]*/
  35130, 41122, 36779, 0,     0,     0,     0,     2936,  12391, 0,       /*[GB+9A359C30, GB+9A359C39]*/
  0,     0,     14849, 0,     28778, 0,     0,     0,     25285, 0,       /*[GB+9A359D30, GB+9A359D39]*/
  0,     0,     0,     0,     0,     0,     0,     28503, 0,     0,       /*[GB+9A359E30, GB+9A359E39]*/
  0,     29889, 8132,  0,     0,     0,     0,     0,     5385,  0,       /*[GB+9A359F30, GB+9A359F39]*/
  0,     0,     0,     0,     0,     0,     15469, 0,     0,     0,       /*[GB+9A35A030, GB+9A35A039]*/
  0,     0,     0,     0,     0,     0,     0,     26083, 0,     0,       /*[GB+9A35A130, GB+9A35A139]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     0,       /*[GB+9A35A230, GB+9A35A239]*/
  36694, 0,     0,     21623, 0,     0,     0,     0,     0,     34842,   /*[GB+9A35A330, GB+9A35A339]*/
  0,     0,     0,     0,     0,     0,     0,     0,     0,     17408,   /*[GB+9A35A430, GB+9A35A439]*/
  0,     0,     29841, 0,     29818, 0,     0,     0,     0,     0,       /*[GB+9A35A530, GB+9A35A539]*/
  0,     0,     0,     0,     0,  