/* Copyright 2006 aQute SARL 
 * Licensed under the Apache License, Version 2.0, see http://www.apache.org/licenses/LICENSE-2.0 */
package aQute.lib.osgi;

import java.io.*;
import java.util.*;
import java.util.jar.*;
import java.util.regex.*;
import java.util.zip.*;

import aQute.lib.reporter.*;

public class Jar {
	public static final Object[]	EMPTY_ARRAY	= new Jar[0];
	Map				resources	= new TreeMap();
	Map				directories	= new TreeMap();
	Manifest		manifest;
	boolean			manifestFirst;
	String			name;
	File			source;
	ZipFile			zipFile;
	long			lastModified;
	Reporter		reporter;

	public Jar(String name) {
		this.name = name;
	}

	public Jar(String name, File dirOrFile) throws ZipException, IOException {
		this(name);
		source = dirOrFile;
		if (dirOrFile.isDirectory())
			FileResource.build(this, dirOrFile, Analyzer.doNotCopy);
		else {
			zipFile = ZipResource.build(this, dirOrFile);
		}
	}

	public Jar(String name, InputStream in, long lastModified)
			throws IOException {
		this(name);
		EmbeddedResource.build(this, in, lastModified);
	}

	public Jar(String name, String path) throws IOException {
		this(name);
		File f = new File(path);
		InputStream in = new FileInputStream(f);
		EmbeddedResource.build(this, in, f.lastModified());
		in.close();
	}

	public Jar(File jar) throws IOException {
		this(jar.getName(), jar);
	}

	public Jar(String string, InputStream resourceAsStream) throws IOException {
		this(string, resourceAsStream, 0);
	}

	public void setName(String name) {
		this.name = name;
	}

	public String toString() {
		return "Jar:" + name;
	}

	public boolean putResource(String path, Resource resource) {
		return putResource(path, resource, true);
	}

	public boolean putResource(String path, Resource resource, boolean overwrite) {
		if (resource.lastModified() > lastModified)
			lastModified = resource.lastModified();

		if (path.equals("META-INF/MANIFEST.MF")) {
			manifest = null;
			if (resources.isEmpty())
				manifestFirst = true;
		}
		resources.put(path, resource);
		String dir = getDirectory(path);
		Map s = (Map) directories.get(dir);
		if (s == null) {
				s = new TreeMap();
				directories.put(dir, s);
				int n = dir.lastIndexOf('/');
				while ( n > 0 ) {
					String dd = dir.substring(0,n);
					if ( directories.containsKey(dd))
						break;
					directories.put(dd,null);
					n = dd.lastIndexOf('/');
				}
		}
		boolean duplicate = s.containsKey(path);
		if (!duplicate || overwrite)
			s.put(path, resource);
		return duplicate;
	}

	public Resource getResource(String path) {
		return (Resource) resources.get(path);
	}

	private String getDirectory(String path) {
		int n = path.lastIndexOf('/');
		if (n < 0)
			return "";

		return path.substring(0, n);
	}

	public Map getDirectories() {
		return directories;
	}

	public Map getResources() {
		return resources;
	}

	public boolean addDirectory(Map directory, boolean overwrite) {
		boolean duplicates = false;
		if ( directory == null)
			return false;
		
		for (Iterator c = directory.entrySet().iterator(); c.hasNext();) {
			Map.Entry entry = (Map.Entry) c.next();
			String key = (String) entry.getKey();
			if (!key.endsWith(".java")) {
				duplicates |= putResource(key, (Resource) entry.getValue(),
						overwrite);
			}
		}
		return duplicates;
	}

	public Manifest getManifest() throws IOException {
		if (manifest == null) {
			Resource manifestResource = getResource("META-INF/MANIFEST.MF");
			if (manifestResource != null) {
				InputStream in = manifestResource.openInputStream();
				manifest = new Manifest(in);
				in.close();
			}
		}
		return manifest;
	}

	public boolean exists(String path) {
		return resources.containsKey(path);
	}

	public void setManifest(Manifest manifest) {
		manifestFirst = true;
		this.manifest = manifest;
	}

	public void write(File file) throws Exception {
		try {
			OutputStream out = new FileOutputStream(file);
			write(out);
			out.close();
		} catch (Throwable t) {
			try {
				file.delete();
			} catch (Exception ee) {
				ee.printStackTrace();
			}
			t.printStackTrace();
		}
	}

	public void write(String file) throws Exception {
		write(new File(file));
	}

	public void write(OutputStream out) throws IOException {
		JarOutputStream jout = new JarOutputStream(out);
		Set done = new HashSet();

		Set directories = new HashSet();
		doManifest(done, jout);
		for (Iterator i = getResources().entrySet().iterator(); i.hasNext();) {
			Map.Entry entry = (Map.Entry) i.next();
			// Skip metainf contents
			if (!done.contains(entry.getKey()))
				writeResource(jout, directories, (String) entry.getKey(),
						(Resource) entry.getValue());
		}
		jout.finish();
	}

	private void doManifest(Set done, JarOutputStream jout) throws IOException {
		JarEntry ze = new JarEntry("META-INF/MANIFEST.MF");
		jout.putNextEntry(ze);
		writeManifest(jout);
		jout.closeEntry();
		done.add(ze.getName());
	}
	
	public void writeManifest(OutputStream out) throws IOException {
		Manifest manifest = clean(getManifest());
		manifest.write(out);		
	}

	private static Manifest clean(Manifest org) {
		Manifest result = new Manifest();
		for ( Iterator i=org.getMainAttributes().entrySet().iterator(); i.hasNext();) {
			Map.Entry entry = (Map.Entry) i.next();
			String nice  = clean((String) entry.getValue());
			result.getMainAttributes().put((Attributes.Name)entry.getKey(), nice );
		}
		for ( Iterator n = org.getEntries().keySet().iterator(); n.hasNext(); ) {
			String name = (String) n.next();
			Attributes attrs = result.getAttributes(name);
			if ( attrs == null ) {
				attrs = new Attributes();
				result.getEntries().put(name, attrs);
			}
			
			for ( Iterator i=org.getAttributes(name).entrySet().iterator(); i.hasNext();) {
				Map.Entry entry = (Map.Entry) i.next();
				String nice  = clean((String) entry.getValue());
				attrs.put((Attributes.Name)entry.getKey(), nice );
			}
		}
		return result;
	}
	
	private static String clean(String s) {
		if ( s.indexOf('\n')<0)
			return s;
		
		StringBuffer sb = new StringBuffer(s);
		for ( int i =0; i<sb.length(); i++ ) {
			if ( sb.charAt(i)=='\n')
				sb.insert(++i, ' ');
		}
		return sb.toString();
	}
	

	private void writeResource(JarOutputStream jout, Set directories,
			String path, Resource resource) throws IOException {
		if (resource == null)
			return;

		createDirectories(directories, jout, path);
		ZipEntry ze = new ZipEntry(path);
		ze.setMethod(ZipEntry.DEFLATED);
		long lastModified = resource.lastModified();
		if (lastModified == 0L)
			lastModified = System.currentTimeMillis();
		ze.setTime(lastModified);
		if (resource.getExtra() != null)
			ze.setExtra(resource.getExtra().getBytes());
		jout.putNextEntry(ze);
		try {
			resource.write(jout);
		} catch (Exception e) {
			throw new IllegalArgumentException("Cannot write resource: " +
					path + " " + e);
		}
		jout.closeEntry();
	}

	void createDirectories(Set directories, JarOutputStream zip, String name)
			throws IOException {
		int index = name.lastIndexOf('/');
		if (index > 0) {
			String path = name.substring(0, index);
			if (directories.contains(path))
				return;
			createDirectories(directories, zip, path);
			ZipEntry ze = new ZipEntry(path + '/');
			zip.putNextEntry(ze);
			zip.closeEntry();
			directories.add(path);
		}
	}

	public String getName() {
		return name;
	}

	/**
	 * Add all the resources in the given jar that match the given filter.
	 * 
	 * @param sub
	 *            the jar
	 * @param filter
	 *            a pattern that should match the resoures in sub to be added
	 */
	public boolean addAll(Jar sub, Pattern filter) {
		boolean dupl = false;
		for (Iterator r = sub.getResources().keySet().iterator(); r.hasNext();) {
			String name = (String) r.next();
			if (filter == null || filter.matcher(name).matches())
				dupl |= putResource(name, sub.getResource(name), true);
		}
		return dupl;
	}

	public void close() {
		if (zipFile != null)
			try {
				zipFile.close();
			} catch (IOException e) {
				// Ignore
			}
		resources = null;
		directories = null;
		manifest = null;
		source = null;
	}

	public long lastModified() {
		return lastModified;
	}

	public void updateModified(long time) {
		if (time > lastModified)
			lastModified = time;
	}

	public void setReporter(Reporter reporter) {
		this.reporter = reporter;
	}

	public boolean hasDirectory(String path) {
		return directories.containsKey(path);
	}

	public List getPackages() {
		List list = new ArrayList( directories.size());
		
		for( Iterator i=directories.keySet().iterator(); i.hasNext(); ) {
			String path = (String) i.next();
			String pack = path.replace('/', '.');
			list.add(pack);
		}
		return list;
	}
	
	public File getSource() {
		return source;
	}

	public boolean addAll(Jar src) {
		return addAll(src,null);
	}

	public boolean rename(String oldPath, String newPath) {
		Resource resource = remove(oldPath);
		if ( resource == null)
			return false;
		
		return putResource(newPath, resource);		
	}
	
	public Resource remove(String path) {
		Resource resource = (Resource) resources.remove(path);
		String dir = getDirectory(path);
		Map mdir = (Map) directories.get(dir);
		// must be != null
		mdir.remove(path);
		return resource;
	}
}
