// REQUIRES: target-is-powerpc64le
// RUN: %clang_builtins %s %librt -o %t && %run %t
#include <stdio.h>
#include "DD.h"

long double __gcc_qmul(long double x, long double y);
int memcmp(const void *, const void *, __typeof__(sizeof(0)));

double testAccuracy();
int testEdgeCases();

int main(int argc, char *argv[]) {
	if (testEdgeCases())
		return 1;

	if (testAccuracy() > 2.0)
		return 1;

	return 0;
}

struct testVector {
	double xhi; double xlo;
	double yhi; double ylo;
	double rhi; double rlo;
};

#define INFINITY __builtin_inf()
#define HUGE 0x1.fffffffffffffp1023
#define QNAN __builtin_nan("")

const struct testVector edgeCases[] = {
{	0.0,		0.0,		0.0,		0.0,		0.0,		0.0			},
{  -0.0,		0.0,		0.0,		0.0,	   -0.0,		0.0			},
{   0.0,		0.0,	   -0.0,		0.0,	   -0.0,		0.0			},
{  -0.0,		0.0,	   -0.0,		0.0,	    0.0,		0.0			},

{   INFINITY,	0.0,		0.0,		0.0,		QNAN,		0.0			},
{  -INFINITY,	0.0,		0.0,		0.0,	    QNAN,		0.0			},
{   INFINITY,	0.0,	   -0.0,		0.0,		QNAN,		0.0			},
{  -INFINITY,	0.0,	   -0.0,		0.0,	    QNAN,		0.0			},
{   INFINITY,	0.0,	    1.0,		0.0,		INFINITY,	0.0			},
{  -INFINITY,	0.0,	    1.0,		0.0,	   -INFINITY,	0.0			},
{   INFINITY,	0.0,	   -1.0,		0.0,	   -INFINITY,	0.0			},
{  -INFINITY,	0.0,	   -1.0,		0.0,	    INFINITY,	0.0			},
{   INFINITY,	0.0,	    HUGE,		0.0,		INFINITY,	0.0			},
{  -INFINITY,	0.0,	    HUGE,		0.0,	   -INFINITY,	0.0			},
{   INFINITY,	0.0,	   -HUGE,		0.0,	   -INFINITY,	0.0			},
{  -INFINITY,	0.0,	   -HUGE,		0.0,	    INFINITY,	0.0			},
{   INFINITY,	0.0,	    INFINITY,	0.0,		INFINITY,	0.0			},
{  -INFINITY,	0.0,	    INFINITY,	0.0,	   -INFINITY,	0.0			},
{   INFINITY,	0.0,	   -INFINITY,	0.0,	   -INFINITY,	0.0			},
{  -INFINITY,	0.0,	   -INFINITY,	0.0,	    INFINITY,	0.0			},
{   0.0,		0.0,		INFINITY,	0.0,		QNAN,		0.0			},
{   0.0,		0.0,	   -INFINITY,	0.0,	    QNAN,		0.0			},
{  -0.0,		0.0,		INFINITY,	0.0,		QNAN,		0.0			},
{  -0.0,		0.0,	   -INFINITY,	0.0,	    QNAN,		0.0			},
{   1.0,		0.0,		INFINITY,	0.0,		INFINITY,	0.0			},
{   1.0,		0.0,	   -INFINITY,	0.0,	   -INFINITY,	0.0			},
{  -1.0,		0.0,		INFINITY,	0.0,	   -INFINITY,	0.0			},
{  -1.0,		0.0,	   -INFINITY,	0.0,	    INFINITY,	0.0			},
{   HUGE,		0.0,		INFINITY,	0.0,		INFINITY,	0.0			},
{   HUGE,		0.0,	   -INFINITY,	0.0,	   -INFINITY,	0.0			},
{  -HUGE,		0.0,		INFINITY,	0.0,	   -INFINITY,	0.0			},
{  -HUGE,		0.0,	   -INFINITY,	0.0,	    INFINITY,	0.0			},

{   QNAN,		0.0,		0.0,		0.0,		QNAN,		0.0			},
{  -QNAN,		0.0,	    0.0,		0.0,	    QNAN,		0.0			},
{   QNAN,		0.0,	   -0.0,		0.0,		QNAN,		0.0			},
{  -QNAN,		0.0,	   -0.0,		0.0,	    QNAN,		0.0			},
{   QNAN,		0.0,		1.0,		0.0,		QNAN,		0.0			},
{  -QNAN,		0.0,	    1.0,		0.0,	    QNAN,		0.0			},
{   QNAN,		0.0,	   -1.0,		0.0,		QNAN,		0.0			},
{  -QNAN,		0.0,	   -1.0,		0.0,	    QNAN,		0.0			},
{   QNAN,		0.0,		HUGE,		0.0,		QNAN,		0.0			},
{  -QNAN,		0.0,	    HUGE,		0.0,	    QNAN,		0.0			},
{   QNAN,		0.0,	   -HUGE,		0.0,		QNAN,		0.0			},
{  -QNAN,		0.0,	   -HUGE,		0.0,	    QNAN,		0.0			},
{   QNAN,		0.0,		INFINITY,	0.0,		QNAN,		0.0			},
{  -QNAN,		0.0,	    INFINITY,	0.0,	    QNAN,		0.0			},
{   QNAN,		0.0,	   -INFINITY,	0.0,		QNAN,		0.0			},
{  -QNAN,		0.0,	   -INFINITY,	0.0,	    QNAN,		0.0			},
{   QNAN,		0.0,		QNAN,		0.0,		QNAN,		0.0			},
{  -QNAN,		0.0,	    QNAN,		0.0,	    QNAN,		0.0			},
{   QNAN,		0.0,	   -QNAN,		0.0,		QNAN,		0.0			},
{  -QNAN,		0.0,	   -QNAN,		0.0,	    QNAN,		0.0			},
{	0.0,		0.0,		QNAN,		0.0,		QNAN,		0.0			},
{	0.0,		0.0,	   -QNAN,		0.0,		QNAN,		0.0			},
{  -0.0,		0.0,		QNAN,		0.0,		QNAN,		0.0			},
{  -0.0,		0.0,	   -QNAN,		0.0,		QNAN,		0.0			},
{	1.0,		0.0,		QNAN,		0.0,		QNAN,		0.0			},
{	1.0,		0.0,	   -QNAN,		0.0,		QNAN,		0.0			},
{  -1.0,		0.0,		QNAN,		0.0,		QNAN,		0.0			},
{  -1.0,		0.0,	   -QNAN,		0.0,		QNAN,		0.0			},
{	HUGE,		0.0,		QNAN,		0.0,		QNAN,		0.0			},
{	HUGE,		0.0,	   -QNAN,		0.0,		QNAN,		0.0			},
{  -HUGE,		0.0,		QNAN,		0.0,		QNAN,		0.0			},
{  -HUGE,		0.0,	   -QNAN,		0.0,		QNAN,		0.0			},
{	INFINITY,	0.0,		QNAN,		0.0,		QNAN,		0.0			},
{	INFINITY,	0.0,	   -QNAN,		0.0,		QNAN,		0.0			},
{  -INFINITY,	0.0,		QNAN,		0.0,		QNAN,		0.0			},
{  -INFINITY,	0.0,	   -QNAN,		0.0,		QNAN,		0.0			},
};

const int numEdgeCases = sizeof(edgeCases) / sizeof(struct testVector);


int testEdgeCases() {
	int i;
	DD a, b, c, r;
	for (i=0; i<numEdgeCases; ++i) {
		a.hi = edgeCases[i].xhi;
		a.lo = edgeCases[i].xlo;
		b.hi = edgeCases[i].yhi;
		b.lo = edgeCases[i].ylo;
		r.hi = edgeCases[i].rhi;
		r.lo = edgeCases[i].rlo;

		int error;

		DD c = { .ld = __gcc_qmul(a.ld, b.ld) };

		if (r.hi != r.hi) {
			if (c.hi == c.hi)
				error = 1;
			else if (c.lo != 0.0)
				error = 1;
			else
				error = 0;
		}

		else if (r.hi == 0.0)
			error = memcmp(&c, &r, sizeof(DD));

		else
			error = ((c.hi != r.hi) || (c.lo != r.lo));

		if (error) {
			printf("Error on edge case %a x %a: expected (%a, %a), got (%a, %a).\n", a.hi, b.hi, r.hi, r.lo, c.hi, c.lo);
			return 1;
		}
	}

	return 0;
}


/*

 Code for generating the test cases, requires the mpfr package to run.

 #include <stdio.h>
 #include <stdlib.h>
 #include <mpfr.h>
 #include <math.h>

 #ifdef __x86_64__
 #define randlength 2
 #else
 #define randlength 4
 #endif

 void printTest(mpfr_t a, mpfr_t b, mpfr_t c) {
 static const double INFINITYD = __builtin_INFINITY();

 MPFR_DECL_INIT(tmp, 53);

 double ahi = mpfr_get_d(a, GMP_RNDN);
 mpfr_set_d(tmp, ahi, GMP_RNDN);
 mpfr_sub(tmp, a, tmp, GMP_RNDN);
 double alo = mpfr_get_d(tmp, GMP_RNDN);
 printf("{%0.13a, %0.13a, ", ahi, alo);

 double bhi = mpfr_get_d(b, GMP_RNDN);
 mpfr_set_d(tmp, bhi, GMP_RNDN);
 mpfr_sub(tmp, b, tmp, GMP_RNDN);
 double blo = mpfr_get_d(tmp, GMP_RNDN);
 printf("%0.13a, %0.13a, ", bhi, blo);

 double chi = mpfr_get_d(c, GMP_RNDN);
 mpfr_set_d(tmp, chi, GMP_RNDN);
 mpfr_sub(tmp, c, tmp, GMP_RNDN);
 double clo = isINFINITY(chi) ? 0.0 : mpfr_get_d(tmp, GMP_RNDN);
 printf("%0.13a, %0.13a},\n", chi, clo);
 }

 int main(int argc, char *argv[]) {

 MPFR_DECL_INIT(a, 106);
 MPFR_DECL_INIT(b, 106);
 MPFR_DECL_INIT(c, 106);

 int exponent_range = atoi(argv[1]);

 int i;
 for (i=0; i<128; ++i) {
 mpfr_random2(a, randlength, exponent_range);
 mpfr_random2(b, randlength, exponent_range);
 mpfr_mul(c, a, b, GMP_RNDN);
 printTest(a, b, c);

 mpfr_neg(b, b, GMP_RNDN);
 mpfr_mul(c, a, b, GMP_RNDN);
 printTest(a, b, c);

 mpfr_neg(a, a, GMP_RNDN);
 mpfr_neg(b, b, GMP_RNDN);
 mpfr_mul(c, a, b, GMP_RNDN);
 printTest(a, b, c);

 mpfr_neg(b, b, GMP_RNDN);
 mpfr_mul(c, a, b, GMP_RNDN);
 printTest(a, b, c);
 }
 return 0;
 }

 */

const struct testVector accuracyTests[] = {
{0x1.00001ffffff00p-784, 0x1.fffff83ff8000p-839, 0x1.00ffc00ffff00p-771, 0x1.fe0007fffff00p-827, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.00001ffffff00p-784, 0x1.fffff83ff8000p-839, -0x1.00ffc00ffff00p-771, -0x1.fe0007fffff00p-827, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.00001ffffff00p-784, -0x1.fffff83ff8000p-839, 0x1.00ffc00ffff00p-771, 0x1.fe0007fffff00p-827, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.00001ffffff00p-784, -0x1.fffff83ff8000p-839, -0x1.00ffc00ffff00p-771, -0x1.fe0007fffff00p-827, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.00001fffff800p-705, 0x1.fffffffe00000p-779, 0x1.3fffffff80000p+666, 0x1.ffffffff00000p+593, 0x1.400027ff7f5ffp-39, 0x1.e0001ffe80000p-111},
{0x1.00001fffff800p-705, 0x1.fffffffe00000p-779, -0x1.3fffffff80000p+666, -0x1.ffffffff00000p+593, -0x1.400027ff7f5ffp-39, -0x1.e0001ffe80000p-111},
{-0x1.00001fffff800p-705, -0x1.fffffffe00000p-779, 0x1.3fffffff80000p+666, 0x1.ffffffff00000p+593, -0x1.400027ff7f5ffp-39, -0x1.e0001ffe80000p-111},
{-0x1.00001fffff800p-705, -0x1.fffffffe00000p-779, -0x1.3fffffff80000p+666, -0x1.ffffffff00000p+593, 0x1.400027ff7f5ffp-39, 0x1.e0001ffe80000p-111},
{0x1.0000000800000p-800, -0x1.0000000000000p-872, 0x1.000003fffc002p-916, -0x1.0000000000000p-1021, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.0000000800000p-800, -0x1.0000000000000p-872, -0x1.000003fffc002p-916, 0x1.0000000000000p-1021, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.0000000800000p-800, 0x1.0000000000000p-872, 0x1.000003fffc002p-916, -0x1.0000000000000p-1021, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.0000000800000p-800, 0x1.0000000000000p-872, -0x1.000003fffc002p-916, 0x1.0000000000000p-1021, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.000003ffffffep-19, -0x1.ffe0080000000p-103, 0x1.0800000000000p-211, -0x1.0000000000000p-307, 0x1.0800041fffffep-230, -0x1.00000020fef08p-286},
{0x1.000003ffffffep-19, -0x1.ffe0080000000p-103, -0x1.0800000000000p-211, 0x1.0000000000000p-307, -0x1.0800041fffffep-230, 0x1.00000020fef08p-286},
{-0x1.000003ffffffep-19, 0x1.ffe0080000000p-103, 0x1.0800000000000p-211, -0x1.0000000000000p-307, -0x1.0800041fffffep-230, 0x1.00000020fef08p-286},
{-0x1.000003ffffffep-19, 0x1.ffe0080000000p-103, -0x1.0800000000000p-211, 0x1.0000000000000p-307, 0x1.0800041fffffep-230, -0x1.00000020fef08p-286},
{0x1.0000007fffc00p-464, 0x1.efe0000000000p-558, 0x1.007fffffe0000p+756, 0x0.0000000000000p+0, 0x1.008000801fbfep+292, -0x1.fffeffff07900p+231},
{0x1.0000007fffc00p-464, 0x1.efe0000000000p-558, -0x1.007fffffe0000p+756, 0x0.0000000000000p+0, -0x1.008000801fbfep+292, 0x1.fffeffff07900p+231},
{-0x1.0000007fffc00p-464, -0x1.efe0000000000p-558, 0x1.007fffffe0000p+756, 0x0.0000000000000p+0, -0x1.008000801fbfep+292, 0x1.fffeffff07900p+231},
{-0x1.0000007fffc00p-464, -0x1.efe0000000000p-558, -0x1.007fffffe0000p+756, 0x0.0000000000000p+0, 0x1.008000801fbfep+292, -0x1.fffeffff07900p+231},
{0x1.3fe0000100000p-20, -0x1.0000000000000p-93, 0x1.3fffffffffe00p-397, 0x1.fe00040000000p-472, 0x1.8fd800013fd80p-417, 0x1.ffffc7d404500p-472},
{0x1.3fe0000100000p-20, -0x1.0000000000000p-93, -0x1.3fffffffffe00p-397, -0x1.fe00040000000p-472, -0x1.8fd800013fd80p-417, -0x1.ffffc7d404500p-472},
{-0x1.3fe0000100000p-20, 0x1.0000000000000p-93, 0x1.3fffffffffe00p-397, 0x1.fe00040000000p-472, -0x1.8fd800013fd80p-417, -0x1.ffffc7d404500p-472},
{-0x1.3fe0000100000p-20, 0x1.0000000000000p-93, -0x1.3fffffffffe00p-397, -0x1.fe00040000000p-472, 0x1.8fd800013fd80p-417, 0x1.ffffc7d404500p-472},
{0x1.0001fff800000p-371, 0x1.ffffff0000ffep-425, 0x1.00ff000000800p-806, -0x1.0000000800000p-872, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.0001fff800000p-371, 0x1.ffffff0000ffep-425, -0x1.00ff000000800p-806, 0x1.0000000800000p-872, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.0001fff800000p-371, -0x1.ffffff0000ffep-425, 0x1.00ff000000800p-806, -0x1.0000000800000p-872, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.0001fff800000p-371, -0x1.ffffff0000ffep-425, -0x1.00ff000000800p-806, 0x1.0000000800000p-872, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.0000800000000p-723, -0x1.ffffffe400000p-786, 0x1.000001ffffffap-94, -0x1.ffe1f80400000p-169, 0x1.0000820000ffap-817, -0x1.0c100109bfa00p-879},
{0x1.0000800000000p-723, -0x1.ffffffe400000p-786, -0x1.000001ffffffap-94, 0x1.ffe1f80400000p-169, -0x1.0000820000ffap-817, 0x1.0c100109bfa00p-879},
{-0x1.0000800000000p-723, 0x1.ffffffe400000p-786, 0x1.000001ffffffap-94, -0x1.ffe1f80400000p-169, -0x1.0000820000ffap-817, 0x1.0c100109bfa00p-879},
{-0x1.0000800000000p-723, 0x1.ffffffe400000p-786, -0x1.000001ffffffap-94, 0x1.ffe1f80400000p-169, 0x1.0000820000ffap-817, -0x1.0c100109bfa00p-879},
{0x1.0000010000000p-343, -0x1.0000000000000p-448, 0x1.007fffffffc00p-425, 0x1.fffffc0000000p-496, 0x1.008001007fc00p-768, -0x1.e000002004000p-835},
{0x1.0000010000000p-343, -0x1.0000000000000p-448, -0x1.007fffffffc00p-425, -0x1.fffffc0000000p-496, -0x1.008001007fc00p-768, 0x1.e000002004000p-835},
{-0x1.0000010000000p-343, 0x1.0000000000000p-448, 0x1.007fffffffc00p-425, 0x1.fffffc0000000p-496, -0x1.008001007fc00p-768, 0x1.e000002004000p-835},
{-0x1.0000010000000p-343, 0x1.0000000000000p-448, -0x1.007fffffffc00p-425, -0x1.fffffc0000000p-496, 0x1.008001007fc00p-768, -0x1.e000002004000p-835},
{0x1.0080000000000p-417, -0x1.0000000000000p-490, 0x1.001ffffffc000p-985, 0x1.0000000000000p-1039, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.0080000000000p-417, -0x1.0000000000000p-490, -0x1.001ffffffc000p-985, -0x1.0000000000000p-1039, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.0080000000000p-417, 0x1.0000000000000p-490, 0x1.001ffffffc000p-985, 0x1.0000000000000p-1039, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.0080000000000p-417, 0x1.0000000000000p-490, -0x1.001ffffffc000p-985, -0x1.0000000000000p-1039, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.000001fffff80p-758, 0x1.fffffc001fffcp-813, 0x1.00001ffff8000p+939, 0x1.00007f001e000p+884, 0x1.000021fff8380p+181, 0x1.7bc05e90173aep+127},
{0x1.000001fffff80p-758, 0x1.fffffc001fffcp-813, -0x1.00001ffff8000p+939, -0x1.00007f001e000p+884, -0x1.000021fff8380p+181, -0x1.7bc05e90173aep+127},
{-0x1.000001fffff80p-758, -0x1.fffffc001fffcp-813, 0x1.00001ffff8000p+939, 0x1.00007f001e000p+884, -0x1.000021fff8380p+181, -0x1.7bc05e90173aep+127},
{-0x1.000001fffff80p-758, -0x1.fffffc001fffcp-813, -0x1.00001ffff8000p+939, -0x1.00007f001e000p+884, 0x1.000021fff8380p+181, 0x1.7bc05e90173aep+127},
{0x1.00f3ffffff000p-189, 0x1.fff8000000000p-257, 0x1.3dfffffffc000p+294, 0x1.ff000003ffe00p+233, 0x1.3f2f17fffabe3p+105, 0x1.02efbc1200d00p+45},
{0x1.00f3ffffff000p-189, 0x1.fff8000000000p-257, -0x1.3dfffffffc000p+294, -0x1.ff000003ffe00p+233, -0x1.3f2f17fffabe3p+105, -0x1.02efbc1200d00p+45},
{-0x1.00f3ffffff000p-189, -0x1.fff8000000000p-257, 0x1.3dfffffffc000p+294, 0x1.ff000003ffe00p+233, -0x1.3f2f17fffabe3p+105, -0x1.02efbc1200d00p+45},
{-0x1.00f3ffffff000p-189, -0x1.fff8000000000p-257, -0x1.3dfffffffc000p+294, -0x1.ff000003ffe00p+233, 0x1.3f2f17fffabe3p+105, 0x1.02efbc1200d00p+45},
{0x1.0000001ffff80p-98, 0x1.fffffdfffff8ep-152, 0x1.0000000000000p+1023, -0x1.0000000000000p+953, 0x1.0000001ffff80p+925, 0x1.fffefdffffd8ep+871},
{0x1.0000001ffff80p-98, 0x1.fffffdfffff8ep-152, -0x1.0000000000000p+1023, 0x1.0000000000000p+953, -0x1.0000001ffff80p+925, -0x1.fffefdffffd8ep+871},
{-0x1.0000001ffff80p-98, -0x1.fffffdfffff8ep-152, 0x1.0000000000000p+1023, -0x1.0000000000000p+953, -0x1.0000001ffff80p+925, -0x1.fffefdffffd8ep+871},
{-0x1.0000001ffff80p-98, -0x1.fffffdfffff8ep-152, -0x1.0000000000000p+1023, 0x1.0000000000000p+953, 0x1.0000001ffff80p+925, 0x1.fffefdffffd8ep+871},
{0x1.001fffffffe00p+730, 0x0.0000000000000p+0, 0x1.0000002000000p+702, -0x1.0000000000000p+602, INFINITY, 0x0.0000000000000p+0},
{0x1.001fffffffe00p+730, 0x0.0000000000000p+0, -0x1.0000002000000p+702, 0x1.0000000000000p+602, -INFINITY, 0x0.0000000000000p+0},
{-0x1.001fffffffe00p+730, 0x0.0000000000000p+0, 0x1.0000002000000p+702, -0x1.0000000000000p+602, -INFINITY, 0x0.0000000000000p+0},
{-0x1.001fffffffe00p+730, 0x0.0000000000000p+0, -0x1.0000002000000p+702, 0x1.0000000000000p+602, INFINITY, 0x0.0000000000000p+0},
{0x1.01ffffffe0000p+902, 0x1.c000000200000p+841, 0x1.0001fff800ffep-251, 0x1.c000000000000p-354, 0x1.020203f7d101ap+651, -0x1.88001fe2ff740p+592},
{0x1.01ffffffe0000p+902, 0x1.c000000200000p+841, -0x1.0001fff800ffep-251, -0x1.c000000000000p-354, -0x1.020203f7d101ap+651, 0x1.88001fe2ff740p+592},
{-0x1.01ffffffe0000p+902, -0x1.c000000200000p+841, 0x1.0001fff800ffep-251, 0x1.c000000000000p-354, -0x1.020203f7d101ap+651, 0x1.88001fe2ff740p+592},
{-0x1.01ffffffe0000p+902, -0x1.c000000200000p+841, -0x1.0001fff800ffep-251, -0x1.c000000000000p-354, 0x1.020203f7d101ap+651, -0x1.88001fe2ff740p+592},
{0x1.00000007ff800p-179, 0x1.ffff800000000p-267, 0x1.00003fe0c4000p+973, -0x1.fffc032000000p+910, 0x1.00003fe8c3820p+794, -0x1.1dd05a045ffe0p+738},
{0x1.00000007ff800p-179, 0x1.ffff800000000p-267, -0x1.00003fe0c4000p+973, 0x1.fffc032000000p+910, -0x1.00003fe8c3820p+794, 0x1.1dd05a045ffe0p+738},
{-0x1.00000007ff800p-179, -0x1.ffff800000000p-267, 0x1.00003fe0c4000p+973, -0x1.fffc032000000p+910, -0x1.00003fe8c3820p+794, 0x1.1dd05a045ffe0p+738},
{-0x1.00000007ff800p-179, -0x1.ffff800000000p-267, -0x1.00003fe0c4000p+973, 0x1.fffc032000000p+910, 0x1.00003fe8c3820p+794, -0x1.1dd05a045ffe0p+738},
{0x1.000001fff8008p-475, -0x1.0000000000000p-557, 0x1.000ffffff8000p+593, 0x1.ffffffffffc00p+538, 0x1.0010020010000p+118, 0x1.fc0023ddedc10p+63},
{0x1.000001fff8008p-475, -0x1.0000000000000p-557, -0x1.000ffffff8000p+593, -0x1.ffffffffffc00p+538, -0x1.0010020010000p+118, -0x1.fc0023ddedc10p+63},
{-0x1.000001fff8008p-475, 0x1.0000000000000p-557, 0x1.000ffffff8000p+593, 0x1.ffffffffffc00p+538, -0x1.0010020010000p+118, -0x1.fc0023ddedc10p+63},
{-0x1.000001fff8008p-475, 0x1.0000000000000p-557, -0x1.000ffffff8000p+593, -0x1.ffffffffffc00p+538, 0x1.0010020010000p+118, 0x1.fc0023ddedc10p+63},
{0x1.007fbfffffcf0p-964, 0x1.ffff803ffe000p-1019, 0x1.0007ffff00000p+383, 0x1.fffff80000000p+313, 0x1.0087c3fcff4f4p-581, 0x1.3c139ab8eff88p-636},
{0x1.007fbfffffcf0p-964, 0x1.ffff803ffe000p-1019, -0x1.0007ffff00000p+383, -0x1.fffff80000000p+313, -0x1.0087c3fcff4f4p-581, -0x1.3c139ab8eff88p-636},
{-0x1.007fbfffffcf0p-964, -0x1.ffff803ffe000p-1019, 0x1.0007ffff00000p+383, 0x1.fffff80000000p+313, -0x1.0087c3fcff4f4p-581, -0x1.3c139ab8eff88p-636},
{-0x1.007fbfffffcf0p-964, -0x1.ffff803ffe000p-1019, -0x1.0007ffff00000p+383, -0x1.fffff80000000p+313, 0x1.0087c3fcff4f4p-581, 0x1.3c139ab8eff88p-636},
{0x1.0000001fc4000p-493, -0x1.fe003ffe00000p-549, 0x1.01ffffffc0004p+997, -0x1.fffc100000000p+936, 0x1.0200001fc3884p+504, -0x1.0cf68c873c5e4p+449},
{0x1.0000001fc4000p-493, -0x1.fe003ffe00000p-549, -0x1.01ffffffc0004p+997, 0x1.fffc100000000p+936, -0x1.0200001fc3884p+504, 0x1.0cf68c873c5e4p+449},
{-0x1.0000001fc4000p-493, 0x1.fe003ffe00000p-549, 0x1.01ffffffc0004p+997, -0x1.fffc100000000p+936, -0x1.0200001fc3884p+504, 0x1.0cf68c873c5e4p+449},
{-0x1.0000001fc4000p-493, 0x1.fe003ffe00000p-549, -0x1.01ffffffc0004p+997, 0x1.fffc100000000p+936, 0x1.0200001fc3884p+504, -0x1.0cf68c873c5e4p+449},
{0x1.0007e3ffe0000p-845, 0x1.fe0ffffffe000p-909, 0x1.0000001000000p+62, -0x1.ffffff0000800p-2, 0x1.0007e40fe07e4p-783, -0x1.00ffe36fecc00p-846},
{0x1.0007e3ffe0000p-845, 0x1.fe0ffffffe000p-909, -0x1.0000001000000p+62, 0x1.ffffff0000800p-2, -0x1.0007e40fe07e4p-783, 0x1.00ffe36fecc00p-846},
{-0x1.0007e3ffe0000p-845, -0x1.fe0ffffffe000p-909, 0x1.0000001000000p+62, -0x1.ffffff0000800p-2, -0x1.0007e40fe07e4p-783, 0x1.00ffe36fecc00p-846},
{-0x1.0007e3ffe0000p-845, -0x1.fe0ffffffe000p-909, -0x1.0000001000000p+62, 0x1.ffffff0000800p-2, 0x1.0007e40fe07e4p-783, -0x1.00ffe36fecc00p-846},
{0x1.0000001000000p-487, -0x1.fffffff804000p-554, 0x1.003ffffff8000p+696, 0x1.f00000007fc00p+638, 0x1.0040000ffc000p+209, 0x1.ecffc01f83c60p+151},
{0x1.0000001000000p-487, -0x1.fffffff804000p-554, -0x1.003ffffff8000p+696, -0x1.f00000007fc00p+638, -0x1.0040000ffc000p+209, -0x1.ecffc01f83c60p+151},
{-0x1.0000001000000p-487, 0x1.fffffff804000p-554, 0x1.003ffffff8000p+696, 0x1.f00000007fc00p+638, -0x1.0040000ffc000p+209, -0x1.ecffc01f83c60p+151},
{-0x1.0000001000000p-487, 0x1.fffffff804000p-554, -0x1.003ffffff8000p+696, -0x1.f00000007fc00p+638, 0x1.0040000ffc000p+209, 0x1.ecffc01f83c60p+151},
{0x1.1ffff80000000p+591, 0x1.ffffff7ff8000p+532, 0x1.00ff000000004p+69, -0x1.ffffff0000400p+6, 0x1.211ed7f808005p+660, -0x1.f1108ffb743c6p+606},
{0x1.1ffff80000000p+591, 0x1.ffffff7ff8000p+532, -0x1.00ff000000004p+69, 0x1.ffffff0000400p+6, -0x1.211ed7f808005p+660, 0x1.f1108ffb743c6p+606},
{-0x1.1ffff80000000p+591, -0x1.ffffff7ff8000p+532, 0x1.00ff000000004p+69, -0x1.ffffff0000400p+6, -0x1.211ed7f808005p+660, 0x1.f1108ffb743c6p+606},
{-0x1.1ffff80000000p+591, -0x1.ffffff7ff8000p+532, -0x1.00ff000000004p+69, 0x1.ffffff0000400p+6, 0x1.211ed7f808005p+660, -0x1.f1108ffb743c6p+606},
{0x1.000000ffff800p-309, 0x1.fffffc0000000p-392, 0x1.0008000000000p+75, -0x1.0000000000000p+7, 0x1.0008010007800p-234, -0x1.0003fff003802p-288},
{0x1.000000ffff800p-309, 0x1.fffffc0000000p-392, -0x1.0008000000000p+75, 0x1.0000000000000p+7, -0x1.0008010007800p-234, 0x1.0003fff003802p-288},
{-0x1.000000ffff800p-309, -0x1.fffffc0000000p-392, 0x1.0008000000000p+75, -0x1.0000000000000p+7, -0x1.0008010007800p-234, 0x1.0003fff003802p-288},
{-0x1.000000ffff800p-309, -0x1.fffffc0000000p-392, -0x1.0008000000000p+75, 0x1.0000000000000p+7, 0x1.0008010007800p-234, -0x1.0003fff003802p-288},
{0x1.0000183ffffc0p+670, 0x1.fff07fcfffffcp+616, 0x1.0000010000000p+363, -0x1.0000000000000p+288, INFINITY, 0x0.0000000000000p+0},
{0x1.0000183ffffc0p+670, 0x1.fff07fcfffffcp+616, -0x1.0000010000000p+363, 0x1.0000000000000p+288, -INFINITY, 0x0.0000000000000p+0},
{-0x1.0000183ffffc0p+670, -0x1.fff07fcfffffcp+616, 0x1.0000010000000p+363, -0x1.0000000000000p+288, -INFINITY, 0x0.0000000000000p+0},
{-0x1.0000183ffffc0p+670, -0x1.fff07fcfffffcp+616, -0x1.0000010000000p+363, 0x1.0000000000000p+288, INFINITY, 0x0.0000000000000p+0},
{0x1.0001000000000p-455, -0x1.fffffff800400p-518, 0x1.0000800000000p-998, -0x1.0000000000000p-1070, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.0001000000000p-455, -0x1.fffffff800400p-518, -0x1.0000800000000p-998, 0x1.0000000000000p-1070, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.0001000000000p-455, 0x1.fffffff800400p-518, 0x1.0000800000000p-998, -0x1.0000000000000p-1070, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.0001000000000p-455, 0x1.fffffff800400p-518, -0x1.0000800000000p-998, 0x1.0000000000000p-1070, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.3ffff800fff81p-572, -0x1.0000000000000p-663, 0x1.1fffffff00000p+890, 0x1.fffffff000000p+813, 0x1.67fff6ffdff72p+318, -0x1.803ff90409202p+264},
{0x1.3ffff800fff81p-572, -0x1.0000000000000p-663, -0x1.1fffffff00000p+890, -0x1.fffffff000000p+813, -0x1.67fff6ffdff72p+318, 0x1.803ff90409202p+264},
{-0x1.3ffff800fff81p-572, 0x1.0000000000000p-663, 0x1.1fffffff00000p+890, 0x1.fffffff000000p+813, -0x1.67fff6ffdff72p+318, 0x1.803ff90409202p+264},
{-0x1.3ffff800fff81p-572, 0x1.0000000000000p-663, -0x1.1fffffff00000p+890, -0x1.fffffff000000p+813, 0x1.67fff6ffdff72p+318, -0x1.803ff90409202p+264},
{0x1.fffffc0000100p-97, -0x1.fffe000040000p-168, 0x1.0000800000000p-692, -0x1.c000fffffffd0p-749, 0x1.00007dffff080p-788, -0x1.b804fc7dfe0a0p-845},
{0x1.fffffc0000100p-97, -0x1.fffe000040000p-168, -0x1.0000800000000p-692, 0x1.c000fffffffd0p-749, -0x1.00007dffff080p-788, 0x1.b804fc7dfe0a0p-845},
{-0x1.fffffc0000100p-97, 0x1.fffe000040000p-168, 0x1.0000800000000p-692, -0x1.c000fffffffd0p-749, -0x1.00007dffff080p-788, 0x1.b804fc7dfe0a0p-845},
{-0x1.fffffc0000100p-97, 0x1.fffe000040000p-168, -0x1.0000800000000p-692, 0x1.c000fffffffd0p-749, 0x1.00007dffff080p-788, -0x1.b804fc7dfe0a0p-845},
{0x1.003f80ff00004p-671, -0x1.ffffc00040000p-742, 0x1.0000ffffff000p+38, 0x1.ffe000003ffe0p-20, 0x1.0040813e7fff0p-633, 0x1.fea97cff90170p-690},
{0x1.003f80ff00004p-671, -0x1.ffffc00040000p-742, -0x1.0000ffffff000p+38, -0x1.ffe000003ffe0p-20, -0x1.0040813e7fff0p-633, -0x1.fea97cff90170p-690},
{-0x1.003f80ff00004p-671, 0x1.ffffc00040000p-742, 0x1.0000ffffff000p+38, 0x1.ffe000003ffe0p-20, -0x1.0040813e7fff0p-633, -0x1.fea97cff90170p-690},
{-0x1.003f80ff00004p-671, 0x1.ffffc00040000p-742, -0x1.0000ffffff000p+38, -0x1.ffe000003ffe0p-20, 0x1.0040813e7fff0p-633, 0x1.fea97cff90170p-690},
{0x1.000003fffff80p+967, 0x1.f007ffe00fc00p+904, 0x1.3ff8007ffffc0p-732, 0x1.e0007ff000000p-798, 0x1.3ff8057fdff40p+235, 0x1.14efd48c70c80p+177},
{0x1.000003fffff80p+967, 0x1.f007ffe00fc00p+904, -0x1.3ff8007ffffc0p-732, -0x1.e0007ff000000p-798, -0x1.3ff8057fdff40p+235, -0x1.14efd48c70c80p+177},
{-0x1.000003fffff80p+967, -0x1.f007ffe00fc00p+904, 0x1.3ff8007ffffc0p-732, 0x1.e0007ff000000p-798, -0x1.3ff8057fdff40p+235, -0x1.14efd48c70c80p+177},
{-0x1.000003fffff80p+967, -0x1.f007ffe00fc00p+904, -0x1.3ff8007ffffc0p-732, -0x1.e0007ff000000p-798, 0x1.3ff8057fdff40p+235, 0x1.14efd48c70c80p+177},
{0x1.ffffffe000000p-621, -0x1.ff80000020000p-677, 0x1.0000ffffff81fp+121, 0x1.ffc7fff800000p+57, 0x1.0000ffefff71fp-499, -0x1.fb72ad806efd0p-556},
{0x1.ffffffe000000p-621, -0x1.ff80000020000p-677, -0x1.0000ffffff81fp+121, -0x1.ffc7fff800000p+57, -0x1.0000ffefff71fp-499, 0x1.fb72ad806efd0p-556},
{-0x1.ffffffe000000p-621, 0x1.ff80000020000p-677, 0x1.0000ffffff81fp+121, 0x1.ffc7fff800000p+57, -0x1.0000ffefff71fp-499, 0x1.fb72ad806efd0p-556},
{-0x1.ffffffe000000p-621, 0x1.ff80000020000p-677, -0x1.0000ffffff81fp+121, -0x1.ffc7fff800000p+57, 0x1.0000ffefff71fp-499, -0x1.fb72ad806efd0p-556},
{0x1.0000000800000p-286, -0x1.ffffc02000000p-364, 0x1.7fffe00800000p-363, -0x1.0000000000000p-468, 0x1.7fffe013ffff0p-649, 0x1.ffffa00013f40p-708},
{0x1.0000000800000p-286, -0x1.ffffc02000000p-364, -0x1.7fffe00800000p-363, 0x1.0000000000000p-468, -0x1.7fffe013ffff0p-649, -0x1.ffffa00013f40p-708},
{-0x1.0000000800000p-286, 0x1.ffffc02000000p-364, 0x1.7fffe00800000p-363, -0x1.0000000000000p-468, -0x1.7fffe013ffff0p-649, -0x1.ffffa00013f40p-708},
{-0x1.0000000800000p-286, 0x1.ffffc02000000p-364, -0x1.7fffe00800000p-363, 0x1.0000000000000p-468, 0x1.7fffe013ffff0p-649, 0x1.ffffa00013f40p-708},
{0x1.0000400000000p+600, -0x1.0000000000000p+509, 0x1.4000000000000p-65, -0x1.ffffffff00020p-123, 0x1.4000500000000p+535, -0x1.00003fffd0010p+478},
{0x1.0000400000000p+600, -0x1.0000000000000p+509, -0x1.4000000000000p-65, 0x1.ffffffff00020p-123, -0x1.4000500000000p+535, 0x1.00003fffd0010p+478},
{-0x1.0000400000000p+600, 0x1.0000000000000p+509, 0x1.4000000000000p-65, -0x1.ffffffff00020p-123, -0x1.4000500000000p+535, 0x1.00003fffd0010p+478},
{-0x1.0000400000000p+600, 0x1.0000000000000p+509, -0x1.4000000000000p-65, 0x1.ffffffff00020p-123, 0x1.4000500000000p+535, -0x1.00003fffd0010p+478},
{0x1.000000fffff80p-307, 0x1.ffffef003e000p-370, 0x1.0000010000000p-490, -0x1.8000000000020p-548, 0x1.000001fffff90p-797, -0x1.702001f7fe060p-855},
{0x1.000000fffff80p-307, 0x1.ffffef003e000p-370, -0x1.0000010000000p-490, 0x1.8000000000020p-548, -0x1.000001fffff90p-797, 0x1.702001f7fe060p-855},
{-0x1.000000fffff80p-307, -0x1.ffffef003e000p-370, 0x1.0000010000000p-490, -0x1.8000000000020p-548, -0x1.000001fffff90p-797, 0x1.702001f7fe060p-855},
{-0x1.000000fffff80p-307, -0x1.ffffef003e000p-370, -0x1.0000010000000p-490, 0x1.8000000000020p-548, 0x1.000001fffff90p-797, -0x1.702001f7fe060p-855},
{0x1.7fffffffc0000p-18, 0x1.fffc000000000p-88, 0x1.0000001fe0000p-469, 0x1.fffffc0001f00p-530, 0x1.8000002f90000p-487, 0x1.00fffbffd1600p-547},
{0x1.7fffffffc0000p-18, 0x1.fffc000000000p-88, -0x1.0000001fe0000p-469, -0x1.fffffc0001f00p-530, -0x1.8000002f90000p-487, -0x1.00fffbffd1600p-547},
{-0x1.7fffffffc0000p-18, -0x1.fffc000000000p-88, 0x1.0000001fe0000p-469, 0x1.fffffc0001f00p-530, -0x1.8000002f90000p-487, -0x1.00fffbffd1600p-547},
{-0x1.7fffffffc0000p-18, -0x1.fffc000000000p-88, -0x1.0000001fe0000p-469, -0x1.fffffc0001f00p-530, 0x1.8000002f90000p-487, 0x1.00fffbffd1600p-547},
{0x1.0000004000000p+707, -0x1.0000000000000p+649, 0x1.8000000000000p+850, -0x1.ffe7ffc001000p+785, INFINITY, 0x0.0000000000000p+0},
{0x1.0000004000000p+707, -0x1.0000000000000p+649, -0x1.8000000000000p+850, 0x1.ffe7ffc001000p+785, -INFINITY, 0x0.0000000000000p+0},
{-0x1.0000004000000p+707, 0x1.0000000000000p+649, 0x1.8000000000000p+850, -0x1.ffe7ffc001000p+785, -INFINITY, 0x0.0000000000000p+0},
{-0x1.0000004000000p+707, 0x1.0000000000000p+649, -0x1.8000000000000p+850, 0x1.ffe7ffc001000p+785, INFINITY, 0x0.0000000000000p+0},
{0x1.0000000800000p+842, -0x1.fffffff400000p+767, 0x1.007fc00007800p+448, 0x1.e0000f80001e0p+390, INFINITY, 0x0.0000000000000p+0},
{0x1.0000000800000p+842, -0x1.fffffff400000p+767, -0x1.007fc00007800p+448, -0x1.e0000f80001e0p+390, -INFINITY, 0x0.0000000000000p+0},
{-0x1.0000000800000p+842, 0x1.fffffff400000p+767, 0x1.007fc00007800p+448, 0x1.e0000f80001e0p+390, -INFINITY, 0x0.0000000000000p+0},
{-0x1.0000000800000p+842, 0x1.fffffff400000p+767, -0x1.007fc00007800p+448, -0x1.e0000f80001e0p+390, INFINITY, 0x0.0000000000000p+0},
{0x1.3fffff0000000p+131, 0x1.fffffe0000000p+68, 0x1.0000200000000p+733, -0x1.0000000000000p+628, 0x1.400026ffffe00p+864, 0x1.00001effffc00p+802},
{0x1.3fffff0000000p+131, 0x1.fffffe0000000p+68, -0x1.0000200000000p+733, 0x1.0000000000000p+628, -0x1.400026ffffe00p+864, -0x1.00001effffc00p+802},
{-0x1.3fffff0000000p+131, -0x1.fffffe0000000p+68, 0x1.0000200000000p+733, -0x1.0000000000000p+628, -0x1.400026ffffe00p+864, -0x1.00001effffc00p+802},
{-0x1.3fffff0000000p+131, -0x1.fffffe0000000p+68, -0x1.0000200000000p+733, 0x1.0000000000000p+628, 0x1.400026ffffe00p+864, 0x1.00001effffc00p+802},
{0x1.7fe0000000008p+383, -0x1.fffff3ff80200p+321, 0x1.000001fff8400p-754, -0x1.fffff80010000p-819, 0x1.7fe002ffb4617p-371, 0x1.fda04808886c8p-425},
{0x1.7fe0000000008p+383, -0x1.fffff3ff80200p+321, -0x1.000001fff8400p-754, 0x1.fffff80010000p-819, -0x1.7fe002ffb4617p-371, -0x1.fda04808886c8p-425},
{-0x1.7fe0000000008p+383, 0x1.fffff3ff80200p+321, 0x1.000001fff8400p-754, -0x1.fffff80010000p-819, -0x1.7fe002ffb4617p-371, -0x1.fda04808886c8p-425},
{-0x1.7fe0000000008p+383, 0x1.fffff3ff80200p+321, -0x1.000001fff8400p-754, 0x1.fffff80010000p-819, 0x1.7fe002ffb4617p-371, 0x1.fda04808886c8p-425},
{0x1.00000fffffff8p-769, 0x1.fffffffc00000p-828, 0x1.0000007001000p+11, -0x1.e00000fff8200p-49, 0x1.0000107001068p-758, 0x1.8ffe315805e80p-817},
{0x1.00000fffffff8p-769, 0x1.fffffffc00000p-828, -0x1.0000007001000p+11, 0x1.e00000fff8200p-49, -0x1.0000107001068p-758, -0x1.8ffe315805e80p-817},
{-0x1.00000fffffff8p-769, -0x1.fffffffc00000p-828, 0x1.0000007001000p+11, -0x1.e00000fff8200p-49, -0x1.0000107001068p-758, -0x1.8ffe315805e80p-817},
{-0x1.00000fffffff8p-769, -0x1.fffffffc00000p-828, -0x1.0000007001000p+11, 0x1.e00000fff8200p-49, 0x1.0000107001068p-758, 0x1.8ffe315805e80p-817},
{0x1.001fffc000400p-304, -0x1.e000000001000p-369, 0x1.0200000000000p+539, -0x1.0000000000000p+436, 0x1.02203fbf80408p+235, -0x1.e3c0000005000p+170},
{0x1.001fffc000400p-304, -0x1.e000000001000p-369, -0x1.0200000000000p+539, 0x1.0000000000000p+436, -0x1.02203fbf80408p+235, 0x1.e3c0000005000p+170},
{-0x1.001fffc000400p-304, 0x1.e000000001000p-369, 0x1.0200000000000p+539, -0x1.0000000000000p+436, -0x1.02203fbf80408p+235, 0x1.e3c0000005000p+170},
{-0x1.001fffc000400p-304, 0x1.e000000001000p-369, -0x1.0200000000000p+539, 0x1.0000000000000p+436, 0x1.02203fbf80408p+235, -0x1.e3c0000005000p+170},
{0x1.0000008000000p+148, -0x1.0000000000000p+78, 0x1.07fff80000002p-186, -0x1.c010000000000p-257, 0x1.07fff883fffc2p-38, -0x1.e407f87000000p-108},
{0x1.0000008000000p+148, -0x1.0000000000000p+78, -0x1.07fff80000002p-186, 0x1.c010000000000p-257, -0x1.07fff883fffc2p-38, 0x1.e407f87000000p-108},
{-0x1.0000008000000p+148, 0x1.0000000000000p+78, 0x1.07fff80000002p-186, -0x1.c010000000000p-257, -0x1.07fff883fffc2p-38, 0x1.e407f87000000p-108},
{-0x1.0000008000000p+148, 0x1.0000000000000p+78, -0x1.07fff80000002p-186, 0x1.c010000000000p-257, 0x1.07fff883fffc2p-38, -0x1.e407f87000000p-108},
{0x1.03ffffffe0000p+267, 0x1.ff80000000000p+178, 0x1.7ffff00000002p+830, -0x1.0000000000000p+737, INFINITY, 0x0.0000000000000p+0},
{0x1.03ffffffe0000p+267, 0x1.ff80000000000p+178, -0x1.7ffff00000002p+830, 0x1.0000000000000p+737, -INFINITY, 0x0.0000000000000p+0},
{-0x1.03ffffffe0000p+267, -0x1.ff80000000000p+178, 0x1.7ffff00000002p+830, -0x1.0000000000000p+737, -INFINITY, 0x0.0000000000000p+0},
{-0x1.03ffffffe0000p+267, -0x1.ff80000000000p+178, -0x1.7ffff00000002p+830, 0x1.0000000000000p+737, INFINITY, 0x0.0000000000000p+0},
{0x1.0008000000000p-881, -0x1.9000000000000p-982, 0x1.0000000700002p-545, -0x1.fffff80200000p-608, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.0008000000000p-881, -0x1.9000000000000p-982, -0x1.0000000700002p-545, 0x1.fffff80200000p-608, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.0008000000000p-881, 0x1.9000000000000p-982, 0x1.0000000700002p-545, -0x1.fffff80200000p-608, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.0008000000000p-881, 0x1.9000000000000p-982, -0x1.0000000700002p-545, 0x1.fffff80200000p-608, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.0010000000000p-465, -0x1.0000000000000p-542, 0x1.007fc00001000p+895, -0x1.ff80000400000p+839, 0x1.008fc7fc01001p+430, -0x1.ffa00007fe400p+374},
{0x1.0010000000000p-465, -0x1.0000000000000p-542, -0x1.007fc00001000p+895, 0x1.ff80000400000p+839, -0x1.008fc7fc01001p+430, 0x1.ffa00007fe400p+374},
{-0x1.0010000000000p-465, 0x1.0000000000000p-542, 0x1.007fc00001000p+895, -0x1.ff80000400000p+839, -0x1.008fc7fc01001p+430, 0x1.ffa00007fe400p+374},
{-0x1.0010000000000p-465, 0x1.0000000000000p-542, -0x1.007fc00001000p+895, 0x1.ff80000400000p+839, 0x1.008fc7fc01001p+430, -0x1.ffa00007fe400p+374},
{0x1.0000008000000p-249, -0x1.fff0000000100p-308, 0x1.7fffff8100000p+89, -0x1.fffcc00000080p+29, 0x1.80000040ffffcp-160, 0x1.9a07df8c40000p-231},
{0x1.0000008000000p-249, -0x1.fff0000000100p-308, -0x1.7fffff8100000p+89, 0x1.fffcc00000080p+29, -0x1.80000040ffffcp-160, -0x1.9a07df8c40000p-231},
{-0x1.0000008000000p-249, 0x1.fff0000000100p-308, 0x1.7fffff8100000p+89, -0x1.fffcc00000080p+29, -0x1.80000040ffffcp-160, -0x1.9a07df8c40000p-231},
{-0x1.0000008000000p-249, 0x1.fff0000000100p-308, -0x1.7fffff8100000p+89, 0x1.fffcc00000080p+29, 0x1.80000040ffffcp-160, 0x1.9a07df8c40000p-231},
{0x1.0000003ffc010p+789, -0x1.ffffc00400000p+720, 0x1.0000c00400000p+535, -0x1.ff80800000000p+454, INFINITY, 0x0.0000000000000p+0},
{0x1.0000003ffc010p+789, -0x1.ffffc00400000p+720, -0x1.0000c00400000p+535, 0x1.ff80800000000p+454, -INFINITY, 0x0.0000000000000p+0},
{-0x1.0000003ffc010p+789, 0x1.ffffc00400000p+720, 0x1.0000c00400000p+535, -0x1.ff80800000000p+454, -INFINITY, 0x0.0000000000000p+0},
{-0x1.0000003ffc010p+789, 0x1.ffffc00400000p+720, -0x1.0000c00400000p+535, 0x1.ff80800000000p+454, INFINITY, 0x0.0000000000000p+0},
{0x1.0000007fff800p+957, 0x1.ffff800007fc0p+898, 0x1.0000007fff000p+26, 0x1.ffffff8000000p-49, 0x1.000000fffe804p+983, 0x1.fa018500063c0p+924},
{0x1.0000007fff800p+957, 0x1.ffff800007fc0p+898, -0x1.0000007fff000p+26, -0x1.ffffff8000000p-49, -0x1.000000fffe804p+983, -0x1.fa018500063c0p+924},
{-0x1.0000007fff800p+957, -0x1.ffff800007fc0p+898, 0x1.0000007fff000p+26, 0x1.ffffff8000000p-49, -0x1.000000fffe804p+983, -0x1.fa018500063c0p+924},
{-0x1.0000007fff800p+957, -0x1.ffff800007fc0p+898, -0x1.0000007fff000p+26, -0x1.ffffff8000000p-49, 0x1.000000fffe804p+983, 0x1.fa018500063c0p+924},
{0x1.4000000000000p-257, -0x1.0000000000000p-339, 0x1.0000001ff8040p+59, -0x1.fff0ffffff008p+3, 0x1.40000027f6050p-198, -0x1.3ff6a01fff604p-253},
{0x1.4000000000000p-257, -0x1.0000000000000p-339, -0x1.0000001ff8040p+59, 0x1.fff0ffffff008p+3, -0x1.40000027f6050p-198, 0x1.3ff6a01fff604p-253},
{-0x1.4000000000000p-257, 0x1.0000000000000p-339, 0x1.0000001ff8040p+59, -0x1.fff0ffffff008p+3, -0x1.40000027f6050p-198, 0x1.3ff6a01fff604p-253},
{-0x1.4000000000000p-257, 0x1.0000000000000p-339, -0x1.0000001ff8040p+59, 0x1.fff0ffffff008p+3, 0x1.40000027f6050p-198, -0x1.3ff6a01fff604p-253},
{0x1.0004000000000p+457, -0x1.fffffffe40000p+386, 0x1.000001ffff1f0p+241, 0x1.ff807ffffc000p+187, 0x1.00040200071f0p+698, 0x1.1e877e01fa00ep+644},
{0x1.0004000000000p+457, -0x1.fffffffe40000p+386, -0x1.000001ffff1f0p+241, -0x1.ff807ffffc000p+187, -0x1.00040200071f0p+698, -0x1.1e877e01fa00ep+644},
{-0x1.0004000000000p+457, 0x1.fffffffe40000p+386, 0x1.000001ffff1f0p+241, 0x1.ff807ffffc000p+187, -0x1.00040200071f0p+698, -0x1.1e877e01fa00ep+644},
{-0x1.0004000000000p+457, 0x1.fffffffe40000p+386, -0x1.000001ffff1f0p+241, -0x1.ff807ffffc000p+187, 0x1.00040200071f0p+698, 0x1.1e877e01fa00ep+644},
{0x1.003fe00030100p-370, -0x1.00003fe002000p-436, 0x1.3ffffff800200p-197, -0x1.0000000000000p-274, 0x1.404fd7f83a350p-567, 0x1.fe8bdf7b033fep-621},
{0x1.003fe00030100p-370, -0x1.00003fe002000p-436, -0x1.3ffffff800200p-197, 0x1.0000000000000p-274, -0x1.404fd7f83a350p-567, -0x1.fe8bdf7b033fep-621},
{-0x1.003fe00030100p-370, 0x1.00003fe002000p-436, 0x1.3ffffff800200p-197, -0x1.0000000000000p-274, -0x1.404fd7f83a350p-567, -0x1.fe8bdf7b033fep-621},
{-0x1.003fe00030100p-370, 0x1.00003fe002000p-436, -0x1.3ffffff800200p-197, 0x1.0000000000000p-274, 0x1.404fd7f83a350p-567, 0x1.fe8bdf7b033fep-621},
{0x1.0001ffc1ffc20p+658, -0x1.fffffc0002000p+592, 0x1.000003fffffc0p-919, 0x1.fffffe0000000p-997, 0x1.000203c207bd0p-261, 0x1.fda2400001fd6p-315},
{0x1.0001ffc1ffc20p+658, -0x1.fffffc0002000p+592, -0x1.000003fffffc0p-919, -0x1.fffffe0000000p-997, -0x1.000203c207bd0p-261, -0x1.fda2400001fd6p-315},
{-0x1.0001ffc1ffc20p+658, 0x1.fffffc0002000p+592, 0x1.000003fffffc0p-919, 0x1.fffffe0000000p-997, -0x1.000203c207bd0p-261, -0x1.fda2400001fd6p-315},
{-0x1.0001ffc1ffc20p+658, 0x1.fffffc0002000p+592, -0x1.000003fffffc0p-919, -0x1.fffffe0000000p-997, 0x1.000203c207bd0p-261, 0x1.fda2400001fd6p-315},
{0x1.0400000000000p-38, -0x1.0000000000000p-92, 0x1.0000200000000p-364, -0x1.ffffffffffc40p-423, 0x1.0400208000000p-402, -0x1.10401ffffffe2p-456},
{0x1.0400000000000p-38, -0x1.0000000000000p-92, -0x1.0000200000000p-364, 0x1.ffffffffffc40p-423, -0x1.0400208000000p-402, 0x1.10401ffffffe2p-456},
{-0x1.0400000000000p-38, 0x1.0000000000000p-92, 0x1.0000200000000p-364, -0x1.ffffffffffc40p-423, -0x1.0400208000000p-402, 0x1.10401ffffffe2p-456},
{-0x1.0400000000000p-38, 0x1.0000000000000p-92, -0x1.0000200000000p-364, 0x1.ffffffffffc40p-423, 0x1.0400208000000p-402, -0x1.10401ffffffe2p-456},
{0x1.00003ff820000p+644, -0x1.ffffffe000100p+589, 0x1.0001f80020000p+462, -0x1.ffffffc1c0040p+403, INFINITY, 0x0.0000000000000p+0},
{0x1.00003ff820000p+644, -0x1.ffffffe000100p+589, -0x1.0001f80020000p+462, 0x1.ffffffc1c0040p+403, -INFINITY, 0x0.0000000000000p+0},
{-0x1.00003ff820000p+644, 0x1.ffffffe000100p+589, 0x1.0001f80020000p+462, -0x1.ffffffc1c0040p+403, -INFINITY, 0x0.0000000000000p+0},
{-0x1.00003ff820000p+644, 0x1.ffffffe000100p+589, -0x1.0001f80020000p+462, 0x1.ffffffc1c0040p+403, INFINITY, 0x0.0000000000000p+0},
{0x1.007ffffffff80p-432, 0x1.fffffffe00000p-506, 0x1.0100000000000p+937, -0x1.0000000000000p+860, 0x1.01807ffffff80p+505, -0x1.ffffe1e100002p+451},
{0x1.007ffffffff80p-432, 0x1.fffffffe00000p-506, -0x1.0100000000000p+937, 0x1.0000000000000p+860, -0x1.01807ffffff80p+505, 0x1.ffffe1e100002p+451},
{-0x1.007ffffffff80p-432, -0x1.fffffffe00000p-506, 0x1.0100000000000p+937, -0x1.0000000000000p+860, -0x1.01807ffffff80p+505, 0x1.ffffe1e100002p+451},
{-0x1.007ffffffff80p-432, -0x1.fffffffe00000p-506, -0x1.0100000000000p+937, 0x1.0000000000000p+860, 0x1.01807ffffff80p+505, -0x1.ffffe1e100002p+451},
{0x1.0000000400000p+229, -0x1.e00003fff8200p+167, 0x1.000ffff802000p-917, -0x1.fffe3fffffe00p-979, 0x1.000ffffc02400p-688, -0x1.7bc3887f27780p-747},
{0x1.0000000400000p+229, -0x1.e00003fff8200p+167, -0x1.000ffff802000p-917, 0x1.fffe3fffffe00p-979, -0x1.000ffffc02400p-688, 0x1.7bc3887f27780p-747},
{-0x1.0000000400000p+229, 0x1.e00003fff8200p+167, 0x1.000ffff802000p-917, -0x1.fffe3fffffe00p-979, -0x1.000ffffc02400p-688, 0x1.7bc3887f27780p-747},
{-0x1.0000000400000p+229, 0x1.e00003fff8200p+167, -0x1.000ffff802000p-917, 0x1.fffe3fffffe00p-979, 0x1.000ffffc02400p-688, -0x1.7bc3887f27780p-747},
{0x1.00003ffffff00p-811, 0x1.ffc003ffffe00p-866, 0x1.0001ff0000000p+595, 0x1.ff8007fffffe0p+540, 0x1.00023f007fb00p-216, 0x1.f7a644b024dd0p-270},
{0x1.00003ffffff00p-811, 0x1.ffc003ffffe00p-866, -0x1.0001ff0000000p+595, -0x1.ff8007fffffe0p+540, -0x1.00023f007fb00p-216, -0x1.f7a644b024dd0p-270},
{-0x1.00003ffffff00p-811, -0x1.ffc003ffffe00p-866, 0x1.0001ff0000000p+595, 0x1.ff8007fffffe0p+540, -0x1.00023f007fb00p-216, -0x1.f7a644b024dd0p-270},
{-0x1.00003ffffff00p-811, -0x1.ffc003ffffe00p-866, -0x1.0001ff0000000p+595, -0x1.ff8007fffffe0p+540, 0x1.00023f007fb00p-216, 0x1.f7a644b024dd0p-270},
{0x1.1000000000000p-978, -0x1.ff00000000000p-1043, 0x1.0200000000000p+352, -0x1.c000000000008p+296, 0x1.1220000000000p-626, -0x1.dd017f0000010p-682},
{0x1.1000000000000p-978, -0x1.ff00000000000p-1043, -0x1.0200000000000p+352, 0x1.c000000000008p+296, -0x1.1220000000000p-626, 0x1.dd017f0000010p-682},
{-0x1.1000000000000p-978, 0x1.ff00000000000p-1043, 0x1.0200000000000p+352, -0x1.c000000000008p+296, -0x1.1220000000000p-626, 0x1.dd017f0000010p-682},
{-0x1.1000000000000p-978, 0x1.ff00000000000p-1043, -0x1.0200000000000p+352, 0x1.c000000000008p+296, 0x1.1220000000000p-626, -0x1.dd017f0000010p-682},
{0x1.1fffffc000000p+620, 0x1.ffffffffffff0p+563, 0x1.007fc00000800p+980, -0x1.fffff03fff000p+920, INFINITY, 0x0.0000000000000p+0},
{0x1.1fffffc000000p+620, 0x1.ffffffffffff0p+563, -0x1.007fc00000800p+980, 0x1.fffff03fff000p+920, -INFINITY, 0x0.0000000000000p+0},
{-0x1.1fffffc000000p+620, -0x1.ffffffffffff0p+563, 0x1.007fc00000800p+980, -0x1.fffff03fff000p+920, -INFINITY, 0x0.0000000000000p+0},
{-0x1.1fffffc000000p+620, -0x1.ffffffffffff0p+563, -0x1.007fc00000800p+980, 0x1.fffff03fff000p+920, INFINITY, 0x0.0000000000000p+0},
{0x1.00007fffff000p+841, 0x1.ffffffff00000p+769, 0x1.1fffffffc0000p+334, 0x1.fff0000000000p+266, INFINITY, 0x0.0000000000000p+0},
{0x1.00007fffff000p+841, 0x1.ffffffff00000p+769, -0x1.1fffffffc0000p+334, -0x1.fff0000000000p+266, -INFINITY, 0x0.0000000000000p+0},
{-0x1.00007fffff000p+841, -0x1.ffffffff00000p+769, 0x1.1fffffffc0000p+334, 0x1.fff0000000000p+266, -INFINITY, 0x0.0000000000000p+0},
{-0x1.00007fffff000p+841, -0x1.ffffffff00000p+769, -0x1.1fffffffc0000p+334, -0x1.fff0000000000p+266, INFINITY, 0x0.0000000000000p+0},
{0x1.0003ffffffffcp+878, 0x1.f000000000000p+777, 0x1.0000003ffff80p+707, 0x1.fffffefff8000p+639, INFINITY, 0x0.0000000000000p+0},
{0x1.0003ffffffffcp+878, 0x1.f000000000000p+777, -0x1.0000003ffff80p+707, -0x1.fffffefff8000p+639, -INFINITY, 0x0.0000000000000p+0},
{-0x1.0003ffffffffcp+878, -0x1.f000000000000p+777, 0x1.0000003ffff80p+707, 0x1.fffffefff8000p+639, -INFINITY, 0x0.0000000000000p+0},
{-0x1.0003ffffffffcp+878, -0x1.f000000000000p+777, -0x1.0000003ffff80p+707, -0x1.fffffefff8000p+639, INFINITY, 0x0.0000000000000p+0},
{0x1.00007ffc00000p+210, 0x1.ffffe00000000p+141, 0x1.3fffffc004000p+887, -0x1.ffffe00010000p+823, INFINITY, 0x0.0000000000000p+0},
{0x1.00007ffc00000p+210, 0x1.ffffe00000000p+141, -0x1.3fffffc004000p+887, 0x1.ffffe00010000p+823, -INFINITY, 0x0.0000000000000p+0},
{-0x1.00007ffc00000p+210, -0x1.ffffe00000000p+141, 0x1.3fffffc004000p+887, -0x1.ffffe00010000p+823, -INFINITY, 0x0.0000000000000p+0},
{-0x1.00007ffc00000p+210, -0x1.ffffe00000000p+141, -0x1.3fffffc004000p+887, 0x1.ffffe00010000p+823, INFINITY, 0x0.0000000000000p+0},
{0x1.0007ffffc0004p+901, -0x1.fffffe0040000p+830, 0x1.00ffffe000400p-637, -0x1.0000000000000p-729, 0x1.010807dfbf004p+264, 0x1.23fdf20002418p+209},
{0x1.0007ffffc0004p+901, -0x1.fffffe0040000p+830, -0x1.00ffffe000400p-637, 0x1.0000000000000p-729, -0x1.010807dfbf004p+264, -0x1.23fdf20002418p+209},
{-0x1.0007ffffc0004p+901, 0x1.fffffe0040000p+830, 0x1.00ffffe000400p-637, -0x1.0000000000000p-729, -0x1.010807dfbf004p+264, -0x1.23fdf20002418p+209},
{-0x1.0007ffffc0004p+901, 0x1.fffffe0040000p+830, -0x1.00ffffe000400p-637, 0x1.0000000000000p-729, 0x1.010807dfbf004p+264, 0x1.23fdf20002418p+209},
{0x1.007ffffffff00p-57, 0x1.fff8003000000p-134, 0x1.01fffffffe000p-715, 0x1.fffc000000000p-803, 0x1.0280fffffdeeep-772, 0x1.041c07d800000p-848},
{0x1.007ffffffff00p-57, 0x1.fff8003000000p-134, -0x1.01fffffffe000p-715, -0x1.fffc000000000p-803, -0x1.0280fffffdeeep-772, -0x1.041c07d800000p-848},
{-0x1.007ffffffff00p-57, -0x1.fff8003000000p-134, 0x1.01fffffffe000p-715, 0x1.fffc000000000p-803, -0x1.0280fffffdeeep-772, -0x1.041c07d800000p-848},
{-0x1.007ffffffff00p-57, -0x1.fff8003000000p-134, -0x1.01fffffffe000p-715, -0x1.fffc000000000p-803, 0x1.0280fffffdeeep-772, 0x1.041c07d800000p-848},
{0x1.00003e0000000p-443, 0x1.ffffffffe0000p-502, 0x1.07ffffff80000p+79, 0x1.8007ffffc0000p+8, 0x1.08003fef7fffep-364, 0x1.4203001095e38p-420},
{0x1.00003e0000000p-443, 0x1.ffffffffe0000p-502, -0x1.07ffffff80000p+79, -0x1.8007ffffc0000p+8, -0x1.08003fef7fffep-364, -0x1.4203001095e38p-420},
{-0x1.00003e0000000p-443, -0x1.ffffffffe0000p-502, 0x1.07ffffff80000p+79, 0x1.8007ffffc0000p+8, -0x1.08003fef7fffep-364, -0x1.4203001095e38p-420},
{-0x1.00003e0000000p-443, -0x1.ffffffffe0000p-502, -0x1.07ffffff80000p+79, -0x1.8007ffffc0000p+8, 0x1.08003fef7fffep-364, 0x1.4203001095e38p-420},
{0x1.00003ff810000p-96, -0x1.fffffffc00800p-160, 0x1.00000ffff8002p+307, -0x1.fffffffe04000p+240, 0x1.00004ff80bffap+211, -0x1.01f8812df9b50p+155},
{0x1.00003ff810000p-96, -0x1.fffffffc00800p-160, -0x1.00000ffff8002p+307, 0x1.fffffffe04000p+240, -0x1.00004ff80bffap+211, 0x1.01f8812df9b50p+155},
{-0x1.00003ff810000p-96, 0x1.fffffffc00800p-160, 0x1.00000ffff8002p+307, -0x1.fffffffe04000p+240, -0x1.00004ff80bffap+211, 0x1.01f8812df9b50p+155},
{-0x1.00003ff810000p-96, 0x1.fffffffc00800p-160, -0x1.00000ffff8002p+307, 0x1.fffffffe04000p+240, 0x1.00004ff80bffap+211, -0x1.01f8812df9b50p+155},
{0x1.0100000000000p-135, -0x1.0000000000000p-222, 0x1.01ffffff00004p-1000, -0x0.0000000000000p+0, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.0100000000000p-135, -0x1.0000000000000p-222, -0x1.01ffffff00004p-1000, 0x0.0000000000000p+0, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.0100000000000p-135, 0x1.0000000000000p-222, 0x1.01ffffff00004p-1000, -0x0.0000000000000p+0, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.0100000000000p-135, 0x1.0000000000000p-222, -0x1.01ffffff00004p-1000, 0x0.0000000000000p+0, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.001fffffff800p+829, 0x1.fffff00000000p+749, 0x1.07ffe00001000p+551, -0x1.c7ffc001fe010p+494, INFINITY, 0x0.0000000000000p+0},
{0x1.001fffffff800p+829, 0x1.fffff00000000p+749, -0x1.07ffe00001000p+551, 0x1.c7ffc001fe010p+494, -INFINITY, 0x0.0000000000000p+0},
{-0x1.001fffffff800p+829, -0x1.fffff00000000p+749, 0x1.07ffe00001000p+551, -0x1.c7ffc001fe010p+494, -INFINITY, 0x0.0000000000000p+0},
{-0x1.001fffffff800p+829, -0x1.fffff00000000p+749, -0x1.07ffe00001000p+551, 0x1.c7ffc001fe010p+494, INFINITY, 0x0.0000000000000p+0},
{0x1.000000ffffff8p-632, 0x1.ffe0000000000p-704, 0x1.1fffffff00001p+944, -0x1.0000000000000p+859, 0x1.2000011effff8p+312, -0x1.fffb5f5800140p+255},
{0x1.000000ffffff8p-632, 0x1.ffe0000000000p-704, -0x1.1fffffff00001p+944, 0x1.0000000000000p+859, -0x1.2000011effff8p+312, 0x1.fffb5f5800140p+255},
{-0x1.000000ffffff8p-632, -0x1.ffe0000000000p-704, 0x1.1fffffff00001p+944, -0x1.0000000000000p+859, -0x1.2000011effff8p+312, 0x1.fffb5f5800140p+255},
{-0x1.000000ffffff8p-632, -0x1.ffe0000000000p-704, -0x1.1fffffff00001p+944, 0x1.0000000000000p+859, 0x1.2000011effff8p+312, -0x1.fffb5f5800140p+255},
{0x1.00000007fff00p-172, 0x1.fffc1ffff87f0p-228, 0x1.0000007c00000p+935, 0x1.fff0000000000p+864, 0x1.00000083fff00p+763, 0x1.77fe1835fdafap+709},
{0x1.00000007fff00p-172, 0x1.fffc1ffff87f0p-228, -0x1.0000007c00000p+935, -0x1.fff0000000000p+864, -0x1.00000083fff00p+763, -0x1.77fe1835fdafap+709},
{-0x1.00000007fff00p-172, -0x1.fffc1ffff87f0p-228, 0x1.0000007c00000p+935, 0x1.fff0000000000p+864, -0x1.00000083fff00p+763, -0x1.77fe1835fdafap+709},
{-0x1.00000007fff00p-172, -0x1.fffc1ffff87f0p-228, -0x1.0000007c00000p+935, -0x1.fff0000000000p+864, 0x1.00000083fff00p+763, 0x1.77fe1835fdafap+709},
{0x1.fffffffe00000p-318, 0x1.e000001fff800p-382, 0x1.0000000000000p+232, -0x1.f000080000000p+147, 0x1.fffffffe00000p-86, 0x1.dfffe11fff000p-150},
{0x1.fffffffe00000p-318, 0x1.e000001fff800p-382, -0x1.0000000000000p+232, 0x1.f000080000000p+147, -0x1.fffffffe00000p-86, -0x1.dfffe11fff000p-150},
{-0x1.fffffffe00000p-318, -0x1.e000001fff800p-382, 0x1.0000000000000p+232, -0x1.f000080000000p+147, -0x1.fffffffe00000p-86, -0x1.dfffe11fff000p-150},
{-0x1.fffffffe00000p-318, -0x1.e000001fff800p-382, -0x1.0000000000000p+232, 0x1.f000080000000p+147, 0x1.fffffffe00000p-86, 0x1.dfffe11fff000p-150},
{0x1.00007fffffe00p+860, 0x1.ffffff8000000p+790, 0x1.001f03f07ff00p-428, 0x1.ffffff8000000p-501, 0x1.001f840001c84p+432, -0x1.f43a8782b15e0p+377},
{0x1.00007fffffe00p+860, 0x1.ffffff8000000p+790, -0x1.001f03f07ff00p-428, -0x1.ffffff8000000p-501, -0x1.001f840001c84p+432, 0x1.f43a8782b15e0p+377},
{-0x1.00007fffffe00p+860, -0x1.ffffff8000000p+790, 0x1.001f03f07ff00p-428, 0x1.ffffff8000000p-501, -0x1.001f840001c84p+432, 0x1.f43a8782b15e0p+377},
{-0x1.00007fffffe00p+860, -0x1.ffffff8000000p+790, -0x1.001f03f07ff00p-428, -0x1.ffffff8000000p-501, 0x1.001f840001c84p+432, -0x1.f43a8782b15e0p+377},
{0x1.00003fffffffcp+10, 0x1.ffe0000700000p-63, 0x1.001f000000040p-31, 0x1.ffc0ffffc0000p-102, 0x1.001f4007c003cp-21, -0x1.aec01b80c3200p-83},
{0x1.00003fffffffcp+10, 0x1.ffe0000700000p-63, -0x1.001f000000040p-31, -0x1.ffc0ffffc0000p-102, -0x1.001f4007c003cp-21, 0x1.aec01b80c3200p-83},
{-0x1.00003fffffffcp+10, -0x1.ffe0000700000p-63, 0x1.001f000000040p-31, 0x1.ffc0ffffc0000p-102, -0x1.001f4007c003cp-21, 0x1.aec01b80c3200p-83},
{-0x1.00003fffffffcp+10, -0x1.ffe0000700000p-63, -0x1.001f000000040p-31, -0x1.ffc0ffffc0000p-102, 0x1.001f4007c003cp-21, -0x1.aec01b80c3200p-83},
{0x1.0000100000000p+594, -0x1.0000000000000p+512, 0x1.0000020000000p+926, -0x1.0000000000000p+834, INFINITY, 0x0.0000000000000p+0},
{0x1.0000100000000p+594, -0x1.0000000000000p+512, -0x1.0000020000000p+926, 0x1.0000000000000p+834, -INFINITY, 0x0.0000000000000p+0},
{-0x1.0000100000000p+594, 0x1.0000000000000p+512, 0x1.0000020000000p+926, -0x1.0000000000000p+834, -INFINITY, 0x0.0000000000000p+0},
{-0x1.0000100000000p+594, 0x1.0000000000000p+512, -0x1.0000020000000p+926, 0x1.0000000000000p+834, INFINITY, 0x0.0000000000000p+0},
{0x1.000007c000000p-87, 0x1.fff8000000000p-143, 0x1.003fff0000200p+65, -0x1.fe0003fc00040p+6, 0x1.004006c1f0184p-22, 0x1.c1aff992783b8p-78},
{0x1.000007c000000p-87, 0x1.fff8000000000p-143, -0x1.003fff0000200p+65, 0x1.fe0003fc00040p+6, -0x1.004006c1f0184p-22, -0x1.c1aff992783b8p-78},
{-0x1.000007c000000p-87, -0x1.fff8000000000p-143, 0x1.003fff0000200p+65, -0x1.fe0003fc00040p+6, -0x1.004006c1f0184p-22, -0x1.c1aff992783b8p-78},
{-0x1.000007c000000p-87, -0x1.fff8000000000p-143, -0x1.003fff0000200p+65, 0x1.fe0003fc00040p+6, 0x1.004006c1f0184p-22, 0x1.c1aff992783b8p-78},
{0x1.00001fffcffc0p+987, -0x1.fffe000000004p+932, 0x1.03fffff800000p-975, 0x1.fffffe0000000p-1043, 0x1.04002077cf3afp+12, -0x1.0396f3f70809ap-42},
{0x1.00001fffcffc0p+987, -0x1.fffe000000004p+932, -0x1.03fffff800000p-975, -0x1.fffffe0000000p-1043, -0x1.04002077cf3afp+12, 0x1.0396f3f70809ap-42},
{-0x1.00001fffcffc0p+987, 0x1.fffe000000004p+932, 0x1.03fffff800000p-975, 0x1.fffffe0000000p-1043, -0x1.04002077cf3afp+12, 0x1.0396f3f70809ap-42},
{-0x1.00001fffcffc0p+987, 0x1.fffe000000004p+932, -0x1.03fffff800000p-975, -0x1.fffffe0000000p-1043, 0x1.04002077cf3afp+12, -0x1.0396f3f70809ap-42},
{0x1.0000008000000p-118, -0x1.ffffff8200000p-175, 0x1.000fff8000000p-555, 0x1.fffffe0000000p-617, 0x1.0010000007ffcp-673, -0x1.f01ffe89f8200p-730},
{0x1.0000008000000p-118, -0x1.ffffff8200000p-175, -0x1.000fff8000000p-555, -0x1.fffffe0000000p-617, -0x1.0010000007ffcp-673, 0x1.f01ffe89f8200p-730},
{-0x1.0000008000000p-118, 0x1.ffffff8200000p-175, 0x1.000fff8000000p-555, 0x1.fffffe0000000p-617, -0x1.0010000007ffcp-673, 0x1.f01ffe89f8200p-730},
{-0x1.0000008000000p-118, 0x1.ffffff8200000p-175, -0x1.000fff8000000p-555, -0x1.fffffe0000000p-617, 0x1.0010000007ffcp-673, -0x1.f01ffe89f8200p-730},
{0x1.00000010001f8p+695, 0x1.fffc007e00000p+621, 0x1.0003fc0000000p-756, 0x1.ffff80ffffc00p-813, 0x1.0003fc10005f4p-61, 0x1.7d824090fe170p-117},
{0x1.00000010001f8p+695, 0x1.fffc007e00000p+621, -0x1.0003fc0000000p-756, -0x1.ffff80ffffc00p-813, -0x1.0003fc10005f4p-61, -0x1.7d824090fe170p-117},
{-0x1.00000010001f8p+695, -0x1.fffc007e00000p+621, 0x1.0003fc0000000p-756, 0x1.ffff80ffffc00p-813, -0x1.0003fc10005f4p-61, -0x1.7d824090fe170p-117},
{-0x1.00000010001f8p+695, -0x1.fffc007e00000p+621, -0x1.0003fc0000000p-756, -0x1.ffff80ffffc00p-813, 0x1.0003fc10005f4p-61, 0x1.7d824090fe170p-117},
{0x1.0000000000000p-985, -0x0.0000000000000p+0, 0x1.0300000001f00p-132, 0x1.fffffff800000p-208, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.0000000000000p-985, -0x0.0000000000000p+0, -0x1.0300000001f00p-132, -0x1.fffffff800000p-208, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.0000000000000p-985, 0x0.0000000000000p+0, 0x1.0300000001f00p-132, 0x1.fffffff800000p-208, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.0000000000000p-985, 0x0.0000000000000p+0, -0x1.0300000001f00p-132, -0x1.fffffff800000p-208, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.07ffffff1fe00p+389, 0x1.fc00000000000p+292, 0x1.0040000000000p+564, -0x1.c000000400000p+503, 0x1.0841ffff1fa7fp+953, 0x1.fc63fffffb100p+899},
{0x1.07ffffff1fe00p+389, 0x1.fc00000000000p+292, -0x1.0040000000000p+564, 0x1.c000000400000p+503, -0x1.0841ffff1fa7fp+953, -0x1.fc63fffffb100p+899},
{-0x1.07ffffff1fe00p+389, -0x1.fc00000000000p+292, 0x1.0040000000000p+564, -0x1.c000000400000p+503, -0x1.0841ffff1fa7fp+953, -0x1.fc63fffffb100p+899},
{-0x1.07ffffff1fe00p+389, -0x1.fc00000000000p+292, -0x1.0040000000000p+564, 0x1.c000000400000p+503, 0x1.0841ffff1fa7fp+953, 0x1.fc63fffffb100p+899},
{0x1.0080000000000p+192, -0x1.0000000000000p+87, 0x1.fffff1ff80000p+667, 0x1.ffffc000fe000p+612, 0x1.007ff8fc3fe00p+860, 0x1.007fdff07f3f4p+805},
{0x1.0080000000000p+192, -0x1.0000000000000p+87, -0x1.fffff1ff80000p+667, -0x1.ffffc000fe000p+612, -0x1.007ff8fc3fe00p+860, -0x1.007fdff07f3f4p+805},
{-0x1.0080000000000p+192, 0x1.0000000000000p+87, 0x1.fffff1ff80000p+667, 0x1.ffffc000fe000p+612, -0x1.007ff8fc3fe00p+860, -0x1.007fdff07f3f4p+805},
{-0x1.0080000000000p+192, 0x1.0000000000000p+87, -0x1.fffff1ff80000p+667, -0x1.ffffc000fe000p+612, 0x1.007ff8fc3fe00p+860, 0x1.007fdff07f3f4p+805},
{0x1.0400000000000p-301, -0x1.0000000000000p-406, 0x1.7fffe00000008p-283, -0x1.e000080000000p-363, 0x1.85ffdf8000008p-584, 0x1.fffffe187ff70p-640},
{0x1.0400000000000p-301, -0x1.0000000000000p-406, -0x1.7fffe00000008p-283, 0x1.e000080000000p-363, -0x1.85ffdf8000008p-584, -0x1.fffffe187ff70p-640},
{-0x1.0400000000000p-301, 0x1.0000000000000p-406, 0x1.7fffe00000008p-283, -0x1.e000080000000p-363, -0x1.85ffdf8000008p-584, -0x1.fffffe187ff70p-640},
{-0x1.0400000000000p-301, 0x1.0000000000000p-406, -0x1.7fffe00000008p-283, 0x1.e000080000000p-363, 0x1.85ffdf8000008p-584, 0x1.fffffe187ff70p-640},
{0x1.000001d000000p+298, -0x1.ff80000200000p+243, 0x1.0000003fffffcp+636, 0x1.feffff8000000p+556, 0x1.0000021000003p+934, 0x1.ff19f68008000p+869},
{0x1.000001d000000p+298, -0x1.ff80000200000p+243, -0x1.0000003fffffcp+636, -0x1.feffff8000000p+556, -0x1.0000021000003p+934, -0x1.ff19f68008000p+869},
{-0x1.000001d000000p+298, 0x1.ff80000200000p+243, 0x1.0000003fffffcp+636, 0x1.feffff8000000p+556, -0x1.0000021000003p+934, -0x1.ff19f68008000p+869},
{-0x1.000001d000000p+298, 0x1.ff80000200000p+243, -0x1.0000003fffffcp+636, -0x1.feffff8000000p+556, 0x1.0000021000003p+934, 0x1.ff19f68008000p+869},
{0x1.001ffffffffc0p+376, 0x1.fffffff03ff00p+315, 0x1.0000010000000p+147, -0x1.fffffff810000p+78, 0x1.002001001ffc0p+523, 0x1.fd7fc1f047e00p+462},
{0x1.001ffffffffc0p+376, 0x1.fffffff03ff00p+315, -0x1.0000010000000p+147, 0x1.fffffff810000p+78, -0x1.002001001ffc0p+523, -0x1.fd7fc1f047e00p+462},
{-0x1.001ffffffffc0p+376, -0x1.fffffff03ff00p+315, 0x1.0000010000000p+147, -0x1.fffffff810000p+78, -0x1.002001001ffc0p+523, -0x1.fd7fc1f047e00p+462},
{-0x1.001ffffffffc0p+376, -0x1.fffffff03ff00p+315, -0x1.0000010000000p+147, 0x1.fffffff810000p+78, 0x1.002001001ffc0p+523, 0x1.fd7fc1f047e00p+462},
{0x1.00000fffc0001p+666, -0x1.fc000007e0040p+607, 0x1.000007fffffc0p+408, 0x1.fc0003fc00000p+348, INFINITY, 0x0.0000000000000p+0},
{0x1.00000fffc0001p+666, -0x1.fc000007e0040p+607, -0x1.000007fffffc0p+408, -0x1.fc0003fc00000p+348, -INFINITY, 0x0.0000000000000p+0},
{-0x1.00000fffc0001p+666, 0x1.fc000007e0040p+607, 0x1.000007fffffc0p+408, 0x1.fc0003fc00000p+348, -INFINITY, 0x0.0000000000000p+0},
{-0x1.00000fffc0001p+666, 0x1.fc000007e0040p+607, -0x1.000007fffffc0p+408, -0x1.fc0003fc00000p+348, INFINITY, 0x0.0000000000000p+0},
{0x1.01ffe000003fep+235, 0x1.ffffff8000000p+169, 0x1.00fffffe00000p+680, 0x1.ffffffe7ff800p+619, 0x1.0301dfddfc406p+915, -0x1.ec00381c54000p+854},
{0x1.01ffe000003fep+235, 0x1.ffffff8000000p+169, -0x1.00fffffe00000p+680, -0x1.ffffffe7ff800p+619, -0x1.0301dfddfc406p+915, 0x1.ec00381c54000p+854},
{-0x1.01ffe000003fep+235, -0x1.ffffff8000000p+169, 0x1.00fffffe00000p+680, 0x1.ffffffe7ff800p+619, -0x1.0301dfddfc406p+915, 0x1.ec00381c54000p+854},
{-0x1.01ffe000003fep+235, -0x1.ffffff8000000p+169, -0x1.00fffffe00000p+680, -0x1.ffffffe7ff800p+619, 0x1.0301dfddfc406p+915, -0x1.ec00381c54000p+854},
{0x1.3fffffffffc00p-546, 0x1.ffffffe000000p-609, 0x1.07c0ffc200000p+282, -0x1.e000003bfffe0p+224, 0x1.49b13fb27fbe1p-264, -0x1.a3a2f827f3a30p-321},
{0x1.3fffffffffc00p-546, 0x1.ffffffe000000p-609, -0x1.07c0ffc200000p+282, 0x1.e000003bfffe0p+224, -0x1.49b13fb27fbe1p-264, 0x1.a3a2f827f3a30p-321},
{-0x1.3fffffffffc00p-546, -0x1.ffffffe000000p-609, 0x1.07c0ffc200000p+282, -0x1.e000003bfffe0p+224, -0x1.49b13fb27fbe1p-264, 0x1.a3a2f827f3a30p-321},
{-0x1.3fffffffffc00p-546, -0x1.ffffffe000000p-609, -0x1.07c0ffc200000p+282, 0x1.e000003bfffe0p+224, 0x1.49b13fb27fbe1p-264, -0x1.a3a2f827f3a30p-321},
{0x1.0000001800000p+988, 0x1.fffffc0000200p+934, 0x1.0000004000000p+36, -0x1.fff0000004000p-31, INFINITY, 0x0.0000000000000p+0},
{0x1.0000001800000p+988, 0x1.fffffc0000200p+934, -0x1.0000004000000p+36, 0x1.fff0000004000p-31, -INFINITY, 0x0.0000000000000p+0},
{-0x1.0000001800000p+988, -0x1.fffffc0000200p+934, 0x1.0000004000000p+36, -0x1.fff0000004000p-31, -INFINITY, 0x0.0000000000000p+0},
{-0x1.0000001800000p+988, -0x1.fffffc0000200p+934, -0x1.0000004000000p+36, 0x1.fff0000004000p-31, INFINITY, 0x0.0000000000000p+0},
{0x1.0000800000000p+726, -0x1.0000000000000p+663, 0x1.0fc007c004000p-506, -0x1.fffe100000000p-570, 0x1.0fc08fa007e00p+220, 0x1.fbe080e080f00p+164},
{0x1.0000800000000p+726, -0x1.0000000000000p+663, -0x1.0fc007c004000p-506, 0x1.fffe100000000p-570, -0x1.0fc08fa007e00p+220, -0x1.fbe080e080f00p+164},
{-0x1.0000800000000p+726, 0x1.0000000000000p+663, 0x1.0fc007c004000p-506, -0x1.fffe100000000p-570, -0x1.0fc08fa007e00p+220, -0x1.fbe080e080f00p+164},
{-0x1.0000800000000p+726, 0x1.0000000000000p+663, -0x1.0fc007c004000p-506, 0x1.fffe100000000p-570, 0x1.0fc08fa007e00p+220, 0x1.fbe080e080f00p+164},
{0x1.00007fffffe00p-706, 0x1.ffffffc003f00p-767, 0x1.0080000000000p-265, -0x1.0000000000000p-323, 0x1.0080803fffdffp-971, -0x1.7fc1001006c00p-1030},
{0x1.00007fffffe00p-706, 0x1.ffffffc003f00p-767, -0x1.0080000000000p-265, 0x1.0000000000000p-323, -0x1.0080803fffdffp-971, 0x1.7fc1001006c00p-1030},
{-0x1.00007fffffe00p-706, -0x1.ffffffc003f00p-767, 0x1.0080000000000p-265, -0x1.0000000000000p-323, -0x1.0080803fffdffp-971, 0x1.7fc1001006c00p-1030},
{-0x1.00007fffffe00p-706, -0x1.ffffffc003f00p-767, -0x1.0080000000000p-265, 0x1.0000000000000p-323, 0x1.0080803fffdffp-971, -0x1.7fc1001006c00p-1030},
{0x1.0000200000000p+215, -0x1.0000000000000p+159, 0x1.00000ffffff80p+645, 0x1.ffffff0007ff8p+589, 0x1.0000300001f80p+860, 0x1.fe005e000fcf0p+803},
{0x1.0000200000000p+215, -0x1.0000000000000p+159, -0x1.00000ffffff80p+645, -0x1.ffffff0007ff8p+589, -0x1.0000300001f80p+860, -0x1.fe005e000fcf0p+803},
{-0x1.0000200000000p+215, 0x1.0000000000000p+159, 0x1.00000ffffff80p+645, 0x1.ffffff0007ff8p+589, -0x1.0000300001f80p+860, -0x1.fe005e000fcf0p+803},
{-0x1.0000200000000p+215, 0x1.0000000000000p+159, -0x1.00000ffffff80p+645, -0x1.ffffff0007ff8p+589, 0x1.0000300001f80p+860, 0x1.fe005e000fcf0p+803},
{0x1.00fe000000000p-121, 0x1.001ffffff0000p-190, 0x1.00007fffff000p+370, 0x0.0000000000000p+0, 0x1.00fe807efeff0p+249, 0x1.00040082003fcp+194},
{0x1.00fe000000000p-121, 0x1.001ffffff0000p-190, -0x1.00007fffff000p+370, 0x0.0000000000000p+0, -0x1.00fe807efeff0p+249, -0x1.00040082003fcp+194},
{-0x1.00fe000000000p-121, -0x1.001ffffff0000p-190, 0x1.00007fffff000p+370, 0x0.0000000000000p+0, -0x1.00fe807efeff0p+249, -0x1.00040082003fcp+194},
{-0x1.00fe000000000p-121, -0x1.001ffffff0000p-190, -0x1.00007fffff000p+370, 0x0.0000000000000p+0, 0x1.00fe807efeff0p+249, 0x1.00040082003fcp+194},
{0x1.01e0000000002p+273, -0x1.fffffffffc200p+214, 0x1.0000080000000p-103, -0x1.0000000000000p-208, 0x1.01e0080f00002p+170, -0x1.fff80ffffc240p+111},
{0x1.01e0000000002p+273, -0x1.fffffffffc200p+214, -0x1.0000080000000p-103, 0x1.0000000000000p-208, -0x1.01e0080f00002p+170, 0x1.fff80ffffc240p+111},
{-0x1.01e0000000002p+273, 0x1.fffffffffc200p+214, 0x1.0000080000000p-103, -0x1.0000000000000p-208, -0x1.01e0080f00002p+170, 0x1.fff80ffffc240p+111},
{-0x1.01e0000000002p+273, 0x1.fffffffffc200p+214, -0x1.0000080000000p-103, 0x1.0000000000000p-208, 0x1.01e0080f00002p+170, -0x1.fff80ffffc240p+111},
{0x1.000000f1fe001p-504, -0x1.0000000000000p-587, 0x1.00003fffffff0p-718, 0x1.fc00000000000p-795, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.000000f1fe001p-504, -0x1.0000000000000p-587, -0x1.00003fffffff0p-718, -0x1.fc00000000000p-795, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.000000f1fe001p-504, 0x1.0000000000000p-587, 0x1.00003fffffff0p-718, 0x1.fc00000000000p-795, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.000000f1fe001p-504, 0x1.0000000000000p-587, -0x1.00003fffffff0p-718, -0x1.fc00000000000p-795, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.0008000000010p+198, -0x1.f8000ffff9000p+133, 0x1.00001ffff0000p+936, 0x1.ffff80001f000p+879, INFINITY, 0x0.0000000000000p+0},
{0x1.0008000000010p+198, -0x1.f8000ffff9000p+133, -0x1.00001ffff0000p+936, -0x1.ffff80001f000p+879, -INFINITY, 0x0.0000000000000p+0},
{-0x1.0008000000010p+198, 0x1.f8000ffff9000p+133, 0x1.00001ffff0000p+936, 0x1.ffff80001f000p+879, -INFINITY, 0x0.0000000000000p+0},
{-0x1.0008000000010p+198, 0x1.f8000ffff9000p+133, -0x1.00001ffff0000p+936, -0x1.ffff80001f000p+879, INFINITY, 0x0.0000000000000p+0},
{0x1.0000100000000p+641, -0x1.0000000000000p+536, 0x1.0038000800000p-759, -0x1.fffffe0000080p-819, 0x1.0038100b80008p-118, -0x1.00000efffff80p-177},
{0x1.0000100000000p+641, -0x1.0000000000000p+536, -0x1.0038000800000p-759, 0x1.fffffe0000080p-819, -0x1.0038100b80008p-118, 0x1.00000efffff80p-177},
{-0x1.0000100000000p+641, 0x1.0000000000000p+536, 0x1.0038000800000p-759, -0x1.fffffe0000080p-819, -0x1.0038100b80008p-118, 0x1.00000efffff80p-177},
{-0x1.0000100000000p+641, 0x1.0000000000000p+536, -0x1.0038000800000p-759, 0x1.fffffe0000080p-819, 0x1.0038100b80008p-118, -0x1.00000efffff80p-177},
{0x1.00003fe008000p-579, -0x1.fffdff0000080p-638, 0x1.ffffffffff800p-344, 0x1.fe00000000000p-442, 0x1.00003fe007c00p-922, -0x1.3fdf077ffec60p-980},
{0x1.00003fe008000p-579, -0x1.fffdff0000080p-638, -0x1.ffffffffff800p-344, -0x1.fe00000000000p-442, -0x1.00003fe007c00p-922, 0x1.3fdf077ffec60p-980},
{-0x1.00003fe008000p-579, 0x1.fffdff0000080p-638, 0x1.ffffffffff800p-344, 0x1.fe00000000000p-442, -0x1.00003fe007c00p-922, 0x1.3fdf077ffec60p-980},
{-0x1.00003fe008000p-579, 0x1.fffdff0000080p-638, -0x1.ffffffffff800p-344, -0x1.fe00000000000p-442, 0x1.00003fe007c00p-922, -0x1.3fdf077ffec60p-980},
{0x1.1fffffff00000p-455, 0x1.ffffff8000000p-527, 0x1.0000100000000p+435, -0x1.0000000000000p+341, 0x1.200011fefffffp-20, 0x1.00000d8000000p-91},
{0x1.1fffffff00000p-455, 0x1.ffffff8000000p-527, -0x1.0000100000000p+435, 0x1.0000000000000p+341, -0x1.200011fefffffp-20, -0x1.00000d8000000p-91},
{-0x1.1fffffff00000p-455, -0x1.ffffff8000000p-527, 0x1.0000100000000p+435, -0x1.0000000000000p+341, -0x1.200011fefffffp-20, -0x1.00000d8000000p-91},
{-0x1.1fffffff00000p-455, -0x1.ffffff8000000p-527, -0x1.0000100000000p+435, 0x1.0000000000000p+341, 0x1.200011fefffffp-20, 0x1.00000d8000000p-91},
{0x1.0ffffc0000010p+917, 0x1.f000000ff0000p+848, 0x1.00003ff002010p+155, -0x1.0000020000000p+90, INFINITY, 0x0.0000000000000p+0},
{0x1.0ffffc0000010p+917, 0x1.f000000ff0000p+848, -0x1.00003ff002010p+155, 0x1.0000020000000p+90, -INFINITY, 0x0.0000000000000p+0},
{-0x1.0ffffc0000010p+917, -0x1.f000000ff0000p+848, 0x1.00003ff002010p+155, -0x1.0000020000000p+90, -INFINITY, 0x0.0000000000000p+0},
{-0x1.0ffffc0000010p+917, -0x1.f000000ff0000p+848, -0x1.00003ff002010p+155, 0x1.0000020000000p+90, INFINITY, 0x0.0000000000000p+0},
{0x1.00ffffffc0000p+784, 0x1.fffffe00003f0p+729, 0x1.1fffffffff020p-125, -0x1.ffff000400000p-200, 0x1.211fffffb7010p+659, 0x1.9ffffeb0072d6p+605},
{0x1.00ffffffc0000p+784, 0x1.fffffe00003f0p+729, -0x1.1fffffffff020p-125, 0x1.ffff000400000p-200, -0x1.211fffffb7010p+659, -0x1.9ffffeb0072d6p+605},
{-0x1.00ffffffc0000p+784, -0x1.fffffe00003f0p+729, 0x1.1fffffffff020p-125, -0x1.ffff000400000p-200, -0x1.211fffffb7010p+659, -0x1.9ffffeb0072d6p+605},
{-0x1.00ffffffc0000p+784, -0x1.fffffe00003f0p+729, -0x1.1fffffffff020p-125, 0x1.ffff000400000p-200, 0x1.211fffffb7010p+659, 0x1.9ffffeb0072d6p+605},
{0x1.0200000000000p+902, -0x1.0000000000000p+849, 0x1.0000700000000p-969, 0x1.ffffffe000000p-1040, 0x1.020070e000000p-67, -0x1.ffffde0000102p-121},
{0x1.0200000000000p+902, -0x1.0000000000000p+849, -0x1.0000700000000p-969, -0x1.ffffffe000000p-1040, -0x1.020070e000000p-67, 0x1.ffffde0000102p-121},
{-0x1.0200000000000p+902, 0x1.0000000000000p+849, 0x1.0000700000000p-969, 0x1.ffffffe000000p-1040, -0x1.020070e000000p-67, 0x1.ffffde0000102p-121},
{-0x1.0200000000000p+902, 0x1.0000000000000p+849, -0x1.0000700000000p-969, -0x1.ffffffe000000p-1040, 0x1.020070e000000p-67, -0x1.ffffde0000102p-121},
{0x1.3fffffc0001fcp+335, 0x1.007fffff00000p+272, 0x1.007fffff80000p+3, 0x1.fffffffe00000p-65, 0x1.409fffbf401fdp+338, 0x1.14f84ffa6b400p+275},
{0x1.3fffffc0001fcp+335, 0x1.007fffff00000p+272, -0x1.007fffff80000p+3, -0x1.fffffffe00000p-65, -0x1.409fffbf401fdp+338, -0x1.14f84ffa6b400p+275},
{-0x1.3fffffc0001fcp+335, -0x1.007fffff00000p+272, 0x1.007fffff80000p+3, 0x1.fffffffe00000p-65, -0x1.409fffbf401fdp+338, -0x1.14f84ffa6b400p+275},
{-0x1.3fffffc0001fcp+335, -0x1.007fffff00000p+272, -0x1.007fffff80000p+3, -0x1.fffffffe00000p-65, 0x1.409fffbf401fdp+338, 0x1.14f84ffa6b400p+275},
{0x1.00000ffffffd0p+189, -0x1.ffffff8200000p+123, 0x1.00000c0000004p-197, -0x1.ff80000800000p-265, 0x1.00001c0000bd4p-8, -0x1.7ff00fc280000p-73},
{0x1.00000ffffffd0p+189, -0x1.ffffff8200000p+123, -0x1.00000c0000004p-197, 0x1.ff80000800000p-265, -0x1.00001c0000bd4p-8, 0x1.7ff00fc280000p-73},
{-0x1.00000ffffffd0p+189, 0x1.ffffff8200000p+123, 0x1.00000c0000004p-197, -0x1.ff80000800000p-265, -0x1.00001c0000bd4p-8, 0x1.7ff00fc280000p-73},
{-0x1.00000ffffffd0p+189, 0x1.ffffff8200000p+123, -0x1.00000c0000004p-197, 0x1.ff80000800000p-265, 0x1.00001c0000bd4p-8, -0x1.7ff00fc280000p-73},
{0x1.18003ffff8000p+281, 0x1.ffffff8000000p+215, 0x1.1ffff80000040p-329, -0x1.0000000000000p-419, 0x1.3b003f3ff5046p-48, 0x1.063fff6e58000p-106},
{0x1.18003ffff8000p+281, 0x1.ffffff8000000p+215, -0x1.1ffff80000040p-329, 0x1.0000000000000p-419, -0x1.3b003f3ff5046p-48, -0x1.063fff6e58000p-106},
{-0x1.18003ffff8000p+281, -0x1.ffffff8000000p+215, 0x1.1ffff80000040p-329, -0x1.0000000000000p-419, -0x1.3b003f3ff5046p-48, -0x1.063fff6e58000p-106},
{-0x1.18003ffff8000p+281, -0x1.ffffff8000000p+215, -0x1.1ffff80000040p-329, 0x1.0000000000000p-419, 0x1.3b003f3ff5046p-48, 0x1.063fff6e58000p-106},
{0x1.0ffffff9ffff0p+735, 0x1.f800008200000p+665, 0x1.000000ffffff8p+876, 0x1.ff0000007ffc0p+817, INFINITY, 0x0.0000000000000p+0},
{0x1.0ffffff9ffff0p+735, 0x1.f800008200000p+665, -0x1.000000ffffff8p+876, -0x1.ff0000007ffc0p+817, -INFINITY, 0x0.0000000000000p+0},
{-0x1.0ffffff9ffff0p+735, -0x1.f800008200000p+665, 0x1.000000ffffff8p+876, 0x1.ff0000007ffc0p+817, -INFINITY, 0x0.0000000000000p+0},
{-0x1.0ffffff9ffff0p+735, -0x1.f800008200000p+665, -0x1.000000ffffff8p+876, -0x1.ff0000007ffc0p+817, INFINITY, 0x0.0000000000000p+0},
{0x1.1fffffe000000p-807, 0x1.ffff800001ffep-861, 0x1.0000008000000p+287, -0x1.fffffc0000008p+231, 0x1.2000006ffffffp-520, 0x1.6fff823001bfap-574},
{0x1.1fffffe000000p-807, 0x1.ffff800001ffep-861, -0x1.0000008000000p+287, 0x1.fffffc0000008p+231, -0x1.2000006ffffffp-520, -0x1.6fff823001bfap-574},
{-0x1.1fffffe000000p-807, -0x1.ffff800001ffep-861, 0x1.0000008000000p+287, -0x1.fffffc0000008p+231, -0x1.2000006ffffffp-520, -0x1.6fff823001bfap-574},
{-0x1.1fffffe000000p-807, -0x1.ffff800001ffep-861, -0x1.0000008000000p+287, 0x1.fffffc0000008p+231, 0x1.2000006ffffffp-520, 0x1.6fff823001bfap-574},
{0x1.0000200000000p-310, -0x1.fffffff804000p-377, 0x1.0000080000000p+614, -0x1.0000000000000p+519, 0x1.0000280001000p+304, -0x1.0000080402000p+238},
{0x1.0000200000000p-310, -0x1.fffffff804000p-377, -0x1.0000080000000p+614, 0x1.0000000000000p+519, -0x1.0000280001000p+304, 0x1.0000080402000p+238},
{-0x1.0000200000000p-310, 0x1.fffffff804000p-377, 0x1.0000080000000p+614, -0x1.0000000000000p+519, -0x1.0000280001000p+304, 0x1.0000080402000p+238},
{-0x1.0000200000000p-310, 0x1.fffffff804000p-377, -0x1.0000080000000p+614, 0x1.0000000000000p+519, 0x1.0000280001000p+304, -0x1.0000080402000p+238},
{0x1.03ffff0000010p-506, -0x1.fffc010000000p-582, 0x1.0000020000000p+657, -0x1.0000000000000p+552, 0x1.04000107ffff0p+151, 0x1.e0003faf80000p+79},
{0x1.03ffff0000010p-506, -0x1.fffc010000000p-582, -0x1.0000020000000p+657, 0x1.0000000000000p+552, -0x1.04000107ffff0p+151, -0x1.e0003faf80000p+79},
{-0x1.03ffff0000010p-506, 0x1.fffc010000000p-582, 0x1.0000020000000p+657, -0x1.0000000000000p+552, -0x1.04000107ffff0p+151, -0x1.e0003faf80000p+79},
{-0x1.03ffff0000010p-506, 0x1.fffc010000000p-582, -0x1.0000020000000p+657, 0x1.0000000000000p+552, 0x1.04000107ffff0p+151, 0x1.e0003faf80000p+79},
{0x1.7ffffffc00000p-455, 0x1.fff80000fc000p-516, 0x1.0010000000000p+908, -0x1.ffffffffc4000p+841, 0x1.8017fffbffc00p+453, 0x1.f417ff811d700p+392},
{0x1.7ffffffc00000p-455, 0x1.fff80000fc000p-516, -0x1.0010000000000p+908, 0x1.ffffffffc4000p+841, -0x1.8017fffbffc00p+453, -0x1.f417ff811d700p+392},
{-0x1.7ffffffc00000p-455, -0x1.fff80000fc000p-516, 0x1.0010000000000p+908, -0x1.ffffffffc4000p+841, -0x1.8017fffbffc00p+453, -0x1.f417ff811d700p+392},
{-0x1.7ffffffc00000p-455, -0x1.fff80000fc000p-516, -0x1.0010000000000p+908, 0x1.ffffffffc4000p+841, 0x1.8017fffbffc00p+453, 0x1.f417ff811d700p+392},
{0x1.000007fffc004p-48, -0x1.8000004000000p-107, 0x1.7fe000001ff00p+482, 0x1.f000000000000p+381, 0x1.7fe00bff19f0ep+434, 0x1.5f0b0077ee250p+377},
{0x1.000007fffc004p-48, -0x1.8000004000000p-107, -0x1.7fe000001ff00p+482, -0x1.f000000000000p+381, -0x1.7fe00bff19f0ep+434, -0x1.5f0b0077ee250p+377},
{-0x1.000007fffc004p-48, 0x1.8000004000000p-107, 0x1.7fe000001ff00p+482, 0x1.f000000000000p+381, -0x1.7fe00bff19f0ep+434, -0x1.5f0b0077ee250p+377},
{-0x1.000007fffc004p-48, 0x1.8000004000000p-107, -0x1.7fe000001ff00p+482, -0x1.f000000000000p+381, 0x1.7fe00bff19f0ep+434, 0x1.5f0b0077ee250p+377},
{0x1.01ffffffffc00p-235, 0x0.0000000000000p+0, 0x1.0ffffff000000p-5, 0x1.fffffff000000p-75, 0x1.121fffefdfbc0p-240, 0x1.81fffff7f0000p-309},
{0x1.01ffffffffc00p-235, 0x0.0000000000000p+0, -0x1.0ffffff000000p-5, -0x1.fffffff000000p-75, -0x1.121fffefdfbc0p-240, -0x1.81fffff7f0000p-309},
{-0x1.01ffffffffc00p-235, 0x0.0000000000000p+0, 0x1.0ffffff000000p-5, 0x1.fffffff000000p-75, -0x1.121fffefdfbc0p-240, -0x1.81fffff7f0000p-309},
{-0x1.01ffffffffc00p-235, 0x0.0000000000000p+0, -0x1.0ffffff000000p-5, -0x1.fffffff000000p-75, 0x1.121fffefdfbc0p-240, 0x1.81fffff7f0000p-309},
{0x1.0000010000000p-572, -0x1.ffffffffe4000p-634, 0x1.0000020000000p-195, -0x1.ff80000100000p-268, 0x1.0000030000020p-767, -0x1.001ffa0012000p-828},
{0x1.0000010000000p-572, -0x1.ffffffffe4000p-634, -0x1.0000020000000p-195, 0x1.ff80000100000p-268, -0x1.0000030000020p-767, 0x1.001ffa0012000p-828},
{-0x1.0000010000000p-572, 0x1.ffffffffe4000p-634, 0x1.0000020000000p-195, -0x1.ff80000100000p-268, -0x1.0000030000020p-767, 0x1.001ffa0012000p-828},
{-0x1.0000010000000p-572, 0x1.ffffffffe4000p-634, -0x1.0000020000000p-195, 0x1.ff80000100000p-268, 0x1.0000030000020p-767, -0x1.001ffa0012000p-828},
{0x1.0008000000000p+139, -0x1.0000000000000p+34, 0x1.000fffff80000p+540, 0x1.fffffffc00000p+484, 0x1.0018007f7ffc0p+679, 0x1.0007fffdffefcp+624},
{0x1.0008000000000p+139, -0x1.0000000000000p+34, -0x1.000fffff80000p+540, -0x1.fffffffc00000p+484, -0x1.0018007f7ffc0p+679, -0x1.0007fffdffefcp+624},
{-0x1.0008000000000p+139, 0x1.0000000000000p+34, 0x1.000fffff80000p+540, 0x1.fffffffc00000p+484, -0x1.0018007f7ffc0p+679, -0x1.0007fffdffefcp+624},
{-0x1.0008000000000p+139, 0x1.0000000000000p+34, -0x1.000fffff80000p+540, -0x1.fffffffc00000p+484, 0x1.0018007f7ffc0p+679, 0x1.0007fffdffefcp+624},
{0x1.007fffffffc00p+663, 0x1.ffffc00000000p+598, 0x1.1ffffffff8000p-133, 0x1.fe001ffffffe0p-189, 0x1.208ffffff7b40p+530, 0x1.000f90f5ffbb4p+475},
{0x1.007fffffffc00p+663, 0x1.ffffc00000000p+598, -0x1.1ffffffff8000p-133, -0x1.fe001ffffffe0p-189, -0x1.208ffffff7b40p+530, -0x1.000f90f5ffbb4p+475},
{-0x1.007fffffffc00p+663, -0x1.ffffc00000000p+598, 0x1.1ffffffff8000p-133, 0x1.fe001ffffffe0p-189, -0x1.208ffffff7b40p+530, -0x1.000f90f5ffbb4p+475},
{-0x1.007fffffffc00p+663, -0x1.ffffc00000000p+598, -0x1.1ffffffff8000p-133, -0x1.fe001ffffffe0p-189, 0x1.208ffffff7b40p+530, 0x1.000f90f5ffbb4p+475},
{0x1.0000000fffe00p-673, 0x1.3fffff87fc000p-733, 0x1.00003ffffc000p-1010, 0x1.0000000000000p-1074, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.0000000fffe00p-673, 0x1.3fffff87fc000p-733, -0x1.00003ffffc000p-1010, -0x1.0000000000000p-1074, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.0000000fffe00p-673, -0x1.3fffff87fc000p-733, 0x1.00003ffffc000p-1010, 0x1.0000000000000p-1074, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.0000000fffe00p-673, -0x1.3fffff87fc000p-733, -0x1.00003ffffc000p-1010, -0x1.0000000000000p-1074, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.0000000800000p+490, -0x1.0000000000000p+385, 0x1.7ffff80000000p+812, 0x1.ffffffffe0000p+756, INFINITY, 0x0.0000000000000p+0},
{0x1.0000000800000p+490, -0x1.0000000000000p+385, -0x1.7ffff80000000p+812, -0x1.ffffffffe0000p+756, -INFINITY, 0x0.0000000000000p+0},
{-0x1.0000000800000p+490, 0x1.0000000000000p+385, 0x1.7ffff80000000p+812, 0x1.ffffffffe0000p+756, -INFINITY, 0x0.0000000000000p+0},
{-0x1.0000000800000p+490, 0x1.0000000000000p+385, -0x1.7ffff80000000p+812, -0x1.ffffffffe0000p+756, INFINITY, 0x0.0000000000000p+0},
{0x1.00fffffff8000p-776, 0x1.fffe000000000p-850, 0x1.0000100000000p+556, -0x1.fffffffffc100p+501, 0x1.0100100ff8000p-220, -0x1.20ffe00014060p-274},
{0x1.00fffffff8000p-776, 0x1.fffe000000000p-850, -0x1.0000100000000p+556, 0x1.fffffffffc100p+501, -0x1.0100100ff8000p-220, 0x1.20ffe00014060p-274},
{-0x1.00fffffff8000p-776, -0x1.fffe000000000p-850, 0x1.0000100000000p+556, -0x1.fffffffffc100p+501, -0x1.0100100ff8000p-220, 0x1.20ffe00014060p-274},
{-0x1.00fffffff8000p-776, -0x1.fffe000000000p-850, -0x1.0000100000000p+556, 0x1.fffffffffc100p+501, 0x1.0100100ff8000p-220, -0x1.20ffe00014060p-274},
{0x1.0007ffffffff0p+441, 0x1.fffffc0000000p+358, 0x1.0003800000080p+407, -0x1.0000000000000p+302, 0x1.000b801c00070p+848, 0x1.e40001ffc6f80p+789},
{0x1.0007ffffffff0p+441, 0x1.fffffc0000000p+358, -0x1.0003800000080p+407, 0x1.0000000000000p+302, -0x1.000b801c00070p+848, -0x1.e40001ffc6f80p+789},
{-0x1.0007ffffffff0p+441, -0x1.fffffc0000000p+358, 0x1.0003800000080p+407, -0x1.0000000000000p+302, -0x1.000b801c00070p+848, -0x1.e40001ffc6f80p+789},
{-0x1.0007ffffffff0p+441, -0x1.fffffc0000000p+358, -0x1.0003800000080p+407, 0x1.0000000000000p+302, 0x1.000b801c00070p+848, 0x1.e40001ffc6f80p+789},
{0x1.000001fffff80p-397, 0x1.ff00000000000p-481, 0x1.0000020000000p+10, -0x1.8002000000000p-66, 0x1.000003fffffc0p-387, -0x1.017e030300000p-455},
{0x1.000001fffff80p-397, 0x1.ff00000000000p-481, -0x1.0000020000000p+10, 0x1.8002000000000p-66, -0x1.000003fffffc0p-387, 0x1.017e030300000p-455},
{-0x1.000001fffff80p-397, -0x1.ff00000000000p-481, 0x1.0000020000000p+10, -0x1.8002000000000p-66, -0x1.000003fffffc0p-387, 0x1.017e030300000p-455},
{-0x1.000001fffff80p-397, -0x1.ff00000000000p-481, -0x1.0000020000000p+10, 0x1.8002000000000p-66, 0x1.000003fffffc0p-387, -0x1.017e030300000p-455},
{0x1.0003fffff0000p+165, 0x1.ffff800007f80p+105, 0x1.000007ffffe00p+981, -0x1.fe0c000004000p+923, INFINITY, 0x0.0000000000000p+0},
{0x1.0003fffff0000p+165, 0x1.ffff800007f80p+105, -0x1.000007ffffe00p+981, 0x1.fe0c000004000p+923, -INFINITY, 0x0.0000000000000p+0},
{-0x1.0003fffff0000p+165, -0x1.ffff800007f80p+105, 0x1.000007ffffe00p+981, -0x1.fe0c000004000p+923, -INFINITY, 0x0.0000000000000p+0},
{-0x1.0003fffff0000p+165, -0x1.ffff800007f80p+105, -0x1.000007ffffe00p+981, 0x1.fe0c000004000p+923, INFINITY, 0x0.0000000000000p+0},
{0x1.00000003f0000p+687, 0x1.ffffffe000000p+609, 0x1.0100000000000p-215, -0x1.fffffff000080p-275, 0x1.01000003f3f00p+472, -0x1.ffff7f77e0100p+412},
{0x1.00000003f0000p+687, 0x1.ffffffe000000p+609, -0x1.0100000000000p-215, 0x1.fffffff000080p-275, -0x1.01000003f3f00p+472, 0x1.ffff7f77e0100p+412},
{-0x1.00000003f0000p+687, -0x1.ffffffe000000p+609, 0x1.0100000000000p-215, -0x1.fffffff000080p-275, -0x1.01000003f3f00p+472, 0x1.ffff7f77e0100p+412},
{-0x1.00000003f0000p+687, -0x1.ffffffe000000p+609, -0x1.0100000000000p-215, 0x1.fffffff000080p-275, 0x1.01000003f3f00p+472, -0x1.ffff7f77e0100p+412},
{0x1.007fffffffc00p-448, 0x1.fffffe0000000p-530, 0x1.0000003fffff8p-643, 0x1.fffff80000000p-714, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.007fffffffc00p-448, 0x1.fffffe0000000p-530, -0x1.0000003fffff8p-643, -0x1.fffff80000000p-714, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.007fffffffc00p-448, -0x1.fffffe0000000p-530, 0x1.0000003fffff8p-643, 0x1.fffff80000000p-714, -0x0.0000000000000p+0, -0x0.0000000000000p+0},
{-0x1.007fffffffc00p-448, -0x1.fffffe0000000p-530, -0x1.0000003fffff8p-643, -0x1.fffff80000000p-714, 0x0.0000000000000p+0, 0x0.0000000000000p+0},
{0x1.00000003ff000p-711, 0x1.ffffff0000000p-773, 0x1.0000ffff80000p+16, 0x1.e0000000fffc0p-43, 0x1.000100037f040p-695, -0x1.7bfef005e8810p-752},
{0x1.00000003ff000p-711, 0x1.ffffff0000000p-773, -0x1.0000ffff80000p+16, -0x1.e0000000fffc0p-43, -0x1.000100037f040p-695, 0x1.7bfef005e8810p-752},
{-0x1.00000003ff000p-711, -0x1.ffffff0000000p-773, 0x1.0000ffff80000p+16, 0x1.e0000000fffc0p-43, -0x1.000100037f040p-695, 0x1.7bfef005e8810p-752},
{-0x1.00000003ff000p-711, -0x1.ffffff0000000p-773, -0x1.0000ffff80000p+16, -0x1.e0000000fffc0p-43, 0x1.000100037f040p-695, -0x1.7bfef005e8810p-752},
{0x1.007ffffffc000p-157, 0x1.ffffff8000000p-224, 0x1.0003fffffffe0p-88, 0x1.fffffff000780p-148, 0x1.008401fffbfdfp-245, -0x1.bf5ffdf221f10p-302},
{0x1.007ffffffc000p-157, 0x1.ffffff8000000p-224, -0x1.0003fffffffe0p-88, -0x1.fffffff000780p-148, -0x1.008401fffbfdfp-245, 0x1.bf5ffdf221f10p-302},
{-0x1.007ffffffc000p-157, -0x1.ffffff8000000p-224, 0x1.0003fffffffe0p-88, 0x1.fffffff000780p-148, -0x1.008401fffbfdfp-245, 0x1.bf5ffdf221f10p-302},
{-0x1.007ffffffc000p-157, -0x1.ffffff8000000p-224, -0x1.0003fffffffe0p-88, -0x1.fffffff000780p-148, 0x1.008401fffbfdfp-245, -0x1.bf5ffdf221f10p-302},
{0x1.07fffffe0003cp-432, 0x1.ffe000007f000p-497, 0x1.1ffff00800000p+307, -0x1.0000000000000p+210, 0x1.28ffef8600045p-125, 0x1.fc3902fc0230cp-179},
{0x1.07fffffe0003cp-432, 0x1.ffe000007f000p-497, -0x1.1ffff00800000p+307, 0x1.0000000000000p+210, -0x1.28ffef8600045p-125, -0x1.fc3902fc0230cp-179},
{-0x1.07fffffe0003cp-432, -0x1.ffe000007f000p-497, 0x1.1ffff00800000p+307, -0x1.0000000000000p+210, -0x1.28ffef8600045p-125, -0x1.fc3902fc0230cp-179},
{-0x1.07fffffe0003cp-432, -0x1.ffe000007f000p-497, -0x1.1ffff00800000p+307, 0x1.0000000000000p+210, 0x1.28ffef8600045p-125, 0x1.fc3902fc0230cp-179},
{0x1.000003fff8000p+711, 0x1.ffff00000003cp+656, 0x1.000001ffff020p+18, -0x1.fff0000200000p-52, 0x1.000005fff70a0p+729, 0x1.f5ff0817ec048p+674},
{0x1.000003fff8000p+711, 0x1.ffff00000003cp+656, -0x1.000001ffff020p+18, 0x1.fff0000200000p-52, -0x1.000005fff70a0p+729, -0x1.f5ff0817ec048p+674},
{-0x1.000003fff8000p+711, -0x1.ffff00000003cp+656, 0x1.000001ffff020p+18, -0x1.fff0000200000p-52, -0x1.000005fff70a0p+729, -0x1.f5ff0817ec048p+674},
{-0x1.000003fff8000p+711, -0x1.ffff00000003cp+656, -0x1.000001ffff020p+18, 0x1.fff0000200000p-52, 0x1.000005fff70a0p+729, 0x1.f5ff0817ec048p+674},
{0x1.00ffe00000000p-302, 0x1.fe00000000000p-362, 0x1.0000100000000p+87, -0x1.ffff800010000p+18, 0x1.00fff00ffe000p-215, 0x1.fcff20403ff00p-275},
{0x1.00ffe00000000p-302, 0x1.fe00000000000p-362, -0x1.0000100000000p+87, 0x1.ffff800010000p+18, -0x1.00fff00ffe000p-215, -0x1.fcff20403ff00p-275},
{-0x1.00ffe00000000p-302, -0x1.fe00000000000p-362, 0x1.0000100000000p+87, -0x1.ffff800010000p+18, -0x1.00fff00ffe000p-215, -0x1.fcff20403ff00p-275},
{-0x1.00ffe00000000p-302, -0x1.fe00000000000p-362, -0x1.0000100000000p+87, 0x1.ffff800010000p+18, 0x1.00fff00ffe000p-215, 0x1.fcff20403ff00p-275},
{0x1.fffffff000008p+565, -0x1.ffffff1000000p+488, 0x1.0001fffffff80p-162, 0x1.fffe000000000p-247, 0x1.0001fff7ffe84p+404, 0x1.00701fdbe7000p+339},
{0x1.fffffff000008p+565, -0x1.ffffff1000000p+488, -0x1.0001fffffff80p-162, -0x1.fffe000000000p-247, -0x1.0001fff7ffe84p+404, -0x1.00701fdbe7000p+339},
{-0x1.fffffff000008p+565, 0x1.ffffff1000000p+488, 0x1.0001fffffff80p-162, 0x1.fffe000000000p-247, -0x1.0001fff7ffe84p+404, -0x1.00701fdbe7000p+339},
{-0x1.fffffff000008p+565, 0x1.ffffff1000000p+488, -0x1.0001fffffff80p-162, -0x1.fffe000000000p-247, 0x1.0001fff7ffe84p+404, 0x1.00701fdbe7000p+339},
};

const int numAccuracyTests = sizeof(accuracyTests) / sizeof(struct testVector);

#include <math.h>

double testAccuracy() {
	int i;
	DD a, b, c, r;

	double worstUlps = 0.5;

	for (i=0; i<numAccuracyTests; ++i) {
		a.hi = accuracyTests[i].xhi;
		a.lo = accuracyTests[i].xlo;
		b.hi = accuracyTests[i].yhi;
		b.lo = accuracyTests[i].ylo;
		r.hi = accuracyTests[i].rhi;
		r.lo = accuracyTests[i].rlo;

		DD c = { .ld = __gcc_qmul(a.ld, b.ld) };

		double error = __builtin_fabs(((r.hi - c.hi) + r.lo) - c.lo);

		if (error != 0.0) {

			int exponent = ilogb(r.hi);
			exponent = (exponent < -1022 ? -1022 : exponent);
			double ulpError = scalbn(error, 106 - exponent);

			if (ulpError > worstUlps) {
#ifdef PRINT_ACCURACY_INFORMATION
				printf("New worst rounding error for (%a,%a) x (%a,%a):\n", a.hi, a.lo, b.hi, b.lo);
				printf("\tExpected: (%a,%a)\n", r.hi, r.lo);
				printf("\tComputed: (%a,%a)\n", c.hi, c.lo);
				printf("\tError: %f ulps\n", ulpError);
#endif
				worstUlps = ulpError;
			}
		}
	}

	return worstUlps;
}
























