// SPDX-License-Identifier: (BSD-3-Clause OR GPL-2.0-only)
/* Copyright(c) 2020 - 2021 Intel Corporation */

#include <adf_accel_devices.h>
#include <adf_common_drv.h>
#include <adf_cfg.h>
#include "adf_c4xxx_aram.h"
#include "adf_c4xxx_hw_data.h"
#include "adf_c4xxx_accel_units.h"
#include "adf_c4xxx_inline.h"

int adf_init_aram_config_c4xxx(struct adf_accel_dev *accel_dev)
{
	u32 aram_size = ADF_C4XXX_2MB_ARAM_SIZE;
	u32 ibuff_mem_needed = 0;
	u32 usable_aram_size = 0;
	struct adf_hw_aram_info *aram_info;
	u32 sa_db_ctl_value = 0;
	void __iomem *aram_csr_base;
	u32 i;
	unsigned long ipsec_algo_group = IPSEC_DEFAUL_ALGO_GROUP;

	if (accel_dev->au_info->num_inline_au > 0)
		if (adf_get_inline_ipsec_algo_group(accel_dev,
						    &ipsec_algo_group))
			return -EFAULT;
	/* Allocate memory for adf_hw_aram_info */
	accel_dev->aram_info = kzalloc(sizeof(*accel_dev->aram_info), GFP_DMA);
	if (!accel_dev->aram_info)
		return -ENOMEM;
	aram_info = accel_dev->aram_info;

	/* Initialise Inline direction */
	aram_info->inline_direction_egress_mask = 0;
	if (accel_dev->au_info->num_inline_au) {
		u8 profile = 0;

		/* Set inline direction bitmap in the ARAM to
		 * inform firmware which ME is egress
		 */
		aram_info->inline_direction_egress_mask =
			accel_dev->au_info->inline_egress_msk;

		if (get_congestion_management_profile(accel_dev, &profile)) {
			kfree(accel_dev->aram_info);
			accel_dev->aram_info = NULL;
			return -EFAULT;
		}
		/* User profile is valid, we can now add it
		 * in the ARAM partition table
		 */
		aram_info->inline_congest_mngt_profile = profile;
	}

	/* Initialise DC ME mask, "1" = ME is used for DC operations */
	aram_info->dc_ae_mask = accel_dev->au_info->dc_ae_msk;

	/* Initialise CY ME mask, "1" = ME is used for CY operations
	 * Since asym service can also be enabled on inline AEs, here
	 * we use the sym ae mask for configuring the cy_ae_msk
	 * and additionally asym one only when not in inline.
	 */
	aram_info->cy_ae_mask = accel_dev->au_info->sym_ae_msk;
	if (!accel_dev->au_info->num_inline_au)
		aram_info->cy_ae_mask |= accel_dev->au_info->asym_ae_msk;

	/* Configure number of long words in the ARAM */
	aram_info->num_aram_lw_entries = ADF_C4XXX_NUM_ARAM_ENTRIES;

	/* Reset region offset values to 0xffffffff */
	aram_info->mmp_region_offset = ~aram_info->mmp_region_offset;
	aram_info->skm_region_offset = ~aram_info->skm_region_offset;
	aram_info->inter_buff_aram_region_offset =
		 ~aram_info->inter_buff_aram_region_offset;

	/* Determine ARAM size */
	aram_csr_base = (&GET_BARS(accel_dev)[ADF_C4XXX_SRAM_BAR])->virt_addr;
	sa_db_ctl_value = ADF_CSR_RD(aram_csr_base, ADF_C4XXX_REG_SA_DB_CTRL);

	aram_size = (sa_db_ctl_value & ADF_C4XXX_SADB_SIZE_BIT)
		? ADF_C4XXX_2MB_ARAM_SIZE : ADF_C4XXX_4MB_ARAM_SIZE;
	dev_info(&GET_DEV(accel_dev),
		 "Total available accelerator memory: %uMB\n",
		 aram_size / ADF_C4XXX_1MB_SIZE);
	sa_db_ctl_value &= ADF_C4XXX_SADB_SIZE_BIT;

	/* Compute MMP region offset */
	aram_info->mmp_region_size = ADF_C4XXX_DEFAULT_MMP_REGION_SIZE;
	aram_info->mmp_region_offset = aram_size - aram_info->mmp_region_size;

	if (accel_dev->au_info->num_cy_au ||
	    accel_dev->au_info->num_inline_au) {
		/* Crypto is available therefore we must
		 * include space in the ARAM for SKM.
		 */
		aram_info->skm_region_size = ADF_C4XXX_DEFAULT_SKM_REGION_SIZE;
		/* Compute SKM region offset */
		aram_info->skm_region_offset = aram_size -
			(aram_info->mmp_region_size +
					aram_info->skm_region_size);
	}

	/* SADB always start at offset 0. */
	if (accel_dev->au_info->num_inline_au) {
		/* Inline is available therefore we must
		 * use remaining ARAM for the SADB.
		 */
		u32 sadb_size = aram_size -
		(aram_info->mmp_region_size + aram_info->skm_region_size);

		/*
		 * When the inline service is enabled, the policy is that
		 * compression gives up it's space in ARAM to allow for a
		 * larger SADB. Compression must use DRAM instead of ARAM.
		 */
		aram_info->inter_buff_aram_region_size = 0;

		/* the SADB size must be an integral multiple of the SA size */
		if (ipsec_algo_group == IPSEC_DEFAUL_ALGO_GROUP) {
			sadb_size = sadb_size - (sadb_size %
			ADF_C4XXX_SA_SIZE_IN_BYTES(ADF_C4XXX_DEFAULT_SA_SIZE));
		} else {
			/* IPSEC_ALGO_GROUP1
			 * Total 2 algo groups.
			 */
			sadb_size = sadb_size - (sadb_size %
			ADF_C4XXX_SA_SIZE_IN_BYTES(ADF_C4XXX_ALGO_GROUP1_SA_SIZE
			));
		}

		aram_info->sadb_region_size = sadb_size;
		sa_db_ctl_value = ADF_C4XXX_SADB_REG_VALUE(accel_dev);
	}

	/* REG_SA_DB_CTRL register initialisation */
	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_REG_SA_DB_CTRL, sa_db_ctl_value);

	/*
	 * REG_SA_CTRL_LOCK register initialisation. We set the lock
	 * bit in order to prevent the REG_SA_DB_CTRL to be
	 * overwritten
	 */
	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_REG_SA_CTRL_LOCK,
		   ADF_C4XXX_DEFAULT_SA_CTRL_LOCKOUT);

	if (accel_dev->au_info->num_dc_au &&
	    !accel_dev->au_info->num_inline_au) {
		/* Compression is available therefore we must see if there is
		 * space in the ARAM for intermediate buffers.
		 */
		aram_info->inter_buff_aram_region_size = 0;
		usable_aram_size = aram_size -
				(aram_info->mmp_region_size
						+ aram_info->skm_region_size);

		for (i = 1; i <= accel_dev->au_info->num_dc_au; i++) {
			if ((i * ADF_C4XXX_AU_COMPR_INTERM_SIZE) >
				usable_aram_size)
				break;

			ibuff_mem_needed = i * ADF_C4XXX_AU_COMPR_INTERM_SIZE;
		}

		/* Set remaining ARAM to intermediate buffers. Firmware handles
		 * fallback to DRAM for cases were number of AU assigned
		 * to compression exceeds available ARAM memory.
		 */
		aram_info->inter_buff_aram_region_size = ibuff_mem_needed;

		/* If ARAM is used for compression set its initial offset. */
		if (aram_info->inter_buff_aram_region_size)
			aram_info->inter_buff_aram_region_offset = 0;
	}

	return 0;
}

void adf_exit_aram_config_c4xxx(struct adf_accel_dev *accel_dev)
{
	kfree(accel_dev->aram_info);
	accel_dev->aram_info = NULL;
}
