// Copyright 2020 The TensorFlow Runtime Authors
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

// Unit test for cuFFT wrapper.
#include "tfrt/gpu/wrapper/fft_wrapper.h"

#include <cmath>

#include "common.h"
#include "gmock/gmock.h"
#include "gtest/gtest.h"
#include "tfrt/gpu/wrapper/cuda_wrapper.h"
#include "tfrt/gpu/wrapper/cufft_wrapper.h"
#include "tfrt/gpu/wrapper/hipfft_wrapper.h"

namespace tfrt {
namespace gpu {
namespace wrapper {
using ::testing::FloatNear;

TEST_P(Test, RealToComplex1D) {
  auto platform = GetParam();
  ASSERT_THAT(Init(platform), IsSuccess());
  TFRT_ASSERT_AND_ASSIGN(auto count, DeviceGetCount(platform));
  ASSERT_GT(count, 0);
  TFRT_ASSERT_AND_ASSIGN(auto device, DeviceGet(platform, 0));
  TFRT_ASSERT_AND_ASSIGN(auto context, CtxCreate(device));
  TFRT_ASSERT_AND_ASSIGN(auto current, CtxGetCurrent());
  TFRT_ASSERT_AND_ASSIGN(auto stream, StreamCreateNonBlocking(current));

  constexpr size_t kWindowSize = 256;
  constexpr size_t kWindowSizeBytesInput = kWindowSize * sizeof(cufftReal);
  constexpr size_t kWindowSizeBytesOutput = kWindowSize * sizeof(cufftComplex);

  // Prepare FFT plan.
  FftType type(CUFFT_R2C, platform);
  FftDirection direction(CUFFT_FORWARD, platform);

  TFRT_ASSERT_AND_ASSIGN(OwningFftHandle handle, FftCreate(current));
  EXPECT_THAT(
      FftMakePlanMany(handle.get(), type, /*batch=*/1, kWindowSize).takeError(),
      IsSuccess());
  EXPECT_THAT(FftSetStream(handle.get(), stream.get()), IsSuccess());

  // Allocate enough for reuse as output.
  MemHostAllocFlags alloc_flags(CU_MEMHOSTALLOC_DEVICEMAP, platform);
  TFRT_ASSERT_AND_ASSIGN(
      auto host_data,
      MemHostAlloc(current, kWindowSizeBytesOutput, alloc_flags));

  const float kPi = acosf(-1.0f);
  auto* input = static_cast<cufftReal*>(host_data.get().raw());
  for (size_t i = 0; i < kWindowSize; ++i)
    input[i] = sinf(2 * kPi * i / kWindowSize);

  // Allocate enough memory for output to be written in place.
  TFRT_ASSERT_AND_ASSIGN(auto device_data,
                         MemAlloc(current, kWindowSizeBytesOutput));

  // Copy data and do transform.
  EXPECT_THAT(MemcpyAsync(current, device_data.get(), host_data.get(),
                          kWindowSizeBytesInput, stream.get()),
              IsSuccess());

  EXPECT_THAT(FftExec(current, handle.get(), device_data.get(),
                      device_data.get(), type, direction),
              IsSuccess());
  EXPECT_THAT(MemcpyAsync(current, host_data.get(), device_data.get(),
                          kWindowSizeBytesOutput, stream.get()),
              IsSuccess());

  EXPECT_THAT(StreamSynchronize(stream.get()), IsSuccess());

  auto* output = static_cast<cufftComplex*>(host_data.get().raw());
  for (size_t i = 0; i < kWindowSize; ++i) {
    EXPECT_THAT(output[i].x, FloatNear(0.0f, 0.1f));
    float y = i == 1 ? -0.5f * kWindowSize : 0.0f;
    EXPECT_THAT(output[i].y, FloatNear(y, 0.1f));
  }
}

}  // namespace wrapper
}  // namespace gpu
}  // namespace tfrt
