#
# Handling of ifcfg files
#
# Copyright (C) 2019 Red Hat, Inc.
#
# This copyrighted material is made available to anyone wishing to use,
# modify, copy, or redistribute it subject to the terms and conditions of
# the GNU General Public License v.2, or (at your option) any later version.
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY expressed or implied, including the implied warranties of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.  You should have received a copy of the
# GNU General Public License along with this program; if not, write to the
# Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
# 02110-1301, USA.  Any Red Hat trademarks that are incorporated in the
# source code or documentation are not subject to the GNU General Public
# License and may only be used or replicated with the express permission of
# Red Hat, Inc.
#

import os

# TODO move to anaconda.core
from pyanaconda.simpleconfig import SimpleConfigFile
from pyanaconda.core import util
from pyanaconda.modules.network.constants import NM_CONNECTION_UUID_LENGTH
from pyanaconda.modules.network.kickstart import default_ks_vlan_interface_name
from pyanaconda.modules.network.utils import is_s390, prefix2netmask
from pyanaconda.modules.network.nm_client import get_iface_from_hwaddr, get_iface_from_connection, \
    get_team_port_config_from_connection, get_team_config_from_connection


from pyanaconda.anaconda_loggers import get_module_logger
log = get_module_logger(__name__)


IFCFG_DIR = "/etc/sysconfig/network-scripts"


class IfcfgFile(SimpleConfigFile):
    """Stores settings of ifcfg configuration file."""
    def __init__(self, path):
        """Create the object.

        :param path: path to the ifcfg file
        :type path: str
        """
        super().__init__(always_quote=True, filename=path)
        self._dirty = False
        self._path = path
        self._loaded = False

    @property
    def uuid(self):
        """The UUID of the ifcfg file."""
        if not self._loaded:
            self.read()
        return self.get('UUID')

    @property
    def path(self):
        """The path of the ifcfg file."""
        return self._path

    def read(self, filename=None):
        """Read the settings from the ifcfg file in."""
        self.reset()
        super().read()
        self._loaded = True
        self._dirty = False

    def write(self, filename=None, use_tmp=False):
        """Write the settings into the ifcfg file."""
        if self._dirty or filename:
            # ifcfg-rh is using inotify IN_CLOSE_WRITE event so we don't use
            # temporary file for new configuration
            super().write(filename, use_tmp=use_tmp)
            self._dirty = False

    def set(self, *args):
        """Set values of given settings of the ifcfg file.

        :param args: tuples containing setting name and its value
        :type args: tuple(str, str), ...
        """
        for (key, data) in args:
            if self.get(key) != data:
                break
        else:
            return
        super().set(*args)
        self._dirty = True

    def unset(self, *args):
        """Unset values of the specified ifcfg file settings.

        :param args: keys to be unset
        :type args: str, ...
        """
        for key in args:
            if self.get(key):
                self._dirty = True
                break
        else:
            return
        super().unset(*args)

    @property
    def is_from_kickstart(self):
        """Is the ifcfg file generated from kickstart?"""
        with open(self.path, 'r') as f:
            return "Generated by parse-kickstart" in f.read()


def get_ifcfg_files_paths(directory):
    rv = []
    for name in os.listdir(directory):
        if name.startswith("ifcfg-"):
            if name == "ifcfg-lo":
                continue
            rv.append(os.path.join(directory, name))
    return rv


def get_ifcfg_file(values, root_path=""):
    """Get ifcfg file specified by values.

    :param values: list required values of settings (KEY, VALUE)
                   of the looked for file
    :type values: list(tuple(str, str))
    :param root_path: search in the filesystem specified by root path
    :type root_path: str
    """
    for file_path in get_ifcfg_files_paths(os.path.normpath(root_path + IFCFG_DIR)):
        ifcfg = IfcfgFile(file_path)
        ifcfg.read()
        for key, value in values:
            if ifcfg.get(key) != value:
                break
        else:
            return ifcfg
    return None


def find_ifcfg_uuid_of_device(nm_client, device_name, hwaddr=None, root_path=""):
    """Get UUID of the ifcfg file of the specified device.

    :param nm_client: instance of NetworkManager client
    :type nm_client: NM.Client
    :param device_name: name of the device
    :type device_name: str
    :param hwaddr: hardware address of the device
    :type hwaddr: str
    :param root_path: search in the filesystem specified by root path
    :type root_path: str
    :returns: uuid of ifcfg file
    :rtype: str
    """
    uuid = None
    ifcfg = get_ifcfg_file_of_device(nm_client, device_name, hwaddr, root_path)
    if ifcfg:
        uuid = ifcfg.uuid
    return uuid


def get_ifcfg_file_of_device(nm_client, device_name, device_hwaddr=None, root_path=""):
    """Get ifcfg file for the device specified by name.

    :param nm_client: instance of NetworkManager client
    :type nm_client: NM.Client
    :param device_name: name of the device
    :type device_name: str
    :param device_hwaddr: hardware address of the device
    :type device_hwaddr: str
    :param root_path: search in the filesystem specified by root path
    :type root_path: str
    :returns: ifcfg file object
    :rtype: IfcfgFile
    """
    # hwaddr is supplementary (--bindto=mac)
    ifcfgs = []
    for file_path in get_ifcfg_files_paths(os.path.normpath(root_path + IFCFG_DIR)):
        ifcfg = IfcfgFile(file_path)
        ifcfg.read()
        device_type = ifcfg.get("TYPE") or ifcfg.get("DEVICETYPE")
        if device_type == "Wireless":
            # TODO check ESSID against active ssid of the device
            pass
        elif device_type in ("Bond", "Team", "Bridge", "Infiniband"):
            if ifcfg.get("DEVICE") == device_name:
                ifcfgs.append(ifcfg)
        elif device_type == "Vlan":
            interface_name = ifcfg.get("DEVICE")
            if interface_name:
                if interface_name == device_name:
                    ifcfgs.append(ifcfg)
            else:
                physdev = ifcfg.get("PHYSDEV")
                if len(physdev) == NM_CONNECTION_UUID_LENGTH:
                    physdev = get_iface_from_connection(nm_client, physdev)
                vlanid = ifcfg.get("VLAN_ID")
                generated_dev_name = default_ks_vlan_interface_name(physdev, vlanid)
                if device_name == generated_dev_name:
                    ifcfgs.append(ifcfg)

        elif device_type == "Ethernet":
            # Ignore slaves
            if ifcfg.get("MASTER") or ifcfg.get("TEAM_MASTER") or ifcfg.get("BRIDGE"):
                continue
            device = ifcfg.get("DEVICE")
            hwaddr = ifcfg.get("HWADDR")
            if device:
                if device == device_name:
                    ifcfgs.append(ifcfg)
            elif hwaddr:
                if device_hwaddr:
                    if device_hwaddr.upper() == hwaddr.upper():
                        ifcfgs.append(ifcfg)
                else:
                    iface = get_iface_from_hwaddr(nm_client, hwaddr)
                    if iface == device_name:
                        ifcfgs.append(ifcfg)
            elif is_s390():
                # s390 setting generated in dracut with net.ifnames=0
                # has neither DEVICE nor HWADDR (#1249750)
                if ifcfg.get("NAME") == device_name:
                    ifcfgs.append(ifcfg)

    if len(ifcfgs) > 1:
        log.debug("Unexpected number of ifcfg files found for %s: %s", device_name,
                  [ifcfg.path for ifcfg in ifcfgs])
    if ifcfgs:
        return ifcfgs[0]
    else:
        log.debug("Ifcfg file for %s not found", device_name)


def get_slaves_from_ifcfgs(nm_client, master_option, master_specs, root_path=""):
    """Get slaves of master specified by master_specs in master_option.

    :param nm_client: instance of NetworkManager client
    :type nm_client: NM.Client
    :param master_option: ifcfg setting containing specification of master
    :type master_option: str
    :param master_specs: a list containing sepcification of master:
                         device name or connection uuid or both
    :type master_specs: list(str)
    :param root_path: search in the filesystem specified by root path
    :type root_path: str
    :returns: slaves specified by interface and ifcfg uuid
    :rtype: set((str,str))
    """
    slaves = set()

    for file_path in get_ifcfg_files_paths(os.path.normpath(root_path + IFCFG_DIR)):
        ifcfg = IfcfgFile(file_path)
        ifcfg.read()
        master = ifcfg.get(master_option)
        if master in master_specs:
            iface = ifcfg.get("DEVICE")
            if not iface:
                hwaddr = ifcfg.get("HWADDR")
                iface = get_iface_from_hwaddr(nm_client, hwaddr)
            if iface:
                slaves.add((iface, ifcfg.get("UUID")))
    return slaves


def get_kickstart_network_data(ifcfg, nm_client, network_data_class, root_path=""):
    """Get kickstart data from ifcfg object.

    :param ifcfg: ifcfg file object
    :type ifcfg: IfcfgFile
    :param nm_client: instance of NetworkManager client
    :type nm_client: NM.Client
    :param network_data_class: pykickstart network command data class
    :type: pykickstart BaseData
    :param root_path: optional root path for ifcfg files to be updated
    :type root_path: str
    :returns: network_data object corresponding to the ifcfg object
    :rtype: network_data_class object instance
    """
    ifcfg.read()
    kwargs = {}

    # no network command for non-virtual device slaves
    if ifcfg.get("TYPE") not in ("Bond", "Team") and ifcfg.get("DEVICETYPE") != "Team":
        if ifcfg.get("MASTER") or ifcfg.get("TEAM_MASTER") or ifcfg.get("BRIDGE"):
            return None

    # no support for wireless
    if ifcfg.get("TYPE") == "Wireless":
        return None

    # ipv4 and ipv6
    if ifcfg.get("ONBOOT") and ifcfg.get("ONBOOT") == "no":
        kwargs["onboot"] = False
    if ifcfg.get('MTU') and ifcfg.get('MTU') != "0":
        kwargs["mtu"] = ifcfg.get('MTU')
    # ipv4
    if not ifcfg.get('BOOTPROTO'):
        kwargs["noipv4"] = True
    else:
        if util.lowerASCII(ifcfg.get('BOOTPROTO')) == 'dhcp':
            kwargs["bootProto"] = "dhcp"
            if ifcfg.get('DHCPCLASS'):
                kwargs["dhcpclass"] = ifcfg.get('DHCPCLASS')
        elif ifcfg.get('IPADDR'):
            kwargs["bootProto"] = "static"
            kwargs["ip"] = ifcfg.get('IPADDR')
            netmask = ifcfg.get('NETMASK')
            prefix = ifcfg.get('PREFIX')
            if not netmask and prefix:
                netmask = prefix2netmask(int(prefix))
            if netmask:
                kwargs["netmask"] = netmask
            # note that --gateway is common for ipv4 and ipv6
            if ifcfg.get('GATEWAY'):
                kwargs["gateway"] = ifcfg.get('GATEWAY')
        elif ifcfg.get('IPADDR0'):
            kwargs["bootProto"] = "static"
            kwargs["ip"] = ifcfg.get('IPADDR0')
            prefix = ifcfg.get('PREFIX0')
            if prefix:
                netmask = prefix2netmask(int(prefix))
                kwargs["netmask"] = netmask
            # note that --gateway is common for ipv4 and ipv6
            if ifcfg.get('GATEWAY0'):
                kwargs["gateway"] = ifcfg.get('GATEWAY0')

    # ipv6
    if not ifcfg.get('IPV6INIT') or ifcfg.get('IPV6INIT') == "no":
        kwargs["noipv6"] = True
    else:
        if ifcfg.get('IPV6_AUTOCONF') in ("yes", ""):
            kwargs["ipv6"] = "auto"
        else:
            if ifcfg.get('IPV6ADDR'):
                kwargs["ipv6"] = ifcfg.get('IPV6ADDR')
                if ifcfg.get('IPV6_DEFAULTGW') \
                        and ifcfg.get('IPV6_DEFAULTGW') != "::":
                    kwargs["ipv6gateway"] = ifcfg.get('IPV6_DEFAULTGW')
            if ifcfg.get('DHCPV6C') == "yes":
                kwargs["ipv6"] = "dhcp"

    # ipv4 and ipv6
    dnsline = ''
    for key in ifcfg.info.keys():
        if util.upperASCII(key).startswith('DNS'):
            if dnsline == '':
                dnsline = ifcfg.get(key)
            else:
                dnsline += "," + ifcfg.get(key)
    if dnsline:
        kwargs["nameserver"] = dnsline

    if ifcfg.get("ETHTOOL_OPTS"):
        kwargs["ethtool"] = ifcfg.get("ETHTOOL_OPTS")

    if ifcfg.get("ESSID"):
        kwargs["essid"] = ifcfg.get("ESSID")

    # hostname
    if ifcfg.get("DHCP_HOSTNAME"):
        kwargs["hostname"] = ifcfg.get("DHCP_HOSTNAME")

    iface = ifcfg.get("DEVICE")
    if not iface:
        hwaddr = ifcfg.get("HWADDR")
        if hwaddr:
            iface = get_iface_from_hwaddr(nm_client, hwaddr)
    if iface:
        kwargs["device"] = iface

    # bonding
    # FIXME: dracut has only BOND_OPTS
    if ifcfg.get("BONDING_MASTER") == "yes" or ifcfg.get("TYPE") == "Bond":
        slaves = sorted(get_slaves_from_ifcfgs(nm_client,
                                               "MASTER",
                                               [ifcfg.get("DEVICE"), ifcfg.get("UUID")],
                                               root_path=root_path))
        if slaves:
            kwargs["bondslaves"] = ",".join(iface for iface, uuid in slaves)
        bondopts = ifcfg.get("BONDING_OPTS")
        if bondopts:
            sep = ","
            if sep in bondopts:
                sep = ";"
            kwargs["bondopts"] = sep.join(bondopts.split())

    # vlan
    if ifcfg.get("VLAN") == "yes" or ifcfg.get("TYPE") == "Vlan":
        physdev = ifcfg.get("PHYSDEV")
        if len(physdev) == NM_CONNECTION_UUID_LENGTH:
            physdev = get_iface_from_connection(nm_client, physdev)
        kwargs["device"] = physdev
        kwargs["vlanid"] = ifcfg.get("VLAN_ID")
        interface_name = ifcfg.get("DEVICE")
        default_name = default_ks_vlan_interface_name(kwargs["device"], kwargs["vlanid"])
        if interface_name and interface_name != default_name:
            kwargs["interfacename"] = interface_name

    # bridging
    if ifcfg.get("TYPE") == "Bridge":
        slaves = sorted(get_slaves_from_ifcfgs(nm_client,
                                               "BRIDGE",
                                               [ifcfg.get("DEVICE"), ifcfg.get("UUID")],
                                               root_path=root_path))
        if slaves:
            kwargs["bridgeslaves"] = ",".join(iface for iface, uuid in slaves)

        bridgeopts = ifcfg.get("BRIDGING_OPTS").replace('_', '-').split()
        if ifcfg.get("STP"):
            bridgeopts.append("%s=%s" % ("stp", ifcfg.get("STP")))
        if ifcfg.get("DELAY"):
            bridgeopts.append("%s=%s" % ("forward-delay", ifcfg.get("DELAY")))
        if bridgeopts:
            kwargs["bridgeopts"] = ",".join(bridgeopts)

    nd = network_data_class(**kwargs)

    # teaming
    if ifcfg.get("TYPE") == "Team" or ifcfg.get("DEVICETYPE") == "Team":
        slaves = sorted(get_slaves_from_ifcfgs(nm_client,
                                               "TEAM_MASTER",
                                               [ifcfg.get("DEVICE"), ifcfg.get("UUID")],
                                               root_path=root_path))
        for iface, uuid in slaves:
            team_port_cfg = get_team_port_config_from_connection(nm_client, uuid) or ""
            nd.teamslaves.append((iface, team_port_cfg))
        teamconfig = get_team_config_from_connection(nm_client, ifcfg.get("UUID"))
        if teamconfig:
            nd.teamconfig = teamconfig
    return nd


def get_master_slaves_from_ifcfgs(nm_client, master_devname, root_path="", uuid=None):
    """Get all slaves based on ifcfg files of given master.

    Master can be identified by device name or uuid. If uuid is not provided
    as argument it will be looked up in master's ifcfg file.

    :param nm_client: instance of NetworkManager client
    :type nm_client: NM.Client
    :param master_devname: name of master device
    :type master_devname: str
    :param root_path: optional root path for ifcfg files
    :type root_path: str
    :param uuid: uuid of master connection (optional)
    :type uuid: str
    :returns: list of slaves represented by tuple (<CONNECTION_NAME>, <UUID>)
    :rtype: list((str, str))
    """
    slaves = []
    # Master can be identified by devname or uuid, try to find master uuid
    if not uuid:
        uuid = find_ifcfg_uuid_of_device(nm_client, master_devname, root_path=root_path)
    for ifcfg_path in get_ifcfg_files_paths(os.path.normpath(root_path + IFCFG_DIR)):
        ifcfg = IfcfgFile(ifcfg_path)
        ifcfg.read()
        master = ifcfg.get("MASTER") or ifcfg.get("TEAM_MASTER") or ifcfg.get("BRIDGE")
        if master and master in (master_devname, uuid):
            slaves.append((ifcfg.get("NAME"), ifcfg.get("UUID")))
    return slaves


def get_ifcfg_files_content(root_path=""):
    fragments = []
    for file_path in get_ifcfg_files_paths(os.path.normpath(root_path + IFCFG_DIR)):
        fragments.append("{}:".format(file_path))
        with open(file_path, "r") as f:
            fragments.append(f.read().strip("\n"))
    return "\n".join(fragments)
