"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.componentCreateTask = exports.generateComponents = exports.promptDetails = void 0;
const tslib_1 = require("tslib");
const fs_1 = tslib_1.__importDefault(require("fs"));
const inquirer_1 = require("inquirer");
const lodash_1 = require("lodash");
const templates_1 = require("../templates");
const pascalCase_1 = require("../utils/pascalCase");
const prompt_1 = require("../utils/prompt");
const task_1 = require("./task");
const componentGroups = [
    { name: 'General', value: 'General' },
    { name: 'Forms', value: 'Forms' },
    { name: 'Panel', value: 'Panel' },
    { name: 'Visualizations', value: 'Visualizations' },
    { name: 'Others', value: 'Others' },
];
const promptDetails = () => {
    return (0, inquirer_1.prompt)([
        (0, prompt_1.promptInput)('name', 'Component name', true),
        (0, prompt_1.promptConfirm)('hasTests', "Generate component's test file?"),
        (0, prompt_1.promptConfirm)('hasStory', "Generate component's story file?"),
        (0, prompt_1.promptConfirm)('isStoryPublic', 'Generate public story? (Selecting "No" will create an internal story)', true, ({ hasStory }) => hasStory),
        (0, prompt_1.promptList)('group', 'Select component group for the story (e.g. Forms, Layout)', () => componentGroups, 0, ({ hasStory }) => hasStory),
    ]);
};
exports.promptDetails = promptDetails;
const generateComponents = ({ details, path }) => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
    const name = (0, pascalCase_1.pascalCase)(details.name);
    const getCompiled = (template) => {
        return (0, lodash_1.template)(template)(Object.assign(Object.assign({}, details), { name }));
    };
    const filePath = `${path}/${name}`;
    let paths = [];
    fs_1.default.writeFileSync(`${filePath}.tsx`, getCompiled(templates_1.componentTpl));
    paths.push(`${filePath}.tsx`);
    if (details.hasTests) {
        fs_1.default.writeFileSync(`${filePath}.test.tsx`, getCompiled(templates_1.testTpl));
        paths.push(`${filePath}.test.tsx`);
    }
    if (details.hasStory) {
        const storyExt = details.isStoryPublic ? '.story.tsx' : '.story.internal.tsx';
        fs_1.default.writeFileSync(`${filePath}${storyExt}`, getCompiled(templates_1.storyTpl));
        fs_1.default.writeFileSync(`${filePath}.mdx`, getCompiled(templates_1.docsTpl));
        paths.push(`${filePath}${storyExt}`, `${filePath}.mdx`);
    }
    console.log('Generated files:');
    console.log(paths.join('\n'));
});
exports.generateComponents = generateComponents;
const componentCreateRunner = () => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
    const destPath = process.cwd();
    const details = yield (0, exports.promptDetails)();
    yield (0, exports.generateComponents)({ details, path: destPath });
});
exports.componentCreateTask = new task_1.Task('component:create', componentCreateRunner);
//# sourceMappingURL=component.create.js.map