"use strict";

var _enhanceArgTypes = require("./enhanceArgTypes");

expect.addSnapshotSerializer({
  print: function print(val) {
    return JSON.stringify(val, null, 2);
  },
  test: function test(val) {
    return typeof val !== 'string';
  }
});

var enhance = function enhance(_ref) {
  var argType = _ref.argType,
      arg = _ref.arg,
      extractedArgTypes = _ref.extractedArgTypes,
      _ref$isArgsStory = _ref.isArgsStory,
      isArgsStory = _ref$isArgsStory === void 0 ? true : _ref$isArgsStory;
  var context = {
    id: 'foo--bar',
    kind: 'foo',
    name: 'bar',
    parameters: {
      component: 'dummy',
      __isArgsStory: isArgsStory,
      docs: {
        extractArgTypes: extractedArgTypes && function () {
          return extractedArgTypes;
        }
      },
      argTypes: argType && {
        input: argType
      },
      args: {
        input: arg
      }
    },
    args: {},
    argTypes: {},
    globals: {}
  };
  return (0, _enhanceArgTypes.enhanceArgTypes)(context);
};

describe('enhanceArgTypes', function () {
  describe('no-args story function', function () {
    it('should no-op', function () {
      expect(enhance({
        argType: {
          foo: 'unmodified',
          type: {
            name: 'number'
          }
        },
        isArgsStory: false
      }).input).toMatchInlineSnapshot("\n        {\n          \"name\": \"input\",\n          \"foo\": \"unmodified\",\n          \"type\": {\n            \"name\": \"number\"\n          }\n        }\n      ");
    });
  });
  describe('args story function', function () {
    describe('single-source input', function () {
      describe('argTypes input', function () {
        it('number', function () {
          expect(enhance({
            argType: {
              type: {
                name: 'number'
              }
            }
          }).input).toMatchInlineSnapshot("\n            {\n              \"name\": \"input\",\n              \"type\": {\n                \"name\": \"number\"\n              }\n            }\n          ");
        });
      });
      describe('extraction from component', function () {
        it('number', function () {
          expect(enhance({
            extractedArgTypes: {
              input: {
                name: 'input',
                type: {
                  name: 'number'
                }
              }
            }
          }).input).toMatchInlineSnapshot("\n            {\n              \"name\": \"input\",\n              \"type\": {\n                \"name\": \"number\"\n              }\n            }\n          ");
        });
      });
      describe('controls input', function () {
        it('range', function () {
          expect(enhance({
            argType: {
              control: {
                type: 'range',
                min: 0,
                max: 100
              }
            }
          }).input).toMatchInlineSnapshot("\n            {\n              \"name\": \"input\",\n              \"control\": {\n                \"type\": \"range\",\n                \"min\": 0,\n                \"max\": 100\n              }\n            }\n          ");
        });
        it('options', function () {
          expect(enhance({
            argType: {
              control: {
                type: 'radio',
                options: [1, 2]
              }
            }
          }).input).toMatchInlineSnapshot("\n            {\n              \"name\": \"input\",\n              \"control\": {\n                \"type\": \"radio\",\n                \"options\": [\n                  1,\n                  2\n                ]\n              }\n            }\n          ");
        });
      });
    });
    describe('mixed-source input', function () {
      it('user-specified argTypes take precedence over extracted argTypes', function () {
        expect(enhance({
          argType: {
            type: {
              name: 'number'
            }
          },
          extractedArgTypes: {
            input: {
              type: {
                name: 'string'
              }
            }
          }
        }).input).toMatchInlineSnapshot("\n          {\n            \"type\": {\n              \"name\": \"number\"\n            },\n            \"name\": \"input\"\n          }\n        ");
      });
      it('user-specified argTypes take precedence over inferred argTypes', function () {
        expect(enhance({
          argType: {
            type: {
              name: 'number'
            }
          },
          arg: 'hello'
        }).input).toMatchInlineSnapshot("\n          {\n            \"name\": \"input\",\n            \"type\": {\n              \"name\": \"number\"\n            }\n          }\n        ");
      });
      it('extracted argTypes take precedence over inferred argTypes', function () {
        expect(enhance({
          extractedArgTypes: {
            input: {
              type: {
                name: 'string'
              }
            }
          },
          arg: 6
        }).input).toMatchInlineSnapshot("\n          {\n            \"type\": {\n              \"name\": \"string\"\n            }\n          }\n        ");
      });
      it('user-specified controls take precedence over inferred controls', function () {
        expect(enhance({
          argType: {
            defaultValue: 5,
            control: {
              type: 'range',
              step: 50
            }
          },
          arg: 3,
          extractedArgTypes: {
            input: {
              name: 'input'
            }
          }
        }).input).toMatchInlineSnapshot("\n          {\n            \"name\": \"input\",\n            \"defaultValue\": 5,\n            \"control\": {\n              \"type\": \"range\",\n              \"step\": 50\n            }\n          }\n        ");
      });
      it('includes extracted argTypes when there are no user-specified argTypes', function () {
        expect(enhance({
          arg: 3,
          extractedArgTypes: {
            input: {
              name: 'input'
            },
            foo: {
              type: {
                name: 'number'
              }
            }
          }
        })).toMatchInlineSnapshot("\n          {\n            \"input\": {\n              \"name\": \"input\"\n            },\n            \"foo\": {\n              \"type\": {\n                \"name\": \"number\"\n              }\n            }\n          }\n        ");
      });
      it('includes extracted argTypes when user-specified argTypes match', function () {
        expect(enhance({
          argType: {
            type: {
              name: 'number'
            }
          },
          extractedArgTypes: {
            input: {
              name: 'input'
            },
            foo: {
              type: {
                name: 'number'
              }
            }
          }
        })).toMatchInlineSnapshot("\n          {\n            \"input\": {\n              \"name\": \"input\",\n              \"type\": {\n                \"name\": \"number\"\n              }\n            },\n            \"foo\": {\n              \"type\": {\n                \"name\": \"number\"\n              }\n            }\n          }\n        ");
      });
      it('excludes extracted argTypes when user-specified argTypes do not match', function () {
        expect(enhance({
          argType: {
            type: {
              name: 'number'
            }
          },
          extractedArgTypes: {
            foo: {
              type: {
                name: 'number'
              }
            }
          }
        })).toMatchInlineSnapshot("\n          {\n            \"foo\": {\n              \"type\": {\n                \"name\": \"number\"\n              }\n            },\n            \"input\": {\n              \"name\": \"input\",\n              \"type\": {\n                \"name\": \"number\"\n              }\n            }\n          }\n        ");
      });
    });
  });
});