# Copyright (C) Jan 2020 Mellanox Technologies Ltd. All rights reserved.
# Copyright (c) 2021 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
#
# This software is available to you under a choice of one of two
# licenses.  You may choose to be licensed under the terms of the GNU
# General Public License (GPL) Version 2, available from the file
# COPYING in the main directory of this source tree, or the
# OpenIB.org BSD license below:
#
#     Redistribution and use in source and binary forms, with or
#     without modification, are permitted provided that the following
#     conditions are met:
#
#      - Redistributions of source code must retain the above
#        copyright notice, this list of conditions and the following
#        disclaimer.
#
#      - Redistributions in binary form must reproduce the above
#        copyright notice, this list of conditions and the following
#        disclaimer in the documentation and/or other materials
#        provided with the distribution.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
# BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
# ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
# CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
# --


#######################################################
#
# resourceparse.py
# Python implementation of the Class ResourceParse
# Generated by Enterprise Architect
# Created on:      16-Dec-2019 14:11:58 AM
# Original author: talve
#
#######################################################
import tools_version
import sys
import os

if sys.version_info[0] < 3:
    print("Error: This tool supports python 3.x only. Exiting...")
    exit(1)

import argparse
from utils import constants as cs
from utils.common_functions import valid_path_arg_type
from ResourceParseManager import ResourceParseManager
from utils.Exceptions import ResourceParseException
from parsers.ResourceParser import PARSER_CLASSES, parser_type

sys.path.append(os.path.join("common"))


class ResourceParse:
    """This class is responsible for the resource dump UI by handling the user inputs and
       and running the right command.
    """

    @classmethod
    def _run_arg_parse(self):
        """This method run the arg parse and return the arguments from the UI.
        """

        # Main parser
        tool_name = os.path.basename(__file__.split('.')[0])

        parser = argparse.ArgumentParser(prog=tool_name, add_help=False)  # todo: fix help

        # Common required arguments
        required_named = parser.add_argument_group('Common required arguments')
        required_named.add_argument("-d", "--dump-file", type=valid_path_arg_type, required=True, help='Location of the dump file used for parsing')

        # Common optional arguments
        optional_named = parser.add_argument_group('Common optional arguments')
        optional_named.add_argument("-p", "--parser", dest="resource_parser", type=parser_type, default=PARSER_CLASSES["adb"], help="Available options: {}. Default: 'adb'. see (Parsing methods) ".format(list(PARSER_CLASSES.keys())))
        optional_named.add_argument('--version', action='version', help="Shows the tool's version and exit", version=tools_version.GetVersionString(tool_name, None))
        optional_named.add_argument("-o", "--out", help='Location of the output file')
        optional_named.add_argument("-v", help='Verbosity notice', dest="verbose", default=0, action='count')

        # Artificial Parser, only to generate help message
        common_description = \
            """Description:
    This tool parses the output of "resourcedump" tool.
    There are several methods of parsing (see --parser).

Parsing methods:
"""
        full_description = common_description + "\n".join(["    {}:\n".format(parsing_method) + parser_class.get_description() for parsing_method, parser_class in PARSER_CLASSES.items()])

        help_parent_parsers = [parser_class.get_arg_parser() for parser_class in PARSER_CLASSES.values() if parser_class.get_arg_parser()]
        help_parser = argparse.ArgumentParser(prog=tool_name, add_help=False, parents=[parser] + help_parent_parsers, description=full_description, formatter_class=argparse.RawDescriptionHelpFormatter)

        class UnifiedHelpAction(argparse._HelpAction):
            def __call__(self, parser, namespace, values, option_string):
                help_parser.print_help()
                parser.exit()

        optional_named.add_argument("-h", "--help", action=UnifiedHelpAction, help="show this help message and exit")

        manager_args, remaining = parser.parse_known_args()
        parsing_method_arg_parser = manager_args.resource_parser.get_arg_parser()
        parser_args = parsing_method_arg_parser.parse_args(remaining) if parsing_method_arg_parser else argparse.Namespace

        return manager_args, parser_args

    def run(self):
        """This method run the parser with the needed arguments
        """
        manager_args, parser_args = self._run_arg_parse()
        creator = ResourceParseManager(manager_args, parser_args)
        creator.parse()


if __name__ == '__main__':
    try:
        ResourceParse().run()
    except ResourceParseException as rpe:
        print("ResourceParse failed!\n{0}.\nExiting...".format(rpe))
        sys.exit(1)
    except Exception as e:
        print("FATAL ERROR!\n{0}.\nExiting...".format(e))
        sys.exit(1)
