/*
 * Decompiled with CFR 0.152.
 */
package org.mozilla.jss.netscape.security.pkcs;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.FileOutputStream;
import java.math.BigInteger;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.security.MessageDigest;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.security.cert.CertificateException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import javax.naming.InvalidNameException;
import javax.naming.ldap.LdapName;
import javax.naming.ldap.Rdn;
import javax.security.auth.x500.X500Principal;
import org.apache.commons.lang3.StringUtils;
import org.mozilla.jss.CryptoManager;
import org.mozilla.jss.asn1.ANY;
import org.mozilla.jss.asn1.ASN1Value;
import org.mozilla.jss.asn1.BMPString;
import org.mozilla.jss.asn1.OBJECT_IDENTIFIER;
import org.mozilla.jss.asn1.OCTET_STRING;
import org.mozilla.jss.asn1.SEQUENCE;
import org.mozilla.jss.asn1.SET;
import org.mozilla.jss.crypto.CryptoStore;
import org.mozilla.jss.crypto.CryptoToken;
import org.mozilla.jss.crypto.EncryptionAlgorithm;
import org.mozilla.jss.crypto.NoSuchItemOnTokenException;
import org.mozilla.jss.crypto.ObjectNotFoundException;
import org.mozilla.jss.crypto.PBEAlgorithm;
import org.mozilla.jss.crypto.PrivateKey;
import org.mozilla.jss.crypto.X509Certificate;
import org.mozilla.jss.netscape.security.pkcs.PKCS12;
import org.mozilla.jss.netscape.security.pkcs.PKCS12CertInfo;
import org.mozilla.jss.netscape.security.pkcs.PKCS12KeyInfo;
import org.mozilla.jss.netscape.security.util.Utils;
import org.mozilla.jss.netscape.security.x509.X509CertImpl;
import org.mozilla.jss.pkcs11.PK11Cert;
import org.mozilla.jss.pkcs11.PK11Store;
import org.mozilla.jss.pkcs12.AuthenticatedSafes;
import org.mozilla.jss.pkcs12.CertBag;
import org.mozilla.jss.pkcs12.PFX;
import org.mozilla.jss.pkcs12.PasswordConverter;
import org.mozilla.jss.pkcs12.SafeBag;
import org.mozilla.jss.pkix.primitive.Attribute;
import org.mozilla.jss.pkix.primitive.EncryptedPrivateKeyInfo;
import org.mozilla.jss.util.Password;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PKCS12Util {
    private static Logger logger = LoggerFactory.getLogger(PKCS12Util.class);
    public static final String NO_ENCRYPTION = "none";
    public static final List<PBEAlgorithm> SUPPORTED_CERT_ENCRYPTIONS = Arrays.asList(null, PBEAlgorithm.PBE_SHA1_RC2_40_CBC);
    public static final List<PBEAlgorithm> SUPPORTED_KEY_ENCRYPTIONS = Arrays.asList(PBEAlgorithm.PBE_PKCS5_PBES2, PBEAlgorithm.PBE_SHA1_DES3_CBC);
    public static final PBEAlgorithm DEFAULT_CERT_ENCRYPTION = SUPPORTED_CERT_ENCRYPTIONS.get(0);
    public static final String DEFAULT_CERT_ENCRYPTION_NAME = "none";
    public static final PBEAlgorithm DEFAULT_KEY_ENCRYPTION = SUPPORTED_KEY_ENCRYPTIONS.get(0);
    public static final String DEFAULT_KEY_ENCRYPTION_NAME = DEFAULT_KEY_ENCRYPTION.toString();
    SecureRandom random;
    PBEAlgorithm certEncryption = DEFAULT_CERT_ENCRYPTION;
    PBEAlgorithm keyEncryption = DEFAULT_KEY_ENCRYPTION;
    boolean trustFlagsEnabled = true;

    public PKCS12Util() throws Exception {
        this.random = SecureRandom.getInstance("pkcs11prng", "Mozilla-JSS");
    }

    public void setCertEncryption(String name) throws Exception {
        for (PBEAlgorithm algorithm : SUPPORTED_CERT_ENCRYPTIONS) {
            if (algorithm == null) {
                if (!"none".equals(name)) continue;
                this.certEncryption = null;
                return;
            }
            if (!algorithm.toString().equals(name)) continue;
            this.certEncryption = algorithm;
            return;
        }
        throw new Exception("Unsupported certificate encryption: " + name);
    }

    public void setCertEncryption(PBEAlgorithm algorithm) throws Exception {
        this.certEncryption = algorithm;
    }

    public PBEAlgorithm getCertEncryption() {
        return this.certEncryption;
    }

    public void setKeyEncryption(String name) throws Exception {
        for (PBEAlgorithm algorithm : SUPPORTED_KEY_ENCRYPTIONS) {
            if (algorithm == null) {
                if (!"none".equals(name)) continue;
                this.keyEncryption = null;
                return;
            }
            if (!algorithm.toString().equals(name)) continue;
            this.keyEncryption = algorithm;
            return;
        }
        throw new Exception("Unsupported key encryption: " + name);
    }

    public void setKeyEncryption(PBEAlgorithm algorithm) throws Exception {
        this.keyEncryption = algorithm;
    }

    public PBEAlgorithm getKeyEncryption() {
        return this.keyEncryption;
    }

    public boolean isTrustFlagsEnabled() {
        return this.trustFlagsEnabled;
    }

    public void setTrustFlagsEnabled(boolean trustFlagsEnabled) {
        this.trustFlagsEnabled = trustFlagsEnabled;
    }

    @Deprecated
    public String getTrustFlags(X509Certificate cert) {
        PK11Cert pk11Cert = (PK11Cert)cert;
        return pk11Cert.getTrustFlags();
    }

    @Deprecated
    public void setTrustFlags(X509Certificate cert, String trustFlags) throws Exception {
        PK11Cert pk11Cert = (PK11Cert)cert;
        pk11Cert.setTrustFlags(trustFlags);
    }

    public void addKeyBag(PKCS12KeyInfo keyInfo, Password password, SEQUENCE encSafeContents) throws Exception {
        ASN1Value content;
        byte[] keyID = keyInfo.getID();
        logger.debug(" - Key ID: " + Utils.HexEncode(keyID));
        byte[] epkiBytes = keyInfo.getEncryptedPrivateKeyInfoBytes();
        if (epkiBytes != null) {
            content = new ANY(epkiBytes);
        } else {
            PrivateKey privateKey = keyInfo.getPrivateKey();
            if (privateKey == null) {
                throw new Exception("Missing private key for " + keyInfo.getFriendlyName());
            }
            CryptoToken token = CryptoManager.getInstance().getInternalKeyStorageToken();
            if (this.keyEncryption == PBEAlgorithm.PBE_SHA1_DES3_CBC) {
                content = this.create_EPKI_with_PBE_SHA1_DES3_CBC(token, privateKey, password);
            } else if (this.keyEncryption == PBEAlgorithm.PBE_PKCS5_PBES2) {
                content = this.create_EPKI_with_PBE_PKCS5_PBES2(token, privateKey, password);
            } else {
                throw new Exception("Unsupported key encryption: " + this.keyEncryption);
            }
        }
        SET keyAttrs = this.createKeyBagAttrs(keyInfo);
        SafeBag safeBag = new SafeBag(SafeBag.PKCS8_SHROUDED_KEY_BAG, content, keyAttrs);
        encSafeContents.addElement(safeBag);
    }

    public ASN1Value create_EPKI_with_PBE_SHA1_DES3_CBC(CryptoToken token, PrivateKey privateKey, Password password) throws Exception {
        byte[] salt = new byte[16];
        this.random.nextBytes(salt);
        return EncryptedPrivateKeyInfo.createPBE(PBEAlgorithm.PBE_SHA1_DES3_CBC, password, salt, 100000, new PasswordConverter(), privateKey, token);
    }

    public ASN1Value create_EPKI_with_PBE_PKCS5_PBES2(CryptoToken token, PrivateKey privateKey, Password password) throws Exception {
        CryptoStore store = token.getCryptoStore();
        byte[] bytes = store.getEncryptedPrivateKeyInfo(null, password, EncryptionAlgorithm.AES_256_CBC, 0, privateKey);
        return new ANY(bytes);
    }

    public void addCertBag(PKCS12CertInfo certInfo, SEQUENCE safeContents) throws Exception {
        byte[] id = certInfo.getID();
        logger.debug(" - Certificate ID: " + Utils.HexEncode(id));
        X509CertImpl cert = certInfo.getCert();
        OCTET_STRING certAsn1 = new OCTET_STRING(cert.getEncoded());
        CertBag certBag = new CertBag(CertBag.X509_CERT_TYPE, certAsn1);
        SET certAttrs = this.createCertBagAttrs(certInfo);
        SafeBag safeBag = new SafeBag(SafeBag.CERT_BAG, certBag, certAttrs);
        safeContents.addElement(safeBag);
    }

    BigInteger createLocalID(X509Certificate cert) throws Exception {
        return this.createLocalID(cert.getEncoded());
    }

    BigInteger createLocalID(byte[] bytes) throws Exception {
        MessageDigest md = MessageDigest.getInstance("SHA");
        md.update(bytes);
        return new BigInteger(1, md.digest());
    }

    SET createKeyBagAttrs(PKCS12KeyInfo keyInfo) throws Exception {
        SET attrs = new SET();
        String friendlyName = keyInfo.getFriendlyName();
        logger.debug("   Friendly name: " + friendlyName);
        SEQUENCE subjectAttr = new SEQUENCE();
        subjectAttr.addElement(SafeBag.FRIENDLY_NAME);
        SET subjectSet = new SET();
        subjectSet.addElement(new BMPString(friendlyName));
        subjectAttr.addElement(subjectSet);
        attrs.addElement(subjectAttr);
        byte[] keyID = keyInfo.getID();
        SEQUENCE localKeyAttr = new SEQUENCE();
        localKeyAttr.addElement(SafeBag.LOCAL_KEY_ID);
        SET localKeySet = new SET();
        localKeySet.addElement(new OCTET_STRING(keyID));
        localKeyAttr.addElement(localKeySet);
        attrs.addElement(localKeyAttr);
        return attrs;
    }

    SET createCertBagAttrs(PKCS12CertInfo certInfo) throws Exception {
        byte[] keyID;
        SET attrs = new SET();
        String friendlyName = certInfo.getFriendlyName();
        logger.debug("   Friendly name: " + friendlyName);
        SEQUENCE nicknameAttr = new SEQUENCE();
        nicknameAttr.addElement(SafeBag.FRIENDLY_NAME);
        SET nicknameSet = new SET();
        nicknameSet.addElement(new BMPString(friendlyName));
        nicknameAttr.addElement(nicknameSet);
        attrs.addElement(nicknameAttr);
        String trustFlags = certInfo.getTrustFlags();
        if (trustFlags != null && this.trustFlagsEnabled) {
            logger.debug("   Trust flags: " + trustFlags);
            SEQUENCE trustFlagsAttr = new SEQUENCE();
            trustFlagsAttr.addElement(PKCS12.CERT_TRUST_FLAGS_OID);
            SET trustFlagsSet = new SET();
            trustFlagsSet.addElement(new BMPString(trustFlags));
            trustFlagsAttr.addElement(trustFlagsSet);
            attrs.addElement(trustFlagsAttr);
        }
        if ((keyID = certInfo.getKeyID()) != null) {
            logger.debug("   Key ID: " + Utils.HexEncode(keyID));
            SEQUENCE localKeyAttr = new SEQUENCE();
            localKeyAttr.addElement(SafeBag.LOCAL_KEY_ID);
            SET localKeySet = new SET();
            localKeySet.addElement(new OCTET_STRING(keyID));
            localKeyAttr.addElement(localKeySet);
            attrs.addElement(localKeyAttr);
        }
        return attrs;
    }

    public void loadFromNSS(PKCS12 pkcs12) throws Exception {
        this.loadFromNSS(pkcs12, true, true);
    }

    public void loadFromNSS(PKCS12 pkcs12, boolean includeKey, boolean includeChain) throws Exception {
        logger.info("Loading certificates and keys from NSS database");
        CryptoManager cm = CryptoManager.getInstance();
        CryptoToken token = cm.getInternalKeyStorageToken();
        CryptoStore store = token.getCryptoStore();
        for (X509Certificate cert : store.getCertificates()) {
            this.loadCertFromNSS(pkcs12, cert, includeKey, includeChain);
        }
    }

    public void loadCertFromNSS(PKCS12 pkcs12, String nickname, boolean includeKey, boolean includeChain) throws Exception {
        this.loadCertFromNSS(pkcs12, nickname, includeKey, includeChain, null);
    }

    public void loadCertFromNSS(PKCS12 pkcs12, String nickname, boolean includeKey, boolean includeChain, String friendlyName) throws Exception {
        CryptoManager cm = CryptoManager.getInstance();
        X509Certificate[] certs = cm.findCertsByNickname(nickname);
        if (certs == null || certs.length == 0) {
            throw new Exception("Certificate not found: " + nickname);
        }
        for (X509Certificate cert : certs) {
            this.loadCertFromNSS(pkcs12, cert, includeKey, includeChain, friendlyName);
        }
    }

    public void loadCertFromNSS(PKCS12 pkcs12, X509Certificate cert, boolean includeKey, boolean includeChain) throws Exception {
        this.loadCertFromNSS(pkcs12, cert, includeKey, includeChain, null);
    }

    public void loadCertFromNSS(PKCS12 pkcs12, X509Certificate cert, boolean includeKey, boolean includeChain, String friendlyName) throws Exception {
        CryptoManager cm = CryptoManager.getInstance();
        PKCS12CertInfo certInfo = this.createCertInfoFromNSS(cert, friendlyName);
        pkcs12.addCertInfo(certInfo, true);
        byte[] id = certInfo.getID();
        logger.debug(" - Certificate ID: " + Utils.HexEncode(id));
        logger.debug("   Friendly name: " + certInfo.getFriendlyName());
        logger.debug("   Trust flags: " + certInfo.getTrustFlags());
        if (includeKey) {
            try {
                PrivateKey privateKey = cm.findPrivKeyByCert(cert);
                PKCS12KeyInfo keyInfo = this.createKeyInfoFromNSS(cert, privateKey, friendlyName);
                pkcs12.addKeyInfo(keyInfo);
                byte[] keyID = keyInfo.getID();
                certInfo.setKeyID(keyID);
                logger.debug("   Key ID: " + Utils.HexEncode(keyID));
            }
            catch (ObjectNotFoundException e) {
                logger.debug("Certificate has no private key");
            }
        }
        if (includeChain) {
            X509Certificate[] certChain = cm.buildCertificateChain(cert);
            if (certChain.length > 1) {
                logger.debug("   Certificate Chain:");
            }
            for (int i = 1; i < certChain.length; ++i) {
                X509Certificate caCert = certChain[i];
                PKCS12CertInfo caCertInfo = this.createCertInfoFromNSS(caCert);
                pkcs12.addCertInfo(caCertInfo, false);
                byte[] caCertID = caCertInfo.getID();
                logger.debug("   - Certificate ID: " + Utils.HexEncode(caCertID));
                logger.debug("     Friendly name: " + caCertInfo.getFriendlyName());
                logger.debug("     Trust flags: " + caCertInfo.getTrustFlags());
            }
        }
    }

    public PKCS12CertInfo createCertInfoFromNSS(X509Certificate cert) throws Exception {
        return this.createCertInfoFromNSS(cert, null);
    }

    public PKCS12CertInfo createCertInfoFromNSS(X509Certificate cert, String friendlyName) throws Exception {
        byte[] id = SafeBag.getLocalKeyIDFromCert(cert.getEncoded());
        if (friendlyName == null) {
            friendlyName = cert.getNickname();
        }
        X509CertImpl certImpl = new X509CertImpl(cert.getEncoded());
        PK11Cert p11Cert = (PK11Cert)cert;
        String trustFlags = p11Cert.getTrustFlags();
        PKCS12CertInfo certInfo = new PKCS12CertInfo();
        certInfo.setID(id);
        certInfo.setFriendlyName(friendlyName);
        certInfo.setCert(certImpl);
        certInfo.setTrustFlags(trustFlags);
        return certInfo;
    }

    public PKCS12KeyInfo createKeyInfoFromNSS(X509Certificate cert, PrivateKey privateKey) throws Exception {
        return this.createKeyInfoFromNSS(cert, privateKey, null);
    }

    public PKCS12KeyInfo createKeyInfoFromNSS(X509Certificate cert, PrivateKey privateKey, String friendlyName) throws Exception {
        byte[] keyID = privateKey.getUniqueID();
        if (friendlyName == null) {
            friendlyName = cert.getNickname();
        }
        PKCS12KeyInfo keyInfo = new PKCS12KeyInfo(privateKey);
        keyInfo.setID(keyID);
        keyInfo.setFriendlyName(friendlyName);
        return keyInfo;
    }

    public PFX generatePFX(PKCS12 pkcs12, Password password) throws Exception {
        byte[] salt;
        logger.info("Generating PKCS #12 data");
        AuthenticatedSafes authSafes = new AuthenticatedSafes();
        Collection<PKCS12KeyInfo> keyInfos = pkcs12.getKeyInfos();
        Collection<PKCS12CertInfo> certInfos = pkcs12.getCertInfos();
        if (!keyInfos.isEmpty()) {
            SEQUENCE keySafeContents = new SEQUENCE();
            for (PKCS12KeyInfo keyInfo : keyInfos) {
                this.addKeyBag(keyInfo, password, keySafeContents);
            }
            authSafes.addSafeContents(keySafeContents);
        }
        if (!certInfos.isEmpty()) {
            SEQUENCE certSafeContents = new SEQUENCE();
            for (PKCS12CertInfo certInfo : certInfos) {
                this.addCertBag(certInfo, certSafeContents);
            }
            if (this.certEncryption == null) {
                authSafes.addSafeContents(certSafeContents);
            } else if (this.certEncryption == PBEAlgorithm.PBE_SHA1_RC2_40_CBC) {
                salt = new byte[16];
                this.random.nextBytes(salt);
                authSafes.addEncryptedSafeContents(this.certEncryption, password, salt, 100000, certSafeContents);
            } else {
                throw new Exception("Unsupported certificate encryption: " + this.certEncryption);
            }
        }
        PFX pfx = new PFX(authSafes);
        salt = new byte[16];
        this.random.nextBytes(salt);
        pfx.computeMacData(password, salt, 100000);
        return pfx;
    }

    public void storeIntoFile(PKCS12 pkcs12, String filename, Password password) throws Exception {
        PFX pfx = this.generatePFX(pkcs12, password);
        logger.info("Storing PKCS #12 data into " + filename);
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        pfx.encode(bos);
        byte[] data = bos.toByteArray();
        try (FileOutputStream fos = new FileOutputStream(filename);){
            fos.write(data);
        }
    }

    public PKCS12KeyInfo getKeyInfo(SafeBag bag, Password password) throws Exception {
        PKCS12KeyInfo keyInfo = new PKCS12KeyInfo(bag.getBagContent().getEncoded());
        SET bagAttrs = bag.getBagAttributes();
        for (int i = 0; bagAttrs != null && i < bagAttrs.size(); ++i) {
            ByteArrayInputStream bis;
            ANY value;
            SET values;
            Attribute attr = (Attribute)bagAttrs.elementAt(i);
            OBJECT_IDENTIFIER oid = attr.getType();
            if (oid.equals(SafeBag.FRIENDLY_NAME)) {
                values = attr.getValues();
                value = (ANY)values.elementAt(0);
                bis = new ByteArrayInputStream(value.getEncoded());
                BMPString friendlyName = (BMPString)new BMPString.Template().decode(bis);
                keyInfo.setFriendlyName(friendlyName.toString());
                logger.debug("   Friendly name: " + keyInfo.getFriendlyName());
                continue;
            }
            if (oid.equals(SafeBag.LOCAL_KEY_ID)) {
                values = attr.getValues();
                value = (ANY)values.elementAt(0);
                bis = new ByteArrayInputStream(value.getEncoded());
                OCTET_STRING keyIdAsn1 = (OCTET_STRING)new OCTET_STRING.Template().decode(bis);
                byte[] keyID = keyIdAsn1.toByteArray();
                keyInfo.setID(keyID);
                continue;
            }
            logger.warn("   " + oid + ": " + attr.getValues());
        }
        return keyInfo;
    }

    public PKCS12CertInfo getCertInfo(SafeBag bag) throws Exception {
        PKCS12CertInfo certInfo = new PKCS12CertInfo();
        CertBag certBag = (CertBag)bag.getInterpretedBagContent();
        OCTET_STRING certStr = (OCTET_STRING)certBag.getInterpretedCert();
        byte[] x509cert = certStr.toByteArray();
        byte[] id = SafeBag.getLocalKeyIDFromCert(x509cert);
        certInfo.setID(id);
        logger.debug("   Certificate ID: " + Utils.HexEncode(id));
        X509CertImpl cert = new X509CertImpl(x509cert);
        certInfo.setCert(cert);
        X500Principal subjectDN = cert.getSubjectX500Principal();
        logger.debug("   Subject DN: " + subjectDN);
        SET bagAttrs = bag.getBagAttributes();
        for (int i = 0; bagAttrs != null && i < bagAttrs.size(); ++i) {
            ByteArrayInputStream bis;
            ANY value;
            SET values;
            Attribute attr = (Attribute)bagAttrs.elementAt(i);
            OBJECT_IDENTIFIER oid = attr.getType();
            if (oid.equals(SafeBag.FRIENDLY_NAME)) {
                values = attr.getValues();
                value = (ANY)values.elementAt(0);
                bis = new ByteArrayInputStream(value.getEncoded());
                BMPString friendlyName = (BMPString)new BMPString.Template().decode(bis);
                certInfo.setFriendlyName(friendlyName.toString());
                logger.debug("   Friendly name: " + certInfo.getFriendlyName());
                continue;
            }
            if (oid.equals(SafeBag.LOCAL_KEY_ID)) {
                values = attr.getValues();
                value = (ANY)values.elementAt(0);
                bis = new ByteArrayInputStream(value.getEncoded());
                OCTET_STRING keyIdAsn1 = (OCTET_STRING)new OCTET_STRING.Template().decode(bis);
                byte[] keyID = keyIdAsn1.toByteArray();
                certInfo.setKeyID(keyID);
                logger.debug("   Key ID: " + Utils.HexEncode(keyID));
                continue;
            }
            if (oid.equals(PKCS12.CERT_TRUST_FLAGS_OID) && this.trustFlagsEnabled) {
                values = attr.getValues();
                value = (ANY)values.elementAt(0);
                ByteArrayInputStream is = new ByteArrayInputStream(value.getEncoded());
                BMPString trustFlagsAsn1 = (BMPString)new BMPString.Template().decode(is);
                String trustFlags = trustFlagsAsn1.toString();
                certInfo.setTrustFlags(trustFlags);
                logger.debug("   Trust flags: " + trustFlags);
                continue;
            }
            logger.warn("   " + oid + ": " + attr.getValues());
        }
        if (certInfo.getFriendlyName() == null) {
            logger.debug("   Generating new friendly name");
            LdapName dn = new LdapName(subjectDN.getName());
            ArrayList<String> values = new ArrayList<String>();
            List<Rdn> rdns = dn.getRdns();
            for (int i = rdns.size() - 1; i >= 0; --i) {
                Rdn rdn = rdns.get(i);
                values.add(rdn.getValue().toString());
            }
            String friendlyName = StringUtils.join(values, (String)" - ");
            certInfo.setFriendlyName(friendlyName);
            logger.debug("   Friendly name: " + friendlyName);
        }
        return certInfo;
    }

    public void getKeyInfos(PKCS12 pkcs12, PFX pfx, Password password) throws Exception {
        logger.debug("Load encrypted private keys:");
        AuthenticatedSafes safes = pfx.getAuthSafes();
        for (int i = 0; i < safes.getSize(); ++i) {
            SEQUENCE contents = safes.getSafeContentsAt(password, i);
            for (int j = 0; j < contents.size(); ++j) {
                SafeBag bag = (SafeBag)contents.elementAt(j);
                OBJECT_IDENTIFIER oid = bag.getBagType();
                if (!oid.equals(SafeBag.PKCS8_SHROUDED_KEY_BAG)) continue;
                logger.debug(" - Private key:");
                PKCS12KeyInfo keyInfo = this.getKeyInfo(bag, password);
                pkcs12.addKeyInfo(keyInfo);
            }
        }
    }

    public void getCertInfos(PKCS12 pkcs12, PFX pfx, Password password) throws Exception {
        logger.debug("Loading certificates:");
        AuthenticatedSafes safes = pfx.getAuthSafes();
        for (int i = 0; i < safes.getSize(); ++i) {
            SEQUENCE contents = safes.getSafeContentsAt(password, i);
            for (int j = 0; j < contents.size(); ++j) {
                SafeBag bag = (SafeBag)contents.elementAt(j);
                OBJECT_IDENTIFIER oid = bag.getBagType();
                if (!oid.equals(SafeBag.CERT_BAG)) continue;
                logger.debug(" - Certificate:");
                PKCS12CertInfo certInfo = this.getCertInfo(bag);
                pkcs12.addCertInfo(certInfo, true);
            }
        }
    }

    public PKCS12 loadFromFile(String filename, Password password) throws Exception {
        logger.info("Loading PKCS #12 file");
        Path path = Paths.get(filename, new String[0]);
        byte[] b = Files.readAllBytes(path);
        return this.loadFromByteArray(b, password);
    }

    public PKCS12 loadFromByteArray(byte[] b, Password password) throws Exception {
        ByteArrayInputStream bis = new ByteArrayInputStream(b);
        PFX pfx = (PFX)new PFX.Template().decode(bis);
        PKCS12 pkcs12 = new PKCS12();
        StringBuffer reason = new StringBuffer();
        boolean valid = pfx.verifyAuthSafes(password, reason);
        if (!valid) {
            throw new Exception("Unable to validate PKCS #12 file: " + reason);
        }
        this.getKeyInfos(pkcs12, pfx, password);
        this.getCertInfos(pkcs12, pfx, password);
        return pkcs12;
    }

    public PKCS12 loadFromFile(String filename) throws Exception {
        return this.loadFromFile(filename, null);
    }

    public PrivateKey.Type getPrivateKeyType(PublicKey publicKey) {
        if (publicKey.getAlgorithm().equals("EC")) {
            return PrivateKey.Type.EC;
        }
        return PrivateKey.Type.RSA;
    }

    public PKCS12CertInfo getCertBySubjectDN(PKCS12 pkcs12, String subjectDN) throws CertificateException {
        for (PKCS12CertInfo certInfo : pkcs12.getCertInfos()) {
            X500Principal certSubjectDN = certInfo.getCert().getSubjectX500Principal();
            try {
                LdapName subjDn;
                LdapName certSubjdn = new LdapName(certSubjectDN.toString());
                if (!certSubjdn.equals(subjDn = new LdapName(subjectDN))) continue;
                return certInfo;
            }
            catch (InvalidNameException e) {
                return null;
            }
        }
        return null;
    }

    public void importKey(PKCS12 pkcs12, Password password, String nickname, PKCS12KeyInfo keyInfo) throws Exception {
        PKCS12CertInfo certInfo = pkcs12.getCertInfoByKeyID(keyInfo.getID());
        if (certInfo == null) {
            logger.debug("Private key has no certificate, ignore");
            return;
        }
        CryptoManager cm = CryptoManager.getInstance();
        CryptoToken token = cm.getInternalKeyStorageToken();
        PK11Store store = (PK11Store)token.getCryptoStore();
        X509CertImpl certImpl = certInfo.getCert();
        X509Certificate cert = cm.importCACertPackage(certImpl.getEncoded());
        PublicKey publicKey = cert.getPublicKey();
        byte[] epkiBytes = keyInfo.getEncryptedPrivateKeyInfoBytes();
        if (epkiBytes == null) {
            logger.debug("No EncryptedPrivateKeyInfo for key '" + keyInfo.getFriendlyName() + "'; skipping key");
        }
        try {
            store.importEncryptedPrivateKeyInfo(null, password, nickname, publicKey, epkiBytes);
        }
        catch (Exception e) {
            store.importEncryptedPrivateKeyInfo(new PasswordConverter(), password, nickname, publicKey, epkiBytes);
        }
        try {
            store.deleteCertOnly(cert);
        }
        catch (NoSuchItemOnTokenException noSuchItemOnTokenException) {
            // empty catch block
        }
    }

    public void storeCertIntoNSS(PKCS12 pkcs12, Password password, PKCS12CertInfo certInfo, boolean overwrite) throws Exception {
        X509Certificate cert;
        CryptoManager cm = CryptoManager.getInstance();
        CryptoToken ct = cm.getInternalKeyStorageToken();
        CryptoStore store = ct.getCryptoStore();
        String nickname = certInfo.getFriendlyName();
        for (X509Certificate cert2 : cm.findCertsByNickname(nickname)) {
            if (!overwrite) {
                return;
            }
            store.deleteCert(cert2);
        }
        X509CertImpl certImpl = certInfo.getCert();
        byte[] keyID = certInfo.getKeyID();
        if (keyID != null) {
            logger.debug("Importing private key for " + certInfo.getFriendlyName());
            PKCS12KeyInfo keyInfo = pkcs12.getKeyInfoByID(keyID);
            this.importKey(pkcs12, password, certInfo.getFriendlyName(), keyInfo);
            logger.debug("Importing user certificate " + certInfo.getFriendlyName());
            cert = cm.importUserCACertPackage(certImpl.getEncoded(), certInfo.getFriendlyName());
        } else {
            logger.debug("Importing CA certificate " + certInfo.getFriendlyName());
            cert = cm.importCACertPackage(certImpl.getEncoded());
        }
        String trustFlags = certInfo.getTrustFlags();
        if (trustFlags != null && this.trustFlagsEnabled) {
            PK11Cert pk11Cert = (PK11Cert)cert;
            pk11Cert.setTrustFlags(trustFlags);
        }
    }

    public void storeCertIntoNSS(PKCS12 pkcs12, Password password, String nickname, boolean overwrite) throws Exception {
        Collection<PKCS12CertInfo> certInfos = pkcs12.getCertInfosByFriendlyName(nickname);
        for (PKCS12CertInfo certInfo : certInfos) {
            this.storeCertIntoNSS(pkcs12, password, certInfo, overwrite);
        }
    }

    public void storeIntoNSS(PKCS12 pkcs12, Password password, boolean overwrite) throws Exception {
        logger.info("Storing data into NSS database");
        for (PKCS12CertInfo certInfo : pkcs12.getCertInfos()) {
            this.storeCertIntoNSS(pkcs12, password, certInfo, overwrite);
        }
    }
}

