// SPDX-License-Identifier: (BSD-3-Clause OR GPL-2.0-only)
/* Copyright(c) 2021 Intel Corporation */

#include <linux/delay.h>
#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/debugfs.h>
#include <linux/errno.h>
#include <linux/device.h>
#include <linux/io.h>
#include <linux/seq_file.h>
#include <adf_accel_devices.h>
#include <adf_common_drv.h>
#include "adf_c4xxx_inline.h"
#include "adf_c4xxx_hw_data.h"

#ifdef CONFIG_DEBUG_FS
struct reg_info {
	size_t offs;
	char *name;
} __packed;

static struct reg_info adf_inline_ctrl_regs[] = {
	{ADF_C4XXX_REG_SA_DB_CTRL,           "reg_sa_db_ctrl          "},
	{ADF_C4XXX_REG_SA_ENTRY_CTRL,        "reg_sa_entry_ctrl       "},
	{ADF_C4XXX_REG_SA_INLINE_CAPABILITY, "reg_sa_inline_capability"},
	{ADF_C4XXX_REG_SA_INLINE_ENABLE,     "reg_sa_inline_enable    "},
	{ADF_C4XXX_REG_SA_LINK_UP,           "reg_sa_link_up          "},
};

static struct reg_info adf_inline_stats_regs64[] = {
	{ADF_C4XXX_MAC_STAT_TX_OCTET,
		"mac_stat_tx_octect           "},
	{ADF_C4XXX_MAC_STAT_TX_FRAME,
		"mac_stat_tx_frame            "},
	{ADF_C4XXX_MAC_STAT_TX_BAD_FRAME,
		"mac_stat_tx_bad_frame        "},
	{ADF_C4XXX_MAC_STAT_TX_FCS_ERROR,
		"mac_stat_tx_fcs_error        "},
	{ADF_C4XXX_MAC_STAT_TX_64,
		"mac_stat_tx_64               "},
	{ADF_C4XXX_MAC_STAT_TX_65,
		"mac_stat_tx_65               "},
	{ADF_C4XXX_MAC_STAT_TX_128,
		"mac_stat_tx_128              "},
	{ADF_C4XXX_MAC_STAT_TX_256,
		"mac_stat_tx_256              "},
	{ADF_C4XXX_MAC_STAT_TX_512,
		"mac_stat_tx_512              "},
	{ADF_C4XXX_MAC_STAT_TX_1024,
		"mac_stat_tx_1024             "},
	{ADF_C4XXX_MAC_STAT_TX_1519,
		"mac_stat_tx_1519             "},
	{ADF_C4XXX_MAC_STAT_TX_JABBER,
		"mac_stat_tx_jabber           "},
	{ADF_C4XXX_MAC_STAT_RX_OCTET,
		"mac_stat_rx_octect           "},
	{ADF_C4XXX_MAC_STAT_RX_FRAME,
		"mac_stat_rx_frame            "},
	{ADF_C4XXX_MAC_STAT_RX_BAD_FRAME,
		"mac_stat_rx_bad_frame        "},
	{ADF_C4XXX_MAC_STAT_RX_FCS_ERROR,
		"mac_stat_rx_fcs_error        "},
	{ADF_C4XXX_MAC_STAT_RX_64,
		"mac_stat_rx_64               "},
	{ADF_C4XXX_MAC_STAT_RX_65,
		"mac_stat_rx_65               "},
	{ADF_C4XXX_MAC_STAT_RX_128,
		"mac_stat_rx_128              "},
	{ADF_C4XXX_MAC_STAT_RX_256,
		"mac_stat_rx_256              "},
	{ADF_C4XXX_MAC_STAT_RX_512,
		"mac_stat_rx_512              "},
	{ADF_C4XXX_MAC_STAT_RX_1024,
		"mac_stat_rx_1024             "},
	{ADF_C4XXX_MAC_STAT_RX_1519,
		"mac_stat_rx_1519             "},
	{ADF_C4XXX_MAC_STAT_RX_OVERSIZE,
		"mac_stat_rx_oversize         "},
	{ADF_C4XXX_MAC_STAT_RX_JABBER,
		"mac_stat_rx_jabber           "},
};

static struct reg_info adf_inline_stats_regs32[] = {
	{ADF_C4XXX_IC_PAR_IPSEC_DESC_COUNT,
		"ic_par_ipsec_desc_count      "},
	{ADF_C4XXX_IC_PAR_MIXED_DESC_COUNT,
		"ic_par_mixed_desc_count      "},
	{ADF_C4XXX_IC_PAR_FULLY_CLEAR_DESC_COUNT,
		"ic_par_fully_clear_desc_count"},
	{ADF_C4XXX_IC_PAR_CLR_COUNT,
		"ic_par_clr_count             "},
	{ADF_C4XXX_IC_CTPB_PKT_COUNT,
		"ic_ctpb_pkt_count            "},
	{ADF_C4XXX_RB_DATA_COUNT,
		"rb_data_count                "},
	{ADF_C4XXX_IC_CLEAR_DESC_COUNT,
		"ic_clear_desc_count          "},
	{ADF_C4XXX_IC_IPSEC_DESC_COUNT,
		"ic_ipsec_desc_count          "},
};

static struct reg_info adf_congest_mngt_common_regs[] = {
	{ADF_C4XXX_BB_BEHTHRESH_OFFSET,		"ic_bb_behthresh"},
	{ADF_C4XXX_BB_BELTHRESH_OFFSET,		"ic_bb_belthresh"},
	{ADF_C4XXX_BEWIP_THRESH_OFFSET,		"ic_bewip_thresh"},
	{ADF_C4XXX_CTPB_THRESH_OFFSET,		"ic_ctpb_thresh"},
	{ADF_C4XXX_CIRQ_OFFSET,			"ic_cirq"},
};

static struct reg_info adf_congest_mngt_domain_regs[] = {
	{ADF_C4XXX_BB_FCHTHRESH_OFFSET,		"ic_bb_fchthresh"},
	{ADF_C4XXX_BB_FCLTHRESH_OFFSET,		"ic_bb_fclthresh"},
	{ADF_C4XXX_Q2MEMAP_OFFSET,		"ic_q2memap"},
};

static DEFINE_MUTEX(egress_read_lock);
static DEFINE_MUTEX(ingress_read_lock);

static void adf_print_congest_mngt_regs(struct seq_file *sfile,
					const u32 offset)
{
	struct adf_accel_dev *accel_dev = sfile->private;
	u32 num_regs, i, j, threshold_offset, regs_offset;
	struct adf_bar *aram_bar = &GET_BARS(accel_dev)[ADF_C4XXX_SRAM_BAR];
	void __iomem *csr = aram_bar->virt_addr;

	num_regs = ARRAY_SIZE(adf_congest_mngt_domain_regs);
	for (i = 0; i < num_regs; i++) {
		for (j = 0; j < ADF_C4XXX_NUM_CONGEST_DOMAINS; j++) {
			if (i == (num_regs - 1)) {
				threshold_offset =
					ADF_C4XXX_NEXT_Q2MEMAP_OFFSET;
			} else {
				threshold_offset =
					ADF_C4XXX_NEXT_FCTHRESH_OFFSET;
			}
			regs_offset = adf_congest_mngt_domain_regs[i].offs
					+ offset;
			seq_printf(sfile, "%s[%u] = 0x%08x\n",
				   adf_congest_mngt_domain_regs[i]
				   .name,
				   j,
				   ADF_CSR_RD(csr, (regs_offset
					      + threshold_offset * j)));
		}
	}

	num_regs = ARRAY_SIZE(adf_congest_mngt_common_regs);
	for (i = 0; i < num_regs; i++) {
		regs_offset = adf_congest_mngt_common_regs[i].offs
				+ offset;
		seq_printf(sfile, "%s = 0x%08x\n",
			   adf_congest_mngt_common_regs[i].name,
			   ADF_CSR_RD(csr, regs_offset));
	}
	seq_puts(sfile, "\n");
}

static void adf_print_inline_ctrl_regs(struct seq_file *sfile)
{
	struct adf_accel_dev *accel_dev = sfile->private;
	u32 num_regs = ARRAY_SIZE(adf_inline_ctrl_regs);
	u32 i = 0;
	struct adf_bar *aram_bar = &GET_BARS(accel_dev)[ADF_C4XXX_SRAM_BAR];
	void __iomem *csr = aram_bar->virt_addr;

	for (i = 0; i < num_regs; i++) {
		seq_printf(sfile, "%s = 0x%08x\n",
			   adf_inline_ctrl_regs[i].name,
			   ADF_CSR_RD(csr, adf_inline_ctrl_regs[i].offs));
	}
	seq_puts(sfile, "\n");
}

static void adf_update_mac_cfg_regs(struct adf_accel_dev *accel_dev,
				    const u32 mac_cfg_offset,
				    const bool unlock)
{
	struct adf_bar *aram_bar = &GET_BARS(accel_dev)[ADF_C4XXX_SRAM_BAR];
	void __iomem *csr = aram_bar->virt_addr;
	u64 reg_val;

	/* Enable statistics requests */
	reg_val = ADF_CSR_RD64(csr, mac_cfg_offset);
	if (unlock) {
		/* if STATS_REQUEST bit is set, we must reset
		 * it to create a 0 to 1 transition
		 */
		if (reg_val & ADF_C4XXX_STATS_REQUEST_ENABLED) {
			/* Reset STATS_REQUEST bit */
			reg_val &= ADF_C4XXX_STATS_REQUEST_DISABLED;
			ADF_CSR_WR64(csr, mac_cfg_offset, reg_val);
		}
		reg_val |= ADF_C4XXX_STATS_REQUEST_ENABLED;
	} else {
		reg_val &= ADF_C4XXX_STATS_REQUEST_DISABLED;
	}
	ADF_CSR_WR64(csr, mac_cfg_offset, reg_val);
}

static int adf_wait_for_stats_ready(struct adf_accel_dev *accel_dev,
				    const u32 mac_ip_offset)
{
	struct adf_bar *aram_bar = &GET_BARS(accel_dev)[ADF_C4XXX_SRAM_BAR];
	void __iomem *csr = aram_bar->virt_addr;
	u64 reg_val;
	u32 num_read = 0;

	do {
		reg_val = ADF_CSR_RD64(csr, mac_ip_offset);
		/* Wait for STATS_READY bit to be asserted */
		if (reg_val & ADF_C4XXX_MAC_STATS_READY) {
			/* Clear the interrupt setting STATS_READY bit */
			reg_val |= ADF_C4XXX_MAC_STATS_READY;
			ADF_CSR_WR64(csr, mac_ip_offset, reg_val);
			return 0;
		}
		msleep(ADF_C4XXX_MAC_STATS_POLLING_INTERVAL);
		num_read++;
	} while (num_read < ADF_C4XXX_MAX_NUM_STAT_READY_READS);
	dev_err(&GET_DEV(accel_dev),
		"Could not read STATS registers! Timeout occurred.\n");
	return -ETIME;
}

static void print_stats(struct seq_file *sfile,
			struct reg_info *stats_regs64,
			const u32 num_regs64,
			struct reg_info *stats_regs32,
			const u32 num_regs32,
			const u32 inline_offset)
{
	u32 i;
	struct adf_accel_dev *accel_dev = sfile->private;
	struct adf_bar *aram_bar = &GET_BARS(accel_dev)[ADF_C4XXX_SRAM_BAR];
	void __iomem *csr = aram_bar->virt_addr;

	/* Display all the 64bit registers */
	for (i = 0; i < num_regs64; i++) {
		seq_printf(sfile, "%s = %llu\n",
			   stats_regs64[i].name,
			   (unsigned long long)ADF_CSR_RD64(csr,
				inline_offset +
				stats_regs64[i].offs));
	}
	/* Display all the 32bit registers */
	for (i = 0; i < num_regs32; i++) {
		seq_printf(sfile, "%s = %d\n",
			   stats_regs32[i].name,
			   ADF_CSR_RD(csr,
				      inline_offset +
				      stats_regs32[i].offs));
	}
}

static void adf_print_inline_stats_regs(struct seq_file *sfile,
					const u32 offset)
{
	struct adf_accel_dev *accel_dev = sfile->private;

	/* Unlock MAC STATS registers */
	adf_update_mac_cfg_regs(accel_dev,
				ADF_C4XXX_MAC_CFG + offset,
				ADF_C4XXX_UNLOCK);
	/* Wait for new data to be loaded in STATS registers */
	if (adf_wait_for_stats_ready(accel_dev,
				     ADF_C4XXX_MAC_IP + offset)) {
		/* Lock MAC STATS registers */
		adf_update_mac_cfg_regs(accel_dev,
					ADF_C4XXX_MAC_CFG + offset,
					ADF_C4XXX_LOCK);
		dev_warn(&GET_DEV(accel_dev),
			 "IPSec egress stats not available.\n");
		return;
	}

	/* Print registers */
	print_stats(sfile,
		    adf_inline_stats_regs64,
		    ARRAY_SIZE(adf_inline_stats_regs64),
		    adf_inline_stats_regs32,
		    ARRAY_SIZE(adf_inline_stats_regs32),
		    offset);

	/* Lock MAC STATS registers */
	adf_update_mac_cfg_regs(accel_dev,
				ADF_C4XXX_MAC_CFG + offset,
				ADF_C4XXX_LOCK);

	seq_puts(sfile, "\n");
}

static void *adf_ingress_start(struct seq_file *sfile, loff_t *pos)
{
	mutex_lock(&ingress_read_lock);

	if (*pos == 0)
		return SEQ_START_TOKEN;
	else
		return NULL;
}

static int adf_ingress_show(struct seq_file *sfile, void *v)
{
	if (v == SEQ_START_TOKEN) {
		/* Display control registers */
		adf_print_inline_ctrl_regs(sfile);
		/* Display congestion profile regsiters */
		adf_print_congest_mngt_regs(sfile,
					    ADF_C4XXX_INLINE_INGRESS_OFFSET);
		/* Display inline ingress status registers */
		adf_print_inline_stats_regs(sfile,
					    ADF_C4XXX_INLINE_INGRESS_OFFSET);
	}

	return 0;
}

static void *adf_ingress_next(struct seq_file *sfile, void *v, loff_t *pos)
{
	return NULL;
}

static void adf_ingress_stop(struct seq_file *sfile, void *v)
{
	mutex_unlock(&ingress_read_lock);
}

static const struct seq_operations adf_ingress_sops = {
	.start = adf_ingress_start,
	.next = adf_ingress_next,
	.stop = adf_ingress_stop,
	.show = adf_ingress_show
};

static void *adf_egress_start(struct seq_file *sfile, loff_t *pos)
{
	mutex_lock(&egress_read_lock);

	if (*pos == 0)
		return SEQ_START_TOKEN;
	else
		return NULL;
}

static int adf_egress_show(struct seq_file *sfile, void *v)
{
	if (v == SEQ_START_TOKEN) {
		/* Display control registers */
		adf_print_inline_ctrl_regs(sfile);
		/* Display congestion profile regsiters */
		adf_print_congest_mngt_regs(sfile,
					    ADF_C4XXX_INLINE_EGRESS_OFFSET);
		/* Display inline egress status registers */
		adf_print_inline_stats_regs(sfile,
					    ADF_C4XXX_INLINE_EGRESS_OFFSET);
	}

	return 0;
}

static void *adf_egress_next(struct seq_file *sfile, void *v, loff_t *pos)
{
	return NULL;
}

static void adf_egress_stop(struct seq_file *sfile, void *v)
{
	mutex_unlock(&egress_read_lock);
}

static const struct seq_operations adf_egress_sops = {
	.start = adf_egress_start,
	.next = adf_egress_next,
	.stop = adf_egress_stop,
	.show = adf_egress_show
};

static int adf_egress_open(struct inode *inode, struct file *file)
{
	int ret = seq_open(file, &adf_egress_sops);

	if (!ret) {
		struct seq_file *seq_f = file->private_data;

		seq_f->private = inode->i_private;
	}
	return ret;
}

static const struct file_operations adf_egress_debug_fops = {
	.owner = THIS_MODULE,
	.open = adf_egress_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = seq_release
};

static int adf_ingress_open(struct inode *inode, struct file *file)
{
	int ret = seq_open(file, &adf_ingress_sops);

	if (!ret) {
		struct seq_file *seq_f = file->private_data;

		seq_f->private = inode->i_private;
	}
	return ret;
}

static const struct file_operations adf_ingress_debug_fops = {
	.owner = THIS_MODULE,
	.open = adf_ingress_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = seq_release
};

static int adf_add_debugfs_egress_ingress(struct adf_accel_dev *accel_dev)
{
	struct dentry *debugfs_current_reg = NULL;

	/* Create Ingress debug file */
	debugfs_current_reg =
		 debugfs_create_file("ingress",
				     0400,
				     accel_dev->debugfs_inline_dir,
				     accel_dev,
				     &adf_ingress_debug_fops);
	if (!debugfs_current_reg) {
		dev_err(&GET_DEV(accel_dev),
			"Could not create debug ingress entry.\n");
		return -EFAULT;
	}

	/* Create Egress debug file */
	debugfs_current_reg =
		debugfs_create_file("egress",
				    0400,
				    accel_dev->debugfs_inline_dir,
				    accel_dev,
				    &adf_egress_debug_fops);
	if (!debugfs_current_reg) {
		adf_exit_debugfs_inline_c4xxx(accel_dev);
		dev_err(&GET_DEV(accel_dev),
			"Could not create debug egress entry.\n");
		return -EFAULT;
	}

	return 0;
}

int adf_init_debugfs_inline_c4xxx(struct adf_accel_dev *accel_dev)
{
	int ret = 0;

	/* Create device debug top level debugfs entry */
	accel_dev->debugfs_inline_dir =
		debugfs_create_dir("inline", accel_dev->debugfs_dir);
	if (!accel_dev->debugfs_inline_dir) {
		dev_err(&GET_DEV(accel_dev),
			"Could not create inline directory.\n");
		return -EFAULT;
	}

	/* Add list ingress and egress entries. */
	ret = adf_add_debugfs_egress_ingress(accel_dev);
	if (ret) {
		adf_exit_debugfs_inline_c4xxx(accel_dev);
		dev_err(&GET_DEV(accel_dev), "Could not create debugfs egress/ingress entries\n");
		return ret;
	}
	return 0;
}

void adf_exit_debugfs_inline_c4xxx(struct adf_accel_dev *accel_dev)
{
	/* Delete "hw_debug" registers */
	debugfs_remove_recursive(accel_dev->debugfs_inline_dir);
	accel_dev->debugfs_inline_dir = NULL;
}

#else
int adf_init_debugfs_inline_c4xxx(struct adf_accel_dev *accel_dev)
{
	return 0;
}

void adf_exit_debugfs_inline_c4xxx(struct adf_accel_dev *accel_dev)
{
}
#endif
