// Copyright 2020 The TensorFlow Runtime Authors
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

// This file implements MLIR operation functions for the test_kernels library.

#include "tfrt/test_kernels/opdefs/test_kernels.h"

#include "mlir/Dialect/Func/IR/FuncOps.h"
#include "mlir/IR/Builders.h"
#include "mlir/IR/BuiltinOps.h"
#include "mlir/IR/BuiltinTypes.h"
#include "mlir/IR/OpImplementation.h"
#include "mlir/IR/TypeUtilities.h"
#include "tfrt/basic_kernels/opdefs/tfrt_base.h"
#include "tfrt/basic_kernels/opdefs/types.h"

namespace tfrt {
namespace test {

namespace {
void createArgs(ArrayRef<OpAsmParser::UnresolvedOperand> operands,
                ArrayRef<Type> types,
                SmallVector<OpAsmParser::Argument> &args) {
  for (auto argAndType : llvm::zip(operands, types)) {
    auto &arg = args.emplace_back();
    arg.ssaName = std::get<0>(argAndType);
    arg.type = std::get<1>(argAndType);
  }
}
}  // namespace

//===----------------------------------------------------------------------===//
// TestDialect Dialect
//===----------------------------------------------------------------------===//

TestDialect::TestDialect(MLIRContext *context)
    : Dialect(/*name=*/"tfrt_test", context, TypeID::get<TestDialect>()) {
  context->getOrLoadDialect<compiler::TFRTDialect>();

  allowUnknownTypes();
  allowUnknownOperations();

  addOperations<
#define GET_OP_LIST
#include "tfrt/test_kernels/opdefs/test_kernels_opdefs.cpp.inc"
      >();
}

// Verify that the specified region contains a tfrt.return operation with the
// specified type list and emit an error if not.
template <typename ResultTypeContainer>
static LogicalResult checkTFRTReturn(Operation *op, Region *region,
                                     ResultTypeContainer result_types) {
  assert(std::distance(region->begin(), region->end()) == 1 &&
         "verifier should already check region size");
  auto *block = &region->front();

  if (block->empty() || block->back().getName().getStringRef() != "tfrt.return")
    return op->emitOpError("expected tfrt.return in body");

  if (!std::equal(block->back().getOperandTypes().begin(),
                  block->back().getOperandTypes().end(), result_types.begin(),
                  result_types.end()))
    return block->back().emitOpError()
           << "operand types don't match '" << op->getName() << "' result";

  return success();
}

//===----------------------------------------------------------------------===//
// DoAsyncOp
//===----------------------------------------------------------------------===//

ParseResult DoAsyncOp::parse(OpAsmParser &parser, OperationState &result) {
  SmallVector<OpAsmParser::UnresolvedOperand, 4> operands;
  if (parser.parseOperandList(operands)) return failure();

  if (succeeded(parser.parseOptionalKeyword("attributes"))) {
    if (parser.parseOptionalAttrDict(result.attributes)) return failure();
  }

  FunctionType types;
  llvm::SMLoc type_loc = parser.getCurrentLocation();
  if (parser.parseColonType(types) ||
      parser.addTypesToList(types.getResults(), result.types))
    return failure();

  // Parse the body region.
  if (parser.resolveOperands(operands, types.getInputs(), type_loc,
                             result.operands))
    return failure();
  SmallVector<OpAsmParser::Argument> args;
  createArgs(operands, types.getInputs(), args);
  Region *body = result.addRegion();
  return parser.parseRegion(*body, args,
                            /*enableNameShadowing=*/true);
}

void DoAsyncOp::print(OpAsmPrinter &p) {
  p << " ";
  p.printOperands(getOperands());
  if (!(*this)->getAttrs().empty()) {
    p << " attributes ";
    p.printOptionalAttrDict((*this)->getAttrs());
  }
  p << " : (";
  interleaveComma(getOperandTypes(), p);
  p << ") -> (";
  interleaveComma(getResultTypes(), p);
  p << ") ";

  // Reuse the argument names provided to the op for the bbarg names within
  // the region.
  p.shadowRegionArgs(getRegion(), getOperands());
  p << ' ';
  p.printRegion(getRegion(), /*printEntryBlockArgs=*/false);
}

LogicalResult DoAsyncOp::verify() {
  DoAsyncOp op = *this;
  return checkTFRTReturn(op, &op.getRegion(), op.getResultTypes());
}

//===----------------------------------------------------------------------===//
// BenchmarkOp
//===----------------------------------------------------------------------===//

// Parse the BenchmarkOp in the following format
// tfrt_test.benchmark "add.i32"(%c : i32, %d : f32)
//       max_count = 100, duration_secs = 1 {
// ...
// }

ParseResult BenchmarkOp::parse(OpAsmParser &parser, OperationState &result) {
  StringAttr nameAttr;
  if (parser.parseAttribute(nameAttr, "name", result.attributes))
    return failure();

  // Parse the operands, e.g. (%c : i32, %d : f32)
  if (parser.parseLParen()) return failure();

  SmallVector<OpAsmParser::UnresolvedOperand, 4> operands;
  SmallVector<Type, 4> types;
  llvm::SMLoc type_loc = parser.getCurrentLocation();

  if (parser.parseOptionalRParen()) {
    // Parse non-empty operands
    do {
      // Parse %c : i32,
      OpAsmParser::UnresolvedOperand operand;
      Type type;

      if (parser.parseOperand(operand) || parser.parseColonType(type))
        return failure();

      operands.push_back(operand);
      types.push_back(type);

    } while (succeeded(parser.parseOptionalComma()));

    if (parser.parseRParen()) return failure();
  }

  if (parser.resolveOperands(operands, types, type_loc, result.operands))
    return failure();

  auto chainType = compiler::ChainType::get(result.getContext());
  if (parser.addTypeToList(chainType, result.types)) return failure();

  auto parseIntegerKeywordAttr = [&]() -> ParseResult {
    StringRef attr;
    Attribute resultAttr;

    return failure(parser.parseKeyword(&attr) || parser.parseEqual() ||
                   parser.parseAttribute(resultAttr,
                                         parser.getBuilder().getIntegerType(32),
                                         attr, result.attributes));
  };

  // Parse the keyword attribute, e.g. max_count = 100, duration_secs = 1
  do {
    if (parseIntegerKeywordAttr()) return failure();
  } while (succeeded(parser.parseOptionalComma()));

  auto setDefaultAttrIfUnset = [&](const char *attr_name, int value) {
    bool found = llvm::any_of(result.attributes,
                              [attr_name](const NamedAttribute &attr) {
                                return attr.getName() == attr_name;
                              });
    if (!found) {
      IntegerAttr default_val = parser.getBuilder().getI32IntegerAttr(value);
      result.addAttribute(attr_name, default_val);
    }
  };

  // Set the default attribute num_warmup_runs to 1 if unset
  setDefaultAttrIfUnset("num_warmup_runs", 1);

  Region *target = result.addRegion();
  SmallVector<OpAsmParser::Argument> args;
  createArgs(operands, types, args);
  return parser.parseRegion(*target, args, /*enableNameShadowing=*/true);
}

// Print the BenchmarkOp in the following format
// tfrt_test.benchmark "add.i32"(%c : i32, %d : f32)
//       max_count = 100, duration_secs = 1 {
// ...
// }
void BenchmarkOp::print(OpAsmPrinter &p) {
  p << " ";

  // Print the name attribute, e.g "add.i32"
  auto name_attr = (*this)->getAttr("name");
  p << name_attr;

  // Print the operands and types, e.g. (%c : i32, %d : f32)
  p << '(';
  llvm::interleaveComma(llvm::zip(getOperands(), getOperandTypes()), p,
                        [&](const auto &it) {
                          p << std::get<0>(it) << " : " << std::get<1>(it);
                        });
  p << ") ";

  bool need_comma = false;

  // Print the attributes, e.g. max_count = 100, duration_secs = 1
  for (auto &name_attr : (*this)->getAttrs()) {
    auto id = name_attr.getName().getValue();
    if (id == "name") continue;

    if (need_comma) p << ", ";

    auto attr = name_attr.getValue();

    p << id << " = ";
    if (auto int_attr = attr.dyn_cast<IntegerAttr>()) {
      int_attr.getValue().print(p.getStream(), /*isSigned=*/false);
    } else {
      emitOpError("Unexpected attribute");
    }

    need_comma = true;
  }

  p << ' ';

  // Print the region
  // Reuse the argument names provided to the op for the bbarg names within
  // the region.
  p.shadowRegionArgs(getRegion(), getOperands());
  p << ' ';
  p.printRegion(getRegion(), /*printEntryBlockArgs=*/false);
}

LogicalResult BenchmarkOp::verify() {
  BenchmarkOp op = *this;
  // Verify that the target benchmark region has exactly one return value.
  auto &region = op.getRegion();
  auto &last_op = region.front().back();
  if (last_op.getName().getStringRef() != "tfrt.return") {
    return op.emitOpError("missing return statement");
  }
  if (last_op.getNumOperands() != 1) {
    return op.emitOpError(
        "incorrect number of return values. One return value is expected");
  }

  return success();
}

//===----------------------------------------------------------------------===//
// SyncBenchmarkOp
//===----------------------------------------------------------------------===//

// Parse the SyncBenchmarkOp in the following format
//
// tfrt_test.sync_benchmark @fibonacci.i32()
//       duration_secs = 1, max_count = 100, num_warmup_runs = 10

ParseResult SyncBenchmarkOp::parse(OpAsmParser &parser,
                                   OperationState &result) {
  SymbolRefAttr targetFnAttr;
  if (parser.parseAttribute(targetFnAttr, "target_fn", result.attributes))
    return failure();

  // Parse the operands, e.g. (%c : i32, %d : f32)
  if (parser.parseLParen()) return failure();

  SmallVector<OpAsmParser::UnresolvedOperand, 4> operands;
  SmallVector<Type, 4> types;
  llvm::SMLoc type_loc = parser.getCurrentLocation();

  if (parser.parseOptionalRParen()) {
    // Parse non-empty operands
    do {
      // Parse %c : i32,
      OpAsmParser::UnresolvedOperand operand;
      Type type;

      if (parser.parseOperand(operand) || parser.parseColonType(type))
        return failure();

      operands.push_back(operand);
      types.push_back(type);

    } while (succeeded(parser.parseOptionalComma()));

    if (parser.parseRParen()) return failure();
  }

  if (parser.resolveOperands(operands, types, type_loc, result.operands))
    return failure();

  auto parseIntegerKeywordAttr = [&]() -> ParseResult {
    StringRef attr;
    Attribute resultAttr;

    return failure(parser.parseKeyword(&attr) || parser.parseEqual() ||
                   parser.parseAttribute(resultAttr,
                                         parser.getBuilder().getIntegerType(32),
                                         attr, result.attributes));
  };

  // Parse the keyword attribute, e.g. max_count = 100, duration_secs = 1
  do {
    if (parseIntegerKeywordAttr()) return failure();
  } while (succeeded(parser.parseOptionalComma()));

  auto setDefaultAttrIfUnset = [&](const char *attr_name, int value) {
    bool found = llvm::any_of(result.attributes,
                              [attr_name](const NamedAttribute &attr) {
                                return attr.getName() == attr_name;
                              });
    if (!found) {
      IntegerAttr default_val = parser.getBuilder().getI32IntegerAttr(value);
      result.addAttribute(attr_name, default_val);
    }
  };

  // Set the default attribute num_warmup_runs to 1 if unset
  setDefaultAttrIfUnset("num_warmup_runs", 1);

  return success();
}

// Print the SyncBenchmarkOp in the following format
// tfrt_test.sync_benchmark @fibonacci.i32()
//       max_count = 100, duration_secs = 1
void SyncBenchmarkOp::print(OpAsmPrinter &p) {
  p << " ";

  // Print the target benchmark function
  p << (*this)->getAttr("target_fn");

  // Print the operands and types, e.g. (%c : i32, %d : f32)
  p << '(';
  llvm::interleaveComma(llvm::zip(getOperands(), getOperandTypes()), p,
                        [&](const auto &it) {
                          p << std::get<0>(it) << " : " << std::get<1>(it);
                        });
  p << ") ";

  bool need_comma = false;

  // Print the attributes, e.g. max_count = 100, duration_secs = 1
  for (auto &name_attr : (*this)->getAttrs()) {
    auto id = name_attr.getName().getValue();
    if (id == "target_fn") continue;

    if (need_comma) p << ", ";

    auto attr = name_attr.getValue();

    p << id << " = ";
    if (auto int_attr = attr.dyn_cast<IntegerAttr>()) {
      int_attr.getValue().print(p.getStream(), /*isSigned=*/false);
    } else {
      emitOpError("Unexpected attribute");
    }

    need_comma = true;
  }
}

LogicalResult SyncBenchmarkOp::verify() {
  SyncBenchmarkOp op = *this;
  auto fnAttr = op->getAttrOfType<FlatSymbolRefAttr>("target_fn");
  if (!fnAttr)
    return op.emitOpError("requires a 'target_fn' symbol reference attribute");

  auto fn = op->getParentOfType<ModuleOp>().lookupSymbol<func::FuncOp>(
      fnAttr.getValue());
  if (!fn)
    return op.emitOpError() << "'" << fnAttr.getValue()
                            << "' does not reference a valid function";

  // Verify that the operand and result types match the callee.
  auto fnType = fn.getFunctionType();
  if (fnType.getNumInputs() != op.getNumOperands())
    return op.emitOpError("incorrect number of operands for callee");

  for (unsigned i = 0, e = fnType.getNumInputs(); i != e; ++i)
    if (op.getOperand(i).getType() != fnType.getInput(i))
      return op.emitOpError("operand type mismatch");

  if (fnType.getNumResults() != 0)
    return op.emitOpError("Target benchmark function must return zero value.");

  return success();
}

}  // namespace test
}  // namespace tfrt

//===----------------------------------------------------------------------===//
// TableGen'd op method definitions
//===----------------------------------------------------------------------===//

#define GET_OP_CLASSES
#include "tfrt/test_kernels/opdefs/test_kernels_opdefs.cpp.inc"
