/*
 * Copyright (C) 2013-2018 Red Hat
 *
 * Author: Nikos Mavrogiannopoulos
 *
 * This file is part of GnuTLS.
 *
 * The GnuTLS is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>
 *
 */

#include "gnutls_int.h"
#include "errors.h"
#include <cipher_int.h>
#include <datum.h>
#include <gnutls/crypto.h>
#include <gnutls/self-test.h>
#include "errors.h"
#include <random.h>
#include <crypto.h>
#include <nettle/sha3.h>

#define STR(tag, tag_size, val) \
	.tag = (uint8_t*)val, \
	.tag_size = (sizeof(val)-1)

#define V(x) (x), (sizeof(x)/sizeof(x[0]))

/* This does check the AES and SHA implementation against test vectors.
 * This should not run under valgrind in order to use the native
 * cpu instructions (AES-NI or padlock).
 */

struct cipher_vectors_st {
	const uint8_t *key;
	unsigned int key_size;

	const uint8_t *plaintext;
	unsigned int plaintext_size;
	const uint8_t *ciphertext;	/* also of plaintext_size */

	const uint8_t *iv;
	unsigned int iv_size;

	const uint8_t *internal_iv;
	unsigned int internal_iv_size;
};

struct cipher_aead_vectors_st {
	unsigned compat_apis;
	const uint8_t *key;
	unsigned int key_size;

	const uint8_t *auth;
	unsigned int auth_size;

	const uint8_t *plaintext;
	unsigned int plaintext_size;
	const uint8_t *ciphertext;	/* also of plaintext_size */

	unsigned int iv_size;
	const uint8_t *iv;
	const uint8_t *tag;
	unsigned tag_size;
	unsigned tag_prepended;
};

const struct cipher_aead_vectors_st chacha_poly1305_vectors[] = {
	{
	 .compat_apis = 1,
	 STR(key, key_size,
	     "\x1c\x92\x40\xa5\xeb\x55\xd3\x8a\xf3\x33\x88\x86\x04\xf6\xb5\xf0\x47\x39\x17\xc1\x40\x2b\x80\x09\x9d\xca\x5c\xbc\x20\x70\x75\xc0"),
	 .auth = (void*)"\xf3\x33\x88\x86\x00\x00\x00\x00\x00\x00\x4e\x91",
	 .auth_size = 12,
	 .plaintext = (void*)"\x49\x6e\x74\x65\x72\x6e\x65\x74\x2d\x44\x72\x61\x66\x74\x73\x20\x61\x72\x65\x20\x64\x72\x61\x66\x74\x20\x64\x6f\x63\x75\x6d\x65\x6e\x74\x73\x20\x76\x61\x6c\x69\x64\x20\x66\x6f\x72\x20\x61\x20\x6d\x61\x78\x69\x6d\x75\x6d\x20\x6f\x66\x20\x73\x69\x78\x20\x6d\x6f\x6e\x74\x68\x73\x20\x61\x6e\x64\x20\x6d\x61\x79\x20\x62\x65\x20\x75\x70\x64\x61\x74\x65\x64\x2c\x20\x72\x65\x70\x6c\x61\x63\x65\x64\x2c\x20\x6f\x72\x20\x6f\x62\x73\x6f\x6c\x65\x74\x65\x64\x20\x62\x79\x20\x6f\x74\x68\x65\x72\x20\x64\x6f\x63\x75\x6d\x65\x6e\x74\x73\x20\x61\x74\x20\x61\x6e\x79\x20\x74\x69\x6d\x65\x2e\x20\x49\x74\x20\x69\x73\x20\x69\x6e\x61\x70\x70\x72\x6f\x70\x72\x69\x61\x74\x65\x20\x74\x6f\x20\x75\x73\x65\x20\x49\x6e\x74\x65\x72\x6e\x65\x74\x2d\x44\x72\x61\x66\x74\x73\x20\x61\x73\x20\x72\x65\x66\x65\x72\x65\x6e\x63\x65\x20\x6d\x61\x74\x65\x72\x69\x61\x6c\x20\x6f\x72\x20\x74\x6f\x20\x63\x69\x74\x65\x20\x74\x68\x65\x6d\x20\x6f\x74\x68\x65\x72\x20\x74\x68\x61\x6e\x20\x61\x73\x20\x2f\xe2\x80\x9c\x77\x6f\x72\x6b\x20\x69\x6e\x20\x70\x72\x6f\x67\x72\x65\x73\x73\x2e\x2f\xe2\x80\x9d",
	 .plaintext_size = 265,
	 .ciphertext = (void*)"\x64\xa0\x86\x15\x75\x86\x1a\xf4\x60\xf0\x62\xc7\x9b\xe6\x43\xbd\x5e\x80\x5c\xfd\x34\x5c\xf3\x89\xf1\x08\x67\x0a\xc7\x6c\x8c\xb2\x4c\x6c\xfc\x18\x75\x5d\x43\xee\xa0\x9e\xe9\x4e\x38\x2d\x26\xb0\xbd\xb7\xb7\x3c\x32\x1b\x01\x00\xd4\xf0\x3b\x7f\x35\x58\x94\xcf\x33\x2f\x83\x0e\x71\x0b\x97\xce\x98\xc8\xa8\x4a\xbd\x0b\x94\x81\x14\xad\x17\x6e\x00\x8d\x33\xbd\x60\xf9\x82\xb1\xff\x37\xc8\x55\x97\x97\xa0\x6e\xf4\xf0\xef\x61\xc1\x86\x32\x4e\x2b\x35\x06\x38\x36\x06\x90\x7b\x6a\x7c\x02\xb0\xf9\xf6\x15\x7b\x53\xc8\x67\xe4\xb9\x16\x6c\x76\x7b\x80\x4d\x46\xa5\x9b\x52\x16\xcd\xe7\xa4\xe9\x90\x40\xc5\xa4\x04\x33\x22\x5e\xe2\x82\xa1\xb0\xa0\x6c\x52\x3e\xaf\x45\x34\xd7\xf8\x3f\xa1\x15\x5b\x00\x47\x71\x8c\xbc\x54\x6a\x0d\x07\x2b\x04\xb3\x56\x4e\xea\x1b\x42\x22\x73\xf5\x48\x27\x1a\x0b\xb2\x31\x60\x53\xfa\x76\x99\x19\x55\xeb\xd6\x31\x59\x43\x4e\xce\xbb\x4e\x46\x6d\xae\x5a\x10\x73\xa6\x72\x76\x27\x09\x7a\x10\x49\xe6\x17\xd9\x1d\x36\x10\x94\xfa\x68\xf0\xff\x77\x98\x71\x30\x30\x5b\xea\xba\x2e\xda\x04\xdf\x99\x7b\x71\x4d\x6c\x6f\x2c\x29\xa6\xad\x5c\xb4\x02\x2b\x02\x70\x9b",
	 STR(iv, iv_size,
	     "\x00\x00\x00\x00\x01\x02\x03\x04\x05\x06\x07\x08"),
	 .tag_size = 16,
	 .tag = (void *)
	 "\xee\xad\x9d\x67\x89\x0c\xbb\x22\x39\x23\x36\xfe\xa1\x85\x1f\x38"},
};

const struct cipher_aead_vectors_st aes128_gcm_vectors[] = {
	{
	 .compat_apis = 1,
	 STR(key, key_size,
	     "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"),
	 .auth = NULL,
	 .auth_size = 0,
	 .plaintext = NULL,
	 .plaintext_size = 0,
	 .ciphertext = NULL,
	 STR(iv, iv_size,
	     "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"),
	 .tag_size = 16,
	 .tag = (void *)
	 "\x58\xe2\xfc\xce\xfa\x7e\x30\x61\x36\x7f\x1d\x57\xa4\xe7\x45\x5a"},
	{
	 .compat_apis = 1,
	 STR(key, key_size,
	     "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"),
	 .auth = NULL,
	 .auth_size = 0,
	 STR(plaintext, plaintext_size,
	     "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"),
	 .ciphertext = (void *)
	 "\x03\x88\xda\xce\x60\xb6\xa3\x92\xf3\x28\xc2\xb9\x71\xb2\xfe\x78",
	 STR(iv, iv_size,
	     "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"),
	 .tag_size = 16,
	 .tag = (void *)
	 "\xab\x6e\x47\xd4\x2c\xec\x13\xbd\xf5\x3a\x67\xb2\x12\x57\xbd\xdf"},
	{
	 .compat_apis = 1,
	 STR(key, key_size,
	     "\xfe\xff\xe9\x92\x86\x65\x73\x1c\x6d\x6a\x8f\x94\x67\x30\x83\x08"),
	 .auth = (void *)
	 "\xfe\xed\xfa\xce\xde\xad\xbe\xef\xfe\xed\xfa\xce\xde\xad\xbe\xef\xab\xad\xda\xd2",
	 .auth_size = 20,
	 STR(plaintext, plaintext_size,
	     "\xd9\x31\x32\x25\xf8\x84\x06\xe5\xa5\x59\x09\xc5\xaf\xf5\x26\x9a\x86\xa7\xa9\x53\x15\x34\xf7\xda\x2e\x4c\x30\x3d\x8a\x31\x8a\x72\x1c\x3c\x0c\x95\x95\x68\x09\x53\x2f\xcf\x0e\x24\x49\xa6\xb5\x25\xb1\x6a\xed\xf5\xaa\x0d\xe6\x57\xba\x63\x7b\x39"),
	 .ciphertext = (void *)
	 "\x42\x83\x1e\xc2\x21\x77\x74\x24\x4b\x72\x21\xb7\x84\xd0\xd4\x9c\xe3\xaa\x21\x2f\x2c\x02\xa4\xe0\x35\xc1\x7e\x23\x29\xac\xa1\x2e\x21\xd5\x14\xb2\x54\x66\x93\x1c\x7d\x8f\x6a\x5a\xac\x84\xaa\x05\x1b\xa3\x0b\x39\x6a\x0a\xac\x97\x3d\x58\xe0\x91",
	 STR(iv, iv_size,
	     "\xca\xfe\xba\xbe\xfa\xce\xdb\xad\xde\xca\xf8\x88"),
	 .tag_size = 16,
	 .tag = (void *)
	 "\x5b\xc9\x4f\xbc\x32\x21\xa5\xdb\x94\xfa\xe9\x5a\xe7\x12\x1a\x47"}
};

const struct cipher_aead_vectors_st aes192_gcm_vectors[] = {
	{
	 .compat_apis = 1,
	 STR(key, key_size,
	     "\xfe\xff\xe9\x92\x86\x65\x73\x1c\x6d\x6a\x8f\x94\x67\x30\x83\x08\xfe\xff\xe9\x92\x86\x65\x73\x1c"),
	 .auth = NULL,
	 .auth_size = 0,
	 STR(plaintext, plaintext_size,
	     "\xd9\x31\x32\x25\xf8\x84\x06\xe5\xa5\x59\x09\xc5\xaf\xf5\x26\x9a\x86\xa7\xa9\x53\x15\x34\xf7\xda\x2e\x4c\x30\x3d\x8a\x31\x8a\x72\x1c\x3c\x0c\x95\x95\x68\x09\x53\x2f\xcf\x0e\x24\x49\xa6\xb5\x25\xb1\x6a\xed\xf5\xaa\x0d\xe6\x57\xba\x63\x7b\x39\x1a\xaf\xd2\x55"),
	 .ciphertext =
	 (uint8_t *)
	 "\x39\x80\xca\x0b\x3c\x00\xe8\x41\xeb\x06\xfa\xc4\x87\x2a\x27\x57\x85\x9e\x1c\xea\xa6\xef\xd9\x84\x62\x85\x93\xb4\x0c\xa1\xe1\x9c\x7d\x77\x3d\x00\xc1\x44\xc5\x25\xac\x61\x9d\x18\xc8\x4a\x3f\x47\x18\xe2\x44\x8b\x2f\xe3\x24\xd9\xcc\xda\x27\x10\xac\xad\xe2\x56",
	 STR(iv, iv_size,
	     "\xca\xfe\xba\xbe\xfa\xce\xdb\xad\xde\xca\xf8\x88"),
	 .tag_size = 16,
	 .tag =
	 (void *)
	 "\x99\x24\xa7\xc8\x58\x73\x36\xbf\xb1\x18\x02\x4d\xb8\x67\x4a\x14"},

};

const struct cipher_aead_vectors_st aes256_gcm_vectors[] = {
	{
	 .compat_apis = 1,
	 STR(key, key_size,
	     "\xfe\xff\xe9\x92\x86\x65\x73\x1c\x6d\x6a\x8f\x94\x67\x30\x83\x08\xfe\xff\xe9\x92\x86\x65\x73\x1c\x6d\x6a\x8f\x94\x67\x30\x83\x08"),
	 .auth = NULL,
	 .auth_size = 0,
	 STR(plaintext, plaintext_size,
	     "\xd9\x31\x32\x25\xf8\x84\x06\xe5\xa5\x59\x09\xc5\xaf\xf5\x26\x9a\x86\xa7\xa9\x53\x15\x34\xf7\xda\x2e\x4c\x30\x3d\x8a\x31\x8a\x72\x1c\x3c\x0c\x95\x95\x68\x09\x53\x2f\xcf\x0e\x24\x49\xa6\xb5\x25\xb1\x6a\xed\xf5\xaa\x0d\xe6\x57\xba\x63\x7b\x39\x1a\xaf\xd2\x55"),
	 .ciphertext =
	 (uint8_t *)
	 "\x52\x2d\xc1\xf0\x99\x56\x7d\x07\xf4\x7f\x37\xa3\x2a\x84\x42\x7d\x64\x3a\x8c\xdc\xbf\xe5\xc0\xc9\x75\x98\xa2\xbd\x25\x55\xd1\xaa\x8c\xb0\x8e\x48\x59\x0d\xbb\x3d\xa7\xb0\x8b\x10\x56\x82\x88\x38\xc5\xf6\x1e\x63\x93\xba\x7a\x0a\xbc\xc9\xf6\x62\x89\x80\x15\xad",
	 STR(iv, iv_size,
	     "\xca\xfe\xba\xbe\xfa\xce\xdb\xad\xde\xca\xf8\x88"),
	 .tag_size = 16,
	 .tag =
	 (void *)
	 "\xb0\x94\xda\xc5\xd9\x34\x71\xbd\xec\x1a\x50\x22\x70\xe3\xcc\x6c"},

};

const struct cipher_aead_vectors_st aes256_ccm_vectors[] = {
	{
	 .compat_apis = 0,
	 STR(key, key_size,
	     "\xfb\x76\x15\xb2\x3d\x80\x89\x1d\xd4\x70\x98\x0b\xc7\x95\x84\xc8\xb2\xfb\x64\xce\x60\x97\x8f\x4d\x17\xfc\xe4\x5a\x49\xe8\x30\xb7"),
	 .auth = NULL,
	 .auth_size = 0,
	 STR(plaintext, plaintext_size,
	     "\xa8\x45\x34\x8e\xc8\xc5\xb5\xf1\x26\xf5\x0e\x76\xfe\xfd\x1b\x1e"),
	 .ciphertext = (void *)
	     "\xcc\x88\x12\x61\xc6\xa7\xfa\x72\xb9\x6a\x17\x39\x17\x6b\x27\x7f",
	 STR(iv, iv_size,
	     "\xdb\xd1\xa3\x63\x60\x24\xb7\xb4\x02\xda\x7d\x6f"),
	 .tag_size = 16,
	 .tag = (void *)
	     "\x34\x72\xe1\x14\x5f\x2c\x0c\xbe\x14\x63\x49\x06\x2c\xf0\xe4\x23"},
	{
	 .compat_apis = 0,
	 STR(key, key_size,
	     "\xfb\x76\x15\xb2\x3d\x80\x89\x1d\xd4\x70\x98\x0b\xc7\x95\x84\xc8\xb2\xfb\x64\xce\x60\x97\x87\x8d\x17\xfc\xe4\x5a\x49\xe8\x30\xb7"),
	 STR(auth, auth_size, "\x36"),
	 STR(plaintext, plaintext_size,
	     "\xa9"),
	 .ciphertext = (void *)
	     "\x9d",
	 STR(iv, iv_size,
	     "\xdb\xd1\xa3\x63\x60\x24\xb7\xb4\x02\xda\x7d\x6f"),
	 .tag_size = 16,
	 .tag = (void *)
	     "\x32\x61\xb1\xcf\x93\x14\x31\xe9\x9a\x32\x80\x67\x38\xec\xbd\x2a"},
};

const struct cipher_aead_vectors_st aes128_ccm_vectors[] = {
	{
	 .compat_apis = 0,
	 STR(key, key_size,
	     "\xC0\xC1\xC2\xC3\xC4\xC5\xC6\xC7\xC8\xC9\xCA\xCB\xCC\xCD\xCE\xCF"),
	 STR(auth, auth_size, "\x08\xD0\x84\x21\x43\x01\x00\x00\x00\x00\x48\xDE\xAC\x02\x05\x00\x00\x00\x55\xCF\x00\x00\x51\x52\x53\x54"),
	 .plaintext = NULL,
	 .plaintext_size = 0,
	 STR(iv, iv_size,
	     "\xAC\xDE\x48\x00\x00\x00\x00\x01\x00\x00\x00\x05\x02"),
	 .tag_size = 8,
	 .tag = (void *)
	     "\x22\x3B\xC1\xEC\x84\x1A\xB5\x53"},
	{
	 .compat_apis = 0,
	 STR(key, key_size,
	     "\x40\x41\x42\x43\x44\x45\x46\x47\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"),
	 STR(auth, auth_size, "\x00\x01\x02\x03\x04\x05\x06\x07"),
	 STR(plaintext, plaintext_size,
	     "\x20\x21\x22\x23"),
	 .ciphertext = (void *)
	     "\x71\x62\x01\x5b",
	 STR(iv, iv_size,
	     "\x10\x11\x12\x13\x14\x15\x16"),
	 .tag_size = 4,
	 .tag = (void *)
	     "\x4d\xac\x25\x5d"},
	/* from rfc3610 */
	{
	 .compat_apis = 0,
	 STR(key, key_size,
	     "\xC0\xC1\xC2\xC3\xC4\xC5\xC6\xC7\xC8\xC9\xCA\xCB\xCC\xCD\xCE\xCF"),
	 STR(auth, auth_size, "\x00\x01\x02\x03\x04\x05\x06\x07"),
	 STR(plaintext, plaintext_size,
	     "\x08\x09\x0A\x0B\x0C\x0D\x0E\x0F\x10\x11\x12\x13\x14\x15\x16\x17\x18\x19\x1A\x1B\x1C\x1D\x1E"),
	 .ciphertext = (void *)
	     "\x01\x35\xD1\xB2\xC9\x5F\x41\xD5\xD1\xD4\xFE\xC1\x85\xD1\x66\xB8\x09\x4E\x99\x9D\xFE\xD9\x6C",
	 STR(iv, iv_size,
	     "\x00\x00\x00\x09\x08\x07\x06\xA0\xA1\xA2\xA3\xA4\xA5"),
	 .tag_size = 10,
	 .tag = (void *)
	     "\x04\x8C\x56\x60\x2C\x97\xAC\xBB\x74\x90"},
	{
	 .compat_apis = 0,
	 STR(key, key_size,
	     "\xC0\xC1\xC2\xC3\xC4\xC5\xC6\xC7\xC8\xC9\xCA\xCB\xCC\xCD\xCE\xCF"),
	 STR(auth, auth_size, "\x00\x01\x02\x03\x04\x05\x06\x07"),
	 STR(plaintext, plaintext_size,
	     "\x08\x09\x0A\x0B\x0C\x0D\x0E\x0F\x10\x11\x12\x13\x14\x15\x16\x17\x18\x19\x1A\x1B\x1C\x1D\x1E"),
	 .ciphertext = (void *)
	     "\x58\x8C\x97\x9A\x61\xC6\x63\xD2\xF0\x66\xD0\xC2\xC0\xF9\x89\x80\x6D\x5F\x6B\x61\xDA\xC3\x84",
	 STR(iv, iv_size,
	     "\x00\x00\x00\x03\x02\x01\x00\xA0\xA1\xA2\xA3\xA4\xA5"),
	 .tag_size = 8,
	 .tag = (void *)
	     "\x17\xE8\xD1\x2C\xFD\xF9\x26\xE0"},
};

const struct cipher_vectors_st aes128_cbc_vectors[] = {
	{
	 STR(key, key_size,
	     "\x2b\x7e\x15\x16\x28\xae\xd2\xa6\xab\xf7\x15\x88\x09\xcf\x4f\x3c"),
	 STR(plaintext, plaintext_size,
	     "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96\xe9\x3d\x7e\x11\x73\x93\x17\x2a"),
	 .ciphertext = (uint8_t *)
	 "\x76\x49\xab\xac\x81\x19\xb2\x46\xce\xe9\x8e\x9b\x12\xe9\x19\x7d",
	 STR(iv, iv_size,
	     "\x00\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0A\x0B\x0C\x0D\x0E\x0F"),
	 },
	{
	 STR(key, key_size,
	     "\x2b\x7e\x15\x16\x28\xae\xd2\xa6\xab\xf7\x15\x88\x09\xcf\x4f\x3c"),
	 STR(plaintext, plaintext_size,
	     "\xae\x2d\x8a\x57\x1e\x03\xac\x9c\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"),
	 .ciphertext =
	 (uint8_t *)
	 "\x50\x86\xcb\x9b\x50\x72\x19\xee\x95\xdb\x11\x3a\x91\x76\x78\xb2",
	 STR(iv, iv_size,
	     "\x76\x49\xab\xac\x81\x19\xb2\x46\xce\xe9\x8e\x9b\x12\xe9\x19\x7d"),
	 },
};

const struct cipher_vectors_st aes192_cbc_vectors[] = {
	{
	 STR(key, key_size,
	     "\x8e\x73\xb0\xf7\xda\x0e\x64\x52\xc8\x10\xf3\x2b\x80\x90\x79\xe5\x62\xf8\xea\xd2\x52\x2c\x6b\x7b"),
	 STR(plaintext, plaintext_size,
	     "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96\xe9\x3d\x7e\x11\x73\x93\x17\x2a"),
	 .ciphertext = (uint8_t *)
	 "\x4f\x02\x1d\xb2\x43\xbc\x63\x3d\x71\x78\x18\x3a\x9f\xa0\x71\xe8",
	 STR(iv, iv_size,
	     "\x00\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0A\x0B\x0C\x0D\x0E\x0F"),
	 },
	{
	 STR(key, key_size,
	     "\x8e\x73\xb0\xf7\xda\x0e\x64\x52\xc8\x10\xf3\x2b\x80\x90\x79\xe5\x62\xf8\xea\xd2\x52\x2c\x6b\x7b"),
	 STR(plaintext, plaintext_size,
	     "\xae\x2d\x8a\x57\x1e\x03\xac\x9c\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"),
	 .ciphertext = (uint8_t *)
	 "\xb4\xd9\xad\xa9\xad\x7d\xed\xf4\xe5\xe7\x38\x76\x3f\x69\x14\x5a",
	 STR(iv, iv_size,
	     "\x4F\x02\x1D\xB2\x43\xBC\x63\x3D\x71\x78\x18\x3A\x9F\xA0\x71\xE8"),
	 },
};

const struct cipher_vectors_st aes256_cbc_vectors[] = {
	{
	 STR(key, key_size,
	     "\x60\x3d\xeb\x10\x15\xca\x71\xbe\x2b\x73\xae\xf0\x85\x7d\x77\x81\x1f\x35\x2c\x07\x3b\x61\x08\xd7\x2d\x98\x10\xa3\x09\x14\xdf\xf4"),
	 STR(plaintext, plaintext_size,
	     "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96\xe9\x3d\x7e\x11\x73\x93\x17\x2a"),
	 .ciphertext = (uint8_t *)
	 "\xF5\x8C\x4C\x04\xD6\xE5\xF1\xBA\x77\x9E\xAB\xFB\x5F\x7B\xFB\xD6",
	 STR(iv, iv_size,
	     "\x00\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0A\x0B\x0C\x0D\x0E\x0F"),
	 },
	{
	 STR(key, key_size,
	     "\x60\x3d\xeb\x10\x15\xca\x71\xbe\x2b\x73\xae\xf0\x85\x7d\x77\x81\x1f\x35\x2c\x07\x3b\x61\x08\xd7\x2d\x98\x10\xa3\x09\x14\xdf\xf4"),
	 STR(plaintext, plaintext_size,
	     "\xae\x2d\x8a\x57\x1e\x03\xac\x9c\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"),
	 .ciphertext = (uint8_t *)
	 "\x9c\xfc\x4e\x96\x7e\xdb\x80\x8d\x67\x9f\x77\x7b\xc6\x70\x2c\x7d",
	 STR(iv, iv_size,
	     "\xF5\x8C\x4C\x04\xD6\xE5\xF1\xBA\x77\x9E\xAB\xFB\x5F\x7B\xFB\xD6"),
	 },
};

const struct cipher_vectors_st tdes_cbc_vectors[] = {
/* First 2 from https://www.cosic.esat.kuleuven.be/nessie/testvectors/bc/des/Triple-Des-3-Key-192-64.unverified.test-vectors */
	{
	 STR(key, key_size,
	     "\x00\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0A\x0B\x0C\x0D\x0E\x0F\x10\x11\x12\x13\x14\x15\x16\x17"),
	 STR(plaintext, plaintext_size,
	     "\x98\x26\x62\x60\x55\x53\x24\x4D"),
	 .ciphertext = (uint8_t *)
	 "\x00\x11\x22\x33\x44\x55\x66\x77",
	 STR(iv, iv_size, "\x00\x00\x00\x00\x00\x00\x00\x00"),
	 },
	{
	 STR(key, key_size,
	     "\x2B\xD6\x45\x9F\x82\xC5\xB3\x00\x95\x2C\x49\x10\x48\x81\xFF\x48\x2B\xD6\x45\x9F\x82\xC5\xB3\x00"),
	 STR(plaintext, plaintext_size,
	     "\x85\x98\x53\x8A\x8E\xCF\x11\x7D"),
	 .ciphertext = (uint8_t *)
	 "\xEA\x02\x47\x14\xAD\x5C\x4D\x84",
	 STR(iv, iv_size, "\x00\x00\x00\x00\x00\x00\x00\x00"),
	 },
};

const struct cipher_vectors_st arcfour_vectors[] = { /* RFC6229 */
	{
	 STR(key, key_size,
	     "\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f\x10\x11\x12\x13\x14\x15\x16\x17\x18"),
	 STR(plaintext, plaintext_size,
	     "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"),
	 .ciphertext = (uint8_t *)
	     "\x05\x95\xe5\x7f\xe5\xf0\xbb\x3c\x70\x6e\xda\xc8\xa4\xb2\xdb\x11",
	 .iv = NULL,
	 .iv_size = 0
	},
	{
	 STR(key, key_size,
	     "\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f\x10\x11\x12\x13\x14\x15\x16\x17\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f\x20"),
	 STR(plaintext, plaintext_size,
	     "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"),
	 .ciphertext = (uint8_t *)
	     "\xea\xa6\xbd\x25\x88\x0b\xf9\x3d\x3f\x5d\x1e\x4c\xa2\x61\x1d\x91",
	 .iv = NULL,
	 .iv_size = 0
	},
	{
	 STR(key, key_size,
	     "\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f\x10"),
	 STR(plaintext, plaintext_size,
	     "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"),
	 .ciphertext = (uint8_t *)
	 "\x9a\xc7\xcc\x9a\x60\x9d\x1e\xf7\xb2\x93\x28\x99\xcd\xe4\x1b\x97",
	 .iv = NULL,
	 .iv_size = 0
	},
};

const struct cipher_vectors_st aes128_cfb8_vectors[] = { /* NIST 800-38a */
	{
	 STR(key, key_size,
	     "\x2b\x7e\x15\x16\x28\xae\xd2\xa6\xab\xf7\x15\x88\x09\xcf\x4f\x3c"),
	 STR(plaintext, plaintext_size,
	     "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
             "\xae\x2d"),
	 .ciphertext = (uint8_t *)
	     "\x3b\x79\x42\x4c\x9c\x0d\xd4\x36\xba\xce\x9e\x0e\xd4\x58\x6a\x4f"
             "\x32\xb9",
	 STR(iv, iv_size,
	     "\x00\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"),
	 /* the least significant 16 bytes of ciphertext */
	 STR(internal_iv, internal_iv_size,
	     "\x42\x4c\x9c\x0d\xd4\x36\xba\xce\x9e\x0e\xd4\x58\x6a\x4f\x32\xb9"),
	 },
};

const struct cipher_vectors_st aes192_cfb8_vectors[] = { /* NIST 800-38a */
	{
	 STR(key, key_size,
	     "\x8e\x73\xb0\xf7\xda\x0e\x64\x52\xc8\x10\xf3\x2b\x80\x90\x79\xe5"
             "\x62\xf8\xea\xd2\x52\x2c\x6b\x7b"),
	 STR(plaintext, plaintext_size,
	     "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
             "\xae\x2d"),
	 .ciphertext = (uint8_t *)
	     "\xcd\xa2\x52\x1e\xf0\xa9\x05\xca\x44\xcd\x05\x7c\xbf\x0d\x47\xa0"
             "\x67\x8a",
	 STR(iv, iv_size,
	     "\x00\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"),
	 /* the least significant 16 bytes of ciphertext */
	 STR(internal_iv, internal_iv_size,
	     "\x52\x1e\xf0\xa9\x05\xca\x44\xcd\x05\x7c\xbf\x0d\x47\xa0\x67\x8a"),
	 },
};

const struct cipher_vectors_st aes256_cfb8_vectors[] = { /* NIST 800-38a */
	{
	 STR(key, key_size,
	     "\x60\x3d\xeb\x10\x15\xca\x71\xbe\x2b\x73\xae\xf0\x85\x7d\x77\x81"
             "\x1f\x35\x2c\x07\x3b\x61\x08\xd7\x2d\x98\x10\xa3\x09\x14\xdf\xf4"),
	 STR(plaintext, plaintext_size,
	     "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
             "\xae\x2d"),
	 .ciphertext = (uint8_t *)
	     "\xdc\x1f\x1a\x85\x20\xa6\x4d\xb5\x5f\xcc\x8a\xc5\x54\x84\x4e\x88"
             "\x97\x00",
	 STR(iv, iv_size,
	     "\x00\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"),
	 /* the least significant 16 bytes of ciphertext */
	 STR(internal_iv, internal_iv_size,
	     "\x1a\x85\x20\xa6\x4d\xb5\x5f\xcc\x8a\xc5\x54\x84\x4e\x88\x97\x00"),
	 },
};

/* GOST 28147-89 vectors come from the testsuite contributed to OpenSSL by
 * Sergey E. Leontiev. CryptoPro-B test vector is just truncated.
 * TC26Z is calculated using Nettle */
const struct cipher_vectors_st gost28147_cpa_cfb_vectors[] = {
	{
	 STR(key, key_size,
	     "\x8d\x5a\x2c\x83\xa7\xc7\x0a\x61\xd6\x1b\x34\xb5\x1f\xdf\x42\x68"
	     "\x66\x71\xa3\x5d\x87\x4c\xfd\x84\x99\x36\x63\xb6\x1e\xd6\x0d\xad"),
	 STR(plaintext, plaintext_size,
	     "\xd2\xfd\xf8\x3a\xc1\xb4\x39\x23\x2e\xaa\xcc\x98\x0a\x02\xda\x33"),
	 .ciphertext = (uint8_t *)
		 "\x88\xb7\x75\x16\x74\xa5\xee\x2d\x14\xfe\x91\x67\xd0\x5c\xcc\x40",
	 STR(iv, iv_size,
	     "\x46\x60\x6f\x0d\x88\x34\x23\x5a"),
	},
};

const struct cipher_vectors_st gost28147_cpb_cfb_vectors[] = {
	{
	 STR(key, key_size,
	     "\x48\x0c\x74\x1b\x02\x6b\x55\xd5\xb6\x6d\xd7\x1d\x40\x48\x05\x6b"
	     "\x6d\xeb\x3c\x29\x0f\x84\x80\x23\xee\x0d\x47\x77\xe3\xfe\x61\xc9"),
	 STR(plaintext, plaintext_size,
	     "\x8c\x9c\x44\x35\xfb\xe9\xa5\xa3\xa0\xae\x28\x56\x91\x10\x8e\x1e"
	     "\xd2\xbb\x18\x53\x81\x27\x0d\xa6\x68\x59\x36\xc5\x81\x62\x9a\x8e"
	     "\x7d\x50\xf1\x6f\x97\x62\x29\xec\x80\x51\xe3\x7d\x6c\xc4\x07\x95"
	     "\x28\x63\xdc\xb4\xb9\x2d\xb8\x13\xb1\x05\xb5\xf9\xeb\x75\x37"),
	 .ciphertext = (uint8_t *)
		 "\x23\xc6\x7f\x20\xa1\x23\x58\xbc\x7b\x05\xdb\x21\x15\xcf\x96\x41"
		 "\xc7\x88\xef\x76\x5c\x49\xdb\x42\xbf\xf3\xc0\xf5\xbd\x5d\xd9\x8e"
		 "\xaf\x3d\xf4\xe4\xda\x88\xbd\xbc\x47\x5d\x76\x07\xc9\x5f\x54\x1d"
		 "\x1d\x6a\xa1\x2e\x18\xd6\x60\x84\x02\x18\x37\x92\x92\x15\xab",
	 STR(iv, iv_size,
	     "\x1f\x3f\x82\x1e\x0d\xd8\x1e\x22"),
	},
};

const struct cipher_vectors_st gost28147_cpc_cfb_vectors[] = {
	{
	 STR(key, key_size,
	     "\x77\xc3\x45\x8e\xf6\x42\xe7\x04\x8e\xfc\x08\xe4\x70\x96\xd6\x05"
	     "\x93\x59\x02\x6d\x6f\x97\xca\xe9\xcf\x89\x44\x4b\xde\x6c\x22\x1d"),
	 STR(plaintext, plaintext_size,
	     "\x07\x9c\x91\xbe"),
	 .ciphertext = (uint8_t *)
		 "\x19\x35\x81\x34",
	 STR(iv, iv_size,
	     "\x43\x7c\x3e\x8e\x2f\x2a\x00\x98"),
	},
};

const struct cipher_vectors_st gost28147_cpd_cfb_vectors[] = {
	{
	 STR(key, key_size,
	     "\x38\x9f\xe8\x37\xff\x9c\x5d\x29\xfc\x48\x55\xa0\x87\xea\xe8\x40"
	     "\x20\x87\x5b\xb2\x01\x15\x55\xa7\xe3\x2d\xcb\x3d\xd6\x59\x04\x73"),
	 STR(plaintext, plaintext_size,
	     "\x2f\x31\xd8\x83\xb4\x20\xe8\x6e\xda"),
	 .ciphertext = (uint8_t *)
		 "\x6d\xa4\xed\x40\x08\x88\x71\xad\x16",
	 STR(iv, iv_size,
	     "\xc5\xa2\xd2\x1f\x2f\xdf\xb8\xeb"),
	},
};

const struct cipher_vectors_st gost28147_tc26z_cfb_vectors[] = {
	{
	 STR(key, key_size,
	     "\x8d\x5a\x2c\x83\xa7\xc7\x0a\x61\xd6\x1b\x34\xb5\x1f\xdf\x42\x68"
	     "\x66\x71\xa3\x5d\x87\x4c\xfd\x84\x99\x36\x63\xb6\x1e\xd6\x0d\xad"),
	 STR(plaintext, plaintext_size,
	     "\xd2\xfd\xf8\x3a\xc1\xb4\x39\x23\x2e\xaa\xcc\x98\x0a\x02\xda\x33"),
	 .ciphertext = (uint8_t *)
		 "\xed\xa7\xf1\x41\x01\x9c\xbd\xcd\x44\x6b\x00\x96\x87\xf7\xc7\xe6",
	 STR(iv, iv_size,
	     "\x46\x60\x6f\x0d\x88\x34\x23\x5a"),
	},
};

const struct cipher_vectors_st gost28147_tc26z_cnt_vectors[] = {
	{
	 STR(key, key_size,
	     "\x59\x9f\x84\xba\xc3\xf3\xd2\xf1\x60\xe1\xe3\xf2\x6a\x96\x1a\xf9"
	     "\x9c\x48\xb2\x4e\xbc\xbb\xbf\x7c\xd8\xf3\xac\xcd\x96\x8d\x28\x6a"),
	 STR(plaintext, plaintext_size,
	     "\x90\xa2\x39\x66\xae\x01\xb9\xa3\x52\x4e\xc8\xed\x6c\xdd\x88\x30"),
	 .ciphertext = (uint8_t *)
		 "\xe8\xb1\x4f\xc7\x30\xdc\x25\xbb\x36\xba\x64\x3c\x17\xdb\xff\x99",
	 STR(iv, iv_size,
	     "\x8d\xaf\xa8\xd1\x58\xed\x05\x8d"),
	}
};

/* Manually calculated to overcome section size differences between TLS cipher
 * and test vectors defined in recommendation for standarization
 * R 1323565.1.017-2018. TLS uses 1024 byte sections, we need more to have at
 * least one rekeying. */
const struct cipher_vectors_st magma_ctr_acpkm_vectors[] = {
	{
	 STR(key, key_size,
	     "\x88\x99\xaa\xbb\xcc\xdd\xee\xff\x00\x11\x22\x33\x44\x55\x66\x77"
	     "\xfe\xdc\xba\x98\x76\x54\x32\x10\x01\x23\x45\x67\x89\xab\xcd\xef"),
	 STR(plaintext, plaintext_size,
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"

	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     ),
	 .ciphertext = (uint8_t *)
	     "\x2a\xb8\x1d\xee\xeb\x1e\x4c\xab\x68\xe1\x04\xc4\xbd\x6b\x94\xea"
	     "\xc0\x4d\xdb\x07\xea\x92\x8f\xa2\x22\x0d\x97\x91\x83\x20\x78\x6b"
	     "\x17\x0d\x8e\xaf\x6e\x3c\xa1\xab\xf2\x92\x93\x9a\xb1\x19\x54\xbb"
	     "\xd3\x3f\x07\x38\x54\x26\x56\xed\xe3\xed\x4d\x54\x42\xb3\x0d\x0d"
	     "\x43\xa1\x2e\xe8\xcf\x83\x81\x30\xe5\x90\xf3\x71\xe6\x7a\x42\xc3"
	     "\xf8\x75\x91\xeb\xb8\xf7\xea\x17\xb3\xc4\xec\x9f\xa0\xca\x54\xd5"
	     "\xbc\x42\x9b\x60\xd8\x90\xce\x61\xc1\xb4\x0d\x20\x52\x45\x0a\xfa"
	     "\x77\xb4\x83\x54\xb4\x43\x77\x11\x50\x8f\xdc\xf4\xc4\x8f\x24\x2b"
	     "\x84\xec\x97\xc9\xc1\x76\x3c\xf3\x17\x3c\x5a\xe8\x57\x86\x41\x22"
	     "\xb6\x5b\x57\x29\xae\x1b\xb0\x75\x06\x95\xe5\xe2\x37\x93\x3b\x5a"
	     "\xbe\x85\x5d\xb0\x16\x70\x5b\x53\x35\x57\x52\xdb\x87\x50\x89\x34"
	     "\xa3\xd2\x1e\xc3\x83\x03\xbd\xdc\xc1\x24\xc3\xc9\x9d\xd7\x1b\x45"
	     "\xfb\xcb\x4e\xd5\x64\x10\x5c\xe6\xc5\x93\x4d\xa9\x9b\xea\xed\xc5"
	     "\xf4\x02\x30\x13\xbc\x41\x97\x81\x05\xb0\xcc\x6a\x39\x83\xe9\x05"
	     "\x25\x11\x6e\x04\x63\x14\x6d\x85\x78\x9a\x09\xa7\x0d\x4c\x7b\xd7"
	     "\xdf\xb1\x33\xeb\xbf\xeb\x92\x58\xf3\xda\x9d\xed\x8e\xf9\x8a\x3c"
	     "\x71\x8d\x19\x67\x1a\x12\x49\xdf\x0e\xb1\x88\x3c\x9e\xf5\x1b\x77"
	     "\x13\x60\x80\x1a\x28\x46\xa0\x48\x31\xec\xc2\xcb\xe7\xe7\x72\xdb"
	     "\x92\x2a\x7b\x04\xd0\xa3\xd0\x17\x8c\xb2\x4d\xb3\xa2\x43\x16\x17"
	     "\x8d\xec\x78\xf0\xdd\xe5\x60\x07\x77\x49\x66\xad\xb3\x36\xdd\xe4"
	     "\xc9\xef\xf7\x6d\xe4\x00\xfc\x9c\x5b\x9d\x8a\x65\x58\xbb\xbd\x5c"
	     "\xfe\x18\xe6\x97\x83\xd1\xae\x20\xed\x12\xac\xe5\x29\x7f\x58\xab"
	     "\x83\xea\xd2\x92\xea\x17\x00\xaf\x2b\x55\x43\x7c\x5b\x17\xf2\x91"
	     "\x0e\x79\x0b\x95\xb4\xe0\x4a\xb0\x55\x62\xe5\x5d\xda\x55\x02\xd6"
	     "\xa1\xea\x99\x73\x16\x59\xb3\x3e\xc9\x86\x92\x1d\xb2\x1a\xf1\x4e"
	     "\x6b\x2e\xeb\x49\x7d\xf6\x35\xc0\x23\xae\xb5\x56\xff\x90\xe7\xc5"
	     "\xc5\x4c\xef\xea\x8f\x9e\xf1\xc2\xd0\x1c\xee\xef\xb7\xac\x0c\x0c"
	     "\x43\xdf\xcc\x8e\x95\xe2\xd3\xd5\x9f\x17\x81\x85\x66\x87\x7e\xdb"
	     "\x5e\x45\xde\xb2\x45\x95\x69\x90\x75\x80\xc1\x5e\x0c\x3d\xe8\xd9"
	     "\x9a\x17\xc5\x62\x49\x90\xa6\x54\x68\xc2\xfa\xb0\x96\xc9\xac\x42"
	     "\x9e\xa9\x6f\x33\x0c\x5b\xc2\xc5\x81\xc1\xc5\x5d\x91\x44\xb7\x48"
	     "\xd0\x32\xc6\x7d\x99\xa1\xad\x43\x6c\xba\xc5\xf5\x10\xbd\xf2\x44"
	     "\x75\x36\xe8\x7e\x91\x21\x8c\x16\xce\x74\xa4\xc2\x3d\x18\x28\x1a"
	     "\x82\x79\x04\x3e\x15\x73\x22\x30\x7a\x33\x7f\x76\x63\xbc\x9b\x0b"
	     "\xcb\x3c\xe2\x3d\x66\x22\x28\x20\x1c\xc3\x43\x12\x89\x2d\x05\x51"
	     "\x0b\x65\x2b\xf3\xca\x0f\x21\xd7\x57\xac\xa9\x4d\x4f\xd0\x03\x13"
	     "\x89\x94\xec\xad\x38\x9b\x95\xaf\xf8\x0a\xc5\xb0\x8b\xb3\x57\xcd"
	     "\xdd\xe7\xa3\xfe\x3e\x84\x35\xad\xe6\x87\x69\xd5\xb7\xc5\xf0\xde"
	     "\xcc\xc7\x81\x59\xd7\x52\x4b\xa1\x77\x8f\xec\x6a\x9d\xcd\x0d\x08"
	     "\xa7\xbf\x32\x6b\x5a\xa3\xc8\x4a\x0b\xde\x8c\xf8\x12\x86\xf1\x6b"
	     "\x78\x86\xaf\xf3\x49\x78\x81\x5b\x9d\x6b\xe9\xf7\x59\xe2\x36\xea"
	     "\xab\x40\x87\x1b\x0b\x2a\x70\x9b\x4c\x29\x3f\x3d\x5a\xfb\x09\x11"
	     "\x9a\x39\xfc\x41\x65\xac\x64\xbd\xfd\x8b\xf3\x09\x1b\x13\xde\x38"
	     "\x5b\x35\x0e\xce\x71\xab\x87\x99\x82\xd4\x26\x5c\x5d\x4d\x5b\xfe"
	     "\xe8\x33\x11\x37\xf8\x3b\x16\x93\x20\xbe\x47\x68\xd9\x19\x94\x17"
	     "\xdf\x56\xce\xbf\x82\x34\x75\x74\x7a\x39\xb3\xb5\xc3\x54\x86\x30"
	     "\x33\x7e\xbd\xe7\xd9\x24\x18\x8f\xbf\xc1\x69\x28\x75\x37\xb6\x90"
	     "\x2e\x4c\x71\x88\xcf\xfe\x90\xd6\xe5\x3d\xf3\xf2\x42\x8f\xee\x76"
	     "\x00\xb8\x84\x5f\xeb\xbc\x5d\x46\xcd\xa6\x13\xaf\x3b\x37\xae\xb4"
	     "\x82\xd5\x37\x88\x86\x94\x0f\xd8\x00\x0b\x97\x86\xa3\x91\x43\x8f"
	     "\x05\x40\x04\xdb\x7c\x17\x5e\xbf\xf0\x07\x29\x94\x34\x04\x6b\x1f"
	     "\x32\xa0\xa3\x30\x1a\xc8\x72\xde\xd7\x42\xfe\xb8\xe2\x2e\x6c\xe9"
	     "\x71\x3e\xe5\x9a\x6c\x83\xc1\x75\xc2\x04\xfb\x8d\x45\x4a\x52\x7a"
	     "\xac\xca\x87\x7a\xf3\x88\x42\x42\xfb\x40\x71\x67\xc0\xa7\xcc\x10"
	     "\x30\x32\xc2\x7e\x3d\x93\x40\xb6\xb2\x53\xcd\xe4\x23\x4b\x5a\x99"
	     "\x20\xfe\xe9\xef\xce\x82\xb0\x0f\x4e\x21\xa6\xc0\xad\x6a\x7b\x08"
	     "\xb9\x33\x6b\xc2\x35\x5e\x0d\x6e\xd5\x9d\x65\xef\x70\x5d\xc7\x02"
	     "\xa8\x4e\x5e\x80\xb1\x9c\xa8\x50\xfd\x25\x44\x0d\xe6\x78\x8c\xd9"
	     "\x18\x01\x43\xd0\xee\x9f\xa7\x68\xd9\x37\xf3\x5e\xcb\xf6\x08\x92"
	     "\x75\x02\x65\xee\xcf\xed\xdc\xe5\x3f\x6d\x9f\x84\x75\x39\x93\x27"
	     "\xff\xc5\x97\x37\x6e\xc7\x60\x9c\x1d\x39\x4d\xbc\x27\xa2\xe5\x85"
	     "\x05\xa4\x3d\x28\xce\x42\x51\x81\xd5\xcb\xbd\x83\x11\x13\x05\x93"
	     "\x4f\x70\xb3\x97\xa9\xfa\x14\x95\x77\xc5\x1e\x40\xd0\x4d\x77\xef"
	     "\xa8\x53\xfc\x26\x30\xd4\xfe\xda\xd0\x74\x58\x38\x21\xc0\xbb\x21"

	     "\xd9\x6a\xf3\xcc\x57\x01\xcb\xbb\x01\x9a\x99\xe6\x4e\xaf\xf2\x32",
	 STR(iv, iv_size,
	     "\x12\x34\x56\x78\x00\x00\x00\x00"),
	}
};

/* Manually calculated to overcome section size differences between TLS cipher
 * and test vectors defined in recommendation for standarization
 * R 1323565.1.017-2018. TLS uses 4096 byte sections, we need more to have at
 * least one rekeying. */
const struct cipher_vectors_st kuznyechik_ctr_acpkm_vectors[] = {
	{
	 STR(key, key_size,
	     "\x88\x99\xaa\xbb\xcc\xdd\xee\xff\x00\x11\x22\x33\x44\x55\x66\x77"
	     "\xfe\xdc\xba\x98\x76\x54\x32\x10\x01\x23\x45\x67\x89\xab\xcd\xef"),
	 STR(plaintext, plaintext_size,
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"
	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	     "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	     "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"

	     "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     ),
	 .ciphertext = (uint8_t *)
	     "\xf1\x95\xd8\xbe\xc1\x0e\xd1\xdb\xd5\x7b\x5f\xa2\x40\xbd\xa1\xb8"
	     "\x85\xee\xe7\x33\xf6\xa1\x3e\x5d\xf3\x3c\xe4\xb3\x3c\x45\xde\xe4"
	     "\xa5\xea\xe8\x8b\xe6\x35\x6e\xd3\xd5\xe8\x77\xf1\x35\x64\xa3\xa5"
	     "\xcb\x91\xfa\xb1\xf2\x0c\xba\xb6\xd1\xc6\xd1\x58\x20\xbd\xba\x73"
	     "\x7e\x08\xf5\x9f\x71\xb8\x56\x08\xe3\x89\x55\xb1\x7b\x53\xed\x27"
	     "\xee\x04\xce\x2c\x02\xae\xf0\x75\x8f\x5d\x64\x77\xea\x0a\x79\xb2"
	     "\x9f\xc0\xb0\x9a\xf0\x93\xd5\xbf\xb9\xdf\xab\x18\x25\xf7\x51\xc9"
	     "\x82\x69\x88\x08\x00\x2b\x35\x8e\x2c\x2f\x7c\x01\xdc\x31\x56\x1d"
	     "\x52\xb4\xca\x07\xbf\x02\xa5\x9a\x80\x88\x91\x19\xf1\x26\x0b\x7f"
	     "\x9d\x3e\x1c\xe5\xf9\x92\x1a\xf5\xb5\x53\xed\x1e\x1a\xdf\xdc\x53"
	     "\x9e\x2e\x79\x65\xbd\x7b\x48\x48\x57\x58\x90\x76\x40\x57\x86\x76"
	     "\x7d\xe3\xb2\x82\x06\xed\xeb\xce\x35\x1b\xcc\x5c\xee\x54\xe4\x3f"
	     "\x53\x4b\xb5\x0d\x07\x0a\x2d\x0c\xb7\xcf\x7c\xf8\x81\xad\x32\x01"
	     "\x26\x50\x0f\xe9\x32\xf3\xd7\xc9\xf0\x86\xab\xb8\xf8\xea\x72\x3a"
	     "\x9a\x17\x48\x12\xa5\xbb\x3b\xa9\x6d\x01\x51\x82\xc0\xe2\x98\x4d"
	     "\x69\x3c\xbd\x67\x6a\xa0\xb1\x7d\x7a\xa8\x2c\xb8\xe2\xb9\xf6\x68"
	     "\xfa\xeb\xae\x0d\x41\xcb\xb0\xfe\xe5\x55\x65\xb6\x5d\x56\xe5\x05"
	     "\x20\x08\x51\x77\x7c\x18\x3a\x25\x69\xaa\xa2\x39\x68\xcf\xd1\x3e"
	     "\x51\x47\xb5\xd8\x13\xcc\xaf\x2d\x31\xd5\x28\xae\x61\x48\xe6\x9a"
	     "\x0a\xd8\x30\x0b\x1f\x7c\xef\x3d\x32\xa5\x40\x05\x41\xdf\x84\x1e"
	     "\xce\xde\xca\x31\x42\xa6\xc4\xbc\x2b\x56\xb7\x7e\xa6\x22\x1f\x85"
	     "\x7b\xe8\x80\x01\xc1\xff\x78\x45\x14\xb5\x4d\x9b\xe3\x21\xf7\xb8"
	     "\x13\xd5\x44\x3d\x6f\x33\x85\x46\x56\x07\x79\x15\xb4\xec\xe5\xfd"
	     "\x34\x35\xc8\x8a\x5f\x4d\x58\x84\x82\x9d\xb6\x4d\x0e\x90\x65\xb0"
	     "\x5e\x61\x83\x30\x3d\x51\x34\xe3\x0e\x6e\x78\x05\xc8\x6d\x1e\x7d"
	     "\x76\xe2\x3d\x85\x33\x82\x42\x7c\x7d\x67\x33\xbb\x60\x5e\x4d\xb0"
	     "\xc8\xd0\x7a\x6c\xbe\x4e\xc8\xc4\x82\x9d\xb6\x25\x56\xc9\x1b\x52"
	     "\x3b\x9b\x4a\xf5\x8f\x67\x12\xd5\x41\x60\xe6\x97\x08\x90\x7b\x17"
	     "\x3b\xf9\x1c\x82\x0d\x15\x0a\x3d\x1b\x27\xd4\x46\xd6\x91\x5c\x74"
	     "\x48\x45\x1a\x4c\x60\x26\xf3\xe3\x1f\xe7\x0a\xfc\x79\x42\x8a\x5f"
	     "\x5f\x5b\x33\xf4\xb3\xda\xbd\x4c\x4a\x10\x5e\x3b\x9d\x09\xf8\xf2"
	     "\x51\x25\x44\x37\x7b\xca\x88\xf0\x88\xa9\x5d\x2e\x9e\x42\xf6\xac"
	     "\x3d\xd6\x01\x06\x06\x5c\x96\x4b\x13\xf5\x5f\xaa\x06\x34\x16\x67"
	     "\xfd\xa5\x49\x12\xcc\x21\x8f\xb4\xb9\x33\xfc\x58\xf0\x36\xeb\xd7"
	     "\x50\xf2\x38\x39\x9e\x93\x7b\xf7\xf5\xc7\x9c\x73\x36\x1f\xbe\xa9"
	     "\x49\xf9\x2c\x4f\x17\xfc\xb3\x64\x51\x68\xaf\xed\xe3\x43\xbf\x2f"
	     "\x3f\x57\xbb\x74\x93\x51\xc4\x93\x1e\x2e\x05\xed\xda\x74\x27\x1b"
	     "\x7b\xa8\x98\x3e\xa6\x62\x75\xbd\x68\xab\x44\xe8\xe4\x2d\x2f\x08"
	     "\x44\x64\x85\xa6\x42\xfe\xdb\x11\xb5\xbc\x1a\x83\xc4\x58\x7a\xd1"
	     "\x0d\x64\xa2\xdb\xd5\x4c\x3a\xd7\x2a\xd8\xc7\x0b\xf5\x33\x80\xe2"
	     "\xaa\xc6\xaf\xad\xee\x39\x83\x1c\xfb\xda\x39\xd6\xe4\xbe\x6f\xde"
	     "\x46\xd1\xdc\xc8\x77\x3b\x42\x3a\x19\x83\xdb\x79\x5e\x9d\xd9\xef"
	     "\xe7\x54\x9e\xfa\xde\x56\x26\x0e\x33\x44\xb6\xac\xc7\x25\x1f\x0b"
	     "\xd0\x12\x4a\x75\x6f\xef\x9f\xb7\xe7\xf1\xc3\xb2\x77\xa7\x01\xc1"
	     "\x24\xf2\x60\xf2\x04\xa4\x07\x47\x25\x81\x2a\x51\x36\x1d\xb3\x42"
	     "\x3c\xc5\x12\xbb\x53\x89\xc3\x00\xb4\xe8\x0e\xd8\xed\x5d\x9c\x18"
	     "\x08\x16\x05\x57\x22\x99\xc1\xcb\x97\x4d\x7d\x42\xb0\xbf\xf3\x35"
	     "\x9e\xc9\xc0\x4e\xe6\x42\xec\x49\x06\x5e\x50\x7f\x6c\x86\x20\x9e"
	     "\xe0\x49\x6a\x1d\x50\x45\x52\x3d\x66\xdc\x66\xb2\xa1\x57\xbb\x1d"
	     "\x65\x30\xf9\x3e\x6c\x8b\xf8\x07\x5e\xf9\xb0\xcf\x15\x1f\xa4\x22"
	     "\xe6\x28\xdf\x40\x44\x27\x80\x99\xbf\xcc\x86\x94\x3d\xe1\x97\xbc"
	     "\x64\x7b\xc9\x8a\xc8\xb7\xdd\x8e\xd9\x95\xfa\x97\x44\x59\x70\xb6"
	     "\xa4\x59\xdd\x94\x2a\x4e\x71\x76\xe1\xf8\xff\x2e\xef\xf7\xf3\x26"
	     "\x0d\xeb\x63\x9b\x34\x74\xcf\xc7\xe4\x99\xb3\x8b\xe5\xf8\xfa\xf8"
	     "\x60\x6a\x8f\x16\xfe\xb3\x12\xe8\x66\x5c\x26\x61\x11\xaa\x14\x85"
	     "\x5e\x4c\x0f\xf8\x47\x99\x49\x4c\x22\xa5\x37\xb7\x4b\xed\x1d\x33"
	     "\x54\xe4\x6f\xfe\x4b\x7e\x95\x44\x56\xa9\xa9\x48\x6f\x40\x08\x42"
	     "\x28\xa0\xd6\x8e\x52\x72\x84\x87\xd5\xcc\x08\xf4\x34\x9b\xc7\xf7"
	     "\x92\xac\x5d\x5a\x30\x4c\x87\xc2\x2c\x94\x68\x10\x03\x02\x8c\x38"
	     "\x6e\xf6\xca\xd1\xd6\x55\x02\x3c\x3a\xf4\x21\x1c\x90\x03\x7c\x33"
	     "\xcd\x81\x93\x3f\x4c\xa1\xe6\x7c\x7d\x27\xaf\x97\x7a\xd8\xb0\x32"
	     "\xcb\x1f\xb6\x5d\xb0\x7c\x9e\xf7\x15\x6a\x29\xc9\xb3\x13\x8e\xb2"
	     "\x2e\xf2\x9d\xb2\x64\xe9\xc7\x13\x86\xc3\x46\x97\x9e\x53\x37\xac"
	     "\xd1\x75\x0b\xe2\x20\x4a\x5e\x3a\x47\xdf\xb3\x67\x78\x23\x22\xa6"
	     "\xb6\x69\x83\x1b\x41\xe1\xdc\xe3\xbb\x63\x8e\x6a\x70\xe2\x91\x30"
	     "\xa2\x48\x43\x07\x9b\xf1\x87\x9d\xd9\x2f\x36\x62\xaf\x51\x70\x10"
	     "\x5d\x68\x42\x6e\x95\xe6\x86\x85\x90\x08\x80\x42\xc6\x11\x7d\x38"
	     "\x37\x94\xc7\x7a\x9d\xec\xff\x87\x15\x0d\x30\x42\x7d\xbf\xb0\x21"
	     "\xa8\x28\xc3\xa9\x56\x31\xda\x2e\x91\x8d\x0d\x78\x93\xe2\x4d\x7f"
	     "\xb3\xdb\x87\x5b\xf2\x2a\x55\x54\xc4\x1a\x47\x9b\x51\x79\x2c\x55"
	     "\xaa\xaf\x57\x89\x90\x32\x26\xfa\xcf\x26\x3d\xbd\xc3\x59\x7e\x4b"
	     "\x53\x7a\xa3\xf8\xec\x31\x3e\x9f\x22\x33\x41\xec\xe5\xf1\x50\x41"
	     "\x57\x64\x37\x0e\x24\x0f\x03\x0c\x93\xaf\xd7\x44\xef\x33\xf9\x96"
	     "\x8e\x63\x9c\x79\x09\xdf\x58\xdb\xcd\xd2\xf2\x81\xed\x97\x28\x5e"
	     "\x6d\x99\x45\xad\xf1\xa4\xe8\x59\xaa\xeb\x59\xbd\xfa\x54\x27\xe2"
	     "\x01\x6d\xfa\x08\xe5\x1a\x46\x92\x31\xa0\xfd\x2e\x25\x21\x5e\x9d"
	     "\x1d\x41\x82\x16\x21\xea\xe8\x29\x5a\xe7\xd4\xd8\x7c\x91\x6d\xa8"
	     "\x69\xa8\x6b\xac\xf9\x3d\x2c\x07\xfa\xaf\x65\x5e\x84\x63\x5c\x1c"
	     "\xb6\x71\x1c\x6e\x1a\x1a\xcf\x5f\x39\x08\x02\xeb\xde\x75\x28\x44"
	     "\x90\x3b\x6d\x09\xf5\xa4\x3a\x00\x99\xc1\x6e\xb8\x23\x47\xb6\xe6"
	     "\x44\x69\x24\x86\x0d\x93\x7d\xc2\x1a\xe7\xf8\x97\x97\x6d\x2f\x7d"
	     "\x6a\x3e\xd7\x6d\x94\xbe\x9b\xfa\x7d\x40\xff\x35\x7e\x2d\x9c\x50"
	     "\x71\xf8\x99\x5f\xf3\x53\x16\x59\x77\x72\x4b\xef\x8d\x21\x8c\x28"
	     "\x15\x8a\x57\x6d\x3d\x35\xf4\x2c\x37\x39\x0e\x97\x8f\xd3\x45\x17"
	     "\xac\x0b\xa2\xaa\x16\xe7\x98\x70\xbd\x91\x4b\x7b\xc9\x14\x5b\x85"
	     "\x29\xe5\xd9\x20\xa6\xf3\x89\x63\xc8\x01\x94\x2c\x20\xb0\xf1\x1c"
	     "\x61\x90\x5b\xe8\xaf\xdb\xe2\x20\x62\x4f\xd0\x3c\x51\x1f\x5f\x14"
	     "\x28\x13\x07\x5e\xbc\x4a\xa0\x46\xa7\x9c\xc7\x00\x24\xb4\x67\x0d"
	     "\x9c\xe8\x05\xa7\xe3\x37\x94\xd3\x97\xc5\xd6\x1d\xb3\x79\x0c\x7f"
	     "\xf0\xeb\x68\xa7\xa9\x75\x30\x3c\xf1\xf8\xfb\x96\x4b\xa5\x99\xb1"
	     "\xf0\x71\xab\xb4\x8a\xa6\x5d\x97\xb5\x6f\x42\x81\xef\x38\x0e\x41"
	     "\x21\xcd\x15\xe4\xf1\xdc\x02\xc2\xe5\xe4\xb2\xf2\xdb\xc5\x42\x0f"
	     "\xeb\x00\xb8\x49\x48\x90\x97\x59\x69\x8c\x3d\x90\xfa\x7f\xaf\x75"
	     "\xff\xe8\x0f\xff\x13\x2f\x7a\xca\x33\xbc\xb0\x41\x7b\xd4\x94\xce"
	     "\xd5\xd7\x53\xe5\x36\x9d\xea\x0c\x13\x60\x63\xd9\x79\x83\x53\x03"
	     "\x3c\x5f\x33\x6b\x16\xd2\xd6\x07\xbd\xbc\xa9\xcd\xd6\x21\x58\x8a"
	     "\x90\xac\x85\x7c\x0a\x85\xf1\x02\xfd\x6c\x5b\xac\x92\xa5\x6f\xfc"
	     "\x02\x60\xb3\x05\x91\x31\x80\xd5\x75\x82\xa8\xab\xcc\x7b\x47\xdb"
	     "\x79\xaf\x85\xea\x75\x2d\xf4\x46\x45\xa6\x24\x35\x2d\xba\x9e\x19"
	     "\x4e\x48\xb3\x5f\x40\x96\x6c\x73\x44\x09\x67\x27\xee\xdb\xdb\xca"
	     "\xf6\x14\x3f\x32\xdf\x3b\xd3\xcb\x87\x48\x6b\x03\xda\xfe\xb5\x20"
	     "\xf8\x72\xd0\xe4\x65\xfd\x14\x5d\x9b\xd3\xc5\x67\x4b\x41\xf4\xcb"
	     "\xac\xe4\xce\x61\xe0\x06\x8e\xb6\xca\xef\xa5\xe4\x70\xd4\xff\x4f"
	     "\x4c\x81\xb7\xc7\xc1\xe3\x1d\x6b\xfe\x44\x37\xfe\x7e\xb4\xec\xda"
	     "\x55\xbc\x26\xdd\x44\x08\x28\x49\x19\xc9\x52\x20\x4a\x5e\xe1\xae"
	     "\x6b\x87\x35\xa2\x9a\x02\x17\xb0\x74\xe1\x19\x92\xca\x5d\x34\x81"
	     "\xd7\x3f\xb1\x14\xe2\x44\x97\x7f\xab\x0a\xc3\x44\x8d\x9c\xe6\x27"
	     "\x84\x5e\x2e\xe1\x0e\x7b\x74\xee\x47\xe2\x6f\x79\x3a\x0d\x25\x90"
	     "\x9b\x32\x13\x78\xd3\xd1\x19\x10\xb7\x88\xc4\xb4\xb8\x0f\x84\x0c"
	     "\xc4\x6d\xb3\xbc\xbc\xea\x1d\x84\xfe\x6f\x95\x1e\x44\x78\xbb\x73"
	     "\xd0\x08\x8b\xbe\xae\x91\x6f\x84\xeb\x6c\x05\x58\x3a\xb7\x68\xee"
	     "\x62\x14\x0b\xba\xd3\xd6\x78\x23\x35\x03\x2f\xed\xac\xf2\x35\xa7"
	     "\x0b\xf5\x37\xa1\xa5\x4d\x57\x0e\x2e\x5e\x0d\x0c\xd6\x04\xc8\x07"
	     "\x60\xe6\xda\xd9\x55\x16\xf6\xc6\x0c\x53\xc0\x46\xa8\x91\xa6\xdd"
	     "\x22\x99\x88\x9f\x23\x93\x64\x81\xdb\x4b\xe8\x99\x1e\x16\x5c\xb5"
	     "\x08\xc0\x74\xf9\x79\xd3\xc1\x6a\xe5\x38\x6b\xcd\xf0\xcb\x1e\xb8"
	     "\x5c\x6b\x14\xd6\x85\x58\xa3\xc8\x3d\x5b\x14\x50\xc4\x70\x97\xb2"
	     "\xcd\x93\xd6\x3a\x42\x2f\x3a\x88\x28\x8a\xd0\xa7\xe9\x98\x87\xbe"
	     "\x66\x89\x69\xd9\xbd\x11\x91\xd3\x9b\xc4\x22\x14\xe6\x8f\xc8\xbd"
	     "\x95\x5d\x3d\xd2\x1d\x10\xbd\x61\x69\x87\xf9\x4a\x9c\x61\x5b\x80"
	     "\x9b\x3f\x32\xce\x17\xcd\x98\x79\xb9\xeb\x45\xf7\x54\x3c\xc9\xa6"
	     "\x2e\xb4\x0b\x0d\x84\x0d\x35\xcc\x64\x5f\x5a\x92\xf8\xcb\xd6\xfa"
	     "\xae\xd9\xce\x9c\x41\x83\xc2\xab\xd5\x73\x21\xea\x95\xd5\x73\x2f"
	     "\xf1\xd0\x71\x95\x45\x07\x7c\x1f\xcc\xba\xf3\xb0\x75\xfa\x1a\xd4"
	     "\x89\x66\x4e\x93\x93\x80\xb7\x67\xd5\xdf\x99\xd8\x43\x6d\xf6\xde"
	     "\x19\xc8\x13\x56\xf8\x63\xe5\xf2\x53\xfd\x2e\x62\x8d\xde\x92\xd4"
	     "\x98\xfc\x1c\x36\x08\x3d\x33\x0f\xee\xbc\xb8\x2f\x49\xa2\xb3\xb4"
	     "\x9b\x0c\x7e\x16\xd4\xe8\xc0\x98\xd0\x5c\xda\x25\xaa\xe4\x60\x95"
	     "\xe6\xc5\x7b\x62\xa5\xe9\xbc\xa1\x5e\x11\x4f\x5d\xb4\x19\x29\x33"
	     "\x6f\xff\xf4\xef\x7f\x62\x2b\xf2\xbd\x47\x90\xcb\xba\xbc\xe8\xa8"
	     "\xd7\xcf\xac\x44\x05\x67\xf3\xd3\x93\x22\xc0\x48\x23\xdf\xef\x54"
	     "\x3d\x64\xc7\x13\xfd\x4f\x0a\x21\x20\x2b\xad\x34\x88\x02\x69\x84"
	     "\xaf\x8a\xfb\xbb\x97\x7d\xca\x0e\x80\xdc\x8a\x50\x51\x49\xc8\x1f"
	     "\xbd\x35\x46\x11\xf6\xdf\x52\x29\xc2\x52\xc4\xae\xd5\x16\xbb\x67"
	     "\x21\x7b\x25\xeb\xc4\x21\x0d\xc7\x3f\x1b\xae\x42\xc2\x21\x58\x1e"
	     "\x32\x46\x0d\x72\x15\x89\x75\xaa\xd9\x0b\x2d\xa7\x25\x3a\x2a\x3d"
	     "\xc6\x46\xd1\xa3\x9d\xed\x52\xd8\xbb\x53\xb9\x1f\x6d\xfc\x63\xb7"
	     "\x28\xa1\x78\x78\x18\x43\xa2\x3d\x6f\xcd\x6a\x40\x6a\xc2\x44\x88"
	     "\x59\x03\xdd\xd3\x47\x8d\x3d\xe2\x24\xc2\x11\x00\xb1\x2e\xc7\x24"
	     "\x81\x4b\xc0\x9c\x90\x64\xb7\x8b\xfb\x3a\xdc\x69\xe1\x4c\xf4\xc4"
	     "\xc7\x78\x5a\xca\xf8\x6f\x16\xa0\xb8\xe3\xb0\x66\x94\xab\x7a\x86"
	     "\x5c\xa8\xb9\x7f\xf7\x6a\x64\x1d\x8f\x03\xcf\x4b\x4b\xb0\xd8\xc2"
	     "\xc2\xea\x02\x5a\xaa\x14\x19\x8f\x60\x49\x5b\xf4\x05\xc8\x98\x30"
	     "\xeb\xbc\x00\x12\xa4\xc0\x27\x5f\x38\x91\x89\x73\xfa\xb7\xc1\x3a"
	     "\xf4\x7c\x18\xe4\xe7\x7f\x8c\x82\x51\xb0\x64\x2b\x9d\xa3\x98\x47"
	     "\x67\xe4\x0f\x1f\x30\x8e\x78\x0a\x3e\xc4\xa8\xc4\x1c\x79\x0a\x09"
	     "\x91\xbd\xa1\x6a\xab\x47\xce\x54\x24\xda\x4b\xa6\x7b\x40\x43\x92"
	     "\xd5\x43\xbb\x50\x53\x54\x1d\xf0\x0c\x87\x01\xc0\x82\x0a\xf0\xd8"
	     "\x7d\xd7\x22\xfc\x33\x59\xf7\x54\x87\x5c\x45\xc8\x2d\x92\x6d\x3a"
	     "\xfc\x90\x03\xa9\x23\xa8\xa3\xa5\x00\x3d\x3f\x7d\x9f\x6c\x7a\xba"
	     "\xdd\xff\xfd\x09\x04\x9b\x82\xea\x9f\x0e\x05\x92\xed\x08\x24\x1d"
	     "\x00\xfa\xac\x61\xf0\xe7\xc7\x37\x7e\x0d\x73\x38\x3e\x7a\x63\xcb"
	     "\x4e\x26\x21\xd6\x50\x12\xe4\x95\x1e\xc7\xd0\xed\xcb\xf9\xbf\x1d"
	     "\x34\x81\x81\xa4\x0e\xeb\xf6\xe8\xc5\x1e\x98\xca\x79\xad\x15\x3c"
	     "\xdc\xf5\xe3\xa9\xfe\xe9\x4c\x6a\xc5\x79\x55\x89\x18\xf2\xd1\x86"
	     "\x4e\xd6\x95\x38\xf8\x78\x0b\xfc\x50\x40\x72\xd4\x3c\x45\xdf\xaa"
	     "\x13\x1e\x9e\x74\xff\xde\xa9\x33\x5e\xa8\xa1\x70\x75\xd7\x7c\x4f"
	     "\x71\xf1\x03\xb3\xfb\x0f\xad\x19\x4f\x5d\xfe\xbb\xc5\x98\x81\x11"
	     "\xee\x24\x24\x13\xa4\xa4\xfe\x40\x4b\x20\x38\x61\x6e\xe3\xec\x05"
	     "\x15\x46\x35\x2d\xd4\x61\x7d\x0f\x4a\x34\xf1\x23\x87\x5e\xc6\x8e"
	     "\xf5\x77\x8c\x03\x4b\x58\xe2\x3b\x35\x08\x2f\x40\x60\xd6\xe3\x3a"
	     "\xf0\xb4\xdd\xb9\x18\xfc\x7d\x79\x7a\xf3\x0a\x5c\x89\x2e\xcc\x30"
	     "\x9b\x8f\x7e\x3c\xd3\x1d\x45\xcb\xfb\xb5\x09\xe8\x6e\xd4\x87\xb0"
	     "\xc5\x5c\x0e\x70\xec\xd9\x04\xb2\xc1\x44\x9b\x78\x4f\x46\x92\x4b"
	     "\x18\x0b\x90\xaa\xe3\xf9\x11\x8d\xf8\x9e\x5e\xc6\x2e\xfb\x9f\xba"
	     "\x31\x49\x9e\x9d\xb7\xb8\x5c\x0b\x09\xcf\x45\xb0\x82\x9d\x6a\xd7"
	     "\xeb\x1d\x79\x6f\x26\x1a\xec\x5d\x16\xb1\xa8\xc5\x90\x00\x9a\x79"
	     "\x51\x9b\xab\x79\x09\xa1\x7f\x65\x74\xa0\xf8\x54\xfd\xf3\x74\x52"
	     "\x80\x09\x25\x33\xa4\xd5\x22\xba\x79\xc0\x77\xe0\x56\xfb\x19\xa9"
	     "\x8f\x23\x16\x29\xa9\x39\x54\x10\x0d\x85\xb9\x20\xd3\x27\xa2\x7d"
	     "\xe4\x81\x9e\x0a\x4a\xc4\xf6\xef\xc7\x0b\xb8\x6a\x50\xa9\xed\x72"
	     "\xb4\x79\x01\xf3\xc3\x89\x5d\x19\x5e\x9a\xf1\x35\x14\xcc\x04\x4c"
	     "\x2d\xe5\x0f\xd8\x07\xee\x50\xee\x3f\xaf\x74\x86\xcd\xc5\xf8\x34"
	     "\x0f\xeb\x32\x7f\x36\xcc\x03\xa8\xbc\xde\xd9\xf8\xfc\x01\x73\x83"
	     "\x74\x4a\x26\xdc\x70\x1a\x76\x71\x5a\xee\xac\x0b\xe0\xdb\x16\xd4"
	     "\x26\x7c\xfb\xb8\x96\x74\x00\x7d\xb8\x5c\x4a\x07\xfc\x9e\xc9\xff"
	     "\x4a\xd0\xaf\x86\x4b\xb0\xc6\x29\x8e\x3f\x2e\x9f\xa0\x33\xd3\xdf"
	     "\xb9\xe0\xd5\x48\xb9\xf0\x08\x7e\xae\xdd\xd0\x2e\xf5\x41\xd8\xa3"
	     "\x9a\xff\x76\x93\x04\x7a\xef\xd2\xf5\xb8\x05\xa1\x55\x8e\xe8\x03"
	     "\x9f\x6e\xd9\xc4\x75\xd0\xfe\x75\xd1\x77\x3f\x22\xbe\x2e\xc7\x27"
	     "\x10\x4e\x2c\xab\x2f\x38\x53\xcb\xe7\x27\x9d\xff\xd8\x86\xb7\xca"
	     "\xc1\x7e\xeb\x23\xd8\x8f\x5e\x85\xa3\x2a\xa3\xe6\xc8\x8f\x71\x30"
	     "\x98\x91\x1d\x11\x8d\x76\xdf\x42\xe8\x72\x6b\x41\x3a\x70\x46\x26"
	     "\x78\x16\xc8\x6a\x21\x66\x81\xdc\xec\xac\x3e\x5d\x8a\x9a\x01\x03"
	     "\xee\xee\x0a\x34\x1a\xad\xdc\xc9\x4f\x3d\x22\x3b\x48\x67\x43\x7d"
	     "\x7e\x53\x08\x91\x26\xbf\x7d\x60\x4a\xf4\xfb\x21\xcf\x74\xc9\xc6"
	     "\x47\x9b\x39\x58\x61\xde\x5d\x3c\xbe\x60\x2e\x2b\x6c\x08\x42\x3e"
	     "\xe3\x03\x76\x6d\xa8\xac\xba\x11\xb2\xce\x0f\xdb\x95\x6a\x33\xf4"
	     "\x00\xc9\x66\x13\xd4\x5f\xab\xe7\x1b\x17\x62\xa1\xe6\x4a\x82\x67"
	     "\x28\xa3\x7b\xf7\x4b\xfc\x57\xcb\x16\x3e\xfe\xb5\xa4\xef\x3b\x35"
	     "\xdd\x14\xd5\x48\x41\x5e\xa5\x62\xcc\x18\xba\x97\x6c\x75\xe9\x39"
	     "\x63\xd4\x9d\xce\x2d\x9d\x4c\xc3\x82\xb1\x31\x69\x91\x90\x3e\x6d"
	     "\x66\xc7\xea\xd3\x88\x02\xd3\x39\x10\x22\x04\x1a\xbf\xc5\x0d\x02"
	     "\xb8\x7c\x9a\x86\xc5\x04\x87\xef\xaf\x75\xe5\xc7\xe6\x21\xba\xa1"
	     "\xbd\x9c\x14\x90\xc3\x99\x80\xbb\x61\xe3\xe2\xc1\x79\xf0\xc6\x46"
	     "\x4f\x38\x86\x35\x93\x1a\x55\xd8\x76\x0f\x7c\xa2\xae\x3b\x6a\xa8"
	     "\xe4\x74\x74\xc2\x50\x38\xcf\xc5\xda\x85\xee\xe0\x2f\x23\x1a\x83"
	     "\x17\x54\xa7\x34\xa0\x1f\xd7\xd3\xb2\x53\xe3\x77\xfe\x8e\x97\x49"
	     "\xc5\x5b\xe0\x5f\xcb\x2c\x90\xe9\x09\x83\x40\x89\x81\x51\xf6\x40"
	     "\x64\x2c\x89\xca\xcf\x61\xda\x04\x0d\xaf\x3b\x48\xf9\x8a\xfa\x74"
	     "\x6a\x6a\xe5\x29\x5e\x3a\x6b\xee\xb5\x5e\xa3\x0c\x78\xf1\xe0\xcb"
	     "\x74\x69\x2c\x7b\xf6\x73\xc1\x53\x20\x8c\x15\x54\x03\x5d\x96\xe3"
	     "\x6a\x8d\x19\x5d\xec\x5a\xd4\xe3\x6b\x1a\x72\x27\x73\x73\xbb\x2e"
	     "\x7c\x4e\xbb\x63\x26\xdb\x9d\x9c\xde\x67\xd8\x50\x7c\x85\x1a\xff"
	     "\xf0\xd0\x12\x4d\x3e\x69\xb1\x2d\x2a\xd1\x28\x4a\xc9\x82\x1e\xd5"
	     "\xfc\xa3\x64\x0b\x2f\xcc\x4c\xef\x17\xa6\xe5\x0b\xd5\x76\xb4\x33"
	     "\xff\x3b\x3d\xc3\xb0\x28\xf7\x38\xf5\x7a\x76\x2c\x84\xf7\xcc\x4f"
	     "\x3a\xdf\x68\xab\xbc\x6b\x24\xac\x84\x8a\x09\x5e\x0d\x89\x9d\x85"
	     "\x98\xf5\x61\x60\x38\xb0\xdd\x86\xe1\xb5\x42\xa0\x6c\x25\x77\x28"
	     "\x8f\xf1\x21\xb3\xa8\xed\x8c\xfd\x18\xcf\x23\xae\x2a\x9f\x1c\x67"
	     "\xc3\x17\x48\x73\x3b\x6e\xbf\x02\x88\xed\x4d\xa2\x86\x07\x6a\x5d"
	     "\x16\x55\x3b\x87\x29\x88\xd0\x8d\xe3\x30\x3b\x89\x22\x5c\xa2\xb6"
	     "\xbb\xcf\xda\xd8\x1f\xe4\x7f\xa5\x50\xb9\x8b\xc2\x61\x73\x86\x61"
	     "\x92\x44\x1b\xc2\x9d\x34\x72\x81\x30\xca\x22\x83\x78\xe0\xa5\xb5"
	     "\x39\xba\xa3\x46\x7a\x4c\xd9\x2b\x5e\xaf\x42\xdf\xba\x80\x4a\x83"
	     "\x9c\x20\xe0\x86\x90\x88\x4e\xbf\x7b\xa6\x44\x3d\x7f\x9c\xbe\xc8"
	     "\xdf\x74\xe8\x2c\xf7\x4b\xc0\x06\x3f\x9e\x23\x30\xe8\x31\xc6\x07"
	     "\x5b\xc9\x26\xf1\x1c\xc8\xbe\xe3\xc7\xcb\x4f\x1b\x37\x09\xf0\xcf"
	     "\x0c\x5e\x1d\xac\x1c\xb8\x91\xcf\x0a\x54\x09\x6c\xa9\x4e\x9f\xa1"
	     "\x36\xde\x41\xa8\x67\xe4\x4c\x65\xc9\x28\x20\x0f\xbd\x49\x74\x7f"
	     "\x83\x26\x48\xb1\xe6\xca\x6f\xd8\x88\x27\x94\xdd\xa4\xaa\x03\x43"
	     "\xc6\x19\xab\x8b\x83\x78\x4f\x2a\x51\x34\x98\xff\x27\x6b\xda\x39"
	     "\xe4\x2e\xf6\xe9\xe7\xd6\x96\x86\x60\x3b\xc3\x80\xb8\xba\x51\x77"
	     "\x8a\x79\x07\x62\xb1\xaa\xa9\x58\x96\x54\x84\x04\x9e\xee\x24\x8d"
	     "\xf1\xac\x34\x03\x43\x8e\xd0\x92\xdc\x4d\xbc\xc7\x15\xdc\x82\xfc"
	     "\x19\x9b\x7e\x43\xe1\x36\xc5\x6b\xfb\xb8\x9e\xf4\x22\x5e\x29\x6f"
	     "\x4c\x45\xdd\xfe\xcb\x17\x77\xee\xd3\x6b\x80\x50\x96\xd0\x07\xa3"
	     "\xb2\xf0\x68\xc7\x40\xeb\x24\x16\x23\xa7\x57\xc4\xbb\x9c\x69\x16"
	     "\x49\x30\xb8\xd9\x4d\x41\x4e\x22\xc6\x24\x7f\x44\x59\x4e\x3c\x46"
	     "\x61\x64\x60\xc5\x82\xc2\x60\xc5\x63\xe7\xb3\xe1\xb9\xd1\x71\x65"
	     "\x32\xaa\x88\xa1\x4e\x98\xdb\x0b\x29\xe8\x5a\xa4\xbc\x61\x20\x62"
	     "\xbc\xea\xb0\x90\x3c\x2f\x29\xef\x74\x87\x4b\x1b\x36\xf7\x01\xde"
	     "\x30\xf0\x00\xe3\xd7\x2d\xf6\x52\x09\xe1\xe5\x84\x13\x04\xfe\x1f"
	     "\x2c\x1c\xeb\xfb\x2c\x40\x7d\x4d\xf5\x9c\x6e\x47\xe9\x2d\xb0\xca"
	     "\x88\x37\xf4\x8d\xbe\x50\xfb\xec\x5e\xee\x5d\x0f\x1a\x71\x42\xe3"
	     "\xc0\x9c\x57\xe3\x50\xba\x98\x0d\x79\xfa\x56\xe9\x7a\x09\x03\x75"
	     "\x21\x0f\x97\x69\xe0\xed\x9d\x10\x8f\x64\xdd\xd5\x10\xdc\x3b\x21"
	     "\x08\xc2\x7e\x7e\xac\x6e\x7c\x0b\xb9\x71\x65\x3e\xfa\x1c\xf3\x13"
	     "\xdc\x03\x9c\xf2\xa4\x21\xf1\xfc\x07\xaa\xef\xa0\x55\xd7\x21\xfc"
	     "\x0c\x17\x6a\x06\x57\x6b\x0e\x8e\x1d\xea\xb2\xc8\x23\xb3\xad\x42"
	     "\x00\x41\xe2\x9c\x1f\x2e\xa8\x51\x19\x3d\xdb\x24\x73\x4a\x8f\x08"
	     "\x74\xb5\x6b\x58\x48\x7a\x4e\x7b\xa6\x5c\x4b\x81\xd6\x38\xf9\xe5"
	     "\x66\xe3\x15\x9a\x74\x49\x68\x8e\x5b\x1e\xdd\x3f\x4f\xc8\xe3\x0c"
	     "\x73\x32\xf0\x35\x16\x8c\xd9\x84\xc3\x9a\x65\x16\x93\x65\x0f\x47"
	     "\x15\x97\xc8\xb4\x6a\x7f\x91\x0a\x05\xf2\x08\x36\xf0\x7e\x2c\xcb"
	     "\x79\xc4\xc4\xcb\x5a\xa0\x25\xc9\xe9\xac\x35\x8f\x56\xf6\x87\x4e"
	     "\x9b\x4b\x9d\x1a\x38\xb0\x43\x41\x94\x07\x17\x78\xec\xd3\x8c\x89"
	     "\x58\x09\xef\x07\xe1\xd8\x65\x97\x3e\xce\x61\xad\x51\x0a\x48\x3d"
	     "\x64\xd5\x6e\xa1\xe5\x86\x06\xb7\x22\xcd\x03\x44\xec\x94\x52\x15"
	     "\x5d\xf6\x9b\xc6\xb6\x22\x49\x5e\xb1\xdb\x01\x24\xd2\xc2\xac\x13"
	     "\xdc\xec\xb3\x93\x4c\x88\x3b\x11\x6d\x77\xab\x44\x67\x8c\xca\xa8"
	     "\x18\x47\xe3\xdc\x58\x13\x45\x43\xb3\x7f\xdf\x3a\xdf\xef\x0e\x4f"
	     "\xc0\xf3\x95\xb6\x7d\xdf\x97\xd7\x0d\x03\x9d\x14\x76\xd6\xae\x5f"
	     "\xb4\x13\x7f\x2a\xdb\x00\xa2\xe7\xfd\x33\xa6\x4d\x3f\xc1\x14\x58"
	     "\x7c\x55\x30\x6a\xae\x71\x0f\x73\xe3\x64\x1f\xc5\x2e\x6f\x82\x82"
	     "\x62\xd2\x02\xde\xf8\x29\x1e\xd5\x71\x3a\xbe\xcd\x43\xf8\x71\x5e"
	     "\xa1\xce\x68\xca\xcb\x23\xfa\x83\xba\xc2\x0f\x9b\x6b\x86\x5d\x9f",
	 STR(iv, iv_size,
	     "\x12\x34\x56\x78\x90\xab\xce\xf0\x00\x00\x00\x00\x00\x00\x00\x00"),
	}
};

const struct cipher_vectors_st aes128_xts_vectors[] = {
	{
	 STR(key, key_size,
	     "\xa1\xb9\x0c\xba\x3f\x06\xac\x35\x3b\x2c\x34\x38\x76\x08\x17\x62"
             "\x09\x09\x23\x02\x6e\x91\x77\x18\x15\xf2\x9d\xab\x01\x93\x2f\x2f"),
	 STR(plaintext, plaintext_size,
	     "\xeb\xab\xce\x95\xb1\x4d\x3c\x8d\x6f\xb3\x50\x39\x07\x90\x31\x1c"),
	 .ciphertext = (uint8_t *)
	     "\x77\x8a\xe8\xb4\x3c\xb9\x8d\x5a\x82\x50\x81\xd5\xbe\x47\x1c\x63",
	 STR(iv, iv_size,
	     "\x4f\xae\xf7\x11\x7c\xda\x59\xc6\x6e\x4b\x92\x01\x3e\x76\x8a\xd5"),
	 },
	{
	 STR(key, key_size,
	     "\x75\x03\x72\xc3\xd8\x2f\x63\x38\x28\x67\xbe\x66\x62\xac\xfa\x4a"
             "\x25\x9b\xe3\xfa\x9b\xc6\x62\xa1\x15\x4f\xfa\xae\xd8\xb4\x48\xa5"),
	 STR(plaintext, plaintext_size,
	     "\xd8\xe3\xa5\x65\x59\xa4\x36\xce\x0d\x8b\x21\x2c\x80\xa8\x8b\x23"
             "\xaf\x62\xb0\xe5\x98\xf2\x08\xe0\x3c\x1f\x2e\x9f\xa5\x63\xa5\x4b"),
	 .ciphertext = (uint8_t *)
	     "\x49\x5f\x78\x55\x53\x5e\xfd\x13\x34\x64\xdc\x9a\x9a\xbf\x8a\x0f"
             "\x28\xfa\xcb\xce\x21\xbd\x3c\x22\x17\x8e\xc4\x89\xb7\x99\xe4\x91",
	 STR(iv, iv_size,
	     "\x93\xa2\x92\x54\xc4\x7e\x42\x60\x66\x96\x21\x30\x7d\x4f\x5c\xd3"),
	 },
};

const struct cipher_vectors_st aes256_xts_vectors[] = {
	{
	 STR(key, key_size,
             "\x1e\xa6\x61\xc5\x8d\x94\x3a\x0e\x48\x01\xe4\x2f\x4b\x09\x47\x14"
             "\x9e\x7f\x9f\x8e\x3e\x68\xd0\xc7\x50\x52\x10\xbd\x31\x1a\x0e\x7c"
             "\xd6\xe1\x3f\xfd\xf2\x41\x8d\x8d\x19\x11\xc0\x04\xcd\xa5\x8d\xa3"
             "\xd6\x19\xb7\xe2\xb9\x14\x1e\x58\x31\x8e\xea\x39\x2c\xf4\x1b\x08"),
	 STR(plaintext, plaintext_size,
	     "\x2e\xed\xea\x52\xcd\x82\x15\xe1\xac\xc6\x47\xe8\x10\xbb\xc3\x64"
             "\x2e\x87\x28\x7f\x8d\x2e\x57\xe3\x6c\x0a\x24\xfb\xc1\x2a\x20\x2e"),
	 .ciphertext = (uint8_t *)
	     "\xcb\xaa\xd0\xe2\xf6\xce\xa3\xf5\x0b\x37\xf9\x34\xd4\x6a\x9b\x13"
             "\x0b\x9d\x54\xf0\x7e\x34\xf3\x6a\xf7\x93\xe8\x6f\x73\xc6\xd7\xdb",
	 STR(iv, iv_size,
	     "\xad\xf8\xd9\x26\x27\x46\x4a\xd2\xf0\x42\x8e\x84\xa9\xf8\x75\x64"),
	 },
};

const struct cipher_aead_vectors_st aes128_siv_vectors[] = {
	{
	 STR(key, key_size,
	     "\x7f\x7e\x7d\x7c\x7b\x7a\x79\x78\x77\x76\x75\x74\x73\x72\x71\x70"
	     "\x40\x41\x42\x43\x44\x45\x46\x47\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"),
	 STR(auth, auth_size,
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xdd\xee\xff"
	     "\xde\xad\xda\xda\xde\xad\xda\xda\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x77\x66\x55\x44\x33\x22\x11\x00"),
	 STR(plaintext, plaintext_size,
	     "\x74\x68\x69\x73\x20\x69\x73\x20\x73\x6f\x6d\x65\x20\x70\x6c\x61"
	     "\x69\x6e\x74\x65\x78\x74\x20\x74\x6f\x20\x65\x6e\x63\x72\x79\x70"
	     "\x74\x20\x75\x73\x69\x6e\x67\x20\x53\x49\x56\x2d\x41\x45\x53"),
	 .ciphertext = (uint8_t *)
	     "\xa4\xff\xb8\x7f\xdb\xa9\x7c\x89\x44\xa6\x23\x25\xf1\x33\xb4\xe0"
	     "\x1c\xa5\x52\x76\xe2\x26\x1c\x1a\x1d\x1d\x42\x48\xd1\xda\x30\xba"
	     "\x52\xb9\xc8\xd7\x95\x5d\x65\xc8\xd2\xce\x6e\xb7\xe3\x67\xd0",
	 STR(iv, iv_size,
	     "\x02\x03\x04"),
	 .tag_size = 16,
	 .tag = (void *)
	     "\xf1\xdb\xa3\x3d\xe5\xb3\x36\x9e\x88\x3f\x67\xb6\xfc\x82\x3c\xee",
	 .tag_prepended = 1,
	}
};

const struct cipher_aead_vectors_st aes256_siv_vectors[] = {
	{
	 STR(key, key_size,
	     "\xc2\x7d\xf2\xfd\xae\xc3\x5d\x4a\x2a\x41\x2a\x50\xc3\xe8\xc4\x7d"
	     "\x2d\x56\x8e\x91\xa3\x8e\x54\x14\x8a\xbd\xc0\xb6\xe8\x6c\xaf\x87"
	     "\x69\x5c\x0a\x8a\xdf\x4c\x5f\x8e\xb2\xc6\xc8\xb1\x36\x52\x98\x64"
	     "\xf3\xb8\x4b\x3a\xe8\xe3\x67\x6c\xe7\x60\xc4\x61\xf3\xa1\x3e\x83"),
	 STR(auth, auth_size,
	     "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xdd\xee\xff"
	     "\xde\xad\xda\xda\xde\xad\xda\xda\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	     "\x77\x66\x55\x44\x33\x22\x11\x00"),
	 STR(plaintext, plaintext_size,
	     "\x74\x68\x69\x73\x20\x69\x73\x20\x73\x6f\x6d\x65\x20\x70\x6c\x61"
	     "\x69\x6e\x74\x65\x78\x74\x20\x74\x6f\x20\x65\x6e\x63\x72\x79\x70"
	     "\x74\x20\x75\x73\x69\x6e\x67\x20\x53\x49\x56\x2d\x41\x45\x53"),
	 .ciphertext = (uint8_t *)
	     "\x50\x93\x3d\xa8\x04\x7b\xc3\x06\xfa\xba\xf0\xc3\xd9\xfa\x84\x71"
	 "\xc7\x0a\x7d\xef\x39\xa2\xf9\x1d\x68\xa2\x02\x1c\x99\xac\x7e\x2a\x24"
	 "\x53\x5a\x13\x4b\xa2\x3e\xc1\x57\x87\xce\xbe\x5c\x53\xcc",
	 STR(iv, iv_size,
	     "\x09\xf9\x11\x02\x9d\x74\xe3\x5b\xd8\x41\x56\xc5\x63\x56\x88\xc0"),
	 .tag_size = 16,
	 .tag = (void *)
	     "\x5a\x97\x9b\x0d\xa5\x8f\xde\x80\x51\x62\x1a\xe6\xbf\x96\xfe\xda",
	 .tag_prepended = 1,
	}
};

const struct cipher_vectors_st chacha20_32_vectors[] = { /* RFC8439 */
	{
	 STR(key, key_size,
	     "\x00\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f\x10\x11\x12\x13\x14\x15\x16\x17\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"),
	 STR(plaintext, plaintext_size,
	     "\x4c\x61\x64\x69\x65\x73\x20\x61\x6e\x64\x20\x47\x65\x6e\x74\x6c\x65\x6d\x65\x6e\x20\x6f\x66\x20\x74\x68\x65\x20\x63\x6c\x61\x73\x73\x20\x6f\x66\x20\x27\x39\x39\x3a\x20\x49\x66\x20\x49\x20\x63\x6f\x75\x6c\x64\x20\x6f\x66\x66\x65\x72\x20\x79\x6f\x75\x20\x6f\x6e\x6c\x79\x20\x6f\x6e\x65\x20\x74\x69\x70\x20\x66\x6f\x72\x20\x74\x68\x65\x20\x66\x75\x74\x75\x72\x65\x2c\x20\x73\x75\x6e\x73\x63\x72\x65\x65\x6e\x20\x77\x6f\x75\x6c\x64\x20\x62\x65\x20\x69\x74\x2e"),
	 .ciphertext = (uint8_t *)
	     "\x6e\x2e\x35\x9a\x25\x68\xf9\x80\x41\xba\x07\x28\xdd\x0d\x69\x81\xe9\x7e\x7a\xec\x1d\x43\x60\xc2\x0a\x27\xaf\xcc\xfd\x9f\xae\x0b\xf9\x1b\x65\xc5\x52\x47\x33\xab\x8f\x59\x3d\xab\xcd\x62\xb3\x57\x16\x39\xd6\x24\xe6\x51\x52\xab\x8f\x53\x0c\x35\x9f\x08\x61\xd8\x07\xca\x0d\xbf\x50\x0d\x6a\x61\x56\xa3\x8e\x08\x8a\x22\xb6\x5e\x52\xbc\x51\x4d\x16\xcc\xf8\x06\x81\x8c\xe9\x1a\xb7\x79\x37\x36\x5a\xf9\x0b\xbf\x74\xa3\x5b\xe6\xb4\x0b\x8e\xed\xf2\x78\x5e\x42\x87\x4d",
	 STR(iv, iv_size,
	     "\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x4a\x00\x00\x00\x00")
	},
};

static int test_cipher(gnutls_cipher_algorithm_t cipher,
		       const struct cipher_vectors_st *vectors,
		       size_t vectors_size, unsigned flags)
{
	gnutls_cipher_hd_t hd;
	int ret;
	unsigned int i;
	uint8_t tmp[4096 + 16];
	gnutls_datum_t key, iv = {NULL, 0};

	for (i = 0; i < vectors_size; i++) {
		key.data = (void *) vectors[i].key;
		key.size = vectors[i].key_size;

		if (vectors[i].iv != NULL) {
			iv.data = (void *) vectors[i].iv;
			iv.size = gnutls_cipher_get_iv_size(cipher);
		}

		if (iv.size != vectors[i].iv_size)
			return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);

		ret = gnutls_cipher_init(&hd, cipher, &key, &iv);
		if (ret < 0) {
			_gnutls_debug_log("error initializing: %s\n",
					  gnutls_cipher_get_name(cipher));
			return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
		}

		ret =
		    gnutls_cipher_encrypt2(hd,
					   vectors[i].plaintext,
					   vectors[i].plaintext_size,
					   tmp, sizeof(tmp));
		if (ret < 0)
			return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);

		if (memcmp
		    (tmp, vectors[i].ciphertext,
		     vectors[i].plaintext_size) != 0) {
			_gnutls_debug_log("%s test vector %d failed!\n",
					  gnutls_cipher_get_name(cipher),
					  i);
			return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
		}

		/* check in-place encryption */
		if (cipher != GNUTLS_CIPHER_ARCFOUR_128) { /* arcfour is stream */
			if (cipher == GNUTLS_CIPHER_MAGMA_CTR_ACPKM ||
			    cipher == GNUTLS_CIPHER_KUZNYECHIK_CTR_ACPKM)
				_gnutls_cipher_set_key(hd, (void*)vectors[i].key, vectors[i].key_size);
			gnutls_cipher_set_iv(hd, (void*)vectors[i].iv, vectors[i].iv_size);

			memcpy(tmp, vectors[i].plaintext, vectors[i].plaintext_size);
			ret = gnutls_cipher_encrypt(hd, tmp, vectors[i].plaintext_size);
			if (ret < 0)
				return
				    gnutls_assert_val
				    (GNUTLS_E_SELF_TEST_ERROR);

			if (memcmp(tmp, vectors[i].ciphertext, vectors[i].plaintext_size) != 0) {
				_gnutls_debug_log("%s vector %d in-place encryption failed!\n", gnutls_cipher_get_name(cipher), i);
				return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
			}
		}

		/* check the internal IV */
		if (vectors[i].internal_iv_size > 0) {
			ret = _gnutls_cipher_get_iv(hd, tmp, sizeof(tmp));
			if (ret < 0)
				return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);

			if (memcmp(tmp, vectors[i].internal_iv, ret) != 0) {
				_gnutls_debug_log("%s vector %d internal IV check failed!\n",
						  gnutls_cipher_get_name(cipher),
						  i);
				return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
			}
		}

		gnutls_cipher_deinit(hd);
	}

	iv.size = gnutls_cipher_get_iv_size(cipher);

	for (i = 0; i < vectors_size; i++) {
		key.data = (void *) vectors[i].key;
		key.size = vectors[i].key_size;

		iv.data = (void *) vectors[i].iv;

		ret = gnutls_cipher_init(&hd, cipher, &key, &iv);
		if (ret < 0)
			return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);

		ret =
		    gnutls_cipher_decrypt2(hd,
					   vectors[i].ciphertext,
					   vectors[i].plaintext_size, tmp,
					   sizeof(tmp));
		if (ret < 0) {
			_gnutls_debug_log
			    ("%s decryption of test vector %d failed!\n",
			     gnutls_cipher_get_name(cipher), i);
			return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
		}

		if (memcmp
		    (tmp, vectors[i].plaintext,
		     vectors[i].plaintext_size) != 0) {
			_gnutls_debug_log("%s test vector %d failed!\n",
					  gnutls_cipher_get_name(cipher),
					  i);
			return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
		}

		/* check in-place decryption */
		if (cipher != GNUTLS_CIPHER_ARCFOUR_128) { /* arcfour is stream */
			if (cipher == GNUTLS_CIPHER_MAGMA_CTR_ACPKM ||
			    cipher == GNUTLS_CIPHER_KUZNYECHIK_CTR_ACPKM)
				_gnutls_cipher_set_key(hd, (void*)vectors[i].key, vectors[i].key_size);
			gnutls_cipher_set_iv(hd, (void*)vectors[i].iv, vectors[i].iv_size);

			memcpy(tmp, vectors[i].ciphertext, vectors[i].plaintext_size);
			ret = gnutls_cipher_decrypt(hd, tmp, vectors[i].plaintext_size);
			if (ret < 0)
				return
				    gnutls_assert_val
				    (GNUTLS_E_SELF_TEST_ERROR);

			if (memcmp(tmp, vectors[i].plaintext, vectors[i].plaintext_size) != 0) {
				_gnutls_debug_log("%s vector %d in-place decryption failed!\n", gnutls_cipher_get_name(cipher), i);
				return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
			}
		}

		gnutls_cipher_deinit(hd);
	}

	_gnutls_debug_log
	    ("%s self check succeeded\n",
	     gnutls_cipher_get_name(cipher));

	return 0;
}

static int test_cipher_all_block_sizes(gnutls_cipher_algorithm_t cipher,
				       const struct cipher_vectors_st *vectors,
				       size_t vectors_size, unsigned flags)
{
	gnutls_cipher_hd_t hd;
	int ret;
	unsigned int i;
	uint8_t tmp[384];
	gnutls_datum_t key, iv = {NULL, 0};
	size_t block;
	size_t offset;

	for (i = 0; i < vectors_size; i++) {
		for (block = 1; block <= vectors[i].plaintext_size; block++) {
			key.data = (void *) vectors[i].key;
			key.size = vectors[i].key_size;

			iv.data = (void *) vectors[i].iv;
			iv.size = gnutls_cipher_get_iv_size(cipher);

			if (iv.size != vectors[i].iv_size)
				return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);

			ret = gnutls_cipher_init(&hd, cipher, &key, &iv);
			if (ret < 0) {
				_gnutls_debug_log("error initializing: %s\n",
						  gnutls_cipher_get_name(cipher));
				return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
			}

			for (offset = 0;
			     offset < vectors[i].plaintext_size;
			     offset += block) {
				ret =
				    gnutls_cipher_encrypt2(hd,
							   vectors[i].plaintext + offset,
							   MIN(block, vectors[i].plaintext_size - offset),
							   tmp + offset,
							   sizeof(tmp) - offset);
				if (ret < 0)
					return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
			}

			if (memcmp
			    (tmp, vectors[i].ciphertext,
			     vectors[i].plaintext_size) != 0) {
				_gnutls_debug_log("%s encryption of test vector %d failed with block size %d/%d!\n",
						  gnutls_cipher_get_name(cipher),
						  i, (int)block, (int)vectors[i].plaintext_size);
				return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
			}

			gnutls_cipher_deinit(hd);
		}
	}

	for (i = 0; i < vectors_size; i++) {
		for (block = 1; block <= vectors[i].plaintext_size; block++) {
			key.data = (void *) vectors[i].key;
			key.size = vectors[i].key_size;

			iv.data = (void *) vectors[i].iv;
			iv.size = gnutls_cipher_get_iv_size(cipher);

			ret = gnutls_cipher_init(&hd, cipher, &key, &iv);
			if (ret < 0)
				return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);

			for (offset = 0;
			     offset + block <= vectors[i].plaintext_size;
			     offset += block) {
				ret =
				    gnutls_cipher_decrypt2(hd,
							   vectors[i].ciphertext + offset,
							   MIN(block, vectors[i].plaintext_size - offset),
							   tmp + offset,
							   sizeof(tmp) - offset);
				if (ret < 0)
					return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
			}

			if (memcmp
			    (tmp, vectors[i].plaintext,
			     vectors[i].plaintext_size) != 0) {
				_gnutls_debug_log("%s decryption of test vector %d failed with block size %d!\n",
						  gnutls_cipher_get_name(cipher),
						  i, (int)block);
				return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
			}

			gnutls_cipher_deinit(hd);
		}
	}

	_gnutls_debug_log
	    ("%s self check succeeded\n",
	     gnutls_cipher_get_name(cipher));

	return 0;
}

/* AEAD modes (compat APIs) */
static int test_cipher_aead_compat(gnutls_cipher_algorithm_t cipher,
			    const struct cipher_aead_vectors_st *vectors,
			    size_t vectors_size)
{
	gnutls_cipher_hd_t hd;
	int ret;
	unsigned int i;
	uint8_t tmp[384];
	uint8_t tmp2[384];
	gnutls_datum_t key, iv;
	unsigned tag_size;

	_gnutls_debug_log("compat: running tests for: %s\n",
				  gnutls_cipher_get_name(cipher));

	for (i = 0; i < vectors_size; i++) {
		memset(tmp, 0, sizeof(tmp));
		key.data = (void *) vectors[i].key;
		key.size = vectors[i].key_size;

		iv.data = (void *) vectors[i].iv;
		iv.size = vectors[i].iv_size;
		tag_size = vectors[i].tag_size;


		if (tag_size > gnutls_cipher_get_tag_size(cipher)) {
			return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
		}

		ret = gnutls_cipher_init(&hd, cipher, &key, &iv);
		if (ret < 0) {
			if (vectors[i].compat_apis == 0) {
				return 0; /* expected */
			} else {
				_gnutls_debug_log("compat: error initializing: %s\n",
					  gnutls_cipher_get_name(cipher));
				return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
			}
		}

		if (vectors[i].compat_apis == 0) {
			_gnutls_debug_log("compat: initialized but shouldn't: %s\n",
				  gnutls_cipher_get_name(cipher));
			return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
		}

		if (vectors[i].auth_size) {
			ret = gnutls_cipher_add_auth(hd, vectors[i].auth, vectors[i].auth_size);
			if (ret < 0)
				return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
		}

		ret = gnutls_cipher_encrypt2(hd, vectors[i].plaintext, vectors[i].plaintext_size,
					     tmp, sizeof(tmp));
		if (ret < 0)
			return
			    gnutls_assert_val
			    (GNUTLS_E_SELF_TEST_ERROR);

		ret = gnutls_cipher_tag(hd, tmp+vectors[i].plaintext_size, tag_size);
		if (ret < 0)
			return
			    gnutls_assert_val
			    (GNUTLS_E_SELF_TEST_ERROR);

		if (memcmp(tmp+vectors[i].plaintext_size, vectors[i].tag, tag_size) != 0) {
			_gnutls_debug_log
			    ("compat: %s test vector %d failed (tag)!\n",
			     gnutls_cipher_get_name(cipher), i);
			return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
		}

		if (vectors[i].plaintext_size > 0) {
			if (memcmp
			    (tmp, vectors[i].ciphertext,
			     vectors[i].plaintext_size) != 0) {
				_gnutls_debug_log
				    ("compat: %s test vector %d failed!\n",
				     gnutls_cipher_get_name(cipher), i);

				return
				    gnutls_assert_val
				    (GNUTLS_E_SELF_TEST_ERROR);
			}
		}

		if (vectors[i].plaintext_size > 0) {
			/* check inplace encryption */
			gnutls_cipher_set_iv(hd, (void*)vectors[i].iv, vectors[i].iv_size);
			memcpy(tmp2, vectors[i].plaintext, vectors[i].plaintext_size);

			ret = gnutls_cipher_encrypt(hd, tmp2, vectors[i].plaintext_size);
			if (ret < 0)
				return
				    gnutls_assert_val
				    (GNUTLS_E_SELF_TEST_ERROR);

			if (memcmp(tmp, tmp2, vectors[i].plaintext_size) != 0) {
				_gnutls_debug_log("compat: %s vector %d in-place encryption failed!\n", gnutls_cipher_get_name(cipher), i);
				return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
			}

			/* check decryption with separate buffers */
			gnutls_cipher_set_iv(hd, (void*)vectors[i].iv, vectors[i].iv_size);

			if (vectors[i].auth_size) {
				ret = gnutls_cipher_add_auth(hd, vectors[i].auth, vectors[i].auth_size);
				if (ret < 0)
					return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
			}

			ret =
			    gnutls_cipher_decrypt2(hd, tmp, vectors[i].plaintext_size,
						   tmp2, sizeof(tmp2));
			if (ret < 0)
				return
				    gnutls_assert_val
				    (GNUTLS_E_SELF_TEST_ERROR);

			if (memcmp(tmp2, vectors[i].plaintext, vectors[i].plaintext_size) != 0) {
				_gnutls_debug_log("compat: %s test vector %d failed (decryption)!\n",
					gnutls_cipher_get_name(cipher), i);
				return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
			}

			/* check in-place decryption */
			if (vectors[i].plaintext_size > 0) {
				gnutls_cipher_set_iv(hd, (void*)vectors[i].iv, vectors[i].iv_size);

				if (vectors[i].auth_size) {
					ret = gnutls_cipher_add_auth(hd, vectors[i].auth, vectors[i].auth_size);
					if (ret < 0)
						return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
				}

				memcpy(tmp2, tmp, vectors[i].plaintext_size);
				ret = gnutls_cipher_decrypt(hd, tmp2, vectors[i].plaintext_size);
				if (ret < 0)
					return
					    gnutls_assert_val
					    (GNUTLS_E_SELF_TEST_ERROR);

				if (memcmp(tmp2, vectors[i].plaintext, vectors[i].plaintext_size) != 0) {
					_gnutls_debug_log("compat: %s vector %d in-place decryption failed!\n", gnutls_cipher_get_name(cipher), i);
					return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
				}
			}
		}

		gnutls_cipher_deinit(hd);
	}

	_gnutls_debug_log
	    ("%s compat self check succeeded\n",
	     gnutls_cipher_get_name(cipher));

	return 0;

}

#define IOV_PARTS 8
/* AEAD modes - scatter read */
static int test_cipher_aead_scatter(gnutls_cipher_algorithm_t cipher,
				    const struct cipher_aead_vectors_st *vectors,
				    size_t vectors_size, unsigned flags)
{
	gnutls_aead_cipher_hd_t hd;
	int ret;
	unsigned int i, z;
	uint8_t tmp[384];
	gnutls_datum_t key, iv;
	size_t s;
	unsigned tag_size;
	giovec_t auth_iov[IOV_PARTS];
	int auth_iov_len;
	int iov_len;
	giovec_t iov[IOV_PARTS];
	const uint8_t *tag;
	uint8_t *ciphertext;

	_gnutls_debug_log("running scatter (iovec) tests for: %s\n",
				  gnutls_cipher_get_name(cipher));

	for (i = 0; i < vectors_size; i++) {
		memset(tmp, 0, sizeof(tmp));
		key.data = (void *) vectors[i].key;
		key.size = vectors[i].key_size;

		iv.data = (void *) vectors[i].iv;
		iv.size = vectors[i].iv_size;
		tag_size = vectors[i].tag_size;

		if (tag_size > gnutls_cipher_get_tag_size(cipher)) {
			return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
		}

		ret = gnutls_aead_cipher_init(&hd, cipher, &key);
		if (ret < 0) {
			_gnutls_debug_log("error initializing: %s\n",
					  gnutls_cipher_get_name(cipher));
			return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
		}

		s = sizeof(tmp);

		/* single vector */
		auth_iov_len = 1;
		auth_iov[0].iov_base = (void*)vectors[i].auth;
		auth_iov[0].iov_len = vectors[i].auth_size;

		iov_len = 1;
		iov[0].iov_base = (void*)vectors[i].plaintext;
		iov[0].iov_len = vectors[i].plaintext_size;

		ret =
		    gnutls_aead_cipher_encryptv(hd,
						iv.data, iv.size,
						auth_iov, auth_iov_len,
						vectors[i].tag_size,
						iov, iov_len,
						tmp, &s);
		if (ret < 0)
			return
			    gnutls_assert_val
			    (GNUTLS_E_SELF_TEST_ERROR);

		if (s != vectors[i].plaintext_size + tag_size) {
			return
			    gnutls_assert_val
			    (GNUTLS_E_SELF_TEST_ERROR);
		}

		if (vectors[i].tag_prepended)
			tag = tmp;
		else
			tag = tmp+vectors[i].plaintext_size;

		if (memcmp(tag, vectors[i].tag, tag_size) != 0) {
			_gnutls_debug_log
			    ("%s test vector %d failed (tag)!\n",
			     gnutls_cipher_get_name(cipher), i);
			return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
		}

		if (vectors[i].tag_prepended)
			ciphertext = tmp+vectors[i].tag_size;
		else
			ciphertext = tmp;

		if (vectors[i].plaintext_size > 0) {
			if (memcmp
			    (ciphertext, vectors[i].ciphertext,
			     vectors[i].plaintext_size) != 0) {
				_gnutls_debug_log
				    ("%s test vector %d failed!\n",
				     gnutls_cipher_get_name(cipher), i);

				return
				    gnutls_assert_val
				    (GNUTLS_E_SELF_TEST_ERROR);
			}
		}

		/* multi-vector */
		auth_iov_len = 0;
		if (vectors[i].auth_size > IOV_PARTS) {
			unsigned split = vectors[i].auth_size / IOV_PARTS;
			assert(split>0);
			for (z=0;z<IOV_PARTS;z++) {
				auth_iov[z].iov_base = (void*)(vectors[i].auth+(z*split));
				if (z==IOV_PARTS-1)
					auth_iov[z].iov_len = vectors[i].auth_size - z*split;
				else
					auth_iov[z].iov_len = split;
				auth_iov_len++;
			}
		} else {
			auth_iov_len = 1;
			auth_iov[0].iov_base = (void*)vectors[i].auth;
			auth_iov[0].iov_len = vectors[i].auth_size;
		}

		iov_len = 0;
		if (vectors[i].plaintext_size > IOV_PARTS) {
			unsigned split = vectors[i].plaintext_size / IOV_PARTS;
			assert(split>0);

			for (z=0;z<IOV_PARTS;z++) {
				iov[z].iov_base = (void*)(vectors[i].plaintext+(z*split));
				if (z==IOV_PARTS-1)
					iov[z].iov_len = vectors[i].plaintext_size - z*split;
				else
					iov[z].iov_len = split;
				iov_len++;
			}
		} else {
			iov_len = 1;
			iov[0].iov_base = (void*)vectors[i].plaintext;
			iov[0].iov_len = vectors[i].plaintext_size;
		}

		s = sizeof(tmp);

		ret =
		    gnutls_aead_cipher_encryptv(hd,
						iv.data, iv.size,
						auth_iov, auth_iov_len,
						vectors[i].tag_size,
						iov, iov_len,
						tmp, &s);
		if (ret < 0)
			return
			    gnutls_assert_val
			    (GNUTLS_E_SELF_TEST_ERROR);

		if (s != vectors[i].plaintext_size + tag_size) {
			return
			    gnutls_assert_val
			    (GNUTLS_E_SELF_TEST_ERROR);
		}

		if (vectors[i].tag_prepended)
			tag = tmp;
		else
			tag = tmp+vectors[i].plaintext_size;

		if (memcmp(tag, vectors[i].tag, tag_size) != 0) {
			_gnutls_debug_log
			    ("%s test vector %d failed (tag)!\n",
			     gnutls_cipher_get_name(cipher), i);
			return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
		}

		if (vectors[i].tag_prepended)
			ciphertext = tmp+vectors[i].tag_size;
		else
			ciphertext = tmp;

		if (vectors[i].plaintext_size > 0) {
			if (memcmp
			    (ciphertext, vectors[i].ciphertext,
			     vectors[i].plaintext_size) != 0) {
				_gnutls_debug_log
				    ("%s test vector %d failed!\n",
				     gnutls_cipher_get_name(cipher), i);

				return
				    gnutls_assert_val
				    (GNUTLS_E_SELF_TEST_ERROR);
			}
		}



		gnutls_aead_cipher_deinit(hd);
	}

	_gnutls_debug_log
	    ("%s scatter self check succeeded\n",
	     gnutls_cipher_get_name(cipher));

	if (flags & GNUTLS_SELF_TEST_FLAG_NO_COMPAT)
		return 0;
	else
		return test_cipher_aead_compat(cipher, vectors, vectors_size);
}

/* AEAD modes */
static int test_cipher_aead(gnutls_cipher_algorithm_t cipher,
			    const struct cipher_aead_vectors_st *vectors,
			    size_t vectors_size, unsigned flags)
{
	gnutls_aead_cipher_hd_t hd;
	int ret;
	unsigned int i;
	uint8_t tmp[384];
	uint8_t tmp2[384];
	gnutls_datum_t key, iv;
	size_t s, s2;
	const uint8_t *tag;
	unsigned tag_size;
	uint8_t *ciphertext;

	_gnutls_debug_log("running tests for: %s\n",
				  gnutls_cipher_get_name(cipher));

	for (i = 0; i < vectors_size; i++) {
		memset(tmp, 0, sizeof(tmp));
		key.data = (void *) vectors[i].key;
		key.size = vectors[i].key_size;

		iv.data = (void *) vectors[i].iv;
		iv.size = vectors[i].iv_size;
		tag_size = vectors[i].tag_size;

		if (tag_size > gnutls_cipher_get_tag_size(cipher)) {
			return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
		}
#if 0
		if (iv.size != vectors[i].iv_size)
			return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
#endif
		ret = gnutls_aead_cipher_init(&hd, cipher, &key);
		if (ret < 0) {
			_gnutls_debug_log("error initializing: %s\n",
					  gnutls_cipher_get_name(cipher));
			return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
		}

		s = sizeof(tmp);

		ret =
		    gnutls_aead_cipher_encrypt(hd,
					   iv.data, iv.size,
					   vectors[i].auth, vectors[i].auth_size,
					   vectors[i].tag_size,
					   vectors[i].plaintext,
					   vectors[i].plaintext_size,
					   tmp, &s);
		if (ret < 0)
			return
			    gnutls_assert_val
			    (GNUTLS_E_SELF_TEST_ERROR);

		if (s != vectors[i].plaintext_size + tag_size) {
			return
			    gnutls_assert_val
			    (GNUTLS_E_SELF_TEST_ERROR);
		}

		if (vectors[i].tag_prepended)
			tag = tmp;
		else
			tag = tmp+vectors[i].plaintext_size;

		if (memcmp(tag, vectors[i].tag, tag_size) != 0) {
			_gnutls_debug_log
			    ("%s test vector %d failed (tag)!\n",
			     gnutls_cipher_get_name(cipher), i);
			return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
		}

		if (vectors[i].tag_prepended)
			ciphertext = tmp+vectors[i].tag_size;
		else
			ciphertext = tmp;

		if (vectors[i].plaintext_size > 0) {
			if (memcmp
			    (ciphertext, vectors[i].ciphertext,
			     vectors[i].plaintext_size) != 0) {
				_gnutls_debug_log
				    ("%s test vector %d failed!\n",
				     gnutls_cipher_get_name(cipher), i);

				return
				    gnutls_assert_val
				    (GNUTLS_E_SELF_TEST_ERROR);
			}
		}

		/* check decryption */
		{
			s2 = sizeof(tmp2);
			ret =
			    gnutls_aead_cipher_decrypt(hd,
						   iv.data, iv.size,
						   vectors[i].auth, vectors[i].auth_size,
						   vectors[i].tag_size,
						   tmp, s,
						   tmp2, &s2);
			if (ret < 0)
				return
				    gnutls_assert_val
				    (GNUTLS_E_SELF_TEST_ERROR);

			if (s2 != vectors[i].plaintext_size && memcmp(tmp, vectors[i].plaintext, vectors[i].plaintext_size) != 0) {
				_gnutls_debug_log("%s test vector %d failed (decryption)!\n",
					gnutls_cipher_get_name(cipher), i);
				return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
			}

			/* test tag verification */
			if (s > 0) {
				(*ciphertext)++;

				s2 = sizeof(tmp2);
				ret =
				    gnutls_aead_cipher_decrypt(hd,
							   iv.data, iv.size,
							   vectors[i].auth, vectors[i].auth_size,
							   vectors[i].tag_size,
							   tmp, s,
							   tmp2, &s2);

				if (ret >= 0) {
					_gnutls_debug_log("%s: tag check failed\n", gnutls_cipher_get_name(cipher));
					return
					    gnutls_assert_val
					    (GNUTLS_E_SELF_TEST_ERROR);
				}
			}
		}

		gnutls_aead_cipher_deinit(hd);
	}

	_gnutls_debug_log
	    ("%s self check succeeded\n",
	     gnutls_cipher_get_name(cipher));

	return test_cipher_aead_scatter(cipher, vectors, vectors_size, flags);
}



struct hash_vectors_st {
	const uint8_t *plaintext;
	unsigned int plaintext_size;
	const uint8_t *output;
	unsigned int output_size;
};

const struct hash_vectors_st md5_vectors[] = {
	{
	 STR(plaintext, plaintext_size, "abcdefghijklmnopqrstuvwxyz"),
	 STR(output, output_size,
	     "\xc3\xfc\xd3\xd7\x61\x92\xe4\x00\x7d\xfb\x49\x6c\xca\x67\xe1\x3b"),
	 },
};

const struct hash_vectors_st sha1_vectors[] = {
	{
	 STR(plaintext, plaintext_size, "what do ya want for nothing?"),
	 STR(output, output_size,
	     "\x8f\x82\x03\x94\xf9\x53\x35\x18\x20\x45\xda\x24\xf3\x4d\xe5\x2b\xf8\xbc\x34\x32"),
	 },
	{
	 STR(plaintext, plaintext_size,
	     "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopqabcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopqabcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq"),
	 STR(output, output_size,
	     "\xbe\xae\xd1\x6d\x65\x8e\xc7\x92\x9e\xdf\xd6\x2b\xfa\xfe\xac\x29\x9f\x0d\x74\x4d"),
	 },
};

const struct hash_vectors_st sha224_vectors[] = {
	{
	 STR(plaintext, plaintext_size,
	     "The quick brown fox jumps over the lazy dog"),
	 STR(output, output_size,
	     "\x73\x0e\x10\x9b\xd7\xa8\xa3\x2b\x1c\xb9\xd9\xa0\x9a\xa2\x32\x5d\x24\x30\x58\x7d\xdb\xc0\xc3\x8b\xad\x91\x15\x25"),
	 },
};

const struct hash_vectors_st sha256_vectors[] = {
	{
	 STR(plaintext, plaintext_size,
	     "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq"),
	 STR(output, output_size,
	     "\x24\x8d\x6a\x61\xd2\x06\x38\xb8\xe5\xc0\x26\x93\x0c\x3e\x60\x39\xa3\x3c\xe4\x59\x64\xff\x21\x67\xf6\xec\xed\xd4\x19\xdb\x06\xc1"),
	 },
	{
	 STR(plaintext, plaintext_size,
	     "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopqabcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopqabcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq"),
	 STR(output, output_size,
	     "\x50\xea\x82\x5d\x96\x84\xf4\x22\x9c\xa2\x9f\x1f\xec\x51\x15\x93\xe2\x81\xe4\x6a\x14\x0d\x81\xe0\x00\x5f\x8f\x68\x86\x69\xa0\x6c"),
	 },
};

const struct hash_vectors_st sha384_vectors[] = {
	{
	 STR(plaintext, plaintext_size,
	     "The quick brown fox jumps over the lazy dog"),
	 STR(output, output_size,
	     "\xca\x73\x7f\x10\x14\xa4\x8f\x4c\x0b\x6d\xd4\x3c\xb1\x77\xb0\xaf\xd9\xe5\x16\x93\x67\x54\x4c\x49\x40\x11\xe3\x31\x7d\xbf\x9a\x50\x9c\xb1\xe5\xdc\x1e\x85\xa9\x41\xbb\xee\x3d\x7f\x2a\xfb\xc9\xb1"),
	 },
};

const struct hash_vectors_st sha512_vectors[] = {
	{
	 STR(plaintext, plaintext_size,
	     "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu"),
	 STR(output, output_size,
	     "\x8e\x95\x9b\x75\xda\xe3\x13\xda\x8c\xf4\xf7\x28\x14\xfc\x14\x3f\x8f\x77\x79\xc6\xeb\x9f\x7f\xa1\x72\x99\xae\xad\xb6\x88\x90\x18\x50\x1d\x28\x9e\x49\x00\xf7\xe4\x33\x1b\x99\xde\xc4\xb5\x43\x3a\xc7\xd3\x29\xee\xb6\xdd\x26\x54\x5e\x96\xe5\x5b\x87\x4b\xe9\x09"),
	 },
};

const struct hash_vectors_st sha3_224_vectors[] = {
	{
	 STR(plaintext, plaintext_size,
	     "\xC1\xEC\xFD\xFC"),
	 STR(output, output_size,
	     "\xA3\x3C\x58\xDF\x8A\x80\x26\xF0\xF9\x59\x19\x66\xBD\x6D\x00\xEE\xD3\xB1\xE8\x29\x58\x0A\xB9\xBE\x26\x8C\xAF\x39"),
	 },
};

const struct hash_vectors_st sha3_256_vectors[] = {
	{
	 STR(plaintext, plaintext_size,
	     "\xC1\xEC\xFD\xFC"),
	 STR(output, output_size,
	     "\xC5\x85\x9B\xE8\x25\x60\xCC\x87\x89\x13\x3F\x7C\x83\x4A\x6E\xE6\x28\xE3\x51\xE5\x04\xE6\x01\xE8\x05\x9A\x06\x67\xFF\x62\xC1\x24"),
	 }
};

const struct hash_vectors_st sha3_384_vectors[] = {
	{
	 STR(plaintext, plaintext_size,
	     "\x4A\x4F\x20\x24\x84\x51\x25\x26"),
	 STR(output, output_size,
	     "\x89\xDB\xF4\xC3\x9B\x8F\xB4\x6F\xDF\x0A\x69\x26\xCE\xC0\x35\x5A\x4B\xDB\xF9\xC6\xA4\x46\xE1\x40\xB7\xC8\xBD\x08\xFF\x6F\x48\x9F\x20\x5D\xAF\x8E\xFF\xE1\x60\xF4\x37\xF6\x74\x91\xEF\x89\x7C\x23"),
	 },
};

const struct hash_vectors_st sha3_512_vectors[] = {
	{
	 STR(plaintext, plaintext_size,
	     "\x82\xE1\x92\xE4\x04\x3D\xDC\xD1\x2E\xCF\x52\x96\x9D\x0F\x80\x7E\xED"),
	 STR(output, output_size,
	     "\x96\x44\xE3\xC9\x0B\x67\xE2\x21\x24\xE9\x6D\xFE\xDC\xE5\x3D\x33\xC4\x60\xF1\x32\x86\x8F\x09\x75\xD1\x8B\x22\xCF\xD5\x9F\x63\x7D\xD8\x5A\xA4\x05\xE3\x98\x08\xA4\x55\x70\xA4\x98\xC0\xB8\xF2\xCB\xA5\x9F\x8E\x14\x37\xEA\xEF\x89\xF2\x0B\x88\x29\x8A\xDF\xA2\xDE"),
	 },
};

const struct hash_vectors_st gostr_94_vectors[] = {
	{
	 STR(plaintext, plaintext_size,
	     "The quick brown fox jumps over the lazy dog"),
	 STR(output, output_size,
	     "\x90\x04\x29\x4a\x36\x1a\x50\x8c\x58\x6f\xe5\x3d\x1f\x1b\x02\x74\x67\x65\xe7\x1b\x76\x54\x72\x78\x6e\x47\x70\xd5\x65\x83\x0a\x76"),
	},
};

/* GOST R 34.11-2012 */
const struct hash_vectors_st streebog_512_vectors[] = {
	{
            STR(plaintext, plaintext_size,
		"\xd1\xe5\x20\xe2\xe5\xf2\xf0\xe8\x2c\x20\xd1\xf2\xf0\xe8\xe1\xee"
		"\xe6\xe8\x20\xe2\xed\xf3\xf6\xe8\x2c\x20\xe2\xe5\xfe\xf2\xfa\x20"
		"\xf1\x20\xec\xee\xf0\xff\x20\xf1\xf2\xf0\xe5\xeb\xe0\xec\xe8\x20"
		"\xed\xe0\x20\xf5\xf0\xe0\xe1\xf0\xfb\xff\x20\xef\xeb\xfa\xea\xfb"
		"\x20\xc8\xe3\xee\xf0\xe5\xe2\xfb"),
            STR(output, output_size,
		"\x1e\x88\xe6\x22\x26\xbf\xca\x6f\x99\x94\xf1\xf2\xd5\x15\x69\xe0"
		"\xda\xf8\x47\x5a\x3b\x0f\xe6\x1a\x53\x00\xee\xe4\x6d\x96\x13\x76"
		"\x03\x5f\xe8\x35\x49\xad\xa2\xb8\x62\x0f\xcd\x7c\x49\x6c\xe5\xb3"
		"\x3f\x0c\xb9\xdd\xdc\x2b\x64\x60\x14\x3b\x03\xda\xba\xc9\xfb\x28"),
	},
	{
            STR(plaintext, plaintext_size,
		"\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
		"\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
		"\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
		"\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
		"\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
		"\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
		"\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
		"\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"),
            STR(output, output_size,
		"\x90\xa1\x61\xd1\x2a\xd3\x09\x49\x8d\x3f\xe5\xd4\x82\x02\xd8\xa4"
		"\xe9\xc4\x06\xd6\xa2\x64\xae\xab\x25\x8a\xc5\xec\xc3\x7a\x79\x62"
		"\xaa\xf9\x58\x7a\x5a\xbb\x09\xb6\xbb\x81\xec\x4b\x37\x52\xa3\xff"
		"\x5a\x83\x8e\xf1\x75\xbe\x57\x72\x05\x6b\xc5\xfe\x54\xfc\xfc\x7e"),
	},
};

/* GOST R 34.11-2012 */
const struct hash_vectors_st streebog_256_vectors[] = {
	{
            STR(plaintext, plaintext_size,
		"\xd1\xe5\x20\xe2\xe5\xf2\xf0\xe8\x2c\x20\xd1\xf2\xf0\xe8\xe1\xee"
		"\xe6\xe8\x20\xe2\xed\xf3\xf6\xe8\x2c\x20\xe2\xe5\xfe\xf2\xfa\x20"
		"\xf1\x20\xec\xee\xf0\xff\x20\xf1\xf2\xf0\xe5\xeb\xe0\xec\xe8\x20"
		"\xed\xe0\x20\xf5\xf0\xe0\xe1\xf0\xfb\xff\x20\xef\xeb\xfa\xea\xfb"
		"\x20\xc8\xe3\xee\xf0\xe5\xe2\xfb"),
            STR(output, output_size,
		"\x9d\xd2\xfe\x4e\x90\x40\x9e\x5d\xa8\x7f\x53\x97\x6d\x74\x05\xb0"
		"\xc0\xca\xc6\x28\xfc\x66\x9a\x74\x1d\x50\x06\x3c\x55\x7e\x8f\x50"),
	},
};

#define HASH_DATA_SIZE 64

/* SHA1 and other hashes */
static int test_digest(gnutls_digest_algorithm_t dig,
		       const struct hash_vectors_st *vectors,
		       size_t vectors_size, unsigned flags)
{
	uint8_t data[HASH_DATA_SIZE];
	unsigned int i;
	int ret;
	size_t data_size;
	gnutls_hash_hd_t hd;
	gnutls_hash_hd_t copy;

	if (_gnutls_digest_exists(dig) == 0)
		return 0;

	for (i = 0; i < vectors_size; i++) {
		ret = gnutls_hash_init(&hd, dig);
		if (ret < 0) {
			_gnutls_debug_log("error initializing: %s\n",
					  gnutls_digest_get_name(dig));
			return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
		}

		ret = gnutls_hash(hd, vectors[i].plaintext, 1);
		if (ret < 0)
			return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);

		copy = gnutls_hash_copy(hd);
		if (!copy) {
			_gnutls_debug_log("copy is not supported for: %s\n",
					  gnutls_digest_get_name(dig));
		}

		ret = gnutls_hash(hd,
				  &vectors[i].plaintext[1],
				  vectors[i].plaintext_size - 1);
		if (ret < 0)
			return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);

		gnutls_hash_deinit(hd, data);

		data_size = gnutls_hash_get_len(dig);
		if (data_size <= 0)
			return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);

		if (data_size != vectors[i].output_size ||
		    memcmp(data, vectors[i].output,
			   vectors[i].output_size) != 0) {
			_gnutls_debug_log("%s test vector %d failed!\n",
					  gnutls_digest_get_name(dig), i);
			return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
		}

		if (copy != NULL) {
			ret = gnutls_hash(copy,
					  &vectors[i].plaintext[1],
					  vectors[i].plaintext_size - 1);
			if (ret < 0)
				return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);

			memset(data, 0xaa, data_size);
			gnutls_hash_deinit(copy, data);

			if (memcmp(data, vectors[i].output,
			    vectors[i].output_size) != 0) {
				_gnutls_debug_log("%s copy test vector %d failed!\n",
						  gnutls_digest_get_name(dig), i);
				return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
			}
		}
	}

	_gnutls_debug_log("%s self check succeeded\n",
			  gnutls_digest_get_name(dig));

	return 0;
}


struct mac_vectors_st {
	const uint8_t *key;
	unsigned int key_size;
	const uint8_t *nonce;
	unsigned int nonce_size;
	const uint8_t *plaintext;
	unsigned int plaintext_size;
	const uint8_t *output;
	unsigned int output_size;
};

const struct mac_vectors_st hmac_md5_vectors[] = {
	{
	 STR(key, key_size, "Jefe"),
	 STR(plaintext, plaintext_size, "what do ya want for nothing?"),
	 STR(output, output_size,
	     "\x75\x0c\x78\x3e\x6a\xb0\xb5\x03\xea\xa8\x6e\x31\x0a\x5d\xb7\x38"),
	 },
};

const struct mac_vectors_st hmac_sha1_vectors[] = {
	{
	 STR(key, key_size,
	     "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"),
	 STR(plaintext, plaintext_size, "Hi There"),
	 STR(output, output_size,
	     "\x67\x5b\x0b\x3a\x1b\x4d\xdf\x4e\x12\x48\x72\xda\x6c\x2f\x63\x2b\xfe\xd9\x57\xe9"),
	 },
};

	    /* from rfc4231 */
const struct mac_vectors_st hmac_sha224_vectors[] = {
	{
	 STR(key, key_size,
	     "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"),
	 STR(plaintext, plaintext_size, "Hi There"),
	 STR(output, output_size,
	     "\x89\x6f\xb1\x12\x8a\xbb\xdf\x19\x68\x32\x10\x7c\xd4\x9d\xf3\x3f\x47\xb4\xb1\x16\x99\x12\xba\x4f\x53\x68\x4b\x22"),
	 },
};

const struct mac_vectors_st hmac_sha256_vectors[] = {
	{
	 STR(key, key_size,
	     "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"),
	 STR(plaintext, plaintext_size, "Hi There"),
	 STR(output, output_size,
	     "\xb0\x34\x4c\x61\xd8\xdb\x38\x53\x5c\xa8\xaf\xce\xaf\x0b\xf1\x2b\x88\x1d\xc2\x00\xc9\x83\x3d\xa7\x26\xe9\x37\x6c\x2e\x32\xcf\xf7"),
	 },
};

const struct mac_vectors_st hmac_sha384_vectors[] = {
	{
	 STR(key, key_size,
	     "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"),
	 STR(plaintext, plaintext_size, "Hi There"),
	 STR(output, output_size,
	     "\xaf\xd0\x39\x44\xd8\x48\x95\x62\x6b\x08\x25\xf4\xab\x46\x90\x7f\x15\xf9\xda\xdb\xe4\x10\x1e\xc6\x82\xaa\x03\x4c\x7c\xeb\xc5\x9c\xfa\xea\x9e\xa9\x07\x6e\xde\x7f\x4a\xf1\x52\xe8\xb2\xfa\x9c\xb6"),
	 },
};

const struct mac_vectors_st hmac_sha512_vectors[] = {
	{
	 STR(key, key_size,
	     "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"),
	 STR(plaintext, plaintext_size, "Hi There"),
	 STR(output, output_size,
	     "\x87\xaa\x7c\xde\xa5\xef\x61\x9d\x4f\xf0\xb4\x24\x1a\x1d\x6c\xb0\x23\x79\xf4\xe2\xce\x4e\xc2\x78\x7a\xd0\xb3\x05\x45\xe1\x7c\xde\xda\xa8\x33\xb7\xd6\xb8\xa7\x02\x03\x8b\x27\x4e\xae\xa3\xf4\xe4\xbe\x9d\x91\x4e\xeb\x61\xf1\x70\x2e\x69\x6c\x20\x3a\x12\x68\x54"),
	 },
};

/* Calculated */
const struct mac_vectors_st hmac_gostr_94_vectors[] = {
	{
	 STR(key, key_size,
	     "\x00\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
	     "\x10\x11\x12\x13\x14\x15\x16\x17\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"),
	 STR(plaintext, plaintext_size,
	     "\x01\x26\xbd\xb8\x78\x00\xaf\x21\x43\x41\x45\x65\x63\x78\x01\x00"),
	 STR(output, output_size,
	     "\xba\xd7\x0b\x61\xc4\x10\x95\xbc\x47\xe1\x14\x1c\xfa\xed\x42\x72"
	     "\x6a\x5c\xee\xbd\x62\xce\x75\xdb\xbb\x9a\xd7\x6c\xda\x9f\x72\xf7"),
	},
};

/* RFC 7836 */
const struct mac_vectors_st hmac_streebog_512_vectors[] = {
	{
	 STR(key, key_size,
	     "\x00\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
	     "\x10\x11\x12\x13\x14\x15\x16\x17\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"),
	 STR(plaintext, plaintext_size,
	     "\x01\x26\xbd\xb8\x78\x00\xaf\x21\x43\x41\x45\x65\x63\x78\x01\x00"),
	 STR(output, output_size,
	     "\xa5\x9b\xab\x22\xec\xae\x19\xc6\x5f\xbd\xe6\xe5\xf4\xe9\xf5\xd8"
	     "\x54\x9d\x31\xf0\x37\xf9\xdf\x9b\x90\x55\x00\xe1\x71\x92\x3a\x77"
	     "\x3d\x5f\x15\x30\xf2\xed\x7e\x96\x4c\xb2\xee\xdc\x29\xe9\xad\x2f"
	     "\x3a\xfe\x93\xb2\x81\x4f\x79\xf5\x00\x0f\xfc\x03\x66\xc2\x51\xe6"),
	},
};

/* RFC 7836 */
const struct mac_vectors_st hmac_streebog_256_vectors[] = {
	{
	 STR(key, key_size,
	     "\x00\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
	     "\x10\x11\x12\x13\x14\x15\x16\x17\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"),
	 STR(plaintext, plaintext_size,
	     "\x01\x26\xbd\xb8\x78\x00\xaf\x21\x43\x41\x45\x65\x63\x78\x01\x00"),
	 STR(output, output_size,
	     "\xa1\xaa\x5f\x7d\xe4\x02\xd7\xb3\xd3\x23\xf2\x99\x1c\x8d\x45\x34"
	     "\x01\x31\x37\x01\x0a\x83\x75\x4f\xd0\xaf\x6d\x7c\xd4\x92\x2e\xd9"),
	},
};

const struct mac_vectors_st aes_cmac_128_vectors[] = { /* NIST SP800-38A */
	{
	 STR(key, key_size,
	     "\x2b\x7e\x15\x16\x28\xae\xd2\xa6\xab\xf7\x15\x88\x09\xcf\x4f\x3c"),
	 STR(plaintext, plaintext_size,
             "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96\xe9\x3d\x7e\x11\x73\x93\x17\x2a"),
	 STR(output, output_size,
	     "\x07\x0a\x16\xb4\x6b\x4d\x41\x44\xf7\x9b\xdd\x9d\xd0\x4a\x28\x7c"),
	 },
};

const struct mac_vectors_st aes_cmac_256_vectors[] = { /* NIST SP800-38A */
	{
	 STR(key, key_size,
	     "\x60\x3d\xeb\x10\x15\xca\x71\xbe\x2b\x73\xae\xf0\x85\x7d\x77\x81"
             "\x1f\x35\x2c\x07\x3b\x61\x08\xd7\x2d\x98\x10\xa3\x09\x14\xdf\xf4"),
	 STR(plaintext, plaintext_size,
             "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96\xe9\x3d\x7e\x11\x73\x93\x17\x2a"),
	 STR(output, output_size,
	     "\x28\xa7\x02\x3f\x45\x2e\x8f\x82\xbd\x4b\xf2\x8d\x8c\x37\xc3\x5c"),
	 },
};

const struct mac_vectors_st aes_gmac_128_vectors[] = { /* NIST test vectors */
	{
	 STR(key, key_size,
	     "\x23\x70\xe3\x20\xd4\x34\x42\x08\xe0\xff\x56\x83\xf2\x43\xb2\x13"),
	 STR(nonce, nonce_size,
	     "\x04\xdb\xb8\x2f\x04\x4d\x30\x83\x1c\x44\x12\x28"),
	 STR(plaintext, plaintext_size,
	     "\xd4\x3a\x8e\x50\x89\xee\xa0\xd0\x26\xc0\x3a\x85\x17\x8b\x27\xda"),
	 STR(output, output_size,
	     "\x2a\x04\x9c\x04\x9d\x25\xaa\x95\x96\x9b\x45\x1d\x93\xc3\x1c\x6e"),
	},
};

const struct mac_vectors_st aes_gmac_192_vectors[] = { /* NIST test vectors */
	{
	 STR(key, key_size,
	     "\xaa\x92\x1c\xb5\xa2\x43\xab\x08\x91\x1f\x32\x89\x26\x6b\x39\xda"
	     "\xb1\x33\xf5\xc4\x20\xa6\xc5\xcd"),
	 STR(nonce, nonce_size,
	     "\x8f\x73\xdb\x68\xda\xee\xed\x2d\x15\x5f\xb1\xa0"),
	 STR(plaintext, plaintext_size,
	     "\x48\x74\x43\xc7\xc1\x4c\xe4\x74\xcb\x3d\x29\x1f\x25\x70\x70\xa2"),
	 STR(output, output_size,
	     "\xb1\x26\x74\xfb\xea\xc6\x88\x9a\x24\x94\x8f\x27\x92\xe3\x0a\x50"),
	},
};

const struct mac_vectors_st aes_gmac_256_vectors[] = { /* NIST test vectors */
	{
	 STR(key, key_size,
	     "\x6d\xfd\xaf\xd6\x70\x3c\x28\x5c\x01\xf1\x4f\xd1\x0a\x60\x12\x86"
	     "\x2b\x2a\xf9\x50\xd4\x73\x3a\xbb\x40\x3b\x2e\x74\x5b\x26\x94\x5d"),
	 STR(nonce, nonce_size,
	     "\x37\x49\xd0\xb3\xd5\xba\xcb\x71\xbe\x06\xad\xe6"),
	 STR(plaintext, plaintext_size,
	     "\xc0\xd2\x49\x87\x19\x92\xe7\x03\x02\xae\x00\x81\x93\xd1\xe8\x9f"),
	 STR(output, output_size,
	     "\x4a\xa4\xcc\x69\xf8\x4e\xe6\xac\x16\xd9\xbf\xb4\xe0\x5d\xe5\x00"),
	},
};

const struct mac_vectors_st gost28147_tc26z_imit_vectors[] = {
	{
		STR(key, key_size,
		    "\x9d\x05\xb7\x9e\x90\xca\xd0\x0a\x2c\xda\xd2\x2e\xf4\xe8\x6f\x5c"
		    "\xf5\xdc\x37\x68\x19\x85\xb3\xbf\xaa\x18\xc1\xc3\x05\x0a\x91\xa2"),
		STR(plaintext, plaintext_size,
		    "\xb5\xa1\xf0\xe3\xce\x2f\x02\x1d\x67\x61\x94\x34\x5c\x41\xe3\x6e"),
		STR(output, output_size,
		    "\x03\xe5\x67\x66"),
	},
};

/* GOST R 34.13-2015 */
const struct mac_vectors_st magma_omac_vectors[] = {
	{
        STR(key, key_size,
	    "\xff\xee\xdd\xcc\xbb\xaa\x99\x88\x77\x66\x55\x44\x33\x22\x11\x00"
	    "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff"),
        STR(plaintext, plaintext_size,
	    "\x92\xde\xf0\x6b\x3c\x13\x0a\x59"
	    "\xdb\x54\xc7\x04\xf8\x18\x9d\x20"
	    "\x4a\x98\xfb\x2e\x67\xa8\x02\x4c"
	    "\x89\x12\x40\x9b\x17\xb5\x7e\x41"),
	STR(output, output_size,
	    "\x15\x4e\x72\x10\x20\x30\xc5\xbb"),
	},
};

/* GOST R 34.13-2015 */
const struct mac_vectors_st kuznyechik_omac_vectors[] = {
	{
        STR(key, key_size,
	    "\x88\x99\xaa\xbb\xcc\xdd\xee\xff\x00\x11\x22\x33\x44\x55\x66\x77"
	    "\xfe\xdc\xba\x98\x76\x54\x32\x10\x01\x23\x45\x67\x89\xab\xcd\xef"),
        STR(plaintext, plaintext_size,
	    "\x11\x22\x33\x44\x55\x66\x77\x00\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
	    "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
	    "\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
	    "\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xee\xff\x0a\x00\x11"),
	STR(output, output_size,
	    "\x33\x6f\x4d\x29\x60\x59\xfb\xe3\x4d\xde\xb3\x5b\x37\x74\x9c\x67"),
	},
};

static int test_mac(gnutls_mac_algorithm_t mac,
		    const struct mac_vectors_st *vectors,
		    size_t vectors_size, unsigned flags)
{
	uint8_t data[HASH_DATA_SIZE];
	unsigned int i;
	int ret;
	size_t data_size;
	gnutls_hmac_hd_t hd;
	gnutls_hmac_hd_t copy;

	for (i = 0; i < vectors_size; i++) {
		ret = gnutls_hmac_init(&hd,
				       mac, vectors[i].key,
				       vectors[i].key_size);

		if (ret < 0) {
			_gnutls_debug_log("error initializing: MAC-%s\n",
					  gnutls_mac_get_name(mac));
			return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
		}

		if (vectors[i].nonce_size)
			gnutls_hmac_set_nonce(hd,
					      vectors[i].nonce,
					      vectors[i].nonce_size);

		ret = gnutls_hmac(hd, vectors[i].plaintext, 1);
		if (ret < 0)
			return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);

		copy = gnutls_hmac_copy(hd);
		if (!copy) {
			_gnutls_debug_log("copy is not supported for: %s\n",
					  gnutls_mac_get_name(mac));
		}

		ret = gnutls_hmac(hd,
				  &vectors[i].plaintext[1],
				  vectors[i].plaintext_size - 1);
		if (ret < 0)
			return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);

		gnutls_hmac_deinit(hd, data);

		data_size = gnutls_hmac_get_len(mac);
		if (data_size <= 0)
			return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);

		if (data_size != vectors[i].output_size ||
		    memcmp(data, vectors[i].output,
			   vectors[i].output_size) != 0) {

			_gnutls_debug_log
			    ("MAC-%s test vector %d failed!\n",
			     gnutls_mac_get_name(mac), i);

			return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
		}

		if (copy != NULL) {
			ret = gnutls_hmac(copy,
					  &vectors[i].plaintext[1],
					  vectors[i].plaintext_size - 1);
			if (ret < 0)
				return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);

			memset(data, 0xaa, data_size);
			gnutls_hmac_deinit(copy, data);

			if (memcmp(data, vectors[i].output,
			    vectors[i].output_size) != 0) {
				_gnutls_debug_log
					("MAC-%s copy test vector %d failed!\n",
					 gnutls_mac_get_name(mac), i);
				return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
			}
		}
	}

	_gnutls_debug_log
	    ("MAC-%s self check succeeded\n",
	     gnutls_mac_get_name(mac));

	return 0;
}

#define CASE(x, func, vectors) case x: \
			ret = func(x, V(vectors), flags); \
			if (!(flags & GNUTLS_SELF_TEST_FLAG_ALL) || ret < 0) \
				return ret

#define CASE2(x, func, func2, vectors) case x:	  \
			ret = func(x, V(vectors), flags); \
			if (!(flags & GNUTLS_SELF_TEST_FLAG_ALL) || ret < 0) \
				return ret; \
			ret = func2(x, V(vectors), flags); \
			if (!(flags & GNUTLS_SELF_TEST_FLAG_ALL) || ret < 0) \
				return ret

#define NON_FIPS_CASE(x, func, vectors) case x: \
			if (_gnutls_fips_mode_enabled() == 0) { \
				ret = func(x, V(vectors), flags); \
				if (!(flags & GNUTLS_SELF_TEST_FLAG_ALL) || ret < 0) \
					return ret; \
			}

/*-
 * gnutls_cipher_self_test:
 * @flags: GNUTLS_SELF_TEST_FLAG flags
 * @cipher: the encryption algorithm to use
 *
 * This function will run self tests on the provided cipher or all
 * available ciphers if @flags is %GNUTLS_SELF_TEST_FLAG_ALL.
 *
 * Returns: Zero or a negative error code on error.
 *
 * Since: 3.3.0-FIPS140
 -*/
int gnutls_cipher_self_test(unsigned flags, gnutls_cipher_algorithm_t cipher)
{
	int ret;

	if (flags & GNUTLS_SELF_TEST_FLAG_ALL)
		cipher = GNUTLS_CIPHER_UNKNOWN;

	switch (cipher) {
	case GNUTLS_CIPHER_UNKNOWN:
		CASE(GNUTLS_CIPHER_AES_128_CCM, test_cipher_aead,
		     aes128_ccm_vectors);
		FALLTHROUGH;
		CASE(GNUTLS_CIPHER_AES_256_CCM, test_cipher_aead,
		     aes256_ccm_vectors);
		FALLTHROUGH;
		CASE(GNUTLS_CIPHER_AES_128_CBC, test_cipher,
		     aes128_cbc_vectors);
		FALLTHROUGH;
		CASE(GNUTLS_CIPHER_AES_192_CBC, test_cipher,
		     aes192_cbc_vectors);
		FALLTHROUGH;
		CASE(GNUTLS_CIPHER_AES_256_CBC, test_cipher,
		     aes256_cbc_vectors);
		FALLTHROUGH;
		NON_FIPS_CASE(GNUTLS_CIPHER_3DES_CBC, test_cipher,
		     tdes_cbc_vectors);
		FALLTHROUGH;
		NON_FIPS_CASE(GNUTLS_CIPHER_ARCFOUR_128, test_cipher,
		     arcfour_vectors);
		FALLTHROUGH;
		CASE(GNUTLS_CIPHER_AES_128_GCM, test_cipher_aead,
		     aes128_gcm_vectors);
		FALLTHROUGH;
		CASE(GNUTLS_CIPHER_AES_192_GCM, test_cipher_aead,
		     aes192_gcm_vectors);
		FALLTHROUGH;
		CASE(GNUTLS_CIPHER_AES_256_GCM, test_cipher_aead,
		     aes256_gcm_vectors);
		FALLTHROUGH;
		NON_FIPS_CASE(GNUTLS_CIPHER_CHACHA20_POLY1305, test_cipher_aead,
		     chacha_poly1305_vectors);
		FALLTHROUGH;
		CASE2(GNUTLS_CIPHER_AES_128_CFB8, test_cipher,
		      test_cipher_all_block_sizes,
		      aes128_cfb8_vectors);
		FALLTHROUGH;
		CASE2(GNUTLS_CIPHER_AES_192_CFB8, test_cipher,
		      test_cipher_all_block_sizes,
		      aes192_cfb8_vectors);
		FALLTHROUGH;
		CASE2(GNUTLS_CIPHER_AES_256_CFB8, test_cipher,
		      test_cipher_all_block_sizes,
		      aes256_cfb8_vectors);
		FALLTHROUGH;
		CASE(GNUTLS_CIPHER_AES_128_XTS, test_cipher,
		     aes128_xts_vectors);
		FALLTHROUGH;
		CASE(GNUTLS_CIPHER_AES_256_XTS, test_cipher,
		     aes256_xts_vectors);
		FALLTHROUGH;
		NON_FIPS_CASE(GNUTLS_CIPHER_AES_128_SIV, test_cipher_aead,
		     aes128_siv_vectors);
		FALLTHROUGH;
		NON_FIPS_CASE(GNUTLS_CIPHER_AES_256_SIV, test_cipher_aead,
		     aes256_siv_vectors);
		FALLTHROUGH;
		NON_FIPS_CASE(GNUTLS_CIPHER_CHACHA20_32, test_cipher,
		     chacha20_32_vectors);
		FALLTHROUGH;
		/* The same test vector for _32 variant should work */
		NON_FIPS_CASE(GNUTLS_CIPHER_CHACHA20_64, test_cipher,
		     chacha20_32_vectors);
#if ENABLE_GOST
		FALLTHROUGH;
		NON_FIPS_CASE(GNUTLS_CIPHER_GOST28147_CPA_CFB, test_cipher,
			      gost28147_cpa_cfb_vectors);
		FALLTHROUGH;
		NON_FIPS_CASE(GNUTLS_CIPHER_GOST28147_CPB_CFB, test_cipher,
			      gost28147_cpb_cfb_vectors);
		FALLTHROUGH;
		NON_FIPS_CASE(GNUTLS_CIPHER_GOST28147_CPC_CFB, test_cipher,
			      gost28147_cpc_cfb_vectors);
		FALLTHROUGH;
		NON_FIPS_CASE(GNUTLS_CIPHER_GOST28147_CPD_CFB, test_cipher,
			      gost28147_cpd_cfb_vectors);
		FALLTHROUGH;
		NON_FIPS_CASE(GNUTLS_CIPHER_GOST28147_TC26Z_CFB, test_cipher,
			      gost28147_tc26z_cfb_vectors);
		FALLTHROUGH;
		NON_FIPS_CASE(GNUTLS_CIPHER_GOST28147_TC26Z_CNT, test_cipher,
			      gost28147_tc26z_cnt_vectors);
		FALLTHROUGH;
		NON_FIPS_CASE(GNUTLS_CIPHER_MAGMA_CTR_ACPKM, test_cipher,
			      magma_ctr_acpkm_vectors);
		FALLTHROUGH;
		NON_FIPS_CASE(GNUTLS_CIPHER_KUZNYECHIK_CTR_ACPKM, test_cipher,
			      kuznyechik_ctr_acpkm_vectors);
#endif
		break;
	default:
		return gnutls_assert_val(GNUTLS_E_NO_SELF_TEST);
	}

	return 0;
}

/*-
 * gnutls_mac_self_test:
 * @flags: GNUTLS_SELF_TEST_FLAG flags
 * @mac: the message authentication algorithm to use
 *
 * This function will run self tests on the provided mac.
 *
 * Returns: Zero or a negative error code on error.
 *
 * Since: 3.3.0-FIPS140
 -*/
int gnutls_mac_self_test(unsigned flags, gnutls_mac_algorithm_t mac)
{
	int ret;

	if (flags & GNUTLS_SELF_TEST_FLAG_ALL)
		mac = GNUTLS_MAC_UNKNOWN;

	switch (mac) {
	case GNUTLS_MAC_UNKNOWN:
		NON_FIPS_CASE(GNUTLS_MAC_MD5, test_mac, hmac_md5_vectors);
		FALLTHROUGH;
		CASE(GNUTLS_MAC_SHA1, test_mac, hmac_sha1_vectors);
		FALLTHROUGH;
		CASE(GNUTLS_MAC_SHA224, test_mac, hmac_sha224_vectors);
		FALLTHROUGH;
		CASE(GNUTLS_MAC_SHA256, test_mac, hmac_sha256_vectors);
		FALLTHROUGH;
		CASE(GNUTLS_MAC_SHA384, test_mac, hmac_sha384_vectors);
		FALLTHROUGH;
		CASE(GNUTLS_MAC_SHA512, test_mac, hmac_sha512_vectors);
#if ENABLE_GOST
		FALLTHROUGH;
		NON_FIPS_CASE(GNUTLS_MAC_GOSTR_94, test_mac, hmac_gostr_94_vectors);
		FALLTHROUGH;
		NON_FIPS_CASE(GNUTLS_MAC_STREEBOG_512, test_mac, hmac_streebog_512_vectors);
		FALLTHROUGH;
		NON_FIPS_CASE(GNUTLS_MAC_STREEBOG_256, test_mac, hmac_streebog_256_vectors);
		FALLTHROUGH;
		NON_FIPS_CASE(GNUTLS_MAC_GOST28147_TC26Z_IMIT, test_mac, gost28147_tc26z_imit_vectors);
		FALLTHROUGH;
		NON_FIPS_CASE(GNUTLS_MAC_MAGMA_OMAC, test_mac, magma_omac_vectors);
		FALLTHROUGH;
		NON_FIPS_CASE(GNUTLS_MAC_KUZNYECHIK_OMAC, test_mac, kuznyechik_omac_vectors);
#endif
		FALLTHROUGH;
		CASE(GNUTLS_MAC_AES_CMAC_128, test_mac, aes_cmac_128_vectors);
		FALLTHROUGH;
		CASE(GNUTLS_MAC_AES_CMAC_256, test_mac, aes_cmac_256_vectors);
		FALLTHROUGH;
		CASE(GNUTLS_MAC_AES_GMAC_128, test_mac, aes_gmac_128_vectors);
		FALLTHROUGH;
		CASE(GNUTLS_MAC_AES_GMAC_192, test_mac, aes_gmac_192_vectors);
		FALLTHROUGH;
		CASE(GNUTLS_MAC_AES_GMAC_256, test_mac, aes_gmac_256_vectors);

		break;
	default:
		return gnutls_assert_val(GNUTLS_E_NO_SELF_TEST);
	}

	return 0;
}

/*-
 * gnutls_digest_self_test:
 * @flags: GNUTLS_SELF_TEST_FLAG flags
 * @digest: the digest algorithm to use
 *
 * This function will run self tests on the provided digest.
 *
 * Returns: Zero or a negative error code on error.
 *
 * Since: 3.3.0-FIPS140
 -*/
int gnutls_digest_self_test(unsigned flags, gnutls_digest_algorithm_t digest)
{
	int ret;

	if (flags & GNUTLS_SELF_TEST_FLAG_ALL)
		digest = GNUTLS_DIG_UNKNOWN;

	switch (digest) {
	case GNUTLS_DIG_UNKNOWN:
		NON_FIPS_CASE(GNUTLS_DIG_MD5, test_digest, md5_vectors);
		FALLTHROUGH;
		CASE(GNUTLS_DIG_SHA1, test_digest, sha1_vectors);
		FALLTHROUGH;
		CASE(GNUTLS_DIG_SHA224, test_digest, sha224_vectors);
		FALLTHROUGH;
		CASE(GNUTLS_DIG_SHA256, test_digest, sha256_vectors);
		FALLTHROUGH;
		CASE(GNUTLS_DIG_SHA384, test_digest, sha384_vectors);
		FALLTHROUGH;
		CASE(GNUTLS_DIG_SHA512, test_digest, sha512_vectors);
#ifdef NETTLE_SHA3_FIPS202
		FALLTHROUGH;
		CASE(GNUTLS_DIG_SHA3_224, test_digest, sha3_224_vectors);
		FALLTHROUGH;
		CASE(GNUTLS_DIG_SHA3_256, test_digest, sha3_256_vectors);
		FALLTHROUGH;
		CASE(GNUTLS_DIG_SHA3_384, test_digest, sha3_384_vectors);
		FALLTHROUGH;
		CASE(GNUTLS_DIG_SHA3_512, test_digest, sha3_512_vectors);
#endif
#if ENABLE_GOST
		FALLTHROUGH;
		NON_FIPS_CASE(GNUTLS_DIG_GOSTR_94, test_digest, gostr_94_vectors);
		FALLTHROUGH;
		NON_FIPS_CASE(GNUTLS_DIG_STREEBOG_512, test_digest, streebog_512_vectors);
		FALLTHROUGH;
		NON_FIPS_CASE(GNUTLS_DIG_STREEBOG_256, test_digest, streebog_256_vectors);
#endif
		break;
	default:
		return gnutls_assert_val(GNUTLS_E_NO_SELF_TEST);
	}

	return 0;
}

struct hkdf_vectors_st {
	const uint8_t *ikm;
	unsigned int ikm_size;
	const uint8_t *salt;
	unsigned int salt_size;
	const uint8_t *prk;
	unsigned int prk_size;
	const uint8_t *info;
	unsigned int info_size;
	const uint8_t *okm;
	unsigned int okm_size;
};

const struct hkdf_vectors_st hkdf_sha256_vectors[] = {
	/* RFC 5869: A.1. Test Case 1: Basic test case with SHA-256 */
	{
		STR(ikm, ikm_size,
		    "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
		    "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"),
		STR(salt, salt_size,
		    "\x00\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0a\x0b\x0c"),
		STR(prk, prk_size,
		    "\x07\x77\x09\x36\x2c\x2e\x32\xdf\x0d\xdc\x3f\x0d\xc4\x7b"
		    "\xba\x63\x90\xb6\xc7\x3b\xb5\x0f\x9c\x31\x22\xec\x84\x4a"
		    "\xd7\xc2\xb3\xe5"),
		STR(info, info_size,
		    "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7\xf8\xf9"),
		STR(okm, okm_size,
		    "\x3c\xb2\x5f\x25\xfa\xac\xd5\x7a\x90\x43\x4f\x64\xd0\x36"
		    "\x2f\x2a\x2d\x2d\x0a\x90\xcf\x1a\x5a\x4c\x5d\xb0\x2d\x56"
		    "\xec\xc4\xc5\xbf\x34\x00\x72\x08\xd5\xb8\x87\x18\x58\x65"),
	},
	/* RFC 5869: A.2. Test Case 2: Test with SHA-256 and longer inputs/outputs */
	{
		STR(ikm, ikm_size,
		    "\x00\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0a\x0b\x0c\x0d"
		    "\x0e\x0f\x10\x11\x12\x13\x14\x15\x16\x17\x18\x19\x1a\x1b"
		    "\x1c\x1d\x1e\x1f\x20\x21\x22\x23\x24\x25\x26\x27\x28\x29"
		    "\x2a\x2b\x2c\x2d\x2e\x2f\x30\x31\x32\x33\x34\x35\x36\x37"
		    "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f\x40\x41\x42\x43\x44\x45"
		    "\x46\x47\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"),
		STR(salt, salt_size,
		    "\x60\x61\x62\x63\x64\x65\x66\x67\x68\x69\x6a\x6b\x6c\x6d"
		    "\x6e\x6f\x70\x71\x72\x73\x74\x75\x76\x77\x78\x79\x7a\x7b"
		    "\x7c\x7d\x7e\x7f\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89"
		    "\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97"
		    "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f\xa0\xa1\xa2\xa3\xa4\xa5"
		    "\xa6\xa7\xa8\xa9\xaa\xab\xac\xad\xae\xaf"),
		STR(prk, prk_size,
		    "\x06\xa6\xb8\x8c\x58\x53\x36\x1a\x06\x10\x4c\x9c\xeb\x35"
		    "\xb4\x5c\xef\x76\x00\x14\x90\x46\x71\x01\x4a\x19\x3f\x40"
		    "\xc1\x5f\xc2\x44"),
		STR(info, info_size,
		    "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7\xb8\xb9\xba\xbb\xbc\xbd"
		    "\xbe\xbf\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7\xc8\xc9\xca\xcb"
		    "\xcc\xcd\xce\xcf\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7\xd8\xd9"
		    "\xda\xdb\xdc\xdd\xde\xdf\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
		    "\xe8\xe9\xea\xeb\xec\xed\xee\xef\xf0\xf1\xf2\xf3\xf4\xf5"
		    "\xf6\xf7\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff"),
		STR(okm, okm_size,
		    "\xb1\x1e\x39\x8d\xc8\x03\x27\xa1\xc8\xe7\xf7\x8c\x59\x6a"
		    "\x49\x34\x4f\x01\x2e\xda\x2d\x4e\xfa\xd8\xa0\x50\xcc\x4c"
		    "\x19\xaf\xa9\x7c\x59\x04\x5a\x99\xca\xc7\x82\x72\x71\xcb"
		    "\x41\xc6\x5e\x59\x0e\x09\xda\x32\x75\x60\x0c\x2f\x09\xb8"
		    "\x36\x77\x93\xa9\xac\xa3\xdb\x71\xcc\x30\xc5\x81\x79\xec"
		    "\x3e\x87\xc1\x4c\x01\xd5\xc1\xf3\x43\x4f\x1d\x87"),
	},
};

static int test_hkdf(gnutls_mac_algorithm_t mac,
		     const struct hkdf_vectors_st *vectors,
		     size_t vectors_size, unsigned flags)
{
	unsigned int i;

	for (i = 0; i < vectors_size; i++) {
		gnutls_datum_t ikm, prk, salt, info;
		uint8_t output[4096];
		int ret;

		ikm.data = (void *) vectors[i].ikm;
		ikm.size = vectors[i].ikm_size;
		salt.data = (void *) vectors[i].salt;
		salt.size = vectors[i].salt_size;

		ret = gnutls_hkdf_extract(mac, &ikm, &salt, output);
		if (ret < 0) {
			_gnutls_debug_log("error extracting HKDF: MAC-%s\n",
					  gnutls_mac_get_name(mac));
			return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
		}

		if (memcmp(output, vectors[i].prk, vectors[i].prk_size) != 0) {
			_gnutls_debug_log
			    ("HKDF extract: MAC-%s test vector failed!\n",
			     gnutls_mac_get_name(mac));

			return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
		}

		prk.data = (void *) vectors[i].prk;
		prk.size = vectors[i].prk_size;
		info.data = (void *) vectors[i].info;
		info.size = vectors[i].info_size;

		ret = gnutls_hkdf_expand(mac, &prk, &info,
					 output, vectors[i].okm_size);
		if (ret < 0) {
			_gnutls_debug_log("error extracting HKDF: MAC-%s\n",
					  gnutls_mac_get_name(mac));
			return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
		}

		if (memcmp(output, vectors[i].okm, vectors[i].okm_size) != 0) {
			_gnutls_debug_log
			    ("HKDF expand: MAC-%s test vector failed!\n",
			     gnutls_mac_get_name(mac));

			return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
		}
	}

	_gnutls_debug_log
	    ("HKDF: MAC-%s self check succeeded\n",
	     gnutls_mac_get_name(mac));

	return 0;
}

/*-
 * gnutls_hkdf_self_test:
 * @flags: GNUTLS_SELF_TEST_FLAG flags
 * @mac: the message authentication algorithm to use
 *
 * This function will run self tests on HKDF with the provided mac.
 *
 * Returns: Zero or a negative error code on error.
 *
 * Since: 3.3.0-FIPS140
 -*/
int gnutls_hkdf_self_test(unsigned flags, gnutls_mac_algorithm_t mac)
{
	int ret;

	if (flags & GNUTLS_SELF_TEST_FLAG_ALL)
		mac = GNUTLS_MAC_UNKNOWN;

	switch (mac) {
	case GNUTLS_MAC_UNKNOWN:
		CASE(GNUTLS_MAC_SHA256, test_hkdf, hkdf_sha256_vectors);

		break;
	default:
		return gnutls_assert_val(GNUTLS_E_NO_SELF_TEST);
	}

	return 0;
}

struct pbkdf2_vectors_st {
	const uint8_t *key;
	size_t key_size;
	const uint8_t *salt;
	size_t salt_size;
	unsigned iter_count;
	const uint8_t *output;
	size_t output_size;
};

const struct pbkdf2_vectors_st pbkdf2_sha256_vectors[] = {
	/* RFC 7914: 11. Test Vectors for PBKDF2 with HMAC-SHA-256 */
	{
		STR(key, key_size, "passwd"),
		STR(salt, salt_size, "salt"),
		.iter_count = 1,
		STR(output, output_size,
		    "\x55\xac\x04\x6e\x56\xe3\x08\x9f\xec\x16\x91\xc2\x25\x44"
		    "\xb6\x05\xf9\x41\x85\x21\x6d\xde\x04\x65\xe6\x8b\x9d\x57"
		    "\xc2\x0d\xac\xbc\x49\xca\x9c\xcc\xf1\x79\xb6\x45\x99\x16"
		    "\x64\xb3\x9d\x77\xef\x31\x7c\x71\xb8\x45\xb1\xe3\x0b\xd5"
		    "\x09\x11\x20\x41\xd3\xa1\x97\x83"),
	},
	/* RFC 7914: 11. Test Vectors for PBKDF2 with HMAC-SHA-256 */
	{
		STR(key, key_size, "Password"),
		STR(salt, salt_size, "NaCl"),
		.iter_count = 80000,
		STR(output, output_size,
		    "\x4d\xdc\xd8\xf6\x0b\x98\xbe\x21\x83\x0c\xee\x5e\xf2\x27"
		    "\x01\xf9\x64\x1a\x44\x18\xd0\x4c\x04\x14\xae\xff\x08\x87"
		    "\x6b\x34\xab\x56\xa1\xd4\x25\xa1\x22\x58\x33\x54\x9a\xdb"
		    "\x84\x1b\x51\xc9\xb3\x17\x6a\x27\x2b\xde\xbb\xa1\xd0\x78"
		    "\x47\x8f\x62\xb3\x97\xf3\x3c\x8d"),
	},
	/* Test vector extracted from:
	 * https://dev.gnupg.org/source/libgcrypt/browse/master/cipher/kdf.c */
	{
		STR(key, key_size, "passwordPASSWORDpassword"),
		STR(salt, salt_size, "saltSALTsaltSALTsaltSALTsaltSALTsalt"),
		.iter_count = 4096,
		STR(output, output_size,
		    "\x34\x8c\x89\xdb\xcb\xd3\x2b\x2f\x32\xd8\x14\xb8\x11\x6e"
		    "\x84\xcf\x2b\x17\x34\x7e\xbc\x18\x00\x18\x1c\x4e\x2a\x1f"
		    "\xb8\xdd\x53\xe1\xc6\x35\x51\x8c\x7d\xac\x47\xe9"),
	},
};

static int test_pbkdf2(gnutls_mac_algorithm_t mac,
		       const struct pbkdf2_vectors_st *vectors,
		       size_t vectors_size, unsigned flags)
{
	unsigned int i;

	for (i = 0; i < vectors_size; i++) {
		gnutls_datum_t key, salt;
		uint8_t output[4096];
		int ret;

		key.data = (void *) vectors[i].key;
		key.size = vectors[i].key_size;
		salt.data = (void *) vectors[i].salt;
		salt.size = vectors[i].salt_size;

		ret = gnutls_pbkdf2(mac, &key, &salt, vectors[i].iter_count,
				    output, vectors[i].output_size);
		if (ret < 0) {
			_gnutls_debug_log("error calculating PBKDF2: MAC-%s\n",
					  gnutls_mac_get_name(mac));
			return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
		}

		if (memcmp(output, vectors[i].output, vectors[i].output_size) != 0) {
			_gnutls_debug_log
			    ("PBKDF2: MAC-%s test vector failed!\n",
			     gnutls_mac_get_name(mac));

			return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
		}
	}

	_gnutls_debug_log
	    ("PBKDF2: MAC-%s self check succeeded\n",
	     gnutls_mac_get_name(mac));

	return 0;
}

/*-
 * gnutls_pbkdf2_self_test:
 * @flags: GNUTLS_SELF_TEST_FLAG flags
 * @mac: the message authentication algorithm to use
 *
 * This function will run self tests on PBKDF2 with the provided mac.
 *
 * Returns: Zero or a negative error code on error.
 *
 * Since: 3.3.0-FIPS140
 -*/
int gnutls_pbkdf2_self_test(unsigned flags, gnutls_mac_algorithm_t mac)
{
	int ret;

	if (flags & GNUTLS_SELF_TEST_FLAG_ALL)
		mac = GNUTLS_MAC_UNKNOWN;

	switch (mac) {
	case GNUTLS_MAC_UNKNOWN:
		CASE(GNUTLS_MAC_SHA256, test_pbkdf2, pbkdf2_sha256_vectors);

		break;
	default:
		return gnutls_assert_val(GNUTLS_E_NO_SELF_TEST);
	}

	return 0;
}

struct tlsprf_vectors_st {
	const uint8_t *key;
	size_t key_size;
	const uint8_t *label;
	size_t label_size;
	const uint8_t *seed;
	size_t seed_size;
	const uint8_t *output;
	size_t output_size;
};

const struct tlsprf_vectors_st tls10prf_vectors[] = {
	/* tests/tls10-prf.c: test1 */
	{
		STR(key, key_size,
		    "\x26\x3b\xdb\xbb\x6f\x6d\x4c\x66\x4e\x05\x8d\x0a\xa9\xd3"
		    "\x21\xbe"),
		STR(label, label_size,
		    "test label"),
		STR(seed, seed_size,
		    "\xb9\x20\x57\x3b\x19\x96\x01\x02\x4f\x04\xd6\xdc\x61\x96"
		    "\x6e\x65"),
		STR(output, output_size,
		    "\x66\x17\x99\x37\x65\xfa\x6c\xa7\x03\xd1\x9e\xc7\x0d\xd5"
		    "\xdd\x16\x0f\xfc\xc0\x77\x25\xfa\xfb\x71\x4a\x9f\x81\x5a"
		    "\x2a\x30\xbf\xb7\xe3\xbb\xfb\x7e\xee\x57\x4b\x3b\x61\x3e"
		    "\xb7\xfe\x80\xee\xc9\x69\x1d\x8c\x1b\x0e\x2d\x9b\x3c\x8b"
		    "\x4b\x02\xb6\xb6\xd6\xdb\x88\xe2\x09\x46\x23\xef\x62\x40"
		    "\x60\x7e\xda\x7a\xbe\x3c\x84\x6e\x82\xa3"),
	},
};

const struct tlsprf_vectors_st tls12prf_sha256_vectors[] = {
	/* tests/tls12-prf.c: sha256_test1 */
	{
		STR(key, key_size,
		    "\x04\x50\xb0\xea\x9e\xcd\x36\x02\xee\x0d\x76\xc5\xc3\xc8"
		    "\x6f\x4a"),
		STR(label, label_size,
		    "test label"),
		STR(seed, seed_size,
		    "\x20\x7a\xcc\x02\x54\xb8\x67\xf5\xb9\x25\xb4\x5a\x33\x60"
		    "\x1d\x8b"),
		STR(output, output_size,
		    "\xae\x67\x9e\x0e\x71\x4f\x59\x75\x76\x37\x68\xb1\x66\x97"
		    "\x9e\x1d"),
	},
	/* tests/tls12-prf.c: sha256_test2 */
	{
		STR(key, key_size,
		    "\x34\x20\x4a\x9d\xf0\xbe\x6e\xb4\xe9\x25\xa8\x02\x7c\xf6"
		    "\xc6\x02"),
		STR(label, label_size,
		    "test label"),
		STR(seed, seed_size,
		    "\x98\xb2\xc4\x0b\xcd\x66\x4c\x83\xbb\x92\x0c\x18\x20\x1a"
		    "\x63\x95"),
		STR(output, output_size,
		    "\xaf\xa9\x31\x24\x53\xc2\x2f\xa8\x3d\x2b\x51\x1b\x37\x2d"
		    "\x73\xa4\x02\xa2\xa6\x28\x73\x23\x9a\x51\xfa\xde\x45\x08"
		    "\x2f\xaf\x3f\xd2\xbb\x7f\xfb\x3e\x9b\xf3\x6e\x28\xb3\x14"
		    "\x1a\xab\xa4\x84\x00\x53\x32\xa9\xf9\xe3\x88\xa4\xd3\x29"
		    "\xf1\x58\x7a\x4b\x31\x7d\xa0\x77\x08\xea\x1b\xa9\x5a\x53"
		    "\xf8\x78\x67\x24\xbd\x83\xce\x4b\x03\xaf"),
	},
	/* tests/tls12-prf.c: sha256_test3 */
	{
		STR(key, key_size,
		    "\xa3\x69\x1a\xa1\xf6\x81\x4b\x80\x59\x2b\xf1\xcf\x2a\xcf"
		    "\x16\x97"),
		STR(label, label_size,
		    "test label"),
		STR(seed, seed_size,
		    "\x55\x23\xd4\x1e\x32\x0e\x69\x4d\x0c\x1f\xf5\x73\x4d\x83"
		    "\x0b\x93\x3e\x46\x92\x70\x71\xc9\x26\x21"),
		STR(output, output_size,
		    "\x6a\xd0\x98\x4f\xa0\x6f\x78\xfe\x16\x1b\xd4\x6d\x7c\x26"
		    "\x1d\xe4\x33\x40\xd7\x28\xdd\xdc\x3d\x0f\xf0\xdd\x7e\x0d"),
	},
	/* tests/tls12-prf.c: sha256_test4 */
	{
		STR(key, key_size,
		    "\x21\x0e\xc9\x37\x06\x97\x07\xe5\x46\x5b\xc4\x6b\xf7\x79"
		    "\xe1\x04\x10\x8b\x18\xfd\xb7\x93\xbe\x7b\x21\x8d\xbf\x14"
		    "\x5c\x86\x41\xf3"),
		STR(label, label_size,
		    "test label"),
		STR(seed, seed_size,
		    "\x1e\x35\x1a\x0b\xaf\x35\xc7\x99\x45\x92\x43\x94\xb8\x81"
		    "\xcf\xe3\x1d\xae\x8f\x1c\x1e\xd5\x4d\x3b"),
		STR(output, output_size,
		    "\x76\x53\xfa\x80\x9c\xde\x3b\x55\x3c\x4a\x17\xe2\xcd\xbc"
		    "\xc9\x18\xf3\x65\x27\xf2\x22\x19\xa7\xd7\xf9\x5d\x97\x24"
		    "\x3f\xf2\xd5\xde\xe8\x26\x5e\xf0\xaf\x03"),
	},
};

const struct tlsprf_vectors_st tls12prf_sha384_vectors[] = {
	/* tests/tls12-prf.c: sha384_test1
	 * https://www.ietf.org/mail-archive/web/tls/current/msg03416.html
	 */
	{
		STR(key, key_size,
		    "\xb8\x0b\x73\x3d\x6c\xee\xfc\xdc\x71\x56\x6e\xa4\x8e\x55"
		    "\x67\xdf"),
		STR(label, label_size,
		    "test label"),
		STR(seed, seed_size,
		    "\xcd\x66\x5c\xf6\xa8\x44\x7d\xd6\xff\x8b\x27\x55\x5e\xdb"
		    "\x74\x65"),
		STR(output, output_size,
		    "\x7b\x0c\x18\xe9\xce\xd4\x10\xed\x18\x04\xf2\xcf\xa3\x4a"
		    "\x33\x6a\x1c\x14\xdf\xfb\x49\x00\xbb\x5f\xd7\x94\x21\x07"
		    "\xe8\x1c\x83\xcd\xe9\xca\x0f\xaa\x60\xbe\x9f\xe3\x4f\x82"
		    "\xb1\x23\x3c\x91\x46\xa0\xe5\x34\xcb\x40\x0f\xed\x27\x00"
		    "\x88\x4f\x9d\xc2\x36\xf8\x0e\xdd\x8b\xfa\x96\x11\x44\xc9"
		    "\xe8\xd7\x92\xec\xa7\x22\xa7\xb3\x2f\xc3\xd4\x16\xd4\x73"
		    "\xeb\xc2\xc5\xfd\x4a\xbf\xda\xd0\x5d\x91\x84\x25\x9b\x5b"
		    "\xf8\xcd\x4d\x90\xfa\x0d\x31\xe2\xde\xc4\x79\xe4\xf1\xa2"
		    "\x60\x66\xf2\xee\xa9\xa6\x92\x36\xa3\xe5\x26\x55\xc9\xe9"
		    "\xae\xe6\x91\xc8\xf3\xa2\x68\x54\x30\x8d\x5e\xaa\x3b\xe8"
		    "\x5e\x09\x90\x70\x3d\x73\xe5\x6f"),
	},
};

static int test_tlsprf(gnutls_mac_algorithm_t mac,
		       const struct tlsprf_vectors_st *vectors,
		       size_t vectors_size, unsigned flags)
{
	unsigned int i;

	for (i = 0; i < vectors_size; i++) {
		char output[4096];
		int ret;

		ret = _gnutls_prf_raw(mac,
				      vectors[i].key_size, vectors[i].key,
				      vectors[i].label_size, (const char *)vectors[i].label,
				      vectors[i].seed_size, vectors[i].seed,
				      vectors[i].output_size, output);
		if (ret < 0) {
			_gnutls_debug_log("error calculating TLS-PRF: MAC-%s\n",
					  gnutls_mac_get_name(mac));
			return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
		}

		if (memcmp(output, vectors[i].output, vectors[i].output_size) != 0) {
			_gnutls_debug_log
			    ("TLS-PRF: MAC-%s test vector failed!\n",
			     gnutls_mac_get_name(mac));

			return gnutls_assert_val(GNUTLS_E_SELF_TEST_ERROR);
		}
	}

	_gnutls_debug_log
	    ("TLS-PRF: MAC-%s self check succeeded\n",
	     gnutls_mac_get_name(mac));

	return 0;
}

/*-
 * gnutls_tlsprf_self_test:
 * @flags: GNUTLS_SELF_TEST_FLAG flags
 * @mac: the message authentication algorithm to use
 *
 * This function will run self tests on TLS-PRF with the provided mac.
 *
 * Returns: Zero or a negative error code on error.
 *
 * Since: 3.3.0-FIPS140
 -*/
int gnutls_tlsprf_self_test(unsigned flags, gnutls_mac_algorithm_t mac)
{
	int ret;

	if (flags & GNUTLS_SELF_TEST_FLAG_ALL)
		mac = GNUTLS_MAC_UNKNOWN;

	switch (mac) {
	case GNUTLS_MAC_UNKNOWN:
		NON_FIPS_CASE(GNUTLS_MAC_MD5_SHA1, test_tlsprf, tls10prf_vectors);
		FALLTHROUGH;
		CASE(GNUTLS_MAC_SHA256, test_tlsprf, tls12prf_sha256_vectors);
		FALLTHROUGH;
		CASE(GNUTLS_MAC_SHA384, test_tlsprf, tls12prf_sha384_vectors);

		break;
	default:
		return gnutls_assert_val(GNUTLS_E_NO_SELF_TEST);
	}

	return 0;
}
