// Copyright 2020 The TensorFlow Runtime Authors
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

//===- bef_executor_driver.cc - Library for bef_executor test driver ------===//
//
// This file implements the test driver library for the bef executor. It opens
// up a given mlir file and then runs it with a host executor.
#include "tfrt/bef_executor_driver/bef_executor_driver.h"

#include <cstdint>
#include <limits>
#include <utility>

#include "llvm/ADT/STLExtras.h"
#include "llvm/ADT/ScopeExit.h"
#include "llvm/Support/DynamicLibrary.h"
#include "llvm/Support/SourceMgr.h"
#include "llvm_derived/Support/raw_ostream.h"
#include "mlir/IR/BuiltinAttributes.h"
#include "mlir/IR/Diagnostics.h"
#include "mlir/IR/Location.h"
#include "mlir/IR/MLIRContext.h"
#include "mlir/Support/FileUtilities.h"
#include "tfrt/bef/bef_buffer.h"
#include "tfrt/bef_executor/bef_file.h"
#include "tfrt/core_runtime/core_runtime.h"
#include "tfrt/core_runtime/tensor_handle.h"
#include "tfrt/host_context/async_value.h"
#include "tfrt/host_context/concurrent_work_queue.h"
#include "tfrt/host_context/function.h"
#include "tfrt/host_context/host_allocator.h"
#include "tfrt/host_context/host_context.h"
#include "tfrt/host_context/kernel_registry.h"
#include "tfrt/host_context/location.h"
#include "tfrt/host_context/profiled_allocator.h"
#include "tfrt/host_context/resource_context.h"
#include "tfrt/host_context/value.h"
#include "tfrt/metrics/common_metrics.h"
#include "tfrt/support/mutex.h"
#include "tfrt/support/string_util.h"
#include "tfrt/tracing/tracing.h"

namespace tfrt {

static void RunBefFunction(
    HostContext* host, const Function& function,
    const std::function<llvm::Expected<ExecutionContext>(
        HostContext*, ResourceContext*)>& create_execution_context,
    bool print_error_code);

int RunBefExecutor(const RunBefConfig& run_config) {
  return RunBefExecutor(
      run_config,
      [](HostContext* host, ResourceContext* resource_context)
          -> llvm::Expected<ExecutionContext> {
        auto req_ctx = RequestContextBuilder(host, resource_context).build();
        if (!req_ctx) return req_ctx.takeError();
        return ExecutionContext{std::move(req_ctx.get())};
      });
}

int RunBefExecutor(
    const RunBefConfig& run_config,
    const std::function<llvm::Expected<ExecutionContext>(
        HostContext*, ResourceContext*)>& create_execution_context) {
  assert(create_execution_context);

  TFRT_TRACE_SCOPE(Default, "Bef Executor");
  metrics::AddTFRTVersionMetric();

  // Set up the input file.
  std::string error_message;
  auto file = mlir::openInputFile(run_config.input_filename, &error_message);
  if (!file) {
    llvm::errs() << error_message << "\n";
    return 1;
  }

  // Parse the input file.
  mlir::MLIRContext context;
  llvm::SourceMgr source_mgr;
  mlir::SourceMgrDiagnosticVerifierHandler source_mgr_handler(source_mgr,
                                                              &context);

  auto get_loc = [&](Optional<DecodedLocation> loc) -> mlir::Location {
    if (!loc) return mlir::UnknownLoc::get(&context);
    if (loc->is<FileLineColLocation>()) {
      auto file_loc = loc->get<FileLineColLocation>();
      return mlir::FileLineColLoc::get(&context, file_loc.filename,
                                       file_loc.line, file_loc.column);
    };
    return mlir::NameLoc::get(
        mlir::StringAttr::get(&context, loc->get<OpaqueLocation>().loc));
  };

  auto decoded_diagnostic_handler = [&](const DecodedDiagnostic& diag) {
    emitError(get_loc(diag.location)) << "runtime error: " << diag.message();
  };

  assert(GetNumReferenceCountedObjects() == 0 &&
         "We have reference-counted objects before we started to do anything");

  std::unique_ptr<HostAllocator> host_allocator;
  switch (run_config.host_allocator_type) {
    case HostAllocatorType::kMalloc:
      host_allocator = CreateMallocAllocator();
      tfrt::outs() << "Choosing malloc.\n";
      break;
    case HostAllocatorType::kTestFixedSizeMalloc:
      host_allocator = tfrt::CreateFixedSizeAllocator();
      tfrt::outs() << "Choosing fixed size malloc.\n";
      break;
    case HostAllocatorType::kProfiledMalloc:
      host_allocator = CreateMallocAllocator();
      host_allocator = CreateProfiledAllocator(std::move(host_allocator));
      tfrt::outs() << "Choosing profiled allocator based on malloc.\n";
      break;
    case HostAllocatorType::kLeakCheckMalloc:
      host_allocator = CreateMallocAllocator();
      host_allocator = CreateLeakCheckAllocator(std::move(host_allocator));
      tfrt::outs() << "Choosing memory leak check allocator.\n";
  }
  tfrt::outs().flush();

  auto buffer = file->getBuffer();

  // Handle BefBuffer alignment.
  //   mlir::openInputFile() should return 4KB aligned buffer when a file is
  //   memory-mapped. When the returned buffer is not aligned by 4KB, it could
  //   be from stdin by pipe operator.
  //   The following logic create an aligned buffer (BefBuffer),
  //   and copy the buffer contents.
  //   The original buffer cannot be released because of source_mgr_handler.
  llvm::ArrayRef<uint8_t> buffer_arr;
  BefBuffer aligned_bef_buffer;
  if (reinterpret_cast<uint64_t>(buffer.data()) % GetRequiredBefAlignment()) {
    aligned_bef_buffer.resize(buffer.size());
    std::memcpy(aligned_bef_buffer.data(), buffer.data(), buffer.size());
    buffer_arr = llvm::ArrayRef<uint8_t>(
        reinterpret_cast<const uint8_t*>(aligned_bef_buffer.data()),
        aligned_bef_buffer.size());
  } else {
    buffer_arr = llvm::ArrayRef<uint8_t>(
        reinterpret_cast<const uint8_t*>(buffer.data()), buffer.size());
  }

  std::unique_ptr<ConcurrentWorkQueue> work_queue =
      CreateWorkQueue(run_config.work_queue_type);
  if (work_queue == nullptr) {
    llvm::errs() << run_config.program_name
                 << ": couldn't create work queue type "
                 << run_config.work_queue_type << "\n";
    return 1;
  }
  tfrt::outs() << "Choosing " << work_queue->name() << " work queue.\n";
  tfrt::outs().flush();

  assert(AsyncValue::GetNumAsyncValueInstances() == 0 &&
         "We have async values allocated before we started to do anything");
  auto async_value_guard = llvm::make_scope_exit([]() {
    assert(AsyncValue::GetNumAsyncValueInstances() == 0 &&
           "All async values should be cleaned up at the end");
    assert(GetNumReferenceCountedObjects() == 0 &&
           "We have live reference-counted objects before exit.");
  });

  auto core_rt =
      CoreRuntime::Create(decoded_diagnostic_handler, std::move(host_allocator),
                          std::move(work_queue));
  if (!core_rt) {
    llvm::errs() << core_rt.takeError();
    return 1;
  }

  auto* host = core_rt.get()->GetHostContext();

  // If there are any libraries specified, load them and see if they have a
  // kernel registration function.
  for (const auto& lib_name : run_config.shared_libs) {
    std::string err;
    auto dyn_lib =
        llvm::sys::DynamicLibrary::getPermanentLibrary(lib_name.c_str(), &err);
    if (!dyn_lib.isValid()) {
      llvm::errs() << run_config.program_name << ": couldn't load library "
                   << err << "\n";
      return 1;
    }

    // The library should specify a kernel registration entrypoint.
    if (auto kernel_reg = dyn_lib.SearchForAddressOfSymbol("RegisterKernels")) {
      reinterpret_cast<void (*)(KernelRegistry*)>(kernel_reg)(
          host->GetMutableRegistry());
    }
  }

  auto bef(BEFFile::Open(buffer_arr, host->GetKernelRegistry(),
                         decoded_diagnostic_handler, host->allocator()));

  if (!bef) {
    return mlir::failed(source_mgr_handler.verify());
  }

  llvm::SmallVector<const Function*, 8> function_list;

  if (run_config.functions.empty()) {
    // No functions specified in the command line. Try to run all functions in
    // the input BEF file.
    bef->GetFunctionList(&function_list);
  } else {
    function_list.reserve(run_config.functions.size());

    for (auto& fn_name : run_config.functions) {
      auto* fn = bef->GetFunction(fn_name);

      if (!fn) {
        llvm::errs() << run_config.program_name << ": couldn't find function "
                     << fn_name << "\n";
        return 1;
      }
      function_list.push_back(fn);
    }
  }

  // Run the init function first if exists.
  auto test_init_function = bef->GetFunction(run_config.test_init_function);

  if (test_init_function) {
    RunBefFunction(host, *test_init_function, create_execution_context,
                   run_config.print_error_code);
  }

  // Loop over each of the functions, running each as a standalone testcase.
  for (auto* fn : function_list) {
    if (fn != test_init_function) {
      RunBefFunction(host, *fn, create_execution_context,
                     run_config.print_error_code);
    }
  }

  bef.reset();
  // Verify the diagnostic handler to make sure that each of the diagnostics
  // matched.
  return mlir::failed(source_mgr_handler.verify());
}

// Utility function to print the result. ValueType is either Value* or
// RCReference<AsyncValue>.
template <typename ValueType>
static void PrintResult(const TypeName& type_name, const ValueType& result) {
  if (type_name.GetName() == "i1") {
    tfrt::outs() << result->template get<bool>();
  } else if (type_name.GetName() == "i32") {
    tfrt::outs() << result->template get<int32_t>();
  } else if (type_name.GetName() == "i64") {
    tfrt::outs() << result->template get<int64_t>();
  } else if (type_name.GetName() == "f32") {
    tfrt::outs() << result->template get<float>();
  } else if (type_name.GetName() == "f64") {
    tfrt::outs() << result->template get<double>();
  } else {
    tfrt::outs() << type_name.GetName() << " value";
  }
}

static void RunSyncBefFunctionHelper(const ExecutionContext& exec_ctx,
                                     const Function& function) {
  TFRT_TRACE_SCOPE(Default, StrCat("Function: ", function.name()));

  llvm::SmallVector<Value, 4> results;
  results.resize(function.result_types().size());

  llvm::SmallVector<Value*, 4> result_ptrs;
  for (auto& value : results) {
    result_ptrs.emplace_back(&value);
  }

  auto error =
      ExecuteSyncBEFFunction(function, exec_ctx, /*arguments=*/{}, result_ptrs);

  // Go ahead and print out the function results that we know about.
  if (error) {
    tfrt::outs() << "'" << function.name() << "' returned ";
    tfrt::outs() << "<<error: " << error << ">>\n";
    tfrt::outs().flush();
  } else if (!results.empty()) {
    tfrt::outs() << "'" << function.name() << "' returned ";
    auto result_types = function.result_types();

    for (int i = 0, e = results.size(); i != e; ++i) {
      auto type_name = result_types[i];

      PrintResult(type_name, &results[i]);

      // Print comma except for the last one.
      if (i != results.size() - 1) {
        tfrt::outs() << ',';
      }
    }

    tfrt::outs() << '\n';
    tfrt::outs().flush();
  }
}

static void RunAsyncBefFunctionHelper(const ExecutionContext& exec_ctx,
                                      const Function& function,
                                      bool print_error_code) {
  TFRT_TRACE_SCOPE(Default, StrCat("Function: ", function.name()));

  // Kick off an execution of the function body.
  llvm::SmallVector<RCReference<AsyncValue>, 4> results;
  results.resize(function.result_types().size());

  function.Execute(exec_ctx, /*arguments=*/{}, results);

  // Block until the function results are fully resolved.
  exec_ctx.host()->Await(results);

  // Go ahead and print out the function results that we know about.
  if (!results.empty()) {
    tfrt::outs() << "'" << function.name() << "' returned ";
    auto result_types = function.result_types();

    for (int i = 0, e = results.size(); i != e; ++i) {
      auto type_name = result_types[i];

      if (auto* error = results[i]->GetErrorIfPresent()) {
        if (print_error_code)
          tfrt::outs() << "<<error: " << error->message()
                       << ", code: " << absl::StatusCodeToString(error->code())
                       << ">>";
        else
          tfrt::outs() << "<<error: " << error->message() << ">>";
      } else {
        PrintResult(type_name, results[i]);
      }

      // Print comma except for the last one.
      if (i != results.size() - 1) {
        tfrt::outs() << ',';
      }
    }

    tfrt::outs() << '\n';
    tfrt::outs().flush();
  }

  // In this test driver, we want to make sure that every function completes
  // all execution before moving on to the next one.  This makes the leak
  // checker work better in the face of side effecting kernels that aren't
  // properly chained together (which is useful for testing).
  exec_ctx.host()->Quiesce();
}

static void RunBefFunction(
    HostContext* host, const Function& function,
    const std::function<llvm::Expected<ExecutionContext>(
        HostContext*, ResourceContext*)>& create_execution_context,
    bool print_error_code) {
  // If the function takes arguments, then we can't run it from this driver.
  if (!function.argument_types().empty()) {
    tfrt::outs() << "--- Not running '" << function.name()
                 << "' because it has arguments.\n";
    tfrt::outs().flush();
    return;
  }

  // Skip anonymous functions.
  if (function.name().empty()) {
    return;
  }

  // Async value leak check before and after running the function.
  size_t before_num_values;
  if (AsyncValue::AsyncValueAllocationTrackingEnabled())
    before_num_values = AsyncValue::GetNumAsyncValueInstances();

  // Actually run the function.
  tfrt::outs() << "--- Running '" << function.name() << "':\n";
  tfrt::outs().flush();

  {
    // Add a ResourceContext ops/kernels to access resources. Shared across
    // kernels in this function, but not across functions.
    ResourceContext resource_context;
    // If any kernel calls RequestContext::Cancel, it will create an extra
    // async value that's stored inside RequestContext which is destroyed
    // only when RequestContext is destroyed.
    auto exec_ctx = create_execution_context(host, &resource_context);
    if (!exec_ctx) {
      llvm::errs() << "Failed to create execution context.\n";
      abort();
    }
    if (function.function_kind() == FunctionKind::kSyncBEFFunction) {
      RunSyncBefFunctionHelper(exec_ctx.get(), function);
    } else {
      RunAsyncBefFunctionHelper(exec_ctx.get(), function, print_error_code);
    }
  }

  if (AsyncValue::AsyncValueAllocationTrackingEnabled()) {
    auto after_num_values = AsyncValue::GetNumAsyncValueInstances();
    if (before_num_values != after_num_values) {
      llvm::errs() << "Evaluation of function '" << function.name()
                   << "' leaked " << (after_num_values - before_num_values)
                   << " async values (before: " << before_num_values
                   << ", after: " << after_num_values << ")!\n";
      abort();
    }
  }
}

}  // namespace tfrt
