package aQute.lib.bundle;

import java.io.*;
import java.util.*;
import java.util.jar.*;
import java.util.zip.*;

public class Bundle extends Jar {
	List					jars				= new Vector();
	Set						categories			= new TreeSet();
	boolean					update;
	Set						exported			= new TreeSet();
	Set						imported			= new TreeSet();
	Set						excluded			= new TreeSet();
	Set						extraImport			= new TreeSet();

	static public final int	PRIVATE				= 1;
	static public final int	EXPORT				= 2;
	static public final int	IMPORT				= 4;
	static public final int	EXCLUDE				= 8;

	static public String	name				= "untitled";
	static public String	outfile				= "untitled.jar";
	static public String	description			= "An untitled bundle: purpose, limitations, manual";
	static public String	copyright			= "(c) 2002, aQute, All Rights Reserved";
	static public String	vendor				= "aQute bv";
	static public String	docurl				= "www.aQute.se/untitled";
	static public String	version				= "1.0.0";

	static public String	BUNDLE_ACTIVATOR	= "Bundle-Activator";
	static public String	BUNDLE_NAME			= "Bundle-Name";
	static public String	BUNDLE_VERSION		= "Bundle-Version";
	static public String	BUNDLE_VENDOR		= "Bundle-Vendor";
	static public String	BUNDLE_DOCURL		= "Bundle-DocURL";
	static public String	BUNDLE_COPYRIGHT	= "Bundle-Copyright";
	static public String	IMPORT_PACKAGE		= "Import-Package";
	static public String	EXPORT_PACKAGE		= "Export-Package";
	static public String	MAIN_CLASS			= "Main-Class";

	static public String	J2B_MAIN_CLASS		= "J2B-Main-Class";
	static public String	J2B_STDIO			= "J2B-Stdio";
	static public String	J2B_SOURCES			= "J2B-Sources";
	static public String	J2B_EXCLUDE_PACKAGE	= "J2B-Exclude-Package";
	static public String	J2B_PUBLISHED		= "J2B-Published";
	static public String	J2B_URI				= "J2B-URI";

	/**
	 * Called when source is set and we update
	 */
	public void prepare() throws Exception {
		if (isUpdate()) {
			parse();
			System.out.println("Updating ");
			exported = parseTypeClause(getHeader(EXPORT_PACKAGE));
			imported = parseTypeClause(getHeader(IMPORT_PACKAGE));
			excluded = parseTypeClause(getHeader(J2B_EXCLUDE_PACKAGE));
			setJarsFromHeader();
			String version = getHeader(BUNDLE_VERSION);
			if (version != null) {
				System.out.println("Old version " + version);
				version = bump(version);
				System.out.println("New version " + version);
				setHeader(BUNDLE_VERSION, version);
			}
		} else
			init();
	}

	String bump(String version) {
		try {
			int index = version.lastIndexOf('.');
			String minor = version.substring(index + 1);
			int n = Integer.parseInt(minor);
			return version.substring(0, index + 1) + (n + 1);
		} catch (NumberFormatException e) {
			return version;
		}
	}

	/**
	 * Called when the JAR files have been added. It will analyze the jars and
	 * merge the information.
	 */

	public void consolidate() throws Exception {
		main = new TreeSet();
		activator = new TreeSet();
		web = new TreeSet();

		setHeader(J2B_SOURCES, join(getJars(), ",\r\n "));

		for (Iterator j = getJars(); j.hasNext();) {
			Jar jar = (Jar) j.next();
			activator.addAll(jar.getActivator());
			main.addAll(jar.getMain());
			web.addAll(jar.getWeb());
		}

		for (Iterator p = getPackages(-1).iterator(); p.hasNext();) {
			Package pack = (Package) p.next();
			int type = getType(pack);
			switch (type) {
				case PRIVATE :
					if (pack.isReference())
						setType(IMPORT, pack);
					break;
			}
		}
	}

	public void parseJar(InputStream in) throws Exception {
		JarInputStream jar = new JarInputStream(in);
		manifest = jar.getManifest();
		if (manifest != null)
			parseManifest(manifest);
		else
			System.out.println("No Manifest");
		jar.close();
	}

	public Set parseTypeClause(String clauses) {
		Set typeset = new TreeSet();
		if (clauses != null) {
			Set packages = split(clauses, ",");
			for (Iterator i = packages.iterator(); i.hasNext();) {
				String clause = ((String) i.next()).trim();
				Package pack = new Package(clause);
				typeset.add(pack);
			}
		}
		return typeset;
	}

	public void setUpdate(boolean update) {
		this.update = update;
	}
	public boolean isUpdate() {
		return update;
	}

	public Set setJarsFromHeader() {
		Set notfound = new TreeSet();

		Set jars = split(getHeader(J2B_SOURCES, ""), ",");
		for (Iterator i = jars.iterator(); i.hasNext();) {
			String file = ((String) i.next()).trim();
			try {
				File f = new File(file);
				if (f.exists()) {
					Jar jar = new Jar(file);
					addJar(jar);
				} else {
					System.err.println("Did not find jar " + f);
					notfound.add(file);
				}
			} catch (Exception e) {
				e.printStackTrace();
				notfound.add(file);
			}
		}
		return notfound;
	}

	public String getOutfile() {
		return outfile;
	}

	public void setOutfile(String outfile) {
		Bundle.outfile = outfile;
	}

	public void addJar(Jar jar) {
		jars.add(jar);
	}

	public void removeJar(Jar jar) {
		jars.remove(jar);
	}

	public Iterator getJars() {
		return jars.iterator();
	}

	// Carefull not to export ANYTHING before the Manifest!!
	// Cannot be read in that case.

	public void build() throws Exception {
		extraImport = new TreeSet();
		TreeMap visited = new TreeMap();
		JarOutputStream out = new JarOutputStream(new FileOutputStream(
				getOutfile()));

		String std = (String) getHeader(J2B_STDIO);
		if (std == null || std.equals("LOG")) {
			extraImport.add(new Package("org.osgi.service.log"));
			extraImport.add(new Package("org.osgi.util.tracker"));
		}

		String pub = (String) getHeader(J2B_PUBLISHED);
		if (pub != null) {
			extraImport.add(new Package("org.osgi.util.tracker"));
			extraImport.add(new Package("org.osgi.service.http"));
		}

		addToJar(out, visited, "META-INF/MANIFEST.MF", getManifest());

		copyResource(out, visited, "support/Activator.class",
				"aQute/lib/bundle/support/Activator.class");
		copyResource(out, visited, "support/XSystem.class",
				"aQute/lib/bundle/support/XSystem.class");
		copyResource(out, visited, "support/XRuntime.class",
				"aQute/lib/bundle/support/XRuntime.class");
		copyResource(out, visited, "support/LogStream.class",
				"aQute/lib/bundle/support/LogStream.class");
		copyResource(out, visited, "support/Publisher.class",
				"aQute/lib/bundle/support/Publisher.class");

		for (Iterator i = getPackages(EXPORT | PRIVATE).iterator(); i.hasNext();) {
			Package c = (Package) i.next();

			for (Iterator r = c.getResources(); r.hasNext();) {
				Resource rsrc = (Resource) r.next();
				System.out.println("  " + rsrc);
				addToJar(out, visited, rsrc.getPath(), rsrc.getBytes());
			}
		}
		out.close();
	}

	void copyResource(JarOutputStream out, Map visited, String resource,
			String path) throws IOException {
		ByteArrayOutputStream bout = new ByteArrayOutputStream();
		InputStream in = getClass().getResourceAsStream(resource);
		byte b[] = new byte[1024];
		int size = in.read(b);
		while (size > 0) {
			bout.write(b, 0, size);
			size = in.read(b);
		}
		byte cbs[] = bout.toByteArray();

		addToJar(out, visited, path, cbs);
	}

	public byte[] getManifest() throws IOException {
		printManifest(System.out);
		ByteArrayOutputStream out = new ByteArrayOutputStream();
		PrintStream pw = new PrintStream(out);
		printManifest(pw);
		pw.close();
		return out.toByteArray();
	}

	void createDirectories(JarOutputStream out, Map visited, String name)
			throws IOException {
		int index = name.lastIndexOf('/');
		if (index > 0) {
			String path = name.substring(0, index);
			if (visited.get(path) != null)
				return;

			createDirectories(out, visited, path);
			JarEntry ze = new JarEntry(path + '/');
			ze.setSize(0);
			CRC32 checksum = new CRC32();
			checksum.update(new byte[0]);
			ze.setCrc(checksum.getValue());
			out.putNextEntry(ze);
			out.closeEntry();
			visited.put(path, name);
		}
	}

	void addToJar(JarOutputStream out, Map set, String path, byte[] actual)
			throws IOException {
		createDirectories(out, set, path);
		try {
			CRC32 checksum = new CRC32();
			checksum.update(actual);

			JarEntry ze = new JarEntry(path);
			ze.setSize(actual.length);
			ze.setCrc(checksum.getValue());
			out.putNextEntry(ze);
			out.write(actual, 0, actual.length);
			out.closeEntry();
			System.out.println("Added " + actual.length);
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	public void printManifest(PrintStream pw) {
		ph(pw, "Manifest-Version", "1.0");
		removeHeader("import-package");
		removeHeader("export-package");
		for (Iterator i = headers.keySet().iterator(); i.hasNext();) {
			String key = (String) i.next();
			String value = getHeader(key);
			if (value != null && value.length() > 0)
				ph(pw, key, value);
		}
		Set exported = getPackages(EXPORT);
		printExportImport(pw, "Export-Package:", exported);
		Set imported = getPackages(IMPORT);
		imported.addAll(extraImport);
		Set filtered = new TreeSet();
		for (Iterator i = imported.iterator(); i.hasNext();) {
			Package pack = (Package) i.next();
			if (!pack.getName().startsWith("java."))
				filtered.add(pack);
		}
		printExportImport(pw, "Import-Package:", filtered);
		if (getHeader(Bundle.BUNDLE_ACTIVATOR) == null)
			pw.println(Bundle.BUNDLE_ACTIVATOR
					+ ": aQute.lib.bundle.support.Activator");
	}

	void ph(PrintStream pw, String header, String value) {
		if (value == null || value.length() < 1)
			return;
		StringBuffer sb = new StringBuffer();
		sb.append(header);
		sb.append(": ");
		sb.append(value);
		String s = format(sb.toString());
		pw.println(sb.toString());
	}

	String format(String s) {
		StringBuffer sb = new StringBuffer();

		int p = 0;
		for (int i = 0; i < s.length(); i++) {
			char c = s.charAt(i);
			switch (c) {
				case '\r' :
				case '\n' :
					if (p == 1)
						break;

					p = 1;
					sb.append("\r\n ");
					break;

				default :
					if (p > 76) {
						p = 1;
						sb.append("\r\n ");
					}
					sb.append(c);
					p++;
					break;
			}
		}
		sb.append("\r\n");
		return sb.toString();
	}

	void printExportImport(PrintStream pw, String header, Set packages) {
		if (packages.size() > 0) {
			pw.print(header);
			String del = " ";
			for (Iterator i = packages.iterator(); i.hasNext();) {
				Package p = (Package) i.next();
				pw.print(del + p.getSpecification());
				del = ",\r\n ";
			}
			pw.print("\r\n");
		}
	}

	public Set split(String s, String separators) {
		StringTokenizer st = new StringTokenizer(s, separators);
		TreeSet set = new TreeSet();

		while (st.hasMoreTokens())
			set.add(st.nextToken().trim());

		return set;
	}

	public String join(Iterator i, String separator) {
		String del = "";
		StringBuffer sb = new StringBuffer();
		while (i.hasNext()) {
			sb.append(del);
			sb.append(i.next());
			del = separator;
		}
		return sb.toString();
	}

	public int getType(Package p) {
		if (exported.contains(p))
			return EXPORT;

		if (imported.contains(p))
			return IMPORT;

		if (excluded.contains(p))
			return EXCLUDE;

		return PRIVATE;
	}

	public void setType(int type, Package p) {
		exported.remove(p);
		imported.remove(p);
		excluded.remove(p);
		switch (type) {
			case EXPORT :
				if (p.isReference())
					throw new IllegalArgumentException(
							"Cannot export a reference");
				exported.add(p);
				return;

			case IMPORT :
				imported.add(p);
				return;

			case EXCLUDE :
				excluded.add(p);
				return;
		}
	}

	public Set getPackages(int type) {
		Set result = new TreeSet();
		for (Iterator j = getJars(); j.hasNext();) {
			Jar jar = (Jar) j.next();
			for (Iterator p = jar.getPackages().iterator(); p.hasNext();) {
				Package pack = (Package) p.next();
				if ((getType(pack) & type) != 0)
					result.add(pack);
			}
		}
		return result;
	}
}