/* Copyright (c) 2023 Intel Corporation

Copyright 2017 The TensorFlow Authors. All Rights Reserved.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
==============================================================================*/

#ifndef ITEX_CORE_COMPILER_XLA_SERVICE_GPU_FFT_THUNK_H_
#define ITEX_CORE_COMPILER_XLA_SERVICE_GPU_FFT_THUNK_H_
#if ITEX_USE_MKL

#include <memory>
#include <vector>

#include "absl/container/flat_hash_map.h"
#include "absl/types/optional.h"
#include "itex/core/compiler/xla/service/buffer_assignment.h"
#include "itex/core/compiler/xla/service/gpu/buffer_allocations.h"
#include "itex/core/compiler/xla/service/gpu/gpu_executable.h"
#include "itex/core/compiler/xla/service/gpu/thunk.h"
#include "itex/core/compiler/xla/service/hlo_instruction.h"
#include "itex/core/compiler/xla/stream_executor/scratch_allocator.h"
#include "itex/core/compiler/xla/types.h"
#include "itex/core/utils/status.h"
#include "mkl.h"  // NOLINT(build/include_subdir)
#include "oneapi/mkl/dfti.hpp"
#include "oneapi/mkl/exceptions.hpp"
#include "protos/xla_data.pb.h"

namespace stream_executor {
namespace fft {

// Specifies FFT input and output types, and the direction.
// R, D, C, and Z stand for SP real, DP real, SP complex, and DP complex.
enum class Type {
  kInvalid,
  kC2CForward,
  kC2CInverse,
  kC2R,
  kR2C,
  kZ2ZForward,
  kZ2ZInverse,
  kZ2D,
  kD2Z
};
}  // namespace fft
}  // namespace stream_executor

namespace itex_xla {
namespace gpu {

// This class stores everything that StreamExecutor needs to launch an FFT.
// It is generated by IrEmitter.
//
// This is thread-compatible.
class FftThunk : public Thunk {
 public:
  // Constructs a thunk for launching an FFT on a stream.
  // Semantics of null hlo_instruction argument are as in Thunk.
  FftThunk(ThunkInfo thunk_info, FftType fft_type,
           absl::Span<const int64_t> fft_length,
           const BufferAllocation::Slice& input_buffer,
           const BufferAllocation::Slice& output_buffer,
           const Shape& input_shape, const Shape& output_shape);

  FftThunk(const FftThunk&) = delete;             // Cannot share fft_plan_
  FftThunk& operator=(const FftThunk&) = delete;  // Cannot share fft_plan_

  // Does the FFT for the thunk on "stream".
  Status ExecuteOnStream(const ExecuteParams& params) override;

 private:
  template <oneapi::mkl::dft::precision P, oneapi::mkl::dft::domain D,
            typename T>
  Status DoFFTInternal(se::Stream* stream, T* in_buffer, T* out_buffer,
                       se::ScratchAllocator& allocator);

  const se::fft::Type fft_type_;
  const std::vector<int64_t> fft_length_;

  double scale_factor_;

  const BufferAllocation::Slice input_buffer_;
  const BufferAllocation::Slice output_buffer_;

  const Shape input_shape_;
  const Shape output_shape_;
};

}  // namespace gpu
}  // namespace itex_xla

#endif  // ITEX_USE_MKL
#endif  // ITEX_CORE_COMPILER_XLA_SERVICE_GPU_FFT_THUNK_H_
